import { IAspect } from 'aws-cdk-lib';
import { IConstruct } from 'constructs';
import { IApplication } from '../application';
import { ApplicationAssociator } from '../application-associator';
/**
 * Aspect class, this will visit each node from the provided construct once.
 *
 * For every stack node visited, this class will be responsible to associate
 * the stack to the application.
 */
declare abstract class StackAssociatorBase implements IAspect {
    protected abstract readonly application: IApplication;
    protected abstract readonly applicationAssociator?: ApplicationAssociator;
    protected readonly sharedAccounts: Set<string>;
    visit(node: IConstruct): void;
    /**
     * Associate a stage stack to the given application.
     *
     * @param node A Stage stack.
     */
    private associate;
    /**
     * Adds an error annotation to a node.
     *
     * @param node The scope to add the error to.
     * @param message The error message.
     */
    private error;
    /**
     * Adds a warning annotation to a node.
     *
     * @param node The scope to add the warning to.
     * @param message The error message.
     */
    private warning;
    /**
     * Handle cross-region association. AppRegistry do not support
     * cross region association at this moment,
     * If any stack is evaluated as cross-region than that of application,
     * we will throw an error.
     *
     * @param node Cfn stack.
     */
    private handleCrossRegionStack;
    /**
     * Handle cross-account association.
     * If any stack is evaluated as cross-account than that of application,
     * then we will share the application to the stack owning account.
     *
     * @param node Cfn stack.
     */
    private handleCrossAccountStack;
}
export declare class CheckedStageStackAssociator extends StackAssociatorBase {
    protected readonly application: IApplication;
    protected readonly applicationAssociator?: ApplicationAssociator;
    constructor(app: ApplicationAssociator);
}
export declare class StageStackAssociator extends StackAssociatorBase {
    protected readonly application: IApplication;
    protected readonly applicationAssociator?: ApplicationAssociator;
    constructor(app: IApplication);
}
export {};
