"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Application = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ram_1 = require("aws-cdk-lib/aws-ram");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const stack_associator_1 = require("./aspects/stack-associator");
const common_1 = require("./common");
const utils_1 = require("./private/utils");
const validation_1 = require("./private/validation");
const aws_servicecatalogappregistry_1 = require("aws-cdk-lib/aws-servicecatalogappregistry");
const APPLICATION_READ_ONLY_RAM_PERMISSION_ARN = 'arn:aws:ram::aws:permission/AWSRAMPermissionServiceCatalogAppRegistryApplicationReadOnly';
const APPLICATION_ALLOW_ACCESS_RAM_PERMISSION_ARN = 'arn:aws:ram::aws:permission/AWSRAMPermissionServiceCatalogAppRegistryApplicationAllowAssociation';
class ApplicationBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.associatedAttributeGroups = new Set();
        this.associatedResources = new Set();
    }
    /**
     * Associate an attribute group with application
     * If the attribute group is already associated, it will ignore duplicate request.
     */
    associateAttributeGroup(attributeGroup) {
        if (!this.associatedAttributeGroups.has(attributeGroup.node.addr)) {
            const hashId = this.generateUniqueHash(attributeGroup.node.addr);
            new aws_servicecatalogappregistry_1.CfnAttributeGroupAssociation(this, `AttributeGroupAssociation${hashId}`, {
                application: this.applicationId,
                attributeGroup: attributeGroup.attributeGroupId,
            });
            this.associatedAttributeGroups.add(attributeGroup.node.addr);
        }
    }
    /**
     * Associate a stack with the application
     * If the resource is already associated, it will ignore duplicate request.
     * A stack can only be associated with one application.
     *
     * @deprecated Use `associateApplicationWithStack` instead.
     */
    associateStack(stack) {
        if (!this.associatedResources.has(stack.node.addr)) {
            const hashId = this.generateUniqueHash(stack.node.addr);
            new aws_servicecatalogappregistry_1.CfnResourceAssociation(this, `ResourceAssociation${hashId}`, {
                application: this.applicationId,
                resource: stack.stackId,
                resourceType: 'CFN_STACK',
            });
            this.associatedResources.add(stack.node.addr);
        }
    }
    /**
     * Associate stack with the application in the stack passed as parameter.
     *
     * A stack can only be associated with one application.
     */
    associateApplicationWithStack(stack) {
        if (!this.associatedResources.has(stack.node.addr)) {
            new aws_servicecatalogappregistry_1.CfnResourceAssociation(stack, 'AppRegistryAssociation', {
                application: stack === cdk.Stack.of(this) ? this.applicationId : this.applicationName ?? this.applicationId,
                resource: stack.stackId,
                resourceType: 'CFN_STACK',
            });
            this.associatedResources.add(stack.node.addr);
            if (stack !== cdk.Stack.of(this) && this.isSameAccount(stack) && !this.isStageScope(stack) && !stack.nested) {
                stack.addDependency(cdk.Stack.of(this));
            }
        }
    }
    /**
     * Share an application with accounts, organizations and OUs, and IAM roles and users.
     * The application will become available to end users within those principals.
     *
     * @param shareOptions The options for the share.
     */
    shareApplication(shareOptions) {
        const principals = common_1.getPrincipalsforSharing(shareOptions);
        const shareName = `RAMShare${common_1.hashValues(aws_cdk_lib_1.Names.nodeUniqueId(this.node), this.node.children.length.toString())}`;
        new aws_ram_1.CfnResourceShare(this, shareName, {
            name: shareName,
            allowExternalPrincipals: false,
            principals: principals,
            resourceArns: [this.applicationArn],
            permissionArns: [this.getApplicationSharePermissionARN(shareOptions)],
        });
    }
    /**
     * Associate all stacks present in construct's aspect with application.
     *
     * NOTE: This method won't automatically register stacks under pipeline stages,
     * and requires association of each pipeline stage by calling this method with stage Construct.
     *
     */
    associateAllStacksInScope(scope) {
        cdk.Aspects.of(scope).add(new stack_associator_1.StageStackAssociator(this));
    }
    /**
     * Get the correct permission ARN based on the SharePermission
     */
    getApplicationSharePermissionARN(shareOptions) {
        switch (shareOptions.sharePermission) {
            case common_1.SharePermission.ALLOW_ACCESS:
                return APPLICATION_ALLOW_ACCESS_RAM_PERMISSION_ARN;
            case common_1.SharePermission.READ_ONLY:
                return APPLICATION_READ_ONLY_RAM_PERMISSION_ARN;
            default:
                return shareOptions.sharePermission ?? APPLICATION_READ_ONLY_RAM_PERMISSION_ARN;
        }
    }
    /**
    *  Checks whether a stack is defined in a Stage or not.
    */
    isStageScope(stack) {
        return !(stack.node.scope instanceof cdk.App) && (stack.node.scope instanceof cdk.Stage);
    }
    /**
     * Verifies if application and the visited node is deployed in different account.
     */
    isSameAccount(stack) {
        return utils_1.isAccountUnresolved(this.env.account, stack.account) || this.env.account === stack.account;
    }
}
/**
 * A Service Catalog AppRegistry Application.
 */
class Application extends ApplicationBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_servicecatalogappregistry_alpha_ApplicationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Application);
            }
            throw error;
        }
        this.validateApplicationProps(props);
        const application = new aws_servicecatalogappregistry_1.CfnApplication(this, 'Resource', {
            name: props.applicationName,
            description: props.description,
        });
        this.applicationArn = application.attrArn;
        this.applicationId = application.attrId;
        this.applicationName = props.applicationName;
        this.nodeAddress = cdk.Names.nodeUniqueId(application.node);
        this.applicationManagerUrl = new cdk.CfnOutput(this, 'ApplicationManagerUrl', {
            value: `https://${this.env.region}.console.aws.amazon.com/systems-manager/appmanager/application/AWS_AppRegistry_Application-${this.applicationName}`,
            description: 'Application manager url for the application created.',
        });
    }
    /**
     * Imports an Application construct that represents an external application.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param applicationArn the Amazon Resource Name of the existing AppRegistry Application
     */
    static fromApplicationArn(scope, id, applicationArn) {
        const arn = cdk.Stack.of(scope).splitArn(applicationArn, cdk.ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME);
        const applicationId = arn.resourceName;
        if (!applicationId) {
            throw new Error('Missing required Application ID from Application ARN: ' + applicationArn);
        }
        class Import extends ApplicationBase {
            constructor() {
                super(...arguments);
                this.applicationArn = applicationArn;
                this.applicationId = applicationId;
                this.applicationName = undefined;
            }
            generateUniqueHash(resourceAddress) {
                return common_1.hashValues(this.applicationArn, resourceAddress);
            }
        }
        return new Import(scope, id, {
            environmentFromArn: applicationArn,
        });
    }
    generateUniqueHash(resourceAddress) {
        return common_1.hashValues(this.nodeAddress, resourceAddress);
    }
    validateApplicationProps(props) {
        validation_1.InputValidator.validateLength(this.node.path, 'application name', 1, 256, props.applicationName);
        validation_1.InputValidator.validateRegex(this.node.path, 'application name', /^[a-zA-Z0-9-_]+$/, props.applicationName);
        validation_1.InputValidator.validateLength(this.node.path, 'application description', 0, 1024, props.description);
    }
}
exports.Application = Application;
_a = JSII_RTTI_SYMBOL_1;
Application[_a] = { fqn: "@aws-cdk/aws-servicecatalogappregistry-alpha.Application", version: "2.63.2-alpha.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwbGljYXRpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhcHBsaWNhdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSxpREFBdUQ7QUFDdkQsbUNBQW1DO0FBQ25DLDZDQUFvQztBQUVwQyxpRUFBa0U7QUFFbEUscUNBQThGO0FBQzlGLDJDQUFzRDtBQUN0RCxxREFBc0Q7QUFDdEQsNkZBQWlJO0FBRWpJLE1BQU0sd0NBQXdDLEdBQUcsMEZBQTBGLENBQUM7QUFDNUksTUFBTSwyQ0FBMkMsR0FBRyxrR0FBa0csQ0FBQztBQWdGdkosTUFBZSxlQUFnQixTQUFRLEdBQUcsQ0FBQyxRQUFRO0lBQW5EOztRQUltQiw4QkFBeUIsR0FBZ0IsSUFBSSxHQUFHLEVBQUUsQ0FBQztRQUNuRCx3QkFBbUIsR0FBZ0IsSUFBSSxHQUFHLEVBQUUsQ0FBQztJQXVIaEUsQ0FBQztJQXJIQzs7O09BR0c7SUFDSSx1QkFBdUIsQ0FBQyxjQUErQjtRQUM1RCxJQUFJLENBQUMsSUFBSSxDQUFDLHlCQUF5QixDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFO1lBQ2pFLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ2pFLElBQUksNERBQTRCLENBQUMsSUFBSSxFQUFFLDRCQUE0QixNQUFNLEVBQUUsRUFBRTtnQkFDM0UsV0FBVyxFQUFFLElBQUksQ0FBQyxhQUFhO2dCQUMvQixjQUFjLEVBQUUsY0FBYyxDQUFDLGdCQUFnQjthQUNoRCxDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMseUJBQXlCLENBQUMsR0FBRyxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDOUQ7S0FDRjtJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWMsQ0FBQyxLQUFnQjtRQUNwQyxJQUFJLENBQUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFO1lBQ2xELE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ3hELElBQUksc0RBQXNCLENBQUMsSUFBSSxFQUFFLHNCQUFzQixNQUFNLEVBQUUsRUFBRTtnQkFDL0QsV0FBVyxFQUFFLElBQUksQ0FBQyxhQUFhO2dCQUMvQixRQUFRLEVBQUUsS0FBSyxDQUFDLE9BQU87Z0JBQ3ZCLFlBQVksRUFBRSxXQUFXO2FBQzFCLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUMvQztLQUNGO0lBRUQ7Ozs7T0FJRztJQUNJLDZCQUE2QixDQUFDLEtBQWdCO1FBQ25ELElBQUksQ0FBQyxJQUFJLENBQUMsbUJBQW1CLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDbEQsSUFBSSxzREFBc0IsQ0FBQyxLQUFLLEVBQUUsd0JBQXdCLEVBQUU7Z0JBQzFELFdBQVcsRUFBRSxLQUFLLEtBQUssR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxlQUFlLElBQUksSUFBSSxDQUFDLGFBQWE7Z0JBQzNHLFFBQVEsRUFBRSxLQUFLLENBQUMsT0FBTztnQkFDdkIsWUFBWSxFQUFFLFdBQVc7YUFDMUIsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzlDLElBQUksS0FBSyxLQUFLLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxJQUFJLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sRUFBRTtnQkFDM0csS0FBSyxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO2FBQ3pDO1NBQ0Y7S0FDRjtJQUVEOzs7OztPQUtHO0lBQ0ksZ0JBQWdCLENBQUMsWUFBMEI7UUFDaEQsTUFBTSxVQUFVLEdBQUcsZ0NBQXVCLENBQUMsWUFBWSxDQUFDLENBQUM7UUFDekQsTUFBTSxTQUFTLEdBQUcsV0FBVyxtQkFBVSxDQUFDLG1CQUFLLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsUUFBUSxFQUFFLENBQUMsRUFBRSxDQUFDO1FBQy9HLElBQUksMEJBQWdCLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUNwQyxJQUFJLEVBQUUsU0FBUztZQUNmLHVCQUF1QixFQUFFLEtBQUs7WUFDOUIsVUFBVSxFQUFFLFVBQVU7WUFDdEIsWUFBWSxFQUFFLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQztZQUNuQyxjQUFjLEVBQUUsQ0FBQyxJQUFJLENBQUMsZ0NBQWdDLENBQUMsWUFBWSxDQUFDLENBQUM7U0FDdEUsQ0FBQyxDQUFDO0tBQ0o7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUIsQ0FBQyxLQUFnQjtRQUMvQyxHQUFHLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxHQUFHLENBQUMsSUFBSSx1Q0FBb0IsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO0tBQzNEO0lBT0Q7O09BRUc7SUFDSyxnQ0FBZ0MsQ0FBQyxZQUEwQjtRQUNqRSxRQUFRLFlBQVksQ0FBQyxlQUFlLEVBQUU7WUFDcEMsS0FBSyx3QkFBZSxDQUFDLFlBQVk7Z0JBQy9CLE9BQU8sMkNBQTJDLENBQUM7WUFDckQsS0FBSyx3QkFBZSxDQUFDLFNBQVM7Z0JBQzVCLE9BQU8sd0NBQXdDLENBQUM7WUFFbEQ7Z0JBQ0UsT0FBTyxZQUFZLENBQUMsZUFBZSxJQUFJLHdDQUF3QyxDQUFDO1NBQ25GO0tBQ0Y7SUFFRDs7TUFFRTtJQUNNLFlBQVksQ0FBQyxLQUFpQjtRQUNwQyxPQUFPLENBQUMsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEtBQUssWUFBWSxHQUFHLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEtBQUssWUFBWSxHQUFHLENBQUMsS0FBSyxDQUFDLENBQUM7S0FDMUY7SUFFRDs7T0FFRztJQUNLLGFBQWEsQ0FBQyxLQUFnQjtRQUNwQyxPQUFPLDJCQUFtQixDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxJQUFJLENBQUMsR0FBRyxDQUFDLE9BQU8sS0FBSyxLQUFLLENBQUMsT0FBTyxDQUFDO0tBQ25HO0NBRUY7QUFFRDs7R0FFRztBQUNILE1BQWEsV0FBWSxTQUFRLGVBQWU7SUF5QzlDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBdUI7UUFDL0QsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQzs7Ozs7OytDQTFDUixXQUFXOzs7O1FBNENwQixJQUFJLENBQUMsd0JBQXdCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFckMsTUFBTSxXQUFXLEdBQUcsSUFBSSw4Q0FBYyxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDdkQsSUFBSSxFQUFFLEtBQUssQ0FBQyxlQUFlO1lBQzNCLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztTQUMvQixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsY0FBYyxHQUFHLFdBQVcsQ0FBQyxPQUFPLENBQUM7UUFDMUMsSUFBSSxDQUFDLGFBQWEsR0FBRyxXQUFXLENBQUMsTUFBTSxDQUFDO1FBQ3hDLElBQUksQ0FBQyxlQUFlLEdBQUcsS0FBSyxDQUFDLGVBQWUsQ0FBQztRQUM3QyxJQUFJLENBQUMsV0FBVyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUU1RCxJQUFJLENBQUMscUJBQXFCLEdBQUcsSUFBSSxHQUFHLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSx1QkFBdUIsRUFBRTtZQUM1RSxLQUFLLEVBQUUsV0FBVyxJQUFJLENBQUMsR0FBRyxDQUFDLE1BQU0sOEZBQThGLElBQUksQ0FBQyxlQUFlLEVBQUU7WUFDckosV0FBVyxFQUFFLHNEQUFzRDtTQUNwRSxDQUFDLENBQUM7S0FDSjtJQTNERDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsa0JBQWtCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsY0FBc0I7UUFDbkYsTUFBTSxHQUFHLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUMsUUFBUSxDQUFDLGNBQWMsRUFBRSxHQUFHLENBQUMsU0FBUyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDM0csTUFBTSxhQUFhLEdBQUcsR0FBRyxDQUFDLFlBQVksQ0FBQztRQUV2QyxJQUFJLENBQUMsYUFBYSxFQUFFO1lBQ2xCLE1BQU0sSUFBSSxLQUFLLENBQUMsd0RBQXdELEdBQUcsY0FBYyxDQUFDLENBQUM7U0FDNUY7UUFFRCxNQUFNLE1BQU8sU0FBUSxlQUFlO1lBQXBDOztnQkFDa0IsbUJBQWMsR0FBRyxjQUFjLENBQUM7Z0JBQ2hDLGtCQUFhLEdBQUcsYUFBYyxDQUFDO2dCQUMvQixvQkFBZSxHQUFHLFNBQVMsQ0FBQztZQUs5QyxDQUFDO1lBSFcsa0JBQWtCLENBQUMsZUFBdUI7Z0JBQ2xELE9BQU8sbUJBQVUsQ0FBQyxJQUFJLENBQUMsY0FBYyxFQUFFLGVBQWUsQ0FBQyxDQUFDO1lBQzFELENBQUM7U0FDRjtRQUVELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUMzQixrQkFBa0IsRUFBRSxjQUFjO1NBQ25DLENBQUMsQ0FBQztLQUNKO0lBaUNTLGtCQUFrQixDQUFDLGVBQXVCO1FBQ2xELE9BQU8sbUJBQVUsQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLGVBQWUsQ0FBQyxDQUFDO0tBQ3REO0lBRU8sd0JBQXdCLENBQUMsS0FBdUI7UUFDdEQsMkJBQWMsQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUUsQ0FBQyxFQUFFLEdBQUcsRUFBRSxLQUFLLENBQUMsZUFBZSxDQUFDLENBQUM7UUFDakcsMkJBQWMsQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUUsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLGVBQWUsQ0FBQyxDQUFDO1FBQzVHLDJCQUFjLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLHlCQUF5QixFQUFFLENBQUMsRUFBRSxJQUFJLEVBQUUsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDO0tBQ3RHOztBQXRFSCxrQ0F1RUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBDZm5SZXNvdXJjZVNoYXJlIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLXJhbSc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgTmFtZXMgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IFN0YWdlU3RhY2tBc3NvY2lhdG9yIH0gZnJvbSAnLi9hc3BlY3RzL3N0YWNrLWFzc29jaWF0b3InO1xuaW1wb3J0IHsgSUF0dHJpYnV0ZUdyb3VwIH0gZnJvbSAnLi9hdHRyaWJ1dGUtZ3JvdXAnO1xuaW1wb3J0IHsgZ2V0UHJpbmNpcGFsc2ZvclNoYXJpbmcsIGhhc2hWYWx1ZXMsIFNoYXJlT3B0aW9ucywgU2hhcmVQZXJtaXNzaW9uIH0gZnJvbSAnLi9jb21tb24nO1xuaW1wb3J0IHsgaXNBY2NvdW50VW5yZXNvbHZlZCB9IGZyb20gJy4vcHJpdmF0ZS91dGlscyc7XG5pbXBvcnQgeyBJbnB1dFZhbGlkYXRvciB9IGZyb20gJy4vcHJpdmF0ZS92YWxpZGF0aW9uJztcbmltcG9ydCB7IENmbkFwcGxpY2F0aW9uLCBDZm5BdHRyaWJ1dGVHcm91cEFzc29jaWF0aW9uLCBDZm5SZXNvdXJjZUFzc29jaWF0aW9uIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLXNlcnZpY2VjYXRhbG9nYXBwcmVnaXN0cnknO1xuXG5jb25zdCBBUFBMSUNBVElPTl9SRUFEX09OTFlfUkFNX1BFUk1JU1NJT05fQVJOID0gJ2Fybjphd3M6cmFtOjphd3M6cGVybWlzc2lvbi9BV1NSQU1QZXJtaXNzaW9uU2VydmljZUNhdGFsb2dBcHBSZWdpc3RyeUFwcGxpY2F0aW9uUmVhZE9ubHknO1xuY29uc3QgQVBQTElDQVRJT05fQUxMT1dfQUNDRVNTX1JBTV9QRVJNSVNTSU9OX0FSTiA9ICdhcm46YXdzOnJhbTo6YXdzOnBlcm1pc3Npb24vQVdTUkFNUGVybWlzc2lvblNlcnZpY2VDYXRhbG9nQXBwUmVnaXN0cnlBcHBsaWNhdGlvbkFsbG93QXNzb2NpYXRpb24nO1xuXG4vKipcbiAqIEEgU2VydmljZSBDYXRhbG9nIEFwcFJlZ2lzdHJ5IEFwcGxpY2F0aW9uLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElBcHBsaWNhdGlvbiBleHRlbmRzIGNkay5JUmVzb3VyY2Uge1xuICAvKipcbiAgICogVGhlIEFSTiBvZiB0aGUgYXBwbGljYXRpb24uXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGFwcGxpY2F0aW9uQXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBJRCBvZiB0aGUgYXBwbGljYXRpb24uXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGFwcGxpY2F0aW9uSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGFwcGxpY2F0aW9uLlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSBhcHBsaWNhdGlvbk5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFzc29jaWF0ZSB0aGlzIGFwcGxpY2F0aW9uIHdpdGggYW4gYXR0cmlidXRlIGdyb3VwLlxuICAgKlxuICAgKiBAcGFyYW0gYXR0cmlidXRlR3JvdXAgQXBwUmVnaXN0cnkgYXR0cmlidXRlIGdyb3VwXG4gICAqL1xuICBhc3NvY2lhdGVBdHRyaWJ1dGVHcm91cChhdHRyaWJ1dGVHcm91cDogSUF0dHJpYnV0ZUdyb3VwKTogdm9pZDtcblxuICAvKipcbiAgICogQXNzb2NpYXRlIHRoaXMgYXBwbGljYXRpb24gd2l0aCBhIENsb3VkRm9ybWF0aW9uIHN0YWNrLlxuICAgKlxuICAgKiBAZGVwcmVjYXRlZCBVc2UgYGFzc29jaWF0ZUFwcGxpY2F0aW9uV2l0aFN0YWNrYCBpbnN0ZWFkLlxuICAgKiBAcGFyYW0gc3RhY2sgYSBDRk4gc3RhY2tcbiAgICovXG4gIGFzc29jaWF0ZVN0YWNrKHN0YWNrOiBjZGsuU3RhY2spOiB2b2lkO1xuXG4gIC8qKlxuICAgKiBBc3NvY2lhdGUgYSBDbG91ZGZvcm1hdGlvbiBzdGF0Y2sgd2l0aCB0aGUgYXBwbGljYXRpb24gaW4gdGhlIGdpdmVuIHN0YWNrLlxuICAgKlxuICAgKiBAcGFyYW0gc3RhY2sgYSBDRk4gc3RhY2tcbiAgICovXG4gIGFzc29jaWF0ZUFwcGxpY2F0aW9uV2l0aFN0YWNrKHN0YWNrOiBjZGsuU3RhY2spOiB2b2lkO1xuXG4gIC8qKlxuICAgKiBTaGFyZSB0aGlzIGFwcGxpY2F0aW9uIHdpdGggb3RoZXIgSUFNIGVudGl0aWVzLCBhY2NvdW50cywgb3IgT1VzLlxuICAgKlxuICAgKiBAcGFyYW0gc2hhcmVPcHRpb25zIFRoZSBvcHRpb25zIGZvciB0aGUgc2hhcmUuXG4gICAqL1xuICBzaGFyZUFwcGxpY2F0aW9uKHNoYXJlT3B0aW9uczogU2hhcmVPcHRpb25zKTogdm9pZDtcblxuICAvKipcbiAgICogQXNzb2NpYXRlIHRoaXMgYXBwbGljYXRpb24gd2l0aCBhbGwgc3RhY2tzIHVuZGVyIHRoZSBjb25zdHJ1Y3Qgbm9kZS5cbiAgICogTk9URTogVGhpcyBtZXRob2Qgd29uJ3QgYXV0b21hdGljYWxseSByZWdpc3RlciBzdGFja3MgdW5kZXIgcGlwZWxpbmUgc3RhZ2VzLFxuICAgKiBhbmQgcmVxdWlyZXMgYXNzb2NpYXRpb24gb2YgZWFjaCBwaXBlbGluZSBzdGFnZSBieSBjYWxsaW5nIHRoaXMgbWV0aG9kIHdpdGggc3RhZ2UgQ29uc3RydWN0LlxuICAgKlxuICAgKiBAcGFyYW0gY29uc3RydWN0IGNkayBDb25zdHJ1Y3RcbiAgICovXG4gIGFzc29jaWF0ZUFsbFN0YWNrc0luU2NvcGUoY29uc3RydWN0OiBDb25zdHJ1Y3QpOiB2b2lkO1xuXG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYSBTZXJ2aWNlIENhdGFsb2cgQXBwUmVnaXN0cnkgQXBwbGljYXRpb25cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBcHBsaWNhdGlvblByb3BzIHtcbiAgLyoqXG4gICAqIEVuZm9yY2VzIGEgcGFydGljdWxhciBwaHlzaWNhbCBhcHBsaWNhdGlvbiBuYW1lLlxuICAgKi9cbiAgcmVhZG9ubHkgYXBwbGljYXRpb25OYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIERlc2NyaXB0aW9uIGZvciBhcHBsaWNhdGlvbi5cbiAgICogQGRlZmF1bHQgLSBObyBkZXNjcmlwdGlvbiBwcm92aWRlZFxuICAgKi9cbiAgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG59XG5cbmFic3RyYWN0IGNsYXNzIEFwcGxpY2F0aW9uQmFzZSBleHRlbmRzIGNkay5SZXNvdXJjZSBpbXBsZW1lbnRzIElBcHBsaWNhdGlvbiB7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBhcHBsaWNhdGlvbkFybjogc3RyaW5nO1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgYXBwbGljYXRpb25JZDogc3RyaW5nO1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgYXBwbGljYXRpb25OYW1lPzogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IGFzc29jaWF0ZWRBdHRyaWJ1dGVHcm91cHM6IFNldDxzdHJpbmc+ID0gbmV3IFNldCgpO1xuICBwcml2YXRlIHJlYWRvbmx5IGFzc29jaWF0ZWRSZXNvdXJjZXM6IFNldDxzdHJpbmc+ID0gbmV3IFNldCgpO1xuXG4gIC8qKlxuICAgKiBBc3NvY2lhdGUgYW4gYXR0cmlidXRlIGdyb3VwIHdpdGggYXBwbGljYXRpb25cbiAgICogSWYgdGhlIGF0dHJpYnV0ZSBncm91cCBpcyBhbHJlYWR5IGFzc29jaWF0ZWQsIGl0IHdpbGwgaWdub3JlIGR1cGxpY2F0ZSByZXF1ZXN0LlxuICAgKi9cbiAgcHVibGljIGFzc29jaWF0ZUF0dHJpYnV0ZUdyb3VwKGF0dHJpYnV0ZUdyb3VwOiBJQXR0cmlidXRlR3JvdXApOiB2b2lkIHtcbiAgICBpZiAoIXRoaXMuYXNzb2NpYXRlZEF0dHJpYnV0ZUdyb3Vwcy5oYXMoYXR0cmlidXRlR3JvdXAubm9kZS5hZGRyKSkge1xuICAgICAgY29uc3QgaGFzaElkID0gdGhpcy5nZW5lcmF0ZVVuaXF1ZUhhc2goYXR0cmlidXRlR3JvdXAubm9kZS5hZGRyKTtcbiAgICAgIG5ldyBDZm5BdHRyaWJ1dGVHcm91cEFzc29jaWF0aW9uKHRoaXMsIGBBdHRyaWJ1dGVHcm91cEFzc29jaWF0aW9uJHtoYXNoSWR9YCwge1xuICAgICAgICBhcHBsaWNhdGlvbjogdGhpcy5hcHBsaWNhdGlvbklkLFxuICAgICAgICBhdHRyaWJ1dGVHcm91cDogYXR0cmlidXRlR3JvdXAuYXR0cmlidXRlR3JvdXBJZCxcbiAgICAgIH0pO1xuICAgICAgdGhpcy5hc3NvY2lhdGVkQXR0cmlidXRlR3JvdXBzLmFkZChhdHRyaWJ1dGVHcm91cC5ub2RlLmFkZHIpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBBc3NvY2lhdGUgYSBzdGFjayB3aXRoIHRoZSBhcHBsaWNhdGlvblxuICAgKiBJZiB0aGUgcmVzb3VyY2UgaXMgYWxyZWFkeSBhc3NvY2lhdGVkLCBpdCB3aWxsIGlnbm9yZSBkdXBsaWNhdGUgcmVxdWVzdC5cbiAgICogQSBzdGFjayBjYW4gb25seSBiZSBhc3NvY2lhdGVkIHdpdGggb25lIGFwcGxpY2F0aW9uLlxuICAgKlxuICAgKiBAZGVwcmVjYXRlZCBVc2UgYGFzc29jaWF0ZUFwcGxpY2F0aW9uV2l0aFN0YWNrYCBpbnN0ZWFkLlxuICAgKi9cbiAgcHVibGljIGFzc29jaWF0ZVN0YWNrKHN0YWNrOiBjZGsuU3RhY2spOiB2b2lkIHtcbiAgICBpZiAoIXRoaXMuYXNzb2NpYXRlZFJlc291cmNlcy5oYXMoc3RhY2subm9kZS5hZGRyKSkge1xuICAgICAgY29uc3QgaGFzaElkID0gdGhpcy5nZW5lcmF0ZVVuaXF1ZUhhc2goc3RhY2subm9kZS5hZGRyKTtcbiAgICAgIG5ldyBDZm5SZXNvdXJjZUFzc29jaWF0aW9uKHRoaXMsIGBSZXNvdXJjZUFzc29jaWF0aW9uJHtoYXNoSWR9YCwge1xuICAgICAgICBhcHBsaWNhdGlvbjogdGhpcy5hcHBsaWNhdGlvbklkLFxuICAgICAgICByZXNvdXJjZTogc3RhY2suc3RhY2tJZCxcbiAgICAgICAgcmVzb3VyY2VUeXBlOiAnQ0ZOX1NUQUNLJyxcbiAgICAgIH0pO1xuICAgICAgdGhpcy5hc3NvY2lhdGVkUmVzb3VyY2VzLmFkZChzdGFjay5ub2RlLmFkZHIpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBBc3NvY2lhdGUgc3RhY2sgd2l0aCB0aGUgYXBwbGljYXRpb24gaW4gdGhlIHN0YWNrIHBhc3NlZCBhcyBwYXJhbWV0ZXIuXG4gICAqXG4gICAqIEEgc3RhY2sgY2FuIG9ubHkgYmUgYXNzb2NpYXRlZCB3aXRoIG9uZSBhcHBsaWNhdGlvbi5cbiAgICovXG4gIHB1YmxpYyBhc3NvY2lhdGVBcHBsaWNhdGlvbldpdGhTdGFjayhzdGFjazogY2RrLlN0YWNrKTogdm9pZCB7XG4gICAgaWYgKCF0aGlzLmFzc29jaWF0ZWRSZXNvdXJjZXMuaGFzKHN0YWNrLm5vZGUuYWRkcikpIHtcbiAgICAgIG5ldyBDZm5SZXNvdXJjZUFzc29jaWF0aW9uKHN0YWNrLCAnQXBwUmVnaXN0cnlBc3NvY2lhdGlvbicsIHtcbiAgICAgICAgYXBwbGljYXRpb246IHN0YWNrID09PSBjZGsuU3RhY2sub2YodGhpcykgPyB0aGlzLmFwcGxpY2F0aW9uSWQgOiB0aGlzLmFwcGxpY2F0aW9uTmFtZSA/PyB0aGlzLmFwcGxpY2F0aW9uSWQsXG4gICAgICAgIHJlc291cmNlOiBzdGFjay5zdGFja0lkLFxuICAgICAgICByZXNvdXJjZVR5cGU6ICdDRk5fU1RBQ0snLFxuICAgICAgfSk7XG5cbiAgICAgIHRoaXMuYXNzb2NpYXRlZFJlc291cmNlcy5hZGQoc3RhY2subm9kZS5hZGRyKTtcbiAgICAgIGlmIChzdGFjayAhPT0gY2RrLlN0YWNrLm9mKHRoaXMpICYmIHRoaXMuaXNTYW1lQWNjb3VudChzdGFjaykgJiYgIXRoaXMuaXNTdGFnZVNjb3BlKHN0YWNrKSAmJiAhc3RhY2submVzdGVkKSB7XG4gICAgICAgIHN0YWNrLmFkZERlcGVuZGVuY3koY2RrLlN0YWNrLm9mKHRoaXMpKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogU2hhcmUgYW4gYXBwbGljYXRpb24gd2l0aCBhY2NvdW50cywgb3JnYW5pemF0aW9ucyBhbmQgT1VzLCBhbmQgSUFNIHJvbGVzIGFuZCB1c2Vycy5cbiAgICogVGhlIGFwcGxpY2F0aW9uIHdpbGwgYmVjb21lIGF2YWlsYWJsZSB0byBlbmQgdXNlcnMgd2l0aGluIHRob3NlIHByaW5jaXBhbHMuXG4gICAqXG4gICAqIEBwYXJhbSBzaGFyZU9wdGlvbnMgVGhlIG9wdGlvbnMgZm9yIHRoZSBzaGFyZS5cbiAgICovXG4gIHB1YmxpYyBzaGFyZUFwcGxpY2F0aW9uKHNoYXJlT3B0aW9uczogU2hhcmVPcHRpb25zKTogdm9pZCB7XG4gICAgY29uc3QgcHJpbmNpcGFscyA9IGdldFByaW5jaXBhbHNmb3JTaGFyaW5nKHNoYXJlT3B0aW9ucyk7XG4gICAgY29uc3Qgc2hhcmVOYW1lID0gYFJBTVNoYXJlJHtoYXNoVmFsdWVzKE5hbWVzLm5vZGVVbmlxdWVJZCh0aGlzLm5vZGUpLCB0aGlzLm5vZGUuY2hpbGRyZW4ubGVuZ3RoLnRvU3RyaW5nKCkpfWA7XG4gICAgbmV3IENmblJlc291cmNlU2hhcmUodGhpcywgc2hhcmVOYW1lLCB7XG4gICAgICBuYW1lOiBzaGFyZU5hbWUsXG4gICAgICBhbGxvd0V4dGVybmFsUHJpbmNpcGFsczogZmFsc2UsXG4gICAgICBwcmluY2lwYWxzOiBwcmluY2lwYWxzLFxuICAgICAgcmVzb3VyY2VBcm5zOiBbdGhpcy5hcHBsaWNhdGlvbkFybl0sXG4gICAgICBwZXJtaXNzaW9uQXJuczogW3RoaXMuZ2V0QXBwbGljYXRpb25TaGFyZVBlcm1pc3Npb25BUk4oc2hhcmVPcHRpb25zKV0sXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQXNzb2NpYXRlIGFsbCBzdGFja3MgcHJlc2VudCBpbiBjb25zdHJ1Y3QncyBhc3BlY3Qgd2l0aCBhcHBsaWNhdGlvbi5cbiAgICpcbiAgICogTk9URTogVGhpcyBtZXRob2Qgd29uJ3QgYXV0b21hdGljYWxseSByZWdpc3RlciBzdGFja3MgdW5kZXIgcGlwZWxpbmUgc3RhZ2VzLFxuICAgKiBhbmQgcmVxdWlyZXMgYXNzb2NpYXRpb24gb2YgZWFjaCBwaXBlbGluZSBzdGFnZSBieSBjYWxsaW5nIHRoaXMgbWV0aG9kIHdpdGggc3RhZ2UgQ29uc3RydWN0LlxuICAgKlxuICAgKi9cbiAgcHVibGljIGFzc29jaWF0ZUFsbFN0YWNrc0luU2NvcGUoc2NvcGU6IENvbnN0cnVjdCk6IHZvaWQge1xuICAgIGNkay5Bc3BlY3RzLm9mKHNjb3BlKS5hZGQobmV3IFN0YWdlU3RhY2tBc3NvY2lhdG9yKHRoaXMpKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSB1bmlxdWUgaWRcbiAgICovXG4gIHByb3RlY3RlZCBhYnN0cmFjdCBnZW5lcmF0ZVVuaXF1ZUhhc2gocmVzb3VyY2VBZGRyZXNzOiBzdHJpbmcpOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEdldCB0aGUgY29ycmVjdCBwZXJtaXNzaW9uIEFSTiBiYXNlZCBvbiB0aGUgU2hhcmVQZXJtaXNzaW9uXG4gICAqL1xuICBwcml2YXRlIGdldEFwcGxpY2F0aW9uU2hhcmVQZXJtaXNzaW9uQVJOKHNoYXJlT3B0aW9uczogU2hhcmVPcHRpb25zKTogc3RyaW5nIHtcbiAgICBzd2l0Y2ggKHNoYXJlT3B0aW9ucy5zaGFyZVBlcm1pc3Npb24pIHtcbiAgICAgIGNhc2UgU2hhcmVQZXJtaXNzaW9uLkFMTE9XX0FDQ0VTUzpcbiAgICAgICAgcmV0dXJuIEFQUExJQ0FUSU9OX0FMTE9XX0FDQ0VTU19SQU1fUEVSTUlTU0lPTl9BUk47XG4gICAgICBjYXNlIFNoYXJlUGVybWlzc2lvbi5SRUFEX09OTFk6XG4gICAgICAgIHJldHVybiBBUFBMSUNBVElPTl9SRUFEX09OTFlfUkFNX1BFUk1JU1NJT05fQVJOO1xuXG4gICAgICBkZWZhdWx0OlxuICAgICAgICByZXR1cm4gc2hhcmVPcHRpb25zLnNoYXJlUGVybWlzc2lvbiA/PyBBUFBMSUNBVElPTl9SRUFEX09OTFlfUkFNX1BFUk1JU1NJT05fQVJOO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAqICBDaGVja3Mgd2hldGhlciBhIHN0YWNrIGlzIGRlZmluZWQgaW4gYSBTdGFnZSBvciBub3QuXG4gICovXG4gIHByaXZhdGUgaXNTdGFnZVNjb3BlKHN0YWNrIDogY2RrLlN0YWNrKTogYm9vbGVhbiB7XG4gICAgcmV0dXJuICEoc3RhY2subm9kZS5zY29wZSBpbnN0YW5jZW9mIGNkay5BcHApICYmIChzdGFjay5ub2RlLnNjb3BlIGluc3RhbmNlb2YgY2RrLlN0YWdlKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBWZXJpZmllcyBpZiBhcHBsaWNhdGlvbiBhbmQgdGhlIHZpc2l0ZWQgbm9kZSBpcyBkZXBsb3llZCBpbiBkaWZmZXJlbnQgYWNjb3VudC5cbiAgICovXG4gIHByaXZhdGUgaXNTYW1lQWNjb3VudChzdGFjazogY2RrLlN0YWNrKTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIGlzQWNjb3VudFVucmVzb2x2ZWQodGhpcy5lbnYuYWNjb3VudCwgc3RhY2suYWNjb3VudCkgfHwgdGhpcy5lbnYuYWNjb3VudCA9PT0gc3RhY2suYWNjb3VudDtcbiAgfVxuXG59XG5cbi8qKlxuICogQSBTZXJ2aWNlIENhdGFsb2cgQXBwUmVnaXN0cnkgQXBwbGljYXRpb24uXG4gKi9cbmV4cG9ydCBjbGFzcyBBcHBsaWNhdGlvbiBleHRlbmRzIEFwcGxpY2F0aW9uQmFzZSB7XG4gIC8qKlxuICAgKiBJbXBvcnRzIGFuIEFwcGxpY2F0aW9uIGNvbnN0cnVjdCB0aGF0IHJlcHJlc2VudHMgYW4gZXh0ZXJuYWwgYXBwbGljYXRpb24uXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZSBUaGUgcGFyZW50IGNyZWF0aW5nIGNvbnN0cnVjdCAodXN1YWxseSBgdGhpc2ApLlxuICAgKiBAcGFyYW0gaWQgVGhlIGNvbnN0cnVjdCdzIG5hbWUuXG4gICAqIEBwYXJhbSBhcHBsaWNhdGlvbkFybiB0aGUgQW1hem9uIFJlc291cmNlIE5hbWUgb2YgdGhlIGV4aXN0aW5nIEFwcFJlZ2lzdHJ5IEFwcGxpY2F0aW9uXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21BcHBsaWNhdGlvbkFybihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhcHBsaWNhdGlvbkFybjogc3RyaW5nKTogSUFwcGxpY2F0aW9uIHtcbiAgICBjb25zdCBhcm4gPSBjZGsuU3RhY2sub2Yoc2NvcGUpLnNwbGl0QXJuKGFwcGxpY2F0aW9uQXJuLCBjZGsuQXJuRm9ybWF0LlNMQVNIX1JFU09VUkNFX1NMQVNIX1JFU09VUkNFX05BTUUpO1xuICAgIGNvbnN0IGFwcGxpY2F0aW9uSWQgPSBhcm4ucmVzb3VyY2VOYW1lO1xuXG4gICAgaWYgKCFhcHBsaWNhdGlvbklkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ01pc3NpbmcgcmVxdWlyZWQgQXBwbGljYXRpb24gSUQgZnJvbSBBcHBsaWNhdGlvbiBBUk46ICcgKyBhcHBsaWNhdGlvbkFybik7XG4gICAgfVxuXG4gICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgQXBwbGljYXRpb25CYXNlIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBhcHBsaWNhdGlvbkFybiA9IGFwcGxpY2F0aW9uQXJuO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGFwcGxpY2F0aW9uSWQgPSBhcHBsaWNhdGlvbklkITtcbiAgICAgIHB1YmxpYyByZWFkb25seSBhcHBsaWNhdGlvbk5hbWUgPSB1bmRlZmluZWQ7XG5cbiAgICAgIHByb3RlY3RlZCBnZW5lcmF0ZVVuaXF1ZUhhc2gocmVzb3VyY2VBZGRyZXNzOiBzdHJpbmcpOiBzdHJpbmcge1xuICAgICAgICByZXR1cm4gaGFzaFZhbHVlcyh0aGlzLmFwcGxpY2F0aW9uQXJuLCByZXNvdXJjZUFkZHJlc3MpO1xuICAgICAgfVxuICAgIH1cblxuICAgIHJldHVybiBuZXcgSW1wb3J0KHNjb3BlLCBpZCwge1xuICAgICAgZW52aXJvbm1lbnRGcm9tQXJuOiBhcHBsaWNhdGlvbkFybixcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBcHBsaWNhdGlvbiBtYW5hZ2VyIFVSTCBmb3IgdGhlIEFwcGxpY2F0aW9uLlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYXBwbGljYXRpb25NYW5hZ2VyVXJsPzogY2RrLkNmbk91dHB1dDtcbiAgcHVibGljIHJlYWRvbmx5IGFwcGxpY2F0aW9uQXJuOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBhcHBsaWNhdGlvbklkOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBhcHBsaWNhdGlvbk5hbWU/OiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgbm9kZUFkZHJlc3M6IHN0cmluZztcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQXBwbGljYXRpb25Qcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLnZhbGlkYXRlQXBwbGljYXRpb25Qcm9wcyhwcm9wcyk7XG5cbiAgICBjb25zdCBhcHBsaWNhdGlvbiA9IG5ldyBDZm5BcHBsaWNhdGlvbih0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBuYW1lOiBwcm9wcy5hcHBsaWNhdGlvbk5hbWUsXG4gICAgICBkZXNjcmlwdGlvbjogcHJvcHMuZGVzY3JpcHRpb24sXG4gICAgfSk7XG5cbiAgICB0aGlzLmFwcGxpY2F0aW9uQXJuID0gYXBwbGljYXRpb24uYXR0ckFybjtcbiAgICB0aGlzLmFwcGxpY2F0aW9uSWQgPSBhcHBsaWNhdGlvbi5hdHRySWQ7XG4gICAgdGhpcy5hcHBsaWNhdGlvbk5hbWUgPSBwcm9wcy5hcHBsaWNhdGlvbk5hbWU7XG4gICAgdGhpcy5ub2RlQWRkcmVzcyA9IGNkay5OYW1lcy5ub2RlVW5pcXVlSWQoYXBwbGljYXRpb24ubm9kZSk7XG5cbiAgICB0aGlzLmFwcGxpY2F0aW9uTWFuYWdlclVybCA9IG5ldyBjZGsuQ2ZuT3V0cHV0KHRoaXMsICdBcHBsaWNhdGlvbk1hbmFnZXJVcmwnLCB7XG4gICAgICB2YWx1ZTogYGh0dHBzOi8vJHt0aGlzLmVudi5yZWdpb259LmNvbnNvbGUuYXdzLmFtYXpvbi5jb20vc3lzdGVtcy1tYW5hZ2VyL2FwcG1hbmFnZXIvYXBwbGljYXRpb24vQVdTX0FwcFJlZ2lzdHJ5X0FwcGxpY2F0aW9uLSR7dGhpcy5hcHBsaWNhdGlvbk5hbWV9YCxcbiAgICAgIGRlc2NyaXB0aW9uOiAnQXBwbGljYXRpb24gbWFuYWdlciB1cmwgZm9yIHRoZSBhcHBsaWNhdGlvbiBjcmVhdGVkLicsXG4gICAgfSk7XG4gIH1cblxuICBwcm90ZWN0ZWQgZ2VuZXJhdGVVbmlxdWVIYXNoKHJlc291cmNlQWRkcmVzczogc3RyaW5nKTogc3RyaW5nIHtcbiAgICByZXR1cm4gaGFzaFZhbHVlcyh0aGlzLm5vZGVBZGRyZXNzLCByZXNvdXJjZUFkZHJlc3MpO1xuICB9XG5cbiAgcHJpdmF0ZSB2YWxpZGF0ZUFwcGxpY2F0aW9uUHJvcHMocHJvcHM6IEFwcGxpY2F0aW9uUHJvcHMpIHtcbiAgICBJbnB1dFZhbGlkYXRvci52YWxpZGF0ZUxlbmd0aCh0aGlzLm5vZGUucGF0aCwgJ2FwcGxpY2F0aW9uIG5hbWUnLCAxLCAyNTYsIHByb3BzLmFwcGxpY2F0aW9uTmFtZSk7XG4gICAgSW5wdXRWYWxpZGF0b3IudmFsaWRhdGVSZWdleCh0aGlzLm5vZGUucGF0aCwgJ2FwcGxpY2F0aW9uIG5hbWUnLCAvXlthLXpBLVowLTktX10rJC8sIHByb3BzLmFwcGxpY2F0aW9uTmFtZSk7XG4gICAgSW5wdXRWYWxpZGF0b3IudmFsaWRhdGVMZW5ndGgodGhpcy5ub2RlLnBhdGgsICdhcHBsaWNhdGlvbiBkZXNjcmlwdGlvbicsIDAsIDEwMjQsIHByb3BzLmRlc2NyaXB0aW9uKTtcbiAgfVxufVxuIl19