'''
# AWS ServiceCatalogAppRegistry Construct Library

<!--BEGIN STABILITY BANNER-->---


![cdk-constructs: Experimental](https://img.shields.io/badge/cdk--constructs-experimental-important.svg?style=for-the-badge)

> The APIs of higher level constructs in this module are experimental and under active development.
> They are subject to non-backward compatible changes or removal in any future version. These are
> not subject to the [Semantic Versioning](https://semver.org/) model and breaking changes will be
> announced in the release notes. This means that while you may use them, you may need to update
> your source code when upgrading to a newer version of this package.

---
<!--END STABILITY BANNER-->

[AWS Service Catalog App Registry](https://docs.aws.amazon.com/servicecatalog/latest/adminguide/appregistry.html)
enables organizations to create and manage repositories of applications and associated resources.

## Table Of Contents

* [Application](#application)
* [Application-Associator](#application-associator)
* [Attribute-Group](#attribute-group)
* [Associations](#associations)

  * [Associating application with an attribute group](#attribute-group-association)
  * [Associating application with a stack](#resource-association)
* [Sharing](#sharing)

  * [Sharing an application](#sharing-an-application)
  * [Sharing an attribute group](#sharing-an-attribute-group)

The `@aws-cdk/aws-servicecatalogappregistry` package contains resources that enable users to automate governance and management of their AWS resources at scale.

```python
import aws_cdk.aws_servicecatalogappregistry_alpha as appreg
```

## Application

An AppRegistry application enables you to define your applications and associated resources.
The application name must be unique at the account level and it's immutable.

```python
application = appreg.Application(self, "MyFirstApplication",
    application_name="MyFirstApplicationName",
    description="description for my application"
)
```

An application that has been created outside of the stack can be imported into your CDK app.
Applications can be imported by their ARN via the `Application.fromApplicationArn()` API:

```python
imported_application = appreg.Application.from_application_arn(self, "MyImportedApplication", "arn:aws:servicecatalog:us-east-1:012345678910:/applications/0aqmvxvgmry0ecc4mjhwypun6i")
```

## Application-Associator

If you want to create an Application named `MyAssociatedApplication` in account `123456789012` and region `us-east-1`
and want to associate all stacks in the `App` scope to `MyAssociatedApplication`, then use as shown in the example below:

```python
app = App()
associated_app = appreg.ApplicationAssociator(app, "AssociatedApplication",
    applications=[appreg.TargetApplication.create_application_stack(
        application_name="MyAssociatedApplication",
        # 'Application containing stacks deployed via CDK.' is the default
        application_description="Associated Application description",
        stack_name="MyAssociatedApplicationStack",
        # AWS Account and Region that are implied by the current CLI configuration is the default
        env=Environment(account="123456789012", region="us-east-1")
    )]
)
```

This will create an application `MyAssociatedApplication` with the `TagKey` as `managedBy` and `TagValue` as `CDK_Application_Associator`.

If you want to re-use an existing Application with ARN: `arn:aws:servicecatalog:us-east-1:123456789012:/applications/applicationId`
and want to associate all stacks in the `App` scope to your imported application, then use as shown in the example below:

```python
app = App()
associated_app = appreg.ApplicationAssociator(app, "AssociatedApplication",
    applications=[appreg.TargetApplication.existing_application_from_arn(
        application_arn_value="arn:aws:servicecatalog:us-east-1:123456789012:/applications/applicationId",
        stack_name="MyAssociatedApplicationStack"
    )]
)
```

If you are using CDK Pipelines to deploy your application, the application stacks will be inside Stages, and
ApplicationAssociator will not be able to find them. Call `associateStage` on each Stage object before adding it to the
Pipeline, as shown in the example below:

```python
import aws_cdk as cdk
import aws_cdk.pipelines as codepipeline
import aws_cdk.aws_codecommit as codecommit
# repo: codecommit.Repository
# pipeline: codepipeline.CodePipeline
# beta: cdk.Stage

class ApplicationPipelineStack(cdk.Stack):
    def __init__(self, scope, id, *, application, description=None, env=None, stackName=None, tags=None, synthesizer=None, terminationProtection=None, analyticsReporting=None, crossRegionReferences=None, permissionsBoundary=None):
        super().__init__(scope, id, application=application, description=description, env=env, stackName=stackName, tags=tags, synthesizer=synthesizer, terminationProtection=terminationProtection, analyticsReporting=analyticsReporting, crossRegionReferences=crossRegionReferences, permissionsBoundary=permissionsBoundary)

        # associate the stage to application associator.
        application.associate_stage(beta)
        pipeline.add_stage(beta)

app = App()
associated_app = appreg.ApplicationAssociator(app, "AssociatedApplication",
    applications=[appreg.TargetApplication.create_application_stack(
        application_name="MyPipelineAssociatedApplication",
        stack_name="MyPipelineAssociatedApplicationStack",
        env=cdk.Environment(account="123456789012", region="us-east-1")
    )]
)

cdk_pipeline = ApplicationPipelineStack(app, "CDKApplicationPipelineStack",
    application=associated_app,
    env=cdk.Environment(account="123456789012", region="us-east-1")
)
```

## Attribute Group

An AppRegistry attribute group acts as a container for user-defined attributes for an application.
Metadata is attached in a machine-readable format to integrate with automated workflows and tools.

```python
attribute_group = appreg.AttributeGroup(self, "MyFirstAttributeGroup",
    attribute_group_name="MyFirstAttributeGroupName",
    description="description for my attribute group",  # the description is optional,
    attributes={
        "project": "foo",
        "team": ["member1", "member2", "member3"],
        "public": False,
        "stages": {
            "alpha": "complete",
            "beta": "incomplete",
            "release": "not started"
        }
    }
)
```

An attribute group that has been created outside of the stack can be imported into your CDK app.
Attribute groups can be imported by their ARN via the `AttributeGroup.fromAttributeGroupArn()` API:

```python
imported_attribute_group = appreg.AttributeGroup.from_attribute_group_arn(self, "MyImportedAttrGroup", "arn:aws:servicecatalog:us-east-1:012345678910:/attribute-groups/0aqmvxvgmry0ecc4mjhwypun6i")
```

## Associations

You can associate your appregistry application with attribute groups and resources.
Resources are CloudFormation stacks that you can associate with an application to group relevant
stacks together to enable metadata rich insights into your applications and resources.
A Cloudformation stack can only be associated with one appregistry application.
If a stack is associated with multiple applications in your app or is already associated with one,
CDK will fail at deploy time.

### Associating application with an attribute group

You can associate an attribute group with an application with the `associateAttributeGroup()` API:

```python
# application: appreg.Application
# attribute_group: appreg.AttributeGroup

application.associate_attribute_group(attribute_group)
```

### Associating application with a Stack

You can associate a stack with an application with the `associateStack()` API:

```python
# application: appreg.Application
app = App()
my_stack = Stack(app, "MyStack")
application.associate_stack(my_stack)
```

## Sharing

You can share your AppRegistry applications and attribute groups with AWS Organizations, Organizational Units (OUs), AWS accounts within an organization, as well as IAM roles and users. AppRegistry requires that AWS Organizations is enabled in an account before deploying a share of an application or attribute group.

### Sharing an application

```python
import aws_cdk.aws_iam as iam
# application: appreg.Application
# my_role: iam.IRole
# my_user: iam.IUser

application.share_application(
    accounts=["123456789012"],
    organization_arns=["arn:aws:organizations::123456789012:organization/o-my-org-id"],
    roles=[my_role],
    users=[my_user]
)
```

E.g., sharing an application with multiple accounts and allowing the accounts to associate resources to the application.

```python
import aws_cdk.aws_iam as iam
# application: appreg.Application

application.share_application(
    accounts=["123456789012", "234567890123"],
    share_permission=appreg.SharePermission.ALLOW_ACCESS
)
```

### Sharing an attribute group

```python
import aws_cdk.aws_iam as iam
# attribute_group: appreg.AttributeGroup
# my_role: iam.IRole
# my_user: iam.IUser

attribute_group.share_attribute_group(
    accounts=["123456789012"],
    organization_arns=["arn:aws:organizations::123456789012:organization/o-my-org-id"],
    roles=[my_role],
    users=[my_user]
)
```

E.g., sharing an application with multiple accounts and allowing the accounts to associate applications to the attribute group.

```python
import aws_cdk.aws_iam as iam
# attribute_group: appreg.AttributeGroup

attribute_group.share_attribute_group(
    accounts=["123456789012", "234567890123"],
    share_permission=appreg.SharePermission.ALLOW_ACCESS
)
```
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from typeguard import check_type

from ._jsii import *

import aws_cdk as _aws_cdk_ceddda9d
import aws_cdk.aws_iam as _aws_cdk_aws_iam_ceddda9d
import constructs as _constructs_77d1e7e8


class ApplicationAssociator(
    _constructs_77d1e7e8.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-servicecatalogappregistry-alpha.ApplicationAssociator",
):
    '''(experimental) An AppRegistry construct to automatically create an application with the given name and description.

    The application name must be unique at the account level per region and it's immutable.
    This construct will automatically associate all stacks in the given scope, however
    in case of a ``Pipeline`` stack, stage underneath the pipeline will not automatically be associated and
    needs to be associated separately.

    If cross account stack is detected, then this construct will automatically share the application to consumer accounts.
    Cross account feature will only work for non environment agnostic stacks.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        app = App()
        associated_app = appreg.ApplicationAssociator(app, "AssociatedApplication",
            applications=[appreg.TargetApplication.create_application_stack(
                application_name="MyAssociatedApplication",
                # 'Application containing stacks deployed via CDK.' is the default
                application_description="Associated Application description",
                stack_name="MyAssociatedApplicationStack",
                # AWS Account and Region that are implied by the current CLI configuration is the default
                env=Environment(account="123456789012", region="us-east-1")
            )]
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_ceddda9d.App,
        id: builtins.str,
        *,
        applications: typing.Sequence["TargetApplication"],
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param applications: (experimental) Application associator properties. Default: - Empty array.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fdf70fe013f8883665f67795a3e892e124b1c97eee77c8691a45e94a03fd3b3f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = ApplicationAssociatorProps(applications=applications)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="appRegistryApplication")
    def app_registry_application(self) -> "IApplication":
        '''(experimental) Get the AppRegistry application.

        :stability: experimental
        '''
        return typing.cast("IApplication", jsii.invoke(self, "appRegistryApplication", []))

    @jsii.member(jsii_name="associateStage")
    def associate_stage(
        self,
        stage: _aws_cdk_ceddda9d.Stage,
    ) -> _aws_cdk_ceddda9d.Stage:
        '''(experimental) Associate this application with the given stage.

        :param stage: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2c562e37ce20cae0e25f66edcda715d4ced81af5946d6243931500bb560b566c)
            check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
        return typing.cast(_aws_cdk_ceddda9d.Stage, jsii.invoke(self, "associateStage", [stage]))

    @jsii.member(jsii_name="isStageAssociated")
    def is_stage_associated(self, stage: _aws_cdk_ceddda9d.Stage) -> builtins.bool:
        '''(experimental) Validates if a stage is already associated to the application.

        :param stage: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e907bed5984ca51a19f28a830c059d5147fb84b71013dd72756e86f116955fef)
            check_type(argname="argument stage", value=stage, expected_type=type_hints["stage"])
        return typing.cast(builtins.bool, jsii.invoke(self, "isStageAssociated", [stage]))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-servicecatalogappregistry-alpha.ApplicationAssociatorProps",
    jsii_struct_bases=[],
    name_mapping={"applications": "applications"},
)
class ApplicationAssociatorProps:
    def __init__(self, *, applications: typing.Sequence["TargetApplication"]) -> None:
        '''(experimental) Properties for Service Catalog AppRegistry Application Associator.

        :param applications: (experimental) Application associator properties. Default: - Empty array.

        :stability: experimental
        :exampleMetadata: infused

        Example::

            app = App()
            associated_app = appreg.ApplicationAssociator(app, "AssociatedApplication",
                applications=[appreg.TargetApplication.create_application_stack(
                    application_name="MyAssociatedApplication",
                    # 'Application containing stacks deployed via CDK.' is the default
                    application_description="Associated Application description",
                    stack_name="MyAssociatedApplicationStack",
                    # AWS Account and Region that are implied by the current CLI configuration is the default
                    env=Environment(account="123456789012", region="us-east-1")
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__30616d49bde91e4252f2107159acc6af6c7e69963df724abfd3d0e3887b7d07c)
            check_type(argname="argument applications", value=applications, expected_type=type_hints["applications"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "applications": applications,
        }

    @builtins.property
    def applications(self) -> typing.List["TargetApplication"]:
        '''(experimental) Application associator properties.

        :default: - Empty array.

        :stability: experimental
        '''
        result = self._values.get("applications")
        assert result is not None, "Required property 'applications' is missing"
        return typing.cast(typing.List["TargetApplication"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApplicationAssociatorProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-servicecatalogappregistry-alpha.ApplicationProps",
    jsii_struct_bases=[],
    name_mapping={"application_name": "applicationName", "description": "description"},
)
class ApplicationProps:
    def __init__(
        self,
        *,
        application_name: builtins.str,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Properties for a Service Catalog AppRegistry Application.

        :param application_name: (experimental) Enforces a particular physical application name.
        :param description: (experimental) Description for application. Default: - No description provided

        :stability: experimental
        :exampleMetadata: infused

        Example::

            application = appreg.Application(self, "MyFirstApplication",
                application_name="MyFirstApplicationName",
                description="description for my application"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6be7bfb8b322f2a1baa56fe1681c794a77811f78b07db4c1d82e108ccfd5d114)
            check_type(argname="argument application_name", value=application_name, expected_type=type_hints["application_name"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "application_name": application_name,
        }
        if description is not None:
            self._values["description"] = description

    @builtins.property
    def application_name(self) -> builtins.str:
        '''(experimental) Enforces a particular physical application name.

        :stability: experimental
        '''
        result = self._values.get("application_name")
        assert result is not None, "Required property 'application_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(experimental) Description for application.

        :default: - No description provided

        :stability: experimental
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApplicationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-servicecatalogappregistry-alpha.AttributeGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "attribute_group_name": "attributeGroupName",
        "attributes": "attributes",
        "description": "description",
    },
)
class AttributeGroupProps:
    def __init__(
        self,
        *,
        attribute_group_name: builtins.str,
        attributes: typing.Mapping[builtins.str, typing.Any],
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Properties for a Service Catalog AppRegistry Attribute Group.

        :param attribute_group_name: (experimental) Enforces a particular physical attribute group name.
        :param attributes: (experimental) A JSON of nested key-value pairs that represent the attributes in the group. Attributes maybe an empty JSON '{}', but must be explicitly stated.
        :param description: (experimental) Description for attribute group. Default: - No description provided

        :stability: experimental
        :exampleMetadata: infused

        Example::

            attribute_group = appreg.AttributeGroup(self, "MyFirstAttributeGroup",
                attribute_group_name="MyFirstAttributeGroupName",
                description="description for my attribute group",  # the description is optional,
                attributes={
                    "project": "foo",
                    "team": ["member1", "member2", "member3"],
                    "public": False,
                    "stages": {
                        "alpha": "complete",
                        "beta": "incomplete",
                        "release": "not started"
                    }
                }
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__3622c2b8fdb683c749e737f012cde1fbfc02d34fadd8639e26812b7ef1494b30)
            check_type(argname="argument attribute_group_name", value=attribute_group_name, expected_type=type_hints["attribute_group_name"])
            check_type(argname="argument attributes", value=attributes, expected_type=type_hints["attributes"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "attribute_group_name": attribute_group_name,
            "attributes": attributes,
        }
        if description is not None:
            self._values["description"] = description

    @builtins.property
    def attribute_group_name(self) -> builtins.str:
        '''(experimental) Enforces a particular physical attribute group name.

        :stability: experimental
        '''
        result = self._values.get("attribute_group_name")
        assert result is not None, "Required property 'attribute_group_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def attributes(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''(experimental) A JSON of nested key-value pairs that represent the attributes in the group.

        Attributes maybe an empty JSON '{}', but must be explicitly stated.

        :stability: experimental
        '''
        result = self._values.get("attributes")
        assert result is not None, "Required property 'attributes' is missing"
        return typing.cast(typing.Mapping[builtins.str, typing.Any], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(experimental) Description for attribute group.

        :default: - No description provided

        :stability: experimental
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AttributeGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-servicecatalogappregistry-alpha.BindTargetApplicationResult",
    jsii_struct_bases=[],
    name_mapping={"application": "application"},
)
class BindTargetApplicationResult:
    def __init__(self, *, application: "IApplication") -> None:
        '''(experimental) Properties for Service Catalog AppRegistry Application Associator to work with.

        :param application: (experimental) Created or imported application.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_servicecatalogappregistry_alpha as servicecatalogappregistry_alpha
            
            # application: servicecatalogappregistry_alpha.Application
            
            bind_target_application_result = servicecatalogappregistry_alpha.BindTargetApplicationResult(
                application=application
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__49f1fbb4ffc705c29064e5565593f9a0e87b46c7dbf27ddfa01428e7ccb10675)
            check_type(argname="argument application", value=application, expected_type=type_hints["application"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "application": application,
        }

    @builtins.property
    def application(self) -> "IApplication":
        '''(experimental) Created or imported application.

        :stability: experimental
        '''
        result = self._values.get("application")
        assert result is not None, "Required property 'application' is missing"
        return typing.cast("IApplication", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BindTargetApplicationResult(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.interface(jsii_type="@aws-cdk/aws-servicecatalogappregistry-alpha.IApplication")
class IApplication(_aws_cdk_ceddda9d.IResource, typing_extensions.Protocol):
    '''(experimental) A Service Catalog AppRegistry Application.

    :stability: experimental
    '''

    @builtins.property
    @jsii.member(jsii_name="applicationArn")
    def application_arn(self) -> builtins.str:
        '''(experimental) The ARN of the application.

        :stability: experimental
        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="applicationId")
    def application_id(self) -> builtins.str:
        '''(experimental) The ID of the application.

        :stability: experimental
        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="applicationName")
    def application_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the application.

        :stability: experimental
        :attribute: true
        '''
        ...

    @jsii.member(jsii_name="associateAllStacksInScope")
    def associate_all_stacks_in_scope(
        self,
        construct: _constructs_77d1e7e8.Construct,
    ) -> None:
        '''(experimental) Associate this application with all stacks under the construct node.

        NOTE: This method won't automatically register stacks under pipeline stages,
        and requires association of each pipeline stage by calling this method with stage Construct.

        :param construct: cdk Construct.

        :stability: experimental
        '''
        ...

    @jsii.member(jsii_name="associateApplicationWithStack")
    def associate_application_with_stack(self, stack: _aws_cdk_ceddda9d.Stack) -> None:
        '''(experimental) Associate a Cloudformation statck with the application in the given stack.

        :param stack: a CFN stack.

        :stability: experimental
        '''
        ...

    @jsii.member(jsii_name="associateAttributeGroup")
    def associate_attribute_group(self, attribute_group: "IAttributeGroup") -> None:
        '''(experimental) Associate this application with an attribute group.

        :param attribute_group: AppRegistry attribute group.

        :stability: experimental
        '''
        ...

    @jsii.member(jsii_name="associateStack")
    def associate_stack(self, stack: _aws_cdk_ceddda9d.Stack) -> None:
        '''(deprecated) Associate this application with a CloudFormation stack.

        :param stack: a CFN stack.

        :deprecated: Use ``associateApplicationWithStack`` instead.

        :stability: deprecated
        '''
        ...

    @jsii.member(jsii_name="shareApplication")
    def share_application(
        self,
        *,
        accounts: typing.Optional[typing.Sequence[builtins.str]] = None,
        organization_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
        roles: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.IRole]] = None,
        share_permission: typing.Optional[typing.Union[builtins.str, "SharePermission"]] = None,
        users: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.IUser]] = None,
    ) -> None:
        '''(experimental) Share this application with other IAM entities, accounts, or OUs.

        :param accounts: (experimental) A list of AWS accounts that the application will be shared with. Default: - No accounts specified for share
        :param organization_arns: (experimental) A list of AWS Organization or Organizational Units (OUs) ARNs that the application will be shared with. Default: - No AWS Organizations or OUs specified for share
        :param roles: (experimental) A list of AWS IAM roles that the application will be shared with. Default: - No IAM roles specified for share
        :param share_permission: (experimental) An option to manage access to the application or attribute group. Default: - Principals will be assigned read only permissions on the application or attribute group.
        :param users: (experimental) A list of AWS IAM users that the application will be shared with. Default: - No IAM Users specified for share

        :stability: experimental
        '''
        ...


class _IApplicationProxy(
    jsii.proxy_for(_aws_cdk_ceddda9d.IResource), # type: ignore[misc]
):
    '''(experimental) A Service Catalog AppRegistry Application.

    :stability: experimental
    '''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-servicecatalogappregistry-alpha.IApplication"

    @builtins.property
    @jsii.member(jsii_name="applicationArn")
    def application_arn(self) -> builtins.str:
        '''(experimental) The ARN of the application.

        :stability: experimental
        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "applicationArn"))

    @builtins.property
    @jsii.member(jsii_name="applicationId")
    def application_id(self) -> builtins.str:
        '''(experimental) The ID of the application.

        :stability: experimental
        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "applicationId"))

    @builtins.property
    @jsii.member(jsii_name="applicationName")
    def application_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the application.

        :stability: experimental
        :attribute: true
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "applicationName"))

    @jsii.member(jsii_name="associateAllStacksInScope")
    def associate_all_stacks_in_scope(
        self,
        construct: _constructs_77d1e7e8.Construct,
    ) -> None:
        '''(experimental) Associate this application with all stacks under the construct node.

        NOTE: This method won't automatically register stacks under pipeline stages,
        and requires association of each pipeline stage by calling this method with stage Construct.

        :param construct: cdk Construct.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__eb25bb53fbac36a075a6c682dbd5e647d5dd5b0436b1b1e29a37205679f60c97)
            check_type(argname="argument construct", value=construct, expected_type=type_hints["construct"])
        return typing.cast(None, jsii.invoke(self, "associateAllStacksInScope", [construct]))

    @jsii.member(jsii_name="associateApplicationWithStack")
    def associate_application_with_stack(self, stack: _aws_cdk_ceddda9d.Stack) -> None:
        '''(experimental) Associate a Cloudformation statck with the application in the given stack.

        :param stack: a CFN stack.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e92370f5ca96b4ac3a6918c6020df4c61044935affc14afe329d14f7b6cfc872)
            check_type(argname="argument stack", value=stack, expected_type=type_hints["stack"])
        return typing.cast(None, jsii.invoke(self, "associateApplicationWithStack", [stack]))

    @jsii.member(jsii_name="associateAttributeGroup")
    def associate_attribute_group(self, attribute_group: "IAttributeGroup") -> None:
        '''(experimental) Associate this application with an attribute group.

        :param attribute_group: AppRegistry attribute group.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__013b061a2eaf66f5f2c454cad6c31f5138ef6eacf9ce64ea456c3acc549750b7)
            check_type(argname="argument attribute_group", value=attribute_group, expected_type=type_hints["attribute_group"])
        return typing.cast(None, jsii.invoke(self, "associateAttributeGroup", [attribute_group]))

    @jsii.member(jsii_name="associateStack")
    def associate_stack(self, stack: _aws_cdk_ceddda9d.Stack) -> None:
        '''(deprecated) Associate this application with a CloudFormation stack.

        :param stack: a CFN stack.

        :deprecated: Use ``associateApplicationWithStack`` instead.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6355fcb9bf431963c17f1d72e211e52e508a3bb085de7597c2deed4b88eda46f)
            check_type(argname="argument stack", value=stack, expected_type=type_hints["stack"])
        return typing.cast(None, jsii.invoke(self, "associateStack", [stack]))

    @jsii.member(jsii_name="shareApplication")
    def share_application(
        self,
        *,
        accounts: typing.Optional[typing.Sequence[builtins.str]] = None,
        organization_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
        roles: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.IRole]] = None,
        share_permission: typing.Optional[typing.Union[builtins.str, "SharePermission"]] = None,
        users: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.IUser]] = None,
    ) -> None:
        '''(experimental) Share this application with other IAM entities, accounts, or OUs.

        :param accounts: (experimental) A list of AWS accounts that the application will be shared with. Default: - No accounts specified for share
        :param organization_arns: (experimental) A list of AWS Organization or Organizational Units (OUs) ARNs that the application will be shared with. Default: - No AWS Organizations or OUs specified for share
        :param roles: (experimental) A list of AWS IAM roles that the application will be shared with. Default: - No IAM roles specified for share
        :param share_permission: (experimental) An option to manage access to the application or attribute group. Default: - Principals will be assigned read only permissions on the application or attribute group.
        :param users: (experimental) A list of AWS IAM users that the application will be shared with. Default: - No IAM Users specified for share

        :stability: experimental
        '''
        share_options = ShareOptions(
            accounts=accounts,
            organization_arns=organization_arns,
            roles=roles,
            share_permission=share_permission,
            users=users,
        )

        return typing.cast(None, jsii.invoke(self, "shareApplication", [share_options]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IApplication).__jsii_proxy_class__ = lambda : _IApplicationProxy


@jsii.interface(
    jsii_type="@aws-cdk/aws-servicecatalogappregistry-alpha.IAttributeGroup"
)
class IAttributeGroup(_aws_cdk_ceddda9d.IResource, typing_extensions.Protocol):
    '''(experimental) A Service Catalog AppRegistry Attribute Group.

    :stability: experimental
    '''

    @builtins.property
    @jsii.member(jsii_name="attributeGroupArn")
    def attribute_group_arn(self) -> builtins.str:
        '''(experimental) The ARN of the attribute group.

        :stability: experimental
        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="attributeGroupId")
    def attribute_group_id(self) -> builtins.str:
        '''(experimental) The ID of the attribute group.

        :stability: experimental
        :attribute: true
        '''
        ...

    @jsii.member(jsii_name="shareAttributeGroup")
    def share_attribute_group(
        self,
        *,
        accounts: typing.Optional[typing.Sequence[builtins.str]] = None,
        organization_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
        roles: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.IRole]] = None,
        share_permission: typing.Optional[typing.Union[builtins.str, "SharePermission"]] = None,
        users: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.IUser]] = None,
    ) -> None:
        '''(experimental) Share the attribute group resource with other IAM entities, accounts, or OUs.

        :param accounts: (experimental) A list of AWS accounts that the application will be shared with. Default: - No accounts specified for share
        :param organization_arns: (experimental) A list of AWS Organization or Organizational Units (OUs) ARNs that the application will be shared with. Default: - No AWS Organizations or OUs specified for share
        :param roles: (experimental) A list of AWS IAM roles that the application will be shared with. Default: - No IAM roles specified for share
        :param share_permission: (experimental) An option to manage access to the application or attribute group. Default: - Principals will be assigned read only permissions on the application or attribute group.
        :param users: (experimental) A list of AWS IAM users that the application will be shared with. Default: - No IAM Users specified for share

        :stability: experimental
        '''
        ...


class _IAttributeGroupProxy(
    jsii.proxy_for(_aws_cdk_ceddda9d.IResource), # type: ignore[misc]
):
    '''(experimental) A Service Catalog AppRegistry Attribute Group.

    :stability: experimental
    '''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-servicecatalogappregistry-alpha.IAttributeGroup"

    @builtins.property
    @jsii.member(jsii_name="attributeGroupArn")
    def attribute_group_arn(self) -> builtins.str:
        '''(experimental) The ARN of the attribute group.

        :stability: experimental
        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "attributeGroupArn"))

    @builtins.property
    @jsii.member(jsii_name="attributeGroupId")
    def attribute_group_id(self) -> builtins.str:
        '''(experimental) The ID of the attribute group.

        :stability: experimental
        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "attributeGroupId"))

    @jsii.member(jsii_name="shareAttributeGroup")
    def share_attribute_group(
        self,
        *,
        accounts: typing.Optional[typing.Sequence[builtins.str]] = None,
        organization_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
        roles: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.IRole]] = None,
        share_permission: typing.Optional[typing.Union[builtins.str, "SharePermission"]] = None,
        users: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.IUser]] = None,
    ) -> None:
        '''(experimental) Share the attribute group resource with other IAM entities, accounts, or OUs.

        :param accounts: (experimental) A list of AWS accounts that the application will be shared with. Default: - No accounts specified for share
        :param organization_arns: (experimental) A list of AWS Organization or Organizational Units (OUs) ARNs that the application will be shared with. Default: - No AWS Organizations or OUs specified for share
        :param roles: (experimental) A list of AWS IAM roles that the application will be shared with. Default: - No IAM roles specified for share
        :param share_permission: (experimental) An option to manage access to the application or attribute group. Default: - Principals will be assigned read only permissions on the application or attribute group.
        :param users: (experimental) A list of AWS IAM users that the application will be shared with. Default: - No IAM Users specified for share

        :stability: experimental
        '''
        share_options = ShareOptions(
            accounts=accounts,
            organization_arns=organization_arns,
            roles=roles,
            share_permission=share_permission,
            users=users,
        )

        return typing.cast(None, jsii.invoke(self, "shareAttributeGroup", [share_options]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IAttributeGroup).__jsii_proxy_class__ = lambda : _IAttributeGroupProxy


@jsii.data_type(
    jsii_type="@aws-cdk/aws-servicecatalogappregistry-alpha.ShareOptions",
    jsii_struct_bases=[],
    name_mapping={
        "accounts": "accounts",
        "organization_arns": "organizationArns",
        "roles": "roles",
        "share_permission": "sharePermission",
        "users": "users",
    },
)
class ShareOptions:
    def __init__(
        self,
        *,
        accounts: typing.Optional[typing.Sequence[builtins.str]] = None,
        organization_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
        roles: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.IRole]] = None,
        share_permission: typing.Optional[typing.Union[builtins.str, "SharePermission"]] = None,
        users: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.IUser]] = None,
    ) -> None:
        '''(experimental) The options that are passed into a share of an Application or Attribute Group.

        :param accounts: (experimental) A list of AWS accounts that the application will be shared with. Default: - No accounts specified for share
        :param organization_arns: (experimental) A list of AWS Organization or Organizational Units (OUs) ARNs that the application will be shared with. Default: - No AWS Organizations or OUs specified for share
        :param roles: (experimental) A list of AWS IAM roles that the application will be shared with. Default: - No IAM roles specified for share
        :param share_permission: (experimental) An option to manage access to the application or attribute group. Default: - Principals will be assigned read only permissions on the application or attribute group.
        :param users: (experimental) A list of AWS IAM users that the application will be shared with. Default: - No IAM Users specified for share

        :stability: experimental
        :exampleMetadata: infused

        Example::

            import aws_cdk.aws_iam as iam
            # application: appreg.Application
            # my_role: iam.IRole
            # my_user: iam.IUser
            
            application.share_application(
                accounts=["123456789012"],
                organization_arns=["arn:aws:organizations::123456789012:organization/o-my-org-id"],
                roles=[my_role],
                users=[my_user]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f20907f839e03d66b58888eece0b2a80abd0c0240c29a95fb103642e09cbfa2e)
            check_type(argname="argument accounts", value=accounts, expected_type=type_hints["accounts"])
            check_type(argname="argument organization_arns", value=organization_arns, expected_type=type_hints["organization_arns"])
            check_type(argname="argument roles", value=roles, expected_type=type_hints["roles"])
            check_type(argname="argument share_permission", value=share_permission, expected_type=type_hints["share_permission"])
            check_type(argname="argument users", value=users, expected_type=type_hints["users"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if accounts is not None:
            self._values["accounts"] = accounts
        if organization_arns is not None:
            self._values["organization_arns"] = organization_arns
        if roles is not None:
            self._values["roles"] = roles
        if share_permission is not None:
            self._values["share_permission"] = share_permission
        if users is not None:
            self._values["users"] = users

    @builtins.property
    def accounts(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) A list of AWS accounts that the application will be shared with.

        :default: - No accounts specified for share

        :stability: experimental
        '''
        result = self._values.get("accounts")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def organization_arns(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) A list of AWS Organization or Organizational Units (OUs) ARNs that the application will be shared with.

        :default: - No AWS Organizations or OUs specified for share

        :stability: experimental
        '''
        result = self._values.get("organization_arns")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def roles(self) -> typing.Optional[typing.List[_aws_cdk_aws_iam_ceddda9d.IRole]]:
        '''(experimental) A list of AWS IAM roles that the application will be shared with.

        :default: - No IAM roles specified for share

        :stability: experimental
        '''
        result = self._values.get("roles")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_aws_iam_ceddda9d.IRole]], result)

    @builtins.property
    def share_permission(
        self,
    ) -> typing.Optional[typing.Union[builtins.str, "SharePermission"]]:
        '''(experimental) An option to manage access to the application or attribute group.

        :default: - Principals will be assigned read only permissions on the application or attribute group.

        :stability: experimental
        '''
        result = self._values.get("share_permission")
        return typing.cast(typing.Optional[typing.Union[builtins.str, "SharePermission"]], result)

    @builtins.property
    def users(self) -> typing.Optional[typing.List[_aws_cdk_aws_iam_ceddda9d.IUser]]:
        '''(experimental) A list of AWS IAM users that the application will be shared with.

        :default: - No IAM Users specified for share

        :stability: experimental
        '''
        result = self._values.get("users")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_aws_iam_ceddda9d.IUser]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ShareOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@aws-cdk/aws-servicecatalogappregistry-alpha.SharePermission")
class SharePermission(enum.Enum):
    '''(experimental) Supported permissions for sharing applications or attribute groups with principals using AWS RAM.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        import aws_cdk.aws_iam as iam
        # application: appreg.Application
        
        application.share_application(
            accounts=["123456789012", "234567890123"],
            share_permission=appreg.SharePermission.ALLOW_ACCESS
        )
    '''

    READ_ONLY = "READ_ONLY"
    '''(experimental) Allows principals in the share to only view the application or attribute group.

    :stability: experimental
    '''
    ALLOW_ACCESS = "ALLOW_ACCESS"
    '''(experimental) Allows principals in the share to associate resources and attribute groups with applications.

    :stability: experimental
    '''


class TargetApplication(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@aws-cdk/aws-servicecatalogappregistry-alpha.TargetApplication",
):
    '''(experimental) Contains static factory methods with which you can build the input needed for application associator to work.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        app = App()
        associated_app = appreg.ApplicationAssociator(app, "AssociatedApplication",
            applications=[appreg.TargetApplication.create_application_stack(
                application_name="MyAssociatedApplication",
                # 'Application containing stacks deployed via CDK.' is the default
                application_description="Associated Application description",
                stack_name="MyAssociatedApplicationStack",
                # AWS Account and Region that are implied by the current CLI configuration is the default
                env=Environment(account="123456789012", region="us-east-1")
            )]
        )
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="createApplicationStack")
    @builtins.classmethod
    def create_application_stack(
        cls,
        *,
        application_name: builtins.str,
        application_description: typing.Optional[builtins.str] = None,
        stack_id: typing.Optional[builtins.str] = None,
        analytics_reporting: typing.Optional[builtins.bool] = None,
        cross_region_references: typing.Optional[builtins.bool] = None,
        description: typing.Optional[builtins.str] = None,
        env: typing.Optional[typing.Union[_aws_cdk_ceddda9d.Environment, typing.Dict[builtins.str, typing.Any]]] = None,
        permissions_boundary: typing.Optional[_aws_cdk_ceddda9d.PermissionsBoundary] = None,
        stack_name: typing.Optional[builtins.str] = None,
        synthesizer: typing.Optional[_aws_cdk_ceddda9d.IStackSynthesizer] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        termination_protection: typing.Optional[builtins.bool] = None,
    ) -> "TargetApplication":
        '''(experimental) Factory method to build the input using the provided application name and stack props.

        :param application_name: (experimental) Enforces a particular physical application name.
        :param application_description: (experimental) Application description. Default: - Application containing stacks deployed via CDK.
        :param stack_id: (experimental) Stack ID in which application will be created or imported. The id of a stack is also the identifier that you use to refer to it in the `AWS CDK Toolkit <https://docs.aws.amazon.com/cdk/v2/guide/cli.html>`_. Default: - ApplicationAssociatorStack
        :param analytics_reporting: Include runtime versioning information in this Stack. Default: ``analyticsReporting`` setting of containing ``App``, or value of 'aws:cdk:version-reporting' context key
        :param cross_region_references: Enable this flag to allow native cross region stack references. Enabling this will create a CloudFormation custom resource in both the producing stack and consuming stack in order to perform the export/import This feature is currently experimental Default: false
        :param description: A description of the stack. Default: - No description.
        :param env: The AWS environment (account/region) where this stack will be deployed. Set the ``region``/``account`` fields of ``env`` to either a concrete value to select the indicated environment (recommended for production stacks), or to the values of environment variables ``CDK_DEFAULT_REGION``/``CDK_DEFAULT_ACCOUNT`` to let the target environment depend on the AWS credentials/configuration that the CDK CLI is executed under (recommended for development stacks). If the ``Stack`` is instantiated inside a ``Stage``, any undefined ``region``/``account`` fields from ``env`` will default to the same field on the encompassing ``Stage``, if configured there. If either ``region`` or ``account`` are not set nor inherited from ``Stage``, the Stack will be considered "*environment-agnostic*"". Environment-agnostic stacks can be deployed to any environment but may not be able to take advantage of all features of the CDK. For example, they will not be able to use environmental context lookups such as ``ec2.Vpc.fromLookup`` and will not automatically translate Service Principals to the right format based on the environment's AWS partition, and other such enhancements. Default: - The environment of the containing ``Stage`` if available, otherwise create the stack will be environment-agnostic.
        :param permissions_boundary: Options for applying a permissions boundary to all IAM Roles and Users created within this Stage. Default: - no permissions boundary is applied
        :param stack_name: Name to deploy the stack with. Default: - Derived from construct path.
        :param synthesizer: Synthesis method to use while deploying this stack. The Stack Synthesizer controls aspects of synthesis and deployment, like how assets are referenced and what IAM roles to use. For more information, see the README of the main CDK package. If not specified, the ``defaultStackSynthesizer`` from ``App`` will be used. If that is not specified, ``DefaultStackSynthesizer`` is used if ``@aws-cdk/core:newStyleStackSynthesis`` is set to ``true`` or the CDK major version is v2. In CDK v1 ``LegacyStackSynthesizer`` is the default if no other synthesizer is specified. Default: - The synthesizer specified on ``App``, or ``DefaultStackSynthesizer`` otherwise.
        :param tags: Stack tags that will be applied to all the taggable resources and the stack itself. Default: {}
        :param termination_protection: Whether to enable termination protection for this stack. Default: false

        :stability: experimental
        '''
        options = CreateTargetApplicationOptions(
            application_name=application_name,
            application_description=application_description,
            stack_id=stack_id,
            analytics_reporting=analytics_reporting,
            cross_region_references=cross_region_references,
            description=description,
            env=env,
            permissions_boundary=permissions_boundary,
            stack_name=stack_name,
            synthesizer=synthesizer,
            tags=tags,
            termination_protection=termination_protection,
        )

        return typing.cast("TargetApplication", jsii.sinvoke(cls, "createApplicationStack", [options]))

    @jsii.member(jsii_name="existingApplicationFromArn")
    @builtins.classmethod
    def existing_application_from_arn(
        cls,
        *,
        application_arn_value: builtins.str,
        stack_id: typing.Optional[builtins.str] = None,
        analytics_reporting: typing.Optional[builtins.bool] = None,
        cross_region_references: typing.Optional[builtins.bool] = None,
        description: typing.Optional[builtins.str] = None,
        env: typing.Optional[typing.Union[_aws_cdk_ceddda9d.Environment, typing.Dict[builtins.str, typing.Any]]] = None,
        permissions_boundary: typing.Optional[_aws_cdk_ceddda9d.PermissionsBoundary] = None,
        stack_name: typing.Optional[builtins.str] = None,
        synthesizer: typing.Optional[_aws_cdk_ceddda9d.IStackSynthesizer] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        termination_protection: typing.Optional[builtins.bool] = None,
    ) -> "TargetApplication":
        '''(experimental) Factory method to build the input using the provided application ARN.

        :param application_arn_value: (experimental) Enforces a particular application arn.
        :param stack_id: (experimental) Stack ID in which application will be created or imported. The id of a stack is also the identifier that you use to refer to it in the `AWS CDK Toolkit <https://docs.aws.amazon.com/cdk/v2/guide/cli.html>`_. Default: - ApplicationAssociatorStack
        :param analytics_reporting: Include runtime versioning information in this Stack. Default: ``analyticsReporting`` setting of containing ``App``, or value of 'aws:cdk:version-reporting' context key
        :param cross_region_references: Enable this flag to allow native cross region stack references. Enabling this will create a CloudFormation custom resource in both the producing stack and consuming stack in order to perform the export/import This feature is currently experimental Default: false
        :param description: A description of the stack. Default: - No description.
        :param env: The AWS environment (account/region) where this stack will be deployed. Set the ``region``/``account`` fields of ``env`` to either a concrete value to select the indicated environment (recommended for production stacks), or to the values of environment variables ``CDK_DEFAULT_REGION``/``CDK_DEFAULT_ACCOUNT`` to let the target environment depend on the AWS credentials/configuration that the CDK CLI is executed under (recommended for development stacks). If the ``Stack`` is instantiated inside a ``Stage``, any undefined ``region``/``account`` fields from ``env`` will default to the same field on the encompassing ``Stage``, if configured there. If either ``region`` or ``account`` are not set nor inherited from ``Stage``, the Stack will be considered "*environment-agnostic*"". Environment-agnostic stacks can be deployed to any environment but may not be able to take advantage of all features of the CDK. For example, they will not be able to use environmental context lookups such as ``ec2.Vpc.fromLookup`` and will not automatically translate Service Principals to the right format based on the environment's AWS partition, and other such enhancements. Default: - The environment of the containing ``Stage`` if available, otherwise create the stack will be environment-agnostic.
        :param permissions_boundary: Options for applying a permissions boundary to all IAM Roles and Users created within this Stage. Default: - no permissions boundary is applied
        :param stack_name: Name to deploy the stack with. Default: - Derived from construct path.
        :param synthesizer: Synthesis method to use while deploying this stack. The Stack Synthesizer controls aspects of synthesis and deployment, like how assets are referenced and what IAM roles to use. For more information, see the README of the main CDK package. If not specified, the ``defaultStackSynthesizer`` from ``App`` will be used. If that is not specified, ``DefaultStackSynthesizer`` is used if ``@aws-cdk/core:newStyleStackSynthesis`` is set to ``true`` or the CDK major version is v2. In CDK v1 ``LegacyStackSynthesizer`` is the default if no other synthesizer is specified. Default: - The synthesizer specified on ``App``, or ``DefaultStackSynthesizer`` otherwise.
        :param tags: Stack tags that will be applied to all the taggable resources and the stack itself. Default: {}
        :param termination_protection: Whether to enable termination protection for this stack. Default: false

        :stability: experimental
        '''
        options = ExistingTargetApplicationOptions(
            application_arn_value=application_arn_value,
            stack_id=stack_id,
            analytics_reporting=analytics_reporting,
            cross_region_references=cross_region_references,
            description=description,
            env=env,
            permissions_boundary=permissions_boundary,
            stack_name=stack_name,
            synthesizer=synthesizer,
            tags=tags,
            termination_protection=termination_protection,
        )

        return typing.cast("TargetApplication", jsii.sinvoke(cls, "existingApplicationFromArn", [options]))

    @jsii.member(jsii_name="bind")
    @abc.abstractmethod
    def bind(
        self,
        scope: _constructs_77d1e7e8.Construct,
    ) -> BindTargetApplicationResult:
        '''(experimental) Called when the ApplicationAssociator is initialized.

        :param scope: -

        :stability: experimental
        '''
        ...


class _TargetApplicationProxy(TargetApplication):
    @jsii.member(jsii_name="bind")
    def bind(
        self,
        scope: _constructs_77d1e7e8.Construct,
    ) -> BindTargetApplicationResult:
        '''(experimental) Called when the ApplicationAssociator is initialized.

        :param scope: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__676c526376ea5a57da1d842b32eeaba48f39014bebf2de107e52abfb0ef30471)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast(BindTargetApplicationResult, jsii.invoke(self, "bind", [scope]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, TargetApplication).__jsii_proxy_class__ = lambda : _TargetApplicationProxy


@jsii.data_type(
    jsii_type="@aws-cdk/aws-servicecatalogappregistry-alpha.TargetApplicationCommonOptions",
    jsii_struct_bases=[_aws_cdk_ceddda9d.StackProps],
    name_mapping={
        "analytics_reporting": "analyticsReporting",
        "cross_region_references": "crossRegionReferences",
        "description": "description",
        "env": "env",
        "permissions_boundary": "permissionsBoundary",
        "stack_name": "stackName",
        "synthesizer": "synthesizer",
        "tags": "tags",
        "termination_protection": "terminationProtection",
        "stack_id": "stackId",
    },
)
class TargetApplicationCommonOptions(_aws_cdk_ceddda9d.StackProps):
    def __init__(
        self,
        *,
        analytics_reporting: typing.Optional[builtins.bool] = None,
        cross_region_references: typing.Optional[builtins.bool] = None,
        description: typing.Optional[builtins.str] = None,
        env: typing.Optional[typing.Union[_aws_cdk_ceddda9d.Environment, typing.Dict[builtins.str, typing.Any]]] = None,
        permissions_boundary: typing.Optional[_aws_cdk_ceddda9d.PermissionsBoundary] = None,
        stack_name: typing.Optional[builtins.str] = None,
        synthesizer: typing.Optional[_aws_cdk_ceddda9d.IStackSynthesizer] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        termination_protection: typing.Optional[builtins.bool] = None,
        stack_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Properties used to define targetapplication.

        :param analytics_reporting: Include runtime versioning information in this Stack. Default: ``analyticsReporting`` setting of containing ``App``, or value of 'aws:cdk:version-reporting' context key
        :param cross_region_references: Enable this flag to allow native cross region stack references. Enabling this will create a CloudFormation custom resource in both the producing stack and consuming stack in order to perform the export/import This feature is currently experimental Default: false
        :param description: A description of the stack. Default: - No description.
        :param env: The AWS environment (account/region) where this stack will be deployed. Set the ``region``/``account`` fields of ``env`` to either a concrete value to select the indicated environment (recommended for production stacks), or to the values of environment variables ``CDK_DEFAULT_REGION``/``CDK_DEFAULT_ACCOUNT`` to let the target environment depend on the AWS credentials/configuration that the CDK CLI is executed under (recommended for development stacks). If the ``Stack`` is instantiated inside a ``Stage``, any undefined ``region``/``account`` fields from ``env`` will default to the same field on the encompassing ``Stage``, if configured there. If either ``region`` or ``account`` are not set nor inherited from ``Stage``, the Stack will be considered "*environment-agnostic*"". Environment-agnostic stacks can be deployed to any environment but may not be able to take advantage of all features of the CDK. For example, they will not be able to use environmental context lookups such as ``ec2.Vpc.fromLookup`` and will not automatically translate Service Principals to the right format based on the environment's AWS partition, and other such enhancements. Default: - The environment of the containing ``Stage`` if available, otherwise create the stack will be environment-agnostic.
        :param permissions_boundary: Options for applying a permissions boundary to all IAM Roles and Users created within this Stage. Default: - no permissions boundary is applied
        :param stack_name: Name to deploy the stack with. Default: - Derived from construct path.
        :param synthesizer: Synthesis method to use while deploying this stack. The Stack Synthesizer controls aspects of synthesis and deployment, like how assets are referenced and what IAM roles to use. For more information, see the README of the main CDK package. If not specified, the ``defaultStackSynthesizer`` from ``App`` will be used. If that is not specified, ``DefaultStackSynthesizer`` is used if ``@aws-cdk/core:newStyleStackSynthesis`` is set to ``true`` or the CDK major version is v2. In CDK v1 ``LegacyStackSynthesizer`` is the default if no other synthesizer is specified. Default: - The synthesizer specified on ``App``, or ``DefaultStackSynthesizer`` otherwise.
        :param tags: Stack tags that will be applied to all the taggable resources and the stack itself. Default: {}
        :param termination_protection: Whether to enable termination protection for this stack. Default: false
        :param stack_id: (experimental) Stack ID in which application will be created or imported. The id of a stack is also the identifier that you use to refer to it in the `AWS CDK Toolkit <https://docs.aws.amazon.com/cdk/v2/guide/cli.html>`_. Default: - ApplicationAssociatorStack

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_servicecatalogappregistry_alpha as servicecatalogappregistry_alpha
            import aws_cdk as cdk
            
            # permissions_boundary: cdk.PermissionsBoundary
            # stack_synthesizer: cdk.StackSynthesizer
            
            target_application_common_options = servicecatalogappregistry_alpha.TargetApplicationCommonOptions(
                analytics_reporting=False,
                cross_region_references=False,
                description="description",
                env=cdk.Environment(
                    account="account",
                    region="region"
                ),
                permissions_boundary=permissions_boundary,
                stack_id="stackId",
                stack_name="stackName",
                synthesizer=stack_synthesizer,
                tags={
                    "tags_key": "tags"
                },
                termination_protection=False
            )
        '''
        if isinstance(env, dict):
            env = _aws_cdk_ceddda9d.Environment(**env)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8e9a8c85d5db3b4596f192c3fe206a4e30e4143fb84b5be58ef7f29131edc782)
            check_type(argname="argument analytics_reporting", value=analytics_reporting, expected_type=type_hints["analytics_reporting"])
            check_type(argname="argument cross_region_references", value=cross_region_references, expected_type=type_hints["cross_region_references"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument env", value=env, expected_type=type_hints["env"])
            check_type(argname="argument permissions_boundary", value=permissions_boundary, expected_type=type_hints["permissions_boundary"])
            check_type(argname="argument stack_name", value=stack_name, expected_type=type_hints["stack_name"])
            check_type(argname="argument synthesizer", value=synthesizer, expected_type=type_hints["synthesizer"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument termination_protection", value=termination_protection, expected_type=type_hints["termination_protection"])
            check_type(argname="argument stack_id", value=stack_id, expected_type=type_hints["stack_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if analytics_reporting is not None:
            self._values["analytics_reporting"] = analytics_reporting
        if cross_region_references is not None:
            self._values["cross_region_references"] = cross_region_references
        if description is not None:
            self._values["description"] = description
        if env is not None:
            self._values["env"] = env
        if permissions_boundary is not None:
            self._values["permissions_boundary"] = permissions_boundary
        if stack_name is not None:
            self._values["stack_name"] = stack_name
        if synthesizer is not None:
            self._values["synthesizer"] = synthesizer
        if tags is not None:
            self._values["tags"] = tags
        if termination_protection is not None:
            self._values["termination_protection"] = termination_protection
        if stack_id is not None:
            self._values["stack_id"] = stack_id

    @builtins.property
    def analytics_reporting(self) -> typing.Optional[builtins.bool]:
        '''Include runtime versioning information in this Stack.

        :default:

        ``analyticsReporting`` setting of containing ``App``, or value of
        'aws:cdk:version-reporting' context key
        '''
        result = self._values.get("analytics_reporting")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def cross_region_references(self) -> typing.Optional[builtins.bool]:
        '''Enable this flag to allow native cross region stack references.

        Enabling this will create a CloudFormation custom resource
        in both the producing stack and consuming stack in order to perform the export/import

        This feature is currently experimental

        :default: false
        '''
        result = self._values.get("cross_region_references")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the stack.

        :default: - No description.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def env(self) -> typing.Optional[_aws_cdk_ceddda9d.Environment]:
        '''The AWS environment (account/region) where this stack will be deployed.

        Set the ``region``/``account`` fields of ``env`` to either a concrete value to
        select the indicated environment (recommended for production stacks), or to
        the values of environment variables
        ``CDK_DEFAULT_REGION``/``CDK_DEFAULT_ACCOUNT`` to let the target environment
        depend on the AWS credentials/configuration that the CDK CLI is executed
        under (recommended for development stacks).

        If the ``Stack`` is instantiated inside a ``Stage``, any undefined
        ``region``/``account`` fields from ``env`` will default to the same field on the
        encompassing ``Stage``, if configured there.

        If either ``region`` or ``account`` are not set nor inherited from ``Stage``, the
        Stack will be considered "*environment-agnostic*"". Environment-agnostic
        stacks can be deployed to any environment but may not be able to take
        advantage of all features of the CDK. For example, they will not be able to
        use environmental context lookups such as ``ec2.Vpc.fromLookup`` and will not
        automatically translate Service Principals to the right format based on the
        environment's AWS partition, and other such enhancements.

        :default:

        - The environment of the containing ``Stage`` if available,
        otherwise create the stack will be environment-agnostic.

        Example::

            # Example automatically generated from non-compiling source. May contain errors.
            # Use a concrete account and region to deploy this stack to:
            # `.account` and `.region` will simply return these values.
            Stack(app, "Stack1",
                env=Environment(
                    account="123456789012",
                    region="us-east-1"
                )
            )
            
            # Use the CLI's current credentials to determine the target environment:
            # `.account` and `.region` will reflect the account+region the CLI
            # is configured to use (based on the user CLI credentials)
            Stack(app, "Stack2",
                env=Environment(
                    account=process.env.CDK_DEFAULT_ACCOUNT,
                    region=process.env.CDK_DEFAULT_REGION
                )
            )
            
            # Define multiple stacks stage associated with an environment
            my_stage = Stage(app, "MyStage",
                env=Environment(
                    account="123456789012",
                    region="us-east-1"
                )
            )
            
            # both of these stacks will use the stage's account/region:
            # `.account` and `.region` will resolve to the concrete values as above
            MyStack(my_stage, "Stack1")
            YourStack(my_stage, "Stack2")
            
            # Define an environment-agnostic stack:
            # `.account` and `.region` will resolve to `{ "Ref": "AWS::AccountId" }` and `{ "Ref": "AWS::Region" }` respectively.
            # which will only resolve to actual values by CloudFormation during deployment.
            MyStack(app, "Stack1")
        '''
        result = self._values.get("env")
        return typing.cast(typing.Optional[_aws_cdk_ceddda9d.Environment], result)

    @builtins.property
    def permissions_boundary(
        self,
    ) -> typing.Optional[_aws_cdk_ceddda9d.PermissionsBoundary]:
        '''Options for applying a permissions boundary to all IAM Roles and Users created within this Stage.

        :default: - no permissions boundary is applied
        '''
        result = self._values.get("permissions_boundary")
        return typing.cast(typing.Optional[_aws_cdk_ceddda9d.PermissionsBoundary], result)

    @builtins.property
    def stack_name(self) -> typing.Optional[builtins.str]:
        '''Name to deploy the stack with.

        :default: - Derived from construct path.
        '''
        result = self._values.get("stack_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def synthesizer(self) -> typing.Optional[_aws_cdk_ceddda9d.IStackSynthesizer]:
        '''Synthesis method to use while deploying this stack.

        The Stack Synthesizer controls aspects of synthesis and deployment,
        like how assets are referenced and what IAM roles to use. For more
        information, see the README of the main CDK package.

        If not specified, the ``defaultStackSynthesizer`` from ``App`` will be used.
        If that is not specified, ``DefaultStackSynthesizer`` is used if
        ``@aws-cdk/core:newStyleStackSynthesis`` is set to ``true`` or the CDK major
        version is v2. In CDK v1 ``LegacyStackSynthesizer`` is the default if no
        other synthesizer is specified.

        :default: - The synthesizer specified on ``App``, or ``DefaultStackSynthesizer`` otherwise.
        '''
        result = self._values.get("synthesizer")
        return typing.cast(typing.Optional[_aws_cdk_ceddda9d.IStackSynthesizer], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Stack tags that will be applied to all the taggable resources and the stack itself.

        :default: {}
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def termination_protection(self) -> typing.Optional[builtins.bool]:
        '''Whether to enable termination protection for this stack.

        :default: false
        '''
        result = self._values.get("termination_protection")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def stack_id(self) -> typing.Optional[builtins.str]:
        '''(experimental) Stack ID in which application will be created or imported.

        The id of a stack is also the identifier that you use to
        refer to it in the `AWS CDK Toolkit <https://docs.aws.amazon.com/cdk/v2/guide/cli.html>`_.

        :default: - ApplicationAssociatorStack

        :stability: experimental
        '''
        result = self._values.get("stack_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "TargetApplicationCommonOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IApplication)
class Application(
    _aws_cdk_ceddda9d.Resource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-servicecatalogappregistry-alpha.Application",
):
    '''(experimental) A Service Catalog AppRegistry Application.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        application = appreg.Application(self, "MyFirstApplication",
            application_name="MyFirstApplicationName",
            description="description for my application"
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        application_name: builtins.str,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param application_name: (experimental) Enforces a particular physical application name.
        :param description: (experimental) Description for application. Default: - No description provided

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d00f48aa19f1be0379562df9ddbae70ebc3518362d0411437cb95dfcdd2d2163)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = ApplicationProps(
            application_name=application_name, description=description
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="fromApplicationArn")
    @builtins.classmethod
    def from_application_arn(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        application_arn: builtins.str,
    ) -> IApplication:
        '''(experimental) Imports an Application construct that represents an external application.

        :param scope: The parent creating construct (usually ``this``).
        :param id: The construct's name.
        :param application_arn: the Amazon Resource Name of the existing AppRegistry Application.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c9f55f3016c1743f9bf17bd3321531f86b5a2fcff42b5b44073089b900c1bc3e)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
            check_type(argname="argument application_arn", value=application_arn, expected_type=type_hints["application_arn"])
        return typing.cast(IApplication, jsii.sinvoke(cls, "fromApplicationArn", [scope, id, application_arn]))

    @jsii.member(jsii_name="associateAllStacksInScope")
    def associate_all_stacks_in_scope(
        self,
        scope: _constructs_77d1e7e8.Construct,
    ) -> None:
        '''(experimental) Associate all stacks present in construct's aspect with application.

        NOTE: This method won't automatically register stacks under pipeline stages,
        and requires association of each pipeline stage by calling this method with stage Construct.

        :param scope: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c009ad74ae52d87dcfb6e1aee6e9569d5807c9f2c9f0bc6691cc089dacece4a8)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        return typing.cast(None, jsii.invoke(self, "associateAllStacksInScope", [scope]))

    @jsii.member(jsii_name="associateApplicationWithStack")
    def associate_application_with_stack(self, stack: _aws_cdk_ceddda9d.Stack) -> None:
        '''(experimental) Associate stack with the application in the stack passed as parameter.

        A stack can only be associated with one application.

        :param stack: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__875f08936c5ec9d623a247709774d3625560a74cad5380509cb4624eecad2f27)
            check_type(argname="argument stack", value=stack, expected_type=type_hints["stack"])
        return typing.cast(None, jsii.invoke(self, "associateApplicationWithStack", [stack]))

    @jsii.member(jsii_name="associateAttributeGroup")
    def associate_attribute_group(self, attribute_group: IAttributeGroup) -> None:
        '''(experimental) Associate an attribute group with application If the attribute group is already associated, it will ignore duplicate request.

        :param attribute_group: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6e09d5799ecd5be55b6bce6f34e72f3f0d7d0c179ac0824b4e961e49687e3871)
            check_type(argname="argument attribute_group", value=attribute_group, expected_type=type_hints["attribute_group"])
        return typing.cast(None, jsii.invoke(self, "associateAttributeGroup", [attribute_group]))

    @jsii.member(jsii_name="associateStack")
    def associate_stack(self, stack: _aws_cdk_ceddda9d.Stack) -> None:
        '''(deprecated) Associate a stack with the application If the resource is already associated, it will ignore duplicate request.

        A stack can only be associated with one application.

        :param stack: -

        :deprecated: Use ``associateApplicationWithStack`` instead.

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__09e36e72eb27ec19021721817f2caa31bde7114fd6069eedbb069641910066e0)
            check_type(argname="argument stack", value=stack, expected_type=type_hints["stack"])
        return typing.cast(None, jsii.invoke(self, "associateStack", [stack]))

    @jsii.member(jsii_name="generateUniqueHash")
    def _generate_unique_hash(self, resource_address: builtins.str) -> builtins.str:
        '''(experimental) Create a unique id.

        :param resource_address: -

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__80266c77c192e878af5d982db7685162f1f646d9574aad4d8a432ef62520b1ab)
            check_type(argname="argument resource_address", value=resource_address, expected_type=type_hints["resource_address"])
        return typing.cast(builtins.str, jsii.invoke(self, "generateUniqueHash", [resource_address]))

    @jsii.member(jsii_name="shareApplication")
    def share_application(
        self,
        *,
        accounts: typing.Optional[typing.Sequence[builtins.str]] = None,
        organization_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
        roles: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.IRole]] = None,
        share_permission: typing.Optional[typing.Union[builtins.str, SharePermission]] = None,
        users: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.IUser]] = None,
    ) -> None:
        '''(experimental) Share an application with accounts, organizations and OUs, and IAM roles and users.

        The application will become available to end users within those principals.

        :param accounts: (experimental) A list of AWS accounts that the application will be shared with. Default: - No accounts specified for share
        :param organization_arns: (experimental) A list of AWS Organization or Organizational Units (OUs) ARNs that the application will be shared with. Default: - No AWS Organizations or OUs specified for share
        :param roles: (experimental) A list of AWS IAM roles that the application will be shared with. Default: - No IAM roles specified for share
        :param share_permission: (experimental) An option to manage access to the application or attribute group. Default: - Principals will be assigned read only permissions on the application or attribute group.
        :param users: (experimental) A list of AWS IAM users that the application will be shared with. Default: - No IAM Users specified for share

        :stability: experimental
        '''
        share_options = ShareOptions(
            accounts=accounts,
            organization_arns=organization_arns,
            roles=roles,
            share_permission=share_permission,
            users=users,
        )

        return typing.cast(None, jsii.invoke(self, "shareApplication", [share_options]))

    @builtins.property
    @jsii.member(jsii_name="applicationArn")
    def application_arn(self) -> builtins.str:
        '''(experimental) The ARN of the application.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "applicationArn"))

    @builtins.property
    @jsii.member(jsii_name="applicationId")
    def application_id(self) -> builtins.str:
        '''(experimental) The ID of the application.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "applicationId"))

    @builtins.property
    @jsii.member(jsii_name="applicationManagerUrl")
    def application_manager_url(self) -> typing.Optional[_aws_cdk_ceddda9d.CfnOutput]:
        '''(experimental) Application manager URL for the Application.

        :stability: experimental
        :attribute: true
        '''
        return typing.cast(typing.Optional[_aws_cdk_ceddda9d.CfnOutput], jsii.get(self, "applicationManagerUrl"))

    @builtins.property
    @jsii.member(jsii_name="applicationName")
    def application_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the application.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "applicationName"))


@jsii.implements(IAttributeGroup)
class AttributeGroup(
    _aws_cdk_ceddda9d.Resource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-servicecatalogappregistry-alpha.AttributeGroup",
):
    '''(experimental) A Service Catalog AppRegistry Attribute Group.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        attribute_group = appreg.AttributeGroup(self, "MyFirstAttributeGroup",
            attribute_group_name="MyFirstAttributeGroupName",
            description="description for my attribute group",  # the description is optional,
            attributes={
                "project": "foo",
                "team": ["member1", "member2", "member3"],
                "public": False,
                "stages": {
                    "alpha": "complete",
                    "beta": "incomplete",
                    "release": "not started"
                }
            }
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        attribute_group_name: builtins.str,
        attributes: typing.Mapping[builtins.str, typing.Any],
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param attribute_group_name: (experimental) Enforces a particular physical attribute group name.
        :param attributes: (experimental) A JSON of nested key-value pairs that represent the attributes in the group. Attributes maybe an empty JSON '{}', but must be explicitly stated.
        :param description: (experimental) Description for attribute group. Default: - No description provided

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__91ce42b356b8555b9df82de3713a41ca962e33062b181b37794c5e3b82d7b12f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = AttributeGroupProps(
            attribute_group_name=attribute_group_name,
            attributes=attributes,
            description=description,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="fromAttributeGroupArn")
    @builtins.classmethod
    def from_attribute_group_arn(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        attribute_group_arn: builtins.str,
    ) -> IAttributeGroup:
        '''(experimental) Imports an attribute group construct that represents an external attribute group.

        :param scope: The parent creating construct (usually ``this``).
        :param id: The construct's name.
        :param attribute_group_arn: the Amazon Resource Name of the existing AppRegistry attribute group.

        :stability: experimental
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d090f29cba3280a4ce331eeac9f6003b4c6b25cc65422ee58257d036241687aa)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
            check_type(argname="argument attribute_group_arn", value=attribute_group_arn, expected_type=type_hints["attribute_group_arn"])
        return typing.cast(IAttributeGroup, jsii.sinvoke(cls, "fromAttributeGroupArn", [scope, id, attribute_group_arn]))

    @jsii.member(jsii_name="getAttributeGroupSharePermissionARN")
    def _get_attribute_group_share_permission_arn(
        self,
        *,
        accounts: typing.Optional[typing.Sequence[builtins.str]] = None,
        organization_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
        roles: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.IRole]] = None,
        share_permission: typing.Optional[typing.Union[builtins.str, SharePermission]] = None,
        users: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.IUser]] = None,
    ) -> builtins.str:
        '''(experimental) Get the correct permission ARN based on the SharePermission.

        :param accounts: (experimental) A list of AWS accounts that the application will be shared with. Default: - No accounts specified for share
        :param organization_arns: (experimental) A list of AWS Organization or Organizational Units (OUs) ARNs that the application will be shared with. Default: - No AWS Organizations or OUs specified for share
        :param roles: (experimental) A list of AWS IAM roles that the application will be shared with. Default: - No IAM roles specified for share
        :param share_permission: (experimental) An option to manage access to the application or attribute group. Default: - Principals will be assigned read only permissions on the application or attribute group.
        :param users: (experimental) A list of AWS IAM users that the application will be shared with. Default: - No IAM Users specified for share

        :stability: experimental
        '''
        share_options = ShareOptions(
            accounts=accounts,
            organization_arns=organization_arns,
            roles=roles,
            share_permission=share_permission,
            users=users,
        )

        return typing.cast(builtins.str, jsii.invoke(self, "getAttributeGroupSharePermissionARN", [share_options]))

    @jsii.member(jsii_name="shareAttributeGroup")
    def share_attribute_group(
        self,
        *,
        accounts: typing.Optional[typing.Sequence[builtins.str]] = None,
        organization_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
        roles: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.IRole]] = None,
        share_permission: typing.Optional[typing.Union[builtins.str, SharePermission]] = None,
        users: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.IUser]] = None,
    ) -> None:
        '''(experimental) Share the attribute group resource with other IAM entities, accounts, or OUs.

        :param accounts: (experimental) A list of AWS accounts that the application will be shared with. Default: - No accounts specified for share
        :param organization_arns: (experimental) A list of AWS Organization or Organizational Units (OUs) ARNs that the application will be shared with. Default: - No AWS Organizations or OUs specified for share
        :param roles: (experimental) A list of AWS IAM roles that the application will be shared with. Default: - No IAM roles specified for share
        :param share_permission: (experimental) An option to manage access to the application or attribute group. Default: - Principals will be assigned read only permissions on the application or attribute group.
        :param users: (experimental) A list of AWS IAM users that the application will be shared with. Default: - No IAM Users specified for share

        :stability: experimental
        '''
        share_options = ShareOptions(
            accounts=accounts,
            organization_arns=organization_arns,
            roles=roles,
            share_permission=share_permission,
            users=users,
        )

        return typing.cast(None, jsii.invoke(self, "shareAttributeGroup", [share_options]))

    @builtins.property
    @jsii.member(jsii_name="attributeGroupArn")
    def attribute_group_arn(self) -> builtins.str:
        '''(experimental) The ARN of the attribute group.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "attributeGroupArn"))

    @builtins.property
    @jsii.member(jsii_name="attributeGroupId")
    def attribute_group_id(self) -> builtins.str:
        '''(experimental) The ID of the attribute group.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "attributeGroupId"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-servicecatalogappregistry-alpha.CreateTargetApplicationOptions",
    jsii_struct_bases=[TargetApplicationCommonOptions],
    name_mapping={
        "analytics_reporting": "analyticsReporting",
        "cross_region_references": "crossRegionReferences",
        "description": "description",
        "env": "env",
        "permissions_boundary": "permissionsBoundary",
        "stack_name": "stackName",
        "synthesizer": "synthesizer",
        "tags": "tags",
        "termination_protection": "terminationProtection",
        "stack_id": "stackId",
        "application_name": "applicationName",
        "application_description": "applicationDescription",
    },
)
class CreateTargetApplicationOptions(TargetApplicationCommonOptions):
    def __init__(
        self,
        *,
        analytics_reporting: typing.Optional[builtins.bool] = None,
        cross_region_references: typing.Optional[builtins.bool] = None,
        description: typing.Optional[builtins.str] = None,
        env: typing.Optional[typing.Union[_aws_cdk_ceddda9d.Environment, typing.Dict[builtins.str, typing.Any]]] = None,
        permissions_boundary: typing.Optional[_aws_cdk_ceddda9d.PermissionsBoundary] = None,
        stack_name: typing.Optional[builtins.str] = None,
        synthesizer: typing.Optional[_aws_cdk_ceddda9d.IStackSynthesizer] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        termination_protection: typing.Optional[builtins.bool] = None,
        stack_id: typing.Optional[builtins.str] = None,
        application_name: builtins.str,
        application_description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Properties used to define New TargetApplication.

        :param analytics_reporting: Include runtime versioning information in this Stack. Default: ``analyticsReporting`` setting of containing ``App``, or value of 'aws:cdk:version-reporting' context key
        :param cross_region_references: Enable this flag to allow native cross region stack references. Enabling this will create a CloudFormation custom resource in both the producing stack and consuming stack in order to perform the export/import This feature is currently experimental Default: false
        :param description: A description of the stack. Default: - No description.
        :param env: The AWS environment (account/region) where this stack will be deployed. Set the ``region``/``account`` fields of ``env`` to either a concrete value to select the indicated environment (recommended for production stacks), or to the values of environment variables ``CDK_DEFAULT_REGION``/``CDK_DEFAULT_ACCOUNT`` to let the target environment depend on the AWS credentials/configuration that the CDK CLI is executed under (recommended for development stacks). If the ``Stack`` is instantiated inside a ``Stage``, any undefined ``region``/``account`` fields from ``env`` will default to the same field on the encompassing ``Stage``, if configured there. If either ``region`` or ``account`` are not set nor inherited from ``Stage``, the Stack will be considered "*environment-agnostic*"". Environment-agnostic stacks can be deployed to any environment but may not be able to take advantage of all features of the CDK. For example, they will not be able to use environmental context lookups such as ``ec2.Vpc.fromLookup`` and will not automatically translate Service Principals to the right format based on the environment's AWS partition, and other such enhancements. Default: - The environment of the containing ``Stage`` if available, otherwise create the stack will be environment-agnostic.
        :param permissions_boundary: Options for applying a permissions boundary to all IAM Roles and Users created within this Stage. Default: - no permissions boundary is applied
        :param stack_name: Name to deploy the stack with. Default: - Derived from construct path.
        :param synthesizer: Synthesis method to use while deploying this stack. The Stack Synthesizer controls aspects of synthesis and deployment, like how assets are referenced and what IAM roles to use. For more information, see the README of the main CDK package. If not specified, the ``defaultStackSynthesizer`` from ``App`` will be used. If that is not specified, ``DefaultStackSynthesizer`` is used if ``@aws-cdk/core:newStyleStackSynthesis`` is set to ``true`` or the CDK major version is v2. In CDK v1 ``LegacyStackSynthesizer`` is the default if no other synthesizer is specified. Default: - The synthesizer specified on ``App``, or ``DefaultStackSynthesizer`` otherwise.
        :param tags: Stack tags that will be applied to all the taggable resources and the stack itself. Default: {}
        :param termination_protection: Whether to enable termination protection for this stack. Default: false
        :param stack_id: (experimental) Stack ID in which application will be created or imported. The id of a stack is also the identifier that you use to refer to it in the `AWS CDK Toolkit <https://docs.aws.amazon.com/cdk/v2/guide/cli.html>`_. Default: - ApplicationAssociatorStack
        :param application_name: (experimental) Enforces a particular physical application name.
        :param application_description: (experimental) Application description. Default: - Application containing stacks deployed via CDK.

        :stability: experimental
        :exampleMetadata: infused

        Example::

            app = App()
            associated_app = appreg.ApplicationAssociator(app, "AssociatedApplication",
                applications=[appreg.TargetApplication.create_application_stack(
                    application_name="MyAssociatedApplication",
                    # 'Application containing stacks deployed via CDK.' is the default
                    application_description="Associated Application description",
                    stack_name="MyAssociatedApplicationStack",
                    # AWS Account and Region that are implied by the current CLI configuration is the default
                    env=Environment(account="123456789012", region="us-east-1")
                )]
            )
        '''
        if isinstance(env, dict):
            env = _aws_cdk_ceddda9d.Environment(**env)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7ed0057efe5c9760fa6e858501ef3eab5baeb0561c53a3dd7c2e611aed0aa51d)
            check_type(argname="argument analytics_reporting", value=analytics_reporting, expected_type=type_hints["analytics_reporting"])
            check_type(argname="argument cross_region_references", value=cross_region_references, expected_type=type_hints["cross_region_references"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument env", value=env, expected_type=type_hints["env"])
            check_type(argname="argument permissions_boundary", value=permissions_boundary, expected_type=type_hints["permissions_boundary"])
            check_type(argname="argument stack_name", value=stack_name, expected_type=type_hints["stack_name"])
            check_type(argname="argument synthesizer", value=synthesizer, expected_type=type_hints["synthesizer"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument termination_protection", value=termination_protection, expected_type=type_hints["termination_protection"])
            check_type(argname="argument stack_id", value=stack_id, expected_type=type_hints["stack_id"])
            check_type(argname="argument application_name", value=application_name, expected_type=type_hints["application_name"])
            check_type(argname="argument application_description", value=application_description, expected_type=type_hints["application_description"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "application_name": application_name,
        }
        if analytics_reporting is not None:
            self._values["analytics_reporting"] = analytics_reporting
        if cross_region_references is not None:
            self._values["cross_region_references"] = cross_region_references
        if description is not None:
            self._values["description"] = description
        if env is not None:
            self._values["env"] = env
        if permissions_boundary is not None:
            self._values["permissions_boundary"] = permissions_boundary
        if stack_name is not None:
            self._values["stack_name"] = stack_name
        if synthesizer is not None:
            self._values["synthesizer"] = synthesizer
        if tags is not None:
            self._values["tags"] = tags
        if termination_protection is not None:
            self._values["termination_protection"] = termination_protection
        if stack_id is not None:
            self._values["stack_id"] = stack_id
        if application_description is not None:
            self._values["application_description"] = application_description

    @builtins.property
    def analytics_reporting(self) -> typing.Optional[builtins.bool]:
        '''Include runtime versioning information in this Stack.

        :default:

        ``analyticsReporting`` setting of containing ``App``, or value of
        'aws:cdk:version-reporting' context key
        '''
        result = self._values.get("analytics_reporting")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def cross_region_references(self) -> typing.Optional[builtins.bool]:
        '''Enable this flag to allow native cross region stack references.

        Enabling this will create a CloudFormation custom resource
        in both the producing stack and consuming stack in order to perform the export/import

        This feature is currently experimental

        :default: false
        '''
        result = self._values.get("cross_region_references")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the stack.

        :default: - No description.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def env(self) -> typing.Optional[_aws_cdk_ceddda9d.Environment]:
        '''The AWS environment (account/region) where this stack will be deployed.

        Set the ``region``/``account`` fields of ``env`` to either a concrete value to
        select the indicated environment (recommended for production stacks), or to
        the values of environment variables
        ``CDK_DEFAULT_REGION``/``CDK_DEFAULT_ACCOUNT`` to let the target environment
        depend on the AWS credentials/configuration that the CDK CLI is executed
        under (recommended for development stacks).

        If the ``Stack`` is instantiated inside a ``Stage``, any undefined
        ``region``/``account`` fields from ``env`` will default to the same field on the
        encompassing ``Stage``, if configured there.

        If either ``region`` or ``account`` are not set nor inherited from ``Stage``, the
        Stack will be considered "*environment-agnostic*"". Environment-agnostic
        stacks can be deployed to any environment but may not be able to take
        advantage of all features of the CDK. For example, they will not be able to
        use environmental context lookups such as ``ec2.Vpc.fromLookup`` and will not
        automatically translate Service Principals to the right format based on the
        environment's AWS partition, and other such enhancements.

        :default:

        - The environment of the containing ``Stage`` if available,
        otherwise create the stack will be environment-agnostic.

        Example::

            # Example automatically generated from non-compiling source. May contain errors.
            # Use a concrete account and region to deploy this stack to:
            # `.account` and `.region` will simply return these values.
            Stack(app, "Stack1",
                env=Environment(
                    account="123456789012",
                    region="us-east-1"
                )
            )
            
            # Use the CLI's current credentials to determine the target environment:
            # `.account` and `.region` will reflect the account+region the CLI
            # is configured to use (based on the user CLI credentials)
            Stack(app, "Stack2",
                env=Environment(
                    account=process.env.CDK_DEFAULT_ACCOUNT,
                    region=process.env.CDK_DEFAULT_REGION
                )
            )
            
            # Define multiple stacks stage associated with an environment
            my_stage = Stage(app, "MyStage",
                env=Environment(
                    account="123456789012",
                    region="us-east-1"
                )
            )
            
            # both of these stacks will use the stage's account/region:
            # `.account` and `.region` will resolve to the concrete values as above
            MyStack(my_stage, "Stack1")
            YourStack(my_stage, "Stack2")
            
            # Define an environment-agnostic stack:
            # `.account` and `.region` will resolve to `{ "Ref": "AWS::AccountId" }` and `{ "Ref": "AWS::Region" }` respectively.
            # which will only resolve to actual values by CloudFormation during deployment.
            MyStack(app, "Stack1")
        '''
        result = self._values.get("env")
        return typing.cast(typing.Optional[_aws_cdk_ceddda9d.Environment], result)

    @builtins.property
    def permissions_boundary(
        self,
    ) -> typing.Optional[_aws_cdk_ceddda9d.PermissionsBoundary]:
        '''Options for applying a permissions boundary to all IAM Roles and Users created within this Stage.

        :default: - no permissions boundary is applied
        '''
        result = self._values.get("permissions_boundary")
        return typing.cast(typing.Optional[_aws_cdk_ceddda9d.PermissionsBoundary], result)

    @builtins.property
    def stack_name(self) -> typing.Optional[builtins.str]:
        '''Name to deploy the stack with.

        :default: - Derived from construct path.
        '''
        result = self._values.get("stack_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def synthesizer(self) -> typing.Optional[_aws_cdk_ceddda9d.IStackSynthesizer]:
        '''Synthesis method to use while deploying this stack.

        The Stack Synthesizer controls aspects of synthesis and deployment,
        like how assets are referenced and what IAM roles to use. For more
        information, see the README of the main CDK package.

        If not specified, the ``defaultStackSynthesizer`` from ``App`` will be used.
        If that is not specified, ``DefaultStackSynthesizer`` is used if
        ``@aws-cdk/core:newStyleStackSynthesis`` is set to ``true`` or the CDK major
        version is v2. In CDK v1 ``LegacyStackSynthesizer`` is the default if no
        other synthesizer is specified.

        :default: - The synthesizer specified on ``App``, or ``DefaultStackSynthesizer`` otherwise.
        '''
        result = self._values.get("synthesizer")
        return typing.cast(typing.Optional[_aws_cdk_ceddda9d.IStackSynthesizer], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Stack tags that will be applied to all the taggable resources and the stack itself.

        :default: {}
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def termination_protection(self) -> typing.Optional[builtins.bool]:
        '''Whether to enable termination protection for this stack.

        :default: false
        '''
        result = self._values.get("termination_protection")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def stack_id(self) -> typing.Optional[builtins.str]:
        '''(experimental) Stack ID in which application will be created or imported.

        The id of a stack is also the identifier that you use to
        refer to it in the `AWS CDK Toolkit <https://docs.aws.amazon.com/cdk/v2/guide/cli.html>`_.

        :default: - ApplicationAssociatorStack

        :stability: experimental
        '''
        result = self._values.get("stack_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def application_name(self) -> builtins.str:
        '''(experimental) Enforces a particular physical application name.

        :stability: experimental
        '''
        result = self._values.get("application_name")
        assert result is not None, "Required property 'application_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def application_description(self) -> typing.Optional[builtins.str]:
        '''(experimental) Application description.

        :default: - Application containing stacks deployed via CDK.

        :stability: experimental
        '''
        result = self._values.get("application_description")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CreateTargetApplicationOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-servicecatalogappregistry-alpha.ExistingTargetApplicationOptions",
    jsii_struct_bases=[TargetApplicationCommonOptions],
    name_mapping={
        "analytics_reporting": "analyticsReporting",
        "cross_region_references": "crossRegionReferences",
        "description": "description",
        "env": "env",
        "permissions_boundary": "permissionsBoundary",
        "stack_name": "stackName",
        "synthesizer": "synthesizer",
        "tags": "tags",
        "termination_protection": "terminationProtection",
        "stack_id": "stackId",
        "application_arn_value": "applicationArnValue",
    },
)
class ExistingTargetApplicationOptions(TargetApplicationCommonOptions):
    def __init__(
        self,
        *,
        analytics_reporting: typing.Optional[builtins.bool] = None,
        cross_region_references: typing.Optional[builtins.bool] = None,
        description: typing.Optional[builtins.str] = None,
        env: typing.Optional[typing.Union[_aws_cdk_ceddda9d.Environment, typing.Dict[builtins.str, typing.Any]]] = None,
        permissions_boundary: typing.Optional[_aws_cdk_ceddda9d.PermissionsBoundary] = None,
        stack_name: typing.Optional[builtins.str] = None,
        synthesizer: typing.Optional[_aws_cdk_ceddda9d.IStackSynthesizer] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        termination_protection: typing.Optional[builtins.bool] = None,
        stack_id: typing.Optional[builtins.str] = None,
        application_arn_value: builtins.str,
    ) -> None:
        '''(experimental) Properties used to define Existing TargetApplication.

        :param analytics_reporting: Include runtime versioning information in this Stack. Default: ``analyticsReporting`` setting of containing ``App``, or value of 'aws:cdk:version-reporting' context key
        :param cross_region_references: Enable this flag to allow native cross region stack references. Enabling this will create a CloudFormation custom resource in both the producing stack and consuming stack in order to perform the export/import This feature is currently experimental Default: false
        :param description: A description of the stack. Default: - No description.
        :param env: The AWS environment (account/region) where this stack will be deployed. Set the ``region``/``account`` fields of ``env`` to either a concrete value to select the indicated environment (recommended for production stacks), or to the values of environment variables ``CDK_DEFAULT_REGION``/``CDK_DEFAULT_ACCOUNT`` to let the target environment depend on the AWS credentials/configuration that the CDK CLI is executed under (recommended for development stacks). If the ``Stack`` is instantiated inside a ``Stage``, any undefined ``region``/``account`` fields from ``env`` will default to the same field on the encompassing ``Stage``, if configured there. If either ``region`` or ``account`` are not set nor inherited from ``Stage``, the Stack will be considered "*environment-agnostic*"". Environment-agnostic stacks can be deployed to any environment but may not be able to take advantage of all features of the CDK. For example, they will not be able to use environmental context lookups such as ``ec2.Vpc.fromLookup`` and will not automatically translate Service Principals to the right format based on the environment's AWS partition, and other such enhancements. Default: - The environment of the containing ``Stage`` if available, otherwise create the stack will be environment-agnostic.
        :param permissions_boundary: Options for applying a permissions boundary to all IAM Roles and Users created within this Stage. Default: - no permissions boundary is applied
        :param stack_name: Name to deploy the stack with. Default: - Derived from construct path.
        :param synthesizer: Synthesis method to use while deploying this stack. The Stack Synthesizer controls aspects of synthesis and deployment, like how assets are referenced and what IAM roles to use. For more information, see the README of the main CDK package. If not specified, the ``defaultStackSynthesizer`` from ``App`` will be used. If that is not specified, ``DefaultStackSynthesizer`` is used if ``@aws-cdk/core:newStyleStackSynthesis`` is set to ``true`` or the CDK major version is v2. In CDK v1 ``LegacyStackSynthesizer`` is the default if no other synthesizer is specified. Default: - The synthesizer specified on ``App``, or ``DefaultStackSynthesizer`` otherwise.
        :param tags: Stack tags that will be applied to all the taggable resources and the stack itself. Default: {}
        :param termination_protection: Whether to enable termination protection for this stack. Default: false
        :param stack_id: (experimental) Stack ID in which application will be created or imported. The id of a stack is also the identifier that you use to refer to it in the `AWS CDK Toolkit <https://docs.aws.amazon.com/cdk/v2/guide/cli.html>`_. Default: - ApplicationAssociatorStack
        :param application_arn_value: (experimental) Enforces a particular application arn.

        :stability: experimental
        :exampleMetadata: infused

        Example::

            app = App()
            associated_app = appreg.ApplicationAssociator(app, "AssociatedApplication",
                applications=[appreg.TargetApplication.existing_application_from_arn(
                    application_arn_value="arn:aws:servicecatalog:us-east-1:123456789012:/applications/applicationId",
                    stack_name="MyAssociatedApplicationStack"
                )]
            )
        '''
        if isinstance(env, dict):
            env = _aws_cdk_ceddda9d.Environment(**env)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__346e88991c2535a9b273e7054098c865f76770acd250aee2bc0348d5774c686e)
            check_type(argname="argument analytics_reporting", value=analytics_reporting, expected_type=type_hints["analytics_reporting"])
            check_type(argname="argument cross_region_references", value=cross_region_references, expected_type=type_hints["cross_region_references"])
            check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            check_type(argname="argument env", value=env, expected_type=type_hints["env"])
            check_type(argname="argument permissions_boundary", value=permissions_boundary, expected_type=type_hints["permissions_boundary"])
            check_type(argname="argument stack_name", value=stack_name, expected_type=type_hints["stack_name"])
            check_type(argname="argument synthesizer", value=synthesizer, expected_type=type_hints["synthesizer"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
            check_type(argname="argument termination_protection", value=termination_protection, expected_type=type_hints["termination_protection"])
            check_type(argname="argument stack_id", value=stack_id, expected_type=type_hints["stack_id"])
            check_type(argname="argument application_arn_value", value=application_arn_value, expected_type=type_hints["application_arn_value"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "application_arn_value": application_arn_value,
        }
        if analytics_reporting is not None:
            self._values["analytics_reporting"] = analytics_reporting
        if cross_region_references is not None:
            self._values["cross_region_references"] = cross_region_references
        if description is not None:
            self._values["description"] = description
        if env is not None:
            self._values["env"] = env
        if permissions_boundary is not None:
            self._values["permissions_boundary"] = permissions_boundary
        if stack_name is not None:
            self._values["stack_name"] = stack_name
        if synthesizer is not None:
            self._values["synthesizer"] = synthesizer
        if tags is not None:
            self._values["tags"] = tags
        if termination_protection is not None:
            self._values["termination_protection"] = termination_protection
        if stack_id is not None:
            self._values["stack_id"] = stack_id

    @builtins.property
    def analytics_reporting(self) -> typing.Optional[builtins.bool]:
        '''Include runtime versioning information in this Stack.

        :default:

        ``analyticsReporting`` setting of containing ``App``, or value of
        'aws:cdk:version-reporting' context key
        '''
        result = self._values.get("analytics_reporting")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def cross_region_references(self) -> typing.Optional[builtins.bool]:
        '''Enable this flag to allow native cross region stack references.

        Enabling this will create a CloudFormation custom resource
        in both the producing stack and consuming stack in order to perform the export/import

        This feature is currently experimental

        :default: false
        '''
        result = self._values.get("cross_region_references")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''A description of the stack.

        :default: - No description.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def env(self) -> typing.Optional[_aws_cdk_ceddda9d.Environment]:
        '''The AWS environment (account/region) where this stack will be deployed.

        Set the ``region``/``account`` fields of ``env`` to either a concrete value to
        select the indicated environment (recommended for production stacks), or to
        the values of environment variables
        ``CDK_DEFAULT_REGION``/``CDK_DEFAULT_ACCOUNT`` to let the target environment
        depend on the AWS credentials/configuration that the CDK CLI is executed
        under (recommended for development stacks).

        If the ``Stack`` is instantiated inside a ``Stage``, any undefined
        ``region``/``account`` fields from ``env`` will default to the same field on the
        encompassing ``Stage``, if configured there.

        If either ``region`` or ``account`` are not set nor inherited from ``Stage``, the
        Stack will be considered "*environment-agnostic*"". Environment-agnostic
        stacks can be deployed to any environment but may not be able to take
        advantage of all features of the CDK. For example, they will not be able to
        use environmental context lookups such as ``ec2.Vpc.fromLookup`` and will not
        automatically translate Service Principals to the right format based on the
        environment's AWS partition, and other such enhancements.

        :default:

        - The environment of the containing ``Stage`` if available,
        otherwise create the stack will be environment-agnostic.

        Example::

            # Example automatically generated from non-compiling source. May contain errors.
            # Use a concrete account and region to deploy this stack to:
            # `.account` and `.region` will simply return these values.
            Stack(app, "Stack1",
                env=Environment(
                    account="123456789012",
                    region="us-east-1"
                )
            )
            
            # Use the CLI's current credentials to determine the target environment:
            # `.account` and `.region` will reflect the account+region the CLI
            # is configured to use (based on the user CLI credentials)
            Stack(app, "Stack2",
                env=Environment(
                    account=process.env.CDK_DEFAULT_ACCOUNT,
                    region=process.env.CDK_DEFAULT_REGION
                )
            )
            
            # Define multiple stacks stage associated with an environment
            my_stage = Stage(app, "MyStage",
                env=Environment(
                    account="123456789012",
                    region="us-east-1"
                )
            )
            
            # both of these stacks will use the stage's account/region:
            # `.account` and `.region` will resolve to the concrete values as above
            MyStack(my_stage, "Stack1")
            YourStack(my_stage, "Stack2")
            
            # Define an environment-agnostic stack:
            # `.account` and `.region` will resolve to `{ "Ref": "AWS::AccountId" }` and `{ "Ref": "AWS::Region" }` respectively.
            # which will only resolve to actual values by CloudFormation during deployment.
            MyStack(app, "Stack1")
        '''
        result = self._values.get("env")
        return typing.cast(typing.Optional[_aws_cdk_ceddda9d.Environment], result)

    @builtins.property
    def permissions_boundary(
        self,
    ) -> typing.Optional[_aws_cdk_ceddda9d.PermissionsBoundary]:
        '''Options for applying a permissions boundary to all IAM Roles and Users created within this Stage.

        :default: - no permissions boundary is applied
        '''
        result = self._values.get("permissions_boundary")
        return typing.cast(typing.Optional[_aws_cdk_ceddda9d.PermissionsBoundary], result)

    @builtins.property
    def stack_name(self) -> typing.Optional[builtins.str]:
        '''Name to deploy the stack with.

        :default: - Derived from construct path.
        '''
        result = self._values.get("stack_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def synthesizer(self) -> typing.Optional[_aws_cdk_ceddda9d.IStackSynthesizer]:
        '''Synthesis method to use while deploying this stack.

        The Stack Synthesizer controls aspects of synthesis and deployment,
        like how assets are referenced and what IAM roles to use. For more
        information, see the README of the main CDK package.

        If not specified, the ``defaultStackSynthesizer`` from ``App`` will be used.
        If that is not specified, ``DefaultStackSynthesizer`` is used if
        ``@aws-cdk/core:newStyleStackSynthesis`` is set to ``true`` or the CDK major
        version is v2. In CDK v1 ``LegacyStackSynthesizer`` is the default if no
        other synthesizer is specified.

        :default: - The synthesizer specified on ``App``, or ``DefaultStackSynthesizer`` otherwise.
        '''
        result = self._values.get("synthesizer")
        return typing.cast(typing.Optional[_aws_cdk_ceddda9d.IStackSynthesizer], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Stack tags that will be applied to all the taggable resources and the stack itself.

        :default: {}
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def termination_protection(self) -> typing.Optional[builtins.bool]:
        '''Whether to enable termination protection for this stack.

        :default: false
        '''
        result = self._values.get("termination_protection")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def stack_id(self) -> typing.Optional[builtins.str]:
        '''(experimental) Stack ID in which application will be created or imported.

        The id of a stack is also the identifier that you use to
        refer to it in the `AWS CDK Toolkit <https://docs.aws.amazon.com/cdk/v2/guide/cli.html>`_.

        :default: - ApplicationAssociatorStack

        :stability: experimental
        '''
        result = self._values.get("stack_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def application_arn_value(self) -> builtins.str:
        '''(experimental) Enforces a particular application arn.

        :stability: experimental
        '''
        result = self._values.get("application_arn_value")
        assert result is not None, "Required property 'application_arn_value' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ExistingTargetApplicationOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


__all__ = [
    "Application",
    "ApplicationAssociator",
    "ApplicationAssociatorProps",
    "ApplicationProps",
    "AttributeGroup",
    "AttributeGroupProps",
    "BindTargetApplicationResult",
    "CreateTargetApplicationOptions",
    "ExistingTargetApplicationOptions",
    "IApplication",
    "IAttributeGroup",
    "ShareOptions",
    "SharePermission",
    "TargetApplication",
    "TargetApplicationCommonOptions",
]

publication.publish()

def _typecheckingstub__fdf70fe013f8883665f67795a3e892e124b1c97eee77c8691a45e94a03fd3b3f(
    scope: _aws_cdk_ceddda9d.App,
    id: builtins.str,
    *,
    applications: typing.Sequence[TargetApplication],
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__2c562e37ce20cae0e25f66edcda715d4ced81af5946d6243931500bb560b566c(
    stage: _aws_cdk_ceddda9d.Stage,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__e907bed5984ca51a19f28a830c059d5147fb84b71013dd72756e86f116955fef(
    stage: _aws_cdk_ceddda9d.Stage,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__30616d49bde91e4252f2107159acc6af6c7e69963df724abfd3d0e3887b7d07c(
    *,
    applications: typing.Sequence[TargetApplication],
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__6be7bfb8b322f2a1baa56fe1681c794a77811f78b07db4c1d82e108ccfd5d114(
    *,
    application_name: builtins.str,
    description: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__3622c2b8fdb683c749e737f012cde1fbfc02d34fadd8639e26812b7ef1494b30(
    *,
    attribute_group_name: builtins.str,
    attributes: typing.Mapping[builtins.str, typing.Any],
    description: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__49f1fbb4ffc705c29064e5565593f9a0e87b46c7dbf27ddfa01428e7ccb10675(
    *,
    application: IApplication,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__eb25bb53fbac36a075a6c682dbd5e647d5dd5b0436b1b1e29a37205679f60c97(
    construct: _constructs_77d1e7e8.Construct,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__e92370f5ca96b4ac3a6918c6020df4c61044935affc14afe329d14f7b6cfc872(
    stack: _aws_cdk_ceddda9d.Stack,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__013b061a2eaf66f5f2c454cad6c31f5138ef6eacf9ce64ea456c3acc549750b7(
    attribute_group: IAttributeGroup,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__6355fcb9bf431963c17f1d72e211e52e508a3bb085de7597c2deed4b88eda46f(
    stack: _aws_cdk_ceddda9d.Stack,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__f20907f839e03d66b58888eece0b2a80abd0c0240c29a95fb103642e09cbfa2e(
    *,
    accounts: typing.Optional[typing.Sequence[builtins.str]] = None,
    organization_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
    roles: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.IRole]] = None,
    share_permission: typing.Optional[typing.Union[builtins.str, SharePermission]] = None,
    users: typing.Optional[typing.Sequence[_aws_cdk_aws_iam_ceddda9d.IUser]] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__676c526376ea5a57da1d842b32eeaba48f39014bebf2de107e52abfb0ef30471(
    scope: _constructs_77d1e7e8.Construct,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__8e9a8c85d5db3b4596f192c3fe206a4e30e4143fb84b5be58ef7f29131edc782(
    *,
    analytics_reporting: typing.Optional[builtins.bool] = None,
    cross_region_references: typing.Optional[builtins.bool] = None,
    description: typing.Optional[builtins.str] = None,
    env: typing.Optional[typing.Union[_aws_cdk_ceddda9d.Environment, typing.Dict[builtins.str, typing.Any]]] = None,
    permissions_boundary: typing.Optional[_aws_cdk_ceddda9d.PermissionsBoundary] = None,
    stack_name: typing.Optional[builtins.str] = None,
    synthesizer: typing.Optional[_aws_cdk_ceddda9d.IStackSynthesizer] = None,
    tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    termination_protection: typing.Optional[builtins.bool] = None,
    stack_id: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__d00f48aa19f1be0379562df9ddbae70ebc3518362d0411437cb95dfcdd2d2163(
    scope: _constructs_77d1e7e8.Construct,
    id: builtins.str,
    *,
    application_name: builtins.str,
    description: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__c9f55f3016c1743f9bf17bd3321531f86b5a2fcff42b5b44073089b900c1bc3e(
    scope: _constructs_77d1e7e8.Construct,
    id: builtins.str,
    application_arn: builtins.str,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__c009ad74ae52d87dcfb6e1aee6e9569d5807c9f2c9f0bc6691cc089dacece4a8(
    scope: _constructs_77d1e7e8.Construct,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__875f08936c5ec9d623a247709774d3625560a74cad5380509cb4624eecad2f27(
    stack: _aws_cdk_ceddda9d.Stack,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__6e09d5799ecd5be55b6bce6f34e72f3f0d7d0c179ac0824b4e961e49687e3871(
    attribute_group: IAttributeGroup,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__09e36e72eb27ec19021721817f2caa31bde7114fd6069eedbb069641910066e0(
    stack: _aws_cdk_ceddda9d.Stack,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__80266c77c192e878af5d982db7685162f1f646d9574aad4d8a432ef62520b1ab(
    resource_address: builtins.str,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__91ce42b356b8555b9df82de3713a41ca962e33062b181b37794c5e3b82d7b12f(
    scope: _constructs_77d1e7e8.Construct,
    id: builtins.str,
    *,
    attribute_group_name: builtins.str,
    attributes: typing.Mapping[builtins.str, typing.Any],
    description: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__d090f29cba3280a4ce331eeac9f6003b4c6b25cc65422ee58257d036241687aa(
    scope: _constructs_77d1e7e8.Construct,
    id: builtins.str,
    attribute_group_arn: builtins.str,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__7ed0057efe5c9760fa6e858501ef3eab5baeb0561c53a3dd7c2e611aed0aa51d(
    *,
    analytics_reporting: typing.Optional[builtins.bool] = None,
    cross_region_references: typing.Optional[builtins.bool] = None,
    description: typing.Optional[builtins.str] = None,
    env: typing.Optional[typing.Union[_aws_cdk_ceddda9d.Environment, typing.Dict[builtins.str, typing.Any]]] = None,
    permissions_boundary: typing.Optional[_aws_cdk_ceddda9d.PermissionsBoundary] = None,
    stack_name: typing.Optional[builtins.str] = None,
    synthesizer: typing.Optional[_aws_cdk_ceddda9d.IStackSynthesizer] = None,
    tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    termination_protection: typing.Optional[builtins.bool] = None,
    stack_id: typing.Optional[builtins.str] = None,
    application_name: builtins.str,
    application_description: typing.Optional[builtins.str] = None,
) -> None:
    """Type checking stubs"""
    pass

def _typecheckingstub__346e88991c2535a9b273e7054098c865f76770acd250aee2bc0348d5774c686e(
    *,
    analytics_reporting: typing.Optional[builtins.bool] = None,
    cross_region_references: typing.Optional[builtins.bool] = None,
    description: typing.Optional[builtins.str] = None,
    env: typing.Optional[typing.Union[_aws_cdk_ceddda9d.Environment, typing.Dict[builtins.str, typing.Any]]] = None,
    permissions_boundary: typing.Optional[_aws_cdk_ceddda9d.PermissionsBoundary] = None,
    stack_name: typing.Optional[builtins.str] = None,
    synthesizer: typing.Optional[_aws_cdk_ceddda9d.IStackSynthesizer] = None,
    tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    termination_protection: typing.Optional[builtins.bool] = None,
    stack_id: typing.Optional[builtins.str] = None,
    application_arn_value: builtins.str,
) -> None:
    """Type checking stubs"""
    pass
