"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const nodeunit_shim_1 = require("nodeunit-shim");
const artifact_1 = require("../lib/artifact");
const pipeline_1 = require("../lib/pipeline");
const validation_1 = require("../lib/validation");
const fake_source_action_1 = require("./fake-source-action");
nodeunit_shim_1.nodeunitShim({
    'name validation'(test) {
        const cases = [
            { name: 'BlahBleep123.@-_', shouldPassValidation: true, explanation: 'should be valid' },
            { name: '', shouldPassValidation: false, explanation: 'the empty string should be invalid' },
            { name: ' BlahBleep', shouldPassValidation: false, explanation: 'spaces should be invalid' },
            { name: '!BlahBleep', shouldPassValidation: false, explanation: '\'!\' should be invalid' },
        ];
        cases.forEach(testCase => {
            const name = testCase.name;
            const validationBlock = () => { validation_1.validateName('test thing', name); };
            if (testCase.shouldPassValidation) {
                test.doesNotThrow(validationBlock, Error, `${name} failed validation but ${testCase.explanation}`);
            }
            else {
                test.throws(validationBlock, Error, `${name} passed validation but ${testCase.explanation}`);
            }
        });
        test.done();
    },
    'Stage validation': {
        'should fail if Stage has no Actions'(test) {
            const stage = stageForTesting();
            test.deepEqual(stage.validate().length, 1);
            test.done();
        },
    },
    'Pipeline validation': {
        'should fail if Pipeline has no Stages'(test) {
            const stack = new cdk.Stack();
            const pipeline = new pipeline_1.Pipeline(stack, 'Pipeline');
            test.deepEqual(cdk.ConstructNode.validate(pipeline.node).length, 1);
            test.done();
        },
        'should fail if Pipeline has a Source Action in a non-first Stage'(test) {
            const stack = new cdk.Stack();
            const pipeline = new pipeline_1.Pipeline(stack, 'Pipeline');
            pipeline.addStage({
                stageName: 'FirstStage',
                actions: [
                    new fake_source_action_1.FakeSourceAction({
                        actionName: 'FakeSource',
                        output: new artifact_1.Artifact(),
                    }),
                ],
            });
            test.deepEqual(cdk.ConstructNode.validate(pipeline.node).length, 1);
            test.done();
        },
    },
});
function stageForTesting() {
    const stack = new cdk.Stack();
    const pipeline = new pipeline_1.Pipeline(stack, 'Pipeline');
    return pipeline.addStage({ stageName: 'stage' });
}
//# sourceMappingURL=data:application/json;base64,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