"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const nodeunit_shim_1 = require("nodeunit-shim");
const codepipeline = require("../lib");
const fake_build_action_1 = require("./fake-build-action");
const fake_source_action_1 = require("./fake-source-action");
/* eslint-disable quote-props */
nodeunit_shim_1.nodeunitShim({
    'Artifacts in CodePipeline': {
        'cannot be created with an empty name'(test) {
            test.throws(() => new codepipeline.Artifact(''), /Artifact name must match regular expression/);
            test.done();
        },
        'without a name, when used as an input without being used as an output first - should fail validation'(test) {
            const stack = new cdk.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'Source',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'Build',
                                input: new codepipeline.Artifact(),
                            }),
                        ],
                    },
                ],
            });
            const errors = validate(stack);
            test.equal(errors.length, 1);
            const error = errors[0];
            test.same(error.source, pipeline);
            test.equal(error.message, "Action 'Build' is using an unnamed input Artifact, which is not being produced in this pipeline");
            test.done();
        },
        'with a name, when used as an input without being used as an output first - should fail validation'(test) {
            const stack = new cdk.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'Source',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'Build',
                                input: new codepipeline.Artifact('named'),
                            }),
                        ],
                    },
                ],
            });
            const errors = validate(stack);
            test.equal(errors.length, 1);
            const error = errors[0];
            test.same(error.source, pipeline);
            test.equal(error.message, "Action 'Build' is using input Artifact 'named', which is not being produced in this pipeline");
            test.done();
        },
        'without a name, when used as an output multiple times - should fail validation'(test) {
            const stack = new cdk.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'Source',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'Build',
                                input: sourceOutput,
                                output: sourceOutput,
                            }),
                        ],
                    },
                ],
            });
            const errors = validate(stack);
            test.equal(errors.length, 1);
            const error = errors[0];
            test.same(error.source, pipeline);
            test.equal(error.message, "Both Actions 'Source' and 'Build' are producting Artifact 'Artifact_Source_Source'. Every artifact can only be produced once.");
            test.done();
        },
        "an Action's output can be used as input for an Action in the same Stage with a higher runOrder"(test) {
            const stack = new cdk.Stack();
            const sourceOutput1 = new codepipeline.Artifact('sourceOutput1');
            const buildOutput1 = new codepipeline.Artifact('buildOutput1');
            const sourceOutput2 = new codepipeline.Artifact('sourceOutput2');
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'source1',
                                output: sourceOutput1,
                            }),
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'source2',
                                output: sourceOutput2,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'build1',
                                input: sourceOutput1,
                                output: buildOutput1,
                            }),
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'build2',
                                input: sourceOutput2,
                                extraInputs: [buildOutput1],
                                output: new codepipeline.Artifact('buildOutput2'),
                                runOrder: 2,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            //
            }));
            test.done();
        },
        'violation of runOrder constraints is detected and reported'(test) {
            const stack = new cdk.Stack();
            const sourceOutput1 = new codepipeline.Artifact('sourceOutput1');
            const buildOutput1 = new codepipeline.Artifact('buildOutput1');
            const sourceOutput2 = new codepipeline.Artifact('sourceOutput2');
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'source1',
                                output: sourceOutput1,
                            }),
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'source2',
                                output: sourceOutput2,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'build1',
                                input: sourceOutput1,
                                output: buildOutput1,
                                runOrder: 3,
                            }),
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'build2',
                                input: sourceOutput2,
                                extraInputs: [buildOutput1],
                                output: new codepipeline.Artifact('buildOutput2'),
                                runOrder: 2,
                            }),
                        ],
                    },
                ],
            });
            const errors = validate(stack);
            test.equal(errors.length, 1);
            const error = errors[0];
            test.same(error.source, pipeline);
            test.equal(error.message, "Stage 2 Action 2 ('Build'/'build2') is consuming input Artifact 'buildOutput1' before it is being produced at Stage 2 Action 3 ('Build'/'build1')");
            test.done();
        },
        'without a name, sanitize the auto stage-action derived name'(test) {
            const stack = new cdk.Stack();
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source.@',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'source1',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'build1',
                                input: sourceOutput,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source.@',
                        'Actions': [
                            {
                                'Name': 'source1',
                                'OutputArtifacts': [
                                    { 'Name': 'Artifact_Source_source1' },
                                ],
                            },
                        ],
                    },
                    {
                        'Name': 'Build',
                        'Actions': [
                            {
                                'Name': 'build1',
                                'InputArtifacts': [
                                    { 'Name': 'Artifact_Source_source1' },
                                ],
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
    },
});
function validate(construct) {
    cdk.ConstructNode.prepare(construct.node);
    return cdk.ConstructNode.validate(construct.node);
}
//# sourceMappingURL=data:application/json;base64,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