"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = void 0;
const events = require("@aws-cdk/aws-events");
const cdk = require("@aws-cdk/core");
const validation = require("./validation");
/**
 * A Stage in a Pipeline.
 *
 * Stages are added to a Pipeline by calling {@link Pipeline#addStage},
 * which returns an instance of {@link codepipeline.IStage}.
 *
 * This class is private to the CodePipeline module.
 */
class Stage {
    /**
     * Create a new Stage.
     */
    constructor(props, pipeline) {
        this._actions = new Array();
        validation.validateName('Stage', props.stageName);
        this.stageName = props.stageName;
        this._pipeline = pipeline;
        this.scope = new cdk.Construct(pipeline, this.stageName);
        for (const action of props.actions || []) {
            this.addAction(action);
        }
    }
    /**
     * Get a duplicate of this stage's list of actions.
     */
    get actionDescriptors() {
        return this._actions.slice();
    }
    get actions() {
        return this._actions.map(actionDescriptor => actionDescriptor.action);
    }
    get pipeline() {
        return this._pipeline;
    }
    render() {
        // first, assign names to output Artifacts who don't have one
        for (const action of this._actions) {
            const outputArtifacts = action.outputs;
            const unnamedOutputs = outputArtifacts.filter(o => !o.artifactName);
            for (const outputArtifact of outputArtifacts) {
                if (!outputArtifact.artifactName) {
                    const unsanitizedArtifactName = `Artifact_${this.stageName}_${action.actionName}` + (unnamedOutputs.length === 1
                        ? ''
                        : '_' + (unnamedOutputs.indexOf(outputArtifact) + 1));
                    const artifactName = sanitizeArtifactName(unsanitizedArtifactName);
                    outputArtifact._setName(artifactName);
                }
            }
        }
        return {
            name: this.stageName,
            actions: this._actions.map(action => this.renderAction(action)),
        };
    }
    addAction(action) {
        const actionName = action.actionProperties.actionName;
        // validate the name
        validation.validateName('Action', actionName);
        // check for duplicate Actions and names
        if (this._actions.find(a => a.actionName === actionName)) {
            throw new Error(`Stage ${this.stageName} already contains an action with name '${actionName}'`);
        }
        this._actions.push(this.attachActionToPipeline(action));
    }
    onStateChange(name, target, options) {
        const rule = new events.Rule(this.scope, name, options);
        rule.addTarget(target);
        rule.addEventPattern({
            detailType: ['CodePipeline Stage Execution State Change'],
            source: ['aws.codepipeline'],
            resources: [this.pipeline.pipelineArn],
            detail: {
                stage: [this.stageName],
            },
        });
        return rule;
    }
    validate() {
        return [
            ...this.validateHasActions(),
            ...this.validateActions(),
        ];
    }
    validateHasActions() {
        if (this._actions.length === 0) {
            return [`Stage '${this.stageName}' must have at least one action`];
        }
        return [];
    }
    validateActions() {
        const ret = new Array();
        for (const action of this.actionDescriptors) {
            ret.push(...this.validateAction(action));
        }
        return ret;
    }
    validateAction(action) {
        return validation.validateArtifactBounds('input', action.inputs, action.artifactBounds.minInputs, action.artifactBounds.maxInputs, action.category, action.provider)
            .concat(validation.validateArtifactBounds('output', action.outputs, action.artifactBounds.minOutputs, action.artifactBounds.maxOutputs, action.category, action.provider));
    }
    attachActionToPipeline(action) {
        // notify the Pipeline of the new Action
        const actionScope = new cdk.Construct(this.scope, action.actionProperties.actionName);
        return this._pipeline._attachActionToPipeline(this, action, actionScope);
    }
    renderAction(action) {
        const outputArtifacts = cdk.Lazy.anyValue({ produce: () => this.renderArtifacts(action.outputs) }, { omitEmptyArray: true });
        const inputArtifacts = cdk.Lazy.anyValue({ produce: () => this.renderArtifacts(action.inputs) }, { omitEmptyArray: true });
        return {
            name: action.actionName,
            inputArtifacts,
            outputArtifacts,
            actionTypeId: {
                category: action.category.toString(),
                version: action.version,
                owner: action.owner,
                provider: action.provider,
            },
            configuration: action.configuration,
            runOrder: action.runOrder,
            roleArn: action.role ? action.role.roleArn : undefined,
            region: action.region,
            namespace: action.namespace,
        };
    }
    renderArtifacts(artifacts) {
        return artifacts
            .filter(a => a.artifactName)
            .map(a => ({ name: a.artifactName }));
    }
}
exports.Stage = Stage;
function sanitizeArtifactName(artifactName) {
    // strip out some characters that are legal in Stage and Action names,
    // but not in Artifact names
    return artifactName.replace(/[@.]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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