"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SelfManagedKafkaEventSource = exports.ManagedKafkaEventSource = exports.AuthenticationMethod = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const stream_1 = require("./stream");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("@aws-cdk/core");
/**
 * The authentication method to use with SelfManagedKafkaEventSource.
 *
 * @stability stable
 */
var AuthenticationMethod;
(function (AuthenticationMethod) {
    AuthenticationMethod["SASL_SCRAM_512_AUTH"] = "SASL_SCRAM_512_AUTH";
    AuthenticationMethod["SASL_SCRAM_256_AUTH"] = "SASL_SCRAM_256_AUTH";
})(AuthenticationMethod = exports.AuthenticationMethod || (exports.AuthenticationMethod = {}));
/**
 * Use a MSK cluster as a streaming source for AWS Lambda.
 *
 * @stability stable
 */
class ManagedKafkaEventSource extends stream_1.StreamEventSource {
    /**
     * @stability stable
     */
    constructor(props) {
        super(props);
        this.innerProps = props;
    }
    /**
     * Called by `lambda.addEventSource` to allow the event source to bind to this function.
     *
     * @stability stable
     */
    bind(target) {
        var _c;
        target.addEventSourceMapping(`KafkaEventSource:${this.innerProps.clusterArn}${this.innerProps.topic}`, this.enrichMappingOptions({
            eventSourceArn: this.innerProps.clusterArn,
            startingPosition: this.innerProps.startingPosition,
            sourceAccessConfigurations: this.sourceAccessConfigurations(),
            kafkaTopic: this.innerProps.topic,
        }));
        if (this.innerProps.secret !== undefined) {
            this.innerProps.secret.grantRead(target);
        }
        target.addToRolePolicy(new iam.PolicyStatement({
            actions: ['kafka:DescribeCluster', 'kafka:GetBootstrapBrokers', 'kafka:ListScramSecrets'],
            resources: [this.innerProps.clusterArn],
        }));
        (_c = target.role) === null || _c === void 0 ? void 0 : _c.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaMSKExecutionRole'));
    }
    sourceAccessConfigurations() {
        const sourceAccessConfigurations = [];
        if (this.innerProps.secret !== undefined) {
            // "Amazon MSK only supports SCRAM-SHA-512 authentication." from https://docs.aws.amazon.com/msk/latest/developerguide/msk-password.html#msk-password-limitations
            sourceAccessConfigurations.push({
                type: lambda.SourceAccessConfigurationType.SASL_SCRAM_512_AUTH,
                uri: this.innerProps.secret.secretArn,
            });
        }
        return sourceAccessConfigurations.length === 0
            ? undefined
            : sourceAccessConfigurations;
    }
}
exports.ManagedKafkaEventSource = ManagedKafkaEventSource;
_a = JSII_RTTI_SYMBOL_1;
ManagedKafkaEventSource[_a] = { fqn: "@aws-cdk/aws-lambda-event-sources.ManagedKafkaEventSource", version: "1.119.0" };
/**
 * Use a self hosted Kafka installation as a streaming source for AWS Lambda.
 *
 * @stability stable
 */
class SelfManagedKafkaEventSource extends stream_1.StreamEventSource {
    /**
     * @stability stable
     */
    constructor(props) {
        super(props);
        if (props.vpc) {
            if (!props.securityGroup) {
                throw new Error('securityGroup must be set when providing vpc');
            }
            if (!props.vpcSubnets) {
                throw new Error('vpcSubnets must be set when providing vpc');
            }
        }
        else if (!props.secret) {
            throw new Error('secret must be set if Kafka brokers accessed over Internet');
        }
        this.innerProps = props;
    }
    /**
     * Called by `lambda.addEventSource` to allow the event source to bind to this function.
     *
     * @stability stable
     */
    bind(target) {
        if (!core_2.Construct.isConstruct(target)) {
            throw new Error('Function is not a construct. Unexpected error.');
        }
        target.addEventSourceMapping(this.mappingId(target), this.enrichMappingOptions({
            kafkaBootstrapServers: this.innerProps.bootstrapServers,
            kafkaTopic: this.innerProps.topic,
            startingPosition: this.innerProps.startingPosition,
            sourceAccessConfigurations: this.sourceAccessConfigurations(),
        }));
        if (this.innerProps.secret !== undefined) {
            this.innerProps.secret.grantRead(target);
        }
    }
    mappingId(target) {
        let hash = crypto.createHash('md5');
        hash.update(JSON.stringify(core_1.Stack.of(target).resolve(this.innerProps.bootstrapServers)));
        const idHash = hash.digest('hex');
        return `KafkaEventSource:${idHash}:${this.innerProps.topic}`;
    }
    sourceAccessConfigurations() {
        var _c;
        let authType;
        switch (this.innerProps.authenticationMethod) {
            case AuthenticationMethod.SASL_SCRAM_256_AUTH:
                authType = lambda.SourceAccessConfigurationType.SASL_SCRAM_256_AUTH;
                break;
            case AuthenticationMethod.SASL_SCRAM_512_AUTH:
            default:
                authType = lambda.SourceAccessConfigurationType.SASL_SCRAM_512_AUTH;
                break;
        }
        const sourceAccessConfigurations = [];
        if (this.innerProps.secret !== undefined) {
            sourceAccessConfigurations.push({ type: authType, uri: this.innerProps.secret.secretArn });
        }
        if (this.innerProps.vpcSubnets !== undefined && this.innerProps.securityGroup !== undefined) {
            sourceAccessConfigurations.push({
                type: lambda.SourceAccessConfigurationType.VPC_SECURITY_GROUP,
                uri: this.innerProps.securityGroup.securityGroupId,
            });
            (_c = this.innerProps.vpc) === null || _c === void 0 ? void 0 : _c.selectSubnets(this.innerProps.vpcSubnets).subnetIds.forEach((id) => {
                sourceAccessConfigurations.push({ type: lambda.SourceAccessConfigurationType.VPC_SUBNET, uri: id });
            });
        }
        return sourceAccessConfigurations.length === 0
            ? undefined
            : sourceAccessConfigurations;
    }
}
exports.SelfManagedKafkaEventSource = SelfManagedKafkaEventSource;
_b = JSII_RTTI_SYMBOL_1;
SelfManagedKafkaEventSource[_b] = { fqn: "@aws-cdk/aws-lambda-event-sources.SelfManagedKafkaEventSource", version: "1.119.0" };
//# sourceMappingURL=data:application/json;base64,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