"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Route53ToAlb = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const defaults = require("@aws-solutions-constructs/core");
const r53 = require("aws-cdk-lib/aws-route53");
const r53t = require("aws-cdk-lib/aws-route53-targets");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const constructs_1 = require("constructs");
/**
 * @summary Configures a Route53 Hosted Zone to route to an Application Load Balancer
 */
class Route53ToAlb extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the Route53ToAlb class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {Route53ToAlbProps} props - user provided props for the construct.
     * @access public
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        defaults.CheckProps(props);
        // NOTE: We don't call CheckAlbProps() here, because this construct creates an ALB
        // with no listener or target, so some of those checks don't apply
        if ((_b = props === null || props === void 0 ? void 0 : props.loadBalancerProps) === null || _b === void 0 ? void 0 : _b.vpc) {
            throw new Error('Specify any existing VPC at the construct level, not within loadBalancerProps.');
        }
        if (props.existingLoadBalancerObj && !props.existingVpc) {
            throw new Error('An existing ALB already exists in a VPC, so that VPC must be passed to the construct in props.existingVpc');
        }
        if (props.existingHostedZoneInterface && !props.publicApi && !props.existingVpc) {
            throw new Error('An existing Private Hosted Zone already exists in a VPC, so that VPC must be passed to the construct in props.existingVpc');
        }
        if (props.existingVpc) {
            this.vpc = props.existingVpc;
        }
        else {
            this.vpc = defaults.buildVpc(scope, {
                defaultVpcProps: props.publicApi ?
                    defaults.DefaultPublicPrivateVpcProps() :
                    // If this is an internal app, we're going to turn on DNS
                    // by default to allow gateway and interface service endpoints
                    defaults.overrideProps(defaults.DefaultIsolatedVpcProps(), { enableDnsHostnames: true, enableDnsSupport: true, }),
                userVpcProps: props.vpcProps,
            });
        }
        if (props.existingHostedZoneInterface) {
            this.hostedZone = props.existingHostedZoneInterface;
        }
        else {
            if (props.publicApi) {
                throw new Error('Public APIs require an existingHostedZone be passed in the Props object.');
            }
            else {
                if (!props.privateHostedZoneProps) {
                    throw new Error('Must supply privateHostedZoneProps to create a private API');
                }
                if (props.privateHostedZoneProps.vpc) {
                    throw new Error('All VPC specs must be provided at the Construct level in Route53ToAlbProps');
                }
                const manufacturedProps = defaults.overrideProps(props.privateHostedZoneProps, { vpc: this.vpc });
                this.hostedZone = new r53.PrivateHostedZone(this, `${id}-zone`, manufacturedProps);
            }
        }
        this.loadBalancer = defaults.ObtainAlb(this, id, this.vpc, props.publicApi, props.existingLoadBalancerObj, props.loadBalancerProps, props.logAlbAccessLogs, props.albLoggingBucketProps);
        // Add the ALB to the HostedZone as a target
        const hostedZoneTarget = new r53t.LoadBalancerTarget(this.loadBalancer);
        new r53.ARecord(this, `${id}-alias`, {
            target: { aliasTarget: hostedZoneTarget },
            zone: this.hostedZone
        });
    }
}
exports.Route53ToAlb = Route53ToAlb;
_a = JSII_RTTI_SYMBOL_1;
Route53ToAlb[_a] = { fqn: "@aws-solutions-constructs/aws-route53-alb.Route53ToAlb", version: "2.3.0" };
//# sourceMappingURL=data:application/json;base64,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