# coding: utf-8

# (C) Copyright IBM Corp. 2019, 2020.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import argparse
from watson_assistant_func import load_logs_from_file
from watson_assistant_func import export_csv_for_intent_recommendation


def parse_args():
    parser = argparse.ArgumentParser(description='Generate intent recommendation CSV from logs JSON file for Watson '
                                                 'Assistant service.')

    # Required arguments
    parser.add_argument(
        '--input_json',
        type=str,
        required=True,
        help="The path of the JSON file of logs, generated by `fetch_logs.py`",
    )
    parser.add_argument(
        '--output_csv',
        type=str,
        required=True,
        help="The path of the CSV file of utterances this script will generate for intent recommendation",
    )

    # Optional arguments
    parser.add_argument(
        '--overwrite',
        type=bool,
        default=False,
        help="If overwrite filename if it exists",
    )
    parser.add_argument(
        '--deduplicate',
        action="store_true",
        help="If set, duplicate utterances are discarded when generating CSV",
    )
    parser.add_argument(
        '--min_length',
        type=int,
        default=3,
        help="Minimum number of tokens of a utterance in the generated CSV.  Any utterance that has less than or "
             "equal to this number is discarded.",
    )
    parser.add_argument(
        '--max_length',
        type=int,
        default=20,
        help="Maximum number of tokens of a utterance in the generated CSV.  Any utterance that has more than or "
             "equal to this number is discarded.",
    )

    return parser.parse_args()


if __name__ == '__main__':
    args = parse_args()
    print(vars(args))

    logs = load_logs_from_file(filename=args.input_json,
                               project=None)

    export_csv_for_intent_recommendation(logs,
                                         filename=args.output_csv,
                                         deduplicate=args.deduplicate,
                                         project=None,
                                         overwrite=args.overwrite,
                                         min_length=args.min_length,
                                         max_length=args.max_length)
