# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ail_typo_squatting']

package_data = \
{'': ['*']}

install_requires = \
['dnspython>=2.2.0,<3.0.0', 'inflect>=5.3.0,<6.0.0', 'pyyaml>=6.0,<7.0']

setup_kwargs = {
    'name': 'ail-typo-squatting',
    'version': '1.2.0',
    'description': 'Generate list of domain name using Domain name permutation engine to feed AIL',
    'long_description': '# ail-typo-squatting\n\nail-typo-squatting is a Python library to generate list of potential typo squatting domains with domain name permutation engine to feed AIL and other systems. \n\nThe tool can be used as a stand-alone tool or to feed other systems.\n\n# Requirements\n\n- Python 3.6+\n- [inflect](https://github.com/jaraco/inflect) library\n- [pyyaml](https://pyyaml.org/wiki/PyYAMLDocumentation)\n\n## Optional\n\n- [dnspython](https://github.com/rthalley/dnspython)\n\n# Installation\n\n## Source install\n\nail-typo-squatting can be install with poetry. If you don\'t have poetry installed, you can do the following `curl -sSL https://raw.githubusercontent.com/python-poetry/poetry/master/get-poetry.py | python`.\n\n~~~bash\n$ poetry install\n$ poetry shell\n$ cd ail-typo-squatting\n$ python typo.py -h\n~~~\n\n## pip installation\n\n~~~bash\n$ pip3 install ail-typo-squatting\n~~~\n\n# Usage\n\n```bash\ndacru@dacru:~/git/ail-typo-squatting/bin$ python3 typo.py --help  \nusage: typo.py [-h] [-v] [-dn DOMAINNAME [DOMAINNAME ...]] [-fdn FILEDOMAINNAME] [-o OUTPUT] [-fo FORMATOUTPUT] [-dnsr] [-l LIMIT] [-var] [-a] [-om] [-repe] [-tra] [-repl] [-drepl] [-cho] [-ki] [-add] [-md] [-sd]\n               [-vs] [-ada] [-bs] [-hg] [-cm] [-hp] [-wt] [-at] [-sub] [-sp] [-cdd]\n\noptions:\n  -h, --help            show this help message and exit\n  -v                    verbose, more display\n  -dn DOMAINNAME [DOMAINNAME ...], --domainName DOMAINNAME [DOMAINNAME ...]\n                        list of domain name\n  -fdn FILEDOMAINNAME, --filedomainName FILEDOMAINNAME\n                        file containing list of domain name\n  -o OUTPUT, --output OUTPUT\n                        path to ouput location\n  -fo FORMATOUTPUT, --formatoutput FORMATOUTPUT\n                        format for the output file, yara - regex - yaml - text. Default: text\n  -dnsr, --dnsresolving\n                        resolve all variation of domain name to see if it\'s up or not\n  -l LIMIT, --limit LIMIT\n                        limit of variations for a domain name\n  -var, --givevariations\n                        give the algo that generate variations\n  -a, --all             Use all algo\n  -om, --omission       Leave out a letter of the domain name\n  -repe, --repetition   Character Repeat\n  -tra, --transposition\n                        Swappe the order of adjacent letters in the domain name\n  -repl, --replacement  Adjacent character replacement to the immediate left and right on the keyboard\n  -drepl, --doublereplacement\n                        Double Character Replacement\n  -cho, --changeorder   Change the order of letters in word\n  -ki, --keyboardinsertion\n                        Adjacent character insertion of letters to the immediate left and right on the keyboard of each letter\n  -add, --addition      Add a character in the domain name\n  -md, --missingdot     Delete a dot from the domain name\n  -sd, --stripdash      Delete of a dash from the domain name\n  -vs, --vowelswap      Swap vowels within the domain name\n  -ada, --adddash       Add a dash between the first and last character in a string\n  -bs, --bitsquatting   The character is substituted with the set of valid characters that can be made after a single bit flip\n  -hg, --homoglyph      One or more characters that look similar to another character but are different are called homogylphs\n  -cm, --commonmisspelling\n                        Change a word by is misspellings\n  -hp, --homophones     Change word by an other who sound the same when spoken\n  -wt, --wrongtld       Change the original top level domain to another\n  -at, --addtld         Adding a tld before the original tld\n  -sub, --subdomain     Insert a dot at varying positions to create subdomain\n  -sp, --singularpluralize\n                        Create by making a singular domain plural and vice versa\n  -cdd, --changedotdash\n                        Change dot to dash\n```\n\n\n\n# Usage example\n\n1. Creation of variations for `ail-project.org` and `circl.lu`, using all algorithm.\n\n```bash\ndacru@dacru:~/git/ail-typo-squatting/bin$ python3 typo.py -dn ail-project.org circl.lu -a -o .\n```\n\n2. Creation of variations for a file who contains domain name, using character omission - subdomain - hyphenation.\n\n````bash\ndacru@dacru:~/git/ail-typo-squatting/bin$ python3 typo.py -fdn domain.txt -co -sub -hyp -o . -fo yara\n````\n\n3. Creation of variations for `ail-project.org` and `circl.lu`, using all algorithm and using dns resolution.\n\n````bash\ndacru@dacru:~/git/ail-typo-squatting/bin$ python3 typo.py -dn ail-project.org circl.lu -a -dnsr -o .\n````\n\n4. Creation of variations for `ail-project.org`  and give the algorithm that generate the variation (**only for text format**).\n\n~~~bash\ndacru@dacru:~/git/ail-typo-squatting/bin$ python3 typo.py -dn ail-project.org -a -o - -var\n~~~\n\n\n\n# Used as a library\n\n## To run all algorithms\n\n~~~~python\nfrom ail_typo_squatting import runAll\nimport math\n\nresultList = list()\ndomainList = ["google.com"]\nformatoutput = "yara"\npathOutput = "."\nfor domain in domainList:\n    resultList = runAll(\n        domain=domain, \n        limit=math.inf, \n        formatoutput=formatoutput, \n        pathOutput=pathOutput, \n        verbose=False, \n        givevariations=False)\n    \n    print(resultList)\n    resultList = list()\n~~~~\n\n\n\n## To run specific algorithm\n\n````python\nfrom ail_typo_squatting import formatOutput, omission, subdomain, addDash\nimport math\n\nresultList = list()\ndomainList = ["google.com"]\nlimit = math.inf\nformatoutput = "yara"\npathOutput = "."\nfor domain in domainList:\n    resultList = omission(domain=domain, resultList=resultList, verbose=False, limit=limit, givevariations=False)\n    \n    resultList = subdomain(domain=domain, resultList=resultList, verbose=False, limit=limit, givevariations=False)\n    \n    resultList = addDash(domain=domain, resultList=resultList, verbose=False, limit=limit, givevariations=False)\n    \n    print(resultList)\n    formatOutput(format=formatoutput, resultList=resultList, domain=domain, pathOutput=pathOutput, givevariations=False)\n    \n    resultList = list()\n````\n\n\n\n# Sample output\n\nThere\'s **4 format** possible for the output file:\n\n- text\n- yara\n- regex\n- sigma\n\n\n\nFor **Text** file, each line is a variation.\n\n````\nail-project.org\nil-project.org\nal-project.org\nai-project.org\nailproject.org\nail-roject.org\nail-poject.org\nail-prject.org\nail-proect.org\nail-projct.org\nail-projet.org\nail-projec.org\naail-project.org\naiil-project.org\n...\n````\n\n\n\nFor **Yara** file, each rule is a variation.\n\n~~~~\nrule ail-project_org {\n\tmeta:\n\t\tdomain = "ail-project.org"\n\tstrings: \n\t\t$s0 = "ail-project.org"\n\t\t$s1 = "il-project.org"\n\t\t$s2 = "al-project.org"\n\t\t$s3 = "ai-project.org"\n\t\t$s4 = "ailproject.org"\n\t\t$s5 = "ail-roject.org"\n\t\t$s6 = "ail-poject.org"\n\t\t$s7 = "ail-prject.org"\n\t\t$s8 = "ail-proect.org"\n\t\t$s9 = "ail-projct.org"\n\t\t$s10 = "ail-projet.org"\n\t\t$s11 = "ail-projec.org"\n\tcondition:\n\t\t any of ($s*)\n}\n~~~~\n\n\n\nFor **Regex** file, each variations is transform into regex and concatenate with other to do only one big regex.\n\n~~~~\nail\\-project\\.org|il\\-project\\.org|al\\-project\\.org|ai\\-project\\.org|ailproject\\.org|ail\\-roject\\.org|ail\\-poject\\.org|ail\\-prject\\.org|ail\\-proect\\.org|ail\\-projct\\.org|ail\\-projet\\.org|ail\\-projec\\.org\n~~~~\n\n\n\nFor **Sigma** file, each variations are list under `variations` key.\n\n~~~~\ntitle: ail-project.org\nvariations:\n- ail-project.org\n- il-project.org\n- al-project.org\n- ai-project.org\n- ailproject.org\n- ail-roject.org\n- ail-poject.org\n- ail-prject.org\n- ail-proect.org\n- ail-projct.org\n- ail-projet.org\n- ail-projec.org\n~~~~\n\n\n\n\n\n## DNS output\n\nIn case DNS resolve is selected, an additional file will be created in JSON format\n\neach keys are variations and may have a field "ip" if the domain name have been resolved. The filed "NotExist" will be there each time with a Boolean value to determine if the domain is existing or not.\n\n````json\n{\n    "circl.lu": {\n        "NotExist": false,\n        "ip": [\n            "185.194.93.14"\n        ]\n    },\n    "ircl.lu": {\n        "NotExist": true\n    },\n    "crcl.lu": {\n        "NotExist": true\n    },\n    "cicl.lu": {\n        "NotExist": true\n    },\n    "cirl.lu": {\n        "NotExist": true\n    },\n    "circ.lu": {\n        "NotExist": true\n    },\n    "ccircl.lu": {\n        "NotExist": true\n    },\n    "ciircl.lu": {\n        "NotExist": true\n    },\n    ...\n}\n````\n\n\n\n# List of algorithms used\n\n\n| Algo               | Description                                                  |\n| :----------------- | :----------------------------------------------------------- |\n| Omission           | These typos are created by leaving out a letter of the domain name, one letter at a time. |\n| Repetition         | These typos are created by repeating a letter of the domain name. |\n| ChangeOrder        | These typos are created by changing the order of letters in the each part of the domain. |\n| Transposition      | These typos are created by swapping the order of adjacent letters in the domain name. |\n| Replacement        | These typos are created by replacing each letter of the domain name with letters to the immediate left and right on the keyboard. (QWERTY, AZERTY, QWERTZ, DVORAK) |\n| Double Replacement | These typos are created by replacing identical, consecutive letters of the domain name with letters to the immediate left and right on the keyboard. |\n| Addition           | These typos are created by add a characters in the domain name. |\n| KeyboardInsertion  | These typos are created by inserting letters to the immediate left and right on the keyboard of each letter. |\n| MissingDot         | These typos are created by deleting a dot from the domain name. |\n| StripDash          | These typos are created by deleting a dash from the domain name. |\n| VowelSwap          | These typos are created by swapping vowels within the domain name except for the first letter. For example, www.google.com becomes www.gaagle.com. |\n| AddDash            | These typos are created by adding a dash between the first and last character in a string. |\n| Bitsquatting       | These typos are created by substituting a character with the set of valid characters that can be made after a single bit flip. For example, facebook.com becomes bacebook.com, dacebook.com, faaebook.com,fabebook.com,facabook.com, etc. |\n| Homoglyph          | These typos are created by replacing characters to another character that look similar but are different.  An example is that the lower case l looks similar to the numeral one, e.g. l vs 1. For example, google.com becomes goog1e.com. |\n| CommonMisspelling  | These typos are created by changing a word by is misspelling. Over 8000 common misspellings from Wikipedia. For example, www.youtube.com becomes www.youtub.com and www.abseil.com becomes www.absail.com. |\n| Homophones         | These typos are created by changing word by an other who sound the same when spoken. Over 450 sets of words that sound the same when spoken. For example, www.base.com becomes www.bass.com. |\n| WrongTld           | These typos are created by changing the original top level domain to another. For example, www.trademe.co.nz becomes www.trademe.co.mz and www.google.com becomes www.google.org Uses the 19 most common top level domains. |\n| AddTld             | These typos are created by adding a tld before the right tld. Example: google.com becomes google.com.it |\n| Subdomain          | These typos are created by placing a dot in the domain name in order to create subdomain. Example: google.com becomes goo.gle.com |\n| SingularPluralize  | These typos are created by making a singular domain plural and vice versa. |\n| ChangeDotDash      | These typos are created by changing a dot to a dash          |\n\n\n\n# Acknowledgment\n\n![](./img/cef.png)\n\nThe project has been co-funded by CEF-TC-2020-2 - 2020-EU-IA-0260 - JTAN - Joint Threat Analysis Network.\n',
    'author': 'David Cruciani',
    'author_email': 'david.cruciani@securitymadein.lu',
    'maintainer': 'Alexandre Dulaunoy',
    'maintainer_email': 'a@foo.be',
    'url': 'https://github.com/ail-project/ail-typo-squatting',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
