"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const types_1 = require("../types");
/**
 * Base class for all other state classes
 */
class State extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.branches = [];
        this.retries = [];
        this.catches = [];
        this.choices = [];
        this.prefixes = [];
        /**
         * States with references to this state.
         *
         * Used for finding complete connected graph that a state is part of.
         */
        this.incomingStates = [];
        this.startState = this;
        this.comment = props.comment;
        this.inputPath = props.inputPath;
        this.parameters = props.parameters;
        this.outputPath = props.outputPath;
        this.resultPath = props.resultPath;
    }
    /**
     * Add a prefix to the stateId of all States found in a construct tree
     */
    static prefixStates(root, prefix) {
        const queue = [root];
        while (queue.length > 0) {
            const el = queue.splice(0, 1)[0];
            if (isPrefixable(el)) {
                el.addPrefix(prefix);
            }
            queue.push(...el.node.children);
        }
    }
    /**
     * Find the set of states reachable through transitions from the given start state.
     * This does not retrieve states from within sub-graphs, such as states within a Parallel state's branch.
     */
    static findReachableStates(start, options = {}) {
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            queue.push(...outgoing);
            ret.add(state);
        }
        return Array.from(ret);
    }
    /**
     * Find the set of end states states reachable through transitions from the given start state
     */
    static findReachableEndStates(start, options = {}) {
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            if (outgoing.length > 0) {
                // We can continue
                queue.push(...outgoing);
            }
            else {
                // Terminal state
                ret.add(state);
            }
        }
        return Array.from(ret);
    }
    /**
     * Return only the states that allow chaining from an array of states
     */
    static filterNextables(states) {
        return states.filter(isNextable);
    }
    get id() {
        return this.node.id;
    }
    /**
     * Tokenized string that evaluates to the state's ID
     */
    get stateId() {
        return this.prefixes.concat(this.id).join('');
    }
    /**
     * Add a prefix to the stateId of this state
     */
    addPrefix(x) {
        if (x !== '') {
            this.prefixes.splice(0, 0, x);
        }
    }
    /**
     * Register this state as part of the given graph
     *
     * Don't call this. It will be called automatically when you work
     * with states normally.
     */
    bindToGraph(graph) {
        if (this.containingGraph === graph) {
            return;
        }
        if (this.containingGraph) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Trying to use state '${this.stateId}' in ${graph}, but is already in ${this.containingGraph}. Every state can only be used in one graph.`);
        }
        this.containingGraph = graph;
        this.whenBoundToGraph(graph);
        for (const incoming of this.incomingStates) {
            incoming.bindToGraph(graph);
        }
        for (const outgoing of this.outgoingTransitions({ includeErrorHandlers: true })) {
            outgoing.bindToGraph(graph);
        }
        for (const branch of this.branches) {
            branch.registerSuperGraph(this.containingGraph);
        }
        if (!!this.iteration) {
            this.iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a retrier to the retry list of this state
     * @internal
     */
    _addRetry(props = {}) {
        this.retries.push({
            ...props,
            errors: props.errors ? props.errors : [types_1.Errors.ALL],
        });
    }
    /**
     * Add an error handler to the catch list of this state
     * @internal
     */
    _addCatch(handler, props = {}) {
        this.catches.push({
            next: handler,
            props: {
                errors: props.errors ? props.errors : [types_1.Errors.ALL],
                resultPath: props.resultPath,
            },
        });
        handler.addIncoming(this);
        if (this.containingGraph) {
            handler.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default transition of this state
     */
    makeNext(next) {
        // Can't be called 'setNext' because of JSII
        if (this._next) {
            throw new Error(`State '${this.id}' already has a next state`);
        }
        this._next = next;
        next.addIncoming(this);
        if (this.containingGraph) {
            next.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a choice branch to this state
     */
    addChoice(condition, next) {
        this.choices.push({ condition, next });
        next.startState.addIncoming(this);
        if (this.containingGraph) {
            next.startState.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a paralle branch to this state
     */
    addBranch(branch) {
        this.branches.push(branch);
        if (this.containingGraph) {
            branch.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a map iterator to this state
     */
    addIterator(iteration) {
        this.iteration = iteration;
        if (this.containingGraph) {
            iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default choice transition of this state
     */
    makeDefault(def) {
        // Can't be called 'setDefault' because of JSII
        if (this.defaultChoice) {
            throw new Error(`Choice '${this.id}' already has a default next state`);
        }
        this.defaultChoice = def;
    }
    /**
     * Render the default next state in ASL JSON format
     */
    renderNextEnd() {
        if (this._next) {
            return { Next: this._next.stateId };
        }
        else {
            return { End: true };
        }
    }
    /**
     * Render the choices in ASL JSON format
     */
    renderChoices() {
        return {
            Choices: renderList(this.choices, renderChoice),
            Default: this.defaultChoice ? this.defaultChoice.stateId : undefined,
        };
    }
    /**
     * Render InputPath/Parameters/OutputPath in ASL JSON format
     */
    renderInputOutput() {
        return {
            InputPath: renderJsonPath(this.inputPath),
            Parameters: this.parameters,
            OutputPath: renderJsonPath(this.outputPath),
        };
    }
    /**
     * Render parallel branches in ASL JSON format
     */
    renderBranches() {
        return {
            Branches: this.branches.map(b => b.toGraphJson()),
        };
    }
    /**
     * Render map iterator in ASL JSON format
     */
    renderIterator() {
        if (!this.iteration) {
            throw new Error('Iterator must not be undefined !');
        }
        return {
            Iterator: this.iteration.toGraphJson(),
        };
    }
    /**
     * Render error recovery options in ASL JSON format
     */
    renderRetryCatch() {
        return {
            Retry: renderList(this.retries, renderRetry),
            Catch: renderList(this.catches, renderCatch),
        };
    }
    /**
     * Called whenever this state is bound to a graph
     *
     * Can be overridden by subclasses.
     */
    whenBoundToGraph(graph) {
        graph.registerState(this);
    }
    /**
     * Add a state to the incoming list
     */
    addIncoming(source) {
        this.incomingStates.push(source);
    }
    /**
     * Return all states this state can transition to
     */
    outgoingTransitions(options) {
        const ret = new Array();
        if (this._next) {
            ret.push(this._next);
        }
        if (this.defaultChoice) {
            ret.push(this.defaultChoice);
        }
        for (const c of this.choices) {
            ret.push(c.next);
        }
        if (options.includeErrorHandlers) {
            for (const c of this.catches) {
                ret.push(c.next);
            }
        }
        return ret;
    }
}
exports.State = State;
/**
 * Render a choice transition
 */
function renderChoice(c) {
    return {
        ...c.condition.renderCondition(),
        Next: c.next.stateId,
    };
}
/**
 * Render a Retry object to ASL
 */
function renderRetry(retry) {
    return {
        ErrorEquals: retry.errors,
        IntervalSeconds: retry.interval && retry.interval.toSeconds(),
        MaxAttempts: retry.maxAttempts,
        BackoffRate: retry.backoffRate,
    };
}
/**
 * Render a Catch object to ASL
 */
function renderCatch(c) {
    return {
        ErrorEquals: c.props.errors,
        ResultPath: renderJsonPath(c.props.resultPath),
        Next: c.next.stateId,
    };
}
/**
 * Render a list or return undefined for an empty list
 */
function renderList(xs, fn) {
    if (xs.length === 0) {
        return undefined;
    }
    return xs.map(fn);
}
exports.renderList = renderList;
/**
 * Render JSON path, respecting the special value DISCARD
 */
function renderJsonPath(jsonPath) {
    if (jsonPath === undefined) {
        return undefined;
    }
    if (jsonPath === types_1.DISCARD) {
        return null;
    }
    if (!jsonPath.startsWith('$')) {
        throw new Error(`Expected JSON path to start with '$', got: ${jsonPath}`);
    }
    return jsonPath;
}
exports.renderJsonPath = renderJsonPath;
/**
 * Whether an object is a Prefixable
 */
function isPrefixable(x) {
    return typeof (x) === 'object' && x.addPrefix;
}
/**
 * Whether an object is INextable
 */
function isNextable(x) {
    return typeof (x) === 'object' && x.next;
}
//# sourceMappingURL=data:application/json;base64,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