"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const parallel_1 = require("./states/parallel");
/**
 * A collection of states to chain onto
 *
 * A Chain has a start and zero or more chainable ends. If there are
 * zero ends, calling next() on the Chain will fail.
 */
class Chain {
    constructor(startState, endStates, lastAdded) {
        this.lastAdded = lastAdded;
        this.id = lastAdded.id;
        this.startState = startState;
        this.endStates = endStates;
    }
    /**
     * Begin a new Chain from one chainable
     */
    static start(state) {
        return new Chain(state.startState, state.endStates, state);
    }
    /**
     * Make a Chain with the start from one chain and the ends from another
     */
    static sequence(start, next) {
        return new Chain(start.startState, next.endStates, next);
    }
    /**
     * Make a Chain with specific start and end states, and a last-added Chainable
     */
    static custom(startState, endStates, lastAdded) {
        return new Chain(startState, endStates, lastAdded);
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        if (this.endStates.length === 0) {
            throw new Error(`Cannot add to chain: last state in chain (${this.lastAdded.id}) does not allow it`);
        }
        for (const endState of this.endStates) {
            endState.next(next);
        }
        return new Chain(this.startState, next.endStates, next);
    }
    /**
     * Return a single state that encompasses all states in the chain
     *
     * This can be used to add error handling to a sequence of states.
     *
     * Be aware that this changes the result of the inner state machine
     * to be an array with the result of the state machine in it. Adjust
     * your paths accordingly. For example, change 'outputPath' to
     * '$[0]'.
     */
    toSingleState(id, props = {}) {
        return new parallel_1.Parallel(this.startState, id, props).branch(this);
    }
}
exports.Chain = Chain;
//# sourceMappingURL=data:application/json;base64,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