"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn = require("@aws-cdk/aws-cloudformation");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const path = require("path");
/**
 * A certificate managed by AWS Certificate Manager.  Will be automatically
 * validated using DNS validation against the specified Route 53 hosted zone.
 *
 * @resource AWS::CertificateManager::Certificate
 * @experimental
 */
class DnsValidatedCertificate extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.domainName = props.domainName;
        this.normalizedZoneName = props.hostedZone.zoneName;
        // Remove trailing `.` from zone name
        if (this.normalizedZoneName.endsWith('.')) {
            this.normalizedZoneName = this.normalizedZoneName.substring(0, this.normalizedZoneName.length - 1);
        }
        // Remove any `/hostedzone/` prefix from the Hosted Zone ID
        this.hostedZoneId = props.hostedZone.hostedZoneId.replace(/^\/hostedzone\//, '');
        const requestorFunction = new lambda.Function(this, 'CertificateRequestorFunction', {
            code: lambda.Code.fromAsset(path.resolve(__dirname, '..', 'lambda-packages', 'dns_validated_certificate_handler', 'lib')),
            handler: 'index.certificateRequestHandler',
            runtime: lambda.Runtime.NODEJS_10_X,
            timeout: cdk.Duration.minutes(15),
            role: props.customResourceRole
        });
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['acm:RequestCertificate', 'acm:DescribeCertificate', 'acm:DeleteCertificate'],
            resources: ['*'],
        }));
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['route53:GetChange'],
            resources: ['*'],
        }));
        requestorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['route53:changeResourceRecordSets'],
            resources: [`arn:${cdk.Stack.of(requestorFunction).partition}:route53:::hostedzone/${this.hostedZoneId}`],
        }));
        const certificate = new cfn.CustomResource(this, 'CertificateRequestorResource', {
            provider: cfn.CustomResourceProvider.lambda(requestorFunction),
            properties: {
                DomainName: props.domainName,
                SubjectAlternativeNames: cdk.Lazy.listValue({ produce: () => props.subjectAlternativeNames }, { omitEmpty: true }),
                HostedZoneId: this.hostedZoneId,
                Region: props.region,
                Route53Endpoint: props.route53Endpoint,
            }
        });
        this.certificateArn = certificate.getAtt('Arn').toString();
    }
    validate() {
        const errors = [];
        // Ensure the zone name is a parent zone of the certificate domain name
        if (!cdk.Token.isUnresolved(this.normalizedZoneName) &&
            this.domainName !== this.normalizedZoneName &&
            !this.domainName.endsWith('.' + this.normalizedZoneName)) {
            errors.push(`DNS zone ${this.normalizedZoneName} is not authoritative for certificate domain name ${this.domainName}`);
        }
        return errors;
    }
}
exports.DnsValidatedCertificate = DnsValidatedCertificate;
//# sourceMappingURL=data:application/json;base64,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