"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const certificatemanager_generated_1 = require("./certificatemanager.generated");
const util_1 = require("./util");
/**
 * A certificate managed by AWS Certificate Manager
 *
 * IMPORTANT: if you are creating a certificate as part of your stack, the stack
 * will not complete creating until you read and follow the instructions in the
 * email that you will receive.
 *
 * ACM will send validation emails to the following addresses:
 *
 *  admin@domain.com
 *  administrator@domain.com
 *  hostmaster@domain.com
 *  postmaster@domain.com
 *  webmaster@domain.com
 *
 * For every domain that you register.
 */
class Certificate extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const allDomainNames = [props.domainName].concat(props.subjectAlternativeNames || []);
        const cert = new certificatemanager_generated_1.CfnCertificate(this, 'Resource', {
            domainName: props.domainName,
            subjectAlternativeNames: props.subjectAlternativeNames,
            domainValidationOptions: allDomainNames.map(domainValidationOption),
            validationMethod: props.validationMethod,
        });
        this.certificateArn = cert.ref;
        /**
         * Return the domain validation options for the given domain
         *
         * Closes over props.
         */
        function domainValidationOption(domainName) {
            let validationDomain = props.validationDomains && props.validationDomains[domainName];
            if (validationDomain === undefined) {
                if (core_1.Token.isUnresolved(domainName)) {
                    throw new Error('When using Tokens for domain names, \'validationDomains\' needs to be supplied');
                }
                validationDomain = util_1.apexDomain(domainName);
            }
            return { domainName, validationDomain };
        }
    }
    /**
     * Import a certificate
     */
    static fromCertificateArn(scope, id, certificateArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.certificateArn = certificateArn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Certificate = Certificate;
/**
 * Method used to assert ownership of the domain
 */
var ValidationMethod;
(function (ValidationMethod) {
    /**
     * Send email to a number of email addresses associated with the domain
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-email.html
     */
    ValidationMethod["EMAIL"] = "EMAIL";
    /**
     * Validate ownership by adding appropriate DNS records
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-dns.html
     */
    ValidationMethod["DNS"] = "DNS";
})(ValidationMethod = exports.ValidationMethod || (exports.ValidationMethod = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2VydGlmaWNhdGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjZXJ0aWZpY2F0ZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLHdDQUFzRTtBQUN0RSxpRkFBZ0U7QUFDaEUsaUNBQW9DO0FBZ0RwQzs7Ozs7Ozs7Ozs7Ozs7OztHQWdCRztBQUNILE1BQWEsV0FBWSxTQUFRLGVBQVE7SUFrQnZDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBdUI7UUFDL0QsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixNQUFNLGNBQWMsR0FBRyxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLHVCQUF1QixJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBRXRGLE1BQU0sSUFBSSxHQUFHLElBQUksNkNBQWMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ2hELFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtZQUM1Qix1QkFBdUIsRUFBRSxLQUFLLENBQUMsdUJBQXVCO1lBQ3RELHVCQUF1QixFQUFFLGNBQWMsQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUM7WUFDbkUsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQjtTQUN6QyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUM7UUFFL0I7Ozs7V0FJRztRQUNILFNBQVMsc0JBQXNCLENBQUMsVUFBa0I7WUFDaEQsSUFBSSxnQkFBZ0IsR0FBRyxLQUFLLENBQUMsaUJBQWlCLElBQUksS0FBSyxDQUFDLGlCQUFpQixDQUFDLFVBQVUsQ0FBQyxDQUFDO1lBQ3RGLElBQUksZ0JBQWdCLEtBQUssU0FBUyxFQUFFO2dCQUNsQyxJQUFJLFlBQUssQ0FBQyxZQUFZLENBQUMsVUFBVSxDQUFDLEVBQUU7b0JBQ2xDLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0ZBQWdGLENBQUMsQ0FBQztpQkFDbkc7Z0JBQ0QsZ0JBQWdCLEdBQUcsaUJBQVUsQ0FBQyxVQUFVLENBQUMsQ0FBQzthQUMzQztZQUVELE9BQU8sRUFBRSxVQUFVLEVBQUUsZ0JBQWdCLEVBQUUsQ0FBQztRQUMxQyxDQUFDO0lBQ0gsQ0FBQztJQTlDRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxjQUFzQjtRQUNuRixNQUFNLE1BQU8sU0FBUSxlQUFRO1lBQTdCOztnQkFDUyxtQkFBYyxHQUFHLGNBQWMsQ0FBQztZQUN6QyxDQUFDO1NBQUE7UUFFRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUMvQixDQUFDO0NBc0NGO0FBakRELGtDQWlEQztBQUVEOztHQUVHO0FBQ0gsSUFBWSxnQkFjWDtBQWRELFdBQVksZ0JBQWdCO0lBQzFCOzs7O09BSUc7SUFDSCxtQ0FBZSxDQUFBO0lBRWY7Ozs7T0FJRztJQUNILCtCQUFXLENBQUE7QUFDYixDQUFDLEVBZFcsZ0JBQWdCLEdBQWhCLHdCQUFnQixLQUFoQix3QkFBZ0IsUUFjM0IiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBDb25zdHJ1Y3QsIElSZXNvdXJjZSwgUmVzb3VyY2UsIFRva2VuIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDZm5DZXJ0aWZpY2F0ZSB9IGZyb20gJy4vY2VydGlmaWNhdGVtYW5hZ2VyLmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBhcGV4RG9tYWluIH0gZnJvbSAnLi91dGlsJztcblxuZXhwb3J0IGludGVyZmFjZSBJQ2VydGlmaWNhdGUgZXh0ZW5kcyBJUmVzb3VyY2Uge1xuICAvKipcbiAgICogVGhlIGNlcnRpZmljYXRlJ3MgQVJOXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGNlcnRpZmljYXRlQXJuOiBzdHJpbmc7XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgeW91ciBjZXJ0aWZpY2F0ZVxuICovXG5leHBvcnQgaW50ZXJmYWNlIENlcnRpZmljYXRlUHJvcHMge1xuICAvKipcbiAgICogRnVsbHktcXVhbGlmaWVkIGRvbWFpbiBuYW1lIHRvIHJlcXVlc3QgYSBjZXJ0aWZpY2F0ZSBmb3IuXG4gICAqXG4gICAqIE1heSBjb250YWluIHdpbGRjYXJkcywgc3VjaCBhcyBgYCouZG9tYWluLmNvbWBgLlxuICAgKi9cbiAgcmVhZG9ubHkgZG9tYWluTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBbHRlcm5hdGl2ZSBkb21haW4gbmFtZXMgb24geW91ciBjZXJ0aWZpY2F0ZS5cbiAgICpcbiAgICogVXNlIHRoaXMgdG8gcmVnaXN0ZXIgYWx0ZXJuYXRpdmUgZG9tYWluIG5hbWVzIHRoYXQgcmVwcmVzZW50IHRoZSBzYW1lIHNpdGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gYWRkaXRpb25hbCBGUUROcyB3aWxsIGJlIGluY2x1ZGVkIGFzIGFsdGVybmF0aXZlIGRvbWFpbiBuYW1lcy5cbiAgICovXG4gIHJlYWRvbmx5IHN1YmplY3RBbHRlcm5hdGl2ZU5hbWVzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFdoYXQgdmFsaWRhdGlvbiBkb21haW4gdG8gdXNlIGZvciBldmVyeSByZXF1ZXN0ZWQgZG9tYWluLlxuICAgKlxuICAgKiBIYXMgdG8gYmUgYSBzdXBlcmRvbWFpbiBvZiB0aGUgcmVxdWVzdGVkIGRvbWFpbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBcGV4IGRvbWFpbiBpcyB1c2VkIGZvciBldmVyeSBkb21haW4gdGhhdCdzIG5vdCBvdmVycmlkZGVuLlxuICAgKi9cbiAgcmVhZG9ubHkgdmFsaWRhdGlvbkRvbWFpbnM/OiB7W2RvbWFpbk5hbWU6IHN0cmluZ106IHN0cmluZ307XG5cbiAgLyoqXG4gICAqIFZhbGlkYXRpb24gbWV0aG9kIHVzZWQgdG8gYXNzZXJ0IGRvbWFpbiBvd25lcnNoaXBcbiAgICpcbiAgICogQGRlZmF1bHQgVmFsaWRhdGlvbk1ldGhvZC5FTUFJTFxuICAgKi9cbiAgcmVhZG9ubHkgdmFsaWRhdGlvbk1ldGhvZD86IFZhbGlkYXRpb25NZXRob2Q7XG59XG5cbi8qKlxuICogQSBjZXJ0aWZpY2F0ZSBtYW5hZ2VkIGJ5IEFXUyBDZXJ0aWZpY2F0ZSBNYW5hZ2VyXG4gKlxuICogSU1QT1JUQU5UOiBpZiB5b3UgYXJlIGNyZWF0aW5nIGEgY2VydGlmaWNhdGUgYXMgcGFydCBvZiB5b3VyIHN0YWNrLCB0aGUgc3RhY2tcbiAqIHdpbGwgbm90IGNvbXBsZXRlIGNyZWF0aW5nIHVudGlsIHlvdSByZWFkIGFuZCBmb2xsb3cgdGhlIGluc3RydWN0aW9ucyBpbiB0aGVcbiAqIGVtYWlsIHRoYXQgeW91IHdpbGwgcmVjZWl2ZS5cbiAqXG4gKiBBQ00gd2lsbCBzZW5kIHZhbGlkYXRpb24gZW1haWxzIHRvIHRoZSBmb2xsb3dpbmcgYWRkcmVzc2VzOlxuICpcbiAqICBhZG1pbkBkb21haW4uY29tXG4gKiAgYWRtaW5pc3RyYXRvckBkb21haW4uY29tXG4gKiAgaG9zdG1hc3RlckBkb21haW4uY29tXG4gKiAgcG9zdG1hc3RlckBkb21haW4uY29tXG4gKiAgd2VibWFzdGVyQGRvbWFpbi5jb21cbiAqXG4gKiBGb3IgZXZlcnkgZG9tYWluIHRoYXQgeW91IHJlZ2lzdGVyLlxuICovXG5leHBvcnQgY2xhc3MgQ2VydGlmaWNhdGUgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElDZXJ0aWZpY2F0ZSB7XG5cbiAgLyoqXG4gICAqIEltcG9ydCBhIGNlcnRpZmljYXRlXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21DZXJ0aWZpY2F0ZUFybihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBjZXJ0aWZpY2F0ZUFybjogc3RyaW5nKTogSUNlcnRpZmljYXRlIHtcbiAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElDZXJ0aWZpY2F0ZSB7XG4gICAgICBwdWJsaWMgY2VydGlmaWNhdGVBcm4gPSBjZXJ0aWZpY2F0ZUFybjtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBjZXJ0aWZpY2F0ZSdzIEFSTlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGNlcnRpZmljYXRlQXJuOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IENlcnRpZmljYXRlUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgY29uc3QgYWxsRG9tYWluTmFtZXMgPSBbcHJvcHMuZG9tYWluTmFtZV0uY29uY2F0KHByb3BzLnN1YmplY3RBbHRlcm5hdGl2ZU5hbWVzIHx8IFtdKTtcblxuICAgIGNvbnN0IGNlcnQgPSBuZXcgQ2ZuQ2VydGlmaWNhdGUodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgZG9tYWluTmFtZTogcHJvcHMuZG9tYWluTmFtZSxcbiAgICAgIHN1YmplY3RBbHRlcm5hdGl2ZU5hbWVzOiBwcm9wcy5zdWJqZWN0QWx0ZXJuYXRpdmVOYW1lcyxcbiAgICAgIGRvbWFpblZhbGlkYXRpb25PcHRpb25zOiBhbGxEb21haW5OYW1lcy5tYXAoZG9tYWluVmFsaWRhdGlvbk9wdGlvbiksXG4gICAgICB2YWxpZGF0aW9uTWV0aG9kOiBwcm9wcy52YWxpZGF0aW9uTWV0aG9kLFxuICAgIH0pO1xuXG4gICAgdGhpcy5jZXJ0aWZpY2F0ZUFybiA9IGNlcnQucmVmO1xuXG4gICAgLyoqXG4gICAgICogUmV0dXJuIHRoZSBkb21haW4gdmFsaWRhdGlvbiBvcHRpb25zIGZvciB0aGUgZ2l2ZW4gZG9tYWluXG4gICAgICpcbiAgICAgKiBDbG9zZXMgb3ZlciBwcm9wcy5cbiAgICAgKi9cbiAgICBmdW5jdGlvbiBkb21haW5WYWxpZGF0aW9uT3B0aW9uKGRvbWFpbk5hbWU6IHN0cmluZyk6IENmbkNlcnRpZmljYXRlLkRvbWFpblZhbGlkYXRpb25PcHRpb25Qcm9wZXJ0eSB7XG4gICAgICBsZXQgdmFsaWRhdGlvbkRvbWFpbiA9IHByb3BzLnZhbGlkYXRpb25Eb21haW5zICYmIHByb3BzLnZhbGlkYXRpb25Eb21haW5zW2RvbWFpbk5hbWVdO1xuICAgICAgaWYgKHZhbGlkYXRpb25Eb21haW4gPT09IHVuZGVmaW5lZCkge1xuICAgICAgICBpZiAoVG9rZW4uaXNVbnJlc29sdmVkKGRvbWFpbk5hbWUpKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdXaGVuIHVzaW5nIFRva2VucyBmb3IgZG9tYWluIG5hbWVzLCBcXCd2YWxpZGF0aW9uRG9tYWluc1xcJyBuZWVkcyB0byBiZSBzdXBwbGllZCcpO1xuICAgICAgICB9XG4gICAgICAgIHZhbGlkYXRpb25Eb21haW4gPSBhcGV4RG9tYWluKGRvbWFpbk5hbWUpO1xuICAgICAgfVxuXG4gICAgICByZXR1cm4geyBkb21haW5OYW1lLCB2YWxpZGF0aW9uRG9tYWluIH07XG4gICAgfVxuICB9XG59XG5cbi8qKlxuICogTWV0aG9kIHVzZWQgdG8gYXNzZXJ0IG93bmVyc2hpcCBvZiB0aGUgZG9tYWluXG4gKi9cbmV4cG9ydCBlbnVtIFZhbGlkYXRpb25NZXRob2Qge1xuICAvKipcbiAgICogU2VuZCBlbWFpbCB0byBhIG51bWJlciBvZiBlbWFpbCBhZGRyZXNzZXMgYXNzb2NpYXRlZCB3aXRoIHRoZSBkb21haW5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYWNtL2xhdGVzdC91c2VyZ3VpZGUvZ3MtYWNtLXZhbGlkYXRlLWVtYWlsLmh0bWxcbiAgICovXG4gIEVNQUlMID0gJ0VNQUlMJyxcblxuICAvKipcbiAgICogVmFsaWRhdGUgb3duZXJzaGlwIGJ5IGFkZGluZyBhcHByb3ByaWF0ZSBETlMgcmVjb3Jkc1xuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hY20vbGF0ZXN0L3VzZXJndWlkZS9ncy1hY20tdmFsaWRhdGUtZG5zLmh0bWxcbiAgICovXG4gIEROUyA9ICdETlMnLFxufSJdfQ==