"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToSqs = void 0;
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
const cdk = require("@aws-cdk/core");
/**
 * @summary The ApiGatewayToSqs class.
 */
class ApiGatewayToSqs extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the ApiGatewayToSqs class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the dead letter queue, if applicable
        this.deadLetterQueue = defaults.buildDeadLetterQueue(this, {
            existingQueueObj: props.existingQueueObj,
            deployDeadLetterQueue: props.deployDeadLetterQueue,
            deadLetterQueueProps: props.deadLetterQueueProps,
            maxReceiveCount: props.maxReceiveCount
        });
        // Setup the queue
        [this.sqsQueue] = defaults.buildQueue(this, 'queue', {
            queueProps: props.queueProps,
            deadLetterQueue: this.deadLetterQueue
        });
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, props.apiGatewayProps);
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        // Setup the API Gateway resource
        const apiGatewayResource = this.apiGateway.root.addResource('message');
        // Create
        let createRequestTemplate = "Action=SendMessage&MessageBody=$util.urlEncode(\"$input.body\")";
        if (props.createRequestTemplate) {
            createRequestTemplate = props.createRequestTemplate;
        }
        if (props.allowCreateOperation && props.allowCreateOperation === true) {
            this.addActionToPolicy("sqs:SendMessage");
            defaults.addProxyMethodToApiResource({
                service: "sqs",
                path: `${cdk.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "POST",
                apiResource: this.apiGateway.root,
                requestTemplate: createRequestTemplate,
                contentType: "'application/x-www-form-urlencoded'"
            });
        }
        // Read
        let readRequestTemplate = "Action=ReceiveMessage";
        if (props.readRequestTemplate) {
            readRequestTemplate = props.readRequestTemplate;
        }
        if (props.allowReadOperation === undefined || props.allowReadOperation === true) {
            this.addActionToPolicy("sqs:ReceiveMessage");
            defaults.addProxyMethodToApiResource({
                service: "sqs",
                path: `${cdk.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "GET",
                apiResource: this.apiGateway.root,
                requestTemplate: readRequestTemplate,
                contentType: "'application/x-www-form-urlencoded'"
            });
        }
        // Delete
        let deleteRequestTemplate = "Action=DeleteMessage&ReceiptHandle=$util.urlEncode($input.params('receiptHandle'))";
        if (props.deleteRequestTemplate) {
            deleteRequestTemplate = props.deleteRequestTemplate;
        }
        if (props.allowDeleteOperation && props.allowDeleteOperation === true) {
            this.addActionToPolicy("sqs:DeleteMessage");
            defaults.addProxyMethodToApiResource({
                service: "sqs",
                path: `${cdk.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "DELETE",
                apiResource: apiGatewayResource,
                requestTemplate: deleteRequestTemplate,
                contentType: "'application/x-www-form-urlencoded'"
            });
        }
    }
    addActionToPolicy(action) {
        this.apiGatewayRole.addToPolicy(new iam.PolicyStatement({
            resources: [
                this.sqsQueue.queueArn
            ],
            actions: [`${action}`]
        }));
    }
}
exports.ApiGatewayToSqs = ApiGatewayToSqs;
//# sourceMappingURL=data:application/json;base64,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