"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const fs = require("fs");
const path = require("path");
const package_json_manager_1 = require("./package-json-manager");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    /**
     * Parcel bundled Lambda asset code
     */
    static parcel(options) {
        var _a, _b, _c, _d, _e, _f;
        // Find project root
        const projectRoot = (_a = options.projectRoot) !== null && _a !== void 0 ? _a : util_1.findUp(`.git${path.sep}`);
        if (!projectRoot) {
            throw new Error('Cannot find project root. Please specify it with `projectRoot`.');
        }
        // Bundling image derived from runtime bundling image (lambci)
        const image = cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../parcel'), {
            buildArgs: {
                IMAGE: options.runtime.bundlingDockerImage.image,
                PARCEL_VERSION: (_b = options.parcelVersion) !== null && _b !== void 0 ? _b : '2.0.0-beta.1',
            },
        });
        const packageJsonManager = new package_json_manager_1.PackageJsonManager(path.dirname(options.entry));
        // Collect external and install modules
        let includeNodeModules;
        let dependencies;
        const externalModules = (_c = options.externalModules) !== null && _c !== void 0 ? _c : ['aws-sdk'];
        if (externalModules || options.nodeModules) {
            const modules = [...externalModules, ...(_d = options.nodeModules) !== null && _d !== void 0 ? _d : []];
            includeNodeModules = {};
            for (const mod of modules) {
                includeNodeModules[mod] = false;
            }
            if (options.nodeModules) {
                dependencies = packageJsonManager.getVersions(options.nodeModules);
            }
        }
        // Configure target in package.json for Parcel
        packageJsonManager.update({
            'cdk-lambda': `${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/index.js`,
            'targets': {
                'cdk-lambda': {
                    context: 'node',
                    includeNodeModules: includeNodeModules !== null && includeNodeModules !== void 0 ? includeNodeModules : true,
                    sourceMap: (_e = options.sourceMaps) !== null && _e !== void 0 ? _e : false,
                    minify: (_f = options.minify) !== null && _f !== void 0 ? _f : false,
                    engines: {
                        node: `>= ${runtimeVersion(options.runtime)}`,
                    },
                },
            },
        });
        // Entry file path relative to container path
        const containerEntryPath = path.join(cdk.AssetStaging.BUNDLING_INPUT_DIR, path.relative(projectRoot, path.resolve(options.entry)));
        const parcelCommand = `parcel build ${containerEntryPath.replace(/\\/g, '/')} --target cdk-lambda${options.cacheDir ? ' --cache-dir /parcel-cache' : ''}`;
        let installer = Installer.NPM;
        let lockfile;
        let depsCommand = '';
        if (dependencies) {
            // Create a dummy package.json for dependencies that we need to install
            fs.writeFileSync(path.join(projectRoot, '.package.json'), JSON.stringify({ dependencies }));
            // Use npm unless we have a yarn.lock.
            if (fs.existsSync(path.join(projectRoot, LockFile.YARN))) {
                installer = Installer.YARN;
                lockfile = LockFile.YARN;
            }
            else if (fs.existsSync(path.join(projectRoot, LockFile.NPM))) {
                lockfile = LockFile.NPM;
            }
            // Move dummy package.json and lock file then install
            depsCommand = chain([
                `mv ${cdk.AssetStaging.BUNDLING_INPUT_DIR}/.package.json ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/package.json`,
                lockfile ? `cp ${cdk.AssetStaging.BUNDLING_INPUT_DIR}/${lockfile} ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${lockfile}` : '',
                `cd ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR} && ${installer} install`,
            ]);
        }
        return lambda.Code.fromAsset(projectRoot, {
            assetHashType: cdk.AssetHashType.BUNDLE,
            bundling: {
                image,
                command: ['bash', '-c', chain([parcelCommand, depsCommand])],
                environment: options.parcelEnvironment,
                volumes: options.cacheDir
                    ? [{ containerPath: '/parcel-cache', hostPath: options.cacheDir }]
                    : [],
                workingDirectory: path.dirname(containerEntryPath).replace(/\\/g, '/'),
            },
        });
    }
}
exports.Bundling = Bundling;
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer || (Installer = {}));
var LockFile;
(function (LockFile) {
    LockFile["NPM"] = "package-lock.json";
    LockFile["YARN"] = "yarn.lock";
})(LockFile || (LockFile = {}));
function runtimeVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return match[1];
}
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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