"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ModelData = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const assets = require("aws-cdk-lib/aws-s3-assets");
const util_1 = require("./private/util");
// The only supported extension for local asset model data
// https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-sagemaker-model-containerdefinition.html#cfn-sagemaker-model-containerdefinition-modeldataurl
const ARTIFACT_EXTENSION = '.tar.gz';
/**
 * Model data represents the source of model artifacts, which will ultimately be loaded from an S3
 * location.
 */
class ModelData {
    /**
     * Constructs model data which is already available within S3.
     * @param bucket The S3 bucket within which the model artifacts are stored
     * @param objectKey The S3 object key at which the model artifacts are stored
     */
    static fromBucket(bucket, objectKey) {
        return new S3ModelData(bucket, objectKey);
    }
    /**
     * Constructs model data that will be uploaded to S3 as part of the CDK app deployment.
     * @param path The local path to a model artifact file as a gzipped tar file
     * @param options The options to further configure the selected asset
     */
    static fromAsset(path, options = {}) {
        return new AssetModelData(path, options);
    }
}
exports.ModelData = ModelData;
_a = JSII_RTTI_SYMBOL_1;
ModelData[_a] = { fqn: "@aws-cdk/aws-sagemaker-alpha.ModelData", version: "2.64.0-alpha.0" };
class S3ModelData extends ModelData {
    constructor(bucket, objectKey) {
        super();
        this.bucket = bucket;
        this.objectKey = objectKey;
    }
    bind(_scope, model) {
        this.bucket.grantRead(model);
        return {
            uri: this.bucket.urlForObject(this.objectKey),
        };
    }
}
class AssetModelData extends ModelData {
    constructor(path, options) {
        super();
        this.path = path;
        this.options = options;
        if (!path.toLowerCase().endsWith(ARTIFACT_EXTENSION)) {
            throw new Error(`Asset must be a gzipped tar file with extension ${ARTIFACT_EXTENSION} (${this.path})`);
        }
    }
    bind(scope, model) {
        // Retain the first instantiation of this asset
        if (!this.asset) {
            this.asset = new assets.Asset(scope, `ModelData${util_1.hashcode(this.path)}`, {
                path: this.path,
                ...this.options,
            });
        }
        this.asset.grantRead(model);
        return {
            uri: this.asset.httpUrl,
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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