"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodejsFunction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const bundling_1 = require("./bundling");
const util_1 = require("./util");
/**
 * A Node.js Lambda function bundled using esbuild.
 *
 * @stability stable
 */
class NodejsFunction extends lambda.Function {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b, _c, _d, _e, _f;
        if (props.runtime && props.runtime.family !== lambda.RuntimeFamily.NODEJS) {
            throw new Error('Only `NODEJS` runtimes are supported.');
        }
        // Find lock file
        let depsLockFilePath;
        if (props.depsLockFilePath) {
            if (!fs.existsSync(props.depsLockFilePath)) {
                throw new Error(`Lock file at ${props.depsLockFilePath} doesn't exist`);
            }
            if (!fs.statSync(props.depsLockFilePath).isFile()) {
                throw new Error('`depsLockFilePath` should point to a file');
            }
            depsLockFilePath = path.resolve(props.depsLockFilePath);
        }
        else {
            const lockFile = (_b = util_1.findUp(util_1.LockFile.YARN)) !== null && _b !== void 0 ? _b : util_1.findUp(util_1.LockFile.NPM);
            if (!lockFile) {
                throw new Error('Cannot find a package lock file (`yarn.lock` or `package-lock.json`). Please specify it with `depsFileLockPath`.');
            }
            depsLockFilePath = lockFile;
        }
        // Entry and defaults
        const entry = path.resolve(findEntry(id, props.entry));
        const handler = (_c = props.handler) !== null && _c !== void 0 ? _c : 'handler';
        const runtime = (_d = props.runtime) !== null && _d !== void 0 ? _d : lambda.Runtime.NODEJS_14_X;
        super(scope, id, {
            ...props,
            runtime,
            code: bundling_1.Bundling.bundle({
                ...(_e = props.bundling) !== null && _e !== void 0 ? _e : {},
                entry,
                runtime,
                depsLockFilePath,
            }),
            handler: `index.${handler}`,
        });
        // Enable connection reuse for aws-sdk
        if ((_f = props.awsSdkConnectionReuse) !== null && _f !== void 0 ? _f : true) {
            this.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });
        }
    }
}
exports.NodejsFunction = NodejsFunction;
_a = JSII_RTTI_SYMBOL_1;
NodejsFunction[_a] = { fqn: "@aws-cdk/aws-lambda-nodejs.NodejsFunction", version: "1.103.0" };
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .ts file named as the defining file with id as suffix (defining-file.id.ts)
 * 3. A .js file name as the defining file with id as suffix (defining-file.id.js)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(jsx?|tsx?)$/.test(entry)) {
            throw new Error('Only JavaScript or TypeScript entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const tsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.ts`);
    if (fs.existsSync(tsHandlerFile)) {
        return tsHandlerFile;
    }
    const jsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.js`);
    if (fs.existsSync(jsHandlerFile)) {
        return jsHandlerFile;
    }
    throw new Error('Cannot find entry file.');
}
/**
 * Finds the name of the file where the `NodejsFunction` is defined
 */
function findDefiningFile() {
    let definingIndex;
    const sites = util_1.callsites();
    for (const [index, site] of sites.entries()) {
        if (site.getFunctionName() === 'NodejsFunction') {
            // The next site is the site where the NodejsFunction was created
            definingIndex = index + 1;
            break;
        }
    }
    if (!definingIndex || !sites[definingIndex]) {
        throw new Error('Cannot find defining file.');
    }
    return sites[definingIndex].getFileName();
}
//# sourceMappingURL=data:application/json;base64,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