"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * A Condition for use in a Choice state branch
 */
class Condition {
    /**
     * Matches if a boolean field has the given value
     */
    static booleanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.BooleanEquals, value);
    }
    /**
     * Matches if a string field has the given value
     */
    static stringEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringEquals, value);
    }
    /**
     * Matches if a string field sorts before a given value
     */
    static stringLessThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringLessThan, value);
    }
    /**
     * Matches if a string field sorts equal to or before a given value
     */
    static stringLessThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringLessThanEquals, value);
    }
    /**
     * Matches if a string field sorts after a given value
     */
    static stringGreaterThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringGreaterThan, value);
    }
    /**
     * Matches if a string field sorts after or equal to a given value
     */
    static stringGreaterThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringGreaterThanEquals, value);
    }
    /**
     * Matches if a numeric field has the given value
     */
    static numberEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericEquals, value);
    }
    /**
     * Matches if a numeric field is less than the given value
     */
    static numberLessThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericLessThan, value);
    }
    /**
     * Matches if a numeric field is less than or equal to the given value
     */
    static numberLessThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericLessThanEquals, value);
    }
    /**
     * Matches if a numeric field is greater than the given value
     */
    static numberGreaterThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericGreaterThan, value);
    }
    /**
     * Matches if a numeric field is greater than or equal to the given value
     */
    static numberGreaterThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericGreaterThanEquals, value);
    }
    /**
     * Matches if a timestamp field is the same time as the given timestamp
     */
    static timestampEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampEquals, value);
    }
    /**
     * Matches if a timestamp field is before the given timestamp
     */
    static timestampLessThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampLessThan, value);
    }
    /**
     * Matches if a timestamp field is before or equal to the given timestamp
     */
    static timestampLessThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampLessThanEquals, value);
    }
    /**
     * Matches if a timestamp field is after the given timestamp
     */
    static timestampGreaterThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampGreaterThan, value);
    }
    /**
     * Matches if a timestamp field is after or equal to the given timestamp
     */
    static timestampGreaterThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampGreaterThanEquals, value);
    }
    /**
     * Combine two or more conditions with a logical AND
     */
    static and(...conditions) {
        return new CompoundCondition(CompoundOperator.And, ...conditions);
    }
    /**
     * Combine two or more conditions with a logical OR
     */
    static or(...conditions) {
        return new CompoundCondition(CompoundOperator.Or, ...conditions);
    }
    /**
     * Negate a condition
     */
    static not(condition) {
        return new NotCondition(condition);
    }
}
exports.Condition = Condition;
/**
 * Comparison Operator types
 */
var ComparisonOperator;
(function (ComparisonOperator) {
    ComparisonOperator[ComparisonOperator["StringEquals"] = 0] = "StringEquals";
    ComparisonOperator[ComparisonOperator["StringLessThan"] = 1] = "StringLessThan";
    ComparisonOperator[ComparisonOperator["StringGreaterThan"] = 2] = "StringGreaterThan";
    ComparisonOperator[ComparisonOperator["StringLessThanEquals"] = 3] = "StringLessThanEquals";
    ComparisonOperator[ComparisonOperator["StringGreaterThanEquals"] = 4] = "StringGreaterThanEquals";
    ComparisonOperator[ComparisonOperator["NumericEquals"] = 5] = "NumericEquals";
    ComparisonOperator[ComparisonOperator["NumericLessThan"] = 6] = "NumericLessThan";
    ComparisonOperator[ComparisonOperator["NumericGreaterThan"] = 7] = "NumericGreaterThan";
    ComparisonOperator[ComparisonOperator["NumericLessThanEquals"] = 8] = "NumericLessThanEquals";
    ComparisonOperator[ComparisonOperator["NumericGreaterThanEquals"] = 9] = "NumericGreaterThanEquals";
    ComparisonOperator[ComparisonOperator["BooleanEquals"] = 10] = "BooleanEquals";
    ComparisonOperator[ComparisonOperator["TimestampEquals"] = 11] = "TimestampEquals";
    ComparisonOperator[ComparisonOperator["TimestampLessThan"] = 12] = "TimestampLessThan";
    ComparisonOperator[ComparisonOperator["TimestampGreaterThan"] = 13] = "TimestampGreaterThan";
    ComparisonOperator[ComparisonOperator["TimestampLessThanEquals"] = 14] = "TimestampLessThanEquals";
    ComparisonOperator[ComparisonOperator["TimestampGreaterThanEquals"] = 15] = "TimestampGreaterThanEquals";
})(ComparisonOperator || (ComparisonOperator = {}));
/**
 * Compound Operator types
 */
var CompoundOperator;
(function (CompoundOperator) {
    CompoundOperator[CompoundOperator["And"] = 0] = "And";
    CompoundOperator[CompoundOperator["Or"] = 1] = "Or";
})(CompoundOperator || (CompoundOperator = {}));
/**
 * Scalar comparison
 */
class VariableComparison extends Condition {
    constructor(variable, comparisonOperator, value) {
        super();
        this.variable = variable;
        this.comparisonOperator = comparisonOperator;
        this.value = value;
        if (!/^\$[.[]/.test(variable)) {
            throw new Error(`Variable reference must start with '$.' or '$[', got '${variable}'`);
        }
    }
    renderCondition() {
        return {
            Variable: this.variable,
            [ComparisonOperator[this.comparisonOperator]]: this.value,
        };
    }
}
/**
 * Logical compound condition
 */
class CompoundCondition extends Condition {
    constructor(operator, ...conditions) {
        super();
        this.operator = operator;
        this.conditions = conditions;
        if (conditions.length === 0) {
            throw new Error('Must supply at least one inner condition for a logical combination');
        }
    }
    renderCondition() {
        return {
            [CompoundOperator[this.operator]]: this.conditions.map(c => c.renderCondition()),
        };
    }
}
/**
 * Logical unary condition
 */
class NotCondition extends Condition {
    constructor(comparisonOperation) {
        super();
        this.comparisonOperation = comparisonOperation;
    }
    renderCondition() {
        return {
            Not: this.comparisonOperation.renderCondition(),
        };
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29uZGl0aW9uLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY29uZGl0aW9uLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUE7O0dBRUc7QUFDSCxNQUFzQixTQUFTO0lBQzdCOztPQUVHO0lBQ0ksTUFBTSxDQUFDLGFBQWEsQ0FBQyxRQUFnQixFQUFFLEtBQWM7UUFDMUQsT0FBTyxJQUFJLGtCQUFrQixDQUFDLFFBQVEsRUFBRSxrQkFBa0IsQ0FBQyxhQUFhLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDbkYsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLFlBQVksQ0FBQyxRQUFnQixFQUFFLEtBQWE7UUFDeEQsT0FBTyxJQUFJLGtCQUFrQixDQUFDLFFBQVEsRUFBRSxrQkFBa0IsQ0FBQyxZQUFZLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDbEYsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLGNBQWMsQ0FBQyxRQUFnQixFQUFFLEtBQWE7UUFDMUQsT0FBTyxJQUFJLGtCQUFrQixDQUFDLFFBQVEsRUFBRSxrQkFBa0IsQ0FBQyxjQUFjLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDcEYsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLG9CQUFvQixDQUFDLFFBQWdCLEVBQUUsS0FBYTtRQUNoRSxPQUFPLElBQUksa0JBQWtCLENBQUMsUUFBUSxFQUFFLGtCQUFrQixDQUFDLG9CQUFvQixFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQzFGLENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxRQUFnQixFQUFFLEtBQWE7UUFDN0QsT0FBTyxJQUFJLGtCQUFrQixDQUFDLFFBQVEsRUFBRSxrQkFBa0IsQ0FBQyxpQkFBaUIsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUN2RixDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsdUJBQXVCLENBQUMsUUFBZ0IsRUFBRSxLQUFhO1FBQ25FLE9BQU8sSUFBSSxrQkFBa0IsQ0FBQyxRQUFRLEVBQUUsa0JBQWtCLENBQUMsdUJBQXVCLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDN0YsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLFlBQVksQ0FBQyxRQUFnQixFQUFFLEtBQWE7UUFDeEQsT0FBTyxJQUFJLGtCQUFrQixDQUFDLFFBQVEsRUFBRSxrQkFBa0IsQ0FBQyxhQUFhLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDbkYsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLGNBQWMsQ0FBQyxRQUFnQixFQUFFLEtBQWE7UUFDMUQsT0FBTyxJQUFJLGtCQUFrQixDQUFDLFFBQVEsRUFBRSxrQkFBa0IsQ0FBQyxlQUFlLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDckYsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLG9CQUFvQixDQUFDLFFBQWdCLEVBQUUsS0FBYTtRQUNoRSxPQUFPLElBQUksa0JBQWtCLENBQUMsUUFBUSxFQUFFLGtCQUFrQixDQUFDLHFCQUFxQixFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQzNGLENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxRQUFnQixFQUFFLEtBQWE7UUFDN0QsT0FBTyxJQUFJLGtCQUFrQixDQUFDLFFBQVEsRUFBRSxrQkFBa0IsQ0FBQyxrQkFBa0IsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUN4RixDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsdUJBQXVCLENBQUMsUUFBZ0IsRUFBRSxLQUFhO1FBQ25FLE9BQU8sSUFBSSxrQkFBa0IsQ0FBQyxRQUFRLEVBQUUsa0JBQWtCLENBQUMsd0JBQXdCLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDOUYsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLGVBQWUsQ0FBQyxRQUFnQixFQUFFLEtBQWE7UUFDM0QsT0FBTyxJQUFJLGtCQUFrQixDQUFDLFFBQVEsRUFBRSxrQkFBa0IsQ0FBQyxlQUFlLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDckYsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLGlCQUFpQixDQUFDLFFBQWdCLEVBQUUsS0FBYTtRQUM3RCxPQUFPLElBQUksa0JBQWtCLENBQUMsUUFBUSxFQUFFLGtCQUFrQixDQUFDLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ3ZGLENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyx1QkFBdUIsQ0FBQyxRQUFnQixFQUFFLEtBQWE7UUFDbkUsT0FBTyxJQUFJLGtCQUFrQixDQUFDLFFBQVEsRUFBRSxrQkFBa0IsQ0FBQyx1QkFBdUIsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUM3RixDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsb0JBQW9CLENBQUMsUUFBZ0IsRUFBRSxLQUFhO1FBQ2hFLE9BQU8sSUFBSSxrQkFBa0IsQ0FBQyxRQUFRLEVBQUUsa0JBQWtCLENBQUMsb0JBQW9CLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDMUYsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLDBCQUEwQixDQUFDLFFBQWdCLEVBQUUsS0FBYTtRQUN0RSxPQUFPLElBQUksa0JBQWtCLENBQUMsUUFBUSxFQUFFLGtCQUFrQixDQUFDLDBCQUEwQixFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ2hHLENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxHQUFHLENBQUMsR0FBRyxVQUF1QjtRQUMxQyxPQUFPLElBQUksaUJBQWlCLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxFQUFFLEdBQUcsVUFBVSxDQUFDLENBQUM7SUFDcEUsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLEVBQUUsQ0FBQyxHQUFHLFVBQXVCO1FBQ3pDLE9BQU8sSUFBSSxpQkFBaUIsQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFLEVBQUUsR0FBRyxVQUFVLENBQUMsQ0FBQztJQUNuRSxDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsR0FBRyxDQUFDLFNBQW9CO1FBQ3BDLE9BQU8sSUFBSSxZQUFZLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDckMsQ0FBQztDQU1GO0FBMUlELDhCQTBJQztBQUVEOztHQUVHO0FBQ0gsSUFBSyxrQkFpQko7QUFqQkQsV0FBSyxrQkFBa0I7SUFDckIsMkVBQVksQ0FBQTtJQUNaLCtFQUFjLENBQUE7SUFDZCxxRkFBaUIsQ0FBQTtJQUNqQiwyRkFBb0IsQ0FBQTtJQUNwQixpR0FBdUIsQ0FBQTtJQUN2Qiw2RUFBYSxDQUFBO0lBQ2IsaUZBQWUsQ0FBQTtJQUNmLHVGQUFrQixDQUFBO0lBQ2xCLDZGQUFxQixDQUFBO0lBQ3JCLG1HQUF3QixDQUFBO0lBQ3hCLDhFQUFhLENBQUE7SUFDYixrRkFBZSxDQUFBO0lBQ2Ysc0ZBQWlCLENBQUE7SUFDakIsNEZBQW9CLENBQUE7SUFDcEIsa0dBQXVCLENBQUE7SUFDdkIsd0dBQTBCLENBQUE7QUFDNUIsQ0FBQyxFQWpCSSxrQkFBa0IsS0FBbEIsa0JBQWtCLFFBaUJ0QjtBQUVEOztHQUVHO0FBQ0gsSUFBSyxnQkFHSjtBQUhELFdBQUssZ0JBQWdCO0lBQ25CLHFEQUFHLENBQUE7SUFDSCxtREFBRSxDQUFBO0FBQ0osQ0FBQyxFQUhJLGdCQUFnQixLQUFoQixnQkFBZ0IsUUFHcEI7QUFFRDs7R0FFRztBQUNILE1BQU0sa0JBQW1CLFNBQVEsU0FBUztJQUN4QyxZQUE2QixRQUFnQixFQUFtQixrQkFBc0MsRUFBbUIsS0FBVTtRQUNqSSxLQUFLLEVBQUUsQ0FBQztRQURtQixhQUFRLEdBQVIsUUFBUSxDQUFRO1FBQW1CLHVCQUFrQixHQUFsQixrQkFBa0IsQ0FBb0I7UUFBbUIsVUFBSyxHQUFMLEtBQUssQ0FBSztRQUVqSSxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRTtZQUM3QixNQUFNLElBQUksS0FBSyxDQUFDLHlEQUF5RCxRQUFRLEdBQUcsQ0FBQyxDQUFDO1NBQ3ZGO0lBQ0gsQ0FBQztJQUVNLGVBQWU7UUFDcEIsT0FBTztZQUNMLFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUTtZQUN2QixDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLEVBQUUsSUFBSSxDQUFDLEtBQUs7U0FDMUQsQ0FBQztJQUNKLENBQUM7Q0FDRjtBQUVEOztHQUVHO0FBQ0gsTUFBTSxpQkFBa0IsU0FBUSxTQUFTO0lBR3ZDLFlBQTZCLFFBQTBCLEVBQUUsR0FBRyxVQUF1QjtRQUNqRixLQUFLLEVBQUUsQ0FBQztRQURtQixhQUFRLEdBQVIsUUFBUSxDQUFrQjtRQUVyRCxJQUFJLENBQUMsVUFBVSxHQUFHLFVBQVUsQ0FBQztRQUM3QixJQUFJLFVBQVUsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzNCLE1BQU0sSUFBSSxLQUFLLENBQUMsb0VBQW9FLENBQUMsQ0FBQztTQUN2RjtJQUNILENBQUM7SUFFTSxlQUFlO1FBQ3BCLE9BQU87WUFDTCxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLGVBQWUsRUFBRSxDQUFDO1NBQ2pGLENBQUM7SUFDSixDQUFDO0NBQ0Y7QUFFRDs7R0FFRztBQUNILE1BQU0sWUFBYSxTQUFRLFNBQVM7SUFDbEMsWUFBNkIsbUJBQThCO1FBQ3pELEtBQUssRUFBRSxDQUFDO1FBRG1CLHdCQUFtQixHQUFuQixtQkFBbUIsQ0FBVztJQUUzRCxDQUFDO0lBRU0sZUFBZTtRQUNwQixPQUFPO1lBQ0wsR0FBRyxFQUFFLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxlQUFlLEVBQUU7U0FDaEQsQ0FBQztJQUNKLENBQUM7Q0FDRiIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQSBDb25kaXRpb24gZm9yIHVzZSBpbiBhIENob2ljZSBzdGF0ZSBicmFuY2hcbiAqL1xuZXhwb3J0IGFic3RyYWN0IGNsYXNzIENvbmRpdGlvbiB7XG4gIC8qKlxuICAgKiBNYXRjaGVzIGlmIGEgYm9vbGVhbiBmaWVsZCBoYXMgdGhlIGdpdmVuIHZhbHVlXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGJvb2xlYW5FcXVhbHModmFyaWFibGU6IHN0cmluZywgdmFsdWU6IGJvb2xlYW4pOiBDb25kaXRpb24ge1xuICAgIHJldHVybiBuZXcgVmFyaWFibGVDb21wYXJpc29uKHZhcmlhYmxlLCBDb21wYXJpc29uT3BlcmF0b3IuQm9vbGVhbkVxdWFscywgdmFsdWUpO1xuICB9XG5cbiAgLyoqXG4gICAqIE1hdGNoZXMgaWYgYSBzdHJpbmcgZmllbGQgaGFzIHRoZSBnaXZlbiB2YWx1ZVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBzdHJpbmdFcXVhbHModmFyaWFibGU6IHN0cmluZywgdmFsdWU6IHN0cmluZyk6IENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIG5ldyBWYXJpYWJsZUNvbXBhcmlzb24odmFyaWFibGUsIENvbXBhcmlzb25PcGVyYXRvci5TdHJpbmdFcXVhbHMsIHZhbHVlKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBNYXRjaGVzIGlmIGEgc3RyaW5nIGZpZWxkIHNvcnRzIGJlZm9yZSBhIGdpdmVuIHZhbHVlXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHN0cmluZ0xlc3NUaGFuKHZhcmlhYmxlOiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcpOiBDb25kaXRpb24ge1xuICAgIHJldHVybiBuZXcgVmFyaWFibGVDb21wYXJpc29uKHZhcmlhYmxlLCBDb21wYXJpc29uT3BlcmF0b3IuU3RyaW5nTGVzc1RoYW4sIHZhbHVlKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBNYXRjaGVzIGlmIGEgc3RyaW5nIGZpZWxkIHNvcnRzIGVxdWFsIHRvIG9yIGJlZm9yZSBhIGdpdmVuIHZhbHVlXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHN0cmluZ0xlc3NUaGFuRXF1YWxzKHZhcmlhYmxlOiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcpOiBDb25kaXRpb24ge1xuICAgIHJldHVybiBuZXcgVmFyaWFibGVDb21wYXJpc29uKHZhcmlhYmxlLCBDb21wYXJpc29uT3BlcmF0b3IuU3RyaW5nTGVzc1RoYW5FcXVhbHMsIHZhbHVlKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBNYXRjaGVzIGlmIGEgc3RyaW5nIGZpZWxkIHNvcnRzIGFmdGVyIGEgZ2l2ZW4gdmFsdWVcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgc3RyaW5nR3JlYXRlclRoYW4odmFyaWFibGU6IHN0cmluZywgdmFsdWU6IHN0cmluZyk6IENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIG5ldyBWYXJpYWJsZUNvbXBhcmlzb24odmFyaWFibGUsIENvbXBhcmlzb25PcGVyYXRvci5TdHJpbmdHcmVhdGVyVGhhbiwgdmFsdWUpO1xuICB9XG5cbiAgLyoqXG4gICAqIE1hdGNoZXMgaWYgYSBzdHJpbmcgZmllbGQgc29ydHMgYWZ0ZXIgb3IgZXF1YWwgdG8gYSBnaXZlbiB2YWx1ZVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBzdHJpbmdHcmVhdGVyVGhhbkVxdWFscyh2YXJpYWJsZTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nKTogQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gbmV3IFZhcmlhYmxlQ29tcGFyaXNvbih2YXJpYWJsZSwgQ29tcGFyaXNvbk9wZXJhdG9yLlN0cmluZ0dyZWF0ZXJUaGFuRXF1YWxzLCB2YWx1ZSk7XG4gIH1cblxuICAvKipcbiAgICogTWF0Y2hlcyBpZiBhIG51bWVyaWMgZmllbGQgaGFzIHRoZSBnaXZlbiB2YWx1ZVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBudW1iZXJFcXVhbHModmFyaWFibGU6IHN0cmluZywgdmFsdWU6IG51bWJlcik6IENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIG5ldyBWYXJpYWJsZUNvbXBhcmlzb24odmFyaWFibGUsIENvbXBhcmlzb25PcGVyYXRvci5OdW1lcmljRXF1YWxzLCB2YWx1ZSk7XG4gIH1cblxuICAvKipcbiAgICogTWF0Y2hlcyBpZiBhIG51bWVyaWMgZmllbGQgaXMgbGVzcyB0aGFuIHRoZSBnaXZlbiB2YWx1ZVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBudW1iZXJMZXNzVGhhbih2YXJpYWJsZTogc3RyaW5nLCB2YWx1ZTogbnVtYmVyKTogQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gbmV3IFZhcmlhYmxlQ29tcGFyaXNvbih2YXJpYWJsZSwgQ29tcGFyaXNvbk9wZXJhdG9yLk51bWVyaWNMZXNzVGhhbiwgdmFsdWUpO1xuICB9XG5cbiAgLyoqXG4gICAqIE1hdGNoZXMgaWYgYSBudW1lcmljIGZpZWxkIGlzIGxlc3MgdGhhbiBvciBlcXVhbCB0byB0aGUgZ2l2ZW4gdmFsdWVcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgbnVtYmVyTGVzc1RoYW5FcXVhbHModmFyaWFibGU6IHN0cmluZywgdmFsdWU6IG51bWJlcik6IENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIG5ldyBWYXJpYWJsZUNvbXBhcmlzb24odmFyaWFibGUsIENvbXBhcmlzb25PcGVyYXRvci5OdW1lcmljTGVzc1RoYW5FcXVhbHMsIHZhbHVlKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBNYXRjaGVzIGlmIGEgbnVtZXJpYyBmaWVsZCBpcyBncmVhdGVyIHRoYW4gdGhlIGdpdmVuIHZhbHVlXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIG51bWJlckdyZWF0ZXJUaGFuKHZhcmlhYmxlOiBzdHJpbmcsIHZhbHVlOiBudW1iZXIpOiBDb25kaXRpb24ge1xuICAgIHJldHVybiBuZXcgVmFyaWFibGVDb21wYXJpc29uKHZhcmlhYmxlLCBDb21wYXJpc29uT3BlcmF0b3IuTnVtZXJpY0dyZWF0ZXJUaGFuLCB2YWx1ZSk7XG4gIH1cblxuICAvKipcbiAgICogTWF0Y2hlcyBpZiBhIG51bWVyaWMgZmllbGQgaXMgZ3JlYXRlciB0aGFuIG9yIGVxdWFsIHRvIHRoZSBnaXZlbiB2YWx1ZVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBudW1iZXJHcmVhdGVyVGhhbkVxdWFscyh2YXJpYWJsZTogc3RyaW5nLCB2YWx1ZTogbnVtYmVyKTogQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gbmV3IFZhcmlhYmxlQ29tcGFyaXNvbih2YXJpYWJsZSwgQ29tcGFyaXNvbk9wZXJhdG9yLk51bWVyaWNHcmVhdGVyVGhhbkVxdWFscywgdmFsdWUpO1xuICB9XG5cbiAgLyoqXG4gICAqIE1hdGNoZXMgaWYgYSB0aW1lc3RhbXAgZmllbGQgaXMgdGhlIHNhbWUgdGltZSBhcyB0aGUgZ2l2ZW4gdGltZXN0YW1wXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHRpbWVzdGFtcEVxdWFscyh2YXJpYWJsZTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nKTogQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gbmV3IFZhcmlhYmxlQ29tcGFyaXNvbih2YXJpYWJsZSwgQ29tcGFyaXNvbk9wZXJhdG9yLlRpbWVzdGFtcEVxdWFscywgdmFsdWUpO1xuICB9XG5cbiAgLyoqXG4gICAqIE1hdGNoZXMgaWYgYSB0aW1lc3RhbXAgZmllbGQgaXMgYmVmb3JlIHRoZSBnaXZlbiB0aW1lc3RhbXBcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgdGltZXN0YW1wTGVzc1RoYW4odmFyaWFibGU6IHN0cmluZywgdmFsdWU6IHN0cmluZyk6IENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIG5ldyBWYXJpYWJsZUNvbXBhcmlzb24odmFyaWFibGUsIENvbXBhcmlzb25PcGVyYXRvci5UaW1lc3RhbXBMZXNzVGhhbiwgdmFsdWUpO1xuICB9XG5cbiAgLyoqXG4gICAqIE1hdGNoZXMgaWYgYSB0aW1lc3RhbXAgZmllbGQgaXMgYmVmb3JlIG9yIGVxdWFsIHRvIHRoZSBnaXZlbiB0aW1lc3RhbXBcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgdGltZXN0YW1wTGVzc1RoYW5FcXVhbHModmFyaWFibGU6IHN0cmluZywgdmFsdWU6IHN0cmluZyk6IENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIG5ldyBWYXJpYWJsZUNvbXBhcmlzb24odmFyaWFibGUsIENvbXBhcmlzb25PcGVyYXRvci5UaW1lc3RhbXBMZXNzVGhhbkVxdWFscywgdmFsdWUpO1xuICB9XG5cbiAgLyoqXG4gICAqIE1hdGNoZXMgaWYgYSB0aW1lc3RhbXAgZmllbGQgaXMgYWZ0ZXIgdGhlIGdpdmVuIHRpbWVzdGFtcFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyB0aW1lc3RhbXBHcmVhdGVyVGhhbih2YXJpYWJsZTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nKTogQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gbmV3IFZhcmlhYmxlQ29tcGFyaXNvbih2YXJpYWJsZSwgQ29tcGFyaXNvbk9wZXJhdG9yLlRpbWVzdGFtcEdyZWF0ZXJUaGFuLCB2YWx1ZSk7XG4gIH1cblxuICAvKipcbiAgICogTWF0Y2hlcyBpZiBhIHRpbWVzdGFtcCBmaWVsZCBpcyBhZnRlciBvciBlcXVhbCB0byB0aGUgZ2l2ZW4gdGltZXN0YW1wXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHRpbWVzdGFtcEdyZWF0ZXJUaGFuRXF1YWxzKHZhcmlhYmxlOiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcpOiBDb25kaXRpb24ge1xuICAgIHJldHVybiBuZXcgVmFyaWFibGVDb21wYXJpc29uKHZhcmlhYmxlLCBDb21wYXJpc29uT3BlcmF0b3IuVGltZXN0YW1wR3JlYXRlclRoYW5FcXVhbHMsIHZhbHVlKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb21iaW5lIHR3byBvciBtb3JlIGNvbmRpdGlvbnMgd2l0aCBhIGxvZ2ljYWwgQU5EXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGFuZCguLi5jb25kaXRpb25zOiBDb25kaXRpb25bXSk6IENvbmRpdGlvbiB7XG4gICAgcmV0dXJuIG5ldyBDb21wb3VuZENvbmRpdGlvbihDb21wb3VuZE9wZXJhdG9yLkFuZCwgLi4uY29uZGl0aW9ucyk7XG4gIH1cblxuICAvKipcbiAgICogQ29tYmluZSB0d28gb3IgbW9yZSBjb25kaXRpb25zIHdpdGggYSBsb2dpY2FsIE9SXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIG9yKC4uLmNvbmRpdGlvbnM6IENvbmRpdGlvbltdKTogQ29uZGl0aW9uIHtcbiAgICByZXR1cm4gbmV3IENvbXBvdW5kQ29uZGl0aW9uKENvbXBvdW5kT3BlcmF0b3IuT3IsIC4uLmNvbmRpdGlvbnMpO1xuICB9XG5cbiAgLyoqXG4gICAqIE5lZ2F0ZSBhIGNvbmRpdGlvblxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBub3QoY29uZGl0aW9uOiBDb25kaXRpb24pOiBDb25kaXRpb24ge1xuICAgIHJldHVybiBuZXcgTm90Q29uZGl0aW9uKGNvbmRpdGlvbik7XG4gIH1cblxuICAvKipcbiAgICogUmVuZGVyIEFtYXpvbiBTdGF0ZXMgTGFuZ3VhZ2UgSlNPTiBmb3IgdGhlIGNvbmRpdGlvblxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlbmRlckNvbmRpdGlvbigpOiBhbnk7XG59XG5cbi8qKlxuICogQ29tcGFyaXNvbiBPcGVyYXRvciB0eXBlc1xuICovXG5lbnVtIENvbXBhcmlzb25PcGVyYXRvciB7XG4gIFN0cmluZ0VxdWFscyxcbiAgU3RyaW5nTGVzc1RoYW4sXG4gIFN0cmluZ0dyZWF0ZXJUaGFuLFxuICBTdHJpbmdMZXNzVGhhbkVxdWFscyxcbiAgU3RyaW5nR3JlYXRlclRoYW5FcXVhbHMsXG4gIE51bWVyaWNFcXVhbHMsXG4gIE51bWVyaWNMZXNzVGhhbixcbiAgTnVtZXJpY0dyZWF0ZXJUaGFuLFxuICBOdW1lcmljTGVzc1RoYW5FcXVhbHMsXG4gIE51bWVyaWNHcmVhdGVyVGhhbkVxdWFscyxcbiAgQm9vbGVhbkVxdWFscyxcbiAgVGltZXN0YW1wRXF1YWxzLFxuICBUaW1lc3RhbXBMZXNzVGhhbixcbiAgVGltZXN0YW1wR3JlYXRlclRoYW4sXG4gIFRpbWVzdGFtcExlc3NUaGFuRXF1YWxzLFxuICBUaW1lc3RhbXBHcmVhdGVyVGhhbkVxdWFscyxcbn1cblxuLyoqXG4gKiBDb21wb3VuZCBPcGVyYXRvciB0eXBlc1xuICovXG5lbnVtIENvbXBvdW5kT3BlcmF0b3Ige1xuICBBbmQsXG4gIE9yLFxufVxuXG4vKipcbiAqIFNjYWxhciBjb21wYXJpc29uXG4gKi9cbmNsYXNzIFZhcmlhYmxlQ29tcGFyaXNvbiBleHRlbmRzIENvbmRpdGlvbiB7XG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgdmFyaWFibGU6IHN0cmluZywgcHJpdmF0ZSByZWFkb25seSBjb21wYXJpc29uT3BlcmF0b3I6IENvbXBhcmlzb25PcGVyYXRvciwgcHJpdmF0ZSByZWFkb25seSB2YWx1ZTogYW55KSB7XG4gICAgc3VwZXIoKTtcbiAgICBpZiAoIS9eXFwkWy5bXS8udGVzdCh2YXJpYWJsZSkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgVmFyaWFibGUgcmVmZXJlbmNlIG11c3Qgc3RhcnQgd2l0aCAnJC4nIG9yICckWycsIGdvdCAnJHt2YXJpYWJsZX0nYCk7XG4gICAgfVxuICB9XG5cbiAgcHVibGljIHJlbmRlckNvbmRpdGlvbigpOiBhbnkge1xuICAgIHJldHVybiB7XG4gICAgICBWYXJpYWJsZTogdGhpcy52YXJpYWJsZSxcbiAgICAgIFtDb21wYXJpc29uT3BlcmF0b3JbdGhpcy5jb21wYXJpc29uT3BlcmF0b3JdXTogdGhpcy52YWx1ZSxcbiAgICB9O1xuICB9XG59XG5cbi8qKlxuICogTG9naWNhbCBjb21wb3VuZCBjb25kaXRpb25cbiAqL1xuY2xhc3MgQ29tcG91bmRDb25kaXRpb24gZXh0ZW5kcyBDb25kaXRpb24ge1xuICBwcml2YXRlIHJlYWRvbmx5IGNvbmRpdGlvbnM6IENvbmRpdGlvbltdO1xuXG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgb3BlcmF0b3I6IENvbXBvdW5kT3BlcmF0b3IsIC4uLmNvbmRpdGlvbnM6IENvbmRpdGlvbltdKSB7XG4gICAgc3VwZXIoKTtcbiAgICB0aGlzLmNvbmRpdGlvbnMgPSBjb25kaXRpb25zO1xuICAgIGlmIChjb25kaXRpb25zLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdNdXN0IHN1cHBseSBhdCBsZWFzdCBvbmUgaW5uZXIgY29uZGl0aW9uIGZvciBhIGxvZ2ljYWwgY29tYmluYXRpb24nKTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgcmVuZGVyQ29uZGl0aW9uKCk6IGFueSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIFtDb21wb3VuZE9wZXJhdG9yW3RoaXMub3BlcmF0b3JdXTogdGhpcy5jb25kaXRpb25zLm1hcChjID0+IGMucmVuZGVyQ29uZGl0aW9uKCkpLFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4gKiBMb2dpY2FsIHVuYXJ5IGNvbmRpdGlvblxuICovXG5jbGFzcyBOb3RDb25kaXRpb24gZXh0ZW5kcyBDb25kaXRpb24ge1xuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IGNvbXBhcmlzb25PcGVyYXRpb246IENvbmRpdGlvbikge1xuICAgIHN1cGVyKCk7XG4gIH1cblxuICBwdWJsaWMgcmVuZGVyQ29uZGl0aW9uKCk6IGFueSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIE5vdDogdGhpcy5jb21wYXJpc29uT3BlcmF0aW9uLnJlbmRlckNvbmRpdGlvbigpLFxuICAgIH07XG4gIH1cbn1cbiJdfQ==