"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomLambdaDeploymentConfig = exports.CustomLambdaDeploymentConfigType = void 0;
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const utils_1 = require("../utils");
/**
 * Lambda Deployment config type.
 */
var CustomLambdaDeploymentConfigType;
(function (CustomLambdaDeploymentConfigType) {
    CustomLambdaDeploymentConfigType["CANARY"] = "Canary";
    CustomLambdaDeploymentConfigType["LINEAR"] = "Linear";
})(CustomLambdaDeploymentConfigType = exports.CustomLambdaDeploymentConfigType || (exports.CustomLambdaDeploymentConfigType = {}));
/**
 * A custom Deployment Configuration for a Lambda Deployment Group.
 *
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class CustomLambdaDeploymentConfig extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.validateParameters(props);
        // In this section we make the argument for the AWS API call
        const deploymentType = 'TimeBased' + props.type.toString();
        const intervalMinutes = props.interval.toMinutes().toString();
        const percentage = props.percentage.toString();
        let routingConfig; // The argument to the AWS API call
        if (props.type == CustomLambdaDeploymentConfigType.CANARY) {
            routingConfig = {
                type: deploymentType,
                timeBasedCanary: {
                    canaryInterval: intervalMinutes,
                    canaryPercentage: percentage,
                },
            };
        }
        else if (props.type == CustomLambdaDeploymentConfigType.LINEAR) {
            routingConfig = {
                type: deploymentType,
                timeBasedLinear: {
                    linearInterval: intervalMinutes,
                    linearPercentage: percentage,
                },
            };
        }
        // Generates the name of the deployment config. It's also what you'll see in the AWS console
        // The name of the config is <construct unique id>.Lambda<deployment type><percentage>Percent<interval>Minutes
        // Unless the user provides an explicit name
        this.deploymentConfigName = props.deploymentConfigName !== undefined
            ? props.deploymentConfigName
            : `${core_1.Names.uniqueId(this)}.Lambda${props.type}${props.percentage}Percent${props.type === CustomLambdaDeploymentConfigType.LINEAR
                ? 'Every'
                : ''}${props.interval.toMinutes()}Minutes`;
        this.deploymentConfigArn = utils_1.arnForDeploymentConfig(this.deploymentConfigName);
        // The AWS Custom Resource that calls CodeDeploy to create and delete the resource
        new custom_resources_1.AwsCustomResource(this, 'DeploymentConfig', {
            onCreate: {
                service: 'CodeDeploy',
                action: 'createDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                    computePlatform: 'Lambda',
                    trafficRoutingConfig: routingConfig,
                },
                // This `resourceName` is the initial physical ID of the config
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.deploymentConfigName),
            },
            onUpdate: {
                service: 'CodeDeploy',
                action: 'createDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                    computePlatform: 'Lambda',
                    trafficRoutingConfig: routingConfig,
                },
                // If `resourceName` is different from the last stack update (or creation),
                // the old config gets deleted and the new one is created
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.deploymentConfigName),
            },
            onDelete: {
                service: 'CodeDeploy',
                action: 'deleteDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                },
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
    }
    // Validate the inputs. The percentage/interval limits come from CodeDeploy
    validateParameters(props) {
        if (!(1 <= props.percentage && props.percentage <= 99)) {
            throw new Error(`Invalid deployment config percentage "${props.percentage.toString()}". \
        Step percentage must be an integer between 1 and 99.`);
        }
        if (props.interval.toMinutes() > 2880) {
            throw new Error(`Invalid deployment config interval "${props.interval.toString()}". \
        Traffic shifting intervals must be positive integers up to 2880 (2 days).`);
        }
    }
}
exports.CustomLambdaDeploymentConfig = CustomLambdaDeploymentConfig;
//# sourceMappingURL=data:application/json;base64,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