"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const s3 = require("aws-cdk-lib/aws-s3");
const lib_1 = require("../lib");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test Case 2 - Test the getter methods
// --------------------------------------------------------------
test('Test properties', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        }
    };
    const app = new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'test-kinesis-firehose-kinesis-analytics', props);
    // Assertions
    expect(app.kinesisAnalytics !== null);
    expect(app.kinesisFirehose !== null);
    expect(app.kinesisFirehoseRole !== null);
    expect(app.kinesisFirehoseLogGroup !== null);
    expect(app.s3Bucket !== null);
    expect(app.s3LoggingBucket !== null);
});
// --------------------------------------------------------------
// Test Case 3 - Override kinesisFirehose props
// --------------------------------------------------------------
test('test kinesisFirehose override ', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'test-firehose-s3', {
        kinesisFirehoseProps: {
            extendedS3DestinationConfiguration: {
                bufferingHints: {
                    intervalInSeconds: 600,
                    sizeInMBs: 55
                },
            }
        },
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        }
    });
    expect(stack).toHaveResourceLike("AWS::KinesisFirehose::DeliveryStream", {
        ExtendedS3DestinationConfiguration: {
            BufferingHints: {
                IntervalInSeconds: 600,
                SizeInMBs: 55
            }
        }
    });
});
// --------------------------------------------------------------
// Test bad call with existingBucket and bucketProps
// --------------------------------------------------------------
test("Test bad call with existingBucket and bucketProps", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testBucket = new s3.Bucket(stack, 'test-bucket', {});
    const app = () => {
        // Helper declaration
        new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, "bad-s3-args", {
            existingBucketObj: testBucket,
            bucketProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY
            },
        });
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// s3 bucket with bucket, loggingBucket, and auto delete objects
// --------------------------------------------------------------
test('s3 bucket with bucket, loggingBucket, and auto delete objects', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'kinsisfirehose-s3-analytics', {
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        },
        loggingBucketProps: {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true
        }
    });
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        AccessControl: "LogDeliveryWrite"
    });
    expect(stack).toHaveResource("Custom::S3AutoDeleteObjects", {
        ServiceToken: {
            "Fn::GetAtt": [
                "CustomS3AutoDeleteObjectsCustomResourceProviderHandler9D90184F",
                "Arn"
            ]
        },
        BucketName: {
            Ref: "kinsisfirehoses3analyticsKinesisFirehoseToS3S3LoggingBucket6EE455EF"
        }
    });
});
// --------------------------------------------------------------
// s3 bucket with one content bucket and no logging bucket
// --------------------------------------------------------------
test('s3 bucket with one content bucket and no logging bucket', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'kinsisfirehose-s3-analytics', {
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        },
        kinesisFirehoseProps: {
            deliveryStreamType: 'KinesisStreamAsSource'
        },
        bucketProps: {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        },
        logS3AccessLogs: false
    });
    expect(stack).toCountResources("AWS::S3::Bucket", 1);
});
//# sourceMappingURL=data:application/json;base64,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