"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseCluster = exports.DatabaseClusterBase = exports.EngineVersion = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const endpoint_1 = require("./endpoint");
const neptune_generated_1 = require("./neptune.generated");
const subnet_group_1 = require("./subnet-group");
/**
 * Possible Instances Types to use in Neptune cluster
 * used for defining {@link DatabaseClusterProps.engineVersion}.
 */
class EngineVersion {
    /**
     * Constructor for specifying a custom engine version
     * @param version the engine version of Neptune
     */
    constructor(version) {
        this.version = version;
    }
}
exports.EngineVersion = EngineVersion;
_a = JSII_RTTI_SYMBOL_1;
EngineVersion[_a] = { fqn: "@aws-cdk/aws-neptune.EngineVersion", version: "1.166.1" };
/**
 * Neptune engine version 1.0.1.0
 */
EngineVersion.V1_0_1_0 = new EngineVersion('1.0.1.0');
/**
 * Neptune engine version 1.0.1.1
 */
EngineVersion.V1_0_1_1 = new EngineVersion('1.0.1.1');
/**
 * Neptune engine version 1.0.1.2
 */
EngineVersion.V1_0_1_2 = new EngineVersion('1.0.1.2');
/**
 * Neptune engine version 1.0.2.1
 */
EngineVersion.V1_0_2_1 = new EngineVersion('1.0.2.1');
/**
 * Neptune engine version 1.0.2.2
 */
EngineVersion.V1_0_2_2 = new EngineVersion('1.0.2.2');
/**
 * Neptune engine version 1.0.3.0
 */
EngineVersion.V1_0_3_0 = new EngineVersion('1.0.3.0');
/**
 * Neptune engine version 1.0.4.0
 */
EngineVersion.V1_0_4_0 = new EngineVersion('1.0.4.0');
/**
 * Neptune engine version 1.0.4.1
 */
EngineVersion.V1_0_4_1 = new EngineVersion('1.0.4.1');
/**
 * Neptune engine version 1.0.5.0
 */
EngineVersion.V1_0_5_0 = new EngineVersion('1.0.5.0');
/**
 * Neptune engine version 1.1.0.0
 */
EngineVersion.V1_1_0_0 = new EngineVersion('1.1.0.0');
/**
 * A new or imported database cluster.
 */
class DatabaseClusterBase extends core_1.Resource {
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromDatabaseClusterAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_neptune_DatabaseClusterAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromDatabaseClusterAttributes);
            }
            throw error;
        }
        class Import extends DatabaseClusterBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: [attrs.securityGroup],
                    defaultPort: this.defaultPort,
                });
                this.clusterIdentifier = attrs.clusterIdentifier;
                this.clusterResourceIdentifier = attrs.clusterResourceIdentifier;
                this.clusterEndpoint = new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port);
                this.clusterReadEndpoint = new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port);
                this.enableIamAuthentication = true;
            }
        }
        return new Import(scope, id);
    }
    grantConnect(grantee) {
        if (this.enableIamAuthentication === false) {
            throw new Error('Cannot grant connect when IAM authentication is disabled');
        }
        this.enableIamAuthentication = true;
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['neptune-db:*'],
            resourceArns: [
                [
                    'arn',
                    core_1.Aws.PARTITION,
                    'neptune-db',
                    core_1.Aws.REGION,
                    core_1.Aws.ACCOUNT_ID,
                    `${this.clusterResourceIdentifier}/*`,
                ].join(':'),
            ],
        });
    }
}
exports.DatabaseClusterBase = DatabaseClusterBase;
_b = JSII_RTTI_SYMBOL_1;
DatabaseClusterBase[_b] = { fqn: "@aws-cdk/aws-neptune.DatabaseClusterBase", version: "1.166.1" };
/**
 * Create a clustered database with a given number of instances.
 *
 * @resource AWS::Neptune::DBCluster
 */
class DatabaseCluster extends DatabaseClusterBase {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Identifiers of the instance
         */
        this.instanceIdentifiers = [];
        /**
         * Endpoints which address each individual instance.
         */
        this.instanceEndpoints = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_neptune_DatabaseClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DatabaseCluster);
            }
            throw error;
        }
        this.vpc = props.vpc;
        this.vpcSubnets = props.vpcSubnets ?? { subnetType: ec2.SubnetType.PRIVATE_WITH_NAT };
        // Determine the subnet(s) to deploy the Neptune cluster to
        const { subnetIds, internetConnectivityEstablished } = this.vpc.selectSubnets(this.vpcSubnets);
        // Cannot test whether the subnets are in different AZs, but at least we can test the amount.
        if (subnetIds.length < 2) {
            throw new Error(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        this.subnetGroup = props.subnetGroup ?? new subnet_group_1.SubnetGroup(this, 'Subnets', {
            description: `Subnets for ${id} database`,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            removalPolicy: props.removalPolicy === core_1.RemovalPolicy.RETAIN ? props.removalPolicy : undefined,
        });
        const securityGroups = props.securityGroups ?? [
            new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'Neptune security group',
                vpc: this.vpc,
            }),
        ];
        // Default to encrypted storage
        const storageEncrypted = props.storageEncrypted ?? true;
        if (props.kmsKey && !storageEncrypted) {
            throw new Error('KMS key supplied but storageEncrypted is false');
        }
        const deletionProtection = props.deletionProtection ?? (props.removalPolicy === core_1.RemovalPolicy.RETAIN ? true : undefined);
        this.enableIamAuthentication = props.iamAuthentication;
        // Create the Neptune cluster
        const cluster = new neptune_generated_1.CfnDBCluster(this, 'Resource', {
            // Basic
            engineVersion: props.engineVersion?.version,
            dbClusterIdentifier: props.dbClusterName,
            dbSubnetGroupName: this.subnetGroup.subnetGroupName,
            port: props.port,
            vpcSecurityGroupIds: securityGroups.map(sg => sg.securityGroupId),
            dbClusterParameterGroupName: props.clusterParameterGroup?.clusterParameterGroupName,
            deletionProtection: deletionProtection,
            associatedRoles: props.associatedRoles ? props.associatedRoles.map(role => ({ roleArn: role.roleArn })) : undefined,
            iamAuthEnabled: core_1.Lazy.any({ produce: () => this.enableIamAuthentication }),
            // Backup
            backupRetentionPeriod: props.backupRetention?.toDays(),
            preferredBackupWindow: props.preferredBackupWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            // Encryption
            kmsKeyId: props.kmsKey?.keyArn,
            storageEncrypted,
        });
        cluster.applyRemovalPolicy(props.removalPolicy, {
            applyToUpdateReplacePolicy: true,
        });
        this.clusterIdentifier = cluster.ref;
        this.clusterResourceIdentifier = cluster.attrClusterResourceId;
        const port = core_1.Token.asNumber(cluster.attrPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpoint, port);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpoint, port);
        // Create the instances
        const instanceCount = props.instances ?? DatabaseCluster.DEFAULT_NUM_INSTANCES;
        if (instanceCount < 1) {
            throw new Error('At least one instance is required');
        }
        for (let i = 0; i < instanceCount; i++) {
            const instanceIndex = i + 1;
            const instanceIdentifier = props.instanceIdentifierBase != null ? `${props.instanceIdentifierBase}${instanceIndex}`
                : props.dbClusterName != null ? `${props.dbClusterName}instance${instanceIndex}` : undefined;
            const instance = new neptune_generated_1.CfnDBInstance(this, `Instance${instanceIndex}`, {
                // Link to cluster
                dbClusterIdentifier: cluster.ref,
                dbInstanceIdentifier: instanceIdentifier,
                // Instance properties
                dbInstanceClass: props.instanceType._instanceType,
                dbParameterGroupName: props.parameterGroup?.parameterGroupName,
                autoMinorVersionUpgrade: props.autoMinorVersionUpgrade === true,
            });
            // We must have a dependency on the NAT gateway provider here to create
            // things in the right order.
            instance.node.addDependency(internetConnectivityEstablished);
            instance.applyRemovalPolicy(props.removalPolicy, {
                applyToUpdateReplacePolicy: true,
            });
            this.instanceIdentifiers.push(instance.ref);
            this.instanceEndpoints.push(new endpoint_1.Endpoint(instance.attrEndpoint, port));
        }
        this.connections = new ec2.Connections({
            defaultPort: ec2.Port.tcp(port),
            securityGroups: securityGroups,
        });
    }
}
exports.DatabaseCluster = DatabaseCluster;
_c = JSII_RTTI_SYMBOL_1;
DatabaseCluster[_c] = { fqn: "@aws-cdk/aws-neptune.DatabaseCluster", version: "1.166.1" };
/**
 * The default number of instances in the Neptune cluster if none are
 * specified
 */
DatabaseCluster.DEFAULT_NUM_INSTANCES = 1;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2x1c3Rlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImNsdXN0ZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBQUEsd0NBQXdDO0FBQ3hDLHdDQUF3QztBQUV4Qyx3Q0FBK0Y7QUFFL0YseUNBQXNDO0FBRXRDLDJEQUFrRTtBQUVsRSxpREFBMkQ7QUFFM0Q7OztHQUdHO0FBQ0gsTUFBYSxhQUFhO0lBMEN4Qjs7O09BR0c7SUFDSCxZQUFtQyxPQUFlO1FBQWYsWUFBTyxHQUFQLE9BQU8sQ0FBUTtLQUFJOztBQTlDeEQsc0NBK0NDOzs7QUE5Q0M7O0dBRUc7QUFDb0Isc0JBQVEsR0FBRyxJQUFJLGFBQWEsQ0FBQyxTQUFTLENBQUMsQ0FBQztBQUMvRDs7R0FFRztBQUNvQixzQkFBUSxHQUFHLElBQUksYUFBYSxDQUFDLFNBQVMsQ0FBQyxDQUFDO0FBQy9EOztHQUVHO0FBQ29CLHNCQUFRLEdBQUcsSUFBSSxhQUFhLENBQUMsU0FBUyxDQUFDLENBQUM7QUFDL0Q7O0dBRUc7QUFDb0Isc0JBQVEsR0FBRyxJQUFJLGFBQWEsQ0FBQyxTQUFTLENBQUMsQ0FBQztBQUMvRDs7R0FFRztBQUNvQixzQkFBUSxHQUFHLElBQUksYUFBYSxDQUFDLFNBQVMsQ0FBQyxDQUFDO0FBQy9EOztHQUVHO0FBQ29CLHNCQUFRLEdBQUcsSUFBSSxhQUFhLENBQUMsU0FBUyxDQUFDLENBQUM7QUFDL0Q7O0dBRUc7QUFDb0Isc0JBQVEsR0FBRyxJQUFJLGFBQWEsQ0FBQyxTQUFTLENBQUMsQ0FBQztBQUMvRDs7R0FFRztBQUNvQixzQkFBUSxHQUFHLElBQUksYUFBYSxDQUFDLFNBQVMsQ0FBQyxDQUFDO0FBQy9EOztHQUVHO0FBQ29CLHNCQUFRLEdBQUcsSUFBSSxhQUFhLENBQUMsU0FBUyxDQUFDLENBQUM7QUFDL0Q7O0dBRUc7QUFDb0Isc0JBQVEsR0FBRyxJQUFJLGFBQWEsQ0FBQyxTQUFTLENBQUMsQ0FBQztBQTJQakU7O0dBRUc7QUFDSCxNQUFzQixtQkFBb0IsU0FBUSxlQUFRO0lBRXhEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLDZCQUE2QixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWdDOzs7Ozs7Ozs7O1FBQ3hHLE1BQU0sTUFBTyxTQUFRLG1CQUFtQjtZQUF4Qzs7Z0JBQ2tCLGdCQUFXLEdBQUcsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO2dCQUN2QyxnQkFBVyxHQUFHLElBQUksR0FBRyxDQUFDLFdBQVcsQ0FBQztvQkFDaEQsY0FBYyxFQUFFLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQztvQkFDckMsV0FBVyxFQUFFLElBQUksQ0FBQyxXQUFXO2lCQUM5QixDQUFDLENBQUM7Z0JBQ2Esc0JBQWlCLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixDQUFDO2dCQUM1Qyw4QkFBeUIsR0FBRyxLQUFLLENBQUMseUJBQXlCLENBQUM7Z0JBQzVELG9CQUFlLEdBQUcsSUFBSSxtQkFBUSxDQUFDLEtBQUssQ0FBQyxzQkFBc0IsRUFBRSxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ3pFLHdCQUFtQixHQUFHLElBQUksbUJBQVEsQ0FBQyxLQUFLLENBQUMscUJBQXFCLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO2dCQUNsRiw0QkFBdUIsR0FBRyxJQUFJLENBQUM7WUFDM0MsQ0FBQztTQUFBO1FBRUQsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7S0FDOUI7SUE2Qk0sWUFBWSxDQUFDLE9BQXVCO1FBQ3pDLElBQUksSUFBSSxDQUFDLHVCQUF1QixLQUFLLEtBQUssRUFBRTtZQUMxQyxNQUFNLElBQUksS0FBSyxDQUFDLDBEQUEwRCxDQUFDLENBQUM7U0FDN0U7UUFFRCxJQUFJLENBQUMsdUJBQXVCLEdBQUcsSUFBSSxDQUFDO1FBQ3BDLE9BQU8sR0FBRyxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUM7WUFDOUIsT0FBTztZQUNQLE9BQU8sRUFBRSxDQUFDLGNBQWMsQ0FBQztZQUN6QixZQUFZLEVBQUU7Z0JBQ1o7b0JBQ0UsS0FBSztvQkFDTCxVQUFHLENBQUMsU0FBUztvQkFDYixZQUFZO29CQUNaLFVBQUcsQ0FBQyxNQUFNO29CQUNWLFVBQUcsQ0FBQyxVQUFVO29CQUNkLEdBQUcsSUFBSSxDQUFDLHlCQUF5QixJQUFJO2lCQUN0QyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUM7YUFDWjtTQUNGLENBQUMsQ0FBQztLQUNKOztBQXJFSCxrREFzRUM7OztBQUVEOzs7O0dBSUc7QUFDSCxNQUFhLGVBQWdCLFNBQVEsbUJBQW1CO0lBK0N0RCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTJCO1FBQ25FLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFibkI7O1dBRUc7UUFDYSx3QkFBbUIsR0FBYSxFQUFFLENBQUM7UUFFbkQ7O1dBRUc7UUFDYSxzQkFBaUIsR0FBZSxFQUFFLENBQUM7Ozs7OzsrQ0EzQ3hDLGVBQWU7Ozs7UUFrRHhCLElBQUksQ0FBQyxHQUFHLEdBQUcsS0FBSyxDQUFDLEdBQUcsQ0FBQztRQUNyQixJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLElBQUksRUFBRSxVQUFVLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDO1FBRXRGLDJEQUEyRDtRQUMzRCxNQUFNLEVBQUUsU0FBUyxFQUFFLCtCQUErQixFQUFFLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBRS9GLDZGQUE2RjtRQUM3RixJQUFJLFNBQVMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ3hCLE1BQU0sSUFBSSxLQUFLLENBQUMsNENBQTRDLFNBQVMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO1NBQ2pGO1FBRUQsSUFBSSxDQUFDLFdBQVcsR0FBRyxLQUFLLENBQUMsV0FBVyxJQUFJLElBQUksMEJBQVcsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQ3ZFLFdBQVcsRUFBRSxlQUFlLEVBQUUsV0FBVztZQUN6QyxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7WUFDYixVQUFVLEVBQUUsSUFBSSxDQUFDLFVBQVU7WUFDM0IsYUFBYSxFQUFFLEtBQUssQ0FBQyxhQUFhLEtBQUssb0JBQWEsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLFNBQVM7U0FDOUYsQ0FBQyxDQUFDO1FBRUgsTUFBTSxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsSUFBSTtZQUM3QyxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLGVBQWUsRUFBRTtnQkFDM0MsV0FBVyxFQUFFLHdCQUF3QjtnQkFDckMsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO2FBQ2QsQ0FBQztTQUNILENBQUM7UUFFRiwrQkFBK0I7UUFDL0IsTUFBTSxnQkFBZ0IsR0FBRyxLQUFLLENBQUMsZ0JBQWdCLElBQUksSUFBSSxDQUFDO1FBRXhELElBQUksS0FBSyxDQUFDLE1BQU0sSUFBSSxDQUFDLGdCQUFnQixFQUFFO1lBQ3JDLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0RBQWdELENBQUMsQ0FBQztTQUNuRTtRQUVELE1BQU0sa0JBQWtCLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsS0FBSyxvQkFBYSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUV6SCxJQUFJLENBQUMsdUJBQXVCLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixDQUFDO1FBRXZELDZCQUE2QjtRQUM3QixNQUFNLE9BQU8sR0FBRyxJQUFJLGdDQUFZLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNqRCxRQUFRO1lBQ1IsYUFBYSxFQUFFLEtBQUssQ0FBQyxhQUFhLEVBQUUsT0FBTztZQUMzQyxtQkFBbUIsRUFBRSxLQUFLLENBQUMsYUFBYTtZQUN4QyxpQkFBaUIsRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLGVBQWU7WUFDbkQsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJO1lBQ2hCLG1CQUFtQixFQUFFLGNBQWMsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDO1lBQ2pFLDJCQUEyQixFQUFFLEtBQUssQ0FBQyxxQkFBcUIsRUFBRSx5QkFBeUI7WUFDbkYsa0JBQWtCLEVBQUUsa0JBQWtCO1lBQ3RDLGVBQWUsRUFBRSxLQUFLLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsZUFBZSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUNuSCxjQUFjLEVBQUUsV0FBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsdUJBQXVCLEVBQUUsQ0FBQztZQUN6RSxTQUFTO1lBQ1QscUJBQXFCLEVBQUUsS0FBSyxDQUFDLGVBQWUsRUFBRSxNQUFNLEVBQUU7WUFDdEQscUJBQXFCLEVBQUUsS0FBSyxDQUFDLHFCQUFxQjtZQUNsRCwwQkFBMEIsRUFBRSxLQUFLLENBQUMsMEJBQTBCO1lBQzVELGFBQWE7WUFDYixRQUFRLEVBQUUsS0FBSyxDQUFDLE1BQU0sRUFBRSxNQUFNO1lBQzlCLGdCQUFnQjtTQUNqQixDQUFDLENBQUM7UUFFSCxPQUFPLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLGFBQWEsRUFBRTtZQUM5QywwQkFBMEIsRUFBRSxJQUFJO1NBQ2pDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxpQkFBaUIsR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDO1FBQ3JDLElBQUksQ0FBQyx5QkFBeUIsR0FBRyxPQUFPLENBQUMscUJBQXFCLENBQUM7UUFFL0QsTUFBTSxJQUFJLEdBQUcsWUFBSyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDOUMsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLG1CQUFRLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNoRSxJQUFJLENBQUMsbUJBQW1CLEdBQUcsSUFBSSxtQkFBUSxDQUFDLE9BQU8sQ0FBQyxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUV4RSx1QkFBdUI7UUFDdkIsTUFBTSxhQUFhLEdBQUcsS0FBSyxDQUFDLFNBQVMsSUFBSSxlQUFlLENBQUMscUJBQXFCLENBQUM7UUFDL0UsSUFBSSxhQUFhLEdBQUcsQ0FBQyxFQUFFO1lBQ3JCLE1BQU0sSUFBSSxLQUFLLENBQUMsbUNBQW1DLENBQUMsQ0FBQztTQUN0RDtRQUVELEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxhQUFhLEVBQUUsQ0FBQyxFQUFFLEVBQUU7WUFDdEMsTUFBTSxhQUFhLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQztZQUU1QixNQUFNLGtCQUFrQixHQUFHLEtBQUssQ0FBQyxzQkFBc0IsSUFBSSxJQUFJLENBQUMsQ0FBQyxDQUFDLEdBQUcsS0FBSyxDQUFDLHNCQUFzQixHQUFHLGFBQWEsRUFBRTtnQkFDakgsQ0FBQyxDQUFDLEtBQUssQ0FBQyxhQUFhLElBQUksSUFBSSxDQUFDLENBQUMsQ0FBQyxHQUFHLEtBQUssQ0FBQyxhQUFhLFdBQVcsYUFBYSxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztZQUUvRixNQUFNLFFBQVEsR0FBRyxJQUFJLGlDQUFhLENBQUMsSUFBSSxFQUFFLFdBQVcsYUFBYSxFQUFFLEVBQUU7Z0JBQ25FLGtCQUFrQjtnQkFDbEIsbUJBQW1CLEVBQUUsT0FBTyxDQUFDLEdBQUc7Z0JBQ2hDLG9CQUFvQixFQUFFLGtCQUFrQjtnQkFDeEMsc0JBQXNCO2dCQUN0QixlQUFlLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxhQUFhO2dCQUNqRCxvQkFBb0IsRUFBRSxLQUFLLENBQUMsY0FBYyxFQUFFLGtCQUFrQjtnQkFDOUQsdUJBQXVCLEVBQUUsS0FBSyxDQUFDLHVCQUF1QixLQUFLLElBQUk7YUFDaEUsQ0FBQyxDQUFDO1lBRUgsdUVBQXVFO1lBQ3ZFLDZCQUE2QjtZQUM3QixRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1lBRTdELFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsYUFBYSxFQUFFO2dCQUMvQywwQkFBMEIsRUFBRSxJQUFJO2FBQ2pDLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQzVDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsSUFBSSxtQkFBUSxDQUFDLFFBQVEsQ0FBQyxZQUFZLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQztTQUN4RTtRQUVELElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSxHQUFHLENBQUMsV0FBVyxDQUFDO1lBQ3JDLFdBQVcsRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUM7WUFDL0IsY0FBYyxFQUFFLGNBQWM7U0FDL0IsQ0FBQyxDQUFDO0tBQ0o7O0FBNUpILDBDQTZKQzs7O0FBM0pDOzs7R0FHRztBQUNvQixxQ0FBcUIsR0FBRyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBlYzIgZnJvbSAnQGF3cy1jZGsvYXdzLWVjMic7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBrbXMgZnJvbSAnQGF3cy1jZGsvYXdzLWttcyc7XG5pbXBvcnQgeyBBd3MsIER1cmF0aW9uLCBJUmVzb3VyY2UsIExhenksIFJlbW92YWxQb2xpY3ksIFJlc291cmNlLCBUb2tlbiB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBFbmRwb2ludCB9IGZyb20gJy4vZW5kcG9pbnQnO1xuaW1wb3J0IHsgSW5zdGFuY2VUeXBlIH0gZnJvbSAnLi9pbnN0YW5jZSc7XG5pbXBvcnQgeyBDZm5EQkNsdXN0ZXIsIENmbkRCSW5zdGFuY2UgfSBmcm9tICcuL25lcHR1bmUuZ2VuZXJhdGVkJztcbmltcG9ydCB7IElDbHVzdGVyUGFyYW1ldGVyR3JvdXAsIElQYXJhbWV0ZXJHcm91cCB9IGZyb20gJy4vcGFyYW1ldGVyLWdyb3VwJztcbmltcG9ydCB7IElTdWJuZXRHcm91cCwgU3VibmV0R3JvdXAgfSBmcm9tICcuL3N1Ym5ldC1ncm91cCc7XG5cbi8qKlxuICogUG9zc2libGUgSW5zdGFuY2VzIFR5cGVzIHRvIHVzZSBpbiBOZXB0dW5lIGNsdXN0ZXJcbiAqIHVzZWQgZm9yIGRlZmluaW5nIHtAbGluayBEYXRhYmFzZUNsdXN0ZXJQcm9wcy5lbmdpbmVWZXJzaW9ufS5cbiAqL1xuZXhwb3J0IGNsYXNzIEVuZ2luZVZlcnNpb24ge1xuICAvKipcbiAgICogTmVwdHVuZSBlbmdpbmUgdmVyc2lvbiAxLjAuMS4wXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFYxXzBfMV8wID0gbmV3IEVuZ2luZVZlcnNpb24oJzEuMC4xLjAnKTtcbiAgLyoqXG4gICAqIE5lcHR1bmUgZW5naW5lIHZlcnNpb24gMS4wLjEuMVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWMV8wXzFfMSA9IG5ldyBFbmdpbmVWZXJzaW9uKCcxLjAuMS4xJyk7XG4gIC8qKlxuICAgKiBOZXB0dW5lIGVuZ2luZSB2ZXJzaW9uIDEuMC4xLjJcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjFfMF8xXzIgPSBuZXcgRW5naW5lVmVyc2lvbignMS4wLjEuMicpO1xuICAvKipcbiAgICogTmVwdHVuZSBlbmdpbmUgdmVyc2lvbiAxLjAuMi4xXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFYxXzBfMl8xID0gbmV3IEVuZ2luZVZlcnNpb24oJzEuMC4yLjEnKTtcbiAgLyoqXG4gICAqIE5lcHR1bmUgZW5naW5lIHZlcnNpb24gMS4wLjIuMlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWMV8wXzJfMiA9IG5ldyBFbmdpbmVWZXJzaW9uKCcxLjAuMi4yJyk7XG4gIC8qKlxuICAgKiBOZXB0dW5lIGVuZ2luZSB2ZXJzaW9uIDEuMC4zLjBcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjFfMF8zXzAgPSBuZXcgRW5naW5lVmVyc2lvbignMS4wLjMuMCcpO1xuICAvKipcbiAgICogTmVwdHVuZSBlbmdpbmUgdmVyc2lvbiAxLjAuNC4wXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFYxXzBfNF8wID0gbmV3IEVuZ2luZVZlcnNpb24oJzEuMC40LjAnKTtcbiAgLyoqXG4gICAqIE5lcHR1bmUgZW5naW5lIHZlcnNpb24gMS4wLjQuMVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBWMV8wXzRfMSA9IG5ldyBFbmdpbmVWZXJzaW9uKCcxLjAuNC4xJyk7XG4gIC8qKlxuICAgKiBOZXB0dW5lIGVuZ2luZSB2ZXJzaW9uIDEuMC41LjBcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVjFfMF81XzAgPSBuZXcgRW5naW5lVmVyc2lvbignMS4wLjUuMCcpO1xuICAvKipcbiAgICogTmVwdHVuZSBlbmdpbmUgdmVyc2lvbiAxLjEuMC4wXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFYxXzFfMF8wID0gbmV3IEVuZ2luZVZlcnNpb24oJzEuMS4wLjAnKTtcblxuICAvKipcbiAgICogQ29uc3RydWN0b3IgZm9yIHNwZWNpZnlpbmcgYSBjdXN0b20gZW5naW5lIHZlcnNpb25cbiAgICogQHBhcmFtIHZlcnNpb24gdGhlIGVuZ2luZSB2ZXJzaW9uIG9mIE5lcHR1bmVcbiAgICovXG4gIHB1YmxpYyBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgdmVyc2lvbjogc3RyaW5nKSB7fVxufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGEgbmV3IGRhdGFiYXNlIGNsdXN0ZXJcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBEYXRhYmFzZUNsdXN0ZXJQcm9wcyB7XG4gIC8qKlxuICAgKiBXaGF0IHZlcnNpb24gb2YgdGhlIGRhdGFiYXNlIHRvIHN0YXJ0XG4gICAqXG4gICAqIEBkZWZhdWx0IC0gVGhlIGRlZmF1bHQgZW5naW5lIHZlcnNpb24uXG4gICAqL1xuICByZWFkb25seSBlbmdpbmVWZXJzaW9uPzogRW5naW5lVmVyc2lvbjtcblxuICAvKipcbiAgICogVGhlIHBvcnQgdGhlIE5lcHR1bmUgY2x1c3RlciB3aWxsIGxpc3RlbiBvblxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFRoZSBkZWZhdWx0IGVuZ2luZSBwb3J0XG4gICAqL1xuICByZWFkb25seSBwb3J0PzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBIb3cgbWFueSBkYXlzIHRvIHJldGFpbiB0aGUgYmFja3VwXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gY2RrLkR1cmF0aW9uLmRheXMoMSlcbiAgICovXG4gIHJlYWRvbmx5IGJhY2t1cFJldGVudGlvbj86IER1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBBIGRhaWx5IHRpbWUgcmFuZ2UgaW4gMjQtaG91cnMgVVRDIGZvcm1hdCBpbiB3aGljaCBiYWNrdXBzIHByZWZlcmFibHkgZXhlY3V0ZS5cbiAgICpcbiAgICogTXVzdCBiZSBhdCBsZWFzdCAzMCBtaW51dGVzIGxvbmcuXG4gICAqXG4gICAqIEV4YW1wbGU6ICcwMTowMC0wMjowMCdcbiAgICpcbiAgICogQGRlZmF1bHQgLSBhIDMwLW1pbnV0ZSB3aW5kb3cgc2VsZWN0ZWQgYXQgcmFuZG9tIGZyb20gYW4gOC1ob3VyIGJsb2NrIG9mXG4gICAqIHRpbWUgZm9yIGVhY2ggQVdTIFJlZ2lvbi4gVG8gc2VlIHRoZSB0aW1lIGJsb2NrcyBhdmFpbGFibGUsIHNlZVxuICAgKi9cbiAgcmVhZG9ubHkgcHJlZmVycmVkQmFja3VwV2luZG93Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgS01TIGtleSBmb3Igc3RvcmFnZSBlbmNyeXB0aW9uLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGRlZmF1bHQgbWFzdGVyIGtleS5cbiAgICovXG4gIHJlYWRvbmx5IGttc0tleT86IGttcy5JS2V5O1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRvIGVuYWJsZSBzdG9yYWdlIGVuY3J5cHRpb25cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgc3RvcmFnZUVuY3J5cHRlZD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIE51bWJlciBvZiBOZXB0dW5lIGNvbXB1dGUgaW5zdGFuY2VzXG4gICAqXG4gICAqIEBkZWZhdWx0IDFcbiAgICovXG4gIHJlYWRvbmx5IGluc3RhbmNlcz86IG51bWJlcjtcblxuICAvKipcbiAgICogQW4gb3B0aW9uYWwgaWRlbnRpZmllciBmb3IgdGhlIGNsdXN0ZXJcbiAgICpcbiAgICogQGRlZmF1bHQgLSBBIG5hbWUgaXMgYXV0b21hdGljYWxseSBnZW5lcmF0ZWQuXG4gICAqL1xuICByZWFkb25seSBkYkNsdXN0ZXJOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBNYXAgQVdTIElkZW50aXR5IGFuZCBBY2Nlc3MgTWFuYWdlbWVudCAoSUFNKSBhY2NvdW50cyB0byBkYXRhYmFzZSBhY2NvdW50c1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIGBmYWxzZWBcbiAgICovXG4gIHJlYWRvbmx5IGlhbUF1dGhlbnRpY2F0aW9uPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQmFzZSBpZGVudGlmaWVyIGZvciBpbnN0YW5jZXNcbiAgICpcbiAgICogRXZlcnkgcmVwbGljYSBpcyBuYW1lZCBieSBhcHBlbmRpbmcgdGhlIHJlcGxpY2EgbnVtYmVyIHRvIHRoaXMgc3RyaW5nLCAxLWJhc2VkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGBkYkNsdXN0ZXJOYW1lYCBpcyB1c2VkIHdpdGggdGhlIHdvcmQgXCJJbnN0YW5jZVwiIGFwcGVuZGVkLiBJZiBgZGJDbHVzdGVyTmFtZWAgaXMgbm90IHByb3ZpZGVkLCB0aGVcbiAgICogaWRlbnRpZmllciBpcyBhdXRvbWF0aWNhbGx5IGdlbmVyYXRlZC5cbiAgICovXG4gIHJlYWRvbmx5IGluc3RhbmNlSWRlbnRpZmllckJhc2U/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFdoYXQgdHlwZSBvZiBpbnN0YW5jZSB0byBzdGFydCBmb3IgdGhlIHJlcGxpY2FzXG4gICAqL1xuICByZWFkb25seSBpbnN0YW5jZVR5cGU6IEluc3RhbmNlVHlwZTtcblxuICAvKipcbiAgICogQSBsaXN0IG9mIEFXUyBJZGVudGl0eSBhbmQgQWNjZXNzIE1hbmFnZW1lbnQgKElBTSkgcm9sZSB0aGF0IGNhbiBiZSB1c2VkIGJ5IHRoZSBjbHVzdGVyIHRvIGFjY2VzcyBvdGhlciBBV1Mgc2VydmljZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gcm9sZSBpcyBhdHRhY2hlZCB0byB0aGUgY2x1c3Rlci5cbiAgICovXG4gIHJlYWRvbmx5IGFzc29jaWF0ZWRSb2xlcz86IGlhbS5JUm9sZVtdO1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgd2hldGhlciB0aGUgREIgY2x1c3RlciBzaG91bGQgaGF2ZSBkZWxldGlvbiBwcm90ZWN0aW9uIGVuYWJsZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdHJ1ZSBpZiBgYHJlbW92YWxQb2xpY3lgYCBpcyBSRVRBSU4sIGZhbHNlIG90aGVyd2lzZVxuICAgKi9cbiAgcmVhZG9ubHkgZGVsZXRpb25Qcm90ZWN0aW9uPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQSB3ZWVrbHkgdGltZSByYW5nZSBpbiB3aGljaCBtYWludGVuYW5jZSBzaG91bGQgcHJlZmVyYWJseSBleGVjdXRlLlxuICAgKlxuICAgKiBNdXN0IGJlIGF0IGxlYXN0IDMwIG1pbnV0ZXMgbG9uZy5cbiAgICpcbiAgICogRXhhbXBsZTogJ3R1ZTowNDoxNy10dWU6MDQ6NDcnXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gMzAtbWludXRlIHdpbmRvdyBzZWxlY3RlZCBhdCByYW5kb20gZnJvbSBhbiA4LWhvdXIgYmxvY2sgb2YgdGltZSBmb3JcbiAgICogZWFjaCBBV1MgUmVnaW9uLCBvY2N1cnJpbmcgb24gYSByYW5kb20gZGF5IG9mIHRoZSB3ZWVrLlxuICAgKi9cbiAgcmVhZG9ubHkgcHJlZmVycmVkTWFpbnRlbmFuY2VXaW5kb3c/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFkZGl0aW9uYWwgcGFyYW1ldGVycyB0byBwYXNzIHRvIHRoZSBkYXRhYmFzZSBlbmdpbmVcbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBwYXJhbWV0ZXIgZ3JvdXAuXG4gICAqL1xuICByZWFkb25seSBjbHVzdGVyUGFyYW1ldGVyR3JvdXA/OiBJQ2x1c3RlclBhcmFtZXRlckdyb3VwO1xuXG4gIC8qKlxuICAgKiBUaGUgREIgcGFyYW1ldGVyIGdyb3VwIHRvIGFzc29jaWF0ZSB3aXRoIHRoZSBpbnN0YW5jZS5cbiAgICpcbiAgICogQGRlZmF1bHQgbm8gcGFyYW1ldGVyIGdyb3VwXG4gICAqL1xuICByZWFkb25seSBwYXJhbWV0ZXJHcm91cD86IElQYXJhbWV0ZXJHcm91cDtcblxuICAvKipcbiAgICogRXhpc3Rpbmcgc3VibmV0IGdyb3VwIGZvciB0aGUgY2x1c3Rlci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBhIG5ldyBzdWJuZXQgZ3JvdXAgd2lsbCBiZSBjcmVhdGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgc3VibmV0R3JvdXA/OiBJU3VibmV0R3JvdXA7XG5cbiAgLyoqXG4gICAqIFdoYXQgc3VibmV0cyB0byBydW4gdGhlIE5lcHR1bmUgaW5zdGFuY2VzIGluLlxuICAgKlxuICAgKiBNdXN0IGJlIGF0IGxlYXN0IDIgc3VibmV0cyBpbiB0d28gZGlmZmVyZW50IEFacy5cbiAgICovXG4gIHJlYWRvbmx5IHZwYzogZWMyLklWcGM7XG5cbiAgLyoqXG4gICAqIFdoZXJlIHRvIHBsYWNlIHRoZSBpbnN0YW5jZXMgd2l0aGluIHRoZSBWUENcbiAgICpcbiAgICogQGRlZmF1bHQgcHJpdmF0ZSBzdWJuZXRzXG4gICAqL1xuICByZWFkb25seSB2cGNTdWJuZXRzPzogZWMyLlN1Ym5ldFNlbGVjdGlvbjtcblxuICAvKipcbiAgICogU2VjdXJpdHkgZ3JvdXAuXG4gICAqXG4gICAqIEBkZWZhdWx0IGEgbmV3IHNlY3VyaXR5IGdyb3VwIGlzIGNyZWF0ZWQuXG4gICAqL1xuICByZWFkb25seSBzZWN1cml0eUdyb3Vwcz86IGVjMi5JU2VjdXJpdHlHcm91cFtdO1xuXG4gIC8qKlxuICAgKiBUaGUgcmVtb3ZhbCBwb2xpY3kgdG8gYXBwbHkgd2hlbiB0aGUgY2x1c3RlciBhbmQgaXRzIGluc3RhbmNlcyBhcmUgcmVtb3ZlZFxuICAgKiBvciByZXBsYWNlZCBkdXJpbmcgYSBzdGFjayB1cGRhdGUsIG9yIHdoZW4gdGhlIHN0YWNrIGlzIGRlbGV0ZWQuIFRoaXNcbiAgICogcmVtb3ZhbCBwb2xpY3kgYWxzbyBhcHBsaWVzIHRvIHRoZSBpbXBsaWNpdCBzZWN1cml0eSBncm91cCBjcmVhdGVkIGZvciB0aGVcbiAgICogY2x1c3RlciBpZiBvbmUgaXMgbm90IHN1cHBsaWVkIGFzIGEgcGFyYW1ldGVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFJldGFpbiBjbHVzdGVyLlxuICAgKi9cbiAgcmVhZG9ubHkgcmVtb3ZhbFBvbGljeT86IFJlbW92YWxQb2xpY3lcblxuICAvKipcbiAgICogSWYgc2V0IHRvIHRydWUsIE5lcHR1bmUgd2lsbCBhdXRvbWF0aWNhbGx5IHVwZGF0ZSB0aGUgZW5naW5lIG9mIHRoZSBlbnRpcmVcbiAgICogY2x1c3RlciB0byB0aGUgbGF0ZXN0IG1pbm9yIHZlcnNpb24gYWZ0ZXIgYSBzdGFiaWxpemF0aW9uIHdpbmRvdyBvZiAyIHRvIDMgd2Vla3MuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGF1dG9NaW5vclZlcnNpb25VcGdyYWRlPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBDcmVhdGUgYSBjbHVzdGVyZWQgZGF0YWJhc2Ugd2l0aCBhIGdpdmVuIG51bWJlciBvZiBpbnN0YW5jZXMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSURhdGFiYXNlQ2x1c3RlciBleHRlbmRzIElSZXNvdXJjZSwgZWMyLklDb25uZWN0YWJsZSB7XG4gIC8qKlxuICAgKiBJZGVudGlmaWVyIG9mIHRoZSBjbHVzdGVyXG4gICAqL1xuICByZWFkb25seSBjbHVzdGVySWRlbnRpZmllcjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBSZXNvdXJjZSBpZGVudGlmaWVyIG9mIHRoZSBjbHVzdGVyXG4gICAqIEBhdHRyaWJ1dGUgQ2x1c3RlclJlc291cmNlSWRcbiAgICovXG4gIHJlYWRvbmx5IGNsdXN0ZXJSZXNvdXJjZUlkZW50aWZpZXI6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGVuZHBvaW50IHRvIHVzZSBmb3IgcmVhZC93cml0ZSBvcGVyYXRpb25zXG4gICAqIEBhdHRyaWJ1dGUgRW5kcG9pbnQsUG9ydFxuICAgKi9cbiAgcmVhZG9ubHkgY2x1c3RlckVuZHBvaW50OiBFbmRwb2ludDtcblxuICAvKipcbiAgICogRW5kcG9pbnQgdG8gdXNlIGZvciBsb2FkLWJhbGFuY2VkIHJlYWQtb25seSBvcGVyYXRpb25zLlxuICAgKiBAYXR0cmlidXRlIFJlYWRFbmRwb2ludFxuICAgKi9cbiAgcmVhZG9ubHkgY2x1c3RlclJlYWRFbmRwb2ludDogRW5kcG9pbnQ7XG5cbiAgLyoqXG4gICAqIEdyYW50IHRoZSBnaXZlbiBpZGVudGl0eSBjb25uZWN0aW9uIGFjY2VzcyB0byB0aGUgZGF0YWJhc2UuXG4gICAqL1xuICBncmFudENvbm5lY3QoZ3JhbnRlZTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQ7XG59XG5cbi8qKlxuICogUHJvcGVydGllcyB0aGF0IGRlc2NyaWJlIGFuIGV4aXN0aW5nIGNsdXN0ZXIgaW5zdGFuY2VcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBEYXRhYmFzZUNsdXN0ZXJBdHRyaWJ1dGVzIHtcbiAgLyoqXG4gICAqIFRoZSBkYXRhYmFzZSBwb3J0XG4gICAqL1xuICByZWFkb25seSBwb3J0OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBzZWN1cml0eSBncm91cCBvZiB0aGUgZGF0YWJhc2UgY2x1c3RlclxuICAgKi9cbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cDogZWMyLklTZWN1cml0eUdyb3VwO1xuXG4gIC8qKlxuICAgKiBJZGVudGlmaWVyIGZvciB0aGUgY2x1c3RlclxuICAgKi9cbiAgcmVhZG9ubHkgY2x1c3RlcklkZW50aWZpZXI6IHN0cmluZztcblxuICAvKipcbiAgICogUmVzb3VyY2UgSWRlbnRpZmllciBmb3IgdGhlIGNsdXN0ZXJcbiAgICovXG4gIHJlYWRvbmx5IGNsdXN0ZXJSZXNvdXJjZUlkZW50aWZpZXI6IHN0cmluZztcblxuICAvKipcbiAgICogQ2x1c3RlciBlbmRwb2ludCBhZGRyZXNzXG4gICAqL1xuICByZWFkb25seSBjbHVzdGVyRW5kcG9pbnRBZGRyZXNzOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFJlYWRlciBlbmRwb2ludCBhZGRyZXNzXG4gICAqL1xuICByZWFkb25seSByZWFkZXJFbmRwb2ludEFkZHJlc3M6IHN0cmluZztcbn1cblxuLyoqXG4gKiBBIG5ldyBvciBpbXBvcnRlZCBkYXRhYmFzZSBjbHVzdGVyLlxuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgRGF0YWJhc2VDbHVzdGVyQmFzZSBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSURhdGFiYXNlQ2x1c3RlciB7XG5cbiAgLyoqXG4gICAqIEltcG9ydCBhbiBleGlzdGluZyBEYXRhYmFzZUNsdXN0ZXIgZnJvbSBwcm9wZXJ0aWVzXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21EYXRhYmFzZUNsdXN0ZXJBdHRyaWJ1dGVzKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGF0dHJzOiBEYXRhYmFzZUNsdXN0ZXJBdHRyaWJ1dGVzKTogSURhdGFiYXNlQ2x1c3RlciB7XG4gICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgRGF0YWJhc2VDbHVzdGVyQmFzZSBpbXBsZW1lbnRzIElEYXRhYmFzZUNsdXN0ZXIge1xuICAgICAgcHVibGljIHJlYWRvbmx5IGRlZmF1bHRQb3J0ID0gZWMyLlBvcnQudGNwKGF0dHJzLnBvcnQpO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGNvbm5lY3Rpb25zID0gbmV3IGVjMi5Db25uZWN0aW9ucyh7XG4gICAgICAgIHNlY3VyaXR5R3JvdXBzOiBbYXR0cnMuc2VjdXJpdHlHcm91cF0sXG4gICAgICAgIGRlZmF1bHRQb3J0OiB0aGlzLmRlZmF1bHRQb3J0LFxuICAgICAgfSk7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgY2x1c3RlcklkZW50aWZpZXIgPSBhdHRycy5jbHVzdGVySWRlbnRpZmllcjtcbiAgICAgIHB1YmxpYyByZWFkb25seSBjbHVzdGVyUmVzb3VyY2VJZGVudGlmaWVyID0gYXR0cnMuY2x1c3RlclJlc291cmNlSWRlbnRpZmllcjtcbiAgICAgIHB1YmxpYyByZWFkb25seSBjbHVzdGVyRW5kcG9pbnQgPSBuZXcgRW5kcG9pbnQoYXR0cnMuY2x1c3RlckVuZHBvaW50QWRkcmVzcywgYXR0cnMucG9ydCk7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgY2x1c3RlclJlYWRFbmRwb2ludCA9IG5ldyBFbmRwb2ludChhdHRycy5yZWFkZXJFbmRwb2ludEFkZHJlc3MsIGF0dHJzLnBvcnQpO1xuICAgICAgcHJvdGVjdGVkIGVuYWJsZUlhbUF1dGhlbnRpY2F0aW9uID0gdHJ1ZTtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIElkZW50aWZpZXIgb2YgdGhlIGNsdXN0ZXJcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBjbHVzdGVySWRlbnRpZmllcjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBSZXNvdXJjZSBpZGVudGlmaWVyIG9mIHRoZSBjbHVzdGVyXG4gICAqL1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgY2x1c3RlclJlc291cmNlSWRlbnRpZmllcjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgZW5kcG9pbnQgdG8gdXNlIGZvciByZWFkL3dyaXRlIG9wZXJhdGlvbnNcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBjbHVzdGVyRW5kcG9pbnQ6IEVuZHBvaW50O1xuXG4gIC8qKlxuICAgKiBFbmRwb2ludCB0byB1c2UgZm9yIGxvYWQtYmFsYW5jZWQgcmVhZC1vbmx5IG9wZXJhdGlvbnMuXG4gICAqL1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgY2x1c3RlclJlYWRFbmRwb2ludDogRW5kcG9pbnQ7XG5cbiAgLyoqXG4gICAqIFRoZSBjb25uZWN0aW9ucyBvYmplY3QgdG8gaW1wbGVtZW50IElDb25uZWN0YWJsZVxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGNvbm5lY3Rpb25zOiBlYzIuQ29ubmVjdGlvbnM7XG5cbiAgcHJvdGVjdGVkIGFic3RyYWN0IGVuYWJsZUlhbUF1dGhlbnRpY2F0aW9uPzogYm9vbGVhbjtcblxuICBwdWJsaWMgZ3JhbnRDb25uZWN0KGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICBpZiAodGhpcy5lbmFibGVJYW1BdXRoZW50aWNhdGlvbiA9PT0gZmFsc2UpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IGdyYW50IGNvbm5lY3Qgd2hlbiBJQU0gYXV0aGVudGljYXRpb24gaXMgZGlzYWJsZWQnKTtcbiAgICB9XG5cbiAgICB0aGlzLmVuYWJsZUlhbUF1dGhlbnRpY2F0aW9uID0gdHJ1ZTtcbiAgICByZXR1cm4gaWFtLkdyYW50LmFkZFRvUHJpbmNpcGFsKHtcbiAgICAgIGdyYW50ZWUsXG4gICAgICBhY3Rpb25zOiBbJ25lcHR1bmUtZGI6KiddLFxuICAgICAgcmVzb3VyY2VBcm5zOiBbXG4gICAgICAgIFtcbiAgICAgICAgICAnYXJuJyxcbiAgICAgICAgICBBd3MuUEFSVElUSU9OLFxuICAgICAgICAgICduZXB0dW5lLWRiJyxcbiAgICAgICAgICBBd3MuUkVHSU9OLFxuICAgICAgICAgIEF3cy5BQ0NPVU5UX0lELFxuICAgICAgICAgIGAke3RoaXMuY2x1c3RlclJlc291cmNlSWRlbnRpZmllcn0vKmAsXG4gICAgICAgIF0uam9pbignOicpLFxuICAgICAgXSxcbiAgICB9KTtcbiAgfVxufVxuXG4vKipcbiAqIENyZWF0ZSBhIGNsdXN0ZXJlZCBkYXRhYmFzZSB3aXRoIGEgZ2l2ZW4gbnVtYmVyIG9mIGluc3RhbmNlcy5cbiAqXG4gKiBAcmVzb3VyY2UgQVdTOjpOZXB0dW5lOjpEQkNsdXN0ZXJcbiAqL1xuZXhwb3J0IGNsYXNzIERhdGFiYXNlQ2x1c3RlciBleHRlbmRzIERhdGFiYXNlQ2x1c3RlckJhc2UgaW1wbGVtZW50cyBJRGF0YWJhc2VDbHVzdGVyIHtcblxuICAvKipcbiAgICogVGhlIGRlZmF1bHQgbnVtYmVyIG9mIGluc3RhbmNlcyBpbiB0aGUgTmVwdHVuZSBjbHVzdGVyIGlmIG5vbmUgYXJlXG4gICAqIHNwZWNpZmllZFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBERUZBVUxUX05VTV9JTlNUQU5DRVMgPSAxO1xuXG4gIHB1YmxpYyByZWFkb25seSBjbHVzdGVySWRlbnRpZmllcjogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgY2x1c3RlckVuZHBvaW50OiBFbmRwb2ludDtcbiAgcHVibGljIHJlYWRvbmx5IGNsdXN0ZXJSZWFkRW5kcG9pbnQ6IEVuZHBvaW50O1xuICBwdWJsaWMgcmVhZG9ubHkgY29ubmVjdGlvbnM6IGVjMi5Db25uZWN0aW9ucztcblxuICAvKipcbiAgICogVGhlIHJlc291cmNlIGlkIGZvciB0aGUgY2x1c3RlcjsgZm9yIGV4YW1wbGU6IGNsdXN0ZXItQUJDRDEyMzRFRkdINTY3OElKS0w5ME1OT1AuIFRoZSBjbHVzdGVyIElEIHVuaXF1ZWx5XG4gICAqIGlkZW50aWZpZXMgdGhlIGNsdXN0ZXIgYW5kIGlzIHVzZWQgaW4gdGhpbmdzIGxpa2UgSUFNIGF1dGhlbnRpY2F0aW9uIHBvbGljaWVzLlxuICAgKiBAYXR0cmlidXRlIENsdXN0ZXJSZXNvdXJjZUlkXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgY2x1c3RlclJlc291cmNlSWRlbnRpZmllcjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgVlBDIHdoZXJlIHRoZSBEQiBzdWJuZXQgZ3JvdXAgaXMgY3JlYXRlZC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB2cGM6IGVjMi5JVnBjO1xuXG4gIC8qKlxuICAgKiBUaGUgc3VibmV0cyB1c2VkIGJ5IHRoZSBEQiBzdWJuZXQgZ3JvdXAuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdnBjU3VibmV0czogZWMyLlN1Ym5ldFNlbGVjdGlvbjtcblxuICAvKipcbiAgICogU3VibmV0IGdyb3VwIHVzZWQgYnkgdGhlIERCXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgc3VibmV0R3JvdXA6IElTdWJuZXRHcm91cDtcblxuICAvKipcbiAgICogSWRlbnRpZmllcnMgb2YgdGhlIGluc3RhbmNlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgaW5zdGFuY2VJZGVudGlmaWVyczogc3RyaW5nW10gPSBbXTtcblxuICAvKipcbiAgICogRW5kcG9pbnRzIHdoaWNoIGFkZHJlc3MgZWFjaCBpbmRpdmlkdWFsIGluc3RhbmNlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGluc3RhbmNlRW5kcG9pbnRzOiBFbmRwb2ludFtdID0gW107XG5cbiAgcHJvdGVjdGVkIGVuYWJsZUlhbUF1dGhlbnRpY2F0aW9uPzogYm9vbGVhbjtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRGF0YWJhc2VDbHVzdGVyUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy52cGMgPSBwcm9wcy52cGM7XG4gICAgdGhpcy52cGNTdWJuZXRzID0gcHJvcHMudnBjU3VibmV0cyA/PyB7IHN1Ym5ldFR5cGU6IGVjMi5TdWJuZXRUeXBlLlBSSVZBVEVfV0lUSF9OQVQgfTtcblxuICAgIC8vIERldGVybWluZSB0aGUgc3VibmV0KHMpIHRvIGRlcGxveSB0aGUgTmVwdHVuZSBjbHVzdGVyIHRvXG4gICAgY29uc3QgeyBzdWJuZXRJZHMsIGludGVybmV0Q29ubmVjdGl2aXR5RXN0YWJsaXNoZWQgfSA9IHRoaXMudnBjLnNlbGVjdFN1Ym5ldHModGhpcy52cGNTdWJuZXRzKTtcblxuICAgIC8vIENhbm5vdCB0ZXN0IHdoZXRoZXIgdGhlIHN1Ym5ldHMgYXJlIGluIGRpZmZlcmVudCBBWnMsIGJ1dCBhdCBsZWFzdCB3ZSBjYW4gdGVzdCB0aGUgYW1vdW50LlxuICAgIGlmIChzdWJuZXRJZHMubGVuZ3RoIDwgMikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBDbHVzdGVyIHJlcXVpcmVzIGF0IGxlYXN0IDIgc3VibmV0cywgZ290ICR7c3VibmV0SWRzLmxlbmd0aH1gKTtcbiAgICB9XG5cbiAgICB0aGlzLnN1Ym5ldEdyb3VwID0gcHJvcHMuc3VibmV0R3JvdXAgPz8gbmV3IFN1Ym5ldEdyb3VwKHRoaXMsICdTdWJuZXRzJywge1xuICAgICAgZGVzY3JpcHRpb246IGBTdWJuZXRzIGZvciAke2lkfSBkYXRhYmFzZWAsXG4gICAgICB2cGM6IHRoaXMudnBjLFxuICAgICAgdnBjU3VibmV0czogdGhpcy52cGNTdWJuZXRzLFxuICAgICAgcmVtb3ZhbFBvbGljeTogcHJvcHMucmVtb3ZhbFBvbGljeSA9PT0gUmVtb3ZhbFBvbGljeS5SRVRBSU4gPyBwcm9wcy5yZW1vdmFsUG9saWN5IDogdW5kZWZpbmVkLFxuICAgIH0pO1xuXG4gICAgY29uc3Qgc2VjdXJpdHlHcm91cHMgPSBwcm9wcy5zZWN1cml0eUdyb3VwcyA/PyBbXG4gICAgICBuZXcgZWMyLlNlY3VyaXR5R3JvdXAodGhpcywgJ1NlY3VyaXR5R3JvdXAnLCB7XG4gICAgICAgIGRlc2NyaXB0aW9uOiAnTmVwdHVuZSBzZWN1cml0eSBncm91cCcsXG4gICAgICAgIHZwYzogdGhpcy52cGMsXG4gICAgICB9KSxcbiAgICBdO1xuXG4gICAgLy8gRGVmYXVsdCB0byBlbmNyeXB0ZWQgc3RvcmFnZVxuICAgIGNvbnN0IHN0b3JhZ2VFbmNyeXB0ZWQgPSBwcm9wcy5zdG9yYWdlRW5jcnlwdGVkID8/IHRydWU7XG5cbiAgICBpZiAocHJvcHMua21zS2V5ICYmICFzdG9yYWdlRW5jcnlwdGVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0tNUyBrZXkgc3VwcGxpZWQgYnV0IHN0b3JhZ2VFbmNyeXB0ZWQgaXMgZmFsc2UnKTtcbiAgICB9XG5cbiAgICBjb25zdCBkZWxldGlvblByb3RlY3Rpb24gPSBwcm9wcy5kZWxldGlvblByb3RlY3Rpb24gPz8gKHByb3BzLnJlbW92YWxQb2xpY3kgPT09IFJlbW92YWxQb2xpY3kuUkVUQUlOID8gdHJ1ZSA6IHVuZGVmaW5lZCk7XG5cbiAgICB0aGlzLmVuYWJsZUlhbUF1dGhlbnRpY2F0aW9uID0gcHJvcHMuaWFtQXV0aGVudGljYXRpb247XG5cbiAgICAvLyBDcmVhdGUgdGhlIE5lcHR1bmUgY2x1c3RlclxuICAgIGNvbnN0IGNsdXN0ZXIgPSBuZXcgQ2ZuREJDbHVzdGVyKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIC8vIEJhc2ljXG4gICAgICBlbmdpbmVWZXJzaW9uOiBwcm9wcy5lbmdpbmVWZXJzaW9uPy52ZXJzaW9uLFxuICAgICAgZGJDbHVzdGVySWRlbnRpZmllcjogcHJvcHMuZGJDbHVzdGVyTmFtZSxcbiAgICAgIGRiU3VibmV0R3JvdXBOYW1lOiB0aGlzLnN1Ym5ldEdyb3VwLnN1Ym5ldEdyb3VwTmFtZSxcbiAgICAgIHBvcnQ6IHByb3BzLnBvcnQsXG4gICAgICB2cGNTZWN1cml0eUdyb3VwSWRzOiBzZWN1cml0eUdyb3Vwcy5tYXAoc2cgPT4gc2cuc2VjdXJpdHlHcm91cElkKSxcbiAgICAgIGRiQ2x1c3RlclBhcmFtZXRlckdyb3VwTmFtZTogcHJvcHMuY2x1c3RlclBhcmFtZXRlckdyb3VwPy5jbHVzdGVyUGFyYW1ldGVyR3JvdXBOYW1lLFxuICAgICAgZGVsZXRpb25Qcm90ZWN0aW9uOiBkZWxldGlvblByb3RlY3Rpb24sXG4gICAgICBhc3NvY2lhdGVkUm9sZXM6IHByb3BzLmFzc29jaWF0ZWRSb2xlcyA/IHByb3BzLmFzc29jaWF0ZWRSb2xlcy5tYXAocm9sZSA9PiAoeyByb2xlQXJuOiByb2xlLnJvbGVBcm4gfSkpIDogdW5kZWZpbmVkLFxuICAgICAgaWFtQXV0aEVuYWJsZWQ6IExhenkuYW55KHsgcHJvZHVjZTogKCkgPT4gdGhpcy5lbmFibGVJYW1BdXRoZW50aWNhdGlvbiB9KSxcbiAgICAgIC8vIEJhY2t1cFxuICAgICAgYmFja3VwUmV0ZW50aW9uUGVyaW9kOiBwcm9wcy5iYWNrdXBSZXRlbnRpb24/LnRvRGF5cygpLFxuICAgICAgcHJlZmVycmVkQmFja3VwV2luZG93OiBwcm9wcy5wcmVmZXJyZWRCYWNrdXBXaW5kb3csXG4gICAgICBwcmVmZXJyZWRNYWludGVuYW5jZVdpbmRvdzogcHJvcHMucHJlZmVycmVkTWFpbnRlbmFuY2VXaW5kb3csXG4gICAgICAvLyBFbmNyeXB0aW9uXG4gICAgICBrbXNLZXlJZDogcHJvcHMua21zS2V5Py5rZXlBcm4sXG4gICAgICBzdG9yYWdlRW5jcnlwdGVkLFxuICAgIH0pO1xuXG4gICAgY2x1c3Rlci5hcHBseVJlbW92YWxQb2xpY3kocHJvcHMucmVtb3ZhbFBvbGljeSwge1xuICAgICAgYXBwbHlUb1VwZGF0ZVJlcGxhY2VQb2xpY3k6IHRydWUsXG4gICAgfSk7XG5cbiAgICB0aGlzLmNsdXN0ZXJJZGVudGlmaWVyID0gY2x1c3Rlci5yZWY7XG4gICAgdGhpcy5jbHVzdGVyUmVzb3VyY2VJZGVudGlmaWVyID0gY2x1c3Rlci5hdHRyQ2x1c3RlclJlc291cmNlSWQ7XG5cbiAgICBjb25zdCBwb3J0ID0gVG9rZW4uYXNOdW1iZXIoY2x1c3Rlci5hdHRyUG9ydCk7XG4gICAgdGhpcy5jbHVzdGVyRW5kcG9pbnQgPSBuZXcgRW5kcG9pbnQoY2x1c3Rlci5hdHRyRW5kcG9pbnQsIHBvcnQpO1xuICAgIHRoaXMuY2x1c3RlclJlYWRFbmRwb2ludCA9IG5ldyBFbmRwb2ludChjbHVzdGVyLmF0dHJSZWFkRW5kcG9pbnQsIHBvcnQpO1xuXG4gICAgLy8gQ3JlYXRlIHRoZSBpbnN0YW5jZXNcbiAgICBjb25zdCBpbnN0YW5jZUNvdW50ID0gcHJvcHMuaW5zdGFuY2VzID8/IERhdGFiYXNlQ2x1c3Rlci5ERUZBVUxUX05VTV9JTlNUQU5DRVM7XG4gICAgaWYgKGluc3RhbmNlQ291bnQgPCAxKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0F0IGxlYXN0IG9uZSBpbnN0YW5jZSBpcyByZXF1aXJlZCcpO1xuICAgIH1cblxuICAgIGZvciAobGV0IGkgPSAwOyBpIDwgaW5zdGFuY2VDb3VudDsgaSsrKSB7XG4gICAgICBjb25zdCBpbnN0YW5jZUluZGV4ID0gaSArIDE7XG5cbiAgICAgIGNvbnN0IGluc3RhbmNlSWRlbnRpZmllciA9IHByb3BzLmluc3RhbmNlSWRlbnRpZmllckJhc2UgIT0gbnVsbCA/IGAke3Byb3BzLmluc3RhbmNlSWRlbnRpZmllckJhc2V9JHtpbnN0YW5jZUluZGV4fWBcbiAgICAgICAgOiBwcm9wcy5kYkNsdXN0ZXJOYW1lICE9IG51bGwgPyBgJHtwcm9wcy5kYkNsdXN0ZXJOYW1lfWluc3RhbmNlJHtpbnN0YW5jZUluZGV4fWAgOiB1bmRlZmluZWQ7XG5cbiAgICAgIGNvbnN0IGluc3RhbmNlID0gbmV3IENmbkRCSW5zdGFuY2UodGhpcywgYEluc3RhbmNlJHtpbnN0YW5jZUluZGV4fWAsIHtcbiAgICAgICAgLy8gTGluayB0byBjbHVzdGVyXG4gICAgICAgIGRiQ2x1c3RlcklkZW50aWZpZXI6IGNsdXN0ZXIucmVmLFxuICAgICAgICBkYkluc3RhbmNlSWRlbnRpZmllcjogaW5zdGFuY2VJZGVudGlmaWVyLFxuICAgICAgICAvLyBJbnN0YW5jZSBwcm9wZXJ0aWVzXG4gICAgICAgIGRiSW5zdGFuY2VDbGFzczogcHJvcHMuaW5zdGFuY2VUeXBlLl9pbnN0YW5jZVR5cGUsXG4gICAgICAgIGRiUGFyYW1ldGVyR3JvdXBOYW1lOiBwcm9wcy5wYXJhbWV0ZXJHcm91cD8ucGFyYW1ldGVyR3JvdXBOYW1lLFxuICAgICAgICBhdXRvTWlub3JWZXJzaW9uVXBncmFkZTogcHJvcHMuYXV0b01pbm9yVmVyc2lvblVwZ3JhZGUgPT09IHRydWUsXG4gICAgICB9KTtcblxuICAgICAgLy8gV2UgbXVzdCBoYXZlIGEgZGVwZW5kZW5jeSBvbiB0aGUgTkFUIGdhdGV3YXkgcHJvdmlkZXIgaGVyZSB0byBjcmVhdGVcbiAgICAgIC8vIHRoaW5ncyBpbiB0aGUgcmlnaHQgb3JkZXIuXG4gICAgICBpbnN0YW5jZS5ub2RlLmFkZERlcGVuZGVuY3koaW50ZXJuZXRDb25uZWN0aXZpdHlFc3RhYmxpc2hlZCk7XG5cbiAgICAgIGluc3RhbmNlLmFwcGx5UmVtb3ZhbFBvbGljeShwcm9wcy5yZW1vdmFsUG9saWN5LCB7XG4gICAgICAgIGFwcGx5VG9VcGRhdGVSZXBsYWNlUG9saWN5OiB0cnVlLFxuICAgICAgfSk7XG5cbiAgICAgIHRoaXMuaW5zdGFuY2VJZGVudGlmaWVycy5wdXNoKGluc3RhbmNlLnJlZik7XG4gICAgICB0aGlzLmluc3RhbmNlRW5kcG9pbnRzLnB1c2gobmV3IEVuZHBvaW50KGluc3RhbmNlLmF0dHJFbmRwb2ludCwgcG9ydCkpO1xuICAgIH1cblxuICAgIHRoaXMuY29ubmVjdGlvbnMgPSBuZXcgZWMyLkNvbm5lY3Rpb25zKHtcbiAgICAgIGRlZmF1bHRQb3J0OiBlYzIuUG9ydC50Y3AocG9ydCksXG4gICAgICBzZWN1cml0eUdyb3Vwczogc2VjdXJpdHlHcm91cHMsXG4gICAgfSk7XG4gIH1cbn1cbiJdfQ==