"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const util_1 = require("./util");
const ESBUILD_VERSION = '0';
/**
 * Bundling with esbuild
 */
class Bundling {
    constructor(props) {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        this.props = props;
        Bundling.runsLocally = (_c = (_a = Bundling.runsLocally) !== null && _a !== void 0 ? _a : (_b = util_1.getEsBuildVersion()) === null || _b === void 0 ? void 0 : _b.startsWith(ESBUILD_VERSION)) !== null && _c !== void 0 ? _c : false;
        const projectRoot = path.dirname(props.depsLockFilePath);
        this.relativeEntryPath = path.relative(projectRoot, path.resolve(props.entry));
        if (props.tsconfig) {
            this.relativeTsconfigPath = path.relative(projectRoot, path.resolve(props.tsconfig));
        }
        this.externals = [
            ...(_d = props.externalModules) !== null && _d !== void 0 ? _d : ['aws-sdk'],
            ...(_e = props.nodeModules) !== null && _e !== void 0 ? _e : [],
        ];
        // Docker bundling
        const shouldBuildImage = props.forceDockerBundling || !Bundling.runsLocally;
        this.image = shouldBuildImage
            ? (_f = props.dockerImage) !== null && _f !== void 0 ? _f : cdk.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
                buildArgs: {
                    ...(_g = props.buildArgs) !== null && _g !== void 0 ? _g : {},
                    IMAGE: props.runtime.bundlingDockerImage.image,
                    ESBUILD_VERSION: (_h = props.esbuildVersion) !== null && _h !== void 0 ? _h : ESBUILD_VERSION,
                },
            }) : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand(cdk.AssetStaging.BUNDLING_INPUT_DIR, cdk.AssetStaging.BUNDLING_OUTPUT_DIR);
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = props.environment;
        // Bundling sets the working directory to cdk.AssetStaging.BUNDLING_INPUT_DIR
        // and we want to force npx to use the globally installed esbuild.
        this.workingDirectory = '/';
        // Local bundling
        if (!props.forceDockerBundling) { // only if Docker is not forced
            const osPlatform = os.platform();
            const createLocalCommand = (outputDir) => this.createBundlingCommand(projectRoot, outputDir, osPlatform);
            this.local = {
                tryBundle(outputDir) {
                    var _a;
                    if (Bundling.runsLocally === false) {
                        process.stderr.write('esbuild cannot run locally. Switching to Docker bundling.\n');
                        return false;
                    }
                    const localCommand = createLocalCommand(outputDir);
                    util_1.exec(osPlatform === 'win32' ? 'cmd' : 'bash', [
                        osPlatform === 'win32' ? '/c' : '-c',
                        localCommand,
                    ], {
                        env: { ...process.env, ...(_a = props.environment) !== null && _a !== void 0 ? _a : {} },
                        stdio: [
                            'ignore',
                            process.stderr,
                            'inherit',
                        ],
                        cwd: path.dirname(props.entry),
                        windowsVerbatimArguments: osPlatform === 'win32',
                    });
                    return true;
                },
            };
        }
    }
    /**
     * esbuild bundled Lambda asset code
     */
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(path.dirname(options.depsLockFilePath), {
            assetHashType: cdk.AssetHashType.OUTPUT,
            bundling: new Bundling(options),
        });
    }
    static clearRunsLocallyCache() {
        this.runsLocally = undefined;
    }
    createBundlingCommand(inputDir, outputDir, osPlatform = 'linux') {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j;
        const pathJoin = osPathJoin(osPlatform);
        const npx = osPlatform === 'win32' ? 'npx.cmd' : 'npx';
        const loaders = Object.entries((_a = this.props.loader) !== null && _a !== void 0 ? _a : {});
        const defines = Object.entries((_b = this.props.define) !== null && _b !== void 0 ? _b : {});
        const esbuildCommand = [
            npx, 'esbuild',
            '--bundle', `"${pathJoin(inputDir, this.relativeEntryPath)}"`,
            `--target=${(_c = this.props.target) !== null && _c !== void 0 ? _c : toTarget(this.props.runtime)}`,
            '--platform=node',
            `--outfile="${pathJoin(outputDir, 'index.js')}"`,
            ...this.props.minify ? ['--minify'] : [],
            ...this.props.sourceMap ? ['--sourcemap'] : [],
            ...this.externals.map(external => `--external:${external}`),
            ...loaders.map(([ext, name]) => `--loader:${ext}=${name}`),
            ...defines.map(([key, value]) => `--define:${key}=${JSON.stringify(value)}`),
            ...this.props.logLevel ? [`--log-level=${this.props.logLevel}`] : [],
            ...this.props.keepNames ? ['--keep-names'] : [],
            ...this.relativeTsconfigPath ? [`--tsconfig=${pathJoin(inputDir, this.relativeTsconfigPath)}`] : [],
            ...this.props.metafile ? [`--metafile=${pathJoin(outputDir, 'index.meta.json')}`] : [],
            ...this.props.banner ? [`--banner='${this.props.banner}'`] : [],
            ...this.props.footer ? [`--footer='${this.props.footer}'`] : [],
        ].join(' ');
        let depsCommand = '';
        if (this.props.nodeModules) {
            // Find 'package.json' closest to entry folder, we are going to extract the
            // modules versions from it.
            const pkgPath = util_1.findUp('package.json', path.dirname(this.props.entry));
            if (!pkgPath) {
                throw new Error('Cannot find a `package.json` in this project. Using `nodeModules` requires a `package.json`.');
            }
            // Determine dependencies versions, lock file and installer
            const dependencies = util_1.extractDependencies(pkgPath, this.props.nodeModules);
            let installer = Installer.NPM;
            let lockFile = util_1.LockFile.NPM;
            if (this.props.depsLockFilePath.endsWith(util_1.LockFile.YARN)) {
                lockFile = util_1.LockFile.YARN;
                installer = Installer.YARN;
            }
            const osCommand = new OsCommand(osPlatform);
            // Create dummy package.json, copy lock file if any and then install
            depsCommand = chain([
                osCommand.writeJson(pathJoin(outputDir, 'package.json'), { dependencies }),
                osCommand.copy(pathJoin(inputDir, lockFile), pathJoin(outputDir, lockFile)),
                osCommand.changeDirectory(outputDir),
                `${installer} install`,
            ]);
        }
        return chain([
            ...(_e = (_d = this.props.commandHooks) === null || _d === void 0 ? void 0 : _d.beforeBundling(inputDir, outputDir)) !== null && _e !== void 0 ? _e : [],
            esbuildCommand,
            ...(_g = (this.props.nodeModules && ((_f = this.props.commandHooks) === null || _f === void 0 ? void 0 : _f.beforeInstall(inputDir, outputDir)))) !== null && _g !== void 0 ? _g : [],
            depsCommand,
            ...(_j = (_h = this.props.commandHooks) === null || _h === void 0 ? void 0 : _h.afterBundling(inputDir, outputDir)) !== null && _j !== void 0 ? _j : [],
        ]);
    }
}
exports.Bundling = Bundling;
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer || (Installer = {}));
/**
 * OS agnostic command
 */
class OsCommand {
    constructor(osPlatform) {
        this.osPlatform = osPlatform;
    }
    writeJson(filePath, data) {
        const stringifiedData = JSON.stringify(data);
        if (this.osPlatform === 'win32') {
            return `echo ^${stringifiedData}^ > ${filePath}`;
        }
        return `echo '${stringifiedData}' > ${filePath}`;
    }
    copy(src, dest) {
        if (this.osPlatform === 'win32') {
            return `copy ${src} ${dest}`;
        }
        return `cp ${src} ${dest}`;
    }
    changeDirectory(dir) {
        return `cd ${dir}`;
    }
}
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
/**
 * Converts a runtime to an esbuild node target
 */
function toTarget(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return `node${match[1]}`;
}
//# sourceMappingURL=data:application/json;base64,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