"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn = require("@aws-cdk/aws-cloudformation");
const cpactions = require("@aws-cdk/aws-codepipeline-actions");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cdk = require("@aws-cdk/core");
/**
 * A class to deploy a stack that is part of a CDK App, using CodePipeline.
 * This composite Action takes care of preparing and executing a CloudFormation ChangeSet.
 *
 * It currently does *not* support stacks that make use of ``Asset``s, and
 * requires the deployed stack is in the same account and region where the
 * CodePipeline is hosted.
 */
class PipelineDeployStackAction {
    constructor(props) {
        this.stack = props.stack;
        const assets = this.stack.node.metadata.filter(md => md.type === cxschema.ArtifactMetadataEntryType.ASSET);
        if (assets.length > 0) {
            // FIXME: Implement the necessary actions to publish assets
            throw new Error(`Cannot deploy the stack ${this.stack.stackName} because it references ${assets.length} asset(s)`);
        }
        const createChangeSetRunOrder = props.createChangeSetRunOrder || 1;
        const executeChangeSetRunOrder = props.executeChangeSetRunOrder || (createChangeSetRunOrder + 1);
        if (createChangeSetRunOrder >= executeChangeSetRunOrder) {
            throw new Error(`createChangeSetRunOrder (${createChangeSetRunOrder}) must be < executeChangeSetRunOrder (${executeChangeSetRunOrder})`);
        }
        const changeSetName = props.changeSetName || 'CDK-CodePipeline-ChangeSet';
        const capabilities = cfnCapabilities(props.adminPermissions, props.capabilities);
        this.prepareChangeSetAction = new cpactions.CloudFormationCreateReplaceChangeSetAction({
            actionName: 'ChangeSet',
            changeSetName,
            runOrder: createChangeSetRunOrder,
            stackName: props.stack.stackName,
            templatePath: props.input.atPath(props.stack.templateFile),
            adminPermissions: props.adminPermissions,
            deploymentRole: props.role,
            capabilities,
        });
        this.executeChangeSetAction = new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: 'Execute',
            changeSetName,
            runOrder: executeChangeSetRunOrder,
            stackName: this.stack.stackName,
        });
    }
    bind(scope, stage, options) {
        if (this.stack.environment !== cdk.Stack.of(scope).environment) {
            // FIXME: Add the necessary to extend to stacks in a different account
            throw new Error('Cross-environment deployment is not supported');
        }
        stage.addAction(this.prepareChangeSetAction);
        this._deploymentRole = this.prepareChangeSetAction.deploymentRole;
        return this.executeChangeSetAction.bind(scope, stage, options);
    }
    get deploymentRole() {
        if (!this._deploymentRole) {
            throw new Error('Use this action in a pipeline first before accessing \'deploymentRole\'');
        }
        return this._deploymentRole;
    }
    /**
     * Add policy statements to the role deploying the stack.
     *
     * This role is passed to CloudFormation and must have the IAM permissions
     * necessary to deploy the stack or you can grant this role `adminPermissions`
     * by using that option during creation. If you do not grant
     * `adminPermissions` you need to identify the proper statements to add to
     * this role based on the CloudFormation Resources in your stack.
     */
    addToDeploymentRolePolicy(statement) {
        this.deploymentRole.addToPolicy(statement);
    }
    onStateChange(name, target, options) {
        return this.executeChangeSetAction.onStateChange(name, target, options);
    }
    get actionProperties() {
        return this.executeChangeSetAction.actionProperties;
    }
}
exports.PipelineDeployStackAction = PipelineDeployStackAction;
function cfnCapabilities(adminPermissions, capabilities) {
    if (adminPermissions && capabilities === undefined) {
        // admin true default capability to NamedIAM and AutoExpand
        return [cfn.CloudFormationCapabilities.NAMED_IAM, cfn.CloudFormationCapabilities.AUTO_EXPAND];
    }
    else if (capabilities === undefined) {
        // else capabilities are undefined set AnonymousIAM and AutoExpand
        return [cfn.CloudFormationCapabilities.ANONYMOUS_IAM, cfn.CloudFormationCapabilities.AUTO_EXPAND];
    }
    else {
        // else capabilities are defined use them
        return capabilities;
    }
}
//# sourceMappingURL=data:application/json;base64,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