# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['aws_lambda_powertools',
 'aws_lambda_powertools.helper',
 'aws_lambda_powertools.logging',
 'aws_lambda_powertools.metrics',
 'aws_lambda_powertools.tracing']

package_data = \
{'': ['*']}

install_requires = \
['aws-xray-sdk>=2.4.3,<3.0.0', 'fastjsonschema>=2.14.4,<3.0.0']

setup_kwargs = {
    'name': 'aws-lambda-powertools',
    'version': '0.6.3',
    'description': 'Python utilities for AWS Lambda functions including but not limited to tracing, logging and custom metric',
    'long_description': '# Lambda Powertools\n\n![PackageStatus](https://img.shields.io/static/v1?label=status&message=beta&color=blueviolet?style=flat-square) ![PythonSupport](https://img.shields.io/static/v1?label=python&message=3.6%20|%203.7|%203.8&color=blue?style=flat-square&logo=python)\n\nA suite of utilities for AWS Lambda Functions that makes tracing with AWS X-Ray, structured logging and creating custom metrics asynchronously easier - Currently available for Python only and compatible with Python >=3.6.\n\n**Status**: Beta\n\n## Features\n\n**Tracing**\n\n> It currently uses AWS X-Ray\n\n* Decorators that capture cold start as annotation, and response and exceptions as metadata\n* Run functions locally without code change to disable tracing\n* Explicitly disable tracing via env var `POWERTOOLS_TRACE_DISABLED="true"`\n\n**Logging**\n\n* Decorators that capture key fields from Lambda context, cold start and structures logging output as JSON\n* Optionally log Lambda request when instructed (disabled by default)\n    - Enable via `POWERTOOLS_LOGGER_LOG_EVENT="true"` or explicitly via decorator param\n* Logs canonical custom metric line to logs that can be consumed asynchronously\n* Log sampling enables DEBUG log level for a percentage of requests (disabled by default)\n    - Enable via `POWERTOOLS_LOGGER_SAMPLE_RATE=0.1`, ranges from 0 to 1, where 0.1 is 10% and 1 is 100%\n\n**Metrics**\n\n* Aggregate up to 100 metrics using a single CloudWatch Embedded Metric Format object (large JSON blob)\n* Context manager to create an one off metric with a different dimension than metrics already aggregated\n* Validate against common metric definitions mistakes (metric unit, values, max dimensions, max metrics, etc)\n* No stack, custom resource, data collection needed — Metrics are created async by CloudWatch EMF\n\n**Environment variables** used across suite of utilities\n\nEnvironment variable | Description | Default | Utility\n------------------------------------------------- | --------------------------------------------------------------------------------- | --------------------------------------------------------------------------------- | -------------------------------------------------\nPOWERTOOLS_SERVICE_NAME | Sets service name used for tracing namespace, metrics dimensions and structured logging | "service_undefined" | all\nPOWERTOOLS_TRACE_DISABLED | Disables tracing | "false" | tracing\nPOWERTOOLS_LOGGER_LOG_EVENT | Logs incoming event | "false" | logging\nPOWERTOOLS_LOGGER_SAMPLE_RATE | Debug log sampling  | 0 | logging\nPOWERTOOLS_METRICS_NAMESPACE | Metrics namespace  | None | metrics\nLOG_LEVEL | Sets logging level | "INFO" | logging\n\n## Usage\n\n### Installation\n\nWith [pip](https://pip.pypa.io/en/latest/index.html) installed, run: ``pip install aws-lambda-powertools``\n\n### Tracing\n\n**Example SAM template using supported environment variables**\n\n```yaml\nGlobals:\n  Function:\n    Tracing: Active # can also be enabled per function\n    Environment:\n        Variables:\n            POWERTOOLS_SERVICE_NAME: "payment" \n            POWERTOOLS_TRACE_DISABLED: "false" \n```\n\n**Pseudo Python Lambda code**\n\n```python\nfrom aws_lambda_powertools.tracing import Tracer\ntracer = Tracer()\n# tracer = Tracer(service="payment") # can also be explicitly defined\n\n@tracer.capture_method\ndef collect_payment(charge_id):\n  # logic\n  ret = requests.post(PAYMENT_ENDPOINT)\n  # custom annotation\n  tracer.put_annotation("PAYMENT_STATUS", "SUCCESS")\n  return ret\n\n@tracer.capture_lambda_handler\ndef handler(event, context)\n  charge_id = event.get(\'charge_id\')\n  payment = collect_payment(charge_id)\n  ...\n```\n\n\n### Logging\n\n**Example SAM template using supported environment variables**\n\n```yaml\nGlobals:\n  Function:\n    Environment:\n        Variables:\n            POWERTOOLS_SERVICE_NAME: "payment" \n            POWERTOOLS_LOGGER_SAMPLE_RATE: 0.1 # enable debug logging for 1% of requests, 0% by default\n            LOG_LEVEL: "INFO"\n```\n\n**Pseudo Python Lambda code**\n\n```python\nfrom aws_lambda_powertools.logging import logger_setup, logger_inject_lambda_context\n\nlogger = logger_setup()  \n# logger_setup(service="payment") # also accept explicit service name\n# logger_setup(level="INFO") # also accept explicit log level\n\n@logger_inject_lambda_context\ndef handler(event, context)\n  logger.info("Collecting payment")\n  ...\n  # You can log entire objects too\n  logger.info({\n    "operation": "collect_payment",\n    "charge_id": event[\'charge_id\']\n  })\n  ...\n```\n\n**Exerpt output in CloudWatch Logs**\n\n```json\n{  \n   "timestamp":"2019-08-22 18:17:33,774",\n   "level":"INFO",\n   "location":"collect.handler:1",\n   "service":"payment",\n   "lambda_function_name":"test",\n   "lambda_function_memory_size":"128",\n   "lambda_function_arn":"arn:aws:lambda:eu-west-1:12345678910:function:test",\n   "lambda_request_id":"52fdfc07-2182-154f-163f-5f0f9a621d72",\n   "cold_start": "true",\n   "message": "Collecting payment"\n}\n\n{  \n   "timestamp":"2019-08-22 18:17:33,774",\n   "level":"INFO",\n   "location":"collect.handler:15",\n   "service":"payment",\n   "lambda_function_name":"test",\n   "lambda_function_memory_size":"128",\n   "lambda_function_arn":"arn:aws:lambda:eu-west-1:12345678910:function:test",\n   "lambda_request_id":"52fdfc07-2182-154f-163f-5f0f9a621d72",\n   "cold_start": "true",\n   "message":{  \n      "operation":"collect_payment",\n      "charge_id": "ch_AZFlk2345C0"\n   }\n}\n```\n\n#### Custom Metrics async\n\n> **NOTE** `log_metric` will be removed once it\'s GA.\n\nThis feature makes use of CloudWatch Embedded Metric Format (EMF) and metrics are created asynchronously by CloudWatch service\n\n> Contrary to `log_metric`, you don\'t need any custom resource or additional CloudFormation stack anymore.\n\nMetrics middleware validates against the minimum necessary for a metric to be published:\n\n* At least of one Metric and Dimension \n* Maximum of 9 dimensions\n* Only one Namespace\n* [Any Metric unit supported by CloudWatch](https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_MetricDatum.html)\n\n**Creating multiple metrics**\n\n`log_metrics` decorator calls the decorated function, so leave that for last decorator or will fail with `SchemaValidationError` if no metrics are recorded.\n\n```python\nfrom aws_lambda_powertools.metrics import Metrics, MetricUnit\n\nmetrics = Metrics()\nmetrics.add_namespace(name="ServerlessAirline")\nmetrics.add_metric(name="ColdStart", unit="Count", value=1)\nmetrics.add_dimension(name="service", value="booking")\n\n@metrics.log_metrics\n@tracer.capture_lambda_handler\ndef lambda_handler(evt, ctx):\n    metrics.add_metric(name="BookingConfirmation", unit="Count", value=1)\n    some_code()\n    return True\n\ndef some_code():\n    metrics.add_metric(name="some_other_metric", unit=MetricUnit.Seconds, value=1)\n    ...\n```\n\nCloudWatch EMF uses the same dimensions across all metrics. If you have metrics that should have different dimensions, use `single_metric` to create a single metric with any dimension you want. Generally, this would be an edge case since you [pay for unique metric](https://aws.amazon.com/cloudwatch/pricing/) \n\n> unique metric = (metric_name + dimension_name + dimension_value)\n\n```python\nfrom aws_lambda_powertools.metrics import MetricUnit, single_metric\n\nwith single_metric(name="ColdStart", unit=MetricUnit.Count, value=1) as metric:\n    metric.add_dimension(name="function_context", value="$LATEST")\n```\n\n## Beta\n\n> **[Progress towards GA](https://github.com/awslabs/aws-lambda-powertools/projects/1)**\n\nThis library may change its API/methods or environment variables as it receives feedback from customers. Currently looking for ideas in the following areas before making it stable:\n\n* **Should Tracer patch all possible imported libraries by default or only AWS SDKs?**\n    - Patching all libraries may have a small performance penalty (~50ms) at cold start\n    - Alternatively, we could patch only AWS SDK if available and to provide a param to patch multiple `Tracer(modules=("boto3", "requests"))` \n',
    'author': 'Amazon Web Services',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/awslabs/aws-lambda-powertools/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
