import * as ec2 from '@aws-cdk/aws-ec2';
import * as iam from '@aws-cdk/aws-iam';
import * as kms from '@aws-cdk/aws-kms';
import * as s3 from '@aws-cdk/aws-s3';
import * as secretsmanager from '@aws-cdk/aws-secretsmanager';
import { Duration, IResource, RemovalPolicy, Resource, SecretValue } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { Endpoint } from './endpoint';
import { IClusterParameterGroup } from './parameter-group';
import { IClusterSubnetGroup } from './subnet-group';
/**
 * Possible Node Types to use in the cluster
 * used for defining {@link ClusterProps.nodeType}.
 */
export declare enum NodeType {
    /**
     * ds2.xlarge
     */
    DS2_XLARGE = "ds2.xlarge",
    /**
     * ds2.8xlarge
     */
    DS2_8XLARGE = "ds2.8xlarge",
    /**
     * dc1.large
     */
    DC1_LARGE = "dc1.large",
    /**
     * dc1.8xlarge
     */
    DC1_8XLARGE = "dc1.8xlarge",
    /**
     * dc2.large
     */
    DC2_LARGE = "dc2.large",
    /**
     * dc2.8xlarge
     */
    DC2_8XLARGE = "dc2.8xlarge",
    /**
     * ra3.xlplus
     */
    RA3_XLPLUS = "ra3.xlplus",
    /**
     * ra3.4xlarge
     */
    RA3_4XLARGE = "ra3.4xlarge",
    /**
     * ra3.16xlarge
     */
    RA3_16XLARGE = "ra3.16xlarge"
}
/**
 * What cluster type to use.
 * Used by {@link ClusterProps.clusterType}
 */
export declare enum ClusterType {
    /**
     * single-node cluster, the {@link ClusterProps.numberOfNodes} parameter is not required
     */
    SINGLE_NODE = "single-node",
    /**
     * multi-node cluster, set the amount of nodes using {@link ClusterProps.numberOfNodes} parameter
     */
    MULTI_NODE = "multi-node"
}
/**
 * Username and password combination
 */
export interface Login {
    /**
     * Username
     */
    readonly masterUsername: string;
    /**
     * Password
     *
     * Do not put passwords in your CDK code directly.
     *
     * @default a Secrets Manager generated password
     */
    readonly masterPassword?: SecretValue;
    /**
     * KMS encryption key to encrypt the generated secret.
     *
     * @default default master key
     */
    readonly encryptionKey?: kms.IKey;
}
/**
 * Options to add the multi user rotation
 */
export interface RotationMultiUserOptions {
    /**
     * The secret to rotate. It must be a JSON string with the following format:
     * ```
     * {
     *   "engine": <required: database engine>,
     *   "host": <required: instance host name>,
     *   "username": <required: username>,
     *   "password": <required: password>,
     *   "dbname": <optional: database name>,
     *   "port": <optional: if not specified, default port will be used>,
     *   "masterarn": <required: the arn of the master secret which will be used to create users/change passwords>
     * }
     * ```
     */
    readonly secret: secretsmanager.ISecret;
    /**
     * Specifies the number of days after the previous rotation before
     * Secrets Manager triggers the next automatic rotation.
     *
     * @default Duration.days(30)
     */
    readonly automaticallyAfter?: Duration;
}
/**
 * Create a Redshift Cluster with a given number of nodes.
 * Implemented by {@link Cluster} via {@link ClusterBase}.
 */
export interface ICluster extends IResource, ec2.IConnectable, secretsmanager.ISecretAttachmentTarget {
    /**
     * Name of the cluster
     *
     * @attribute ClusterName
     */
    readonly clusterName: string;
    /**
     * The endpoint to use for read/write operations
     *
     * @attribute EndpointAddress,EndpointPort
     */
    readonly clusterEndpoint: Endpoint;
}
/**
 * Properties that describe an existing cluster instance
 */
export interface ClusterAttributes {
    /**
     * The security groups of the redshift cluster
     *
     * @default no security groups will be attached to the import
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * Identifier for the cluster
     */
    readonly clusterName: string;
    /**
     * Cluster endpoint address
     */
    readonly clusterEndpointAddress: string;
    /**
     * Cluster endpoint port
     */
    readonly clusterEndpointPort: number;
}
/**
 * Properties for a new database cluster
 */
export interface ClusterProps {
    /**
     * An optional identifier for the cluster
     *
     * @default - A name is automatically generated.
     */
    readonly clusterName?: string;
    /**
     * Additional parameters to pass to the database engine
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-parameter-groups.html
     *
     * @default - No parameter group.
     */
    readonly parameterGroup?: IClusterParameterGroup;
    /**
     * Number of compute nodes in the cluster. Only specify this property for multi-node clusters.
     *
     * Value must be at least 2 and no more than 100.
     *
     * @default - 2 if `clusterType` is ClusterType.MULTI_NODE, undefined otherwise
     */
    readonly numberOfNodes?: number;
    /**
     * The node type to be provisioned for the cluster.
     *
     * @default {@link NodeType.DC2_LARGE}
     */
    readonly nodeType?: NodeType;
    /**
     * Settings for the individual instances that are launched
     *
     * @default {@link ClusterType.MULTI_NODE}
     */
    readonly clusterType?: ClusterType;
    /**
     * What port to listen on
     *
     * @default - The default for the engine is used.
     */
    readonly port?: number;
    /**
     * Whether to enable encryption of data at rest in the cluster.
     *
     * @default true
     */
    readonly encrypted?: boolean;
    /**
     * The KMS key to use for encryption of data at rest.
     *
     * @default - AWS-managed key, if encryption at rest is enabled
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * A preferred maintenance window day/time range. Should be specified as a range ddd:hh24:mi-ddd:hh24:mi (24H Clock UTC).
     *
     * Example: 'Sun:23:45-Mon:00:15'
     *
     * @default - 30-minute window selected at random from an 8-hour block of time for
     * each AWS Region, occurring on a random day of the week.
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_UpgradeDBInstance.Maintenance.html#Concepts.DBMaintenance
     */
    readonly preferredMaintenanceWindow?: string;
    /**
     * The VPC to place the cluster in.
     */
    readonly vpc: ec2.IVpc;
    /**
     * Where to place the instances within the VPC
     *
     * @default - private subnets
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * Security group.
     *
     * @default - a new security group is created.
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * A cluster subnet group to use with this cluster.
     *
     * @default - a new subnet group will be created.
     */
    readonly subnetGroup?: IClusterSubnetGroup;
    /**
     * Username and password for the administrative user
     */
    readonly masterUser: Login;
    /**
     * A list of AWS Identity and Access Management (IAM) role that can be used by the cluster to access other AWS services.
     * Specify a maximum of 10 roles.
     *
     * @default - No role is attached to the cluster.
     */
    readonly roles?: iam.IRole[];
    /**
     * Name of a database which is automatically created inside the cluster
     *
     * @default - default_db
     */
    readonly defaultDatabaseName?: string;
    /**
     * Bucket to send logs to.
     * Logging information includes queries and connection attempts, for the specified Amazon Redshift cluster.
     *
     * @default - No Logs
     */
    readonly loggingBucket?: s3.IBucket;
    /**
     * Prefix used for logging
     *
     * @default - no prefix
     */
    readonly loggingKeyPrefix?: string;
    /**
     * The removal policy to apply when the cluster and its instances are removed
     * from the stack or replaced during an update.
     *
     * @default RemovalPolicy.RETAIN
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * Whether to make cluster publicly accessible.
     *
     * @default false
     */
    readonly publiclyAccessible?: boolean;
}
/**
 * A new or imported clustered database.
 */
declare abstract class ClusterBase extends Resource implements ICluster {
    /**
     * Name of the cluster
     */
    abstract readonly clusterName: string;
    /**
     * The endpoint to use for read/write operations
     */
    abstract readonly clusterEndpoint: Endpoint;
    /**
     * Access to the network connections
     */
    abstract readonly connections: ec2.Connections;
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget(): secretsmanager.SecretAttachmentTargetProps;
}
/**
 * Create a Redshift cluster a given number of nodes.
 *
 * @resource AWS::Redshift::Cluster
 */
export declare class Cluster extends ClusterBase {
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromClusterAttributes(scope: Construct, id: string, attrs: ClusterAttributes): ICluster;
    /**
     * Identifier of the cluster
     */
    readonly clusterName: string;
    /**
     * The endpoint to use for read/write operations
     */
    readonly clusterEndpoint: Endpoint;
    /**
     * Access to the network connections
     */
    readonly connections: ec2.Connections;
    /**
     * The secret attached to this cluster
     */
    readonly secret?: secretsmanager.ISecret;
    private readonly singleUserRotationApplication;
    private readonly multiUserRotationApplication;
    /**
     * The VPC where the DB subnet group is created.
     */
    private readonly vpc;
    /**
     * The subnets used by the DB subnet group.
     */
    private readonly vpcSubnets?;
    constructor(scope: Construct, id: string, props: ClusterProps);
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter?: Duration): secretsmanager.SecretRotation;
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id: string, options: RotationMultiUserOptions): secretsmanager.SecretRotation;
    private validateNodeCount;
}
export {};
