"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToSqs = void 0;
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
const cdk = require("@aws-cdk/core");
/**
 * @summary The ApiGatewayToSqs class.
 */
class ApiGatewayToSqs extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the ApiGatewayToSqs class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the dead letter queue, if applicable
        this.deadLetterQueue = defaults.buildDeadLetterQueue(this, {
            existingQueueObj: props.existingQueueObj,
            deployDeadLetterQueue: props.deployDeadLetterQueue,
            deadLetterQueueProps: props.deadLetterQueueProps,
            maxReceiveCount: props.maxReceiveCount
        });
        // Setup the queue
        [this.sqsQueue] = defaults.buildQueue(this, 'queue', {
            queueProps: props.queueProps,
            deadLetterQueue: this.deadLetterQueue
        });
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, props.apiGatewayProps);
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        // Setup the API Gateway resource
        const apiGatewayResource = this.apiGateway.root.addResource('message');
        // Create
        let createRequestTemplate = "Action=SendMessage&MessageBody=$util.urlEncode(\"$input.body\")";
        if (props.createRequestTemplate) {
            createRequestTemplate = props.createRequestTemplate;
        }
        if (props.allowCreateOperation && props.allowCreateOperation === true) {
            this.addActionToPolicy("sqs:SendMessage");
            defaults.addProxyMethodToApiResource({
                service: "sqs",
                path: `${cdk.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "POST",
                apiResource: this.apiGateway.root,
                requestTemplate: createRequestTemplate,
                contentType: "'application/x-www-form-urlencoded'"
            });
        }
        // Read
        let readRequestTemplate = "Action=ReceiveMessage";
        if (props.readRequestTemplate) {
            readRequestTemplate = props.readRequestTemplate;
        }
        if (props.allowReadOperation === undefined || props.allowReadOperation === true) {
            this.addActionToPolicy("sqs:ReceiveMessage");
            defaults.addProxyMethodToApiResource({
                service: "sqs",
                path: `${cdk.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "GET",
                apiResource: this.apiGateway.root,
                requestTemplate: readRequestTemplate,
                contentType: "'application/x-www-form-urlencoded'"
            });
        }
        // Delete
        let deleteRequestTemplate = "Action=DeleteMessage&ReceiptHandle=$util.urlEncode($input.params('receiptHandle'))";
        if (props.deleteRequestTemplate) {
            deleteRequestTemplate = props.deleteRequestTemplate;
        }
        if (props.allowDeleteOperation && props.allowDeleteOperation === true) {
            this.addActionToPolicy("sqs:DeleteMessage");
            defaults.addProxyMethodToApiResource({
                service: "sqs",
                path: `${cdk.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "DELETE",
                apiResource: apiGatewayResource,
                requestTemplate: deleteRequestTemplate,
                contentType: "'application/x-www-form-urlencoded'"
            });
        }
    }
    addActionToPolicy(action) {
        this.apiGatewayRole.addToPolicy(new iam.PolicyStatement({
            resources: [
                this.sqsQueue.queueArn
            ],
            actions: [`${action}`]
        }));
    }
}
exports.ApiGatewayToSqs = ApiGatewayToSqs;
//# sourceMappingURL=data:application/json;base64,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