"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const s3 = require("@aws-cdk/aws-s3");
const lib_1 = require("../lib");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test Case 1 - Pattern deployment w/ default properties
// --------------------------------------------------------------
test('Pattern deployment w/ default properties', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ticker_symbol',
                                sqlType: 'VARCHAR(4)',
                                mapping: '$.ticker_symbol'
                            }, {
                                name: 'sector',
                                sqlType: 'VARCHAR(16)',
                                mapping: '$.sector'
                            }, {
                                name: 'change',
                                sqlType: 'REAL',
                                mapping: '$.change'
                            }, {
                                name: 'price',
                                sqlType: 'REAL',
                                mapping: '$.price'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        },
        bucketProps: {
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        }
    };
    new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'test-firehose-s3-and-analytics-stack', props);
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test Case 2 - Test the getter methods
// --------------------------------------------------------------
test('Test properties', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        }
    };
    const app = new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'test-kinesis-firehose-kinesis-analytics', props);
    // Assertions
    expect(app.kinesisAnalytics !== null);
    expect(app.kinesisFirehose !== null);
    expect(app.kinesisFirehoseRole !== null);
    expect(app.kinesisFirehoseLogGroup !== null);
    expect(app.s3Bucket !== null);
    expect(app.s3LoggingBucket !== null);
});
// --------------------------------------------------------------
// Test Case 3 - Override kinesisFirehose props
// --------------------------------------------------------------
test('test kinesisFirehose override ', () => {
    const stack = new core_1.Stack();
    new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'test-firehose-s3', {
        kinesisFirehoseProps: {
            extendedS3DestinationConfiguration: {
                bufferingHints: {
                    intervalInSeconds: 600,
                    sizeInMBs: 55
                },
            }
        },
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        }
    });
    expect(stack).toHaveResourceLike("AWS::KinesisFirehose::DeliveryStream", {
        ExtendedS3DestinationConfiguration: {
            BufferingHints: {
                IntervalInSeconds: 600,
                SizeInMBs: 55
            }
        }
    });
});
// --------------------------------------------------------------
// Test bad call with existingBucket and bucketProps
// --------------------------------------------------------------
test("Test bad call with existingBucket and bucketProps", () => {
    // Stack
    const stack = new core_1.Stack();
    const testBucket = new s3.Bucket(stack, 'test-bucket', {});
    const app = () => {
        // Helper declaration
        new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, "bad-s3-args", {
            existingBucketObj: testBucket,
            bucketProps: {
                removalPolicy: core_1.RemovalPolicy.DESTROY
            },
        });
    };
    // Assertion
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,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