# Copyright 2022 Amethyst Reese
# Licensed under the MIT license

import textwrap
from pathlib import Path

from jinja2 import Template

from .project import Project


class GeneratedFile:
    FILENAME: str = "FAKE.md"
    TEMPLATE: str = "FAKE FILE, DO NOT COMMIT!"

    def __init__(self, project: Project):
        self.project = project

    def generate(self) -> str:
        tags = self.project.tags
        template = Template(textwrap.dedent(self.TEMPLATE))
        output = template.render(
            project=self.project,
            tags=tags,
            len=len,
        )
        return output

    def write(self) -> Path:
        content = self.generate()
        fpath = Path(self.FILENAME.format(project=self.project))
        fpath.write_text(content)
        return fpath


class Changelog(GeneratedFile):
    FILENAME = "CHANGELOG.md"
    TEMPLATE = """
        {{- project.name }}
        {{ "=" * len(project.name) }}

        [![Generated by attribution][attribution-badge]][attribution-url]

        {% for tag in tags %}
        {{ tag.name }}
        {{ "-" * len(tag.name) }}

        {{ tag.message if tag.message else "" }}
        {% if tag.shortlog -%}
        ```text
        $ {{ tag.shortlog_cmd }}
        {{ tag.shortlog }}
        ```
        {%- endif %}

        {% endfor -%}

        [attribution-badge]:
            https://img.shields.io/badge/generated%20by-attribution-informational
        [attribution-url]: https://attribution.omnilib.dev

    """


class Contributors(GeneratedFile):
    FILENAME = "CONTRIBUTORS"
    TEMPLATE = """
        Contributors
        ============

        {{ project.shortlog }}
        """


class VersionFile(GeneratedFile):
    FILENAME = "{project.package}/__version__.py"
    TEMPLATE = '''\
        """
        This file is automatically generated by attribution.

        Do not edit manually. Get more info at https://attribution.omnilib.dev
        """

        __version__ = "{{ project.latest.version }}"

        '''
