"""
This file is part of Automunge which is released under the BSD 3-Clause license.
See file LICENSE or go to https://github.com/Automunge/AutoMunge for full license details.

contact available via automunge.com

Copyright (C) 2018, 2019, 2020, 2021, 2022, 2023 Nicholas Teague - All Rights Reserved

patent pending
"""

#global imports
import numpy as np
import pandas as pd

#this mutes the pandas defragmented dataframe warning
#which is not a bug it is a performance indication
#we are muting until pandas can roll out a fix
from warnings import simplefilter
simplefilter(action="ignore", category=pd.errors.PerformanceWarning)

#imports for automunge
import random
import types
#dt used for time stamp, other datetime methods built on pandas
import datetime as dt

#imports for _evalcategory, _getNArows
# from scipy.stats import shapiro
# from scipy.stats import skew

#imports for _predictinfill, _predictpostinfill, _trainFSmodel, _shuffleaccuracy
from sklearn.ensemble import RandomForestRegressor
from sklearn.ensemble import RandomForestClassifier
from sklearn.model_selection import GridSearchCV
from sklearn.model_selection import RandomizedSearchCV
from sklearn.metrics import mean_squared_error
from sklearn.metrics import mean_squared_log_error
# from sklearn.metrics import roc_auc_score
from sklearn.metrics import accuracy_score
from sklearn.metrics import f1_score
#stats may be used for cases where user elects RandomSearchCV hyperparameter tuning
from scipy import stats

#imports for PCA dimensionality reduction applied with application
# from sklearn.decomposition import PCA
# from sklearn.decomposition import SparsePCA
# from sklearn.decomposition import KernelPCA

#imports for DP family of transforms e.g. _process_DP** (DP transforms use np.random)
# import numpy as np

#imports for _process_bxcx 
# from scipy.stats import boxcox

#imports for _process_qttf
# from sklearn.preprocessing import QuantileTransformer

#imports for process_hldy 
# from pandas.tseries.holiday import USFederalHolidayCalendar

#we also have imports for auto ML options in the support functions with their application
#(this allows us to include the option without having their library install as a dependancy)
# import sys
# from flaml import AutoML
# from catboost import CatBoostClassifier
# from catboost import CatBoostRegressor
# import tensorflow as tf
# from xgboost import XGBClassifier
# from xgboost import XGBRegressor
# import optuna

class AutoMunge:
  """
  The AutoMunge class defined functions are broadly organized as:
  #01 - automunge(.) related Function Blocks
  #02 - postmunge(.) related Function Blocks
  #03 - inversion related Function Blocks
  #04 - experimental model training Function Blocks

  Under each of these headers, we've aggregated function definitions by theme
  Where each of these "Function Blocks" can be navigated to by a control F search

  Listed here are all of these Function Blocks with names of included functions

  #________________________________________________
  #01 - automunge(.) related Function Blocks
  #________________________________________________

  #__FunctionBlock: FamilyTrees (transform_dict)
  __assembletransformdict

  #__FunctionBlock: category definitions (process_dict)
  __assembleprocessdict

  #__FunctionBlock: automunge process family functions
  __processfamily
  __circleoflife
  __dictupdate
  __populate_columnkey_dict
  __processcousin
  __processparent
  __grab_params
  __custom_process_wrapper
  __check_for_protected_features
  __messy_data_prep
  __apply_processfamily
  __prepare_labelsencoding_dict

  #__FunctionBlock: suffix overlap and defaultinfill support functions
  __apply_defaultinfill
  __set_indexcolumn
  __set_Binary_column
  __set_PCA_column
  __df_copy_train
  __df_check_suffixoverlap
  __suffix_overlap_final_aggregation_and_printouts

  #__FunctionBlock: automunge dualprocess, singleprocess, and custom_train functions
  _process_NArw
  _process_numerical
  _process_dxdt
  _process_dxd2
  _process_shft
  _process_MADn
  _process_mnmx
  _process_mnm3
  _process_mxab
  _process_retn
  _process_mean
  _process_binary
  _custom_train_onht
  _process_smth
  __GPS_parse
  _custom_train_GPS1
  _process_mlti
  _process_cnsl
  _process_lngt
  _process_bnst
  _process_UPCS
  _process_splt
  _process_spl2
  _process_sp19
  _process_sbst
  _process_sbs3
  _process_hash
  _process_hs10
  _process_srch
  _process_src2
  _process_src3
  _process_src4
  _process_aggt
  _process_strn
  _process_strg
  _process_nmrc
  _process_nmr4
  _custom_train_ordl
  _process_maxb
  _process_ucct
  _custom_train_1010
  _process_bshr
  _process_wkdy
  _process_hldy
  _process_wkds
  _process_mnts
  _process_tmzn
  _process_tmsc
  _process_time
  _process_qttf
  _process_bxcx
  _process_log0
  _process_logn
  _process_sqrt
  _process_addd
  _process_sbtr
  _process_mltp
  _process_divd
  _process_rais
  _process_absl
  _process_pwrs
  _process_pwor
  _process_bins
  _process_bsor
  _process_bnwd
  _process_bnwo
  _process_bnep
  _process_bneo
  _process_tlbn
  _process_bkt1
  _process_bkt2
  _process_bkt3
  _process_bkt4
  _process_DPnb
  _process_DPmm
  _process_DPrt
  _process_DPbn
  _process_DPod
  _process_DPmc
  _process_DPsk
  _process_qbt1
  _process_null
  _process_copy
  _process_excl
  _process_exc2
  _process_exc5
  _process_shfl
  _custom_train_trig

  #__FunctionBlock: evalcategory and NArows
  __evalcategory
  __getNArows
  __assemble_excluded_from_postmunge_getNArows

  #__FunctionBlock: parsing support functions
  __parsedate
  __is_number
  __is_number_comma
  __is_number_EU
  __parsenumeric

  #__FunctionBlock: ML infill master functions
  __MLinfillfunction
  __predictinfill
  __createMLinfillsets
  __insertinfill
  __inspect_ML_cmnd
  __assemble_param_sets
  __optuna_XG1

  #__FunctionBlock: halting criteria support functions
  __consolidate_multicolumn
  __populate_halt_dict
  __calc_stop_result

  #__FunctionBlock: ML infill leakage support
  __check_for_leakage
  __append_full_exclude
  __convert_leakage_dict
  __convert_leakage_sets

  #__FunctionBlock: ML infill stochastic impute support
  __stochastic_impute
  __stochastic_impute_categoric
  __stochastic_impute_numeric

  #__FunctionBlock: ML infill autoML training, inference, and support
  __assemble_autoMLer
  __autoMLer_cleanup
  __populateMLinfilldefaults
  __initRandomForestClassifier
  __initRandomForestRegressor
  _train_randomforest_classifier
  _predict_randomforest_classifier
  _train_randomforest_regressor
  _predict_randomforest_regressor
  _train_autogluon_classifier
  _train_autogluon_regressor
  _predict_autogluon_classifier
  _predict_autogluon_regressor
  _train_flaml_classifier
  _predict_flaml_classifier
  _train_flaml_regressor
  _predict_flaml_regressor
  _train_catboost_classifier
  _predict_catboost_classifier
  _train_catboost_regressor
  _predict_catboost_regressor
  _train_xgboost_classifier
  _train_xgboost_regressor
  __train_xgboost
  _predict_xgboost_classifier
  _predict_xgboost_regressor
  __predict_xgboost

  #__FunctionBlock: customML support functions
  _train_customML_classifier
  _train_customML_regressor
  __train_customML
  _predict_customML_classifier
  _predict_customML_regressor
  __predict_customML
  __call_default_function

  #__FunctionBlock: customML default inference functions
  __customML_tensorflow_defaultpredict_classification
  __customML_tensorflow_defaultpredict_regression
  __customML_xgboost_defaultpredict
  __customML_catboost_defaultpredict
  __customML_flaml_defaultpredict
  __customML_randomforest_defaultpredict

  #__FunctionBlock: data translations support
  __convert_onehot_to_singlecolumn
  __sequential_range_integer
  __sequential_range_integer_invert
  __convert_singlecolumn_to_onehot
  __convert_1010_to_onehot
  __convert_onehot_to_1010
  __onehot_support

  #__FunctionBlock: oversampling support
  __LabelSetGenerator
  __LabelFrequencyLevelizer

  #__FunctionBlock: automunge feature importance
  __trainFSmodel
  __createFSsets
  __createFSsets2
  __shuffleaccuracy
  __assemblemadethecut
  __featureselect

  #__FunctionBlock: assigninfill support functions
  __assemblepostprocess_assigninfill
  __assemble_sorted_columns_by_NaN_dict
  __apply_am_infill
  __apply_pm_infill
  __zeroinfillfunction
  __oneinfillfunction
  __negzeroinfillfunction
  __naninfillfunction
  __adjinfillfunction
  __interpinfillfunction
  __train_medianinfillfunction
  __test_medianinfillfunction
  __train_meaninfillfunction
  __test_meaninfillfunction
  __train_modeinfillfunction
  __test_modeinfillfunction
  __train_lcinfillfunction

  #__FunctionBlock: PCA support functions
  __evalPCA
  __populatePCAdefaults
  __initSparsePCA
  __initKernelPCA
  __initPCA
  __boolexcl
  __createPCAsets
  __PCAfunction

  #__FunctionBlock: validation functions
  __check_am_miscparameters
  __check_pm_miscparameters
  __check_df_type
  __check_np_shape
  __check_ML_infill
  __check_ML_infill_2
  __validate_allvalidnumeric
  __check_assigncat
  __check_assigncat2
  __check_assigncat3
  __create_inverse_assigncat
  __check_assigninfill
  __check_transformdict000
  __check_transformdict00
  __check_transformdict0
  __check_transformdict
  __check_transformdict2
  __check_transform_dict_roots
  __check_haltingproblem
  __check_offspring
  __check_assignnan
  __check_assignnan_injections
  __check_ML_cmnd
  __check_sampling_dict
  __check_entropy_seeds
  __check_random_generator
  __check_assignparam
  __check_columnheaders
  __check_processdict
  __check_processdict3
  __check_processdict3_support
  __check_processdict4

  #__FunctionBlock: entropy_seeds and random_generator support
  __initialize_sampling_report_dict
  __populate_sampling_report_dict
  __prepare_seeds
  __random_parameters_params_append
  __populate_randomseed
  __sample_from_parameter_list
  __sample_from_scipy_stats
  __get_nprandom
  __erase_seeds

  #__FunctionBlock: functionpointer support
  __grab_functionpointer_entries_support
  __grab_functionpointer_entries

  #__FunctionBlock: column assignments string conversion
  __assigncat_str_convert
  __assigninfill_str_convert
  __parameter_str_convert
  __assignparam_str_convert
  __assignnan_str_convert
  __assignnan_list_convert
  __convert_floatnan_to_npnan

  #__FunctionBlock: label smoothing support
  __apply_LabelSmoothing
  __postapply_LabelSmoothing

  #__FunctionBlock: Binary categoric consolidations support
  __BinaryConsolidate
  __postBinaryConsolidate
  __Binary_convert
  __postBinary_convert
  __masterBinaryinvert
  __meta_inverseprocess_Binary
  __inverseprocess_Binary

  #__FunctionBlock: data type management
  __floatprecision_transform
  __convert_to_nan
  __assignnan_convert
  __assignnan_inject

  #__FunctionBlock: data set partitioning and shuffling
  __df_split
  __df_split_specified
  __df_shuffle
  __df_shuffle_series
  __dupl_rows_consolidate

  #__FunctionBlock: populate returned reports
  __populate_columntype_report
  __populate_column_map_report

  #__FunctionBlock: various automunge utility functions
  __list_replace
  __list_sorting
  __autowhere
  __autocopy
  __autoprint
  __logger_append
  __column_convert_support
  __orig_headers_support

  #__FunctionBlock: encryption utility functions
  __encrypt_postprocess_dict
  __decrypt_postprocess_dict

  #__FunctionBlock: automunge ppd_append
  __ppd_append_support

  #__FunctionBlock: automunge(.) definition
  automunge

  #________________________________________________
  #02 - postmunge(.) related Function Blocks
  #________________________________________________

  #__FunctionBlock: postmunge process family functions
  __postprocessfamily
  __postcircleoflife
  __postprocesscousin
  __postprocessparent
  __custom_postprocess_wrapper
  __apply_postprocessfamily

  #__FunctionBlock: postmunge postprocess and custom_test functions
  _postprocess_numerical
  _postprocess_MADn
  _postprocess_mnmx
  _postprocess_mnm3
  _postprocess_mxab
  _postprocess_retn
  _postprocess_mean
  _postprocess_binary
  _custom_test_onht
  _postprocess_smth
  _custom_test_GPS1
  _postprocess_mlti
  _postprocess_cnsl
  _postprocess_splt
  _postprocess_spl2
  _postprocess_sp19
  _postprocess_sbst
  _postprocess_sbs3
  _postprocess_hash
  _postprocess_hs10
  _postprocess_srch
  _postprocess_src2
  _postprocess_src3
  _postprocess_src4
  _postprocess_nmr4
  _custom_test_ordl
  _postprocess_maxb
  _postprocess_ucct
  _custom_test_1010
  _postprocess_tmsc
  _postprocess_time
  _postprocess_qttf
  _postprocess_bxcx
  _postprocess_log0
  _postprocess_logn
  _postprocess_sqrt
  _postprocess_addd
  _postprocess_sbtr
  _postprocess_mltp
  _postprocess_divd
  _postprocess_rais
  _postprocess_absl
  _postprocess_pwrs
  _postprocess_pwor
  _postprocess_bins
  _postprocess_bsor
  _postprocess_bnwd
  _postprocess_bnwo
  _postprocess_bnep
  _postprocess_bneo
  _postprocess_tlbn
  _postprocess_bkt1
  _postprocess_bkt2
  _postprocess_bkt3
  _postprocess_bkt4
  _postprocess_DPnb
  _postprocess_DPmm
  _postprocess_DPrt
  _postprocess_DPbn
  _postprocess_DPod
  _postprocess_DPmc
  _postprocess_DPsk
  _postprocess_exc2
  _postprocess_exc5

  #__FunctionBlock: postmunge ML infill
  __createpostMLinfillsets
  __predictpostinfill
  __postMLinfillfunction

  #__FunctionBlock: postmunge PCA
  __postcreatePCAsets
  __postPCAfunction

  #__FunctionBlock: postmunge feature selection
  __postfeatureselect

  #__FunctionBlock: postmunge driftreport
  __PCA_drift_stats
  __Binary_drift_stats
  __prepare_driftreport

  #__FunctionBlock: postmunge(.) definition
  postmunge

  #________________________________________________
  #03 - inversion related Function Blocks
  #________________________________________________

  #__FunctionBlock: data structure assembly support functions
  __populate_categorytree
  __populate_family
  __get_categorylist
  __populate_inverse_categorytree
  __populate_inverse_family
  __populate_inputcolumn_dict
  __populate_labelsencoding_dict_support
  __populate_labelsencoding_dict_support2
  __populate_labelsencoding_dict_support3

  #__FunctionBlock: inversion preprocessing support
  __LS_invert
  __inversion_header_support

  #__FunctionBlock: custom inversion support
  __custom_inverseprocess_wrapper

  #__FunctionBlock: postmunge inversion inverseprocess and custom_inversion functions
  _inverseprocess_nmbr
  _inverseprocess_year
  _inverseprocess_mean
  _inverseprocess_MADn
  _inverseprocess_MAD3
  _inverseprocess_mnmx
  _inverseprocess_mnm3
  _inverseprocess_mxab
  _inverseprocess_retn
  _inverseprocess_shft
  _inverseprocess_qttf
  _inverseprocess_log0
  _inverseprocess_logn
  _inverseprocess_addd
  _inverseprocess_sbtr
  _inverseprocess_mltp
  _inverseprocess_divd
  _inverseprocess_rais
  _inverseprocess_absl
  _inverseprocess_sqrt
  _inverseprocess_bnst
  _inverseprocess_UPCS
  _inverseprocess_excl
  _inverseprocess_pwr2
  _inverseprocess_pwor
  _inverseprocess_bins
  _inverseprocess_bsor
  _inverseprocess_bnwd
  _inverseprocess_bnwo
  _inverseprocess_bnep
  _inverseprocess_bneo
  _inverseprocess_tlbn
  _inverseprocess_bkt1
  _inverseprocess_bkt2
  _inverseprocess_bkt3
  _inverseprocess_bkt4
  _custom_inversion_onht
  _inverseprocess_smth
  _custom_inversion_GPS1
  _inverseprocess_mlti
  _inverseprocess_cnsl
  _custom_inversion_ordl
  _inverseprocess_strg
  _inverseprocess_bnry
  _custom_inversion_1010
  _inverseprocess_splt
  _inverseprocess_spl2
  _inverseprocess_sp19
  _inverseprocess_sbst
  _inverseprocess_sbs3
  _inverseprocess_srch
  _inverseprocess_src2
  _inverseprocess_src3
  _inverseprocess_src4
  _inverseprocess_nmrc
  _inverseprocess_DPmc
  _inverseprocess_qbt1
  _custom_inversion_trig

  #__FunctionBlock: master inversion functions
  __df_inversion
  __df_inversion_meta
  __inversion_parent

  #________________________________________________
  #04 - experimental model training Function Blocks
  #________________________________________________

  #__FunctionBlock: model training Function Blocks
  __check_model_parameters
  automodel

  #__FunctionBlock: model inference Function Blocks
  autoinference
  """

  #________________________________________________
  #01 - automunge(.) related Function Blocks
  #________________________________________________

  #__FunctionBlock: FamilyTrees (transform_dict)

  def __assembletransformdict(self, binstransform, NArw_marker):
    """
    #populates the transform_dict data structure
    #which is the internal library that is subsequently consolidated 
    #with any user passed transformdict
      
    #transform_dict is for purposes of populating
    #for each transformation category's use as a root category
    #a "family tree" set of associated transformation categories
    #which are for purposes of specifying the type and order of transformation functions
    #to be applied when a transformation category is assigned as a root category
      
    #we'll refer to the category key to a family as the "root category"
    #we'll refer to a transformation category entered into 
    #a family tree primitive as a "tree category"

    #a transformation category may serve as both a root category
    #and a tree category

    #each transformation category will have a set of properties assigned
    #in the corresponding process_dict data structure
    #including associated transformation functions, data properties, and etc.

    #a root category may be assigned to a column with the user passed assigncat
    #or when not specified may be determined under automation via _evalcategory

    #when applying transformations
    #the transformation functions associated with a root category
    #will not be applied unless that same category is populated as a tree category

    #the family tree primitives are for purposes of specifying order of transformations
    #as may include generations and branches of derivations
    #as well as for managing column retentions in the returned data
    #(as in some cases intermediate stages of transformations may or may not have desired retention)

    #the family tree primitives can be distinguished by types of 
    #upstream/downstream, supplement/replace, offsping/no offspring

    #___________
    #'parents' :
    #upstream / first generation / replaces column / with offspring

    #'siblings':
    #upstream / first generation / supplements column / with offspring

    #'auntsuncles' :
    #upstream / first generation / replaces column / no offspring

    #'cousins' :
    #upstream / first generation / supplements column / no offspring

    #'children' :
    #downstream parents / offspring generations / replaces column / with offspring

    #'niecesnephews' :
    #downstream siblings / offspring generations / supplements column / with offspring

    #'coworkers' :
    #downstream auntsuncles / offspring generations / replaces column / no offspring

    #'friends' :
    #downstream cousins / offspring generations / supplements column / no offspring
    #___________

    #each of the family tree primitives associated with a root category
    #may have entries of zero, one, or more transformation categories
      
    #when a root category is assigned to a column
    #the upstream primitives are inspected
      
    #when a tree category is found 
    #as an entry to an upstream primitive associated with the root category
    #the transformation functions associated with the tree category are performed

    #if any tree categories are populated in the upstream replacement primitives
    #their inclusion supersedes supplement primitive entries
    #and so the input column to the transformation is not retained in the returned set
    #with the column replacement either achieved by an inplace transformation
    #or subsequent deletion operation

    #when a tree category is found
    #as an entry to an upstream primitive with offspring
    #after the associated transformation function is performed
    #the downstream primitives of the family tree of the tree category is inspected
    #and those downstream primitives are treated as a subsequent generation's upstream primitives
    #where the input column to that subsequent generation is the column returned 
    #from the transformation function associated with the upstream tree category

    #this is an easy point of confusion so as further clarification on this point
    #the downstream primitives associated with a root category
    #will not be inspected when root category is applied
    #unless that root category is also entered as a tree category entry
    #in one of the root category's upstream primitives with offspring
    """

    transform_dict = {}

    #initialize bins based on what was passed through application of automunge(.)
    if binstransform is True:
      bint = 'bint'
    else:
      bint = None
        
    if NArw_marker is True:
      NArw = 'NArw'
    else:
      NArw = None

    #initialize trasnform_dict. Note in a future extension the range of categories
    #is intended to be built out
    transform_dict.update({'nmbr' : {'parents'       : ['nmbr'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : [bint]}})
    
    transform_dict.update({'dxdt' : {'parents'       : ['dxdt'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})
    
    transform_dict.update({'d2dt' : {'parents'       : ['d2dt'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['dxdt'],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})
    
    transform_dict.update({'d3dt' : {'parents'       : ['d3dt'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['d2dt'],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})

    transform_dict.update({'d4dt' : {'parents'       : ['d4dt'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['d3dt'],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})

    transform_dict.update({'d5dt' : {'parents'       : ['d5dt'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['d4dt'],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})

    transform_dict.update({'d6dt' : {'parents'       : ['d6dt'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['d5dt'],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})
    
    transform_dict.update({'dxd2' : {'parents'       : ['dxd2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})
    
    transform_dict.update({'d2d2' : {'parents'       : ['d2d2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['dxd2'],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})
    
    transform_dict.update({'d3d2' : {'parents'       : ['d3d2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['d2d2'],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})

    transform_dict.update({'d4d2' : {'parents'       : ['d4d2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['d3d2'],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})

    transform_dict.update({'d5d2' : {'parents'       : ['d5d2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['d4d2'],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})

    transform_dict.update({'d6d2' : {'parents'       : ['d6d2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['d5d2'],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})
    
    transform_dict.update({'nmdx' : {'parents'       : ['nmdx'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['dxdt'],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})
    
    transform_dict.update({'nmd2' : {'parents'       : ['nmd2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['d2dt'],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})
    
    transform_dict.update({'nmd3' : {'parents'       : ['nmd3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['d3dt'],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})

    transform_dict.update({'nmd4' : {'parents'       : ['nmd4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['d4dt'],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})

    transform_dict.update({'nmd5' : {'parents'       : ['nmd5'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['d5dt'],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})

    transform_dict.update({'nmd6' : {'parents'       : ['nmd6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['d6dt'],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})
    
    transform_dict.update({'mmdx' : {'parents'       : ['mmdx'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['nbr2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['nbr2'],
                                     'friends'       : []}})
    
    transform_dict.update({'mmd2' : {'parents'       : ['mmd2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['nbr2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['mmdx'],
                                     'coworkers'     : ['nbr2'],
                                     'friends'       : []}})
    
    transform_dict.update({'mmd3' : {'parents'       : ['mmd3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['nbr2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['mmd2'],
                                     'coworkers'     : ['nbr2'],
                                     'friends'       : []}})

    transform_dict.update({'mmd4' : {'parents'       : ['mmd4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['nbr2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['mmd3'],
                                     'coworkers'     : ['nbr2'],
                                     'friends'       : []}})

    transform_dict.update({'mmd5' : {'parents'       : ['mmd5'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['nbr2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['mmd4'],
                                     'coworkers'     : ['nbr2'],
                                     'friends'       : []}})

    transform_dict.update({'mmd6' : {'parents'       : ['mmd6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['nbr2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['mmd5'],
                                     'coworkers'     : ['nbr2'],
                                     'friends'       : []}})
    
    transform_dict.update({'dddt' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['dddt', 'exc2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'ddd2' : {'parents'       : ['ddd2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['exc2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['dddt'],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'ddd3' : {'parents'       : ['ddd3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['exc2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['ddd2'],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'ddd4' : {'parents'       : ['ddd4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['exc2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['ddd3'],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'ddd5' : {'parents'       : ['ddd5'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['exc2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['ddd4'],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'ddd6' : {'parents'       : ['ddd6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['exc2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['ddd5'],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'dedt' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['dedt', 'exc2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'ded2' : {'parents'       : ['ded2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['exc2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['dedt'],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'ded3' : {'parents'       : ['ded3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['exc2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['ded2'],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'ded4' : {'parents'       : ['ded4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['exc2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['ded3'],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'ded5' : {'parents'       : ['ded5'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['exc2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['ded4'],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'ded6' : {'parents'       : ['ded6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['exc2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['ded5'],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'shft' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['shft'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'shf2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['shf2'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'shf3' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['shf3'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'shf4' : {'parents'       : ['shf4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})
  
    transform_dict.update({'shf5' : {'parents'       : ['shf5'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})
    
    transform_dict.update({'shf6' : {'parents'       : ['shf6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})

    transform_dict.update({'shf7' : {'parents'       : ['shf4', 'shf5'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})
    
    transform_dict.update({'shf8' : {'parents'       : ['shf4', 'shf5', 'shf6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['retn'],
                                     'friends'       : []}})

    transform_dict.update({'bnry' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bnry'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'bnr2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bnr2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'onht' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['onht'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'text' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['text'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'txt2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['text'],
                                     'cousins'       : [NArw, 'splt'],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'txt3' : {'parents'       : ['txt3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['text'],
                                     'friends'       : []}})

    #mlti primarily intended for use as a downstream tree category
    transform_dict.update({'mlti' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['nmbr'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    #mlto primarily intended for use as a downstream tree category
    transform_dict.update({'mlto' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['ord3'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'mlt1' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mlt1'],
                                     'cousins'       : [NArw],
                                     'children'      : ['cnsl'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'mlt2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mlt2'],
                                     'cousins'       : [NArw],
                                     'children'      : ['cns2'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'mlt3' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mlt3'],
                                     'cousins'       : [NArw],
                                     'children'      : ['cns3'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'cnsl' : {'parents'       : ['mlt1'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})

    transform_dict.update({'cns2' : {'parents'       : ['mlt2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['onht'],
                                     'friends'       : []}})

    transform_dict.update({'cns3' : {'parents'       : ['mlt3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['ord3'],
                                     'friends'       : []}})

    transform_dict.update({'smth' : {'parents'       : ['smt0'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['smth'],
                                     'friends'       : []}})
  
    transform_dict.update({'smt0' : {'parents'       : ['smt0'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['smth'],
                                     'friends'       : []}})
  
    transform_dict.update({'fsmh' : {'parents'       : ['fsm0'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['fsmh'],
                                     'friends'       : []}})
    
    transform_dict.update({'fsm0' : {'parents'       : ['fsm0'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['fsmh'],
                                     'friends'       : []}})

    transform_dict.update({'GPS1' : {'parents'       : ['GPS1'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['mlti'],
                                     'friends'       : []}})

    transform_dict.update({'GPS2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['GPS2'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'GPS3' : {'parents'       : ['GPS3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['mlti'],
                                     'friends'       : []}})

    transform_dict.update({'GPS4' : {'parents'       : ['GPS4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['mlti'],
                                     'friends'       : []}})

    transform_dict.update({'GPS5' : {'parents'       : ['GPS5'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['mlto'],
                                     'friends'       : []}})

    transform_dict.update({'GPS6' : {'parents'       : ['GPS6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['mlto', 'mlti'],
                                     'friends'       : []}})

    transform_dict.update({'lngt' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['lngt'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'lngm' : {'parents'       : ['lngm'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['mnmx'],
                                     'friends'       : []}})
  
    transform_dict.update({'lnlg' : {'parents'       : ['lnlg'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['logn'],
                                     'friends'       : []}})

    transform_dict.update({'bnst' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bnst'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['ord3'],
                                     'friends'       : []}})
  
    transform_dict.update({'bnso' : {'parents'       : ['bnso'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['ord3'],
                                     'friends'       : []}})

    transform_dict.update({'UPCS' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['UPCS'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'Unht' : {'parents'       : ['Unht'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['onht'],
                                     'friends'       : []}})
  
    transform_dict.update({'Utxt' : {'parents'       : ['Utxt'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['text'],
                                     'friends'       : []}})
    
    transform_dict.update({'Utx2' : {'parents'       : ['Utx2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['text'],
                                     'friends'       : ['splt']}})

    transform_dict.update({'Utx3' : {'parents'       : ['Utx3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['txt3'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'Ucct' : {'parents'       : ['Ucct'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['ucct', 'ord3'],
                                     'friends'       : []}})
    
    transform_dict.update({'Uord' : {'parents'       : ['Uord'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['ordl'],
                                     'friends'       : []}})
        
    transform_dict.update({'Uor2' : {'parents'       : ['Uor2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['ord2'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'Uor3' : {'parents'       : ['Uor3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['ord3'],
                                     'friends'       : []}})
    
    transform_dict.update({'Uor6' : {'parents'       : ['Uor6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['spl6'],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['ord3'],
                                     'friends'       : []}})
    
    transform_dict.update({'U101' : {'parents'       : ['U101'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})
    
    transform_dict.update({'splt' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['splt'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'spl2' : {'parents'       : ['spl2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['ord3'],
                                     'friends'       : []}})
    
    transform_dict.update({'spl5' : {'parents'       : ['spl5'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['ord3'],
                                     'friends'       : []}})
    
    transform_dict.update({'spl6' : {'parents'       : ['spl6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['splt'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : ['ord3']}})
    
    transform_dict.update({'spl7' : {'parents'       : ['spl7'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['ord3'],
                                     'friends'       : []}})

    transform_dict.update({'spl8' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['spl8'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'spl9' : {'parents'       : ['spl9'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['ord3'],
                                     'friends'       : []}})

    transform_dict.update({'sp10' : {'parents'       : ['sp10'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['ord3'],
                                     'friends'       : []}})
    
    transform_dict.update({'sp11' : {'parents'       : ['sp11'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['spl5'],
                                     'coworkers'     : ['ord3'],
                                     'friends'       : []}})
    
    transform_dict.update({'sp12' : {'parents'       : ['sp12'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['sp11'],
                                     'coworkers'     : ['ord3'],
                                     'friends'       : []}})
    
    transform_dict.update({'sp13' : {'parents'       : ['sp13'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['sp10'],
                                     'coworkers'     : ['ord3'],
                                     'friends'       : []}})
    
    transform_dict.update({'sp14' : {'parents'       : ['sp14'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['sp13'],
                                     'coworkers'     : ['ord3'],
                                     'friends'       : []}})
    
    transform_dict.update({'sp15' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['sp15'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'sp16' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['sp16'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'sp17' : {'parents'       : ['sp17'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['spl5'],
                                     'coworkers'     : ['ord3'],
                                     'friends'       : []}})
    
    transform_dict.update({'sp18' : {'parents'       : ['sp18'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : ['sp17'],
                                     'coworkers'     : ['ord3'],
                                     'friends'       : []}})

    transform_dict.update({'sp19' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['sp19'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'sp20' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['sp20'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'sbst' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['sbst'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'sbs2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['sbs2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'sbs3' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['sbs3'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'sbs4' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['sbs4'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'hash' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['hash'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'hsh2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['hsh2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'hs10' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['hs10'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'Uhsh' : {'parents'       : ['Uhsh'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['hash'],
                                     'friends'       : []}})

    transform_dict.update({'Uhs2' : {'parents'       : ['Uhs2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['hsh2'],
                                     'friends'       : []}})
    
    transform_dict.update({'Uh10' : {'parents'       : ['Uh10'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['hs10'],
                                     'friends'       : []}})
    
    transform_dict.update({'srch' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['srch'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'src2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['src2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'src3' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['src3'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'src4' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['src4'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'aggt' : {'parents'       : ['aggt'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['ord3'],
                                     'friends'       : []}})
    
    transform_dict.update({'strn' : {'parents'       : ['strn'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['ord3'],
                                     'friends'       : []}})

    transform_dict.update({'strg' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['strg'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'nmrc' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['nmrc'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'nmr2' : {'parents'       : ['nmr2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['nmbr'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'nmr3' : {'parents'       : ['nmr3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['mnmx'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'nmr4' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['nmr4'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'nmr5' : {'parents'       : ['nmr5'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['nmbr'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'nmr6' : {'parents'       : ['nmr6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['mnmx'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'nmr7' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['nmr7'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'nmr8' : {'parents'       : ['nmr8'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['nmbr'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'nmr9' : {'parents'       : ['nmr9'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['mnmx'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'nmcm' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['nmcm'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'nmc2' : {'parents'       : ['nmc2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['nmbr'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'nmc3' : {'parents'       : ['nmc3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['mnmx'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'nmc4' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['nmc4'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'nmc5' : {'parents'       : ['nmc5'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['nmbr'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'nmc6' : {'parents'       : ['nmc6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['mnmx'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'nmc7' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['nmc7'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'nmc8' : {'parents'       : ['nmc8'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['nmbr'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'nmc9' : {'parents'       : ['nmc9'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['mnmx'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'nmEU' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['nmEU'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'nmE2' : {'parents'       : ['nmE2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['nmbr'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'nmE3' : {'parents'       : ['nmE3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['mnmx'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'nmE4' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['nmE4'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'nmE5' : {'parents'       : ['nmE5'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['nmbr'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'nmE6' : {'parents'       : ['nmE6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['mnmx'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'nmE7' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['nmE7'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'nmE8' : {'parents'       : ['nmE8'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['nmbr'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'nmE9' : {'parents'       : ['nmE9'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['mnmx'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'ors7' : {'parents'       : ['spl6', 'nmr2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['ord3'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'ors5' : {'parents'       : ['spl5'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['ord3'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'ors6' : {'parents'       : ['spl6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['ord3'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'ordl' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['ordl'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
        
    transform_dict.update({'ord2' : {'parents'       : ['ord2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['mnmx'],
                                     'friends'       : []}})
    
    transform_dict.update({'ord3' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['ord3'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'ord5' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['ord5'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'maxb' : {'parents'       : ['or3b'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'or3b' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['or3b'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['maxb'],
                                     'friends'       : []}})
  
    transform_dict.update({'matx' : {'parents'       : ['or3c'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['onht'],
                                     'friends'       : []}})
    
    transform_dict.update({'or3c' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['or3c'],
                                     'cousins'       : [NArw],
                                     'children'      : ['matx'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'ma10' : {'parents'       : ['or3d'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})
    
    transform_dict.update({'or3d' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['or3d'],
                                     'cousins'       : [NArw],
                                     'children'      : ['ma10'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'ucct' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['ucct'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
        
    transform_dict.update({'ord4' : {'parents'       : ['ord4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['mnmx'],
                                     'friends'       : []}})
    
    transform_dict.update({'ors2' : {'parents'       : ['spl2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['ord3'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'or10' : {'parents'       : ['ord4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['1010'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['mnmx'],
                                     'friends'       : []}})
    
    transform_dict.update({'or11' : {'parents'       : ['sp11'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['1010'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'or12' : {'parents'       : ['nmr2'],
                                     'siblings'      : ['sp11'],
                                     'auntsuncles'   : ['1010'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'or13' : {'parents'       : ['sp12'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['1010'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'or14' : {'parents'       : ['nmr2'],
                                     'siblings'      : ['sp12'],
                                     'auntsuncles'   : ['1010'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'or15' : {'parents'       : ['or15'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['sp13'],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})
  
    transform_dict.update({'or16' : {'parents'       : ['or16'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['nmr2'],
                                     'niecesnephews' : ['sp13'],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})
    
    transform_dict.update({'or17' : {'parents'       : ['or17'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['sp14'],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})
    
    transform_dict.update({'or18' : {'parents'       : ['or18'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['nmr2'],
                                     'niecesnephews' : ['sp14'],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})

    transform_dict.update({'or19' : {'parents'       : ['or19'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['nmc8'],
                                     'niecesnephews' : ['sp13'],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})
    
    transform_dict.update({'or20' : {'parents'       : ['or20'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['nmc8'],
                                     'niecesnephews' : ['sp14'],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})
    
    transform_dict.update({'or21' : {'parents'       : ['or21'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['nmc8'],
                                     'niecesnephews' : ['sp17'],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})
    
    transform_dict.update({'or22' : {'parents'       : ['or22'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['nmc8'],
                                     'niecesnephews' : ['sp18'],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})

    transform_dict.update({'or23' : {'parents'       : ['or23'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['nmcm', 'sp19', 'ord3'],
                                     'friends'       : []}})
    
    transform_dict.update({'om10' : {'parents'       : ['ord4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['1010', 'mnmx'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['mnmx'],
                                     'friends'       : []}})

    transform_dict.update({'mmor' : {'parents'       : ['ord4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mnmx'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'1010' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['1010'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'10mz' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['10mz'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'null' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['null'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'NArw' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['NArw'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'NAr2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['NAr2'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'NAr3' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['NAr3'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'NAr4' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['NAr4'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'NAr5' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['NAr5'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'nbr2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['nbr2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'nbr3' : {'parents'       : ['nbr3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : ['bint']}})

    transform_dict.update({'nbr4' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['nbr4'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'MADn' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['MADn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'MAD2' : {'parents'       : ['MAD2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['nmbr'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'MAD3' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['MAD3'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mnmx' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mnmx'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mnm2' : {'parents'       : ['nmbr'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mnmx'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mnm3' : {'parents'       : ['nmbr'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mnm3'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mnm4' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mnm4'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mnm5' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mnmx'],
                                     'cousins'       : ['nmbr', NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mnm6' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mnm6'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mnm7' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mnmx', 'bins'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'mxab' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mxab'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'retn' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'rtbn' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn', 'bsor'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'rtb2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn', 'bins'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'mean' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mean'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mea2' : {'parents'       : ['nmbr'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mean'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mea3' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mean', 'bins'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'tmsc' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['tmsc'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'time' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['year'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'tmzn' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['tmzn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'date' : {'parents'       : ['date'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['year', 'mnth', 'days', 'hour', 'mint', 'scnd'],
                                     'friends'       : []}})
  
    transform_dict.update({'dat2' : {'parents'       : ['dat2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['bshr', 'wkdy', 'hldy'],
                                     'friends'       : []}})
    
    transform_dict.update({'dat3' : {'parents'       : ['dat3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['year', 'mnsn', 'mncs', 'dysn', 'dycs', 'hrsn', 'hrcs', 'misn', 'mics', 'scsn', 'sccs'],
                                     'friends'       : []}})
    
    transform_dict.update({'dat4' : {'parents'       : ['dat4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['year', 'mdsn', 'mdcs', 'hmss', 'hmsc'],
                                     'friends'       : []}})
    
    transform_dict.update({'dat5' : {'parents'       : ['dat5'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['year', 'mdsn', 'mdcs', 'dysn', 'dycs', 'hmss', 'hmsc'],
                                     'friends'       : []}})
    
    transform_dict.update({'dat6' : {'parents'       : ['dat6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['year', 'mdsn', 'mdcs', 'hmss', 'hmsc', 'bshr', 'wkdy', 'hldy'],
                                     'friends'       : []}})
    
    transform_dict.update({'year' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['year'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'yea2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['year', 'yrsn', 'yrcs', 'mdsn', 'mdcs'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'yrcs' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['yrcs'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'yrsn' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['yrsn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mnth' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mnth'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'mnt2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mnsn', 'mncs'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mnt3' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mnsn', 'mncs', 'dysn', 'dycs', 'hrsn', 'hrcs', 'misn', 'mics', 'scsn', 'sccs'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mnt4' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mdsn', 'mdcs'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mnt5' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mdsn', 'mdcs', 'hmss', 'hmsc'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mnt6' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mdsn', 'mdcs', 'dysn', 'dycs', 'hmss', 'hmsc'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mnsn' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mnsn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mncs' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mncs'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mdsn' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mdsn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mdcs' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mdcs'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'days' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['days'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'day2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['dysn', 'dycs'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'day3' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['dysn', 'dycs', 'hrsn', 'hrcs', 'misn', 'mics', 'scsn', 'sccs'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'day4' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['dhms', 'dhmc'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'day5' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['dhms', 'dhmc', 'hmss', 'hmsc'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'dysn' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['dysn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'dycs' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['dycs'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'dhms' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['dhms'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'dhmc' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['dhmc'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'hour' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['hour'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'hrs2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['hrsn', 'hrcs'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'hrs3' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['hrsn', 'hrcs', 'misn', 'mics', 'scsn', 'sccs'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'hrs4' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['hmss', 'hmsc'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'hrsn' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['hrsn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'hrcs' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['hrcs'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'hmss' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['hmss'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'hmsc' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['hmsc'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mint' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mint'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'min2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['misn', 'mics'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'min3' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['misn', 'mics', 'scsn', 'sccs'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'min4' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mssn', 'mscs'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'misn' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['misn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mics' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mics'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mssn' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mssn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mscs' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mscs'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'scnd' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['scnd'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'scn2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['scsn', 'sccs'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'scsn' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['scsn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'sccs' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['sccs'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'qttf' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['qttf'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'qtt2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['qtt2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'DPqt' : {'parents'       : ['qtt1'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'qtt1' : {'parents'       : ['qtt1'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DPqt'],
                                     'friends'       : []}})
    
    transform_dict.update({'bxcx' : {'parents'       : ['bxcx'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['nmbr'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'bxc2' : {'parents'       : ['bxc2'],
                                     'siblings'      : ['nmbr'],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['nmbr'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'bxc3' : {'parents'       : ['bxc3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['nmbr'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'bxc4' : {'parents'       : ['bxc4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['nbr2'],
                                     'friends'       : []}})

    transform_dict.update({'bxc5' : {'parents'       : ['bxc5'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mnmx'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['nbr2', 'bins'],
                                     'friends'       : []}})

    transform_dict.update({'DPbx' : {'parents'       : ['bxc6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'bxc6' : {'parents'       : ['bxc6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['bxc7'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'bxc7' : {'parents'       : ['bxc7'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DPbx'],
                                     'friends'       : []}})

    transform_dict.update({'ntgr' : {'parents'       : ['ntgr'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn', '1010', 'ordl'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['mnmx'],
                                     'friends'       : []}})
    
    transform_dict.update({'ntg2' : {'parents'       : ['ntg2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn', '1010', 'ordl', 'pwr2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['mnmx'],
                                     'friends'       : []}})
    
    transform_dict.update({'ntg3' : {'parents'       : ['ntg3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['retn', 'ordl', 'por2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['mnmx'],
                                     'friends'       : []}})
    
    transform_dict.update({'pwrs' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['pwrs'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'pwr2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['pwr2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'log0' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['log0'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'log1' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['log0', 'pwr2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'logn' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['logn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'lgnm' : {'parents'       : ['lgnm'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['nmbr'],
                                     'friends'       : []}})
    
    transform_dict.update({'sqrt' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['sqrt'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'addd' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['addd'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'sbtr' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['sbtr'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'mltp' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['mltp'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'divd' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['divd'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'rais' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['rais'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'absl' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['absl'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'bkt1' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bkt1'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'bkt2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bkt2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'bkt3' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bkt3'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'bkt4' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bkt4'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'wkdy' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['wkdy'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'bshr' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bshr'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'hldy' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['hldy'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'wkds' : {'parents'       : ['wkds'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['text'],
                                     'friends'       : []}})
  
    transform_dict.update({'wkdo' : {'parents'       : ['wkdo'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['ordl'],
                                     'friends'       : []}})
    
    transform_dict.update({'mnts' : {'parents'       : ['mnts'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['text'],
                                     'friends'       : []}})
  
    transform_dict.update({'mnto' : {'parents'       : ['mnto'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['ordl'],
                                     'friends'       : []}})
    
    transform_dict.update({'bins' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bins'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'bint' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bint'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'bsor' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bsor'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'btor' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['btor'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'bnwd' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bnwd'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'bnwK' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bnwK'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'bnwM' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bnwM'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'bnwo' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bnwo'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'bnKo' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bnKo'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'bnMo' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bnMo'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})    
    
    transform_dict.update({'bnep' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bnep'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'bne7' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bne7'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'bne9' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bne9'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'bneo' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bneo'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'bn7o' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bn7o'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'bn9o' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bn9o'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'tlbn' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['tlbn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'pwor' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['pwor'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'por2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['por2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'por3' : {'parents'       : ['por3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})

    transform_dict.update({'bkb3' : {'parents'       : ['bkb3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})
  
    transform_dict.update({'bkb4' : {'parents'       : ['bkb4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})
    
    transform_dict.update({'bsbn' : {'parents'       : ['bsbn'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})
    
    transform_dict.update({'bnwb' : {'parents'       : ['bnwb'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})
    
    transform_dict.update({'bnKb' : {'parents'       : ['bnKb'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})

    transform_dict.update({'bnMb' : {'parents'       : ['bnMb'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})
    
    transform_dict.update({'bneb' : {'parents'       : ['bneb'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})

    transform_dict.update({'bn7b' : {'parents'       : ['bn7b'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})
    
    transform_dict.update({'bn9b' : {'parents'       : ['bn9b'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})
    
    transform_dict.update({'pwbn' : {'parents'       : ['pwbn'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['1010'],
                                     'friends'       : []}})

    transform_dict.update({'DTnb' : {'parents'       : ['DTn3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DTn3 primarily intended for use as a tree category
    transform_dict.update({'DTn3' : {'parents'       : ['DTn3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DTnb'],
                                     'friends'       : []}})

    transform_dict.update({'DTmm' : {'parents'       : ['DTm2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DTm2 primarily intended for use as a tree category
    transform_dict.update({'DTm2' : {'parents'       : ['DTm2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DTmm'],
                                     'friends'       : []}})

    transform_dict.update({'DTrt' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['DTrt'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'DTbn' : {'parents'       : ['DTb2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DTb2 primarily intended for use as a tree category
    transform_dict.update({'DTb2' : {'parents'       : ['DTb2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DTbn'],
                                     'friends'       : []}})
    
    transform_dict.update({'DTod' : {'parents'       : ['DTo4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DTo4 primarily intended for use as a tree category
    transform_dict.update({'DTo4' : {'parents'       : ['DTo4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DTod'],
                                     'friends'       : []}})

    transform_dict.update({'DToh' : {'parents'       : ['DTo5'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DTo5 primarily intended for use as a tree category
    transform_dict.update({'DTo5' : {'parents'       : ['DTo5'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DToh'],
                                     'friends'       : []}})

    transform_dict.update({'DT10' : {'parents'       : ['DTo6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DTo6 primarily intended for use as a tree category
    transform_dict.update({'DTo6' : {'parents'       : ['DTo6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DT10'],
                                     'friends'       : []}})

    transform_dict.update({'DTh1' : {'parents'       : ['DTh1'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DTmc'],
                                     'friends'       : []}})

    #DTmc primarily intended for use as a downstream tree category
    transform_dict.update({'DTmc' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['ordl'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'DTmp' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['DTmp'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DTo7 primarily intended for use as a downstream tree category
    transform_dict.update({'DTo7' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['ordl'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'DThs' : {'parents'       : ['DThs'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['mThs'],
                                     'friends'       : []}})
  
    transform_dict.update({'DTh2' : {'parents'       : ['DTh2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DTo7'],
                                     'friends'       : []}})

    transform_dict.update({'DTns' : {'parents'       : ['DTn4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    #DTn4 primarily intended for use as a tree category
    transform_dict.update({'DTn4' : {'parents'       : ['DTn4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DTns'],
                                     'friends'       : []}})

    transform_dict.update({'DT1s' : {'parents'       : ['DTo8'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DTo8 primarily intended for use as a tree category
    transform_dict.update({'DTo8' : {'parents'       : ['DTo8'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DT1s'],
                                     'friends'       : []}})

    transform_dict.update({'DTsk' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['DTsk'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'DTne' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['DTne'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'DTse' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['DTse'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #mThs primarily intended for use as a downstream tree category
    transform_dict.update({'mThs' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['ordl'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'DBnb' : {'parents'       : ['DBn3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DTn3 primarily intended for use as a tree category
    transform_dict.update({'DBn3' : {'parents'       : ['DBn3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DBnb'],
                                     'friends'       : []}})

    transform_dict.update({'DBmm' : {'parents'       : ['DBm2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DTm2 primarily intended for use as a tree category
    transform_dict.update({'DBm2' : {'parents'       : ['DBm2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DBmm'],
                                     'friends'       : []}})

    transform_dict.update({'DBrt' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['DBrt'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'DBbn' : {'parents'       : ['DBb2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DTb2 primarily intended for use as a tree category
    transform_dict.update({'DBb2' : {'parents'       : ['DBb2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DBbn'],
                                     'friends'       : []}})
    
    transform_dict.update({'DBod' : {'parents'       : ['DBo4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DTo4 primarily intended for use as a tree category
    transform_dict.update({'DBo4' : {'parents'       : ['DBo4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DBod'],
                                     'friends'       : []}})

    transform_dict.update({'DBoh' : {'parents'       : ['DBo5'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DTo5 primarily intended for use as a tree category
    transform_dict.update({'DBo5' : {'parents'       : ['DBo5'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DBoh'],
                                     'friends'       : []}})

    transform_dict.update({'DB10' : {'parents'       : ['DBo6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DTo6 primarily intended for use as a tree category
    transform_dict.update({'DBo6' : {'parents'       : ['DBo6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DB10'],
                                     'friends'       : []}})

    transform_dict.update({'DBh1' : {'parents'       : ['DBh1'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DBmc'],
                                     'friends'       : []}})

    #DTmc primarily intended for use as a downstream tree category
    transform_dict.update({'DBmc' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['ordl'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'DBmp' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['DBmp'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DTo7 primarily intended for use as a downstream tree category
    transform_dict.update({'DBo7' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['ordl'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'DBhs' : {'parents'       : ['DBhs'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['mBhs'],
                                     'friends'       : []}})
  
    transform_dict.update({'DBh2' : {'parents'       : ['DBh2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DBo7'],
                                     'friends'       : []}})

    transform_dict.update({'DBns' : {'parents'       : ['DBn4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    #DTn4 primarily intended for use as a tree category
    transform_dict.update({'DBn4' : {'parents'       : ['DBn4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DBns'],
                                     'friends'       : []}})

    transform_dict.update({'DB1s' : {'parents'       : ['DBo8'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DTo8 primarily intended for use as a tree category
    transform_dict.update({'DBo8' : {'parents'       : ['DBo8'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DB1s'],
                                     'friends'       : []}})

    transform_dict.update({'DBsk' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['DBsk'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'DBne' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['DBne'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'DBse' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['DBse'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #mThs primarily intended for use as a downstream tree category
    transform_dict.update({'mBhs' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['ordl'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'DPnb' : {'parents'       : ['DPn3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DPn3 primarily intended for use as a tree category
    transform_dict.update({'DPn3' : {'parents'       : ['DPn3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DPnb'],
                                     'friends'       : []}})

    transform_dict.update({'DPmm' : {'parents'       : ['DPm2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DPm2 primarily intended for use as a tree category
    transform_dict.update({'DPm2' : {'parents'       : ['DPm2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DPmm'],
                                     'friends'       : []}})

    transform_dict.update({'DPrt' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['DPrt'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'DLnb' : {'parents'       : ['DLn3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DLn3 primarily intended for use as a tree category
    transform_dict.update({'DLn3' : {'parents'       : ['DLn3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DLnb'],
                                     'friends'       : []}})

    transform_dict.update({'DLmm' : {'parents'       : ['DLm2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    #DLm2 primarily intended for use as a tree category
    transform_dict.update({'DLm2' : {'parents'       : ['DLm2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DLmm'],
                                     'friends'       : []}})

    transform_dict.update({'DLrt' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['DLrt'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'DPbn' : {'parents'       : ['DPb2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DPb2 primarily intended for use as a tree category
    transform_dict.update({'DPb2' : {'parents'       : ['DPb2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DPbn'],
                                     'friends'       : []}})
    
    transform_dict.update({'DPod' : {'parents'       : ['DPo4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DPo4 primarily intended for use as a tree category
    transform_dict.update({'DPo4' : {'parents'       : ['DPo4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DPod'],
                                     'friends'       : []}})

    transform_dict.update({'DPoh' : {'parents'       : ['DPo5'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DPo5 primarily intended for use as a tree category
    transform_dict.update({'DPo5' : {'parents'       : ['DPo5'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DPoh'],
                                     'friends'       : []}})

    transform_dict.update({'DP10' : {'parents'       : ['DPo6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DPo6 primarily intended for use as a tree category
    transform_dict.update({'DPo6' : {'parents'       : ['DPo6'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DP10'],
                                     'friends'       : []}})

    transform_dict.update({'DPh1' : {'parents'       : ['DPh1'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DPmc'],
                                     'friends'       : []}})

    #DPmc primarily intended for use as a downstream tree category
    transform_dict.update({'DPmc' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['ordl'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'DPmp' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['DPmp'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DPo7 primarily intended for use as a downstream tree category
    transform_dict.update({'DPo7' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['ordl'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'DPhs' : {'parents'       : ['DPhs'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['mlhs'],
                                     'friends'       : []}})
  
    transform_dict.update({'DPh2' : {'parents'       : ['DPh2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DPo7'],
                                     'friends'       : []}})

    transform_dict.update({'DPns' : {'parents'       : ['DPn4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    #DPn4 primarily intended for use as a tree category
    transform_dict.update({'DPn4' : {'parents'       : ['DPn4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DPns'],
                                     'friends'       : []}})

    transform_dict.update({'DP1s' : {'parents'       : ['DPo8'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #DPo8 primarily intended for use as a tree category
    transform_dict.update({'DPo8' : {'parents'       : ['DPo8'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['DP1s'],
                                     'friends'       : []}})

    transform_dict.update({'DPsk' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['DPsk'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'DPne' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['DPne'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'DPse' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['DPse'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'DPpc' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['DPpc'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    #mlhs primarily intended for use as a downstream tree category
    transform_dict.update({'mlhs' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['ordl'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'qbt1' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['qbt1'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'qbt2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['qbt2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'qbt3' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['qbt3'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'qbt4' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['qbt4'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'nmqb' : {'parents'       : ['nmqb'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['qbt1'],
                                     'friends'       : []}})
  
    transform_dict.update({'nmq2' : {'parents'       : ['nmq2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : ['qbt1']}})
  
    transform_dict.update({'mmqb' : {'parents'       : ['mmqb'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['qbt3'],
                                     'friends'       : []}})
    
    transform_dict.update({'mmq2' : {'parents'       : ['mmq2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : ['qbt3']}})
    
    transform_dict.update({'copy' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['copy'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'excl' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['excl'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #for exc2 without NArw use exc6
    transform_dict.update({'exc2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['exc2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    #exc3 is like exc2 with downstream standard deviation bins
    transform_dict.update({'exc3' : {'parents'       : ['exc3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : ['bins']}})
    
    #exc4 is like exc2 with downstream power of ten bins
    transform_dict.update({'exc4' : {'parents'       : ['exc4'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : ['pwr2']}})
    
    #exc5 is passthrough integer categoric, for use without NArw use exc7
    transform_dict.update({'exc5' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['exc5'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'exc6' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['exc2'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'exc7' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['exc7'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    #exc8 is passthrough integer continuous, for use without NArw use exc9
    transform_dict.update({'exc8' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['exc8'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'exc9' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['exc9'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'shfl' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['shfl'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'sint' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['sint'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'cost' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['cost'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'tant' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['tant'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'arsn' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['arsn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'arcs' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['arcs'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'artn' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['artn'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'nmbd' : {'parents'       : ['nmbr'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : [bint]}})

    transform_dict.update({'101d' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['101d'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'ordd' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['ordd'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'texd' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['texd'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'bnrd' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['bnrd'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'datd' : {'parents'       : ['datd'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['year', 'mdsn', 'mdcs', 'hmss', 'hmsc', 'bshr', 'wkdy', 'hldy'],
                                     'friends'       : []}})
    
    transform_dict.update({'nuld' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['nuld'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'lbnm' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['lbnm'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'lbnb' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['lbnb'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'lb10' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['lb10'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'lbor' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['lbor'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'lbos' : {'parents'       : ['lbos'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['strg'],
                                     'friends'       : []}})
    
    transform_dict.update({'lbte' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['lbte'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'lbbn' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['lbbn'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'lbsm' : {'parents'       : ['smt0'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['lbsm'],
                                     'friends'       : []}})
  
    transform_dict.update({'lbfs' : {'parents'       : ['fsm0'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['lbfs'],
                                     'friends'       : []}})
    
    transform_dict.update({'lbda' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['year', 'mdsn', 'mdcs', 'hmss', 'hmsc', 'bshr', 'wkdy', 'hldy'],
                                     'cousins'       : [],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'lgnr' : {'parents'       : ['lgnr', 'sgn3'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : ['lgn2'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
  
    transform_dict.update({'lgn2' : {'parents'       : ['lgn2'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['qbt5'],
                                     'friends'       : []}})
    
    transform_dict.update({'sgn1' : {'parents'       : ['sgn1'],
                                     'siblings'      : [],
                                     'auntsuncles'   : [],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : ['sgn2'],
                                     'friends'       : []}})
    
    transform_dict.update({'qbt5' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['qbt5'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})
    
    transform_dict.update({'sgn2' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['sgn2'],
                                     'cousins'       : [NArw],
                                     'children'      : [],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'sgn3' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['sgn3'],
                                     'cousins'       : [NArw],
                                     'children'      : ['sgn4'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    transform_dict.update({'sgn4' : {'parents'       : [],
                                     'siblings'      : [],
                                     'auntsuncles'   : ['sgn4'],
                                     'cousins'       : [NArw],
                                     'children'      : ['sgn1'],
                                     'niecesnephews' : [],
                                     'coworkers'     : [],
                                     'friends'       : []}})

    return transform_dict

  #__FunctionBlock: category definitions (process_dict)
  
  def __assembleprocessdict(self):
    '''
    #creates a dictionary storing all of the processing functions for each
    #category. Note that the convention is that every dualprocess entry 
    #(to process both train and test data in automunge) is meant
    #to have a corresponding postprocess entry (to process the test set in 
    #postmunge). If the dualprocess/postprocess pair aren't included a 
    #singleprocess function should be instead which processes a single column
    #at a time and is neutral to whether that set is from train or test data.
      
    #note that the functionpointer entry is currently only available for user passed processdict
    #this internal library process_dict does not accept functionpointer entries
      
    #A user should pass either a pair of processing functions to both 
    #dualprocess and postprocess, or alternatively just a single processing
    #function to singleprocess, and omit or pass None to those not used.
    #A user can also pass an inversion function to inverseprocess if available.
    #Most of the transforms defined internal to the library follow this convention.

    #dualprocess: for passing a processing function in which normalization 
    #             parameters are derived from properties of the training set
    #             and jointly process the train set and if available corresponding test set

    #singleprocess: for passing a processing function in which no normalization
    #               parameters are needed from the train set to process the
    #               test set, such that train and test sets processed separately

    #postprocess: for passing a processing function in which normalization 
    #             parameters originally derived from the train set are applied
    #             to separately process a corresponding test set
    #             An entry should correspond to the dualprocess entry.

    #inverseprocess: for passing a processing function used to invert
    #                a corresponding forward pass transform
    #                An entry should correspond to the dualprocess or singleprocess entry.

    #__________________________________________________________________________
    #Alternative streamlined processing function conventions are also available 
    #which may be populated as entries to custom_train / custom_test / custom_inversion.
    #These conventions are documented in the readme section "Custom Transformation Functions".
    #In cases of redundancy custom_train entry specifications take precedence 
    #over dualprocess/singleprocess/postprocess entries.

    #custom_train: for passing a train set processing function in which normalization parameters
    #              are derived from properties of the training set. Will be used to process both 
    #              train and test data when custom_test not provided (in which case similar to singleprocess convention).

    #custom_test: for passing a test set processing function in which normalization parameters
    #             that were derived from properties of the training set are used to process the test data.
    #             When omitted custom_train will be used to process both the train and test data.
    #             An entry should correspond to the custom_train entry.

    #custom_inversion: for passing a processing function used to invert
    #                  a corresponding forward pass transform
    #                  An entry should correspond to the custom_train entry.

    #___________________________________________________________________________
    #The processdict also specifies various properties associated with the transformations. 
    #At a minimum, a user needs to specify NArowtype and MLinfilltype or otherwise
    #include a functionpointer entry.

    #___________________________________________________________________________
    #NArowtype: classifies the type of entries that are targets for infill.
    #           can be entries of {'numeric', 'integer', 'justNaN', 'exclude', 
    #                              'positivenumeric', 'nonnegativenumeric', 
    #                              'nonzeronumeric', 'parsenumeric', 'datetime'}
    #           Note that in the custom_train convention this is used to apply data type casting prior to the transform.
    # - 'numeric' for source columns with expected numeric entries
    # - 'integer' for source columns with expected integer entries
    # - 'justNaN' for source columns that may have expected entries other than numeric
    # - 'binary' similar to justNaN but only the top two most frequent entries are considered valid
    # - 'exclude' for source columns that aren't needing NArow columns derived
    # - 'totalexclude' for source columns that aren't needing NArow columns derived, 
    #                  also excluded from assignnan global option and nan conversions for missing data
    # - 'positivenumeric' for source columns with expected positive numeric entries
    # - 'nonnegativenumeric' for source columns with expected non-negative numeric (zero allowed)
    # - 'nonzeronumeric' for source columns with allowed positive and negative but no zero
    # - 'parsenumeric' marks for infill strings that don't contain any numeric characters
    # - 'datetime' marks for infill cells that aren't recognized as datetime objects

    # ** Note that NArowtype also is used as basis for metrics evaluated in drift assessment of source columns
    # ** Note that by default any np.inf values are converted to NaN for infill
    # ** Note that by default python None entries are treated as targets for infill

    #___________________________________________________________________________
    #MLinfilltype: classifies data types of the returned set, 
    #              as may determine what types of models are trained for ML infill
    #              can be entries {'numeric', 'singlct', 'binary', 'multirt', 'concurrent_act', 'concurrent_nmbr', 
    #                              '1010', 'exclude', 'boolexclude', 'ordlexclude', 'totalexclude'}
    #              'numeric' single columns with numeric entries for regression (signed floats)
    #              'singlct' for single column sets with ordinal entries (nonnegative integer classification)
    #              'integer' for single column sets with integer entries (signed integer regression)
    #              'binary'  single column sets with boolean entries (0/1)
    #              'multirt' categoric multicolumn sets with boolean entries (0/1), up to one activation per row
    #              '1010'    for multicolumn sets with binary encoding via 1010, boolean integer entries (0/1), 
    #                        with distinct encoding representations by the set of activations
    #              'concurrent_act' for multicolumn sets with boolean integer entries as may have 
    #                               multiple entries in the same row, different from 1010 
    #                               in that columns are independent
    #              'concurrent_ordl' for multicolumn sets with ordinal encoded entries (nonnegative integer classification)
    #              'concurrent_nmbr' for multicolumn sets with numeric entries (signed floats)
    #              'exclude' for columns which will be excluded from infill, included in other features' ML infill bases
    #                        returned data should be numerically encoded
    #              'boolexclude' boolean integer set suitable for Binary transform but excluded from all infill 
    #                            (e.g. NArw entries), included in other features' ML infill bases
    #              'ordlexclude' ordinal set excluded from infill (note that in some cases in library 
    #                            ordlexclude may return a multi-column set), included in other features' ML infill bases
    #              'totalexclude' for complete passthroughs (excl) without datatype conversions, infill, 
    #                             excluded from other features' ML infill bases

    #___________________________________________________________________________
    #Other optional entries for processdict include:
    #info_retention, inplace_option, defaultparams, labelctgy, 
    #defaultinfill, dtype_convert, functionpointer, and noise_transform.

    #___________________________________________________________________________
    #info_retention: boolean marker associated with an inversion operation that helps inversion prioritize
    #transformation paths with full information recovery. (May pass as True when there is no information loss.)

    #___________________________________________________________________________
    #inplace_option: boolean marker indicating whether a transform supports the inplace parameter received in params.
    #                When not specified this is assumed as True (which is always valid for the custom_train convention).
    #                In other words, in dualprocess/singleprocess convention, if your transform does not support inplace,
    #                need to specify inplace_option as False

    #___________________________________________________________________________
    #defaultparams: a dictionary recording any default assignparam assignments associated with the category. 
    #               Note that deviations in user specifications to assignparam as part of an automunge(.) call
    #               take precedence over defaultparams. Note that when applying functionpointer defaultparams
    #               from the pointer target are also populated when not previously specified.

    #___________________________________________________________________________
    #defaultinfill: this option serves to specify a default infill
    #               applied after NArowtype data type casting and preceding the transformation function.
    #               (defaultinfill is a precursor to ML infill or other infills applied based on assigninfill)
    #               defaults to 'adjinfill' when not specified, can also pass as one of
    #               {'adjinfill', 'meaninfill', 'medianinfill', 'modeinfill', 'lcinfill', 
    #                'interpinfill', 'zeroinfill', 'oneinfill', 'naninfill', 'negzeroinfill'}
    #               Note that 'meaninfill' and 'medianinfill' only work with numeric data (based on NArowtype).
    #               Note that for 'datetime' NArowtype, defaultinfill only supports 'adjinfill' or 'naninfill'
    #               Note that 'naninfill' is intended for cases where user wishes to apply their own default infill 
    #               as part of a custom_train entry

    #___________________________________________________________________________
    #dtype_convert: this option is intended for the custom_train convention, accepts boolean entries,
    #               defaults to True when not specified, False turns off a data type conversion
    #               that is applied after custom_train transformation functions based on MLinfilltype.
    #               May also be used to deactivate a floatprecision conversion for any category. 
    #               This option primarily included to support special cases and not intended for wide use.

    #___________________________________________________________________________
    #labelctgy: an optional entry, should be a string entry of a single transformation category 
    #           as entered in the family tree when the category of the processdict entry is used as a root category. 
    #           Used to determine a basis of feature selection for cases where root 
    #           category is applied to a label set resulting in a set returned in multiple configurations. 
    #           Also used in label frequency levelizer. 
    #           Note that since this is only used for small edge case populating a labelctgy entry is optional. 
    #           If one is not assigned, an arbitrary entry will be accessed from the family tree. 
    #           This option primarily included to support special cases.

    #___________________________________________________________________________
    #functionpointer: A functionpointer entry 
    #                 may be entered in lieu of any or all of these other entries **.
    #                 The functionpointer should be populated with a category that has its own processdict entry 
    #                 (or a category that has its own process_dict entry internal to the library)
    #                 The functionpointer inspects the pointer target and passes those specifications 
    #                 to the origin processdict entry unless previously specified.
    #                 The functionpointer is intended as a shortcut for specifying processdict entries
    #                 that may be helpful in cases where a new entry is very similar to some existing entry.
    #                 (**As the exception labelctgy not accessed from functionpointer 
    #                 since it is specific to a root category's family tree.)

    #___________________________________________________________________________
    #noise_transform: this option serves to specify the noise injection types for noise transforms
    #                 used to support an entropy seeding based on sampling_dict['sampling_type'] specification
    #                 defaults to False when not specified, can also pass as one of
    #                 {'numeric', 'categoric', 'binary', False}
    #                 numeric is for transforms similar to DPnb/DPmm/DPrt which have a binomial and distribution sampling
    #                 categoric is for transforms similar to DPod/DPmc which have a binomial and a choice sampling
    #                 binary is for transforms similar to an alternate DPbn configuration which only have a binomial sampling
    #                 False is for transforms without sampling_dict['sampling_type'] specification support

    #___________________________________________________________________________
    #Other clarifications:
    #Note that NArowtype is associated with transformation inputs
    #including for a category's use as a root category and as a tree category
    #MLinfilltype is associated with transformation outputs
    #for a category's use as a tree category
    '''

    process_dict = {}
    
    process_dict.update({'nmbr' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'dxdt' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxdt,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'d2dt' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxdt,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'d3dt' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxdt,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'d4dt' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxdt,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'d5dt' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxdt,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'d6dt' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxdt,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'dxd2' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxd2,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'d2d2' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxd2,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'d3d2' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxd2,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'d4d2' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxd2,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'d5d2' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxd2,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'d6d2' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxd2,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'nmdx' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'nmd2' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'nmd3' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'nmd4' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'nmd5' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'nmd6' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'mmdx' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxdt,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'mmd2' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxdt,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'mmd3' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxdt,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'mmd4' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxdt,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'mmd5' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxdt,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'mmd6' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxdt,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'dddt' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxdt,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'dddt'}})
    process_dict.update({'ddd2' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxdt,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'dddt'}})
    process_dict.update({'ddd3' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxdt,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'dddt'}})
    process_dict.update({'ddd4' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxdt,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'dddt'}})
    process_dict.update({'ddd5' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxdt,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'dddt'}})
    process_dict.update({'ddd6' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxdt,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'dddt'}})
    process_dict.update({'dedt' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxd2,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'dedt'}})
    process_dict.update({'ded2' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxd2,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'dedt'}})
    process_dict.update({'ded3' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxd2,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'dedt'}})
    process_dict.update({'ded4' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxd2,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'dedt'}})
    process_dict.update({'ded5' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxd2,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'dedt'}})
    process_dict.update({'ded6' : {'dualprocess' : None,
                                  'singleprocess' : self._process_dxd2,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'dedt'}})
    process_dict.update({'shft' : {'dualprocess' : None,
                                  'singleprocess' : self._process_shft,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_shft,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'periods' : 1},
                                  'NArowtype' : 'exclude',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'shft'}})
    process_dict.update({'shf2' : {'dualprocess' : None,
                                  'singleprocess' : self._process_shft,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_shft,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'periods' : 2},
                                  'NArowtype' : 'exclude',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'shf2'}})
    process_dict.update({'shf3' : {'dualprocess' : None,
                                  'singleprocess' : self._process_shft,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_shft,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'periods' : 3},
                                  'NArowtype' : 'exclude',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'shf3'}})
    process_dict.update({'shf4' : {'dualprocess' : None,
                                  'singleprocess' : self._process_shft,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_shft,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'periods' : 1},
                                  'NArowtype' : 'exclude',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'shf5' : {'dualprocess' : None,
                                  'singleprocess' : self._process_shft,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_shft,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'periods' : 2},
                                  'NArowtype' : 'exclude',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'shf6' : {'dualprocess' : None,
                                  'singleprocess' : self._process_shft,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_shft,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'periods' : 3},
                                  'NArowtype' : 'exclude',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'shf7' : {'dualprocess' : None,
                                  'singleprocess' : self._process_shft,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_shft,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'NArowtype' : 'exclude',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'shf8' : {'dualprocess' : None,
                                  'singleprocess' : self._process_shft,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_shft,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'NArowtype' : 'exclude',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'nbr2' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nbr2'}})
    process_dict.update({'nbr3' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nbr3'}})
    process_dict.update({'nbr4' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'zeroinfill',
                                  'defaultparams' : {'abs_zero' : False},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'MADn' : {'dualprocess' : self._process_MADn,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_MADn,
                                  'inverseprocess' : self._inverseprocess_MADn,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'defaultparams' : {'center' : 'mean'},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'MADn'}})
    process_dict.update({'MAD2' : {'dualprocess' : self._process_MADn,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_MADn,
                                  'inverseprocess' : self._inverseprocess_MADn,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'defaultparams' : {'center' : 'mean'},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'MAD2'}})
    process_dict.update({'MAD3' : {'dualprocess' : self._process_MADn,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_MADn,
                                  'inverseprocess' : self._inverseprocess_MADn,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'defaultparams' : {'center' : 'max'},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'MAD3'}})
    process_dict.update({'mnmx' : {'dualprocess' : self._process_mnmx,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mnmx,
                                  'inverseprocess' : self._inverseprocess_mnmx,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'mnm2' : {'dualprocess' : self._process_mnmx,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mnmx,
                                  'inverseprocess' : self._inverseprocess_mnmx,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'mnm3' : {'dualprocess' : self._process_mnm3,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mnm3,
                                  'inverseprocess' : self._inverseprocess_mnm3,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnm3'}})
    process_dict.update({'mnm4' : {'dualprocess' : self._process_mnm3,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mnm3,
                                  'inverseprocess' : self._inverseprocess_mnm3,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnm4'}})
    process_dict.update({'mnm5' : {'dualprocess' : self._process_mnmx,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mnmx,
                                  'inverseprocess' : self._inverseprocess_mnmx,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'mnm6' : {'dualprocess' : self._process_mnmx,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mnmx,
                                  'inverseprocess' : self._inverseprocess_mnmx,
                                  'defaultparams' : {'floor' : True},
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnm6'}})
    process_dict.update({'mnm7' : {'dualprocess' : self._process_mnmx,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mnmx,
                                  'inverseprocess' : self._inverseprocess_mnmx,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'mxab' : {'dualprocess' : self._process_mxab,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mxab,
                                  'inverseprocess' : self._inverseprocess_mxab,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mxab'}})
    process_dict.update({'retn' : {'dualprocess' : self._process_retn,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_retn,
                                  'inverseprocess' : self._inverseprocess_retn,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'rtbn' : {'dualprocess' : self._process_retn,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_retn,
                                  'inverseprocess' : self._inverseprocess_retn,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'rtb2' : {'dualprocess' : self._process_retn,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_retn,
                                  'inverseprocess' : self._inverseprocess_retn,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'retn'}})
    process_dict.update({'mean' : {'dualprocess' : self._process_mean,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mean,
                                  'inverseprocess' : self._inverseprocess_mean,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mean'}})
    process_dict.update({'mea2' : {'dualprocess' : self._process_mean,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mean,
                                  'inverseprocess' : self._inverseprocess_mean,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mean'}})
    process_dict.update({'mea3' : {'dualprocess' : self._process_mean,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mean,
                                  'inverseprocess' : self._inverseprocess_mean,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mean'}})
    process_dict.update({'bnry' : {'dualprocess' : self._process_binary,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_binary,
                                  'inverseprocess' : self._inverseprocess_bnry,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'infillconvention' : 'onevalue'},
                                  'NArowtype' : 'binary',
                                  'MLinfilltype' : 'binary',
                                  'labelctgy' : 'bnry'}})
    process_dict.update({'bnr2' : {'dualprocess' : self._process_binary,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_binary,
                                  'inverseprocess' : self._inverseprocess_bnry,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'infillconvention' : 'zerovalue'},
                                  'NArowtype' : 'binary',
                                  'MLinfilltype' : 'binary',
                                  'labelctgy' : 'bnr2'}})
    process_dict.update({'onht' : {'custom_train' : self._custom_train_onht,
                                  'custom_test' : self._custom_test_onht,
                                  'custom_inversion' : self._custom_inversion_onht,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'suffix_convention' : 'onht'},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'onht'}})
    process_dict.update({'text' : {'custom_train' : self._custom_train_onht,
                                  'custom_test' : self._custom_test_onht,
                                  'custom_inversion' : self._custom_inversion_onht,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'text'}})
    process_dict.update({'txt2' : {'custom_train' : self._custom_train_onht,
                                  'custom_test' : self._custom_test_onht,
                                  'custom_inversion' : self._custom_inversion_onht,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'text'}})
    process_dict.update({'txt3' : {'dualprocess' : self._process_spl2,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_spl2,
                                  'inverseprocess' : self._inverseprocess_spl2,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'text'}})
    process_dict.update({'mlti' : {'dualprocess' : self._process_mlti,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mlti,
                                  'inverseprocess' : self._inverseprocess_mlti,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'concurrent_nmbr',
                                  'labelctgy' : 'mlti'}})
    process_dict.update({'mlto' : {'dualprocess' : self._process_mlti,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mlti,
                                  'inverseprocess' : self._inverseprocess_mlti,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultparams' : {'dtype' : 'conditionalinteger',
                                                     'norm_category' : 'ord3',
                                                     'norm_params' : {'str_convert' : False}},
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'concurrent_ordl',
                                  'labelctgy' : 'mlto'}})
    process_dict.update({'mlt1' : {'dualprocess' : self._process_mlti,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mlti,
                                  'inverseprocess' : self._inverseprocess_mlti,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultparams' : {'dtype' : 'conditionalinteger',
                                                     'norm_category' : 'ord3',
                                                     'norm_params' : {'str_convert' : True}},
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'concurrent_ordl',
                                  'labelctgy' : 'mlt1'}})
    process_dict.update({'mlt2' : {'dualprocess' : self._process_mlti,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mlti,
                                  'inverseprocess' : self._inverseprocess_mlti,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultparams' : {'dtype' : 'conditionalinteger',
                                                     'norm_category' : 'ord3',
                                                     'norm_params' : {'str_convert' : True}},
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'concurrent_ordl',
                                  'labelctgy' : 'mlt2'}})
    process_dict.update({'mlt3' : {'dualprocess' : self._process_mlti,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mlti,
                                  'inverseprocess' : self._inverseprocess_mlti,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultparams' : {'dtype' : 'conditionalinteger',
                                                     'norm_category' : 'ord3',
                                                     'norm_params' : {'str_convert' : True}},
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'concurrent_ordl',
                                  'labelctgy' : 'mlt3'}})
    process_dict.update({'cnsl' : {'dualprocess' : self._process_cnsl,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_cnsl,
                                  'inverseprocess' : self._inverseprocess_cnsl,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultparams' : {'messy_data_headers' : [],
                                                     'pad_character' : '_',
                                                     'upstream_ordinal' : True},
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'exclude',
                                  'labelctgy' : '1010'}})
    process_dict.update({'cns2' : {'dualprocess' : self._process_cnsl,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_cnsl,
                                  'inverseprocess' : self._inverseprocess_cnsl,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultparams' : {'messy_data_headers' : [],
                                                     'pad_character' : '_',
                                                     'upstream_ordinal' : True},
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'exclude',
                                  'labelctgy' : 'onht'}})
    process_dict.update({'cns3' : {'dualprocess' : self._process_cnsl,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_cnsl,
                                  'inverseprocess' : self._inverseprocess_cnsl,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultparams' : {'messy_data_headers' : [],
                                                     'pad_character' : '_',
                                                     'upstream_ordinal' : True},
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'exclude',
                                  'labelctgy' : 'ord3'}})
    process_dict.update({'smt0' : {'custom_train' : self._custom_train_onht,
                                  'custom_test' : self._custom_test_onht,
                                  'custom_inversion' : self._custom_inversion_onht,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'smth'}})
    process_dict.update({'smth' : {'dualprocess' : self._process_smth,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_smth,
                                  'inverseprocess' : self._inverseprocess_smth,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'exclude',
                                  'labelctgy' : 'smth'}})
    process_dict.update({'fsm0' : {'custom_train' : self._custom_train_onht,
                                  'custom_test' : self._custom_test_onht,
                                  'custom_inversion' : self._custom_inversion_onht,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'fsmh'}})
    process_dict.update({'fsmh' : {'dualprocess' : self._process_smth,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_smth,
                                  'inverseprocess' : self._inverseprocess_smth,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'LSfit' : True},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'exclude',
                                  'labelctgy' : 'fsmh'}})
    process_dict.update({'GPS1' : {'custom_train' : self._custom_train_GPS1,
                                  'custom_test' : None,
                                  'custom_inversion' : self._custom_inversion_GPS1,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'GPS_convention' : 'default'},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'concurrent_nmbr',
                                  'labelctgy' : 'mlti'}})
    process_dict.update({'GPS2' : {'custom_train' : self._custom_train_GPS1,
                                  'custom_test' : None,
                                  'custom_inversion' : self._custom_inversion_GPS1,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'GPS_convention' : 'default'},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'concurrent_nmbr',
                                  'labelctgy' : 'mlti'}})
    process_dict.update({'GPS3' : {'custom_train' : self._custom_train_GPS1,
                                  'custom_test' : None,
                                  'custom_inversion' : self._custom_inversion_GPS1,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'GPS_convention' : 'nonunique'},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'concurrent_nmbr',
                                  'labelctgy' : 'mlti'}})
    process_dict.update({'GPS4' : {'custom_train' : self._custom_train_GPS1,
                                  'custom_test' : self._custom_test_GPS1,
                                  'custom_inversion' : self._custom_inversion_GPS1,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'GPS_convention' : 'nonunique'},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'concurrent_nmbr',
                                  'labelctgy' : 'mlti'}})
    process_dict.update({'GPS5' : {'custom_train' : self._custom_train_GPS1,
                                  'custom_test' : None,
                                  'custom_inversion' : self._custom_inversion_GPS1,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'GPS_convention' : 'nonunique'},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'concurrent_nmbr',
                                  'labelctgy' : 'mlto'}})
    process_dict.update({'GPS6' : {'custom_train' : self._custom_train_GPS1,
                                  'custom_test' : None,
                                  'custom_inversion' : self._custom_inversion_GPS1,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'GPS_convention' : 'nonunique'},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'concurrent_nmbr',
                                  'labelctgy' : 'mlto'}})
    process_dict.update({'lngt' : {'dualprocess' : None,
                                  'singleprocess' : self._process_lngt,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'integer',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'lngm' : {'dualprocess' : None,
                                  'singleprocess' : self._process_lngt,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'integer',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'lnlg' : {'dualprocess' : None,
                                  'singleprocess' : self._process_lngt,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'integer',
                                  'labelctgy' : 'log0'}})
    process_dict.update({'bnst' : {'dualprocess' : None,
                                  'singleprocess' : self._process_bnst,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_bnst,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'bnst'}})
    process_dict.update({'bnso' : {'dualprocess' : None,
                                  'singleprocess' : self._process_bnst,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_bnst,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'ord3'}})
    process_dict.update({'UPCS' : {'dualprocess' : None,
                                  'singleprocess' : self._process_UPCS,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'UPCS'}})
    process_dict.update({'Unht' : {'dualprocess' : None,
                                  'singleprocess' : self._process_UPCS,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'onht'}})
    process_dict.update({'Utxt' : {'dualprocess' : None,
                                  'singleprocess' : self._process_UPCS,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'text'}})
    process_dict.update({'Utx2' : {'dualprocess' : None,
                                  'singleprocess' : self._process_UPCS,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'text'}})
    process_dict.update({'Utx3' : {'dualprocess' : None,
                                  'singleprocess' : self._process_UPCS,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'text'}})
    process_dict.update({'Ucct' : {'dualprocess' : None,
                                  'singleprocess' : self._process_UPCS,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'ord3'}})
    process_dict.update({'Uord' : {'dualprocess' : None,
                                  'singleprocess' : self._process_UPCS,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'ordl'}})
    process_dict.update({'Uor2' : {'dualprocess' : None,
                                  'singleprocess' : self._process_UPCS,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'Uor3' : {'dualprocess' : None,
                                  'singleprocess' : self._process_UPCS,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'ord3'}})
    process_dict.update({'Uor6' : {'dualprocess' : None,
                                  'singleprocess' : self._process_UPCS,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'ord3'}})
    process_dict.update({'U101' : {'dualprocess' : None,
                                  'singleprocess' : self._process_UPCS,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : '1010'}})
    process_dict.update({'splt' : {'dualprocess' : self._process_splt,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_splt,
                                  'inverseprocess' : self._inverseprocess_splt,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'splt'}})
    process_dict.update({'spl2' : {'dualprocess' : self._process_spl2,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_spl2,
                                  'inverseprocess' : self._inverseprocess_spl2,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'test_same_as_train' : False,
                                                     'consolidate_nonoverlaps' : False},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'ord3'}})
    process_dict.update({'spl5' : {'dualprocess' : self._process_spl2,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_spl2,
                                  'inverseprocess' : self._inverseprocess_spl2,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'test_same_as_train' : False,
                                                     'consolidate_nonoverlaps' : True},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'ord3'}})
    process_dict.update({'spl6' : {'dualprocess' : self._process_spl2,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_spl2,
                                  'inverseprocess' : self._inverseprocess_spl2,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'ord3'}})
    process_dict.update({'spl7' : {'dualprocess' : self._process_spl2,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_spl2,
                                  'inverseprocess' : self._inverseprocess_spl2,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'test_same_as_train' : False,
                                                     'consolidate_nonoverlaps' : True,
                                                     'minsplit' : 1},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'ord3'}})
    process_dict.update({'spl8' : {'dualprocess' : self._process_splt,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_splt,
                                  'inverseprocess' : self._inverseprocess_splt,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'test_same_as_train' : True},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'spl8'}})
    process_dict.update({'spl9' : {'dualprocess' : self._process_spl2,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_spl2,
                                  'inverseprocess' : self._inverseprocess_spl2,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'test_same_as_train' : True,
                                                     'consolidate_nonoverlaps' : False},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'ord3'}})
    process_dict.update({'sp10' : {'dualprocess' : self._process_spl2,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_spl2,
                                  'inverseprocess' : self._inverseprocess_spl2,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'test_same_as_train' : True,
                                                     'consolidate_nonoverlaps' : True},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'ord3'}})
    process_dict.update({'sp11' : {'dualprocess' : self._process_spl2,
                                   'singleprocess' : None,
                                   'postprocess' : self._postprocess_spl2,
                                   'inverseprocess' : self._inverseprocess_spl2,
                                   'info_retention' : False,
                                   'inplace_option' : False,
                                   'defaultinfill' : 'naninfill',
                                   'defaultparams' : {'test_same_as_train' : False,
                                                      'consolidate_nonoverlaps' : False},
                                   'NArowtype' : 'justNaN',
                                   'MLinfilltype' : 'totalexclude',
                                   'labelctgy' : 'ord3'}})
    process_dict.update({'sp12' : {'dualprocess' : self._process_spl2,
                                   'singleprocess' : None,
                                   'postprocess' : self._postprocess_spl2,
                                   'inverseprocess' : self._inverseprocess_spl2,
                                   'info_retention' : False,
                                   'inplace_option' : False,
                                   'defaultinfill' : 'naninfill',
                                   'defaultparams' : {'test_same_as_train' : False,
                                                      'consolidate_nonoverlaps' : False},
                                   'NArowtype' : 'justNaN',
                                   'MLinfilltype' : 'totalexclude',
                                   'labelctgy' : 'ord3'}})
    process_dict.update({'sp13' : {'dualprocess' : self._process_spl2,
                                   'singleprocess' : None,
                                   'postprocess' : self._postprocess_spl2,
                                   'inverseprocess' : self._inverseprocess_spl2,
                                   'info_retention' : False,
                                   'inplace_option' : False,
                                   'defaultinfill' : 'naninfill',
                                   'defaultparams' : {'test_same_as_train' : True,
                                                      'consolidate_nonoverlaps' : False},
                                   'NArowtype' : 'justNaN',
                                   'MLinfilltype' : 'totalexclude',
                                   'labelctgy' : 'ord3'}})
    process_dict.update({'sp14' : {'dualprocess' : self._process_spl2,
                                   'singleprocess' : None,
                                   'postprocess' : self._postprocess_spl2,
                                   'inverseprocess' : self._inverseprocess_spl2,
                                   'info_retention' : False,
                                   'inplace_option' : False,
                                   'defaultinfill' : 'naninfill',
                                   'defaultparams' : {'test_same_as_train' : True,
                                                      'consolidate_nonoverlaps' : False},
                                   'NArowtype' : 'justNaN',
                                   'MLinfilltype' : 'totalexclude',
                                   'labelctgy' : 'ord3'}})
    process_dict.update({'sp15' : {'dualprocess' : self._process_splt,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_splt,
                                  'inverseprocess' : self._inverseprocess_splt,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'concurrent_activations': True,
                                                     'test_same_as_train' : False},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'concurrent_act',
                                  'labelctgy' : 'sp15'}})
    process_dict.update({'sp16' : {'dualprocess' : self._process_splt,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_splt,
                                  'inverseprocess' : self._inverseprocess_splt,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'concurrent_activations': True,
                                                     'test_same_as_train' : True},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'concurrent_act',
                                  'labelctgy' : 'sp16'}})
    process_dict.update({'sp17' : {'dualprocess' : self._process_spl2,
                                   'singleprocess' : None,
                                   'postprocess' : self._postprocess_spl2,
                                   'inverseprocess' : self._inverseprocess_spl2,
                                   'info_retention' : False,
                                   'inplace_option' : False,
                                   'defaultinfill' : 'naninfill',
                                   'defaultparams' : {'test_same_as_train' : False,
                                                      'consolidate_nonoverlaps' : False},
                                   'NArowtype' : 'justNaN',
                                   'MLinfilltype' : 'totalexclude',
                                   'labelctgy' : 'ord3'}})
    process_dict.update({'sp18' : {'dualprocess' : self._process_spl2,
                                   'singleprocess' : None,
                                   'postprocess' : self._postprocess_spl2,
                                   'inverseprocess' : self._inverseprocess_spl2,
                                   'info_retention' : False,
                                   'inplace_option' : False,
                                   'defaultinfill' : 'naninfill',
                                   'defaultparams' : {'test_same_as_train' : False,
                                                      'consolidate_nonoverlaps' : False},
                                   'NArowtype' : 'justNaN',
                                   'MLinfilltype' : 'totalexclude',
                                   'labelctgy' : 'ord3'}})
    process_dict.update({'sp19' : {'dualprocess' : self._process_sp19,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_sp19,
                                  'inverseprocess' : self._inverseprocess_sp19,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : '1010',
                                  'labelctgy' : 'sp19'}})
    process_dict.update({'sp20' : {'dualprocess' : self._process_sp19,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_sp19,
                                  'inverseprocess' : self._inverseprocess_sp19,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'test_same_as_train' : True},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : '1010',
                                  'labelctgy' : 'sp20'}})
    process_dict.update({'sbst' : {'dualprocess' : self._process_sbst,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_sbst,
                                  'inverseprocess' : self._inverseprocess_sbst,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'test_same_as_train' : False},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'concurrent_act',
                                  'labelctgy' : 'sbst'}})
    process_dict.update({'sbs2' : {'dualprocess' : self._process_sbst,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_sbst,
                                  'inverseprocess' : self._inverseprocess_sbst,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'test_same_as_train' : True},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'concurrent_act',
                                  'labelctgy' : 'sbs2'}})
    process_dict.update({'sbs3' : {'dualprocess' : self._process_sbs3,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_sbs3,
                                  'inverseprocess' : self._inverseprocess_sbs3,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'test_same_as_train' : False},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : '1010',
                                  'labelctgy' : 'sbs3'}})
    process_dict.update({'sbs4' : {'dualprocess' : self._process_sbs3,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_sbs3,
                                  'inverseprocess' : self._inverseprocess_sbs3,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'test_same_as_train' : True},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : '1010',
                                  'labelctgy' : 'sbs4'}})
    process_dict.update({'hash' : {'dualprocess' : self._process_hash,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_hash,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'ordlexclude',
                                  'labelctgy' : 'hash'}})
    process_dict.update({'hsh2' : {'dualprocess' : self._process_hash,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_hash,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'space' : '',
                                                     'excluded_characters' : []},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'ordlexclude',
                                  'labelctgy' : 'hsh2'}})
    process_dict.update({'hs10' : {'dualprocess' : self._process_hs10,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_hs10,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'boolexclude',
                                  'labelctgy' : 'hs10'}})
    process_dict.update({'Uhsh' : {'dualprocess' : None,
                                  'singleprocess' : self._process_UPCS,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'hash'}})
    process_dict.update({'Uhs2' : {'dualprocess' : None,
                                  'singleprocess' : self._process_UPCS,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'hsh2'}})
    process_dict.update({'Uh10' : {'dualprocess' : None,
                                  'singleprocess' : self._process_UPCS,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'hs10'}})
    process_dict.update({'srch' : {'dualprocess' : self._process_srch,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_srch,
                                  'inverseprocess' : self._inverseprocess_srch,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'concurrent_act',
                                  'labelctgy' : 'srch'}})
    process_dict.update({'src2' : {'dualprocess' : self._process_src2,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_src2,
                                  'inverseprocess' : self._inverseprocess_src2,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'concurrent_act',
                                  'labelctgy' : 'src2'}})
    process_dict.update({'src3' : {'dualprocess' : self._process_src3,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_src3,
                                  'inverseprocess' : self._inverseprocess_src3,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'concurrent_act',
                                  'labelctgy' : 'src3'}})
    process_dict.update({'src4' : {'dualprocess' : self._process_src4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_src4,
                                  'inverseprocess' : self._inverseprocess_src4,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'src4'}})
    process_dict.update({'aggt' : {'dualprocess' : None,
                                  'singleprocess' : self._process_aggt,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'ord3'}})
    process_dict.update({'strn' : {'dualprocess' : None,
                                  'singleprocess' : self._process_strn,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'ord3'}})
    process_dict.update({'strg' : {'dualprocess' : None,
                                  'singleprocess' : self._process_strg,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_strg,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'integer',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'strg'}})
    process_dict.update({'nmrc' : {'dualprocess' : None,
                                  'singleprocess' : self._process_nmrc,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'numbers'},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmrc'}})
    process_dict.update({'nmr2' : {'dualprocess' : None,
                                  'singleprocess' : self._process_nmrc,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'numbers'},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'nmr3' : {'dualprocess' : None,
                                  'singleprocess' : self._process_nmrc,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'numbers'},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'nmr4' : {'dualprocess' : self._process_nmr4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_nmr4,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'numbers',
                                                     'test_same_as_train' : True},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmr4'}})
    process_dict.update({'nmr5' : {'dualprocess' : self._process_nmr4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_nmr4,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'numbers',
                                                     'test_same_as_train' : True},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'nmr6' : {'dualprocess' : self._process_nmr4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_nmr4,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'numbers',
                                                     'test_same_as_train' : True},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'nmr7' : {'dualprocess' : self._process_nmr4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_nmr4,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'numbers',
                                                     'test_same_as_train' : False},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmr7'}})
    process_dict.update({'nmr8' : {'dualprocess' : self._process_nmr4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_nmr4,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'numbers',
                                                     'test_same_as_train' : False},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'nmr9' : {'dualprocess' : self._process_nmr4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_nmr4,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'numbers',
                                                     'test_same_as_train' : False},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'nmcm' : {'dualprocess' : None,
                                  'singleprocess' : self._process_nmrc,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'commas'},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmcm'}})
    process_dict.update({'nmc2' : {'dualprocess' : None,
                                  'singleprocess' : self._process_nmrc,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'commas'},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'nmc3' : {'dualprocess' : None,
                                  'singleprocess' : self._process_nmrc,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'commas'},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'nmc4' : {'dualprocess' : self._process_nmr4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_nmr4,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'commas',
                                                     'test_same_as_train' : True},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmc4'}})
    process_dict.update({'nmc5' : {'dualprocess' : self._process_nmr4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_nmr4,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'commas',
                                                     'test_same_as_train' : True},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'nmc6' : {'dualprocess' : self._process_nmr4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_nmr4,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'commas',
                                                     'test_same_as_train' : True},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'nmc7' : {'dualprocess' : self._process_nmr4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_nmr4,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'commas',
                                                     'test_same_as_train' : False},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmc7'}})
    process_dict.update({'nmc8' : {'dualprocess' : self._process_nmr4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_nmr4,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'commas',
                                                     'test_same_as_train' : False},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'nmc9' : {'dualprocess' : self._process_nmr4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_nmr4,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'commas',
                                                     'test_same_as_train' : False},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'nmEU' : {'dualprocess' : None,
                                  'singleprocess' : self._process_nmrc,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'spaces'},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmEU'}})
    process_dict.update({'nmE2' : {'dualprocess' : None,
                                  'singleprocess' : self._process_nmrc,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'spaces'},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'nmE3' : {'dualprocess' : None,
                                  'singleprocess' : self._process_nmrc,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'spaces'},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'nmE4' : {'dualprocess' : self._process_nmr4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_nmr4,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'spaces',
                                                     'test_same_as_train' : True},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmE4'}})
    process_dict.update({'nmE5' : {'dualprocess' : self._process_nmr4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_nmr4,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'spaces',
                                                     'test_same_as_train' : True},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'nmE6' : {'dualprocess' : self._process_nmr4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_nmr4,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'spaces',
                                                     'test_same_as_train' : True},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'nmE7' : {'dualprocess' : self._process_nmr4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_nmr4,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'spaces',
                                                     'test_same_as_train' : False},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmE7'}})
    process_dict.update({'nmE8' : {'dualprocess' : self._process_nmr4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_nmr4,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'spaces',
                                                     'test_same_as_train' : False},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'nmE9' : {'dualprocess' : self._process_nmr4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_nmr4,
                                  'inverseprocess' : self._inverseprocess_nmrc,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'convention' : 'spaces',
                                                     'test_same_as_train' : False},
                                  'NArowtype' : 'parsenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'ors7' : {'dualprocess' : self._process_spl2,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_spl2,
                                  'inverseprocess' : self._inverseprocess_spl2,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'test_same_as_train' : False,
                                                     'consolidate_nonoverlaps' : True},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'ord3'}})
    process_dict.update({'ors5' : {'dualprocess' : self._process_spl2,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_spl2,
                                  'inverseprocess' : self._inverseprocess_spl2,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'test_same_as_train' : False,
                                                     'consolidate_nonoverlaps' : True},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'ord3'}})
    process_dict.update({'ors6' : {'dualprocess' : self._process_spl2,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_spl2,
                                  'inverseprocess' : self._inverseprocess_spl2,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'test_same_as_train' : False,
                                                     'consolidate_nonoverlaps' : True},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'ord3'}})
    process_dict.update({'ordl' : {'custom_train' : self._custom_train_ordl,
                                  'custom_test' : self._custom_test_ordl,
                                  'custom_inversion' : self._custom_inversion_ordl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'frequency_sort' : False},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'ordl'}})
    process_dict.update({'ord2' : {'custom_train' : self._custom_train_ordl,
                                  'custom_test' : self._custom_test_ordl,
                                  'custom_inversion' : self._custom_inversion_ordl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'frequency_sort' : False},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'ord3' : {'custom_train' : self._custom_train_ordl,
                                  'custom_test' : self._custom_test_ordl,
                                  'custom_inversion' : self._custom_inversion_ordl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'ord3'}})
    process_dict.update({'ord5' : {'custom_train' : self._custom_train_ordl,
                                  'custom_test' : self._custom_test_ordl,
                                  'custom_inversion' : self._custom_inversion_ordl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'frequency_sort' : False,
                                                     'null_activation' : False},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'ord5'}})
    process_dict.update({'maxb' : {'dualprocess' : self._process_maxb,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_maxb,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'maxb'}})
    process_dict.update({'or3b' : {'custom_train' : self._custom_train_ordl,
                                  'custom_test' : self._custom_test_ordl,
                                  'custom_inversion' : self._custom_inversion_ordl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'maxb'}})
    process_dict.update({'matx' : {'dualprocess' : self._process_maxb,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_maxb,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'onht'}})
    process_dict.update({'or3c' : {'custom_train' : self._custom_train_ordl,
                                  'custom_test' : self._custom_test_ordl,
                                  'custom_inversion' : self._custom_inversion_ordl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'onht'}})
    process_dict.update({'ma10' : {'dualprocess' : self._process_maxb,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_maxb,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : '1010'}})
    process_dict.update({'or3d' : {'custom_train' : self._custom_train_ordl,
                                  'custom_test' : self._custom_test_ordl,
                                  'custom_inversion' : self._custom_inversion_ordl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : '1010'}})
    process_dict.update({'ucct' : {'dualprocess' : self._process_ucct,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_ucct,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'ucct'}})
    process_dict.update({'ord4' : {'custom_train' : self._custom_train_ordl,
                                  'custom_test' : self._custom_test_ordl,
                                  'custom_inversion' : self._custom_inversion_ordl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'ors2' : {'dualprocess' : self._process_spl2,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_spl2,
                                  'inverseprocess' : self._inverseprocess_spl2,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'test_same_as_train' : False,
                                                     'consolidate_nonoverlaps' : False},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'ord3'}})
    process_dict.update({'or10' : {'custom_train' : self._custom_train_ordl,
                                  'custom_test' : self._custom_test_ordl,
                                  'custom_inversion' : self._custom_inversion_ordl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'or11' : {'custom_train' : self._custom_train_1010,
                                   'custom_test' : self._custom_test_1010,
                                   'custom_inversion' : self._custom_inversion_1010,
                                   'info_retention' : True,
                                   'inplace_option' : True,
                                   'defaultinfill' : 'naninfill',
                                   'NArowtype' : 'justNaN',
                                   'MLinfilltype' : '1010',
                                   'labelctgy' : 'ord3'}})
    process_dict.update({'or12' : {'custom_train' : self._custom_train_1010,
                                   'custom_test' : self._custom_test_1010,
                                   'custom_inversion' : self._custom_inversion_1010,
                                   'info_retention' : True,
                                   'inplace_option' : True,
                                   'defaultinfill' : 'naninfill',
                                   'NArowtype' : 'justNaN',
                                   'MLinfilltype' : '1010',
                                   'labelctgy' : 'ord3'}})
    process_dict.update({'or13' : {'custom_train' : self._custom_train_1010,
                                   'custom_test' : self._custom_test_1010,
                                   'custom_inversion' : self._custom_inversion_1010,
                                   'info_retention' : True,
                                   'inplace_option' : True,
                                   'defaultinfill' : 'naninfill',
                                   'NArowtype' : 'justNaN',
                                   'MLinfilltype' : '1010',
                                   'labelctgy' : 'ord3'}})
    process_dict.update({'or14' : {'custom_train' : self._custom_train_1010,
                                   'custom_test' : self._custom_test_1010,
                                   'custom_inversion' : self._custom_inversion_1010,
                                   'info_retention' : True,
                                   'inplace_option' : True,
                                   'defaultinfill' : 'naninfill',
                                   'NArowtype' : 'justNaN',
                                   'MLinfilltype' : '1010',
                                   'labelctgy' : 'ord3'}})
    process_dict.update({'or15' : {'dualprocess' : None,
                                   'singleprocess' : self._process_UPCS,
                                   'postprocess' : None,
                                   'inverseprocess' : self._inverseprocess_UPCS,
                                   'info_retention' : False,
                                   'inplace_option' : True,
                                   'defaultinfill' : 'naninfill',
                                   'NArowtype' : 'justNaN',
                                   'MLinfilltype' : 'totalexclude',
                                   'labelctgy' : 'ord3'}})
    process_dict.update({'or16' : {'dualprocess' : None,
                                   'singleprocess' : self._process_UPCS,
                                   'postprocess' : None,
                                   'inverseprocess' : self._inverseprocess_UPCS,
                                   'info_retention' : False,
                                   'inplace_option' : True,
                                   'defaultinfill' : 'naninfill',
                                   'NArowtype' : 'justNaN',
                                   'MLinfilltype' : 'totalexclude',
                                   'labelctgy' : 'ord3'}})
    process_dict.update({'or17' : {'dualprocess' : None,
                                   'singleprocess' : self._process_UPCS,
                                   'postprocess' : None,
                                   'inverseprocess' : self._inverseprocess_UPCS,
                                   'info_retention' : False,
                                   'inplace_option' : True,
                                   'defaultinfill' : 'naninfill',
                                   'NArowtype' : 'justNaN',
                                   'MLinfilltype' : 'totalexclude',
                                   'labelctgy' : 'ord3'}})
    process_dict.update({'or18' : {'dualprocess' : None,
                                   'singleprocess' : self._process_UPCS,
                                   'postprocess' : None,
                                   'inverseprocess' : self._inverseprocess_UPCS,
                                   'info_retention' : False,
                                   'inplace_option' : True,
                                   'defaultinfill' : 'naninfill',
                                   'NArowtype' : 'justNaN',
                                   'MLinfilltype' : 'totalexclude',
                                   'labelctgy' : 'ord3'}})
    process_dict.update({'or19' : {'dualprocess' : None,
                                   'singleprocess' : self._process_UPCS,
                                   'postprocess' : None,
                                   'inverseprocess' : self._inverseprocess_UPCS,
                                   'info_retention' : False,
                                   'inplace_option' : True,
                                   'defaultinfill' : 'naninfill',
                                   'NArowtype' : 'justNaN',
                                   'MLinfilltype' : 'totalexclude',
                                   'labelctgy' : 'ord3'}})
    process_dict.update({'or20' : {'dualprocess' : None,
                                   'singleprocess' : self._process_UPCS,
                                   'postprocess' : None,
                                   'inverseprocess' : self._inverseprocess_UPCS,
                                   'info_retention' : False,
                                   'inplace_option' : True,
                                   'defaultinfill' : 'naninfill',
                                   'NArowtype' : 'justNaN',
                                   'MLinfilltype' : 'totalexclude',
                                   'labelctgy' : 'ord3'}})
    process_dict.update({'or21' : {'dualprocess' : None,
                                   'singleprocess' : self._process_UPCS,
                                   'postprocess' : None,
                                   'inverseprocess' : self._inverseprocess_UPCS,
                                   'info_retention' : False,
                                   'inplace_option' : True,
                                   'defaultinfill' : 'naninfill',
                                   'NArowtype' : 'justNaN',
                                   'MLinfilltype' : 'totalexclude',
                                   'labelctgy' : 'ord3'}})
    process_dict.update({'or22' : {'dualprocess' : None,
                                   'singleprocess' : self._process_UPCS,
                                   'postprocess' : None,
                                   'inverseprocess' : self._inverseprocess_UPCS,
                                   'info_retention' : False,
                                   'inplace_option' : True,
                                   'defaultinfill' : 'naninfill',
                                   'NArowtype' : 'justNaN',
                                   'MLinfilltype' : 'totalexclude',
                                   'labelctgy' : 'ord3'}})
    process_dict.update({'or23' : {'dualprocess' : None,
                                   'singleprocess' : self._process_UPCS,
                                   'postprocess' : None,
                                   'inverseprocess' : self._inverseprocess_UPCS,
                                   'info_retention' : False,
                                   'inplace_option' : True,
                                   'defaultinfill' : 'naninfill',
                                   'NArowtype' : 'justNaN',
                                   'MLinfilltype' : 'totalexclude',
                                   'labelctgy' : 'ord3'}})
    process_dict.update({'om10' : {'custom_train' : self._custom_train_ordl,
                                  'custom_test' : self._custom_test_ordl,
                                  'custom_inversion' : self._custom_inversion_ordl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'mmor' : {'custom_train' : self._custom_train_ordl,
                                  'custom_test' : self._custom_test_ordl,
                                  'custom_inversion' : self._custom_inversion_ordl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'1010' : {'custom_train' : self._custom_train_1010,
                                  'custom_test' : self._custom_test_1010,
                                  'custom_inversion' : self._custom_inversion_1010,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : '1010',
                                  'labelctgy' : '1010'}})
    process_dict.update({'10mz' : {'custom_train' : self._custom_train_1010,
                                  'custom_test' : self._custom_test_1010,
                                  'custom_inversion' : self._custom_inversion_1010,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'max_zero' : True},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : '1010',
                                  'labelctgy' : '10mz'}})
    process_dict.update({'qttf' : {'dualprocess' : self._process_qttf,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_qttf,
                                  'inverseprocess' : self._inverseprocess_qttf,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'meaninfill',
                                  'defaultparams' : {'output_distribution' : 'normal'},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'qttf'}})
    process_dict.update({'qtt2' : {'dualprocess' : self._process_qttf,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_qttf,
                                  'inverseprocess' : self._inverseprocess_qttf,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'meaninfill',
                                  'defaultparams' : {'output_distribution' : 'uniform'},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'qtt2'}})
    process_dict.update({'qtt1' : {'dualprocess' : self._process_qttf,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_qttf,
                                  'inverseprocess' : self._inverseprocess_qttf,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'meaninfill',
                                  'defaultparams' : {'output_distribution' : 'normal'},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DPqt'}})
    process_dict.update({'DPqt' : {'dualprocess' : self._process_DPnb,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPnb,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'noise_transform' : 'numeric',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DPqt'}})
    process_dict.update({'bxcx' : {'dualprocess' : self._process_bxcx,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bxcx,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'positivenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'tmsc' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'tmsc'}})
    process_dict.update({'time' : {'dualprocess' : self._process_time,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_time,
                                  'inverseprocess' : self._inverseprocess_year,
                                  'info_retention' : False,
                                  'defaultinfill' : 'adjinfill',
                                  'defaultparams' : {'scale' : 'year',
                                                     'normalization' : 'zscore'},
                                  'inplace_option' : True,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'time'}})
    process_dict.update({'tmzn' : {'dualprocess' : None,
                                  'singleprocess' : self._process_tmzn,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'tmzn'}})
    process_dict.update({'date' : {'dualprocess' : None,
                                  'singleprocess' : self._process_tmzn,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'year'}})
    process_dict.update({'dat2' : {'dualprocess' : None,
                                  'singleprocess' : self._process_tmzn,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'hldy'}})
    process_dict.update({'dat3' : {'dualprocess' : None,
                                  'singleprocess' : self._process_tmzn,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'year'}})
    process_dict.update({'dat4' : {'dualprocess' : None,
                                  'singleprocess' : self._process_tmzn,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'year'}})
    process_dict.update({'dat5' : {'dualprocess' : None,
                                  'singleprocess' : self._process_tmzn,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'year'}})
    process_dict.update({'dat6' : {'dualprocess' : None,
                                  'singleprocess' : self._process_tmzn,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'year'}})
    process_dict.update({'year' : {'dualprocess' : self._process_time,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_time,
                                  'inverseprocess' : self._inverseprocess_year,
                                  'info_retention' : False,
                                  'defaultinfill' : 'adjinfill',
                                  'defaultparams' : {'scale' : 'year',
                                                     'normalization' : 'zscore'},
                                  'inplace_option' : True,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'year'}})
    process_dict.update({'yea2' : {'dualprocess' : self._process_time,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_time,
                                  'inverseprocess' : self._inverseprocess_year,
                                  'info_retention' : False,
                                  'defaultinfill' : 'adjinfill',
                                  'defaultparams' : {'scale' : 'year',
                                                     'normalization' : 'zscore'},
                                  'inplace_option' : True,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'year'}})
    process_dict.update({'yrsn' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {'scale' : 'year',
                                                     'function' : 'sin'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'yrsn'}})
    process_dict.update({'yrcs' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {'scale' : 'year',
                                                     'function' : 'cos'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'yrcs'}})
    process_dict.update({'mnth' : {'dualprocess' : self._process_time,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_time,
                                  'defaultparams' : {'scale' : 'month',
                                                     'normalization' : 'zscore'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnth'}})
    process_dict.update({'mnt2' : {'dualprocess' : None,
                                  'singleprocess' : None,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnsn'}})
    process_dict.update({'mnt3' : {'dualprocess' : None,
                                  'singleprocess' : None,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnsn'}})
    process_dict.update({'mnt4' : {'dualprocess' : None,
                                  'singleprocess' : None,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mdsn'}})
    process_dict.update({'mnt5' : {'dualprocess' : None,
                                  'singleprocess' : None,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mdsn'}})
    process_dict.update({'mnt6' : {'dualprocess' : None,
                                  'singleprocess' : None,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mdsn'}})
    process_dict.update({'mnsn' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {'scale' : 'month',
                                                     'function' : 'sin'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mnsn'}})
    process_dict.update({'mncs' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {'scale' : 'month',
                                                     'function' : 'cos'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mncs'}})
    process_dict.update({'mdsn' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {'scale' : 'monthday',
                                                     'function' : 'sin'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mdsn'}})
    process_dict.update({'mdcs' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {'scale' : 'monthday',
                                                     'function' : 'cos'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mdcs'}})
    process_dict.update({'days' : {'dualprocess' : self._process_time,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_time,
                                  'defaultparams' : {'scale' : 'day',
                                                     'normalization' : 'zscore'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'days'}})
    process_dict.update({'day2' : {'dualprocess' : None,
                                  'singleprocess' : None,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'dysn'}})
    process_dict.update({'day3' : {'dualprocess' : None,
                                  'singleprocess' : None,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'dysn'}})
    process_dict.update({'day4' : {'dualprocess' : None,
                                  'singleprocess' : None,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'dhms'}})
    process_dict.update({'day5' : {'dualprocess' : None,
                                  'singleprocess' : None,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'dhms'}})
    process_dict.update({'dysn' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {'scale' : 'day',
                                                     'function' : 'sin'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'dysn'}})
    process_dict.update({'dycs' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {'scale' : 'day',
                                                     'function' : 'cos'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'dycs'}})
    process_dict.update({'dhms' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {'scale' : 'dayhourminute',
                                                     'function' : 'sin'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'dhms'}})
    process_dict.update({'dhmc' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {'scale' : 'dayhourminute',
                                                     'function' : 'cos'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'dhmc'}})
    process_dict.update({'hour' : {'dualprocess' : self._process_time,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_time,
                                  'defaultparams' : {'scale' : 'hour',
                                                     'normalization' : 'zscore'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'hour'}})
    process_dict.update({'hrs2' : {'dualprocess' : None,
                                  'singleprocess' : None,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'hrsn'}})
    process_dict.update({'hrs3' : {'dualprocess' : None,
                                  'singleprocess' : None,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'hrsn'}})
    process_dict.update({'hrs4' : {'dualprocess' : None,
                                  'singleprocess' : None,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'hmss'}})
    process_dict.update({'hrsn' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {'scale' : 'hour',
                                                     'function' : 'sin'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'hrsn'}})
    process_dict.update({'hrcs' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {'scale' : 'hour',
                                                     'function' : 'cos'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'hrcs'}})
    process_dict.update({'hmss' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {'scale' : 'hourminutesecond',
                                                     'function' : 'sin'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'hmss'}})
    process_dict.update({'hmsc' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {'scale' : 'hourminutesecond',
                                                     'function' : 'cos'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'hmsc'}})
    process_dict.update({'mint' : {'dualprocess' : self._process_time,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_time,
                                  'defaultparams' : {'scale' : 'minute',
                                                     'normalization' : 'zscore'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mint'}})
    process_dict.update({'min2' : {'dualprocess' : None,
                                  'singleprocess' : None,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'misn'}})
    process_dict.update({'min3' : {'dualprocess' : None,
                                  'singleprocess' : None,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'misn'}})
    process_dict.update({'min4' : {'dualprocess' : None,
                                  'singleprocess' : None,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mssn'}})
    process_dict.update({'misn' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {'scale' : 'minute',
                                                     'function' : 'sin'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'misn'}})
    process_dict.update({'mics' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {'scale' : 'minute',
                                                     'function' : 'cos'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mics'}})
    process_dict.update({'mssn' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {'scale' : 'minutesecond',
                                                     'function' : 'sin'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mssn'}})
    process_dict.update({'mscs' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {'scale' : 'minutesecond',
                                                     'function' : 'cos'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mscs'}})
    process_dict.update({'scnd' : {'dualprocess' : self._process_time,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_time,
                                  'defaultparams' : {'scale' : 'second',
                                                     'normalization' : 'zscore'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'scnd'}})
    process_dict.update({'scn2' : {'dualprocess' : None,
                                  'singleprocess' : None,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'scsn'}})
    process_dict.update({'scsn' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {'scale' : 'second',
                                                     'function' : 'sin'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'scsn'}})
    process_dict.update({'sccs' : {'dualprocess' : self._process_tmsc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tmsc,
                                  'defaultparams' : {'scale' : 'second',
                                                     'function' : 'cos'},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'sccs'}})
    process_dict.update({'bxc2' : {'dualprocess' : self._process_bxcx,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bxcx,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'nonzeronumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'bxc3' : {'dualprocess' : self._process_bxcx,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bxcx,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'positivenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'bxc4' : {'dualprocess' : self._process_bxcx,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bxcx,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'positivenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nbr2'}})
    process_dict.update({'bxc5' : {'dualprocess' : self._process_bxcx,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bxcx,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'positivenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nbr2'}})
    process_dict.update({'bxc6' : {'dualprocess' : self._process_bxcx,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bxcx,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'positivenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DPbx'}})
    process_dict.update({'bxc7' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DPbx'}})
    process_dict.update({'DPbx' : {'dualprocess' : self._process_DPnb,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPnb,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'noise_transform' : 'numeric',
                                  'NArowtype' : 'positivenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DPbx'}})
    process_dict.update({'ntgr' : {'custom_train' : self._custom_train_ordl,
                                  'custom_test' : self._custom_test_ordl,
                                  'custom_inversion' : self._custom_inversion_ordl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'integer',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'ntg2' : {'custom_train' : self._custom_train_ordl,
                                  'custom_test' : self._custom_test_ordl,
                                  'custom_inversion' : self._custom_inversion_ordl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'integer',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'ntg3' : {'custom_train' : self._custom_train_ordl,
                                  'custom_test' : self._custom_test_ordl,
                                  'custom_inversion' : self._custom_inversion_ordl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'integer',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'mnmx'}})
    process_dict.update({'pwrs' : {'dualprocess' : self._process_pwrs,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_pwrs,
                                  'inverseprocess' : self._inverseprocess_pwr2,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultparams' : {'negvalues' : False},
                                  'NArowtype' : 'positivenumeric',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'pwrs'}})
    process_dict.update({'pwr2' : {'dualprocess' : self._process_pwrs,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_pwrs,
                                  'inverseprocess' : self._inverseprocess_pwr2,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultparams' : {'negvalues' : True},
                                  'NArowtype' : 'nonzeronumeric',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'pwr2'}})
    process_dict.update({'log0' : {'dualprocess' : self._process_log0,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_log0,
                                  'inverseprocess' : self._inverseprocess_log0,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'positivenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'log0'}})
    process_dict.update({'log1' : {'dualprocess' : self._process_log0,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_log0,
                                  'inverseprocess' : self._inverseprocess_log0,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'positivenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'log0'}})
    process_dict.update({'logn' : {'dualprocess' : self._process_logn,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_logn,
                                  'inverseprocess' : self._inverseprocess_logn,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'positivenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'logn'}})
    process_dict.update({'lgnm' : {'dualprocess' : self._process_logn,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_logn,
                                  'inverseprocess' : self._inverseprocess_logn,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'positivenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'sqrt' : {'dualprocess' : self._process_sqrt,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_sqrt,
                                  'inverseprocess' : self._inverseprocess_sqrt,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'nonnegativenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'sqrt'}})
    process_dict.update({'addd' : {'dualprocess' : self._process_addd,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_addd,
                                  'inverseprocess' : self._inverseprocess_addd,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'addd'}})
    process_dict.update({'sbtr' : {'dualprocess' : self._process_sbtr,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_sbtr,
                                  'inverseprocess' : self._inverseprocess_sbtr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'sbtr'}})
    process_dict.update({'mltp' : {'dualprocess' : self._process_mltp,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mltp,
                                  'inverseprocess' : self._inverseprocess_mltp,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'mltp'}})
    process_dict.update({'divd' : {'dualprocess' : self._process_divd,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_divd,
                                  'inverseprocess' : self._inverseprocess_divd,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'divd'}})
    process_dict.update({'rais' : {'dualprocess' : self._process_rais,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_rais,
                                  'inverseprocess' : self._inverseprocess_rais,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'rais'}})
    process_dict.update({'absl' : {'dualprocess' : self._process_absl,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_absl,
                                  'inverseprocess' : self._inverseprocess_absl,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'absl'}})
    process_dict.update({'bkt1' : {'dualprocess' : self._process_bkt1,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bkt1,
                                  'inverseprocess' : self._inverseprocess_bkt1,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'bkt1'}})
    process_dict.update({'bkt2' : {'dualprocess' : self._process_bkt2,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bkt2,
                                  'inverseprocess' : self._inverseprocess_bkt2,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'bkt2'}})
    process_dict.update({'bkt3' : {'dualprocess' : self._process_bkt3,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bkt3,
                                  'inverseprocess' : self._inverseprocess_bkt3,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'bkt3'}})
    process_dict.update({'bkt4' : {'dualprocess' : self._process_bkt4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bkt4,
                                  'inverseprocess' : self._inverseprocess_bkt4,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'bkt4'}})
    process_dict.update({'wkdy' : {'dualprocess' : None,
                                  'singleprocess' : self._process_wkdy,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'binary',
                                  'labelctgy' : 'wkdy'}})
    process_dict.update({'bshr' : {'dualprocess' : None,
                                  'singleprocess' : self._process_bshr,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'binary',
                                  'labelctgy' : 'bshr'}})
    process_dict.update({'hldy' : {'dualprocess' : None,
                                  'singleprocess' : self._process_hldy,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'binary',
                                  'labelctgy' : 'hldy'}})
    process_dict.update({'wkds' : {'dualprocess' : None,
                                  'singleprocess' : self._process_wkds,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'text'}})
    process_dict.update({'wkdo' : {'dualprocess' : None,
                                  'singleprocess' : self._process_wkds,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'ordl'}})
    process_dict.update({'mnts' : {'dualprocess' : None,
                                  'singleprocess' : self._process_mnts,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'text'}})
    process_dict.update({'mnto' : {'dualprocess' : None,
                                  'singleprocess' : self._process_mnts,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'ordl'}})
    process_dict.update({'bins' : {'dualprocess' : self._process_bins,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bins,
                                  'inverseprocess' : self._inverseprocess_bins,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'bins'}})
    process_dict.update({'bint' : {'dualprocess' : self._process_bins,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bins,
                                  'inverseprocess' : self._inverseprocess_bins,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'meaninfill',
                                  'defaultparams' : {'normalizedinput' : True},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'bint'}})
    process_dict.update({'bsor' : {'dualprocess' : self._process_bsor,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bsor,
                                  'inverseprocess' : self._inverseprocess_bsor,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'bsor'}})
    process_dict.update({'btor' : {'dualprocess' : self._process_bsor,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bsor,
                                  'inverseprocess' : self._inverseprocess_bsor,
                                  'info_retention' : False,
                                  'defaultparams' : {'normalizedinput' : True},
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'btor'}})
    process_dict.update({'bnwd' : {'dualprocess' : self._process_bnwd,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bnwd,
                                  'inverseprocess' : self._inverseprocess_bnwd,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'bnwd'}})
    process_dict.update({'bnwK' : {'dualprocess' : self._process_bnwd,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bnwd,
                                  'inverseprocess' : self._inverseprocess_bnwd,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'meaninfill',
                                  'defaultparams' : {'width':1000},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'bnwK'}})
    process_dict.update({'bnwM' : {'dualprocess' : self._process_bnwd,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bnwd,
                                  'inverseprocess' : self._inverseprocess_bnwd,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'meaninfill',
                                  'defaultparams' : {'width':1000000},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'bnwM'}})
    process_dict.update({'bnwo' : {'dualprocess' : self._process_bnwo,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bnwo,
                                  'inverseprocess' : self._inverseprocess_bnwo,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'bnwo'}})
    process_dict.update({'bnKo' : {'dualprocess' : self._process_bnwo,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bnwo,
                                  'inverseprocess' : self._inverseprocess_bnwo,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'defaultparams' : {'width':1000},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'bnKo'}})
    process_dict.update({'bnMo' : {'dualprocess' : self._process_bnwo,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bnwo,
                                  'inverseprocess' : self._inverseprocess_bnwo,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'defaultparams' : {'width':1000000},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'bnMo'}})
    process_dict.update({'bnep' : {'dualprocess' : self._process_bnep,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bnep,
                                  'inverseprocess' : self._inverseprocess_bnep,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'bnep'}})
    process_dict.update({'bne7' : {'dualprocess' : self._process_bnep,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bnep,
                                  'inverseprocess' : self._inverseprocess_bnep,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultparams' : {'bincount':7},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'bne7'}})
    process_dict.update({'bne9' : {'dualprocess' : self._process_bnep,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bnep,
                                  'inverseprocess' : self._inverseprocess_bnep,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultparams' : {'bincount':9},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'bne9'}})
    process_dict.update({'bneo' : {'dualprocess' : self._process_bneo,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bneo,
                                  'inverseprocess' : self._inverseprocess_bneo,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'bneo'}})
    process_dict.update({'bn7o' : {'dualprocess' : self._process_bneo,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bneo,
                                  'inverseprocess' : self._inverseprocess_bneo,
                                  'info_retention' : False,
                                  'defaultinfill' : 'adjinfill',
                                  'defaultparams' : {'bincount':7},
                                  'inplace_option' : True,
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'bn7o'}})
    process_dict.update({'bn9o' : {'dualprocess' : self._process_bneo,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bneo,
                                  'inverseprocess' : self._inverseprocess_bneo,
                                  'info_retention' : False,
                                  'defaultinfill' : 'adjinfill',
                                  'defaultparams' : {'bincount':9},
                                  'inplace_option' : True,
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'bn9o'}})
    process_dict.update({'tlbn' : {'dualprocess' : self._process_tlbn,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_tlbn,
                                  'inverseprocess' : self._inverseprocess_tlbn,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'concurrent_nmbr',
                                  'labelctgy' : 'tlbn'}})
    process_dict.update({'pwor' : {'dualprocess' : self._process_pwor,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_pwor,
                                  'inverseprocess' : self._inverseprocess_pwor,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultparams' : {'negvalues' : False},
                                  'NArowtype' : 'positivenumeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'pwor'}})
    process_dict.update({'por2' : {'dualprocess' : self._process_pwor,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_pwor,
                                  'inverseprocess' : self._inverseprocess_pwor,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultparams' : {'negvalues' : True},
                                  'NArowtype' : 'nonzeronumeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'por2'}})
    process_dict.update({'por3' : {'dualprocess' : self._process_pwor,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_pwor,
                                  'inverseprocess' : self._inverseprocess_pwor,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultparams' : {'negvalues' : True},
                                  'NArowtype' : 'nonzeronumeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : '1010'}})
    process_dict.update({'bkb3' : {'dualprocess' : self._process_bkt3,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bkt3,
                                  'inverseprocess' : self._inverseprocess_bkt3,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : '1010'}})
    process_dict.update({'bkb4' : {'dualprocess' : self._process_bkt4,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bkt4,
                                  'inverseprocess' : self._inverseprocess_bkt4,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : '1010'}})
    process_dict.update({'bsbn' : {'dualprocess' : self._process_bsor,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bsor,
                                  'inverseprocess' : self._inverseprocess_bsor,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : '1010'}})
    process_dict.update({'bnwb' : {'dualprocess' : self._process_bnwo,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bnwo,
                                  'inverseprocess' : self._inverseprocess_bnwo,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : '1010'}})
    process_dict.update({'bnKb' : {'dualprocess' : self._process_bnwo,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bnwo,
                                  'inverseprocess' : self._inverseprocess_bnwo,
                                  'info_retention' : False,
                                  'defaultinfill' : 'meaninfill',
                                  'defaultparams' : {'width':1000},
                                  'inplace_option' : True,
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : '1010'}})
    process_dict.update({'bnMb' : {'dualprocess' : self._process_bnwo,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bnwo,
                                  'inverseprocess' : self._inverseprocess_bnwo,
                                  'info_retention' : False,
                                  'defaultinfill' : 'meaninfill',
                                  'defaultparams' : {'width':1000000},
                                  'inplace_option' : True,
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : '1010'}})
    process_dict.update({'bneb' : {'dualprocess' : self._process_bneo,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bneo,
                                  'inverseprocess' : self._inverseprocess_bneo,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : '1010'}})
    process_dict.update({'bn7b' : {'dualprocess' : self._process_bneo,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bneo,
                                  'inverseprocess' : self._inverseprocess_bneo,
                                  'info_retention' : False,
                                  'defaultinfill' : 'adjinfill',
                                  'defaultparams' : {'bincount':7},
                                  'inplace_option' : True,
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : '1010'}})
    process_dict.update({'bn9b' : {'dualprocess' : self._process_bneo,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bneo,
                                  'inverseprocess' : self._inverseprocess_bneo,
                                  'info_retention' : False,
                                  'defaultinfill' : 'adjinfill',
                                  'defaultparams' : {'bincount':9},
                                  'inplace_option' : True,
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : '1010'}})
    process_dict.update({'pwbn' : {'dualprocess' : self._process_pwor,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_pwor,
                                  'inverseprocess' : self._inverseprocess_pwor,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'NArowtype' : 'positivenumeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : '1010'}})
    process_dict.update({'DTn3' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DTnb'}})
    process_dict.update({'DTnb' : {'dualprocess' : self._process_DPnb,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPnb,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'trainnoise' : False,
                                                     'testnoise' : True},
                                  'noise_transform' : 'numeric',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DTnb'}})
    process_dict.update({'DTm2' : {'dualprocess' : self._process_mnmx,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mnmx,
                                  'inverseprocess' : self._inverseprocess_mnmx,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DTmm'}})
    process_dict.update({'DTmm' : {'dualprocess' : self._process_DPmm,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmm,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultparams' : {'trainnoise' : False,
                                                     'testnoise' : True},
                                  'noise_transform' : 'numeric',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DTmm'}})
    process_dict.update({'DTrt' : {'dualprocess' : self._process_DPrt,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPrt,
                                  'inverseprocess' : self._inverseprocess_retn,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultparams' : {'trainnoise' : False,
                                                     'testnoise' : True},
                                  'noise_transform' : 'numeric',
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DTrt'}})
    process_dict.update({'DTb2' : {'dualprocess' : self._process_binary,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_binary,
                                  'inverseprocess' : self._inverseprocess_bnry,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'binary',
                                  'MLinfilltype' : 'binary',
                                  'labelctgy' : 'DTbn'}})
    process_dict.update({'DTbn' : {'dualprocess' : self._process_DPod,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPod,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'trainnoise' : False,
                                                     'testnoise' : True},
                                  'noise_transform' : 'categoric',
                                  'NArowtype' : 'binary',
                                  'MLinfilltype' : 'binary',
                                  'labelctgy' : 'DTbn'}})
    process_dict.update({'DTo4' : {'custom_train' : self._custom_train_ordl,
                                  'custom_test' : self._custom_test_ordl,
                                  'custom_inversion' : self._custom_inversion_ordl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'DTod'}})
    process_dict.update({'DTod' : {'dualprocess' : self._process_DPod,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPod,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'trainnoise' : False,
                                                     'testnoise' : True},
                                  'noise_transform' : 'categoric',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'DTod'}})
    process_dict.update({'DTo5' : {'custom_train' : self._custom_train_onht,
                                  'custom_test' : self._custom_test_onht,
                                  'custom_inversion' : self._custom_inversion_onht,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'suffix_convention' : 'onht'},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'DToh'}})
    process_dict.update({'DToh' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'trainnoise' : False,
                                                     'testnoise' : True},
                                  'noise_transform' : 'categoric',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'DToh'}})
    process_dict.update({'DT10' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'trainnoise' : False,
                                                     'testnoise' : True},
                                  'noise_transform' : 'categoric',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : '1010',
                                  'labelctgy' : 'DT10'}})
    process_dict.update({'DTo6' : {'custom_train' : self._custom_train_1010,
                                  'custom_test' : self._custom_test_1010,
                                  'custom_inversion' : self._custom_inversion_1010,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : '1010',
                                  'labelctgy' : 'DT10'}})
    process_dict.update({'DTh1' : {'dualprocess' : self._process_hs10,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_hs10,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'boolexclude',
                                  'labelctgy' : 'DTmc'}})
    #if DTmc to be applied downstream of multirt or 1010 MLinfilltype recommend recasting MLinfilltype to match
    process_dict.update({'DTmc' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'trainnoise' : False,
                                                     'testnoise' : True},
                                  'noise_transform' : 'categoric',
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'boolexclude',
                                  'labelctgy' : 'DTmc'}})
    #note that inplace set to False when receiving messy_data_headers specification
    process_dict.update({'DTmp' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'messy_data_headers' : []},
                                  'noise_transform' : 'categoric',
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'exclude',
                                  'MLinfilltype' : 'exclude',
                                  'labelctgy' : 'DTmp'}})
    process_dict.update({'DTo7' : {'dualprocess' : self._process_DPod,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPod,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'upstream_hsh2' : True,
                                                     'trainnoise' : False,
                                                     'testnoise' : True},
                                  'noise_transform' : 'categoric',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'DTo7'}})
    process_dict.update({'DThs' : {'dualprocess' : self._process_hash,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_hash,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'ordlexclude',
                                  'labelctgy' : 'mThs'}})
    process_dict.update({'DTh2' : {'dualprocess' : self._process_hash,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_hash,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'space' : '',
                                                     'excluded_characters' : []},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'ordlexclude',
                                  'labelctgy' : 'DTod'}})
    process_dict.update({'DTn4' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DTns'}})
    process_dict.update({'DTns' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'noise_transform' : 'categoric',
                                  'defaultparams' : {'swap_noise' : True,
                                                     'trainnoise' : False,
                                                     'testnoise' : True},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DTns'}})
    process_dict.update({'DTo8' : {'custom_train' : self._custom_train_1010,
                                  'custom_test' : self._custom_test_1010,
                                  'custom_inversion' : self._custom_inversion_1010,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : '1010',
                                  'labelctgy' : 'DT1s'}})
    process_dict.update({'DT1s' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'noise_transform' : 'categoric',
                                  'defaultparams' : {'swap_noise' : True,
                                                     'trainnoise' : False,
                                                     'testnoise' : True},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : '1010',
                                  'labelctgy' : 'DT1s'}})
    process_dict.update({'DTsk' : {'dualprocess' : self._process_DPsk,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPsk,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultparams' : {'trainnoise' : False,
                                                     'testnoise' : True},
                                  'noise_transform' : 'binary',
                                  'NArowtype' : 'totalexclude',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'DTsk'}})
    process_dict.update({'DTne' : {'dualprocess' : self._process_DPnb,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPnb,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'rescale_sigmas' : True,
                                                     'trainnoise' : False,
                                                     'testnoise' : True},
                                  'noise_transform' : 'numeric',
                                  'NArowtype' : 'totalexclude',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'DTne'}})
    process_dict.update({'DTse' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'swap_noise' : True,
                                                     'trainnoise' : False,
                                                     'testnoise' : True},
                                  'noise_transform' : 'categoric',
                                  'NArowtype' : 'totalexclude',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'DTse'}})
    #note that the default norm_category DPod is intended for use as a downstream tree category
    process_dict.update({'mThs' : {'dualprocess' : self._process_mlti,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mlti,
                                  'inverseprocess' : self._inverseprocess_mlti,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'noise_transform' : 'categoric',
                                  'defaultparams' : {'dtype' : 'mThs',
                                                     'flip_prob' : 0.03,
                                                     'testnoise' : False,
                                                     'norm_params' : {'test_flip_prob' : 0.01,
                                                                      'trainnoise' : False,
                                                                      'testnoise' : True,
                                                                      'test_weighted' : True},
                                                     'norm_category' : 'DTod'},
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'ordlexclude',
                                  'labelctgy' : 'mThs'}})
    process_dict.update({'DBn3' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DBnb'}})
    process_dict.update({'DBnb' : {'dualprocess' : self._process_DPnb,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPnb,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'trainnoise' : True,
                                                     'testnoise' : True},
                                  'noise_transform' : 'numeric',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DBnb'}})
    process_dict.update({'DBm2' : {'dualprocess' : self._process_mnmx,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mnmx,
                                  'inverseprocess' : self._inverseprocess_mnmx,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DBmm'}})
    process_dict.update({'DBmm' : {'dualprocess' : self._process_DPmm,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmm,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultparams' : {'trainnoise' : True,
                                                     'testnoise' : True},
                                  'noise_transform' : 'numeric',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DBmm'}})
    process_dict.update({'DBrt' : {'dualprocess' : self._process_DPrt,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPrt,
                                  'inverseprocess' : self._inverseprocess_retn,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultparams' : {'trainnoise' : True,
                                                     'testnoise' : True},
                                  'noise_transform' : 'numeric',
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DBrt'}})
    process_dict.update({'DBb2' : {'dualprocess' : self._process_binary,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_binary,
                                  'inverseprocess' : self._inverseprocess_bnry,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'binary',
                                  'MLinfilltype' : 'binary',
                                  'labelctgy' : 'DBbn'}})
    process_dict.update({'DBbn' : {'dualprocess' : self._process_DPod,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPod,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'trainnoise' : True,
                                                     'testnoise' : True},
                                  'noise_transform' : 'categoric',
                                  'NArowtype' : 'binary',
                                  'MLinfilltype' : 'binary',
                                  'labelctgy' : 'DBbn'}})
    process_dict.update({'DBo4' : {'custom_train' : self._custom_train_ordl,
                                  'custom_test' : self._custom_test_ordl,
                                  'custom_inversion' : self._custom_inversion_ordl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'DBod'}})
    process_dict.update({'DBod' : {'dualprocess' : self._process_DPod,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPod,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'trainnoise' : True,
                                                     'testnoise' : True},
                                  'noise_transform' : 'categoric',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'DBod'}})
    process_dict.update({'DBo5' : {'custom_train' : self._custom_train_onht,
                                  'custom_test' : self._custom_test_onht,
                                  'custom_inversion' : self._custom_inversion_onht,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'suffix_convention' : 'onht'},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'DBoh'}})
    process_dict.update({'DBoh' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'trainnoise' : True,
                                                     'testnoise' : True},
                                  'noise_transform' : 'categoric',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'DBoh'}})
    process_dict.update({'DB10' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'trainnoise' : True,
                                                     'testnoise' : True},
                                  'noise_transform' : 'categoric',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : '1010',
                                  'labelctgy' : 'DB10'}})
    process_dict.update({'DBo6' : {'custom_train' : self._custom_train_1010,
                                  'custom_test' : self._custom_test_1010,
                                  'custom_inversion' : self._custom_inversion_1010,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : '1010',
                                  'labelctgy' : 'DB10'}})
    process_dict.update({'DBh1' : {'dualprocess' : self._process_hs10,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_hs10,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'boolexclude',
                                  'labelctgy' : 'DBmc'}})
    #if DTmc to be applied downstream of multirt or 1010 MLinfilltype recommend recasting MLinfilltype to match
    process_dict.update({'DBmc' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'trainnoise' : True,
                                                     'testnoise' : True},
                                  'noise_transform' : 'categoric',
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'boolexclude',
                                  'labelctgy' : 'DBmc'}})
    #note that inplace set to False when receiving messy_data_headers specification
    process_dict.update({'DBmp' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'messy_data_headers' : []},
                                  'noise_transform' : 'categoric',
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'exclude',
                                  'MLinfilltype' : 'exclude',
                                  'labelctgy' : 'DBmp'}})
    process_dict.update({'DBo7' : {'dualprocess' : self._process_DPod,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPod,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'upstream_hsh2' : True,
                                                     'trainnoise' : True,
                                                     'testnoise' : True},
                                  'noise_transform' : 'categoric',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'DBo7'}})
    process_dict.update({'DBhs' : {'dualprocess' : self._process_hash,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_hash,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'ordlexclude',
                                  'labelctgy' : 'mBhs'}})
    process_dict.update({'DBh2' : {'dualprocess' : self._process_hash,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_hash,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'space' : '',
                                                     'excluded_characters' : []},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'ordlexclude',
                                  'labelctgy' : 'DBod'}})
    process_dict.update({'DBn4' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DBns'}})
    process_dict.update({'DBns' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'noise_transform' : 'categoric',
                                  'defaultparams' : {'swap_noise' : True,
                                                     'trainnoise' : True,
                                                     'testnoise' : True},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DBns'}})
    process_dict.update({'DBo8' : {'custom_train' : self._custom_train_1010,
                                  'custom_test' : self._custom_test_1010,
                                  'custom_inversion' : self._custom_inversion_1010,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : '1010',
                                  'labelctgy' : 'DB1s'}})
    process_dict.update({'DB1s' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'noise_transform' : 'categoric',
                                  'defaultparams' : {'swap_noise' : True,
                                                     'trainnoise' : True,
                                                     'testnoise' : True},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : '1010',
                                  'labelctgy' : 'DB1s'}})
    process_dict.update({'DBsk' : {'dualprocess' : self._process_DPsk,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPsk,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'defaultparams' : {'trainnoise' : True,
                                                     'testnoise' : True},
                                  'noise_transform' : 'binary',
                                  'NArowtype' : 'totalexclude',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'DBsk'}})
    process_dict.update({'DBne' : {'dualprocess' : self._process_DPnb,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPnb,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'rescale_sigmas' : True,
                                                     'trainnoise' : True,
                                                     'testnoise' : True},
                                  'noise_transform' : 'numeric',
                                  'NArowtype' : 'totalexclude',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'DBne'}})
    process_dict.update({'DBse' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'swap_noise' : True,
                                                     'trainnoise' : True,
                                                     'testnoise' : True},
                                  'noise_transform' : 'categoric',
                                  'NArowtype' : 'totalexclude',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'DBse'}})
    #note that the default norm_category DPod is intended for use as a downstream tree category
    process_dict.update({'mBhs' : {'dualprocess' : self._process_mlti,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mlti,
                                  'inverseprocess' : self._inverseprocess_mlti,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'noise_transform' : 'categoric',
                                  'defaultparams' : {'dtype' : 'mThs',
                                                     'flip_prob' : 0.03,
                                                     'testnoise' : False,
                                                     'norm_params' : {'flip_prob' : 0.03,
                                                                      'test_flip_prob' : 0.01,
                                                                      'trainnoise' : True,
                                                                      'testnoise' : True,
                                                                      'weighted' : True,
                                                                      'test_weighted' : True},
                                                     'norm_category' : 'DBod'},
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'ordlexclude',
                                  'labelctgy' : 'mBhs'}})
    process_dict.update({'DPn3' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DPnb'}})
    process_dict.update({'DPnb' : {'dualprocess' : self._process_DPnb,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPnb,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'noise_transform' : 'numeric',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DPnb'}})
    process_dict.update({'DPm2' : {'dualprocess' : self._process_mnmx,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mnmx,
                                  'inverseprocess' : self._inverseprocess_mnmx,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DPmm'}})
    process_dict.update({'DPmm' : {'dualprocess' : self._process_DPmm,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmm,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'noise_transform' : 'numeric',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DPmm'}})
    process_dict.update({'DPrt' : {'dualprocess' : self._process_DPrt,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPrt,
                                  'inverseprocess' : self._inverseprocess_retn,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'noise_transform' : 'numeric',
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DPrt'}})
    process_dict.update({'DLn3' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DLnb'}})
    process_dict.update({'DLnb' : {'dualprocess' : self._process_DPnb,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPnb,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'noise_transform' : 'numeric',
                                  'defaultparams' : {'noisedistribution' : 'laplace'},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DLnb'}})
    process_dict.update({'DLm2' : {'dualprocess' : self._process_mnmx,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mnmx,
                                  'inverseprocess' : self._inverseprocess_mnmx,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DLmm'}})
    process_dict.update({'DLmm' : {'dualprocess' : self._process_DPmm,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmm,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'noise_transform' : 'numeric',
                                  'defaultparams' : {'noisedistribution' : 'laplace'},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DLmm'}})
    process_dict.update({'DLrt' : {'dualprocess' : self._process_DPrt,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPrt,
                                  'inverseprocess' : self._inverseprocess_retn,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'noise_transform' : 'numeric',
                                  'defaultinfill' : 'meaninfill',
                                  'defaultparams' : {'noisedistribution' : 'laplace'},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DLrt'}})
    process_dict.update({'DPb2' : {'dualprocess' : self._process_binary,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_binary,
                                  'inverseprocess' : self._inverseprocess_bnry,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'binary',
                                  'MLinfilltype' : 'binary',
                                  'labelctgy' : 'DPbn'}})
    process_dict.update({'DPbn' : {'dualprocess' : self._process_DPod,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPod,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'noise_transform' : 'categoric',
                                  'NArowtype' : 'binary',
                                  'MLinfilltype' : 'binary',
                                  'labelctgy' : 'DPbn'}})
    process_dict.update({'DPo4' : {'custom_train' : self._custom_train_ordl,
                                  'custom_test' : self._custom_test_ordl,
                                  'custom_inversion' : self._custom_inversion_ordl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'DPod'}})
    process_dict.update({'DPod' : {'dualprocess' : self._process_DPod,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPod,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'noise_transform' : 'categoric',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'DPod'}})
    process_dict.update({'DPo5' : {'custom_train' : self._custom_train_onht,
                                  'custom_test' : self._custom_test_onht,
                                  'custom_inversion' : self._custom_inversion_onht,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'suffix_convention' : 'onht'},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'DPoh'}})
    process_dict.update({'DPoh' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'noise_transform' : 'categoric',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'DPoh'}})
    process_dict.update({'DP10' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'noise_transform' : 'categoric',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : '1010',
                                  'labelctgy' : 'DP10'}})
    process_dict.update({'DPo6' : {'custom_train' : self._custom_train_1010,
                                  'custom_test' : self._custom_test_1010,
                                  'custom_inversion' : self._custom_inversion_1010,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : '1010',
                                  'labelctgy' : 'DP10'}})
    process_dict.update({'DPh1' : {'dualprocess' : self._process_hs10,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_hs10,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'boolexclude',
                                  'labelctgy' : 'DPmc'}})
    #if DPmc to be applied downstream of multirt or 1010 MLinfilltype recommend recasting MLinfilltype to match
    process_dict.update({'DPmc' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'noise_transform' : 'categoric',
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'boolexclude',
                                  'labelctgy' : 'DPmc'}})
    #note that inplace set to False when receiving messy_data_headers specification
    process_dict.update({'DPmp' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'messy_data_headers' : []},
                                  'noise_transform' : 'categoric',
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'exclude',
                                  'MLinfilltype' : 'exclude',
                                  'labelctgy' : 'DPmp'}})
    process_dict.update({'DPo7' : {'dualprocess' : self._process_DPod,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPod,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'noise_transform' : 'categoric',
                                  'defaultparams' : {'upstream_hsh2' : True},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'DPo7'}})
    process_dict.update({'DPhs' : {'dualprocess' : self._process_hash,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_hash,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'ordlexclude',
                                  'labelctgy' : 'mlhs'}})
    process_dict.update({'DPh2' : {'dualprocess' : self._process_hash,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_hash,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'space' : '',
                                                     'excluded_characters' : []},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'ordlexclude',
                                  'labelctgy' : 'DPod'}})
    process_dict.update({'DPn4' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DPns'}})
    process_dict.update({'DPns' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'noise_transform' : 'categoric',
                                  'defaultparams' : {'swap_noise' : True},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'DPns'}})
    process_dict.update({'DPo8' : {'custom_train' : self._custom_train_1010,
                                  'custom_test' : self._custom_test_1010,
                                  'custom_inversion' : self._custom_inversion_1010,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : '1010',
                                  'labelctgy' : 'DP1s'}})
    process_dict.update({'DP1s' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'noise_transform' : 'categoric',
                                  'defaultparams' : {'swap_noise' : True},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : '1010',
                                  'labelctgy' : 'DP1s'}})
    process_dict.update({'DPsk' : {'dualprocess' : self._process_DPsk,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPsk,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : False,
                                  'inplace_option' : False,
                                  'noise_transform' : 'binary',
                                  'NArowtype' : 'totalexclude',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'DPsk'}})
    process_dict.update({'DPne' : {'dualprocess' : self._process_DPnb,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPnb,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'noise_transform' : 'numeric',
                                  'defaultparams' : {'rescale_sigmas' : True},
                                  'NArowtype' : 'totalexclude',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'DPne'}})
    process_dict.update({'DPse' : {'dualprocess' : self._process_DPmc,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPmc,
                                  'inverseprocess' : self._inverseprocess_DPmc,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'noise_transform' : 'categoric',
                                  'defaultparams' : {'swap_noise' : True},
                                  'NArowtype' : 'totalexclude',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'DPse'}})
    process_dict.update({'DPpc' : {'dualprocess' : self._process_DPod,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_DPod,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'noise_transform' : 'categoric',
                                  'defaultparams' : {'passthrough' : True},
                                  'NArowtype' : 'totalexclude',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'DPpc'}})
    #note that the default norm_category DPod is intended for use as a downstream tree category
    process_dict.update({'mlhs' : {'dualprocess' : self._process_mlti,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mlti,
                                  'inverseprocess' : self._inverseprocess_mlti,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'noise_transform' : 'categoric',
                                  'defaultparams' : {'dtype' : 'mlhs',
                                                     'flip_prob' : 0.03,
                                                     'testnoise' : False,
                                                     'norm_params' : {'flip_prob' : 0.03,
                                                                      'testnoise' : False,
                                                                      'weighted' : True},
                                                     'norm_category' : 'DPod'},
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'ordlexclude',
                                  'labelctgy' : 'mlhs'}})
    process_dict.update({'qbt1' : {'dualprocess' : None,
                                  'singleprocess' : self._process_qbt1,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_qbt1,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'boolexclude',
                                  'labelctgy' : 'qbt1'}})
    process_dict.update({'qbt2' : {'dualprocess' : None,
                                  'singleprocess' : self._process_qbt1,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_qbt1,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'defaultparams' : {'integer_bits' : 15,
                                                     'fractional_bits' : 0},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'boolexclude',
                                  'labelctgy' : 'qbt2'}})
    process_dict.update({'qbt3' : {'dualprocess' : None,
                                  'singleprocess' : self._process_qbt1,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_qbt1,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'zeroinfill',
                                  'defaultparams' : {'sign_bit' : False,
                                                     'fractional_bits' : 13},
                                  'NArowtype' : 'nonnegativenumeric',
                                  'MLinfilltype' : 'boolexclude',
                                  'labelctgy' : 'qbt3'}})
    process_dict.update({'qbt4' : {'dualprocess' : None,
                                  'singleprocess' : self._process_qbt1,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_qbt1,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'zeroinfill',
                                  'defaultparams' : {'sign_bit' : False,
                                                     'integer_bits' : 16,
                                                     'fractional_bits' : 0},
                                  'NArowtype' : 'nonnegativenumeric',
                                  'MLinfilltype' : 'boolexclude',
                                  'labelctgy' : 'qbt4'}})
    process_dict.update({'nmqb' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'qbt1'}})
    process_dict.update({'nmq2' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'qbt1'}})
    process_dict.update({'mmqb' : {'dualprocess' : self._process_mnmx,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mnmx,
                                  'inverseprocess' : self._inverseprocess_mnmx,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'qbt3'}})
    process_dict.update({'mmq2' : {'dualprocess' : self._process_mnmx,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mnmx,
                                  'inverseprocess' : self._inverseprocess_mnmx,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'qbt3'}})
    process_dict.update({'NArw' : {'dualprocess' : None,
                                  'singleprocess' : self._process_NArw,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'boolexclude',
                                  'labelctgy' : 'NArw'}})
    process_dict.update({'NAr2' : {'dualprocess' : None,
                                  'singleprocess' : self._process_NArw,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'boolexclude',
                                  'labelctgy' : 'NAr2'}})
    process_dict.update({'NAr3' : {'dualprocess' : None,
                                  'singleprocess' : self._process_NArw,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'positivenumeric',
                                  'MLinfilltype' : 'boolexclude',
                                  'labelctgy' : 'NAr3'}})
    process_dict.update({'NAr4' : {'dualprocess' : None,
                                  'singleprocess' : self._process_NArw,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'nonnegativenumeric',
                                  'MLinfilltype' : 'boolexclude',
                                  'labelctgy' : 'NAr4'}})
    process_dict.update({'NAr5' : {'dualprocess' : None,
                                  'singleprocess' : self._process_NArw,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'integer',
                                  'MLinfilltype' : 'boolexclude',
                                  'labelctgy' : 'NAr5'}})
    process_dict.update({'null' : {'dualprocess' : None,
                                  'singleprocess' : self._process_null,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'totalexclude',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : None}})
    process_dict.update({'copy' : {'dualprocess' : None,
                                  'singleprocess' : self._process_copy,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'NArowtype' : 'exclude',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'copy'}})
    process_dict.update({'excl' : {'dualprocess' : None,
                                  'singleprocess' : self._process_excl,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_excl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'NArowtype' : 'totalexclude',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'excl'}})
    process_dict.update({'exc2' : {'dualprocess' : self._process_exc2,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_exc2,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'exc2'}})
    process_dict.update({'exc3' : {'dualprocess' : self._process_exc2,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_exc2,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'exc3'}})
    process_dict.update({'exc4' : {'dualprocess' : self._process_exc2,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_exc2,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'exc4'}})
    process_dict.update({'exc5' : {'dualprocess' : self._process_exc5,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_exc5,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'defaultparams' : {'integertype' : 'singlct'},
                                  'NArowtype' : 'integer',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'exc5'}})
    process_dict.update({'exc6' : {'dualprocess' : self._process_exc2,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_exc2,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'exc6'}})
    process_dict.update({'exc7' : {'dualprocess' : self._process_exc5,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_exc5,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'defaultparams' : {'integertype' : 'singlct'},
                                  'NArowtype' : 'integer',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'exc7'}})
    process_dict.update({'exc8' : {'dualprocess' : self._process_exc5,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_exc5,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'defaultparams' : {'integertype' : 'integer'},
                                  'NArowtype' : 'integer',
                                  'MLinfilltype' : 'integer',
                                  'labelctgy' : 'exc8'}})
    process_dict.update({'exc9' : {'dualprocess' : self._process_exc5,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_exc5,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'defaultparams' : {'integertype' : 'integer'},
                                  'NArowtype' : 'integer',
                                  'MLinfilltype' : 'integer',
                                  'labelctgy' : 'exc9'}})
    process_dict.update({'shfl' : {'dualprocess' : None,
                                  'singleprocess' : self._process_shfl,
                                  'postprocess' : None,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'exclude',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'shfl'}})
    process_dict.update({'sint' : {'custom_train' : self._custom_train_trig,
                                  'custom_inversion' : self._custom_inversion_trig,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultparams' : {'operation' : 'sin'},
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'sint'}})
    process_dict.update({'cost' : {'custom_train' : self._custom_train_trig,
                                  'custom_inversion' : self._custom_inversion_trig,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultparams' : {'operation' : 'cos'},
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'cost'}})
    process_dict.update({'tant' : {'custom_train' : self._custom_train_trig,
                                  'custom_inversion' : self._custom_inversion_trig,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultparams' : {'operation' : 'tan'},
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'tant'}})
    process_dict.update({'arsn' : {'custom_train' : self._custom_train_trig,
                                  'custom_inversion' : self._custom_inversion_trig,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultparams' : {'operation' : 'arcsin'},
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'arsn'}})
    process_dict.update({'arcs' : {'custom_train' : self._custom_train_trig,
                                  'custom_inversion' : self._custom_inversion_trig,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultparams' : {'operation' : 'arccos'},
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'arcs'}})
    process_dict.update({'artn' : {'custom_train' : self._custom_train_trig,
                                  'custom_inversion' : self._custom_inversion_trig,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultparams' : {'operation' : 'arctan'},
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'artn'}})
    process_dict.update({'nmbd' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'nmbr'}})
    process_dict.update({'101d' : {'custom_train' : self._custom_train_1010,
                                  'custom_test' : self._custom_test_1010,
                                  'custom_inversion' : self._custom_inversion_1010,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : '1010',
                                  'labelctgy' : '101d'}})
    process_dict.update({'ordd' : {'custom_train' : self._custom_train_ordl,
                                  'custom_test' : self._custom_test_ordl,
                                  'custom_inversion' : self._custom_inversion_ordl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'ordd'}})
    process_dict.update({'texd' : {'custom_train' : self._custom_train_onht,
                                  'custom_test' : self._custom_test_onht,
                                  'custom_inversion' : self._custom_inversion_onht,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'texd'}})
    process_dict.update({'bnrd' : {'dualprocess' : self._process_binary,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_binary,
                                  'inverseprocess' : self._inverseprocess_bnry,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'binary',
                                  'MLinfilltype' : 'binary',
                                  'labelctgy' : 'bnrd'}})
    process_dict.update({'datd' : {'dualprocess' : None,
                                  'singleprocess' : None,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'mdsn'}})
    process_dict.update({'nuld' : {'dualprocess' : None,
                                  'singleprocess' : self._process_null,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'totalexclude',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'nuld'}})
    process_dict.update({'lbnm' : {'dualprocess' : self._process_exc2,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_exc2,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'adjinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'lbnm'}})
    process_dict.update({'lbnb' : {'dualprocess' : self._process_numerical,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_numerical,
                                  'inverseprocess' : self._inverseprocess_nmbr,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'lbnb'}})
    process_dict.update({'lb10' : {'custom_train' : self._custom_train_1010,
                                  'custom_test' : self._custom_test_1010,
                                  'custom_inversion' : self._custom_inversion_1010,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : '1010',
                                  'labelctgy' : 'lb10'}})
    process_dict.update({'lbor' : {'custom_train' : self._custom_train_ordl,
                                  'custom_test' : self._custom_test_ordl,
                                  'custom_inversion' : self._custom_inversion_ordl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'null_activation' : False},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'lbor'}})
    process_dict.update({'lbos' : {'custom_train' : self._custom_train_ordl,
                                  'custom_test' : self._custom_test_ordl,
                                  'custom_inversion' : self._custom_inversion_ordl,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'null_activation' : False},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'strg'}})
    process_dict.update({'lbte' : {'custom_train' : self._custom_train_onht,
                                  'custom_test' : self._custom_test_onht,
                                  'custom_inversion' : self._custom_inversion_onht,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'multirt',
                                  'labelctgy' : 'lbte'}})
    process_dict.update({'lbbn' : {'dualprocess' : self._process_binary,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_binary,
                                  'inverseprocess' : self._inverseprocess_bnry,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'invert' : True},
                                  'NArowtype' : 'binary',
                                  'MLinfilltype' : 'binary',
                                  'labelctgy' : 'lbbn'}})
    process_dict.update({'lbsm' : {'dualprocess' : self._process_smth,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_smth,
                                  'inverseprocess' : self._inverseprocess_smth,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'exclude',
                                  'labelctgy' : 'lbsm'}})
    process_dict.update({'lbfs' : {'dualprocess' : self._process_smth,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_smth,
                                  'inverseprocess' : self._inverseprocess_smth,
                                  'info_retention' : True,
                                  'inplace_option' : False,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'LSfit' : True},
                                  'NArowtype' : 'justNaN',
                                  'MLinfilltype' : 'exclude',
                                  'labelctgy' : 'lbfs'}})
    process_dict.update({'lbda' : {'dualprocess' : None,
                                  'singleprocess' : None,
                                  'postprocess' : None,
                                  'inplace_option' : False,
                                  'NArowtype' : 'datetime',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'mdsn'}})
    process_dict.update({'lgnr' : {'dualprocess' : self._process_absl,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_absl,
                                  'inverseprocess' : self._inverseprocess_absl,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'nonzeronumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'qbt5'}})
    process_dict.update({'lgn2' : {'dualprocess' : self._process_logn,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_logn,
                                  'inverseprocess' : self._inverseprocess_logn,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'NArowtype' : 'positivenumeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'qbt5'}})
    process_dict.update({'qbt5' : {'dualprocess' : None,
                                  'singleprocess' : self._process_qbt1,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_qbt1,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'negzeroinfill',
                                  'defaultparams' : {'integer_bits' : 4,
                                                     'fractional_bits' : 3,
                                                     'sign_bit' : True},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'boolexclude',
                                  'labelctgy' : 'qbt5'}})
    process_dict.update({'sgn3' : {'dualprocess' : None,
                                  'singleprocess' : self._process_copy,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'suffix' : ''},
                                  'NArowtype' : 'exclude',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'sgn2'}})
    process_dict.update({'sgn4' : {'dualprocess' : None,
                                  'singleprocess' : self._process_copy,
                                  'postprocess' : None,
                                  'inverseprocess' : self._inverseprocess_UPCS,
                                  'info_retention' : True,
                                  'inplace_option' : True,
                                  'defaultparams' : {'suffix' : ''},
                                  'NArowtype' : 'exclude',
                                  'MLinfilltype' : 'totalexclude',
                                  'labelctgy' : 'sgn2'}})
    process_dict.update({'sgn1' : {'dualprocess' : self._process_mltp,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_mltp,
                                  'inverseprocess' : self._inverseprocess_mltp,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'meaninfill',
                                  'defaultparams' : {'multiply' : -1},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'numeric',
                                  'labelctgy' : 'sgn2'}})
    process_dict.update({'sgn2' : {'dualprocess' : self._process_bkt3,
                                  'singleprocess' : None,
                                  'postprocess' : self._postprocess_bkt3,
                                  'inverseprocess' : self._inverseprocess_bkt3,
                                  'info_retention' : False,
                                  'inplace_option' : True,
                                  'defaultinfill' : 'naninfill',
                                  'defaultparams' : {'buckets' : [0]},
                                  'NArowtype' : 'numeric',
                                  'MLinfilltype' : 'singlct',
                                  'labelctgy' : 'sgn2'}})

    return process_dict

  #__FunctionBlock: automunge process family functions

  def __processfamily(self, df_train, df_test, column, origcategory, \
                    transform_dict, postprocess_dict, assign_param, logger):
    '''
    #as automunge runs a for loop through each column in automunge, this is the master 
    #processing function applied which runs through the different family primitives
    #populated in the transform_dict by assembletransformdict
    #processfamily is only called with column as an inputcolumn and origcategory as a root category
    #downstream generations are applied within processparent
    
    #we will run in order of
    #siblings, cousins, parents, auntsuncles
    '''

    inplaceperformed = False
    
    #final upstream transform from parents or auntsuncles is elligible for inplace
    #when a replacement transform is to be applied
    final_upstream = False
    if len(transform_dict[origcategory]['auntsuncles']) == 0:
      if len(transform_dict[origcategory]['parents']) > 0:
        final_upstream = transform_dict[origcategory]['parents'][-1]
    else:
      if len(transform_dict[origcategory]['auntsuncles']) > 0:
        final_upstream = transform_dict[origcategory]['auntsuncles'][-1]

    #process the siblings (with downstream, supplemental)
    for sibling in transform_dict[origcategory]['siblings']:

      if sibling != None:
        #note we use the processparent function here
        df_train, df_test, postprocess_dict, inplaceperformed = \
        self.__processparent(df_train, df_test, column, sibling, origcategory, final_upstream, \
                          transform_dict, postprocess_dict, assign_param, logger)
    
    #process the cousins (no downstream, supplemental)
    for cousin in transform_dict[origcategory]['cousins']:
      
      #this if statement kind of a placeholder such as for validation of primitive entry
      if cousin != None:

        #note we use the processcousin function here
        df_train, df_test, postprocess_dict, inplaceperformed = \
        self.__processcousin(df_train, df_test, column, cousin, origcategory, final_upstream, \
                            transform_dict, postprocess_dict, assign_param, logger)

    #process the parents (with downstream, with replacement)
    for parent in transform_dict[origcategory]['parents']:

      if parent != None:

        df_train, df_test, postprocess_dict, inplaceperformed = \
        self.__processparent(df_train, df_test, column, parent, origcategory, final_upstream, \
                          transform_dict, postprocess_dict, assign_param, logger)
        
    #process the auntsuncles (no downstream, with replacement)
    for auntuncle in transform_dict[origcategory]['auntsuncles']:

      if auntuncle != None:

        #note we use the processcousin function here
        df_train, df_test, postprocess_dict, inplaceperformed = \
        self.__processcousin(df_train, df_test, column, auntuncle, origcategory, final_upstream, \
                            transform_dict, postprocess_dict, assign_param, logger)

    #if a replacement primitive was applied and inplace wasn't performed register the column to orig_noinplace
    if len(transform_dict[origcategory]['auntsuncles']) \
    + len(transform_dict[origcategory]['parents']) > 0 \
    and inplaceperformed is False:
      if column not in postprocess_dict['orig_noinplace']:
        postprocess_dict['orig_noinplace'] = postprocess_dict['orig_noinplace'] | {column}

    return df_train, df_test, postprocess_dict

  def __circleoflife(self, df_train, df_test, column, category, \
                    transform_dict, postprocess_dict, templist1):
    '''
    #This function deletes source column for cases where family primitives 
    #included replacement, with maintenance of the associated data structures.
    
    #templist1 is the list of df_train columns before processfamily
    '''

    #if we had replacement transformations performed on first generation \
    #then delete the original column
    if len(transform_dict[category]['auntsuncles']) \
    + len(transform_dict[category]['parents']) > 0:
      
      if column in postprocess_dict['orig_noinplace']:
        del df_train[column]
        del df_test[column]

    #if we had replacement transformations performed on downstream generation \
    #then delete the associated parent column 
    
    newcolumns = set(df_train) - set(templist1)
    
    #this one is for columns replaced as part of inplace operation
    if len(newcolumns) > 0:
      anewcolumn = list(newcolumns)[0]
      temp_columnslist = self.__autocopy(postprocess_dict['column_dict'][anewcolumn]['columnslist'])
      for newcolumn in temp_columnslist:
        if postprocess_dict['column_dict'][newcolumn]['deletecolumn'] == 'inplace':
          for newcolumn2 in temp_columnslist:
            if newcolumn in postprocess_dict['column_dict'][newcolumn2]['columnslist']:        
              postprocess_dict['column_dict'][newcolumn2]['columnslist'].remove(newcolumn)
    
    #this one is for columns we manually delete
    for newcolumn in newcolumns:
      if postprocess_dict['column_dict'][newcolumn]['deletecolumn'] is True:
        for newcolumn2 in newcolumns:
          if newcolumn in postprocess_dict['column_dict'][newcolumn2]['columnslist']:
            postprocess_dict['column_dict'][newcolumn2]['columnslist'].remove(newcolumn)
          
        #now we'll delete column
        #note this only worksa on single column  parents, need to incioroprate categorylist
        #for multicolumn parents (future extension)
        if newcolumn in df_train.columns:
          del df_train[newcolumn]
          del df_test[newcolumn]

    return df_train, df_test, postprocess_dict

  def __dictupdate(self, column, column_dict, postprocess_dict):
    '''
    #dictupdate function takes as input column_dict, postprocess_dict, then for cases
    #where origcolumn is the same for the two combines the columnslist and the 
    #normalization_dict, then appends the column_dict onto the postprocess_dict
    #returns the column_dict and postprocess_dict. Note that the passed column name
    #"column" is the column name prior to the applicaiton of processing, and the
    #name of the column after the last processing function is saved as a key
    #in the column_dict
    '''

    #(reason for "key2" instead of key1 is some shuffling during editing)
    for key2 in column_dict:

      #first address carry-though of origcolumn and origcategory from parent to child
      if column in postprocess_dict['column_dict']:

        #if column is not origcolumn in postprocess_dict
        if postprocess_dict['column_dict'][column]['origcolumn'] \
        != column:

          #assign origcolumn from postprocess_dict to column_dict
          column_dict[key2]['origcolumn'] = \
          postprocess_dict['column_dict'][column]['origcolumn']

          #assign origcategory from postprocess_dict to column_dict
          column_dict[key2]['origcategory'] = \
          postprocess_dict['column_dict'][column]['origcategory']

      for key1 in postprocess_dict['column_dict']:

        #if origcolumn is the same between column_dict saved in postprocess_dict and
        #the column_dict outputed from our processing, we'll combine a few values
        if postprocess_dict['column_dict'][key1]['origcolumn'] == column_dict[key2]['origcolumn']:
          #first we'll combine the columnslist capturing all columns 
          #originating from same origcolumn for these two sets
          postprocess_dict['column_dict'][key1]['columnslist'] = \
          list(set(postprocess_dict['column_dict'][key1]['columnslist'])|set(column_dict[key2]['columnslist']))
          #apply that value to the column_dict columnslist as well
          column_dict[key2]['columnslist'] = postprocess_dict['column_dict'][key1]['columnslist']

    #now append column_dict onto postprocess_dict
    postprocess_dict['column_dict'].update(column_dict)

    #return column_dict, postprocess_dict
    return postprocess_dict

  def __populate_columnkey_dict(self, column_dict_list, postprocess_dict, transformationcategory):
    """
    #columnkey_dict is used in postprocess functions
    #to derive a normkey when returned column isn't known or may return emtpy set
    #
    #populates in the form:
    #columnkey_dict.update({inputcolumn : {transformationcategory : categorylist}})
    #where categorylist_aggregate is a combination of all categorylists 
    #derived from some functions with recorded category applied to a given input column
    #
    #Note that convention is if the same transformation is applied multiple times in same family tree
    #each application will have a unique input column, otherwise transfomation will be via a unique process_dict transformationcategory
    #  
    #Here column_dict_list is the list of dictionaries returned from a single processing function
    """

    if len(column_dict_list) > 0:

      inputcolumn = column_dict_list[0][list(column_dict_list[0])[0]]['inputcolumn']
      if inputcolumn not in postprocess_dict['columnkey_dict']:
        postprocess_dict['columnkey_dict'].update({inputcolumn : {}})
        
      categorylist = column_dict_list[0][list(column_dict_list[0])[0]]['categorylist']

      if transformationcategory not in postprocess_dict['columnkey_dict'][inputcolumn]:
        postprocess_dict['columnkey_dict'][inputcolumn].update({transformationcategory : categorylist})
      
    return postprocess_dict
  
  def __processcousin(self, df_train, df_test, column, cousin, origcategory, final_upstream, \
                     transform_dict, postprocess_dict, assign_param, logger):
    '''
    #cousin is one of the primitives for processfamily function, and it involves
    #transformations without downstream derivations without replacement of source column
    #although this same funciton can be used with the auntsuncles primitive
    #by following with a deletion of original column, also this funciton can be
    #used on the niecesnephews primitive downstream of parents or siblings since 
    #they don't have children (they're way to young for that)
    #note the processing funcitons are accessed through the process_dict
    #note that validation of processing functions in user passed processdict takes place in _check_processdict4
    
    #note that the presence of a custom_train processing function for a tree category
    #takes precedence over a dualprocess function
    #which takes precedence over a singleprocess function

    #reminder the format of assign_param is e.g.
    #assignparam = {'splt' : {'column1' : {'minsplit' : 4}}, \
    #               'spl2' : {'column2' : {'minsplit' : 3}}}

    '''

    process_dict = postprocess_dict['process_dict']
    
    inplaceperformed = False
    inplacecandidate = False
    if final_upstream == cousin:
      inplacecandidate = True

    params = self.__grab_params(assign_param, cousin, column, process_dict[cousin], postprocess_dict)

    #now populate params entries associated with entropy seeding for noise transforms
    params, postprocess_dict = \
    self.__random_parameters_params_append(params, postprocess_dict, \
                                      column, cousin, 'traintest', \
                                      df_train.shape[0], df_test.shape[0], \
                                      postprocess_dict['printstatus'], df_train, df_test)
    
    if inplacecandidate is True:
      if 'inplace_option' not in process_dict[cousin] \
      or 'inplace_option' in process_dict[cousin] \
      and process_dict[cousin]['inplace_option'] is True:
        if 'inplace' not in params:
          inplaceperformed = True
          params.update({'inplace' : True})
        elif ('inplace' in params and params['inplace'] != False):
          inplaceperformed = True
          params.update({'inplace' : True})
        else:
          inplaceperformed = False
    else:
      #user cannot manually specify inplace as True by design
      if ('inplace' in params and params['inplace'] is True):
        inplaceperformed = False
        params.update({'inplace' : False})
    
    #if this is a custom process function
    #(convention is that 'custom_train' is populated in both scenarios for dualprocess or singleprocess)
    if 'custom_train' in process_dict[cousin] \
    and process_dict[cousin]['custom_train'] != None:
      
      #note that _custom_process_wrapper accesses the copy of process_dict saved within postprocess_dict
      df_train, df_test, column_dict_list = \
      self.__custom_process_wrapper(df_train, df_test, column, origcategory, \
                                   cousin, postprocess_dict, params)
      
      #columnkey_dict used in postprocess functions in a few cases to derive a normkey
      postprocess_dict = self.__populate_columnkey_dict(column_dict_list, postprocess_dict, cousin)
    
    #elif this is a dual process function
    elif 'dualprocess' in process_dict[cousin] \
    and process_dict[cousin]['dualprocess'] != None:

      df_train, df_test, column_dict_list = \
      process_dict[cousin]['dualprocess'](df_train, df_test, column, origcategory, \
                                          cousin, postprocess_dict, params)
      
      #columnkey_dict used in postprocess functions in a few cases to derive a normkey
      postprocess_dict = self.__populate_columnkey_dict(column_dict_list, postprocess_dict, cousin)

    #else if this is a single process function process train and test seperately
    elif 'singleprocess' in process_dict[cousin] \
    and process_dict[cousin]['singleprocess'] != None:

      df_train, column_dict_list =  \
      process_dict[cousin]['singleprocess'](df_train, column, origcategory, \
                                            cousin, postprocess_dict, params)

      df_test, _1 = \
      process_dict[cousin]['singleprocess'](df_test, column, origcategory, \
                                            cousin, postprocess_dict, params)
      
      #columnkey_dict used in postprocess functions in a few cases to derive a normkey
      postprocess_dict = self.__populate_columnkey_dict(column_dict_list, postprocess_dict, cousin)

    else:

      #_____the following 40 rows is a validation result aggregation for cases where processdict was speciifed without a callable trasnformation function
      #since we populate results with unique integer identifier performing here a little dictionary munging
      # if postprocess_dict['printstatus'] != 'silent':
        
      self.__autoprint('warning', 'Please note that a tree category was accessed as category ', cousin)
      self.__autoprint('warning', 'Originating from generations of root category ', origcategory)
      self.__autoprint('warning', 'Which did not have processing functions populated in processdict')
      self.__autoprint('warning', 'Processing functions in custom_train or dual/single/post process convention ')
      self.__autoprint('warning', 'Are required when a category is accessed as a tree category entry in a family tree.')
      self.__autoprint('warning', 'Otherwise tree category is treated as None and no downstream generations are accessed.')
      self.__autoprint('warning', '')

      column_dict_list = []

      #populate validation result
      #we'll have convention of populating an entry each time this validation result identified
      #with a key of an integer iterated with each occurance
      #and populated with details of treecategory without processing functions and associated root category 
      # treecategory_with_empty_processing_functions_valresult = {i : {'treecategory' : treecategory,
      #                                                                'rootcategory' : rootcategory}}

      if 'treecategory_with_empty_processing_functions_valresult' not in logger_dict['validations']:
        logger_dict['validations'].update({'treecategory_with_empty_processing_functions_valresult' : {}})

      #validation results reported with key of an integer incremented with each occurance
      #get value of current max entry 
      #(here using pandas as a hack since python doesn't have clean max item in list, works bacause are numeric)
      max_valkey_i = pd.DataFrame({'temp' : list(logger_dict['validations']['treecategory_with_empty_processing_functions_valresult'])}).max()

      #this returns max_valkey_i as a series with one integer entry, with entry as NaN when no entries present
      if (max_valkey_i != max_valkey_i).iat[0]:
        max_valkey_i = -1
      else:
        max_valkey_i = int(max_valkey_i)

      #initialize our valitation result reporting, which notes the tree category without processing functions
      #as well as the root category in whose generations this tree category was found
      treecategory_with_empty_processing_functions_valresult = {max_valkey_i+1 : {'treecategory' : cousin,
                                                                                  'rootcategory' : origcategory}}

      #populate the validation results in ['treecategory_with_empty_processing_functions_valresult']
      logger_dict['validations']['treecategory_with_empty_processing_functions_valresult'].update(
        treecategory_with_empty_processing_functions_valresult
        )
      logger = self.__logger_append(logger)
      #_____

    #update the columnslist and normalization_dict for both column_dict and postprocess_dict
    for column_dict in column_dict_list:
      postprocess_dict = self.__dictupdate(column, column_dict, postprocess_dict)

    return df_train, df_test, postprocess_dict, inplaceperformed

  def __processparent(self, df_train, df_test, column, parent, origcategory, final_upstream, \
                    transform_dict, postprocess_dict, assign_param, logger):
    '''
    #parent is one of the primitives for processfamily function, and it involves
    #transformations with downstream derivations with replacement of source column
    #although this same funciton can be used with the siblinga primitive
    #by not following with a deletion of original column, also this funciton can be
    #used on the children primitive downstream of parents or siblings, allowing
    #the children to have children of their own, you know, grandchildren and stuff.
    #note the processing functions are accessed through the process_dict
    #note that validation of processing functions in user passed processdict takes place in _check_processdict4
    
    #note that the presence of a custom_train processing function for a tree category
    #takes precedence over a dualprocess function
    #which takes precedence over a singleprocess function
    
    #reminder the format of assign_param is e.g.
    #assignparam = {'splt' : {'column1' : {'minsplit' : 4}}, \
    #               'spl2' : {'column2' : {'minsplit' : 3}}}
    
    #we want to apply in order of
    #upstream process, niecesnephews, friends, children, coworkers
    '''

    process_dict = postprocess_dict['process_dict']

    #upstream process
    
    inplaceperformed = False
    inplacecandidate = False
    if final_upstream == parent:
      inplacecandidate = True
    
    params = self.__grab_params(assign_param, parent, column, process_dict[parent], postprocess_dict)

    #now populate params entries associated with entropy seeding for noise transforms
    params, postprocess_dict = \
    self.__random_parameters_params_append(params, postprocess_dict, \
                                          column, parent, 'traintest', \
                                          df_train.shape[0], df_test.shape[0], \
                                          postprocess_dict['printstatus'], df_train, df_test)
    
    if inplacecandidate is True:
      if 'inplace_option' not in process_dict[parent] \
      or 'inplace_option' in process_dict[parent] \
      and process_dict[parent]['inplace_option'] is True:
        if 'inplace' not in params:
          inplaceperformed = True
          params.update({'inplace' : True})
        elif ('inplace' in params and params['inplace'] != False):
          inplaceperformed = True
          params.update({'inplace' : True})
        else:
          inplaceperformed = False
    else:
      #user cannot manually specify inplace as True by design
      if ('inplace' in params and params['inplace'] is True):
        inplaceperformed = False
        params.update({'inplace' : False})
    
    #if this is a custom process function 
    #(convention is that 'custom_train' is populated in both scenarios for dualprocess or singleprocess)
    if 'custom_train' in process_dict[parent] \
    and process_dict[parent]['custom_train'] != None:
      
      #note that _custom_process_wrapper accesses the copy of process_dict saved within postprocess_dict
      df_train, df_test, column_dict_list = \
      self.__custom_process_wrapper(df_train, df_test, column, origcategory, \
                                   parent, postprocess_dict, params)
      
      #columnkey_dict used in postprocess functions in a few cases to derive a normkey
      postprocess_dict = self.__populate_columnkey_dict(column_dict_list, postprocess_dict, parent)
    
    #elif this is a dual process function
    elif 'dualprocess' in process_dict[parent] \
    and process_dict[parent]['dualprocess'] != None:

      df_train, df_test, column_dict_list = \
      process_dict[parent]['dualprocess'](df_train, df_test, column, origcategory, \
                                          parent, postprocess_dict, params)
      
      #columnkey_dict used in postprocess functions in a few cases to derive a normkey
      postprocess_dict = self.__populate_columnkey_dict(column_dict_list, postprocess_dict, parent)

    #else if this is a single process function process train and test seperately
    elif 'singleprocess' in process_dict[parent] \
    and process_dict[parent]['singleprocess'] != None:

      df_train, column_dict_list =  \
      process_dict[parent]['singleprocess'](df_train, column, origcategory, \
                                          parent, postprocess_dict, params)

      df_test, _1 = \
      process_dict[parent]['singleprocess'](df_test, column, origcategory, \
                                          parent, postprocess_dict, params)
      
      #columnkey_dict used in postprocess functions in a few cases to derive a normkey
      postprocess_dict = self.__populate_columnkey_dict(column_dict_list, postprocess_dict, parent)

    else:

      #_____aggregating validaiton results for non callable functions in processdict entry
      # if postprocess_dict['printstatus'] != 'silent':

      self.__autoprint('warning', 'Please note that a tree category was accessed as category ', parent)
      self.__autoprint('warning', 'Originating from generations of root category ', origcategory)
      self.__autoprint('warning', 'Which did not have processing functions populated in processdict')
      self.__autoprint('warning', 'Processing functions in custom_train or dual/single/post process convention ')
      self.__autoprint('warning', 'Are required when a category is accessed as a tree category entry in a family tree.')
      self.__autoprint('warning', 'Otherwise tree category is treated as None and no downstream generations are accessed.')
      self.__autoprint('warning', '')

      column_dict_list = []

      #populate validation result
      #we'll have convention of populating an entry each time this validation result identified
      #with a key of an integer iterated with each occurance
      #and populated with details of treecategory without processing functions and associated root category 
      # treecategory_with_empty_processing_functions_valresult = {i : {'treecategory' : treecategory,
      #                                                                'rootcategory' : rootcategory}}

      if 'treecategory_with_empty_processing_functions_valresult' not in logger_dict['validations']:
        logger_dict['validations'].update({'treecategory_with_empty_processing_functions_valresult' : {}})

      #validation results reported with key of an integer incremented with each occurance
      #get value of current max entry
      #(here using pandas as a hack since python doesn't have clean max item in list, works bacause are numeric)
      max_valkey_i = pd.DataFrame({'temp' : list(logger_dict['validations']['treecategory_with_empty_processing_functions_valresult'])}).max()

      #this returns max_valkey_i as a series with one integer entry, with entry as NaN when no entries present
      if (max_valkey_i != max_valkey_i).iat[0]:
        max_valkey_i = -1
      else:
        max_valkey_i = int(max_valkey_i)

      #initialize our valitation result reporting, which notes the tree category without processing functions
      #as well as the root category in whose generations this tree category was found
      treecategory_with_empty_processing_functions_valresult = {max_valkey_i+1 : {'treecategory' : parent,
                                                                                  'rootcategory' : origcategory}}

      #populate the validation results in ['treecategory_with_empty_processing_functions_valresult']
      logger_dict['validations']['treecategory_with_empty_processing_functions_valresult'].update(
        treecategory_with_empty_processing_functions_valresult
        )
      logger = self.__logger_append(logger)
      #_____

    #update the columnslist and normalization_dict for both column_dict and postprocess_dict
    for column_dict in column_dict_list:
      postprocess_dict = self.__dictupdate(column, column_dict, postprocess_dict)

    #we take an arbitrary column returned from upstream transform as first entry in categorylist
    #to serve as parentcolumn input to next generation transform
    #if downstream transform includes support for multi-column input
    #they can use parentcolumn to access from column_dict the upstream categorylist and normalization_dict
    if len(column_dict_list) > 0:
      column_dict = column_dict_list[0]
      support_column = list(column_dict.keys())[0]
      support_categorylist = column_dict[support_column]['categorylist']
      parentcolumn = support_categorylist[0]
      
      #only proceed to next generation if parent did not return an empty set

      #if transform_dict[parent] != None:

      #initialize in case no downstream performed
      parent_inplaceperformed = False

      #process any children

      #final downstream transform from coworkers or children is elligible for inplace
      #when a replacement transform is to be applied
      final_downstream = False
      if len(transform_dict[parent]['coworkers']) == 0:
        if len(transform_dict[parent]['children']) > 0:
          final_downstream = transform_dict[parent]['children'][-1]
      else:
        if len(transform_dict[parent]['coworkers']) > 0:
          final_downstream = transform_dict[parent]['coworkers'][-1]

      #process any niecesnephews
      #note the function applied is comparable to processsibling, just a different
      #parent column
      for niecenephew in transform_dict[parent]['niecesnephews']:

        if niecenephew != None:

          #process the niecenephew
          #note the function applied is processparent (using recursion)
          #parent column
          df_train, df_test, postprocess_dict, parent_inplaceperformed = \
          self.__processparent(df_train, df_test, parentcolumn, niecenephew, origcategory, final_downstream, \
                             transform_dict, postprocess_dict, assign_param, logger)

      #process any friends
      for friend in transform_dict[parent]['friends']:

        if friend != None:

          #process the friend
          #note the function applied is processcousin
          df_train, df_test, postprocess_dict, parent_inplaceperformed = \
          self.__processcousin(df_train, df_test, parentcolumn, friend, origcategory, final_downstream, \
                             transform_dict, postprocess_dict, assign_param, logger)

      for child in transform_dict[parent]['children']:

        if child != None:

          #process the child
          #note the function applied is processparent (using recursion)
          #parent column
          df_train, df_test, postprocess_dict, parent_inplaceperformed = \
          self.__processparent(df_train, df_test, parentcolumn, child, origcategory, final_downstream, \
                             transform_dict, postprocess_dict, assign_param, logger)

      #process any coworkers
      for coworker in transform_dict[parent]['coworkers']:

        if coworker != None:

          #process the coworker
          #note the function applied is processcousin
          df_train, df_test, postprocess_dict, parent_inplaceperformed = \
          self.__processcousin(df_train, df_test, parentcolumn, coworker, origcategory, final_downstream, \
                             transform_dict, postprocess_dict, assign_param, logger)

      #if we had replacement transformations performed then mark column for deletion
      if len(transform_dict[parent]['children']) \
      + len(transform_dict[parent]['coworkers']) > 0 \
      and parent_inplaceperformed is False:
        #here we'll only address downstream generaitons
        if parentcolumn in postprocess_dict['column_dict']:
          for parentcolumn_categorylist_entry in postprocess_dict['column_dict'][parentcolumn]['categorylist']:
            postprocess_dict['column_dict'][parentcolumn_categorylist_entry]['deletecolumn'] = True
        else:
          if parentcolumn not in postprocess_dict['orig_noinplace']:
            postprocess_dict['orig_noinplace'] = postprocess_dict['orig_noinplace'] | {parentcolumn}
      elif parent_inplaceperformed is True:
        if parentcolumn in postprocess_dict['column_dict']:
          for parentcolumn_categorylist_entry in postprocess_dict['column_dict'][parentcolumn]['categorylist']:
            postprocess_dict['column_dict'][parentcolumn_categorylist_entry]['deletecolumn'] = 'inplace'

    return df_train, df_test, postprocess_dict, inplaceperformed

  def __grab_params(self, assign_param, category, column, processdict_entry, postprocess_dict):
    """
    #In order of precendence, parameters assigned to distinct 
    #category/column configurations take precedence 
    #to default_assignparam assigned to categories which take precendence 
    #to global_assignparam assigned to all transformations which take precendence 
    #to parameters set as defaultparams in processdict definition.
    """

    params = {}
    
    if 'defaultparams' in processdict_entry:
      
      #key are parameters
      for key in processdict_entry['defaultparams']:
        
        params.update({key : processdict_entry['defaultparams'][key]})

    #if assign_param is not empty
    if bool(assign_param):
      
      if 'global_assignparam' in assign_param:
        
        #key are parameters
        for key in assign_param['global_assignparam']:
          
          params.update({key : assign_param['global_assignparam'][key]})
          
      if 'default_assignparam' in assign_param:
        
        if category in assign_param['default_assignparam']:
            
          #key are parameters
          for key in assign_param['default_assignparam'][category]:

            params.update({key : assign_param['default_assignparam'][category][key]})
                
      if category in assign_param:
        
        #distinct category/column configurations can either be assigned
        #using the source column or the derived column serving as input to the transform
        #in case both are present the dervied column specifciation takes precedence
          
        #derived column specification with suffix takes precendence over input column specification
        if column in assign_param[category]:

          #key are parameters
          for key in assign_param[category][column]:

            params.update({key : assign_param[category][column][key]})

        #we won't use a specified source column entry if the derived column was already specfied
        elif column in postprocess_dict['column_dict'] \
        and postprocess_dict['column_dict'][column]['origcolumn'] in assign_param[category]:

          #key are parameters
          for key in assign_param[category][postprocess_dict['column_dict'][column]['origcolumn']]:

            params.update({key : assign_param[category][postprocess_dict['column_dict'][column]['origcolumn']][key]})
            
      #messy_data_headers refers to input features in multi column encodings
      if 'messy_data_headers' in assign_param:
        
        #we'll know it is an inputcolumn if not in column_dict
        if column in assign_param['messy_data_headers'] \
        and column not in postprocess_dict['column_dict']:

          #messy convention doesn't have inplace support
          params.update({'inplace' : False})
          params.update({'messy_data_headers' : assign_param['messy_data_headers'][column]})
        
    return params

  def __custom_process_wrapper(self, mdf_train, mdf_test, column, category, \
                              treecategory, postprocess_dict, params = {}):
    """
    A wrapper for custom transformation functions applied in automunge
    Where custom transformations follow templates of custom_train_template 
    and custom_test_template
    
    The purpose of this wrapper is to extend the minimized versions of the 
    custom transform templates
    To include other conventions of the library
    Such as default infill, populating data structures, inplace option, 
    suffix overlap detection, and etc
    
    The form of the _custom_process_wrapper inputs/outputs 
    will be similar to what is applied for dualprocess functions
    
    When a processdict has entries for both custom_train_template 
    and custom_test_template
    custom_train_template will be applied to mdf_train 
    and custom_test_template will be applied to mdf_test
    
    Otherwise if the processdict only has an entry for custom_train_template
    custom_train_template will be applied to both mdf_train and mdf_test
    And the normalization_dict populated based on mdf_train will be the saved version
    
    Receives dataframes of a train and test set as mdf_train and mdf_test
    column is the received column that will serve as target for the transformation
    category is the root category for original upstream primitive entries
    treecategory is the category entry to a family tree primitive that 
    was used to access this transformation
    postprocess_dict is the dictionary data structure passed between transforms
    (Note that at this point postprocess_dict will include an entry for the 
    process_dict including consolidated processdict entries)
    params are the parameters passed through assignparam associated with 
    this specific categoy and column
    
    Returns the resulting transformed dataframes mdf_train and mdf_test
    and a populated column_dict_list data structure
    Where column_dict_list has a column_dict entry for each column returned from transform
    And where general convention in library is that the same normalization_dict 
    is saved in each column_dict populated in a transform*
    (*as an asterisk, in a few cases in library the normalization_dict may differ between column_dicts 
    populated in a transform returning a multi column set due to returned column specific drift stats)
    
    Note that this wrapper includes application of a default infill per any processdict entry for 'defaultinfill'
    Or otherwise performs adjinfill as default
    As well as other infill conventions determined based on the NArowtype associated with the treecategory
    
    We'll also create seperately a similar wrapper for postprocess functions applied 
    in postmunge (_custom_postprocess_wrapper)
    And likewise a wrapper for custom inversions that may be performed 
    in postmunge (_custom_inverseprocess_wrapper)
    """

    suffixoverlap_results = {}
    custom_process_wrapper_dict = {}
    
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False

    suffix = treecategory
      
    #note this suffixcolumn convention won't work with the two suffix edge cases of text and excl transform
    #which is fine since this is only for transforms passed through custom_train / custom_test
    suffixcolumn = column + '_' + suffix
    
    #grab a list of current columns for a suffix overlap check performed after applying transforms
    initial_columns = list(mdf_train)
    
    if inplace is not True:
      
      #copy source column into new column
      mdf_train, suffixoverlap_results = \
      self.__df_copy_train(mdf_train, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])

      mdf_test[suffixcolumn] = mdf_test[column].copy()
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_train.rename(columns = {column : suffixcolumn}, inplace = True)
      mdf_test.rename(columns = {column : suffixcolumn}, inplace = True)
      
    #___
      
    #now that we've created a new column, we'll apply a default infill
    #contingent on the 'NArowtype' of the tree category serving as basis for this transform
    NArowtype = postprocess_dict['process_dict'][treecategory]['NArowtype']
    MLinfilltype = postprocess_dict['process_dict'][treecategory]['MLinfilltype']
    
    #first to apply default infill we'll convert any nonvalid entries to nan
    
    if NArowtype in {'numeric'}:
      
      #convert all values to either numeric or NaN
      mdf_train[suffixcolumn] = pd.to_numeric(mdf_train[suffixcolumn], errors='coerce')
      mdf_test[suffixcolumn] = pd.to_numeric(mdf_test[suffixcolumn], errors='coerce')
      
    elif NArowtype in {'integer'}:
      
      #convert all values to either numeric or NaN
      mdf_train[suffixcolumn] = pd.to_numeric(mdf_train[suffixcolumn], errors='coerce')
      mdf_test[suffixcolumn] = pd.to_numeric(mdf_test[suffixcolumn], errors='coerce')
      
      #non integers are subject to infill
      mdf_train = self.__autowhere(mdf_train, suffixcolumn, mdf_train[suffixcolumn] == mdf_train[suffixcolumn].round(), alternative=np.nan, specified='alternative')
      mdf_test = self.__autowhere(mdf_test, suffixcolumn, mdf_test[suffixcolumn] == mdf_test[suffixcolumn].round(), alternative=np.nan, specified='alternative')
    
    elif NArowtype in {'positivenumeric'}:
      
      #convert all values to either numeric or NaN
      mdf_train[suffixcolumn] = pd.to_numeric(mdf_train[suffixcolumn], errors='coerce')
      mdf_test[suffixcolumn] = pd.to_numeric(mdf_test[suffixcolumn], errors='coerce')
      
      #values <= 0 subject to infill
      mdf_train.loc[mdf_train[suffixcolumn] <= 0, (suffixcolumn)] = np.nan
      mdf_test.loc[mdf_test[suffixcolumn] <= 0, (suffixcolumn)] = np.nan
      
    elif NArowtype in {'nonnegativenumeric'}:
      
      #convert all values to either numeric or NaN
      mdf_train[suffixcolumn] = pd.to_numeric(mdf_train[suffixcolumn], errors='coerce')
      mdf_test[suffixcolumn] = pd.to_numeric(mdf_test[suffixcolumn], errors='coerce')
      
      #values < 0 subject to infill
      mdf_train.loc[mdf_train[suffixcolumn] < 0, (suffixcolumn)] = np.nan
      mdf_test.loc[mdf_test[suffixcolumn] < 0, (suffixcolumn)] = np.nan
      
    elif NArowtype in {'nonzeronumeric'}:
      
      #convert all values to either numeric or NaN
      mdf_train[suffixcolumn] = pd.to_numeric(mdf_train[suffixcolumn], errors='coerce')
      mdf_test[suffixcolumn] = pd.to_numeric(mdf_test[suffixcolumn], errors='coerce')
      
      #values == 0 subject to infill
      mdf_train.loc[mdf_train[suffixcolumn] == 0, (suffixcolumn)] = np.nan
      mdf_test.loc[mdf_test[suffixcolumn] == 0, (suffixcolumn)] = np.nan
      
    elif NArowtype in {'datetime'}:
      
      #convert values to datetime
      mdf_train[suffixcolumn] = pd.to_datetime(mdf_train[suffixcolumn], errors = 'coerce', utc=True)
      mdf_test[suffixcolumn] = pd.to_datetime(mdf_test[suffixcolumn], errors = 'coerce', utc=True)
      
    elif NArowtype in {'justNaN', 'binary', 'exclude', 'totalexclude', 'parsenumeric'}:
      
      #nonvalid entries are already nan
      pass
    
    #___
    
    #Now that all nonvalid entries are cast as nan, we'll perform our infill
    #which will default to adjacent cell infill unless otherwise specified in processdict
    
    mdf_train, defaultinfill_dict = \
    self.__apply_defaultinfill(mdf_train, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    mdf_test, _1 = \
    self.__apply_defaultinfill(mdf_test, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)
    
    custom_process_wrapper_dict.update({'defaultinfill_dict' : defaultinfill_dict})
    
    #___
      
    #great now we can apply the custom_train_template to training data
    #custom_train_template will be saved as entry for 'custom_train'
    mdf_train, normalization_dict = \
    postprocess_dict['process_dict'][treecategory]['custom_train'](mdf_train, suffixcolumn, params)
    
    returned_columns = list(mdf_train)
    
    #newcolumns_list is list of returned columns including suffixcolumn
    #may be empty when suffixcolumn not returned
    #this might be a different order of columns vs the dataframe, we'll then recover the order
    #the ^ operation is to accomodate both inplace scenarios
    newcolumns_set = (set(initial_columns) ^ set(returned_columns)) - set(initial_columns)
    newcolumns_list = []
    #this is to recover the order of columns as found in returned_columns
    for returned_column in returned_columns:
      if returned_column in newcolumns_set:
        newcolumns_list.append(returned_column)

    #to be consistent with postmunge, if newcolumns_list is empty then we'll just delete the suffix column in mdf_test
    if len(newcolumns_list) == 0:
      del mdf_test[suffixcolumn]
    else:
      #We'll have convention that if a corresponding custom_test_template wasn't populated
      #custom_test entry will either not be included or cast as None, in which case we apply custom_train to test data
      if 'custom_test' in postprocess_dict['process_dict'][treecategory] \
      and postprocess_dict['process_dict'][treecategory]['custom_test'] != None:
        mdf_test = \
        postprocess_dict['process_dict'][treecategory]['custom_test'](mdf_test, suffixcolumn, normalization_dict)
      else:
        mdf_test, _1 = \
        postprocess_dict['process_dict'][treecategory]['custom_train'](mdf_test, suffixcolumn, params)
      
    #___
    
    #Now run some suffix overlap tests
    
    #If any temporary columns were created but not returned
    #the convention is user should log in a normalization_dict entry under 'tempcolumns' for suffix overlap detection
    if 'tempcolumns' in normalization_dict:
      tempcolumns = normalization_dict['tempcolumns']
      if not isinstance(tempcolumns, list):
        tempcolumns = [tempcolumns]
        
      suffixoverlap_results = self.__df_check_suffixoverlap(initial_columns, \
                                                           tempcolumns, \
                                                           suffixoverlap_results = suffixoverlap_results, \
                                                           printstatus = postprocess_dict['printstatus'])
    
    #now check suffix overlap for returned columns (noting that suffixcolumn already checked above)
    suffixoverlap_results = self.__df_check_suffixoverlap(initial_columns, \
                                                         newcolumns_list, \
                                                         suffixoverlap_results = suffixoverlap_results, \
                                                         printstatus = postprocess_dict['printstatus'])
    
    #add entries to custom_process_wrapper_dict associated with suffix and inplace for reference
    custom_process_wrapper_dict.update({'suffix'  : suffix,
                                        'inplace' : inplace})
    
    #___
    
    #we'll perform one more infill via adjinfill in case user defined transform had unforseen edge case
    
    if MLinfilltype not in {'exclude', 'boolexclude', 'ordlexclude', 'totalexclude'}:

      #apply ffill to replace NArows with value from adjacent cell in preceding row
      mdf_train[newcolumns_list] = mdf_train[newcolumns_list].fillna(method='ffill')
      mdf_test[newcolumns_list] = mdf_test[newcolumns_list].fillna(method='ffill')

      #we'll follow with a bfill just in case first row had a nan
      mdf_train[newcolumns_list] = mdf_train[newcolumns_list].fillna(method='bfill')
      mdf_test[newcolumns_list] = mdf_test[newcolumns_list].fillna(method='bfill')
      
      #finally if prior infill still resulted in nan we'll just plug with 0
      mdf_train[newcolumns_list] = mdf_train[newcolumns_list].fillna(0)
      mdf_test[newcolumns_list] = mdf_test[newcolumns_list].fillna(0)
    
    #___
    
    #now perform a datatype conversion based on MLinfilltype
    
    dtype_convert = True
    #user can deactivate dtype convert in processdict if desired
    if 'dtype_convert' in postprocess_dict['process_dict'][treecategory]:
      if postprocess_dict['process_dict'][treecategory]['dtype_convert'] is False:
        dtype_convert = False
        
    #we'll store any parameters for conditional dtype in custom_process_wrapper_dict
    custom_process_wrapper_dict.update({'dtype_convert' : dtype_convert})
    
    if dtype_convert is True:

      if MLinfilltype in {'numeric', 'concurrent_nmbr', 'exclude'}:
        #datatype conversion performed elsewhere based on floatprecision parameter
        pass

      if MLinfilltype in {'binary', 'multirt', '1010', 'concurrent_act', 'boolexclude'}:
        #datatype cast as np.int8 since entries are boolean integers
        mdf_train[newcolumns_list] = mdf_train[newcolumns_list].astype(np.int8)
        mdf_test[newcolumns_list] = mdf_test[newcolumns_list].astype(np.int8)
      
      if MLinfilltype in {'singlct', 'ordlexclude', 'concurrent_ordl'}:
        #ordinal sets are given a conditional dtype based on max activation
        #this can be deactivated in processdict with dtype_convert if desired
        
        #for concurrent_ordl we'll populate a dicitonary of max_encodings for each newcolumn
        max_encoding_for_dtype_convert = {}

        for newcolumn in newcolumns_list:

          max_encoding = mdf_train[newcolumn].max()
          max_encoding_for_dtype_convert.update({newcolumn : max_encoding})

          if max_encoding <= 255:
            mdf_train[newcolumn] = mdf_train[newcolumn].astype(np.uint8)
            mdf_test[newcolumn] = mdf_test[newcolumn].astype(np.uint8)
          elif max_encoding <= 65535:
            mdf_train[newcolumn] = mdf_train[newcolumn].astype(np.uint16)
            mdf_test[newcolumn] = mdf_test[newcolumn].astype(np.uint16)
          else:
            mdf_train[newcolumn] = mdf_train[newcolumn].astype(np.uint32)
            mdf_test[newcolumn] = mdf_test[newcolumn].astype(np.uint32)

        custom_process_wrapper_dict.update({'max_encoding_for_dtype_convert' : max_encoding_for_dtype_convert})

      if MLinfilltype in {'integer', 'totalexclude'}:
        #no conversion, assumes any conversion takes place in transformation function
        pass

    #___
    
    #Now populate the returned column_dict_list
    #(note that the column_dict entry 'custom_process_wrapper_dict' is unique to the custom_train convention)
    
    column_dict_list = []
    
    newcolumn_normalization_dict = {}
    if len(newcolumns_list) > 0:
      newcolumn_normalization_dict = {newcolumns_list[0] : normalization_dict}
    
    for newcolumn in newcolumns_list:

      if newcolumn != newcolumns_list[0]:
        newcolumn_normalization_dict = {}
      
      column_dict = {newcolumn : {'category' : treecategory, \
                                 'origcategory' : category, \
                                 'normalization_dict' : newcolumn_normalization_dict, \
                                 'origcolumn' : column, \
                                 'inputcolumn' : column, \
                                 'columnslist' : newcolumns_list, \
                                 'categorylist' : newcolumns_list, \
                                 'infillmodel' : False, \
                                 'infillcomplete' : False, \
                                 'suffixoverlap_results' : suffixoverlap_results, \
                                 'deletecolumn' : False, \
                                 'custom_process_wrapper_dict' : custom_process_wrapper_dict}}

      column_dict_list.append(column_dict)

    return mdf_train, mdf_test, column_dict_list

  def __check_for_protected_features(self, assign_param, processdict):
    """
    for cases where any protected_feature have specification
    either in assign_param or processdict
    those features are prepared in a seperate for loop through columns
    to ensure available in original form when noise trasnforms are applied
    """

    protected_features = []
    
    for entry_1 in assign_param:
      #global specification {global_assignparam : {param : value}}
      if entry_1 == 'global_assignparam':
        if 'protected_feature' in assign_param[entry_1]:
          if assign_param[entry_1]['protected_feature'] is not False:
            protected_features.append(assign_param[entry_1]['protected_feature'])
            
      #default category specification {default_assignparam : {category : {param : value}}}
      elif entry_1 == 'default_assignparam':
        for entry_2 in assign_param[entry_1]:
          if 'protected_feature' in assign_param[entry_1][entry_2] \
          and assign_param[entry_1][entry_2]['protected_feature'] is not False:
            protected_features.append(assign_param[entry_1][entry_2]['protected_feature'])

      #target specification {category : {column : {param : value}}}
      else:
        for entry_2 in assign_param[entry_1]:
          if 'protected_feature' in assign_param[entry_1][entry_2] \
          and assign_param[entry_1][entry_2]['protected_feature'] is not False:
            protected_features.append(assign_param[entry_1][entry_2]['protected_feature'])

    #user specified processdict may have protected_feature specifciation in defaultparams
    for root in processdict:
      if 'defaultparams' in processdict[root]:
        if 'protected_feature' in processdict[root]['defaultparams']:
          if processdict[root]['defaultparams']['protected_feature'] is not False:
            protected_features.append(processdict[root]['defaultparams']['protected_feature'])

    #consolidate redundancies
    #these are sorted outside of this function for consistent order
    protected_features = list(set(protected_features))

    return protected_features

  def __messy_data_prep(self, protected_features, assigncat, assign_param, postprocess_dict, printstatus):
    """
    messy_data refers to non tidy feature sets with more than one column per feature
    we'll have convention that to designate column aggregations to treat as single feature
    user can pass a set of input headers to assigncat as a {set} of headers in place of a string
    
    this funciton serves multiple purposes assocaited with the messy_data convenion
    assigncat is inspected to identify messy specification as sets of strings within the specification list
    and replaces messy sets in assigncat with one of those entries
    
    the other entries from a messy set are added to protected_features
    and also added to assigncat['null'] 
    
    and also added to assign_param['messy_data_headers']
    in the form
    {inputcolumn : {messy_set}}
    
    which is accessed in __grab_params
    where the root category is used to selectively pass to tree categories for that root
    as params['messy_data_headers'] = messy_set
    and also with params['inplace'] = False
    
    then with the protected features the others are prepared in a seperate for loop and deleted with the null
    """

    messy_data_headers_dict = {}
    add_to_null_set = set()
    already_populated_set = set()
    
    messy_set_overlap_valresult = False
    
    #inspect assigncat to identify messy sets
    #convention is is user assigned a single set they are embeded in list brackets in __assigncat_str_convert
    for rootcat in assigncat:
      
      current_list = assigncat[rootcat]
      
      for current_list_set in current_list:
        
        if isinstance(current_list_set, set):
          
          if len(current_list_set - already_populated_set) < len(current_list_set):
            messy_set_overlap_valresult = True
            # if printstatus != 'silent':
            self.__autoprint('warning', "Warning of potential error from overlapping messy_data set specifications in assigncat")
            self.__autoprint('warning', '')   
          already_populated_set = already_populated_set | current_list_set
          
          #access element from set
          for inputcolumn in sorted(list(current_list_set)):
            break
          
          #populate messy_data_headers_dict
          messy_data_headers_dict.update({inputcolumn : sorted(list(current_list_set))})
    
          #add entries to protected_features and assigncat[null]
          for current_list_set_entry in current_list_set:
          
            if current_list_set_entry != inputcolumn:
              
              protected_features = protected_features + [current_list_set_entry]
              
              add_to_null_set = add_to_null_set | {current_list_set_entry}
              
          #replace the asisgncat set specification with the excerpt
          assigncat[rootcat][current_list.index(current_list_set)] = inputcolumn
              
    #now add messy_data_headers_dict to assign_param
    assign_param.update({'messy_data_headers' : messy_data_headers_dict})
    
    #then the null specifications of the new protected_features are added to assigncat
    if 'null' not in assigncat:
      assigncat.update({'null':list(add_to_null_set)})
    else:
      assigncat['null'] = assigncat['null'] + list(add_to_null_set)

    return protected_features, assigncat, assign_param, messy_set_overlap_valresult

  def __apply_processfamily(self,
                            df_train,
                            df_test,
                            columns_train,
                            postprocess_dict,
                            transform_dict,
                            mirror_dict,
                            assigncat,
                            inverse_assigncat,
                            final_assigncat,
                            assignnan,
                            masterNArows_train,
                            masterNArows_test,
                            randomseed,
                            eval_ratio,
                            numbercategoryheuristic,
                            powertransform,
                            evalcat,
                            drift_dict,
                            assign_param,
                            printstatus,
                            inplace,
                            logger,
                            labels = False,
                           ):
    """
    Aggregates the processfamily functions and surrouding actions
    Including evalcategory, getNArows, populating mirror_dict, etc
    the labels case is to distinguish between applying function to features or labels
    """

    #For each column, perform processing 
    #based on either category assignments in assigncat 
    #or under automation based on train set feature evaluation in _evalcategory
    for column in columns_train:
      #
      categorycomplete = False

      if bool(assigncat) is True:
    
        if column in inverse_assigncat:
        
          category = inverse_assigncat[column]
          categorycomplete = True

          #printout display progress
          # if printstatus is True:
          if labels is False:
            self.__autoprint('debug', "evaluating column: ", column)
          else:
            self.__autoprint('debug', "evaluating label column: ", column)

          #assigncat has special case, can assign distinct columns to automated evaluation
          #if user assigned column to 'eval' or 'ptfm'
          #such as to perform eval when default is powertransform or visa versa
          #with _evalcategory distinction based on temp_powertransform_for_evalcategory_call
          if category in {'eval', 'ptfm'}:

            if category == 'eval':
              temp_powertransform_for_evalcategory_call = False
            if category == 'ptfm':
              temp_powertransform_for_evalcategory_call = True

            if evalcat is False:
              category = self.__evalcategory(df_train, column, randomseed, eval_ratio, \
                                           numbercategoryheuristic, temp_powertransform_for_evalcategory_call, labels)
            elif callable(evalcat):
              category = evalcat(df_train, column, randomseed, eval_ratio, \
                                 numbercategoryheuristic, temp_powertransform_for_evalcategory_call, labels)
      #
      if categorycomplete is False:

        #printout display progress
        # if printstatus is True:
        if labels is False:
          self.__autoprint('debug', "evaluating column: ", column)
        else:
          self.__autoprint('debug', "evaluating labels column: ", column)

        if evalcat is False:
          category = self.__evalcategory(df_train, column, randomseed, eval_ratio, \
                                       numbercategoryheuristic, powertransform, labels)
        elif callable(evalcat):
          category = evalcat(df_train, column, randomseed, eval_ratio, \
                             numbercategoryheuristic, powertransform, labels)
          
        #populate the result in the final_assigncat as informational resource
        if category in final_assigncat:
          final_assigncat[category].append(column)
        else:
          final_assigncat.update({category:[column]})

      #Previously had a few methods here to validate consistensy of data between train
      #and test sets. Found it was introducing too much complexity and was having trouble
      #keeping track of all the edge cases. So let's just make outright assumption that
      #test data if passed is consistently formatted as train data (for now)
      #added benefit that this reduces running time

      ##
      #to support the postprocess_dict entry below, let's first create a temp
      #list of columns
      #using a list instead of set here to maintain order, even though set would be a little quicker
      templist1 = list(df_train)
      #tempset1 is similar to templist1, but is based on column_dict so includes derivations that were subject to replacement
      tempset1 = set(postprocess_dict['column_dict'])

      #Before calling getNArows, we'll allow user to designate either by category or column 
      #designated source column values that will be converted to nan for treatment as infill
      #where in case of specification redundancy column designation takes precedence
      #and where category is referring to the root category associated with a column
      #and global just means this value treated universally as nan
      #where values are passed in automunge(.) parameter assignnan e.g.
      #assignnan = {'categories':{'cat1':[], 'cat2':[]}, 'columns':{'col1':[], 'col2':[]}, 'global':[]}

      df_train = self.__assignnan_convert(df_train, column, category, assignnan, postprocess_dict)
      df_test = self.__assignnan_convert(df_test, column, category, assignnan, postprocess_dict)

      #we also have convention that infinity values are by default subjected to infill
      #based on understanding that ML libraries in general do not accept thesae kind of values
      #as well as the python None value

      convert_to_nan_list = [np.inf, -np.inf, None, float("NaN")]
      df_train = self.__convert_to_nan(df_train, column, category, postprocess_dict, convert_to_nan_list)
      df_test = self.__convert_to_nan(df_test, column, category, postprocess_dict, convert_to_nan_list)
      
      if labels is False:

        #create NArows (column of True/False where True coresponds to missing data)
        trainNArows, drift_dict = self.__getNArows(df_train, column, category, postprocess_dict, drift_dict=drift_dict, driftassess=True)
        testNArows = self.__getNArows(df_test, column, category, postprocess_dict)

        #now append that NArows onto a master NA rows df
        #these df's are used to support application of infill 
        #such as for partitioning data sets for ML infill and identifying infill targets
        masterNArows_train = pd.concat([masterNArows_train, trainNArows], axis=1)
        masterNArows_test = pd.concat([masterNArows_test, testNArows], axis=1)

      #printout display progress
      # if printstatus is True:
      if labels is False:
        self.__autoprint('debug', "processing column: ", column)
      else:
        self.__autoprint('debug', "processing labels column: ", column)
      self.__autoprint('debug', "    root category: ", category)

      ##
      #now process family
      df_train, df_test, postprocess_dict = \
      self.__processfamily(df_train, df_test, column, category, \
                        transform_dict, postprocess_dict, assign_param, logger)

      ##
      #now delete columns that were subject to replacement when inplace transform wasn't available
      df_train, df_test, postprocess_dict = \
      self.__circleoflife(df_train, df_test, column, category, \
                        transform_dict, postprocess_dict, templist1)
      ##
      #here's another templist to support the postprocess_dict entry below
      templist2 = list(df_train)
      tempset2 = set(postprocess_dict['column_dict'])

      #ok now we're going to pick one of the new entries in templist2 to serve 
      #as a "columnkey" for pulling datas from the postprocess_dict down the road
      #similar to but retains order: columnkeylist = list(set(templist2) - set(templist1))
      columnkeylist = []
      for templist2_entry in templist2:
        if templist2_entry not in templist1:
          columnkeylist.append(templist2_entry)

      #allderivedlist is similar to columnkeylist but includes headers of derivations that were subject to replacement
      allderivedlist = tempset2 - tempset1
      allderivedlist = \
      self.__list_sorting(columnkeylist, allderivedlist)

      #an arbitrary columnkey is populated in postprocess_dict['origcolumn'] with columnkeylist
      if len(columnkeylist) == 0:
        columnkey = column
      else:
        columnkey = columnkeylist[0]

      ##
      if labels is False:
        origcolumn_type = 'train'
      else:
        origcolumn_type = 'label'
        
      postprocess_dict['origcolumn'].update({column : {'type' : origcolumn_type, \
                                                       'category' : category, \
                                                       'columnkeylist' : columnkeylist, \
                                                       'allderivedlist' : allderivedlist, \
                                                       'columnkey' : columnkey}})

      #populate mirror_dict, which will serve as the returned form of transform_dict and process_dict returned in postprocess_dict
      #mirror_dict is used so that user preserves privacy of custom entries that were not inspected
      mirror_dict, _1 = \
      self.__populate_labelsencoding_dict_support2(mirror_dict, postprocess_dict, transform_dict, category, 0)

      ##
      #printout display progress
      # if printstatus is True:
      self.__autoprint('debug', " returned columns:")
      self.__autoprint('debug', postprocess_dict['origcolumn'][column]['columnkeylist'])
      self.__autoprint('debug', "")

      if inplace is not True:
        #this defragments the dataframes
        df_train = df_train.copy()
        df_test = df_test.copy()
        
    #one more populate mirror_dict for mlti categories which are categories inspected outside of a root category tree such as for mlti transform
    for mlti_entry in postprocess_dict['mlti_categories']:
      mirror_dict, _1 = \
      self.__populate_labelsencoding_dict_support2(mirror_dict, postprocess_dict, transform_dict, mlti_entry, 0)

    return df_train, \
           df_test, \
           postprocess_dict, \
           mirror_dict, \
           final_assigncat, \
           masterNArows_train, \
           masterNArows_test

  def __prepare_labelsencoding_dict(self, postprocess_dict, processdict, transform_dict, mirror_dict, labels_column_listofcolumns, printstatus, logger):
    """
    labelsencoding_dict is used to support public label inversion with enncryption
    in the privacy_encode = False or True case
    note that we use the term labelsencoding_dict in a few support funcitons
    that also serve other purposes for populating the "mirror_dict"
    the purpose of labelsencoding_dict is to extract portions of the column_dict
    relavant to the label features
    including also relevant entries for transform_dict and process_dict
    and any other postprocess_dict entries needed to support label inversion
    """

    labelsencoding_dict = {'transforms' : {}}
    
    for column in labels_column_listofcolumns:
      
      category = postprocess_dict['origcolumn'][column]['category']
      columnkeylist = postprocess_dict['origcolumn'][column]['columnkeylist']

      #labelsencoding_dict is returned in postprocess_dict
      #this is redundant with information stored in postprocess_dict['column_dict']
      #labelsencoding_dict is used for public label inversion without encryption key
      #first we'll populate some entries and then final aggregations performed after fully populating postprocess_dict
      labelsencoding_dict['transforms'].update({column : {category:{}}})
      for finalcolumn_label in columnkeylist:

        #this populates a labelsnormalization_dict as {returnedcolumn : {}}
        #which later in the workflow are translated to column_dict entries in __populate_labelsencoding_dict_support3
        labelsencoding_dict['transforms'][column][category].update({finalcolumn_label : {}})

        #this access any upstream normalization_dict's for columns that were subject to replacement
        inputcolumn = postprocess_dict['column_dict'][finalcolumn_label]['inputcolumn']
        labelsencoding_dict = self.__populate_labelsencoding_dict_support(labelsencoding_dict, postprocess_dict, column, inputcolumn)

      #now populate any transform_dict or process_dict entries that were inspected for label transforms
      labelsencoding_dict, _0 = \
      self.__populate_labelsencoding_dict_support2(labelsencoding_dict, postprocess_dict, transform_dict, category, 0)

      #now that we know the root label category, we'll verify that if this was a custom processdict entry
      #it either includes a labelctgy entry or we'll otherwise populate one based on family tree
      #returning any update in postprocess_dict['process_dict']
      postprocess_dict, mirror_dict, check_processdict3_valresult, check_processdict3_validlabelctgy_valresult = \
      self.__check_processdict3(category, processdict, postprocess_dict, transform_dict, mirror_dict, printstatus)

      logger_dict['validations'].update({'check_processdict3_valresult' : check_processdict3_valresult,
                                                              'check_processdict3_validlabelctgy_valresult' : check_processdict3_validlabelctgy_valresult})
      logger = self.__logger_append(logger)
      
    return labelsencoding_dict, mirror_dict, postprocess_dict

  #__FunctionBlock: suffix overlap and defaultinfill support functions

  def __apply_defaultinfill(self, df, suffixcolumn, postprocess_dict, treecategory = False, defaultinfill_dict = False):
    """
    Applies default infill based on defaultinfill processdict entry
    Where if defaultinfill not populated defers to adjinfill
    
    This function can optionally be called internal to the transformation functions
    In dual/single/post process convention
    Where it is intended for application after suffixcolumn creation
    And after data type casting based on NArowtype
    
    Accepts input of train or test data as df
    suffixcolumn is the target column
    postprocess_dict is used to access process_dict
    treecategory is the tree category associated with the transform, which can be passed as False within postprocess functions
    (treecategory won't be inspected when defaultinfill_dict is populated)
    defaultinfill_dict passed as False for train data, test data receives entries populated from train data
    
    note that defaultinfill_dict should be externally stored in normalization_dict
    
    If custom default infill desired for a transform can designate naninfill (or simply omit this function call)
    
    defaultinfill support / default entry for a transform 
    will be noted in the familytrees process_dict population

    note that zeroinfill and oneinfill may not return a final 0/1 value if further operations applied in transform
    for final returned 0/1 can use assigninfill

    note that negzeroinfill intended for use in qbt1 family of transforms
    
    note that when column is pandas category dtype if an infill value not already a registered entry we add it with add_categories
    """    

    #if this is test data then we are expecting a populated defaultinfill_dict from the train data
    #otherwise we'll initialize
    
    #which will default to adjacent cell infill unless otherwise specified in processdict
    #here for simplicity we'll run through each defaultinfill twice
    #first if is for test data, the else will be for train data
    
    traindata = False
    if defaultinfill_dict is not False and 'defaultinfill' in defaultinfill_dict:
      
      traindata = True
      
      defaultinfill = defaultinfill_dict['defaultinfill']
      
      if defaultinfill in {'adjinfill'}:

        #apply ffill to replace NArows with value from adjacent cell in preceding row
        df[suffixcolumn] = df[suffixcolumn].fillna(method='ffill')

        #we'll follow with a bfill just in case first row had a nan
        df[suffixcolumn] = df[suffixcolumn].fillna(method='bfill')
        
      elif defaultinfill in {'interpinfill'}:
        
        df[suffixcolumn] = pd.to_numeric(df[suffixcolumn], errors='coerce').interpolate(method='linear', 
                                                                                        axis=0, 
                                                                                        limit=None, 
                                                                                        inplace=False,
                                                                                        limit_direction='forward',
                                                                                        limit_area=None,
                                                                                        downcast=None,
                                                                                       )
        
        #we'll follow with a adj just in case first or last row had a nan
        df[suffixcolumn] = df[suffixcolumn].fillna(method='ffill')
        df[suffixcolumn] = df[suffixcolumn].fillna(method='bfill')

      elif defaultinfill in {'meaninfill'}:

        infill_mean = defaultinfill_dict['infill_mean']
        
        if df[suffixcolumn].dtype.name == 'category':
          if infill_mean not in df[suffixcolumn].cat.categories:
            df[suffixcolumn] = df[suffixcolumn].cat.add_categories([infill_mean])

        df[suffixcolumn] = df[suffixcolumn].fillna(infill_mean)

      elif defaultinfill in {'medianinfill'}:

        infill_median = defaultinfill_dict['infill_median']
        
        if df[suffixcolumn].dtype.name == 'category':
          if infill_median not in df[suffixcolumn].cat.categories:
            df[suffixcolumn] = df[suffixcolumn].cat.add_categories([infill_median])

        df[suffixcolumn] = df[suffixcolumn].fillna(infill_median)

      elif defaultinfill in {'modeinfill'}:

        infill_mode = defaultinfill_dict['infill_mode']
        
        if df[suffixcolumn].dtype.name == 'category':
          if infill_mode not in df[suffixcolumn].cat.categories:
            df[suffixcolumn] = df[suffixcolumn].cat.add_categories([infill_mode])

        df[suffixcolumn] = df[suffixcolumn].fillna(infill_mode)

      elif defaultinfill in {'lcinfill'}:

        infill_lc = defaultinfill_dict['infill_lc']
        
        if df[suffixcolumn].dtype.name == 'category':
          if infill_lc not in df[suffixcolumn].cat.categories:
            df[suffixcolumn] = df[suffixcolumn].cat.add_categories([infill_lc])

        df[suffixcolumn] = df[suffixcolumn].fillna(infill_lc)
        
      elif defaultinfill in {'zeroinfill'}:
        
        if df[suffixcolumn].dtype.name == 'category':
          if 0 not in df[suffixcolumn].cat.categories:
            df[suffixcolumn] = df[suffixcolumn].cat.add_categories([0])

        df[suffixcolumn] = df[suffixcolumn].fillna(0)

      elif defaultinfill in {'oneinfill'}:
        
        if df[suffixcolumn].dtype.name == 'category':
          if 1 not in df[suffixcolumn].cat.categories:
            df[suffixcolumn] = df[suffixcolumn].cat.add_categories([1])
        
        df[suffixcolumn] = df[suffixcolumn].fillna(1)

      elif defaultinfill in {'negzeroinfill'}:
        
        if df[suffixcolumn].dtype.name == 'category':
          if -0. not in df[suffixcolumn].cat.categories:
            df[suffixcolumn] = df[suffixcolumn].cat.add_categories([-0.])

        df[suffixcolumn] = df[suffixcolumn].fillna(-0.)

      elif defaultinfill in {'naninfill'}:
        #naninfill is intended for cases when user wishes to apply a custom default infill inside of the transform
        pass

      #finally if prior infill still resulted in nan we'll just plug with 0
      if defaultinfill not in {'naninfill'}:
        
        if df[suffixcolumn].dtype.name == 'category':
          if 0 not in df[suffixcolumn].cat.categories:
            df[suffixcolumn] = df[suffixcolumn].cat.add_categories([0])
        
        df[suffixcolumn] = df[suffixcolumn].fillna(0)

    #else if this is train data we'll also populate the defaultinfill_dict
    else:
      
      #we'll apply a default infill
      #contingent on the 'NArowtype' of the tree category serving as basis for this transform
      NArowtype = postprocess_dict['process_dict'][treecategory]['NArowtype']

      defaultinfill = 'adjinfill'
      if 'defaultinfill' in postprocess_dict['process_dict'][treecategory]:
        if isinstance(postprocess_dict['process_dict'][treecategory]['defaultinfill'], str) \
        and postprocess_dict['process_dict'][treecategory]['defaultinfill'] \
        in {'adjinfill', 'interpinfill', 'meaninfill', 'medianinfill', 'modeinfill', 'lcinfill', 'zeroinfill', 'oneinfill', 'negzeroinfill', 'naninfill'}:
          defaultinfill = postprocess_dict['process_dict'][treecategory]['defaultinfill']

      #a few special cases to accomodate NArowtype / defaultinfill compatibilities

      #'meaninfill' and 'medianinfill' intended for numeric data, if not a numeric NArowtype adjinfill applied
      if defaultinfill in {'meaninfill', 'medianinfill', 'interpinfill'}:
        if NArowtype not in {'numeric', 'integer', 'positivenumeric', 'nonnegativenumeric', 'nonzeronumeric'}:
          defaultinfill = 'adjinfill'

      #'datetime' NArowtype only accepts adjinfill or naninfill
      if NArowtype in {'datetime'}:
        if defaultinfill not in {'adjinfill', 'naninfill'}:
          defaultinfill = 'adjinfill'

      #no infill performed for 'exclude' or 'totalexclude' NArowtype
      if NArowtype in {'exclude', 'totalexclude'}:
        defaultinfill = 'naninfill'

      #initialize a dictionary to pass default infill parameters between automunge and postmunge in column_dict
      defaultinfill_dict = {'defaultinfill' : defaultinfill}
    
      if defaultinfill in {'adjinfill'}:

        #apply ffill to replace NArows with value from adjacent cell in preceding row
        df[suffixcolumn] = df[suffixcolumn].fillna(method='ffill')

        #we'll follow with a bfill just in case first row had a nan
        df[suffixcolumn] = df[suffixcolumn].fillna(method='bfill')
        
      elif defaultinfill in {'interpinfill'}:
        
        df[suffixcolumn] = pd.to_numeric(df[suffixcolumn], errors='coerce').interpolate(method='linear', 
                                                                                        axis=0, 
                                                                                        limit=None, 
                                                                                        inplace=False,
                                                                                        limit_direction='forward',
                                                                                        limit_area=None,
                                                                                        downcast=None,
                                                                                       )
        
        #we'll follow with a adj just in case first or last row had a nan
        df[suffixcolumn] = df[suffixcolumn].fillna(method='ffill')
        df[suffixcolumn] = df[suffixcolumn].fillna(method='bfill')

      elif defaultinfill in {'meaninfill'}:

        infill_mean = df[suffixcolumn].mean()
        defaultinfill_dict.update({'infill_mean' : infill_mean})
        
        if df[suffixcolumn].dtype.name == 'category':
          if infill_mean not in df[suffixcolumn].cat.categories:
            df[suffixcolumn] = df[suffixcolumn].cat.add_categories([infill_mean])

        df[suffixcolumn] = df[suffixcolumn].fillna(infill_mean)

      elif defaultinfill in {'medianinfill'}:

        infill_median = df[suffixcolumn].median()
        defaultinfill_dict.update({'infill_median' : infill_median})
        
        if df[suffixcolumn].dtype.name == 'category':
          if infill_median not in df[suffixcolumn].cat.categories:
            df[suffixcolumn] = df[suffixcolumn].cat.add_categories([infill_median])

        df[suffixcolumn] = df[suffixcolumn].fillna(infill_median)

      elif defaultinfill in {'modeinfill'}:

        infill_mode = df[suffixcolumn].mode()

        if len(infill_mode) > 0:
          infill_mode = infill_mode[0]
        else:
          infill_mode = 0

        defaultinfill_dict.update({'infill_mode' : infill_mode})
        
        if df[suffixcolumn].dtype.name == 'category':
          if infill_mode not in df[suffixcolumn].cat.categories:
            df[suffixcolumn] = df[suffixcolumn].cat.add_categories([infill_mode])

        df[suffixcolumn] = df[suffixcolumn].fillna(infill_mode)

      elif defaultinfill in {'lcinfill'}:

        #(asdf as used here is just an arbitrary string)
        tempstring = 'asdf'
        if tempstring in df:
          while tempstring in df:
            tempstring += 'z'
        #defaultinfill of lcinfill inserts infill as dtype str when bytes entries are present 
        #to avoid edge case for bytes type entries with value_counts sorting
        if (df[suffixcolumn] == df[suffixcolumn].astype(bytes)).sum() > 0:
          mode_valuecounts_list = pd.DataFrame(df[suffixcolumn].astype(str).value_counts())
        else:
          mode_valuecounts_list = pd.DataFrame(df[suffixcolumn].value_counts())
        mode_valuecounts_list = mode_valuecounts_list.rename_axis(tempstring).sort_values(by = [suffixcolumn, tempstring], ascending = [False, True])
        mode_valuecounts_list = list(mode_valuecounts_list.index)

        if len(mode_valuecounts_list) > 0:
          infill_lc = mode_valuecounts_list[-1]
        else:
          infill_lc = 0

        defaultinfill_dict.update({'infill_lc' : infill_lc})
        
        if df[suffixcolumn].dtype.name == 'category':
          if infill_lc not in df[suffixcolumn].cat.categories:
            df[suffixcolumn] = df[suffixcolumn].cat.add_categories([infill_lc])

        df[suffixcolumn] = df[suffixcolumn].fillna(infill_lc)

      elif defaultinfill in {'zeroinfill'}:
        
        if df[suffixcolumn].dtype.name == 'category':
          if 0 not in df[suffixcolumn].cat.categories:
            df[suffixcolumn] = df[suffixcolumn].cat.add_categories([0])

        #note that transform may not return a 0 value, for final returned 0 use assigninfill
        df[suffixcolumn] = df[suffixcolumn].fillna(0)

      elif defaultinfill in {'oneinfill'}:
        
        if df[suffixcolumn].dtype.name == 'category':
          if 1 not in df[suffixcolumn].cat.categories:
            df[suffixcolumn] = df[suffixcolumn].cat.add_categories([1])

        #note that transform may not return a 1 value, for final returned 1 use assigninfill
        df[suffixcolumn] = df[suffixcolumn].fillna(1)

      elif defaultinfill in {'negzeroinfill'}:
        
        if df[suffixcolumn].dtype.name == 'category':
          if -0. not in df[suffixcolumn].cat.categories:
            df[suffixcolumn] = df[suffixcolumn].cat.add_categories([-0.])

        df[suffixcolumn] = df[suffixcolumn].fillna(-0.)

      elif defaultinfill in {'naninfill'}:
        #naninfill is intended for cases when user wishes to apply a custom default infill inside of the transform
        #note an adjinfill is applied later in this function, for final returned nan use assigninfill
        pass

      #finally if prior infill still resulted in nan we'll just plug with 0
      if defaultinfill not in {'naninfill'}:
        
        if df[suffixcolumn].dtype.name == 'category':
          if 0 not in df[suffixcolumn].cat.categories:
            df[suffixcolumn] = df[suffixcolumn].cat.add_categories([0])

        df[suffixcolumn] = df[suffixcolumn].fillna(0)
    
    return df, defaultinfill_dict

  def __set_indexcolumn(self, trainID_column, testID_column, application_number, indexcolumn_base = 'Automunge_index'):
    """
    #this either sets indexcolumn as 'Automunge_index' 
    #or 'Automunge_index_' + str(application_number) if 'Automunge_index' is already in ID sets
    #(this helps with a rare potential workflow when data sets are repeatedly run through automunge)
    """

    indexcolumn_valresult = False

    fullset = set()
    
    indexcolumn = indexcolumn_base
    if isinstance(trainID_column, list):
      fullset = fullset | set(trainID_column)
      if indexcolumn_base in trainID_column:
        indexcolumn = indexcolumn_base + '_' + str(application_number)
    elif indexcolumn_base == trainID_column:
      fullset = fullset | {trainID_column}
      indexcolumn = indexcolumn_base + '_' + str(application_number)
        
    testindexcolumn = indexcolumn_base
    if isinstance(testID_column, list):
      fullset = fullset | set(testID_column)
      if indexcolumn_base in testID_column:
        testindexcolumn = indexcolumn_base + '_' + str(application_number)
    elif indexcolumn_base == testID_column:
      fullset = fullset | {testID_column}
      testindexcolumn = indexcolumn_base + '_' + str(application_number)
      
    if testindexcolumn != indexcolumn:
      if len(testindexcolumn) > len(indexcolumn):
        indexcolumn = testindexcolumn

    #this is a very remote edge case, just being comprehensive
    if indexcolumn in fullset:
      
      while indexcolumn in fullset:
        indexcolumn = indexcolumn + ','

    if indexcolumn != indexcolumn_base:
      indexcolumn_valresult = True
    
    return indexcolumn, indexcolumn_valresult

  def __set_Binary_column(self, postprocess_dict, suffixrange, root = 'Binary_'):
    """
    #this support function used in Binary dimensionality reduction
    #which Binary and PCA are unique in library in that they create new column headers by means other than suffix appention
    #so we will compare the root of the new column
    #which for Binary will be 'Binary_'
    #to input columns logged in postprocess_dict['origcolumn']
    #and returned columns logged in postprocess_dict['column_dict']
    #and if overlap found add an appender to the root as string of the application_number
    #which is a 12 digit integer derived by random sample for each automunge(.) call
    #similar to what is done for Automunge_index returned in ID sets
    #note that we'll also check for overlaps between potential configurations of root plus suffix
    #if overlap found a validation result will be returned
    #suffixrange is an integer >= 0 which sets depth of how many overlaps will be checked
    #which we know the number of returned columns from Binary will be <= a value as function of nunique
    """

    #root column will be base of the new column header
    rootcolumn = root
    set_Binary_column_valresult = False
    
    #___
    
    #permutations are potential derivaitons from the root based on different Binary options
    #note that to ensure comprehensive, we set a suffixrange depth of inspection as per _Binary_convert
    permutations = {rootcolumn, rootcolumn+'_ord3'}
    for i in range(suffixrange):
      _1010_permutation = rootcolumn + '_1010_' + str(i)
      _onht_permutation = rootcolumn + '_onht_' + str(i)
      permutations = permutations | {_1010_permutation, _onht_permutation}
      
    origcolumns = set(postprocess_dict['origcolumn'])
    returnedcolumns = set(postprocess_dict['column_dict'])
    fullcolumns = origcolumns | returnedcolumns
    
    #if overlap between fullcolumns and root and permutations add application number appender
    if len(permutations & fullcolumns) > 0:
      rootcolumn = rootcolumn + str(postprocess_dict['application_number'])
      set_Binary_column_valresult = True
      
    #___
    
    if set_Binary_column_valresult is True:
      #now one more check to be comprehensive, this is for a very remote edge case
      permutations = {rootcolumn, rootcolumn+'_ord3'}
      for i in range(suffixrange):
        _1010_permutation = rootcolumn + '_1010_' + str(i)
        _onht_permutation = rootcolumn + '_onht_' + str(i)
        permutations = permutations | {_1010_permutation, _onht_permutation}

      if len(permutations & fullcolumns) > 0:
        while len(permutations & fullcolumns) > 0:

          rootcolumn = rootcolumn + ','

          permutations = {rootcolumn, rootcolumn+'_ord3'}
          for i in range(suffixrange):
            _1010_permutation = rootcolumn + '_1010_' + str(i)
            _onht_permutation = rootcolumn + '_onht_' + str(i)
            permutations = permutations | {_1010_permutation, _onht_permutation}
          
    #___
    
    return rootcolumn, set_Binary_column_valresult

  def __set_PCA_column(self, newcolumncount, postprocess_dict, root='PCA_'):
    """
    #this support function used in PCA dimensionality reduction
    #which Binary and PCA are unique in library in that they create new column headers by means other than suffix appention
    #so we will compare the root of the new column
    #which for PCA will be 'PCA_'
    #to input columns logged in postprocess_dict['origcolumn']
    #and returned columns logged in postprocess_dict['column_dict']
    #and if overlap found add an appender to the root as string of the application_number
    #which is a 12 digit integer derived by random sample for each automunge(.) call
    #similar to what is done for Automunge_index returned in ID sets
    #if overlap found a validation result will be returned
    """

    #the returned column names check for overlap with existing columns to accomodate edge case
    origcolumns = set(postprocess_dict['origcolumn'])
    returnedcolumns = set(postprocess_dict['column_dict'])
    fullcolumns = origcolumns | returnedcolumns
    PCA_columns_valresult = False
    
    #generate a list of column names for the conversion to pandas
    columnnames = [root + '_' + str(y) for y in range(newcolumncount)]
    
    #if overlap found, add the application_number to column headers
    if len((set(columnnames) | {root}) & fullcolumns) > 0:
      PCA_columns_valresult = True
      
      columnnames = [(root + str(postprocess_dict['application_number']) + '_' + str(y)) for y in range(newcolumncount)]
      
    #one more to be comprehensive, this is for a very remote edge case
    if PCA_columns_valresult is True:
      i=1
      if len((set(columnnames) | {root}) & fullcolumns) > 0:
        while len(set(columnnames) & fullcolumns) > 0:
          columnnames = [(root + str(postprocess_dict['application_number']) + i*',' + '_' + str(y)) for y in range(newcolumncount)]
          i+=1
        
    return columnnames, PCA_columns_valresult

  def __df_copy_train(self, df_train, column, newcolumn, suffixoverlap_results = {}, printstatus = False):
    """
    #performs a copy operation to add column to a df_train
    #Before any new columns added to df_train
    #checks that they are not already present in df_train
    #if so returns error message and logs in suffixoverlap_results
    """

    #test for overlap error
    if newcolumn in df_train.columns:
      
      # if printstatus != 'silent':
      self.__autoprint('warning', "*****************")
      self.__autoprint('warning', "Warning of suffix overlap error")
      self.__autoprint('warning', "When creating new column: ", newcolumn)
      self.__autoprint('warning', "The column was already found present in df_train headers.")
      self.__autoprint('warning', "")
      self.__autoprint('warning', "Some potential quick fixes for this error include:")
      self.__autoprint('warning', "- rename columns to integers before passing to automunge(.)")
      self.__autoprint('warning', "- strip underscores '_' from column header titles.")
      self.__autoprint('warning', "(convention is all suffix appenders include an underscore)")
      self.__autoprint('warning', "")
      self.__autoprint('warning', "Please note any updates to column headers will need to be carried through to assignment parameters.")
      self.__autoprint('warning', "*****************")
      self.__autoprint('warning', "")
      
      suffixoverlap_results.update({newcolumn : True})
      
    else:
      
      df_train[newcolumn] = df_train[column].copy()
      
      suffixoverlap_results.update({newcolumn : False})
    
    return df_train, suffixoverlap_results

  def __df_check_suffixoverlap(self, df_train, newcolumns, suffixoverlap_results = {}, printstatus = False):
    """
    #checks that newcolumns list are not already present in df_train
    #logs in suffixoverlap_results

    #we'll have convention that df_train can either be passed as the dataframe
    #or alternatively as a list of column headers
    """

    #this will result in a set of column headers for both cases (if df_train passed as list or dataframe)
    origcolumns = set(df_train)

    if not isinstance(newcolumns, list):
      newcolumns = [newcolumns]
    
    for newcolumn in newcolumns:
      
      if newcolumn in origcolumns:
        
        # if printstatus != 'silent':
        self.__autoprint('warning', "*****************")
        self.__autoprint('warning', "Warning of suffix overlap error")
        self.__autoprint('warning', "When creating new column: ", newcolumn)
        self.__autoprint('warning', "The column was already found present in df_train headers.")
        self.__autoprint('warning', "")
        self.__autoprint('warning', "Some potential quick fixes for this error include:")
        self.__autoprint('warning', "- rename columns to integers before passing to automunge(.)")
        self.__autoprint('warning', "- strip underscores '_' from column header titles.")
        self.__autoprint('warning', "(convention is all suffix appenders include an underscore)")
        self.__autoprint('warning', "")
        self.__autoprint('warning', "Please note any updates to column headers will need to be carried through to assignment parameters.")
        self.__autoprint('warning', "*****************")
        self.__autoprint('warning', "")

        suffixoverlap_results.update({newcolumn : True})

      else:

        suffixoverlap_results.update({newcolumn : False})
        
    return suffixoverlap_results

  def __suffix_overlap_final_aggregation_and_printouts(self, postprocess_dict, logger_dict, logger):
    """
    #Performs a final round of printouts in case of identified suffix overlap error
    #Also aggregates the validation results stored in column_dict
    #to those store with logging_dict
    
    #suffixoverlap_aggregated_result is added to give single boolean signal
    #for presence of suffix overlap error
    #(False is good)
    """

    suffixoverlap_aggregated_result = False
    
    #then at completion of automunge(.), aggregate the suffixoverlap results
    #and do an additional printout if any column overlap error to be sure user sees message
    for entry1 in postprocess_dict['column_dict']:
      for entry2 in postprocess_dict['column_dict'][entry1]['suffixoverlap_results']:
        if postprocess_dict['column_dict'][entry1]['suffixoverlap_results'][entry2] is True:
          
          suffixoverlap_aggregated_result = True
          # if postprocess_dict['printstatus'] != 'silent':
          self.__autoprint('warning', "*****************")
          self.__autoprint('warning', "Warning of suffix overlap error")
          self.__autoprint('warning', "When creating new column: ", entry2)
          self.__autoprint('warning', "The column was already found present in df_train headers.")
          self.__autoprint('warning', "")
          self.__autoprint('warning', "Some potential quick fixes for this error include:")
          self.__autoprint('warning', "- rename columns to integers before passing to automunge(.)")
          self.__autoprint('warning', "- strip underscores '_' from column header titles.")
          self.__autoprint('warning', "(convention is all suffix appenders include an underscore)")
          self.__autoprint('warning', "")
          self.__autoprint('warning', "Please note any updates to column headers will need to be carried through to assignment parameters.")
          self.__autoprint('warning', "*****************")
          self.__autoprint('warning', "")
      
      logger_dict['validations']['suffixoverlap_results'].update(
      postprocess_dict['column_dict'][entry1]['suffixoverlap_results'])
      logger = self.__logger_append(logger)

    #note that we automatically mitigate this channel now by revision of the PCA column root
    if 'PCA_suffixoverlap_results' in logger_dict['validations']:
      for entry1 in logger_dict['validations']['PCA_suffixoverlap_results']:
        if logger_dict['validations']['PCA_suffixoverlap_results'][entry1] is True:

            suffixoverlap_aggregated_result = True
            # if postprocess_dict['printstatus'] != 'silent':
            self.__autoprint('warning', "*****************")
            self.__autoprint('warning', "Warning of suffix overlap error")
            self.__autoprint('warning', "When creating PCA column: ", entry1)
            self.__autoprint('warning', "The column was already found present in df_train headers.")
            self.__autoprint('warning', "")
            self.__autoprint('warning', "Note that PCA returned columns are of form: PCAcol0")
            self.__autoprint('warning', "Where # is integer")
            self.__autoprint('warning', "This form of column header should be avoided in passed data.")
            self.__autoprint('warning', "")

    #note that we automatically mitigate this channel now by revision of the Binary column root
    if 'Binary_suffixoverlap_results' in logger_dict['validations']:
      for entry1 in logger_dict['validations']['Binary_suffixoverlap_results']:
        if logger_dict['validations']['Binary_suffixoverlap_results'][entry1] is True:

            suffixoverlap_aggregated_result = True
            # if postprocess_dict['printstatus'] != 'silent':
            self.__autoprint('warning', "*****************")
            self.__autoprint('warning', "Warning of suffix overlap error")
            self.__autoprint('warning', "When creating Binary column: ", entry1)
            self.__autoprint('warning', "The column was already found present in df_train headers.")
            self.__autoprint('warning', "")
            self.__autoprint('warning', "Note that Binary returned columns are of form: Binary_1010_#")
            self.__autoprint('warning', "Where # is integer")
            self.__autoprint('warning', "This error might have occured if you passed data including column header 'Binary' to '1010' transform")
            self.__autoprint('warning', "This form of column header should be avoided in passed data.")
            self.__autoprint('warning', "")

    if 'excl_suffixoverlap_results' in logger_dict['validations']:
      for entry1 in logger_dict['validations']['excl_suffixoverlap_results']:
        if logger_dict['validations']['excl_suffixoverlap_results'][entry1] is True:

            suffixoverlap_aggregated_result = True
            # if postprocess_dict['printstatus'] != 'silent':
            self.__autoprint('warning', "*****************")
            self.__autoprint('warning', "Warning of suffix overlap error")
            self.__autoprint('warning', "When removing '_excl' suffix for column: ", entry1)
            self.__autoprint('warning', "The column without suffix was already found present in df_train headers.")
            self.__autoprint('warning', "")
            
    logger_dict['validations'].update({'suffixoverlap_aggregated_result':suffixoverlap_aggregated_result})
    logger = self.__logger_append(logger)

    if 'indexcolumn_valresult' in logger_dict['validations'] \
    and logger_dict['validations']['indexcolumn_valresult'] is True:
      # if postprocess_dict['printstatus'] != 'silent':
      self.__autoprint('warning', "please note that the Automunge_index returned in the ID sets had a header update to avoid overlap")
      self.__autoprint('warning', "updated Automunge_index header is: ", postprocess_dict['indexcolumn'])
      self.__autoprint('warning', '')

    if 'PCA_columns_valresult' in logger_dict['validations'] \
    and logger_dict['validations']['PCA_columns_valresult'] is True:
      # if postprocess_dict['printstatus'] != 'silent':
      self.__autoprint('warning', "please note that the PCA derivation had a root header update to avoid overlap")
      self.__autoprint('warning', "returned PCA columns are: ", postprocess_dict['returned_PCA_columns'])
      self.__autoprint('warning', '')

    if 'set_Binary_column_valresult' in logger_dict['validations'] \
    and logger_dict['validations']['set_Binary_column_valresult'] is True:
      # if postprocess_dict['printstatus'] != 'silent':
      self.__autoprint('warning', "please note that the Binary derivation had a root header update to avoid overlap")
      self.__autoprint('warning', "returned Binary columns are: ", postprocess_dict['returned_Binary_columns'])
      self.__autoprint('warning', "returned label Binary columns are: ", postprocess_dict['final_returned_labelBinary_columns'])
      self.__autoprint('warning', '')
    
    return postprocess_dict

  #__FunctionBlock: automunge dualprocess, singleprocess, and custom_train functions

  def _process_NArw(self, df, column, category, treecategory, postprocess_dict, params = {}):
    '''
    #processing funciton that creates a boolean column indicating 1 for rows
    #corresponding to missing or improperly formated data in source column
    #note this uses the NArows function which has a category specific approach
    #returns same dataframe with new column of name column + '_NArw'
    #note this is a "singleprocess" function since is applied to single dataframe
    '''

    suffixoverlap_results = {}

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    suffixcolumn = column + '_' + suffix
    
    suffixoverlap_results = \
    self.__df_check_suffixoverlap(df, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])

    df[suffixcolumn] = self.__getNArows(df, column, category, postprocess_dict)

    #change NArows data type to 8-bit (1 byte) integers for memory savings
    df[suffixcolumn] = df[suffixcolumn].astype(np.int8)

    #create list of columns
    nmbrcolumns = [suffixcolumn]
    
    #for drift report
    pct_NArw = df[suffixcolumn].sum() / df[suffixcolumn].shape[0]

    #create normalization dictionary
    NArwnormalization_dict = {suffixcolumn : {'pct_NArw':pct_NArw, 'suffix' : suffix}}

    #store some values in the nmbr_dict{} for use later in ML infill methods
    column_dict_list = []
    
    for nc in nmbrcolumns:

      if nc != nmbrcolumns[0]:
        NArwnormalization_dict = {}

      column_dict = { nc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : NArwnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : nmbrcolumns, \
                           'categorylist' : [nc], \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)

    return df, column_dict_list

  def _process_numerical(self, mdf_train, mdf_test, column, category, \
                              treecategory, postprocess_dict, params = {}):
    '''
    #process_numerical(mdf_train, mdf_test, column, category)
    #function to normalize data to mean of 0 and standard deviation of 1 \
    #z score normalization) 
    #takes as arguement pandas dataframe of training and test data (mdf_train), (mdf_test)\
    #and the name of the column string ('column') and parent category (category)
    #replaces missing or improperly formatted data with mean of remaining values
    #returns same dataframes with new column of name column + '_nmbr'
    #note this is a "dualprocess" function since is applied to both dataframes
    #expect this approach works better when the numerical distribution is thin tailed
    #if only have training but not test data handy, use same training data for both dataframe inputs
    '''

    suffixoverlap_results = {}
    
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
    
    #initialize parameters
    #offset is just an added constant applied after multiplier
    if 'offset' in params:
      offset = params['offset']
    else:
      offset = 0
    
    #multiplier scales the set by multiplication prior to offset
    if 'multiplier' in params:
      multiplier = params['multiplier']
    else:
      multiplier = 1
    
    #cap can be passed as True for max of training data or as a specific value prior to normalization, False for no cap
    if 'cap' in params:
      cap = params['cap']
    else:
      cap = False
      
    #floor can be passed as True for min of training data or as a specific value prior to normalization, False for no floor
    if 'floor' in params:
      floor = params['floor']
    else:
      floor = False
      
    #stdev_cap can be passed as integer or float for setting cap based on number of standard deviations from mean
    #only performed with cap=False and floor=False, default of stdev_cap=False deactivates
    if 'stdev_cap' in params:
      stdev_cap = params['stdev_cap']
    else:
      stdev_cap = False

    #abs_zero accepts boolean defaulting to True for conversion of negative zero to positive
    #which is in place to ensure defaultinfill of negzeroinfill returns distinct encoding
    if 'abs_zero' in params:
      abs_zero = params['abs_zero']
    else:
      abs_zero = True

    suffixcolumn = column + '_' + suffix

    if inplace is not True:
      
      #copy source column into new column
      mdf_train, suffixoverlap_results = \
      self.__df_copy_train(mdf_train, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])

      mdf_test[suffixcolumn] = mdf_test[column].copy()
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_train.rename(columns = {column : suffixcolumn}, inplace = True)
      mdf_test.rename(columns = {column : suffixcolumn}, inplace = True)

    #convert all values to either numeric or NaN
    mdf_train[suffixcolumn] = pd.to_numeric(mdf_train[suffixcolumn], errors='coerce')
    mdf_test[suffixcolumn] = pd.to_numeric(mdf_test[suffixcolumn], errors='coerce')
    
    #a few more metrics collected for driftreport
    #get maximum value of training column
    maximum = mdf_train[suffixcolumn].max()
    #get minimum value of training column
    minimum = mdf_train[suffixcolumn].min()
    #median and median absolute deviation
    median = mdf_train[suffixcolumn].median()
    # MAD = mdf_train[suffixcolumn].mad()
    MAD = (mdf_train[suffixcolumn] - mdf_train[suffixcolumn].median(axis=0)).abs().median()
    
    #if cap/floor set based on number of standard deviations, measure statistics and set variables cap and/or floor
    if stdev_cap is not False and cap is False and floor is False:
      
      pre_capfloor_mean = mdf_train[suffixcolumn].mean()
      pre_capfloor_stdev = mdf_train[suffixcolumn].std()
      
      if pre_capfloor_mean != pre_capfloor_mean:
        pre_capfloor_mean = 0
      if pre_capfloor_stdev != pre_capfloor_stdev:
        pre_capfloor_stdev = 1

      cap = pre_capfloor_mean + stdev_cap * pre_capfloor_stdev
      floor = pre_capfloor_mean - stdev_cap * pre_capfloor_stdev
    
    #if cap < maximum, maximum = cap
    if cap is not False and cap is not True:
      if cap < maximum:
        maximum = cap
    if floor is not False and floor is not True:
      if floor > minimum:
        minimum = floor
        
    #cap and floor application
    if cap is True:
      cap = maximum
    if floor is True:
      floor = minimum
      
    if cap is not False:
      #replace values in test > cap with cap
      mdf_train.loc[mdf_train[suffixcolumn] > cap, (suffixcolumn)] \
      = cap
      
      mdf_test.loc[mdf_test[suffixcolumn] > cap, (suffixcolumn)] \
      = cap
    
    if floor is not False:
      #replace values in test < floor with floor
      mdf_train.loc[mdf_train[suffixcolumn] < floor, (suffixcolumn)] \
      = floor
      
      mdf_test.loc[mdf_test[suffixcolumn] < floor, (suffixcolumn)] \
      = floor

    #get mean of training data
    mean = mdf_train[suffixcolumn].mean()
    if mean != mean:
      mean = 0

    #subtract mean from column for both train and test
    mdf_train[suffixcolumn] = mdf_train[suffixcolumn] - mean
    mdf_test[suffixcolumn] = mdf_test[suffixcolumn] - mean

    #get standard deviation of training data
    std = mdf_train[suffixcolumn].std()
    
    #special case, if standard deviation is 0 we'll set it to 1 to avoid division by 0
    if std == 0:
      std = 1

    #divide column values by std for both training and test data
    #offset, multiplier are parameters that defaults to zero, one
    mdf_train[suffixcolumn] = mdf_train[suffixcolumn] / std * multiplier + offset
    mdf_test[suffixcolumn] = mdf_test[suffixcolumn] / std * multiplier + offset

    #replace any negative zero floats with positive zero. Negative zero is reserved for default infill
    if abs_zero is True:
      mdf_train = self.__autowhere(mdf_train, suffixcolumn, mdf_train[suffixcolumn] == 0, 0, specified='replacement')
      mdf_test = self.__autowhere(mdf_test, suffixcolumn, mdf_test[suffixcolumn] == 0, 0, specified='replacement')

    #apply defaultinfill based on processdict entry
    #this will default to negzeroinfill
    mdf_train, defaultinfill_dict = \
    self.__apply_defaultinfill(mdf_train, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    mdf_test, _1 = \
    self.__apply_defaultinfill(mdf_test, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)

#     #change data type for memory savings
#     mdf_train[suffixcolumn] = mdf_train[suffixcolumn].astype(np.float32)
#     mdf_test[suffixcolumn] = mdf_test[suffixcolumn].astype(np.float32)

    #create list of columns
    nmbrcolumns = [suffixcolumn]

    nmbrnormalization_dict = {suffixcolumn : {'mean' : mean, 'std' : std, \
                                              'median' : median, 'MAD' : MAD, \
                                              'max' : maximum, 'min' : minimum, \
                                              'offset' : offset, 'multiplier': multiplier, \
                                              'cap' : cap, 'floor' : floor, \
                                              'abs_zero' : abs_zero, \
                                              'inplace' : inplace, 'suffix' : suffix,
                                              'defaultinfill_dict' : defaultinfill_dict}}

    #store some values in the nmbr_dict{} for use later in ML infill methods
    column_dict_list = []

    for nc in nmbrcolumns:

      if nc != nmbrcolumns[0]:
        nmbrnormalization_dict = {}

      column_dict = { nc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : nmbrnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : nmbrcolumns, \
                           'categorylist' : nmbrcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
        
    return mdf_train, mdf_test, column_dict_list
  
  def _process_dxdt(self, df, column, category, treecategory, postprocess_dict, params = {}):
    '''
    #process_dxdt(df, column, category, postprocess_dict)
    #function to translate a continues variable into a bounded variable
    #by taking delta of row from preceding row
    #assumes the rows are not shuffled and represent a continuous funciton 
    #with consistent time steps
    
    #for missing values, uses adjacent cell infill as default
    '''

    suffixoverlap_results = {}
    
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False
    
    #initialize parameters
    if 'periods' in params:
      periods = params['periods']
    else:
      periods = 1

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory

    suffixcolumn = column + '_' + suffix
    
    if inplace is not True:
      
      #copy source column into new column
      df, suffixoverlap_results = \
      self.__df_copy_train(df, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(df, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      df.rename(columns = {column : suffixcolumn}, inplace = True)
    
    #convert all values to either numeric or NaN
    df[suffixcolumn] = pd.to_numeric(df[suffixcolumn], errors='coerce')

    #apply defaultinfill based on processdict entry
    df, defaultinfill_dict = \
    self.__apply_defaultinfill(df, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    
    #subtract preceding row
    df[suffixcolumn] = df[suffixcolumn] - df[suffixcolumn].shift(periods = periods)
    
    #first row will have a nan so just one more backfill
    df[suffixcolumn] = df[suffixcolumn].fillna(method='bfill')
    
    #then one more infill with to address scenario when data wasn't numeric
    #get arbitrary cell value, if one is nan then all will be
    value = df[suffixcolumn].iat[0]
    if value != value:
      value = 0

      df[suffixcolumn] = df[suffixcolumn].fillna(value)
    
    #create list of columns
    nmbrcolumns = [suffixcolumn]

    #grab some driftreport metrics
    #note that if this function implemented for data streams at scale it may be appropriate
    #to consider creating an alternate to dxdt without the driftreport metrics for postmunge efficiency
    positiveratio = df[df[suffixcolumn] >= 0].shape[0] / df[suffixcolumn].shape[0]
    negativeratio = df[df[suffixcolumn] < 0].shape[0] / df[suffixcolumn].shape[0]
    zeroratio = df[df[suffixcolumn] == 0].shape[0] / df[suffixcolumn].shape[0]
    minimum = df[suffixcolumn].min()
    maximum = df[suffixcolumn].max()
    mean = df[suffixcolumn].mean()
    std = df[suffixcolumn].std()

    nmbrnormalization_dict = {suffixcolumn : {'positiveratio' : positiveratio, \
                                              'negativeratio' : negativeratio, \
                                              'zeroratio' : zeroratio, \
                                              'minimum' : minimum, \
                                              'maximum' : maximum, \
                                              'mean' : mean, \
                                              'std' : std, \
                                              'periods' : periods, \
                                              'inplace' : inplace,
                                              'defaultinfill_dict' : defaultinfill_dict}}

    #store some values in the nmbr_dict{} for use later in ML infill methods
    column_dict_list = []

    for nc in nmbrcolumns:

      if nc != nmbrcolumns[0]:
        nmbrnormalization_dict = {}
      
      column_dict = { nc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : nmbrnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : nmbrcolumns, \
                           'categorylist' : nmbrcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
        
    return df, column_dict_list

  def _process_dxd2(self, df, column, category, treecategory, postprocess_dict, params = {}):
    '''
    #process_dxd2(df, column, category, postprocess_dict)
    #function to translate a continues variable into a bounded variable
    #by taking delta of average of last two rows minus 
    #average of preceding two rows before that
    #should take a littel noise out of noisy data
    #assumes the rows are not shuffled and represent a continuous funciton 
    #with consistent time steps
    
    #for missing values, uses adjacent cell infill as default
    '''

    suffixoverlap_results = {}
    
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False
    
    #initialize parameters
    if 'periods' in params:
      periods = params['periods']
    else:
      periods = 2

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory

    suffixcolumn = column + '_' + suffix
    
    if inplace is not True:
      
      #copy source column into new column
      df, suffixoverlap_results = \
      self.__df_copy_train(df, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(df, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      df.rename(columns = {column : suffixcolumn}, inplace = True)
    
    #convert all values to either numeric or NaN
    df[suffixcolumn] = pd.to_numeric(df[suffixcolumn], errors='coerce')
    
    #apply defaultinfill based on processdict entry
    df, defaultinfill_dict = \
    self.__apply_defaultinfill(df, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    
#     #we're going to take difference of average of last two rows with two rows preceding
#     df[column + '_dxd2'] = (df[column + '_dxd2'] + df[column + '_dxd2'].shift()) / 2 \
#                            - ((df[column + '_dxd2'].shift(periods=2) + df[column + '_dxd2'].shift(periods=3)) / 2)

    suffixoverlap_results = \
    self.__df_check_suffixoverlap(df, [column + '_temp1'], suffixoverlap_results, postprocess_dict['printstatus'])

    df[column + '_temp1'] = df[suffixcolumn].copy()
    # df_train['number7_temp3'] = df_train['number7'].copy()

    for i in range(periods-1):
      df[column + '_temp1'] = df[column + '_temp1'] + df[suffixcolumn].shift(periods = i+1)

    df[suffixcolumn] = (df[column + '_temp1'] - df[column + '_temp1'].shift(periods = periods)) / periods
    
    #first row will have a nan so just one more backfill
    df[suffixcolumn] = df[suffixcolumn].fillna(method='bfill')
    
    #then one more infill with to address scenario when data wasn't numeric
    #get arbitrary cell value, if one is nan then all will be
    value = df[suffixcolumn].iat[0]
    if value != value:
      value = 0

      df[suffixcolumn] = df[suffixcolumn].fillna(value)
    
    del df[column + '_temp1']
    
    #create list of columns
    nmbrcolumns = [suffixcolumn]

    #grab some driftreport metrics
    #note that if this function implemented for data streams at scale it may be appropriate
    #to consider creating an alternate to dxd2 without the driftreport metrics for postmunge efficiency
    positiveratio = df[df[suffixcolumn] >= 0].shape[0] / df[suffixcolumn].shape[0]
    negativeratio = df[df[suffixcolumn] < 0].shape[0] / df[suffixcolumn].shape[0]
    zeroratio = df[df[suffixcolumn] == 0].shape[0] / df[suffixcolumn].shape[0]
    minimum = df[suffixcolumn].min()
    maximum = df[suffixcolumn].max()
    mean = df[suffixcolumn].mean()
    std = df[suffixcolumn].std()
  
    nmbrnormalization_dict = {suffixcolumn : {'positiveratio' : positiveratio, \
                                              'negativeratio' : negativeratio, \
                                              'zeroratio' : zeroratio, \
                                              'minimum' : minimum, \
                                              'maximum' : maximum, \
                                              'mean' : mean, \
                                              'std' : std, \
                                              'periods' : periods, \
                                              'inplace' : inplace,
                                              'suffix' : suffix,
                                              'defaultinfill_dict' : defaultinfill_dict}}

    #store some values in the nmbr_dict{} for use later in ML infill methods
    column_dict_list = []

    for nc in nmbrcolumns:

      if nc != nmbrcolumns[0]:
        nmbrnormalization_dict = {}

      column_dict = { nc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : nmbrnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : nmbrcolumns, \
                           'categorylist' : nmbrcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
        
    return df, column_dict_list

  def _process_shft(self, df, column, category, treecategory, postprocess_dict, params = {}):
    '''
    #process_shft(df, column, category, postprocess_dict)
    #function to shift a sequential set forward by one or more time steps    
    #for missing values, uses adjacent cell infill as default
    #accepts parameter 'periods' for number of time steps, defaults to one
    #accepts parameter 'suffix' for column suffix appender
    #such as may be useful if applying this transform to the same column more than once

    #shft family of transforms is special case in the library since an aggregated NArw
    #wouldn't align with missing data entries (as rows are shifted by some period)
    #since we need to set NArowtype to exclude (which prevents aggregation of missing data markers)
    #we will omit defaultinfill support which does not apply infill to exclude NArowtype
    #and apply adjinfill here in transform
    '''

    suffixoverlap_results = {}
    
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False
    
    #initialize parameters
    if 'periods' in params:
      periods = params['periods']
    else:
      periods = 1
    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    shft_column = column + '_' + suffix
    
    if inplace is not True:
      
      #copy source column into new column
      df, suffixoverlap_results = \
      self.__df_copy_train(df, column, shft_column, suffixoverlap_results, postprocess_dict['printstatus'])
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(df, shft_column, suffixoverlap_results, postprocess_dict['printstatus'])
      
      df.rename(columns = {column : shft_column}, inplace = True)
    
    #convert all values to either numeric or NaN
    df[shft_column] = pd.to_numeric(df[shft_column], errors='coerce')
    
    #shift from preceding row
    df[shft_column] = df[shft_column].shift(periods = periods)

    #apply ffill to replace NArows with value from adjacent cell in pre4ceding row
    df[shft_column] = df[shft_column].fillna(method='ffill')
    
    #we'll follow with a bfill just in case first row had a nan
    df[shft_column] = df[shft_column].fillna(method='bfill')

    #then a final infill in case data was non-numeric
    df[shft_column] = df[shft_column].fillna(0)

    #create list of columns
    nmbrcolumns = [shft_column]

    #grab some driftreport metrics
    #note that if this function implemented for data streams at scale it may be appropriate
    #to consider creating an alternate to dxdt without the driftreport metrics for postmunge efficiency
    positiveratio = df[df[shft_column] >= 0].shape[0] / df[shft_column].shape[0]
    negativeratio = df[df[shft_column] < 0].shape[0] / df[shft_column].shape[0]
    zeroratio = df[df[shft_column] == 0].shape[0] / df[shft_column].shape[0]
    minimum = df[shft_column].min()
    maximum = df[shft_column].max()
    mean = df[shft_column].mean()
    std = df[shft_column].std()

    nmbrnormalization_dict = {shft_column :      {'positiveratio' : positiveratio, \
                                                  'negativeratio' : negativeratio, \
                                                  'zeroratio' : zeroratio, \
                                                  'minimum' : minimum, \
                                                  'maximum' : maximum, \
                                                  'mean' : mean, \
                                                  'std' : std, \
                                                  'periods' : periods, \
                                                  'suffix' : suffix, \
                                                  'inplace' : inplace}}

    #store some values in the nmbr_dict{} for use later in ML infill methods
    column_dict_list = []

    for nc in nmbrcolumns:

      if nc != nmbrcolumns[0]:
        nmbrnormalization_dict = {}
      
      column_dict = { nc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : nmbrnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : nmbrcolumns, \
                           'categorylist' : nmbrcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
        
    return df, column_dict_list

  def _process_MADn(self, mdf_train, mdf_test, column, category, \
                              treecategory, postprocess_dict, params = {}):
    '''
    #process_MADn(mdf_train, mdf_test, column, category)
    #function to normalize data to mean of 0 and mean absolute deviation of 1
    #takes as arguement pandas dataframe of training and test data (mdf_train), (mdf_test)\
    #and the name of the column string ('column') and parent category (category)
    #replaces missing or improperly formatted data with mean of remaining values
    #returns same dataframes with new column of name column + '_MADn'
    #note this is a "dualprocess" function since is applied to both train and test dataframes
    #expect this approach works better than z-score for when the numerical distribution isn't thin tailed
    #if only have training but not test data handy, use same training data for both dataframe inputs
    '''

    suffixoverlap_results = {}
    
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    #center can be passed as 'mean' or 'max', which serves as subtractor in numerator of normalization 
    #MADn applies mean, MAD3 applies max (max option inspired by discussions at Real World Risk Institute)
    if 'center' in params:
      center = params['center']
    else:
      center = 'mean'
      
    suffixcolumn = column + '_' + suffix
    
    if inplace is not True:
      
      #copy source column into new column
      mdf_train, suffixoverlap_results = \
      self.__df_copy_train(mdf_train, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])

      mdf_test[suffixcolumn] = mdf_test[column].copy()
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_train.rename(columns = {column : suffixcolumn}, inplace = True)
      mdf_test.rename(columns = {column : suffixcolumn}, inplace = True)

    #convert all values to either numeric or NaN
    mdf_train[suffixcolumn] = pd.to_numeric(mdf_train[suffixcolumn], errors='coerce')
    mdf_test[suffixcolumn] = pd.to_numeric(mdf_test[suffixcolumn], errors='coerce')
    
    #a few more metrics collected for driftreport
    #get maximum value of training column
    maximum = mdf_train[suffixcolumn].max()
    #get minimum value of training column
    minimum = mdf_train[suffixcolumn].min()
    median = mdf_train[suffixcolumn].median()

    #get mean of training data
    mean = mdf_train[suffixcolumn].mean() 
    if mean != mean:
      mean = 0

    #apply defaultinfill based on processdict entry
    mdf_train, defaultinfill_dict = \
    self.__apply_defaultinfill(mdf_train, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    mdf_test, _1 = \
    self.__apply_defaultinfill(mdf_test, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)
    
    if center == 'mean':

      #subtract mean from column for both train and test
      mdf_train[suffixcolumn] = mdf_train[suffixcolumn] - mean
      mdf_test[suffixcolumn] = mdf_test[suffixcolumn] - mean
      
    if center == 'max':
      
      if maximum != maximum:
        maximum = 0

      #subtract maximum from column for both train and test
      mdf_train[suffixcolumn] = mdf_train[suffixcolumn] - maximum
      mdf_test[suffixcolumn] = mdf_test[suffixcolumn] - maximum

    #get mean absolute deviation of training data
    # MAD = mdf_train[suffixcolumn].mad()
    MAD = (mdf_train[suffixcolumn] - mdf_train[suffixcolumn].median(axis=0)).abs().median()
    
    #special case to avoid div by 0
    if MAD == 0:
      MAD = 1

    #divide column values by mad for both training and test data
    mdf_train[suffixcolumn] = mdf_train[suffixcolumn] / MAD
    mdf_test[suffixcolumn] = mdf_test[suffixcolumn] / MAD

#     #change data type for memory savings
#     mdf_train[suffixcolumn] = mdf_train[suffixcolumn].astype(np.float32)
#     mdf_test[suffixcolumn] = mdf_test[suffixcolumn].astype(np.float32)

    #create list of columns
    nmbrcolumns = [suffixcolumn]

    nmbrnormalization_dict = {suffixcolumn : {'mean' : mean, 'MAD' : MAD, 'center' : center, \
                                              'maximum':maximum, 'minimum':minimum, 'median' : median, \
                                              'inplace' : inplace, 'suffix' : suffix, \
                                              'defaultinfill_dict' : defaultinfill_dict}}

    #store some values in the nmbr_dict{} for use later in ML infill methods
    column_dict_list = []

    for nc in nmbrcolumns:

      if nc != nmbrcolumns[0]:
        nmbrnormalization_dict = {}

      column_dict = { nc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : nmbrnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : nmbrcolumns, \
                           'categorylist' : nmbrcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
    
    return mdf_train, mdf_test, column_dict_list

  def _process_mnmx(self, mdf_train, mdf_test, column, category, \
                         treecategory, postprocess_dict, params = {}):
    '''
    #process_mnmx(mdf_train, mdf_test, column, category)
    #function to scale data to minimum of 0 and maximum of 1 \
    #based on min/max values from training set for this column
    #takes as arguement pandas dataframe of training and test data (mdf_train), (mdf_test)\
    #and the name of the column string ('column') and parent category (category)
    #replaces missing or improperly formatted data with mean of remaining values
    #returns same dataframes with new column of name suffixcolumn
    #note this is a "dualprocess" function since is applied to both dataframes
    #expect this approach works better when the numerical distribution is thin tailed
    #if only have training but not test data handy, use same training data for both
    #dataframe inputs
    '''

    suffixoverlap_results = {}
    
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False
    
    #for cap ands floor, False means not applied, True means based on set's found max/min in train set
    
    #initialize parameters
    #cap can be passed as True for max of training data or as a specific value prior to normalizaiton, False for no cap
    if 'cap' in params:
      cap = params['cap']
    else:
      cap = False
      
    #floor can be passed as True for min of training data or as a specific value prior to normalizaiton, False for no floor
    if 'floor' in params:
      floor = params['floor']
    else:
      floor = False
      
    #stdev_cap can be passed as integer or float for setting cap based on number of standard deviations from mean
    #only performed with cap=False and floor=False, default of stdev_cap=False deactivates
    if 'stdev_cap' in params:
      stdev_cap = params['stdev_cap']
    else:
      stdev_cap = False

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    suffixcolumn = column + '_' + suffix
    
    if inplace is not True:
      
      #copy source column into new column
      mdf_train, suffixoverlap_results = \
      self.__df_copy_train(mdf_train, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])

      mdf_test[suffixcolumn] = mdf_test[column].copy()
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_train.rename(columns = {column : suffixcolumn}, inplace = True)
      mdf_test.rename(columns = {column : suffixcolumn}, inplace = True)

    #convert all values to either numeric or NaN
    mdf_train[suffixcolumn] = pd.to_numeric(mdf_train[suffixcolumn], errors='coerce')
    mdf_test[suffixcolumn] = pd.to_numeric(mdf_test[suffixcolumn], errors='coerce')
    
    #a few more metrics collected for driftreport
    #get standard deviation of training data
    std = mdf_train[suffixcolumn].std()
    if std != std or std == 0:
      std = 1

    #get mean of training data
    mean = mdf_train[suffixcolumn].mean()   
    if mean != mean:
      mean = 0

    #additional drift stats
    median = mdf_train[suffixcolumn].median()
    # MAD = mdf_train[suffixcolumn].mad()
    MAD = (mdf_train[suffixcolumn] - mdf_train[suffixcolumn].median(axis=0)).abs().median()

    #apply defaultinfill based on processdict entry
    mdf_train, defaultinfill_dict = \
    self.__apply_defaultinfill(mdf_train, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    mdf_test, _1 = \
    self.__apply_defaultinfill(mdf_test, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)
    
    #get maximum value of training column
    maximum = mdf_train[suffixcolumn].max()
    
    #get minimum value of training column
    minimum = mdf_train[suffixcolumn].min()
    
    #if cap/floor set based on number of standard deviations, measure statistics and set variables cap and/or floor
    if stdev_cap is not False and cap is False and floor is False:
      
      pre_capfloor_mean = mean
      pre_capfloor_stdev = std
      
      if pre_capfloor_mean != pre_capfloor_mean:
        pre_capfloor_mean = 0
      if pre_capfloor_stdev != pre_capfloor_stdev:
        pre_capfloor_stdev = 1

      cap = pre_capfloor_mean + stdev_cap * pre_capfloor_stdev
      floor = pre_capfloor_mean - stdev_cap * pre_capfloor_stdev
    
    #if cap < maximum, maximum = cap
    if cap is not False and cap is not True:
      if cap < maximum:
        maximum = cap
    if floor is not False and floor is not True:
      if floor > minimum:
        minimum = floor
    
    #avoid outlier div by zero when max = min
    maxminusmin = maximum - minimum
    if maxminusmin == 0:
      maxminusmin = 1
    
    #perform min-max scaling to train and test sets using values from train
    mdf_train[suffixcolumn] = (mdf_train[suffixcolumn] - minimum) / \
                                  (maxminusmin)
    
    mdf_test[suffixcolumn] = (mdf_test[suffixcolumn] - minimum) / \
                                 (maxminusmin)

    #cap and floor application
    if cap is True:
      cap = maximum
    if floor is True:
      floor = minimum
    
    if cap is not False:
      #replace values in test > cap with cap
      mdf_train.loc[mdf_train[suffixcolumn] > (cap - minimum)/maxminusmin, (suffixcolumn)] \
      = (cap - minimum)/maxminusmin
      
      mdf_test.loc[mdf_test[suffixcolumn] > (cap - minimum)/maxminusmin, (suffixcolumn)] \
      = (cap - minimum)/maxminusmin
    
    if floor is not False:
      #replace values in test < floor with floor
      mdf_train.loc[mdf_train[suffixcolumn] < (floor - minimum)/maxminusmin, (suffixcolumn)] \
      = (floor - minimum)/maxminusmin
      
      mdf_test.loc[mdf_test[suffixcolumn] < (floor - minimum)/maxminusmin, (suffixcolumn)] \
      = (floor - minimum)/maxminusmin
    
    #create list of columns
    nmbrcolumns = [suffixcolumn]

    nmbrnormalization_dict = {suffixcolumn : {'minimum' : minimum, \
                                              'maximum' : maximum, \
                                              'maxminusmin' : maxminusmin, \
                                              'mean' : mean, \
                                              'std' : std, \
                                              'median' : median, \
                                              'MAD' : MAD, \
                                              'cap' : cap, \
                                              'floor' : floor, \
                                              'suffix' : suffix, \
                                              'inplace' : inplace, \
                                              'defaultinfill_dict' : defaultinfill_dict}}

    #store some values in the nmbr_dict{} for use later in ML infill methods
    column_dict_list = []

    for nc in nmbrcolumns:

      if nc != nmbrcolumns[0]:
        nmbrnormalization_dict = {}

      column_dict = { nc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : nmbrnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : nmbrcolumns, \
                           'categorylist' : nmbrcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
        
    return mdf_train, mdf_test, column_dict_list

  def _process_mnm3(self, mdf_train, mdf_test, column, category, \
                         treecategory, postprocess_dict, params = {}):
    '''
    #process_mnmx(mdf_train, mdf_test, column, category)
    #function to scale data to minimum of 0 and maximum of 1 \
    #after replacing extreme values above the 0.99 quantile with
    #the value of 0.99 quantile and extreme values below the 0.01
    #quantile with the value of 0.01 quantile
    #(accepts parameters qmax and qmin to customize these 0.99/0.01 values)
    #takes as arguement pandas dataframe of training and test data (mdf_train), (mdf_test)\
    #and the name of the column string ('column') and parent category (category)
    #replaces missing or improperly formatted data with mean of remaining values
    #returns same dataframes with new column of name suffixcolumn
    #note this is a "dualprocess" function since is applied to both dataframes
    '''

    suffixoverlap_results = {}
    
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False

    #initialize parameters
    if 'qmax' in params:
      qmax = params['qmax']
    else:
      qmax = 0.99
      
    if 'qmin' in params:
      qmin = params['qmin']
    else:
      qmin = 0.01

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    suffixcolumn = column + '_' + suffix
    
    if inplace is not True:
      
      #copy source column into new column
      mdf_train, suffixoverlap_results = \
      self.__df_copy_train(mdf_train, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])

      mdf_test[suffixcolumn] = mdf_test[column].copy()
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_train.rename(columns = {column : suffixcolumn}, inplace = True)
      mdf_test.rename(columns = {column : suffixcolumn}, inplace = True)

    #convert all values to either numeric or NaN
    mdf_train[suffixcolumn] = pd.to_numeric(mdf_train[suffixcolumn], errors='coerce')
    mdf_test[suffixcolumn] = pd.to_numeric(mdf_test[suffixcolumn], errors='coerce')
    
    #a few more metrics collected for driftreport
    #get standard deviation of training data
    std = mdf_train[suffixcolumn].std()

    #get maximum value of training column
    quantilemax = mdf_train[suffixcolumn].quantile(qmax)
    
    if quantilemax != quantilemax:
      quantilemax = 0

    #get minimum value of training column
    quantilemin = mdf_train[suffixcolumn].quantile(qmin)
    
    if quantilemin != quantilemin:
      quantilemin = 0

    #replace values > quantilemax with quantilemax
    mdf_train.loc[mdf_train[suffixcolumn] > quantilemax, (suffixcolumn)] \
    = quantilemax
    mdf_test.loc[mdf_test[suffixcolumn] > quantilemax, (suffixcolumn)] \
    = quantilemax
    #replace values < quantile10 with quantile10
    mdf_train.loc[mdf_train[suffixcolumn] < quantilemin, (suffixcolumn)] \
    = quantilemin
    mdf_test.loc[mdf_test[suffixcolumn] < quantilemin, (suffixcolumn)] \
    = quantilemin

    #note this step is now performed after the quantile evaluation / replacement

    #get mean of training data
    mean = mdf_train[suffixcolumn].mean()    
    if mean != mean:
      mean = 0
    
    #apply defaultinfill based on processdict entry
    mdf_train, defaultinfill_dict = \
    self.__apply_defaultinfill(mdf_train, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    mdf_test, _1 = \
    self.__apply_defaultinfill(mdf_test, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)
    
    #avoid outlier div by zero when max = min
    maxminusmin = quantilemax - quantilemin
    if maxminusmin == 0:
      maxminusmin = 1

    #perform min-max scaling to train and test sets using values from train
    mdf_train[suffixcolumn] = (mdf_train[suffixcolumn] - quantilemin) / \
                                  (maxminusmin)

    mdf_test[suffixcolumn] = (mdf_test[suffixcolumn] - quantilemin) / \
                                 (maxminusmin)

#     #change data type for memory savings
#     mdf_train[suffixcolumn] = mdf_train[suffixcolumn].astype(np.float32)
#     mdf_test[suffixcolumn] = mdf_test[suffixcolumn].astype(np.float32)
    
    #create list of columns
    nmbrcolumns = [suffixcolumn]

    nmbrnormalization_dict = {suffixcolumn : {'quantilemin' : quantilemin, \
                                              'quantilemax' : quantilemax, \
                                              'maxminusmin' : maxminusmin, \
                                              'mean' : mean, \
                                              'std' : std, \
                                              'qmax' : qmax, \
                                              'qmin' : qmin, \
                                              'suffix' : suffix, \
                                              'defaultinfill_dict' : defaultinfill_dict, \
                                              'inplace' : inplace}}

    #store some values in the nmbr_dict{} for use later in ML infill methods
    column_dict_list = []

    for nc in nmbrcolumns:

      if nc != nmbrcolumns[0]:
        nmbrnormalization_dict = {}

      column_dict = { nc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : nmbrnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : nmbrcolumns, \
                           'categorylist' : nmbrcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)

    return mdf_train, mdf_test, column_dict_list

  def _process_mxab(self, mdf_train, mdf_test, column, category, \
                         treecategory, postprocess_dict, params = {}):
    '''
    #process_mxab(mdf_train, mdf_test, column, category)
    #function to scale data to minimum of -1 and maximum of 1 \
    #based on division by max absolute values from training set.
    '''

    suffixoverlap_results = {}
    
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    suffixcolumn = column + '_' + suffix
    
    if inplace is not True:
      
      #copy source column into new column
      mdf_train, suffixoverlap_results = \
      self.__df_copy_train(mdf_train, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])

      mdf_test[suffixcolumn] = mdf_test[column].copy()
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_train.rename(columns = {column : suffixcolumn}, inplace = True)
      mdf_test.rename(columns = {column : suffixcolumn}, inplace = True)

    #convert all values to either numeric or NaN
    mdf_train[suffixcolumn] = pd.to_numeric(mdf_train[suffixcolumn], errors='coerce')
    mdf_test[suffixcolumn] = pd.to_numeric(mdf_test[suffixcolumn], errors='coerce')
    
    #a few more metrics collected for driftreport
    #get standard deviation of training data
    std = mdf_train[suffixcolumn].std()
    median = mdf_train[suffixcolumn].median()
    # MAD = mdf_train[suffixcolumn].mad()
    MAD = (mdf_train[suffixcolumn] - mdf_train[suffixcolumn].median(axis=0)).abs().median()

    #get mean of training data
    mean = mdf_train[suffixcolumn].mean()   
    if mean != mean:
      mean = 0

    #apply defaultinfill based on processdict entry
    mdf_train, defaultinfill_dict = \
    self.__apply_defaultinfill(mdf_train, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    mdf_test, _1 = \
    self.__apply_defaultinfill(mdf_test, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)
    
    #get maximum value of training column
    maximum = mdf_train[suffixcolumn].max()
    
    #get minimum value of training column
    minimum = mdf_train[suffixcolumn].min()
    
    #get max absolute
    maxabs = max(abs(maximum), abs(minimum))
    
    #avoid outlier div by zero when max = min
    if maxabs == 0:
      maxabs = 1
    
    #perform maxabs scaling to train and test sets using values from train
    mdf_train[suffixcolumn] = mdf_train[suffixcolumn] / \
                                  (maxabs)
    
    mdf_test[suffixcolumn] = mdf_test[suffixcolumn] / \
                                 (maxabs)
    
    #create list of columns
    nmbrcolumns = [suffixcolumn]

    nmbrnormalization_dict = {suffixcolumn : {'minimum' : minimum, \
                                              'maximum' : maximum, \
                                              'maxabs' : maxabs, \
                                              'mean' : mean, \
                                              'std' : std, \
                                              'median' : median, \
                                              'MAD' : MAD, \
                                              'suffix' : suffix, \
                                              'defaultinfill_dict' : defaultinfill_dict, 
                                              'inplace' : inplace}}

    #store some values in the nmbr_dict{} for use later in ML infill methods
    column_dict_list = []

    for nc in nmbrcolumns:

      if nc != nmbrcolumns[0]:
        nmbrnormalization_dict = {}

      column_dict = { nc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : nmbrnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : nmbrcolumns, \
                           'categorylist' : nmbrcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
        
    return mdf_train, mdf_test, column_dict_list
  
  def _process_retn(self, mdf_train, mdf_test, column, category, treecategory, postprocess_dict, params = {}):
    """
    #process_retn(mdf_train, mdf_test, column, category)
    #function to scale data as follows:
    
    # if max >= 0 and min <= 0:
    #   #scaling based on 
    #   x = x / (max - min)

    # elif max >= 0 and min >= 0:
    #   #traditional min/max
    #   x = (x - min) / (max - min)
    
    # elif max <= 0 and min <= 0:
    #   #max/min (retains negative values)
    #   x = (x - max) / (max - min)
    
    #replaces missing or improperly formatted data with mean of remaining values
    
    #returns same dataframes with new column of name suffixcolumn
    #note this is a "dualprocess" function since is applied to both dataframes
    
    #note with parameters divisor can also be set as standard deviation
    #also aprameters accepted for cap/floor/mulitplier/offset
    #where cap/floor based on pretransform values
    #multiplier/offset based on posttransform values, muoltiplier applied betfore offset
    """

    suffixoverlap_results = {}
    
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False
    
    #initialize parameters
    
    #accepts divisor parameters of 'minmax' or 'std', eg divisor for normalization equation
    #note that standard deviation doesn't have same properties for sign retention when all values > or < 0
    if 'divisor' in params:
      divisor = params['divisor']
    else:
      divisor = 'minmax'
    
    #offset is just an added constant applied after multiplier
    if 'offset' in params:
      offset = params['offset']
    else:
      offset = 0
    
    #multiplier scales the set by multiplication prior to offset
    if 'multiplier' in params:
      multiplier = params['multiplier']
    else:
      multiplier = 1
    
    #cap can be passed as True for max of training data or as a specific value prior to normalization, False for no cap
    if 'cap' in params:
      cap = params['cap']
    else:
      cap = False
    
    #floor can be passed as True for min of training data or as a specific value prior to normalization, False for no floor
    if 'floor' in params:
      floor = params['floor']
    else:
      floor = False
      
    #stdev_cap can be passed as integer or float for setting cap based on number of standard deviations from mean
    #only performed with cap=False and floor=False, default of stdev_cap=False deactivates
    if 'stdev_cap' in params:
      stdev_cap = params['stdev_cap']
    else:
      stdev_cap = False

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    suffixcolumn = column + '_' + suffix
    
    if inplace is not True:
      
      #copy source column into new column
      mdf_train, suffixoverlap_results = \
      self.__df_copy_train(mdf_train, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])

      mdf_test[suffixcolumn] = mdf_test[column].copy()
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_train.rename(columns = {column : suffixcolumn}, inplace = True)
      mdf_test.rename(columns = {column : suffixcolumn}, inplace = True)

    #convert all values to either numeric or NaN
    mdf_train[suffixcolumn] = pd.to_numeric(mdf_train[suffixcolumn], errors='coerce')
    mdf_test[suffixcolumn] = pd.to_numeric(mdf_test[suffixcolumn], errors='coerce')
    
    #a few more metrics collected for driftreport
    #stdev, median, mad
    std = mdf_train[suffixcolumn].std()
    median = mdf_train[suffixcolumn].median()
    # mad = mdf_train[suffixcolumn].mad()
    mad = (mdf_train[suffixcolumn] - mdf_train[suffixcolumn].median(axis=0)).abs().median()
    
    #get maximum value of training column
    maximum = mdf_train[suffixcolumn].max()
    
    #get minimum value of training column
    minimum = mdf_train[suffixcolumn].min()
    
    #avoid outlier div by zero when max = min
    maxminusmin = maximum - minimum
    if maxminusmin == 0 or maxminusmin != maxminusmin:
      maxminusmin = 1
      
    if std != std or std == 0:
      std = 1
      
    if mad != mad or mad == 0:
      mad = 1
      
    #if cap/floor set based on number of standard deviations, measure statistics and set variables cap and/or floor
    if stdev_cap is not False and cap is False and floor is False:
      
      pre_capfloor_mean = mdf_train[suffixcolumn].mean()
      pre_capfloor_stdev = std
      
      if pre_capfloor_mean != pre_capfloor_mean:
        pre_capfloor_mean = 0
      if pre_capfloor_stdev != pre_capfloor_stdev:
        pre_capfloor_stdev = 1

      cap = pre_capfloor_mean + stdev_cap * pre_capfloor_stdev
      floor = pre_capfloor_mean - stdev_cap * pre_capfloor_stdev
      
    #if cap < maximum, maximum = cap
    if cap is not False and cap is not True:
      if cap < maximum:
        maximum = cap
    if floor is not False and floor is not True:
      if floor > minimum:
        minimum = floor
        
    #cap and floor application
    if cap is True:
      cap = maximum
    if floor is True:
      floor = minimum
      
    if cap is not False:
      #replace values in test > cap with cap
      mdf_train.loc[mdf_train[suffixcolumn] > cap, (suffixcolumn)] \
      = cap
      
      mdf_test.loc[mdf_test[suffixcolumn] > cap, (suffixcolumn)] \
      = cap
    
    if floor is not False:
      #replace values in test < floor with floor
      mdf_train.loc[mdf_train[suffixcolumn] < floor, (suffixcolumn)] \
      = floor
      
      mdf_test.loc[mdf_test[suffixcolumn] < floor, (suffixcolumn)] \
      = floor
      
    #get mean of training data
    mean = mdf_train[suffixcolumn].mean()
    if mean != mean:
      mean = 0

    #apply defaultinfill based on processdict entry
    mdf_train, defaultinfill_dict = \
    self.__apply_defaultinfill(mdf_train, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    mdf_test, _1 = \
    self.__apply_defaultinfill(mdf_test, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)
    
    #edge case (only neccesary so scalingapproach is assigned)
    if maximum != maximum:
      maximum = 0
    if minimum != minimum:
      minimum = 0
    
    #divisor
    if divisor not in {'minmax', 'std', 'mad'}:
      # if postprocess_dict['printstatus'] != 'silent':
      self.__autoprint('warning', "Error: retn transform parameter 'divisor' only accepts entries of 'minmax' 'mad' or 'std'")
      self.__autoprint('warning', '')
    if divisor == 'minmax':
      divisor = maxminusmin
    elif divisor == 'mad':
      divisor = mad
    else:
      divisor = std
      
    if divisor == 0 or divisor != divisor:
      divisor = 1
    
    #driftreport metric scalingapproach returned as 'retn' or 'mnmx' or 'mxmn'
    #where mnmx is for cases where all values in train set are positive
    #mxmn is for cases where all values in train set are negative
    
    if maximum >= 0 and minimum <= 0:
      
      mdf_train[suffixcolumn] = (mdf_train[suffixcolumn]) / \
                                    (divisor) * multiplier + offset
      
      mdf_test[suffixcolumn] = (mdf_test[suffixcolumn]) / \
                                    (divisor) * multiplier + offset
      
      scalingapproach = 'retn'
      
    elif maximum >= 0 and minimum >= 0:
    
      #perform min-max scaling to train and test sets using values from train
      mdf_train[suffixcolumn] = (mdf_train[suffixcolumn] - minimum) / \
                                    (divisor) * multiplier + offset

      mdf_test[suffixcolumn] = (mdf_test[suffixcolumn] - minimum) / \
                                   (divisor) * multiplier + offset
      
      scalingapproach = 'mnmx'
      
    elif maximum <= 0 and minimum <= 0:
    
      #perform min-max scaling to train and test sets using values from train
      mdf_train[suffixcolumn] = (mdf_train[suffixcolumn] - maximum) / \
                                    (divisor) * multiplier + offset

      mdf_test[suffixcolumn] = (mdf_test[suffixcolumn] - maximum) / \
                                   (divisor) * multiplier + offset
      
      scalingapproach = 'mxmn'

    #create list of columns
    nmbrcolumns = [suffixcolumn]

    nmbrnormalization_dict = {suffixcolumn : {'minimum' : minimum, \
                                              'maximum' : maximum, \
                                              'mean' : mean, \
                                              'std' : std, \
                                              'median' : median, \
                                              'MAD' : mad, \
                                              'scalingapproach' : scalingapproach, \
                                              'offset' : offset, \
                                              'multiplier': multiplier, \
                                              'cap' : cap, \
                                              'floor' : floor, \
                                              'divisor' : divisor, \
                                              'suffix' : suffix, \
                                              'defaultinfill_dict' : defaultinfill_dict, 
                                              'inplace' : inplace}}

    #store some values in the nmbr_dict{} for use later in ML infill methods
    column_dict_list = []

    for nc in nmbrcolumns:

      if nc != nmbrcolumns[0]:
        nmbrnormalization_dict = {}

      column_dict = { nc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : nmbrnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : nmbrcolumns, \
                           'categorylist' : nmbrcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
        
    return mdf_train, mdf_test, column_dict_list

  def _process_mean(self, mdf_train, mdf_test, column, category, treecategory, postprocess_dict, params = {}):
    '''
    #process_mean(mdf_train, mdf_test, column, category)
    #function to scale data to minimum of 0 and maximum of 1 \
    #based on min/max values from training set for this column
    #takes as arguement pandas dataframe of training and test data (mdf_train), (mdf_test)\
    #and the name of the column string ('column') and parent category (category)
    #replaces missing or improperly formatted data with mean of remaining values
    #returns same dataframes with new column of name suffixcolumn
    #note this is a "dualprocess" function since is applied to both dataframes
    #expect this approach works better when the numerical distribution is thin tailed
    #if only have training but not test data handy, use same training data for both
    #dataframe inputs
    '''

    suffixoverlap_results = {}
    
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False
    
    #initialize parameters
    if 'offset' in params:
      offset = params['offset']
    else:
      offset = 0
      
    if 'multiplier' in params:
      multiplier = params['multiplier']
    else:
      multiplier = 1
    
    if 'cap' in params:
      cap = params['cap']
    else:
      cap = False
      
    if 'floor' in params:
      floor = params['floor']
    else:
      floor = False
      
    #stdev_cap can be passed as integer or float for setting cap based on number of standard deviations from mean
    #only performed with cap=False and floor=False, default of stdev_cap=False deactivates
    if 'stdev_cap' in params:
      stdev_cap = params['stdev_cap']
    else:
      stdev_cap = False

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    suffixcolumn = column + '_' + suffix
    
    if inplace is not True:
      
      #copy source column into new column
      mdf_train, suffixoverlap_results = \
      self.__df_copy_train(mdf_train, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])

      mdf_test[suffixcolumn] = mdf_test[column].copy()
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_train.rename(columns = {column : suffixcolumn}, inplace = True)
      mdf_test.rename(columns = {column : suffixcolumn}, inplace = True)

    #convert all values to either numeric or NaN
    mdf_train[suffixcolumn] = pd.to_numeric(mdf_train[suffixcolumn], errors='coerce')
    mdf_test[suffixcolumn] = pd.to_numeric(mdf_test[suffixcolumn], errors='coerce')
    
    #get maximum value of training column
    maximum = mdf_train[suffixcolumn].max()
    
    #get minimum value of training column
    minimum = mdf_train[suffixcolumn].min()
    
    #if cap/floor set based on number of standard deviations, measure statistics and set variables cap and/or floor
    if stdev_cap is not False and cap is False and floor is False:
      
      pre_capfloor_mean = mdf_train[suffixcolumn].mean()
      pre_capfloor_stdev = mdf_train[suffixcolumn].std()
      
      if pre_capfloor_mean != pre_capfloor_mean:
        pre_capfloor_mean = 0
      if pre_capfloor_stdev != pre_capfloor_stdev:
        pre_capfloor_stdev = 1

      cap = pre_capfloor_mean + stdev_cap * pre_capfloor_stdev
      floor = pre_capfloor_mean - stdev_cap * pre_capfloor_stdev
    
    #if cap < maximum, maximum = cap
    if cap is not False and cap is not True:
      if cap < maximum:
        maximum = cap
    if floor is not False and floor is not True:
      if floor > minimum:
        minimum = floor
        
    #cap and floor application
    if cap is True:
      cap = maximum
    if floor is True:
      floor = minimum
      
    if cap is not False:
      #replace values in test > cap with cap
      mdf_train.loc[mdf_train[suffixcolumn] > cap, (suffixcolumn)] \
      = cap
      
      mdf_test.loc[mdf_test[suffixcolumn] > cap, (suffixcolumn)] \
      = cap
    
    if floor is not False:
      #replace values in test < floor with floor
      mdf_train.loc[mdf_train[suffixcolumn] < floor, (suffixcolumn)] \
      = floor
      
      mdf_test.loc[mdf_test[suffixcolumn] < floor, (suffixcolumn)] \
      = floor
    
    #a few more metrics collected for driftreport
    #get standard deviation of training data
    std = mdf_train[suffixcolumn].std()
    #median and median absolute deviation
    median = mdf_train[suffixcolumn].median()
    # MAD = mdf_train[suffixcolumn].mad()
    MAD = (mdf_train[suffixcolumn] - mdf_train[suffixcolumn].median(axis=0)).abs().median()

    #get mean of training data
    mean = mdf_train[suffixcolumn].mean()
    if mean != mean:
      mean = 0
      
    #apply defaultinfill based on processdict entry
    mdf_train, defaultinfill_dict = \
    self.__apply_defaultinfill(mdf_train, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    mdf_test, _1 = \
    self.__apply_defaultinfill(mdf_test, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)
    
    #avoid outlier div by zero when max = min
    maxminusmin = maximum - minimum
    if maxminusmin == 0 or maxminusmin != maxminusmin:
      maxminusmin = 1
    
    #perform min-max scaling to train and test sets using values from train
    mdf_train[suffixcolumn] = (mdf_train[suffixcolumn] - mean) / \
                                  (maxminusmin) * multiplier + offset
    
    mdf_test[suffixcolumn] = (mdf_test[suffixcolumn] - mean) / \
                                 (maxminusmin) * multiplier + offset

#     #change data type for memory savings
#     mdf_train[suffixcolumn] = mdf_train[suffixcolumn].astype(np.float32)
#     mdf_test[suffixcolumn] = mdf_test[suffixcolumn].astype(np.float32)
    
    #create list of columns
    nmbrcolumns = [suffixcolumn]

    nmbrnormalization_dict = {suffixcolumn : {'minimum' : minimum, \
                                              'maximum' : maximum, \
                                              'maxminusmin' : maxminusmin, \
                                              'mean' : mean, \
                                              'std' : std, \
                                              'median' : median, \
                                              'MAD' : MAD, \
                                              'offset' : offset, \
                                              'multiplier': multiplier, \
                                              'cap' : cap, \
                                              'floor' : floor, \
                                              'suffix' : suffix, \
                                              'defaultinfill_dict' : defaultinfill_dict,
                                              'inplace' : inplace}}

    #store some values in the nmbr_dict{} for use later in ML infill methods
    column_dict_list = []

    for nc in nmbrcolumns:

      if nc != nmbrcolumns[0]:
        nmbrnormalization_dict = {}

      column_dict = { nc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : nmbrnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : nmbrcolumns, \
                           'categorylist' : nmbrcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
        
    return mdf_train, mdf_test, column_dict_list

  def _process_binary(self, mdf_train, mdf_test, column, category, \
                           treecategory, postprocess_dict, params = {}):
    '''
    #process_binary(mdf, column, missing)
    #converts binary classification values to 0 or 1
    #takes as arguement a pandas dataframe (mdf_train, mdf_test), \
    #the name of the column string ('column') \
    #and the category from parent columkn (category)
    #fills missing valules with most common value
    #returns same dataframes with new column of name suffixcolumn
    #note this is a "dualprocess" function since is applied to both dataframes
    '''

    suffixoverlap_results = {}
    
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False
      
    #str_convert provides consistent encodings between numbers and string equivalent, eg 2 == '2'
    #str convert defaults to True to avoid edge case associated with bytes type entries
    #accepts one of {True, False, 'conditional_on_bytes'}
    #where conditional_on_bytes sets as False unless bytes entries are present in train data then sets to True
    if 'str_convert' in params:
      str_convert = params['str_convert']
    else:
      str_convert = True

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    #can pass as 'onevalue' or 'zerovalue'
    #where onevalue used for bnry and zerovalue used for bnr2
    if 'infillconvention' in params:
      infillconvention = params['infillconvention']
    else:
      infillconvention = 'onevalue'

    if 'invert' in params:
      invert = params['invert']
    else:
      invert = False
      
    suffixcolumn = column + '_' + suffix
    
    if inplace is not True:
      
      #copy source column into new column
      mdf_train, suffixoverlap_results = \
      self.__df_copy_train(mdf_train, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])

      mdf_test[suffixcolumn] = mdf_test[column].copy()
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_train.rename(columns = {column : suffixcolumn}, inplace = True)
      mdf_test.rename(columns = {column : suffixcolumn}, inplace = True)
    
    #conditional_on_bytes str_convert scenario resets to True if train set bytes entries present, otherwise resets to False
    if str_convert == 'conditional_on_bytes' \
    and (mdf_train[suffixcolumn] == mdf_train[suffixcolumn].astype(bytes)).sum() > 0:
      str_convert = True
    elif str_convert == 'conditional_on_bytes':
      str_convert = False

    if str_convert is True:
      mdf_train[suffixcolumn] = mdf_train[suffixcolumn].astype(str)
      mdf_test[suffixcolumn] = mdf_test[suffixcolumn].astype(str)

    tempstring = 'asdf'
    if tempstring in mdf_train:
      while tempstring in mdf_train:
        tempstring += 'z'
    #valuecounts is a list of unique entries sorted by frequency (from most to least) and then alphabetic, excluding nan
    #note this sorting has an edge case for bytes type entries when str_convert passed as False
    valuecounts = pd.DataFrame(mdf_train[suffixcolumn].value_counts())
    valuecounts = valuecounts.rename_axis(tempstring).sort_values(by = [suffixcolumn, tempstring], ascending = [False, True])
    valuecounts = list(valuecounts.index)
    
    if len(valuecounts) > 0:

      if len(valuecounts) > 1:
        if infillconvention == 'onevalue':
          binary_missing_plug = valuecounts[0]
        elif infillconvention == 'zerovalue':
          binary_missing_plug = valuecounts[1]

      if len(valuecounts) == 1:
        #making an executive decision here to deviate from standardinfill of most common value
        #for this edge case where a column evaluated as binary has only single value and NaN's
        if valuecounts[0] == valuecounts[0]:
          binary_missing_plug = np.nan
        else:
          binary_missing_plug = 'zzzinfill'
      #test for nan
      elif binary_missing_plug != binary_missing_plug:
        if infillconvention == 'onevalue':
          binary_missing_plug = valuecounts[1]
        elif infillconvention == 'zerovalue':
          binary_missing_plug = valuecounts[0]

      #edge case when applying this transform to set with >2 values
      #this only comes up when caluclating driftreport in postmunge
      extravalues = []
      if len(valuecounts) > 2:
        i=0
        for value in valuecounts:
          if i>1:
            extravalues.append(value)
          i+=1

      #replace nan in valuecounts with binary_missing_plug so we can sort
      valuecounts = [x if x == x else binary_missing_plug for x in valuecounts]
  #     #convert everything to string for sort
  #     valuecounts = [str(x) for x in valuecounts]

      #note LabelBinarizer encodes alphabetically, with 1 assigned to first and 0 to second
      #we'll take different approach of going by most common value to 1 unless 0 or 1
      #are already in the set then we'll defer to keeping those designations in place
      #there's some added complexity here to deal with edge case of passing this function
      #to a set with >2 values as we might run into when caluclating drift in postmunge

  #     valuecounts.sort()
  #     valuecounts = sorted(valuecounts)
      #in case this includes both strings and integers for instance we'll sort this way
  #     valuecounts = sorted(valuecounts, key=lambda p: str(p))

      #we'll save these in the normalization dictionary for future reference
      onevalue = valuecounts[0]
      if len(valuecounts) > 1:
        zerovalue = valuecounts[1]
      else:
        zerovalue = binary_missing_plug

      #special case for when the source column is already encoded as 0/1

      if len(valuecounts) <= 2:

        if 0 in valuecounts:
          zerovalue = 0
          if 1 in valuecounts:
            onevalue = 1
          else:
            if valuecounts[0] == 0:
              if len(valuecounts) > 1:
                onevalue = valuecounts[1]
              else:
                onevalue = binary_missing_plug

        if 1 in valuecounts:
          if 0 not in valuecounts:
            if valuecounts[0] != 1:
              onevalue = 1
              zerovalue = valuecounts[0]

      #edge case same as above but when values of 0 or 1. are in set and 
      #len(valuecounts) > 2
      if len(valuecounts) > 2:
        valuecounts2 = valuecounts[:2]

        if 0 in valuecounts2:
          zerovalue = 0
          if 1 in valuecounts2:
            onevalue = 1
          else:
            if valuecounts2[0] == 0:
              if len(valuecounts) > 1:
                onevalue = valuecounts2[1]
              else:
                onevalue = binary_missing_plug

        if 1 in valuecounts2:
          if 0 not in valuecounts2:
            if valuecounts2[0] != 1:
              onevalue = 1
              zerovalue = valuecounts2[0]

      #edge case that might come up in drift report
      if binary_missing_plug not in [onevalue, zerovalue]:
        if infillconvention == 'onevalue':
          binary_missing_plug = onevalue
        elif infillconvention == 'zerovalue':
          binary_missing_plug = zerovalue

      #edge case when applying this transform to set with >2 values
      #this only comes up when caluclating driftreport in postmunge
      if len(valuecounts) > 2:
        for value in extravalues:

          mdf_train = self.__autowhere(mdf_train, suffixcolumn, mdf_train[suffixcolumn] == value, binary_missing_plug, specified='replacement')
          mdf_test = self.__autowhere(mdf_test, suffixcolumn, mdf_test[suffixcolumn] == value, binary_missing_plug, specified='replacement')

      #apply defaultinfill based on processdict entry
      #(this will default to naninfill in which casee the next line will dictate)
      mdf_train, defaultinfill_dict = \
      self.__apply_defaultinfill(mdf_train, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
      mdf_test, _1 = \
      self.__apply_defaultinfill(mdf_test, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)

      #replace missing data with specified classification
      mdf_train[suffixcolumn] = mdf_train[suffixcolumn].fillna(binary_missing_plug)
      mdf_test[suffixcolumn] = mdf_test[suffixcolumn].fillna(binary_missing_plug)

      #this addressess issue where nunique for mdftest > than that for mdf_train
      #note is currently an oportunity for improvement that NArows won't identify these poinsts as candiadates
      #for user specified infill, and as currently addressed will default to infill with most common value
      #in the mean time a workaround could be for user to manually replace extra values with nan prior to
      #postmunge application such as if they want to apply ML infill
      #this will only be an issue when nunique for df_train == 2, and nunique for df_test > 2
      #if len(mdf_test[suffixcolumn].unique()) > 2:
      uniqueintest = mdf_test[suffixcolumn].unique()
      for unique in uniqueintest:
        if unique not in {onevalue, zerovalue}:

          mdf_test = self.__autowhere(mdf_test, suffixcolumn, mdf_test[suffixcolumn] == unique, binary_missing_plug, specified='replacement')

      #convert column to binary 0/1 classification (replaces scikit LabelBinarizer)
      if invert is False:
        mdf_train = self.__autowhere(mdf_train, suffixcolumn, mdf_train[suffixcolumn] == onevalue, 1, 0)
        mdf_test = self.__autowhere(mdf_test, suffixcolumn, mdf_test[suffixcolumn] == onevalue, 1, 0)
      elif invert is True:
        mdf_train = self.__autowhere(mdf_train, suffixcolumn, mdf_train[suffixcolumn] == onevalue, 0, 1)
        mdf_test = self.__autowhere(mdf_test, suffixcolumn, mdf_test[suffixcolumn] == onevalue, 0, 1)

      #create list of columns
      bnrycolumns = [suffixcolumn]

      #change data types to 8-bit (1 byte) integers for memory savings
      mdf_train[suffixcolumn] = mdf_train[suffixcolumn].astype(np.int8)
      mdf_test[suffixcolumn] = mdf_test[suffixcolumn].astype(np.int8)

      #a few more metrics collected for driftreport
      oneratio = mdf_train[suffixcolumn].sum() / mdf_train[suffixcolumn].shape[0]
      zeroratio = (mdf_train[suffixcolumn].shape[0] - mdf_train[suffixcolumn].sum() )\
                  / mdf_train[suffixcolumn].shape[0]

      #create list of columns associated with categorical transform (blank for now)
      categorylist = []
    
    else:
      mdf_train.loc[:, suffixcolumn] = 0
      mdf_test.loc[:, suffixcolumn] = 0
      
      binary_missing_plug = 0
      onevalue = 1
      zerovalue = 0
      extravalues = 0
      oneratio = 0
      zeroratio = 0
      bnrycolumns = [suffixcolumn]
      defaultinfill_dict = {'defaultinfill' : 'naninfill'}

  #     bnrynormalization_dict = {suffixcolumn : {'missing' : binary_missing_plug, \
  #                                                   'onevalue' : onevalue, \
  #                                                   'zerovalue' : zerovalue}}
    
    bnrynormalization_dict = {suffixcolumn : {'missing' : binary_missing_plug, \
                                              'infillconvention' : infillconvention, \
                                              1 : onevalue, \
                                              0 : zerovalue, \
                                              'extravalues' : extravalues, \
                                              'oneratio' : oneratio, \
                                              'zeroratio' : zeroratio, \
                                              'str_convert' : str_convert, \
                                              'suffix' : suffix, \
                                              'defaultinfill_dict' : defaultinfill_dict,
                                              'invert' : invert,
                                              'inplace' : inplace}}

    #store some values in the column_dict{} for use later in ML infill methods
    column_dict_list = []

    for bc in bnrycolumns:

      if bc != bnrycolumns[0]:
        bnrynormalization_dict = {}

      column_dict = { bc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : bnrynormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : bnrycolumns, \
                           'categorylist' : bnrycolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)

    return mdf_train, mdf_test, column_dict_list

  def _custom_train_onht(self, df, column, normalization_dict):
    """
    #a rewrite of onht transform in the custom train convention
    #note that this version combines text vs onht suffix conentions, now distinguishable by parameter
    #this should benefit latency
    #return comparable form of output
    #and accept comaprable parameters
    #primary difference is cleaner code and trimmed a little fat
    """

    #suffix_convention is to distinguish between text and onht suffixes
    #eg text: column + _str(entry) vs onht: column + _str(int)
    #(originally text followed conention column(minus suffix) + _entry)
    #suffix_convention accepts one of {'text', 'onht'}, and defaults to text
    if 'suffix_convention' in normalization_dict:
      suffix_convention = normalization_dict['suffix_convention']
    else:
      suffix_convention = 'text'
      normalization_dict.update({'suffix_convention' : suffix_convention})
    
    #note that when suffix_convention passed as text, 
    #str_convert hard resets to True, and null_activation hard resets to False
    if suffix_convention == 'text':
      normalization_dict.update({'str_convert' : True})
      normalization_dict.update({'null_activation' : False})
      
    #str_convert provides consistent encodings between numbers and string equivalent, eg 2 == '2'
    #str_convert defaults to True to avoid edge case assocaited with bytes type entries
    #accepts one of {True, False, 'conditional_on_bytes'}
    #where conditional_on_bytes sets as False unless bytes entries are present in train data then sets to True
    if 'str_convert' in normalization_dict:
      str_convert = normalization_dict['str_convert']
    else:
      str_convert = True
      normalization_dict.update({'str_convert' : str_convert})
    
    #ordered_overide is boolean to indicate if order of integer encoding basis will 
    #defer to cases when a column is a pandas categorical ordered set
    if 'ordered_overide' in normalization_dict:
      ordered_overide = normalization_dict['ordered_overide']
    else:
      ordered_overide = True
      normalization_dict.update({'ordered_overide' : ordered_overide})

    #all_activations is to base the full set of activations on user specification instead of training set
    if 'all_activations' in normalization_dict:
      #accepts False or a list of activation targets
      all_activations = normalization_dict['all_activations']
    else:
      all_activations = False
      normalization_dict.update({'all_activations' : all_activations})

    #add_activations is to include additional columns for entry activations even when not found in train set
    if 'add_activations' in normalization_dict:
      #accepts False or a list of added activation targets
      add_activations = normalization_dict['add_activations']
    else:
      add_activations = False
      normalization_dict.update({'add_activations' : add_activations})

    #less_activations is to remove entry activaiton columns even when entry found in train set
    if 'less_activations' in normalization_dict:
      #accepts False or a list of removed activation targets
      less_activations = normalization_dict['less_activations']
    else:
      less_activations = False
      normalization_dict.update({'less_activations' : less_activations})

    #consolidated_activations is to consolidate entries to a single common activation
    if 'consolidated_activations' in normalization_dict:
      #accepts False or a list of consolidated activation targets or a list of lists
      consolidated_activations = normalization_dict['consolidated_activations']
    else:
      consolidated_activations = False
      normalization_dict.update({'consolidated_activations' : consolidated_activations})
      
    #null_activation is to have a distinct column for missing data
    #note that when activated, entries in test set not found in train set will still be returned without activation
    #also accepts as 'Binary', which is used in Binary dimensionality reduction to return from inversion all zeros
    #for cases where test activation sets don't match any found in train activation sets
    #note that in Binary scenario we don't accept NaN entries in input
    if 'null_activation' in normalization_dict:
      null_activation = normalization_dict['null_activation']
    else:
      null_activation = False
      normalization_dict.update({'null_activation' : null_activation})
      
    #frequency_sort changes the sorting of encodings from alphabetical to frequency of entries in train set
    if 'frequency_sort' in normalization_dict:
      frequency_sort = normalization_dict['frequency_sort']
    else:
      frequency_sort = True
      normalization_dict.update({'frequency_sort' : frequency_sort})
      
    #_____
    
    #for every derivation related to the set labels_train, we'll remove missing_marker and add once prior to assembling binaryencoding_dict
    #which helps accomodate a few peculiarities related to python sets with NaN inclusion
    missing_marker = np.nan
    if null_activation == 'Binary':
      missing_marker = '0' * len(str(df[column].iat[0]))

    normalization_dict.update({'missing_marker' : missing_marker})

    #conditional_on_bytes str_convert scenario resets to True if train set bytes entries present, otherwise resets to False
    if str_convert == 'conditional_on_bytes' \
    and (df[column] == df[column].astype(bytes)).sum() > 0:
      str_convert = True
      normalization_dict.update({'str_convert' : str_convert})
    elif str_convert == 'conditional_on_bytes':
      str_convert = False
      normalization_dict.update({'str_convert' : str_convert})
    
    #labels_train will be adjusted through derivation and serves as basis for binarization encoding
    labels_train = set()
    
    ordered = False
    
    #ordered_overide is not compatible with activation parameters
    if ordered_overide:
      if df[column].dtype.name == 'category' and df[column].cat.ordered:
        ordered = True
        labels_train_ordered = list(df[column].cat.categories)
        #by convention NaN is reserved for use with missing data
        labels_train_ordered = [x for x in labels_train_ordered if x==x]
        if str_convert is True:
          labels_train_ordered = [str(x) for x in labels_train_ordered]
        
    #frequency_sort derives a sorting order based on frequency of entries found in set
    if ordered is False and frequency_sort is True:
      ordered = True
      
      tempstring = 'asdf'
      if tempstring in df:
        while tempstring in df:
          tempstring += 'z'
      
      if str_convert is True:
        labels_train_ordered = pd.DataFrame(df[column].astype(str).value_counts())
      else:
        labels_train_ordered = pd.DataFrame(df[column].value_counts())
      #note this sorting has an edge case for bytes type entries when str_convert passed as False
      labels_train_ordered = labels_train_ordered.rename_axis(tempstring).sort_values(by = [column, tempstring], ascending = [False, True])
      labels_train_ordered = list(labels_train_ordered.index)
      #by convention NaN is reserved for use with missing data
      labels_train_ordered = [x for x in labels_train_ordered if x==x]
      if str_convert is True:
        labels_train_ordered = [str(x) for x in labels_train_ordered]
    
    #_____
            
    if df[column].dtype.name == 'category':
      labels_train = set(df[column].cat.categories)
      labels_train = {x for x in labels_train if x==x}

    if labels_train == set():  
      labels_train = set(df[column].unique())
      labels_train = {x for x in labels_train if x==x}

    #if str_convert parameter activated replace numerical with string equivalent (for common encoding between e.g. 2=='2')
    if str_convert is True:
      #note this set excludes missing_marker
      labels_train = {str(x) for x in labels_train}
      #only convert non-NaN entries in target column
      df = self.__autowhere(df, column, df[column] == df[column], df[column].astype(str), specified='replacement')

    #now we have a few activation set related parameters, applied by adjusting labels_train
    #we'll have convention that in cases where activation parameters are assigned, will overide ordered_overide (for alphabetic sorting)

    if all_activations is not False or less_activations is not False:
      #labels_train_orig is a support record that won't be returned
      labels_train_orig = labels_train.copy()

    if all_activations is not False:
      all_activations = {x for x in set(all_activations) if x==x}
      labels_train = all_activations

    if add_activations is not False:
      add_activations = {x for x in set(add_activations) if x==x}
      labels_train = labels_train | add_activations

    if less_activations is not False:
      less_activations = {x for x in set(less_activations) if x==x}
      labels_train = labels_train - less_activations

    #______

    #now we'll take account for any activation consolidations from consolidated_activations parameter

    #as part of this implementation, we next want to derive
    #a version of labels_train excluding consolidations (labels_train)
    #a list of consolidations associated with each returned_consolidation mapped to the returned_consolidation (consolidation_translate_dict)
    #and an inverse_consolidation_translate_dict mapping consolidated entries to their activations
    #which we'll then apply with a replace operation

    labels_train_before_consolidation = labels_train.copy()
    consolidation_translate_dict = {}
    inverse_consolidation_translate_dict = {}

    if consolidated_activations is not False:

      #if user passes a single tier list instead of list of lists we'll embed in a list
      if isinstance(consolidated_activations, list) and len(consolidated_activations) > 0:
        if not isinstance(consolidated_activations[0], list):
          consolidated_activations = [consolidated_activations]
          normalization_dict.update({'consolidated_activations' : consolidated_activations})

      for consolidation_list in consolidated_activations:

        #here is where we add any consolidation targets that weren't present in labels_train
        if str_convert is True:
          consolidation_list = [str(x) for x in consolidation_list if x==x]
          labels_train = labels_train | set(consolidation_list)
        else:
          #by convention missing data marker not elligible for inclusion in consolidation_list due to NaN/set peculiarities
          #consolidations with NaN can be accomodated by assignnan to treat desired entries as missing data
          consolidation_list = [x for x in consolidation_list if x==x]
          labels_train = labels_train | set(consolidation_list)

        #no prepare a version of labels_train excluding consolidations (labels_train)

        #we'll take the first entry in list as the returned activation (relevant to normalization_dict)
        returned_consolidation = consolidation_list[0]

        #now remove consolidated entries from labels_train
        #and map a list of consolidations associated with each returned_consolidation to the returned_consolidation (consolidation_translate_dict)
        for consolidation_list in consolidated_activations:

          if len(consolidation_list) > 1:

            #we'll take the first entry in list as the returned activation (relevant to normalization_dict)
            returned_consolidation = consolidation_list[0]

            labels_train = labels_train - set(consolidation_list[1:])

            consolidation_translate_dict.update({returned_consolidation : consolidation_list[1:]})

      #now populate an inverse_consolidation_translate_dict mapping consolidated entries to their activations
      for key,value in consolidation_translate_dict.items():
        for consolidation_list_entry in value:
          inverse_consolidation_translate_dict.update({consolidation_list_entry : key})

      #we can then apply a replace to convert consolidated items to their targeted activations
      df[column] = df[column].astype('object').replace(inverse_consolidation_translate_dict)

    del consolidation_translate_dict
    normalization_dict.update({'inverse_consolidation_translate_dict' : inverse_consolidation_translate_dict})
    del inverse_consolidation_translate_dict

    #____

    #there are a few activation parameter scenarios where we may want to replace train set entries with missing data marker
    if all_activations is not False or less_activations is not False:
      extra_entries = labels_train_orig - labels_train
      extra_entries = list({x for x in extra_entries if x==x})
      if len(extra_entries) > 0:
        plug_dict = dict(zip(extra_entries, [missing_marker] * len(extra_entries)))
        df[column] = df[column].astype('object').replace(plug_dict)

      del labels_train_orig

    #____

    #now prepare our onehot encoding
    
    #if there is a particular order to encodings we'll sort labels_train on basis of labels_train_ordered
    if ordered is True:
      #this converts labels_train to a sorted list
      labels_train = self.__list_sorting(labels_train_ordered, labels_train)
    
    elif ordered is False:
      #convert labels_train to list 
      #and add the missing data marker to first position which will result in all zero binarized representation
      labels_train = list(labels_train)
      labels_train = sorted(labels_train, key=str)
      
    #add our missing_marker, note adding as last position will result in last column even in ordered scenario
    if null_activation is True:
      labels_train = labels_train + [missing_marker]

    #one hot encoding support function
    df_cat = self.__onehot_support(df, column, scenario=1, activations_list=labels_train)
    
    #change data types to int8 for memory savings
    for activation_column in labels_train:
      df_cat[activation_column] = df_cat[activation_column].astype(np.int8)
    
    labels_dict = {}
    onht_returned_columns = []
    if suffix_convention == 'onht':
      i = 0
      for entry in labels_train:
        labels_dict.update({entry : column + '_' + str(i)})
        onht_returned_columns.append(column + '_' + str(i))
        i += 1
    elif suffix_convention == 'text':
      for entry in labels_train:
        #str conversion is to accomodate missing data marker which will return as column + '_nan'
        labels_dict.update({entry : column + '_' + entry})
        onht_returned_columns.append(column + '_' + entry)
      
    normalization_dict.update({'labels_dict' : labels_dict,
                               'onht_returned_columns' : onht_returned_columns})
      
    #concatinate the sparse set with the rest of our training data
    df = pd.concat([df, df_cat], axis=1)
    
    del df[column]
    
    #now convert coloumn headers from text convention to onht convention
    df = df.rename(columns=labels_dict)

    #this is a drift stat for activaiton ratios
    onht_activations_dict = {}
    for onht_returned_column in onht_returned_columns:
      sumcalc = (df[onht_returned_column] == 1).sum() 
      ratio = sumcalc / df[onht_returned_column].shape[0]
      onht_activations_dict.update({onht_returned_column : ratio})
      
    normalization_dict.update({'onht_activations_dict' : onht_activations_dict})
    
    return df, normalization_dict

  def _process_smth(self, mdf_train, mdf_test, column, category, treecategory, postprocess_dict, params = {}):
    '''
    #process_smth(mdf_train, mdf_test, column, category, postprocess_dict, params = {})
    #intended for applicant downstream of a multirt MLinfilltype encoding (e.g. one hot encoding)
    #since this is specifically intended as a downstream trasnform, we'll assume infill already applied
    #applies label smoothing
    #accepts parameters for activation value (float 0.5-1), 
    #LSfit parameter to activate fitted smoothing
    #testsmooth parameter to activate consistently smoothing test data
    '''

    suffixoverlap_results = {}
      
    if 'activation' in params:
      activation = params['activation']
    else:
      activation = 0.9

    if 'LSfit' in params:
      LSfit = params['LSfit']
    else:
      LSfit = False
      
    if 'testsmooth' in params:
      testsmooth = params['testsmooth']
    else:
      testsmooth = False

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False
      
    #this function is intended to be applied donstream of a multirt encoding
    #meaning there may be multiple columns serving as target
    #we'll access the upstream categorylist stored in postprocess_dict
    inputtextcolumns = postprocess_dict['column_dict'][column]['categorylist']
    
    #the returned columns will each have consistent suffix appending
    textcolumns = [(x + '_' + suffix) for x in inputtextcolumns]
    
    #this maps between received and returned columns
    textlabelsdict = dict(zip(inputtextcolumns, textcolumns))
    
    #inplace convention is a little different in that we are inspecting suffixoverlap before copying
    suffixoverlap_results = \
    self.__df_check_suffixoverlap(mdf_train, textcolumns, suffixoverlap_results, postprocess_dict['printstatus'])
    
    if inplace is not True:
      
      for inputtextcolumn in inputtextcolumns:
        
        mdf_train[textlabelsdict[inputtextcolumn]] = mdf_train[inputtextcolumn].copy()
        mdf_test[textlabelsdict[inputtextcolumn]] = mdf_test[inputtextcolumn].copy()
    
    else:
      
      mdf_train.rename(columns = textlabelsdict, inplace = True)
      mdf_test.rename(columns = textlabelsdict, inplace = True)
    
    #___
    
    #now apply label smoothing
#     category = 'smth'
    LSfitparams_dict = {}

    categorycomplete_dict = dict(zip(textcolumns, [False]*len(textcolumns)))

    for labelsmoothingcolumn in textcolumns:

      if categorycomplete_dict[labelsmoothingcolumn] is False:

        mdf_train, categorycomplete_dict, LSfitparams_dict = \
        self.__apply_LabelSmoothing(mdf_train, 
                                  labelsmoothingcolumn, 
                                  activation, 
                                  textcolumns, 
                                  category, 
                                  categorycomplete_dict, 
                                  LSfit, 
                                  LSfitparams_dict)

    #smoothing not applied to test data consistent with postmunge convention
    #(postmunge can apply based on traindata parameter or by activating testsmooth)
    if testsmooth is True:
      
      categorycomplete_test_dict = dict(zip(textcolumns, [False]*len(textcolumns)))
      
      for labelsmoothingcolumn in textcolumns:

        if categorycomplete_test_dict[labelsmoothingcolumn] is False:

          mdf_test, categorycomplete_dict = \
          self.__postapply_LabelSmoothing(mdf_test, 
                                        labelsmoothingcolumn, 
                                        categorycomplete_test_dict, 
                                        LSfitparams_dict)
    
    column_dict_list = []
    textnormalization_dict = {}

    if len(textcolumns) > 0:
      textnormalization_dict = {textcolumns[0] : {'textlabelsdict' : textlabelsdict, \
                                      'textcolumns' : textcolumns, \
                                      'inputtextcolumns' : inputtextcolumns, \
                                      'LSfitparams_dict' : LSfitparams_dict, \
                                      'activation' : activation, \
                                      'LSfit' : LSfit, \
                                      'testsmooth' : testsmooth, \
                                      'suffix' : suffix, \
                                      'inplace' : inplace}}

    for tc in textcolumns:

      if tc != textcolumns[0]:
        textnormalization_dict = {}
      
      column_dict = {tc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : textnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : textcolumns, \
                           'categorylist' : textcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
    
    return mdf_train, mdf_test, column_dict_list

  def __GPS_parse(self, coordinates, firstcomma, seccondcomma, negdirection):
    """
    This is a support function used in GPS1

    based on "$GPGGA message"
    applied to one set of coordinates
    receives coordinates as a str for parsing

    note that we are applying the full conversion in this function
    instead of extracting serately support columns for degrees, minutes, and direction
    and applying pandas operations to convert entries in parallel
    it is possible doing the latter may be more efficient
    for now we are following the full conversion in this function approach for cleaner code
    we speculate some further optimizations for processing efficiency may be possible
    by conducting portions of the DDMM operations in pandas instead of to each entry seperately with .transform
    which would require seperate parsing functions to populate each support column
    but is also possible the benefit would be offset by the redundant parsing
    """

    comma_counter = 0
    maxparsed_address = 0
    str_length = len(coordinates)
    parsing_complete = False
    DDMM = np.nan
    direction = np.nan
    latt = np.nan

    for address in range(str_length):

      if address > maxparsed_address and parsing_complete is False:

        current_char = coordinates[address]
        maxparsed_address = address

        if current_char == ',':
          comma_counter += 1

        else:

          #direction is single character string as one of {'N', 'S'}
          if comma_counter == seccondcomma:

            for address3 in range(address, str_length):
              if comma_counter == seccondcomma:
                current_char3 = coordinates[address3]

                if current_char3 == ',':
                  comma_counter += 1
                  maxparsed_address = address3
                  parsing_complete = True

                  excerpt = coordinates[address : address3]

                  if len(excerpt) > 0 and excerpt in {'N', 'S', 'E', 'W'}:

                    direction = excerpt

          #latitude in the DDMM.MMMMM (variable decimal) populated after second comma
          if comma_counter == firstcomma:

            for address2 in range(address, str_length):
              if comma_counter == firstcomma:
                current_char2 = coordinates[address2]

                if current_char2 == ',':
                  comma_counter += 1
                  maxparsed_address = address2

                  excerpt = coordinates[address : address2]

                  if len(excerpt) > 0 and self.__is_number(excerpt):
#                       if len(excerpt) > 0 and _is_number(excerpt):
                    DDMM = excerpt

    degrees = np.nan
    minutes = np.nan

    #the potential scenarios we will accomodate for characters preceding the decimal are DDMM. / DDDMM.
    if DDMM == DDMM:
      dot_index = np.nan
      if '.' in list(DDMM):
        dot_index = list(DDMM).index('.')
        if dot_index == 4:
          degrees = int(DDMM[0:2])
          minutes = float(DDMM[2:])
        elif dot_index == 5:
          degrees = int(DDMM[0:3])
          minutes = float(DDMM[3:])
      elif len(DDMM) == 4:
        degrees = int(DDMM[0:2])
        minutes = float(DDMM[2:])
      elif len(DDMM) == 5:
        degrees = int(DDMM[0:3])
        minutes = float(DDMM[3:])

      latt = degrees*60 + minutes

      if direction == negdirection:
        latt *= -1

    return latt
  
  def _custom_train_GPS1(self, df, column, normalization_dict):
    """
    GPS1 is for converting sets of received GPS coordinates into format of two columns of lattitude and longitude
    accepts input of string GPS coordinates
    
    accepts parameter GPS_convention, which currently only supports the base configuration of 'default'
    which in future extensions may allow selection between alternate GPS reporting conventions
    'default' is based on structure of the "$GPGGA message" which was output from an RTK GPS receiver
    which follows NMEA conventions, and has lattitude in between commas 2-3, and longitude between 4-5
    reference description available at https://www.gpsworld.com/what-exactly-is-gps-nmea-data/
    allows for variations in precisions of reported coordinates (i.e. number of significant figures)
    or variations in degree magnitude, such as between DDMM. or DDDMM.
    relies on comma seperated inputs
    accepts parameter comma_addresses to designate locations for lattitude/direction/longitude/direction
    which consistent with the demonstration defaults to [2,3,4,5]
    i.e. lattitude is after comma 2, direction after comma 3, longitude after 4, direction after 5
    also accepts parameter comma_count, defaulting to 14, which is used for inversion
    returns lattitude and longitude coordinates as +/- floats in units of arc minutes
    
    we believe there are other conventions for GPS reporting
    we don't yet have sufficient insight into mainstream practice to formalize alternatives
    this implementation is intended as a proof of concept
    
    note that in cases where comma seperated entries are blank, nonnumeric, or omitted
    the returned entries may be represented as NaN for missing data
    in the base root category definition, this transform is followed by a mlti transform 
    for independent normalization of the lattitude and longitude sets
    this will return consistent output independent of data properties of two columns 
    with suffixes _latt and _long to distinguish lattitude and longitude
    so no corresponding _custom_test function is needed

    note that the NArw aggregation will only recognize received NaN points
    (e.g. it won't recognize cases where lattitude or longitude weren't recorded)
    so if ML infill is desired on returned sets, received missing data should be NaN encoded

    Note that this function uses the support function _GPS_parse
    """

    #GPS_convention is to distinguish between conventions for format of received GPS coordinates
    #default is based on structure of the "$GPGGA message"
    #which was output from an RTK GPS receiver
    #in 'default' each row is individually parsed
    #in 'nonunique', only unique entries are parsed
    if 'GPS_convention' in normalization_dict:
      GPS_convention = normalization_dict['GPS_convention']
    else:
      GPS_convention = 'default'
      normalization_dict.update({'GPS_convention' : GPS_convention})
      
    if 'comma_addresses' in normalization_dict:
      comma_addresses = normalization_dict['comma_addresses']
    else:
      #this corresponds to default where DDMM.MMMM lattitude after comma 2, lattitude direction after comma 3
      #DDMM.MMMM longitude after comma 4, longitude direction after comma 5
      comma_addresses = [2,3,4,5]
      normalization_dict.update({'comma_addresses' : comma_addresses})

    #comma_count used in inversion to pad out the recovered form with commas
    if 'comma_count' in normalization_dict:
      comma_count = normalization_dict['comma_count']
    else:
      comma_count = 14
      normalization_dict.update({'comma_count' : comma_count})

    #_____

    def default_GPS_parse_latt(coordinates1):
      return self.__GPS_parse(coordinates1, comma_addresses[0], comma_addresses[1], 'S')

    def default_GPS_parse_long(coordinates1):
      return self.__GPS_parse(coordinates1, comma_addresses[2], comma_addresses[3], 'W')

    #_____

    #in the default scenario, each row is parsed, which may be appropriate for all unique entries
    if GPS_convention == 'default':

      #here are returned column headers
      latt_column = column + '_latt'
      long_column = column + '_long'

      normalization_dict.update({'latt_column' : latt_column,
                                 'long_column' : long_column})

      # df[column] = df[column].astype(str)

      df[latt_column] = pd.Series(df[column].astype(str)).transform(default_GPS_parse_latt)

      df[long_column] = pd.Series(df[column].astype(str)).transform(default_GPS_parse_long)
      
      del df[column]

    #_____
    
    #in the nonunique scenario, only unique entries are parsed, which may be appropriate for common points
    if GPS_convention == 'nonunique':

      unique_entries = list(df[column].astype(str).unique())

      latt_replace_dict = {}
      long_replace_dict = {}

      for unique_entry in unique_entries:

        unique_latt = default_GPS_parse_latt(unique_entry)
        latt_replace_dict.update({unique_entry : unique_latt})

        unique_long = default_GPS_parse_long(unique_entry)
        long_replace_dict.update({unique_entry : unique_long})

      #here are returned column headers
      latt_column = column + '_latt'
      long_column = column + '_long'

      df[latt_column] = df[column].astype(str).astype('object').replace(latt_replace_dict)
      df[long_column] = df[column].astype(str).astype('object').replace(long_replace_dict)

      del df[column]

      normalization_dict.update({'latt_column' : latt_column,
                                 'long_column' : long_column,
                                 'latt_replace_dict' : latt_replace_dict,
                                 'long_replace_dict' : long_replace_dict})

    return df, normalization_dict

  def _process_mlti(self, mdf_train, mdf_test, column, category, treecategory, postprocess_dict, params = {}):
    '''
    #intended for applicant downstream of a concurrent_nmbr MLinfilltype encoding 
    #(e.g. downstream of multicolumn continuous numeric sets)
    #since this is specifically intended as a downstream trasnform, we'll assume infill already applied
    #applies normalization to each of the columns in upstream transform returned categorylist
    #the type of normalization applied is based on assignparam parameter norm_category
    #which defaults to nmbr, as in trasnforms are accessed from nmbr process_dict entry
    #parameters to the normalization trasnform can also be passed as dictionary to parameter norm_params, e.g. in the form
    #assignparam = {'mlti' : {'(column)' : {'norm_params' : {'(parameter)' : '(value)'}}}
    #where '(column)' is input column associated with the root category
    #and parameter / value are associated with the norm_category
    #the normalizaiton applied treats the norm_category as a tree category without offspring
    #and the input columns are either retained or replaced based on parameter norm_retain, defaulting ot False for replaced
    #note that if an alternate treatment is desired where to apply a family tree of transforms to each column
    #user should instead structure upstream trasnform as a set of numeric mlinfilltype trasnforms.
    '''

    suffixoverlap_results = {}
      
    if 'norm_category' in params:
      norm_category = params['norm_category']
    else:
      norm_category = 'nmbr'

    postprocess_dict['mlti_categories'] = postprocess_dict['mlti_categories'] | {norm_category}

    if 'norm_params' in params:
      norm_params = params['norm_params']
    else:
      norm_params = {}

    #dtype accepts one of {'float', 'conditionalinteger', 'mlhs'}
    #where 'float' is for use with mlti applied with concurrent_nmbr MLinfilltype
    #and 'conditionalinteger' is for use with concurrent_ordl MLinfilltype
    #and 'hash' is for use when upstream trasnform is an ordinal hashing (uses different method to identify encoding space for conditional type)
    #(may be needed for transforms in custom_trian convention, otherwise can just leave as 'float' to defer to transform basis)
    #note conditionalinteger is only applied when processdict[norm_category]['dtype_convert'] is not False
    if 'dtype' in params:
      dtype = params['dtype']
    else:
      dtype = 'float'

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False
      
    if 'messy_data_headers' in params:
      messy_data_headers = params['messy_data_headers']
      if messy_data_headers == []:
        messy_data_headers = [column]
    else:
      messy_data_headers = False
      
    #________

    #when applying mlti as a noise transform mlhs for DPhs
    #redundant noise parameters need to be passed both to mlti params and to norm_params
    #mlti params for purposes of deriving entroy seed requirements
    #and norm_params for purposes of passing to norm_category
    #(as alternative to norm_params global_assignparam params or default_assignparam will also work)

    if 'trainnoise' in params:
      trainnoise = params['trainnoise']
    else:
      trainnoise = True

    if 'testnoise' in params:
      testnoise = params['testnoise']
    else:
      testnoise = False

    if 'flip_prob' in params:
      flip_prob = params['flip_prob']
    else:
      #default also populated in __random_parameters_params_append
      flip_prob = 0.03
      
    if 'test_flip_prob' in params:
      test_flip_prob = params['test_flip_prob']
    else:
      #default also populated in __random_parameters_params_append
      test_flip_prob = 0.01

    #note that random_generator accessed from automunge(.) parameter and not passed to postmunge
    #postmunge(.) has a corresponding parameter to support
    if 'random_generator' in params:
      random_generator = params['random_generator']
    else:
      random_generator = np.random.PCG64
    
#     #note that in the random_generator == False scenario 
#     #__random_parameters_params_append will have set to np.random.PCG64
#     nprandom_dict = {'custom' : random_generator,
#                      'default': np.random.PCG64}
      
    #note that sampling_resource_dict populated externally 
    #based on automunge(.) sampling_dict and entropy_seeds parameters 
    #and not passed to postmunge
    #postmunge(.) has corresponding parameters to support
    if 'sampling_resource_dict' in params:
      sampling_resource_dict = params['sampling_resource_dict']
    # else:
    #as configured sampling_resource_dict is always populated for noise transforms in params by __random_parameters_params_append based on process_dict['noise_transform']  specification
    else:
      #mlti doesn't register the missing_process_dict_noise_transform validation result since it is also used for non-noise trasnforms
      sampling_resource_dict = {'seeding_type' : 'supplemental_seeds', 
                                'binomial_train' : 'custom',
                                'binomial_train_seeds' : [],
                                'binomial_train_call_count' : 0,
                                'binomial_train_sample_count' : 0,
                                'binomial_test' : 'custom',
                                'binomial_test_seeds' : [],
                                'binomial_test_call_count' : 0,
                                'binomial_test_sample_count' : 0,
                                'distribution_train' : 'custom',
                                'distribution_train_seeds' : [],
                                'distribution_train_call_count' : 0,
                                'distribution_train_sample_count' : 0,
                                'distribution_test' : 'custom',
                                'distribution_test_seeds' : [],
                                'distribution_test_call_count' : 0,
                                'distribution_test_sample_count' : 0,
                                'choice_train' : 'custom',
                                'choice_train_seeds' : [],
                                'choice_train_call_count' : 0,
                                'choice_train_sample_count' : 0,
                                'choice_test' : 'custom',
                                'choice_test_seeds' : [],
                                'choice_test_call_count' : 0,
                                'choice_test_sample_count' : 0,
                                'parameterlist_train' : 'custom',
                                'parameterlist_train_seeds' : [],
                                'parameterlist_train_call_count' : 0,
                                'parameterlist_train_sample_count' : 0,
                                'parameterlist_test' : 'custom',
                                'parameterlist_test_seeds' : [],
                                'parameterlist_test_call_count' : 0,
                                'parameterlist_test_sample_count' : 0,
                                'statsdistribution_train' : 'custom',
                                'statsdistribution_train_seeds' : [],
                                'statsdistribution_train_call_count' : 0,
                                'statsdistribution_train_sample_count' : 0,
                                'statsdistribution_test' : 'custom',
                                'statsdistribution_test_seeds' : [],
                                'statsdistribution_test_call_count' : 0,
                                'statsdistribution_test_sample_count' : 0,
                                'random_generator_accepts_seeds' : True,
                                }

    #________

    #this is to support accessing defaultparams from process_dict based on norm_category
    mlti_norm_params = self.__grab_params({norm_category : {column : norm_params}}, 
                                          norm_category, 
                                          column, 
                                          postprocess_dict['process_dict'][norm_category],
                                          postprocess_dict
                                         )
      
    #this function is intended to be applied donstream of a multirt encoding
    #meaning there may be multiple columns serving as target
    #we'll access the upstream categorylist stored in postprocess_dict
    #or otherwise if this is an upstream trasnform just apply to the received column
    if column in postprocess_dict['column_dict']:
      inputtextcolumns = postprocess_dict['column_dict'][column]['categorylist']
    else:
      #origcolumn scenario when not applied as downstream transform
      inputtextcolumns = [column]
      
    if isinstance(messy_data_headers, list) and column not in postprocess_dict['column_dict']:
      inputtextcolumns = messy_data_headers.copy()
    
    #the returned columns will each have consistent suffix appending
    textcolumns = [(x + '_' + suffix) for x in inputtextcolumns]

    #now update the mlti_norm_params based on any relevant assignparam specifications to global and default
    assignparam = postprocess_dict['assign_param']
    if 'global_assignparam' in assignparam:
      mlti_norm_params.update(assignparam['global_assignparam'])
    if 'default_assignparam' in assignparam:
      if norm_category in assignparam['default_assignparam']:
        mlti_norm_params.update(assignparam['default_assignparam'][norm_category])
        
    #for now operating on convention that the same entropy seeding parameters are passed to each instance
    #traindata inspection is applicable to postmunge, in automunge will not be present
    if 'noise_transform' in postprocess_dict['process_dict'][norm_category] \
    and postprocess_dict['process_dict'][norm_category]['noise_transform'] is not False \
    and ( 'traindata' not in postprocess_dict or 'traindata' in postprocess_dict \
    and postprocess_dict['traindata'] not in {'train_no_noise', 'test_no_noise'}):
      
      mlti_norm_params.update({'random_generator' : random_generator,
                               'sampling_resource_dict' : sampling_resource_dict})

    #textcolumns are never returned, they just used to applky the intermediate suffix appender associated with this tree category
    #so we'll always pass inplace as True in norm_params and if not accepted based on inplace_option in process_dict delete textcolumns
    mlti_norm_params.update({'inplace' : True})
    
    #this maps between received and returned columns
    textlabelsdict = dict(zip(inputtextcolumns, textcolumns))
    
    #inplace convention is a little different in that we are inspecting suffixoverlap before copying
    suffixoverlap_results = \
    self.__df_check_suffixoverlap(mdf_train, textcolumns, suffixoverlap_results, postprocess_dict['printstatus'])
    
    if inplace is not True:
      
      for inputtextcolumn in inputtextcolumns:
        
        mdf_train[textlabelsdict[inputtextcolumn]] = mdf_train[inputtextcolumn].copy()
        mdf_test[textlabelsdict[inputtextcolumn]] = mdf_test[inputtextcolumn].copy()
    
    else:
      
      mdf_train.rename(columns = textlabelsdict, inplace = True)
      mdf_test.rename(columns = textlabelsdict, inplace = True)
      
    mlti_norm_params_column_dict = {}
    for inputcolumn in textcolumns:
      mlti_norm_params_column_dict.update({inputcolumn : self.__autocopy(mlti_norm_params)})
    
    #now apply one of custom_train / custom_test or dualprocess based on norm_category processdict entry
    
    norm_column_dict_list = []
    norm_columnkey_dict = {'columnkey_dict' : {}}
    
    if 'custom_train' in postprocess_dict['process_dict'][norm_category] \
    and callable(postprocess_dict['process_dict'][norm_category]['custom_train']):
      
      for inputcolumn in textcolumns:
        
        #any assignparam specifications to specific inputcolumn added here
        if norm_category in assignparam:
          if inputcolumn in assignparam[norm_category]:
            mlti_norm_params_column_dict[inputcolumn].update(assignparam[norm_category][inputcolumn])
            
        #for entropy seeding we'll shuffle for each application
        if 'sampling_resource_dict' in mlti_norm_params_column_dict[inputcolumn]:
          for entry in mlti_norm_params_column_dict[inputcolumn]['sampling_resource_dict']:
            if entry[-5:] == 'seeds' \
            and isinstance(mlti_norm_params_column_dict[inputcolumn]['sampling_resource_dict'][entry], type(np.array([]))):
              np.random.shuffle(mlti_norm_params_column_dict[inputcolumn]['sampling_resource_dict'][entry])
        
        mdf_train, mdf_test, column_dict_list_portion = \
        self.__custom_process_wrapper(mdf_train, mdf_test, inputcolumn, category, \
                                     norm_category, postprocess_dict, mlti_norm_params_column_dict[inputcolumn])
    
        norm_column_dict_list += self.__autocopy(column_dict_list_portion)
      
        norm_columnkey_dict = self.__populate_columnkey_dict(column_dict_list_portion, norm_columnkey_dict, norm_category)

      if 'inplace_option' in postprocess_dict['process_dict'][norm_category] \
      and postprocess_dict['process_dict'][norm_category]['inplace_option'] is False:
        for textcolumn in textcolumns:
          del mdf_train[textcolumn]
          del mdf_test[textcolumn]

    #elif this is a dual process function
    elif 'dualprocess' in postprocess_dict['process_dict'][norm_category] \
    and callable(postprocess_dict['process_dict'][norm_category]['dualprocess']):
      
      for inputcolumn in textcolumns:
        
        #any assignparam specifications to specific inputcolumn added here
        if norm_category in assignparam:
          if inputcolumn in assignparam[norm_category]:
            mlti_norm_params_column_dict[inputcolumn].update(assignparam[norm_category][inputcolumn])
            
        #for entropy seeding we'll shuffle for each application
        if 'sampling_resource_dict' in mlti_norm_params_column_dict[inputcolumn]:
          for entry in mlti_norm_params_column_dict[inputcolumn]['sampling_resource_dict']:
            if entry[-5:] == 'seeds' \
            and isinstance(mlti_norm_params_column_dict[inputcolumn]['sampling_resource_dict'][entry], type(np.array([]))):
              np.random.shuffle(mlti_norm_params_column_dict[inputcolumn]['sampling_resource_dict'][entry])

        mdf_train, mdf_test, column_dict_list_portion = \
        postprocess_dict['process_dict'][norm_category]['dualprocess'](mdf_train, mdf_test, inputcolumn, category, \
                                                                       norm_category, postprocess_dict, mlti_norm_params_column_dict[inputcolumn])

        norm_column_dict_list += self.__autocopy(column_dict_list_portion)
      
        norm_columnkey_dict = self.__populate_columnkey_dict(column_dict_list_portion, norm_columnkey_dict, norm_category)

      if 'inplace_option' in postprocess_dict['process_dict'][norm_category] \
      and postprocess_dict['process_dict'][norm_category]['inplace_option'] is False:
        for textcolumn in textcolumns:
          del mdf_train[textcolumn]
          del mdf_test[textcolumn]
    
    #else if this is a single process function process train and test seperately
    elif 'singleprocess' in postprocess_dict['process_dict'][norm_category] \
    and callable(postprocess_dict['process_dict'][norm_category]['singleprocess']):
      
      for inputcolumn in textcolumns:

        #any assignparam specifications to specific inputcolumn added here
        if norm_category in assignparam:
          if inputcolumn in assignparam[norm_category]:
            mlti_norm_params_column_dict[inputcolumn].update(assignparam[norm_category][inputcolumn])
            
        #for entropy seeding we'll shuffle for each application
        if 'sampling_resource_dict' in mlti_norm_params_column_dict[inputcolumn]:
          for entry in mlti_norm_params_column_dict[inputcolumn]['sampling_resource_dict']:
            if entry[-5:] == 'seeds' \
            and isinstance(mlti_norm_params_column_dict[inputcolumn]['sampling_resource_dict'][entry], type(np.array([]))):
              np.random.shuffle(mlti_norm_params_column_dict[inputcolumn]['sampling_resource_dict'][entry])

        mdf_train, column_dict_list_portion =  \
        postprocess_dict['process_dict'][norm_category]['singleprocess'](mdf_train, inputcolumn, category, \
                                                                         norm_category, postprocess_dict, mlti_norm_params_column_dict[inputcolumn])

        mdf_test, _1 = \
        postprocess_dict['process_dict'][norm_category]['singleprocess'](mdf_test, inputcolumn, category, \
                                                                         norm_category, postprocess_dict, mlti_norm_params_column_dict[inputcolumn])

        norm_column_dict_list += self.__autocopy(column_dict_list_portion)
      
        norm_columnkey_dict = self.__populate_columnkey_dict(column_dict_list_portion, norm_columnkey_dict, norm_category)

      if 'inplace_option' in postprocess_dict['process_dict'][norm_category] \
      and postprocess_dict['process_dict'][norm_category]['inplace_option'] is False:
        for textcolumn in textcolumns:
          del mdf_train[textcolumn]
          del mdf_test[textcolumn]
    
    final_returned_columns = []
    for norm_column_dict_list_entry in norm_column_dict_list:
      final_returned_columns.append(list(norm_column_dict_list_entry)[0])

    #if this is an ordinal encoded set returned form transform in custom_train convention we may need to apply dtype convert
    max_encoding_for_dtype_convert_dict = {}
    if dtype in {'conditionalinteger', 'mlhs'}:

      if 'dtype_convert' not in postprocess_dict['process_dict'][norm_category] \
      or 'dtype_convert' in postprocess_dict['process_dict'][norm_category] \
      and postprocess_dict['process_dict'][norm_category]['dtype_convert'] is not False:

        for final_returned_column in final_returned_columns:

          max_encoding_for_dtype_convert = mdf_train[final_returned_column].max()

          #max encoding space for dtype = mlhs scnenario assumes upstream hashing and norm_category that retains size of encoding space
          if dtype == 'mlhs':

            if column in postprocess_dict['column_dict'] \
            and column in postprocess_dict['column_dict'][column]['normalization_dict'] \
            and 'vocab_size' in postprocess_dict['column_dict'][column]['normalization_dict'][column]:
              max_encoding_for_dtype_convert = postprocess_dict['column_dict'][column]['normalization_dict'][column]['vocab_size']

          #save the max encoding with key of final_returned_column for use in postmunge
          max_encoding_for_dtype_convert_dict.update({final_returned_column : max_encoding_for_dtype_convert})

          if max_encoding_for_dtype_convert <= 255:
            mdf_train[final_returned_column] = mdf_train[final_returned_column].astype(np.uint8)
            mdf_test[final_returned_column] = mdf_test[final_returned_column].astype(np.uint8)
          elif max_encoding_for_dtype_convert <= 65535:
            mdf_train[final_returned_column] = mdf_train[final_returned_column].astype(np.uint16)
            mdf_test[final_returned_column] = mdf_test[final_returned_column].astype(np.uint16)
          else:
            mdf_train[final_returned_column] = mdf_train[final_returned_column].astype(np.uint32)
            mdf_test[final_returned_column] = mdf_test[final_returned_column].astype(np.uint32)
    
    #_____
    #entropy seeding support
      
    #erase seeds
    sampling_resource_dict = self.__erase_seeds(sampling_resource_dict)
    
    if 'noise_transform' in postprocess_dict['process_dict'][norm_category] \
    and postprocess_dict['process_dict'][norm_category]['noise_transform'] is not False \
    and ( 'traindata' not in postprocess_dict or 'traindata' in postprocess_dict \
    and postprocess_dict['traindata'] not in {'train_no_noise', 'test_no_noise'}):
      
      #erase seeds that were saved in mlti_norm_params_column_dict
      for inputcolumn in textcolumns:
        if 'sampling_resource_dict' in mlti_norm_params_column_dict[inputcolumn]:
          mlti_norm_params_column_dict[inputcolumn]['sampling_resource_dict'] = \
          self.__erase_seeds(mlti_norm_params_column_dict[inputcolumn]['sampling_resource_dict'])
        
      #aggregate counts into the returned sampling_resource_dict from norm_category reports
      for norm_column_dict in norm_column_dict_list:
        for norm_column_dict_key in norm_column_dict:
          if 'normalization_dict' in norm_column_dict[norm_column_dict_key] \
          and norm_column_dict_key in norm_column_dict[norm_column_dict_key]['normalization_dict'] \
          and 'sampling_resource_dict' in norm_column_dict[norm_column_dict_key]['normalization_dict'][norm_column_dict_key]:
            sampling_resource_dict_single = norm_column_dict[norm_column_dict_key]['normalization_dict'][norm_column_dict_key]['sampling_resource_dict']
            
            #here are the counts we'll aggregate, which will have integer entries (seeds will be empty list or array)
            aggregated_counts = []
            for entry in sampling_resource_dict:
              if isinstance(sampling_resource_dict[entry], (int, float)):
                aggregated_counts.append(entry)
                
            for aggregated_count in aggregated_counts:
              if aggregated_count in sampling_resource_dict_single:
                sampling_resource_dict[aggregated_count] += sampling_resource_dict_single[aggregated_count]
    #_____

    textnormalization_dict = {}
    if len(final_returned_columns) > 0:
      textnormalization_dict = {final_returned_columns[0] : {'norm_category' : norm_category, \
                                                              'norm_params' : mlti_norm_params_column_dict, \
                                                              'textlabelsdict' : textlabelsdict, \
                                                              'textcolumns' : textcolumns, \
                                                              'inputtextcolumns' : inputtextcolumns, \
                                                              'norm_columnkey_dict' : norm_columnkey_dict, \
                                                              'norm_column_dict_list' : norm_column_dict_list, \
                                                              'messy_data_headers' : messy_data_headers, \
                                                              'dtype' : dtype, \
                                                              'max_encoding_for_dtype_convert_dict' : max_encoding_for_dtype_convert_dict, \
                                                              'final_returned_columns' : final_returned_columns, \
                                                              'sampling_resource_dict' : sampling_resource_dict, \
                                                              'trainnoise' : trainnoise,
                                                              'testnoise' : testnoise,
                                                              'flip_prob' : flip_prob,
                                                              'test_flip_prob' : test_flip_prob,
                                                              'suffix' : suffix, \
                                                              'inplace' : inplace}}

    column_dict_list = []
    for tc in final_returned_columns:

      if tc != final_returned_columns[0]:
        textnormalization_dict = {}
      
      column_dict = {tc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : textnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : final_returned_columns, \
                           'categorylist' : final_returned_columns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
    
    return mdf_train, mdf_test, column_dict_list

  def _process_cnsl(self, mdf_train, mdf_test, column, category, treecategory, postprocess_dict, params = {}):
    '''
    #process_cnsl(mdf_train, mdf_test, column, category, postprocess_dict, params = {})
    #function to consolidate multiple categoric columns into a single categoric column
    #this is similar to what is done in Binary
    #now that we have the messy data convention from 8.10 wanted to get comparable functionality in family trees
    #recommend applying an upstream mlto to align to a common form between input columns as ordinal
    #note that this will convert all entries into strings prior to consolidation
    
    #note that cnsl pads out the representation of each input with underscore characters
    #which relies on convention of ordinal encoded input to avoid edge case
    #based on padding to width of max string with in encoding space
    '''

    suffixoverlap_results = {}
    
    #initialize parameters

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    if 'messy_data_headers' in params:
      messy_data_headers = params['messy_data_headers']
      if messy_data_headers == []:
        messy_data_headers = [column]
    else:
      messy_data_headers = False
      
#     if 'inplace' in params:
#       inplace = params['inplace']
#     else:
#       inplace = False
      
    if 'pad_character' in params:
      pad_character = params['pad_character']
    else:
      pad_character = '_'
      
    if 'upstream_ordinal' in params:
      upstream_ordinal = params['upstream_ordinal']
    else:
      upstream_ordinal = True
      
    #this function is intended to be applied donstream of a potentially multi column encoding 
    #meaning there may be multiple columns serving as target
    #we'll access the upstream categorylist stored in postprocess_dict
    #or otherwise if this is an upstream trasnform just apply to the received column
    if column in postprocess_dict['column_dict']:
      inputtextcolumns = postprocess_dict['column_dict'][column]['categorylist']
    else:
      #origcolumn scenario when not applied as downstream transform
      #note that this scenario also is automatically applied 
      #when this transform applied downstream of a concurrent MLinfilltype with mlti
      #since at that point column won't be in column_dict
      #which is desired behavior for that use case
      inputtextcolumns = [column]
      
    if isinstance(messy_data_headers, list) and column not in postprocess_dict['column_dict']:
      inputtextcolumns = messy_data_headers.copy()
    
    #the returned columns will each have consistent suffix appending
    textcolumn = (column + '_' + suffix)
    
    #we don't offer inplace support for most DP transforms since they usually involve support columns
    #we might be able to make inplace work for this one
    
    #check suffixoverlap before copying
    suffixoverlap_results = \
    self.__df_check_suffixoverlap(mdf_train, textcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
    
    mdf_train[textcolumn] = ''
    mdf_test[textcolumn] = ''
    
    #we'll do two versions with and without integer conversion to remove float trailing 0's
    if upstream_ordinal is True:

      #find the max string length to pad out with underscroe character
      #an alternate character can be selected by parameter pad_character
      unique_list = []
      for inputtextcolumn in inputtextcolumns:
        unique_list += list(mdf_train[inputtextcolumn].unique())
      #get longest length entry
      longest_len = len([str(int(x)) for x in unique_list if unique_list.index(x) == 0 or len(str(int(x))) > len(str(int(unique_list[unique_list.index(x)-1])))])
      
    else:

      #find the max string length to pad out with underscroe character
      #an alternate character can be selected by parameter pad_character
      unique_list = list(mdf_train[inputtextcolumns].unique())
      #get longest length entry
      longest_len = len([str(x) for x in unique_list if unique_list.index(x) == 0 or len(int(x)) > len(str(unique_list[unique_list.index(x)-1]))])

    #add a contingency factor for case where encoding space wasn't fully represented
    #this scenario is not common in our library
    #as used here if ordinal encoding space found in train set was 10^3, this will support encodings up to 10^10
    longest_len += 7
    
    #now consolidate with padding
    #we'll do two versions with and without integer conversion to remove float trailing 0's  
    for inputtextcolumn in inputtextcolumns:

      if upstream_ordinal is True:
        mdf_train[textcolumn] += mdf_train[inputtextcolumn].astype(int).astype(str).str.pad(longest_len, side='left', fillchar=pad_character).str.slice(start=0, stop=longest_len)
        mdf_test[textcolumn] += mdf_test[inputtextcolumn].astype(int).astype(str).str.pad(longest_len, side='left', fillchar=pad_character).str.slice(start=0, stop=longest_len)
        
      else:
        mdf_train[textcolumn] += mdf_train[inputtextcolumn].astype(str).str.pad(longest_len, side='left', fillchar=pad_character).str.slice(start=0, stop=longest_len)
        mdf_test[textcolumn] += mdf_test[inputtextcolumn].astype(str).str.pad(longest_len, side='left', fillchar=pad_character).str.slice(start=0, stop=longest_len)

    column_dict_list = []

    normalization_dict = {}

    normalization_dict = {textcolumn : {'longest_len' : longest_len,
                                        'pad_character' : pad_character,
                                        'upstream_ordinal' : upstream_ordinal,
                                        'messy_data_headers' : messy_data_headers,
                                        'suffix' : suffix,
                                        'inputtextcolumns' : inputtextcolumns,
                                       }}
    
    for textcolumn in [textcolumn]:
    
      column_dict = {textcolumn : {'category' : treecategory, \
                                   'origcategory' : category, \
                                   'normalization_dict' : normalization_dict, \
                                   'origcolumn' : column, \
                                   'inputcolumn' : column, \
                                   'columnslist' : [textcolumn], \
                                   'categorylist' : [textcolumn], \
                                   'infillmodel' : False, \
                                   'infillcomplete' : False, \
                                   'suffixoverlap_results' : suffixoverlap_results, \
                                   'deletecolumn' : False}}

      column_dict_list.append(column_dict)
    
    return mdf_train, mdf_test, column_dict_list

  def _process_lngt(self, df, column, category, treecategory, postprocess_dict, params = {}):
    '''
    #processing funciton that length of string for each entry
    #such as a heuristic for information content
    #default infill is len(str(np.nan)) = 3
    #note this is a "singleprocess" function since is applied to single dataframe
    '''

    suffixoverlap_results = {}
    
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    suffixcolumn = column + '_' + suffix
    
    if inplace is not True:
      
      #copy source column into new column
      df, suffixoverlap_results = \
      self.__df_copy_train(df, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(df, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      df.rename(columns = {column : suffixcolumn}, inplace = True)
    
    #apply defaultinfill based on processdict entry
    #(this will default to naninfill)
    df, defaultinfill_dict = \
    self.__apply_defaultinfill(df, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)

    df[suffixcolumn] = df[suffixcolumn].astype(str).transform(len)
    
    #grab a fe4w driftreport metrics:
    #get maximum value of training column
    maximum = df[suffixcolumn].max()
    
    #get minimum value of training column
    minimum = df[suffixcolumn].min()
    
    #get minimum value of training column
    mean = df[suffixcolumn].mean()
    
    #get standard deviation of training column
    std = df[suffixcolumn].std()

    #create list of columns
    columns = [suffixcolumn]

    #library defaults to int32 data type for integer mlinfilltype
    df[suffixcolumn] = df[suffixcolumn].astype(np.int32)

    #create normalization dictionary
    normalization_dict = {suffixcolumn : {'maximum' : maximum, \
                                          'minimum' : minimum, \
                                          'mean' : mean, \
                                          'std' : std, \
                                          'suffix' : suffix, \
                                          'defaultinfill_dict' : defaultinfill_dict,
                                          'inplace' : inplace }}

    #store some values in the nmbr_dict{} for use later in ML infill methods
    column_dict_list = []
    
    for nc in columns:

      if nc != columns[0]:
        normalization_dict = {}

      column_dict = { nc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : normalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : columns, \
                           'categorylist' : columns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)

    return df, column_dict_list

  def _process_bnst(self, df, column, category, treecategory, postprocess_dict, params = {}):
    """
    intended for application downstream of a 1010, multirt, or concurrent_act MLinfilltype encoding (with boolean integer entries)
    bnst converts multicolumn categoric representations to a single column representation
    returning a single column string aggregation of the received multi column input
    this may be useful if the upstream categoric transform isn't available in ordinal form
    as some  learning libraries may not accept multi column categoric encodings for label sets
    note that an additional downstream ordinal encoding could be specified in family tree to convert output to numeric
    
    note that bnst is very similar to what has been used in prior transforms such as inverseprocess_1010 which preceded custom_inversion_1010
    
    parameter upstreaminteger should be passed as True when upstream contains integers (such as boolean integer or ordinal int)
    else can pass as False when upstream contains string entries
    
    since reducing the number of columns does not support inplace
    
    implemented in the singleprocess convention
    """

    suffixoverlap_results = {}
    
    if 'upstreaminteger' in params:
      upstreaminteger = params['upstreaminteger']
    else:
      upstreaminteger = True
    
    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
    
    #this function is intended to be applied donstream of a 1010, multirt, or concurrent encoding
    #meaning there may be multiple columns serving as target
    #we'll access the upstream categorylist stored in postprocess_dict
    if column in postprocess_dict['column_dict']:
      inputtextcolumns = postprocess_dict['column_dict'][column]['categorylist']
    else:
      inputtextcolumns = [column]
    
    #note that column will be the first column in the upstream categorylist
    suffixcolumn = column + '_' + suffix
    
    #check for suffix overlap before initizlizing suffixcolumn
    suffixoverlap_results = \
    self.__df_check_suffixoverlap(df, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
    
    df[suffixcolumn] = ''

    for inputtextcolumn in inputtextcolumns:
      
      if upstreaminteger is True:

        #only supports inputtextcolumn containing entries of 0 and 1, assumes infill performed upstream
        #partly motivated by inversion requires a fixed character width 
        df[suffixcolumn] = df[suffixcolumn] + pd.to_numeric(df[inputtextcolumn], errors='coerce').fillna(0).abs().astype(int).clip(lower=0, upper=1).astype(str)

      #we need a fixed character width for inversion        
      else:
        df[suffixcolumn] = df[suffixcolumn] + df[inputtextcolumn].astype(str)
        
    normalization_dict = {suffixcolumn : {'upstreaminteger' : upstreaminteger, \
                                          'suffix' : suffix, \
                                          'inputtextcolumns' : inputtextcolumns}}

    #store some values in the nmbr_dict{} for use later in ML infill methods
    column_dict_list = []

    for nc in [suffixcolumn]:

      if nc != [suffixcolumn][0]:
        normalization_dict = {}
      
      column_dict = { nc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : normalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : [suffixcolumn], \
                           'categorylist' : [suffixcolumn], \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
        
    return df, column_dict_list
  
  def _process_UPCS(self, df, column, category, treecategory, postprocess_dict, params = {}):
    '''
    #processing funciton that converts columns to uppercase strings
    #such as to allow consistnet encoding if data has upper/lower case discrepencies
    #default infill is a distinct entry as string NAN
    #note that with assigninfill this can be converted to other infill methods
    #returns same dataframe with new column of name suffixcolumn
    #note this is a "singleprocess" function since is applied to single dataframe
    '''

    suffixoverlap_results = {}
    
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False
    
    if 'activate' in params:
      activate = params['activate']
    else:
      activate = True

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    suffixcolumn = column + '_' + suffix
    
    if inplace is not True:
      
      #copy source column into new column
      df, suffixoverlap_results = \
      self.__df_copy_train(df, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(df, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      df.rename(columns = {column : suffixcolumn}, inplace = True)

    #apply defaultinfill based on processdict entry
    #(this will default to naninfill)
    df, defaultinfill_dict = \
    self.__apply_defaultinfill(df, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    
    #convert to uppercase string based on activate parameter
    if activate is True:
      #convert column to uppercase string except for nan infill points
      df = \
      self.__autowhere(df, suffixcolumn, df[suffixcolumn] == df[suffixcolumn], df[suffixcolumn].astype(str).str.upper(), specified='replacement')

    #create list of columns
    UPCScolumns = [suffixcolumn]

    #create normalization dictionary
    normalization_dict = {suffixcolumn : {'activate' : activate, 
                                          'defaultinfill_dict' : defaultinfill_dict, 
                                          'suffix' : suffix}}

    #store some values in the nmbr_dict{} for use later in ML infill methods
    column_dict_list = []
    
    for nc in UPCScolumns:

      if nc != UPCScolumns[0]:
        normalization_dict = {}

      column_dict = { nc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : normalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : UPCScolumns, \
                           'categorylist' : UPCScolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False, \
                           'inplace' : inplace}}

      column_dict_list.append(column_dict)

    return df, column_dict_list

  def _process_splt(self, mdf_train, mdf_test, column, category, \
                         treecategory, postprocess_dict, params = {}):
    '''
    #process_splt(mdf_train, mdf_test, column, category)
    #preprocess column with categorical entries as strings
    #identifies overlaps of subsets of those strings and records
    #as a new boolan column
    #for example, if a categoical set consisted of unique values 
    #['west', 'north', 'northeast']
    #then a new column would be created idenitifying cells which included 
    #'north' in their entries
    #(here for north and northeast)
    #returns as column titled origcolumn_splt_entry    
    #missing values are ignored by default
    '''

    suffixoverlap_results = {}
    
    #overlap_lengths = [20, 19, 18, 17, 16, 15, 14, 13, 12, 11, 10, 9, 8, 7 , 6, 5]
    
    if 'minsplit' in params:
      minsplit = params['minsplit'] - 1
    else:
      minsplit = 4
      
    if 'space_and_punctuation' in params:
      space_and_punctuation = params['space_and_punctuation']
    else:
      space_and_punctuation = True
      
    if 'excluded_characters' in params:
      excluded_characters = params['excluded_characters']
    else:
      excluded_characters = [' ', ',', '.', '?', '!', '(', ')']
      
    if 'concurrent_activations' in params:
      concurrent_activations = params['concurrent_activations']
    else:
      concurrent_activations = False
      
    if 'int_headers' in params:
      int_headers = params['int_headers']
    else:
      int_headers = False
      
    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    if 'test_same_as_train' in params:
      test_same_as_train = params['test_same_as_train']
    else:
      test_same_as_train = False
    
    #first we find overlaps from mdf_train
    
    unique_list = list(mdf_train[column].unique())

    unique_list = list(map(str, unique_list))
    
    maxlength = max(len(x) for x in unique_list)
    
    overlap_lengths = list(range(maxlength - 1, minsplit, -1))

    overlap_dict = {}

    #we'll populate overlap_dict as
    #{extract_with_overlap : [list of associate categories with that overlap]}

    #we'll cycle through the overlap lengths and only record an overlap 
    #if it is not a subset of those already recorded
    
    for overlap_length in overlap_lengths:

      for unique in unique_list:

        len_unique = len(unique)

        if len_unique >= overlap_length:

          nbr_iterations = len_unique - overlap_length + 1

          for i in range(nbr_iterations):

            extract = unique[i:(overlap_length+i)]

            extract_already_in_overlap_dict = False

            for key in overlap_dict:

              len_key = len(key)

              if len_key >= overlap_length:

                nbr_iterations3 = len_key - overlap_length + 1

                for k in range(nbr_iterations3):

                  extract3 = key[k:(overlap_length+k)]
                  
                  if concurrent_activations is False:

                    if extract == extract3:

                      extract_already_in_overlap_dict = True
                      
                      break
                      
                  elif concurrent_activations is True:
                    
                    if extract == extract3 and unique in overlap_dict[key]:

                      extract_already_in_overlap_dict = True
                      
                      break
                      
                if extract_already_in_overlap_dict is True:
                  
                  break

            if extract_already_in_overlap_dict is False:

              for unique2 in unique_list:

                if unique2 != unique:

                  len_unique2 = len(unique2)

                  nbr_iterations2 = len_unique2 - overlap_length + 1

                  for j in range(nbr_iterations2):

                    extract2 = unique2[j:(overlap_length+j)]

                    #________
                    
                    if space_and_punctuation is True:

                      if extract2 == extract:

                        if extract in overlap_dict:

                          if unique2 not in overlap_dict[extract]:

                            overlap_dict[extract].append(unique2)
                            
                            if concurrent_activations is False:

                              break

                          if unique not in overlap_dict[extract]:

                            overlap_dict[extract].append(unique)
                            
                            if concurrent_activations is False:

                              break

                        #else if we don't have a key for extract
                        else:

                          overlap_dict.update({extract : [unique, unique2]})
                          
                          if concurrent_activations is False:

                            break
                          
                    elif space_and_punctuation is False:
                      
                      for scrub_punctuation in excluded_characters:
                        
                        extract2 = extract2.replace(scrub_punctuation, '')
                        
                      #if any punctuation was scrubbed these two extracts will be different lengths
                      if extract2 == extract:

                        if extract in overlap_dict:

                          if unique2 not in overlap_dict[extract]:

                            overlap_dict[extract].append(unique2)
                            
                            if concurrent_activations is False:

                              break

                          if unique not in overlap_dict[extract]:

                            overlap_dict[extract].append(unique)
                            
                            if concurrent_activations is False:

                              break

                        #else if we don't have a key for extract
                        else:

                          overlap_dict.update({extract : [unique, unique2]})
                          
                          if concurrent_activations is False:

                            break
     
    #now for mdf_test we'll only consider those overlaps already identified from train set
    
    if test_same_as_train is True:
      test_overlap_dict = overlap_dict
    
    elif test_same_as_train is False:

      unique_list_test = list(mdf_test[column].unique())

      unique_list_test = list(map(str, unique_list_test))

      test_overlap_dict = {}

      train_keys = list(overlap_dict)

      train_keys.sort(key = len, reverse=True)

      for key in train_keys:

        test_overlap_dict.update({key:[]})

      for dict_key in train_keys:

        for unique_test in unique_list_test:

          len_key = len(dict_key)

          if len(unique_test) >= len_key:

            nbr_iterations4 = len(unique_test) - len_key + 1

            for l in range(nbr_iterations4):

              extract4 = unique_test[l:(len_key+l)]

              if extract4 == dict_key:

                test_overlap_dict[dict_key].append(unique_test)

                if concurrent_activations is False:

                  break
    
    newcolumns = []

    for dict_key in overlap_dict:

      newcolumn = column + '_' + suffix + '_' + dict_key
      
      mdf_train, suffixoverlap_results = \
      self.__df_copy_train(mdf_train, column, newcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_test[newcolumn] = mdf_test[column].copy()

      #apply defaultinfill based on processdict entry
      #(this will default to naninfill)
      mdf_train, defaultinfill_dict = \
      self.__apply_defaultinfill(mdf_train, newcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
      mdf_test, _1 = \
      self.__apply_defaultinfill(mdf_test, newcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)

      mdf_train[newcolumn] = mdf_train[newcolumn].astype(str)
      mdf_test[newcolumn] = mdf_test[newcolumn].astype(str)

      mdf_train[newcolumn] = mdf_train[newcolumn].isin(overlap_dict[dict_key])
      mdf_train[newcolumn] = mdf_train[newcolumn].astype(np.int8)
      
      mdf_test[newcolumn] = mdf_test[newcolumn].isin(test_overlap_dict[dict_key])
      mdf_test[newcolumn] = mdf_test[newcolumn].astype(np.int8)

      newcolumns.append(newcolumn)
      
    preint_newcolumns = newcolumns.copy()
      
    if int_headers is True:
      
      int_labels_dict = {}
      i = 0
      for entry in newcolumns:
        int_labels_dict.update({entry : column + '_' + suffix + '_' + str(i)})
        i += 1
        
      newcolumns = [int_labels_dict[entry] for entry in newcolumns]
        
      #now convert column headers from string to int convention
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, newcolumns, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_train = mdf_train.rename(columns=int_labels_dict)
      mdf_test  = mdf_test.rename(columns=int_labels_dict)

      inverse_int_labels_dict = {value:key for key,value in int_labels_dict.items()}
      for key in inverse_int_labels_dict:
        inverse_int_labels_dict[key] = inverse_int_labels_dict[key][len(column) + 1:]
        
    else:
      int_labels_dict = False
      inverse_int_labels_dict = False

    textnormalization_dict = {}
    if len(newcolumns) > 0:
      textnormalization_dict = {newcolumns[0] : {'suffix' : suffix, \
                                                'test_same_as_train' : test_same_as_train, \
                                                'overlap_dict' : overlap_dict, \
                                                'splt_newcolumns_splt'   : newcolumns, \
                                                'minsplit' : minsplit, \
                                                'concurrent_activations' : concurrent_activations, \
                                                'preint_newcolumns' : preint_newcolumns, \
                                                'int_headers' : int_headers, \
                                                'int_labels_dict' : int_labels_dict, \
                                                'defaultinfill_dict' : defaultinfill_dict,
                                                'inverse_int_labels_dict' : inverse_int_labels_dict}}
    
    column_dict_list = []

    for tc in newcolumns:

      if tc != newcolumns[0]:
        textnormalization_dict = {}
      
      column_dict = {tc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : textnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : newcolumns, \
                           'categorylist' : newcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
      
    if len(newcolumns) == 0:
      
      column_dict_list = []
    
    return mdf_train, mdf_test, column_dict_list

  def _process_spl2(self, mdf_train, mdf_test, column, category, \
                         treecategory, postprocess_dict, params = {}):
    '''
    #process_spl2(mdf_train, mdf_test, column, category)
    #preprocess column with categorical entries as strings
    #identifies overlaps of subsets of those strings and replaces entries 
    #with their redecued overlap
    #replaces entries without overlap to 0 (unique to spl5)
    #for example, if a categorical set consisted of unique values 
    #['west', 'north', 'northeast']
    #then a new column would be created in which the entry 'north' 
    #replaced cells with north in their entries
    #(here for north and northeast)
    #and cells with west would be set to 0
    #returns as column titled origcolumn_spl2
    #missing values are ignored by default
    #this alternative to splt may be benficial for instance if one wanted 
    #to follow with an ordl encoding
    '''

    suffixoverlap_results = {}
    
#     overlap_lengths = [20, 19, 18, 17, 16, 15, 14, 13, 12, 11, 10, 9, 8, 7 , 6, 5]

    if 'minsplit' in params:
        
      minsplit = params['minsplit'] - 1
    
    else:
      
      minsplit = 4
      
    if 'space_and_punctuation' in params:
      space_and_punctuation = params['space_and_punctuation']
    else:
      space_and_punctuation = True
      
    if 'excluded_characters' in params:
      excluded_characters = params['excluded_characters']
    else:
      excluded_characters = [' ', ',', '.', '?', '!', '(', ')']
      
    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    if 'test_same_as_train' in params:
      test_same_as_train = params['test_same_as_train']
    else:
      test_same_as_train = False
      
    if 'consolidate_nonoverlaps' in params:
      consolidate_nonoverlaps = params['consolidate_nonoverlaps']
    else:
      consolidate_nonoverlaps = False
    
    #first we find overlaps from mdf_train
    
    unique_list = list(mdf_train[column].unique())

    unique_list = list(map(str, unique_list))
    
    maxlength = max(len(x) for x in unique_list)
    
    overlap_lengths = list(range(maxlength - 1, minsplit, -1))

    overlap_dict = {}

    #we'll populate overlap_dict as
    #{extract_with_overlap : [list of associate categories with that overlap]}

    #we'll cycle through the overlap lengths and only record an overlap 
    #if it is not a subset of those already recorded
    
    for overlap_length in overlap_lengths:

      for unique in unique_list:

        len_unique = len(unique)

        if len_unique >= overlap_length:

          nbr_iterations = len_unique - overlap_length + 1

          for i in range(nbr_iterations):

            extract = unique[i:(overlap_length+i)]

            extract_already_in_overlap_dict = False

            for key in overlap_dict:

              len_key = len(key)

              if len_key >= overlap_length:

                nbr_iterations3 = len_key - overlap_length + 1

                for k in range(nbr_iterations3):

                  extract3 = key[k:(overlap_length+k)]

                  if extract == extract3:

                    extract_already_in_overlap_dict = True
                    
                    break
                    
                if extract_already_in_overlap_dict is True:
                  
                  break

            if extract_already_in_overlap_dict is False:

              for unique2 in unique_list:

                if unique2 != unique:

                  len_unique2 = len(unique2)

                  nbr_iterations2 = len_unique2 - overlap_length + 1

                  for j in range(nbr_iterations2):

                    extract2 = unique2[j:(overlap_length+j)]

                    #________
                    
                    if space_and_punctuation is True:

                      if extract2 == extract:

                        if extract in overlap_dict:

                          if unique2 not in overlap_dict[extract]:

                            overlap_dict[extract].append(unique2)
                            
                            break

                          if unique not in overlap_dict[extract]:

                            overlap_dict[extract].append(unique)
                            
                            break

                        #else if we don't have a key for extract
                        else:

                          overlap_dict.update({extract : [unique, unique2]})
                          
                          break
                          
                    elif space_and_punctuation is False:
                      
                      for scrub_punctuation in excluded_characters:
                        
                        extract2 = extract2.replace(scrub_punctuation, '')
                        
                      #if any punctuation was scrubbed these two extracts will be different lengths
                      if extract2 == extract:

                        if extract in overlap_dict:

                          if unique2 not in overlap_dict[extract]:

                            overlap_dict[extract].append(unique2)
                            
                            break

                          if unique not in overlap_dict[extract]:

                            overlap_dict[extract].append(unique)
                            
                            break

                        #else if we don't have a key for extract
                        else:

                          overlap_dict.update({extract : [unique, unique2]})
                          
                          break
      
    #now for mdf_test we'll only consider those overlaps already 
    #identified from train set
    
    if test_same_as_train is True:
      test_overlap_dict = overlap_dict
    
    elif test_same_as_train is False:
    
      unique_list_test = list(mdf_test[column].unique())

      unique_list_test = list(map(str, unique_list_test))

      test_overlap_dict = {}

      train_keys = list(overlap_dict)

      train_keys.sort(key = len, reverse=True)

      for key in train_keys:

        test_overlap_dict.update({key:[]})

      for dict_key in train_keys:

        for unique_test in unique_list_test:

          len_key = len(dict_key)

          if len(unique_test) >= len_key:

            nbr_iterations4 = len(unique_test) - len_key + 1

            for l in range(nbr_iterations4):

              extract4 = unique_test[l:(len_key+l)]

              if extract4 == dict_key:

                test_overlap_dict[dict_key].append(unique_test)

                break
    
    #so that was all comparable to splt, now for spl2 we'll create a new 
    #dictionary structred as
    #{original unique value : overlap extract for replacement}
    
    #since one original unique value may have entries as multiple overlaps, 
    #we'll prioritize overlaps with
    #longer string lengths and then alphabetical
    
    spl2_overlap_dict = {}
    
    overlap_key_list = list(overlap_dict)
    
    overlap_key_list.sort()
    overlap_key_list.sort(key = len, reverse=True)
    
    for overlap_key in overlap_key_list:
      
      for entry in overlap_dict[overlap_key]:
        
        if entry not in spl2_overlap_dict:
          
          spl2_overlap_dict.update({entry : overlap_key})
    
    #here's where we identify values to set to 0 for spl5
    spl5_zero_dict = {}
    if consolidate_nonoverlaps is True:
      for entry in unique_list:
        if entry not in spl2_overlap_dict:
          spl5_zero_dict.update({entry : 0})
    
    #then we'll do same for test set
    
    spl2_test_overlap_dict = {}
    
    test_overlap_key_list = list(test_overlap_dict)
    
    test_overlap_key_list.sort()
    test_overlap_key_list.sort(key = len, reverse=True)
    
    for overlap_key in test_overlap_key_list:
      
      for entry in test_overlap_dict[overlap_key]:
        
        if entry not in spl2_test_overlap_dict:
          
          spl2_test_overlap_dict.update({entry : overlap_key})
    
    #here's where we identify values to set to 0 for spl5
    spl5_test_zero_dict = {}
    if consolidate_nonoverlaps is True:

      if test_same_as_train is True:
        unique_list_test = list(mdf_test[column].unique())
        unique_list_test = list(map(str, unique_list_test))

      for entry in unique_list_test:
        if entry not in spl2_test_overlap_dict:
          spl5_test_zero_dict.update({entry : 0})
    
    newcolumns = []

#     for dict_key in overlap_dict:

    newcolumn = column + '_' + suffix
    
    mdf_train, suffixoverlap_results = \
    self.__df_copy_train(mdf_train, column, newcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
    
    mdf_test[newcolumn] = mdf_test[column].copy()

    #apply defaultinfill based on processdict entry
    mdf_train, defaultinfill_dict = \
    self.__apply_defaultinfill(mdf_train, newcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    mdf_test, _1 = \
    self.__apply_defaultinfill(mdf_test, newcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)
    
    mdf_train[newcolumn] = mdf_train[newcolumn].astype(str)
    mdf_test[newcolumn] = mdf_test[newcolumn].astype(str)

    mdf_train[newcolumn] = mdf_train[newcolumn].replace(spl2_overlap_dict)
    mdf_train[newcolumn] = mdf_train[newcolumn].replace(spl5_zero_dict)

#       mdf_train[newcolumn] = mdf_train[column].isin(overlap_dict[dict_key])
#       mdf_train[newcolumn] = mdf_train[newcolumn].astype(np.int8)

    mdf_test[newcolumn] = mdf_test[newcolumn].replace(spl2_test_overlap_dict)
    mdf_test[newcolumn] = mdf_test[newcolumn].replace(spl5_test_zero_dict)

#       mdf_test[newcolumn] = mdf_test[column].isin(test_overlap_dict[dict_key])
#       mdf_test[newcolumn] = mdf_test[newcolumn].astype(np.int8)

    newcolumns.append(newcolumn)

    textnormalization_dict = {}
    if len(newcolumns) > 0:
      textnormalization_dict = {newcolumns[0] : {'suffix' : suffix, \
                                                'test_same_as_train' : test_same_as_train, \
                                                'consolidate_nonoverlaps' : consolidate_nonoverlaps, \
                                                'overlap_dict' : overlap_dict, \
                                                'spl2_newcolumns'   : newcolumns, 
                                                'spl2_overlap_dict' : spl2_overlap_dict, \
                                                'spl2_test_overlap_dict' : spl2_test_overlap_dict, \
                                                'spl5_zero_dict' : spl5_zero_dict, \
                                                'defaultinfill_dict' : defaultinfill_dict,
                                                'minsplit' : minsplit}}
    
    column_dict_list = []

    for tc in newcolumns:

      if tc != newcolumns[0]:
        textnormalization_dict = {}
      
      column_dict = {tc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : textnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : newcolumns, \
                           'categorylist' : newcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
      
    if len(newcolumns) == 0:
      
      column_dict_list = []

    return mdf_train, mdf_test, column_dict_list

  def _process_sp19(self, mdf_train, mdf_test, column, category, \
                         treecategory, postprocess_dict, params = {}):
    '''
    #process_splt(mdf_train, mdf_test, column, category)
    #preprocess column with categorical entries as strings
    #identifies overlaps of subsets of those strings and records
    #as a new boolan column
    #for example, if a categoical set consisted of unique values 
    #['west', 'north', 'northeast']
    #then a new column would be created idenitifying cells which included 
    #'north' in their entries
    #(here for north and northeast)
    #returns as column titled origcolumn_splt_entry    
    #missing values are ignored by default
    
    #sp15 is comparable to splt but multiple concurrent activations allowed
    #so requires a different MLinfilltype in processdict
    
    #sp19 is comparable to sp15 but with a returned binary encoding aggregation
    '''

    suffixoverlap_results = {}
    
    #overlap_lengths = [20, 19, 18, 17, 16, 15, 14, 13, 12, 11, 10, 9, 8, 7 , 6, 5]
    
    if 'minsplit' in params:
      minsplit = params['minsplit'] - 1
    else:
      minsplit = 4
      
    if 'space_and_punctuation' in params:
      space_and_punctuation = params['space_and_punctuation']
    else:
      space_and_punctuation = True
      
    if 'excluded_characters' in params:
      excluded_characters = params['excluded_characters']
    else:
      excluded_characters = [' ', ',', '.', '?', '!', '(', ')']
      
    if 'int_headers' in params:
      int_headers = params['int_headers']
    else:
      int_headers = False
      
    #note that same MLinfilltype in processdict ('1010')
    #may be used for both configurations but applying concurrent_activations = False
    #with sp11 is less efficient then running splt
    if 'concurrent_activations' in params:
      concurrent_activations = params['concurrent_activations']
    else:
      concurrent_activations = True
      
    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    if 'test_same_as_train' in params:
      test_same_as_train = params['test_same_as_train']
    else:
      test_same_as_train = False
    
    #first we find overlaps from mdf_train
    
    unique_list = list(mdf_train[column].unique())

    unique_list = list(map(str, unique_list))
    
    maxlength = max(len(x) for x in unique_list)
    
    overlap_lengths = list(range(maxlength - 1, minsplit, -1))

    overlap_dict = {}

    #we'll populate overlap_dict as
    #{extract_with_overlap : [list of associate categories with that overlap]}

    #we'll cycle through the overlap lengths and only record an overlap 
    #if it is not a subset of those already recorded
    
    for overlap_length in overlap_lengths:

      for unique in unique_list:

        len_unique = len(unique)

        if len_unique >= overlap_length:

          nbr_iterations = len_unique - overlap_length + 1

          for i in range(nbr_iterations):

            extract = unique[i:(overlap_length+i)]

            extract_already_in_overlap_dict = False

            for key in overlap_dict:

              len_key = len(key)

              if len_key >= overlap_length:

                nbr_iterations3 = len_key - overlap_length + 1

                for k in range(nbr_iterations3):

                  extract3 = key[k:(overlap_length+k)]
                  
                  if concurrent_activations is False:

                    if extract == extract3:

                      extract_already_in_overlap_dict = True
                      
                      break
                      
                  elif concurrent_activations is True:
                    
                    if extract == extract3 and unique in overlap_dict[key]:

                      extract_already_in_overlap_dict = True
                      
                      break
                      
                if extract_already_in_overlap_dict is True:
                  
                  break

            if extract_already_in_overlap_dict is False:

              for unique2 in unique_list:

                if unique2 != unique:

                  len_unique2 = len(unique2)

                  nbr_iterations2 = len_unique2 - overlap_length + 1

                  for j in range(nbr_iterations2):

                    extract2 = unique2[j:(overlap_length+j)]

                    #________
                    
                    if space_and_punctuation is True:

                      if extract2 == extract:

                        if extract in overlap_dict:

                          if unique2 not in overlap_dict[extract]:

                            overlap_dict[extract].append(unique2)
                            
                            if concurrent_activations is False:

                              break

                          if unique not in overlap_dict[extract]:

                            overlap_dict[extract].append(unique)
                            
                            if concurrent_activations is False:

                              break

                        #else if we don't have a key for extract
                        else:

                          overlap_dict.update({extract : [unique, unique2]})
                          
                          if concurrent_activations is False:

                            break
                          
                    elif space_and_punctuation is False:
                      
                      for scrub_punctuation in excluded_characters:
                        
                        extract2 = extract2.replace(scrub_punctuation, '')
                        
                      #if any punctuation was scrubbed these two extracts will be different lengths
                      if extract2 == extract:

                        if extract in overlap_dict:

                          if unique2 not in overlap_dict[extract]:

                            overlap_dict[extract].append(unique2)
                            
                            if concurrent_activations is False:

                              break

                          if unique not in overlap_dict[extract]:

                            overlap_dict[extract].append(unique)
                            
                            if concurrent_activations is False:

                              break

                        #else if we don't have a key for extract
                        else:

                          overlap_dict.update({extract : [unique, unique2]})
                          
                          if concurrent_activations is False:

                            break
        
    #now for mdf_test we'll only consider those overlaps already identified from train set
    
    if test_same_as_train is True:
      test_overlap_dict = overlap_dict
    
    elif test_same_as_train is False:

      unique_list_test = list(mdf_test[column].unique())

      unique_list_test = list(map(str, unique_list_test))

      test_overlap_dict = {}

      train_keys = list(overlap_dict)

      train_keys.sort(key = len, reverse=True)

      for key in train_keys:

        test_overlap_dict.update({key:[]})

      for dict_key in train_keys:

        for unique_test in unique_list_test:

          len_key = len(dict_key)

          if len(unique_test) >= len_key:

            nbr_iterations4 = len(unique_test) - len_key + 1

            for l in range(nbr_iterations4):

              extract4 = unique_test[l:(len_key+l)]

              if extract4 == dict_key:

                test_overlap_dict[dict_key].append(unique_test)

                if concurrent_activations is False:

                  break
    
    newcolumns = []

    for dict_key in overlap_dict:

      newcolumn = column + '_sp15_' + dict_key
      
      mdf_train, suffixoverlap_results = \
      self.__df_copy_train(mdf_train, column, newcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
#       mdf_train[newcolumn] = mdf_train[column].copy()
      
      mdf_test[newcolumn] = mdf_test[column].copy()

      #apply defaultinfill based on processdict entry
      #(this will default to naninfill)
      mdf_train, defaultinfill_dict = \
      self.__apply_defaultinfill(mdf_train, newcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
      mdf_test, _1 = \
      self.__apply_defaultinfill(mdf_test, newcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)

      mdf_train[newcolumn] = mdf_train[newcolumn].astype(str)
      mdf_test[newcolumn] = mdf_test[newcolumn].astype(str)

      mdf_train[newcolumn] = mdf_train[newcolumn].isin(overlap_dict[dict_key])
      mdf_train[newcolumn] = mdf_train[newcolumn].astype(np.int8)
      
      mdf_test[newcolumn] = mdf_test[newcolumn].isin(test_overlap_dict[dict_key])
      mdf_test[newcolumn] = mdf_test[newcolumn].astype(np.int8)

      newcolumns.append(newcolumn)
      
    preint_newcolumns = newcolumns.copy()
      
    if int_headers is True:
      
      int_labels_dict = {}
      i = 0
      for entry in newcolumns:
        int_labels_dict.update({entry : column + '_sp15_' + str(i)})
        i += 1
        
      #now convert column headers from string to int convention
      mdf_train = mdf_train.rename(columns=int_labels_dict)
      mdf_test  = mdf_test.rename(columns=int_labels_dict)

      newcolumns = [int_labels_dict[entry] for entry in newcolumns]

      inverse_int_labels_dict = {value:key for key,value in int_labels_dict.items()}
      for key in inverse_int_labels_dict:
        inverse_int_labels_dict[key] = inverse_int_labels_dict[key][len(column) + 1:]
        
    else:
      int_labels_dict = False
      inverse_int_labels_dict = False
    
    column_dict_list = []
    
    #begin binary encoding of set, leaving the int_headers here out of convenience, not really needed
    
    if len(newcolumns) > 0:
      
      sp19_column = column + '_' + suffix
    
      #aggregate collection of activations as string set
      #the suffix 'activations_' is to avoid potential of overlap with binary encoding and aggregated activations
      mdf_train[sp19_column] = 'activations_'
      mdf_test[sp19_column] = 'activations_'

      for entry in newcolumns:
        mdf_train[sp19_column] = mdf_train[sp19_column] + mdf_train[entry].astype(str)
        mdf_test[sp19_column] = mdf_test[sp19_column] + mdf_test[entry].astype(str)

      #extract categories for column labels
      #note that .unique() extracts the labels as a numpy array
      labels_train = list(mdf_train[sp19_column].unique())
      labels_train.sort()
      labels_test = list(mdf_test[sp19_column].unique())
      labels_test.sort()

      #labels_train is a list of strings, insert missing data marker
      labels_train = labels_train + [np.nan]
      labels_test = labels_test + [np.nan]

      #get length of the list
      listlength = len(labels_train)

      #calculate number of columns we'll need
      binary_column_count = int(np.ceil(np.log2(listlength)))

      #initialize dictionaryt to store encodings
      binary_encoding_dict = {}
      encoding_list = []

      for i in range(listlength):

        #this converts the integer i to binary encoding
        #where f is an f string for inserting the column coount into the string to designate length of encoding
        #0 is to pad out the encoding with 0's for the length
        #and b is telling it to convert to binary 
        #note this returns a string
        encoding = format(i, f"0{binary_column_count}b")

        if i < len(labels_train):

          #store the encoding in a dictionary
          binary_encoding_dict.update({labels_train[i] : encoding})

          #store the encoding in a list for checking in next step
          encoding_list.append(encoding)

      #clear up memory
      del encoding_list
  #     del overlap_list

      #new driftreport metric _1010_activations_dict
      _1010_activations_dict = {}
      for key in binary_encoding_dict:
        sumcalc = (mdf_train[sp19_column] == key).sum() 
        ratio = sumcalc / mdf_train[sp19_column].shape[0]
        _1010_activations_dict.update({key:ratio})

      #replace the cateogries in train set via ordinal trasnformation
      mdf_train[sp19_column] = mdf_train[sp19_column].astype('object').replace(binary_encoding_dict) 

      #in test set, we'll need to strike any categories that weren't present in train
      #first let'/s identify what applies
      testspecificcategories = list(set(labels_test)-set(labels_train))

      #so we'll just replace those items with our plug value
      testplug_dict = dict(zip(testspecificcategories, [np.nan] * len(testspecificcategories)))
      mdf_test[sp19_column] = mdf_test[sp19_column].astype('object').replace(testplug_dict)  

      #now we'll apply the 1010 transformation to the test set
      mdf_test[sp19_column] = mdf_test[sp19_column].astype('object').replace(binary_encoding_dict)    

      #ok let's create a list of columns to store each entry of the binary encoding
      _1010_columnlist = []

      for i in range(binary_column_count):

        _1010_columnlist.append(column + '_' + suffix + '_' + str(i))

      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, _1010_columnlist, suffixoverlap_results, postprocess_dict['printstatus'])

      #now let's store the encoding
      i=0
      for _1010_column in _1010_columnlist:

        mdf_train[_1010_column] = mdf_train[sp19_column].str.slice(i,i+1).astype(np.int8)

        mdf_test[_1010_column] = mdf_test[sp19_column].str.slice(i,i+1).astype(np.int8)

        i+=1

      #now delete the support column
      del mdf_train[sp19_column]
      del mdf_test[sp19_column]

      for entry in newcolumns:
        del mdf_train[entry]
        del mdf_test[entry]

      #now store the column_dict entries
      categorylist = _1010_columnlist

      normalization_dict = {}
      if len(categorylist) > 0:
        normalization_dict = {categorylist[0] : {'suffix' : suffix, \
                                                'test_same_as_train' : test_same_as_train, \
                                                '_1010_binary_encoding_dict' : binary_encoding_dict, \
                                                '_1010_binary_column_count' : binary_column_count, \
                                                '_1010_activations_dict' : _1010_activations_dict, \
                                                'categorylist' : categorylist, \
                                                'overlap_dict' : overlap_dict, \
                                                'splt_newcolumns_sp19'   : newcolumns, \
                                                'minsplit' : minsplit, \
                                                'concurrent_activations' : concurrent_activations, \
                                                'preint_newcolumns' : preint_newcolumns, \
                                                'int_headers' : int_headers, \
                                                'int_labels_dict' : int_labels_dict, \
                                                'defaultinfill_dict' : defaultinfill_dict,
                                                'inverse_int_labels_dict' : inverse_int_labels_dict}}

      column_dict_list = []

      for tc in categorylist:

        if tc != categorylist[0]:
          normalization_dict = {}

        column_dict = {tc : {'category' : treecategory, \
                             'origcategory' : category, \
                             'normalization_dict' : normalization_dict, \
                             'origcolumn' : column, \
                             'inputcolumn' : column, \
                             'columnslist' : categorylist, \
                             'categorylist' : categorylist, \
                             'infillmodel' : False, \
                             'infillcomplete' : False, \
                             'suffixoverlap_results' : suffixoverlap_results, \
                             'deletecolumn' : False}}

        column_dict_list.append(column_dict)
      
    else:
      
      column_dict_list = []
    
    return mdf_train, mdf_test, column_dict_list

  def _process_sbst(self, mdf_train, mdf_test, column, category, \
                         treecategory, postprocess_dict, params = {}):
    '''
    #process_sbst(mdf_train, mdf_test, column, category)
    #preprocess column with categorical entries as strings
    #identifies cases where a full unique value is present 
    #as a subset of a longer length unique value
    #and returns one-hot activations to aggregate those cases
    
    #accepts parameters concurrent_activations to allow mulitple activations
    #actually let's make concurrent activations the default
    #and int_headers for privacy preserving headers
    
    #this differs from other string parsing functions in that
    #only complete entries are checked for presence as subsets in other entries
    '''

    suffixoverlap_results = {}
      
    if 'concurrent_activations' in params:
      concurrent_activations = params['concurrent_activations']
    else:
      concurrent_activations = True
      
    if 'int_headers' in params:
      int_headers = params['int_headers']
    else:
      int_headers = False

    if 'minsplit' in params:
      minsplit = params['minsplit']
    else:
      minsplit = 1
      
    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    if 'test_same_as_train' in params:
      test_same_as_train = params['test_same_as_train']
    else:
      test_same_as_train = False
    
    #first we find overlaps from mdf_train
    
    unique_list = list(mdf_train[column].unique())

    unique_list = list(map(str, unique_list))
    
    unique_list = sorted(unique_list, key=len, reverse=True)
    
#     maxlength = max(len(x) for x in unique_list)
    
#     minlength = min(len(x) for x in unique_list)
    
#     overlap_lengths = list(range(maxlength - 1, minlength, -1))

    overlap_dict = {}

    #we'll populate overlap_dict as
    #{extract_with_overlap : [list of associate categories with that overlap]}

    #we'll cycle through the overlap lengths and only record an overlap 
    #if it is not a subset of those already recorded
    
    #unique is what we are searching for
    for unique in unique_list:
      len_unique = len(unique)

      if len_unique >= minsplit:
      
        #unique2 is where we are searching
        for unique2 in unique_list:
          len_unique2 = len(unique2)
          
          if len_unique2 > len_unique:
            
            nbr_iterations = len_unique2 - len_unique + 1
            
            for i in range(nbr_iterations):
              
              extract = unique2[i:(len_unique+i)]
              
              extract_already_in_overlap_dict = False
                    
              if extract_already_in_overlap_dict is False:
                
                if extract == unique:
                  
                  if extract in overlap_dict:
                    
                    if unique2 not in overlap_dict[extract]:
                      
                      overlap_dict[extract].append(unique2)
                      
                      if concurrent_activations is False:

                        break
                        
                    # if unique not in overlap_dict[extract]:
                      
                    #   overlap_dict[extract].append(unique)

                    #   if concurrent_activations is False:

                    #     break
                        
                  #else if we don't have a key for extract
                  else:

                    overlap_dict.update({extract : [unique, unique2]})

                    if concurrent_activations is False:

                      break
                    
    #now for mdf_test we'll only consider those overlaps already identified from train set
    
    if test_same_as_train is True:
      test_overlap_dict = overlap_dict
    
    elif test_same_as_train is False:

      unique_list_test = list(mdf_test[column].unique())

      unique_list_test = list(map(str, unique_list_test))

      unique_list_test = sorted(unique_list_test, key=len, reverse=True)

      test_overlap_dict = {}

      train_keys = list(overlap_dict)

      train_keys.sort(key = len, reverse=True)

      for key in train_keys:

        test_overlap_dict.update({key:[]})

      for dict_key in train_keys:

        for unique_test in unique_list_test:

          len_key = len(dict_key)

          if len(unique_test) >= len_key:

            nbr_iterations4 = len(unique_test) - len_key + 1

            for l in range(nbr_iterations4):

              extract4 = unique_test[l:(len_key+l)]

              if extract4 == dict_key:

                test_overlap_dict[dict_key].append(unique_test)

                if concurrent_activations is False:

                  break
                
    newcolumns = []

    for dict_key in overlap_dict:

      newcolumn = column + '_' + suffix + '_' + dict_key
      
      mdf_train, suffixoverlap_results = \
      self.__df_copy_train(mdf_train, column, newcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
#       mdf_train[newcolumn] = mdf_train[column].copy()
  
      mdf_test[newcolumn] = mdf_test[column].copy()

      #apply defaultinfill based on processdict entry
      #(this will default to naninfill)
      mdf_train, defaultinfill_dict = \
      self.__apply_defaultinfill(mdf_train, newcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
      mdf_test, _1 = \
      self.__apply_defaultinfill(mdf_test, newcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)

      mdf_train[newcolumn] = mdf_train[newcolumn].astype(str)
      mdf_test[newcolumn] = mdf_test[newcolumn].astype(str)

      mdf_train[newcolumn] = mdf_train[newcolumn].isin(overlap_dict[dict_key])
      mdf_train[newcolumn] = mdf_train[newcolumn].astype(np.int8)
      
      mdf_test[newcolumn] = mdf_test[newcolumn].isin(test_overlap_dict[dict_key])
      mdf_test[newcolumn] = mdf_test[newcolumn].astype(np.int8)

      newcolumns.append(newcolumn)
      
    preint_newcolumns = newcolumns.copy()
      
    if int_headers is True:
      
      int_labels_dict = {}
      i = 0
      for entry in newcolumns:
        int_labels_dict.update({entry : column + '_' + suffix + '_' + str(i)})
        i += 1
        
      newcolumns = [int_labels_dict[entry] for entry in newcolumns]
        
      #now convert column headers from string to int convention
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, newcolumns, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_train = mdf_train.rename(columns=int_labels_dict)
      mdf_test  = mdf_test.rename(columns=int_labels_dict)

      inverse_int_labels_dict = {value:key for key,value in int_labels_dict.items()}
      for key in inverse_int_labels_dict:
        inverse_int_labels_dict[key] = inverse_int_labels_dict[key][len(column) + 1:]
        
    else:
      int_labels_dict = False
      inverse_int_labels_dict = False

    textnormalization_dict = {}
    if len(newcolumns) > 0:
      textnormalization_dict = {newcolumns[0] : {'suffix' : suffix, \
                                                'test_same_as_train' : test_same_as_train, \
                                                'overlap_dict' : overlap_dict, \
                                                'splt_newcolumns_sbst'   : newcolumns, \
                                                'minsplit' : minsplit, \
                                                'concurrent_activations' : concurrent_activations, \
                                                'preint_newcolumns' : preint_newcolumns, \
                                                'int_headers' : int_headers, \
                                                'int_labels_dict' : int_labels_dict, \
                                                'defaultinfill_dict' : defaultinfill_dict,
                                                'inverse_int_labels_dict' : inverse_int_labels_dict}}
    
    column_dict_list = []

    for tc in newcolumns:

      if tc != newcolumns[0]:
        textnormalization_dict = {}
      
      column_dict = {tc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : textnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : newcolumns, \
                           'categorylist' : newcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
      
    if len(newcolumns) == 0:
      
      column_dict_list = []
    
    return mdf_train, mdf_test, column_dict_list

  def _process_sbs3(self, mdf_train, mdf_test, column, category, \
                         treecategory, postprocess_dict, params = {}):
    '''
    #process_sbst(mdf_train, mdf_test, column, category)
    #preprocess column with categorical entries as strings
    #identifies cases where a full unique value is present 
    #as a subset of a longer length unique value
    #and returns one-hot activations to aggregate those cases
    
    #accepts parameters concurrent_activations to allow mulitple activations
    #actually let's make concurrent activations the default
    #and int_headers for privacy preserving headers
    
    #this differs from other string parsing functions in that
    #only complete entries are checked for presence as subsets in other entries
    
    #sbs3 is comparable to sbst but with a returned binary encoding aggregation
    '''

    suffixoverlap_results = {}
      
    if 'concurrent_activations' in params:
      concurrent_activations = params['concurrent_activations']
    else:
      concurrent_activations = True
      
    if 'int_headers' in params:
      int_headers = params['int_headers']
    else:
      int_headers = False

    if 'minsplit' in params:
      minsplit = params['minsplit']
    else:
      minsplit = 1
      
    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    if 'test_same_as_train' in params:
      test_same_as_train = params['test_same_as_train']
    else:
      test_same_as_train = False
    
    #first we find overlaps from mdf_train
    
    unique_list = list(mdf_train[column].unique())

    unique_list = list(map(str, unique_list))
    
    unique_list = sorted(unique_list, key=len, reverse=True)
    
#     maxlength = max(len(x) for x in unique_list)
    
#     minlength = min(len(x) for x in unique_list)
    
#     overlap_lengths = list(range(maxlength - 1, minlength, -1))

    overlap_dict = {}

    #we'll populate overlap_dict as
    #{extract_with_overlap : [list of associate categories with that overlap]}

    #we'll cycle through the overlap lengths and only record an overlap 
    #if it is not a subset of those already recorded
    
    #unique is what we are searching for
    for unique in unique_list:
      len_unique = len(unique)

      if len_unique >= minsplit:
      
        #unique2 is where we are searching
        for unique2 in unique_list:
          len_unique2 = len(unique2)
          
          if len_unique2 > len_unique:
            
            nbr_iterations = len_unique2 - len_unique + 1
            
            for i in range(nbr_iterations):
              
              extract = unique2[i:(len_unique+i)]
              
              extract_already_in_overlap_dict = False
                    
              if extract_already_in_overlap_dict is False:
                
                if extract == unique:
                  
                  if extract in overlap_dict:
                    
                    if unique2 not in overlap_dict[extract]:
                      
                      overlap_dict[extract].append(unique2)
                      
                      if concurrent_activations is False:

                        break
                        
                    # if unique not in overlap_dict[extract]:
                      
                    #   overlap_dict[extract].append(unique)

                    #   if concurrent_activations is False:

                    #     break
                        
                  #else if we don't have a key for extract
                  else:

                    overlap_dict.update({extract : [unique, unique2]})

                    if concurrent_activations is False:

                      break
                    
    #now for mdf_test we'll only consider those overlaps already identified from train set
    
    if test_same_as_train is True:
      test_overlap_dict = overlap_dict
    
    elif test_same_as_train is False:

      unique_list_test = list(mdf_test[column].unique())

      unique_list_test = list(map(str, unique_list_test))

      unique_list_test = sorted(unique_list_test, key=len, reverse=True)

      test_overlap_dict = {}

      train_keys = list(overlap_dict)

      train_keys.sort(key = len, reverse=True)

      for key in train_keys:

        test_overlap_dict.update({key:[]})

      for dict_key in train_keys:

        for unique_test in unique_list_test:

          len_key = len(dict_key)

          if len(unique_test) >= len_key:

            nbr_iterations4 = len(unique_test) - len_key + 1

            for l in range(nbr_iterations4):

              extract4 = unique_test[l:(len_key+l)]

              if extract4 == dict_key:

                test_overlap_dict[dict_key].append(unique_test)

                if concurrent_activations is False:

                  break
                
    newcolumns = []

    for dict_key in overlap_dict:

      newcolumn = column + '_sbst_' + dict_key
      
      mdf_train, suffixoverlap_results = \
      self.__df_copy_train(mdf_train, column, newcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
#       mdf_train[newcolumn] = mdf_train[column].copy()
  
      mdf_test[newcolumn] = mdf_test[column].copy()

      #apply defaultinfill based on processdict entry
      #(this will default to naninfill)
      mdf_train, defaultinfill_dict = \
      self.__apply_defaultinfill(mdf_train, newcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
      mdf_test, _1 = \
      self.__apply_defaultinfill(mdf_test, newcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)

      mdf_train[newcolumn] = mdf_train[newcolumn].astype(str)
      mdf_test[newcolumn] = mdf_test[newcolumn].astype(str)

      mdf_train[newcolumn] = mdf_train[newcolumn].isin(overlap_dict[dict_key])
      mdf_train[newcolumn] = mdf_train[newcolumn].astype(np.int8)
      
      mdf_test[newcolumn] = mdf_test[newcolumn].isin(test_overlap_dict[dict_key])
      mdf_test[newcolumn] = mdf_test[newcolumn].astype(np.int8)

      newcolumns.append(newcolumn)
      
    preint_newcolumns = newcolumns.copy()
      
    if int_headers is True:
      
      int_labels_dict = {}
      i = 0
      for entry in newcolumns:
        int_labels_dict.update({entry : column + '_sbst_' + str(i)})
        i += 1
        
      newcolumns = [int_labels_dict[entry] for entry in newcolumns]
        
      #now convert column headers from string to int convention
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, newcolumns, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_train = mdf_train.rename(columns=int_labels_dict)
      mdf_test  = mdf_test.rename(columns=int_labels_dict)

      inverse_int_labels_dict = {value:key for key,value in int_labels_dict.items()}
      for key in inverse_int_labels_dict:
        inverse_int_labels_dict[key] = inverse_int_labels_dict[key][len(column) + 1:]
        
    else:
      int_labels_dict = False
      inverse_int_labels_dict = False
    
    column_dict_list = []
    
    #begin binary encoding of set, leaving the int_headers here out of convenience, not really needed
    
    if len(newcolumns) > 0:
      
      sbs3_column = column + suffix

      #aggregate collection of activations as string set
      #the suffix 'activations_' is to avoid potential of overlap with binary encoding and aggregated activations
      mdf_train[sbs3_column] = 'activations_'
      mdf_test[sbs3_column] = 'activations_'

      for entry in newcolumns:
        mdf_train[sbs3_column] = mdf_train[sbs3_column] + mdf_train[entry].astype(str)
        mdf_test[sbs3_column] = mdf_test[sbs3_column] + mdf_test[entry].astype(str)

      #extract categories for column labels
      #note that .unique() extracts the labels as a numpy array
      labels_train = list(mdf_train[sbs3_column].unique())
      labels_train.sort()
      labels_test = list(mdf_test[sbs3_column].unique())
      labels_test.sort()

      #labels_train is a list of strings, insert missing data marker
      labels_train = labels_train + [np.nan]
      labels_test = labels_test + [np.nan]

      #get length of the list
      listlength = len(labels_train)

      #calculate number of columns we'll need
      binary_column_count = int(np.ceil(np.log2(listlength)))

      #initialize dictionaryt to store encodings
      binary_encoding_dict = {}
      encoding_list = []

      for i in range(listlength):

        #this converts the integer i to binary encoding
        #where f is an f string for inserting the column coount into the string to designate length of encoding
        #0 is to pad out the encoding with 0's for the length
        #and b is telling it to convert to binary 
        #note this returns a string
        encoding = format(i, f"0{binary_column_count}b")

        if i < len(labels_train):

          #store the encoding in a dictionary
          binary_encoding_dict.update({labels_train[i] : encoding})

          #store the encoding in a list for checking in next step
          encoding_list.append(encoding)

      #clear up memory
      del encoding_list
  #     del overlap_list

      #new driftreport metric _1010_activations_dict
      _1010_activations_dict = {}
      for key in binary_encoding_dict:
        sumcalc = (mdf_train[sbs3_column] == key).sum() 
        ratio = sumcalc / mdf_train[sbs3_column].shape[0]
        _1010_activations_dict.update({key:ratio})

      #replace the cateogries in train set via ordinal trasnformation
      mdf_train[sbs3_column] = mdf_train[sbs3_column].astype('object').replace(binary_encoding_dict) 

      #in test set, we'll need to strike any categories that weren't present in train
      #first let'/s identify what applies
      testspecificcategories = list(set(labels_test)-set(labels_train))

      #so we'll just replace those items with our plug value
      testplug_dict = dict(zip(testspecificcategories, [np.nan] * len(testspecificcategories)))
      mdf_test[sbs3_column] = mdf_test[sbs3_column].astype('object').replace(testplug_dict)  

      #now we'll apply the 1010 transformation to the test set
      mdf_test[sbs3_column] = mdf_test[sbs3_column].astype('object').replace(binary_encoding_dict)    

      #ok let's create a list of columns to store each entry of the binary encoding
      _1010_columnlist = []

      for i in range(binary_column_count):

        _1010_columnlist.append(column + '_' + suffix + '_' + str(i))

      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, _1010_columnlist, suffixoverlap_results, postprocess_dict['printstatus'])

      #now let's store the encoding
      i=0
      for _1010_column in _1010_columnlist:

        mdf_train[_1010_column] = mdf_train[sbs3_column].str.slice(i,i+1).astype(np.int8)

        mdf_test[_1010_column] = mdf_test[sbs3_column].str.slice(i,i+1).astype(np.int8)

        i+=1

      #now delete the support column
      del mdf_train[sbs3_column]
      del mdf_test[sbs3_column]

      for entry in newcolumns:
        del mdf_train[entry]
        del mdf_test[entry]

      #now store the column_dict entries
      categorylist = _1010_columnlist

      normalization_dict = {}
      if len(categorylist) > 0:
        normalization_dict = {categorylist[0] : {'suffix' : suffix, \
                                                'test_same_as_train' : test_same_as_train, \
                                                '_1010_binary_encoding_dict' : binary_encoding_dict, \
                                                '_1010_binary_column_count' : binary_column_count, \
                                                '_1010_activations_dict' : _1010_activations_dict, \
                                                'categorylist' : categorylist, \
                                                'overlap_dict' : overlap_dict, \
                                                'splt_newcolumns_sbs3'   : newcolumns, \
                                                'concurrent_activations' : concurrent_activations, \
                                                'minsplit' : minsplit, \
                                                'preint_newcolumns' : preint_newcolumns, \
                                                'int_headers' : int_headers, \
                                                'int_labels_dict' : int_labels_dict, \
                                                'defaultinfill_dict' : defaultinfill_dict,
                                                'inverse_int_labels_dict' : inverse_int_labels_dict}}

      column_dict_list = []

      for tc in categorylist:

        if tc != categorylist[0]:
          normalization_dict = {}

        column_dict = {tc : {'category' : treecategory, \
                             'origcategory' : category, \
                             'normalization_dict' : normalization_dict, \
                             'origcolumn' : column, \
                             'inputcolumn' : column, \
                             'columnslist' : categorylist, \
                             'categorylist' : categorylist, \
                             'infillmodel' : False, \
                             'infillcomplete' : False, \
                             'suffixoverlap_results' : suffixoverlap_results, \
                             'deletecolumn' : False}}

        column_dict_list.append(column_dict)
      
    else:
      
      column_dict_list = []
    
    return mdf_train, mdf_test, column_dict_list

  def _process_hash(self, mdf_train, mdf_test, column, category, treecategory, postprocess_dict, params = {}):
    """
    #applies the "hashing trick" to encode categoric sets
    #returning a set of columns with integers corresponding to words from set vocabulary
    #this is intended for sets with very high cardinality
    #note that the same integer may be returned in different columns 
    #for same word found in different entries
    #works by segregating entries into a list of words based on space seperator
    #stripping any special characters
    #and hashing each word with hashlib md5 hashing algorithm
    #which is converted to integer and taken remainder from a division by vocab_size
    #where vocab_size is based on heuristic
    #the heuristic derives vocab_size based on number of unique entries found in train set times the multipler
    #where if that result is greater than the cap then the heuristic reverts to the cap as vocab_size
    #where for hash the number of unique entries is calculated after extracting words from entries
    #note that if vocab_size is not large enough some of words may be returned with encoding overlap
    #returns set of columns with suffix appenders '_hash_#' where # is integer
    #unless only returning one column then suffix appender is just '_hash'
    #note that entries with fewer words than max word count are padded out with 0
    #also accepts parameter for excluded_characters, space
    #uppercase conversion if desired is performed externally by the UPCS transform
    #if space passed as '' then word extraction doesn't take place
    #user can manually specify a vocab_size with vocab-size parameter
    """

    suffixoverlap_results = {}
    
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False
      
    if 'vocab_size' in params:
      vocab_size = params['vocab_size']
    else:
      vocab_size = False
      
    if 'heuristic_multiplier' in params:
      heuristic_multiplier = params['heuristic_multiplier']
    else:
      heuristic_multiplier = 2
      
    if 'heuristic_cap' in params:
      heuristic_cap = params['heuristic_cap']
    else:
      heuristic_cap = 1024
    
    if 'excluded_characters' in params:
      excluded_characters = params['excluded_characters']
    else:
      excluded_characters = [',', '.', '?', '!', '(', ')']

    if 'space' in params:
      space = params['space']
    else:
      space = ' '

    if 'max_column_count' in params:
      max_column_count = params['max_column_count']
    else:
      max_column_count = False

    #salt can be passed as arbitrary string to ensure privacy of encoding basis
    if 'salt' in params:
      salt = params['salt']
    else:
      salt = ''
      
    #accepts either 'hash' or 'md5', 
    #where hash is quicker since uses native python function instead of hashlib
    if 'hash_alg' in params:
      hash_alg = params['hash_alg']
    else:
      hash_alg = 'hash'

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    suffixcolumn = column + '_' + suffix

    if inplace is not True:
      
      #copy source column into new column
      mdf_train, suffixoverlap_results = \
      self.__df_copy_train(mdf_train, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_test[suffixcolumn] = mdf_test[column].copy()
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_train.rename(columns = {column : suffixcolumn}, inplace = True)
      mdf_test.rename(columns = {column : suffixcolumn}, inplace = True)

    #apply defaultinfill based on processdict entry
    #(this will default to naninfill)
    mdf_train, defaultinfill_dict = \
    self.__apply_defaultinfill(mdf_train, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    mdf_test, _1 = \
    self.__apply_defaultinfill(mdf_test, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)    
      
    #convert column to string, note this means that missing data converted to 'nan'
    mdf_train[suffixcolumn] = mdf_train[suffixcolumn].astype(str)
    mdf_test[suffixcolumn] = mdf_test[suffixcolumn].astype(str)
    
    #now scrub special characters
    for scrub_punctuation in excluded_characters:
      mdf_train[suffixcolumn] = mdf_train[suffixcolumn].str.replace(scrub_punctuation, '', regex=False)
      mdf_test[suffixcolumn] = mdf_test[suffixcolumn].str.replace(scrub_punctuation, '', regex=False)
      
    if hash_alg == 'md5':
      from hashlib import md5
    
    #define some support functions
    def _assemble_wordlist(string):
      """
      converts a string to list of words by splitting words from space characters
      assumes any desired special characters have already been stripped
      """

      wordlist = []
      j = 0
      
      if max_column_count is False:
        for i in range(len(string)+1):
          if i < len(string):
            if string[i] == space:
              if i > 0:
                if string[j] != space:
                  wordlist.append(string[j:i])
              j = i+1

          else:
            if j < len(string):
              if string[j] != space:
                wordlist.append(string[j:i])
      
      #else if we have a cap on number of returned columns
      else:
        wordlist_length = 0
        for i in range(len(string)+1):
          if i < len(string):
            if string[i] == space:
              if i > 0:
                if string[j] != space:
                  wordlist.append(string[j:i])
                  wordlist_length += 1
                  if wordlist_length == max_column_count - 1:
                    j = i+1
                    break
              j = i+1

          else:
            if j < len(string):
              if string[j] != space:
                wordlist.append(string[j:i])
                wordlist_length += 1
                j = i+1
                if wordlist_length == max_column_count - 1:
                  break

        if wordlist_length == max_column_count - 1:
          if j < len(string):
            wordlist.append(string[j:len(string)])

      return wordlist
    
    #now convert entries to lists of words
    #e.g. this converts "Two words" to ['Two', 'words']
    #if you don't want to split words can pass space = ''
    if space != '':
      mdf_train[suffixcolumn] = mdf_train[suffixcolumn].transform(_assemble_wordlist)
      mdf_test[suffixcolumn] = mdf_test[suffixcolumn].transform(_assemble_wordlist)
    else:
      mdf_train[suffixcolumn] = mdf_train[suffixcolumn].transform(lambda x: [x])
      mdf_test[suffixcolumn] = mdf_test[suffixcolumn].transform(lambda x: [x])
      
    #if user didn't specify vocab_size then derive based on heuristic
    if vocab_size is False:
      #now let's derive vocab_size from train set, first convert all entries to a list of lists
      temp_list = pd.Series(mdf_train[suffixcolumn]).tolist()
      #this flattens the list of lists
      temp_list = [item for items in temp_list for item in items]
      #consolidate redundant entries
      temp_list = set(temp_list)
      #get length
      vocab_size = len(temp_list)
      del temp_list
      #calculate the vocab_size based on heuristic_multiplier and heuristic_cap
      vocab_size = int(vocab_size * heuristic_multiplier)
      if vocab_size > heuristic_cap:
        vocab_size = int(heuristic_cap)
        
    def _md5_hash(wordlist):
      """
      applies either md5 hashing to the list of words or python default hashing
      this conversion to integers is known as "the hashing trick"
      md5 is partly inspired by tensorflow keras_preprocessing hashing_trick function
      requires importing from hashlib import md5 which is performed above
      here n is the range of integers for vocabulary
      0 is reserved for use to pad lists of shorter length
      (salt and hash_alg are accessible even though not explicitly passed)
      """
      if hash_alg == 'md5':
        #this is the line (excluding the salt option) that was partly inspired by a line in a Keras function
        return [int(md5((salt + word).encode()).hexdigest(), 16) % (vocab_size-1) + 1 for word in wordlist]
      else:
        return [hash(salt + word) % (vocab_size-1) + 1 for word in wordlist]
        
    #now apply hashing to convert to integers based on vocab_size
    mdf_train[suffixcolumn] = mdf_train[suffixcolumn].transform(_md5_hash)
    mdf_test[suffixcolumn] = mdf_test[suffixcolumn].transform(_md5_hash)
    
    #get max length, i.e. for entry with most words
    max_length = mdf_train[suffixcolumn].transform(len).max()
    
    def _pad_hash(hash_list):
      """
      ensures hashing lists are all same length by padding shorter length lists with 0
      """
      padcount = max_length - len(hash_list)
      if padcount >= 0:
        pad = []
        for i in range(padcount):
          pad = pad + [0]
        hash_list = hash_list + pad
      else:
        #for test data we'll trim if max_length greater than max_length from train data
        hash_list = hash_list[:max_length]

      return hash_list
        
    #the other entries are padded out with 0 to reach same length, if a train entry has longer length it is trimmed
    mdf_train[suffixcolumn] = \
    mdf_train[suffixcolumn].transform(_pad_hash)
    mdf_test[suffixcolumn] = \
    mdf_test[suffixcolumn].transform(_pad_hash)
    
    if max_length > 1:

      hashcolumns = []
      for i in range(max_length):

        hash_column = column + '_' + suffix + '_' + str(i)

        hashcolumns += [hash_column]

        #check for column header overlap
        suffixoverlap_results = \
        self.__df_check_suffixoverlap(mdf_train, hash_column, suffixoverlap_results, postprocess_dict['printstatus'])

        #now populate the column with i'th entry from hashed list
        mdf_train[hash_column] = mdf_train[suffixcolumn].transform(lambda x: x[i])
        mdf_test[hash_column] = mdf_test[suffixcolumn].transform(lambda x: x[i])

      #remove support column
      del mdf_train[suffixcolumn]
      del mdf_test[suffixcolumn]
      
    else:
      hashcolumns = [suffixcolumn]
      
      #now populate the column with i'th entry from hashed list
      mdf_train[suffixcolumn] = mdf_train[suffixcolumn].transform(lambda x: x[0])
      mdf_test[suffixcolumn] = mdf_test[suffixcolumn].transform(lambda x: x[0])

    #returned data type is conditional on the size of encoding space
    for hashcolumn in hashcolumns:

      max_encoding = vocab_size - 1
      if max_encoding <= 255:
        mdf_train[hashcolumn] = mdf_train[hashcolumn].astype(np.uint8)
        mdf_test[hashcolumn] = mdf_test[hashcolumn].astype(np.uint8)
      elif max_encoding <= 65535:
        mdf_train[hashcolumn] = mdf_train[hashcolumn].astype(np.uint16)
        mdf_test[hashcolumn] = mdf_test[hashcolumn].astype(np.uint16)
      else:
        mdf_train[hashcolumn] = mdf_train[hashcolumn].astype(np.uint32)
        mdf_test[hashcolumn] = mdf_test[hashcolumn].astype(np.uint32)
    
    hashnormalization_dict = {}
    if len(hashcolumns) > 0:
      hashnormalization_dict = {hashcolumns[0] : {'hashcolumns' : hashcolumns, \
                                                  'col_count' : max_length, \
                                                  'vocab_size_hash' : vocab_size, \
                                                  'heuristic_multiplier' : heuristic_multiplier, \
                                                  'heuristic_cap' : heuristic_cap, \
                                                  'max_length' : max_length, \
                                                  'excluded_characters' : excluded_characters, \
                                                  'space' : space, \
                                                  'salt' : salt, \
                                                  'max_column_count' : max_column_count, \
                                                  'hash_alg' : hash_alg, \
                                                  'suffix' : suffix, \
                                                  'defaultinfill_dict' : defaultinfill_dict,
                                                  'inplace' : inplace}}

    column_dict_list = []

    for hc in hashcolumns:

      if hc != hashcolumns[0]:
        hashnormalization_dict = {}
      
      column_dict = { hc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : hashnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : hashcolumns, \
                           'categorylist' : hashcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
      
    return mdf_train, mdf_test, column_dict_list

  def _process_hs10(self, mdf_train, mdf_test, column, category, treecategory, postprocess_dict, params = {}):
    """
    #applies the "hashing trick" to encode categoric sets
    #returning a set of columns binary encoded corresponding to integers returned from hash
    #this is intended for sets with very high cardinality
    #note that the same activation set may be returned for different entries
    #works by hashing each entry with hashlib md5 hashing algorithm
    #which is converted to integer and taken remainder from a division by vocab_size
    #where vocab_size is passed parameter intended to align with vocabulary size defaulting to 128
    #note that if vocab_size is not large enough some of words may be returned with encoding overlap
    #returns set of columns with suffix appenders '_hs10_#' where # is integer
    #uppercase conversion if desired is performed externally by the UPCS transform
    #applies a heuristic to
    #set a vocab_size based on number unique entries times heuristic_multiplier parameter which defaults to 2
    #also accepts heuristic_cap parameter where if unique * heuristic_muyltipler > heuristic_cap
    #then vocab_size = heuristic_cap
    #or user can manually specify a vocab_size instead of relying on heuristic
    """
    
    suffixoverlap_results = {}
    
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False
      
    if 'vocab_size' in params:
      vocab_size = params['vocab_size']
    else:
      vocab_size = False
      
    if 'heuristic_multiplier' in params:
      heuristic_multiplier = params['heuristic_multiplier']
    else:
      heuristic_multiplier = 2
      
    if 'heuristic_cap' in params:
      heuristic_cap = params['heuristic_cap']
    else:
      heuristic_cap = 1024

    #salt can be passed as arbitrary string to ensure privacy of encoding basis
    if 'salt' in params:
      salt = params['salt']
    else:
      salt = ''

    #a list of strings that are scrubbed from entries e.g. punctuations
    if 'excluded_characters' in params:
      excluded_characters = params['excluded_characters']
    else:
      excluded_characters = []
      
    #accepts either 'hash' or 'md5', 
    #where hash is quicker since uses native python function instead of hashlib 
    #(md5 was the original basis, hash is new default)
    if 'hash_alg' in params:
      hash_alg = params['hash_alg']
    else:
      hash_alg = 'hash'

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    suffixcolumn = column + '_' + suffix

    if inplace is not True:
      
      #copy source column into new column
      mdf_train, suffixoverlap_results = \
      self.__df_copy_train(mdf_train, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_test[suffixcolumn] = mdf_test[column].copy()
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_train.rename(columns = {column : suffixcolumn}, inplace = True)
      mdf_test.rename(columns = {column : suffixcolumn}, inplace = True)

    #apply defaultinfill based on processdict entry
    #(this will default to naninfill)
    mdf_train, defaultinfill_dict = \
    self.__apply_defaultinfill(mdf_train, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    mdf_test, _1 = \
    self.__apply_defaultinfill(mdf_test, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)
      
    #convert column to string, note this means that missing data converted to 'nan'
    mdf_train[suffixcolumn] = mdf_train[suffixcolumn].astype(str)
    mdf_test[suffixcolumn] = mdf_test[suffixcolumn].astype(str)

    #now scrub special characters
    for scrub_punctuation in excluded_characters:
      mdf_train[suffixcolumn] = mdf_train[suffixcolumn].str.replace(scrub_punctuation, '', regex=False)
      mdf_test[suffixcolumn] = mdf_test[suffixcolumn].str.replace(scrub_punctuation, '', regex=False)
    
    if vocab_size is False:
      #calculate the vocab_size based on heuristic_multiplier and heuristic_cap
      vocab_size = int(mdf_train[suffixcolumn].nunique() * heuristic_multiplier)
      if vocab_size > heuristic_cap:
        vocab_size = int(heuristic_cap)
      
    if hash_alg == 'md5':
      from hashlib import md5
    
    def _md5_hash(entry):
      """
      applies hashing to the list of words
      this conversion to ingtegers is known as "the hashing trick"
      requires importing from hashlib import md5 if hash_alg = "md5"
      here n is the range of integers for vocabulary
      similar comment to _process_hash that md5 was partly inspired by a line in a keras function
      """
      if hash_alg == 'md5':
        return int(md5((salt + entry).encode()).hexdigest(), 16) % (vocab_size)
      else:
        return hash(salt + entry) % (vocab_size)

    #now apply hashing to convert to integers based on vocab_size
    mdf_train[suffixcolumn] = mdf_train[suffixcolumn].transform(_md5_hash)
    mdf_test[suffixcolumn] = mdf_test[suffixcolumn].transform(_md5_hash)

    binary_column_count = int(np.ceil(np.log2(vocab_size)))
    
    #convert integer encoding to binary
    mdf_train[suffixcolumn] = mdf_train[suffixcolumn].transform(bin)
    mdf_test[suffixcolumn] = mdf_test[suffixcolumn].transform(bin)
    
    #convert format to string of digits
    mdf_train[suffixcolumn] = mdf_train[suffixcolumn].str[2:]
    mdf_test[suffixcolumn] = mdf_test[suffixcolumn].str[2:]
    
    #pad out zeros
    mdf_train[suffixcolumn] = mdf_train[suffixcolumn].str.zfill(binary_column_count)
    mdf_test[suffixcolumn] = mdf_test[suffixcolumn].str.zfill(binary_column_count)
    
    hashcolumns = []
    for i in range(binary_column_count):

      hash_column = column + '_' + suffix + '_' + str(i)
      
      hashcolumns += [hash_column]
      
      #check for column header overlap
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, hash_column, suffixoverlap_results, postprocess_dict['printstatus'])
      
      #now populate the column with i'th entry from hashed list
      mdf_train[hash_column] = mdf_train[suffixcolumn].str[i].astype(np.int8)
      mdf_test[hash_column] = mdf_test[suffixcolumn].str[i].astype(np.int8)
    
    #remove support column
    del mdf_train[suffixcolumn]
    del mdf_test[suffixcolumn]

    hashnormalization_dict = {}
    if len(hashcolumns) > 0:
      hashnormalization_dict = {hashcolumns[0] : {'hashcolumns' : hashcolumns, \
                                                  'col_count' : binary_column_count, \
                                                  'vocab_size' : vocab_size, \
                                                  'heuristic_multiplier' : heuristic_multiplier, \
                                                  'heuristic_cap' : heuristic_cap, \
                                                  'salt' : salt, \
                                                  'excluded_characters' : excluded_characters, \
                                                  'hash_alg' : hash_alg, \
                                                  'suffix' : suffix, \
                                                  'defaultinfill_dict' : defaultinfill_dict,
                                                  'inplace' : inplace}}
    
    column_dict_list = []

    for hc in hashcolumns:

      if hc != hashcolumns[0]:
        hashnormalization_dict = {}
      
      column_dict = { hc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : hashnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : hashcolumns, \
                           'categorylist' : hashcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
      
    return mdf_train, mdf_test, column_dict_list

  def _process_srch(self, mdf_train, mdf_test, column, category, \
                         treecategory, postprocess_dict, params = {}):
    """
    #process_srch(mdf_train, mdf_test, column, category)
    #preprocess column with categorical entries as strings
    #relies on user passed list of strings in search parameter
    #string parses unique entries to identify overlaps with search strings
    #when overlap found returns a column with boolean activation identifiers
    
    #note this differs from original srch in that makes use of pandas str.contains
    #which is expected to be more efficient for unbounded sets
    
    #for example, if a categoical set consisted of unique values 
    #['west', 'north', 'northwest']
    #and a user passed the search parameter as ['west']
    #then a new column would be returned 
    #with activations corresponding to entries of 'west' and 'northwest'
    
    #note that search parameter can include lists of search terms embedded in the list
    #which embedded lists will be aggregated to a single activation
    #for example if we want single activation for female names could pass search = [['Ms.', 'Miss', 'Mrs']] etc
    
    #note this returns all zeros in a column if search value not found
    
    #note returned coluymns are named by search term, e
    #e.g. column + '_srch_' + str(search)
    
    #note that search terms are converted to strings and compared to columns cast as strings

    #missing values are ignored by default
    """

    suffixoverlap_results = {}
        
    if 'search' in params:
      search = params['search']
    else:
      search = []
      
    if 'case' in params:
      case = params['case']
    else:
      case = True

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False
      
    #for this transform, suffixcolumn is not returned in final set
    #applying so that we can allow defaultinfill support without overwriting the input column
    
    suffixcolumn = column + '_' + suffix
    
    if inplace is not True:
      
      #copy source column into new column
      mdf_train, suffixoverlap_results = \
      self.__df_copy_train(mdf_train, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])

      mdf_test[suffixcolumn] = mdf_test[column].copy()
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_train.rename(columns = {column : suffixcolumn}, inplace = True)
      mdf_test.rename(columns = {column : suffixcolumn}, inplace = True)
      
    #apply defaultinfill based on processdict entry
    #(this will default to naninfill)
    mdf_train, defaultinfill_dict = \
    self.__apply_defaultinfill(mdf_train, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    mdf_test, _1 = \
    self.__apply_defaultinfill(mdf_test, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)
      
    #we'll create mirror to account for any embdded lists of search terms for aggregation
    search_preflattening = search.copy()
    #this is kind of hacky just to reuse code below resetting this list to repopulate
    search = []
    aggregated_dict = {}
    
    for entry in search_preflattening:
      if type(entry) != type([]):
        search.append(str(entry))
      else:
        aggregated_dict.update({str(entry[-1]):[]})
        for entry2 in entry[0:-1]:
          search.append(entry2)
          aggregated_dict[str(entry[-1])].append(str(entry2))
        for entry2 in entry[-1:]:
          search.append(entry2)
    
    newcolumns = []
    search_dict = {}
    for searchitem in search:
      search_dict.update({column + '_' + suffix + '_' + str(searchitem) : str(searchitem)})
      
    for newcolumn in search_dict:
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, newcolumn, suffixoverlap_results, postprocess_dict['printstatus'])

      mdf_train = \
      self.__autowhere(mdf_train, newcolumn, mdf_train[suffixcolumn].astype(str).str.contains(search_dict[newcolumn], case=case, regex=False), 1, specified='replacement')

      mdf_test = \
      self.__autowhere(mdf_test, newcolumn, mdf_test[suffixcolumn].astype(str).str.contains(search_dict[newcolumn], case=case, regex=False), 1, specified='replacement')
    
    newcolumns = list(search_dict)
    
    #now we'll address any aggregations fo search terms
    #from search parameter passed with embedded list of search terms
          
    #then after populating activations, we'll put this below
    #inverse_search_dict has key of search term and value of column for activations
    inverse_search_dict = {value:key for key,value in search_dict.items()}
    newcolumns_before_aggregation = newcolumns.copy()
    
    #now we consolidate activations
    #note that this only runs when aggregated_dict was populated with an embedded list of search terms
    for aggregated_dict_key in aggregated_dict:
      if aggregated_dict_key in inverse_search_dict:
        aggregated_dict_key_column = inverse_search_dict[aggregated_dict_key]
        
        for target_for_aggregation in aggregated_dict[aggregated_dict_key]:
          target_for_aggregation_column = inverse_search_dict[target_for_aggregation]

          mdf_train = \
          self.__autowhere(mdf_train, aggregated_dict_key_column, mdf_train[target_for_aggregation_column] == 1, 1, specified='replacement')
          mdf_test = \
          self.__autowhere(mdf_test, aggregated_dict_key_column, mdf_test[target_for_aggregation_column] == 1, 1, specified='replacement')
          
          del mdf_train[target_for_aggregation_column]
          del mdf_test[target_for_aggregation_column]
          
          newcolumns.remove(target_for_aggregation_column)
    
    for newcolumn in newcolumns:

      mdf_train[newcolumn] = mdf_train[newcolumn].astype(np.int8)
      mdf_test[newcolumn] = mdf_test[newcolumn].astype(np.int8)
      
    #remove temporary support column
    del mdf_train[suffixcolumn]
    del mdf_test[suffixcolumn]

    textnormalization_dict = {}
    if len(newcolumns) > 0:
      textnormalization_dict = {newcolumns[0] : {'search_dict' : search_dict, \
                                                'inverse_search_dict' : inverse_search_dict, \
                                                'srch_newcolumns_srch'   : newcolumns, \
                                                'newcolumns_before_aggregation' : newcolumns_before_aggregation, \
                                                'search' : search, \
                                                'search_preflattening' : search_preflattening, \
                                                'aggregated_dict' : aggregated_dict, \
                                                'suffix' : suffix, \
                                                'inplace' : inplace, \
                                                'defaultinfill_dict' : defaultinfill_dict, \
                                                'case' : case}}
    
    column_dict_list = []

    for tc in newcolumns:

      if tc != newcolumns[0]:
        textnormalization_dict = {}
      
      column_dict = {tc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : textnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : newcolumns, \
                           'categorylist' : newcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
      
    if len(newcolumns) == 0:
      
      column_dict_list = []
    
    return mdf_train, mdf_test, column_dict_list

  def _process_src2(self, mdf_train, mdf_test, column, category, \
                        treecategory, postprocess_dict, params = {}):
    """
    #process_src2(mdf_train, mdf_test, column, category)
    #preprocess column with categorical entries as strings
    #relies on user passed list of strings in search parameter
    #string parses unique entries to identify overlaps with search strings
    #when overlap found returns a column with boolean activation identifiers
    
    #for example, if a categoical set consisted of unique values 
    #['west', 'north', 'northwest']
    #and a user passed the search parameter as ['west']
    #then a new column would be returned 
    #with activations corresponding to entries of 'west' and 'northwest'

    #missing values are ignored by default
    
    #assumes that unique values of test set are same or subset of train set
    #for more efficient application in postmunge
    """

    suffixoverlap_results = {}
        
    if 'search' in params:
      search = params['search']
    else:
      search = []
      
    if 'case' in params:
      case = params['case']
    else:
      case = True

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False
      
    #for this transform, suffixcolumn is not returned in final set
    #applying so that we can allow defaultinfill support without overwriting the input column
    
    suffixcolumn = column + '_' + suffix
    
    if inplace is not True:
      
      #copy source column into new column
      mdf_train, suffixoverlap_results = \
      self.__df_copy_train(mdf_train, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])

      mdf_test[suffixcolumn] = mdf_test[column].copy()
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_train.rename(columns = {column : suffixcolumn}, inplace = True)
      mdf_test.rename(columns = {column : suffixcolumn}, inplace = True)
      
    #apply defaultinfill based on processdict entry
    #(this will default to naninfill)
    mdf_train, defaultinfill_dict = \
    self.__apply_defaultinfill(mdf_train, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    mdf_test, _1 = \
    self.__apply_defaultinfill(mdf_test, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)
    
    #convert to uppercase string when case sensitivity not desired based on case parameter
    if case is False:
      #convert column to uppercase string except for nan infill points
      mdf_train = \
      self.__autowhere(mdf_train, suffixcolumn, mdf_train[suffixcolumn] == mdf_train[suffixcolumn], mdf_train[suffixcolumn].astype(str).str.upper(), specified='replacement')
      mdf_test = \
      self.__autowhere(mdf_test, suffixcolumn, mdf_test[suffixcolumn] == mdf_test[suffixcolumn], mdf_test[suffixcolumn].astype(str).str.upper(), specified='replacement')
    
    #first we find overlaps from mdf_train
    
    unique_list = list(mdf_train[suffixcolumn].unique())

    unique_list = list(map(str, unique_list))
    
#     maxlength = max(len(x) for x in unique_list)
    
#     overlap_lengths = list(range(maxlength - 1, minsplit, -1))

    #we'll create mirror to account for any embdded lists of search terms for aggregation
    search_preflattening = search.copy()
    #this is kind of hacky just to reuse code below resetting this list to repopulate
    search = []
    aggregated_dict = {}
    
    for entry in search_preflattening:
      if type(entry) != type([]):
        search.append(str(entry))
      else:
        aggregated_dict.update({str(entry[-1]):[]})
        for entry2 in entry[0:-1]:
          search.append(entry2)
          aggregated_dict[str(entry[-1])].append(str(entry2))
        for entry2 in entry[-1:]:
          search.append(entry2)
          
    #when case sensitivity not desired convert search terms to uppercase strings
    if case is False:
      for i in range(len(search)):
        search[i] = str(search[i]).upper()
      #similarly convert aggregated_dict keys and value lists to uppercase
      aggregated_dict_preconvert = self.__autocopy(aggregated_dict)
      aggregated_dict = {}
      for key, value in aggregated_dict_preconvert.items():
        for i in range(len(value)):
          value[i] = str(value[i]).upper()
        aggregated_dict.update({str(key).upper() : value})

    #we'll populate overlap_dict as
    #{search_string : [list of associate categories with that overlap found]}
    
    overlap_dict = {}
    
    for search_string in search:
      
      overlap_dict.update({search_string : []})
    
    for search_string in search:
      
      len_search_string = len(search_string)
    
      for unique in unique_list:
        
        len_unique = len(unique)
        
        if len_unique >= len_search_string:
          
          nbr_iterations = len_unique - len_search_string
          
          for i in range(nbr_iterations + 1):
            
            extract = unique[i:(len_search_string+i)]
            
            if extract in search:
              
              overlap_dict[extract].append(unique)
    
#     #now for mdf_test
    
#     unique_list_test = list(mdf_test[column].unique())

#     unique_list_test = list(map(str, unique_list_test))

#     test_overlap_dict = {}
    
#     for search_string in search:
      
#       test_overlap_dict.update({search_string : []})
    

#     train_keys = list(overlap_dict)

#     train_keys.sort(key = len, reverse=True)

#     for dict_key in train_keys:

#       for unique_test in unique_list_test:

#         len_key = len(dict_key)

#         if len(unique_test) >= len_key:

#           nbr_iterations4 = len(unique_test) - len_key

#           for l in range(nbr_iterations4 + 1):

#             extract4 = unique_test[l:(len_key+l)]

#             if extract4 == dict_key:

#               test_overlap_dict[dict_key].append(unique_test)
    
    newcolumns = []

    for dict_key in overlap_dict:
      
      if len(overlap_dict[dict_key]) > 0:

        newcolumn = column + '_' + suffix + '_' + dict_key

        mdf_train, suffixoverlap_results = \
        self.__df_copy_train(mdf_train, suffixcolumn, newcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
        
        mdf_test[newcolumn] = mdf_test[suffixcolumn].copy()

        mdf_train[newcolumn] = mdf_train[newcolumn].astype(str)
        mdf_test[newcolumn] = mdf_test[newcolumn].astype(str)

        mdf_train[newcolumn] = mdf_train[newcolumn].isin(overlap_dict[dict_key])
#         mdf_train[newcolumn] = mdf_train[newcolumn].astype(np.int8)

        mdf_test[newcolumn] = mdf_test[newcolumn].isin(overlap_dict[dict_key])
#         mdf_test[newcolumn] = mdf_test[newcolumn].astype(np.int8)

        newcolumns.append(newcolumn)
    
    #now in case there are any aggregated activations, inspired by approach in srch
    inverse_search_dict = dict(zip(search, newcolumns))
    newcolumns_before_aggregation = newcolumns.copy()
    
    #now we consolidate activations
    #note that this only runs when aggregated_dict was populated with an embedded list of search terms
    for aggregated_dict_key in aggregated_dict:
      if aggregated_dict_key in inverse_search_dict:
        aggregated_dict_key_column = inverse_search_dict[aggregated_dict_key]
        
        for target_for_aggregation in aggregated_dict[aggregated_dict_key]:
          target_for_aggregation_column = inverse_search_dict[target_for_aggregation]

          mdf_train = \
          self.__autowhere(mdf_train, aggregated_dict_key_column, mdf_train[target_for_aggregation_column] == 1, 1, specified='replacement')
          mdf_test = \
          self.__autowhere(mdf_test, aggregated_dict_key_column, mdf_test[target_for_aggregation_column] == 1, 1, specified='replacement')
          
          del mdf_train[target_for_aggregation_column]
          del mdf_test[target_for_aggregation_column]
          
          newcolumns.remove(target_for_aggregation_column)
        
    for newcolumn in newcolumns:
      mdf_train[newcolumn] = mdf_train[newcolumn].astype(np.int8)
      mdf_test[newcolumn] = mdf_test[newcolumn].astype(np.int8)
      
    #remove temporary support column
    del mdf_train[suffixcolumn]
    del mdf_test[suffixcolumn]

    textnormalization_dict = {}
    if len(newcolumns) > 0:
      textnormalization_dict = {newcolumns[0] : {'overlap_dict' : overlap_dict, \
                                                'src2_newcolumns_src2'   : newcolumns, \
                                                'newcolumns_before_aggregation' : newcolumns_before_aggregation, \
                                                'search' : search, \
                                                'inverse_search_dict' : inverse_search_dict, \
                                                'aggregated_dict' : aggregated_dict, \
                                                'case' : case, \
                                                'suffix' : suffix, \
                                                'inplace' : inplace, \
                                                'defaultinfill_dict' : defaultinfill_dict,
                                                'search_preflattening' : search_preflattening}}
    
    column_dict_list = []

    for tc in newcolumns:

      if tc != newcolumns[0]:
        textnormalization_dict = {}
      
      column_dict = {tc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : textnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : newcolumns, \
                           'categorylist' : newcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
      
    if len(newcolumns) == 0:
      
      column_dict_list = []
    
    return mdf_train, mdf_test, column_dict_list
  
  def _process_src3(self, mdf_train, mdf_test, column, category, \
                         treecategory, postprocess_dict, params = {}):
    """
    #process_src3(mdf_train, mdf_test, column, category)
    #preprocess column with categorical entries as strings
    #relies on user passed list of strings in search parameter
    #string parses unique entries to identify overlaps with search strings
    #when overlap found returns a column with boolean activation identifiers
    
    #for example, if a categoical set consisted of unique values 
    #['west', 'north', 'northwest']
    #and a user passed the search parameter as ['west']
    #then a new column would be returned 
    #with activations corresponding to entries of 'west' and 'northwest'

    #missing values are ignored by default
    
    #where srch is preferred for unbounded range of unique values
    
    #and src2 preferred when have bounded range of unique values for both train & test
    
    #and speculation is that src3 may be preferred when have a bounded
    #range of unique values but still want capacity to handle values in 
    #test set not found in train set
    """

    suffixoverlap_results = {}
        
    if 'search' in params:
      search = params['search']
    else:
      search = []
      
    if 'case' in params:
      case = params['case']
    else:
      case = True

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False
      
    #for this transform, suffixcolumn is not returned in final set
    #applying so that we can allow defaultinfill support without overwriting the input column
    
    suffixcolumn = column + '_' + suffix
    
    if inplace is not True:
      
      #copy source column into new column
      mdf_train, suffixoverlap_results = \
      self.__df_copy_train(mdf_train, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])

      mdf_test[suffixcolumn] = mdf_test[column].copy()
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_train.rename(columns = {column : suffixcolumn}, inplace = True)
      mdf_test.rename(columns = {column : suffixcolumn}, inplace = True)
      
    #apply defaultinfill based on processdict entry
    #(this will default to naninfill)
    mdf_train, defaultinfill_dict = \
    self.__apply_defaultinfill(mdf_train, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    mdf_test, _1 = \
    self.__apply_defaultinfill(mdf_test, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)
    
    #convert to uppercase string when case sensitivity not desired based on case parameter
    if case is False:
      #convert column to uppercase string except for nan infill points
      mdf_train = \
      self.__autowhere(mdf_train, suffixcolumn, mdf_train[suffixcolumn] == mdf_train[suffixcolumn], mdf_train[suffixcolumn].astype(str).str.upper(), specified='replacement')
      mdf_test = \
      self.__autowhere(mdf_test, suffixcolumn, mdf_test[suffixcolumn] == mdf_test[suffixcolumn], mdf_test[suffixcolumn].astype(str).str.upper(), specified='replacement')
    
    #first we find overlaps from mdf_train
    
    unique_list = list(mdf_train[suffixcolumn].unique())

    unique_list = list(map(str, unique_list))
    
#     maxlength = max(len(x) for x in unique_list)
    
#     overlap_lengths = list(range(maxlength - 1, minsplit, -1))

    #we'll create mirror to account for any embdded lists of search terms for aggregation
    search_preflattening = search.copy()
    #this is kind of hacky just to reuse code below resetting this list to repopulate
    search = []
    aggregated_dict = {}
    
    for entry in search_preflattening:
      if type(entry) != type([]):
        search.append(str(entry))
      else:
        aggregated_dict.update({str(entry[-1]):[]})
        for entry2 in entry[0:-1]:
          search.append(entry2)
          aggregated_dict[str(entry[-1])].append(str(entry2))
        for entry2 in entry[-1:]:
          search.append(entry2)

    #when case sensitivity not desired convert search terms to uppercase strings
    if case is False:
      for i in range(len(search)):
        search[i] = str(search[i]).upper()
      
      #similarly convert aggregated_dict keys and value lists to uppercase
      aggregated_dict_preconvert = self.__autocopy(aggregated_dict)
      aggregated_dict = {}
      for key, value in aggregated_dict_preconvert.items():
        for i in range(len(value)):
          value[i] = str(value[i]).upper()
        aggregated_dict.update({str(key).upper() : value})

    #we'll populate overlap_dict as
    #{search_string : [list of associate categories with that overlap found]}
    
    overlap_dict = {}
    
    for search_string in search:
      
      overlap_dict.update({search_string : []})
    
    for search_string in search:
      
      len_search_string = len(search_string)
    
      for unique in unique_list:
        
        len_unique = len(unique)
        
        if len_unique >= len_search_string:
          
          nbr_iterations = len_unique - len_search_string
          
          for i in range(nbr_iterations + 1):
            
            extract = unique[i:(len_search_string+i)]
            
            if extract in search:
              
              overlap_dict[extract].append(unique)
           
    #now for mdf_test
    
    unique_list_test = list(mdf_test[suffixcolumn].unique())

    unique_list_test = list(map(str, unique_list_test))

    test_overlap_dict = {}
    
    for search_string in search:
      
      test_overlap_dict.update({search_string : []})
    
    train_keys = list(overlap_dict)

    train_keys.sort(key = len, reverse=True)

    for dict_key in train_keys:

      for unique_test in unique_list_test:

        len_key = len(dict_key)

        if len(unique_test) >= len_key:

          nbr_iterations4 = len(unique_test) - len_key

          for l in range(nbr_iterations4 + 1):

            extract4 = unique_test[l:(len_key+l)]

            if extract4 == dict_key:

              test_overlap_dict[dict_key].append(unique_test)
    
    newcolumns = []

    for dict_key in overlap_dict:
      
      if len(overlap_dict[dict_key]) > 0:

        newcolumn = column + '_' + suffix + '_' + dict_key

        mdf_train, suffixoverlap_results = \
        self.__df_copy_train(mdf_train, suffixcolumn, newcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
        
        mdf_test[newcolumn] = mdf_test[suffixcolumn].copy()

        mdf_train[newcolumn] = mdf_train[newcolumn].astype(str)
        mdf_test[newcolumn] = mdf_test[newcolumn].astype(str)

        mdf_train[newcolumn] = mdf_train[newcolumn].isin(overlap_dict[dict_key])
        mdf_train[newcolumn] = mdf_train[newcolumn].astype(np.int8)

        mdf_test[newcolumn] = mdf_test[newcolumn].isin(test_overlap_dict[dict_key])
        mdf_test[newcolumn] = mdf_test[newcolumn].astype(np.int8)

        newcolumns.append(newcolumn)
        
    #now in case there are any aggregated activations, inspired by approach in srch
    inverse_search_dict = dict(zip(search, newcolumns))
    newcolumns_before_aggregation = newcolumns.copy()
    
    #now we consolidate activations
    #note that this only runs when aggregated_dict was populated with an embedded list of search terms
    for aggregated_dict_key in aggregated_dict:
      if aggregated_dict_key in inverse_search_dict:
        aggregated_dict_key_column = inverse_search_dict[aggregated_dict_key]
        
        for target_for_aggregation in aggregated_dict[aggregated_dict_key]:
          target_for_aggregation_column = inverse_search_dict[target_for_aggregation]

          mdf_train = \
          self.__autowhere(mdf_train, aggregated_dict_key_column, mdf_train[target_for_aggregation_column] == 1, 1, specified='replacement')
          mdf_test = \
          self.__autowhere(mdf_test, aggregated_dict_key_column, mdf_test[target_for_aggregation_column] == 1, 1, specified='replacement')
          
          del mdf_train[target_for_aggregation_column]
          del mdf_test[target_for_aggregation_column]
          
          newcolumns.remove(target_for_aggregation_column)
        
    #remove temporary support column
    del mdf_train[suffixcolumn]
    del mdf_test[suffixcolumn]

    textnormalization_dict = {}
    if len(newcolumns) > 0:
      textnormalization_dict = {newcolumns[0] : {'overlap_dict' : overlap_dict, \
                                                'srch_newcolumns_src3'   : newcolumns, \
                                                'aggregated_dict' : aggregated_dict, \
                                                'inverse_search_dict' : inverse_search_dict, \
                                                'case' : case, \
                                                'suffix' : suffix, \
                                                'inplace': inplace, \
                                                'defaultinfill_dict' : defaultinfill_dict, \
                                                'search' : search}}
    
    column_dict_list = []

    for tc in newcolumns:

      if tc != newcolumns[0]:
        textnormalization_dict = {}
      
      column_dict = {tc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : textnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : newcolumns, \
                           'categorylist' : newcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
      
    if len(newcolumns) == 0:
      
      column_dict_list = []
    
    return mdf_train, mdf_test, column_dict_list

  def _process_src4(self, mdf_train, mdf_test, column, category, \
                         treecategory, postprocess_dict, params = {}):
    """
    #process_src4(mdf_train, mdf_test, column, category)
    #preprocess column with categorical entries as strings
    #relies on user passed list of strings in search parameter
    #string parses unique entries to identify overlaps with search strings
    #when overlap found returns a column with boolean activation identifiers
    
    #note this differs from original srch in that makes use of pandas str.contains
    #which is expected to be more efficient for unbounded sets
    
    #for example, if a categoical set consisted of unique values 
    #['west', 'north', 'northwest']
    #and a user passed the search parameter as ['west']
    #then a new column would be returned 
    #with activations corresponding to entries of 'west' and 'northwest'
    
    #note this returns all zeros in a column if search value not found
    
    #note returned coluymns are named by search term, e
    #e.g. column + '_srch_' + str(search)
    
    #note that search terms are converted to strings and compared to columns cast as strings

    #missing values are ignored by default
    
    #src4 builds on the srch by converting to an ordinal activation
    #with 0 reserved for no activations
    #note that if an entry was activated for multiple search terms
    #the order of entries in search parameter will dictate the final encoding
    #(e.g. entries at end of list are prioritized over beginning)
    """

    suffixoverlap_results = {}
        
    if 'search' in params:
      search = params['search']
    else:
      search = []
      
    if 'case' in params:
      case = params['case']
    else:
      case = True
      
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory

    #for this transform, suffixcolumn is not returned in final set
    #applying so that we can allow defaultinfill support without overwriting the input column
    
    suffixcolumn = column + '_' + suffix
    
    if inplace is not True:
      
      #copy source column into new column
      mdf_train, suffixoverlap_results = \
      self.__df_copy_train(mdf_train, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])

      mdf_test[suffixcolumn] = mdf_test[column].copy()
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_train.rename(columns = {column : suffixcolumn}, inplace = True)
      mdf_test.rename(columns = {column : suffixcolumn}, inplace = True)

    if mdf_train[suffixcolumn].dtype.name == 'category':
      mdf_train[suffixcolumn] = mdf_train[suffixcolumn].astype('object')
      mdf_test[suffixcolumn] = mdf_test[suffixcolumn].astype('object')
      
    #apply defaultinfill based on processdict entry
    #(this will default to naninfill)
    mdf_train, defaultinfill_dict = \
    self.__apply_defaultinfill(mdf_train, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    mdf_test, _1 = \
    self.__apply_defaultinfill(mdf_test, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)
      
    #we'll create mirror to account for any embdded lists of search terms for aggregation
    search_preflattening = search.copy()
    #this is kind of hacky just to reuse code below resetting this list to repopulate
    search = []
    aggregated_dict = {}
    
    for entry in search_preflattening:
      if type(entry) != type([]):
        search.append(str(entry))
      else:
        aggregated_dict.update({str(entry[-1]):[]})
        for entry2 in entry[0:-1]:
          search.append(entry2)
          aggregated_dict[str(entry[-1])].append(str(entry2))
        for entry2 in entry[-1:]:
          search.append(entry2)
    
    newcolumns = []
    search_dict = {}
    for searchitem in search:
      search_dict.update({column + '_' + suffix + '_' + str(searchitem) : str(searchitem)})
      
    for newcolumn in search_dict:
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, newcolumn, suffixoverlap_results, postprocess_dict['printstatus'])

      mdf_train = \
      self.__autowhere(mdf_train, newcolumn, mdf_train[suffixcolumn].astype(str).str.contains(search_dict[newcolumn], case=case, regex=False), 1, specified='replacement')

      mdf_test = \
      self.__autowhere(mdf_test, newcolumn, mdf_test[suffixcolumn].astype(str).str.contains(search_dict[newcolumn], case=case, regex=False), 1, specified='replacement')
    
    newcolumns = list(search_dict)

#     for newcolumn in newcolumns:

#       mdf_train[newcolumn] = mdf_train[newcolumn].astype(np.int8)
#       mdf_test[newcolumn] = mdf_test[newcolumn].astype(np.int8)
      
    #ok now let's convert to ordinal for src4
    ordl_dict1 = {}
    ordl_dict2 = {}
    
    #reserve zero for no activations
    i = 1
    for newcolumn in newcolumns:
      ordl_dict1.update({i : newcolumn})
      ordl_dict2.update({newcolumn : i})
      i += 1
      
    mdf_train.loc[:, suffixcolumn] = 0
    mdf_test.loc[:, suffixcolumn] = 0
    
    for newcolumn in newcolumns:

      mdf_train = \
      self.__autowhere(mdf_train, suffixcolumn, mdf_train[newcolumn] == 1, ordl_dict2[newcolumn],  specified='replacement')
      mdf_test = \
      self.__autowhere(mdf_test, suffixcolumn, mdf_test[newcolumn] == 1, ordl_dict2[newcolumn],  specified='replacement')
      
      del mdf_train[newcolumn]
      del mdf_test[newcolumn]
      
    #now we'll address any aggregations fo search terms
    #from search parameter passed with embedded list of search terms
          
    #then after populating activations, we'll put this below
    #inverse_search_dict has key of search term and value of column for activations
    inverse_search_dict = {value:key for key,value in search_dict.items()}
#     newcolumns_before_aggregation = newcolumns.copy()
      
    #now we consolidate activations
    #note that this only runs when aggregated_dict was populated with an embedded list of search terms
    for aggregated_dict_key in aggregated_dict:
      if aggregated_dict_key in inverse_search_dict:
        aggregated_dict_key_column = inverse_search_dict[aggregated_dict_key]
        aggregated_dict_key_encoding = ordl_dict2[aggregated_dict_key_column]
        
        for target_for_aggregation in aggregated_dict[aggregated_dict_key]:
          target_for_aggregation_column = inverse_search_dict[target_for_aggregation]
          target_for_aggregation_encoding = ordl_dict2[target_for_aggregation_column]

          mdf_train = \
          self.__autowhere(mdf_train, suffixcolumn, mdf_train[suffixcolumn] == target_for_aggregation_encoding, aggregated_dict_key_encoding, specified='replacement')
          mdf_test = \
          self.__autowhere(mdf_test, suffixcolumn, mdf_test[suffixcolumn] == target_for_aggregation_encoding, aggregated_dict_key_encoding, specified='replacement')

    #we'll base the integer type on number of ordinal entries
    max_encoding = len(ordl_dict1)
    if max_encoding <= 255:
      mdf_train[suffixcolumn] = mdf_train[suffixcolumn].astype(np.uint8)
      mdf_test[suffixcolumn] = mdf_test[suffixcolumn].astype(np.uint8)
    elif max_encoding <= 65535:
      mdf_train[suffixcolumn] = mdf_train[suffixcolumn].astype(np.uint16)
      mdf_test[suffixcolumn] = mdf_test[suffixcolumn].astype(np.uint16)
    else:
      mdf_train[suffixcolumn] = mdf_train[suffixcolumn].astype(np.uint32)
      mdf_test[suffixcolumn] = mdf_test[suffixcolumn].astype(np.uint32)
    
    column_dict_list = []
    
    #newcolumns are based on the original srch transform
    #src4_newcolumns are after consolidating to ordinal encoding (single entry)
    src4_newcolumns = [suffixcolumn]

    textnormalization_dict = {}
    if len(src4_newcolumns) > 0:
      textnormalization_dict = {src4_newcolumns[0] : {'search_dict' : search_dict, \
                                                      'inverse_search_dict' : inverse_search_dict, \
                                                      'srch_newcolumns_src4' : newcolumns, \
                                                      'src4_newcolumns' : src4_newcolumns, \
                                                      'search' : search, \
                                                      'search_preflattening' : search_preflattening, \
                                                      'aggregated_dict' : aggregated_dict, \
                                                      'case' : case, \
                                                      'ordl_dict1' : ordl_dict1, \
                                                      'activations_list' : list(ordl_dict1), \
                                                      'defaultinfill_dict' : defaultinfill_dict, \
                                                      'suffix' : suffix, \
                                                      'inplace': inplace, \
                                                      'ordl_dict2' : ordl_dict2}}

    for tc in src4_newcolumns:

      if tc != src4_newcolumns[0]:
        textnormalization_dict = {}
      
      column_dict = {tc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : textnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : src4_newcolumns, \
                           'categorylist' : src4_newcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)

    return mdf_train, mdf_test, column_dict_list
  
  def _process_aggt(self, df, column, category, treecategory, postprocess_dict, params = {}):
    """
    #process_aggt(mdf_train, mdf_test, column, category)
    #preprocess column with categorical entries as strings
    #and aggregates differently spelled duplicates into single representation
    #based on user passed parameter 'aggregate'
    #which is a list of lists, where sublists are the aggregation groups
    #and the final representation will be the final item in list
    #note also supports passing aggregate as a single list of terms without embedded lists
    """

    suffixoverlap_results = {}
    
    if 'aggregate' in params:
      aggregate = params['aggregate']
    else:
      aggregate = [[]]

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    suffixcolumn = column + '_' + suffix
      
    df, suffixoverlap_results = \
    self.__df_copy_train(df, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])

    #apply defaultinfill based on processdict entry
    #(this will default to naninfill)
    df, defaultinfill_dict = \
    self.__apply_defaultinfill(df, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)

    for sublist in aggregate:
      
      if not isinstance(sublist, list):
        
        sublist = aggregate
      
        length_sublist = len(sublist)

        for i in range(length_sublist-1):

          df = \
          self.__autowhere(df, suffixcolumn, df[suffixcolumn] == sublist[i], sublist[-1], specified='replacement')
          
        break
      
      else:
        
        length_sublist = len(sublist)

        for i in range(length_sublist-1):

          df = \
          self.__autowhere(df, suffixcolumn, df[suffixcolumn] == sublist[i], sublist[-1], specified='replacement')

    normalization_dict = {suffixcolumn : {'aggregate' : aggregate, 'suffix' : suffix, 'defaultinfill_dict' : defaultinfill_dict}}
    
    nmbrcolumns = [suffixcolumn]
    
    #store some values in the nmbr_dict{} for use later in ML infill methods
    column_dict_list = []

    for nc in nmbrcolumns:

      if nc != nmbrcolumns[0]:
        normalization_dict = {}
      
      column_dict = { nc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : normalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : nmbrcolumns, \
                           'categorylist' : nmbrcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
        
    return df, column_dict_list

  def _process_strn(self, df, column, category, treecategory, postprocess_dict, params = {}):
    """
    #process_strn(df, column, category, postprocess_dict)
    #parses string entries and if any strings present returns longest string
    #i.e. character subsets excluding numerical entries
    #note that since this transform doesn't return numerically enocded data
    #we are leaving infill in place as np.nan
    """

    suffixoverlap_results = {}

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    suffixcolumn = column + '_' + suffix
    
    unique_list = list(df[column].unique())

    unique_list = list(map(str, unique_list))
    
    maxlength = max(len(x) for x in unique_list)
    
    overlap_lengths = list(range(maxlength, 0, -1))

    overlap_dict = {}
    
    for overlap_length in overlap_lengths:

      for unique in unique_list:
        
        if unique not in overlap_dict:

          len_unique = len(unique)

          if len_unique >= overlap_length:
            
            if overlap_length > 1:

              nbr_iterations = len_unique - overlap_length

              for i in range(nbr_iterations + 1):
                
                if unique not in overlap_dict:

                  extract = unique[i:(overlap_length+i)]
                  
                  has_number = False
                  
                  for j in range(len(extract)):
                    
                    if self.__is_number(extract[j]):
                      
                      has_number = True

  #                 extract_already_in_overlap_dict = False

                  if has_number is False:

                    overlap_dict.update({unique : extract})
                
            #else if overlap_length == 1    
            else:
              
              nbr_iterations = len_unique - overlap_length
              
              in_dict = False

              for i in range(nbr_iterations + 1):
                
                if unique not in overlap_dict:

                  extract = unique[i:(overlap_length+i)]

  #                 extract_already_in_overlap_dict = False
  
                  has_number = False
                  
                  for j in range(len(extract)):
                    
                    if self.__is_number(extract[j]):
                      
                      has_number = True

  #                 extract_already_in_overlap_dict = False

                  if has_number is False:
      
                    in_dict = True

                    overlap_dict.update({unique : extract})
                  
              if in_dict is False:

                overlap_dict.update({unique : np.nan})

    suffixoverlap_results = \
    self.__df_check_suffixoverlap(df, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
    
    df[suffixcolumn] = df[column].copy()

    #apply defaultinfill based on processdict entry
    #(this will default to naninfill)
    df, defaultinfill_dict = \
    self.__apply_defaultinfill(df, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)

    df[suffixcolumn] = df[suffixcolumn].astype(str)
    df[suffixcolumn] = df[suffixcolumn].replace(overlap_dict)
    
#     #a few more metrics collected for driftreport
#     #get maximum value of training column
#     maximum = df[column + '_nmrc'].max()
#     #get minimum value of training column
#     minimum = df[column + '_nmrc'].min()
    
    #create list of columns
    nmbrcolumns = [suffixcolumn]

    nmbrnormalization_dict = {suffixcolumn : {'overlap_dict' : overlap_dict, \
                                              'defaultinfill_dict' : defaultinfill_dict,
                                              'suffix' : suffix}}

    #store some values in the nmbr_dict{} for use later in ML infill methods
    column_dict_list = []
    
    for nc in nmbrcolumns:

      if nc != nmbrcolumns[0]:
        nmbrnormalization_dict = {}

      column_dict = { nc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : nmbrnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : nmbrcolumns, \
                           'categorylist' : nmbrcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
        
    return df, column_dict_list

  def _process_strg(self, df, column, category, treecategory, postprocess_dict, params = {}):
    '''
    #str function
    #accepts input of integer categoric sets, such as from an ordinal transform
    #and converts to strings for purposes of categoric recognition in some downstream libaries
    #(eg some libraries will treat integer label sets as targets for regression instead of classificaiton)
    #does not perform infill, just converts entries to string
    '''

    suffixoverlap_results = {}

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    strg_column = column + '_' + suffix
    
    df, suffixoverlap_results = \
    self.__df_copy_train(df, column, strg_column, suffixoverlap_results, postprocess_dict['printstatus'])

    #apply defaultinfill based on processdict entry
    #(this will default to naninfill)
    df, defaultinfill_dict = \
    self.__apply_defaultinfill(df, strg_column, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    
    df[strg_column] = df[strg_column].astype(str)

    column_dict_list = []

    column_dict = {strg_column : {'category' : treecategory, \
                                 'origcategory' : category, \
                                 'normalization_dict' : {strg_column:{'suffix' : suffix, 'defaultinfill_dict' : defaultinfill_dict}}, \
                                 'origcolumn' : column, \
                                 'inputcolumn' : column, \
                                 'columnslist' : [strg_column], \
                                 'categorylist' : [strg_column], \
                                 'infillmodel' : False, \
                                 'infillcomplete' : False, \
                                 'suffixoverlap_results' : suffixoverlap_results, \
                                 'deletecolumn' : False}}
    
    #now append column_dict onto postprocess_dict
    column_dict_list.append(column_dict)

    return df, column_dict_list

  def _process_nmrc(self, df, column, category, treecategory, postprocess_dict, params = {}):
    """
    #process_nmrc(df, column, category, postprocess_dict)
    #parses string entries and if any numbers present returns numbers
    #entries without numbers present subject to infill
    #accepts parameters 
    #convention as numbers/commas/spaces
    #suffix for column suffix identifier
    """

    suffixoverlap_results = {}
    
    if 'convention' in params:
      #accepts numbers/commas/spaces
      convention = params['convention']
    else:
      convention = 'numbers'
      
    if 'suffix' in params:
      #accepts string for suffix appender
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    nmrc_column = column + '_' + suffix
    
    df, suffixoverlap_results = \
    self.__df_copy_train(df, column, nmrc_column, suffixoverlap_results, postprocess_dict['printstatus'])
    
    unique_list = list(df[nmrc_column].unique())

    unique_list = list(map(str, unique_list))
    
    maxlength = max(len(x) for x in unique_list)
    
    overlap_lengths = list(range(maxlength, 0, -1))

    overlap_dict = {}
    
    for overlap_length in overlap_lengths:

      for unique in unique_list:
        
        if unique not in overlap_dict:

          len_unique = len(unique)

          if len_unique >= overlap_length:
            
            if overlap_length > 1:

              nbr_iterations = len_unique - overlap_length

              for i in range(nbr_iterations + 1):
                
                if unique not in overlap_dict:

                  extract = unique[i:(overlap_length+i)]

  #                 extract_already_in_overlap_dict = False
                  
                  if convention == 'numbers':
                  
                    if self.__is_number(extract):

                      overlap_dict.update({unique : float(extract)})
              
                  elif convention == 'commas':
                  
                    if self.__is_number_comma(extract):

                      overlap_dict.update({unique : float(extract.replace(',',''))})
                      
                  elif convention == 'spaces':
                  
                    if self.__is_number_EU(extract):

                      overlap_dict.update({unique : float(extract[0] + extract[1:-1].replace(' ','').replace('.','').replace(',','.') + extract[-1])})
                      
            #else if overlap_length == 1    
            else:
              
              nbr_iterations = len_unique - overlap_length
              
              in_dict = False

              for i in range(nbr_iterations + 1):
                
                if unique not in overlap_dict:

                  extract = unique[i:(overlap_length+i)]

  #                 extract_already_in_overlap_dict = False

                  if self.__is_number(extract):

                    in_dict = True

                    overlap_dict.update({unique : float(extract)})

              if in_dict is False:

                overlap_dict.update({unique : np.nan})

    #apply defaultinfill based on processdict entry
    #(this will default to naninfill)
    df, defaultinfill_dict = \
    self.__apply_defaultinfill(df, nmrc_column, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    
    df[nmrc_column] = df[nmrc_column].astype(str)
    df[nmrc_column] = df[nmrc_column].replace(overlap_dict)

    df[nmrc_column] = pd.to_numeric(df[nmrc_column], errors='coerce')
    
    #get mean of training data
    mean = df[nmrc_column].mean()
    if mean != mean:
      mean = 0
      
    #replace missing data with training set mean as default infill
    df[nmrc_column] = df[nmrc_column].fillna(mean)
    
    #a few more metrics collected for driftreport
    #get maximum value of training column
    maximum = df[nmrc_column].max()
    #get minimum value of training column
    minimum = df[nmrc_column].min()
    
    #create list of columns
    nmbrcolumns = [nmrc_column]

    #populate data structures
    nmbrnormalization_dict = {nmrc_column : {'overlap_dict' : overlap_dict, \
                                            'mean' : mean, \
                                            'maximum' : maximum, \
                                            'minimum' : minimum, \
                                            'convention' : convention, \
                                            'defaultinfill_dict' : defaultinfill_dict,
                                            'suffix' : suffix }}
    
    column_dict_list = []
    
    for nc in nmbrcolumns:

      if nc != nmbrcolumns[0]:
        nmbrnormalization_dict = {}

      column_dict = { nc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : nmbrnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : nmbrcolumns, \
                           'categorylist' : nmbrcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
    
    return df, column_dict_list

  def _process_nmr4(self, mdf_train, mdf_test, column, category, \
                         treecategory, postprocess_dict, params = {}):
    """
    #extract numeric partitions from categoric entries, test treated differently than train
    #accepts parameters
    #convention as numbers/commas/spaces
    #suffix for column suffix identifier
    #test_same_as_train as True/False
    #where True copiues overlap_dict from train for test, False parses test entries not found in train
    """

    suffixoverlap_results = {}
    
    if 'convention' in params:
      #accepts numbers/commas/spaces
      convention = params['convention']
    else:
      convention = 'numbers'
      
    if 'suffix' in params:
      #accepts string for suffix appender
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    if 'test_same_as_train' in params:
      #accepts boolean
      test_same_as_train = params['test_same_as_train']
    else:
      test_same_as_train = True
      
    nmrc_column = column + '_' + suffix
    
    mdf_train, suffixoverlap_results = \
    self.__df_copy_train(mdf_train, column, nmrc_column, suffixoverlap_results, postprocess_dict['printstatus'])
    
    mdf_test[nmrc_column] = mdf_test[column].copy()
    
    #begin parsing train set
    
    unique_list = list(mdf_train[nmrc_column].unique())

    unique_list = list(map(str, unique_list))
    
    maxlength = max(len(x) for x in unique_list)
    
    overlap_lengths = list(range(maxlength, 0, -1))

    overlap_dict = {}
    
    for overlap_length in overlap_lengths:

      for unique in unique_list:
        
        if unique not in overlap_dict:

          len_unique = len(unique)

          if len_unique >= overlap_length:
            
            if overlap_length > 1:

              nbr_iterations = len_unique - overlap_length

              for i in range(nbr_iterations + 1):
                
                if unique not in overlap_dict:

                  extract = unique[i:(overlap_length+i)]

  #                 extract_already_in_overlap_dict = False
                  
                  if convention == 'numbers':
                  
                    if self.__is_number(extract):

                      overlap_dict.update({unique : float(extract)})
              
                  elif convention == 'commas':
                  
                    if self.__is_number_comma(extract):

                      overlap_dict.update({unique : float(extract.replace(',',''))})
                      
                  elif convention == 'spaces':
                  
                    if self.__is_number_EU(extract):

                      overlap_dict.update({unique : float(extract[0] + extract[1:-1].replace(' ','').replace('.','').replace(',','.') + extract[-1])})
                      
            #else if overlap_length == 1    
            else:
              
              nbr_iterations = len_unique - overlap_length
              
              in_dict = False

              for i in range(nbr_iterations + 1):
                
                if unique not in overlap_dict:

                  extract = unique[i:(overlap_length+i)]

  #                 extract_already_in_overlap_dict = False
  
                  if self.__is_number(extract):

                    in_dict = True

                    overlap_dict.update({unique : float(extract)})

              if in_dict is False:

                overlap_dict.update({unique : np.nan})

    #apply defaultinfill based on processdict entry
    mdf_train, defaultinfill_dict = \
    self.__apply_defaultinfill(mdf_train, nmrc_column, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
                
    mdf_train[nmrc_column] = mdf_train[nmrc_column].astype(str)
    mdf_train[nmrc_column] = mdf_train[nmrc_column].replace(overlap_dict)
    
    #now test set
    test_unique_list = list(mdf_test[nmrc_column].unique())
    test_unique_list = list(map(str, test_unique_list))
    extra_test_unique = list(set(test_unique_list) - set(unique_list))

    test_overlap_dict = self.__autocopy(overlap_dict)
    
    if test_same_as_train is True:
      
      for test_unique in extra_test_unique:
        test_overlap_dict.update({str(test_unique) : np.nan})
      
    elif test_same_as_train is False:
      
      testmaxlength = max(len(x) for x in unique_list)

      overlap_lengths = list(range(testmaxlength, 0, -1))

  #     overlap_dict = {}

      for overlap_length in overlap_lengths:

        for unique in extra_test_unique:

          if unique not in test_overlap_dict:

            len_unique = len(unique)

            if len_unique >= overlap_length:

              if overlap_length > 1:

                nbr_iterations = len_unique - overlap_length

                for i in range(nbr_iterations + 1):

                  if unique not in test_overlap_dict:

                    extract = unique[i:(overlap_length+i)]

    #                 extract_already_in_overlap_dict = False
                    
                    if convention == 'numbers':
                    
                      if self.__is_number(extract):

                        test_overlap_dict.update({unique : float(extract)})
                  
                    elif convention == 'commas':
                    
                      if self.__is_number_comma(extract):

                        test_overlap_dict.update({unique : float(extract.replace(',',''))})
                        
                    elif convention == 'spaces':
                    
                      if self.__is_number_EU(extract):

                        test_overlap_dict.update({unique : float(extract[0] + extract[1:-1].replace(' ','').replace('.','').replace(',','.') + extract[-1])})

              #else if overlap_length == 1    
              else:

                nbr_iterations = len_unique - overlap_length

                in_dict = False

                for i in range(nbr_iterations + 1):

                  if unique not in test_overlap_dict:

                    extract = unique[i:(overlap_length+i)]

    #                 extract_already_in_overlap_dict = False
                    
                    if self.__is_number(extract):

                      in_dict = True

                      test_overlap_dict.update({unique : float(extract)})
                    
                if in_dict is False:

                  test_overlap_dict.update({unique : np.nan})

    #apply defaultinfill to test data based on processdict entry
    mdf_test, _1 = \
    self.__apply_defaultinfill(mdf_test, nmrc_column, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)

    #great now that test_overlap_dict is populated
    mdf_test[nmrc_column] = mdf_test[nmrc_column].astype(str)
    mdf_test[nmrc_column] = mdf_test[nmrc_column].replace(test_overlap_dict)

    mdf_train[nmrc_column] = pd.to_numeric(mdf_train[nmrc_column], errors='coerce')
    mdf_test[nmrc_column] = pd.to_numeric(mdf_test[nmrc_column], errors='coerce')

    #get mean of training data
    mean = mdf_train[nmrc_column].mean()
    if mean != mean:
      mean = 0

    #replace missing data with training set mean as default infill
    mdf_train[nmrc_column] = mdf_train[nmrc_column].fillna(mean)
    mdf_test[nmrc_column] = mdf_test[nmrc_column].fillna(mean)
    
    #a few more metrics collected for driftreport
    maximum = mdf_train[nmrc_column].max()
    minimum = mdf_train[nmrc_column].min()
    
    #create list of columns
    nmbrcolumns = [nmrc_column]
    
    #populate data structures
    nmbrnormalization_dict = {nmrc_column : {'overlap_dict' : overlap_dict, \
                                            'mean' : mean, \
                                            'maximum' : maximum, \
                                            'minimum' : minimum, \
                                            'unique_list' : unique_list, \
                                            'maxlength' : maxlength, \
                                            'convention' : convention, \
                                            'suffix' : suffix, \
                                            'defaultinfill_dict' : defaultinfill_dict,
                                            'test_same_as_train' : test_same_as_train}}
    
    column_dict_list = []
    
    for nc in nmbrcolumns:

      if nc != nmbrcolumns[0]:
        nmbrnormalization_dict = {}

      column_dict = { nc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : nmbrnormalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : nmbrcolumns, \
                           'categorylist' : nmbrcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
      
    return mdf_train, mdf_test, column_dict_list

  def _custom_train_ordl(self, df, column, normalization_dict):
    """
    #a rewrite of ordl transform in the custom train convention
    #this should benefit latency
    #return comparable form of output
    #and accept comaprable parameters
    #primary difference is cleaner code and trimmed a little fat
    #and returned missing data representation now defaults to 0

    #please note that due to its use in Binary dimensionality reduction,
    #the ordl normalization_dict has a few reserved strings
    """

    #ordered_overide is boolean to indicate if order of integer encoding basis will 
    #defer to cases when a column is a pandas categorical ordered set
    if 'ordered_overide' in normalization_dict:
      ordered_overide = normalization_dict['ordered_overide']
    else:
      ordered_overide = True
      normalization_dict.update({'ordered_overide' : ordered_overide})
      
    #frequency_sort changes the sorting of encodings from alphabetical to frequency of entries in train set
    #note that frequency_sort=True is default for ord3 and frequency_sort=False is default for ordl
    if 'frequency_sort' in normalization_dict:
      frequency_sort = normalization_dict['frequency_sort']
    else:
      frequency_sort = True
      normalization_dict.update({'frequency_sort' : frequency_sort})

    #all_activations is to base the full set of activations on user specification instead of training set
    if 'all_activations' in normalization_dict:
      #accepts False or a list of activation targets
      all_activations = normalization_dict['all_activations']
    else:
      all_activations = False
      normalization_dict.update({'all_activations' : all_activations})

    #add_activations is to include additional columns for entry activations even when not found in train set
    if 'add_activations' in normalization_dict:
      #accepts False or a list of added activation targets
      add_activations = normalization_dict['add_activations']
    else:
      add_activations = False
      normalization_dict.update({'add_activations' : add_activations})

    #less_activations is to remove entry activaiton columns even when entry found in train set
    if 'less_activations' in normalization_dict:
      #accepts False or a list of removed activation targets
      less_activations = normalization_dict['less_activations']
    else:
      less_activations = False
      normalization_dict.update({'less_activations' : less_activations})

    #consolidated_activations is to consolidate entries to a single common activation
    if 'consolidated_activations' in normalization_dict:
      #accepts False or a list of consolidated activation targets or a list of lists
      consolidated_activations = normalization_dict['consolidated_activations']
    else:
      consolidated_activations = False
      normalization_dict.update({'consolidated_activations' : consolidated_activations})
      
    #str_convert provides consistent encodings between numbers and string equivalent, eg 2 == '2'
    #str_convert defaults to True to avoid edge case with bytes type entries
    #accepts one of {True, False, 'conditional_on_bytes'}
    #where conditional_on_bytes sets as False unless bytes entries are present in train data then sets to True
    if 'str_convert' in normalization_dict:
      str_convert = normalization_dict['str_convert']
    else:
      str_convert = True
      normalization_dict.update({'str_convert' : str_convert})

    #null_activation is to have a distinct activation for missing data
    #which defaults to the 0 integer
    #note that when deactivated, missing data is grouped into whichever else entry is for the zero bucket
    #also accepts as 'Binary', which is used in Binary dimensionality reduction to return from inversion all zeros
    #for cases where test activation sets don't match any found in train activation sets
    #note that in Binary scenario we don't accept NaN entries in input
    if 'null_activation' in normalization_dict:
      null_activation = normalization_dict['null_activation']
    else:
      null_activation = True
      normalization_dict.update({'null_activation' : null_activation})
      
    #_____
    
    #for every derivation related to the set labels_train, we'll remove missing_marker and add once prior to assembling binaryencoding_dict
    #which helps accomodate a few peculiarities related to python sets with NaN inclusion
    missing_marker = np.nan
    if null_activation == 'Binary':
      missing_marker = '0' * len(str(df[column].iat[0]))
    normalization_dict.update({'missing_marker' : missing_marker})

    #conditional_on_bytes str_convert scenario resets to True if train set bytes entries present, otherwise resets to False
    if str_convert == 'conditional_on_bytes' \
    and (df[column] == df[column].astype(bytes)).sum() > 0:
      str_convert = True
      normalization_dict.update({'str_convert' : str_convert})
    elif str_convert == 'conditional_on_bytes':
      str_convert = False
      normalization_dict.update({'str_convert' : str_convert})
    
    #labels_train will be adjusted through derivation and serves as basis for binarization encoding
    labels_train = set()
    
    ordered = False

    #ordered_overide is not compatible with activation parameters
    if ordered_overide:
      if df[column].dtype.name == 'category' and df[column].cat.ordered:
        ordered = True
        labels_train_ordered = list(df[column].cat.categories)
        #by convention NaN is reserved for use with missing data
        labels_train_ordered = [x for x in labels_train_ordered if x==x]
        if str_convert is True:
          labels_train_ordered = [str(x) for x in labels_train_ordered]
        
    #frequency_sort derives a sorting order based on frequency of entries found in set
    if ordered is False and frequency_sort is True:
      ordered = True
      
      tempstring = 'asdf'
      if tempstring in df:
        while tempstring in df:
          tempstring += 'z'
      
      if str_convert is True:
        labels_train_ordered = pd.DataFrame(df[column].astype(str).value_counts())
      else:
        labels_train_ordered = pd.DataFrame(df[column].value_counts())
      #note this sorting has an edge case for bytes type entries when str_convert passed as False
      labels_train_ordered = labels_train_ordered.rename_axis(tempstring).sort_values(by = [column, tempstring], ascending = [False, True])
      labels_train_ordered = list(labels_train_ordered.index)
      #by convention NaN is reserved for use with missing data
      labels_train_ordered = [x for x in labels_train_ordered if x==x]
      if str_convert is True:
        labels_train_ordered = [str(x) for x in labels_train_ordered]
    
    #_____
            
    if df[column].dtype.name == 'category':
      labels_train = set(df[column].cat.categories)
      labels_train = {x for x in labels_train if x==x}

    if labels_train == set():  
      labels_train = set(df[column].unique())
      labels_train = {x for x in labels_train if x==x}

    #if str_convert parameter activated replace numerical with string equivalent (for common encoding between e.g. 2=='2')
    if str_convert is True:
      #note this set excludes missing_marker
      labels_train = {str(x) for x in labels_train}
      #only convert non-NaN entries in target column
#         df.loc[df[column] == df[column], (column)] = df[column].astype(str)
      df = \
      self.__autowhere(df, column, df[column] == df[column], df[column].astype(str), specified='replacement')

    #now we have a few activation set related parameters, applied by adjusting labels_train
    #we'll have convention that in cases where activation parameters are assigned, will overide ordered_overide (for alphabetic sorting)

    if all_activations is not False or less_activations is not False:
      #labels_train_orig is a support record that won't be returned
      labels_train_orig = labels_train.copy()

    if all_activations is not False:
      all_activations = {x for x in set(all_activations) if x==x}
      labels_train = all_activations

    if add_activations is not False:
      add_activations = {x for x in set(add_activations) if x==x}
      labels_train = labels_train | add_activations

    if less_activations is not False:
      less_activations = {x for x in set(less_activations) if x==x}
      labels_train = labels_train - less_activations

    #______

    #now we'll take account for any activation consolidations from consolidated_activations parameter

    #as part of this implementation, we next want to derive
    #a version of labels_train excluding consolidations (labels_train)
    #a list of consolidations associated with each returned_consolidation mapped to the returned_consolidation (consolidation_translate_dict)
    #and an inverse_consolidation_translate_dict mapping consolidated entries to their activations
    #which we'll then apply with a replace operation

    labels_train_before_consolidation = labels_train.copy()
    consolidation_translate_dict = {}
    inverse_consolidation_translate_dict = {}

    if consolidated_activations is not False:

      #if user passes a single tier list instead of list of lists we'll embed in a list
      if isinstance(consolidated_activations, list) and len(consolidated_activations) > 0:
        if not isinstance(consolidated_activations[0], list):
          consolidated_activations = [consolidated_activations]
          normalization_dict.update({'consolidated_activations' : consolidated_activations})

      for consolidation_list in consolidated_activations:

        #here is where we add any consolidation targets that weren't present in labels_train
        if str_convert is True:
          consolidation_list = [str(x) for x in consolidation_list if x==x]
          labels_train = labels_train | set(consolidation_list)
        else:
          #by convention missing data marker not elligible for inclusion in consolidation_list due to NaN/set peculiarities
          #consolidations with NaN can be accomodated by assignnan to treat desired entries as missing data
          consolidation_list = [x for x in consolidation_list if x==x]
          labels_train = labels_train | set(consolidation_list)

        #no prepare a version of labels_train excluding consolidations (labels_train)

        #we'll take the first entry in list as the returned activation (relevant to normalization_dict)
        returned_consolidation = consolidation_list[0]

        #now remove consolidated entries from labels_train
        #and map a list of consolidations associated with each returned_consolidation to the returned_consolidation (consolidation_translate_dict)
        for consolidation_list in consolidated_activations:

          if len(consolidation_list) > 1:

            #we'll take the first entry in list as the returned activation (relevant to normalization_dict)
            returned_consolidation = consolidation_list[0]

            labels_train = labels_train - set(consolidation_list[1:])

            consolidation_translate_dict.update({returned_consolidation : consolidation_list[1:]})

      #now populate an inverse_consolidation_translate_dict mapping consolidated entries to their activations
      for key,value in consolidation_translate_dict.items():
        for consolidation_list_entry in value:
          inverse_consolidation_translate_dict.update({consolidation_list_entry : key})

      #we can then apply a replace to convert consolidated items to their targeted activations
      df[column] = df[column].astype('object').replace(inverse_consolidation_translate_dict)

    del consolidation_translate_dict
    normalization_dict.update({'inverse_consolidation_translate_dict' : inverse_consolidation_translate_dict})
    del inverse_consolidation_translate_dict

    #____

    #there are a few activation parameter scenarios where we may want to replace train set entries with missing data marker
    if all_activations is not False or less_activations is not False:
      extra_entries = labels_train_orig - labels_train
      extra_entries = list({x for x in extra_entries if x==x})
      if len(extra_entries) > 0:
        plug_dict = dict(zip(extra_entries, [missing_marker] * len(extra_entries)))
        df[column] = df[column].astype('object').replace(plug_dict)

      del labels_train_orig

    #____

    #now prepare our ordinal encoding
    
    #if there is a particular order to encodings we'll sort labels_train on basis of labels_train_ordered
    if ordered is True:
      #this converts labels_train to a sorted list
      labels_train = self.__list_sorting(labels_train_ordered, labels_train)
    
    elif ordered is False:
      #convert labels_train to list 
      #and add the missing data marker to first position which will result in all zero binarized representation
      labels_train = list(labels_train)
      labels_train = sorted(labels_train, key=str)
      
    #add our missing_marker, note adding as first position will result in 0 representation even in ordered scenario
    if null_activation is True:
      labels_train = [missing_marker] + labels_train
    elif null_activation == 'Binary':
      if missing_marker not in labels_train:
        labels_train = [missing_marker] + labels_train
      
    #get length of the list, then zip a dictionary from list and range(length)
    #the range values will be our ordinal points to replace the categories
    listlength = len(labels_train)
    ordinal_dict = dict(zip(labels_train, range(listlength)))
    
    if null_activation is False:
      ordinal_dict.update({np.nan : 0})
    
    normalization_dict.update({'ordinal_dict' : ordinal_dict})
    
    #replace the categories in train set via ordinal trasnformation
    df[column] = df[column].astype('object').replace(ordinal_dict)
    
    #driftreport metric ordl_activations_dict
    ordl_activations_dict = {}
    for key in ordinal_dict:
      sumcalc = (df[column] == ordinal_dict[key]).sum() 
      ratio = sumcalc / df[column].shape[0]
      ordl_activations_dict.update({key:ratio})
    normalization_dict.update({'ordl_activations_dict' : ordl_activations_dict})
    
    return df, normalization_dict

  def _process_maxb(self, mdf_train, mdf_test, column, category, \
                   treecategory, postprocess_dict, params = {}):
    '''
    #process_maxb
    #function to translate an ord3 ordinal encoded categoric set
    #to a reduced number of activations based on a maxbincount parameter
    #where maxbincount can be passed as integer
    #for setting the maximum number of activations
    #alternately, user can set parameter minentrycount
    #as integer of the minimum number of entries in train set to include actiations
    #finally, minentryratio can also be passed as a float between 0-1
    #to designate the minimum ratio of entries in an activation to register
    #consolidated activations will be aggregated into the top activation
    #eg if maxbincount is 4, 0-3 will be retained and reaminder aggregated into 4
    #for missing values, uses adjacent cell infill as default
    #we'll set default values as False signalling not applied
    '''

    suffixoverlap_results = {}
    
    if 'inplace' in params:
      inplace = params['inplace']
    else:
      inplace = False
    
    #initialize parameters
    if 'maxbincount' in params:
      maxbincount = params['maxbincount']
    else:
      maxbincount = False
      
    #initialize parameters
    if 'minentrycount' in params:
      minentrycount = params['minentrycount']
    else:
      minentrycount = False
      
    #initialize parameters
    if 'minentryratio' in params:
      minentryratio = params['minentryratio']
    else:
      minentryratio = False

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    suffixcolumn = column + '_' + suffix
    
    if inplace is not True:
      
      #copy source column into new column
      mdf_train, suffixoverlap_results = \
      self.__df_copy_train(mdf_train, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])

      mdf_test[suffixcolumn] = mdf_test[column].copy()
    
    else:
      
      suffixoverlap_results = \
      self.__df_check_suffixoverlap(mdf_train, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
      
      mdf_train.rename(columns = {column : suffixcolumn}, inplace = True)
      mdf_test.rename(columns = {column : suffixcolumn}, inplace = True)
    
    #convert all values to either numeric or NaN
    mdf_train[suffixcolumn] = pd.to_numeric(mdf_train[suffixcolumn], errors='coerce')
    mdf_test[suffixcolumn] = pd.to_numeric(mdf_test[suffixcolumn], errors='coerce')
    
    #non integers are subject to infill
    mdf_train = \
    self.__autowhere(mdf_train, suffixcolumn, mdf_train[suffixcolumn] == mdf_train[suffixcolumn].round(), alternative = np.nan, specified='alternative')
    mdf_test = \
    self.__autowhere(mdf_test, suffixcolumn, mdf_test[suffixcolumn] == mdf_test[suffixcolumn].round(), alternative = np.nan, specified='alternative')
    
    #apply defaultinfill based on processdict entry
    #(this will default to adjinfill)
    mdf_train, defaultinfill_dict = \
    self.__apply_defaultinfill(mdf_train, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    mdf_test, _1 = \
    self.__apply_defaultinfill(mdf_test, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)
    
    #get maximum train set activation which for ord3 will be least frequent entry
    maxactivation = int(mdf_train[suffixcolumn].max())
    
    #first we'll inspect maxbincount
    bincount_maxactivation = maxactivation
    if maxbincount is not False:
      
      #current max activation
      if maxactivation > maxbincount:
        
        bincount_maxactivation = maxbincount

        mdf_train = \
        self.__autowhere(mdf_train, suffixcolumn, mdf_train[suffixcolumn] < bincount_maxactivation, alternative = bincount_maxactivation, specified='alternative')
        mdf_test = \
        self.__autowhere(mdf_test, suffixcolumn, mdf_test[suffixcolumn] < bincount_maxactivation, alternative = bincount_maxactivation, specified='alternative')
    
    #then inspect minentrycount
    count_maxactivation = maxactivation
    if minentrycount is not False:
      
      if minentrycount >= 1:

        entry_counts = {}
        for i in range(maxactivation + 1):

          count = mdf_train[mdf_train[suffixcolumn] == i].shape[0]
          entry_counts.update({i : count})
          
        for i in entry_counts:
          
          if entry_counts[i] < minentrycount:
            
            count_maxactivation = i
            break
            
        if count_maxactivation < maxactivation:

          mdf_train = \
          self.__autowhere(mdf_train, suffixcolumn, mdf_train[suffixcolumn] < count_maxactivation, alternative = count_maxactivation, specified='alternative')
          mdf_test = \
          self.__autowhere(mdf_test, suffixcolumn, mdf_test[suffixcolumn] < count_maxactivation, alternative = count_maxactivation, specified='alternative')
      
    #else if minentrycount passed as a ratio
    ratio_maxactivation = maxactivation
    if minentryratio is not False:

      if minentryratio > 0. and minentryratio < 1.:

        train_row_count = mdf_train.shape[0]

        entry_ratios = {}
        for i in range(maxactivation + 1):

          ratio = mdf_train[mdf_train[suffixcolumn] == i].shape[0] / train_row_count
          entry_ratios.update({i : ratio})

        for i in entry_ratios:

          if entry_ratios[i] < minentryratio:

            ratio_maxactivation = i
            break

        if ratio_maxactivation < maxactivation:

          mdf_train = \
          self.__autowhere(mdf_train, suffixcolumn, mdf_train[suffixcolumn] < ratio_maxactivation, alternative = ratio_maxactivation, specified='alternative')
          mdf_test = \
          self.__autowhere(mdf_test, suffixcolumn, mdf_test[suffixcolumn] < ratio_maxactivation, alternative = ratio_maxactivation, specified='alternative')

    #create list of columns
    nmbrcolumns = [suffixcolumn]

    #grab some driftreport metrics
    new_maxactivation = maxactivation
    if bincount_maxactivation < new_maxactivation:
      new_maxactivation = bincount_maxactivation
    if count_maxactivation < new_maxactivation:
      new_maxactivation = count_maxactivation
    if ratio_maxactivation < new_maxactivation:
      new_maxactivation = ratio_maxactivation
    
    consolidation_count = 0
    if new_maxactivation < maxactivation:
      consolidation_count = mdf_train[mdf_train[suffixcolumn] == new_maxactivation].shape[0]
      
    #set integer type based on encoding depth
    if new_maxactivation <= 255:
      mdf_train[suffixcolumn] = mdf_train[suffixcolumn].astype(np.uint8)
      mdf_test[suffixcolumn] = mdf_test[suffixcolumn].astype(np.uint8)
    elif new_maxactivation <= 65535:
      mdf_train[suffixcolumn] = mdf_train[suffixcolumn].astype(np.uint16)
      mdf_test[suffixcolumn] = mdf_test[suffixcolumn].astype(np.uint16)
    else:
      mdf_train[suffixcolumn] = mdf_train[suffixcolumn].astype(np.uint32)
      mdf_test[suffixcolumn] = mdf_test[suffixcolumn].astype(np.uint32)

    normalization_dict = {suffixcolumn : {'bincount_maxactivation' : bincount_maxactivation, \
                                          'count_maxactivation' : count_maxactivation, \
                                          'ratio_maxactivation' : ratio_maxactivation, \
                                          'new_maxactivation' : new_maxactivation, \
                                          'orig_maxactivation' : maxactivation, \
                                          'consolidation_count' : consolidation_count, \
                                          'maxbincount' : maxbincount, \
                                          'minentrycount' : minentrycount, \
                                          'minentryratio' : minentryratio, \
                                          'suffix' : suffix, \
                                          'defaultinfill_dict' : defaultinfill_dict,
                                          'inplace' : inplace}}
    
    #store some values in the nmbr_dict{} for use later in ML infill methods
    column_dict_list = []

    for nc in nmbrcolumns:

      if nc != nmbrcolumns[0]:
        normalization_dict = {}
      
      column_dict = { nc : {'category' : treecategory, \
                           'origcategory' : category, \
                           'normalization_dict' : normalization_dict, \
                           'origcolumn' : column, \
                           'inputcolumn' : column, \
                           'columnslist' : nmbrcolumns, \
                           'categorylist' : nmbrcolumns, \
                           'infillmodel' : False, \
                           'infillcomplete' : False, \
                           'suffixoverlap_results' : suffixoverlap_results, \
                           'deletecolumn' : False}}

      column_dict_list.append(column_dict)
        
    return mdf_train, mdf_test, column_dict_list

  def _process_ucct(self, mdf_train, mdf_test, column, category, \
                         treecategory, postprocess_dict, params = {}):
    '''
    #process_ucct(mdf_train, mdf_test, column, category)
    #preprocess column with categories into unique class count sets
    #normalized by total row count
    #e.g. for each class in train set, 
    #counts instances and divides by total train set row count
    #(so values will fall in range 0-1)
    #test sets recive comparable encoding
    '''

    suffixoverlap_results = {}

    if 'suffix' in params:
      suffix = params['suffix']
    else:
      suffix = treecategory
      
    suffixcolumn = column + '_' + suffix
    
    #create new column for trasnformation
    mdf_train, suffixoverlap_results = \
    self.__df_copy_train(mdf_train, column, suffixcolumn, suffixoverlap_results, postprocess_dict['printstatus'])
    
    mdf_test[suffixcolumn] = mdf_test[column].copy()
    
    #convert column to object
    mdf_train[suffixcolumn] = mdf_train[suffixcolumn].astype('object')
    mdf_test[suffixcolumn] = mdf_test[suffixcolumn].astype('object')

    #apply defaultinfill based on processdict entry
    #(this will default to naninfill)
    mdf_train, defaultinfill_dict = \
    self.__apply_defaultinfill(mdf_train, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=False)
    mdf_test, _1 = \
    self.__apply_defaultinfill(mdf_test, suffixcolumn, postprocess_dict, treecategory=treecategory, defaultinfill_dict=defaultinfill_dict)

    mdf_train = \
    self.__autowhere(mdf_train, suffixcolumn, mdf_train[suffixcolumn] == mdf_train[suffixcolumn], mdf_train[suffixcolumn].astype(str), specified='replacement')
    mdf_test = \
    self.__autowhere(mdf_test, suffixcolumn, mdf_test[suffixcolumn] == mdf_test[suffixcolumn], mdf_test[suffixcolumn].astype(str), specified='replacement')
    
    labels_train = set(mdf_train[suffixcolumn].unique())
    origlen = len(labels_train)
    nanincluded = False
    labels_train = {x for x in labels_train if x==x}
    if len(labels_train) < origlen:
      nanincluded = True
  
    labels_test = set(mdf_test[suffixcolumn].unique())
    labels_test = {x for x in labels_test if x==x}
    
    #____
    
    #assemble the ordinal_dict
    #with key of class and value of normalized unique class count
    ordinal_dict = {}
    rowcount = mdf_train.shape[0]
    
    for item in labels_train:
      item_count = mdf_train[mdf_train[suffixcolumn] == item].shape[0]
      ordinal_dict.update({item: item_count / rowcount})

    ordinal_nan_value = 0
    if nanincluded:
      item_count = mdf_train[mdf_train[suffixcolumn].isna()].shape[0]
      ordinal_nan_value = item_count / rowcount
    
    #replace the cateogries in train set via ordinal trasnformation
    mdf_train[suffixcolumn] = mdf_train[suffixcolumn].astype('object').replace(ordinal_dict)
    mdf_train[suffixcolumn] = mdf_train[suffixcolumn].fillna(ordinal_nan_value)
    
    #in test set, we'll need to strike any categories that weren't present in train
    #first let'/s identify what applies
    testspecificcategories = list(set(labels_test)-set(labels_train))
    
    #so we'll just replace those items with our plug value
    testplug_dict = dict(zip(testspecificcat