"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = void 0;
const events = require("@aws-cdk/aws-events");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const validation = require("./validation");
/**
 * A Stage in a Pipeline.
 *
 * Stages are added to a Pipeline by calling {@link Pipeline#addStage},
 * which returns an instance of {@link codepipeline.IStage}.
 *
 * This class is private to the CodePipeline module.
 */
class Stage {
    /**
     * Create a new Stage.
     */
    constructor(props, pipeline) {
        this._actions = new Array();
        validation.validateName('Stage', props.stageName);
        this.stageName = props.stageName;
        this._pipeline = pipeline;
        this.scope = new cdk.Construct(pipeline, this.stageName);
        for (const action of props.actions || []) {
            this.addAction(action);
        }
    }
    /**
     * Get a duplicate of this stage's list of actions.
     */
    get actionDescriptors() {
        return this._actions.slice();
    }
    get actions() {
        return this._actions.map(actionDescriptor => actionDescriptor.action);
    }
    get pipeline() {
        return this._pipeline;
    }
    render() {
        // first, assign names to output Artifacts who don't have one
        for (const action of this._actions) {
            const outputArtifacts = action.outputs;
            const unnamedOutputs = outputArtifacts.filter(o => !o.artifactName);
            for (const outputArtifact of outputArtifacts) {
                if (!outputArtifact.artifactName) {
                    const unsanitizedArtifactName = `Artifact_${this.stageName}_${action.actionName}` + (unnamedOutputs.length === 1
                        ? ''
                        : '_' + (unnamedOutputs.indexOf(outputArtifact) + 1));
                    const artifactName = sanitizeArtifactName(unsanitizedArtifactName);
                    outputArtifact._setName(artifactName);
                }
            }
        }
        return {
            name: this.stageName,
            actions: this._actions.map(action => this.renderAction(action)),
        };
    }
    addAction(action) {
        const actionName = action.actionProperties.actionName;
        // validate the name
        validation.validateName('Action', actionName);
        // check for duplicate Actions and names
        if (this._actions.find(a => a.actionName === actionName)) {
            throw new Error(`Stage ${this.stageName} already contains an action with name '${actionName}'`);
        }
        this._actions.push(this.attachActionToPipeline(action));
    }
    onStateChange(name, target, options) {
        const rule = new events.Rule(this.scope, name, options);
        rule.addTarget(target);
        rule.addEventPattern({
            detailType: ['CodePipeline Stage Execution State Change'],
            source: ['aws.codepipeline'],
            resources: [this.pipeline.pipelineArn],
            detail: {
                stage: [this.stageName],
            },
        });
        return rule;
    }
    validate() {
        return [
            ...this.validateHasActions(),
            ...this.validateActions(),
        ];
    }
    validateHasActions() {
        if (this._actions.length === 0) {
            return [`Stage '${this.stageName}' must have at least one action`];
        }
        return [];
    }
    validateActions() {
        const ret = new Array();
        for (const action of this.actionDescriptors) {
            ret.push(...this.validateAction(action));
        }
        return ret;
    }
    validateAction(action) {
        return validation.validateArtifactBounds('input', action.inputs, action.artifactBounds.minInputs, action.artifactBounds.maxInputs, action.category, action.provider)
            .concat(validation.validateArtifactBounds('output', action.outputs, action.artifactBounds.minOutputs, action.artifactBounds.maxOutputs, action.category, action.provider));
    }
    attachActionToPipeline(action) {
        // notify the Pipeline of the new Action
        //
        // It may be that a construct already exists with the given action name (CDK Pipelines
        // may do this to maintain construct tree compatibility between versions).
        //
        // If so, we simply reuse it.
        let actionScope = constructs_1.Node.of(this.scope).tryFindChild(action.actionProperties.actionName);
        if (!actionScope) {
            let id = action.actionProperties.actionName;
            if (core_1.Token.isUnresolved(id)) {
                id = findUniqueConstructId(this.scope, action.actionProperties.provider);
            }
            actionScope = new cdk.Construct(this.scope, id);
        }
        return this._pipeline._attachActionToPipeline(this, action, actionScope);
    }
    renderAction(action) {
        const outputArtifacts = cdk.Lazy.any({ produce: () => this.renderArtifacts(action.outputs) }, { omitEmptyArray: true });
        const inputArtifacts = cdk.Lazy.any({ produce: () => this.renderArtifacts(action.inputs) }, { omitEmptyArray: true });
        return {
            name: action.actionName,
            inputArtifacts,
            outputArtifacts,
            actionTypeId: {
                category: action.category.toString(),
                version: action.version,
                owner: action.owner,
                provider: action.provider,
            },
            configuration: action.configuration,
            runOrder: action.runOrder,
            roleArn: action.role ? action.role.roleArn : undefined,
            region: action.region,
            namespace: action.namespace,
        };
    }
    renderArtifacts(artifacts) {
        return artifacts
            .filter(a => a.artifactName)
            .map(a => ({ name: a.artifactName }));
    }
}
exports.Stage = Stage;
function sanitizeArtifactName(artifactName) {
    // strip out some characters that are legal in Stage and Action names,
    // but not in Artifact names
    return artifactName.replace(/[@.]/g, '');
}
function findUniqueConstructId(scope, prefix) {
    let current = prefix;
    let ctr = 1;
    while (constructs_1.Node.of(scope).tryFindChild(current) !== undefined) {
        current = `${prefix}${++ctr}`;
    }
    return current;
}
//# sourceMappingURL=data:application/json;base64,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