"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PackageJsonManager = void 0;
const fs = require("fs");
const path = require("path");
const util_1 = require("./util");
/**
 * A package.json manager to act on the closest package.json file.
 *
 * Configuring the bundler requires to manipulate the package.json and then
 * restore it.
 */
class PackageJsonManager {
    constructor() {
        const pkgPath = util_1.findClosestPathContaining('package.json');
        if (!pkgPath) {
            throw new Error('Cannot find a `package.json` in this project.');
        }
        this.pkgPath = path.join(pkgPath, 'package.json');
        this.pkg = fs.readFileSync(this.pkgPath);
        this.pkgJson = JSON.parse(this.pkg.toString());
    }
    /**
     * Update the package.json
     */
    update(data) {
        fs.writeFileSync(this.pkgPath, JSON.stringify({
            ...this.pkgJson,
            ...data,
        }, null, 2));
    }
    /**
     * Restore the package.json to the original
     */
    restore() {
        fs.writeFileSync(this.pkgPath, this.pkg);
    }
    /**
     * Extract versions for a list of modules
     */
    getVersions(modules) {
        var _a, _b, _c;
        const dependencies = {};
        const allDependencies = {
            ...(_a = this.pkgJson.dependencies) !== null && _a !== void 0 ? _a : {},
            ...(_b = this.pkgJson.devDependencies) !== null && _b !== void 0 ? _b : {},
            ...(_c = this.pkgJson.peerDependencies) !== null && _c !== void 0 ? _c : {},
        };
        for (const mod of modules) {
            if (!allDependencies[mod]) {
                throw new Error(`Cannot extract version for ${mod} in package.json`);
            }
            dependencies[mod] = allDependencies[mod];
        }
        return dependencies;
    }
}
exports.PackageJsonManager = PackageJsonManager;
//# sourceMappingURL=data:application/json;base64,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