"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const fs = require("fs");
const path = require("path");
const package_json_manager_1 = require("./package-json-manager");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    /**
     * Parcel bundled Lambda asset code
     */
    static parcel(options) {
        var _a, _b, _c, _d, _e, _f;
        // Find project root
        const projectRoot = (_a = options.projectRoot) !== null && _a !== void 0 ? _a : util_1.findClosestPathContaining(`.git${path.sep}`);
        if (!projectRoot) {
            throw new Error('Cannot find project root. Please specify it with `projectRoot`.');
        }
        // Bundling image derived from runtime bundling image (lambci)
        const image = cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../parcel'), {
            buildArgs: {
                IMAGE: options.runtime.bundlingDockerImage.image,
                PARCEL_VERSION: (_b = options.parcelVersion) !== null && _b !== void 0 ? _b : '2.0.0-beta.1',
            },
        });
        const packageJsonManager = new package_json_manager_1.PackageJsonManager();
        // Collect external and install modules
        let includeNodeModules;
        let dependencies;
        const externalModules = (_c = options.externalModules) !== null && _c !== void 0 ? _c : ['aws-sdk'];
        if (externalModules || options.nodeModules) {
            const modules = [...externalModules, ...(_d = options.nodeModules) !== null && _d !== void 0 ? _d : []];
            includeNodeModules = {};
            for (const mod of modules) {
                includeNodeModules[mod] = false;
            }
            if (options.nodeModules) {
                dependencies = packageJsonManager.getVersions(options.nodeModules);
            }
        }
        // Configure target in package.json for Parcel
        packageJsonManager.update({
            'cdk-lambda': `${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/index.js`,
            'targets': {
                'cdk-lambda': {
                    context: 'node',
                    includeNodeModules: includeNodeModules !== null && includeNodeModules !== void 0 ? includeNodeModules : true,
                    sourceMap: (_e = options.sourceMaps) !== null && _e !== void 0 ? _e : false,
                    minify: (_f = options.minify) !== null && _f !== void 0 ? _f : false,
                    engines: {
                        node: `>= ${runtimeVersion(options.runtime)}`,
                    },
                },
            },
        });
        // Entry file path relative to container path
        const containerEntryPath = path.join(cdk.AssetStaging.BUNDLING_INPUT_DIR, path.relative(projectRoot, path.resolve(options.entry)));
        const parcelCommand = `parcel build ${containerEntryPath.replace(/\\/g, '/')} --target cdk-lambda${options.cacheDir ? ' --cache-dir /parcel-cache' : ''}`;
        let installer = Installer.NPM;
        let lockfile;
        let depsCommand = '';
        if (dependencies) {
            // Create a dummy package.json for dependencies that we need to install
            fs.writeFileSync(path.join(projectRoot, '.package.json'), JSON.stringify({ dependencies }));
            // Use npm unless we have a yarn.lock.
            if (fs.existsSync(path.join(projectRoot, LockFile.YARN))) {
                installer = Installer.YARN;
                lockfile = LockFile.YARN;
            }
            else if (fs.existsSync(path.join(projectRoot, LockFile.NPM))) {
                lockfile = LockFile.NPM;
            }
            // Move dummy package.json and lock file then install
            depsCommand = chain([
                `mv ${cdk.AssetStaging.BUNDLING_INPUT_DIR}/.package.json ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/package.json`,
                lockfile ? `cp ${cdk.AssetStaging.BUNDLING_INPUT_DIR}/${lockfile} ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${lockfile}` : '',
                `cd ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR} && ${installer} install`,
            ]);
        }
        return lambda.Code.fromAsset(projectRoot, {
            assetHashType: cdk.AssetHashType.BUNDLE,
            bundling: {
                image,
                command: ['bash', '-c', chain([parcelCommand, depsCommand])],
                environment: options.parcelEnvironment,
                volumes: options.cacheDir
                    ? [{ containerPath: '/parcel-cache', hostPath: options.cacheDir }]
                    : [],
                workingDirectory: path.dirname(containerEntryPath).replace(/\\/g, '/'),
            },
        });
    }
}
exports.Bundling = Bundling;
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer || (Installer = {}));
var LockFile;
(function (LockFile) {
    LockFile["NPM"] = "package-lock.json";
    LockFile["YARN"] = "yarn.lock";
})(LockFile || (LockFile = {}));
function runtimeVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return match[1];
}
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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