"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
const events = require("@aws-cdk/aws-events");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
function deployNewStack(stack) {
    const props = {
        eventRuleProps: {
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        }
    };
    return new lib_1.EventsRuleToSqs(stack, 'test-events-rule-sqs', props);
}
test('snapshot test EventsRuleToSqs default params', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('check the sqs queue properties', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    expect(stack).toHaveResource('AWS::SQS::Queue', {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "testeventsrulesqsEncryptionKey19AB0C02",
                "Arn"
            ]
        },
        RedrivePolicy: {
            deadLetterTargetArn: {
                "Fn::GetAtt": [
                    "testeventsrulesqsdeadLetterQueueA4A15A1C",
                    "Arn"
                ]
            },
            maxReceiveCount: 15
        }
    });
});
test('check the sqs queue properties with existing KMS key', () => {
    const stack = new cdk.Stack();
    const key = defaults.buildEncryptionKey(stack, {
        description: 'my-key'
    });
    const props = {
        eventRuleProps: {
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        },
        encryptionKey: key
    };
    new lib_1.EventsRuleToSqs(stack, 'test-events-rule-sqs', props);
    expect(stack).toHaveResource('AWS::SQS::Queue', {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "EncryptionKey1B843E66",
                "Arn"
            ]
        },
        RedrivePolicy: {
            deadLetterTargetArn: {
                "Fn::GetAtt": [
                    "testeventsrulesqsdeadLetterQueueA4A15A1C",
                    "Arn"
                ]
            },
            maxReceiveCount: 15
        }
    });
    expect(stack).toHaveResource('AWS::KMS::Key', {
        KeyPolicy: {
            Statement: [
                {
                    Action: [
                        "kms:Create*",
                        "kms:Describe*",
                        "kms:Enable*",
                        "kms:List*",
                        "kms:Put*",
                        "kms:Update*",
                        "kms:Revoke*",
                        "kms:Disable*",
                        "kms:Get*",
                        "kms:Delete*",
                        "kms:ScheduleKeyDeletion",
                        "kms:CancelKeyDeletion",
                        "kms:GenerateDataKey",
                        "kms:TagResource",
                        "kms:UntagResource"
                    ],
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition"
                                    },
                                    ":iam::",
                                    {
                                        Ref: "AWS::AccountId"
                                    },
                                    ":root"
                                ]
                            ]
                        }
                    },
                    Resource: "*"
                },
                {
                    Action: [
                        "kms:Decrypt",
                        "kms:Encrypt",
                        "kms:ReEncrypt*",
                        "kms:GenerateDataKey*"
                    ],
                    Effect: "Allow",
                    Principal: {
                        Service: "events.amazonaws.com"
                    },
                    Resource: "*"
                }
            ],
            Version: "2012-10-17"
        },
        Description: "my-key",
        EnableKeyRotation: true
    });
});
test('check if the event rule has permission/policy in place in sqs queue for it to be able to send messages to the queue.', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    expect(stack).toHaveResource('AWS::SQS::QueuePolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "sqs:DeleteMessage",
                        "sqs:ReceiveMessage",
                        "sqs:SendMessage",
                        "sqs:GetQueueAttributes",
                        "sqs:RemovePermission",
                        "sqs:AddPermission",
                        "sqs:SetQueueAttributes",
                    ],
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition",
                                    },
                                    ":iam::",
                                    {
                                        Ref: "AWS::AccountId",
                                    },
                                    ":root"
                                ],
                            ],
                        },
                    },
                    Resource: {
                        "Fn::GetAtt": [
                            "testeventsrulesqsqueueAACD0364",
                            "Arn",
                        ],
                    },
                    Sid: "QueueOwnerOnlyAccess",
                },
                {
                    Action: "SQS:*",
                    Condition: {
                        Bool: {
                            "aws:SecureTransport": "false",
                        },
                    },
                    Effect: "Deny",
                    Principal: {
                        AWS: "*"
                    },
                    Resource: {
                        "Fn::GetAtt": [
                            "testeventsrulesqsqueueAACD0364",
                            "Arn",
                        ],
                    },
                    Sid: "HttpsOnly",
                },
                {
                    Action: [
                        "sqs:SendMessage",
                        "sqs:GetQueueAttributes",
                        "sqs:GetQueueUrl"
                    ],
                    Effect: "Allow",
                    Principal: {
                        Service: "events.amazonaws.com"
                    },
                    Resource: {
                        "Fn::GetAtt": [
                            "testeventsrulesqsqueueAACD0364",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Queues: [
            {
                Ref: "testeventsrulesqsqueueAACD0364",
            }
        ]
    });
});
test('check if the dead letter queue policy is setup', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    expect(stack).toHaveResource('AWS::SQS::QueuePolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "sqs:DeleteMessage",
                        "sqs:ReceiveMessage",
                        "sqs:SendMessage",
                        "sqs:GetQueueAttributes",
                        "sqs:RemovePermission",
                        "sqs:AddPermission",
                        "sqs:SetQueueAttributes",
                    ],
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition",
                                    },
                                    ":iam::",
                                    {
                                        Ref: "AWS::AccountId"
                                    },
                                    ":root"
                                ],
                            ],
                        },
                    },
                    Resource: {
                        "Fn::GetAtt": [
                            "testeventsrulesqsdeadLetterQueueA4A15A1C",
                            "Arn",
                        ],
                    },
                    Sid: "QueueOwnerOnlyAccess",
                },
                {
                    Action: "SQS:*",
                    Condition: {
                        Bool: {
                            "aws:SecureTransport": "false",
                        },
                    },
                    Effect: "Deny",
                    Principal: {
                        AWS: "*"
                    },
                    Resource: {
                        "Fn::GetAtt": [
                            "testeventsrulesqsdeadLetterQueueA4A15A1C",
                            "Arn",
                        ],
                    },
                    Sid: "HttpsOnly",
                },
            ],
            Version: "2012-10-17",
        },
        Queues: [
            {
                Ref: "testeventsrulesqsdeadLetterQueueA4A15A1C",
            },
        ]
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewStack(stack);
    expect(construct.eventsRule !== null);
    expect(construct.sqsQueue !== null);
    expect(construct.encryptionKey !== null);
    expect(construct.deadLetterQueue !== null);
});
//# sourceMappingURL=data:application/json;base64,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