"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const validation = require("./validation");
/**
 * An output artifact of an action. Artifacts can be used as input by some actions.
 */
class Artifact {
    constructor(artifactName) {
        validation.validateArtifactName(artifactName);
        this._artifactName = artifactName;
    }
    /**
     * A static factory method used to create instances of the Artifact class.
     * Mainly meant to be used from `decdk`.
     *
     * @param name the (required) name of the Artifact
     */
    static artifact(name) {
        return new Artifact(name);
    }
    get artifactName() {
        return this._artifactName;
    }
    /**
     * Returns an ArtifactPath for a file within this artifact.
     * CfnOutput is in the form "<artifact-name>::<file-name>"
     * @param fileName The name of the file
     */
    atPath(fileName) {
        return new ArtifactPath(this, fileName);
    }
    /**
     * The artifact attribute for the name of the S3 bucket where the artifact is stored.
     */
    get bucketName() {
        return artifactAttribute(this, 'BucketName');
    }
    /**
     * The artifact attribute for The name of the .zip file that contains the artifact that is
     * generated by AWS CodePipeline, such as 1ABCyZZ.zip.
     */
    get objectKey() {
        return artifactAttribute(this, 'ObjectKey');
    }
    /**
     * The artifact attribute of the Amazon Simple Storage Service (Amazon S3) URL of the artifact,
     * such as https://s3-us-west-2.amazonaws.com/artifactstorebucket-yivczw8jma0c/test/TemplateSo/1ABCyZZ.zip.
     */
    get url() {
        return artifactAttribute(this, 'URL');
    }
    /**
     * Returns a token for a value inside a JSON file within this artifact.
     * @param jsonFile The JSON file name.
     * @param keyName The hash key.
     */
    getParam(jsonFile, keyName) {
        return artifactGetParam(this, jsonFile, keyName);
    }
    /**
     * Returns the location of the .zip file in S3 that this Artifact represents.
     * Used by Lambda's `CfnParametersCode` when being deployed in a CodePipeline.
     */
    get s3Location() {
        return {
            bucketName: this.bucketName,
            objectKey: this.objectKey,
        };
    }
    toString() {
        return this.artifactName;
    }
    /** @internal */
    _setName(name) {
        if (this._artifactName) {
            throw new Error(`Artifact already has name '${this._artifactName}', cannot override it`);
        }
        else {
            this._artifactName = name;
        }
    }
}
exports.Artifact = Artifact;
/**
 * A specific file within an output artifact.
 *
 * The most common use case for this is specifying the template file
 * for a CloudFormation action.
 */
class ArtifactPath {
    constructor(artifact, fileName) {
        this.artifact = artifact;
        this.fileName = fileName;
    }
    static artifactPath(artifactName, fileName) {
        return new ArtifactPath(Artifact.artifact(artifactName), fileName);
    }
    get location() {
        const artifactName = this.artifact.artifactName
            ? this.artifact.artifactName
            : core_1.Lazy.stringValue({ produce: () => this.artifact.artifactName });
        return `${artifactName}::${this.fileName}`;
    }
}
exports.ArtifactPath = ArtifactPath;
function artifactAttribute(artifact, attributeName) {
    const lazyArtifactName = core_1.Lazy.stringValue({ produce: () => artifact.artifactName });
    return core_1.Token.asString({ 'Fn::GetArtifactAtt': [lazyArtifactName, attributeName] });
}
function artifactGetParam(artifact, jsonFile, keyName) {
    const lazyArtifactName = core_1.Lazy.stringValue({ produce: () => artifact.artifactName });
    return core_1.Token.asString({ 'Fn::GetParam': [lazyArtifactName, jsonFile, keyName] });
}
//# sourceMappingURL=data:application/json;base64,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