"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BucketNotifications = void 0;
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const bucket_1 = require("../bucket");
const destination_1 = require("../destination");
const notifications_resource_handler_1 = require("./notifications-resource-handler");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
/**
 * A custom CloudFormation resource that updates bucket notifications for a
 * bucket. The reason we need it is because the AWS::S3::Bucket notification
 * configuration is defined on the bucket itself, which makes it impossible to
 * provision notifications at the same time as the target (since
 * PutBucketNotifications validates the targets).
 *
 * Since only a single BucketNotifications resource is allowed for each Bucket,
 * this construct is not exported in the public API of this module. Instead, it
 * is created just-in-time by `s3.Bucket.onEvent`, so a 1:1 relationship is
 * ensured.
 *
 * @see
 * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-s3-bucket-notificationconfig.html
 */
class BucketNotifications extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaNotifications = new Array();
        this.queueNotifications = new Array();
        this.topicNotifications = new Array();
        this.bucket = props.bucket;
    }
    /**
     * Adds a notification subscription for this bucket.
     * If this is the first notification, a BucketNotification resource is added to the stack.
     *
     * @param event The type of event
     * @param target The target construct
     * @param filters A set of S3 key filters
     */
    addNotification(event, target, ...filters) {
        const resource = this.createResourceOnce();
        // resolve target. this also provides an opportunity for the target to e.g. update
        // policies to allow this notification to happen.
        const targetProps = target.bind(this, this.bucket);
        const commonConfig = {
            Events: [event],
            Filter: renderFilters(filters),
        };
        // if the target specifies any dependencies, add them to the custom resource.
        // for example, the SNS topic policy must be created /before/ the notification resource.
        // otherwise, S3 won't be able to confirm the subscription.
        if (targetProps.dependencies) {
            resource.node.addDependency(...targetProps.dependencies);
        }
        // based on the target type, add the the correct configurations array
        switch (targetProps.type) {
            case destination_1.BucketNotificationDestinationType.LAMBDA:
                this.lambdaNotifications.push({ ...commonConfig, LambdaFunctionArn: targetProps.arn });
                break;
            case destination_1.BucketNotificationDestinationType.QUEUE:
                this.queueNotifications.push({ ...commonConfig, QueueArn: targetProps.arn });
                break;
            case destination_1.BucketNotificationDestinationType.TOPIC:
                this.topicNotifications.push({ ...commonConfig, TopicArn: targetProps.arn });
                break;
            default:
                throw new Error('Unsupported notification target type:' + destination_1.BucketNotificationDestinationType[targetProps.type]);
        }
    }
    renderNotificationConfiguration() {
        return {
            LambdaFunctionConfigurations: this.lambdaNotifications.length > 0 ? this.lambdaNotifications : undefined,
            QueueConfigurations: this.queueNotifications.length > 0 ? this.queueNotifications : undefined,
            TopicConfigurations: this.topicNotifications.length > 0 ? this.topicNotifications : undefined,
        };
    }
    /**
     * Defines the bucket notifications resources in the stack only once.
     * This is called lazily as we add notifications, so that if notifications are not added,
     * there is no notifications resource.
     */
    createResourceOnce() {
        if (!this.resource) {
            const handler = notifications_resource_handler_1.NotificationsResourceHandler.singleton(this);
            const managed = this.bucket instanceof bucket_1.Bucket;
            if (!managed) {
                handler.role.addToPolicy(new iam.PolicyStatement({
                    actions: ['s3:GetBucketNotification'],
                    resources: ['*'],
                }));
            }
            this.resource = new cdk.CfnResource(this, 'Resource', {
                type: 'Custom::S3BucketNotifications',
                properties: {
                    ServiceToken: handler.functionArn,
                    BucketName: this.bucket.bucketName,
                    NotificationConfiguration: cdk.Lazy.any({ produce: () => this.renderNotificationConfiguration() }),
                    Managed: managed,
                },
            });
        }
        return this.resource;
    }
}
exports.BucketNotifications = BucketNotifications;
function renderFilters(filters) {
    if (!filters || filters.length === 0) {
        return undefined;
    }
    const renderedRules = new Array();
    let hasPrefix = false;
    let hasSuffix = false;
    for (const rule of filters) {
        if (!rule.suffix && !rule.prefix) {
            throw new Error('NotificationKeyFilter must specify `prefix` and/or `suffix`');
        }
        if (rule.suffix) {
            if (hasSuffix) {
                throw new Error('Cannot specify more than one suffix rule in a filter.');
            }
            renderedRules.push({ Name: 'suffix', Value: rule.suffix });
            hasSuffix = true;
        }
        if (rule.prefix) {
            if (hasPrefix) {
                throw new Error('Cannot specify more than one prefix rule in a filter.');
            }
            renderedRules.push({ Name: 'prefix', Value: rule.prefix });
            hasPrefix = true;
        }
    }
    return {
        Key: {
            FilterRules: renderedRules,
        },
    };
}
//# sourceMappingURL=data:application/json;base64,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