"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const stream_1 = require("./stream");
/**
 * Use an Amazon Kinesis stream as an event source for AWS Lambda.
 */
class KinesisEventSource extends stream_1.StreamEventSource {
    constructor(stream, props) {
        super(props);
        this.stream = stream;
        this._eventSourceMappingId = undefined;
        if (this.props.batchSize !== undefined && (this.props.batchSize < 1 || this.props.batchSize > 10000)) {
            throw new Error(`Maximum batch size must be between 1 and 10000 inclusive (given ${this.props.batchSize})`);
        }
    }
    bind(target) {
        const eventSourceMapping = target.addEventSourceMapping(`KinesisEventSource:${this.stream.node.uniqueId}`, this.enrichMappingOptions({ eventSourceArn: this.stream.streamArn }));
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        this.stream.grantRead(target);
        // The `grantRead` API provides all the permissions recommended by the Kinesis team for reading a stream.
        // `DescribeStream` permissions are not required to read a stream as it's covered by the `DescribeStreamSummary`
        // and `SubscribeToShard` APIs.
        // The Lambda::EventSourceMapping resource validates against the `DescribeStream` permission. So we add it explicitly.
        // FIXME This permission can be removed when the event source mapping resource drops it from validation.
        this.stream.grant(target, 'kinesis:DescribeStream');
    }
    /**
     * The identifier for this EventSourceMapping
     */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('KinesisEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.KinesisEventSource = KinesisEventSource;
//# sourceMappingURL=data:application/json;base64,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