"""
.. _emoji:

Emoji: Extract, Analyze, and Get Insights
=========================================

An emoji is worth a thousand words!
Regular expressions and helper functionality to aid in extracting and finding
emoji from text.

=======================   ====================================================================
:class:`EMOJI_ENTRIES`    A dictionary of the full emoji list together with unicode code points, textual name, group, and sub-group. Based on v13.1 https://unicode.org/Public/emoji/13.1/emoji-test.txt
:class:`emoji_df`         The same dictionary as a pandas DataFrame
:func:`extract_emoji`     A function for extracting and summarizing emoji in a text list, with statistics about frequencies and usage.
:func:`emoji_search`      A function for searching across names, groups, and sub-groups to find emoji based on your keywords of choice.
:class:`EMOJI_RAW`        A regular expression to extract the full list. See here on how it was developed: https://www.kaggle.com/eliasdabbas/how-to-create-a-python-regex-to-extract-emoji
=======================   ====================================================================
"""
__all__ = ['emoji_df', 'extract_emoji', 'emoji_search']

import re
from collections import Counter, namedtuple

import pandas as pd


EmojiEntry = namedtuple('EmojiEntry', ['codepoint', 'status', 'emoji','name', 'group', 'sub_group'])


EMOJI_ENTRIES = {
    '😀': EmojiEntry(codepoint='1F600', status='fully-qualified', emoji='😀', name='grinning face', group='Smileys & Emotion', sub_group='face-smiling'),
    '😃': EmojiEntry(codepoint='1F603', status='fully-qualified', emoji='😃', name='grinning face with big eyes', group='Smileys & Emotion', sub_group='face-smiling'),
    '😄': EmojiEntry(codepoint='1F604', status='fully-qualified', emoji='😄', name='grinning face with smiling eyes', group='Smileys & Emotion', sub_group='face-smiling'),
    '😁': EmojiEntry(codepoint='1F601', status='fully-qualified', emoji='😁', name='beaming face with smiling eyes', group='Smileys & Emotion', sub_group='face-smiling'),
    '😆': EmojiEntry(codepoint='1F606', status='fully-qualified', emoji='😆', name='grinning squinting face', group='Smileys & Emotion', sub_group='face-smiling'),
    '😅': EmojiEntry(codepoint='1F605', status='fully-qualified', emoji='😅', name='grinning face with sweat', group='Smileys & Emotion', sub_group='face-smiling'),
    '🤣': EmojiEntry(codepoint='1F923', status='fully-qualified', emoji='🤣', name='rolling on the floor laughing', group='Smileys & Emotion', sub_group='face-smiling'),
    '😂': EmojiEntry(codepoint='1F602', status='fully-qualified', emoji='😂', name='face with tears of joy', group='Smileys & Emotion', sub_group='face-smiling'),
    '🙂': EmojiEntry(codepoint='1F642', status='fully-qualified', emoji='🙂', name='slightly smiling face', group='Smileys & Emotion', sub_group='face-smiling'),
    '🙃': EmojiEntry(codepoint='1F643', status='fully-qualified', emoji='🙃', name='upside-down face', group='Smileys & Emotion', sub_group='face-smiling'),
    '😉': EmojiEntry(codepoint='1F609', status='fully-qualified', emoji='😉', name='winking face', group='Smileys & Emotion', sub_group='face-smiling'),
    '😊': EmojiEntry(codepoint='1F60A', status='fully-qualified', emoji='😊', name='smiling face with smiling eyes', group='Smileys & Emotion', sub_group='face-smiling'),
    '😇': EmojiEntry(codepoint='1F607', status='fully-qualified', emoji='😇', name='smiling face with halo', group='Smileys & Emotion', sub_group='face-smiling'),
    '🥰': EmojiEntry(codepoint='1F970', status='fully-qualified', emoji='🥰', name='smiling face with hearts', group='Smileys & Emotion', sub_group='face-affection'),
    '😍': EmojiEntry(codepoint='1F60D', status='fully-qualified', emoji='😍', name='smiling face with heart-eyes', group='Smileys & Emotion', sub_group='face-affection'),
    '🤩': EmojiEntry(codepoint='1F929', status='fully-qualified', emoji='🤩', name='star-struck', group='Smileys & Emotion', sub_group='face-affection'),
    '😘': EmojiEntry(codepoint='1F618', status='fully-qualified', emoji='😘', name='face blowing a kiss', group='Smileys & Emotion', sub_group='face-affection'),
    '😗': EmojiEntry(codepoint='1F617', status='fully-qualified', emoji='😗', name='kissing face', group='Smileys & Emotion', sub_group='face-affection'),
    '☺️': EmojiEntry(codepoint='263A FE0F', status='fully-qualified', emoji='☺️', name='smiling face', group='Smileys & Emotion', sub_group='face-affection'),
    '☺': EmojiEntry(codepoint='263A', status='unqualified', emoji='☺', name='smiling face', group='Smileys & Emotion', sub_group='face-affection'),
    '😚': EmojiEntry(codepoint='1F61A', status='fully-qualified', emoji='😚', name='kissing face with closed eyes', group='Smileys & Emotion', sub_group='face-affection'),
    '😙': EmojiEntry(codepoint='1F619', status='fully-qualified', emoji='😙', name='kissing face with smiling eyes', group='Smileys & Emotion', sub_group='face-affection'),
    '🥲': EmojiEntry(codepoint='1F972', status='fully-qualified', emoji='🥲', name='smiling face with tear', group='Smileys & Emotion', sub_group='face-affection'),
    '😋': EmojiEntry(codepoint='1F60B', status='fully-qualified', emoji='😋', name='face savoring food', group='Smileys & Emotion', sub_group='face-tongue'),
    '😛': EmojiEntry(codepoint='1F61B', status='fully-qualified', emoji='😛', name='face with tongue', group='Smileys & Emotion', sub_group='face-tongue'),
    '😜': EmojiEntry(codepoint='1F61C', status='fully-qualified', emoji='😜', name='winking face with tongue', group='Smileys & Emotion', sub_group='face-tongue'),
    '🤪': EmojiEntry(codepoint='1F92A', status='fully-qualified', emoji='🤪', name='zany face', group='Smileys & Emotion', sub_group='face-tongue'),
    '😝': EmojiEntry(codepoint='1F61D', status='fully-qualified', emoji='😝', name='squinting face with tongue', group='Smileys & Emotion', sub_group='face-tongue'),
    '🤑': EmojiEntry(codepoint='1F911', status='fully-qualified', emoji='🤑', name='money-mouth face', group='Smileys & Emotion', sub_group='face-tongue'),
    '🤗': EmojiEntry(codepoint='1F917', status='fully-qualified', emoji='🤗', name='hugging face', group='Smileys & Emotion', sub_group='face-hand'),
    '🤭': EmojiEntry(codepoint='1F92D', status='fully-qualified', emoji='🤭', name='face with hand over mouth', group='Smileys & Emotion', sub_group='face-hand'),
    '🤫': EmojiEntry(codepoint='1F92B', status='fully-qualified', emoji='🤫', name='shushing face', group='Smileys & Emotion', sub_group='face-hand'),
    '🤔': EmojiEntry(codepoint='1F914', status='fully-qualified', emoji='🤔', name='thinking face', group='Smileys & Emotion', sub_group='face-hand'),
    '🤐': EmojiEntry(codepoint='1F910', status='fully-qualified', emoji='🤐', name='zipper-mouth face', group='Smileys & Emotion', sub_group='face-neutral-skeptical'),
    '🤨': EmojiEntry(codepoint='1F928', status='fully-qualified', emoji='🤨', name='face with raised eyebrow', group='Smileys & Emotion', sub_group='face-neutral-skeptical'),
    '😐': EmojiEntry(codepoint='1F610', status='fully-qualified', emoji='😐', name='neutral face', group='Smileys & Emotion', sub_group='face-neutral-skeptical'),
    '😑': EmojiEntry(codepoint='1F611', status='fully-qualified', emoji='😑', name='expressionless face', group='Smileys & Emotion', sub_group='face-neutral-skeptical'),
    '😶': EmojiEntry(codepoint='1F636', status='fully-qualified', emoji='😶', name='face without mouth', group='Smileys & Emotion', sub_group='face-neutral-skeptical'),
    '😶‍🌫️': EmojiEntry(codepoint='1F636 200D 1F32B FE0F', status='fully-qualified', emoji='😶\u200d🌫️', name='face in clouds', group='Smileys & Emotion', sub_group='face-neutral-skeptical'),
    '😶‍🌫': EmojiEntry(codepoint='1F636 200D 1F32B', status='minimally-qualified', emoji='😶\u200d🌫', name='face in clouds', group='Smileys & Emotion', sub_group='face-neutral-skeptical'),
    '😏': EmojiEntry(codepoint='1F60F', status='fully-qualified', emoji='😏', name='smirking face', group='Smileys & Emotion', sub_group='face-neutral-skeptical'),
    '😒': EmojiEntry(codepoint='1F612', status='fully-qualified', emoji='😒', name='unamused face', group='Smileys & Emotion', sub_group='face-neutral-skeptical'),
    '🙄': EmojiEntry(codepoint='1F644', status='fully-qualified', emoji='🙄', name='face with rolling eyes', group='Smileys & Emotion', sub_group='face-neutral-skeptical'),
    '😬': EmojiEntry(codepoint='1F62C', status='fully-qualified', emoji='😬', name='grimacing face', group='Smileys & Emotion', sub_group='face-neutral-skeptical'),
    '😮‍💨': EmojiEntry(codepoint='1F62E 200D 1F4A8', status='fully-qualified', emoji='😮\u200d💨', name='face exhaling', group='Smileys & Emotion', sub_group='face-neutral-skeptical'),
    '🤥': EmojiEntry(codepoint='1F925', status='fully-qualified', emoji='🤥', name='lying face', group='Smileys & Emotion', sub_group='face-neutral-skeptical'),
    '😌': EmojiEntry(codepoint='1F60C', status='fully-qualified', emoji='😌', name='relieved face', group='Smileys & Emotion', sub_group='face-sleepy'),
    '😔': EmojiEntry(codepoint='1F614', status='fully-qualified', emoji='😔', name='pensive face', group='Smileys & Emotion', sub_group='face-sleepy'),
    '😪': EmojiEntry(codepoint='1F62A', status='fully-qualified', emoji='😪', name='sleepy face', group='Smileys & Emotion', sub_group='face-sleepy'),
    '🤤': EmojiEntry(codepoint='1F924', status='fully-qualified', emoji='🤤', name='drooling face', group='Smileys & Emotion', sub_group='face-sleepy'),
    '😴': EmojiEntry(codepoint='1F634', status='fully-qualified', emoji='😴', name='sleeping face', group='Smileys & Emotion', sub_group='face-sleepy'),
    '😷': EmojiEntry(codepoint='1F637', status='fully-qualified', emoji='😷', name='face with medical mask', group='Smileys & Emotion', sub_group='face-unwell'),
    '🤒': EmojiEntry(codepoint='1F912', status='fully-qualified', emoji='🤒', name='face with thermometer', group='Smileys & Emotion', sub_group='face-unwell'),
    '🤕': EmojiEntry(codepoint='1F915', status='fully-qualified', emoji='🤕', name='face with head-bandage', group='Smileys & Emotion', sub_group='face-unwell'),
    '🤢': EmojiEntry(codepoint='1F922', status='fully-qualified', emoji='🤢', name='nauseated face', group='Smileys & Emotion', sub_group='face-unwell'),
    '🤮': EmojiEntry(codepoint='1F92E', status='fully-qualified', emoji='🤮', name='face vomiting', group='Smileys & Emotion', sub_group='face-unwell'),
    '🤧': EmojiEntry(codepoint='1F927', status='fully-qualified', emoji='🤧', name='sneezing face', group='Smileys & Emotion', sub_group='face-unwell'),
    '🥵': EmojiEntry(codepoint='1F975', status='fully-qualified', emoji='🥵', name='hot face', group='Smileys & Emotion', sub_group='face-unwell'),
    '🥶': EmojiEntry(codepoint='1F976', status='fully-qualified', emoji='🥶', name='cold face', group='Smileys & Emotion', sub_group='face-unwell'),
    '🥴': EmojiEntry(codepoint='1F974', status='fully-qualified', emoji='🥴', name='woozy face', group='Smileys & Emotion', sub_group='face-unwell'),
    '😵': EmojiEntry(codepoint='1F635', status='fully-qualified', emoji='😵', name='knocked-out face', group='Smileys & Emotion', sub_group='face-unwell'),
    '😵‍💫': EmojiEntry(codepoint='1F635 200D 1F4AB', status='fully-qualified', emoji='😵\u200d💫', name='face with spiral eyes', group='Smileys & Emotion', sub_group='face-unwell'),
    '🤯': EmojiEntry(codepoint='1F92F', status='fully-qualified', emoji='🤯', name='exploding head', group='Smileys & Emotion', sub_group='face-unwell'),
    '🤠': EmojiEntry(codepoint='1F920', status='fully-qualified', emoji='🤠', name='cowboy hat face', group='Smileys & Emotion', sub_group='face-hat'),
    '🥳': EmojiEntry(codepoint='1F973', status='fully-qualified', emoji='🥳', name='partying face', group='Smileys & Emotion', sub_group='face-hat'),
    '🥸': EmojiEntry(codepoint='1F978', status='fully-qualified', emoji='🥸', name='disguised face', group='Smileys & Emotion', sub_group='face-hat'),
    '😎': EmojiEntry(codepoint='1F60E', status='fully-qualified', emoji='😎', name='smiling face with sunglasses', group='Smileys & Emotion', sub_group='face-glasses'),
    '🤓': EmojiEntry(codepoint='1F913', status='fully-qualified', emoji='🤓', name='nerd face', group='Smileys & Emotion', sub_group='face-glasses'),
    '🧐': EmojiEntry(codepoint='1F9D0', status='fully-qualified', emoji='🧐', name='face with monocle', group='Smileys & Emotion', sub_group='face-glasses'),
    '😕': EmojiEntry(codepoint='1F615', status='fully-qualified', emoji='😕', name='confused face', group='Smileys & Emotion', sub_group='face-concerned'),
    '😟': EmojiEntry(codepoint='1F61F', status='fully-qualified', emoji='😟', name='worried face', group='Smileys & Emotion', sub_group='face-concerned'),
    '🙁': EmojiEntry(codepoint='1F641', status='fully-qualified', emoji='🙁', name='slightly frowning face', group='Smileys & Emotion', sub_group='face-concerned'),
    '☹️': EmojiEntry(codepoint='2639 FE0F', status='fully-qualified', emoji='☹️', name='frowning face', group='Smileys & Emotion', sub_group='face-concerned'),
    '☹': EmojiEntry(codepoint='2639', status='unqualified', emoji='☹', name='frowning face', group='Smileys & Emotion', sub_group='face-concerned'),
    '😮': EmojiEntry(codepoint='1F62E', status='fully-qualified', emoji='😮', name='face with open mouth', group='Smileys & Emotion', sub_group='face-concerned'),
    '😯': EmojiEntry(codepoint='1F62F', status='fully-qualified', emoji='😯', name='hushed face', group='Smileys & Emotion', sub_group='face-concerned'),
    '😲': EmojiEntry(codepoint='1F632', status='fully-qualified', emoji='😲', name='astonished face', group='Smileys & Emotion', sub_group='face-concerned'),
    '😳': EmojiEntry(codepoint='1F633', status='fully-qualified', emoji='😳', name='flushed face', group='Smileys & Emotion', sub_group='face-concerned'),
    '🥺': EmojiEntry(codepoint='1F97A', status='fully-qualified', emoji='🥺', name='pleading face', group='Smileys & Emotion', sub_group='face-concerned'),
    '😦': EmojiEntry(codepoint='1F626', status='fully-qualified', emoji='😦', name='frowning face with open mouth', group='Smileys & Emotion', sub_group='face-concerned'),
    '😧': EmojiEntry(codepoint='1F627', status='fully-qualified', emoji='😧', name='anguished face', group='Smileys & Emotion', sub_group='face-concerned'),
    '😨': EmojiEntry(codepoint='1F628', status='fully-qualified', emoji='😨', name='fearful face', group='Smileys & Emotion', sub_group='face-concerned'),
    '😰': EmojiEntry(codepoint='1F630', status='fully-qualified', emoji='😰', name='anxious face with sweat', group='Smileys & Emotion', sub_group='face-concerned'),
    '😥': EmojiEntry(codepoint='1F625', status='fully-qualified', emoji='😥', name='sad but relieved face', group='Smileys & Emotion', sub_group='face-concerned'),
    '😢': EmojiEntry(codepoint='1F622', status='fully-qualified', emoji='😢', name='crying face', group='Smileys & Emotion', sub_group='face-concerned'),
    '😭': EmojiEntry(codepoint='1F62D', status='fully-qualified', emoji='😭', name='loudly crying face', group='Smileys & Emotion', sub_group='face-concerned'),
    '😱': EmojiEntry(codepoint='1F631', status='fully-qualified', emoji='😱', name='face screaming in fear', group='Smileys & Emotion', sub_group='face-concerned'),
    '😖': EmojiEntry(codepoint='1F616', status='fully-qualified', emoji='😖', name='confounded face', group='Smileys & Emotion', sub_group='face-concerned'),
    '😣': EmojiEntry(codepoint='1F623', status='fully-qualified', emoji='😣', name='persevering face', group='Smileys & Emotion', sub_group='face-concerned'),
    '😞': EmojiEntry(codepoint='1F61E', status='fully-qualified', emoji='😞', name='disappointed face', group='Smileys & Emotion', sub_group='face-concerned'),
    '😓': EmojiEntry(codepoint='1F613', status='fully-qualified', emoji='😓', name='downcast face with sweat', group='Smileys & Emotion', sub_group='face-concerned'),
    '😩': EmojiEntry(codepoint='1F629', status='fully-qualified', emoji='😩', name='weary face', group='Smileys & Emotion', sub_group='face-concerned'),
    '😫': EmojiEntry(codepoint='1F62B', status='fully-qualified', emoji='😫', name='tired face', group='Smileys & Emotion', sub_group='face-concerned'),
    '🥱': EmojiEntry(codepoint='1F971', status='fully-qualified', emoji='🥱', name='yawning face', group='Smileys & Emotion', sub_group='face-concerned'),
    '😤': EmojiEntry(codepoint='1F624', status='fully-qualified', emoji='😤', name='face with steam from nose', group='Smileys & Emotion', sub_group='face-negative'),
    '😡': EmojiEntry(codepoint='1F621', status='fully-qualified', emoji='😡', name='pouting face', group='Smileys & Emotion', sub_group='face-negative'),
    '😠': EmojiEntry(codepoint='1F620', status='fully-qualified', emoji='😠', name='angry face', group='Smileys & Emotion', sub_group='face-negative'),
    '🤬': EmojiEntry(codepoint='1F92C', status='fully-qualified', emoji='🤬', name='face with symbols on mouth', group='Smileys & Emotion', sub_group='face-negative'),
    '😈': EmojiEntry(codepoint='1F608', status='fully-qualified', emoji='😈', name='smiling face with horns', group='Smileys & Emotion', sub_group='face-negative'),
    '👿': EmojiEntry(codepoint='1F47F', status='fully-qualified', emoji='👿', name='angry face with horns', group='Smileys & Emotion', sub_group='face-negative'),
    '💀': EmojiEntry(codepoint='1F480', status='fully-qualified', emoji='💀', name='skull', group='Smileys & Emotion', sub_group='face-negative'),
    '☠️': EmojiEntry(codepoint='2620 FE0F', status='fully-qualified', emoji='☠️', name='skull and crossbones', group='Smileys & Emotion', sub_group='face-negative'),
    '☠': EmojiEntry(codepoint='2620', status='unqualified', emoji='☠', name='skull and crossbones', group='Smileys & Emotion', sub_group='face-negative'),
    '💩': EmojiEntry(codepoint='1F4A9', status='fully-qualified', emoji='💩', name='pile of poo', group='Smileys & Emotion', sub_group='face-costume'),
    '🤡': EmojiEntry(codepoint='1F921', status='fully-qualified', emoji='🤡', name='clown face', group='Smileys & Emotion', sub_group='face-costume'),
    '👹': EmojiEntry(codepoint='1F479', status='fully-qualified', emoji='👹', name='ogre', group='Smileys & Emotion', sub_group='face-costume'),
    '👺': EmojiEntry(codepoint='1F47A', status='fully-qualified', emoji='👺', name='goblin', group='Smileys & Emotion', sub_group='face-costume'),
    '👻': EmojiEntry(codepoint='1F47B', status='fully-qualified', emoji='👻', name='ghost', group='Smileys & Emotion', sub_group='face-costume'),
    '👽': EmojiEntry(codepoint='1F47D', status='fully-qualified', emoji='👽', name='alien', group='Smileys & Emotion', sub_group='face-costume'),
    '👾': EmojiEntry(codepoint='1F47E', status='fully-qualified', emoji='👾', name='alien monster', group='Smileys & Emotion', sub_group='face-costume'),
    '🤖': EmojiEntry(codepoint='1F916', status='fully-qualified', emoji='🤖', name='robot', group='Smileys & Emotion', sub_group='face-costume'),
    '😺': EmojiEntry(codepoint='1F63A', status='fully-qualified', emoji='😺', name='grinning cat', group='Smileys & Emotion', sub_group='cat-face'),
    '😸': EmojiEntry(codepoint='1F638', status='fully-qualified', emoji='😸', name='grinning cat with smiling eyes', group='Smileys & Emotion', sub_group='cat-face'),
    '😹': EmojiEntry(codepoint='1F639', status='fully-qualified', emoji='😹', name='cat with tears of joy', group='Smileys & Emotion', sub_group='cat-face'),
    '😻': EmojiEntry(codepoint='1F63B', status='fully-qualified', emoji='😻', name='smiling cat with heart-eyes', group='Smileys & Emotion', sub_group='cat-face'),
    '😼': EmojiEntry(codepoint='1F63C', status='fully-qualified', emoji='😼', name='cat with wry smile', group='Smileys & Emotion', sub_group='cat-face'),
    '😽': EmojiEntry(codepoint='1F63D', status='fully-qualified', emoji='😽', name='kissing cat', group='Smileys & Emotion', sub_group='cat-face'),
    '🙀': EmojiEntry(codepoint='1F640', status='fully-qualified', emoji='🙀', name='weary cat', group='Smileys & Emotion', sub_group='cat-face'),
    '😿': EmojiEntry(codepoint='1F63F', status='fully-qualified', emoji='😿', name='crying cat', group='Smileys & Emotion', sub_group='cat-face'),
    '😾': EmojiEntry(codepoint='1F63E', status='fully-qualified', emoji='😾', name='pouting cat', group='Smileys & Emotion', sub_group='cat-face'),
    '🙈': EmojiEntry(codepoint='1F648', status='fully-qualified', emoji='🙈', name='see-no-evil monkey', group='Smileys & Emotion', sub_group='monkey-face'),
    '🙉': EmojiEntry(codepoint='1F649', status='fully-qualified', emoji='🙉', name='hear-no-evil monkey', group='Smileys & Emotion', sub_group='monkey-face'),
    '🙊': EmojiEntry(codepoint='1F64A', status='fully-qualified', emoji='🙊', name='speak-no-evil monkey', group='Smileys & Emotion', sub_group='monkey-face'),
    '💋': EmojiEntry(codepoint='1F48B', status='fully-qualified', emoji='💋', name='kiss mark', group='Smileys & Emotion', sub_group='emotion'),
    '💌': EmojiEntry(codepoint='1F48C', status='fully-qualified', emoji='💌', name='love letter', group='Smileys & Emotion', sub_group='emotion'),
    '💘': EmojiEntry(codepoint='1F498', status='fully-qualified', emoji='💘', name='heart with arrow', group='Smileys & Emotion', sub_group='emotion'),
    '💝': EmojiEntry(codepoint='1F49D', status='fully-qualified', emoji='💝', name='heart with ribbon', group='Smileys & Emotion', sub_group='emotion'),
    '💖': EmojiEntry(codepoint='1F496', status='fully-qualified', emoji='💖', name='sparkling heart', group='Smileys & Emotion', sub_group='emotion'),
    '💗': EmojiEntry(codepoint='1F497', status='fully-qualified', emoji='💗', name='growing heart', group='Smileys & Emotion', sub_group='emotion'),
    '💓': EmojiEntry(codepoint='1F493', status='fully-qualified', emoji='💓', name='beating heart', group='Smileys & Emotion', sub_group='emotion'),
    '💞': EmojiEntry(codepoint='1F49E', status='fully-qualified', emoji='💞', name='revolving hearts', group='Smileys & Emotion', sub_group='emotion'),
    '💕': EmojiEntry(codepoint='1F495', status='fully-qualified', emoji='💕', name='two hearts', group='Smileys & Emotion', sub_group='emotion'),
    '💟': EmojiEntry(codepoint='1F49F', status='fully-qualified', emoji='💟', name='heart decoration', group='Smileys & Emotion', sub_group='emotion'),
    '❣️': EmojiEntry(codepoint='2763 FE0F', status='fully-qualified', emoji='❣️', name='heart exclamation', group='Smileys & Emotion', sub_group='emotion'),
    '❣': EmojiEntry(codepoint='2763', status='unqualified', emoji='❣', name='heart exclamation', group='Smileys & Emotion', sub_group='emotion'),
    '💔': EmojiEntry(codepoint='1F494', status='fully-qualified', emoji='💔', name='broken heart', group='Smileys & Emotion', sub_group='emotion'),
    '❤️‍🔥': EmojiEntry(codepoint='2764 FE0F 200D 1F525', status='fully-qualified', emoji='❤️\u200d🔥', name='heart on fire', group='Smileys & Emotion', sub_group='emotion'),
    '❤‍🔥': EmojiEntry(codepoint='2764 200D 1F525', status='unqualified', emoji='❤\u200d🔥', name='heart on fire', group='Smileys & Emotion', sub_group='emotion'),
    '❤️‍🩹': EmojiEntry(codepoint='2764 FE0F 200D 1FA79', status='fully-qualified', emoji='❤️\u200d🩹', name='mending heart', group='Smileys & Emotion', sub_group='emotion'),
    '❤‍🩹': EmojiEntry(codepoint='2764 200D 1FA79', status='unqualified', emoji='❤\u200d🩹', name='mending heart', group='Smileys & Emotion', sub_group='emotion'),
    '❤️': EmojiEntry(codepoint='2764 FE0F', status='fully-qualified', emoji='❤️', name='red heart', group='Smileys & Emotion', sub_group='emotion'),
    '❤': EmojiEntry(codepoint='2764', status='unqualified', emoji='❤', name='red heart', group='Smileys & Emotion', sub_group='emotion'),
    '🧡': EmojiEntry(codepoint='1F9E1', status='fully-qualified', emoji='🧡', name='orange heart', group='Smileys & Emotion', sub_group='emotion'),
    '💛': EmojiEntry(codepoint='1F49B', status='fully-qualified', emoji='💛', name='yellow heart', group='Smileys & Emotion', sub_group='emotion'),
    '💚': EmojiEntry(codepoint='1F49A', status='fully-qualified', emoji='💚', name='green heart', group='Smileys & Emotion', sub_group='emotion'),
    '💙': EmojiEntry(codepoint='1F499', status='fully-qualified', emoji='💙', name='blue heart', group='Smileys & Emotion', sub_group='emotion'),
    '💜': EmojiEntry(codepoint='1F49C', status='fully-qualified', emoji='💜', name='purple heart', group='Smileys & Emotion', sub_group='emotion'),
    '🤎': EmojiEntry(codepoint='1F90E', status='fully-qualified', emoji='🤎', name='brown heart', group='Smileys & Emotion', sub_group='emotion'),
    '🖤': EmojiEntry(codepoint='1F5A4', status='fully-qualified', emoji='🖤', name='black heart', group='Smileys & Emotion', sub_group='emotion'),
    '🤍': EmojiEntry(codepoint='1F90D', status='fully-qualified', emoji='🤍', name='white heart', group='Smileys & Emotion', sub_group='emotion'),
    '💯': EmojiEntry(codepoint='1F4AF', status='fully-qualified', emoji='💯', name='hundred points', group='Smileys & Emotion', sub_group='emotion'),
    '💢': EmojiEntry(codepoint='1F4A2', status='fully-qualified', emoji='💢', name='anger symbol', group='Smileys & Emotion', sub_group='emotion'),
    '💥': EmojiEntry(codepoint='1F4A5', status='fully-qualified', emoji='💥', name='collision', group='Smileys & Emotion', sub_group='emotion'),
    '💫': EmojiEntry(codepoint='1F4AB', status='fully-qualified', emoji='💫', name='dizzy', group='Smileys & Emotion', sub_group='emotion'),
    '💦': EmojiEntry(codepoint='1F4A6', status='fully-qualified', emoji='💦', name='sweat droplets', group='Smileys & Emotion', sub_group='emotion'),
    '💨': EmojiEntry(codepoint='1F4A8', status='fully-qualified', emoji='💨', name='dashing away', group='Smileys & Emotion', sub_group='emotion'),
    '🕳️': EmojiEntry(codepoint='1F573 FE0F', status='fully-qualified', emoji='🕳️', name='hole', group='Smileys & Emotion', sub_group='emotion'),
    '🕳': EmojiEntry(codepoint='1F573', status='unqualified', emoji='🕳', name='hole', group='Smileys & Emotion', sub_group='emotion'),
    '💣': EmojiEntry(codepoint='1F4A3', status='fully-qualified', emoji='💣', name='bomb', group='Smileys & Emotion', sub_group='emotion'),
    '💬': EmojiEntry(codepoint='1F4AC', status='fully-qualified', emoji='💬', name='speech balloon', group='Smileys & Emotion', sub_group='emotion'),
    '👁️‍🗨️': EmojiEntry(codepoint='1F441 FE0F 200D 1F5E8 FE0F', status='fully-qualified', emoji='👁️\u200d🗨️', name='eye in speech bubble', group='Smileys & Emotion', sub_group='emotion'),
    '👁‍🗨️': EmojiEntry(codepoint='1F441 200D 1F5E8 FE0F', status='unqualified', emoji='👁\u200d🗨️', name='eye in speech bubble', group='Smileys & Emotion', sub_group='emotion'),
    '👁️‍🗨': EmojiEntry(codepoint='1F441 FE0F 200D 1F5E8', status='unqualified', emoji='👁️\u200d🗨', name='eye in speech bubble', group='Smileys & Emotion', sub_group='emotion'),
    '👁‍🗨': EmojiEntry(codepoint='1F441 200D 1F5E8', status='unqualified', emoji='👁\u200d🗨', name='eye in speech bubble', group='Smileys & Emotion', sub_group='emotion'),
    '🗨️': EmojiEntry(codepoint='1F5E8 FE0F', status='fully-qualified', emoji='🗨️', name='left speech bubble', group='Smileys & Emotion', sub_group='emotion'),
    '🗨': EmojiEntry(codepoint='1F5E8', status='unqualified', emoji='🗨', name='left speech bubble', group='Smileys & Emotion', sub_group='emotion'),
    '🗯️': EmojiEntry(codepoint='1F5EF FE0F', status='fully-qualified', emoji='🗯️', name='right anger bubble', group='Smileys & Emotion', sub_group='emotion'),
    '🗯': EmojiEntry(codepoint='1F5EF', status='unqualified', emoji='🗯', name='right anger bubble', group='Smileys & Emotion', sub_group='emotion'),
    '💭': EmojiEntry(codepoint='1F4AD', status='fully-qualified', emoji='💭', name='thought balloon', group='Smileys & Emotion', sub_group='emotion'),
    '💤': EmojiEntry(codepoint='1F4A4', status='fully-qualified', emoji='💤', name='zzz', group='Smileys & Emotion', sub_group='emotion'),
    '👋': EmojiEntry(codepoint='1F44B', status='fully-qualified', emoji='👋', name='waving hand', group='People & Body', sub_group='hand-fingers-open'),
    '👋🏻': EmojiEntry(codepoint='1F44B 1F3FB', status='fully-qualified', emoji='👋🏻', name='waving hand: light skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '👋🏼': EmojiEntry(codepoint='1F44B 1F3FC', status='fully-qualified', emoji='👋🏼', name='waving hand: medium-light skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '👋🏽': EmojiEntry(codepoint='1F44B 1F3FD', status='fully-qualified', emoji='👋🏽', name='waving hand: medium skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '👋🏾': EmojiEntry(codepoint='1F44B 1F3FE', status='fully-qualified', emoji='👋🏾', name='waving hand: medium-dark skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '👋🏿': EmojiEntry(codepoint='1F44B 1F3FF', status='fully-qualified', emoji='👋🏿', name='waving hand: dark skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '🤚': EmojiEntry(codepoint='1F91A', status='fully-qualified', emoji='🤚', name='raised back of hand', group='People & Body', sub_group='hand-fingers-open'),
    '🤚🏻': EmojiEntry(codepoint='1F91A 1F3FB', status='fully-qualified', emoji='🤚🏻', name='raised back of hand: light skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '🤚🏼': EmojiEntry(codepoint='1F91A 1F3FC', status='fully-qualified', emoji='🤚🏼', name='raised back of hand: medium-light skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '🤚🏽': EmojiEntry(codepoint='1F91A 1F3FD', status='fully-qualified', emoji='🤚🏽', name='raised back of hand: medium skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '🤚🏾': EmojiEntry(codepoint='1F91A 1F3FE', status='fully-qualified', emoji='🤚🏾', name='raised back of hand: medium-dark skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '🤚🏿': EmojiEntry(codepoint='1F91A 1F3FF', status='fully-qualified', emoji='🤚🏿', name='raised back of hand: dark skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '🖐️': EmojiEntry(codepoint='1F590 FE0F', status='fully-qualified', emoji='🖐️', name='hand with fingers splayed', group='People & Body', sub_group='hand-fingers-open'),
    '🖐': EmojiEntry(codepoint='1F590', status='unqualified', emoji='🖐', name='hand with fingers splayed', group='People & Body', sub_group='hand-fingers-open'),
    '🖐🏻': EmojiEntry(codepoint='1F590 1F3FB', status='fully-qualified', emoji='🖐🏻', name='hand with fingers splayed: light skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '🖐🏼': EmojiEntry(codepoint='1F590 1F3FC', status='fully-qualified', emoji='🖐🏼', name='hand with fingers splayed: medium-light skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '🖐🏽': EmojiEntry(codepoint='1F590 1F3FD', status='fully-qualified', emoji='🖐🏽', name='hand with fingers splayed: medium skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '🖐🏾': EmojiEntry(codepoint='1F590 1F3FE', status='fully-qualified', emoji='🖐🏾', name='hand with fingers splayed: medium-dark skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '🖐🏿': EmojiEntry(codepoint='1F590 1F3FF', status='fully-qualified', emoji='🖐🏿', name='hand with fingers splayed: dark skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '✋': EmojiEntry(codepoint='270B', status='fully-qualified', emoji='✋', name='raised hand', group='People & Body', sub_group='hand-fingers-open'),
    '✋🏻': EmojiEntry(codepoint='270B 1F3FB', status='fully-qualified', emoji='✋🏻', name='raised hand: light skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '✋🏼': EmojiEntry(codepoint='270B 1F3FC', status='fully-qualified', emoji='✋🏼', name='raised hand: medium-light skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '✋🏽': EmojiEntry(codepoint='270B 1F3FD', status='fully-qualified', emoji='✋🏽', name='raised hand: medium skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '✋🏾': EmojiEntry(codepoint='270B 1F3FE', status='fully-qualified', emoji='✋🏾', name='raised hand: medium-dark skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '✋🏿': EmojiEntry(codepoint='270B 1F3FF', status='fully-qualified', emoji='✋🏿', name='raised hand: dark skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '🖖': EmojiEntry(codepoint='1F596', status='fully-qualified', emoji='🖖', name='vulcan salute', group='People & Body', sub_group='hand-fingers-open'),
    '🖖🏻': EmojiEntry(codepoint='1F596 1F3FB', status='fully-qualified', emoji='🖖🏻', name='vulcan salute: light skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '🖖🏼': EmojiEntry(codepoint='1F596 1F3FC', status='fully-qualified', emoji='🖖🏼', name='vulcan salute: medium-light skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '🖖🏽': EmojiEntry(codepoint='1F596 1F3FD', status='fully-qualified', emoji='🖖🏽', name='vulcan salute: medium skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '🖖🏾': EmojiEntry(codepoint='1F596 1F3FE', status='fully-qualified', emoji='🖖🏾', name='vulcan salute: medium-dark skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '🖖🏿': EmojiEntry(codepoint='1F596 1F3FF', status='fully-qualified', emoji='🖖🏿', name='vulcan salute: dark skin tone', group='People & Body', sub_group='hand-fingers-open'),
    '👌': EmojiEntry(codepoint='1F44C', status='fully-qualified', emoji='👌', name='OK hand', group='People & Body', sub_group='hand-fingers-partial'),
    '👌🏻': EmojiEntry(codepoint='1F44C 1F3FB', status='fully-qualified', emoji='👌🏻', name='OK hand: light skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '👌🏼': EmojiEntry(codepoint='1F44C 1F3FC', status='fully-qualified', emoji='👌🏼', name='OK hand: medium-light skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '👌🏽': EmojiEntry(codepoint='1F44C 1F3FD', status='fully-qualified', emoji='👌🏽', name='OK hand: medium skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '👌🏾': EmojiEntry(codepoint='1F44C 1F3FE', status='fully-qualified', emoji='👌🏾', name='OK hand: medium-dark skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '👌🏿': EmojiEntry(codepoint='1F44C 1F3FF', status='fully-qualified', emoji='👌🏿', name='OK hand: dark skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤌': EmojiEntry(codepoint='1F90C', status='fully-qualified', emoji='🤌', name='pinched fingers', group='People & Body', sub_group='hand-fingers-partial'),
    '🤌🏻': EmojiEntry(codepoint='1F90C 1F3FB', status='fully-qualified', emoji='🤌🏻', name='pinched fingers: light skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤌🏼': EmojiEntry(codepoint='1F90C 1F3FC', status='fully-qualified', emoji='🤌🏼', name='pinched fingers: medium-light skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤌🏽': EmojiEntry(codepoint='1F90C 1F3FD', status='fully-qualified', emoji='🤌🏽', name='pinched fingers: medium skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤌🏾': EmojiEntry(codepoint='1F90C 1F3FE', status='fully-qualified', emoji='🤌🏾', name='pinched fingers: medium-dark skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤌🏿': EmojiEntry(codepoint='1F90C 1F3FF', status='fully-qualified', emoji='🤌🏿', name='pinched fingers: dark skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤏': EmojiEntry(codepoint='1F90F', status='fully-qualified', emoji='🤏', name='pinching hand', group='People & Body', sub_group='hand-fingers-partial'),
    '🤏🏻': EmojiEntry(codepoint='1F90F 1F3FB', status='fully-qualified', emoji='🤏🏻', name='pinching hand: light skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤏🏼': EmojiEntry(codepoint='1F90F 1F3FC', status='fully-qualified', emoji='🤏🏼', name='pinching hand: medium-light skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤏🏽': EmojiEntry(codepoint='1F90F 1F3FD', status='fully-qualified', emoji='🤏🏽', name='pinching hand: medium skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤏🏾': EmojiEntry(codepoint='1F90F 1F3FE', status='fully-qualified', emoji='🤏🏾', name='pinching hand: medium-dark skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤏🏿': EmojiEntry(codepoint='1F90F 1F3FF', status='fully-qualified', emoji='🤏🏿', name='pinching hand: dark skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '✌️': EmojiEntry(codepoint='270C FE0F', status='fully-qualified', emoji='✌️', name='victory hand', group='People & Body', sub_group='hand-fingers-partial'),
    '✌': EmojiEntry(codepoint='270C', status='unqualified', emoji='✌', name='victory hand', group='People & Body', sub_group='hand-fingers-partial'),
    '✌🏻': EmojiEntry(codepoint='270C 1F3FB', status='fully-qualified', emoji='✌🏻', name='victory hand: light skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '✌🏼': EmojiEntry(codepoint='270C 1F3FC', status='fully-qualified', emoji='✌🏼', name='victory hand: medium-light skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '✌🏽': EmojiEntry(codepoint='270C 1F3FD', status='fully-qualified', emoji='✌🏽', name='victory hand: medium skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '✌🏾': EmojiEntry(codepoint='270C 1F3FE', status='fully-qualified', emoji='✌🏾', name='victory hand: medium-dark skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '✌🏿': EmojiEntry(codepoint='270C 1F3FF', status='fully-qualified', emoji='✌🏿', name='victory hand: dark skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤞': EmojiEntry(codepoint='1F91E', status='fully-qualified', emoji='🤞', name='crossed fingers', group='People & Body', sub_group='hand-fingers-partial'),
    '🤞🏻': EmojiEntry(codepoint='1F91E 1F3FB', status='fully-qualified', emoji='🤞🏻', name='crossed fingers: light skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤞🏼': EmojiEntry(codepoint='1F91E 1F3FC', status='fully-qualified', emoji='🤞🏼', name='crossed fingers: medium-light skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤞🏽': EmojiEntry(codepoint='1F91E 1F3FD', status='fully-qualified', emoji='🤞🏽', name='crossed fingers: medium skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤞🏾': EmojiEntry(codepoint='1F91E 1F3FE', status='fully-qualified', emoji='🤞🏾', name='crossed fingers: medium-dark skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤞🏿': EmojiEntry(codepoint='1F91E 1F3FF', status='fully-qualified', emoji='🤞🏿', name='crossed fingers: dark skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤟': EmojiEntry(codepoint='1F91F', status='fully-qualified', emoji='🤟', name='love-you gesture', group='People & Body', sub_group='hand-fingers-partial'),
    '🤟🏻': EmojiEntry(codepoint='1F91F 1F3FB', status='fully-qualified', emoji='🤟🏻', name='love-you gesture: light skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤟🏼': EmojiEntry(codepoint='1F91F 1F3FC', status='fully-qualified', emoji='🤟🏼', name='love-you gesture: medium-light skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤟🏽': EmojiEntry(codepoint='1F91F 1F3FD', status='fully-qualified', emoji='🤟🏽', name='love-you gesture: medium skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤟🏾': EmojiEntry(codepoint='1F91F 1F3FE', status='fully-qualified', emoji='🤟🏾', name='love-you gesture: medium-dark skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤟🏿': EmojiEntry(codepoint='1F91F 1F3FF', status='fully-qualified', emoji='🤟🏿', name='love-you gesture: dark skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤘': EmojiEntry(codepoint='1F918', status='fully-qualified', emoji='🤘', name='sign of the horns', group='People & Body', sub_group='hand-fingers-partial'),
    '🤘🏻': EmojiEntry(codepoint='1F918 1F3FB', status='fully-qualified', emoji='🤘🏻', name='sign of the horns: light skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤘🏼': EmojiEntry(codepoint='1F918 1F3FC', status='fully-qualified', emoji='🤘🏼', name='sign of the horns: medium-light skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤘🏽': EmojiEntry(codepoint='1F918 1F3FD', status='fully-qualified', emoji='🤘🏽', name='sign of the horns: medium skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤘🏾': EmojiEntry(codepoint='1F918 1F3FE', status='fully-qualified', emoji='🤘🏾', name='sign of the horns: medium-dark skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤘🏿': EmojiEntry(codepoint='1F918 1F3FF', status='fully-qualified', emoji='🤘🏿', name='sign of the horns: dark skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤙': EmojiEntry(codepoint='1F919', status='fully-qualified', emoji='🤙', name='call me hand', group='People & Body', sub_group='hand-fingers-partial'),
    '🤙🏻': EmojiEntry(codepoint='1F919 1F3FB', status='fully-qualified', emoji='🤙🏻', name='call me hand: light skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤙🏼': EmojiEntry(codepoint='1F919 1F3FC', status='fully-qualified', emoji='🤙🏼', name='call me hand: medium-light skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤙🏽': EmojiEntry(codepoint='1F919 1F3FD', status='fully-qualified', emoji='🤙🏽', name='call me hand: medium skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤙🏾': EmojiEntry(codepoint='1F919 1F3FE', status='fully-qualified', emoji='🤙🏾', name='call me hand: medium-dark skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '🤙🏿': EmojiEntry(codepoint='1F919 1F3FF', status='fully-qualified', emoji='🤙🏿', name='call me hand: dark skin tone', group='People & Body', sub_group='hand-fingers-partial'),
    '👈': EmojiEntry(codepoint='1F448', status='fully-qualified', emoji='👈', name='backhand index pointing left', group='People & Body', sub_group='hand-single-finger'),
    '👈🏻': EmojiEntry(codepoint='1F448 1F3FB', status='fully-qualified', emoji='👈🏻', name='backhand index pointing left: light skin tone', group='People & Body', sub_group='hand-single-finger'),
    '👈🏼': EmojiEntry(codepoint='1F448 1F3FC', status='fully-qualified', emoji='👈🏼', name='backhand index pointing left: medium-light skin tone', group='People & Body', sub_group='hand-single-finger'),
    '👈🏽': EmojiEntry(codepoint='1F448 1F3FD', status='fully-qualified', emoji='👈🏽', name='backhand index pointing left: medium skin tone', group='People & Body', sub_group='hand-single-finger'),
    '👈🏾': EmojiEntry(codepoint='1F448 1F3FE', status='fully-qualified', emoji='👈🏾', name='backhand index pointing left: medium-dark skin tone', group='People & Body', sub_group='hand-single-finger'),
    '👈🏿': EmojiEntry(codepoint='1F448 1F3FF', status='fully-qualified', emoji='👈🏿', name='backhand index pointing left: dark skin tone', group='People & Body', sub_group='hand-single-finger'),
    '👉': EmojiEntry(codepoint='1F449', status='fully-qualified', emoji='👉', name='backhand index pointing right', group='People & Body', sub_group='hand-single-finger'),
    '👉🏻': EmojiEntry(codepoint='1F449 1F3FB', status='fully-qualified', emoji='👉🏻', name='backhand index pointing right: light skin tone', group='People & Body', sub_group='hand-single-finger'),
    '👉🏼': EmojiEntry(codepoint='1F449 1F3FC', status='fully-qualified', emoji='👉🏼', name='backhand index pointing right: medium-light skin tone', group='People & Body', sub_group='hand-single-finger'),
    '👉🏽': EmojiEntry(codepoint='1F449 1F3FD', status='fully-qualified', emoji='👉🏽', name='backhand index pointing right: medium skin tone', group='People & Body', sub_group='hand-single-finger'),
    '👉🏾': EmojiEntry(codepoint='1F449 1F3FE', status='fully-qualified', emoji='👉🏾', name='backhand index pointing right: medium-dark skin tone', group='People & Body', sub_group='hand-single-finger'),
    '👉🏿': EmojiEntry(codepoint='1F449 1F3FF', status='fully-qualified', emoji='👉🏿', name='backhand index pointing right: dark skin tone', group='People & Body', sub_group='hand-single-finger'),
    '👆': EmojiEntry(codepoint='1F446', status='fully-qualified', emoji='👆', name='backhand index pointing up', group='People & Body', sub_group='hand-single-finger'),
    '👆🏻': EmojiEntry(codepoint='1F446 1F3FB', status='fully-qualified', emoji='👆🏻', name='backhand index pointing up: light skin tone', group='People & Body', sub_group='hand-single-finger'),
    '👆🏼': EmojiEntry(codepoint='1F446 1F3FC', status='fully-qualified', emoji='👆🏼', name='backhand index pointing up: medium-light skin tone', group='People & Body', sub_group='hand-single-finger'),
    '👆🏽': EmojiEntry(codepoint='1F446 1F3FD', status='fully-qualified', emoji='👆🏽', name='backhand index pointing up: medium skin tone', group='People & Body', sub_group='hand-single-finger'),
    '👆🏾': EmojiEntry(codepoint='1F446 1F3FE', status='fully-qualified', emoji='👆🏾', name='backhand index pointing up: medium-dark skin tone', group='People & Body', sub_group='hand-single-finger'),
    '👆🏿': EmojiEntry(codepoint='1F446 1F3FF', status='fully-qualified', emoji='👆🏿', name='backhand index pointing up: dark skin tone', group='People & Body', sub_group='hand-single-finger'),
    '🖕': EmojiEntry(codepoint='1F595', status='fully-qualified', emoji='🖕', name='middle finger', group='People & Body', sub_group='hand-single-finger'),
    '🖕🏻': EmojiEntry(codepoint='1F595 1F3FB', status='fully-qualified', emoji='🖕🏻', name='middle finger: light skin tone', group='People & Body', sub_group='hand-single-finger'),
    '🖕🏼': EmojiEntry(codepoint='1F595 1F3FC', status='fully-qualified', emoji='🖕🏼', name='middle finger: medium-light skin tone', group='People & Body', sub_group='hand-single-finger'),
    '🖕🏽': EmojiEntry(codepoint='1F595 1F3FD', status='fully-qualified', emoji='🖕🏽', name='middle finger: medium skin tone', group='People & Body', sub_group='hand-single-finger'),
    '🖕🏾': EmojiEntry(codepoint='1F595 1F3FE', status='fully-qualified', emoji='🖕🏾', name='middle finger: medium-dark skin tone', group='People & Body', sub_group='hand-single-finger'),
    '🖕🏿': EmojiEntry(codepoint='1F595 1F3FF', status='fully-qualified', emoji='🖕🏿', name='middle finger: dark skin tone', group='People & Body', sub_group='hand-single-finger'),
    '👇': EmojiEntry(codepoint='1F447', status='fully-qualified', emoji='👇', name='backhand index pointing down', group='People & Body', sub_group='hand-single-finger'),
    '👇🏻': EmojiEntry(codepoint='1F447 1F3FB', status='fully-qualified', emoji='👇🏻', name='backhand index pointing down: light skin tone', group='People & Body', sub_group='hand-single-finger'),
    '👇🏼': EmojiEntry(codepoint='1F447 1F3FC', status='fully-qualified', emoji='👇🏼', name='backhand index pointing down: medium-light skin tone', group='People & Body', sub_group='hand-single-finger'),
    '👇🏽': EmojiEntry(codepoint='1F447 1F3FD', status='fully-qualified', emoji='👇🏽', name='backhand index pointing down: medium skin tone', group='People & Body', sub_group='hand-single-finger'),
    '👇🏾': EmojiEntry(codepoint='1F447 1F3FE', status='fully-qualified', emoji='👇🏾', name='backhand index pointing down: medium-dark skin tone', group='People & Body', sub_group='hand-single-finger'),
    '👇🏿': EmojiEntry(codepoint='1F447 1F3FF', status='fully-qualified', emoji='👇🏿', name='backhand index pointing down: dark skin tone', group='People & Body', sub_group='hand-single-finger'),
    '☝️': EmojiEntry(codepoint='261D FE0F', status='fully-qualified', emoji='☝️', name='index pointing up', group='People & Body', sub_group='hand-single-finger'),
    '☝': EmojiEntry(codepoint='261D', status='unqualified', emoji='☝', name='index pointing up', group='People & Body', sub_group='hand-single-finger'),
    '☝🏻': EmojiEntry(codepoint='261D 1F3FB', status='fully-qualified', emoji='☝🏻', name='index pointing up: light skin tone', group='People & Body', sub_group='hand-single-finger'),
    '☝🏼': EmojiEntry(codepoint='261D 1F3FC', status='fully-qualified', emoji='☝🏼', name='index pointing up: medium-light skin tone', group='People & Body', sub_group='hand-single-finger'),
    '☝🏽': EmojiEntry(codepoint='261D 1F3FD', status='fully-qualified', emoji='☝🏽', name='index pointing up: medium skin tone', group='People & Body', sub_group='hand-single-finger'),
    '☝🏾': EmojiEntry(codepoint='261D 1F3FE', status='fully-qualified', emoji='☝🏾', name='index pointing up: medium-dark skin tone', group='People & Body', sub_group='hand-single-finger'),
    '☝🏿': EmojiEntry(codepoint='261D 1F3FF', status='fully-qualified', emoji='☝🏿', name='index pointing up: dark skin tone', group='People & Body', sub_group='hand-single-finger'),
    '👍': EmojiEntry(codepoint='1F44D', status='fully-qualified', emoji='👍', name='thumbs up', group='People & Body', sub_group='hand-fingers-closed'),
    '👍🏻': EmojiEntry(codepoint='1F44D 1F3FB', status='fully-qualified', emoji='👍🏻', name='thumbs up: light skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '👍🏼': EmojiEntry(codepoint='1F44D 1F3FC', status='fully-qualified', emoji='👍🏼', name='thumbs up: medium-light skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '👍🏽': EmojiEntry(codepoint='1F44D 1F3FD', status='fully-qualified', emoji='👍🏽', name='thumbs up: medium skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '👍🏾': EmojiEntry(codepoint='1F44D 1F3FE', status='fully-qualified', emoji='👍🏾', name='thumbs up: medium-dark skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '👍🏿': EmojiEntry(codepoint='1F44D 1F3FF', status='fully-qualified', emoji='👍🏿', name='thumbs up: dark skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '👎': EmojiEntry(codepoint='1F44E', status='fully-qualified', emoji='👎', name='thumbs down', group='People & Body', sub_group='hand-fingers-closed'),
    '👎🏻': EmojiEntry(codepoint='1F44E 1F3FB', status='fully-qualified', emoji='👎🏻', name='thumbs down: light skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '👎🏼': EmojiEntry(codepoint='1F44E 1F3FC', status='fully-qualified', emoji='👎🏼', name='thumbs down: medium-light skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '👎🏽': EmojiEntry(codepoint='1F44E 1F3FD', status='fully-qualified', emoji='👎🏽', name='thumbs down: medium skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '👎🏾': EmojiEntry(codepoint='1F44E 1F3FE', status='fully-qualified', emoji='👎🏾', name='thumbs down: medium-dark skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '👎🏿': EmojiEntry(codepoint='1F44E 1F3FF', status='fully-qualified', emoji='👎🏿', name='thumbs down: dark skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '✊': EmojiEntry(codepoint='270A', status='fully-qualified', emoji='✊', name='raised fist', group='People & Body', sub_group='hand-fingers-closed'),
    '✊🏻': EmojiEntry(codepoint='270A 1F3FB', status='fully-qualified', emoji='✊🏻', name='raised fist: light skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '✊🏼': EmojiEntry(codepoint='270A 1F3FC', status='fully-qualified', emoji='✊🏼', name='raised fist: medium-light skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '✊🏽': EmojiEntry(codepoint='270A 1F3FD', status='fully-qualified', emoji='✊🏽', name='raised fist: medium skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '✊🏾': EmojiEntry(codepoint='270A 1F3FE', status='fully-qualified', emoji='✊🏾', name='raised fist: medium-dark skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '✊🏿': EmojiEntry(codepoint='270A 1F3FF', status='fully-qualified', emoji='✊🏿', name='raised fist: dark skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '👊': EmojiEntry(codepoint='1F44A', status='fully-qualified', emoji='👊', name='oncoming fist', group='People & Body', sub_group='hand-fingers-closed'),
    '👊🏻': EmojiEntry(codepoint='1F44A 1F3FB', status='fully-qualified', emoji='👊🏻', name='oncoming fist: light skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '👊🏼': EmojiEntry(codepoint='1F44A 1F3FC', status='fully-qualified', emoji='👊🏼', name='oncoming fist: medium-light skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '👊🏽': EmojiEntry(codepoint='1F44A 1F3FD', status='fully-qualified', emoji='👊🏽', name='oncoming fist: medium skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '👊🏾': EmojiEntry(codepoint='1F44A 1F3FE', status='fully-qualified', emoji='👊🏾', name='oncoming fist: medium-dark skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '👊🏿': EmojiEntry(codepoint='1F44A 1F3FF', status='fully-qualified', emoji='👊🏿', name='oncoming fist: dark skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '🤛': EmojiEntry(codepoint='1F91B', status='fully-qualified', emoji='🤛', name='left-facing fist', group='People & Body', sub_group='hand-fingers-closed'),
    '🤛🏻': EmojiEntry(codepoint='1F91B 1F3FB', status='fully-qualified', emoji='🤛🏻', name='left-facing fist: light skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '🤛🏼': EmojiEntry(codepoint='1F91B 1F3FC', status='fully-qualified', emoji='🤛🏼', name='left-facing fist: medium-light skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '🤛🏽': EmojiEntry(codepoint='1F91B 1F3FD', status='fully-qualified', emoji='🤛🏽', name='left-facing fist: medium skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '🤛🏾': EmojiEntry(codepoint='1F91B 1F3FE', status='fully-qualified', emoji='🤛🏾', name='left-facing fist: medium-dark skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '🤛🏿': EmojiEntry(codepoint='1F91B 1F3FF', status='fully-qualified', emoji='🤛🏿', name='left-facing fist: dark skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '🤜': EmojiEntry(codepoint='1F91C', status='fully-qualified', emoji='🤜', name='right-facing fist', group='People & Body', sub_group='hand-fingers-closed'),
    '🤜🏻': EmojiEntry(codepoint='1F91C 1F3FB', status='fully-qualified', emoji='🤜🏻', name='right-facing fist: light skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '🤜🏼': EmojiEntry(codepoint='1F91C 1F3FC', status='fully-qualified', emoji='🤜🏼', name='right-facing fist: medium-light skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '🤜🏽': EmojiEntry(codepoint='1F91C 1F3FD', status='fully-qualified', emoji='🤜🏽', name='right-facing fist: medium skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '🤜🏾': EmojiEntry(codepoint='1F91C 1F3FE', status='fully-qualified', emoji='🤜🏾', name='right-facing fist: medium-dark skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '🤜🏿': EmojiEntry(codepoint='1F91C 1F3FF', status='fully-qualified', emoji='🤜🏿', name='right-facing fist: dark skin tone', group='People & Body', sub_group='hand-fingers-closed'),
    '👏': EmojiEntry(codepoint='1F44F', status='fully-qualified', emoji='👏', name='clapping hands', group='People & Body', sub_group='hands'),
    '👏🏻': EmojiEntry(codepoint='1F44F 1F3FB', status='fully-qualified', emoji='👏🏻', name='clapping hands: light skin tone', group='People & Body', sub_group='hands'),
    '👏🏼': EmojiEntry(codepoint='1F44F 1F3FC', status='fully-qualified', emoji='👏🏼', name='clapping hands: medium-light skin tone', group='People & Body', sub_group='hands'),
    '👏🏽': EmojiEntry(codepoint='1F44F 1F3FD', status='fully-qualified', emoji='👏🏽', name='clapping hands: medium skin tone', group='People & Body', sub_group='hands'),
    '👏🏾': EmojiEntry(codepoint='1F44F 1F3FE', status='fully-qualified', emoji='👏🏾', name='clapping hands: medium-dark skin tone', group='People & Body', sub_group='hands'),
    '👏🏿': EmojiEntry(codepoint='1F44F 1F3FF', status='fully-qualified', emoji='👏🏿', name='clapping hands: dark skin tone', group='People & Body', sub_group='hands'),
    '🙌': EmojiEntry(codepoint='1F64C', status='fully-qualified', emoji='🙌', name='raising hands', group='People & Body', sub_group='hands'),
    '🙌🏻': EmojiEntry(codepoint='1F64C 1F3FB', status='fully-qualified', emoji='🙌🏻', name='raising hands: light skin tone', group='People & Body', sub_group='hands'),
    '🙌🏼': EmojiEntry(codepoint='1F64C 1F3FC', status='fully-qualified', emoji='🙌🏼', name='raising hands: medium-light skin tone', group='People & Body', sub_group='hands'),
    '🙌🏽': EmojiEntry(codepoint='1F64C 1F3FD', status='fully-qualified', emoji='🙌🏽', name='raising hands: medium skin tone', group='People & Body', sub_group='hands'),
    '🙌🏾': EmojiEntry(codepoint='1F64C 1F3FE', status='fully-qualified', emoji='🙌🏾', name='raising hands: medium-dark skin tone', group='People & Body', sub_group='hands'),
    '🙌🏿': EmojiEntry(codepoint='1F64C 1F3FF', status='fully-qualified', emoji='🙌🏿', name='raising hands: dark skin tone', group='People & Body', sub_group='hands'),
    '👐': EmojiEntry(codepoint='1F450', status='fully-qualified', emoji='👐', name='open hands', group='People & Body', sub_group='hands'),
    '👐🏻': EmojiEntry(codepoint='1F450 1F3FB', status='fully-qualified', emoji='👐🏻', name='open hands: light skin tone', group='People & Body', sub_group='hands'),
    '👐🏼': EmojiEntry(codepoint='1F450 1F3FC', status='fully-qualified', emoji='👐🏼', name='open hands: medium-light skin tone', group='People & Body', sub_group='hands'),
    '👐🏽': EmojiEntry(codepoint='1F450 1F3FD', status='fully-qualified', emoji='👐🏽', name='open hands: medium skin tone', group='People & Body', sub_group='hands'),
    '👐🏾': EmojiEntry(codepoint='1F450 1F3FE', status='fully-qualified', emoji='👐🏾', name='open hands: medium-dark skin tone', group='People & Body', sub_group='hands'),
    '👐🏿': EmojiEntry(codepoint='1F450 1F3FF', status='fully-qualified', emoji='👐🏿', name='open hands: dark skin tone', group='People & Body', sub_group='hands'),
    '🤲': EmojiEntry(codepoint='1F932', status='fully-qualified', emoji='🤲', name='palms up together', group='People & Body', sub_group='hands'),
    '🤲🏻': EmojiEntry(codepoint='1F932 1F3FB', status='fully-qualified', emoji='🤲🏻', name='palms up together: light skin tone', group='People & Body', sub_group='hands'),
    '🤲🏼': EmojiEntry(codepoint='1F932 1F3FC', status='fully-qualified', emoji='🤲🏼', name='palms up together: medium-light skin tone', group='People & Body', sub_group='hands'),
    '🤲🏽': EmojiEntry(codepoint='1F932 1F3FD', status='fully-qualified', emoji='🤲🏽', name='palms up together: medium skin tone', group='People & Body', sub_group='hands'),
    '🤲🏾': EmojiEntry(codepoint='1F932 1F3FE', status='fully-qualified', emoji='🤲🏾', name='palms up together: medium-dark skin tone', group='People & Body', sub_group='hands'),
    '🤲🏿': EmojiEntry(codepoint='1F932 1F3FF', status='fully-qualified', emoji='🤲🏿', name='palms up together: dark skin tone', group='People & Body', sub_group='hands'),
    '🤝': EmojiEntry(codepoint='1F91D', status='fully-qualified', emoji='🤝', name='handshake', group='People & Body', sub_group='hands'),
    '🙏': EmojiEntry(codepoint='1F64F', status='fully-qualified', emoji='🙏', name='folded hands', group='People & Body', sub_group='hands'),
    '🙏🏻': EmojiEntry(codepoint='1F64F 1F3FB', status='fully-qualified', emoji='🙏🏻', name='folded hands: light skin tone', group='People & Body', sub_group='hands'),
    '🙏🏼': EmojiEntry(codepoint='1F64F 1F3FC', status='fully-qualified', emoji='🙏🏼', name='folded hands: medium-light skin tone', group='People & Body', sub_group='hands'),
    '🙏🏽': EmojiEntry(codepoint='1F64F 1F3FD', status='fully-qualified', emoji='🙏🏽', name='folded hands: medium skin tone', group='People & Body', sub_group='hands'),
    '🙏🏾': EmojiEntry(codepoint='1F64F 1F3FE', status='fully-qualified', emoji='🙏🏾', name='folded hands: medium-dark skin tone', group='People & Body', sub_group='hands'),
    '🙏🏿': EmojiEntry(codepoint='1F64F 1F3FF', status='fully-qualified', emoji='🙏🏿', name='folded hands: dark skin tone', group='People & Body', sub_group='hands'),
    '✍️': EmojiEntry(codepoint='270D FE0F', status='fully-qualified', emoji='✍️', name='writing hand', group='People & Body', sub_group='hand-prop'),
    '✍': EmojiEntry(codepoint='270D', status='unqualified', emoji='✍', name='writing hand', group='People & Body', sub_group='hand-prop'),
    '✍🏻': EmojiEntry(codepoint='270D 1F3FB', status='fully-qualified', emoji='✍🏻', name='writing hand: light skin tone', group='People & Body', sub_group='hand-prop'),
    '✍🏼': EmojiEntry(codepoint='270D 1F3FC', status='fully-qualified', emoji='✍🏼', name='writing hand: medium-light skin tone', group='People & Body', sub_group='hand-prop'),
    '✍🏽': EmojiEntry(codepoint='270D 1F3FD', status='fully-qualified', emoji='✍🏽', name='writing hand: medium skin tone', group='People & Body', sub_group='hand-prop'),
    '✍🏾': EmojiEntry(codepoint='270D 1F3FE', status='fully-qualified', emoji='✍🏾', name='writing hand: medium-dark skin tone', group='People & Body', sub_group='hand-prop'),
    '✍🏿': EmojiEntry(codepoint='270D 1F3FF', status='fully-qualified', emoji='✍🏿', name='writing hand: dark skin tone', group='People & Body', sub_group='hand-prop'),
    '💅': EmojiEntry(codepoint='1F485', status='fully-qualified', emoji='💅', name='nail polish', group='People & Body', sub_group='hand-prop'),
    '💅🏻': EmojiEntry(codepoint='1F485 1F3FB', status='fully-qualified', emoji='💅🏻', name='nail polish: light skin tone', group='People & Body', sub_group='hand-prop'),
    '💅🏼': EmojiEntry(codepoint='1F485 1F3FC', status='fully-qualified', emoji='💅🏼', name='nail polish: medium-light skin tone', group='People & Body', sub_group='hand-prop'),
    '💅🏽': EmojiEntry(codepoint='1F485 1F3FD', status='fully-qualified', emoji='💅🏽', name='nail polish: medium skin tone', group='People & Body', sub_group='hand-prop'),
    '💅🏾': EmojiEntry(codepoint='1F485 1F3FE', status='fully-qualified', emoji='💅🏾', name='nail polish: medium-dark skin tone', group='People & Body', sub_group='hand-prop'),
    '💅🏿': EmojiEntry(codepoint='1F485 1F3FF', status='fully-qualified', emoji='💅🏿', name='nail polish: dark skin tone', group='People & Body', sub_group='hand-prop'),
    '🤳': EmojiEntry(codepoint='1F933', status='fully-qualified', emoji='🤳', name='selfie', group='People & Body', sub_group='hand-prop'),
    '🤳🏻': EmojiEntry(codepoint='1F933 1F3FB', status='fully-qualified', emoji='🤳🏻', name='selfie: light skin tone', group='People & Body', sub_group='hand-prop'),
    '🤳🏼': EmojiEntry(codepoint='1F933 1F3FC', status='fully-qualified', emoji='🤳🏼', name='selfie: medium-light skin tone', group='People & Body', sub_group='hand-prop'),
    '🤳🏽': EmojiEntry(codepoint='1F933 1F3FD', status='fully-qualified', emoji='🤳🏽', name='selfie: medium skin tone', group='People & Body', sub_group='hand-prop'),
    '🤳🏾': EmojiEntry(codepoint='1F933 1F3FE', status='fully-qualified', emoji='🤳🏾', name='selfie: medium-dark skin tone', group='People & Body', sub_group='hand-prop'),
    '🤳🏿': EmojiEntry(codepoint='1F933 1F3FF', status='fully-qualified', emoji='🤳🏿', name='selfie: dark skin tone', group='People & Body', sub_group='hand-prop'),
    '💪': EmojiEntry(codepoint='1F4AA', status='fully-qualified', emoji='💪', name='flexed biceps', group='People & Body', sub_group='body-parts'),
    '💪🏻': EmojiEntry(codepoint='1F4AA 1F3FB', status='fully-qualified', emoji='💪🏻', name='flexed biceps: light skin tone', group='People & Body', sub_group='body-parts'),
    '💪🏼': EmojiEntry(codepoint='1F4AA 1F3FC', status='fully-qualified', emoji='💪🏼', name='flexed biceps: medium-light skin tone', group='People & Body', sub_group='body-parts'),
    '💪🏽': EmojiEntry(codepoint='1F4AA 1F3FD', status='fully-qualified', emoji='💪🏽', name='flexed biceps: medium skin tone', group='People & Body', sub_group='body-parts'),
    '💪🏾': EmojiEntry(codepoint='1F4AA 1F3FE', status='fully-qualified', emoji='💪🏾', name='flexed biceps: medium-dark skin tone', group='People & Body', sub_group='body-parts'),
    '💪🏿': EmojiEntry(codepoint='1F4AA 1F3FF', status='fully-qualified', emoji='💪🏿', name='flexed biceps: dark skin tone', group='People & Body', sub_group='body-parts'),
    '🦾': EmojiEntry(codepoint='1F9BE', status='fully-qualified', emoji='🦾', name='mechanical arm', group='People & Body', sub_group='body-parts'),
    '🦿': EmojiEntry(codepoint='1F9BF', status='fully-qualified', emoji='🦿', name='mechanical leg', group='People & Body', sub_group='body-parts'),
    '🦵': EmojiEntry(codepoint='1F9B5', status='fully-qualified', emoji='🦵', name='leg', group='People & Body', sub_group='body-parts'),
    '🦵🏻': EmojiEntry(codepoint='1F9B5 1F3FB', status='fully-qualified', emoji='🦵🏻', name='leg: light skin tone', group='People & Body', sub_group='body-parts'),
    '🦵🏼': EmojiEntry(codepoint='1F9B5 1F3FC', status='fully-qualified', emoji='🦵🏼', name='leg: medium-light skin tone', group='People & Body', sub_group='body-parts'),
    '🦵🏽': EmojiEntry(codepoint='1F9B5 1F3FD', status='fully-qualified', emoji='🦵🏽', name='leg: medium skin tone', group='People & Body', sub_group='body-parts'),
    '🦵🏾': EmojiEntry(codepoint='1F9B5 1F3FE', status='fully-qualified', emoji='🦵🏾', name='leg: medium-dark skin tone', group='People & Body', sub_group='body-parts'),
    '🦵🏿': EmojiEntry(codepoint='1F9B5 1F3FF', status='fully-qualified', emoji='🦵🏿', name='leg: dark skin tone', group='People & Body', sub_group='body-parts'),
    '🦶': EmojiEntry(codepoint='1F9B6', status='fully-qualified', emoji='🦶', name='foot', group='People & Body', sub_group='body-parts'),
    '🦶🏻': EmojiEntry(codepoint='1F9B6 1F3FB', status='fully-qualified', emoji='🦶🏻', name='foot: light skin tone', group='People & Body', sub_group='body-parts'),
    '🦶🏼': EmojiEntry(codepoint='1F9B6 1F3FC', status='fully-qualified', emoji='🦶🏼', name='foot: medium-light skin tone', group='People & Body', sub_group='body-parts'),
    '🦶🏽': EmojiEntry(codepoint='1F9B6 1F3FD', status='fully-qualified', emoji='🦶🏽', name='foot: medium skin tone', group='People & Body', sub_group='body-parts'),
    '🦶🏾': EmojiEntry(codepoint='1F9B6 1F3FE', status='fully-qualified', emoji='🦶🏾', name='foot: medium-dark skin tone', group='People & Body', sub_group='body-parts'),
    '🦶🏿': EmojiEntry(codepoint='1F9B6 1F3FF', status='fully-qualified', emoji='🦶🏿', name='foot: dark skin tone', group='People & Body', sub_group='body-parts'),
    '👂': EmojiEntry(codepoint='1F442', status='fully-qualified', emoji='👂', name='ear', group='People & Body', sub_group='body-parts'),
    '👂🏻': EmojiEntry(codepoint='1F442 1F3FB', status='fully-qualified', emoji='👂🏻', name='ear: light skin tone', group='People & Body', sub_group='body-parts'),
    '👂🏼': EmojiEntry(codepoint='1F442 1F3FC', status='fully-qualified', emoji='👂🏼', name='ear: medium-light skin tone', group='People & Body', sub_group='body-parts'),
    '👂🏽': EmojiEntry(codepoint='1F442 1F3FD', status='fully-qualified', emoji='👂🏽', name='ear: medium skin tone', group='People & Body', sub_group='body-parts'),
    '👂🏾': EmojiEntry(codepoint='1F442 1F3FE', status='fully-qualified', emoji='👂🏾', name='ear: medium-dark skin tone', group='People & Body', sub_group='body-parts'),
    '👂🏿': EmojiEntry(codepoint='1F442 1F3FF', status='fully-qualified', emoji='👂🏿', name='ear: dark skin tone', group='People & Body', sub_group='body-parts'),
    '🦻': EmojiEntry(codepoint='1F9BB', status='fully-qualified', emoji='🦻', name='ear with hearing aid', group='People & Body', sub_group='body-parts'),
    '🦻🏻': EmojiEntry(codepoint='1F9BB 1F3FB', status='fully-qualified', emoji='🦻🏻', name='ear with hearing aid: light skin tone', group='People & Body', sub_group='body-parts'),
    '🦻🏼': EmojiEntry(codepoint='1F9BB 1F3FC', status='fully-qualified', emoji='🦻🏼', name='ear with hearing aid: medium-light skin tone', group='People & Body', sub_group='body-parts'),
    '🦻🏽': EmojiEntry(codepoint='1F9BB 1F3FD', status='fully-qualified', emoji='🦻🏽', name='ear with hearing aid: medium skin tone', group='People & Body', sub_group='body-parts'),
    '🦻🏾': EmojiEntry(codepoint='1F9BB 1F3FE', status='fully-qualified', emoji='🦻🏾', name='ear with hearing aid: medium-dark skin tone', group='People & Body', sub_group='body-parts'),
    '🦻🏿': EmojiEntry(codepoint='1F9BB 1F3FF', status='fully-qualified', emoji='🦻🏿', name='ear with hearing aid: dark skin tone', group='People & Body', sub_group='body-parts'),
    '👃': EmojiEntry(codepoint='1F443', status='fully-qualified', emoji='👃', name='nose', group='People & Body', sub_group='body-parts'),
    '👃🏻': EmojiEntry(codepoint='1F443 1F3FB', status='fully-qualified', emoji='👃🏻', name='nose: light skin tone', group='People & Body', sub_group='body-parts'),
    '👃🏼': EmojiEntry(codepoint='1F443 1F3FC', status='fully-qualified', emoji='👃🏼', name='nose: medium-light skin tone', group='People & Body', sub_group='body-parts'),
    '👃🏽': EmojiEntry(codepoint='1F443 1F3FD', status='fully-qualified', emoji='👃🏽', name='nose: medium skin tone', group='People & Body', sub_group='body-parts'),
    '👃🏾': EmojiEntry(codepoint='1F443 1F3FE', status='fully-qualified', emoji='👃🏾', name='nose: medium-dark skin tone', group='People & Body', sub_group='body-parts'),
    '👃🏿': EmojiEntry(codepoint='1F443 1F3FF', status='fully-qualified', emoji='👃🏿', name='nose: dark skin tone', group='People & Body', sub_group='body-parts'),
    '🧠': EmojiEntry(codepoint='1F9E0', status='fully-qualified', emoji='🧠', name='brain', group='People & Body', sub_group='body-parts'),
    '🫀': EmojiEntry(codepoint='1FAC0', status='fully-qualified', emoji='🫀', name='anatomical heart', group='People & Body', sub_group='body-parts'),
    '🫁': EmojiEntry(codepoint='1FAC1', status='fully-qualified', emoji='🫁', name='lungs', group='People & Body', sub_group='body-parts'),
    '🦷': EmojiEntry(codepoint='1F9B7', status='fully-qualified', emoji='🦷', name='tooth', group='People & Body', sub_group='body-parts'),
    '🦴': EmojiEntry(codepoint='1F9B4', status='fully-qualified', emoji='🦴', name='bone', group='People & Body', sub_group='body-parts'),
    '👀': EmojiEntry(codepoint='1F440', status='fully-qualified', emoji='👀', name='eyes', group='People & Body', sub_group='body-parts'),
    '👁️': EmojiEntry(codepoint='1F441 FE0F', status='fully-qualified', emoji='👁️', name='eye', group='People & Body', sub_group='body-parts'),
    '👁': EmojiEntry(codepoint='1F441', status='unqualified', emoji='👁', name='eye', group='People & Body', sub_group='body-parts'),
    '👅': EmojiEntry(codepoint='1F445', status='fully-qualified', emoji='👅', name='tongue', group='People & Body', sub_group='body-parts'),
    '👄': EmojiEntry(codepoint='1F444', status='fully-qualified', emoji='👄', name='mouth', group='People & Body', sub_group='body-parts'),
    '👶': EmojiEntry(codepoint='1F476', status='fully-qualified', emoji='👶', name='baby', group='People & Body', sub_group='person'),
    '👶🏻': EmojiEntry(codepoint='1F476 1F3FB', status='fully-qualified', emoji='👶🏻', name='baby: light skin tone', group='People & Body', sub_group='person'),
    '👶🏼': EmojiEntry(codepoint='1F476 1F3FC', status='fully-qualified', emoji='👶🏼', name='baby: medium-light skin tone', group='People & Body', sub_group='person'),
    '👶🏽': EmojiEntry(codepoint='1F476 1F3FD', status='fully-qualified', emoji='👶🏽', name='baby: medium skin tone', group='People & Body', sub_group='person'),
    '👶🏾': EmojiEntry(codepoint='1F476 1F3FE', status='fully-qualified', emoji='👶🏾', name='baby: medium-dark skin tone', group='People & Body', sub_group='person'),
    '👶🏿': EmojiEntry(codepoint='1F476 1F3FF', status='fully-qualified', emoji='👶🏿', name='baby: dark skin tone', group='People & Body', sub_group='person'),
    '🧒': EmojiEntry(codepoint='1F9D2', status='fully-qualified', emoji='🧒', name='child', group='People & Body', sub_group='person'),
    '🧒🏻': EmojiEntry(codepoint='1F9D2 1F3FB', status='fully-qualified', emoji='🧒🏻', name='child: light skin tone', group='People & Body', sub_group='person'),
    '🧒🏼': EmojiEntry(codepoint='1F9D2 1F3FC', status='fully-qualified', emoji='🧒🏼', name='child: medium-light skin tone', group='People & Body', sub_group='person'),
    '🧒🏽': EmojiEntry(codepoint='1F9D2 1F3FD', status='fully-qualified', emoji='🧒🏽', name='child: medium skin tone', group='People & Body', sub_group='person'),
    '🧒🏾': EmojiEntry(codepoint='1F9D2 1F3FE', status='fully-qualified', emoji='🧒🏾', name='child: medium-dark skin tone', group='People & Body', sub_group='person'),
    '🧒🏿': EmojiEntry(codepoint='1F9D2 1F3FF', status='fully-qualified', emoji='🧒🏿', name='child: dark skin tone', group='People & Body', sub_group='person'),
    '👦': EmojiEntry(codepoint='1F466', status='fully-qualified', emoji='👦', name='boy', group='People & Body', sub_group='person'),
    '👦🏻': EmojiEntry(codepoint='1F466 1F3FB', status='fully-qualified', emoji='👦🏻', name='boy: light skin tone', group='People & Body', sub_group='person'),
    '👦🏼': EmojiEntry(codepoint='1F466 1F3FC', status='fully-qualified', emoji='👦🏼', name='boy: medium-light skin tone', group='People & Body', sub_group='person'),
    '👦🏽': EmojiEntry(codepoint='1F466 1F3FD', status='fully-qualified', emoji='👦🏽', name='boy: medium skin tone', group='People & Body', sub_group='person'),
    '👦🏾': EmojiEntry(codepoint='1F466 1F3FE', status='fully-qualified', emoji='👦🏾', name='boy: medium-dark skin tone', group='People & Body', sub_group='person'),
    '👦🏿': EmojiEntry(codepoint='1F466 1F3FF', status='fully-qualified', emoji='👦🏿', name='boy: dark skin tone', group='People & Body', sub_group='person'),
    '👧': EmojiEntry(codepoint='1F467', status='fully-qualified', emoji='👧', name='girl', group='People & Body', sub_group='person'),
    '👧🏻': EmojiEntry(codepoint='1F467 1F3FB', status='fully-qualified', emoji='👧🏻', name='girl: light skin tone', group='People & Body', sub_group='person'),
    '👧🏼': EmojiEntry(codepoint='1F467 1F3FC', status='fully-qualified', emoji='👧🏼', name='girl: medium-light skin tone', group='People & Body', sub_group='person'),
    '👧🏽': EmojiEntry(codepoint='1F467 1F3FD', status='fully-qualified', emoji='👧🏽', name='girl: medium skin tone', group='People & Body', sub_group='person'),
    '👧🏾': EmojiEntry(codepoint='1F467 1F3FE', status='fully-qualified', emoji='👧🏾', name='girl: medium-dark skin tone', group='People & Body', sub_group='person'),
    '👧🏿': EmojiEntry(codepoint='1F467 1F3FF', status='fully-qualified', emoji='👧🏿', name='girl: dark skin tone', group='People & Body', sub_group='person'),
    '🧑': EmojiEntry(codepoint='1F9D1', status='fully-qualified', emoji='🧑', name='person', group='People & Body', sub_group='person'),
    '🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FB', status='fully-qualified', emoji='🧑🏻', name='person: light skin tone', group='People & Body', sub_group='person'),
    '🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FC', status='fully-qualified', emoji='🧑🏼', name='person: medium-light skin tone', group='People & Body', sub_group='person'),
    '🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FD', status='fully-qualified', emoji='🧑🏽', name='person: medium skin tone', group='People & Body', sub_group='person'),
    '🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FE', status='fully-qualified', emoji='🧑🏾', name='person: medium-dark skin tone', group='People & Body', sub_group='person'),
    '🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FF', status='fully-qualified', emoji='🧑🏿', name='person: dark skin tone', group='People & Body', sub_group='person'),
    '👱': EmojiEntry(codepoint='1F471', status='fully-qualified', emoji='👱', name='person: blond hair', group='People & Body', sub_group='person'),
    '👱🏻': EmojiEntry(codepoint='1F471 1F3FB', status='fully-qualified', emoji='👱🏻', name='person: light skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏼': EmojiEntry(codepoint='1F471 1F3FC', status='fully-qualified', emoji='👱🏼', name='person: medium-light skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏽': EmojiEntry(codepoint='1F471 1F3FD', status='fully-qualified', emoji='👱🏽', name='person: medium skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏾': EmojiEntry(codepoint='1F471 1F3FE', status='fully-qualified', emoji='👱🏾', name='person: medium-dark skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏿': EmojiEntry(codepoint='1F471 1F3FF', status='fully-qualified', emoji='👱🏿', name='person: dark skin tone, blond hair', group='People & Body', sub_group='person'),
    '👨': EmojiEntry(codepoint='1F468', status='fully-qualified', emoji='👨', name='man', group='People & Body', sub_group='person'),
    '👨🏻': EmojiEntry(codepoint='1F468 1F3FB', status='fully-qualified', emoji='👨🏻', name='man: light skin tone', group='People & Body', sub_group='person'),
    '👨🏼': EmojiEntry(codepoint='1F468 1F3FC', status='fully-qualified', emoji='👨🏼', name='man: medium-light skin tone', group='People & Body', sub_group='person'),
    '👨🏽': EmojiEntry(codepoint='1F468 1F3FD', status='fully-qualified', emoji='👨🏽', name='man: medium skin tone', group='People & Body', sub_group='person'),
    '👨🏾': EmojiEntry(codepoint='1F468 1F3FE', status='fully-qualified', emoji='👨🏾', name='man: medium-dark skin tone', group='People & Body', sub_group='person'),
    '👨🏿': EmojiEntry(codepoint='1F468 1F3FF', status='fully-qualified', emoji='👨🏿', name='man: dark skin tone', group='People & Body', sub_group='person'),
    '🧔': EmojiEntry(codepoint='1F9D4', status='fully-qualified', emoji='🧔', name='person: beard', group='People & Body', sub_group='person'),
    '🧔🏻': EmojiEntry(codepoint='1F9D4 1F3FB', status='fully-qualified', emoji='🧔🏻', name='person: light skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏼': EmojiEntry(codepoint='1F9D4 1F3FC', status='fully-qualified', emoji='🧔🏼', name='person: medium-light skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏽': EmojiEntry(codepoint='1F9D4 1F3FD', status='fully-qualified', emoji='🧔🏽', name='person: medium skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏾': EmojiEntry(codepoint='1F9D4 1F3FE', status='fully-qualified', emoji='🧔🏾', name='person: medium-dark skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏿': EmojiEntry(codepoint='1F9D4 1F3FF', status='fully-qualified', emoji='🧔🏿', name='person: dark skin tone, beard', group='People & Body', sub_group='person'),
    '🧔‍♂️': EmojiEntry(codepoint='1F9D4 200D 2642 FE0F', status='fully-qualified', emoji='🧔\u200d♂️', name='man: beard', group='People & Body', sub_group='person'),
    '🧔‍♂': EmojiEntry(codepoint='1F9D4 200D 2642', status='minimally-qualified', emoji='🧔\u200d♂', name='man: beard', group='People & Body', sub_group='person'),
    '🧔🏻‍♂️': EmojiEntry(codepoint='1F9D4 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🧔🏻\u200d♂️', name='man: light skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏻‍♂': EmojiEntry(codepoint='1F9D4 1F3FB 200D 2642', status='minimally-qualified', emoji='🧔🏻\u200d♂', name='man: light skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏼‍♂️': EmojiEntry(codepoint='1F9D4 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🧔🏼\u200d♂️', name='man: medium-light skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏼‍♂': EmojiEntry(codepoint='1F9D4 1F3FC 200D 2642', status='minimally-qualified', emoji='🧔🏼\u200d♂', name='man: medium-light skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏽‍♂️': EmojiEntry(codepoint='1F9D4 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🧔🏽\u200d♂️', name='man: medium skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏽‍♂': EmojiEntry(codepoint='1F9D4 1F3FD 200D 2642', status='minimally-qualified', emoji='🧔🏽\u200d♂', name='man: medium skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏾‍♂️': EmojiEntry(codepoint='1F9D4 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🧔🏾\u200d♂️', name='man: medium-dark skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏾‍♂': EmojiEntry(codepoint='1F9D4 1F3FE 200D 2642', status='minimally-qualified', emoji='🧔🏾\u200d♂', name='man: medium-dark skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏿‍♂️': EmojiEntry(codepoint='1F9D4 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🧔🏿\u200d♂️', name='man: dark skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏿‍♂': EmojiEntry(codepoint='1F9D4 1F3FF 200D 2642', status='minimally-qualified', emoji='🧔🏿\u200d♂', name='man: dark skin tone, beard', group='People & Body', sub_group='person'),
    '🧔‍♀️': EmojiEntry(codepoint='1F9D4 200D 2640 FE0F', status='fully-qualified', emoji='🧔\u200d♀️', name='woman: beard', group='People & Body', sub_group='person'),
    '🧔‍♀': EmojiEntry(codepoint='1F9D4 200D 2640', status='minimally-qualified', emoji='🧔\u200d♀', name='woman: beard', group='People & Body', sub_group='person'),
    '🧔🏻‍♀️': EmojiEntry(codepoint='1F9D4 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🧔🏻\u200d♀️', name='woman: light skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏻‍♀': EmojiEntry(codepoint='1F9D4 1F3FB 200D 2640', status='minimally-qualified', emoji='🧔🏻\u200d♀', name='woman: light skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏼‍♀️': EmojiEntry(codepoint='1F9D4 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🧔🏼\u200d♀️', name='woman: medium-light skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏼‍♀': EmojiEntry(codepoint='1F9D4 1F3FC 200D 2640', status='minimally-qualified', emoji='🧔🏼\u200d♀', name='woman: medium-light skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏽‍♀️': EmojiEntry(codepoint='1F9D4 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🧔🏽\u200d♀️', name='woman: medium skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏽‍♀': EmojiEntry(codepoint='1F9D4 1F3FD 200D 2640', status='minimally-qualified', emoji='🧔🏽\u200d♀', name='woman: medium skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏾‍♀️': EmojiEntry(codepoint='1F9D4 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🧔🏾\u200d♀️', name='woman: medium-dark skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏾‍♀': EmojiEntry(codepoint='1F9D4 1F3FE 200D 2640', status='minimally-qualified', emoji='🧔🏾\u200d♀', name='woman: medium-dark skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏿‍♀️': EmojiEntry(codepoint='1F9D4 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🧔🏿\u200d♀️', name='woman: dark skin tone, beard', group='People & Body', sub_group='person'),
    '🧔🏿‍♀': EmojiEntry(codepoint='1F9D4 1F3FF 200D 2640', status='minimally-qualified', emoji='🧔🏿\u200d♀', name='woman: dark skin tone, beard', group='People & Body', sub_group='person'),
    '👨‍🦰': EmojiEntry(codepoint='1F468 200D 1F9B0', status='fully-qualified', emoji='👨\u200d🦰', name='man: red hair', group='People & Body', sub_group='person'),
    '👨🏻‍🦰': EmojiEntry(codepoint='1F468 1F3FB 200D 1F9B0', status='fully-qualified', emoji='👨🏻\u200d🦰', name='man: light skin tone, red hair', group='People & Body', sub_group='person'),
    '👨🏼‍🦰': EmojiEntry(codepoint='1F468 1F3FC 200D 1F9B0', status='fully-qualified', emoji='👨🏼\u200d🦰', name='man: medium-light skin tone, red hair', group='People & Body', sub_group='person'),
    '👨🏽‍🦰': EmojiEntry(codepoint='1F468 1F3FD 200D 1F9B0', status='fully-qualified', emoji='👨🏽\u200d🦰', name='man: medium skin tone, red hair', group='People & Body', sub_group='person'),
    '👨🏾‍🦰': EmojiEntry(codepoint='1F468 1F3FE 200D 1F9B0', status='fully-qualified', emoji='👨🏾\u200d🦰', name='man: medium-dark skin tone, red hair', group='People & Body', sub_group='person'),
    '👨🏿‍🦰': EmojiEntry(codepoint='1F468 1F3FF 200D 1F9B0', status='fully-qualified', emoji='👨🏿\u200d🦰', name='man: dark skin tone, red hair', group='People & Body', sub_group='person'),
    '👨‍🦱': EmojiEntry(codepoint='1F468 200D 1F9B1', status='fully-qualified', emoji='👨\u200d🦱', name='man: curly hair', group='People & Body', sub_group='person'),
    '👨🏻‍🦱': EmojiEntry(codepoint='1F468 1F3FB 200D 1F9B1', status='fully-qualified', emoji='👨🏻\u200d🦱', name='man: light skin tone, curly hair', group='People & Body', sub_group='person'),
    '👨🏼‍🦱': EmojiEntry(codepoint='1F468 1F3FC 200D 1F9B1', status='fully-qualified', emoji='👨🏼\u200d🦱', name='man: medium-light skin tone, curly hair', group='People & Body', sub_group='person'),
    '👨🏽‍🦱': EmojiEntry(codepoint='1F468 1F3FD 200D 1F9B1', status='fully-qualified', emoji='👨🏽\u200d🦱', name='man: medium skin tone, curly hair', group='People & Body', sub_group='person'),
    '👨🏾‍🦱': EmojiEntry(codepoint='1F468 1F3FE 200D 1F9B1', status='fully-qualified', emoji='👨🏾\u200d🦱', name='man: medium-dark skin tone, curly hair', group='People & Body', sub_group='person'),
    '👨🏿‍🦱': EmojiEntry(codepoint='1F468 1F3FF 200D 1F9B1', status='fully-qualified', emoji='👨🏿\u200d🦱', name='man: dark skin tone, curly hair', group='People & Body', sub_group='person'),
    '👨‍🦳': EmojiEntry(codepoint='1F468 200D 1F9B3', status='fully-qualified', emoji='👨\u200d🦳', name='man: white hair', group='People & Body', sub_group='person'),
    '👨🏻‍🦳': EmojiEntry(codepoint='1F468 1F3FB 200D 1F9B3', status='fully-qualified', emoji='👨🏻\u200d🦳', name='man: light skin tone, white hair', group='People & Body', sub_group='person'),
    '👨🏼‍🦳': EmojiEntry(codepoint='1F468 1F3FC 200D 1F9B3', status='fully-qualified', emoji='👨🏼\u200d🦳', name='man: medium-light skin tone, white hair', group='People & Body', sub_group='person'),
    '👨🏽‍🦳': EmojiEntry(codepoint='1F468 1F3FD 200D 1F9B3', status='fully-qualified', emoji='👨🏽\u200d🦳', name='man: medium skin tone, white hair', group='People & Body', sub_group='person'),
    '👨🏾‍🦳': EmojiEntry(codepoint='1F468 1F3FE 200D 1F9B3', status='fully-qualified', emoji='👨🏾\u200d🦳', name='man: medium-dark skin tone, white hair', group='People & Body', sub_group='person'),
    '👨🏿‍🦳': EmojiEntry(codepoint='1F468 1F3FF 200D 1F9B3', status='fully-qualified', emoji='👨🏿\u200d🦳', name='man: dark skin tone, white hair', group='People & Body', sub_group='person'),
    '👨‍🦲': EmojiEntry(codepoint='1F468 200D 1F9B2', status='fully-qualified', emoji='👨\u200d🦲', name='man: bald', group='People & Body', sub_group='person'),
    '👨🏻‍🦲': EmojiEntry(codepoint='1F468 1F3FB 200D 1F9B2', status='fully-qualified', emoji='👨🏻\u200d🦲', name='man: light skin tone, bald', group='People & Body', sub_group='person'),
    '👨🏼‍🦲': EmojiEntry(codepoint='1F468 1F3FC 200D 1F9B2', status='fully-qualified', emoji='👨🏼\u200d🦲', name='man: medium-light skin tone, bald', group='People & Body', sub_group='person'),
    '👨🏽‍🦲': EmojiEntry(codepoint='1F468 1F3FD 200D 1F9B2', status='fully-qualified', emoji='👨🏽\u200d🦲', name='man: medium skin tone, bald', group='People & Body', sub_group='person'),
    '👨🏾‍🦲': EmojiEntry(codepoint='1F468 1F3FE 200D 1F9B2', status='fully-qualified', emoji='👨🏾\u200d🦲', name='man: medium-dark skin tone, bald', group='People & Body', sub_group='person'),
    '👨🏿‍🦲': EmojiEntry(codepoint='1F468 1F3FF 200D 1F9B2', status='fully-qualified', emoji='👨🏿\u200d🦲', name='man: dark skin tone, bald', group='People & Body', sub_group='person'),
    '👩': EmojiEntry(codepoint='1F469', status='fully-qualified', emoji='👩', name='woman', group='People & Body', sub_group='person'),
    '👩🏻': EmojiEntry(codepoint='1F469 1F3FB', status='fully-qualified', emoji='👩🏻', name='woman: light skin tone', group='People & Body', sub_group='person'),
    '👩🏼': EmojiEntry(codepoint='1F469 1F3FC', status='fully-qualified', emoji='👩🏼', name='woman: medium-light skin tone', group='People & Body', sub_group='person'),
    '👩🏽': EmojiEntry(codepoint='1F469 1F3FD', status='fully-qualified', emoji='👩🏽', name='woman: medium skin tone', group='People & Body', sub_group='person'),
    '👩🏾': EmojiEntry(codepoint='1F469 1F3FE', status='fully-qualified', emoji='👩🏾', name='woman: medium-dark skin tone', group='People & Body', sub_group='person'),
    '👩🏿': EmojiEntry(codepoint='1F469 1F3FF', status='fully-qualified', emoji='👩🏿', name='woman: dark skin tone', group='People & Body', sub_group='person'),
    '👩‍🦰': EmojiEntry(codepoint='1F469 200D 1F9B0', status='fully-qualified', emoji='👩\u200d🦰', name='woman: red hair', group='People & Body', sub_group='person'),
    '👩🏻‍🦰': EmojiEntry(codepoint='1F469 1F3FB 200D 1F9B0', status='fully-qualified', emoji='👩🏻\u200d🦰', name='woman: light skin tone, red hair', group='People & Body', sub_group='person'),
    '👩🏼‍🦰': EmojiEntry(codepoint='1F469 1F3FC 200D 1F9B0', status='fully-qualified', emoji='👩🏼\u200d🦰', name='woman: medium-light skin tone, red hair', group='People & Body', sub_group='person'),
    '👩🏽‍🦰': EmojiEntry(codepoint='1F469 1F3FD 200D 1F9B0', status='fully-qualified', emoji='👩🏽\u200d🦰', name='woman: medium skin tone, red hair', group='People & Body', sub_group='person'),
    '👩🏾‍🦰': EmojiEntry(codepoint='1F469 1F3FE 200D 1F9B0', status='fully-qualified', emoji='👩🏾\u200d🦰', name='woman: medium-dark skin tone, red hair', group='People & Body', sub_group='person'),
    '👩🏿‍🦰': EmojiEntry(codepoint='1F469 1F3FF 200D 1F9B0', status='fully-qualified', emoji='👩🏿\u200d🦰', name='woman: dark skin tone, red hair', group='People & Body', sub_group='person'),
    '🧑‍🦰': EmojiEntry(codepoint='1F9D1 200D 1F9B0', status='fully-qualified', emoji='🧑\u200d🦰', name='person: red hair', group='People & Body', sub_group='person'),
    '🧑🏻‍🦰': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F9B0', status='fully-qualified', emoji='🧑🏻\u200d🦰', name='person: light skin tone, red hair', group='People & Body', sub_group='person'),
    '🧑🏼‍🦰': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F9B0', status='fully-qualified', emoji='🧑🏼\u200d🦰', name='person: medium-light skin tone, red hair', group='People & Body', sub_group='person'),
    '🧑🏽‍🦰': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F9B0', status='fully-qualified', emoji='🧑🏽\u200d🦰', name='person: medium skin tone, red hair', group='People & Body', sub_group='person'),
    '🧑🏾‍🦰': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F9B0', status='fully-qualified', emoji='🧑🏾\u200d🦰', name='person: medium-dark skin tone, red hair', group='People & Body', sub_group='person'),
    '🧑🏿‍🦰': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F9B0', status='fully-qualified', emoji='🧑🏿\u200d🦰', name='person: dark skin tone, red hair', group='People & Body', sub_group='person'),
    '👩‍🦱': EmojiEntry(codepoint='1F469 200D 1F9B1', status='fully-qualified', emoji='👩\u200d🦱', name='woman: curly hair', group='People & Body', sub_group='person'),
    '👩🏻‍🦱': EmojiEntry(codepoint='1F469 1F3FB 200D 1F9B1', status='fully-qualified', emoji='👩🏻\u200d🦱', name='woman: light skin tone, curly hair', group='People & Body', sub_group='person'),
    '👩🏼‍🦱': EmojiEntry(codepoint='1F469 1F3FC 200D 1F9B1', status='fully-qualified', emoji='👩🏼\u200d🦱', name='woman: medium-light skin tone, curly hair', group='People & Body', sub_group='person'),
    '👩🏽‍🦱': EmojiEntry(codepoint='1F469 1F3FD 200D 1F9B1', status='fully-qualified', emoji='👩🏽\u200d🦱', name='woman: medium skin tone, curly hair', group='People & Body', sub_group='person'),
    '👩🏾‍🦱': EmojiEntry(codepoint='1F469 1F3FE 200D 1F9B1', status='fully-qualified', emoji='👩🏾\u200d🦱', name='woman: medium-dark skin tone, curly hair', group='People & Body', sub_group='person'),
    '👩🏿‍🦱': EmojiEntry(codepoint='1F469 1F3FF 200D 1F9B1', status='fully-qualified', emoji='👩🏿\u200d🦱', name='woman: dark skin tone, curly hair', group='People & Body', sub_group='person'),
    '🧑‍🦱': EmojiEntry(codepoint='1F9D1 200D 1F9B1', status='fully-qualified', emoji='🧑\u200d🦱', name='person: curly hair', group='People & Body', sub_group='person'),
    '🧑🏻‍🦱': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F9B1', status='fully-qualified', emoji='🧑🏻\u200d🦱', name='person: light skin tone, curly hair', group='People & Body', sub_group='person'),
    '🧑🏼‍🦱': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F9B1', status='fully-qualified', emoji='🧑🏼\u200d🦱', name='person: medium-light skin tone, curly hair', group='People & Body', sub_group='person'),
    '🧑🏽‍🦱': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F9B1', status='fully-qualified', emoji='🧑🏽\u200d🦱', name='person: medium skin tone, curly hair', group='People & Body', sub_group='person'),
    '🧑🏾‍🦱': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F9B1', status='fully-qualified', emoji='🧑🏾\u200d🦱', name='person: medium-dark skin tone, curly hair', group='People & Body', sub_group='person'),
    '🧑🏿‍🦱': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F9B1', status='fully-qualified', emoji='🧑🏿\u200d🦱', name='person: dark skin tone, curly hair', group='People & Body', sub_group='person'),
    '👩‍🦳': EmojiEntry(codepoint='1F469 200D 1F9B3', status='fully-qualified', emoji='👩\u200d🦳', name='woman: white hair', group='People & Body', sub_group='person'),
    '👩🏻‍🦳': EmojiEntry(codepoint='1F469 1F3FB 200D 1F9B3', status='fully-qualified', emoji='👩🏻\u200d🦳', name='woman: light skin tone, white hair', group='People & Body', sub_group='person'),
    '👩🏼‍🦳': EmojiEntry(codepoint='1F469 1F3FC 200D 1F9B3', status='fully-qualified', emoji='👩🏼\u200d🦳', name='woman: medium-light skin tone, white hair', group='People & Body', sub_group='person'),
    '👩🏽‍🦳': EmojiEntry(codepoint='1F469 1F3FD 200D 1F9B3', status='fully-qualified', emoji='👩🏽\u200d🦳', name='woman: medium skin tone, white hair', group='People & Body', sub_group='person'),
    '👩🏾‍🦳': EmojiEntry(codepoint='1F469 1F3FE 200D 1F9B3', status='fully-qualified', emoji='👩🏾\u200d🦳', name='woman: medium-dark skin tone, white hair', group='People & Body', sub_group='person'),
    '👩🏿‍🦳': EmojiEntry(codepoint='1F469 1F3FF 200D 1F9B3', status='fully-qualified', emoji='👩🏿\u200d🦳', name='woman: dark skin tone, white hair', group='People & Body', sub_group='person'),
    '🧑‍🦳': EmojiEntry(codepoint='1F9D1 200D 1F9B3', status='fully-qualified', emoji='🧑\u200d🦳', name='person: white hair', group='People & Body', sub_group='person'),
    '🧑🏻‍🦳': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F9B3', status='fully-qualified', emoji='🧑🏻\u200d🦳', name='person: light skin tone, white hair', group='People & Body', sub_group='person'),
    '🧑🏼‍🦳': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F9B3', status='fully-qualified', emoji='🧑🏼\u200d🦳', name='person: medium-light skin tone, white hair', group='People & Body', sub_group='person'),
    '🧑🏽‍🦳': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F9B3', status='fully-qualified', emoji='🧑🏽\u200d🦳', name='person: medium skin tone, white hair', group='People & Body', sub_group='person'),
    '🧑🏾‍🦳': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F9B3', status='fully-qualified', emoji='🧑🏾\u200d🦳', name='person: medium-dark skin tone, white hair', group='People & Body', sub_group='person'),
    '🧑🏿‍🦳': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F9B3', status='fully-qualified', emoji='🧑🏿\u200d🦳', name='person: dark skin tone, white hair', group='People & Body', sub_group='person'),
    '👩‍🦲': EmojiEntry(codepoint='1F469 200D 1F9B2', status='fully-qualified', emoji='👩\u200d🦲', name='woman: bald', group='People & Body', sub_group='person'),
    '👩🏻‍🦲': EmojiEntry(codepoint='1F469 1F3FB 200D 1F9B2', status='fully-qualified', emoji='👩🏻\u200d🦲', name='woman: light skin tone, bald', group='People & Body', sub_group='person'),
    '👩🏼‍🦲': EmojiEntry(codepoint='1F469 1F3FC 200D 1F9B2', status='fully-qualified', emoji='👩🏼\u200d🦲', name='woman: medium-light skin tone, bald', group='People & Body', sub_group='person'),
    '👩🏽‍🦲': EmojiEntry(codepoint='1F469 1F3FD 200D 1F9B2', status='fully-qualified', emoji='👩🏽\u200d🦲', name='woman: medium skin tone, bald', group='People & Body', sub_group='person'),
    '👩🏾‍🦲': EmojiEntry(codepoint='1F469 1F3FE 200D 1F9B2', status='fully-qualified', emoji='👩🏾\u200d🦲', name='woman: medium-dark skin tone, bald', group='People & Body', sub_group='person'),
    '👩🏿‍🦲': EmojiEntry(codepoint='1F469 1F3FF 200D 1F9B2', status='fully-qualified', emoji='👩🏿\u200d🦲', name='woman: dark skin tone, bald', group='People & Body', sub_group='person'),
    '🧑‍🦲': EmojiEntry(codepoint='1F9D1 200D 1F9B2', status='fully-qualified', emoji='🧑\u200d🦲', name='person: bald', group='People & Body', sub_group='person'),
    '🧑🏻‍🦲': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F9B2', status='fully-qualified', emoji='🧑🏻\u200d🦲', name='person: light skin tone, bald', group='People & Body', sub_group='person'),
    '🧑🏼‍🦲': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F9B2', status='fully-qualified', emoji='🧑🏼\u200d🦲', name='person: medium-light skin tone, bald', group='People & Body', sub_group='person'),
    '🧑🏽‍🦲': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F9B2', status='fully-qualified', emoji='🧑🏽\u200d🦲', name='person: medium skin tone, bald', group='People & Body', sub_group='person'),
    '🧑🏾‍🦲': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F9B2', status='fully-qualified', emoji='🧑🏾\u200d🦲', name='person: medium-dark skin tone, bald', group='People & Body', sub_group='person'),
    '🧑🏿‍🦲': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F9B2', status='fully-qualified', emoji='🧑🏿\u200d🦲', name='person: dark skin tone, bald', group='People & Body', sub_group='person'),
    '👱‍♀️': EmojiEntry(codepoint='1F471 200D 2640 FE0F', status='fully-qualified', emoji='👱\u200d♀️', name='woman: blond hair', group='People & Body', sub_group='person'),
    '👱‍♀': EmojiEntry(codepoint='1F471 200D 2640', status='minimally-qualified', emoji='👱\u200d♀', name='woman: blond hair', group='People & Body', sub_group='person'),
    '👱🏻‍♀️': EmojiEntry(codepoint='1F471 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='👱🏻\u200d♀️', name='woman: light skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏻‍♀': EmojiEntry(codepoint='1F471 1F3FB 200D 2640', status='minimally-qualified', emoji='👱🏻\u200d♀', name='woman: light skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏼‍♀️': EmojiEntry(codepoint='1F471 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='👱🏼\u200d♀️', name='woman: medium-light skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏼‍♀': EmojiEntry(codepoint='1F471 1F3FC 200D 2640', status='minimally-qualified', emoji='👱🏼\u200d♀', name='woman: medium-light skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏽‍♀️': EmojiEntry(codepoint='1F471 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='👱🏽\u200d♀️', name='woman: medium skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏽‍♀': EmojiEntry(codepoint='1F471 1F3FD 200D 2640', status='minimally-qualified', emoji='👱🏽\u200d♀', name='woman: medium skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏾‍♀️': EmojiEntry(codepoint='1F471 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='👱🏾\u200d♀️', name='woman: medium-dark skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏾‍♀': EmojiEntry(codepoint='1F471 1F3FE 200D 2640', status='minimally-qualified', emoji='👱🏾\u200d♀', name='woman: medium-dark skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏿‍♀️': EmojiEntry(codepoint='1F471 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='👱🏿\u200d♀️', name='woman: dark skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏿‍♀': EmojiEntry(codepoint='1F471 1F3FF 200D 2640', status='minimally-qualified', emoji='👱🏿\u200d♀', name='woman: dark skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱‍♂️': EmojiEntry(codepoint='1F471 200D 2642 FE0F', status='fully-qualified', emoji='👱\u200d♂️', name='man: blond hair', group='People & Body', sub_group='person'),
    '👱‍♂': EmojiEntry(codepoint='1F471 200D 2642', status='minimally-qualified', emoji='👱\u200d♂', name='man: blond hair', group='People & Body', sub_group='person'),
    '👱🏻‍♂️': EmojiEntry(codepoint='1F471 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='👱🏻\u200d♂️', name='man: light skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏻‍♂': EmojiEntry(codepoint='1F471 1F3FB 200D 2642', status='minimally-qualified', emoji='👱🏻\u200d♂', name='man: light skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏼‍♂️': EmojiEntry(codepoint='1F471 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='👱🏼\u200d♂️', name='man: medium-light skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏼‍♂': EmojiEntry(codepoint='1F471 1F3FC 200D 2642', status='minimally-qualified', emoji='👱🏼\u200d♂', name='man: medium-light skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏽‍♂️': EmojiEntry(codepoint='1F471 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='👱🏽\u200d♂️', name='man: medium skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏽‍♂': EmojiEntry(codepoint='1F471 1F3FD 200D 2642', status='minimally-qualified', emoji='👱🏽\u200d♂', name='man: medium skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏾‍♂️': EmojiEntry(codepoint='1F471 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='👱🏾\u200d♂️', name='man: medium-dark skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏾‍♂': EmojiEntry(codepoint='1F471 1F3FE 200D 2642', status='minimally-qualified', emoji='👱🏾\u200d♂', name='man: medium-dark skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏿‍♂️': EmojiEntry(codepoint='1F471 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='👱🏿\u200d♂️', name='man: dark skin tone, blond hair', group='People & Body', sub_group='person'),
    '👱🏿‍♂': EmojiEntry(codepoint='1F471 1F3FF 200D 2642', status='minimally-qualified', emoji='👱🏿\u200d♂', name='man: dark skin tone, blond hair', group='People & Body', sub_group='person'),
    '🧓': EmojiEntry(codepoint='1F9D3', status='fully-qualified', emoji='🧓', name='older person', group='People & Body', sub_group='person'),
    '🧓🏻': EmojiEntry(codepoint='1F9D3 1F3FB', status='fully-qualified', emoji='🧓🏻', name='older person: light skin tone', group='People & Body', sub_group='person'),
    '🧓🏼': EmojiEntry(codepoint='1F9D3 1F3FC', status='fully-qualified', emoji='🧓🏼', name='older person: medium-light skin tone', group='People & Body', sub_group='person'),
    '🧓🏽': EmojiEntry(codepoint='1F9D3 1F3FD', status='fully-qualified', emoji='🧓🏽', name='older person: medium skin tone', group='People & Body', sub_group='person'),
    '🧓🏾': EmojiEntry(codepoint='1F9D3 1F3FE', status='fully-qualified', emoji='🧓🏾', name='older person: medium-dark skin tone', group='People & Body', sub_group='person'),
    '🧓🏿': EmojiEntry(codepoint='1F9D3 1F3FF', status='fully-qualified', emoji='🧓🏿', name='older person: dark skin tone', group='People & Body', sub_group='person'),
    '👴': EmojiEntry(codepoint='1F474', status='fully-qualified', emoji='👴', name='old man', group='People & Body', sub_group='person'),
    '👴🏻': EmojiEntry(codepoint='1F474 1F3FB', status='fully-qualified', emoji='👴🏻', name='old man: light skin tone', group='People & Body', sub_group='person'),
    '👴🏼': EmojiEntry(codepoint='1F474 1F3FC', status='fully-qualified', emoji='👴🏼', name='old man: medium-light skin tone', group='People & Body', sub_group='person'),
    '👴🏽': EmojiEntry(codepoint='1F474 1F3FD', status='fully-qualified', emoji='👴🏽', name='old man: medium skin tone', group='People & Body', sub_group='person'),
    '👴🏾': EmojiEntry(codepoint='1F474 1F3FE', status='fully-qualified', emoji='👴🏾', name='old man: medium-dark skin tone', group='People & Body', sub_group='person'),
    '👴🏿': EmojiEntry(codepoint='1F474 1F3FF', status='fully-qualified', emoji='👴🏿', name='old man: dark skin tone', group='People & Body', sub_group='person'),
    '👵': EmojiEntry(codepoint='1F475', status='fully-qualified', emoji='👵', name='old woman', group='People & Body', sub_group='person'),
    '👵🏻': EmojiEntry(codepoint='1F475 1F3FB', status='fully-qualified', emoji='👵🏻', name='old woman: light skin tone', group='People & Body', sub_group='person'),
    '👵🏼': EmojiEntry(codepoint='1F475 1F3FC', status='fully-qualified', emoji='👵🏼', name='old woman: medium-light skin tone', group='People & Body', sub_group='person'),
    '👵🏽': EmojiEntry(codepoint='1F475 1F3FD', status='fully-qualified', emoji='👵🏽', name='old woman: medium skin tone', group='People & Body', sub_group='person'),
    '👵🏾': EmojiEntry(codepoint='1F475 1F3FE', status='fully-qualified', emoji='👵🏾', name='old woman: medium-dark skin tone', group='People & Body', sub_group='person'),
    '👵🏿': EmojiEntry(codepoint='1F475 1F3FF', status='fully-qualified', emoji='👵🏿', name='old woman: dark skin tone', group='People & Body', sub_group='person'),
    '🙍': EmojiEntry(codepoint='1F64D', status='fully-qualified', emoji='🙍', name='person frowning', group='People & Body', sub_group='person-gesture'),
    '🙍🏻': EmojiEntry(codepoint='1F64D 1F3FB', status='fully-qualified', emoji='🙍🏻', name='person frowning: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏼': EmojiEntry(codepoint='1F64D 1F3FC', status='fully-qualified', emoji='🙍🏼', name='person frowning: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏽': EmojiEntry(codepoint='1F64D 1F3FD', status='fully-qualified', emoji='🙍🏽', name='person frowning: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏾': EmojiEntry(codepoint='1F64D 1F3FE', status='fully-qualified', emoji='🙍🏾', name='person frowning: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏿': EmojiEntry(codepoint='1F64D 1F3FF', status='fully-qualified', emoji='🙍🏿', name='person frowning: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍‍♂️': EmojiEntry(codepoint='1F64D 200D 2642 FE0F', status='fully-qualified', emoji='🙍\u200d♂️', name='man frowning', group='People & Body', sub_group='person-gesture'),
    '🙍‍♂': EmojiEntry(codepoint='1F64D 200D 2642', status='minimally-qualified', emoji='🙍\u200d♂', name='man frowning', group='People & Body', sub_group='person-gesture'),
    '🙍🏻‍♂️': EmojiEntry(codepoint='1F64D 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🙍🏻\u200d♂️', name='man frowning: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏻‍♂': EmojiEntry(codepoint='1F64D 1F3FB 200D 2642', status='minimally-qualified', emoji='🙍🏻\u200d♂', name='man frowning: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏼‍♂️': EmojiEntry(codepoint='1F64D 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🙍🏼\u200d♂️', name='man frowning: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏼‍♂': EmojiEntry(codepoint='1F64D 1F3FC 200D 2642', status='minimally-qualified', emoji='🙍🏼\u200d♂', name='man frowning: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏽‍♂️': EmojiEntry(codepoint='1F64D 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🙍🏽\u200d♂️', name='man frowning: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏽‍♂': EmojiEntry(codepoint='1F64D 1F3FD 200D 2642', status='minimally-qualified', emoji='🙍🏽\u200d♂', name='man frowning: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏾‍♂️': EmojiEntry(codepoint='1F64D 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🙍🏾\u200d♂️', name='man frowning: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏾‍♂': EmojiEntry(codepoint='1F64D 1F3FE 200D 2642', status='minimally-qualified', emoji='🙍🏾\u200d♂', name='man frowning: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏿‍♂️': EmojiEntry(codepoint='1F64D 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🙍🏿\u200d♂️', name='man frowning: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏿‍♂': EmojiEntry(codepoint='1F64D 1F3FF 200D 2642', status='minimally-qualified', emoji='🙍🏿\u200d♂', name='man frowning: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍‍♀️': EmojiEntry(codepoint='1F64D 200D 2640 FE0F', status='fully-qualified', emoji='🙍\u200d♀️', name='woman frowning', group='People & Body', sub_group='person-gesture'),
    '🙍‍♀': EmojiEntry(codepoint='1F64D 200D 2640', status='minimally-qualified', emoji='🙍\u200d♀', name='woman frowning', group='People & Body', sub_group='person-gesture'),
    '🙍🏻‍♀️': EmojiEntry(codepoint='1F64D 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🙍🏻\u200d♀️', name='woman frowning: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏻‍♀': EmojiEntry(codepoint='1F64D 1F3FB 200D 2640', status='minimally-qualified', emoji='🙍🏻\u200d♀', name='woman frowning: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏼‍♀️': EmojiEntry(codepoint='1F64D 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🙍🏼\u200d♀️', name='woman frowning: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏼‍♀': EmojiEntry(codepoint='1F64D 1F3FC 200D 2640', status='minimally-qualified', emoji='🙍🏼\u200d♀', name='woman frowning: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏽‍♀️': EmojiEntry(codepoint='1F64D 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🙍🏽\u200d♀️', name='woman frowning: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏽‍♀': EmojiEntry(codepoint='1F64D 1F3FD 200D 2640', status='minimally-qualified', emoji='🙍🏽\u200d♀', name='woman frowning: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏾‍♀️': EmojiEntry(codepoint='1F64D 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🙍🏾\u200d♀️', name='woman frowning: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏾‍♀': EmojiEntry(codepoint='1F64D 1F3FE 200D 2640', status='minimally-qualified', emoji='🙍🏾\u200d♀', name='woman frowning: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏿‍♀️': EmojiEntry(codepoint='1F64D 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🙍🏿\u200d♀️', name='woman frowning: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙍🏿‍♀': EmojiEntry(codepoint='1F64D 1F3FF 200D 2640', status='minimally-qualified', emoji='🙍🏿\u200d♀', name='woman frowning: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎': EmojiEntry(codepoint='1F64E', status='fully-qualified', emoji='🙎', name='person pouting', group='People & Body', sub_group='person-gesture'),
    '🙎🏻': EmojiEntry(codepoint='1F64E 1F3FB', status='fully-qualified', emoji='🙎🏻', name='person pouting: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏼': EmojiEntry(codepoint='1F64E 1F3FC', status='fully-qualified', emoji='🙎🏼', name='person pouting: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏽': EmojiEntry(codepoint='1F64E 1F3FD', status='fully-qualified', emoji='🙎🏽', name='person pouting: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏾': EmojiEntry(codepoint='1F64E 1F3FE', status='fully-qualified', emoji='🙎🏾', name='person pouting: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏿': EmojiEntry(codepoint='1F64E 1F3FF', status='fully-qualified', emoji='🙎🏿', name='person pouting: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎‍♂️': EmojiEntry(codepoint='1F64E 200D 2642 FE0F', status='fully-qualified', emoji='🙎\u200d♂️', name='man pouting', group='People & Body', sub_group='person-gesture'),
    '🙎‍♂': EmojiEntry(codepoint='1F64E 200D 2642', status='minimally-qualified', emoji='🙎\u200d♂', name='man pouting', group='People & Body', sub_group='person-gesture'),
    '🙎🏻‍♂️': EmojiEntry(codepoint='1F64E 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🙎🏻\u200d♂️', name='man pouting: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏻‍♂': EmojiEntry(codepoint='1F64E 1F3FB 200D 2642', status='minimally-qualified', emoji='🙎🏻\u200d♂', name='man pouting: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏼‍♂️': EmojiEntry(codepoint='1F64E 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🙎🏼\u200d♂️', name='man pouting: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏼‍♂': EmojiEntry(codepoint='1F64E 1F3FC 200D 2642', status='minimally-qualified', emoji='🙎🏼\u200d♂', name='man pouting: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏽‍♂️': EmojiEntry(codepoint='1F64E 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🙎🏽\u200d♂️', name='man pouting: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏽‍♂': EmojiEntry(codepoint='1F64E 1F3FD 200D 2642', status='minimally-qualified', emoji='🙎🏽\u200d♂', name='man pouting: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏾‍♂️': EmojiEntry(codepoint='1F64E 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🙎🏾\u200d♂️', name='man pouting: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏾‍♂': EmojiEntry(codepoint='1F64E 1F3FE 200D 2642', status='minimally-qualified', emoji='🙎🏾\u200d♂', name='man pouting: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏿‍♂️': EmojiEntry(codepoint='1F64E 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🙎🏿\u200d♂️', name='man pouting: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏿‍♂': EmojiEntry(codepoint='1F64E 1F3FF 200D 2642', status='minimally-qualified', emoji='🙎🏿\u200d♂', name='man pouting: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎‍♀️': EmojiEntry(codepoint='1F64E 200D 2640 FE0F', status='fully-qualified', emoji='🙎\u200d♀️', name='woman pouting', group='People & Body', sub_group='person-gesture'),
    '🙎‍♀': EmojiEntry(codepoint='1F64E 200D 2640', status='minimally-qualified', emoji='🙎\u200d♀', name='woman pouting', group='People & Body', sub_group='person-gesture'),
    '🙎🏻‍♀️': EmojiEntry(codepoint='1F64E 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🙎🏻\u200d♀️', name='woman pouting: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏻‍♀': EmojiEntry(codepoint='1F64E 1F3FB 200D 2640', status='minimally-qualified', emoji='🙎🏻\u200d♀', name='woman pouting: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏼‍♀️': EmojiEntry(codepoint='1F64E 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🙎🏼\u200d♀️', name='woman pouting: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏼‍♀': EmojiEntry(codepoint='1F64E 1F3FC 200D 2640', status='minimally-qualified', emoji='🙎🏼\u200d♀', name='woman pouting: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏽‍♀️': EmojiEntry(codepoint='1F64E 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🙎🏽\u200d♀️', name='woman pouting: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏽‍♀': EmojiEntry(codepoint='1F64E 1F3FD 200D 2640', status='minimally-qualified', emoji='🙎🏽\u200d♀', name='woman pouting: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏾‍♀️': EmojiEntry(codepoint='1F64E 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🙎🏾\u200d♀️', name='woman pouting: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏾‍♀': EmojiEntry(codepoint='1F64E 1F3FE 200D 2640', status='minimally-qualified', emoji='🙎🏾\u200d♀', name='woman pouting: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏿‍♀️': EmojiEntry(codepoint='1F64E 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🙎🏿\u200d♀️', name='woman pouting: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙎🏿‍♀': EmojiEntry(codepoint='1F64E 1F3FF 200D 2640', status='minimally-qualified', emoji='🙎🏿\u200d♀', name='woman pouting: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅': EmojiEntry(codepoint='1F645', status='fully-qualified', emoji='🙅', name='person gesturing NO', group='People & Body', sub_group='person-gesture'),
    '🙅🏻': EmojiEntry(codepoint='1F645 1F3FB', status='fully-qualified', emoji='🙅🏻', name='person gesturing NO: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏼': EmojiEntry(codepoint='1F645 1F3FC', status='fully-qualified', emoji='🙅🏼', name='person gesturing NO: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏽': EmojiEntry(codepoint='1F645 1F3FD', status='fully-qualified', emoji='🙅🏽', name='person gesturing NO: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏾': EmojiEntry(codepoint='1F645 1F3FE', status='fully-qualified', emoji='🙅🏾', name='person gesturing NO: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏿': EmojiEntry(codepoint='1F645 1F3FF', status='fully-qualified', emoji='🙅🏿', name='person gesturing NO: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅‍♂️': EmojiEntry(codepoint='1F645 200D 2642 FE0F', status='fully-qualified', emoji='🙅\u200d♂️', name='man gesturing NO', group='People & Body', sub_group='person-gesture'),
    '🙅‍♂': EmojiEntry(codepoint='1F645 200D 2642', status='minimally-qualified', emoji='🙅\u200d♂', name='man gesturing NO', group='People & Body', sub_group='person-gesture'),
    '🙅🏻‍♂️': EmojiEntry(codepoint='1F645 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🙅🏻\u200d♂️', name='man gesturing NO: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏻‍♂': EmojiEntry(codepoint='1F645 1F3FB 200D 2642', status='minimally-qualified', emoji='🙅🏻\u200d♂', name='man gesturing NO: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏼‍♂️': EmojiEntry(codepoint='1F645 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🙅🏼\u200d♂️', name='man gesturing NO: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏼‍♂': EmojiEntry(codepoint='1F645 1F3FC 200D 2642', status='minimally-qualified', emoji='🙅🏼\u200d♂', name='man gesturing NO: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏽‍♂️': EmojiEntry(codepoint='1F645 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🙅🏽\u200d♂️', name='man gesturing NO: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏽‍♂': EmojiEntry(codepoint='1F645 1F3FD 200D 2642', status='minimally-qualified', emoji='🙅🏽\u200d♂', name='man gesturing NO: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏾‍♂️': EmojiEntry(codepoint='1F645 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🙅🏾\u200d♂️', name='man gesturing NO: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏾‍♂': EmojiEntry(codepoint='1F645 1F3FE 200D 2642', status='minimally-qualified', emoji='🙅🏾\u200d♂', name='man gesturing NO: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏿‍♂️': EmojiEntry(codepoint='1F645 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🙅🏿\u200d♂️', name='man gesturing NO: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏿‍♂': EmojiEntry(codepoint='1F645 1F3FF 200D 2642', status='minimally-qualified', emoji='🙅🏿\u200d♂', name='man gesturing NO: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅‍♀️': EmojiEntry(codepoint='1F645 200D 2640 FE0F', status='fully-qualified', emoji='🙅\u200d♀️', name='woman gesturing NO', group='People & Body', sub_group='person-gesture'),
    '🙅‍♀': EmojiEntry(codepoint='1F645 200D 2640', status='minimally-qualified', emoji='🙅\u200d♀', name='woman gesturing NO', group='People & Body', sub_group='person-gesture'),
    '🙅🏻‍♀️': EmojiEntry(codepoint='1F645 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🙅🏻\u200d♀️', name='woman gesturing NO: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏻‍♀': EmojiEntry(codepoint='1F645 1F3FB 200D 2640', status='minimally-qualified', emoji='🙅🏻\u200d♀', name='woman gesturing NO: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏼‍♀️': EmojiEntry(codepoint='1F645 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🙅🏼\u200d♀️', name='woman gesturing NO: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏼‍♀': EmojiEntry(codepoint='1F645 1F3FC 200D 2640', status='minimally-qualified', emoji='🙅🏼\u200d♀', name='woman gesturing NO: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏽‍♀️': EmojiEntry(codepoint='1F645 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🙅🏽\u200d♀️', name='woman gesturing NO: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏽‍♀': EmojiEntry(codepoint='1F645 1F3FD 200D 2640', status='minimally-qualified', emoji='🙅🏽\u200d♀', name='woman gesturing NO: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏾‍♀️': EmojiEntry(codepoint='1F645 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🙅🏾\u200d♀️', name='woman gesturing NO: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏾‍♀': EmojiEntry(codepoint='1F645 1F3FE 200D 2640', status='minimally-qualified', emoji='🙅🏾\u200d♀', name='woman gesturing NO: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏿‍♀️': EmojiEntry(codepoint='1F645 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🙅🏿\u200d♀️', name='woman gesturing NO: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙅🏿‍♀': EmojiEntry(codepoint='1F645 1F3FF 200D 2640', status='minimally-qualified', emoji='🙅🏿\u200d♀', name='woman gesturing NO: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆': EmojiEntry(codepoint='1F646', status='fully-qualified', emoji='🙆', name='person gesturing OK', group='People & Body', sub_group='person-gesture'),
    '🙆🏻': EmojiEntry(codepoint='1F646 1F3FB', status='fully-qualified', emoji='🙆🏻', name='person gesturing OK: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏼': EmojiEntry(codepoint='1F646 1F3FC', status='fully-qualified', emoji='🙆🏼', name='person gesturing OK: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏽': EmojiEntry(codepoint='1F646 1F3FD', status='fully-qualified', emoji='🙆🏽', name='person gesturing OK: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏾': EmojiEntry(codepoint='1F646 1F3FE', status='fully-qualified', emoji='🙆🏾', name='person gesturing OK: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏿': EmojiEntry(codepoint='1F646 1F3FF', status='fully-qualified', emoji='🙆🏿', name='person gesturing OK: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆‍♂️': EmojiEntry(codepoint='1F646 200D 2642 FE0F', status='fully-qualified', emoji='🙆\u200d♂️', name='man gesturing OK', group='People & Body', sub_group='person-gesture'),
    '🙆‍♂': EmojiEntry(codepoint='1F646 200D 2642', status='minimally-qualified', emoji='🙆\u200d♂', name='man gesturing OK', group='People & Body', sub_group='person-gesture'),
    '🙆🏻‍♂️': EmojiEntry(codepoint='1F646 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🙆🏻\u200d♂️', name='man gesturing OK: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏻‍♂': EmojiEntry(codepoint='1F646 1F3FB 200D 2642', status='minimally-qualified', emoji='🙆🏻\u200d♂', name='man gesturing OK: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏼‍♂️': EmojiEntry(codepoint='1F646 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🙆🏼\u200d♂️', name='man gesturing OK: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏼‍♂': EmojiEntry(codepoint='1F646 1F3FC 200D 2642', status='minimally-qualified', emoji='🙆🏼\u200d♂', name='man gesturing OK: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏽‍♂️': EmojiEntry(codepoint='1F646 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🙆🏽\u200d♂️', name='man gesturing OK: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏽‍♂': EmojiEntry(codepoint='1F646 1F3FD 200D 2642', status='minimally-qualified', emoji='🙆🏽\u200d♂', name='man gesturing OK: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏾‍♂️': EmojiEntry(codepoint='1F646 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🙆🏾\u200d♂️', name='man gesturing OK: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏾‍♂': EmojiEntry(codepoint='1F646 1F3FE 200D 2642', status='minimally-qualified', emoji='🙆🏾\u200d♂', name='man gesturing OK: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏿‍♂️': EmojiEntry(codepoint='1F646 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🙆🏿\u200d♂️', name='man gesturing OK: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏿‍♂': EmojiEntry(codepoint='1F646 1F3FF 200D 2642', status='minimally-qualified', emoji='🙆🏿\u200d♂', name='man gesturing OK: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆‍♀️': EmojiEntry(codepoint='1F646 200D 2640 FE0F', status='fully-qualified', emoji='🙆\u200d♀️', name='woman gesturing OK', group='People & Body', sub_group='person-gesture'),
    '🙆‍♀': EmojiEntry(codepoint='1F646 200D 2640', status='minimally-qualified', emoji='🙆\u200d♀', name='woman gesturing OK', group='People & Body', sub_group='person-gesture'),
    '🙆🏻‍♀️': EmojiEntry(codepoint='1F646 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🙆🏻\u200d♀️', name='woman gesturing OK: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏻‍♀': EmojiEntry(codepoint='1F646 1F3FB 200D 2640', status='minimally-qualified', emoji='🙆🏻\u200d♀', name='woman gesturing OK: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏼‍♀️': EmojiEntry(codepoint='1F646 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🙆🏼\u200d♀️', name='woman gesturing OK: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏼‍♀': EmojiEntry(codepoint='1F646 1F3FC 200D 2640', status='minimally-qualified', emoji='🙆🏼\u200d♀', name='woman gesturing OK: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏽‍♀️': EmojiEntry(codepoint='1F646 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🙆🏽\u200d♀️', name='woman gesturing OK: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏽‍♀': EmojiEntry(codepoint='1F646 1F3FD 200D 2640', status='minimally-qualified', emoji='🙆🏽\u200d♀', name='woman gesturing OK: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏾‍♀️': EmojiEntry(codepoint='1F646 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🙆🏾\u200d♀️', name='woman gesturing OK: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏾‍♀': EmojiEntry(codepoint='1F646 1F3FE 200D 2640', status='minimally-qualified', emoji='🙆🏾\u200d♀', name='woman gesturing OK: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏿‍♀️': EmojiEntry(codepoint='1F646 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🙆🏿\u200d♀️', name='woman gesturing OK: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙆🏿‍♀': EmojiEntry(codepoint='1F646 1F3FF 200D 2640', status='minimally-qualified', emoji='🙆🏿\u200d♀', name='woman gesturing OK: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '💁': EmojiEntry(codepoint='1F481', status='fully-qualified', emoji='💁', name='person tipping hand', group='People & Body', sub_group='person-gesture'),
    '💁🏻': EmojiEntry(codepoint='1F481 1F3FB', status='fully-qualified', emoji='💁🏻', name='person tipping hand: light skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏼': EmojiEntry(codepoint='1F481 1F3FC', status='fully-qualified', emoji='💁🏼', name='person tipping hand: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏽': EmojiEntry(codepoint='1F481 1F3FD', status='fully-qualified', emoji='💁🏽', name='person tipping hand: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏾': EmojiEntry(codepoint='1F481 1F3FE', status='fully-qualified', emoji='💁🏾', name='person tipping hand: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏿': EmojiEntry(codepoint='1F481 1F3FF', status='fully-qualified', emoji='💁🏿', name='person tipping hand: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '💁‍♂️': EmojiEntry(codepoint='1F481 200D 2642 FE0F', status='fully-qualified', emoji='💁\u200d♂️', name='man tipping hand', group='People & Body', sub_group='person-gesture'),
    '💁‍♂': EmojiEntry(codepoint='1F481 200D 2642', status='minimally-qualified', emoji='💁\u200d♂', name='man tipping hand', group='People & Body', sub_group='person-gesture'),
    '💁🏻‍♂️': EmojiEntry(codepoint='1F481 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='💁🏻\u200d♂️', name='man tipping hand: light skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏻‍♂': EmojiEntry(codepoint='1F481 1F3FB 200D 2642', status='minimally-qualified', emoji='💁🏻\u200d♂', name='man tipping hand: light skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏼‍♂️': EmojiEntry(codepoint='1F481 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='💁🏼\u200d♂️', name='man tipping hand: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏼‍♂': EmojiEntry(codepoint='1F481 1F3FC 200D 2642', status='minimally-qualified', emoji='💁🏼\u200d♂', name='man tipping hand: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏽‍♂️': EmojiEntry(codepoint='1F481 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='💁🏽\u200d♂️', name='man tipping hand: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏽‍♂': EmojiEntry(codepoint='1F481 1F3FD 200D 2642', status='minimally-qualified', emoji='💁🏽\u200d♂', name='man tipping hand: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏾‍♂️': EmojiEntry(codepoint='1F481 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='💁🏾\u200d♂️', name='man tipping hand: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏾‍♂': EmojiEntry(codepoint='1F481 1F3FE 200D 2642', status='minimally-qualified', emoji='💁🏾\u200d♂', name='man tipping hand: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏿‍♂️': EmojiEntry(codepoint='1F481 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='💁🏿\u200d♂️', name='man tipping hand: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏿‍♂': EmojiEntry(codepoint='1F481 1F3FF 200D 2642', status='minimally-qualified', emoji='💁🏿\u200d♂', name='man tipping hand: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '💁‍♀️': EmojiEntry(codepoint='1F481 200D 2640 FE0F', status='fully-qualified', emoji='💁\u200d♀️', name='woman tipping hand', group='People & Body', sub_group='person-gesture'),
    '💁‍♀': EmojiEntry(codepoint='1F481 200D 2640', status='minimally-qualified', emoji='💁\u200d♀', name='woman tipping hand', group='People & Body', sub_group='person-gesture'),
    '💁🏻‍♀️': EmojiEntry(codepoint='1F481 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='💁🏻\u200d♀️', name='woman tipping hand: light skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏻‍♀': EmojiEntry(codepoint='1F481 1F3FB 200D 2640', status='minimally-qualified', emoji='💁🏻\u200d♀', name='woman tipping hand: light skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏼‍♀️': EmojiEntry(codepoint='1F481 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='💁🏼\u200d♀️', name='woman tipping hand: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏼‍♀': EmojiEntry(codepoint='1F481 1F3FC 200D 2640', status='minimally-qualified', emoji='💁🏼\u200d♀', name='woman tipping hand: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏽‍♀️': EmojiEntry(codepoint='1F481 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='💁🏽\u200d♀️', name='woman tipping hand: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏽‍♀': EmojiEntry(codepoint='1F481 1F3FD 200D 2640', status='minimally-qualified', emoji='💁🏽\u200d♀', name='woman tipping hand: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏾‍♀️': EmojiEntry(codepoint='1F481 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='💁🏾\u200d♀️', name='woman tipping hand: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏾‍♀': EmojiEntry(codepoint='1F481 1F3FE 200D 2640', status='minimally-qualified', emoji='💁🏾\u200d♀', name='woman tipping hand: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏿‍♀️': EmojiEntry(codepoint='1F481 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='💁🏿\u200d♀️', name='woman tipping hand: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '💁🏿‍♀': EmojiEntry(codepoint='1F481 1F3FF 200D 2640', status='minimally-qualified', emoji='💁🏿\u200d♀', name='woman tipping hand: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋': EmojiEntry(codepoint='1F64B', status='fully-qualified', emoji='🙋', name='person raising hand', group='People & Body', sub_group='person-gesture'),
    '🙋🏻': EmojiEntry(codepoint='1F64B 1F3FB', status='fully-qualified', emoji='🙋🏻', name='person raising hand: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏼': EmojiEntry(codepoint='1F64B 1F3FC', status='fully-qualified', emoji='🙋🏼', name='person raising hand: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏽': EmojiEntry(codepoint='1F64B 1F3FD', status='fully-qualified', emoji='🙋🏽', name='person raising hand: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏾': EmojiEntry(codepoint='1F64B 1F3FE', status='fully-qualified', emoji='🙋🏾', name='person raising hand: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏿': EmojiEntry(codepoint='1F64B 1F3FF', status='fully-qualified', emoji='🙋🏿', name='person raising hand: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋‍♂️': EmojiEntry(codepoint='1F64B 200D 2642 FE0F', status='fully-qualified', emoji='🙋\u200d♂️', name='man raising hand', group='People & Body', sub_group='person-gesture'),
    '🙋‍♂': EmojiEntry(codepoint='1F64B 200D 2642', status='minimally-qualified', emoji='🙋\u200d♂', name='man raising hand', group='People & Body', sub_group='person-gesture'),
    '🙋🏻‍♂️': EmojiEntry(codepoint='1F64B 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🙋🏻\u200d♂️', name='man raising hand: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏻‍♂': EmojiEntry(codepoint='1F64B 1F3FB 200D 2642', status='minimally-qualified', emoji='🙋🏻\u200d♂', name='man raising hand: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏼‍♂️': EmojiEntry(codepoint='1F64B 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🙋🏼\u200d♂️', name='man raising hand: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏼‍♂': EmojiEntry(codepoint='1F64B 1F3FC 200D 2642', status='minimally-qualified', emoji='🙋🏼\u200d♂', name='man raising hand: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏽‍♂️': EmojiEntry(codepoint='1F64B 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🙋🏽\u200d♂️', name='man raising hand: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏽‍♂': EmojiEntry(codepoint='1F64B 1F3FD 200D 2642', status='minimally-qualified', emoji='🙋🏽\u200d♂', name='man raising hand: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏾‍♂️': EmojiEntry(codepoint='1F64B 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🙋🏾\u200d♂️', name='man raising hand: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏾‍♂': EmojiEntry(codepoint='1F64B 1F3FE 200D 2642', status='minimally-qualified', emoji='🙋🏾\u200d♂', name='man raising hand: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏿‍♂️': EmojiEntry(codepoint='1F64B 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🙋🏿\u200d♂️', name='man raising hand: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏿‍♂': EmojiEntry(codepoint='1F64B 1F3FF 200D 2642', status='minimally-qualified', emoji='🙋🏿\u200d♂', name='man raising hand: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋‍♀️': EmojiEntry(codepoint='1F64B 200D 2640 FE0F', status='fully-qualified', emoji='🙋\u200d♀️', name='woman raising hand', group='People & Body', sub_group='person-gesture'),
    '🙋‍♀': EmojiEntry(codepoint='1F64B 200D 2640', status='minimally-qualified', emoji='🙋\u200d♀', name='woman raising hand', group='People & Body', sub_group='person-gesture'),
    '🙋🏻‍♀️': EmojiEntry(codepoint='1F64B 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🙋🏻\u200d♀️', name='woman raising hand: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏻‍♀': EmojiEntry(codepoint='1F64B 1F3FB 200D 2640', status='minimally-qualified', emoji='🙋🏻\u200d♀', name='woman raising hand: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏼‍♀️': EmojiEntry(codepoint='1F64B 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🙋🏼\u200d♀️', name='woman raising hand: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏼‍♀': EmojiEntry(codepoint='1F64B 1F3FC 200D 2640', status='minimally-qualified', emoji='🙋🏼\u200d♀', name='woman raising hand: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏽‍♀️': EmojiEntry(codepoint='1F64B 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🙋🏽\u200d♀️', name='woman raising hand: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏽‍♀': EmojiEntry(codepoint='1F64B 1F3FD 200D 2640', status='minimally-qualified', emoji='🙋🏽\u200d♀', name='woman raising hand: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏾‍♀️': EmojiEntry(codepoint='1F64B 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🙋🏾\u200d♀️', name='woman raising hand: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏾‍♀': EmojiEntry(codepoint='1F64B 1F3FE 200D 2640', status='minimally-qualified', emoji='🙋🏾\u200d♀', name='woman raising hand: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏿‍♀️': EmojiEntry(codepoint='1F64B 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🙋🏿\u200d♀️', name='woman raising hand: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙋🏿‍♀': EmojiEntry(codepoint='1F64B 1F3FF 200D 2640', status='minimally-qualified', emoji='🙋🏿\u200d♀', name='woman raising hand: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏': EmojiEntry(codepoint='1F9CF', status='fully-qualified', emoji='🧏', name='deaf person', group='People & Body', sub_group='person-gesture'),
    '🧏🏻': EmojiEntry(codepoint='1F9CF 1F3FB', status='fully-qualified', emoji='🧏🏻', name='deaf person: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏼': EmojiEntry(codepoint='1F9CF 1F3FC', status='fully-qualified', emoji='🧏🏼', name='deaf person: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏽': EmojiEntry(codepoint='1F9CF 1F3FD', status='fully-qualified', emoji='🧏🏽', name='deaf person: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏾': EmojiEntry(codepoint='1F9CF 1F3FE', status='fully-qualified', emoji='🧏🏾', name='deaf person: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏿': EmojiEntry(codepoint='1F9CF 1F3FF', status='fully-qualified', emoji='🧏🏿', name='deaf person: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏‍♂️': EmojiEntry(codepoint='1F9CF 200D 2642 FE0F', status='fully-qualified', emoji='🧏\u200d♂️', name='deaf man', group='People & Body', sub_group='person-gesture'),
    '🧏‍♂': EmojiEntry(codepoint='1F9CF 200D 2642', status='minimally-qualified', emoji='🧏\u200d♂', name='deaf man', group='People & Body', sub_group='person-gesture'),
    '🧏🏻‍♂️': EmojiEntry(codepoint='1F9CF 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🧏🏻\u200d♂️', name='deaf man: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏻‍♂': EmojiEntry(codepoint='1F9CF 1F3FB 200D 2642', status='minimally-qualified', emoji='🧏🏻\u200d♂', name='deaf man: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏼‍♂️': EmojiEntry(codepoint='1F9CF 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🧏🏼\u200d♂️', name='deaf man: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏼‍♂': EmojiEntry(codepoint='1F9CF 1F3FC 200D 2642', status='minimally-qualified', emoji='🧏🏼\u200d♂', name='deaf man: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏽‍♂️': EmojiEntry(codepoint='1F9CF 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🧏🏽\u200d♂️', name='deaf man: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏽‍♂': EmojiEntry(codepoint='1F9CF 1F3FD 200D 2642', status='minimally-qualified', emoji='🧏🏽\u200d♂', name='deaf man: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏾‍♂️': EmojiEntry(codepoint='1F9CF 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🧏🏾\u200d♂️', name='deaf man: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏾‍♂': EmojiEntry(codepoint='1F9CF 1F3FE 200D 2642', status='minimally-qualified', emoji='🧏🏾\u200d♂', name='deaf man: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏿‍♂️': EmojiEntry(codepoint='1F9CF 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🧏🏿\u200d♂️', name='deaf man: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏿‍♂': EmojiEntry(codepoint='1F9CF 1F3FF 200D 2642', status='minimally-qualified', emoji='🧏🏿\u200d♂', name='deaf man: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏‍♀️': EmojiEntry(codepoint='1F9CF 200D 2640 FE0F', status='fully-qualified', emoji='🧏\u200d♀️', name='deaf woman', group='People & Body', sub_group='person-gesture'),
    '🧏‍♀': EmojiEntry(codepoint='1F9CF 200D 2640', status='minimally-qualified', emoji='🧏\u200d♀', name='deaf woman', group='People & Body', sub_group='person-gesture'),
    '🧏🏻‍♀️': EmojiEntry(codepoint='1F9CF 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🧏🏻\u200d♀️', name='deaf woman: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏻‍♀': EmojiEntry(codepoint='1F9CF 1F3FB 200D 2640', status='minimally-qualified', emoji='🧏🏻\u200d♀', name='deaf woman: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏼‍♀️': EmojiEntry(codepoint='1F9CF 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🧏🏼\u200d♀️', name='deaf woman: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏼‍♀': EmojiEntry(codepoint='1F9CF 1F3FC 200D 2640', status='minimally-qualified', emoji='🧏🏼\u200d♀', name='deaf woman: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏽‍♀️': EmojiEntry(codepoint='1F9CF 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🧏🏽\u200d♀️', name='deaf woman: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏽‍♀': EmojiEntry(codepoint='1F9CF 1F3FD 200D 2640', status='minimally-qualified', emoji='🧏🏽\u200d♀', name='deaf woman: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏾‍♀️': EmojiEntry(codepoint='1F9CF 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🧏🏾\u200d♀️', name='deaf woman: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏾‍♀': EmojiEntry(codepoint='1F9CF 1F3FE 200D 2640', status='minimally-qualified', emoji='🧏🏾\u200d♀', name='deaf woman: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏿‍♀️': EmojiEntry(codepoint='1F9CF 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🧏🏿\u200d♀️', name='deaf woman: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🧏🏿‍♀': EmojiEntry(codepoint='1F9CF 1F3FF 200D 2640', status='minimally-qualified', emoji='🧏🏿\u200d♀', name='deaf woman: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇': EmojiEntry(codepoint='1F647', status='fully-qualified', emoji='🙇', name='person bowing', group='People & Body', sub_group='person-gesture'),
    '🙇🏻': EmojiEntry(codepoint='1F647 1F3FB', status='fully-qualified', emoji='🙇🏻', name='person bowing: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏼': EmojiEntry(codepoint='1F647 1F3FC', status='fully-qualified', emoji='🙇🏼', name='person bowing: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏽': EmojiEntry(codepoint='1F647 1F3FD', status='fully-qualified', emoji='🙇🏽', name='person bowing: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏾': EmojiEntry(codepoint='1F647 1F3FE', status='fully-qualified', emoji='🙇🏾', name='person bowing: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏿': EmojiEntry(codepoint='1F647 1F3FF', status='fully-qualified', emoji='🙇🏿', name='person bowing: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇‍♂️': EmojiEntry(codepoint='1F647 200D 2642 FE0F', status='fully-qualified', emoji='🙇\u200d♂️', name='man bowing', group='People & Body', sub_group='person-gesture'),
    '🙇‍♂': EmojiEntry(codepoint='1F647 200D 2642', status='minimally-qualified', emoji='🙇\u200d♂', name='man bowing', group='People & Body', sub_group='person-gesture'),
    '🙇🏻‍♂️': EmojiEntry(codepoint='1F647 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🙇🏻\u200d♂️', name='man bowing: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏻‍♂': EmojiEntry(codepoint='1F647 1F3FB 200D 2642', status='minimally-qualified', emoji='🙇🏻\u200d♂', name='man bowing: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏼‍♂️': EmojiEntry(codepoint='1F647 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🙇🏼\u200d♂️', name='man bowing: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏼‍♂': EmojiEntry(codepoint='1F647 1F3FC 200D 2642', status='minimally-qualified', emoji='🙇🏼\u200d♂', name='man bowing: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏽‍♂️': EmojiEntry(codepoint='1F647 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🙇🏽\u200d♂️', name='man bowing: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏽‍♂': EmojiEntry(codepoint='1F647 1F3FD 200D 2642', status='minimally-qualified', emoji='🙇🏽\u200d♂', name='man bowing: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏾‍♂️': EmojiEntry(codepoint='1F647 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🙇🏾\u200d♂️', name='man bowing: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏾‍♂': EmojiEntry(codepoint='1F647 1F3FE 200D 2642', status='minimally-qualified', emoji='🙇🏾\u200d♂', name='man bowing: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏿‍♂️': EmojiEntry(codepoint='1F647 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🙇🏿\u200d♂️', name='man bowing: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏿‍♂': EmojiEntry(codepoint='1F647 1F3FF 200D 2642', status='minimally-qualified', emoji='🙇🏿\u200d♂', name='man bowing: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇‍♀️': EmojiEntry(codepoint='1F647 200D 2640 FE0F', status='fully-qualified', emoji='🙇\u200d♀️', name='woman bowing', group='People & Body', sub_group='person-gesture'),
    '🙇‍♀': EmojiEntry(codepoint='1F647 200D 2640', status='minimally-qualified', emoji='🙇\u200d♀', name='woman bowing', group='People & Body', sub_group='person-gesture'),
    '🙇🏻‍♀️': EmojiEntry(codepoint='1F647 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🙇🏻\u200d♀️', name='woman bowing: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏻‍♀': EmojiEntry(codepoint='1F647 1F3FB 200D 2640', status='minimally-qualified', emoji='🙇🏻\u200d♀', name='woman bowing: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏼‍♀️': EmojiEntry(codepoint='1F647 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🙇🏼\u200d♀️', name='woman bowing: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏼‍♀': EmojiEntry(codepoint='1F647 1F3FC 200D 2640', status='minimally-qualified', emoji='🙇🏼\u200d♀', name='woman bowing: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏽‍♀️': EmojiEntry(codepoint='1F647 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🙇🏽\u200d♀️', name='woman bowing: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏽‍♀': EmojiEntry(codepoint='1F647 1F3FD 200D 2640', status='minimally-qualified', emoji='🙇🏽\u200d♀', name='woman bowing: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏾‍♀️': EmojiEntry(codepoint='1F647 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🙇🏾\u200d♀️', name='woman bowing: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏾‍♀': EmojiEntry(codepoint='1F647 1F3FE 200D 2640', status='minimally-qualified', emoji='🙇🏾\u200d♀', name='woman bowing: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏿‍♀️': EmojiEntry(codepoint='1F647 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🙇🏿\u200d♀️', name='woman bowing: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🙇🏿‍♀': EmojiEntry(codepoint='1F647 1F3FF 200D 2640', status='minimally-qualified', emoji='🙇🏿\u200d♀', name='woman bowing: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦': EmojiEntry(codepoint='1F926', status='fully-qualified', emoji='🤦', name='person facepalming', group='People & Body', sub_group='person-gesture'),
    '🤦🏻': EmojiEntry(codepoint='1F926 1F3FB', status='fully-qualified', emoji='🤦🏻', name='person facepalming: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏼': EmojiEntry(codepoint='1F926 1F3FC', status='fully-qualified', emoji='🤦🏼', name='person facepalming: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏽': EmojiEntry(codepoint='1F926 1F3FD', status='fully-qualified', emoji='🤦🏽', name='person facepalming: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏾': EmojiEntry(codepoint='1F926 1F3FE', status='fully-qualified', emoji='🤦🏾', name='person facepalming: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏿': EmojiEntry(codepoint='1F926 1F3FF', status='fully-qualified', emoji='🤦🏿', name='person facepalming: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦‍♂️': EmojiEntry(codepoint='1F926 200D 2642 FE0F', status='fully-qualified', emoji='🤦\u200d♂️', name='man facepalming', group='People & Body', sub_group='person-gesture'),
    '🤦‍♂': EmojiEntry(codepoint='1F926 200D 2642', status='minimally-qualified', emoji='🤦\u200d♂', name='man facepalming', group='People & Body', sub_group='person-gesture'),
    '🤦🏻‍♂️': EmojiEntry(codepoint='1F926 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🤦🏻\u200d♂️', name='man facepalming: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏻‍♂': EmojiEntry(codepoint='1F926 1F3FB 200D 2642', status='minimally-qualified', emoji='🤦🏻\u200d♂', name='man facepalming: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏼‍♂️': EmojiEntry(codepoint='1F926 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🤦🏼\u200d♂️', name='man facepalming: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏼‍♂': EmojiEntry(codepoint='1F926 1F3FC 200D 2642', status='minimally-qualified', emoji='🤦🏼\u200d♂', name='man facepalming: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏽‍♂️': EmojiEntry(codepoint='1F926 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🤦🏽\u200d♂️', name='man facepalming: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏽‍♂': EmojiEntry(codepoint='1F926 1F3FD 200D 2642', status='minimally-qualified', emoji='🤦🏽\u200d♂', name='man facepalming: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏾‍♂️': EmojiEntry(codepoint='1F926 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🤦🏾\u200d♂️', name='man facepalming: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏾‍♂': EmojiEntry(codepoint='1F926 1F3FE 200D 2642', status='minimally-qualified', emoji='🤦🏾\u200d♂', name='man facepalming: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏿‍♂️': EmojiEntry(codepoint='1F926 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🤦🏿\u200d♂️', name='man facepalming: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏿‍♂': EmojiEntry(codepoint='1F926 1F3FF 200D 2642', status='minimally-qualified', emoji='🤦🏿\u200d♂', name='man facepalming: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦‍♀️': EmojiEntry(codepoint='1F926 200D 2640 FE0F', status='fully-qualified', emoji='🤦\u200d♀️', name='woman facepalming', group='People & Body', sub_group='person-gesture'),
    '🤦‍♀': EmojiEntry(codepoint='1F926 200D 2640', status='minimally-qualified', emoji='🤦\u200d♀', name='woman facepalming', group='People & Body', sub_group='person-gesture'),
    '🤦🏻‍♀️': EmojiEntry(codepoint='1F926 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🤦🏻\u200d♀️', name='woman facepalming: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏻‍♀': EmojiEntry(codepoint='1F926 1F3FB 200D 2640', status='minimally-qualified', emoji='🤦🏻\u200d♀', name='woman facepalming: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏼‍♀️': EmojiEntry(codepoint='1F926 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🤦🏼\u200d♀️', name='woman facepalming: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏼‍♀': EmojiEntry(codepoint='1F926 1F3FC 200D 2640', status='minimally-qualified', emoji='🤦🏼\u200d♀', name='woman facepalming: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏽‍♀️': EmojiEntry(codepoint='1F926 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🤦🏽\u200d♀️', name='woman facepalming: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏽‍♀': EmojiEntry(codepoint='1F926 1F3FD 200D 2640', status='minimally-qualified', emoji='🤦🏽\u200d♀', name='woman facepalming: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏾‍♀️': EmojiEntry(codepoint='1F926 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🤦🏾\u200d♀️', name='woman facepalming: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏾‍♀': EmojiEntry(codepoint='1F926 1F3FE 200D 2640', status='minimally-qualified', emoji='🤦🏾\u200d♀', name='woman facepalming: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏿‍♀️': EmojiEntry(codepoint='1F926 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🤦🏿\u200d♀️', name='woman facepalming: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🤦🏿‍♀': EmojiEntry(codepoint='1F926 1F3FF 200D 2640', status='minimally-qualified', emoji='🤦🏿\u200d♀', name='woman facepalming: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷': EmojiEntry(codepoint='1F937', status='fully-qualified', emoji='🤷', name='person shrugging', group='People & Body', sub_group='person-gesture'),
    '🤷🏻': EmojiEntry(codepoint='1F937 1F3FB', status='fully-qualified', emoji='🤷🏻', name='person shrugging: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏼': EmojiEntry(codepoint='1F937 1F3FC', status='fully-qualified', emoji='🤷🏼', name='person shrugging: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏽': EmojiEntry(codepoint='1F937 1F3FD', status='fully-qualified', emoji='🤷🏽', name='person shrugging: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏾': EmojiEntry(codepoint='1F937 1F3FE', status='fully-qualified', emoji='🤷🏾', name='person shrugging: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏿': EmojiEntry(codepoint='1F937 1F3FF', status='fully-qualified', emoji='🤷🏿', name='person shrugging: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷‍♂️': EmojiEntry(codepoint='1F937 200D 2642 FE0F', status='fully-qualified', emoji='🤷\u200d♂️', name='man shrugging', group='People & Body', sub_group='person-gesture'),
    '🤷‍♂': EmojiEntry(codepoint='1F937 200D 2642', status='minimally-qualified', emoji='🤷\u200d♂', name='man shrugging', group='People & Body', sub_group='person-gesture'),
    '🤷🏻‍♂️': EmojiEntry(codepoint='1F937 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🤷🏻\u200d♂️', name='man shrugging: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏻‍♂': EmojiEntry(codepoint='1F937 1F3FB 200D 2642', status='minimally-qualified', emoji='🤷🏻\u200d♂', name='man shrugging: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏼‍♂️': EmojiEntry(codepoint='1F937 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🤷🏼\u200d♂️', name='man shrugging: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏼‍♂': EmojiEntry(codepoint='1F937 1F3FC 200D 2642', status='minimally-qualified', emoji='🤷🏼\u200d♂', name='man shrugging: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏽‍♂️': EmojiEntry(codepoint='1F937 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🤷🏽\u200d♂️', name='man shrugging: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏽‍♂': EmojiEntry(codepoint='1F937 1F3FD 200D 2642', status='minimally-qualified', emoji='🤷🏽\u200d♂', name='man shrugging: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏾‍♂️': EmojiEntry(codepoint='1F937 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🤷🏾\u200d♂️', name='man shrugging: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏾‍♂': EmojiEntry(codepoint='1F937 1F3FE 200D 2642', status='minimally-qualified', emoji='🤷🏾\u200d♂', name='man shrugging: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏿‍♂️': EmojiEntry(codepoint='1F937 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🤷🏿\u200d♂️', name='man shrugging: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏿‍♂': EmojiEntry(codepoint='1F937 1F3FF 200D 2642', status='minimally-qualified', emoji='🤷🏿\u200d♂', name='man shrugging: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷‍♀️': EmojiEntry(codepoint='1F937 200D 2640 FE0F', status='fully-qualified', emoji='🤷\u200d♀️', name='woman shrugging', group='People & Body', sub_group='person-gesture'),
    '🤷‍♀': EmojiEntry(codepoint='1F937 200D 2640', status='minimally-qualified', emoji='🤷\u200d♀', name='woman shrugging', group='People & Body', sub_group='person-gesture'),
    '🤷🏻‍♀️': EmojiEntry(codepoint='1F937 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🤷🏻\u200d♀️', name='woman shrugging: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏻‍♀': EmojiEntry(codepoint='1F937 1F3FB 200D 2640', status='minimally-qualified', emoji='🤷🏻\u200d♀', name='woman shrugging: light skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏼‍♀️': EmojiEntry(codepoint='1F937 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🤷🏼\u200d♀️', name='woman shrugging: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏼‍♀': EmojiEntry(codepoint='1F937 1F3FC 200D 2640', status='minimally-qualified', emoji='🤷🏼\u200d♀', name='woman shrugging: medium-light skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏽‍♀️': EmojiEntry(codepoint='1F937 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🤷🏽\u200d♀️', name='woman shrugging: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏽‍♀': EmojiEntry(codepoint='1F937 1F3FD 200D 2640', status='minimally-qualified', emoji='🤷🏽\u200d♀', name='woman shrugging: medium skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏾‍♀️': EmojiEntry(codepoint='1F937 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🤷🏾\u200d♀️', name='woman shrugging: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏾‍♀': EmojiEntry(codepoint='1F937 1F3FE 200D 2640', status='minimally-qualified', emoji='🤷🏾\u200d♀', name='woman shrugging: medium-dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏿‍♀️': EmojiEntry(codepoint='1F937 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🤷🏿\u200d♀️', name='woman shrugging: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🤷🏿‍♀': EmojiEntry(codepoint='1F937 1F3FF 200D 2640', status='minimally-qualified', emoji='🤷🏿\u200d♀', name='woman shrugging: dark skin tone', group='People & Body', sub_group='person-gesture'),
    '🧑‍⚕️': EmojiEntry(codepoint='1F9D1 200D 2695 FE0F', status='fully-qualified', emoji='🧑\u200d⚕️', name='health worker', group='People & Body', sub_group='person-role'),
    '🧑‍⚕': EmojiEntry(codepoint='1F9D1 200D 2695', status='minimally-qualified', emoji='🧑\u200d⚕', name='health worker', group='People & Body', sub_group='person-role'),
    '🧑🏻‍⚕️': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2695 FE0F', status='fully-qualified', emoji='🧑🏻\u200d⚕️', name='health worker: light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏻‍⚕': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2695', status='minimally-qualified', emoji='🧑🏻\u200d⚕', name='health worker: light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏼‍⚕️': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2695 FE0F', status='fully-qualified', emoji='🧑🏼\u200d⚕️', name='health worker: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏼‍⚕': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2695', status='minimally-qualified', emoji='🧑🏼\u200d⚕', name='health worker: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏽‍⚕️': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2695 FE0F', status='fully-qualified', emoji='🧑🏽\u200d⚕️', name='health worker: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏽‍⚕': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2695', status='minimally-qualified', emoji='🧑🏽\u200d⚕', name='health worker: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏾‍⚕️': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2695 FE0F', status='fully-qualified', emoji='🧑🏾\u200d⚕️', name='health worker: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏾‍⚕': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2695', status='minimally-qualified', emoji='🧑🏾\u200d⚕', name='health worker: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏿‍⚕️': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2695 FE0F', status='fully-qualified', emoji='🧑🏿\u200d⚕️', name='health worker: dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏿‍⚕': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2695', status='minimally-qualified', emoji='🧑🏿\u200d⚕', name='health worker: dark skin tone', group='People & Body', sub_group='person-role'),
    '👨‍⚕️': EmojiEntry(codepoint='1F468 200D 2695 FE0F', status='fully-qualified', emoji='👨\u200d⚕️', name='man health worker', group='People & Body', sub_group='person-role'),
    '👨‍⚕': EmojiEntry(codepoint='1F468 200D 2695', status='minimally-qualified', emoji='👨\u200d⚕', name='man health worker', group='People & Body', sub_group='person-role'),
    '👨🏻‍⚕️': EmojiEntry(codepoint='1F468 1F3FB 200D 2695 FE0F', status='fully-qualified', emoji='👨🏻\u200d⚕️', name='man health worker: light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏻‍⚕': EmojiEntry(codepoint='1F468 1F3FB 200D 2695', status='minimally-qualified', emoji='👨🏻\u200d⚕', name='man health worker: light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏼‍⚕️': EmojiEntry(codepoint='1F468 1F3FC 200D 2695 FE0F', status='fully-qualified', emoji='👨🏼\u200d⚕️', name='man health worker: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏼‍⚕': EmojiEntry(codepoint='1F468 1F3FC 200D 2695', status='minimally-qualified', emoji='👨🏼\u200d⚕', name='man health worker: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏽‍⚕️': EmojiEntry(codepoint='1F468 1F3FD 200D 2695 FE0F', status='fully-qualified', emoji='👨🏽\u200d⚕️', name='man health worker: medium skin tone', group='People & Body', sub_group='person-role'),
    '👨🏽‍⚕': EmojiEntry(codepoint='1F468 1F3FD 200D 2695', status='minimally-qualified', emoji='👨🏽\u200d⚕', name='man health worker: medium skin tone', group='People & Body', sub_group='person-role'),
    '👨🏾‍⚕️': EmojiEntry(codepoint='1F468 1F3FE 200D 2695 FE0F', status='fully-qualified', emoji='👨🏾\u200d⚕️', name='man health worker: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏾‍⚕': EmojiEntry(codepoint='1F468 1F3FE 200D 2695', status='minimally-qualified', emoji='👨🏾\u200d⚕', name='man health worker: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏿‍⚕️': EmojiEntry(codepoint='1F468 1F3FF 200D 2695 FE0F', status='fully-qualified', emoji='👨🏿\u200d⚕️', name='man health worker: dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏿‍⚕': EmojiEntry(codepoint='1F468 1F3FF 200D 2695', status='minimally-qualified', emoji='👨🏿\u200d⚕', name='man health worker: dark skin tone', group='People & Body', sub_group='person-role'),
    '👩‍⚕️': EmojiEntry(codepoint='1F469 200D 2695 FE0F', status='fully-qualified', emoji='👩\u200d⚕️', name='woman health worker', group='People & Body', sub_group='person-role'),
    '👩‍⚕': EmojiEntry(codepoint='1F469 200D 2695', status='minimally-qualified', emoji='👩\u200d⚕', name='woman health worker', group='People & Body', sub_group='person-role'),
    '👩🏻‍⚕️': EmojiEntry(codepoint='1F469 1F3FB 200D 2695 FE0F', status='fully-qualified', emoji='👩🏻\u200d⚕️', name='woman health worker: light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏻‍⚕': EmojiEntry(codepoint='1F469 1F3FB 200D 2695', status='minimally-qualified', emoji='👩🏻\u200d⚕', name='woman health worker: light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏼‍⚕️': EmojiEntry(codepoint='1F469 1F3FC 200D 2695 FE0F', status='fully-qualified', emoji='👩🏼\u200d⚕️', name='woman health worker: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏼‍⚕': EmojiEntry(codepoint='1F469 1F3FC 200D 2695', status='minimally-qualified', emoji='👩🏼\u200d⚕', name='woman health worker: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏽‍⚕️': EmojiEntry(codepoint='1F469 1F3FD 200D 2695 FE0F', status='fully-qualified', emoji='👩🏽\u200d⚕️', name='woman health worker: medium skin tone', group='People & Body', sub_group='person-role'),
    '👩🏽‍⚕': EmojiEntry(codepoint='1F469 1F3FD 200D 2695', status='minimally-qualified', emoji='👩🏽\u200d⚕', name='woman health worker: medium skin tone', group='People & Body', sub_group='person-role'),
    '👩🏾‍⚕️': EmojiEntry(codepoint='1F469 1F3FE 200D 2695 FE0F', status='fully-qualified', emoji='👩🏾\u200d⚕️', name='woman health worker: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏾‍⚕': EmojiEntry(codepoint='1F469 1F3FE 200D 2695', status='minimally-qualified', emoji='👩🏾\u200d⚕', name='woman health worker: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏿‍⚕️': EmojiEntry(codepoint='1F469 1F3FF 200D 2695 FE0F', status='fully-qualified', emoji='👩🏿\u200d⚕️', name='woman health worker: dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏿‍⚕': EmojiEntry(codepoint='1F469 1F3FF 200D 2695', status='minimally-qualified', emoji='👩🏿\u200d⚕', name='woman health worker: dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑‍🎓': EmojiEntry(codepoint='1F9D1 200D 1F393', status='fully-qualified', emoji='🧑\u200d🎓', name='student', group='People & Body', sub_group='person-role'),
    '🧑🏻‍🎓': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F393', status='fully-qualified', emoji='🧑🏻\u200d🎓', name='student: light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏼‍🎓': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F393', status='fully-qualified', emoji='🧑🏼\u200d🎓', name='student: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏽‍🎓': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F393', status='fully-qualified', emoji='🧑🏽\u200d🎓', name='student: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏾‍🎓': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F393', status='fully-qualified', emoji='🧑🏾\u200d🎓', name='student: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏿‍🎓': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F393', status='fully-qualified', emoji='🧑🏿\u200d🎓', name='student: dark skin tone', group='People & Body', sub_group='person-role'),
    '👨‍🎓': EmojiEntry(codepoint='1F468 200D 1F393', status='fully-qualified', emoji='👨\u200d🎓', name='man student', group='People & Body', sub_group='person-role'),
    '👨🏻‍🎓': EmojiEntry(codepoint='1F468 1F3FB 200D 1F393', status='fully-qualified', emoji='👨🏻\u200d🎓', name='man student: light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏼‍🎓': EmojiEntry(codepoint='1F468 1F3FC 200D 1F393', status='fully-qualified', emoji='👨🏼\u200d🎓', name='man student: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏽‍🎓': EmojiEntry(codepoint='1F468 1F3FD 200D 1F393', status='fully-qualified', emoji='👨🏽\u200d🎓', name='man student: medium skin tone', group='People & Body', sub_group='person-role'),
    '👨🏾‍🎓': EmojiEntry(codepoint='1F468 1F3FE 200D 1F393', status='fully-qualified', emoji='👨🏾\u200d🎓', name='man student: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏿‍🎓': EmojiEntry(codepoint='1F468 1F3FF 200D 1F393', status='fully-qualified', emoji='👨🏿\u200d🎓', name='man student: dark skin tone', group='People & Body', sub_group='person-role'),
    '👩‍🎓': EmojiEntry(codepoint='1F469 200D 1F393', status='fully-qualified', emoji='👩\u200d🎓', name='woman student', group='People & Body', sub_group='person-role'),
    '👩🏻‍🎓': EmojiEntry(codepoint='1F469 1F3FB 200D 1F393', status='fully-qualified', emoji='👩🏻\u200d🎓', name='woman student: light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏼‍🎓': EmojiEntry(codepoint='1F469 1F3FC 200D 1F393', status='fully-qualified', emoji='👩🏼\u200d🎓', name='woman student: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏽‍🎓': EmojiEntry(codepoint='1F469 1F3FD 200D 1F393', status='fully-qualified', emoji='👩🏽\u200d🎓', name='woman student: medium skin tone', group='People & Body', sub_group='person-role'),
    '👩🏾‍🎓': EmojiEntry(codepoint='1F469 1F3FE 200D 1F393', status='fully-qualified', emoji='👩🏾\u200d🎓', name='woman student: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏿‍🎓': EmojiEntry(codepoint='1F469 1F3FF 200D 1F393', status='fully-qualified', emoji='👩🏿\u200d🎓', name='woman student: dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑‍🏫': EmojiEntry(codepoint='1F9D1 200D 1F3EB', status='fully-qualified', emoji='🧑\u200d🏫', name='teacher', group='People & Body', sub_group='person-role'),
    '🧑🏻‍🏫': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F3EB', status='fully-qualified', emoji='🧑🏻\u200d🏫', name='teacher: light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏼‍🏫': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F3EB', status='fully-qualified', emoji='🧑🏼\u200d🏫', name='teacher: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏽‍🏫': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F3EB', status='fully-qualified', emoji='🧑🏽\u200d🏫', name='teacher: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏾‍🏫': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F3EB', status='fully-qualified', emoji='🧑🏾\u200d🏫', name='teacher: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏿‍🏫': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F3EB', status='fully-qualified', emoji='🧑🏿\u200d🏫', name='teacher: dark skin tone', group='People & Body', sub_group='person-role'),
    '👨‍🏫': EmojiEntry(codepoint='1F468 200D 1F3EB', status='fully-qualified', emoji='👨\u200d🏫', name='man teacher', group='People & Body', sub_group='person-role'),
    '👨🏻‍🏫': EmojiEntry(codepoint='1F468 1F3FB 200D 1F3EB', status='fully-qualified', emoji='👨🏻\u200d🏫', name='man teacher: light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏼‍🏫': EmojiEntry(codepoint='1F468 1F3FC 200D 1F3EB', status='fully-qualified', emoji='👨🏼\u200d🏫', name='man teacher: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏽‍🏫': EmojiEntry(codepoint='1F468 1F3FD 200D 1F3EB', status='fully-qualified', emoji='👨🏽\u200d🏫', name='man teacher: medium skin tone', group='People & Body', sub_group='person-role'),
    '👨🏾‍🏫': EmojiEntry(codepoint='1F468 1F3FE 200D 1F3EB', status='fully-qualified', emoji='👨🏾\u200d🏫', name='man teacher: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏿‍🏫': EmojiEntry(codepoint='1F468 1F3FF 200D 1F3EB', status='fully-qualified', emoji='👨🏿\u200d🏫', name='man teacher: dark skin tone', group='People & Body', sub_group='person-role'),
    '👩‍🏫': EmojiEntry(codepoint='1F469 200D 1F3EB', status='fully-qualified', emoji='👩\u200d🏫', name='woman teacher', group='People & Body', sub_group='person-role'),
    '👩🏻‍🏫': EmojiEntry(codepoint='1F469 1F3FB 200D 1F3EB', status='fully-qualified', emoji='👩🏻\u200d🏫', name='woman teacher: light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏼‍🏫': EmojiEntry(codepoint='1F469 1F3FC 200D 1F3EB', status='fully-qualified', emoji='👩🏼\u200d🏫', name='woman teacher: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏽‍🏫': EmojiEntry(codepoint='1F469 1F3FD 200D 1F3EB', status='fully-qualified', emoji='👩🏽\u200d🏫', name='woman teacher: medium skin tone', group='People & Body', sub_group='person-role'),
    '👩🏾‍🏫': EmojiEntry(codepoint='1F469 1F3FE 200D 1F3EB', status='fully-qualified', emoji='👩🏾\u200d🏫', name='woman teacher: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏿‍🏫': EmojiEntry(codepoint='1F469 1F3FF 200D 1F3EB', status='fully-qualified', emoji='👩🏿\u200d🏫', name='woman teacher: dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑‍⚖️': EmojiEntry(codepoint='1F9D1 200D 2696 FE0F', status='fully-qualified', emoji='🧑\u200d⚖️', name='judge', group='People & Body', sub_group='person-role'),
    '🧑‍⚖': EmojiEntry(codepoint='1F9D1 200D 2696', status='minimally-qualified', emoji='🧑\u200d⚖', name='judge', group='People & Body', sub_group='person-role'),
    '🧑🏻‍⚖️': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2696 FE0F', status='fully-qualified', emoji='🧑🏻\u200d⚖️', name='judge: light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏻‍⚖': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2696', status='minimally-qualified', emoji='🧑🏻\u200d⚖', name='judge: light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏼‍⚖️': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2696 FE0F', status='fully-qualified', emoji='🧑🏼\u200d⚖️', name='judge: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏼‍⚖': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2696', status='minimally-qualified', emoji='🧑🏼\u200d⚖', name='judge: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏽‍⚖️': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2696 FE0F', status='fully-qualified', emoji='🧑🏽\u200d⚖️', name='judge: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏽‍⚖': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2696', status='minimally-qualified', emoji='🧑🏽\u200d⚖', name='judge: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏾‍⚖️': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2696 FE0F', status='fully-qualified', emoji='🧑🏾\u200d⚖️', name='judge: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏾‍⚖': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2696', status='minimally-qualified', emoji='🧑🏾\u200d⚖', name='judge: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏿‍⚖️': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2696 FE0F', status='fully-qualified', emoji='🧑🏿\u200d⚖️', name='judge: dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏿‍⚖': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2696', status='minimally-qualified', emoji='🧑🏿\u200d⚖', name='judge: dark skin tone', group='People & Body', sub_group='person-role'),
    '👨‍⚖️': EmojiEntry(codepoint='1F468 200D 2696 FE0F', status='fully-qualified', emoji='👨\u200d⚖️', name='man judge', group='People & Body', sub_group='person-role'),
    '👨‍⚖': EmojiEntry(codepoint='1F468 200D 2696', status='minimally-qualified', emoji='👨\u200d⚖', name='man judge', group='People & Body', sub_group='person-role'),
    '👨🏻‍⚖️': EmojiEntry(codepoint='1F468 1F3FB 200D 2696 FE0F', status='fully-qualified', emoji='👨🏻\u200d⚖️', name='man judge: light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏻‍⚖': EmojiEntry(codepoint='1F468 1F3FB 200D 2696', status='minimally-qualified', emoji='👨🏻\u200d⚖', name='man judge: light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏼‍⚖️': EmojiEntry(codepoint='1F468 1F3FC 200D 2696 FE0F', status='fully-qualified', emoji='👨🏼\u200d⚖️', name='man judge: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏼‍⚖': EmojiEntry(codepoint='1F468 1F3FC 200D 2696', status='minimally-qualified', emoji='👨🏼\u200d⚖', name='man judge: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏽‍⚖️': EmojiEntry(codepoint='1F468 1F3FD 200D 2696 FE0F', status='fully-qualified', emoji='👨🏽\u200d⚖️', name='man judge: medium skin tone', group='People & Body', sub_group='person-role'),
    '👨🏽‍⚖': EmojiEntry(codepoint='1F468 1F3FD 200D 2696', status='minimally-qualified', emoji='👨🏽\u200d⚖', name='man judge: medium skin tone', group='People & Body', sub_group='person-role'),
    '👨🏾‍⚖️': EmojiEntry(codepoint='1F468 1F3FE 200D 2696 FE0F', status='fully-qualified', emoji='👨🏾\u200d⚖️', name='man judge: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏾‍⚖': EmojiEntry(codepoint='1F468 1F3FE 200D 2696', status='minimally-qualified', emoji='👨🏾\u200d⚖', name='man judge: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏿‍⚖️': EmojiEntry(codepoint='1F468 1F3FF 200D 2696 FE0F', status='fully-qualified', emoji='👨🏿\u200d⚖️', name='man judge: dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏿‍⚖': EmojiEntry(codepoint='1F468 1F3FF 200D 2696', status='minimally-qualified', emoji='👨🏿\u200d⚖', name='man judge: dark skin tone', group='People & Body', sub_group='person-role'),
    '👩‍⚖️': EmojiEntry(codepoint='1F469 200D 2696 FE0F', status='fully-qualified', emoji='👩\u200d⚖️', name='woman judge', group='People & Body', sub_group='person-role'),
    '👩‍⚖': EmojiEntry(codepoint='1F469 200D 2696', status='minimally-qualified', emoji='👩\u200d⚖', name='woman judge', group='People & Body', sub_group='person-role'),
    '👩🏻‍⚖️': EmojiEntry(codepoint='1F469 1F3FB 200D 2696 FE0F', status='fully-qualified', emoji='👩🏻\u200d⚖️', name='woman judge: light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏻‍⚖': EmojiEntry(codepoint='1F469 1F3FB 200D 2696', status='minimally-qualified', emoji='👩🏻\u200d⚖', name='woman judge: light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏼‍⚖️': EmojiEntry(codepoint='1F469 1F3FC 200D 2696 FE0F', status='fully-qualified', emoji='👩🏼\u200d⚖️', name='woman judge: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏼‍⚖': EmojiEntry(codepoint='1F469 1F3FC 200D 2696', status='minimally-qualified', emoji='👩🏼\u200d⚖', name='woman judge: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏽‍⚖️': EmojiEntry(codepoint='1F469 1F3FD 200D 2696 FE0F', status='fully-qualified', emoji='👩🏽\u200d⚖️', name='woman judge: medium skin tone', group='People & Body', sub_group='person-role'),
    '👩🏽‍⚖': EmojiEntry(codepoint='1F469 1F3FD 200D 2696', status='minimally-qualified', emoji='👩🏽\u200d⚖', name='woman judge: medium skin tone', group='People & Body', sub_group='person-role'),
    '👩🏾‍⚖️': EmojiEntry(codepoint='1F469 1F3FE 200D 2696 FE0F', status='fully-qualified', emoji='👩🏾\u200d⚖️', name='woman judge: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏾‍⚖': EmojiEntry(codepoint='1F469 1F3FE 200D 2696', status='minimally-qualified', emoji='👩🏾\u200d⚖', name='woman judge: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏿‍⚖️': EmojiEntry(codepoint='1F469 1F3FF 200D 2696 FE0F', status='fully-qualified', emoji='👩🏿\u200d⚖️', name='woman judge: dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏿‍⚖': EmojiEntry(codepoint='1F469 1F3FF 200D 2696', status='minimally-qualified', emoji='👩🏿\u200d⚖', name='woman judge: dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑‍🌾': EmojiEntry(codepoint='1F9D1 200D 1F33E', status='fully-qualified', emoji='🧑\u200d🌾', name='farmer', group='People & Body', sub_group='person-role'),
    '🧑🏻‍🌾': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F33E', status='fully-qualified', emoji='🧑🏻\u200d🌾', name='farmer: light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏼‍🌾': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F33E', status='fully-qualified', emoji='🧑🏼\u200d🌾', name='farmer: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏽‍🌾': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F33E', status='fully-qualified', emoji='🧑🏽\u200d🌾', name='farmer: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏾‍🌾': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F33E', status='fully-qualified', emoji='🧑🏾\u200d🌾', name='farmer: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏿‍🌾': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F33E', status='fully-qualified', emoji='🧑🏿\u200d🌾', name='farmer: dark skin tone', group='People & Body', sub_group='person-role'),
    '👨‍🌾': EmojiEntry(codepoint='1F468 200D 1F33E', status='fully-qualified', emoji='👨\u200d🌾', name='man farmer', group='People & Body', sub_group='person-role'),
    '👨🏻‍🌾': EmojiEntry(codepoint='1F468 1F3FB 200D 1F33E', status='fully-qualified', emoji='👨🏻\u200d🌾', name='man farmer: light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏼‍🌾': EmojiEntry(codepoint='1F468 1F3FC 200D 1F33E', status='fully-qualified', emoji='👨🏼\u200d🌾', name='man farmer: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏽‍🌾': EmojiEntry(codepoint='1F468 1F3FD 200D 1F33E', status='fully-qualified', emoji='👨🏽\u200d🌾', name='man farmer: medium skin tone', group='People & Body', sub_group='person-role'),
    '👨🏾‍🌾': EmojiEntry(codepoint='1F468 1F3FE 200D 1F33E', status='fully-qualified', emoji='👨🏾\u200d🌾', name='man farmer: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏿‍🌾': EmojiEntry(codepoint='1F468 1F3FF 200D 1F33E', status='fully-qualified', emoji='👨🏿\u200d🌾', name='man farmer: dark skin tone', group='People & Body', sub_group='person-role'),
    '👩‍🌾': EmojiEntry(codepoint='1F469 200D 1F33E', status='fully-qualified', emoji='👩\u200d🌾', name='woman farmer', group='People & Body', sub_group='person-role'),
    '👩🏻‍🌾': EmojiEntry(codepoint='1F469 1F3FB 200D 1F33E', status='fully-qualified', emoji='👩🏻\u200d🌾', name='woman farmer: light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏼‍🌾': EmojiEntry(codepoint='1F469 1F3FC 200D 1F33E', status='fully-qualified', emoji='👩🏼\u200d🌾', name='woman farmer: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏽‍🌾': EmojiEntry(codepoint='1F469 1F3FD 200D 1F33E', status='fully-qualified', emoji='👩🏽\u200d🌾', name='woman farmer: medium skin tone', group='People & Body', sub_group='person-role'),
    '👩🏾‍🌾': EmojiEntry(codepoint='1F469 1F3FE 200D 1F33E', status='fully-qualified', emoji='👩🏾\u200d🌾', name='woman farmer: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏿‍🌾': EmojiEntry(codepoint='1F469 1F3FF 200D 1F33E', status='fully-qualified', emoji='👩🏿\u200d🌾', name='woman farmer: dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑‍🍳': EmojiEntry(codepoint='1F9D1 200D 1F373', status='fully-qualified', emoji='🧑\u200d🍳', name='cook', group='People & Body', sub_group='person-role'),
    '🧑🏻‍🍳': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F373', status='fully-qualified', emoji='🧑🏻\u200d🍳', name='cook: light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏼‍🍳': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F373', status='fully-qualified', emoji='🧑🏼\u200d🍳', name='cook: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏽‍🍳': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F373', status='fully-qualified', emoji='🧑🏽\u200d🍳', name='cook: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏾‍🍳': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F373', status='fully-qualified', emoji='🧑🏾\u200d🍳', name='cook: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏿‍🍳': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F373', status='fully-qualified', emoji='🧑🏿\u200d🍳', name='cook: dark skin tone', group='People & Body', sub_group='person-role'),
    '👨‍🍳': EmojiEntry(codepoint='1F468 200D 1F373', status='fully-qualified', emoji='👨\u200d🍳', name='man cook', group='People & Body', sub_group='person-role'),
    '👨🏻‍🍳': EmojiEntry(codepoint='1F468 1F3FB 200D 1F373', status='fully-qualified', emoji='👨🏻\u200d🍳', name='man cook: light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏼‍🍳': EmojiEntry(codepoint='1F468 1F3FC 200D 1F373', status='fully-qualified', emoji='👨🏼\u200d🍳', name='man cook: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏽‍🍳': EmojiEntry(codepoint='1F468 1F3FD 200D 1F373', status='fully-qualified', emoji='👨🏽\u200d🍳', name='man cook: medium skin tone', group='People & Body', sub_group='person-role'),
    '👨🏾‍🍳': EmojiEntry(codepoint='1F468 1F3FE 200D 1F373', status='fully-qualified', emoji='👨🏾\u200d🍳', name='man cook: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏿‍🍳': EmojiEntry(codepoint='1F468 1F3FF 200D 1F373', status='fully-qualified', emoji='👨🏿\u200d🍳', name='man cook: dark skin tone', group='People & Body', sub_group='person-role'),
    '👩‍🍳': EmojiEntry(codepoint='1F469 200D 1F373', status='fully-qualified', emoji='👩\u200d🍳', name='woman cook', group='People & Body', sub_group='person-role'),
    '👩🏻‍🍳': EmojiEntry(codepoint='1F469 1F3FB 200D 1F373', status='fully-qualified', emoji='👩🏻\u200d🍳', name='woman cook: light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏼‍🍳': EmojiEntry(codepoint='1F469 1F3FC 200D 1F373', status='fully-qualified', emoji='👩🏼\u200d🍳', name='woman cook: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏽‍🍳': EmojiEntry(codepoint='1F469 1F3FD 200D 1F373', status='fully-qualified', emoji='👩🏽\u200d🍳', name='woman cook: medium skin tone', group='People & Body', sub_group='person-role'),
    '👩🏾‍🍳': EmojiEntry(codepoint='1F469 1F3FE 200D 1F373', status='fully-qualified', emoji='👩🏾\u200d🍳', name='woman cook: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏿‍🍳': EmojiEntry(codepoint='1F469 1F3FF 200D 1F373', status='fully-qualified', emoji='👩🏿\u200d🍳', name='woman cook: dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑‍🔧': EmojiEntry(codepoint='1F9D1 200D 1F527', status='fully-qualified', emoji='🧑\u200d🔧', name='mechanic', group='People & Body', sub_group='person-role'),
    '🧑🏻‍🔧': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F527', status='fully-qualified', emoji='🧑🏻\u200d🔧', name='mechanic: light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏼‍🔧': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F527', status='fully-qualified', emoji='🧑🏼\u200d🔧', name='mechanic: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏽‍🔧': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F527', status='fully-qualified', emoji='🧑🏽\u200d🔧', name='mechanic: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏾‍🔧': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F527', status='fully-qualified', emoji='🧑🏾\u200d🔧', name='mechanic: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏿‍🔧': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F527', status='fully-qualified', emoji='🧑🏿\u200d🔧', name='mechanic: dark skin tone', group='People & Body', sub_group='person-role'),
    '👨‍🔧': EmojiEntry(codepoint='1F468 200D 1F527', status='fully-qualified', emoji='👨\u200d🔧', name='man mechanic', group='People & Body', sub_group='person-role'),
    '👨🏻‍🔧': EmojiEntry(codepoint='1F468 1F3FB 200D 1F527', status='fully-qualified', emoji='👨🏻\u200d🔧', name='man mechanic: light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏼‍🔧': EmojiEntry(codepoint='1F468 1F3FC 200D 1F527', status='fully-qualified', emoji='👨🏼\u200d🔧', name='man mechanic: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏽‍🔧': EmojiEntry(codepoint='1F468 1F3FD 200D 1F527', status='fully-qualified', emoji='👨🏽\u200d🔧', name='man mechanic: medium skin tone', group='People & Body', sub_group='person-role'),
    '👨🏾‍🔧': EmojiEntry(codepoint='1F468 1F3FE 200D 1F527', status='fully-qualified', emoji='👨🏾\u200d🔧', name='man mechanic: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏿‍🔧': EmojiEntry(codepoint='1F468 1F3FF 200D 1F527', status='fully-qualified', emoji='👨🏿\u200d🔧', name='man mechanic: dark skin tone', group='People & Body', sub_group='person-role'),
    '👩‍🔧': EmojiEntry(codepoint='1F469 200D 1F527', status='fully-qualified', emoji='👩\u200d🔧', name='woman mechanic', group='People & Body', sub_group='person-role'),
    '👩🏻‍🔧': EmojiEntry(codepoint='1F469 1F3FB 200D 1F527', status='fully-qualified', emoji='👩🏻\u200d🔧', name='woman mechanic: light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏼‍🔧': EmojiEntry(codepoint='1F469 1F3FC 200D 1F527', status='fully-qualified', emoji='👩🏼\u200d🔧', name='woman mechanic: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏽‍🔧': EmojiEntry(codepoint='1F469 1F3FD 200D 1F527', status='fully-qualified', emoji='👩🏽\u200d🔧', name='woman mechanic: medium skin tone', group='People & Body', sub_group='person-role'),
    '👩🏾‍🔧': EmojiEntry(codepoint='1F469 1F3FE 200D 1F527', status='fully-qualified', emoji='👩🏾\u200d🔧', name='woman mechanic: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏿‍🔧': EmojiEntry(codepoint='1F469 1F3FF 200D 1F527', status='fully-qualified', emoji='👩🏿\u200d🔧', name='woman mechanic: dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑‍🏭': EmojiEntry(codepoint='1F9D1 200D 1F3ED', status='fully-qualified', emoji='🧑\u200d🏭', name='factory worker', group='People & Body', sub_group='person-role'),
    '🧑🏻‍🏭': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F3ED', status='fully-qualified', emoji='🧑🏻\u200d🏭', name='factory worker: light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏼‍🏭': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F3ED', status='fully-qualified', emoji='🧑🏼\u200d🏭', name='factory worker: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏽‍🏭': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F3ED', status='fully-qualified', emoji='🧑🏽\u200d🏭', name='factory worker: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏾‍🏭': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F3ED', status='fully-qualified', emoji='🧑🏾\u200d🏭', name='factory worker: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏿‍🏭': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F3ED', status='fully-qualified', emoji='🧑🏿\u200d🏭', name='factory worker: dark skin tone', group='People & Body', sub_group='person-role'),
    '👨‍🏭': EmojiEntry(codepoint='1F468 200D 1F3ED', status='fully-qualified', emoji='👨\u200d🏭', name='man factory worker', group='People & Body', sub_group='person-role'),
    '👨🏻‍🏭': EmojiEntry(codepoint='1F468 1F3FB 200D 1F3ED', status='fully-qualified', emoji='👨🏻\u200d🏭', name='man factory worker: light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏼‍🏭': EmojiEntry(codepoint='1F468 1F3FC 200D 1F3ED', status='fully-qualified', emoji='👨🏼\u200d🏭', name='man factory worker: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏽‍🏭': EmojiEntry(codepoint='1F468 1F3FD 200D 1F3ED', status='fully-qualified', emoji='👨🏽\u200d🏭', name='man factory worker: medium skin tone', group='People & Body', sub_group='person-role'),
    '👨🏾‍🏭': EmojiEntry(codepoint='1F468 1F3FE 200D 1F3ED', status='fully-qualified', emoji='👨🏾\u200d🏭', name='man factory worker: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏿‍🏭': EmojiEntry(codepoint='1F468 1F3FF 200D 1F3ED', status='fully-qualified', emoji='👨🏿\u200d🏭', name='man factory worker: dark skin tone', group='People & Body', sub_group='person-role'),
    '👩‍🏭': EmojiEntry(codepoint='1F469 200D 1F3ED', status='fully-qualified', emoji='👩\u200d🏭', name='woman factory worker', group='People & Body', sub_group='person-role'),
    '👩🏻‍🏭': EmojiEntry(codepoint='1F469 1F3FB 200D 1F3ED', status='fully-qualified', emoji='👩🏻\u200d🏭', name='woman factory worker: light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏼‍🏭': EmojiEntry(codepoint='1F469 1F3FC 200D 1F3ED', status='fully-qualified', emoji='👩🏼\u200d🏭', name='woman factory worker: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏽‍🏭': EmojiEntry(codepoint='1F469 1F3FD 200D 1F3ED', status='fully-qualified', emoji='👩🏽\u200d🏭', name='woman factory worker: medium skin tone', group='People & Body', sub_group='person-role'),
    '👩🏾‍🏭': EmojiEntry(codepoint='1F469 1F3FE 200D 1F3ED', status='fully-qualified', emoji='👩🏾\u200d🏭', name='woman factory worker: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏿‍🏭': EmojiEntry(codepoint='1F469 1F3FF 200D 1F3ED', status='fully-qualified', emoji='👩🏿\u200d🏭', name='woman factory worker: dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑‍💼': EmojiEntry(codepoint='1F9D1 200D 1F4BC', status='fully-qualified', emoji='🧑\u200d💼', name='office worker', group='People & Body', sub_group='person-role'),
    '🧑🏻‍💼': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F4BC', status='fully-qualified', emoji='🧑🏻\u200d💼', name='office worker: light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏼‍💼': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F4BC', status='fully-qualified', emoji='🧑🏼\u200d💼', name='office worker: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏽‍💼': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F4BC', status='fully-qualified', emoji='🧑🏽\u200d💼', name='office worker: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏾‍💼': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F4BC', status='fully-qualified', emoji='🧑🏾\u200d💼', name='office worker: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏿‍💼': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F4BC', status='fully-qualified', emoji='🧑🏿\u200d💼', name='office worker: dark skin tone', group='People & Body', sub_group='person-role'),
    '👨‍💼': EmojiEntry(codepoint='1F468 200D 1F4BC', status='fully-qualified', emoji='👨\u200d💼', name='man office worker', group='People & Body', sub_group='person-role'),
    '👨🏻‍💼': EmojiEntry(codepoint='1F468 1F3FB 200D 1F4BC', status='fully-qualified', emoji='👨🏻\u200d💼', name='man office worker: light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏼‍💼': EmojiEntry(codepoint='1F468 1F3FC 200D 1F4BC', status='fully-qualified', emoji='👨🏼\u200d💼', name='man office worker: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏽‍💼': EmojiEntry(codepoint='1F468 1F3FD 200D 1F4BC', status='fully-qualified', emoji='👨🏽\u200d💼', name='man office worker: medium skin tone', group='People & Body', sub_group='person-role'),
    '👨🏾‍💼': EmojiEntry(codepoint='1F468 1F3FE 200D 1F4BC', status='fully-qualified', emoji='👨🏾\u200d💼', name='man office worker: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏿‍💼': EmojiEntry(codepoint='1F468 1F3FF 200D 1F4BC', status='fully-qualified', emoji='👨🏿\u200d💼', name='man office worker: dark skin tone', group='People & Body', sub_group='person-role'),
    '👩‍💼': EmojiEntry(codepoint='1F469 200D 1F4BC', status='fully-qualified', emoji='👩\u200d💼', name='woman office worker', group='People & Body', sub_group='person-role'),
    '👩🏻‍💼': EmojiEntry(codepoint='1F469 1F3FB 200D 1F4BC', status='fully-qualified', emoji='👩🏻\u200d💼', name='woman office worker: light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏼‍💼': EmojiEntry(codepoint='1F469 1F3FC 200D 1F4BC', status='fully-qualified', emoji='👩🏼\u200d💼', name='woman office worker: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏽‍💼': EmojiEntry(codepoint='1F469 1F3FD 200D 1F4BC', status='fully-qualified', emoji='👩🏽\u200d💼', name='woman office worker: medium skin tone', group='People & Body', sub_group='person-role'),
    '👩🏾‍💼': EmojiEntry(codepoint='1F469 1F3FE 200D 1F4BC', status='fully-qualified', emoji='👩🏾\u200d💼', name='woman office worker: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏿‍💼': EmojiEntry(codepoint='1F469 1F3FF 200D 1F4BC', status='fully-qualified', emoji='👩🏿\u200d💼', name='woman office worker: dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑‍🔬': EmojiEntry(codepoint='1F9D1 200D 1F52C', status='fully-qualified', emoji='🧑\u200d🔬', name='scientist', group='People & Body', sub_group='person-role'),
    '🧑🏻‍🔬': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F52C', status='fully-qualified', emoji='🧑🏻\u200d🔬', name='scientist: light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏼‍🔬': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F52C', status='fully-qualified', emoji='🧑🏼\u200d🔬', name='scientist: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏽‍🔬': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F52C', status='fully-qualified', emoji='🧑🏽\u200d🔬', name='scientist: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏾‍🔬': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F52C', status='fully-qualified', emoji='🧑🏾\u200d🔬', name='scientist: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏿‍🔬': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F52C', status='fully-qualified', emoji='🧑🏿\u200d🔬', name='scientist: dark skin tone', group='People & Body', sub_group='person-role'),
    '👨‍🔬': EmojiEntry(codepoint='1F468 200D 1F52C', status='fully-qualified', emoji='👨\u200d🔬', name='man scientist', group='People & Body', sub_group='person-role'),
    '👨🏻‍🔬': EmojiEntry(codepoint='1F468 1F3FB 200D 1F52C', status='fully-qualified', emoji='👨🏻\u200d🔬', name='man scientist: light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏼‍🔬': EmojiEntry(codepoint='1F468 1F3FC 200D 1F52C', status='fully-qualified', emoji='👨🏼\u200d🔬', name='man scientist: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏽‍🔬': EmojiEntry(codepoint='1F468 1F3FD 200D 1F52C', status='fully-qualified', emoji='👨🏽\u200d🔬', name='man scientist: medium skin tone', group='People & Body', sub_group='person-role'),
    '👨🏾‍🔬': EmojiEntry(codepoint='1F468 1F3FE 200D 1F52C', status='fully-qualified', emoji='👨🏾\u200d🔬', name='man scientist: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏿‍🔬': EmojiEntry(codepoint='1F468 1F3FF 200D 1F52C', status='fully-qualified', emoji='👨🏿\u200d🔬', name='man scientist: dark skin tone', group='People & Body', sub_group='person-role'),
    '👩‍🔬': EmojiEntry(codepoint='1F469 200D 1F52C', status='fully-qualified', emoji='👩\u200d🔬', name='woman scientist', group='People & Body', sub_group='person-role'),
    '👩🏻‍🔬': EmojiEntry(codepoint='1F469 1F3FB 200D 1F52C', status='fully-qualified', emoji='👩🏻\u200d🔬', name='woman scientist: light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏼‍🔬': EmojiEntry(codepoint='1F469 1F3FC 200D 1F52C', status='fully-qualified', emoji='👩🏼\u200d🔬', name='woman scientist: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏽‍🔬': EmojiEntry(codepoint='1F469 1F3FD 200D 1F52C', status='fully-qualified', emoji='👩🏽\u200d🔬', name='woman scientist: medium skin tone', group='People & Body', sub_group='person-role'),
    '👩🏾‍🔬': EmojiEntry(codepoint='1F469 1F3FE 200D 1F52C', status='fully-qualified', emoji='👩🏾\u200d🔬', name='woman scientist: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏿‍🔬': EmojiEntry(codepoint='1F469 1F3FF 200D 1F52C', status='fully-qualified', emoji='👩🏿\u200d🔬', name='woman scientist: dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑‍💻': EmojiEntry(codepoint='1F9D1 200D 1F4BB', status='fully-qualified', emoji='🧑\u200d💻', name='technologist', group='People & Body', sub_group='person-role'),
    '🧑🏻‍💻': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F4BB', status='fully-qualified', emoji='🧑🏻\u200d💻', name='technologist: light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏼‍💻': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F4BB', status='fully-qualified', emoji='🧑🏼\u200d💻', name='technologist: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏽‍💻': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F4BB', status='fully-qualified', emoji='🧑🏽\u200d💻', name='technologist: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏾‍💻': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F4BB', status='fully-qualified', emoji='🧑🏾\u200d💻', name='technologist: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏿‍💻': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F4BB', status='fully-qualified', emoji='🧑🏿\u200d💻', name='technologist: dark skin tone', group='People & Body', sub_group='person-role'),
    '👨‍💻': EmojiEntry(codepoint='1F468 200D 1F4BB', status='fully-qualified', emoji='👨\u200d💻', name='man technologist', group='People & Body', sub_group='person-role'),
    '👨🏻‍💻': EmojiEntry(codepoint='1F468 1F3FB 200D 1F4BB', status='fully-qualified', emoji='👨🏻\u200d💻', name='man technologist: light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏼‍💻': EmojiEntry(codepoint='1F468 1F3FC 200D 1F4BB', status='fully-qualified', emoji='👨🏼\u200d💻', name='man technologist: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏽‍💻': EmojiEntry(codepoint='1F468 1F3FD 200D 1F4BB', status='fully-qualified', emoji='👨🏽\u200d💻', name='man technologist: medium skin tone', group='People & Body', sub_group='person-role'),
    '👨🏾‍💻': EmojiEntry(codepoint='1F468 1F3FE 200D 1F4BB', status='fully-qualified', emoji='👨🏾\u200d💻', name='man technologist: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏿‍💻': EmojiEntry(codepoint='1F468 1F3FF 200D 1F4BB', status='fully-qualified', emoji='👨🏿\u200d💻', name='man technologist: dark skin tone', group='People & Body', sub_group='person-role'),
    '👩‍💻': EmojiEntry(codepoint='1F469 200D 1F4BB', status='fully-qualified', emoji='👩\u200d💻', name='woman technologist', group='People & Body', sub_group='person-role'),
    '👩🏻‍💻': EmojiEntry(codepoint='1F469 1F3FB 200D 1F4BB', status='fully-qualified', emoji='👩🏻\u200d💻', name='woman technologist: light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏼‍💻': EmojiEntry(codepoint='1F469 1F3FC 200D 1F4BB', status='fully-qualified', emoji='👩🏼\u200d💻', name='woman technologist: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏽‍💻': EmojiEntry(codepoint='1F469 1F3FD 200D 1F4BB', status='fully-qualified', emoji='👩🏽\u200d💻', name='woman technologist: medium skin tone', group='People & Body', sub_group='person-role'),
    '👩🏾‍💻': EmojiEntry(codepoint='1F469 1F3FE 200D 1F4BB', status='fully-qualified', emoji='👩🏾\u200d💻', name='woman technologist: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏿‍💻': EmojiEntry(codepoint='1F469 1F3FF 200D 1F4BB', status='fully-qualified', emoji='👩🏿\u200d💻', name='woman technologist: dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑‍🎤': EmojiEntry(codepoint='1F9D1 200D 1F3A4', status='fully-qualified', emoji='🧑\u200d🎤', name='singer', group='People & Body', sub_group='person-role'),
    '🧑🏻‍🎤': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F3A4', status='fully-qualified', emoji='🧑🏻\u200d🎤', name='singer: light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏼‍🎤': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F3A4', status='fully-qualified', emoji='🧑🏼\u200d🎤', name='singer: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏽‍🎤': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F3A4', status='fully-qualified', emoji='🧑🏽\u200d🎤', name='singer: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏾‍🎤': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F3A4', status='fully-qualified', emoji='🧑🏾\u200d🎤', name='singer: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏿‍🎤': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F3A4', status='fully-qualified', emoji='🧑🏿\u200d🎤', name='singer: dark skin tone', group='People & Body', sub_group='person-role'),
    '👨‍🎤': EmojiEntry(codepoint='1F468 200D 1F3A4', status='fully-qualified', emoji='👨\u200d🎤', name='man singer', group='People & Body', sub_group='person-role'),
    '👨🏻‍🎤': EmojiEntry(codepoint='1F468 1F3FB 200D 1F3A4', status='fully-qualified', emoji='👨🏻\u200d🎤', name='man singer: light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏼‍🎤': EmojiEntry(codepoint='1F468 1F3FC 200D 1F3A4', status='fully-qualified', emoji='👨🏼\u200d🎤', name='man singer: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏽‍🎤': EmojiEntry(codepoint='1F468 1F3FD 200D 1F3A4', status='fully-qualified', emoji='👨🏽\u200d🎤', name='man singer: medium skin tone', group='People & Body', sub_group='person-role'),
    '👨🏾‍🎤': EmojiEntry(codepoint='1F468 1F3FE 200D 1F3A4', status='fully-qualified', emoji='👨🏾\u200d🎤', name='man singer: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏿‍🎤': EmojiEntry(codepoint='1F468 1F3FF 200D 1F3A4', status='fully-qualified', emoji='👨🏿\u200d🎤', name='man singer: dark skin tone', group='People & Body', sub_group='person-role'),
    '👩‍🎤': EmojiEntry(codepoint='1F469 200D 1F3A4', status='fully-qualified', emoji='👩\u200d🎤', name='woman singer', group='People & Body', sub_group='person-role'),
    '👩🏻‍🎤': EmojiEntry(codepoint='1F469 1F3FB 200D 1F3A4', status='fully-qualified', emoji='👩🏻\u200d🎤', name='woman singer: light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏼‍🎤': EmojiEntry(codepoint='1F469 1F3FC 200D 1F3A4', status='fully-qualified', emoji='👩🏼\u200d🎤', name='woman singer: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏽‍🎤': EmojiEntry(codepoint='1F469 1F3FD 200D 1F3A4', status='fully-qualified', emoji='👩🏽\u200d🎤', name='woman singer: medium skin tone', group='People & Body', sub_group='person-role'),
    '👩🏾‍🎤': EmojiEntry(codepoint='1F469 1F3FE 200D 1F3A4', status='fully-qualified', emoji='👩🏾\u200d🎤', name='woman singer: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏿‍🎤': EmojiEntry(codepoint='1F469 1F3FF 200D 1F3A4', status='fully-qualified', emoji='👩🏿\u200d🎤', name='woman singer: dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑‍🎨': EmojiEntry(codepoint='1F9D1 200D 1F3A8', status='fully-qualified', emoji='🧑\u200d🎨', name='artist', group='People & Body', sub_group='person-role'),
    '🧑🏻‍🎨': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F3A8', status='fully-qualified', emoji='🧑🏻\u200d🎨', name='artist: light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏼‍🎨': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F3A8', status='fully-qualified', emoji='🧑🏼\u200d🎨', name='artist: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏽‍🎨': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F3A8', status='fully-qualified', emoji='🧑🏽\u200d🎨', name='artist: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏾‍🎨': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F3A8', status='fully-qualified', emoji='🧑🏾\u200d🎨', name='artist: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏿‍🎨': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F3A8', status='fully-qualified', emoji='🧑🏿\u200d🎨', name='artist: dark skin tone', group='People & Body', sub_group='person-role'),
    '👨‍🎨': EmojiEntry(codepoint='1F468 200D 1F3A8', status='fully-qualified', emoji='👨\u200d🎨', name='man artist', group='People & Body', sub_group='person-role'),
    '👨🏻‍🎨': EmojiEntry(codepoint='1F468 1F3FB 200D 1F3A8', status='fully-qualified', emoji='👨🏻\u200d🎨', name='man artist: light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏼‍🎨': EmojiEntry(codepoint='1F468 1F3FC 200D 1F3A8', status='fully-qualified', emoji='👨🏼\u200d🎨', name='man artist: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏽‍🎨': EmojiEntry(codepoint='1F468 1F3FD 200D 1F3A8', status='fully-qualified', emoji='👨🏽\u200d🎨', name='man artist: medium skin tone', group='People & Body', sub_group='person-role'),
    '👨🏾‍🎨': EmojiEntry(codepoint='1F468 1F3FE 200D 1F3A8', status='fully-qualified', emoji='👨🏾\u200d🎨', name='man artist: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏿‍🎨': EmojiEntry(codepoint='1F468 1F3FF 200D 1F3A8', status='fully-qualified', emoji='👨🏿\u200d🎨', name='man artist: dark skin tone', group='People & Body', sub_group='person-role'),
    '👩‍🎨': EmojiEntry(codepoint='1F469 200D 1F3A8', status='fully-qualified', emoji='👩\u200d🎨', name='woman artist', group='People & Body', sub_group='person-role'),
    '👩🏻‍🎨': EmojiEntry(codepoint='1F469 1F3FB 200D 1F3A8', status='fully-qualified', emoji='👩🏻\u200d🎨', name='woman artist: light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏼‍🎨': EmojiEntry(codepoint='1F469 1F3FC 200D 1F3A8', status='fully-qualified', emoji='👩🏼\u200d🎨', name='woman artist: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏽‍🎨': EmojiEntry(codepoint='1F469 1F3FD 200D 1F3A8', status='fully-qualified', emoji='👩🏽\u200d🎨', name='woman artist: medium skin tone', group='People & Body', sub_group='person-role'),
    '👩🏾‍🎨': EmojiEntry(codepoint='1F469 1F3FE 200D 1F3A8', status='fully-qualified', emoji='👩🏾\u200d🎨', name='woman artist: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏿‍🎨': EmojiEntry(codepoint='1F469 1F3FF 200D 1F3A8', status='fully-qualified', emoji='👩🏿\u200d🎨', name='woman artist: dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑‍✈️': EmojiEntry(codepoint='1F9D1 200D 2708 FE0F', status='fully-qualified', emoji='🧑\u200d✈️', name='pilot', group='People & Body', sub_group='person-role'),
    '🧑‍✈': EmojiEntry(codepoint='1F9D1 200D 2708', status='minimally-qualified', emoji='🧑\u200d✈', name='pilot', group='People & Body', sub_group='person-role'),
    '🧑🏻‍✈️': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2708 FE0F', status='fully-qualified', emoji='🧑🏻\u200d✈️', name='pilot: light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏻‍✈': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2708', status='minimally-qualified', emoji='🧑🏻\u200d✈', name='pilot: light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏼‍✈️': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2708 FE0F', status='fully-qualified', emoji='🧑🏼\u200d✈️', name='pilot: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏼‍✈': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2708', status='minimally-qualified', emoji='🧑🏼\u200d✈', name='pilot: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏽‍✈️': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2708 FE0F', status='fully-qualified', emoji='🧑🏽\u200d✈️', name='pilot: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏽‍✈': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2708', status='minimally-qualified', emoji='🧑🏽\u200d✈', name='pilot: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏾‍✈️': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2708 FE0F', status='fully-qualified', emoji='🧑🏾\u200d✈️', name='pilot: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏾‍✈': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2708', status='minimally-qualified', emoji='🧑🏾\u200d✈', name='pilot: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏿‍✈️': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2708 FE0F', status='fully-qualified', emoji='🧑🏿\u200d✈️', name='pilot: dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏿‍✈': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2708', status='minimally-qualified', emoji='🧑🏿\u200d✈', name='pilot: dark skin tone', group='People & Body', sub_group='person-role'),
    '👨‍✈️': EmojiEntry(codepoint='1F468 200D 2708 FE0F', status='fully-qualified', emoji='👨\u200d✈️', name='man pilot', group='People & Body', sub_group='person-role'),
    '👨‍✈': EmojiEntry(codepoint='1F468 200D 2708', status='minimally-qualified', emoji='👨\u200d✈', name='man pilot', group='People & Body', sub_group='person-role'),
    '👨🏻‍✈️': EmojiEntry(codepoint='1F468 1F3FB 200D 2708 FE0F', status='fully-qualified', emoji='👨🏻\u200d✈️', name='man pilot: light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏻‍✈': EmojiEntry(codepoint='1F468 1F3FB 200D 2708', status='minimally-qualified', emoji='👨🏻\u200d✈', name='man pilot: light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏼‍✈️': EmojiEntry(codepoint='1F468 1F3FC 200D 2708 FE0F', status='fully-qualified', emoji='👨🏼\u200d✈️', name='man pilot: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏼‍✈': EmojiEntry(codepoint='1F468 1F3FC 200D 2708', status='minimally-qualified', emoji='👨🏼\u200d✈', name='man pilot: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏽‍✈️': EmojiEntry(codepoint='1F468 1F3FD 200D 2708 FE0F', status='fully-qualified', emoji='👨🏽\u200d✈️', name='man pilot: medium skin tone', group='People & Body', sub_group='person-role'),
    '👨🏽‍✈': EmojiEntry(codepoint='1F468 1F3FD 200D 2708', status='minimally-qualified', emoji='👨🏽\u200d✈', name='man pilot: medium skin tone', group='People & Body', sub_group='person-role'),
    '👨🏾‍✈️': EmojiEntry(codepoint='1F468 1F3FE 200D 2708 FE0F', status='fully-qualified', emoji='👨🏾\u200d✈️', name='man pilot: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏾‍✈': EmojiEntry(codepoint='1F468 1F3FE 200D 2708', status='minimally-qualified', emoji='👨🏾\u200d✈', name='man pilot: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏿‍✈️': EmojiEntry(codepoint='1F468 1F3FF 200D 2708 FE0F', status='fully-qualified', emoji='👨🏿\u200d✈️', name='man pilot: dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏿‍✈': EmojiEntry(codepoint='1F468 1F3FF 200D 2708', status='minimally-qualified', emoji='👨🏿\u200d✈', name='man pilot: dark skin tone', group='People & Body', sub_group='person-role'),
    '👩‍✈️': EmojiEntry(codepoint='1F469 200D 2708 FE0F', status='fully-qualified', emoji='👩\u200d✈️', name='woman pilot', group='People & Body', sub_group='person-role'),
    '👩‍✈': EmojiEntry(codepoint='1F469 200D 2708', status='minimally-qualified', emoji='👩\u200d✈', name='woman pilot', group='People & Body', sub_group='person-role'),
    '👩🏻‍✈️': EmojiEntry(codepoint='1F469 1F3FB 200D 2708 FE0F', status='fully-qualified', emoji='👩🏻\u200d✈️', name='woman pilot: light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏻‍✈': EmojiEntry(codepoint='1F469 1F3FB 200D 2708', status='minimally-qualified', emoji='👩🏻\u200d✈', name='woman pilot: light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏼‍✈️': EmojiEntry(codepoint='1F469 1F3FC 200D 2708 FE0F', status='fully-qualified', emoji='👩🏼\u200d✈️', name='woman pilot: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏼‍✈': EmojiEntry(codepoint='1F469 1F3FC 200D 2708', status='minimally-qualified', emoji='👩🏼\u200d✈', name='woman pilot: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏽‍✈️': EmojiEntry(codepoint='1F469 1F3FD 200D 2708 FE0F', status='fully-qualified', emoji='👩🏽\u200d✈️', name='woman pilot: medium skin tone', group='People & Body', sub_group='person-role'),
    '👩🏽‍✈': EmojiEntry(codepoint='1F469 1F3FD 200D 2708', status='minimally-qualified', emoji='👩🏽\u200d✈', name='woman pilot: medium skin tone', group='People & Body', sub_group='person-role'),
    '👩🏾‍✈️': EmojiEntry(codepoint='1F469 1F3FE 200D 2708 FE0F', status='fully-qualified', emoji='👩🏾\u200d✈️', name='woman pilot: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏾‍✈': EmojiEntry(codepoint='1F469 1F3FE 200D 2708', status='minimally-qualified', emoji='👩🏾\u200d✈', name='woman pilot: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏿‍✈️': EmojiEntry(codepoint='1F469 1F3FF 200D 2708 FE0F', status='fully-qualified', emoji='👩🏿\u200d✈️', name='woman pilot: dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏿‍✈': EmojiEntry(codepoint='1F469 1F3FF 200D 2708', status='minimally-qualified', emoji='👩🏿\u200d✈', name='woman pilot: dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑‍🚀': EmojiEntry(codepoint='1F9D1 200D 1F680', status='fully-qualified', emoji='🧑\u200d🚀', name='astronaut', group='People & Body', sub_group='person-role'),
    '🧑🏻‍🚀': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F680', status='fully-qualified', emoji='🧑🏻\u200d🚀', name='astronaut: light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏼‍🚀': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F680', status='fully-qualified', emoji='🧑🏼\u200d🚀', name='astronaut: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏽‍🚀': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F680', status='fully-qualified', emoji='🧑🏽\u200d🚀', name='astronaut: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏾‍🚀': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F680', status='fully-qualified', emoji='🧑🏾\u200d🚀', name='astronaut: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏿‍🚀': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F680', status='fully-qualified', emoji='🧑🏿\u200d🚀', name='astronaut: dark skin tone', group='People & Body', sub_group='person-role'),
    '👨‍🚀': EmojiEntry(codepoint='1F468 200D 1F680', status='fully-qualified', emoji='👨\u200d🚀', name='man astronaut', group='People & Body', sub_group='person-role'),
    '👨🏻‍🚀': EmojiEntry(codepoint='1F468 1F3FB 200D 1F680', status='fully-qualified', emoji='👨🏻\u200d🚀', name='man astronaut: light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏼‍🚀': EmojiEntry(codepoint='1F468 1F3FC 200D 1F680', status='fully-qualified', emoji='👨🏼\u200d🚀', name='man astronaut: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏽‍🚀': EmojiEntry(codepoint='1F468 1F3FD 200D 1F680', status='fully-qualified', emoji='👨🏽\u200d🚀', name='man astronaut: medium skin tone', group='People & Body', sub_group='person-role'),
    '👨🏾‍🚀': EmojiEntry(codepoint='1F468 1F3FE 200D 1F680', status='fully-qualified', emoji='👨🏾\u200d🚀', name='man astronaut: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏿‍🚀': EmojiEntry(codepoint='1F468 1F3FF 200D 1F680', status='fully-qualified', emoji='👨🏿\u200d🚀', name='man astronaut: dark skin tone', group='People & Body', sub_group='person-role'),
    '👩‍🚀': EmojiEntry(codepoint='1F469 200D 1F680', status='fully-qualified', emoji='👩\u200d🚀', name='woman astronaut', group='People & Body', sub_group='person-role'),
    '👩🏻‍🚀': EmojiEntry(codepoint='1F469 1F3FB 200D 1F680', status='fully-qualified', emoji='👩🏻\u200d🚀', name='woman astronaut: light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏼‍🚀': EmojiEntry(codepoint='1F469 1F3FC 200D 1F680', status='fully-qualified', emoji='👩🏼\u200d🚀', name='woman astronaut: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏽‍🚀': EmojiEntry(codepoint='1F469 1F3FD 200D 1F680', status='fully-qualified', emoji='👩🏽\u200d🚀', name='woman astronaut: medium skin tone', group='People & Body', sub_group='person-role'),
    '👩🏾‍🚀': EmojiEntry(codepoint='1F469 1F3FE 200D 1F680', status='fully-qualified', emoji='👩🏾\u200d🚀', name='woman astronaut: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏿‍🚀': EmojiEntry(codepoint='1F469 1F3FF 200D 1F680', status='fully-qualified', emoji='👩🏿\u200d🚀', name='woman astronaut: dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑‍🚒': EmojiEntry(codepoint='1F9D1 200D 1F692', status='fully-qualified', emoji='🧑\u200d🚒', name='firefighter', group='People & Body', sub_group='person-role'),
    '🧑🏻‍🚒': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F692', status='fully-qualified', emoji='🧑🏻\u200d🚒', name='firefighter: light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏼‍🚒': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F692', status='fully-qualified', emoji='🧑🏼\u200d🚒', name='firefighter: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏽‍🚒': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F692', status='fully-qualified', emoji='🧑🏽\u200d🚒', name='firefighter: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏾‍🚒': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F692', status='fully-qualified', emoji='🧑🏾\u200d🚒', name='firefighter: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏿‍🚒': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F692', status='fully-qualified', emoji='🧑🏿\u200d🚒', name='firefighter: dark skin tone', group='People & Body', sub_group='person-role'),
    '👨‍🚒': EmojiEntry(codepoint='1F468 200D 1F692', status='fully-qualified', emoji='👨\u200d🚒', name='man firefighter', group='People & Body', sub_group='person-role'),
    '👨🏻‍🚒': EmojiEntry(codepoint='1F468 1F3FB 200D 1F692', status='fully-qualified', emoji='👨🏻\u200d🚒', name='man firefighter: light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏼‍🚒': EmojiEntry(codepoint='1F468 1F3FC 200D 1F692', status='fully-qualified', emoji='👨🏼\u200d🚒', name='man firefighter: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏽‍🚒': EmojiEntry(codepoint='1F468 1F3FD 200D 1F692', status='fully-qualified', emoji='👨🏽\u200d🚒', name='man firefighter: medium skin tone', group='People & Body', sub_group='person-role'),
    '👨🏾‍🚒': EmojiEntry(codepoint='1F468 1F3FE 200D 1F692', status='fully-qualified', emoji='👨🏾\u200d🚒', name='man firefighter: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏿‍🚒': EmojiEntry(codepoint='1F468 1F3FF 200D 1F692', status='fully-qualified', emoji='👨🏿\u200d🚒', name='man firefighter: dark skin tone', group='People & Body', sub_group='person-role'),
    '👩‍🚒': EmojiEntry(codepoint='1F469 200D 1F692', status='fully-qualified', emoji='👩\u200d🚒', name='woman firefighter', group='People & Body', sub_group='person-role'),
    '👩🏻‍🚒': EmojiEntry(codepoint='1F469 1F3FB 200D 1F692', status='fully-qualified', emoji='👩🏻\u200d🚒', name='woman firefighter: light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏼‍🚒': EmojiEntry(codepoint='1F469 1F3FC 200D 1F692', status='fully-qualified', emoji='👩🏼\u200d🚒', name='woman firefighter: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏽‍🚒': EmojiEntry(codepoint='1F469 1F3FD 200D 1F692', status='fully-qualified', emoji='👩🏽\u200d🚒', name='woman firefighter: medium skin tone', group='People & Body', sub_group='person-role'),
    '👩🏾‍🚒': EmojiEntry(codepoint='1F469 1F3FE 200D 1F692', status='fully-qualified', emoji='👩🏾\u200d🚒', name='woman firefighter: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏿‍🚒': EmojiEntry(codepoint='1F469 1F3FF 200D 1F692', status='fully-qualified', emoji='👩🏿\u200d🚒', name='woman firefighter: dark skin tone', group='People & Body', sub_group='person-role'),
    '👮': EmojiEntry(codepoint='1F46E', status='fully-qualified', emoji='👮', name='police officer', group='People & Body', sub_group='person-role'),
    '👮🏻': EmojiEntry(codepoint='1F46E 1F3FB', status='fully-qualified', emoji='👮🏻', name='police officer: light skin tone', group='People & Body', sub_group='person-role'),
    '👮🏼': EmojiEntry(codepoint='1F46E 1F3FC', status='fully-qualified', emoji='👮🏼', name='police officer: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👮🏽': EmojiEntry(codepoint='1F46E 1F3FD', status='fully-qualified', emoji='👮🏽', name='police officer: medium skin tone', group='People & Body', sub_group='person-role'),
    '👮🏾': EmojiEntry(codepoint='1F46E 1F3FE', status='fully-qualified', emoji='👮🏾', name='police officer: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👮🏿': EmojiEntry(codepoint='1F46E 1F3FF', status='fully-qualified', emoji='👮🏿', name='police officer: dark skin tone', group='People & Body', sub_group='person-role'),
    '👮‍♂️': EmojiEntry(codepoint='1F46E 200D 2642 FE0F', status='fully-qualified', emoji='👮\u200d♂️', name='man police officer', group='People & Body', sub_group='person-role'),
    '👮‍♂': EmojiEntry(codepoint='1F46E 200D 2642', status='minimally-qualified', emoji='👮\u200d♂', name='man police officer', group='People & Body', sub_group='person-role'),
    '👮🏻‍♂️': EmojiEntry(codepoint='1F46E 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='👮🏻\u200d♂️', name='man police officer: light skin tone', group='People & Body', sub_group='person-role'),
    '👮🏻‍♂': EmojiEntry(codepoint='1F46E 1F3FB 200D 2642', status='minimally-qualified', emoji='👮🏻\u200d♂', name='man police officer: light skin tone', group='People & Body', sub_group='person-role'),
    '👮🏼‍♂️': EmojiEntry(codepoint='1F46E 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='👮🏼\u200d♂️', name='man police officer: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👮🏼‍♂': EmojiEntry(codepoint='1F46E 1F3FC 200D 2642', status='minimally-qualified', emoji='👮🏼\u200d♂', name='man police officer: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👮🏽‍♂️': EmojiEntry(codepoint='1F46E 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='👮🏽\u200d♂️', name='man police officer: medium skin tone', group='People & Body', sub_group='person-role'),
    '👮🏽‍♂': EmojiEntry(codepoint='1F46E 1F3FD 200D 2642', status='minimally-qualified', emoji='👮🏽\u200d♂', name='man police officer: medium skin tone', group='People & Body', sub_group='person-role'),
    '👮🏾‍♂️': EmojiEntry(codepoint='1F46E 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='👮🏾\u200d♂️', name='man police officer: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👮🏾‍♂': EmojiEntry(codepoint='1F46E 1F3FE 200D 2642', status='minimally-qualified', emoji='👮🏾\u200d♂', name='man police officer: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👮🏿‍♂️': EmojiEntry(codepoint='1F46E 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='👮🏿\u200d♂️', name='man police officer: dark skin tone', group='People & Body', sub_group='person-role'),
    '👮🏿‍♂': EmojiEntry(codepoint='1F46E 1F3FF 200D 2642', status='minimally-qualified', emoji='👮🏿\u200d♂', name='man police officer: dark skin tone', group='People & Body', sub_group='person-role'),
    '👮‍♀️': EmojiEntry(codepoint='1F46E 200D 2640 FE0F', status='fully-qualified', emoji='👮\u200d♀️', name='woman police officer', group='People & Body', sub_group='person-role'),
    '👮‍♀': EmojiEntry(codepoint='1F46E 200D 2640', status='minimally-qualified', emoji='👮\u200d♀', name='woman police officer', group='People & Body', sub_group='person-role'),
    '👮🏻‍♀️': EmojiEntry(codepoint='1F46E 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='👮🏻\u200d♀️', name='woman police officer: light skin tone', group='People & Body', sub_group='person-role'),
    '👮🏻‍♀': EmojiEntry(codepoint='1F46E 1F3FB 200D 2640', status='minimally-qualified', emoji='👮🏻\u200d♀', name='woman police officer: light skin tone', group='People & Body', sub_group='person-role'),
    '👮🏼‍♀️': EmojiEntry(codepoint='1F46E 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='👮🏼\u200d♀️', name='woman police officer: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👮🏼‍♀': EmojiEntry(codepoint='1F46E 1F3FC 200D 2640', status='minimally-qualified', emoji='👮🏼\u200d♀', name='woman police officer: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👮🏽‍♀️': EmojiEntry(codepoint='1F46E 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='👮🏽\u200d♀️', name='woman police officer: medium skin tone', group='People & Body', sub_group='person-role'),
    '👮🏽‍♀': EmojiEntry(codepoint='1F46E 1F3FD 200D 2640', status='minimally-qualified', emoji='👮🏽\u200d♀', name='woman police officer: medium skin tone', group='People & Body', sub_group='person-role'),
    '👮🏾‍♀️': EmojiEntry(codepoint='1F46E 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='👮🏾\u200d♀️', name='woman police officer: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👮🏾‍♀': EmojiEntry(codepoint='1F46E 1F3FE 200D 2640', status='minimally-qualified', emoji='👮🏾\u200d♀', name='woman police officer: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👮🏿‍♀️': EmojiEntry(codepoint='1F46E 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='👮🏿\u200d♀️', name='woman police officer: dark skin tone', group='People & Body', sub_group='person-role'),
    '👮🏿‍♀': EmojiEntry(codepoint='1F46E 1F3FF 200D 2640', status='minimally-qualified', emoji='👮🏿\u200d♀', name='woman police officer: dark skin tone', group='People & Body', sub_group='person-role'),
    '🕵️': EmojiEntry(codepoint='1F575 FE0F', status='fully-qualified', emoji='🕵️', name='detective', group='People & Body', sub_group='person-role'),
    '🕵': EmojiEntry(codepoint='1F575', status='unqualified', emoji='🕵', name='detective', group='People & Body', sub_group='person-role'),
    '🕵🏻': EmojiEntry(codepoint='1F575 1F3FB', status='fully-qualified', emoji='🕵🏻', name='detective: light skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏼': EmojiEntry(codepoint='1F575 1F3FC', status='fully-qualified', emoji='🕵🏼', name='detective: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏽': EmojiEntry(codepoint='1F575 1F3FD', status='fully-qualified', emoji='🕵🏽', name='detective: medium skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏾': EmojiEntry(codepoint='1F575 1F3FE', status='fully-qualified', emoji='🕵🏾', name='detective: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏿': EmojiEntry(codepoint='1F575 1F3FF', status='fully-qualified', emoji='🕵🏿', name='detective: dark skin tone', group='People & Body', sub_group='person-role'),
    '🕵️‍♂️': EmojiEntry(codepoint='1F575 FE0F 200D 2642 FE0F', status='fully-qualified', emoji='🕵️\u200d♂️', name='man detective', group='People & Body', sub_group='person-role'),
    '🕵‍♂️': EmojiEntry(codepoint='1F575 200D 2642 FE0F', status='unqualified', emoji='🕵\u200d♂️', name='man detective', group='People & Body', sub_group='person-role'),
    '🕵️‍♂': EmojiEntry(codepoint='1F575 FE0F 200D 2642', status='unqualified', emoji='🕵️\u200d♂', name='man detective', group='People & Body', sub_group='person-role'),
    '🕵‍♂': EmojiEntry(codepoint='1F575 200D 2642', status='unqualified', emoji='🕵\u200d♂', name='man detective', group='People & Body', sub_group='person-role'),
    '🕵🏻‍♂️': EmojiEntry(codepoint='1F575 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🕵🏻\u200d♂️', name='man detective: light skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏻‍♂': EmojiEntry(codepoint='1F575 1F3FB 200D 2642', status='minimally-qualified', emoji='🕵🏻\u200d♂', name='man detective: light skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏼‍♂️': EmojiEntry(codepoint='1F575 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🕵🏼\u200d♂️', name='man detective: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏼‍♂': EmojiEntry(codepoint='1F575 1F3FC 200D 2642', status='minimally-qualified', emoji='🕵🏼\u200d♂', name='man detective: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏽‍♂️': EmojiEntry(codepoint='1F575 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🕵🏽\u200d♂️', name='man detective: medium skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏽‍♂': EmojiEntry(codepoint='1F575 1F3FD 200D 2642', status='minimally-qualified', emoji='🕵🏽\u200d♂', name='man detective: medium skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏾‍♂️': EmojiEntry(codepoint='1F575 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🕵🏾\u200d♂️', name='man detective: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏾‍♂': EmojiEntry(codepoint='1F575 1F3FE 200D 2642', status='minimally-qualified', emoji='🕵🏾\u200d♂', name='man detective: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏿‍♂️': EmojiEntry(codepoint='1F575 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🕵🏿\u200d♂️', name='man detective: dark skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏿‍♂': EmojiEntry(codepoint='1F575 1F3FF 200D 2642', status='minimally-qualified', emoji='🕵🏿\u200d♂', name='man detective: dark skin tone', group='People & Body', sub_group='person-role'),
    '🕵️‍♀️': EmojiEntry(codepoint='1F575 FE0F 200D 2640 FE0F', status='fully-qualified', emoji='🕵️\u200d♀️', name='woman detective', group='People & Body', sub_group='person-role'),
    '🕵‍♀️': EmojiEntry(codepoint='1F575 200D 2640 FE0F', status='unqualified', emoji='🕵\u200d♀️', name='woman detective', group='People & Body', sub_group='person-role'),
    '🕵️‍♀': EmojiEntry(codepoint='1F575 FE0F 200D 2640', status='unqualified', emoji='🕵️\u200d♀', name='woman detective', group='People & Body', sub_group='person-role'),
    '🕵‍♀': EmojiEntry(codepoint='1F575 200D 2640', status='unqualified', emoji='🕵\u200d♀', name='woman detective', group='People & Body', sub_group='person-role'),
    '🕵🏻‍♀️': EmojiEntry(codepoint='1F575 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🕵🏻\u200d♀️', name='woman detective: light skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏻‍♀': EmojiEntry(codepoint='1F575 1F3FB 200D 2640', status='minimally-qualified', emoji='🕵🏻\u200d♀', name='woman detective: light skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏼‍♀️': EmojiEntry(codepoint='1F575 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🕵🏼\u200d♀️', name='woman detective: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏼‍♀': EmojiEntry(codepoint='1F575 1F3FC 200D 2640', status='minimally-qualified', emoji='🕵🏼\u200d♀', name='woman detective: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏽‍♀️': EmojiEntry(codepoint='1F575 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🕵🏽\u200d♀️', name='woman detective: medium skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏽‍♀': EmojiEntry(codepoint='1F575 1F3FD 200D 2640', status='minimally-qualified', emoji='🕵🏽\u200d♀', name='woman detective: medium skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏾‍♀️': EmojiEntry(codepoint='1F575 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🕵🏾\u200d♀️', name='woman detective: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏾‍♀': EmojiEntry(codepoint='1F575 1F3FE 200D 2640', status='minimally-qualified', emoji='🕵🏾\u200d♀', name='woman detective: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏿‍♀️': EmojiEntry(codepoint='1F575 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🕵🏿\u200d♀️', name='woman detective: dark skin tone', group='People & Body', sub_group='person-role'),
    '🕵🏿‍♀': EmojiEntry(codepoint='1F575 1F3FF 200D 2640', status='minimally-qualified', emoji='🕵🏿\u200d♀', name='woman detective: dark skin tone', group='People & Body', sub_group='person-role'),
    '💂': EmojiEntry(codepoint='1F482', status='fully-qualified', emoji='💂', name='guard', group='People & Body', sub_group='person-role'),
    '💂🏻': EmojiEntry(codepoint='1F482 1F3FB', status='fully-qualified', emoji='💂🏻', name='guard: light skin tone', group='People & Body', sub_group='person-role'),
    '💂🏼': EmojiEntry(codepoint='1F482 1F3FC', status='fully-qualified', emoji='💂🏼', name='guard: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '💂🏽': EmojiEntry(codepoint='1F482 1F3FD', status='fully-qualified', emoji='💂🏽', name='guard: medium skin tone', group='People & Body', sub_group='person-role'),
    '💂🏾': EmojiEntry(codepoint='1F482 1F3FE', status='fully-qualified', emoji='💂🏾', name='guard: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '💂🏿': EmojiEntry(codepoint='1F482 1F3FF', status='fully-qualified', emoji='💂🏿', name='guard: dark skin tone', group='People & Body', sub_group='person-role'),
    '💂‍♂️': EmojiEntry(codepoint='1F482 200D 2642 FE0F', status='fully-qualified', emoji='💂\u200d♂️', name='man guard', group='People & Body', sub_group='person-role'),
    '💂‍♂': EmojiEntry(codepoint='1F482 200D 2642', status='minimally-qualified', emoji='💂\u200d♂', name='man guard', group='People & Body', sub_group='person-role'),
    '💂🏻‍♂️': EmojiEntry(codepoint='1F482 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='💂🏻\u200d♂️', name='man guard: light skin tone', group='People & Body', sub_group='person-role'),
    '💂🏻‍♂': EmojiEntry(codepoint='1F482 1F3FB 200D 2642', status='minimally-qualified', emoji='💂🏻\u200d♂', name='man guard: light skin tone', group='People & Body', sub_group='person-role'),
    '💂🏼‍♂️': EmojiEntry(codepoint='1F482 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='💂🏼\u200d♂️', name='man guard: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '💂🏼‍♂': EmojiEntry(codepoint='1F482 1F3FC 200D 2642', status='minimally-qualified', emoji='💂🏼\u200d♂', name='man guard: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '💂🏽‍♂️': EmojiEntry(codepoint='1F482 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='💂🏽\u200d♂️', name='man guard: medium skin tone', group='People & Body', sub_group='person-role'),
    '💂🏽‍♂': EmojiEntry(codepoint='1F482 1F3FD 200D 2642', status='minimally-qualified', emoji='💂🏽\u200d♂', name='man guard: medium skin tone', group='People & Body', sub_group='person-role'),
    '💂🏾‍♂️': EmojiEntry(codepoint='1F482 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='💂🏾\u200d♂️', name='man guard: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '💂🏾‍♂': EmojiEntry(codepoint='1F482 1F3FE 200D 2642', status='minimally-qualified', emoji='💂🏾\u200d♂', name='man guard: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '💂🏿‍♂️': EmojiEntry(codepoint='1F482 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='💂🏿\u200d♂️', name='man guard: dark skin tone', group='People & Body', sub_group='person-role'),
    '💂🏿‍♂': EmojiEntry(codepoint='1F482 1F3FF 200D 2642', status='minimally-qualified', emoji='💂🏿\u200d♂', name='man guard: dark skin tone', group='People & Body', sub_group='person-role'),
    '💂‍♀️': EmojiEntry(codepoint='1F482 200D 2640 FE0F', status='fully-qualified', emoji='💂\u200d♀️', name='woman guard', group='People & Body', sub_group='person-role'),
    '💂‍♀': EmojiEntry(codepoint='1F482 200D 2640', status='minimally-qualified', emoji='💂\u200d♀', name='woman guard', group='People & Body', sub_group='person-role'),
    '💂🏻‍♀️': EmojiEntry(codepoint='1F482 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='💂🏻\u200d♀️', name='woman guard: light skin tone', group='People & Body', sub_group='person-role'),
    '💂🏻‍♀': EmojiEntry(codepoint='1F482 1F3FB 200D 2640', status='minimally-qualified', emoji='💂🏻\u200d♀', name='woman guard: light skin tone', group='People & Body', sub_group='person-role'),
    '💂🏼‍♀️': EmojiEntry(codepoint='1F482 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='💂🏼\u200d♀️', name='woman guard: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '💂🏼‍♀': EmojiEntry(codepoint='1F482 1F3FC 200D 2640', status='minimally-qualified', emoji='💂🏼\u200d♀', name='woman guard: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '💂🏽‍♀️': EmojiEntry(codepoint='1F482 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='💂🏽\u200d♀️', name='woman guard: medium skin tone', group='People & Body', sub_group='person-role'),
    '💂🏽‍♀': EmojiEntry(codepoint='1F482 1F3FD 200D 2640', status='minimally-qualified', emoji='💂🏽\u200d♀', name='woman guard: medium skin tone', group='People & Body', sub_group='person-role'),
    '💂🏾‍♀️': EmojiEntry(codepoint='1F482 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='💂🏾\u200d♀️', name='woman guard: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '💂🏾‍♀': EmojiEntry(codepoint='1F482 1F3FE 200D 2640', status='minimally-qualified', emoji='💂🏾\u200d♀', name='woman guard: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '💂🏿‍♀️': EmojiEntry(codepoint='1F482 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='💂🏿\u200d♀️', name='woman guard: dark skin tone', group='People & Body', sub_group='person-role'),
    '💂🏿‍♀': EmojiEntry(codepoint='1F482 1F3FF 200D 2640', status='minimally-qualified', emoji='💂🏿\u200d♀', name='woman guard: dark skin tone', group='People & Body', sub_group='person-role'),
    '🥷': EmojiEntry(codepoint='1F977', status='fully-qualified', emoji='🥷', name='ninja', group='People & Body', sub_group='person-role'),
    '🥷🏻': EmojiEntry(codepoint='1F977 1F3FB', status='fully-qualified', emoji='🥷🏻', name='ninja: light skin tone', group='People & Body', sub_group='person-role'),
    '🥷🏼': EmojiEntry(codepoint='1F977 1F3FC', status='fully-qualified', emoji='🥷🏼', name='ninja: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🥷🏽': EmojiEntry(codepoint='1F977 1F3FD', status='fully-qualified', emoji='🥷🏽', name='ninja: medium skin tone', group='People & Body', sub_group='person-role'),
    '🥷🏾': EmojiEntry(codepoint='1F977 1F3FE', status='fully-qualified', emoji='🥷🏾', name='ninja: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🥷🏿': EmojiEntry(codepoint='1F977 1F3FF', status='fully-qualified', emoji='🥷🏿', name='ninja: dark skin tone', group='People & Body', sub_group='person-role'),
    '👷': EmojiEntry(codepoint='1F477', status='fully-qualified', emoji='👷', name='construction worker', group='People & Body', sub_group='person-role'),
    '👷🏻': EmojiEntry(codepoint='1F477 1F3FB', status='fully-qualified', emoji='👷🏻', name='construction worker: light skin tone', group='People & Body', sub_group='person-role'),
    '👷🏼': EmojiEntry(codepoint='1F477 1F3FC', status='fully-qualified', emoji='👷🏼', name='construction worker: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👷🏽': EmojiEntry(codepoint='1F477 1F3FD', status='fully-qualified', emoji='👷🏽', name='construction worker: medium skin tone', group='People & Body', sub_group='person-role'),
    '👷🏾': EmojiEntry(codepoint='1F477 1F3FE', status='fully-qualified', emoji='👷🏾', name='construction worker: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👷🏿': EmojiEntry(codepoint='1F477 1F3FF', status='fully-qualified', emoji='👷🏿', name='construction worker: dark skin tone', group='People & Body', sub_group='person-role'),
    '👷‍♂️': EmojiEntry(codepoint='1F477 200D 2642 FE0F', status='fully-qualified', emoji='👷\u200d♂️', name='man construction worker', group='People & Body', sub_group='person-role'),
    '👷‍♂': EmojiEntry(codepoint='1F477 200D 2642', status='minimally-qualified', emoji='👷\u200d♂', name='man construction worker', group='People & Body', sub_group='person-role'),
    '👷🏻‍♂️': EmojiEntry(codepoint='1F477 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='👷🏻\u200d♂️', name='man construction worker: light skin tone', group='People & Body', sub_group='person-role'),
    '👷🏻‍♂': EmojiEntry(codepoint='1F477 1F3FB 200D 2642', status='minimally-qualified', emoji='👷🏻\u200d♂', name='man construction worker: light skin tone', group='People & Body', sub_group='person-role'),
    '👷🏼‍♂️': EmojiEntry(codepoint='1F477 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='👷🏼\u200d♂️', name='man construction worker: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👷🏼‍♂': EmojiEntry(codepoint='1F477 1F3FC 200D 2642', status='minimally-qualified', emoji='👷🏼\u200d♂', name='man construction worker: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👷🏽‍♂️': EmojiEntry(codepoint='1F477 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='👷🏽\u200d♂️', name='man construction worker: medium skin tone', group='People & Body', sub_group='person-role'),
    '👷🏽‍♂': EmojiEntry(codepoint='1F477 1F3FD 200D 2642', status='minimally-qualified', emoji='👷🏽\u200d♂', name='man construction worker: medium skin tone', group='People & Body', sub_group='person-role'),
    '👷🏾‍♂️': EmojiEntry(codepoint='1F477 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='👷🏾\u200d♂️', name='man construction worker: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👷🏾‍♂': EmojiEntry(codepoint='1F477 1F3FE 200D 2642', status='minimally-qualified', emoji='👷🏾\u200d♂', name='man construction worker: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👷🏿‍♂️': EmojiEntry(codepoint='1F477 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='👷🏿\u200d♂️', name='man construction worker: dark skin tone', group='People & Body', sub_group='person-role'),
    '👷🏿‍♂': EmojiEntry(codepoint='1F477 1F3FF 200D 2642', status='minimally-qualified', emoji='👷🏿\u200d♂', name='man construction worker: dark skin tone', group='People & Body', sub_group='person-role'),
    '👷‍♀️': EmojiEntry(codepoint='1F477 200D 2640 FE0F', status='fully-qualified', emoji='👷\u200d♀️', name='woman construction worker', group='People & Body', sub_group='person-role'),
    '👷‍♀': EmojiEntry(codepoint='1F477 200D 2640', status='minimally-qualified', emoji='👷\u200d♀', name='woman construction worker', group='People & Body', sub_group='person-role'),
    '👷🏻‍♀️': EmojiEntry(codepoint='1F477 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='👷🏻\u200d♀️', name='woman construction worker: light skin tone', group='People & Body', sub_group='person-role'),
    '👷🏻‍♀': EmojiEntry(codepoint='1F477 1F3FB 200D 2640', status='minimally-qualified', emoji='👷🏻\u200d♀', name='woman construction worker: light skin tone', group='People & Body', sub_group='person-role'),
    '👷🏼‍♀️': EmojiEntry(codepoint='1F477 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='👷🏼\u200d♀️', name='woman construction worker: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👷🏼‍♀': EmojiEntry(codepoint='1F477 1F3FC 200D 2640', status='minimally-qualified', emoji='👷🏼\u200d♀', name='woman construction worker: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👷🏽‍♀️': EmojiEntry(codepoint='1F477 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='👷🏽\u200d♀️', name='woman construction worker: medium skin tone', group='People & Body', sub_group='person-role'),
    '👷🏽‍♀': EmojiEntry(codepoint='1F477 1F3FD 200D 2640', status='minimally-qualified', emoji='👷🏽\u200d♀', name='woman construction worker: medium skin tone', group='People & Body', sub_group='person-role'),
    '👷🏾‍♀️': EmojiEntry(codepoint='1F477 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='👷🏾\u200d♀️', name='woman construction worker: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👷🏾‍♀': EmojiEntry(codepoint='1F477 1F3FE 200D 2640', status='minimally-qualified', emoji='👷🏾\u200d♀', name='woman construction worker: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👷🏿‍♀️': EmojiEntry(codepoint='1F477 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='👷🏿\u200d♀️', name='woman construction worker: dark skin tone', group='People & Body', sub_group='person-role'),
    '👷🏿‍♀': EmojiEntry(codepoint='1F477 1F3FF 200D 2640', status='minimally-qualified', emoji='👷🏿\u200d♀', name='woman construction worker: dark skin tone', group='People & Body', sub_group='person-role'),
    '🤴': EmojiEntry(codepoint='1F934', status='fully-qualified', emoji='🤴', name='prince', group='People & Body', sub_group='person-role'),
    '🤴🏻': EmojiEntry(codepoint='1F934 1F3FB', status='fully-qualified', emoji='🤴🏻', name='prince: light skin tone', group='People & Body', sub_group='person-role'),
    '🤴🏼': EmojiEntry(codepoint='1F934 1F3FC', status='fully-qualified', emoji='🤴🏼', name='prince: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🤴🏽': EmojiEntry(codepoint='1F934 1F3FD', status='fully-qualified', emoji='🤴🏽', name='prince: medium skin tone', group='People & Body', sub_group='person-role'),
    '🤴🏾': EmojiEntry(codepoint='1F934 1F3FE', status='fully-qualified', emoji='🤴🏾', name='prince: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🤴🏿': EmojiEntry(codepoint='1F934 1F3FF', status='fully-qualified', emoji='🤴🏿', name='prince: dark skin tone', group='People & Body', sub_group='person-role'),
    '👸': EmojiEntry(codepoint='1F478', status='fully-qualified', emoji='👸', name='princess', group='People & Body', sub_group='person-role'),
    '👸🏻': EmojiEntry(codepoint='1F478 1F3FB', status='fully-qualified', emoji='👸🏻', name='princess: light skin tone', group='People & Body', sub_group='person-role'),
    '👸🏼': EmojiEntry(codepoint='1F478 1F3FC', status='fully-qualified', emoji='👸🏼', name='princess: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👸🏽': EmojiEntry(codepoint='1F478 1F3FD', status='fully-qualified', emoji='👸🏽', name='princess: medium skin tone', group='People & Body', sub_group='person-role'),
    '👸🏾': EmojiEntry(codepoint='1F478 1F3FE', status='fully-qualified', emoji='👸🏾', name='princess: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👸🏿': EmojiEntry(codepoint='1F478 1F3FF', status='fully-qualified', emoji='👸🏿', name='princess: dark skin tone', group='People & Body', sub_group='person-role'),
    '👳': EmojiEntry(codepoint='1F473', status='fully-qualified', emoji='👳', name='person wearing turban', group='People & Body', sub_group='person-role'),
    '👳🏻': EmojiEntry(codepoint='1F473 1F3FB', status='fully-qualified', emoji='👳🏻', name='person wearing turban: light skin tone', group='People & Body', sub_group='person-role'),
    '👳🏼': EmojiEntry(codepoint='1F473 1F3FC', status='fully-qualified', emoji='👳🏼', name='person wearing turban: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👳🏽': EmojiEntry(codepoint='1F473 1F3FD', status='fully-qualified', emoji='👳🏽', name='person wearing turban: medium skin tone', group='People & Body', sub_group='person-role'),
    '👳🏾': EmojiEntry(codepoint='1F473 1F3FE', status='fully-qualified', emoji='👳🏾', name='person wearing turban: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👳🏿': EmojiEntry(codepoint='1F473 1F3FF', status='fully-qualified', emoji='👳🏿', name='person wearing turban: dark skin tone', group='People & Body', sub_group='person-role'),
    '👳‍♂️': EmojiEntry(codepoint='1F473 200D 2642 FE0F', status='fully-qualified', emoji='👳\u200d♂️', name='man wearing turban', group='People & Body', sub_group='person-role'),
    '👳‍♂': EmojiEntry(codepoint='1F473 200D 2642', status='minimally-qualified', emoji='👳\u200d♂', name='man wearing turban', group='People & Body', sub_group='person-role'),
    '👳🏻‍♂️': EmojiEntry(codepoint='1F473 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='👳🏻\u200d♂️', name='man wearing turban: light skin tone', group='People & Body', sub_group='person-role'),
    '👳🏻‍♂': EmojiEntry(codepoint='1F473 1F3FB 200D 2642', status='minimally-qualified', emoji='👳🏻\u200d♂', name='man wearing turban: light skin tone', group='People & Body', sub_group='person-role'),
    '👳🏼‍♂️': EmojiEntry(codepoint='1F473 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='👳🏼\u200d♂️', name='man wearing turban: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👳🏼‍♂': EmojiEntry(codepoint='1F473 1F3FC 200D 2642', status='minimally-qualified', emoji='👳🏼\u200d♂', name='man wearing turban: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👳🏽‍♂️': EmojiEntry(codepoint='1F473 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='👳🏽\u200d♂️', name='man wearing turban: medium skin tone', group='People & Body', sub_group='person-role'),
    '👳🏽‍♂': EmojiEntry(codepoint='1F473 1F3FD 200D 2642', status='minimally-qualified', emoji='👳🏽\u200d♂', name='man wearing turban: medium skin tone', group='People & Body', sub_group='person-role'),
    '👳🏾‍♂️': EmojiEntry(codepoint='1F473 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='👳🏾\u200d♂️', name='man wearing turban: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👳🏾‍♂': EmojiEntry(codepoint='1F473 1F3FE 200D 2642', status='minimally-qualified', emoji='👳🏾\u200d♂', name='man wearing turban: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👳🏿‍♂️': EmojiEntry(codepoint='1F473 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='👳🏿\u200d♂️', name='man wearing turban: dark skin tone', group='People & Body', sub_group='person-role'),
    '👳🏿‍♂': EmojiEntry(codepoint='1F473 1F3FF 200D 2642', status='minimally-qualified', emoji='👳🏿\u200d♂', name='man wearing turban: dark skin tone', group='People & Body', sub_group='person-role'),
    '👳‍♀️': EmojiEntry(codepoint='1F473 200D 2640 FE0F', status='fully-qualified', emoji='👳\u200d♀️', name='woman wearing turban', group='People & Body', sub_group='person-role'),
    '👳‍♀': EmojiEntry(codepoint='1F473 200D 2640', status='minimally-qualified', emoji='👳\u200d♀', name='woman wearing turban', group='People & Body', sub_group='person-role'),
    '👳🏻‍♀️': EmojiEntry(codepoint='1F473 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='👳🏻\u200d♀️', name='woman wearing turban: light skin tone', group='People & Body', sub_group='person-role'),
    '👳🏻‍♀': EmojiEntry(codepoint='1F473 1F3FB 200D 2640', status='minimally-qualified', emoji='👳🏻\u200d♀', name='woman wearing turban: light skin tone', group='People & Body', sub_group='person-role'),
    '👳🏼‍♀️': EmojiEntry(codepoint='1F473 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='👳🏼\u200d♀️', name='woman wearing turban: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👳🏼‍♀': EmojiEntry(codepoint='1F473 1F3FC 200D 2640', status='minimally-qualified', emoji='👳🏼\u200d♀', name='woman wearing turban: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👳🏽‍♀️': EmojiEntry(codepoint='1F473 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='👳🏽\u200d♀️', name='woman wearing turban: medium skin tone', group='People & Body', sub_group='person-role'),
    '👳🏽‍♀': EmojiEntry(codepoint='1F473 1F3FD 200D 2640', status='minimally-qualified', emoji='👳🏽\u200d♀', name='woman wearing turban: medium skin tone', group='People & Body', sub_group='person-role'),
    '👳🏾‍♀️': EmojiEntry(codepoint='1F473 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='👳🏾\u200d♀️', name='woman wearing turban: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👳🏾‍♀': EmojiEntry(codepoint='1F473 1F3FE 200D 2640', status='minimally-qualified', emoji='👳🏾\u200d♀', name='woman wearing turban: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👳🏿‍♀️': EmojiEntry(codepoint='1F473 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='👳🏿\u200d♀️', name='woman wearing turban: dark skin tone', group='People & Body', sub_group='person-role'),
    '👳🏿‍♀': EmojiEntry(codepoint='1F473 1F3FF 200D 2640', status='minimally-qualified', emoji='👳🏿\u200d♀', name='woman wearing turban: dark skin tone', group='People & Body', sub_group='person-role'),
    '👲': EmojiEntry(codepoint='1F472', status='fully-qualified', emoji='👲', name='person with skullcap', group='People & Body', sub_group='person-role'),
    '👲🏻': EmojiEntry(codepoint='1F472 1F3FB', status='fully-qualified', emoji='👲🏻', name='person with skullcap: light skin tone', group='People & Body', sub_group='person-role'),
    '👲🏼': EmojiEntry(codepoint='1F472 1F3FC', status='fully-qualified', emoji='👲🏼', name='person with skullcap: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👲🏽': EmojiEntry(codepoint='1F472 1F3FD', status='fully-qualified', emoji='👲🏽', name='person with skullcap: medium skin tone', group='People & Body', sub_group='person-role'),
    '👲🏾': EmojiEntry(codepoint='1F472 1F3FE', status='fully-qualified', emoji='👲🏾', name='person with skullcap: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👲🏿': EmojiEntry(codepoint='1F472 1F3FF', status='fully-qualified', emoji='👲🏿', name='person with skullcap: dark skin tone', group='People & Body', sub_group='person-role'),
    '🧕': EmojiEntry(codepoint='1F9D5', status='fully-qualified', emoji='🧕', name='woman with headscarf', group='People & Body', sub_group='person-role'),
    '🧕🏻': EmojiEntry(codepoint='1F9D5 1F3FB', status='fully-qualified', emoji='🧕🏻', name='woman with headscarf: light skin tone', group='People & Body', sub_group='person-role'),
    '🧕🏼': EmojiEntry(codepoint='1F9D5 1F3FC', status='fully-qualified', emoji='🧕🏼', name='woman with headscarf: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧕🏽': EmojiEntry(codepoint='1F9D5 1F3FD', status='fully-qualified', emoji='🧕🏽', name='woman with headscarf: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧕🏾': EmojiEntry(codepoint='1F9D5 1F3FE', status='fully-qualified', emoji='🧕🏾', name='woman with headscarf: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧕🏿': EmojiEntry(codepoint='1F9D5 1F3FF', status='fully-qualified', emoji='🧕🏿', name='woman with headscarf: dark skin tone', group='People & Body', sub_group='person-role'),
    '🤵': EmojiEntry(codepoint='1F935', status='fully-qualified', emoji='🤵', name='person in tuxedo', group='People & Body', sub_group='person-role'),
    '🤵🏻': EmojiEntry(codepoint='1F935 1F3FB', status='fully-qualified', emoji='🤵🏻', name='person in tuxedo: light skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏼': EmojiEntry(codepoint='1F935 1F3FC', status='fully-qualified', emoji='🤵🏼', name='person in tuxedo: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏽': EmojiEntry(codepoint='1F935 1F3FD', status='fully-qualified', emoji='🤵🏽', name='person in tuxedo: medium skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏾': EmojiEntry(codepoint='1F935 1F3FE', status='fully-qualified', emoji='🤵🏾', name='person in tuxedo: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏿': EmojiEntry(codepoint='1F935 1F3FF', status='fully-qualified', emoji='🤵🏿', name='person in tuxedo: dark skin tone', group='People & Body', sub_group='person-role'),
    '🤵‍♂️': EmojiEntry(codepoint='1F935 200D 2642 FE0F', status='fully-qualified', emoji='🤵\u200d♂️', name='man in tuxedo', group='People & Body', sub_group='person-role'),
    '🤵‍♂': EmojiEntry(codepoint='1F935 200D 2642', status='minimally-qualified', emoji='🤵\u200d♂', name='man in tuxedo', group='People & Body', sub_group='person-role'),
    '🤵🏻‍♂️': EmojiEntry(codepoint='1F935 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🤵🏻\u200d♂️', name='man in tuxedo: light skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏻‍♂': EmojiEntry(codepoint='1F935 1F3FB 200D 2642', status='minimally-qualified', emoji='🤵🏻\u200d♂', name='man in tuxedo: light skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏼‍♂️': EmojiEntry(codepoint='1F935 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🤵🏼\u200d♂️', name='man in tuxedo: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏼‍♂': EmojiEntry(codepoint='1F935 1F3FC 200D 2642', status='minimally-qualified', emoji='🤵🏼\u200d♂', name='man in tuxedo: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏽‍♂️': EmojiEntry(codepoint='1F935 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🤵🏽\u200d♂️', name='man in tuxedo: medium skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏽‍♂': EmojiEntry(codepoint='1F935 1F3FD 200D 2642', status='minimally-qualified', emoji='🤵🏽\u200d♂', name='man in tuxedo: medium skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏾‍♂️': EmojiEntry(codepoint='1F935 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🤵🏾\u200d♂️', name='man in tuxedo: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏾‍♂': EmojiEntry(codepoint='1F935 1F3FE 200D 2642', status='minimally-qualified', emoji='🤵🏾\u200d♂', name='man in tuxedo: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏿‍♂️': EmojiEntry(codepoint='1F935 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🤵🏿\u200d♂️', name='man in tuxedo: dark skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏿‍♂': EmojiEntry(codepoint='1F935 1F3FF 200D 2642', status='minimally-qualified', emoji='🤵🏿\u200d♂', name='man in tuxedo: dark skin tone', group='People & Body', sub_group='person-role'),
    '🤵‍♀️': EmojiEntry(codepoint='1F935 200D 2640 FE0F', status='fully-qualified', emoji='🤵\u200d♀️', name='woman in tuxedo', group='People & Body', sub_group='person-role'),
    '🤵‍♀': EmojiEntry(codepoint='1F935 200D 2640', status='minimally-qualified', emoji='🤵\u200d♀', name='woman in tuxedo', group='People & Body', sub_group='person-role'),
    '🤵🏻‍♀️': EmojiEntry(codepoint='1F935 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🤵🏻\u200d♀️', name='woman in tuxedo: light skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏻‍♀': EmojiEntry(codepoint='1F935 1F3FB 200D 2640', status='minimally-qualified', emoji='🤵🏻\u200d♀', name='woman in tuxedo: light skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏼‍♀️': EmojiEntry(codepoint='1F935 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🤵🏼\u200d♀️', name='woman in tuxedo: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏼‍♀': EmojiEntry(codepoint='1F935 1F3FC 200D 2640', status='minimally-qualified', emoji='🤵🏼\u200d♀', name='woman in tuxedo: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏽‍♀️': EmojiEntry(codepoint='1F935 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🤵🏽\u200d♀️', name='woman in tuxedo: medium skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏽‍♀': EmojiEntry(codepoint='1F935 1F3FD 200D 2640', status='minimally-qualified', emoji='🤵🏽\u200d♀', name='woman in tuxedo: medium skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏾‍♀️': EmojiEntry(codepoint='1F935 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🤵🏾\u200d♀️', name='woman in tuxedo: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏾‍♀': EmojiEntry(codepoint='1F935 1F3FE 200D 2640', status='minimally-qualified', emoji='🤵🏾\u200d♀', name='woman in tuxedo: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏿‍♀️': EmojiEntry(codepoint='1F935 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🤵🏿\u200d♀️', name='woman in tuxedo: dark skin tone', group='People & Body', sub_group='person-role'),
    '🤵🏿‍♀': EmojiEntry(codepoint='1F935 1F3FF 200D 2640', status='minimally-qualified', emoji='🤵🏿\u200d♀', name='woman in tuxedo: dark skin tone', group='People & Body', sub_group='person-role'),
    '👰': EmojiEntry(codepoint='1F470', status='fully-qualified', emoji='👰', name='person with veil', group='People & Body', sub_group='person-role'),
    '👰🏻': EmojiEntry(codepoint='1F470 1F3FB', status='fully-qualified', emoji='👰🏻', name='person with veil: light skin tone', group='People & Body', sub_group='person-role'),
    '👰🏼': EmojiEntry(codepoint='1F470 1F3FC', status='fully-qualified', emoji='👰🏼', name='person with veil: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👰🏽': EmojiEntry(codepoint='1F470 1F3FD', status='fully-qualified', emoji='👰🏽', name='person with veil: medium skin tone', group='People & Body', sub_group='person-role'),
    '👰🏾': EmojiEntry(codepoint='1F470 1F3FE', status='fully-qualified', emoji='👰🏾', name='person with veil: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👰🏿': EmojiEntry(codepoint='1F470 1F3FF', status='fully-qualified', emoji='👰🏿', name='person with veil: dark skin tone', group='People & Body', sub_group='person-role'),
    '👰‍♂️': EmojiEntry(codepoint='1F470 200D 2642 FE0F', status='fully-qualified', emoji='👰\u200d♂️', name='man with veil', group='People & Body', sub_group='person-role'),
    '👰‍♂': EmojiEntry(codepoint='1F470 200D 2642', status='minimally-qualified', emoji='👰\u200d♂', name='man with veil', group='People & Body', sub_group='person-role'),
    '👰🏻‍♂️': EmojiEntry(codepoint='1F470 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='👰🏻\u200d♂️', name='man with veil: light skin tone', group='People & Body', sub_group='person-role'),
    '👰🏻‍♂': EmojiEntry(codepoint='1F470 1F3FB 200D 2642', status='minimally-qualified', emoji='👰🏻\u200d♂', name='man with veil: light skin tone', group='People & Body', sub_group='person-role'),
    '👰🏼‍♂️': EmojiEntry(codepoint='1F470 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='👰🏼\u200d♂️', name='man with veil: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👰🏼‍♂': EmojiEntry(codepoint='1F470 1F3FC 200D 2642', status='minimally-qualified', emoji='👰🏼\u200d♂', name='man with veil: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👰🏽‍♂️': EmojiEntry(codepoint='1F470 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='👰🏽\u200d♂️', name='man with veil: medium skin tone', group='People & Body', sub_group='person-role'),
    '👰🏽‍♂': EmojiEntry(codepoint='1F470 1F3FD 200D 2642', status='minimally-qualified', emoji='👰🏽\u200d♂', name='man with veil: medium skin tone', group='People & Body', sub_group='person-role'),
    '👰🏾‍♂️': EmojiEntry(codepoint='1F470 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='👰🏾\u200d♂️', name='man with veil: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👰🏾‍♂': EmojiEntry(codepoint='1F470 1F3FE 200D 2642', status='minimally-qualified', emoji='👰🏾\u200d♂', name='man with veil: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👰🏿‍♂️': EmojiEntry(codepoint='1F470 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='👰🏿\u200d♂️', name='man with veil: dark skin tone', group='People & Body', sub_group='person-role'),
    '👰🏿‍♂': EmojiEntry(codepoint='1F470 1F3FF 200D 2642', status='minimally-qualified', emoji='👰🏿\u200d♂', name='man with veil: dark skin tone', group='People & Body', sub_group='person-role'),
    '👰‍♀️': EmojiEntry(codepoint='1F470 200D 2640 FE0F', status='fully-qualified', emoji='👰\u200d♀️', name='woman with veil', group='People & Body', sub_group='person-role'),
    '👰‍♀': EmojiEntry(codepoint='1F470 200D 2640', status='minimally-qualified', emoji='👰\u200d♀', name='woman with veil', group='People & Body', sub_group='person-role'),
    '👰🏻‍♀️': EmojiEntry(codepoint='1F470 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='👰🏻\u200d♀️', name='woman with veil: light skin tone', group='People & Body', sub_group='person-role'),
    '👰🏻‍♀': EmojiEntry(codepoint='1F470 1F3FB 200D 2640', status='minimally-qualified', emoji='👰🏻\u200d♀', name='woman with veil: light skin tone', group='People & Body', sub_group='person-role'),
    '👰🏼‍♀️': EmojiEntry(codepoint='1F470 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='👰🏼\u200d♀️', name='woman with veil: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👰🏼‍♀': EmojiEntry(codepoint='1F470 1F3FC 200D 2640', status='minimally-qualified', emoji='👰🏼\u200d♀', name='woman with veil: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👰🏽‍♀️': EmojiEntry(codepoint='1F470 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='👰🏽\u200d♀️', name='woman with veil: medium skin tone', group='People & Body', sub_group='person-role'),
    '👰🏽‍♀': EmojiEntry(codepoint='1F470 1F3FD 200D 2640', status='minimally-qualified', emoji='👰🏽\u200d♀', name='woman with veil: medium skin tone', group='People & Body', sub_group='person-role'),
    '👰🏾‍♀️': EmojiEntry(codepoint='1F470 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='👰🏾\u200d♀️', name='woman with veil: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👰🏾‍♀': EmojiEntry(codepoint='1F470 1F3FE 200D 2640', status='minimally-qualified', emoji='👰🏾\u200d♀', name='woman with veil: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👰🏿‍♀️': EmojiEntry(codepoint='1F470 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='👰🏿\u200d♀️', name='woman with veil: dark skin tone', group='People & Body', sub_group='person-role'),
    '👰🏿‍♀': EmojiEntry(codepoint='1F470 1F3FF 200D 2640', status='minimally-qualified', emoji='👰🏿\u200d♀', name='woman with veil: dark skin tone', group='People & Body', sub_group='person-role'),
    '🤰': EmojiEntry(codepoint='1F930', status='fully-qualified', emoji='🤰', name='pregnant woman', group='People & Body', sub_group='person-role'),
    '🤰🏻': EmojiEntry(codepoint='1F930 1F3FB', status='fully-qualified', emoji='🤰🏻', name='pregnant woman: light skin tone', group='People & Body', sub_group='person-role'),
    '🤰🏼': EmojiEntry(codepoint='1F930 1F3FC', status='fully-qualified', emoji='🤰🏼', name='pregnant woman: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🤰🏽': EmojiEntry(codepoint='1F930 1F3FD', status='fully-qualified', emoji='🤰🏽', name='pregnant woman: medium skin tone', group='People & Body', sub_group='person-role'),
    '🤰🏾': EmojiEntry(codepoint='1F930 1F3FE', status='fully-qualified', emoji='🤰🏾', name='pregnant woman: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🤰🏿': EmojiEntry(codepoint='1F930 1F3FF', status='fully-qualified', emoji='🤰🏿', name='pregnant woman: dark skin tone', group='People & Body', sub_group='person-role'),
    '🤱': EmojiEntry(codepoint='1F931', status='fully-qualified', emoji='🤱', name='breast-feeding', group='People & Body', sub_group='person-role'),
    '🤱🏻': EmojiEntry(codepoint='1F931 1F3FB', status='fully-qualified', emoji='🤱🏻', name='breast-feeding: light skin tone', group='People & Body', sub_group='person-role'),
    '🤱🏼': EmojiEntry(codepoint='1F931 1F3FC', status='fully-qualified', emoji='🤱🏼', name='breast-feeding: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🤱🏽': EmojiEntry(codepoint='1F931 1F3FD', status='fully-qualified', emoji='🤱🏽', name='breast-feeding: medium skin tone', group='People & Body', sub_group='person-role'),
    '🤱🏾': EmojiEntry(codepoint='1F931 1F3FE', status='fully-qualified', emoji='🤱🏾', name='breast-feeding: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🤱🏿': EmojiEntry(codepoint='1F931 1F3FF', status='fully-qualified', emoji='🤱🏿', name='breast-feeding: dark skin tone', group='People & Body', sub_group='person-role'),
    '👩‍🍼': EmojiEntry(codepoint='1F469 200D 1F37C', status='fully-qualified', emoji='👩\u200d🍼', name='woman feeding baby', group='People & Body', sub_group='person-role'),
    '👩🏻‍🍼': EmojiEntry(codepoint='1F469 1F3FB 200D 1F37C', status='fully-qualified', emoji='👩🏻\u200d🍼', name='woman feeding baby: light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏼‍🍼': EmojiEntry(codepoint='1F469 1F3FC 200D 1F37C', status='fully-qualified', emoji='👩🏼\u200d🍼', name='woman feeding baby: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👩🏽‍🍼': EmojiEntry(codepoint='1F469 1F3FD 200D 1F37C', status='fully-qualified', emoji='👩🏽\u200d🍼', name='woman feeding baby: medium skin tone', group='People & Body', sub_group='person-role'),
    '👩🏾‍🍼': EmojiEntry(codepoint='1F469 1F3FE 200D 1F37C', status='fully-qualified', emoji='👩🏾\u200d🍼', name='woman feeding baby: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👩🏿‍🍼': EmojiEntry(codepoint='1F469 1F3FF 200D 1F37C', status='fully-qualified', emoji='👩🏿\u200d🍼', name='woman feeding baby: dark skin tone', group='People & Body', sub_group='person-role'),
    '👨‍🍼': EmojiEntry(codepoint='1F468 200D 1F37C', status='fully-qualified', emoji='👨\u200d🍼', name='man feeding baby', group='People & Body', sub_group='person-role'),
    '👨🏻‍🍼': EmojiEntry(codepoint='1F468 1F3FB 200D 1F37C', status='fully-qualified', emoji='👨🏻\u200d🍼', name='man feeding baby: light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏼‍🍼': EmojiEntry(codepoint='1F468 1F3FC 200D 1F37C', status='fully-qualified', emoji='👨🏼\u200d🍼', name='man feeding baby: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '👨🏽‍🍼': EmojiEntry(codepoint='1F468 1F3FD 200D 1F37C', status='fully-qualified', emoji='👨🏽\u200d🍼', name='man feeding baby: medium skin tone', group='People & Body', sub_group='person-role'),
    '👨🏾‍🍼': EmojiEntry(codepoint='1F468 1F3FE 200D 1F37C', status='fully-qualified', emoji='👨🏾\u200d🍼', name='man feeding baby: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '👨🏿‍🍼': EmojiEntry(codepoint='1F468 1F3FF 200D 1F37C', status='fully-qualified', emoji='👨🏿\u200d🍼', name='man feeding baby: dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑‍🍼': EmojiEntry(codepoint='1F9D1 200D 1F37C', status='fully-qualified', emoji='🧑\u200d🍼', name='person feeding baby', group='People & Body', sub_group='person-role'),
    '🧑🏻‍🍼': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F37C', status='fully-qualified', emoji='🧑🏻\u200d🍼', name='person feeding baby: light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏼‍🍼': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F37C', status='fully-qualified', emoji='🧑🏼\u200d🍼', name='person feeding baby: medium-light skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏽‍🍼': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F37C', status='fully-qualified', emoji='🧑🏽\u200d🍼', name='person feeding baby: medium skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏾‍🍼': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F37C', status='fully-qualified', emoji='🧑🏾\u200d🍼', name='person feeding baby: medium-dark skin tone', group='People & Body', sub_group='person-role'),
    '🧑🏿‍🍼': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F37C', status='fully-qualified', emoji='🧑🏿\u200d🍼', name='person feeding baby: dark skin tone', group='People & Body', sub_group='person-role'),
    '👼': EmojiEntry(codepoint='1F47C', status='fully-qualified', emoji='👼', name='baby angel', group='People & Body', sub_group='person-fantasy'),
    '👼🏻': EmojiEntry(codepoint='1F47C 1F3FB', status='fully-qualified', emoji='👼🏻', name='baby angel: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '👼🏼': EmojiEntry(codepoint='1F47C 1F3FC', status='fully-qualified', emoji='👼🏼', name='baby angel: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '👼🏽': EmojiEntry(codepoint='1F47C 1F3FD', status='fully-qualified', emoji='👼🏽', name='baby angel: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '👼🏾': EmojiEntry(codepoint='1F47C 1F3FE', status='fully-qualified', emoji='👼🏾', name='baby angel: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '👼🏿': EmojiEntry(codepoint='1F47C 1F3FF', status='fully-qualified', emoji='👼🏿', name='baby angel: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🎅': EmojiEntry(codepoint='1F385', status='fully-qualified', emoji='🎅', name='Santa Claus', group='People & Body', sub_group='person-fantasy'),
    '🎅🏻': EmojiEntry(codepoint='1F385 1F3FB', status='fully-qualified', emoji='🎅🏻', name='Santa Claus: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🎅🏼': EmojiEntry(codepoint='1F385 1F3FC', status='fully-qualified', emoji='🎅🏼', name='Santa Claus: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🎅🏽': EmojiEntry(codepoint='1F385 1F3FD', status='fully-qualified', emoji='🎅🏽', name='Santa Claus: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🎅🏾': EmojiEntry(codepoint='1F385 1F3FE', status='fully-qualified', emoji='🎅🏾', name='Santa Claus: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🎅🏿': EmojiEntry(codepoint='1F385 1F3FF', status='fully-qualified', emoji='🎅🏿', name='Santa Claus: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🤶': EmojiEntry(codepoint='1F936', status='fully-qualified', emoji='🤶', name='Mrs. Claus', group='People & Body', sub_group='person-fantasy'),
    '🤶🏻': EmojiEntry(codepoint='1F936 1F3FB', status='fully-qualified', emoji='🤶🏻', name='Mrs. Claus: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🤶🏼': EmojiEntry(codepoint='1F936 1F3FC', status='fully-qualified', emoji='🤶🏼', name='Mrs. Claus: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🤶🏽': EmojiEntry(codepoint='1F936 1F3FD', status='fully-qualified', emoji='🤶🏽', name='Mrs. Claus: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🤶🏾': EmojiEntry(codepoint='1F936 1F3FE', status='fully-qualified', emoji='🤶🏾', name='Mrs. Claus: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🤶🏿': EmojiEntry(codepoint='1F936 1F3FF', status='fully-qualified', emoji='🤶🏿', name='Mrs. Claus: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧑‍🎄': EmojiEntry(codepoint='1F9D1 200D 1F384', status='fully-qualified', emoji='🧑\u200d🎄', name='mx claus', group='People & Body', sub_group='person-fantasy'),
    '🧑🏻‍🎄': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F384', status='fully-qualified', emoji='🧑🏻\u200d🎄', name='mx claus: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧑🏼‍🎄': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F384', status='fully-qualified', emoji='🧑🏼\u200d🎄', name='mx claus: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧑🏽‍🎄': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F384', status='fully-qualified', emoji='🧑🏽\u200d🎄', name='mx claus: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧑🏾‍🎄': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F384', status='fully-qualified', emoji='🧑🏾\u200d🎄', name='mx claus: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧑🏿‍🎄': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F384', status='fully-qualified', emoji='🧑🏿\u200d🎄', name='mx claus: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸': EmojiEntry(codepoint='1F9B8', status='fully-qualified', emoji='🦸', name='superhero', group='People & Body', sub_group='person-fantasy'),
    '🦸🏻': EmojiEntry(codepoint='1F9B8 1F3FB', status='fully-qualified', emoji='🦸🏻', name='superhero: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏼': EmojiEntry(codepoint='1F9B8 1F3FC', status='fully-qualified', emoji='🦸🏼', name='superhero: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏽': EmojiEntry(codepoint='1F9B8 1F3FD', status='fully-qualified', emoji='🦸🏽', name='superhero: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏾': EmojiEntry(codepoint='1F9B8 1F3FE', status='fully-qualified', emoji='🦸🏾', name='superhero: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏿': EmojiEntry(codepoint='1F9B8 1F3FF', status='fully-qualified', emoji='🦸🏿', name='superhero: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸‍♂️': EmojiEntry(codepoint='1F9B8 200D 2642 FE0F', status='fully-qualified', emoji='🦸\u200d♂️', name='man superhero', group='People & Body', sub_group='person-fantasy'),
    '🦸‍♂': EmojiEntry(codepoint='1F9B8 200D 2642', status='minimally-qualified', emoji='🦸\u200d♂', name='man superhero', group='People & Body', sub_group='person-fantasy'),
    '🦸🏻‍♂️': EmojiEntry(codepoint='1F9B8 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🦸🏻\u200d♂️', name='man superhero: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏻‍♂': EmojiEntry(codepoint='1F9B8 1F3FB 200D 2642', status='minimally-qualified', emoji='🦸🏻\u200d♂', name='man superhero: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏼‍♂️': EmojiEntry(codepoint='1F9B8 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🦸🏼\u200d♂️', name='man superhero: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏼‍♂': EmojiEntry(codepoint='1F9B8 1F3FC 200D 2642', status='minimally-qualified', emoji='🦸🏼\u200d♂', name='man superhero: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏽‍♂️': EmojiEntry(codepoint='1F9B8 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🦸🏽\u200d♂️', name='man superhero: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏽‍♂': EmojiEntry(codepoint='1F9B8 1F3FD 200D 2642', status='minimally-qualified', emoji='🦸🏽\u200d♂', name='man superhero: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏾‍♂️': EmojiEntry(codepoint='1F9B8 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🦸🏾\u200d♂️', name='man superhero: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏾‍♂': EmojiEntry(codepoint='1F9B8 1F3FE 200D 2642', status='minimally-qualified', emoji='🦸🏾\u200d♂', name='man superhero: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏿‍♂️': EmojiEntry(codepoint='1F9B8 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🦸🏿\u200d♂️', name='man superhero: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏿‍♂': EmojiEntry(codepoint='1F9B8 1F3FF 200D 2642', status='minimally-qualified', emoji='🦸🏿\u200d♂', name='man superhero: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸‍♀️': EmojiEntry(codepoint='1F9B8 200D 2640 FE0F', status='fully-qualified', emoji='🦸\u200d♀️', name='woman superhero', group='People & Body', sub_group='person-fantasy'),
    '🦸‍♀': EmojiEntry(codepoint='1F9B8 200D 2640', status='minimally-qualified', emoji='🦸\u200d♀', name='woman superhero', group='People & Body', sub_group='person-fantasy'),
    '🦸🏻‍♀️': EmojiEntry(codepoint='1F9B8 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🦸🏻\u200d♀️', name='woman superhero: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏻‍♀': EmojiEntry(codepoint='1F9B8 1F3FB 200D 2640', status='minimally-qualified', emoji='🦸🏻\u200d♀', name='woman superhero: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏼‍♀️': EmojiEntry(codepoint='1F9B8 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🦸🏼\u200d♀️', name='woman superhero: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏼‍♀': EmojiEntry(codepoint='1F9B8 1F3FC 200D 2640', status='minimally-qualified', emoji='🦸🏼\u200d♀', name='woman superhero: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏽‍♀️': EmojiEntry(codepoint='1F9B8 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🦸🏽\u200d♀️', name='woman superhero: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏽‍♀': EmojiEntry(codepoint='1F9B8 1F3FD 200D 2640', status='minimally-qualified', emoji='🦸🏽\u200d♀', name='woman superhero: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏾‍♀️': EmojiEntry(codepoint='1F9B8 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🦸🏾\u200d♀️', name='woman superhero: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏾‍♀': EmojiEntry(codepoint='1F9B8 1F3FE 200D 2640', status='minimally-qualified', emoji='🦸🏾\u200d♀', name='woman superhero: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏿‍♀️': EmojiEntry(codepoint='1F9B8 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🦸🏿\u200d♀️', name='woman superhero: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦸🏿‍♀': EmojiEntry(codepoint='1F9B8 1F3FF 200D 2640', status='minimally-qualified', emoji='🦸🏿\u200d♀', name='woman superhero: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹': EmojiEntry(codepoint='1F9B9', status='fully-qualified', emoji='🦹', name='supervillain', group='People & Body', sub_group='person-fantasy'),
    '🦹🏻': EmojiEntry(codepoint='1F9B9 1F3FB', status='fully-qualified', emoji='🦹🏻', name='supervillain: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏼': EmojiEntry(codepoint='1F9B9 1F3FC', status='fully-qualified', emoji='🦹🏼', name='supervillain: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏽': EmojiEntry(codepoint='1F9B9 1F3FD', status='fully-qualified', emoji='🦹🏽', name='supervillain: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏾': EmojiEntry(codepoint='1F9B9 1F3FE', status='fully-qualified', emoji='🦹🏾', name='supervillain: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏿': EmojiEntry(codepoint='1F9B9 1F3FF', status='fully-qualified', emoji='🦹🏿', name='supervillain: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹‍♂️': EmojiEntry(codepoint='1F9B9 200D 2642 FE0F', status='fully-qualified', emoji='🦹\u200d♂️', name='man supervillain', group='People & Body', sub_group='person-fantasy'),
    '🦹‍♂': EmojiEntry(codepoint='1F9B9 200D 2642', status='minimally-qualified', emoji='🦹\u200d♂', name='man supervillain', group='People & Body', sub_group='person-fantasy'),
    '🦹🏻‍♂️': EmojiEntry(codepoint='1F9B9 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🦹🏻\u200d♂️', name='man supervillain: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏻‍♂': EmojiEntry(codepoint='1F9B9 1F3FB 200D 2642', status='minimally-qualified', emoji='🦹🏻\u200d♂', name='man supervillain: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏼‍♂️': EmojiEntry(codepoint='1F9B9 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🦹🏼\u200d♂️', name='man supervillain: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏼‍♂': EmojiEntry(codepoint='1F9B9 1F3FC 200D 2642', status='minimally-qualified', emoji='🦹🏼\u200d♂', name='man supervillain: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏽‍♂️': EmojiEntry(codepoint='1F9B9 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🦹🏽\u200d♂️', name='man supervillain: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏽‍♂': EmojiEntry(codepoint='1F9B9 1F3FD 200D 2642', status='minimally-qualified', emoji='🦹🏽\u200d♂', name='man supervillain: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏾‍♂️': EmojiEntry(codepoint='1F9B9 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🦹🏾\u200d♂️', name='man supervillain: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏾‍♂': EmojiEntry(codepoint='1F9B9 1F3FE 200D 2642', status='minimally-qualified', emoji='🦹🏾\u200d♂', name='man supervillain: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏿‍♂️': EmojiEntry(codepoint='1F9B9 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🦹🏿\u200d♂️', name='man supervillain: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏿‍♂': EmojiEntry(codepoint='1F9B9 1F3FF 200D 2642', status='minimally-qualified', emoji='🦹🏿\u200d♂', name='man supervillain: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹‍♀️': EmojiEntry(codepoint='1F9B9 200D 2640 FE0F', status='fully-qualified', emoji='🦹\u200d♀️', name='woman supervillain', group='People & Body', sub_group='person-fantasy'),
    '🦹‍♀': EmojiEntry(codepoint='1F9B9 200D 2640', status='minimally-qualified', emoji='🦹\u200d♀', name='woman supervillain', group='People & Body', sub_group='person-fantasy'),
    '🦹🏻‍♀️': EmojiEntry(codepoint='1F9B9 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🦹🏻\u200d♀️', name='woman supervillain: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏻‍♀': EmojiEntry(codepoint='1F9B9 1F3FB 200D 2640', status='minimally-qualified', emoji='🦹🏻\u200d♀', name='woman supervillain: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏼‍♀️': EmojiEntry(codepoint='1F9B9 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🦹🏼\u200d♀️', name='woman supervillain: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏼‍♀': EmojiEntry(codepoint='1F9B9 1F3FC 200D 2640', status='minimally-qualified', emoji='🦹🏼\u200d♀', name='woman supervillain: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏽‍♀️': EmojiEntry(codepoint='1F9B9 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🦹🏽\u200d♀️', name='woman supervillain: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏽‍♀': EmojiEntry(codepoint='1F9B9 1F3FD 200D 2640', status='minimally-qualified', emoji='🦹🏽\u200d♀', name='woman supervillain: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏾‍♀️': EmojiEntry(codepoint='1F9B9 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🦹🏾\u200d♀️', name='woman supervillain: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏾‍♀': EmojiEntry(codepoint='1F9B9 1F3FE 200D 2640', status='minimally-qualified', emoji='🦹🏾\u200d♀', name='woman supervillain: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏿‍♀️': EmojiEntry(codepoint='1F9B9 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🦹🏿\u200d♀️', name='woman supervillain: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🦹🏿‍♀': EmojiEntry(codepoint='1F9B9 1F3FF 200D 2640', status='minimally-qualified', emoji='🦹🏿\u200d♀', name='woman supervillain: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙': EmojiEntry(codepoint='1F9D9', status='fully-qualified', emoji='🧙', name='mage', group='People & Body', sub_group='person-fantasy'),
    '🧙🏻': EmojiEntry(codepoint='1F9D9 1F3FB', status='fully-qualified', emoji='🧙🏻', name='mage: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏼': EmojiEntry(codepoint='1F9D9 1F3FC', status='fully-qualified', emoji='🧙🏼', name='mage: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏽': EmojiEntry(codepoint='1F9D9 1F3FD', status='fully-qualified', emoji='🧙🏽', name='mage: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏾': EmojiEntry(codepoint='1F9D9 1F3FE', status='fully-qualified', emoji='🧙🏾', name='mage: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏿': EmojiEntry(codepoint='1F9D9 1F3FF', status='fully-qualified', emoji='🧙🏿', name='mage: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙‍♂️': EmojiEntry(codepoint='1F9D9 200D 2642 FE0F', status='fully-qualified', emoji='🧙\u200d♂️', name='man mage', group='People & Body', sub_group='person-fantasy'),
    '🧙‍♂': EmojiEntry(codepoint='1F9D9 200D 2642', status='minimally-qualified', emoji='🧙\u200d♂', name='man mage', group='People & Body', sub_group='person-fantasy'),
    '🧙🏻‍♂️': EmojiEntry(codepoint='1F9D9 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🧙🏻\u200d♂️', name='man mage: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏻‍♂': EmojiEntry(codepoint='1F9D9 1F3FB 200D 2642', status='minimally-qualified', emoji='🧙🏻\u200d♂', name='man mage: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏼‍♂️': EmojiEntry(codepoint='1F9D9 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🧙🏼\u200d♂️', name='man mage: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏼‍♂': EmojiEntry(codepoint='1F9D9 1F3FC 200D 2642', status='minimally-qualified', emoji='🧙🏼\u200d♂', name='man mage: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏽‍♂️': EmojiEntry(codepoint='1F9D9 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🧙🏽\u200d♂️', name='man mage: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏽‍♂': EmojiEntry(codepoint='1F9D9 1F3FD 200D 2642', status='minimally-qualified', emoji='🧙🏽\u200d♂', name='man mage: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏾‍♂️': EmojiEntry(codepoint='1F9D9 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🧙🏾\u200d♂️', name='man mage: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏾‍♂': EmojiEntry(codepoint='1F9D9 1F3FE 200D 2642', status='minimally-qualified', emoji='🧙🏾\u200d♂', name='man mage: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏿‍♂️': EmojiEntry(codepoint='1F9D9 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🧙🏿\u200d♂️', name='man mage: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏿‍♂': EmojiEntry(codepoint='1F9D9 1F3FF 200D 2642', status='minimally-qualified', emoji='🧙🏿\u200d♂', name='man mage: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙‍♀️': EmojiEntry(codepoint='1F9D9 200D 2640 FE0F', status='fully-qualified', emoji='🧙\u200d♀️', name='woman mage', group='People & Body', sub_group='person-fantasy'),
    '🧙‍♀': EmojiEntry(codepoint='1F9D9 200D 2640', status='minimally-qualified', emoji='🧙\u200d♀', name='woman mage', group='People & Body', sub_group='person-fantasy'),
    '🧙🏻‍♀️': EmojiEntry(codepoint='1F9D9 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🧙🏻\u200d♀️', name='woman mage: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏻‍♀': EmojiEntry(codepoint='1F9D9 1F3FB 200D 2640', status='minimally-qualified', emoji='🧙🏻\u200d♀', name='woman mage: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏼‍♀️': EmojiEntry(codepoint='1F9D9 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🧙🏼\u200d♀️', name='woman mage: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏼‍♀': EmojiEntry(codepoint='1F9D9 1F3FC 200D 2640', status='minimally-qualified', emoji='🧙🏼\u200d♀', name='woman mage: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏽‍♀️': EmojiEntry(codepoint='1F9D9 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🧙🏽\u200d♀️', name='woman mage: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏽‍♀': EmojiEntry(codepoint='1F9D9 1F3FD 200D 2640', status='minimally-qualified', emoji='🧙🏽\u200d♀', name='woman mage: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏾‍♀️': EmojiEntry(codepoint='1F9D9 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🧙🏾\u200d♀️', name='woman mage: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏾‍♀': EmojiEntry(codepoint='1F9D9 1F3FE 200D 2640', status='minimally-qualified', emoji='🧙🏾\u200d♀', name='woman mage: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏿‍♀️': EmojiEntry(codepoint='1F9D9 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🧙🏿\u200d♀️', name='woman mage: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧙🏿‍♀': EmojiEntry(codepoint='1F9D9 1F3FF 200D 2640', status='minimally-qualified', emoji='🧙🏿\u200d♀', name='woman mage: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚': EmojiEntry(codepoint='1F9DA', status='fully-qualified', emoji='🧚', name='fairy', group='People & Body', sub_group='person-fantasy'),
    '🧚🏻': EmojiEntry(codepoint='1F9DA 1F3FB', status='fully-qualified', emoji='🧚🏻', name='fairy: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏼': EmojiEntry(codepoint='1F9DA 1F3FC', status='fully-qualified', emoji='🧚🏼', name='fairy: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏽': EmojiEntry(codepoint='1F9DA 1F3FD', status='fully-qualified', emoji='🧚🏽', name='fairy: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏾': EmojiEntry(codepoint='1F9DA 1F3FE', status='fully-qualified', emoji='🧚🏾', name='fairy: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏿': EmojiEntry(codepoint='1F9DA 1F3FF', status='fully-qualified', emoji='🧚🏿', name='fairy: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚‍♂️': EmojiEntry(codepoint='1F9DA 200D 2642 FE0F', status='fully-qualified', emoji='🧚\u200d♂️', name='man fairy', group='People & Body', sub_group='person-fantasy'),
    '🧚‍♂': EmojiEntry(codepoint='1F9DA 200D 2642', status='minimally-qualified', emoji='🧚\u200d♂', name='man fairy', group='People & Body', sub_group='person-fantasy'),
    '🧚🏻‍♂️': EmojiEntry(codepoint='1F9DA 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🧚🏻\u200d♂️', name='man fairy: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏻‍♂': EmojiEntry(codepoint='1F9DA 1F3FB 200D 2642', status='minimally-qualified', emoji='🧚🏻\u200d♂', name='man fairy: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏼‍♂️': EmojiEntry(codepoint='1F9DA 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🧚🏼\u200d♂️', name='man fairy: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏼‍♂': EmojiEntry(codepoint='1F9DA 1F3FC 200D 2642', status='minimally-qualified', emoji='🧚🏼\u200d♂', name='man fairy: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏽‍♂️': EmojiEntry(codepoint='1F9DA 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🧚🏽\u200d♂️', name='man fairy: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏽‍♂': EmojiEntry(codepoint='1F9DA 1F3FD 200D 2642', status='minimally-qualified', emoji='🧚🏽\u200d♂', name='man fairy: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏾‍♂️': EmojiEntry(codepoint='1F9DA 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🧚🏾\u200d♂️', name='man fairy: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏾‍♂': EmojiEntry(codepoint='1F9DA 1F3FE 200D 2642', status='minimally-qualified', emoji='🧚🏾\u200d♂', name='man fairy: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏿‍♂️': EmojiEntry(codepoint='1F9DA 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🧚🏿\u200d♂️', name='man fairy: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏿‍♂': EmojiEntry(codepoint='1F9DA 1F3FF 200D 2642', status='minimally-qualified', emoji='🧚🏿\u200d♂', name='man fairy: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚‍♀️': EmojiEntry(codepoint='1F9DA 200D 2640 FE0F', status='fully-qualified', emoji='🧚\u200d♀️', name='woman fairy', group='People & Body', sub_group='person-fantasy'),
    '🧚‍♀': EmojiEntry(codepoint='1F9DA 200D 2640', status='minimally-qualified', emoji='🧚\u200d♀', name='woman fairy', group='People & Body', sub_group='person-fantasy'),
    '🧚🏻‍♀️': EmojiEntry(codepoint='1F9DA 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🧚🏻\u200d♀️', name='woman fairy: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏻‍♀': EmojiEntry(codepoint='1F9DA 1F3FB 200D 2640', status='minimally-qualified', emoji='🧚🏻\u200d♀', name='woman fairy: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏼‍♀️': EmojiEntry(codepoint='1F9DA 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🧚🏼\u200d♀️', name='woman fairy: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏼‍♀': EmojiEntry(codepoint='1F9DA 1F3FC 200D 2640', status='minimally-qualified', emoji='🧚🏼\u200d♀', name='woman fairy: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏽‍♀️': EmojiEntry(codepoint='1F9DA 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🧚🏽\u200d♀️', name='woman fairy: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏽‍♀': EmojiEntry(codepoint='1F9DA 1F3FD 200D 2640', status='minimally-qualified', emoji='🧚🏽\u200d♀', name='woman fairy: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏾‍♀️': EmojiEntry(codepoint='1F9DA 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🧚🏾\u200d♀️', name='woman fairy: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏾‍♀': EmojiEntry(codepoint='1F9DA 1F3FE 200D 2640', status='minimally-qualified', emoji='🧚🏾\u200d♀', name='woman fairy: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏿‍♀️': EmojiEntry(codepoint='1F9DA 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🧚🏿\u200d♀️', name='woman fairy: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧚🏿‍♀': EmojiEntry(codepoint='1F9DA 1F3FF 200D 2640', status='minimally-qualified', emoji='🧚🏿\u200d♀', name='woman fairy: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛': EmojiEntry(codepoint='1F9DB', status='fully-qualified', emoji='🧛', name='vampire', group='People & Body', sub_group='person-fantasy'),
    '🧛🏻': EmojiEntry(codepoint='1F9DB 1F3FB', status='fully-qualified', emoji='🧛🏻', name='vampire: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏼': EmojiEntry(codepoint='1F9DB 1F3FC', status='fully-qualified', emoji='🧛🏼', name='vampire: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏽': EmojiEntry(codepoint='1F9DB 1F3FD', status='fully-qualified', emoji='🧛🏽', name='vampire: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏾': EmojiEntry(codepoint='1F9DB 1F3FE', status='fully-qualified', emoji='🧛🏾', name='vampire: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏿': EmojiEntry(codepoint='1F9DB 1F3FF', status='fully-qualified', emoji='🧛🏿', name='vampire: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛‍♂️': EmojiEntry(codepoint='1F9DB 200D 2642 FE0F', status='fully-qualified', emoji='🧛\u200d♂️', name='man vampire', group='People & Body', sub_group='person-fantasy'),
    '🧛‍♂': EmojiEntry(codepoint='1F9DB 200D 2642', status='minimally-qualified', emoji='🧛\u200d♂', name='man vampire', group='People & Body', sub_group='person-fantasy'),
    '🧛🏻‍♂️': EmojiEntry(codepoint='1F9DB 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🧛🏻\u200d♂️', name='man vampire: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏻‍♂': EmojiEntry(codepoint='1F9DB 1F3FB 200D 2642', status='minimally-qualified', emoji='🧛🏻\u200d♂', name='man vampire: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏼‍♂️': EmojiEntry(codepoint='1F9DB 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🧛🏼\u200d♂️', name='man vampire: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏼‍♂': EmojiEntry(codepoint='1F9DB 1F3FC 200D 2642', status='minimally-qualified', emoji='🧛🏼\u200d♂', name='man vampire: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏽‍♂️': EmojiEntry(codepoint='1F9DB 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🧛🏽\u200d♂️', name='man vampire: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏽‍♂': EmojiEntry(codepoint='1F9DB 1F3FD 200D 2642', status='minimally-qualified', emoji='🧛🏽\u200d♂', name='man vampire: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏾‍♂️': EmojiEntry(codepoint='1F9DB 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🧛🏾\u200d♂️', name='man vampire: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏾‍♂': EmojiEntry(codepoint='1F9DB 1F3FE 200D 2642', status='minimally-qualified', emoji='🧛🏾\u200d♂', name='man vampire: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏿‍♂️': EmojiEntry(codepoint='1F9DB 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🧛🏿\u200d♂️', name='man vampire: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏿‍♂': EmojiEntry(codepoint='1F9DB 1F3FF 200D 2642', status='minimally-qualified', emoji='🧛🏿\u200d♂', name='man vampire: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛‍♀️': EmojiEntry(codepoint='1F9DB 200D 2640 FE0F', status='fully-qualified', emoji='🧛\u200d♀️', name='woman vampire', group='People & Body', sub_group='person-fantasy'),
    '🧛‍♀': EmojiEntry(codepoint='1F9DB 200D 2640', status='minimally-qualified', emoji='🧛\u200d♀', name='woman vampire', group='People & Body', sub_group='person-fantasy'),
    '🧛🏻‍♀️': EmojiEntry(codepoint='1F9DB 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🧛🏻\u200d♀️', name='woman vampire: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏻‍♀': EmojiEntry(codepoint='1F9DB 1F3FB 200D 2640', status='minimally-qualified', emoji='🧛🏻\u200d♀', name='woman vampire: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏼‍♀️': EmojiEntry(codepoint='1F9DB 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🧛🏼\u200d♀️', name='woman vampire: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏼‍♀': EmojiEntry(codepoint='1F9DB 1F3FC 200D 2640', status='minimally-qualified', emoji='🧛🏼\u200d♀', name='woman vampire: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏽‍♀️': EmojiEntry(codepoint='1F9DB 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🧛🏽\u200d♀️', name='woman vampire: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏽‍♀': EmojiEntry(codepoint='1F9DB 1F3FD 200D 2640', status='minimally-qualified', emoji='🧛🏽\u200d♀', name='woman vampire: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏾‍♀️': EmojiEntry(codepoint='1F9DB 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🧛🏾\u200d♀️', name='woman vampire: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏾‍♀': EmojiEntry(codepoint='1F9DB 1F3FE 200D 2640', status='minimally-qualified', emoji='🧛🏾\u200d♀', name='woman vampire: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏿‍♀️': EmojiEntry(codepoint='1F9DB 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🧛🏿\u200d♀️', name='woman vampire: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧛🏿‍♀': EmojiEntry(codepoint='1F9DB 1F3FF 200D 2640', status='minimally-qualified', emoji='🧛🏿\u200d♀', name='woman vampire: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜': EmojiEntry(codepoint='1F9DC', status='fully-qualified', emoji='🧜', name='merperson', group='People & Body', sub_group='person-fantasy'),
    '🧜🏻': EmojiEntry(codepoint='1F9DC 1F3FB', status='fully-qualified', emoji='🧜🏻', name='merperson: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏼': EmojiEntry(codepoint='1F9DC 1F3FC', status='fully-qualified', emoji='🧜🏼', name='merperson: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏽': EmojiEntry(codepoint='1F9DC 1F3FD', status='fully-qualified', emoji='🧜🏽', name='merperson: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏾': EmojiEntry(codepoint='1F9DC 1F3FE', status='fully-qualified', emoji='🧜🏾', name='merperson: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏿': EmojiEntry(codepoint='1F9DC 1F3FF', status='fully-qualified', emoji='🧜🏿', name='merperson: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜‍♂️': EmojiEntry(codepoint='1F9DC 200D 2642 FE0F', status='fully-qualified', emoji='🧜\u200d♂️', name='merman', group='People & Body', sub_group='person-fantasy'),
    '🧜‍♂': EmojiEntry(codepoint='1F9DC 200D 2642', status='minimally-qualified', emoji='🧜\u200d♂', name='merman', group='People & Body', sub_group='person-fantasy'),
    '🧜🏻‍♂️': EmojiEntry(codepoint='1F9DC 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🧜🏻\u200d♂️', name='merman: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏻‍♂': EmojiEntry(codepoint='1F9DC 1F3FB 200D 2642', status='minimally-qualified', emoji='🧜🏻\u200d♂', name='merman: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏼‍♂️': EmojiEntry(codepoint='1F9DC 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🧜🏼\u200d♂️', name='merman: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏼‍♂': EmojiEntry(codepoint='1F9DC 1F3FC 200D 2642', status='minimally-qualified', emoji='🧜🏼\u200d♂', name='merman: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏽‍♂️': EmojiEntry(codepoint='1F9DC 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🧜🏽\u200d♂️', name='merman: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏽‍♂': EmojiEntry(codepoint='1F9DC 1F3FD 200D 2642', status='minimally-qualified', emoji='🧜🏽\u200d♂', name='merman: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏾‍♂️': EmojiEntry(codepoint='1F9DC 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🧜🏾\u200d♂️', name='merman: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏾‍♂': EmojiEntry(codepoint='1F9DC 1F3FE 200D 2642', status='minimally-qualified', emoji='🧜🏾\u200d♂', name='merman: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏿‍♂️': EmojiEntry(codepoint='1F9DC 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🧜🏿\u200d♂️', name='merman: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏿‍♂': EmojiEntry(codepoint='1F9DC 1F3FF 200D 2642', status='minimally-qualified', emoji='🧜🏿\u200d♂', name='merman: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜‍♀️': EmojiEntry(codepoint='1F9DC 200D 2640 FE0F', status='fully-qualified', emoji='🧜\u200d♀️', name='mermaid', group='People & Body', sub_group='person-fantasy'),
    '🧜‍♀': EmojiEntry(codepoint='1F9DC 200D 2640', status='minimally-qualified', emoji='🧜\u200d♀', name='mermaid', group='People & Body', sub_group='person-fantasy'),
    '🧜🏻‍♀️': EmojiEntry(codepoint='1F9DC 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🧜🏻\u200d♀️', name='mermaid: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏻‍♀': EmojiEntry(codepoint='1F9DC 1F3FB 200D 2640', status='minimally-qualified', emoji='🧜🏻\u200d♀', name='mermaid: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏼‍♀️': EmojiEntry(codepoint='1F9DC 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🧜🏼\u200d♀️', name='mermaid: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏼‍♀': EmojiEntry(codepoint='1F9DC 1F3FC 200D 2640', status='minimally-qualified', emoji='🧜🏼\u200d♀', name='mermaid: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏽‍♀️': EmojiEntry(codepoint='1F9DC 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🧜🏽\u200d♀️', name='mermaid: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏽‍♀': EmojiEntry(codepoint='1F9DC 1F3FD 200D 2640', status='minimally-qualified', emoji='🧜🏽\u200d♀', name='mermaid: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏾‍♀️': EmojiEntry(codepoint='1F9DC 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🧜🏾\u200d♀️', name='mermaid: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏾‍♀': EmojiEntry(codepoint='1F9DC 1F3FE 200D 2640', status='minimally-qualified', emoji='🧜🏾\u200d♀', name='mermaid: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏿‍♀️': EmojiEntry(codepoint='1F9DC 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🧜🏿\u200d♀️', name='mermaid: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧜🏿‍♀': EmojiEntry(codepoint='1F9DC 1F3FF 200D 2640', status='minimally-qualified', emoji='🧜🏿\u200d♀', name='mermaid: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝': EmojiEntry(codepoint='1F9DD', status='fully-qualified', emoji='🧝', name='elf', group='People & Body', sub_group='person-fantasy'),
    '🧝🏻': EmojiEntry(codepoint='1F9DD 1F3FB', status='fully-qualified', emoji='🧝🏻', name='elf: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏼': EmojiEntry(codepoint='1F9DD 1F3FC', status='fully-qualified', emoji='🧝🏼', name='elf: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏽': EmojiEntry(codepoint='1F9DD 1F3FD', status='fully-qualified', emoji='🧝🏽', name='elf: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏾': EmojiEntry(codepoint='1F9DD 1F3FE', status='fully-qualified', emoji='🧝🏾', name='elf: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏿': EmojiEntry(codepoint='1F9DD 1F3FF', status='fully-qualified', emoji='🧝🏿', name='elf: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝‍♂️': EmojiEntry(codepoint='1F9DD 200D 2642 FE0F', status='fully-qualified', emoji='🧝\u200d♂️', name='man elf', group='People & Body', sub_group='person-fantasy'),
    '🧝‍♂': EmojiEntry(codepoint='1F9DD 200D 2642', status='minimally-qualified', emoji='🧝\u200d♂', name='man elf', group='People & Body', sub_group='person-fantasy'),
    '🧝🏻‍♂️': EmojiEntry(codepoint='1F9DD 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🧝🏻\u200d♂️', name='man elf: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏻‍♂': EmojiEntry(codepoint='1F9DD 1F3FB 200D 2642', status='minimally-qualified', emoji='🧝🏻\u200d♂', name='man elf: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏼‍♂️': EmojiEntry(codepoint='1F9DD 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🧝🏼\u200d♂️', name='man elf: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏼‍♂': EmojiEntry(codepoint='1F9DD 1F3FC 200D 2642', status='minimally-qualified', emoji='🧝🏼\u200d♂', name='man elf: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏽‍♂️': EmojiEntry(codepoint='1F9DD 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🧝🏽\u200d♂️', name='man elf: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏽‍♂': EmojiEntry(codepoint='1F9DD 1F3FD 200D 2642', status='minimally-qualified', emoji='🧝🏽\u200d♂', name='man elf: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏾‍♂️': EmojiEntry(codepoint='1F9DD 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🧝🏾\u200d♂️', name='man elf: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏾‍♂': EmojiEntry(codepoint='1F9DD 1F3FE 200D 2642', status='minimally-qualified', emoji='🧝🏾\u200d♂', name='man elf: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏿‍♂️': EmojiEntry(codepoint='1F9DD 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🧝🏿\u200d♂️', name='man elf: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏿‍♂': EmojiEntry(codepoint='1F9DD 1F3FF 200D 2642', status='minimally-qualified', emoji='🧝🏿\u200d♂', name='man elf: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝‍♀️': EmojiEntry(codepoint='1F9DD 200D 2640 FE0F', status='fully-qualified', emoji='🧝\u200d♀️', name='woman elf', group='People & Body', sub_group='person-fantasy'),
    '🧝‍♀': EmojiEntry(codepoint='1F9DD 200D 2640', status='minimally-qualified', emoji='🧝\u200d♀', name='woman elf', group='People & Body', sub_group='person-fantasy'),
    '🧝🏻‍♀️': EmojiEntry(codepoint='1F9DD 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🧝🏻\u200d♀️', name='woman elf: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏻‍♀': EmojiEntry(codepoint='1F9DD 1F3FB 200D 2640', status='minimally-qualified', emoji='🧝🏻\u200d♀', name='woman elf: light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏼‍♀️': EmojiEntry(codepoint='1F9DD 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🧝🏼\u200d♀️', name='woman elf: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏼‍♀': EmojiEntry(codepoint='1F9DD 1F3FC 200D 2640', status='minimally-qualified', emoji='🧝🏼\u200d♀', name='woman elf: medium-light skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏽‍♀️': EmojiEntry(codepoint='1F9DD 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🧝🏽\u200d♀️', name='woman elf: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏽‍♀': EmojiEntry(codepoint='1F9DD 1F3FD 200D 2640', status='minimally-qualified', emoji='🧝🏽\u200d♀', name='woman elf: medium skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏾‍♀️': EmojiEntry(codepoint='1F9DD 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🧝🏾\u200d♀️', name='woman elf: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏾‍♀': EmojiEntry(codepoint='1F9DD 1F3FE 200D 2640', status='minimally-qualified', emoji='🧝🏾\u200d♀', name='woman elf: medium-dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏿‍♀️': EmojiEntry(codepoint='1F9DD 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🧝🏿\u200d♀️', name='woman elf: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧝🏿‍♀': EmojiEntry(codepoint='1F9DD 1F3FF 200D 2640', status='minimally-qualified', emoji='🧝🏿\u200d♀', name='woman elf: dark skin tone', group='People & Body', sub_group='person-fantasy'),
    '🧞': EmojiEntry(codepoint='1F9DE', status='fully-qualified', emoji='🧞', name='genie', group='People & Body', sub_group='person-fantasy'),
    '🧞‍♂️': EmojiEntry(codepoint='1F9DE 200D 2642 FE0F', status='fully-qualified', emoji='🧞\u200d♂️', name='man genie', group='People & Body', sub_group='person-fantasy'),
    '🧞‍♂': EmojiEntry(codepoint='1F9DE 200D 2642', status='minimally-qualified', emoji='🧞\u200d♂', name='man genie', group='People & Body', sub_group='person-fantasy'),
    '🧞‍♀️': EmojiEntry(codepoint='1F9DE 200D 2640 FE0F', status='fully-qualified', emoji='🧞\u200d♀️', name='woman genie', group='People & Body', sub_group='person-fantasy'),
    '🧞‍♀': EmojiEntry(codepoint='1F9DE 200D 2640', status='minimally-qualified', emoji='🧞\u200d♀', name='woman genie', group='People & Body', sub_group='person-fantasy'),
    '🧟': EmojiEntry(codepoint='1F9DF', status='fully-qualified', emoji='🧟', name='zombie', group='People & Body', sub_group='person-fantasy'),
    '🧟‍♂️': EmojiEntry(codepoint='1F9DF 200D 2642 FE0F', status='fully-qualified', emoji='🧟\u200d♂️', name='man zombie', group='People & Body', sub_group='person-fantasy'),
    '🧟‍♂': EmojiEntry(codepoint='1F9DF 200D 2642', status='minimally-qualified', emoji='🧟\u200d♂', name='man zombie', group='People & Body', sub_group='person-fantasy'),
    '🧟‍♀️': EmojiEntry(codepoint='1F9DF 200D 2640 FE0F', status='fully-qualified', emoji='🧟\u200d♀️', name='woman zombie', group='People & Body', sub_group='person-fantasy'),
    '🧟‍♀': EmojiEntry(codepoint='1F9DF 200D 2640', status='minimally-qualified', emoji='🧟\u200d♀', name='woman zombie', group='People & Body', sub_group='person-fantasy'),
    '💆': EmojiEntry(codepoint='1F486', status='fully-qualified', emoji='💆', name='person getting massage', group='People & Body', sub_group='person-activity'),
    '💆🏻': EmojiEntry(codepoint='1F486 1F3FB', status='fully-qualified', emoji='💆🏻', name='person getting massage: light skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏼': EmojiEntry(codepoint='1F486 1F3FC', status='fully-qualified', emoji='💆🏼', name='person getting massage: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏽': EmojiEntry(codepoint='1F486 1F3FD', status='fully-qualified', emoji='💆🏽', name='person getting massage: medium skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏾': EmojiEntry(codepoint='1F486 1F3FE', status='fully-qualified', emoji='💆🏾', name='person getting massage: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏿': EmojiEntry(codepoint='1F486 1F3FF', status='fully-qualified', emoji='💆🏿', name='person getting massage: dark skin tone', group='People & Body', sub_group='person-activity'),
    '💆‍♂️': EmojiEntry(codepoint='1F486 200D 2642 FE0F', status='fully-qualified', emoji='💆\u200d♂️', name='man getting massage', group='People & Body', sub_group='person-activity'),
    '💆‍♂': EmojiEntry(codepoint='1F486 200D 2642', status='minimally-qualified', emoji='💆\u200d♂', name='man getting massage', group='People & Body', sub_group='person-activity'),
    '💆🏻‍♂️': EmojiEntry(codepoint='1F486 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='💆🏻\u200d♂️', name='man getting massage: light skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏻‍♂': EmojiEntry(codepoint='1F486 1F3FB 200D 2642', status='minimally-qualified', emoji='💆🏻\u200d♂', name='man getting massage: light skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏼‍♂️': EmojiEntry(codepoint='1F486 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='💆🏼\u200d♂️', name='man getting massage: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏼‍♂': EmojiEntry(codepoint='1F486 1F3FC 200D 2642', status='minimally-qualified', emoji='💆🏼\u200d♂', name='man getting massage: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏽‍♂️': EmojiEntry(codepoint='1F486 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='💆🏽\u200d♂️', name='man getting massage: medium skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏽‍♂': EmojiEntry(codepoint='1F486 1F3FD 200D 2642', status='minimally-qualified', emoji='💆🏽\u200d♂', name='man getting massage: medium skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏾‍♂️': EmojiEntry(codepoint='1F486 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='💆🏾\u200d♂️', name='man getting massage: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏾‍♂': EmojiEntry(codepoint='1F486 1F3FE 200D 2642', status='minimally-qualified', emoji='💆🏾\u200d♂', name='man getting massage: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏿‍♂️': EmojiEntry(codepoint='1F486 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='💆🏿\u200d♂️', name='man getting massage: dark skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏿‍♂': EmojiEntry(codepoint='1F486 1F3FF 200D 2642', status='minimally-qualified', emoji='💆🏿\u200d♂', name='man getting massage: dark skin tone', group='People & Body', sub_group='person-activity'),
    '💆‍♀️': EmojiEntry(codepoint='1F486 200D 2640 FE0F', status='fully-qualified', emoji='💆\u200d♀️', name='woman getting massage', group='People & Body', sub_group='person-activity'),
    '💆‍♀': EmojiEntry(codepoint='1F486 200D 2640', status='minimally-qualified', emoji='💆\u200d♀', name='woman getting massage', group='People & Body', sub_group='person-activity'),
    '💆🏻‍♀️': EmojiEntry(codepoint='1F486 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='💆🏻\u200d♀️', name='woman getting massage: light skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏻‍♀': EmojiEntry(codepoint='1F486 1F3FB 200D 2640', status='minimally-qualified', emoji='💆🏻\u200d♀', name='woman getting massage: light skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏼‍♀️': EmojiEntry(codepoint='1F486 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='💆🏼\u200d♀️', name='woman getting massage: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏼‍♀': EmojiEntry(codepoint='1F486 1F3FC 200D 2640', status='minimally-qualified', emoji='💆🏼\u200d♀', name='woman getting massage: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏽‍♀️': EmojiEntry(codepoint='1F486 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='💆🏽\u200d♀️', name='woman getting massage: medium skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏽‍♀': EmojiEntry(codepoint='1F486 1F3FD 200D 2640', status='minimally-qualified', emoji='💆🏽\u200d♀', name='woman getting massage: medium skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏾‍♀️': EmojiEntry(codepoint='1F486 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='💆🏾\u200d♀️', name='woman getting massage: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏾‍♀': EmojiEntry(codepoint='1F486 1F3FE 200D 2640', status='minimally-qualified', emoji='💆🏾\u200d♀', name='woman getting massage: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏿‍♀️': EmojiEntry(codepoint='1F486 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='💆🏿\u200d♀️', name='woman getting massage: dark skin tone', group='People & Body', sub_group='person-activity'),
    '💆🏿‍♀': EmojiEntry(codepoint='1F486 1F3FF 200D 2640', status='minimally-qualified', emoji='💆🏿\u200d♀', name='woman getting massage: dark skin tone', group='People & Body', sub_group='person-activity'),
    '💇': EmojiEntry(codepoint='1F487', status='fully-qualified', emoji='💇', name='person getting haircut', group='People & Body', sub_group='person-activity'),
    '💇🏻': EmojiEntry(codepoint='1F487 1F3FB', status='fully-qualified', emoji='💇🏻', name='person getting haircut: light skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏼': EmojiEntry(codepoint='1F487 1F3FC', status='fully-qualified', emoji='💇🏼', name='person getting haircut: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏽': EmojiEntry(codepoint='1F487 1F3FD', status='fully-qualified', emoji='💇🏽', name='person getting haircut: medium skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏾': EmojiEntry(codepoint='1F487 1F3FE', status='fully-qualified', emoji='💇🏾', name='person getting haircut: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏿': EmojiEntry(codepoint='1F487 1F3FF', status='fully-qualified', emoji='💇🏿', name='person getting haircut: dark skin tone', group='People & Body', sub_group='person-activity'),
    '💇‍♂️': EmojiEntry(codepoint='1F487 200D 2642 FE0F', status='fully-qualified', emoji='💇\u200d♂️', name='man getting haircut', group='People & Body', sub_group='person-activity'),
    '💇‍♂': EmojiEntry(codepoint='1F487 200D 2642', status='minimally-qualified', emoji='💇\u200d♂', name='man getting haircut', group='People & Body', sub_group='person-activity'),
    '💇🏻‍♂️': EmojiEntry(codepoint='1F487 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='💇🏻\u200d♂️', name='man getting haircut: light skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏻‍♂': EmojiEntry(codepoint='1F487 1F3FB 200D 2642', status='minimally-qualified', emoji='💇🏻\u200d♂', name='man getting haircut: light skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏼‍♂️': EmojiEntry(codepoint='1F487 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='💇🏼\u200d♂️', name='man getting haircut: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏼‍♂': EmojiEntry(codepoint='1F487 1F3FC 200D 2642', status='minimally-qualified', emoji='💇🏼\u200d♂', name='man getting haircut: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏽‍♂️': EmojiEntry(codepoint='1F487 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='💇🏽\u200d♂️', name='man getting haircut: medium skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏽‍♂': EmojiEntry(codepoint='1F487 1F3FD 200D 2642', status='minimally-qualified', emoji='💇🏽\u200d♂', name='man getting haircut: medium skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏾‍♂️': EmojiEntry(codepoint='1F487 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='💇🏾\u200d♂️', name='man getting haircut: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏾‍♂': EmojiEntry(codepoint='1F487 1F3FE 200D 2642', status='minimally-qualified', emoji='💇🏾\u200d♂', name='man getting haircut: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏿‍♂️': EmojiEntry(codepoint='1F487 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='💇🏿\u200d♂️', name='man getting haircut: dark skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏿‍♂': EmojiEntry(codepoint='1F487 1F3FF 200D 2642', status='minimally-qualified', emoji='💇🏿\u200d♂', name='man getting haircut: dark skin tone', group='People & Body', sub_group='person-activity'),
    '💇‍♀️': EmojiEntry(codepoint='1F487 200D 2640 FE0F', status='fully-qualified', emoji='💇\u200d♀️', name='woman getting haircut', group='People & Body', sub_group='person-activity'),
    '💇‍♀': EmojiEntry(codepoint='1F487 200D 2640', status='minimally-qualified', emoji='💇\u200d♀', name='woman getting haircut', group='People & Body', sub_group='person-activity'),
    '💇🏻‍♀️': EmojiEntry(codepoint='1F487 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='💇🏻\u200d♀️', name='woman getting haircut: light skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏻‍♀': EmojiEntry(codepoint='1F487 1F3FB 200D 2640', status='minimally-qualified', emoji='💇🏻\u200d♀', name='woman getting haircut: light skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏼‍♀️': EmojiEntry(codepoint='1F487 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='💇🏼\u200d♀️', name='woman getting haircut: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏼‍♀': EmojiEntry(codepoint='1F487 1F3FC 200D 2640', status='minimally-qualified', emoji='💇🏼\u200d♀', name='woman getting haircut: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏽‍♀️': EmojiEntry(codepoint='1F487 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='💇🏽\u200d♀️', name='woman getting haircut: medium skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏽‍♀': EmojiEntry(codepoint='1F487 1F3FD 200D 2640', status='minimally-qualified', emoji='💇🏽\u200d♀', name='woman getting haircut: medium skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏾‍♀️': EmojiEntry(codepoint='1F487 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='💇🏾\u200d♀️', name='woman getting haircut: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏾‍♀': EmojiEntry(codepoint='1F487 1F3FE 200D 2640', status='minimally-qualified', emoji='💇🏾\u200d♀', name='woman getting haircut: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏿‍♀️': EmojiEntry(codepoint='1F487 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='💇🏿\u200d♀️', name='woman getting haircut: dark skin tone', group='People & Body', sub_group='person-activity'),
    '💇🏿‍♀': EmojiEntry(codepoint='1F487 1F3FF 200D 2640', status='minimally-qualified', emoji='💇🏿\u200d♀', name='woman getting haircut: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🚶': EmojiEntry(codepoint='1F6B6', status='fully-qualified', emoji='🚶', name='person walking', group='People & Body', sub_group='person-activity'),
    '🚶🏻': EmojiEntry(codepoint='1F6B6 1F3FB', status='fully-qualified', emoji='🚶🏻', name='person walking: light skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏼': EmojiEntry(codepoint='1F6B6 1F3FC', status='fully-qualified', emoji='🚶🏼', name='person walking: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏽': EmojiEntry(codepoint='1F6B6 1F3FD', status='fully-qualified', emoji='🚶🏽', name='person walking: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏾': EmojiEntry(codepoint='1F6B6 1F3FE', status='fully-qualified', emoji='🚶🏾', name='person walking: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏿': EmojiEntry(codepoint='1F6B6 1F3FF', status='fully-qualified', emoji='🚶🏿', name='person walking: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🚶‍♂️': EmojiEntry(codepoint='1F6B6 200D 2642 FE0F', status='fully-qualified', emoji='🚶\u200d♂️', name='man walking', group='People & Body', sub_group='person-activity'),
    '🚶‍♂': EmojiEntry(codepoint='1F6B6 200D 2642', status='minimally-qualified', emoji='🚶\u200d♂', name='man walking', group='People & Body', sub_group='person-activity'),
    '🚶🏻‍♂️': EmojiEntry(codepoint='1F6B6 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🚶🏻\u200d♂️', name='man walking: light skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏻‍♂': EmojiEntry(codepoint='1F6B6 1F3FB 200D 2642', status='minimally-qualified', emoji='🚶🏻\u200d♂', name='man walking: light skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏼‍♂️': EmojiEntry(codepoint='1F6B6 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🚶🏼\u200d♂️', name='man walking: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏼‍♂': EmojiEntry(codepoint='1F6B6 1F3FC 200D 2642', status='minimally-qualified', emoji='🚶🏼\u200d♂', name='man walking: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏽‍♂️': EmojiEntry(codepoint='1F6B6 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🚶🏽\u200d♂️', name='man walking: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏽‍♂': EmojiEntry(codepoint='1F6B6 1F3FD 200D 2642', status='minimally-qualified', emoji='🚶🏽\u200d♂', name='man walking: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏾‍♂️': EmojiEntry(codepoint='1F6B6 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🚶🏾\u200d♂️', name='man walking: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏾‍♂': EmojiEntry(codepoint='1F6B6 1F3FE 200D 2642', status='minimally-qualified', emoji='🚶🏾\u200d♂', name='man walking: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏿‍♂️': EmojiEntry(codepoint='1F6B6 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🚶🏿\u200d♂️', name='man walking: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏿‍♂': EmojiEntry(codepoint='1F6B6 1F3FF 200D 2642', status='minimally-qualified', emoji='🚶🏿\u200d♂', name='man walking: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🚶‍♀️': EmojiEntry(codepoint='1F6B6 200D 2640 FE0F', status='fully-qualified', emoji='🚶\u200d♀️', name='woman walking', group='People & Body', sub_group='person-activity'),
    '🚶‍♀': EmojiEntry(codepoint='1F6B6 200D 2640', status='minimally-qualified', emoji='🚶\u200d♀', name='woman walking', group='People & Body', sub_group='person-activity'),
    '🚶🏻‍♀️': EmojiEntry(codepoint='1F6B6 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🚶🏻\u200d♀️', name='woman walking: light skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏻‍♀': EmojiEntry(codepoint='1F6B6 1F3FB 200D 2640', status='minimally-qualified', emoji='🚶🏻\u200d♀', name='woman walking: light skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏼‍♀️': EmojiEntry(codepoint='1F6B6 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🚶🏼\u200d♀️', name='woman walking: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏼‍♀': EmojiEntry(codepoint='1F6B6 1F3FC 200D 2640', status='minimally-qualified', emoji='🚶🏼\u200d♀', name='woman walking: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏽‍♀️': EmojiEntry(codepoint='1F6B6 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🚶🏽\u200d♀️', name='woman walking: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏽‍♀': EmojiEntry(codepoint='1F6B6 1F3FD 200D 2640', status='minimally-qualified', emoji='🚶🏽\u200d♀', name='woman walking: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏾‍♀️': EmojiEntry(codepoint='1F6B6 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🚶🏾\u200d♀️', name='woman walking: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏾‍♀': EmojiEntry(codepoint='1F6B6 1F3FE 200D 2640', status='minimally-qualified', emoji='🚶🏾\u200d♀', name='woman walking: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏿‍♀️': EmojiEntry(codepoint='1F6B6 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🚶🏿\u200d♀️', name='woman walking: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🚶🏿‍♀': EmojiEntry(codepoint='1F6B6 1F3FF 200D 2640', status='minimally-qualified', emoji='🚶🏿\u200d♀', name='woman walking: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧍': EmojiEntry(codepoint='1F9CD', status='fully-qualified', emoji='🧍', name='person standing', group='People & Body', sub_group='person-activity'),
    '🧍🏻': EmojiEntry(codepoint='1F9CD 1F3FB', status='fully-qualified', emoji='🧍🏻', name='person standing: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏼': EmojiEntry(codepoint='1F9CD 1F3FC', status='fully-qualified', emoji='🧍🏼', name='person standing: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏽': EmojiEntry(codepoint='1F9CD 1F3FD', status='fully-qualified', emoji='🧍🏽', name='person standing: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏾': EmojiEntry(codepoint='1F9CD 1F3FE', status='fully-qualified', emoji='🧍🏾', name='person standing: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏿': EmojiEntry(codepoint='1F9CD 1F3FF', status='fully-qualified', emoji='🧍🏿', name='person standing: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧍‍♂️': EmojiEntry(codepoint='1F9CD 200D 2642 FE0F', status='fully-qualified', emoji='🧍\u200d♂️', name='man standing', group='People & Body', sub_group='person-activity'),
    '🧍‍♂': EmojiEntry(codepoint='1F9CD 200D 2642', status='minimally-qualified', emoji='🧍\u200d♂', name='man standing', group='People & Body', sub_group='person-activity'),
    '🧍🏻‍♂️': EmojiEntry(codepoint='1F9CD 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🧍🏻\u200d♂️', name='man standing: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏻‍♂': EmojiEntry(codepoint='1F9CD 1F3FB 200D 2642', status='minimally-qualified', emoji='🧍🏻\u200d♂', name='man standing: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏼‍♂️': EmojiEntry(codepoint='1F9CD 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🧍🏼\u200d♂️', name='man standing: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏼‍♂': EmojiEntry(codepoint='1F9CD 1F3FC 200D 2642', status='minimally-qualified', emoji='🧍🏼\u200d♂', name='man standing: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏽‍♂️': EmojiEntry(codepoint='1F9CD 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🧍🏽\u200d♂️', name='man standing: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏽‍♂': EmojiEntry(codepoint='1F9CD 1F3FD 200D 2642', status='minimally-qualified', emoji='🧍🏽\u200d♂', name='man standing: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏾‍♂️': EmojiEntry(codepoint='1F9CD 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🧍🏾\u200d♂️', name='man standing: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏾‍♂': EmojiEntry(codepoint='1F9CD 1F3FE 200D 2642', status='minimally-qualified', emoji='🧍🏾\u200d♂', name='man standing: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏿‍♂️': EmojiEntry(codepoint='1F9CD 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🧍🏿\u200d♂️', name='man standing: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏿‍♂': EmojiEntry(codepoint='1F9CD 1F3FF 200D 2642', status='minimally-qualified', emoji='🧍🏿\u200d♂', name='man standing: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧍‍♀️': EmojiEntry(codepoint='1F9CD 200D 2640 FE0F', status='fully-qualified', emoji='🧍\u200d♀️', name='woman standing', group='People & Body', sub_group='person-activity'),
    '🧍‍♀': EmojiEntry(codepoint='1F9CD 200D 2640', status='minimally-qualified', emoji='🧍\u200d♀', name='woman standing', group='People & Body', sub_group='person-activity'),
    '🧍🏻‍♀️': EmojiEntry(codepoint='1F9CD 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🧍🏻\u200d♀️', name='woman standing: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏻‍♀': EmojiEntry(codepoint='1F9CD 1F3FB 200D 2640', status='minimally-qualified', emoji='🧍🏻\u200d♀', name='woman standing: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏼‍♀️': EmojiEntry(codepoint='1F9CD 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🧍🏼\u200d♀️', name='woman standing: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏼‍♀': EmojiEntry(codepoint='1F9CD 1F3FC 200D 2640', status='minimally-qualified', emoji='🧍🏼\u200d♀', name='woman standing: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏽‍♀️': EmojiEntry(codepoint='1F9CD 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🧍🏽\u200d♀️', name='woman standing: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏽‍♀': EmojiEntry(codepoint='1F9CD 1F3FD 200D 2640', status='minimally-qualified', emoji='🧍🏽\u200d♀', name='woman standing: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏾‍♀️': EmojiEntry(codepoint='1F9CD 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🧍🏾\u200d♀️', name='woman standing: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏾‍♀': EmojiEntry(codepoint='1F9CD 1F3FE 200D 2640', status='minimally-qualified', emoji='🧍🏾\u200d♀', name='woman standing: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏿‍♀️': EmojiEntry(codepoint='1F9CD 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🧍🏿\u200d♀️', name='woman standing: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧍🏿‍♀': EmojiEntry(codepoint='1F9CD 1F3FF 200D 2640', status='minimally-qualified', emoji='🧍🏿\u200d♀', name='woman standing: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧎': EmojiEntry(codepoint='1F9CE', status='fully-qualified', emoji='🧎', name='person kneeling', group='People & Body', sub_group='person-activity'),
    '🧎🏻': EmojiEntry(codepoint='1F9CE 1F3FB', status='fully-qualified', emoji='🧎🏻', name='person kneeling: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏼': EmojiEntry(codepoint='1F9CE 1F3FC', status='fully-qualified', emoji='🧎🏼', name='person kneeling: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏽': EmojiEntry(codepoint='1F9CE 1F3FD', status='fully-qualified', emoji='🧎🏽', name='person kneeling: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏾': EmojiEntry(codepoint='1F9CE 1F3FE', status='fully-qualified', emoji='🧎🏾', name='person kneeling: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏿': EmojiEntry(codepoint='1F9CE 1F3FF', status='fully-qualified', emoji='🧎🏿', name='person kneeling: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧎‍♂️': EmojiEntry(codepoint='1F9CE 200D 2642 FE0F', status='fully-qualified', emoji='🧎\u200d♂️', name='man kneeling', group='People & Body', sub_group='person-activity'),
    '🧎‍♂': EmojiEntry(codepoint='1F9CE 200D 2642', status='minimally-qualified', emoji='🧎\u200d♂', name='man kneeling', group='People & Body', sub_group='person-activity'),
    '🧎🏻‍♂️': EmojiEntry(codepoint='1F9CE 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🧎🏻\u200d♂️', name='man kneeling: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏻‍♂': EmojiEntry(codepoint='1F9CE 1F3FB 200D 2642', status='minimally-qualified', emoji='🧎🏻\u200d♂', name='man kneeling: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏼‍♂️': EmojiEntry(codepoint='1F9CE 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🧎🏼\u200d♂️', name='man kneeling: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏼‍♂': EmojiEntry(codepoint='1F9CE 1F3FC 200D 2642', status='minimally-qualified', emoji='🧎🏼\u200d♂', name='man kneeling: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏽‍♂️': EmojiEntry(codepoint='1F9CE 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🧎🏽\u200d♂️', name='man kneeling: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏽‍♂': EmojiEntry(codepoint='1F9CE 1F3FD 200D 2642', status='minimally-qualified', emoji='🧎🏽\u200d♂', name='man kneeling: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏾‍♂️': EmojiEntry(codepoint='1F9CE 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🧎🏾\u200d♂️', name='man kneeling: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏾‍♂': EmojiEntry(codepoint='1F9CE 1F3FE 200D 2642', status='minimally-qualified', emoji='🧎🏾\u200d♂', name='man kneeling: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏿‍♂️': EmojiEntry(codepoint='1F9CE 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🧎🏿\u200d♂️', name='man kneeling: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏿‍♂': EmojiEntry(codepoint='1F9CE 1F3FF 200D 2642', status='minimally-qualified', emoji='🧎🏿\u200d♂', name='man kneeling: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧎‍♀️': EmojiEntry(codepoint='1F9CE 200D 2640 FE0F', status='fully-qualified', emoji='🧎\u200d♀️', name='woman kneeling', group='People & Body', sub_group='person-activity'),
    '🧎‍♀': EmojiEntry(codepoint='1F9CE 200D 2640', status='minimally-qualified', emoji='🧎\u200d♀', name='woman kneeling', group='People & Body', sub_group='person-activity'),
    '🧎🏻‍♀️': EmojiEntry(codepoint='1F9CE 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🧎🏻\u200d♀️', name='woman kneeling: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏻‍♀': EmojiEntry(codepoint='1F9CE 1F3FB 200D 2640', status='minimally-qualified', emoji='🧎🏻\u200d♀', name='woman kneeling: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏼‍♀️': EmojiEntry(codepoint='1F9CE 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🧎🏼\u200d♀️', name='woman kneeling: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏼‍♀': EmojiEntry(codepoint='1F9CE 1F3FC 200D 2640', status='minimally-qualified', emoji='🧎🏼\u200d♀', name='woman kneeling: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏽‍♀️': EmojiEntry(codepoint='1F9CE 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🧎🏽\u200d♀️', name='woman kneeling: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏽‍♀': EmojiEntry(codepoint='1F9CE 1F3FD 200D 2640', status='minimally-qualified', emoji='🧎🏽\u200d♀', name='woman kneeling: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏾‍♀️': EmojiEntry(codepoint='1F9CE 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🧎🏾\u200d♀️', name='woman kneeling: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏾‍♀': EmojiEntry(codepoint='1F9CE 1F3FE 200D 2640', status='minimally-qualified', emoji='🧎🏾\u200d♀', name='woman kneeling: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏿‍♀️': EmojiEntry(codepoint='1F9CE 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🧎🏿\u200d♀️', name='woman kneeling: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧎🏿‍♀': EmojiEntry(codepoint='1F9CE 1F3FF 200D 2640', status='minimally-qualified', emoji='🧎🏿\u200d♀', name='woman kneeling: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧑‍🦯': EmojiEntry(codepoint='1F9D1 200D 1F9AF', status='fully-qualified', emoji='🧑\u200d🦯', name='person with white cane', group='People & Body', sub_group='person-activity'),
    '🧑🏻‍🦯': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F9AF', status='fully-qualified', emoji='🧑🏻\u200d🦯', name='person with white cane: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧑🏼‍🦯': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F9AF', status='fully-qualified', emoji='🧑🏼\u200d🦯', name='person with white cane: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧑🏽‍🦯': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F9AF', status='fully-qualified', emoji='🧑🏽\u200d🦯', name='person with white cane: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧑🏾‍🦯': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F9AF', status='fully-qualified', emoji='🧑🏾\u200d🦯', name='person with white cane: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧑🏿‍🦯': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F9AF', status='fully-qualified', emoji='🧑🏿\u200d🦯', name='person with white cane: dark skin tone', group='People & Body', sub_group='person-activity'),
    '👨‍🦯': EmojiEntry(codepoint='1F468 200D 1F9AF', status='fully-qualified', emoji='👨\u200d🦯', name='man with white cane', group='People & Body', sub_group='person-activity'),
    '👨🏻‍🦯': EmojiEntry(codepoint='1F468 1F3FB 200D 1F9AF', status='fully-qualified', emoji='👨🏻\u200d🦯', name='man with white cane: light skin tone', group='People & Body', sub_group='person-activity'),
    '👨🏼‍🦯': EmojiEntry(codepoint='1F468 1F3FC 200D 1F9AF', status='fully-qualified', emoji='👨🏼\u200d🦯', name='man with white cane: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '👨🏽‍🦯': EmojiEntry(codepoint='1F468 1F3FD 200D 1F9AF', status='fully-qualified', emoji='👨🏽\u200d🦯', name='man with white cane: medium skin tone', group='People & Body', sub_group='person-activity'),
    '👨🏾‍🦯': EmojiEntry(codepoint='1F468 1F3FE 200D 1F9AF', status='fully-qualified', emoji='👨🏾\u200d🦯', name='man with white cane: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '👨🏿‍🦯': EmojiEntry(codepoint='1F468 1F3FF 200D 1F9AF', status='fully-qualified', emoji='👨🏿\u200d🦯', name='man with white cane: dark skin tone', group='People & Body', sub_group='person-activity'),
    '👩‍🦯': EmojiEntry(codepoint='1F469 200D 1F9AF', status='fully-qualified', emoji='👩\u200d🦯', name='woman with white cane', group='People & Body', sub_group='person-activity'),
    '👩🏻‍🦯': EmojiEntry(codepoint='1F469 1F3FB 200D 1F9AF', status='fully-qualified', emoji='👩🏻\u200d🦯', name='woman with white cane: light skin tone', group='People & Body', sub_group='person-activity'),
    '👩🏼‍🦯': EmojiEntry(codepoint='1F469 1F3FC 200D 1F9AF', status='fully-qualified', emoji='👩🏼\u200d🦯', name='woman with white cane: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '👩🏽‍🦯': EmojiEntry(codepoint='1F469 1F3FD 200D 1F9AF', status='fully-qualified', emoji='👩🏽\u200d🦯', name='woman with white cane: medium skin tone', group='People & Body', sub_group='person-activity'),
    '👩🏾‍🦯': EmojiEntry(codepoint='1F469 1F3FE 200D 1F9AF', status='fully-qualified', emoji='👩🏾\u200d🦯', name='woman with white cane: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '👩🏿‍🦯': EmojiEntry(codepoint='1F469 1F3FF 200D 1F9AF', status='fully-qualified', emoji='👩🏿\u200d🦯', name='woman with white cane: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧑‍🦼': EmojiEntry(codepoint='1F9D1 200D 1F9BC', status='fully-qualified', emoji='🧑\u200d🦼', name='person in motorized wheelchair', group='People & Body', sub_group='person-activity'),
    '🧑🏻‍🦼': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F9BC', status='fully-qualified', emoji='🧑🏻\u200d🦼', name='person in motorized wheelchair: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧑🏼‍🦼': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F9BC', status='fully-qualified', emoji='🧑🏼\u200d🦼', name='person in motorized wheelchair: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧑🏽‍🦼': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F9BC', status='fully-qualified', emoji='🧑🏽\u200d🦼', name='person in motorized wheelchair: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧑🏾‍🦼': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F9BC', status='fully-qualified', emoji='🧑🏾\u200d🦼', name='person in motorized wheelchair: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧑🏿‍🦼': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F9BC', status='fully-qualified', emoji='🧑🏿\u200d🦼', name='person in motorized wheelchair: dark skin tone', group='People & Body', sub_group='person-activity'),
    '👨‍🦼': EmojiEntry(codepoint='1F468 200D 1F9BC', status='fully-qualified', emoji='👨\u200d🦼', name='man in motorized wheelchair', group='People & Body', sub_group='person-activity'),
    '👨🏻‍🦼': EmojiEntry(codepoint='1F468 1F3FB 200D 1F9BC', status='fully-qualified', emoji='👨🏻\u200d🦼', name='man in motorized wheelchair: light skin tone', group='People & Body', sub_group='person-activity'),
    '👨🏼‍🦼': EmojiEntry(codepoint='1F468 1F3FC 200D 1F9BC', status='fully-qualified', emoji='👨🏼\u200d🦼', name='man in motorized wheelchair: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '👨🏽‍🦼': EmojiEntry(codepoint='1F468 1F3FD 200D 1F9BC', status='fully-qualified', emoji='👨🏽\u200d🦼', name='man in motorized wheelchair: medium skin tone', group='People & Body', sub_group='person-activity'),
    '👨🏾‍🦼': EmojiEntry(codepoint='1F468 1F3FE 200D 1F9BC', status='fully-qualified', emoji='👨🏾\u200d🦼', name='man in motorized wheelchair: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '👨🏿‍🦼': EmojiEntry(codepoint='1F468 1F3FF 200D 1F9BC', status='fully-qualified', emoji='👨🏿\u200d🦼', name='man in motorized wheelchair: dark skin tone', group='People & Body', sub_group='person-activity'),
    '👩‍🦼': EmojiEntry(codepoint='1F469 200D 1F9BC', status='fully-qualified', emoji='👩\u200d🦼', name='woman in motorized wheelchair', group='People & Body', sub_group='person-activity'),
    '👩🏻‍🦼': EmojiEntry(codepoint='1F469 1F3FB 200D 1F9BC', status='fully-qualified', emoji='👩🏻\u200d🦼', name='woman in motorized wheelchair: light skin tone', group='People & Body', sub_group='person-activity'),
    '👩🏼‍🦼': EmojiEntry(codepoint='1F469 1F3FC 200D 1F9BC', status='fully-qualified', emoji='👩🏼\u200d🦼', name='woman in motorized wheelchair: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '👩🏽‍🦼': EmojiEntry(codepoint='1F469 1F3FD 200D 1F9BC', status='fully-qualified', emoji='👩🏽\u200d🦼', name='woman in motorized wheelchair: medium skin tone', group='People & Body', sub_group='person-activity'),
    '👩🏾‍🦼': EmojiEntry(codepoint='1F469 1F3FE 200D 1F9BC', status='fully-qualified', emoji='👩🏾\u200d🦼', name='woman in motorized wheelchair: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '👩🏿‍🦼': EmojiEntry(codepoint='1F469 1F3FF 200D 1F9BC', status='fully-qualified', emoji='👩🏿\u200d🦼', name='woman in motorized wheelchair: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧑‍🦽': EmojiEntry(codepoint='1F9D1 200D 1F9BD', status='fully-qualified', emoji='🧑\u200d🦽', name='person in manual wheelchair', group='People & Body', sub_group='person-activity'),
    '🧑🏻‍🦽': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F9BD', status='fully-qualified', emoji='🧑🏻\u200d🦽', name='person in manual wheelchair: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧑🏼‍🦽': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F9BD', status='fully-qualified', emoji='🧑🏼\u200d🦽', name='person in manual wheelchair: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧑🏽‍🦽': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F9BD', status='fully-qualified', emoji='🧑🏽\u200d🦽', name='person in manual wheelchair: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧑🏾‍🦽': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F9BD', status='fully-qualified', emoji='🧑🏾\u200d🦽', name='person in manual wheelchair: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧑🏿‍🦽': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F9BD', status='fully-qualified', emoji='🧑🏿\u200d🦽', name='person in manual wheelchair: dark skin tone', group='People & Body', sub_group='person-activity'),
    '👨‍🦽': EmojiEntry(codepoint='1F468 200D 1F9BD', status='fully-qualified', emoji='👨\u200d🦽', name='man in manual wheelchair', group='People & Body', sub_group='person-activity'),
    '👨🏻‍🦽': EmojiEntry(codepoint='1F468 1F3FB 200D 1F9BD', status='fully-qualified', emoji='👨🏻\u200d🦽', name='man in manual wheelchair: light skin tone', group='People & Body', sub_group='person-activity'),
    '👨🏼‍🦽': EmojiEntry(codepoint='1F468 1F3FC 200D 1F9BD', status='fully-qualified', emoji='👨🏼\u200d🦽', name='man in manual wheelchair: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '👨🏽‍🦽': EmojiEntry(codepoint='1F468 1F3FD 200D 1F9BD', status='fully-qualified', emoji='👨🏽\u200d🦽', name='man in manual wheelchair: medium skin tone', group='People & Body', sub_group='person-activity'),
    '👨🏾‍🦽': EmojiEntry(codepoint='1F468 1F3FE 200D 1F9BD', status='fully-qualified', emoji='👨🏾\u200d🦽', name='man in manual wheelchair: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '👨🏿‍🦽': EmojiEntry(codepoint='1F468 1F3FF 200D 1F9BD', status='fully-qualified', emoji='👨🏿\u200d🦽', name='man in manual wheelchair: dark skin tone', group='People & Body', sub_group='person-activity'),
    '👩‍🦽': EmojiEntry(codepoint='1F469 200D 1F9BD', status='fully-qualified', emoji='👩\u200d🦽', name='woman in manual wheelchair', group='People & Body', sub_group='person-activity'),
    '👩🏻‍🦽': EmojiEntry(codepoint='1F469 1F3FB 200D 1F9BD', status='fully-qualified', emoji='👩🏻\u200d🦽', name='woman in manual wheelchair: light skin tone', group='People & Body', sub_group='person-activity'),
    '👩🏼‍🦽': EmojiEntry(codepoint='1F469 1F3FC 200D 1F9BD', status='fully-qualified', emoji='👩🏼\u200d🦽', name='woman in manual wheelchair: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '👩🏽‍🦽': EmojiEntry(codepoint='1F469 1F3FD 200D 1F9BD', status='fully-qualified', emoji='👩🏽\u200d🦽', name='woman in manual wheelchair: medium skin tone', group='People & Body', sub_group='person-activity'),
    '👩🏾‍🦽': EmojiEntry(codepoint='1F469 1F3FE 200D 1F9BD', status='fully-qualified', emoji='👩🏾\u200d🦽', name='woman in manual wheelchair: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '👩🏿‍🦽': EmojiEntry(codepoint='1F469 1F3FF 200D 1F9BD', status='fully-qualified', emoji='👩🏿\u200d🦽', name='woman in manual wheelchair: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🏃': EmojiEntry(codepoint='1F3C3', status='fully-qualified', emoji='🏃', name='person running', group='People & Body', sub_group='person-activity'),
    '🏃🏻': EmojiEntry(codepoint='1F3C3 1F3FB', status='fully-qualified', emoji='🏃🏻', name='person running: light skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏼': EmojiEntry(codepoint='1F3C3 1F3FC', status='fully-qualified', emoji='🏃🏼', name='person running: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏽': EmojiEntry(codepoint='1F3C3 1F3FD', status='fully-qualified', emoji='🏃🏽', name='person running: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏾': EmojiEntry(codepoint='1F3C3 1F3FE', status='fully-qualified', emoji='🏃🏾', name='person running: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏿': EmojiEntry(codepoint='1F3C3 1F3FF', status='fully-qualified', emoji='🏃🏿', name='person running: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🏃‍♂️': EmojiEntry(codepoint='1F3C3 200D 2642 FE0F', status='fully-qualified', emoji='🏃\u200d♂️', name='man running', group='People & Body', sub_group='person-activity'),
    '🏃‍♂': EmojiEntry(codepoint='1F3C3 200D 2642', status='minimally-qualified', emoji='🏃\u200d♂', name='man running', group='People & Body', sub_group='person-activity'),
    '🏃🏻‍♂️': EmojiEntry(codepoint='1F3C3 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🏃🏻\u200d♂️', name='man running: light skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏻‍♂': EmojiEntry(codepoint='1F3C3 1F3FB 200D 2642', status='minimally-qualified', emoji='🏃🏻\u200d♂', name='man running: light skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏼‍♂️': EmojiEntry(codepoint='1F3C3 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🏃🏼\u200d♂️', name='man running: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏼‍♂': EmojiEntry(codepoint='1F3C3 1F3FC 200D 2642', status='minimally-qualified', emoji='🏃🏼\u200d♂', name='man running: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏽‍♂️': EmojiEntry(codepoint='1F3C3 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🏃🏽\u200d♂️', name='man running: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏽‍♂': EmojiEntry(codepoint='1F3C3 1F3FD 200D 2642', status='minimally-qualified', emoji='🏃🏽\u200d♂', name='man running: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏾‍♂️': EmojiEntry(codepoint='1F3C3 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🏃🏾\u200d♂️', name='man running: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏾‍♂': EmojiEntry(codepoint='1F3C3 1F3FE 200D 2642', status='minimally-qualified', emoji='🏃🏾\u200d♂', name='man running: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏿‍♂️': EmojiEntry(codepoint='1F3C3 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🏃🏿\u200d♂️', name='man running: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏿‍♂': EmojiEntry(codepoint='1F3C3 1F3FF 200D 2642', status='minimally-qualified', emoji='🏃🏿\u200d♂', name='man running: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🏃‍♀️': EmojiEntry(codepoint='1F3C3 200D 2640 FE0F', status='fully-qualified', emoji='🏃\u200d♀️', name='woman running', group='People & Body', sub_group='person-activity'),
    '🏃‍♀': EmojiEntry(codepoint='1F3C3 200D 2640', status='minimally-qualified', emoji='🏃\u200d♀', name='woman running', group='People & Body', sub_group='person-activity'),
    '🏃🏻‍♀️': EmojiEntry(codepoint='1F3C3 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🏃🏻\u200d♀️', name='woman running: light skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏻‍♀': EmojiEntry(codepoint='1F3C3 1F3FB 200D 2640', status='minimally-qualified', emoji='🏃🏻\u200d♀', name='woman running: light skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏼‍♀️': EmojiEntry(codepoint='1F3C3 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🏃🏼\u200d♀️', name='woman running: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏼‍♀': EmojiEntry(codepoint='1F3C3 1F3FC 200D 2640', status='minimally-qualified', emoji='🏃🏼\u200d♀', name='woman running: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏽‍♀️': EmojiEntry(codepoint='1F3C3 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🏃🏽\u200d♀️', name='woman running: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏽‍♀': EmojiEntry(codepoint='1F3C3 1F3FD 200D 2640', status='minimally-qualified', emoji='🏃🏽\u200d♀', name='woman running: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏾‍♀️': EmojiEntry(codepoint='1F3C3 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🏃🏾\u200d♀️', name='woman running: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏾‍♀': EmojiEntry(codepoint='1F3C3 1F3FE 200D 2640', status='minimally-qualified', emoji='🏃🏾\u200d♀', name='woman running: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏿‍♀️': EmojiEntry(codepoint='1F3C3 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🏃🏿\u200d♀️', name='woman running: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🏃🏿‍♀': EmojiEntry(codepoint='1F3C3 1F3FF 200D 2640', status='minimally-qualified', emoji='🏃🏿\u200d♀', name='woman running: dark skin tone', group='People & Body', sub_group='person-activity'),
    '💃': EmojiEntry(codepoint='1F483', status='fully-qualified', emoji='💃', name='woman dancing', group='People & Body', sub_group='person-activity'),
    '💃🏻': EmojiEntry(codepoint='1F483 1F3FB', status='fully-qualified', emoji='💃🏻', name='woman dancing: light skin tone', group='People & Body', sub_group='person-activity'),
    '💃🏼': EmojiEntry(codepoint='1F483 1F3FC', status='fully-qualified', emoji='💃🏼', name='woman dancing: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '💃🏽': EmojiEntry(codepoint='1F483 1F3FD', status='fully-qualified', emoji='💃🏽', name='woman dancing: medium skin tone', group='People & Body', sub_group='person-activity'),
    '💃🏾': EmojiEntry(codepoint='1F483 1F3FE', status='fully-qualified', emoji='💃🏾', name='woman dancing: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '💃🏿': EmojiEntry(codepoint='1F483 1F3FF', status='fully-qualified', emoji='💃🏿', name='woman dancing: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🕺': EmojiEntry(codepoint='1F57A', status='fully-qualified', emoji='🕺', name='man dancing', group='People & Body', sub_group='person-activity'),
    '🕺🏻': EmojiEntry(codepoint='1F57A 1F3FB', status='fully-qualified', emoji='🕺🏻', name='man dancing: light skin tone', group='People & Body', sub_group='person-activity'),
    '🕺🏼': EmojiEntry(codepoint='1F57A 1F3FC', status='fully-qualified', emoji='🕺🏼', name='man dancing: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🕺🏽': EmojiEntry(codepoint='1F57A 1F3FD', status='fully-qualified', emoji='🕺🏽', name='man dancing: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🕺🏾': EmojiEntry(codepoint='1F57A 1F3FE', status='fully-qualified', emoji='🕺🏾', name='man dancing: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🕺🏿': EmojiEntry(codepoint='1F57A 1F3FF', status='fully-qualified', emoji='🕺🏿', name='man dancing: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🕴️': EmojiEntry(codepoint='1F574 FE0F', status='fully-qualified', emoji='🕴️', name='person in suit levitating', group='People & Body', sub_group='person-activity'),
    '🕴': EmojiEntry(codepoint='1F574', status='unqualified', emoji='🕴', name='person in suit levitating', group='People & Body', sub_group='person-activity'),
    '🕴🏻': EmojiEntry(codepoint='1F574 1F3FB', status='fully-qualified', emoji='🕴🏻', name='person in suit levitating: light skin tone', group='People & Body', sub_group='person-activity'),
    '🕴🏼': EmojiEntry(codepoint='1F574 1F3FC', status='fully-qualified', emoji='🕴🏼', name='person in suit levitating: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🕴🏽': EmojiEntry(codepoint='1F574 1F3FD', status='fully-qualified', emoji='🕴🏽', name='person in suit levitating: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🕴🏾': EmojiEntry(codepoint='1F574 1F3FE', status='fully-qualified', emoji='🕴🏾', name='person in suit levitating: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🕴🏿': EmojiEntry(codepoint='1F574 1F3FF', status='fully-qualified', emoji='🕴🏿', name='person in suit levitating: dark skin tone', group='People & Body', sub_group='person-activity'),
    '👯': EmojiEntry(codepoint='1F46F', status='fully-qualified', emoji='👯', name='people with bunny ears', group='People & Body', sub_group='person-activity'),
    '👯‍♂️': EmojiEntry(codepoint='1F46F 200D 2642 FE0F', status='fully-qualified', emoji='👯\u200d♂️', name='men with bunny ears', group='People & Body', sub_group='person-activity'),
    '👯‍♂': EmojiEntry(codepoint='1F46F 200D 2642', status='minimally-qualified', emoji='👯\u200d♂', name='men with bunny ears', group='People & Body', sub_group='person-activity'),
    '👯‍♀️': EmojiEntry(codepoint='1F46F 200D 2640 FE0F', status='fully-qualified', emoji='👯\u200d♀️', name='women with bunny ears', group='People & Body', sub_group='person-activity'),
    '👯‍♀': EmojiEntry(codepoint='1F46F 200D 2640', status='minimally-qualified', emoji='👯\u200d♀', name='women with bunny ears', group='People & Body', sub_group='person-activity'),
    '🧖': EmojiEntry(codepoint='1F9D6', status='fully-qualified', emoji='🧖', name='person in steamy room', group='People & Body', sub_group='person-activity'),
    '🧖🏻': EmojiEntry(codepoint='1F9D6 1F3FB', status='fully-qualified', emoji='🧖🏻', name='person in steamy room: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏼': EmojiEntry(codepoint='1F9D6 1F3FC', status='fully-qualified', emoji='🧖🏼', name='person in steamy room: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏽': EmojiEntry(codepoint='1F9D6 1F3FD', status='fully-qualified', emoji='🧖🏽', name='person in steamy room: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏾': EmojiEntry(codepoint='1F9D6 1F3FE', status='fully-qualified', emoji='🧖🏾', name='person in steamy room: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏿': EmojiEntry(codepoint='1F9D6 1F3FF', status='fully-qualified', emoji='🧖🏿', name='person in steamy room: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧖‍♂️': EmojiEntry(codepoint='1F9D6 200D 2642 FE0F', status='fully-qualified', emoji='🧖\u200d♂️', name='man in steamy room', group='People & Body', sub_group='person-activity'),
    '🧖‍♂': EmojiEntry(codepoint='1F9D6 200D 2642', status='minimally-qualified', emoji='🧖\u200d♂', name='man in steamy room', group='People & Body', sub_group='person-activity'),
    '🧖🏻‍♂️': EmojiEntry(codepoint='1F9D6 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🧖🏻\u200d♂️', name='man in steamy room: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏻‍♂': EmojiEntry(codepoint='1F9D6 1F3FB 200D 2642', status='minimally-qualified', emoji='🧖🏻\u200d♂', name='man in steamy room: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏼‍♂️': EmojiEntry(codepoint='1F9D6 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🧖🏼\u200d♂️', name='man in steamy room: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏼‍♂': EmojiEntry(codepoint='1F9D6 1F3FC 200D 2642', status='minimally-qualified', emoji='🧖🏼\u200d♂', name='man in steamy room: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏽‍♂️': EmojiEntry(codepoint='1F9D6 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🧖🏽\u200d♂️', name='man in steamy room: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏽‍♂': EmojiEntry(codepoint='1F9D6 1F3FD 200D 2642', status='minimally-qualified', emoji='🧖🏽\u200d♂', name='man in steamy room: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏾‍♂️': EmojiEntry(codepoint='1F9D6 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🧖🏾\u200d♂️', name='man in steamy room: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏾‍♂': EmojiEntry(codepoint='1F9D6 1F3FE 200D 2642', status='minimally-qualified', emoji='🧖🏾\u200d♂', name='man in steamy room: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏿‍♂️': EmojiEntry(codepoint='1F9D6 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🧖🏿\u200d♂️', name='man in steamy room: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏿‍♂': EmojiEntry(codepoint='1F9D6 1F3FF 200D 2642', status='minimally-qualified', emoji='🧖🏿\u200d♂', name='man in steamy room: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧖‍♀️': EmojiEntry(codepoint='1F9D6 200D 2640 FE0F', status='fully-qualified', emoji='🧖\u200d♀️', name='woman in steamy room', group='People & Body', sub_group='person-activity'),
    '🧖‍♀': EmojiEntry(codepoint='1F9D6 200D 2640', status='minimally-qualified', emoji='🧖\u200d♀', name='woman in steamy room', group='People & Body', sub_group='person-activity'),
    '🧖🏻‍♀️': EmojiEntry(codepoint='1F9D6 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🧖🏻\u200d♀️', name='woman in steamy room: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏻‍♀': EmojiEntry(codepoint='1F9D6 1F3FB 200D 2640', status='minimally-qualified', emoji='🧖🏻\u200d♀', name='woman in steamy room: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏼‍♀️': EmojiEntry(codepoint='1F9D6 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🧖🏼\u200d♀️', name='woman in steamy room: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏼‍♀': EmojiEntry(codepoint='1F9D6 1F3FC 200D 2640', status='minimally-qualified', emoji='🧖🏼\u200d♀', name='woman in steamy room: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏽‍♀️': EmojiEntry(codepoint='1F9D6 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🧖🏽\u200d♀️', name='woman in steamy room: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏽‍♀': EmojiEntry(codepoint='1F9D6 1F3FD 200D 2640', status='minimally-qualified', emoji='🧖🏽\u200d♀', name='woman in steamy room: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏾‍♀️': EmojiEntry(codepoint='1F9D6 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🧖🏾\u200d♀️', name='woman in steamy room: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏾‍♀': EmojiEntry(codepoint='1F9D6 1F3FE 200D 2640', status='minimally-qualified', emoji='🧖🏾\u200d♀', name='woman in steamy room: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏿‍♀️': EmojiEntry(codepoint='1F9D6 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🧖🏿\u200d♀️', name='woman in steamy room: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧖🏿‍♀': EmojiEntry(codepoint='1F9D6 1F3FF 200D 2640', status='minimally-qualified', emoji='🧖🏿\u200d♀', name='woman in steamy room: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧗': EmojiEntry(codepoint='1F9D7', status='fully-qualified', emoji='🧗', name='person climbing', group='People & Body', sub_group='person-activity'),
    '🧗🏻': EmojiEntry(codepoint='1F9D7 1F3FB', status='fully-qualified', emoji='🧗🏻', name='person climbing: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏼': EmojiEntry(codepoint='1F9D7 1F3FC', status='fully-qualified', emoji='🧗🏼', name='person climbing: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏽': EmojiEntry(codepoint='1F9D7 1F3FD', status='fully-qualified', emoji='🧗🏽', name='person climbing: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏾': EmojiEntry(codepoint='1F9D7 1F3FE', status='fully-qualified', emoji='🧗🏾', name='person climbing: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏿': EmojiEntry(codepoint='1F9D7 1F3FF', status='fully-qualified', emoji='🧗🏿', name='person climbing: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧗‍♂️': EmojiEntry(codepoint='1F9D7 200D 2642 FE0F', status='fully-qualified', emoji='🧗\u200d♂️', name='man climbing', group='People & Body', sub_group='person-activity'),
    '🧗‍♂': EmojiEntry(codepoint='1F9D7 200D 2642', status='minimally-qualified', emoji='🧗\u200d♂', name='man climbing', group='People & Body', sub_group='person-activity'),
    '🧗🏻‍♂️': EmojiEntry(codepoint='1F9D7 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🧗🏻\u200d♂️', name='man climbing: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏻‍♂': EmojiEntry(codepoint='1F9D7 1F3FB 200D 2642', status='minimally-qualified', emoji='🧗🏻\u200d♂', name='man climbing: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏼‍♂️': EmojiEntry(codepoint='1F9D7 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🧗🏼\u200d♂️', name='man climbing: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏼‍♂': EmojiEntry(codepoint='1F9D7 1F3FC 200D 2642', status='minimally-qualified', emoji='🧗🏼\u200d♂', name='man climbing: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏽‍♂️': EmojiEntry(codepoint='1F9D7 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🧗🏽\u200d♂️', name='man climbing: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏽‍♂': EmojiEntry(codepoint='1F9D7 1F3FD 200D 2642', status='minimally-qualified', emoji='🧗🏽\u200d♂', name='man climbing: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏾‍♂️': EmojiEntry(codepoint='1F9D7 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🧗🏾\u200d♂️', name='man climbing: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏾‍♂': EmojiEntry(codepoint='1F9D7 1F3FE 200D 2642', status='minimally-qualified', emoji='🧗🏾\u200d♂', name='man climbing: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏿‍♂️': EmojiEntry(codepoint='1F9D7 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🧗🏿\u200d♂️', name='man climbing: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏿‍♂': EmojiEntry(codepoint='1F9D7 1F3FF 200D 2642', status='minimally-qualified', emoji='🧗🏿\u200d♂', name='man climbing: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧗‍♀️': EmojiEntry(codepoint='1F9D7 200D 2640 FE0F', status='fully-qualified', emoji='🧗\u200d♀️', name='woman climbing', group='People & Body', sub_group='person-activity'),
    '🧗‍♀': EmojiEntry(codepoint='1F9D7 200D 2640', status='minimally-qualified', emoji='🧗\u200d♀', name='woman climbing', group='People & Body', sub_group='person-activity'),
    '🧗🏻‍♀️': EmojiEntry(codepoint='1F9D7 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🧗🏻\u200d♀️', name='woman climbing: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏻‍♀': EmojiEntry(codepoint='1F9D7 1F3FB 200D 2640', status='minimally-qualified', emoji='🧗🏻\u200d♀', name='woman climbing: light skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏼‍♀️': EmojiEntry(codepoint='1F9D7 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🧗🏼\u200d♀️', name='woman climbing: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏼‍♀': EmojiEntry(codepoint='1F9D7 1F3FC 200D 2640', status='minimally-qualified', emoji='🧗🏼\u200d♀', name='woman climbing: medium-light skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏽‍♀️': EmojiEntry(codepoint='1F9D7 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🧗🏽\u200d♀️', name='woman climbing: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏽‍♀': EmojiEntry(codepoint='1F9D7 1F3FD 200D 2640', status='minimally-qualified', emoji='🧗🏽\u200d♀', name='woman climbing: medium skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏾‍♀️': EmojiEntry(codepoint='1F9D7 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🧗🏾\u200d♀️', name='woman climbing: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏾‍♀': EmojiEntry(codepoint='1F9D7 1F3FE 200D 2640', status='minimally-qualified', emoji='🧗🏾\u200d♀', name='woman climbing: medium-dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏿‍♀️': EmojiEntry(codepoint='1F9D7 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🧗🏿\u200d♀️', name='woman climbing: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🧗🏿‍♀': EmojiEntry(codepoint='1F9D7 1F3FF 200D 2640', status='minimally-qualified', emoji='🧗🏿\u200d♀', name='woman climbing: dark skin tone', group='People & Body', sub_group='person-activity'),
    '🤺': EmojiEntry(codepoint='1F93A', status='fully-qualified', emoji='🤺', name='person fencing', group='People & Body', sub_group='person-sport'),
    '🏇': EmojiEntry(codepoint='1F3C7', status='fully-qualified', emoji='🏇', name='horse racing', group='People & Body', sub_group='person-sport'),
    '🏇🏻': EmojiEntry(codepoint='1F3C7 1F3FB', status='fully-qualified', emoji='🏇🏻', name='horse racing: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏇🏼': EmojiEntry(codepoint='1F3C7 1F3FC', status='fully-qualified', emoji='🏇🏼', name='horse racing: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏇🏽': EmojiEntry(codepoint='1F3C7 1F3FD', status='fully-qualified', emoji='🏇🏽', name='horse racing: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏇🏾': EmojiEntry(codepoint='1F3C7 1F3FE', status='fully-qualified', emoji='🏇🏾', name='horse racing: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏇🏿': EmojiEntry(codepoint='1F3C7 1F3FF', status='fully-qualified', emoji='🏇🏿', name='horse racing: dark skin tone', group='People & Body', sub_group='person-sport'),
    '⛷️': EmojiEntry(codepoint='26F7 FE0F', status='fully-qualified', emoji='⛷️', name='skier', group='People & Body', sub_group='person-sport'),
    '⛷': EmojiEntry(codepoint='26F7', status='unqualified', emoji='⛷', name='skier', group='People & Body', sub_group='person-sport'),
    '🏂': EmojiEntry(codepoint='1F3C2', status='fully-qualified', emoji='🏂', name='snowboarder', group='People & Body', sub_group='person-sport'),
    '🏂🏻': EmojiEntry(codepoint='1F3C2 1F3FB', status='fully-qualified', emoji='🏂🏻', name='snowboarder: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏂🏼': EmojiEntry(codepoint='1F3C2 1F3FC', status='fully-qualified', emoji='🏂🏼', name='snowboarder: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏂🏽': EmojiEntry(codepoint='1F3C2 1F3FD', status='fully-qualified', emoji='🏂🏽', name='snowboarder: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏂🏾': EmojiEntry(codepoint='1F3C2 1F3FE', status='fully-qualified', emoji='🏂🏾', name='snowboarder: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏂🏿': EmojiEntry(codepoint='1F3C2 1F3FF', status='fully-qualified', emoji='🏂🏿', name='snowboarder: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏌️': EmojiEntry(codepoint='1F3CC FE0F', status='fully-qualified', emoji='🏌️', name='person golfing', group='People & Body', sub_group='person-sport'),
    '🏌': EmojiEntry(codepoint='1F3CC', status='unqualified', emoji='🏌', name='person golfing', group='People & Body', sub_group='person-sport'),
    '🏌🏻': EmojiEntry(codepoint='1F3CC 1F3FB', status='fully-qualified', emoji='🏌🏻', name='person golfing: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏼': EmojiEntry(codepoint='1F3CC 1F3FC', status='fully-qualified', emoji='🏌🏼', name='person golfing: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏽': EmojiEntry(codepoint='1F3CC 1F3FD', status='fully-qualified', emoji='🏌🏽', name='person golfing: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏾': EmojiEntry(codepoint='1F3CC 1F3FE', status='fully-qualified', emoji='🏌🏾', name='person golfing: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏿': EmojiEntry(codepoint='1F3CC 1F3FF', status='fully-qualified', emoji='🏌🏿', name='person golfing: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏌️‍♂️': EmojiEntry(codepoint='1F3CC FE0F 200D 2642 FE0F', status='fully-qualified', emoji='🏌️\u200d♂️', name='man golfing', group='People & Body', sub_group='person-sport'),
    '🏌‍♂️': EmojiEntry(codepoint='1F3CC 200D 2642 FE0F', status='unqualified', emoji='🏌\u200d♂️', name='man golfing', group='People & Body', sub_group='person-sport'),
    '🏌️‍♂': EmojiEntry(codepoint='1F3CC FE0F 200D 2642', status='unqualified', emoji='🏌️\u200d♂', name='man golfing', group='People & Body', sub_group='person-sport'),
    '🏌‍♂': EmojiEntry(codepoint='1F3CC 200D 2642', status='unqualified', emoji='🏌\u200d♂', name='man golfing', group='People & Body', sub_group='person-sport'),
    '🏌🏻‍♂️': EmojiEntry(codepoint='1F3CC 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🏌🏻\u200d♂️', name='man golfing: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏻‍♂': EmojiEntry(codepoint='1F3CC 1F3FB 200D 2642', status='minimally-qualified', emoji='🏌🏻\u200d♂', name='man golfing: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏼‍♂️': EmojiEntry(codepoint='1F3CC 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🏌🏼\u200d♂️', name='man golfing: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏼‍♂': EmojiEntry(codepoint='1F3CC 1F3FC 200D 2642', status='minimally-qualified', emoji='🏌🏼\u200d♂', name='man golfing: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏽‍♂️': EmojiEntry(codepoint='1F3CC 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🏌🏽\u200d♂️', name='man golfing: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏽‍♂': EmojiEntry(codepoint='1F3CC 1F3FD 200D 2642', status='minimally-qualified', emoji='🏌🏽\u200d♂', name='man golfing: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏾‍♂️': EmojiEntry(codepoint='1F3CC 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🏌🏾\u200d♂️', name='man golfing: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏾‍♂': EmojiEntry(codepoint='1F3CC 1F3FE 200D 2642', status='minimally-qualified', emoji='🏌🏾\u200d♂', name='man golfing: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏿‍♂️': EmojiEntry(codepoint='1F3CC 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🏌🏿\u200d♂️', name='man golfing: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏿‍♂': EmojiEntry(codepoint='1F3CC 1F3FF 200D 2642', status='minimally-qualified', emoji='🏌🏿\u200d♂', name='man golfing: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏌️‍♀️': EmojiEntry(codepoint='1F3CC FE0F 200D 2640 FE0F', status='fully-qualified', emoji='🏌️\u200d♀️', name='woman golfing', group='People & Body', sub_group='person-sport'),
    '🏌‍♀️': EmojiEntry(codepoint='1F3CC 200D 2640 FE0F', status='unqualified', emoji='🏌\u200d♀️', name='woman golfing', group='People & Body', sub_group='person-sport'),
    '🏌️‍♀': EmojiEntry(codepoint='1F3CC FE0F 200D 2640', status='unqualified', emoji='🏌️\u200d♀', name='woman golfing', group='People & Body', sub_group='person-sport'),
    '🏌‍♀': EmojiEntry(codepoint='1F3CC 200D 2640', status='unqualified', emoji='🏌\u200d♀', name='woman golfing', group='People & Body', sub_group='person-sport'),
    '🏌🏻‍♀️': EmojiEntry(codepoint='1F3CC 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🏌🏻\u200d♀️', name='woman golfing: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏻‍♀': EmojiEntry(codepoint='1F3CC 1F3FB 200D 2640', status='minimally-qualified', emoji='🏌🏻\u200d♀', name='woman golfing: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏼‍♀️': EmojiEntry(codepoint='1F3CC 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🏌🏼\u200d♀️', name='woman golfing: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏼‍♀': EmojiEntry(codepoint='1F3CC 1F3FC 200D 2640', status='minimally-qualified', emoji='🏌🏼\u200d♀', name='woman golfing: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏽‍♀️': EmojiEntry(codepoint='1F3CC 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🏌🏽\u200d♀️', name='woman golfing: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏽‍♀': EmojiEntry(codepoint='1F3CC 1F3FD 200D 2640', status='minimally-qualified', emoji='🏌🏽\u200d♀', name='woman golfing: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏾‍♀️': EmojiEntry(codepoint='1F3CC 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🏌🏾\u200d♀️', name='woman golfing: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏾‍♀': EmojiEntry(codepoint='1F3CC 1F3FE 200D 2640', status='minimally-qualified', emoji='🏌🏾\u200d♀', name='woman golfing: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏿‍♀️': EmojiEntry(codepoint='1F3CC 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🏌🏿\u200d♀️', name='woman golfing: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏌🏿‍♀': EmojiEntry(codepoint='1F3CC 1F3FF 200D 2640', status='minimally-qualified', emoji='🏌🏿\u200d♀', name='woman golfing: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏄': EmojiEntry(codepoint='1F3C4', status='fully-qualified', emoji='🏄', name='person surfing', group='People & Body', sub_group='person-sport'),
    '🏄🏻': EmojiEntry(codepoint='1F3C4 1F3FB', status='fully-qualified', emoji='🏄🏻', name='person surfing: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏼': EmojiEntry(codepoint='1F3C4 1F3FC', status='fully-qualified', emoji='🏄🏼', name='person surfing: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏽': EmojiEntry(codepoint='1F3C4 1F3FD', status='fully-qualified', emoji='🏄🏽', name='person surfing: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏾': EmojiEntry(codepoint='1F3C4 1F3FE', status='fully-qualified', emoji='🏄🏾', name='person surfing: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏿': EmojiEntry(codepoint='1F3C4 1F3FF', status='fully-qualified', emoji='🏄🏿', name='person surfing: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏄‍♂️': EmojiEntry(codepoint='1F3C4 200D 2642 FE0F', status='fully-qualified', emoji='🏄\u200d♂️', name='man surfing', group='People & Body', sub_group='person-sport'),
    '🏄‍♂': EmojiEntry(codepoint='1F3C4 200D 2642', status='minimally-qualified', emoji='🏄\u200d♂', name='man surfing', group='People & Body', sub_group='person-sport'),
    '🏄🏻‍♂️': EmojiEntry(codepoint='1F3C4 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🏄🏻\u200d♂️', name='man surfing: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏻‍♂': EmojiEntry(codepoint='1F3C4 1F3FB 200D 2642', status='minimally-qualified', emoji='🏄🏻\u200d♂', name='man surfing: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏼‍♂️': EmojiEntry(codepoint='1F3C4 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🏄🏼\u200d♂️', name='man surfing: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏼‍♂': EmojiEntry(codepoint='1F3C4 1F3FC 200D 2642', status='minimally-qualified', emoji='🏄🏼\u200d♂', name='man surfing: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏽‍♂️': EmojiEntry(codepoint='1F3C4 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🏄🏽\u200d♂️', name='man surfing: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏽‍♂': EmojiEntry(codepoint='1F3C4 1F3FD 200D 2642', status='minimally-qualified', emoji='🏄🏽\u200d♂', name='man surfing: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏾‍♂️': EmojiEntry(codepoint='1F3C4 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🏄🏾\u200d♂️', name='man surfing: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏾‍♂': EmojiEntry(codepoint='1F3C4 1F3FE 200D 2642', status='minimally-qualified', emoji='🏄🏾\u200d♂', name='man surfing: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏿‍♂️': EmojiEntry(codepoint='1F3C4 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🏄🏿\u200d♂️', name='man surfing: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏿‍♂': EmojiEntry(codepoint='1F3C4 1F3FF 200D 2642', status='minimally-qualified', emoji='🏄🏿\u200d♂', name='man surfing: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏄‍♀️': EmojiEntry(codepoint='1F3C4 200D 2640 FE0F', status='fully-qualified', emoji='🏄\u200d♀️', name='woman surfing', group='People & Body', sub_group='person-sport'),
    '🏄‍♀': EmojiEntry(codepoint='1F3C4 200D 2640', status='minimally-qualified', emoji='🏄\u200d♀', name='woman surfing', group='People & Body', sub_group='person-sport'),
    '🏄🏻‍♀️': EmojiEntry(codepoint='1F3C4 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🏄🏻\u200d♀️', name='woman surfing: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏻‍♀': EmojiEntry(codepoint='1F3C4 1F3FB 200D 2640', status='minimally-qualified', emoji='🏄🏻\u200d♀', name='woman surfing: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏼‍♀️': EmojiEntry(codepoint='1F3C4 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🏄🏼\u200d♀️', name='woman surfing: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏼‍♀': EmojiEntry(codepoint='1F3C4 1F3FC 200D 2640', status='minimally-qualified', emoji='🏄🏼\u200d♀', name='woman surfing: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏽‍♀️': EmojiEntry(codepoint='1F3C4 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🏄🏽\u200d♀️', name='woman surfing: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏽‍♀': EmojiEntry(codepoint='1F3C4 1F3FD 200D 2640', status='minimally-qualified', emoji='🏄🏽\u200d♀', name='woman surfing: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏾‍♀️': EmojiEntry(codepoint='1F3C4 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🏄🏾\u200d♀️', name='woman surfing: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏾‍♀': EmojiEntry(codepoint='1F3C4 1F3FE 200D 2640', status='minimally-qualified', emoji='🏄🏾\u200d♀', name='woman surfing: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏿‍♀️': EmojiEntry(codepoint='1F3C4 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🏄🏿\u200d♀️', name='woman surfing: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏄🏿‍♀': EmojiEntry(codepoint='1F3C4 1F3FF 200D 2640', status='minimally-qualified', emoji='🏄🏿\u200d♀', name='woman surfing: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚣': EmojiEntry(codepoint='1F6A3', status='fully-qualified', emoji='🚣', name='person rowing boat', group='People & Body', sub_group='person-sport'),
    '🚣🏻': EmojiEntry(codepoint='1F6A3 1F3FB', status='fully-qualified', emoji='🚣🏻', name='person rowing boat: light skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏼': EmojiEntry(codepoint='1F6A3 1F3FC', status='fully-qualified', emoji='🚣🏼', name='person rowing boat: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏽': EmojiEntry(codepoint='1F6A3 1F3FD', status='fully-qualified', emoji='🚣🏽', name='person rowing boat: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏾': EmojiEntry(codepoint='1F6A3 1F3FE', status='fully-qualified', emoji='🚣🏾', name='person rowing boat: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏿': EmojiEntry(codepoint='1F6A3 1F3FF', status='fully-qualified', emoji='🚣🏿', name='person rowing boat: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚣‍♂️': EmojiEntry(codepoint='1F6A3 200D 2642 FE0F', status='fully-qualified', emoji='🚣\u200d♂️', name='man rowing boat', group='People & Body', sub_group='person-sport'),
    '🚣‍♂': EmojiEntry(codepoint='1F6A3 200D 2642', status='minimally-qualified', emoji='🚣\u200d♂', name='man rowing boat', group='People & Body', sub_group='person-sport'),
    '🚣🏻‍♂️': EmojiEntry(codepoint='1F6A3 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🚣🏻\u200d♂️', name='man rowing boat: light skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏻‍♂': EmojiEntry(codepoint='1F6A3 1F3FB 200D 2642', status='minimally-qualified', emoji='🚣🏻\u200d♂', name='man rowing boat: light skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏼‍♂️': EmojiEntry(codepoint='1F6A3 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🚣🏼\u200d♂️', name='man rowing boat: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏼‍♂': EmojiEntry(codepoint='1F6A3 1F3FC 200D 2642', status='minimally-qualified', emoji='🚣🏼\u200d♂', name='man rowing boat: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏽‍♂️': EmojiEntry(codepoint='1F6A3 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🚣🏽\u200d♂️', name='man rowing boat: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏽‍♂': EmojiEntry(codepoint='1F6A3 1F3FD 200D 2642', status='minimally-qualified', emoji='🚣🏽\u200d♂', name='man rowing boat: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏾‍♂️': EmojiEntry(codepoint='1F6A3 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🚣🏾\u200d♂️', name='man rowing boat: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏾‍♂': EmojiEntry(codepoint='1F6A3 1F3FE 200D 2642', status='minimally-qualified', emoji='🚣🏾\u200d♂', name='man rowing boat: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏿‍♂️': EmojiEntry(codepoint='1F6A3 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🚣🏿\u200d♂️', name='man rowing boat: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏿‍♂': EmojiEntry(codepoint='1F6A3 1F3FF 200D 2642', status='minimally-qualified', emoji='🚣🏿\u200d♂', name='man rowing boat: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚣‍♀️': EmojiEntry(codepoint='1F6A3 200D 2640 FE0F', status='fully-qualified', emoji='🚣\u200d♀️', name='woman rowing boat', group='People & Body', sub_group='person-sport'),
    '🚣‍♀': EmojiEntry(codepoint='1F6A3 200D 2640', status='minimally-qualified', emoji='🚣\u200d♀', name='woman rowing boat', group='People & Body', sub_group='person-sport'),
    '🚣🏻‍♀️': EmojiEntry(codepoint='1F6A3 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🚣🏻\u200d♀️', name='woman rowing boat: light skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏻‍♀': EmojiEntry(codepoint='1F6A3 1F3FB 200D 2640', status='minimally-qualified', emoji='🚣🏻\u200d♀', name='woman rowing boat: light skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏼‍♀️': EmojiEntry(codepoint='1F6A3 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🚣🏼\u200d♀️', name='woman rowing boat: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏼‍♀': EmojiEntry(codepoint='1F6A3 1F3FC 200D 2640', status='minimally-qualified', emoji='🚣🏼\u200d♀', name='woman rowing boat: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏽‍♀️': EmojiEntry(codepoint='1F6A3 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🚣🏽\u200d♀️', name='woman rowing boat: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏽‍♀': EmojiEntry(codepoint='1F6A3 1F3FD 200D 2640', status='minimally-qualified', emoji='🚣🏽\u200d♀', name='woman rowing boat: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏾‍♀️': EmojiEntry(codepoint='1F6A3 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🚣🏾\u200d♀️', name='woman rowing boat: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏾‍♀': EmojiEntry(codepoint='1F6A3 1F3FE 200D 2640', status='minimally-qualified', emoji='🚣🏾\u200d♀', name='woman rowing boat: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏿‍♀️': EmojiEntry(codepoint='1F6A3 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🚣🏿\u200d♀️', name='woman rowing boat: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚣🏿‍♀': EmojiEntry(codepoint='1F6A3 1F3FF 200D 2640', status='minimally-qualified', emoji='🚣🏿\u200d♀', name='woman rowing boat: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏊': EmojiEntry(codepoint='1F3CA', status='fully-qualified', emoji='🏊', name='person swimming', group='People & Body', sub_group='person-sport'),
    '🏊🏻': EmojiEntry(codepoint='1F3CA 1F3FB', status='fully-qualified', emoji='🏊🏻', name='person swimming: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏼': EmojiEntry(codepoint='1F3CA 1F3FC', status='fully-qualified', emoji='🏊🏼', name='person swimming: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏽': EmojiEntry(codepoint='1F3CA 1F3FD', status='fully-qualified', emoji='🏊🏽', name='person swimming: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏾': EmojiEntry(codepoint='1F3CA 1F3FE', status='fully-qualified', emoji='🏊🏾', name='person swimming: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏿': EmojiEntry(codepoint='1F3CA 1F3FF', status='fully-qualified', emoji='🏊🏿', name='person swimming: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏊‍♂️': EmojiEntry(codepoint='1F3CA 200D 2642 FE0F', status='fully-qualified', emoji='🏊\u200d♂️', name='man swimming', group='People & Body', sub_group='person-sport'),
    '🏊‍♂': EmojiEntry(codepoint='1F3CA 200D 2642', status='minimally-qualified', emoji='🏊\u200d♂', name='man swimming', group='People & Body', sub_group='person-sport'),
    '🏊🏻‍♂️': EmojiEntry(codepoint='1F3CA 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🏊🏻\u200d♂️', name='man swimming: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏻‍♂': EmojiEntry(codepoint='1F3CA 1F3FB 200D 2642', status='minimally-qualified', emoji='🏊🏻\u200d♂', name='man swimming: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏼‍♂️': EmojiEntry(codepoint='1F3CA 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🏊🏼\u200d♂️', name='man swimming: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏼‍♂': EmojiEntry(codepoint='1F3CA 1F3FC 200D 2642', status='minimally-qualified', emoji='🏊🏼\u200d♂', name='man swimming: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏽‍♂️': EmojiEntry(codepoint='1F3CA 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🏊🏽\u200d♂️', name='man swimming: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏽‍♂': EmojiEntry(codepoint='1F3CA 1F3FD 200D 2642', status='minimally-qualified', emoji='🏊🏽\u200d♂', name='man swimming: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏾‍♂️': EmojiEntry(codepoint='1F3CA 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🏊🏾\u200d♂️', name='man swimming: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏾‍♂': EmojiEntry(codepoint='1F3CA 1F3FE 200D 2642', status='minimally-qualified', emoji='🏊🏾\u200d♂', name='man swimming: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏿‍♂️': EmojiEntry(codepoint='1F3CA 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🏊🏿\u200d♂️', name='man swimming: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏿‍♂': EmojiEntry(codepoint='1F3CA 1F3FF 200D 2642', status='minimally-qualified', emoji='🏊🏿\u200d♂', name='man swimming: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏊‍♀️': EmojiEntry(codepoint='1F3CA 200D 2640 FE0F', status='fully-qualified', emoji='🏊\u200d♀️', name='woman swimming', group='People & Body', sub_group='person-sport'),
    '🏊‍♀': EmojiEntry(codepoint='1F3CA 200D 2640', status='minimally-qualified', emoji='🏊\u200d♀', name='woman swimming', group='People & Body', sub_group='person-sport'),
    '🏊🏻‍♀️': EmojiEntry(codepoint='1F3CA 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🏊🏻\u200d♀️', name='woman swimming: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏻‍♀': EmojiEntry(codepoint='1F3CA 1F3FB 200D 2640', status='minimally-qualified', emoji='🏊🏻\u200d♀', name='woman swimming: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏼‍♀️': EmojiEntry(codepoint='1F3CA 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🏊🏼\u200d♀️', name='woman swimming: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏼‍♀': EmojiEntry(codepoint='1F3CA 1F3FC 200D 2640', status='minimally-qualified', emoji='🏊🏼\u200d♀', name='woman swimming: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏽‍♀️': EmojiEntry(codepoint='1F3CA 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🏊🏽\u200d♀️', name='woman swimming: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏽‍♀': EmojiEntry(codepoint='1F3CA 1F3FD 200D 2640', status='minimally-qualified', emoji='🏊🏽\u200d♀', name='woman swimming: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏾‍♀️': EmojiEntry(codepoint='1F3CA 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🏊🏾\u200d♀️', name='woman swimming: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏾‍♀': EmojiEntry(codepoint='1F3CA 1F3FE 200D 2640', status='minimally-qualified', emoji='🏊🏾\u200d♀', name='woman swimming: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏿‍♀️': EmojiEntry(codepoint='1F3CA 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🏊🏿\u200d♀️', name='woman swimming: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏊🏿‍♀': EmojiEntry(codepoint='1F3CA 1F3FF 200D 2640', status='minimally-qualified', emoji='🏊🏿\u200d♀', name='woman swimming: dark skin tone', group='People & Body', sub_group='person-sport'),
    '⛹️': EmojiEntry(codepoint='26F9 FE0F', status='fully-qualified', emoji='⛹️', name='person bouncing ball', group='People & Body', sub_group='person-sport'),
    '⛹': EmojiEntry(codepoint='26F9', status='unqualified', emoji='⛹', name='person bouncing ball', group='People & Body', sub_group='person-sport'),
    '⛹🏻': EmojiEntry(codepoint='26F9 1F3FB', status='fully-qualified', emoji='⛹🏻', name='person bouncing ball: light skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏼': EmojiEntry(codepoint='26F9 1F3FC', status='fully-qualified', emoji='⛹🏼', name='person bouncing ball: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏽': EmojiEntry(codepoint='26F9 1F3FD', status='fully-qualified', emoji='⛹🏽', name='person bouncing ball: medium skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏾': EmojiEntry(codepoint='26F9 1F3FE', status='fully-qualified', emoji='⛹🏾', name='person bouncing ball: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏿': EmojiEntry(codepoint='26F9 1F3FF', status='fully-qualified', emoji='⛹🏿', name='person bouncing ball: dark skin tone', group='People & Body', sub_group='person-sport'),
    '⛹️‍♂️': EmojiEntry(codepoint='26F9 FE0F 200D 2642 FE0F', status='fully-qualified', emoji='⛹️\u200d♂️', name='man bouncing ball', group='People & Body', sub_group='person-sport'),
    '⛹‍♂️': EmojiEntry(codepoint='26F9 200D 2642 FE0F', status='unqualified', emoji='⛹\u200d♂️', name='man bouncing ball', group='People & Body', sub_group='person-sport'),
    '⛹️‍♂': EmojiEntry(codepoint='26F9 FE0F 200D 2642', status='unqualified', emoji='⛹️\u200d♂', name='man bouncing ball', group='People & Body', sub_group='person-sport'),
    '⛹‍♂': EmojiEntry(codepoint='26F9 200D 2642', status='unqualified', emoji='⛹\u200d♂', name='man bouncing ball', group='People & Body', sub_group='person-sport'),
    '⛹🏻‍♂️': EmojiEntry(codepoint='26F9 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='⛹🏻\u200d♂️', name='man bouncing ball: light skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏻‍♂': EmojiEntry(codepoint='26F9 1F3FB 200D 2642', status='minimally-qualified', emoji='⛹🏻\u200d♂', name='man bouncing ball: light skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏼‍♂️': EmojiEntry(codepoint='26F9 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='⛹🏼\u200d♂️', name='man bouncing ball: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏼‍♂': EmojiEntry(codepoint='26F9 1F3FC 200D 2642', status='minimally-qualified', emoji='⛹🏼\u200d♂', name='man bouncing ball: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏽‍♂️': EmojiEntry(codepoint='26F9 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='⛹🏽\u200d♂️', name='man bouncing ball: medium skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏽‍♂': EmojiEntry(codepoint='26F9 1F3FD 200D 2642', status='minimally-qualified', emoji='⛹🏽\u200d♂', name='man bouncing ball: medium skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏾‍♂️': EmojiEntry(codepoint='26F9 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='⛹🏾\u200d♂️', name='man bouncing ball: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏾‍♂': EmojiEntry(codepoint='26F9 1F3FE 200D 2642', status='minimally-qualified', emoji='⛹🏾\u200d♂', name='man bouncing ball: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏿‍♂️': EmojiEntry(codepoint='26F9 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='⛹🏿\u200d♂️', name='man bouncing ball: dark skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏿‍♂': EmojiEntry(codepoint='26F9 1F3FF 200D 2642', status='minimally-qualified', emoji='⛹🏿\u200d♂', name='man bouncing ball: dark skin tone', group='People & Body', sub_group='person-sport'),
    '⛹️‍♀️': EmojiEntry(codepoint='26F9 FE0F 200D 2640 FE0F', status='fully-qualified', emoji='⛹️\u200d♀️', name='woman bouncing ball', group='People & Body', sub_group='person-sport'),
    '⛹‍♀️': EmojiEntry(codepoint='26F9 200D 2640 FE0F', status='unqualified', emoji='⛹\u200d♀️', name='woman bouncing ball', group='People & Body', sub_group='person-sport'),
    '⛹️‍♀': EmojiEntry(codepoint='26F9 FE0F 200D 2640', status='unqualified', emoji='⛹️\u200d♀', name='woman bouncing ball', group='People & Body', sub_group='person-sport'),
    '⛹‍♀': EmojiEntry(codepoint='26F9 200D 2640', status='unqualified', emoji='⛹\u200d♀', name='woman bouncing ball', group='People & Body', sub_group='person-sport'),
    '⛹🏻‍♀️': EmojiEntry(codepoint='26F9 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='⛹🏻\u200d♀️', name='woman bouncing ball: light skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏻‍♀': EmojiEntry(codepoint='26F9 1F3FB 200D 2640', status='minimally-qualified', emoji='⛹🏻\u200d♀', name='woman bouncing ball: light skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏼‍♀️': EmojiEntry(codepoint='26F9 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='⛹🏼\u200d♀️', name='woman bouncing ball: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏼‍♀': EmojiEntry(codepoint='26F9 1F3FC 200D 2640', status='minimally-qualified', emoji='⛹🏼\u200d♀', name='woman bouncing ball: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏽‍♀️': EmojiEntry(codepoint='26F9 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='⛹🏽\u200d♀️', name='woman bouncing ball: medium skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏽‍♀': EmojiEntry(codepoint='26F9 1F3FD 200D 2640', status='minimally-qualified', emoji='⛹🏽\u200d♀', name='woman bouncing ball: medium skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏾‍♀️': EmojiEntry(codepoint='26F9 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='⛹🏾\u200d♀️', name='woman bouncing ball: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏾‍♀': EmojiEntry(codepoint='26F9 1F3FE 200D 2640', status='minimally-qualified', emoji='⛹🏾\u200d♀', name='woman bouncing ball: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏿‍♀️': EmojiEntry(codepoint='26F9 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='⛹🏿\u200d♀️', name='woman bouncing ball: dark skin tone', group='People & Body', sub_group='person-sport'),
    '⛹🏿‍♀': EmojiEntry(codepoint='26F9 1F3FF 200D 2640', status='minimally-qualified', emoji='⛹🏿\u200d♀', name='woman bouncing ball: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏋️': EmojiEntry(codepoint='1F3CB FE0F', status='fully-qualified', emoji='🏋️', name='person lifting weights', group='People & Body', sub_group='person-sport'),
    '🏋': EmojiEntry(codepoint='1F3CB', status='unqualified', emoji='🏋', name='person lifting weights', group='People & Body', sub_group='person-sport'),
    '🏋🏻': EmojiEntry(codepoint='1F3CB 1F3FB', status='fully-qualified', emoji='🏋🏻', name='person lifting weights: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏼': EmojiEntry(codepoint='1F3CB 1F3FC', status='fully-qualified', emoji='🏋🏼', name='person lifting weights: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏽': EmojiEntry(codepoint='1F3CB 1F3FD', status='fully-qualified', emoji='🏋🏽', name='person lifting weights: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏾': EmojiEntry(codepoint='1F3CB 1F3FE', status='fully-qualified', emoji='🏋🏾', name='person lifting weights: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏿': EmojiEntry(codepoint='1F3CB 1F3FF', status='fully-qualified', emoji='🏋🏿', name='person lifting weights: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏋️‍♂️': EmojiEntry(codepoint='1F3CB FE0F 200D 2642 FE0F', status='fully-qualified', emoji='🏋️\u200d♂️', name='man lifting weights', group='People & Body', sub_group='person-sport'),
    '🏋‍♂️': EmojiEntry(codepoint='1F3CB 200D 2642 FE0F', status='unqualified', emoji='🏋\u200d♂️', name='man lifting weights', group='People & Body', sub_group='person-sport'),
    '🏋️‍♂': EmojiEntry(codepoint='1F3CB FE0F 200D 2642', status='unqualified', emoji='🏋️\u200d♂', name='man lifting weights', group='People & Body', sub_group='person-sport'),
    '🏋‍♂': EmojiEntry(codepoint='1F3CB 200D 2642', status='unqualified', emoji='🏋\u200d♂', name='man lifting weights', group='People & Body', sub_group='person-sport'),
    '🏋🏻‍♂️': EmojiEntry(codepoint='1F3CB 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🏋🏻\u200d♂️', name='man lifting weights: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏻‍♂': EmojiEntry(codepoint='1F3CB 1F3FB 200D 2642', status='minimally-qualified', emoji='🏋🏻\u200d♂', name='man lifting weights: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏼‍♂️': EmojiEntry(codepoint='1F3CB 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🏋🏼\u200d♂️', name='man lifting weights: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏼‍♂': EmojiEntry(codepoint='1F3CB 1F3FC 200D 2642', status='minimally-qualified', emoji='🏋🏼\u200d♂', name='man lifting weights: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏽‍♂️': EmojiEntry(codepoint='1F3CB 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🏋🏽\u200d♂️', name='man lifting weights: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏽‍♂': EmojiEntry(codepoint='1F3CB 1F3FD 200D 2642', status='minimally-qualified', emoji='🏋🏽\u200d♂', name='man lifting weights: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏾‍♂️': EmojiEntry(codepoint='1F3CB 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🏋🏾\u200d♂️', name='man lifting weights: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏾‍♂': EmojiEntry(codepoint='1F3CB 1F3FE 200D 2642', status='minimally-qualified', emoji='🏋🏾\u200d♂', name='man lifting weights: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏿‍♂️': EmojiEntry(codepoint='1F3CB 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🏋🏿\u200d♂️', name='man lifting weights: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏿‍♂': EmojiEntry(codepoint='1F3CB 1F3FF 200D 2642', status='minimally-qualified', emoji='🏋🏿\u200d♂', name='man lifting weights: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏋️‍♀️': EmojiEntry(codepoint='1F3CB FE0F 200D 2640 FE0F', status='fully-qualified', emoji='🏋️\u200d♀️', name='woman lifting weights', group='People & Body', sub_group='person-sport'),
    '🏋‍♀️': EmojiEntry(codepoint='1F3CB 200D 2640 FE0F', status='unqualified', emoji='🏋\u200d♀️', name='woman lifting weights', group='People & Body', sub_group='person-sport'),
    '🏋️‍♀': EmojiEntry(codepoint='1F3CB FE0F 200D 2640', status='unqualified', emoji='🏋️\u200d♀', name='woman lifting weights', group='People & Body', sub_group='person-sport'),
    '🏋‍♀': EmojiEntry(codepoint='1F3CB 200D 2640', status='unqualified', emoji='🏋\u200d♀', name='woman lifting weights', group='People & Body', sub_group='person-sport'),
    '🏋🏻‍♀️': EmojiEntry(codepoint='1F3CB 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🏋🏻\u200d♀️', name='woman lifting weights: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏻‍♀': EmojiEntry(codepoint='1F3CB 1F3FB 200D 2640', status='minimally-qualified', emoji='🏋🏻\u200d♀', name='woman lifting weights: light skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏼‍♀️': EmojiEntry(codepoint='1F3CB 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🏋🏼\u200d♀️', name='woman lifting weights: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏼‍♀': EmojiEntry(codepoint='1F3CB 1F3FC 200D 2640', status='minimally-qualified', emoji='🏋🏼\u200d♀', name='woman lifting weights: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏽‍♀️': EmojiEntry(codepoint='1F3CB 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🏋🏽\u200d♀️', name='woman lifting weights: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏽‍♀': EmojiEntry(codepoint='1F3CB 1F3FD 200D 2640', status='minimally-qualified', emoji='🏋🏽\u200d♀', name='woman lifting weights: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏾‍♀️': EmojiEntry(codepoint='1F3CB 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🏋🏾\u200d♀️', name='woman lifting weights: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏾‍♀': EmojiEntry(codepoint='1F3CB 1F3FE 200D 2640', status='minimally-qualified', emoji='🏋🏾\u200d♀', name='woman lifting weights: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏿‍♀️': EmojiEntry(codepoint='1F3CB 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🏋🏿\u200d♀️', name='woman lifting weights: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🏋🏿‍♀': EmojiEntry(codepoint='1F3CB 1F3FF 200D 2640', status='minimally-qualified', emoji='🏋🏿\u200d♀', name='woman lifting weights: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚴': EmojiEntry(codepoint='1F6B4', status='fully-qualified', emoji='🚴', name='person biking', group='People & Body', sub_group='person-sport'),
    '🚴🏻': EmojiEntry(codepoint='1F6B4 1F3FB', status='fully-qualified', emoji='🚴🏻', name='person biking: light skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏼': EmojiEntry(codepoint='1F6B4 1F3FC', status='fully-qualified', emoji='🚴🏼', name='person biking: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏽': EmojiEntry(codepoint='1F6B4 1F3FD', status='fully-qualified', emoji='🚴🏽', name='person biking: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏾': EmojiEntry(codepoint='1F6B4 1F3FE', status='fully-qualified', emoji='🚴🏾', name='person biking: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏿': EmojiEntry(codepoint='1F6B4 1F3FF', status='fully-qualified', emoji='🚴🏿', name='person biking: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚴‍♂️': EmojiEntry(codepoint='1F6B4 200D 2642 FE0F', status='fully-qualified', emoji='🚴\u200d♂️', name='man biking', group='People & Body', sub_group='person-sport'),
    '🚴‍♂': EmojiEntry(codepoint='1F6B4 200D 2642', status='minimally-qualified', emoji='🚴\u200d♂', name='man biking', group='People & Body', sub_group='person-sport'),
    '🚴🏻‍♂️': EmojiEntry(codepoint='1F6B4 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🚴🏻\u200d♂️', name='man biking: light skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏻‍♂': EmojiEntry(codepoint='1F6B4 1F3FB 200D 2642', status='minimally-qualified', emoji='🚴🏻\u200d♂', name='man biking: light skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏼‍♂️': EmojiEntry(codepoint='1F6B4 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🚴🏼\u200d♂️', name='man biking: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏼‍♂': EmojiEntry(codepoint='1F6B4 1F3FC 200D 2642', status='minimally-qualified', emoji='🚴🏼\u200d♂', name='man biking: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏽‍♂️': EmojiEntry(codepoint='1F6B4 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🚴🏽\u200d♂️', name='man biking: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏽‍♂': EmojiEntry(codepoint='1F6B4 1F3FD 200D 2642', status='minimally-qualified', emoji='🚴🏽\u200d♂', name='man biking: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏾‍♂️': EmojiEntry(codepoint='1F6B4 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🚴🏾\u200d♂️', name='man biking: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏾‍♂': EmojiEntry(codepoint='1F6B4 1F3FE 200D 2642', status='minimally-qualified', emoji='🚴🏾\u200d♂', name='man biking: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏿‍♂️': EmojiEntry(codepoint='1F6B4 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🚴🏿\u200d♂️', name='man biking: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏿‍♂': EmojiEntry(codepoint='1F6B4 1F3FF 200D 2642', status='minimally-qualified', emoji='🚴🏿\u200d♂', name='man biking: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚴‍♀️': EmojiEntry(codepoint='1F6B4 200D 2640 FE0F', status='fully-qualified', emoji='🚴\u200d♀️', name='woman biking', group='People & Body', sub_group='person-sport'),
    '🚴‍♀': EmojiEntry(codepoint='1F6B4 200D 2640', status='minimally-qualified', emoji='🚴\u200d♀', name='woman biking', group='People & Body', sub_group='person-sport'),
    '🚴🏻‍♀️': EmojiEntry(codepoint='1F6B4 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🚴🏻\u200d♀️', name='woman biking: light skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏻‍♀': EmojiEntry(codepoint='1F6B4 1F3FB 200D 2640', status='minimally-qualified', emoji='🚴🏻\u200d♀', name='woman biking: light skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏼‍♀️': EmojiEntry(codepoint='1F6B4 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🚴🏼\u200d♀️', name='woman biking: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏼‍♀': EmojiEntry(codepoint='1F6B4 1F3FC 200D 2640', status='minimally-qualified', emoji='🚴🏼\u200d♀', name='woman biking: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏽‍♀️': EmojiEntry(codepoint='1F6B4 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🚴🏽\u200d♀️', name='woman biking: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏽‍♀': EmojiEntry(codepoint='1F6B4 1F3FD 200D 2640', status='minimally-qualified', emoji='🚴🏽\u200d♀', name='woman biking: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏾‍♀️': EmojiEntry(codepoint='1F6B4 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🚴🏾\u200d♀️', name='woman biking: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏾‍♀': EmojiEntry(codepoint='1F6B4 1F3FE 200D 2640', status='minimally-qualified', emoji='🚴🏾\u200d♀', name='woman biking: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏿‍♀️': EmojiEntry(codepoint='1F6B4 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🚴🏿\u200d♀️', name='woman biking: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚴🏿‍♀': EmojiEntry(codepoint='1F6B4 1F3FF 200D 2640', status='minimally-qualified', emoji='🚴🏿\u200d♀', name='woman biking: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚵': EmojiEntry(codepoint='1F6B5', status='fully-qualified', emoji='🚵', name='person mountain biking', group='People & Body', sub_group='person-sport'),
    '🚵🏻': EmojiEntry(codepoint='1F6B5 1F3FB', status='fully-qualified', emoji='🚵🏻', name='person mountain biking: light skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏼': EmojiEntry(codepoint='1F6B5 1F3FC', status='fully-qualified', emoji='🚵🏼', name='person mountain biking: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏽': EmojiEntry(codepoint='1F6B5 1F3FD', status='fully-qualified', emoji='🚵🏽', name='person mountain biking: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏾': EmojiEntry(codepoint='1F6B5 1F3FE', status='fully-qualified', emoji='🚵🏾', name='person mountain biking: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏿': EmojiEntry(codepoint='1F6B5 1F3FF', status='fully-qualified', emoji='🚵🏿', name='person mountain biking: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚵‍♂️': EmojiEntry(codepoint='1F6B5 200D 2642 FE0F', status='fully-qualified', emoji='🚵\u200d♂️', name='man mountain biking', group='People & Body', sub_group='person-sport'),
    '🚵‍♂': EmojiEntry(codepoint='1F6B5 200D 2642', status='minimally-qualified', emoji='🚵\u200d♂', name='man mountain biking', group='People & Body', sub_group='person-sport'),
    '🚵🏻‍♂️': EmojiEntry(codepoint='1F6B5 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🚵🏻\u200d♂️', name='man mountain biking: light skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏻‍♂': EmojiEntry(codepoint='1F6B5 1F3FB 200D 2642', status='minimally-qualified', emoji='🚵🏻\u200d♂', name='man mountain biking: light skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏼‍♂️': EmojiEntry(codepoint='1F6B5 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🚵🏼\u200d♂️', name='man mountain biking: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏼‍♂': EmojiEntry(codepoint='1F6B5 1F3FC 200D 2642', status='minimally-qualified', emoji='🚵🏼\u200d♂', name='man mountain biking: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏽‍♂️': EmojiEntry(codepoint='1F6B5 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🚵🏽\u200d♂️', name='man mountain biking: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏽‍♂': EmojiEntry(codepoint='1F6B5 1F3FD 200D 2642', status='minimally-qualified', emoji='🚵🏽\u200d♂', name='man mountain biking: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏾‍♂️': EmojiEntry(codepoint='1F6B5 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🚵🏾\u200d♂️', name='man mountain biking: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏾‍♂': EmojiEntry(codepoint='1F6B5 1F3FE 200D 2642', status='minimally-qualified', emoji='🚵🏾\u200d♂', name='man mountain biking: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏿‍♂️': EmojiEntry(codepoint='1F6B5 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🚵🏿\u200d♂️', name='man mountain biking: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏿‍♂': EmojiEntry(codepoint='1F6B5 1F3FF 200D 2642', status='minimally-qualified', emoji='🚵🏿\u200d♂', name='man mountain biking: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚵‍♀️': EmojiEntry(codepoint='1F6B5 200D 2640 FE0F', status='fully-qualified', emoji='🚵\u200d♀️', name='woman mountain biking', group='People & Body', sub_group='person-sport'),
    '🚵‍♀': EmojiEntry(codepoint='1F6B5 200D 2640', status='minimally-qualified', emoji='🚵\u200d♀', name='woman mountain biking', group='People & Body', sub_group='person-sport'),
    '🚵🏻‍♀️': EmojiEntry(codepoint='1F6B5 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🚵🏻\u200d♀️', name='woman mountain biking: light skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏻‍♀': EmojiEntry(codepoint='1F6B5 1F3FB 200D 2640', status='minimally-qualified', emoji='🚵🏻\u200d♀', name='woman mountain biking: light skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏼‍♀️': EmojiEntry(codepoint='1F6B5 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🚵🏼\u200d♀️', name='woman mountain biking: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏼‍♀': EmojiEntry(codepoint='1F6B5 1F3FC 200D 2640', status='minimally-qualified', emoji='🚵🏼\u200d♀', name='woman mountain biking: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏽‍♀️': EmojiEntry(codepoint='1F6B5 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🚵🏽\u200d♀️', name='woman mountain biking: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏽‍♀': EmojiEntry(codepoint='1F6B5 1F3FD 200D 2640', status='minimally-qualified', emoji='🚵🏽\u200d♀', name='woman mountain biking: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏾‍♀️': EmojiEntry(codepoint='1F6B5 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🚵🏾\u200d♀️', name='woman mountain biking: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏾‍♀': EmojiEntry(codepoint='1F6B5 1F3FE 200D 2640', status='minimally-qualified', emoji='🚵🏾\u200d♀', name='woman mountain biking: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏿‍♀️': EmojiEntry(codepoint='1F6B5 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🚵🏿\u200d♀️', name='woman mountain biking: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🚵🏿‍♀': EmojiEntry(codepoint='1F6B5 1F3FF 200D 2640', status='minimally-qualified', emoji='🚵🏿\u200d♀', name='woman mountain biking: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤸': EmojiEntry(codepoint='1F938', status='fully-qualified', emoji='🤸', name='person cartwheeling', group='People & Body', sub_group='person-sport'),
    '🤸🏻': EmojiEntry(codepoint='1F938 1F3FB', status='fully-qualified', emoji='🤸🏻', name='person cartwheeling: light skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏼': EmojiEntry(codepoint='1F938 1F3FC', status='fully-qualified', emoji='🤸🏼', name='person cartwheeling: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏽': EmojiEntry(codepoint='1F938 1F3FD', status='fully-qualified', emoji='🤸🏽', name='person cartwheeling: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏾': EmojiEntry(codepoint='1F938 1F3FE', status='fully-qualified', emoji='🤸🏾', name='person cartwheeling: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏿': EmojiEntry(codepoint='1F938 1F3FF', status='fully-qualified', emoji='🤸🏿', name='person cartwheeling: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤸‍♂️': EmojiEntry(codepoint='1F938 200D 2642 FE0F', status='fully-qualified', emoji='🤸\u200d♂️', name='man cartwheeling', group='People & Body', sub_group='person-sport'),
    '🤸‍♂': EmojiEntry(codepoint='1F938 200D 2642', status='minimally-qualified', emoji='🤸\u200d♂', name='man cartwheeling', group='People & Body', sub_group='person-sport'),
    '🤸🏻‍♂️': EmojiEntry(codepoint='1F938 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🤸🏻\u200d♂️', name='man cartwheeling: light skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏻‍♂': EmojiEntry(codepoint='1F938 1F3FB 200D 2642', status='minimally-qualified', emoji='🤸🏻\u200d♂', name='man cartwheeling: light skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏼‍♂️': EmojiEntry(codepoint='1F938 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🤸🏼\u200d♂️', name='man cartwheeling: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏼‍♂': EmojiEntry(codepoint='1F938 1F3FC 200D 2642', status='minimally-qualified', emoji='🤸🏼\u200d♂', name='man cartwheeling: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏽‍♂️': EmojiEntry(codepoint='1F938 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🤸🏽\u200d♂️', name='man cartwheeling: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏽‍♂': EmojiEntry(codepoint='1F938 1F3FD 200D 2642', status='minimally-qualified', emoji='🤸🏽\u200d♂', name='man cartwheeling: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏾‍♂️': EmojiEntry(codepoint='1F938 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🤸🏾\u200d♂️', name='man cartwheeling: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏾‍♂': EmojiEntry(codepoint='1F938 1F3FE 200D 2642', status='minimally-qualified', emoji='🤸🏾\u200d♂', name='man cartwheeling: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏿‍♂️': EmojiEntry(codepoint='1F938 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🤸🏿\u200d♂️', name='man cartwheeling: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏿‍♂': EmojiEntry(codepoint='1F938 1F3FF 200D 2642', status='minimally-qualified', emoji='🤸🏿\u200d♂', name='man cartwheeling: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤸‍♀️': EmojiEntry(codepoint='1F938 200D 2640 FE0F', status='fully-qualified', emoji='🤸\u200d♀️', name='woman cartwheeling', group='People & Body', sub_group='person-sport'),
    '🤸‍♀': EmojiEntry(codepoint='1F938 200D 2640', status='minimally-qualified', emoji='🤸\u200d♀', name='woman cartwheeling', group='People & Body', sub_group='person-sport'),
    '🤸🏻‍♀️': EmojiEntry(codepoint='1F938 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🤸🏻\u200d♀️', name='woman cartwheeling: light skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏻‍♀': EmojiEntry(codepoint='1F938 1F3FB 200D 2640', status='minimally-qualified', emoji='🤸🏻\u200d♀', name='woman cartwheeling: light skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏼‍♀️': EmojiEntry(codepoint='1F938 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🤸🏼\u200d♀️', name='woman cartwheeling: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏼‍♀': EmojiEntry(codepoint='1F938 1F3FC 200D 2640', status='minimally-qualified', emoji='🤸🏼\u200d♀', name='woman cartwheeling: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏽‍♀️': EmojiEntry(codepoint='1F938 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🤸🏽\u200d♀️', name='woman cartwheeling: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏽‍♀': EmojiEntry(codepoint='1F938 1F3FD 200D 2640', status='minimally-qualified', emoji='🤸🏽\u200d♀', name='woman cartwheeling: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏾‍♀️': EmojiEntry(codepoint='1F938 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🤸🏾\u200d♀️', name='woman cartwheeling: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏾‍♀': EmojiEntry(codepoint='1F938 1F3FE 200D 2640', status='minimally-qualified', emoji='🤸🏾\u200d♀', name='woman cartwheeling: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏿‍♀️': EmojiEntry(codepoint='1F938 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🤸🏿\u200d♀️', name='woman cartwheeling: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤸🏿‍♀': EmojiEntry(codepoint='1F938 1F3FF 200D 2640', status='minimally-qualified', emoji='🤸🏿\u200d♀', name='woman cartwheeling: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤼': EmojiEntry(codepoint='1F93C', status='fully-qualified', emoji='🤼', name='people wrestling', group='People & Body', sub_group='person-sport'),
    '🤼‍♂️': EmojiEntry(codepoint='1F93C 200D 2642 FE0F', status='fully-qualified', emoji='🤼\u200d♂️', name='men wrestling', group='People & Body', sub_group='person-sport'),
    '🤼‍♂': EmojiEntry(codepoint='1F93C 200D 2642', status='minimally-qualified', emoji='🤼\u200d♂', name='men wrestling', group='People & Body', sub_group='person-sport'),
    '🤼‍♀️': EmojiEntry(codepoint='1F93C 200D 2640 FE0F', status='fully-qualified', emoji='🤼\u200d♀️', name='women wrestling', group='People & Body', sub_group='person-sport'),
    '🤼‍♀': EmojiEntry(codepoint='1F93C 200D 2640', status='minimally-qualified', emoji='🤼\u200d♀', name='women wrestling', group='People & Body', sub_group='person-sport'),
    '🤽': EmojiEntry(codepoint='1F93D', status='fully-qualified', emoji='🤽', name='person playing water polo', group='People & Body', sub_group='person-sport'),
    '🤽🏻': EmojiEntry(codepoint='1F93D 1F3FB', status='fully-qualified', emoji='🤽🏻', name='person playing water polo: light skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏼': EmojiEntry(codepoint='1F93D 1F3FC', status='fully-qualified', emoji='🤽🏼', name='person playing water polo: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏽': EmojiEntry(codepoint='1F93D 1F3FD', status='fully-qualified', emoji='🤽🏽', name='person playing water polo: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏾': EmojiEntry(codepoint='1F93D 1F3FE', status='fully-qualified', emoji='🤽🏾', name='person playing water polo: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏿': EmojiEntry(codepoint='1F93D 1F3FF', status='fully-qualified', emoji='🤽🏿', name='person playing water polo: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤽‍♂️': EmojiEntry(codepoint='1F93D 200D 2642 FE0F', status='fully-qualified', emoji='🤽\u200d♂️', name='man playing water polo', group='People & Body', sub_group='person-sport'),
    '🤽‍♂': EmojiEntry(codepoint='1F93D 200D 2642', status='minimally-qualified', emoji='🤽\u200d♂', name='man playing water polo', group='People & Body', sub_group='person-sport'),
    '🤽🏻‍♂️': EmojiEntry(codepoint='1F93D 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🤽🏻\u200d♂️', name='man playing water polo: light skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏻‍♂': EmojiEntry(codepoint='1F93D 1F3FB 200D 2642', status='minimally-qualified', emoji='🤽🏻\u200d♂', name='man playing water polo: light skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏼‍♂️': EmojiEntry(codepoint='1F93D 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🤽🏼\u200d♂️', name='man playing water polo: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏼‍♂': EmojiEntry(codepoint='1F93D 1F3FC 200D 2642', status='minimally-qualified', emoji='🤽🏼\u200d♂', name='man playing water polo: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏽‍♂️': EmojiEntry(codepoint='1F93D 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🤽🏽\u200d♂️', name='man playing water polo: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏽‍♂': EmojiEntry(codepoint='1F93D 1F3FD 200D 2642', status='minimally-qualified', emoji='🤽🏽\u200d♂', name='man playing water polo: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏾‍♂️': EmojiEntry(codepoint='1F93D 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🤽🏾\u200d♂️', name='man playing water polo: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏾‍♂': EmojiEntry(codepoint='1F93D 1F3FE 200D 2642', status='minimally-qualified', emoji='🤽🏾\u200d♂', name='man playing water polo: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏿‍♂️': EmojiEntry(codepoint='1F93D 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🤽🏿\u200d♂️', name='man playing water polo: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏿‍♂': EmojiEntry(codepoint='1F93D 1F3FF 200D 2642', status='minimally-qualified', emoji='🤽🏿\u200d♂', name='man playing water polo: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤽‍♀️': EmojiEntry(codepoint='1F93D 200D 2640 FE0F', status='fully-qualified', emoji='🤽\u200d♀️', name='woman playing water polo', group='People & Body', sub_group='person-sport'),
    '🤽‍♀': EmojiEntry(codepoint='1F93D 200D 2640', status='minimally-qualified', emoji='🤽\u200d♀', name='woman playing water polo', group='People & Body', sub_group='person-sport'),
    '🤽🏻‍♀️': EmojiEntry(codepoint='1F93D 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🤽🏻\u200d♀️', name='woman playing water polo: light skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏻‍♀': EmojiEntry(codepoint='1F93D 1F3FB 200D 2640', status='minimally-qualified', emoji='🤽🏻\u200d♀', name='woman playing water polo: light skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏼‍♀️': EmojiEntry(codepoint='1F93D 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🤽🏼\u200d♀️', name='woman playing water polo: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏼‍♀': EmojiEntry(codepoint='1F93D 1F3FC 200D 2640', status='minimally-qualified', emoji='🤽🏼\u200d♀', name='woman playing water polo: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏽‍♀️': EmojiEntry(codepoint='1F93D 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🤽🏽\u200d♀️', name='woman playing water polo: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏽‍♀': EmojiEntry(codepoint='1F93D 1F3FD 200D 2640', status='minimally-qualified', emoji='🤽🏽\u200d♀', name='woman playing water polo: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏾‍♀️': EmojiEntry(codepoint='1F93D 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🤽🏾\u200d♀️', name='woman playing water polo: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏾‍♀': EmojiEntry(codepoint='1F93D 1F3FE 200D 2640', status='minimally-qualified', emoji='🤽🏾\u200d♀', name='woman playing water polo: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏿‍♀️': EmojiEntry(codepoint='1F93D 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🤽🏿\u200d♀️', name='woman playing water polo: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤽🏿‍♀': EmojiEntry(codepoint='1F93D 1F3FF 200D 2640', status='minimally-qualified', emoji='🤽🏿\u200d♀', name='woman playing water polo: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤾': EmojiEntry(codepoint='1F93E', status='fully-qualified', emoji='🤾', name='person playing handball', group='People & Body', sub_group='person-sport'),
    '🤾🏻': EmojiEntry(codepoint='1F93E 1F3FB', status='fully-qualified', emoji='🤾🏻', name='person playing handball: light skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏼': EmojiEntry(codepoint='1F93E 1F3FC', status='fully-qualified', emoji='🤾🏼', name='person playing handball: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏽': EmojiEntry(codepoint='1F93E 1F3FD', status='fully-qualified', emoji='🤾🏽', name='person playing handball: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏾': EmojiEntry(codepoint='1F93E 1F3FE', status='fully-qualified', emoji='🤾🏾', name='person playing handball: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏿': EmojiEntry(codepoint='1F93E 1F3FF', status='fully-qualified', emoji='🤾🏿', name='person playing handball: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤾‍♂️': EmojiEntry(codepoint='1F93E 200D 2642 FE0F', status='fully-qualified', emoji='🤾\u200d♂️', name='man playing handball', group='People & Body', sub_group='person-sport'),
    '🤾‍♂': EmojiEntry(codepoint='1F93E 200D 2642', status='minimally-qualified', emoji='🤾\u200d♂', name='man playing handball', group='People & Body', sub_group='person-sport'),
    '🤾🏻‍♂️': EmojiEntry(codepoint='1F93E 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🤾🏻\u200d♂️', name='man playing handball: light skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏻‍♂': EmojiEntry(codepoint='1F93E 1F3FB 200D 2642', status='minimally-qualified', emoji='🤾🏻\u200d♂', name='man playing handball: light skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏼‍♂️': EmojiEntry(codepoint='1F93E 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🤾🏼\u200d♂️', name='man playing handball: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏼‍♂': EmojiEntry(codepoint='1F93E 1F3FC 200D 2642', status='minimally-qualified', emoji='🤾🏼\u200d♂', name='man playing handball: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏽‍♂️': EmojiEntry(codepoint='1F93E 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🤾🏽\u200d♂️', name='man playing handball: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏽‍♂': EmojiEntry(codepoint='1F93E 1F3FD 200D 2642', status='minimally-qualified', emoji='🤾🏽\u200d♂', name='man playing handball: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏾‍♂️': EmojiEntry(codepoint='1F93E 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🤾🏾\u200d♂️', name='man playing handball: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏾‍♂': EmojiEntry(codepoint='1F93E 1F3FE 200D 2642', status='minimally-qualified', emoji='🤾🏾\u200d♂', name='man playing handball: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏿‍♂️': EmojiEntry(codepoint='1F93E 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🤾🏿\u200d♂️', name='man playing handball: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏿‍♂': EmojiEntry(codepoint='1F93E 1F3FF 200D 2642', status='minimally-qualified', emoji='🤾🏿\u200d♂', name='man playing handball: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤾‍♀️': EmojiEntry(codepoint='1F93E 200D 2640 FE0F', status='fully-qualified', emoji='🤾\u200d♀️', name='woman playing handball', group='People & Body', sub_group='person-sport'),
    '🤾‍♀': EmojiEntry(codepoint='1F93E 200D 2640', status='minimally-qualified', emoji='🤾\u200d♀', name='woman playing handball', group='People & Body', sub_group='person-sport'),
    '🤾🏻‍♀️': EmojiEntry(codepoint='1F93E 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🤾🏻\u200d♀️', name='woman playing handball: light skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏻‍♀': EmojiEntry(codepoint='1F93E 1F3FB 200D 2640', status='minimally-qualified', emoji='🤾🏻\u200d♀', name='woman playing handball: light skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏼‍♀️': EmojiEntry(codepoint='1F93E 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🤾🏼\u200d♀️', name='woman playing handball: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏼‍♀': EmojiEntry(codepoint='1F93E 1F3FC 200D 2640', status='minimally-qualified', emoji='🤾🏼\u200d♀', name='woman playing handball: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏽‍♀️': EmojiEntry(codepoint='1F93E 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🤾🏽\u200d♀️', name='woman playing handball: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏽‍♀': EmojiEntry(codepoint='1F93E 1F3FD 200D 2640', status='minimally-qualified', emoji='🤾🏽\u200d♀', name='woman playing handball: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏾‍♀️': EmojiEntry(codepoint='1F93E 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🤾🏾\u200d♀️', name='woman playing handball: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏾‍♀': EmojiEntry(codepoint='1F93E 1F3FE 200D 2640', status='minimally-qualified', emoji='🤾🏾\u200d♀', name='woman playing handball: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏿‍♀️': EmojiEntry(codepoint='1F93E 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🤾🏿\u200d♀️', name='woman playing handball: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤾🏿‍♀': EmojiEntry(codepoint='1F93E 1F3FF 200D 2640', status='minimally-qualified', emoji='🤾🏿\u200d♀', name='woman playing handball: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤹': EmojiEntry(codepoint='1F939', status='fully-qualified', emoji='🤹', name='person juggling', group='People & Body', sub_group='person-sport'),
    '🤹🏻': EmojiEntry(codepoint='1F939 1F3FB', status='fully-qualified', emoji='🤹🏻', name='person juggling: light skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏼': EmojiEntry(codepoint='1F939 1F3FC', status='fully-qualified', emoji='🤹🏼', name='person juggling: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏽': EmojiEntry(codepoint='1F939 1F3FD', status='fully-qualified', emoji='🤹🏽', name='person juggling: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏾': EmojiEntry(codepoint='1F939 1F3FE', status='fully-qualified', emoji='🤹🏾', name='person juggling: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏿': EmojiEntry(codepoint='1F939 1F3FF', status='fully-qualified', emoji='🤹🏿', name='person juggling: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤹‍♂️': EmojiEntry(codepoint='1F939 200D 2642 FE0F', status='fully-qualified', emoji='🤹\u200d♂️', name='man juggling', group='People & Body', sub_group='person-sport'),
    '🤹‍♂': EmojiEntry(codepoint='1F939 200D 2642', status='minimally-qualified', emoji='🤹\u200d♂', name='man juggling', group='People & Body', sub_group='person-sport'),
    '🤹🏻‍♂️': EmojiEntry(codepoint='1F939 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🤹🏻\u200d♂️', name='man juggling: light skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏻‍♂': EmojiEntry(codepoint='1F939 1F3FB 200D 2642', status='minimally-qualified', emoji='🤹🏻\u200d♂', name='man juggling: light skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏼‍♂️': EmojiEntry(codepoint='1F939 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🤹🏼\u200d♂️', name='man juggling: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏼‍♂': EmojiEntry(codepoint='1F939 1F3FC 200D 2642', status='minimally-qualified', emoji='🤹🏼\u200d♂', name='man juggling: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏽‍♂️': EmojiEntry(codepoint='1F939 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🤹🏽\u200d♂️', name='man juggling: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏽‍♂': EmojiEntry(codepoint='1F939 1F3FD 200D 2642', status='minimally-qualified', emoji='🤹🏽\u200d♂', name='man juggling: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏾‍♂️': EmojiEntry(codepoint='1F939 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🤹🏾\u200d♂️', name='man juggling: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏾‍♂': EmojiEntry(codepoint='1F939 1F3FE 200D 2642', status='minimally-qualified', emoji='🤹🏾\u200d♂', name='man juggling: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏿‍♂️': EmojiEntry(codepoint='1F939 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🤹🏿\u200d♂️', name='man juggling: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏿‍♂': EmojiEntry(codepoint='1F939 1F3FF 200D 2642', status='minimally-qualified', emoji='🤹🏿\u200d♂', name='man juggling: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤹‍♀️': EmojiEntry(codepoint='1F939 200D 2640 FE0F', status='fully-qualified', emoji='🤹\u200d♀️', name='woman juggling', group='People & Body', sub_group='person-sport'),
    '🤹‍♀': EmojiEntry(codepoint='1F939 200D 2640', status='minimally-qualified', emoji='🤹\u200d♀', name='woman juggling', group='People & Body', sub_group='person-sport'),
    '🤹🏻‍♀️': EmojiEntry(codepoint='1F939 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🤹🏻\u200d♀️', name='woman juggling: light skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏻‍♀': EmojiEntry(codepoint='1F939 1F3FB 200D 2640', status='minimally-qualified', emoji='🤹🏻\u200d♀', name='woman juggling: light skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏼‍♀️': EmojiEntry(codepoint='1F939 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🤹🏼\u200d♀️', name='woman juggling: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏼‍♀': EmojiEntry(codepoint='1F939 1F3FC 200D 2640', status='minimally-qualified', emoji='🤹🏼\u200d♀', name='woman juggling: medium-light skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏽‍♀️': EmojiEntry(codepoint='1F939 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🤹🏽\u200d♀️', name='woman juggling: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏽‍♀': EmojiEntry(codepoint='1F939 1F3FD 200D 2640', status='minimally-qualified', emoji='🤹🏽\u200d♀', name='woman juggling: medium skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏾‍♀️': EmojiEntry(codepoint='1F939 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🤹🏾\u200d♀️', name='woman juggling: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏾‍♀': EmojiEntry(codepoint='1F939 1F3FE 200D 2640', status='minimally-qualified', emoji='🤹🏾\u200d♀', name='woman juggling: medium-dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏿‍♀️': EmojiEntry(codepoint='1F939 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🤹🏿\u200d♀️', name='woman juggling: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🤹🏿‍♀': EmojiEntry(codepoint='1F939 1F3FF 200D 2640', status='minimally-qualified', emoji='🤹🏿\u200d♀', name='woman juggling: dark skin tone', group='People & Body', sub_group='person-sport'),
    '🧘': EmojiEntry(codepoint='1F9D8', status='fully-qualified', emoji='🧘', name='person in lotus position', group='People & Body', sub_group='person-resting'),
    '🧘🏻': EmojiEntry(codepoint='1F9D8 1F3FB', status='fully-qualified', emoji='🧘🏻', name='person in lotus position: light skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏼': EmojiEntry(codepoint='1F9D8 1F3FC', status='fully-qualified', emoji='🧘🏼', name='person in lotus position: medium-light skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏽': EmojiEntry(codepoint='1F9D8 1F3FD', status='fully-qualified', emoji='🧘🏽', name='person in lotus position: medium skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏾': EmojiEntry(codepoint='1F9D8 1F3FE', status='fully-qualified', emoji='🧘🏾', name='person in lotus position: medium-dark skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏿': EmojiEntry(codepoint='1F9D8 1F3FF', status='fully-qualified', emoji='🧘🏿', name='person in lotus position: dark skin tone', group='People & Body', sub_group='person-resting'),
    '🧘‍♂️': EmojiEntry(codepoint='1F9D8 200D 2642 FE0F', status='fully-qualified', emoji='🧘\u200d♂️', name='man in lotus position', group='People & Body', sub_group='person-resting'),
    '🧘‍♂': EmojiEntry(codepoint='1F9D8 200D 2642', status='minimally-qualified', emoji='🧘\u200d♂', name='man in lotus position', group='People & Body', sub_group='person-resting'),
    '🧘🏻‍♂️': EmojiEntry(codepoint='1F9D8 1F3FB 200D 2642 FE0F', status='fully-qualified', emoji='🧘🏻\u200d♂️', name='man in lotus position: light skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏻‍♂': EmojiEntry(codepoint='1F9D8 1F3FB 200D 2642', status='minimally-qualified', emoji='🧘🏻\u200d♂', name='man in lotus position: light skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏼‍♂️': EmojiEntry(codepoint='1F9D8 1F3FC 200D 2642 FE0F', status='fully-qualified', emoji='🧘🏼\u200d♂️', name='man in lotus position: medium-light skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏼‍♂': EmojiEntry(codepoint='1F9D8 1F3FC 200D 2642', status='minimally-qualified', emoji='🧘🏼\u200d♂', name='man in lotus position: medium-light skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏽‍♂️': EmojiEntry(codepoint='1F9D8 1F3FD 200D 2642 FE0F', status='fully-qualified', emoji='🧘🏽\u200d♂️', name='man in lotus position: medium skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏽‍♂': EmojiEntry(codepoint='1F9D8 1F3FD 200D 2642', status='minimally-qualified', emoji='🧘🏽\u200d♂', name='man in lotus position: medium skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏾‍♂️': EmojiEntry(codepoint='1F9D8 1F3FE 200D 2642 FE0F', status='fully-qualified', emoji='🧘🏾\u200d♂️', name='man in lotus position: medium-dark skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏾‍♂': EmojiEntry(codepoint='1F9D8 1F3FE 200D 2642', status='minimally-qualified', emoji='🧘🏾\u200d♂', name='man in lotus position: medium-dark skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏿‍♂️': EmojiEntry(codepoint='1F9D8 1F3FF 200D 2642 FE0F', status='fully-qualified', emoji='🧘🏿\u200d♂️', name='man in lotus position: dark skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏿‍♂': EmojiEntry(codepoint='1F9D8 1F3FF 200D 2642', status='minimally-qualified', emoji='🧘🏿\u200d♂', name='man in lotus position: dark skin tone', group='People & Body', sub_group='person-resting'),
    '🧘‍♀️': EmojiEntry(codepoint='1F9D8 200D 2640 FE0F', status='fully-qualified', emoji='🧘\u200d♀️', name='woman in lotus position', group='People & Body', sub_group='person-resting'),
    '🧘‍♀': EmojiEntry(codepoint='1F9D8 200D 2640', status='minimally-qualified', emoji='🧘\u200d♀', name='woman in lotus position', group='People & Body', sub_group='person-resting'),
    '🧘🏻‍♀️': EmojiEntry(codepoint='1F9D8 1F3FB 200D 2640 FE0F', status='fully-qualified', emoji='🧘🏻\u200d♀️', name='woman in lotus position: light skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏻‍♀': EmojiEntry(codepoint='1F9D8 1F3FB 200D 2640', status='minimally-qualified', emoji='🧘🏻\u200d♀', name='woman in lotus position: light skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏼‍♀️': EmojiEntry(codepoint='1F9D8 1F3FC 200D 2640 FE0F', status='fully-qualified', emoji='🧘🏼\u200d♀️', name='woman in lotus position: medium-light skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏼‍♀': EmojiEntry(codepoint='1F9D8 1F3FC 200D 2640', status='minimally-qualified', emoji='🧘🏼\u200d♀', name='woman in lotus position: medium-light skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏽‍♀️': EmojiEntry(codepoint='1F9D8 1F3FD 200D 2640 FE0F', status='fully-qualified', emoji='🧘🏽\u200d♀️', name='woman in lotus position: medium skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏽‍♀': EmojiEntry(codepoint='1F9D8 1F3FD 200D 2640', status='minimally-qualified', emoji='🧘🏽\u200d♀', name='woman in lotus position: medium skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏾‍♀️': EmojiEntry(codepoint='1F9D8 1F3FE 200D 2640 FE0F', status='fully-qualified', emoji='🧘🏾\u200d♀️', name='woman in lotus position: medium-dark skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏾‍♀': EmojiEntry(codepoint='1F9D8 1F3FE 200D 2640', status='minimally-qualified', emoji='🧘🏾\u200d♀', name='woman in lotus position: medium-dark skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏿‍♀️': EmojiEntry(codepoint='1F9D8 1F3FF 200D 2640 FE0F', status='fully-qualified', emoji='🧘🏿\u200d♀️', name='woman in lotus position: dark skin tone', group='People & Body', sub_group='person-resting'),
    '🧘🏿‍♀': EmojiEntry(codepoint='1F9D8 1F3FF 200D 2640', status='minimally-qualified', emoji='🧘🏿\u200d♀', name='woman in lotus position: dark skin tone', group='People & Body', sub_group='person-resting'),
    '🛀': EmojiEntry(codepoint='1F6C0', status='fully-qualified', emoji='🛀', name='person taking bath', group='People & Body', sub_group='person-resting'),
    '🛀🏻': EmojiEntry(codepoint='1F6C0 1F3FB', status='fully-qualified', emoji='🛀🏻', name='person taking bath: light skin tone', group='People & Body', sub_group='person-resting'),
    '🛀🏼': EmojiEntry(codepoint='1F6C0 1F3FC', status='fully-qualified', emoji='🛀🏼', name='person taking bath: medium-light skin tone', group='People & Body', sub_group='person-resting'),
    '🛀🏽': EmojiEntry(codepoint='1F6C0 1F3FD', status='fully-qualified', emoji='🛀🏽', name='person taking bath: medium skin tone', group='People & Body', sub_group='person-resting'),
    '🛀🏾': EmojiEntry(codepoint='1F6C0 1F3FE', status='fully-qualified', emoji='🛀🏾', name='person taking bath: medium-dark skin tone', group='People & Body', sub_group='person-resting'),
    '🛀🏿': EmojiEntry(codepoint='1F6C0 1F3FF', status='fully-qualified', emoji='🛀🏿', name='person taking bath: dark skin tone', group='People & Body', sub_group='person-resting'),
    '🛌': EmojiEntry(codepoint='1F6CC', status='fully-qualified', emoji='🛌', name='person in bed', group='People & Body', sub_group='person-resting'),
    '🛌🏻': EmojiEntry(codepoint='1F6CC 1F3FB', status='fully-qualified', emoji='🛌🏻', name='person in bed: light skin tone', group='People & Body', sub_group='person-resting'),
    '🛌🏼': EmojiEntry(codepoint='1F6CC 1F3FC', status='fully-qualified', emoji='🛌🏼', name='person in bed: medium-light skin tone', group='People & Body', sub_group='person-resting'),
    '🛌🏽': EmojiEntry(codepoint='1F6CC 1F3FD', status='fully-qualified', emoji='🛌🏽', name='person in bed: medium skin tone', group='People & Body', sub_group='person-resting'),
    '🛌🏾': EmojiEntry(codepoint='1F6CC 1F3FE', status='fully-qualified', emoji='🛌🏾', name='person in bed: medium-dark skin tone', group='People & Body', sub_group='person-resting'),
    '🛌🏿': EmojiEntry(codepoint='1F6CC 1F3FF', status='fully-qualified', emoji='🛌🏿', name='person in bed: dark skin tone', group='People & Body', sub_group='person-resting'),
    '🧑‍🤝‍🧑': EmojiEntry(codepoint='1F9D1 200D 1F91D 200D 1F9D1', status='fully-qualified', emoji='🧑\u200d🤝\u200d🧑', name='people holding hands', group='People & Body', sub_group='family'),
    '🧑🏻‍🤝‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F91D 200D 1F9D1 1F3FB', status='fully-qualified', emoji='🧑🏻\u200d🤝\u200d🧑🏻', name='people holding hands: light skin tone', group='People & Body', sub_group='family'),
    '🧑🏻‍🤝‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F91D 200D 1F9D1 1F3FC', status='fully-qualified', emoji='🧑🏻\u200d🤝\u200d🧑🏼', name='people holding hands: light skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏻‍🤝‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F91D 200D 1F9D1 1F3FD', status='fully-qualified', emoji='🧑🏻\u200d🤝\u200d🧑🏽', name='people holding hands: light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏻‍🤝‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F91D 200D 1F9D1 1F3FE', status='fully-qualified', emoji='🧑🏻\u200d🤝\u200d🧑🏾', name='people holding hands: light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏻‍🤝‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FB 200D 1F91D 200D 1F9D1 1F3FF', status='fully-qualified', emoji='🧑🏻\u200d🤝\u200d🧑🏿', name='people holding hands: light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍🤝‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F91D 200D 1F9D1 1F3FB', status='fully-qualified', emoji='🧑🏼\u200d🤝\u200d🧑🏻', name='people holding hands: medium-light skin tone, light skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍🤝‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F91D 200D 1F9D1 1F3FC', status='fully-qualified', emoji='🧑🏼\u200d🤝\u200d🧑🏼', name='people holding hands: medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍🤝‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F91D 200D 1F9D1 1F3FD', status='fully-qualified', emoji='🧑🏼\u200d🤝\u200d🧑🏽', name='people holding hands: medium-light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍🤝‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F91D 200D 1F9D1 1F3FE', status='fully-qualified', emoji='🧑🏼\u200d🤝\u200d🧑🏾', name='people holding hands: medium-light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍🤝‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FC 200D 1F91D 200D 1F9D1 1F3FF', status='fully-qualified', emoji='🧑🏼\u200d🤝\u200d🧑🏿', name='people holding hands: medium-light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍🤝‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F91D 200D 1F9D1 1F3FB', status='fully-qualified', emoji='🧑🏽\u200d🤝\u200d🧑🏻', name='people holding hands: medium skin tone, light skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍🤝‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F91D 200D 1F9D1 1F3FC', status='fully-qualified', emoji='🧑🏽\u200d🤝\u200d🧑🏼', name='people holding hands: medium skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍🤝‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F91D 200D 1F9D1 1F3FD', status='fully-qualified', emoji='🧑🏽\u200d🤝\u200d🧑🏽', name='people holding hands: medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍🤝‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F91D 200D 1F9D1 1F3FE', status='fully-qualified', emoji='🧑🏽\u200d🤝\u200d🧑🏾', name='people holding hands: medium skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍🤝‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FD 200D 1F91D 200D 1F9D1 1F3FF', status='fully-qualified', emoji='🧑🏽\u200d🤝\u200d🧑🏿', name='people holding hands: medium skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍🤝‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F91D 200D 1F9D1 1F3FB', status='fully-qualified', emoji='🧑🏾\u200d🤝\u200d🧑🏻', name='people holding hands: medium-dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍🤝‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F91D 200D 1F9D1 1F3FC', status='fully-qualified', emoji='🧑🏾\u200d🤝\u200d🧑🏼', name='people holding hands: medium-dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍🤝‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F91D 200D 1F9D1 1F3FD', status='fully-qualified', emoji='🧑🏾\u200d🤝\u200d🧑🏽', name='people holding hands: medium-dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍🤝‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F91D 200D 1F9D1 1F3FE', status='fully-qualified', emoji='🧑🏾\u200d🤝\u200d🧑🏾', name='people holding hands: medium-dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍🤝‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FE 200D 1F91D 200D 1F9D1 1F3FF', status='fully-qualified', emoji='🧑🏾\u200d🤝\u200d🧑🏿', name='people holding hands: medium-dark skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍🤝‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F91D 200D 1F9D1 1F3FB', status='fully-qualified', emoji='🧑🏿\u200d🤝\u200d🧑🏻', name='people holding hands: dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍🤝‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F91D 200D 1F9D1 1F3FC', status='fully-qualified', emoji='🧑🏿\u200d🤝\u200d🧑🏼', name='people holding hands: dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍🤝‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F91D 200D 1F9D1 1F3FD', status='fully-qualified', emoji='🧑🏿\u200d🤝\u200d🧑🏽', name='people holding hands: dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍🤝‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F91D 200D 1F9D1 1F3FE', status='fully-qualified', emoji='🧑🏿\u200d🤝\u200d🧑🏾', name='people holding hands: dark skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍🤝‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FF 200D 1F91D 200D 1F9D1 1F3FF', status='fully-qualified', emoji='🧑🏿\u200d🤝\u200d🧑🏿', name='people holding hands: dark skin tone', group='People & Body', sub_group='family'),
    '👭': EmojiEntry(codepoint='1F46D', status='fully-qualified', emoji='👭', name='women holding hands', group='People & Body', sub_group='family'),
    '👭🏻': EmojiEntry(codepoint='1F46D 1F3FB', status='fully-qualified', emoji='👭🏻', name='women holding hands: light skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍🤝‍👩🏼': EmojiEntry(codepoint='1F469 1F3FB 200D 1F91D 200D 1F469 1F3FC', status='fully-qualified', emoji='👩🏻\u200d🤝\u200d👩🏼', name='women holding hands: light skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍🤝‍👩🏽': EmojiEntry(codepoint='1F469 1F3FB 200D 1F91D 200D 1F469 1F3FD', status='fully-qualified', emoji='👩🏻\u200d🤝\u200d👩🏽', name='women holding hands: light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍🤝‍👩🏾': EmojiEntry(codepoint='1F469 1F3FB 200D 1F91D 200D 1F469 1F3FE', status='fully-qualified', emoji='👩🏻\u200d🤝\u200d👩🏾', name='women holding hands: light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍🤝‍👩🏿': EmojiEntry(codepoint='1F469 1F3FB 200D 1F91D 200D 1F469 1F3FF', status='fully-qualified', emoji='👩🏻\u200d🤝\u200d👩🏿', name='women holding hands: light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍🤝‍👩🏻': EmojiEntry(codepoint='1F469 1F3FC 200D 1F91D 200D 1F469 1F3FB', status='fully-qualified', emoji='👩🏼\u200d🤝\u200d👩🏻', name='women holding hands: medium-light skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👭🏼': EmojiEntry(codepoint='1F46D 1F3FC', status='fully-qualified', emoji='👭🏼', name='women holding hands: medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍🤝‍👩🏽': EmojiEntry(codepoint='1F469 1F3FC 200D 1F91D 200D 1F469 1F3FD', status='fully-qualified', emoji='👩🏼\u200d🤝\u200d👩🏽', name='women holding hands: medium-light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍🤝‍👩🏾': EmojiEntry(codepoint='1F469 1F3FC 200D 1F91D 200D 1F469 1F3FE', status='fully-qualified', emoji='👩🏼\u200d🤝\u200d👩🏾', name='women holding hands: medium-light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍🤝‍👩🏿': EmojiEntry(codepoint='1F469 1F3FC 200D 1F91D 200D 1F469 1F3FF', status='fully-qualified', emoji='👩🏼\u200d🤝\u200d👩🏿', name='women holding hands: medium-light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍🤝‍👩🏻': EmojiEntry(codepoint='1F469 1F3FD 200D 1F91D 200D 1F469 1F3FB', status='fully-qualified', emoji='👩🏽\u200d🤝\u200d👩🏻', name='women holding hands: medium skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍🤝‍👩🏼': EmojiEntry(codepoint='1F469 1F3FD 200D 1F91D 200D 1F469 1F3FC', status='fully-qualified', emoji='👩🏽\u200d🤝\u200d👩🏼', name='women holding hands: medium skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👭🏽': EmojiEntry(codepoint='1F46D 1F3FD', status='fully-qualified', emoji='👭🏽', name='women holding hands: medium skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍🤝‍👩🏾': EmojiEntry(codepoint='1F469 1F3FD 200D 1F91D 200D 1F469 1F3FE', status='fully-qualified', emoji='👩🏽\u200d🤝\u200d👩🏾', name='women holding hands: medium skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍🤝‍👩🏿': EmojiEntry(codepoint='1F469 1F3FD 200D 1F91D 200D 1F469 1F3FF', status='fully-qualified', emoji='👩🏽\u200d🤝\u200d👩🏿', name='women holding hands: medium skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍🤝‍👩🏻': EmojiEntry(codepoint='1F469 1F3FE 200D 1F91D 200D 1F469 1F3FB', status='fully-qualified', emoji='👩🏾\u200d🤝\u200d👩🏻', name='women holding hands: medium-dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍🤝‍👩🏼': EmojiEntry(codepoint='1F469 1F3FE 200D 1F91D 200D 1F469 1F3FC', status='fully-qualified', emoji='👩🏾\u200d🤝\u200d👩🏼', name='women holding hands: medium-dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍🤝‍👩🏽': EmojiEntry(codepoint='1F469 1F3FE 200D 1F91D 200D 1F469 1F3FD', status='fully-qualified', emoji='👩🏾\u200d🤝\u200d👩🏽', name='women holding hands: medium-dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👭🏾': EmojiEntry(codepoint='1F46D 1F3FE', status='fully-qualified', emoji='👭🏾', name='women holding hands: medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍🤝‍👩🏿': EmojiEntry(codepoint='1F469 1F3FE 200D 1F91D 200D 1F469 1F3FF', status='fully-qualified', emoji='👩🏾\u200d🤝\u200d👩🏿', name='women holding hands: medium-dark skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍🤝‍👩🏻': EmojiEntry(codepoint='1F469 1F3FF 200D 1F91D 200D 1F469 1F3FB', status='fully-qualified', emoji='👩🏿\u200d🤝\u200d👩🏻', name='women holding hands: dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍🤝‍👩🏼': EmojiEntry(codepoint='1F469 1F3FF 200D 1F91D 200D 1F469 1F3FC', status='fully-qualified', emoji='👩🏿\u200d🤝\u200d👩🏼', name='women holding hands: dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍🤝‍👩🏽': EmojiEntry(codepoint='1F469 1F3FF 200D 1F91D 200D 1F469 1F3FD', status='fully-qualified', emoji='👩🏿\u200d🤝\u200d👩🏽', name='women holding hands: dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍🤝‍👩🏾': EmojiEntry(codepoint='1F469 1F3FF 200D 1F91D 200D 1F469 1F3FE', status='fully-qualified', emoji='👩🏿\u200d🤝\u200d👩🏾', name='women holding hands: dark skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👭🏿': EmojiEntry(codepoint='1F46D 1F3FF', status='fully-qualified', emoji='👭🏿', name='women holding hands: dark skin tone', group='People & Body', sub_group='family'),
    '👫': EmojiEntry(codepoint='1F46B', status='fully-qualified', emoji='👫', name='woman and man holding hands', group='People & Body', sub_group='family'),
    '👫🏻': EmojiEntry(codepoint='1F46B 1F3FB', status='fully-qualified', emoji='👫🏻', name='woman and man holding hands: light skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍🤝‍👨🏼': EmojiEntry(codepoint='1F469 1F3FB 200D 1F91D 200D 1F468 1F3FC', status='fully-qualified', emoji='👩🏻\u200d🤝\u200d👨🏼', name='woman and man holding hands: light skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍🤝‍👨🏽': EmojiEntry(codepoint='1F469 1F3FB 200D 1F91D 200D 1F468 1F3FD', status='fully-qualified', emoji='👩🏻\u200d🤝\u200d👨🏽', name='woman and man holding hands: light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍🤝‍👨🏾': EmojiEntry(codepoint='1F469 1F3FB 200D 1F91D 200D 1F468 1F3FE', status='fully-qualified', emoji='👩🏻\u200d🤝\u200d👨🏾', name='woman and man holding hands: light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍🤝‍👨🏿': EmojiEntry(codepoint='1F469 1F3FB 200D 1F91D 200D 1F468 1F3FF', status='fully-qualified', emoji='👩🏻\u200d🤝\u200d👨🏿', name='woman and man holding hands: light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍🤝‍👨🏻': EmojiEntry(codepoint='1F469 1F3FC 200D 1F91D 200D 1F468 1F3FB', status='fully-qualified', emoji='👩🏼\u200d🤝\u200d👨🏻', name='woman and man holding hands: medium-light skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👫🏼': EmojiEntry(codepoint='1F46B 1F3FC', status='fully-qualified', emoji='👫🏼', name='woman and man holding hands: medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍🤝‍👨🏽': EmojiEntry(codepoint='1F469 1F3FC 200D 1F91D 200D 1F468 1F3FD', status='fully-qualified', emoji='👩🏼\u200d🤝\u200d👨🏽', name='woman and man holding hands: medium-light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍🤝‍👨🏾': EmojiEntry(codepoint='1F469 1F3FC 200D 1F91D 200D 1F468 1F3FE', status='fully-qualified', emoji='👩🏼\u200d🤝\u200d👨🏾', name='woman and man holding hands: medium-light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍🤝‍👨🏿': EmojiEntry(codepoint='1F469 1F3FC 200D 1F91D 200D 1F468 1F3FF', status='fully-qualified', emoji='👩🏼\u200d🤝\u200d👨🏿', name='woman and man holding hands: medium-light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍🤝‍👨🏻': EmojiEntry(codepoint='1F469 1F3FD 200D 1F91D 200D 1F468 1F3FB', status='fully-qualified', emoji='👩🏽\u200d🤝\u200d👨🏻', name='woman and man holding hands: medium skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍🤝‍👨🏼': EmojiEntry(codepoint='1F469 1F3FD 200D 1F91D 200D 1F468 1F3FC', status='fully-qualified', emoji='👩🏽\u200d🤝\u200d👨🏼', name='woman and man holding hands: medium skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👫🏽': EmojiEntry(codepoint='1F46B 1F3FD', status='fully-qualified', emoji='👫🏽', name='woman and man holding hands: medium skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍🤝‍👨🏾': EmojiEntry(codepoint='1F469 1F3FD 200D 1F91D 200D 1F468 1F3FE', status='fully-qualified', emoji='👩🏽\u200d🤝\u200d👨🏾', name='woman and man holding hands: medium skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍🤝‍👨🏿': EmojiEntry(codepoint='1F469 1F3FD 200D 1F91D 200D 1F468 1F3FF', status='fully-qualified', emoji='👩🏽\u200d🤝\u200d👨🏿', name='woman and man holding hands: medium skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍🤝‍👨🏻': EmojiEntry(codepoint='1F469 1F3FE 200D 1F91D 200D 1F468 1F3FB', status='fully-qualified', emoji='👩🏾\u200d🤝\u200d👨🏻', name='woman and man holding hands: medium-dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍🤝‍👨🏼': EmojiEntry(codepoint='1F469 1F3FE 200D 1F91D 200D 1F468 1F3FC', status='fully-qualified', emoji='👩🏾\u200d🤝\u200d👨🏼', name='woman and man holding hands: medium-dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍🤝‍👨🏽': EmojiEntry(codepoint='1F469 1F3FE 200D 1F91D 200D 1F468 1F3FD', status='fully-qualified', emoji='👩🏾\u200d🤝\u200d👨🏽', name='woman and man holding hands: medium-dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👫🏾': EmojiEntry(codepoint='1F46B 1F3FE', status='fully-qualified', emoji='👫🏾', name='woman and man holding hands: medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍🤝‍👨🏿': EmojiEntry(codepoint='1F469 1F3FE 200D 1F91D 200D 1F468 1F3FF', status='fully-qualified', emoji='👩🏾\u200d🤝\u200d👨🏿', name='woman and man holding hands: medium-dark skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍🤝‍👨🏻': EmojiEntry(codepoint='1F469 1F3FF 200D 1F91D 200D 1F468 1F3FB', status='fully-qualified', emoji='👩🏿\u200d🤝\u200d👨🏻', name='woman and man holding hands: dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍🤝‍👨🏼': EmojiEntry(codepoint='1F469 1F3FF 200D 1F91D 200D 1F468 1F3FC', status='fully-qualified', emoji='👩🏿\u200d🤝\u200d👨🏼', name='woman and man holding hands: dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍🤝‍👨🏽': EmojiEntry(codepoint='1F469 1F3FF 200D 1F91D 200D 1F468 1F3FD', status='fully-qualified', emoji='👩🏿\u200d🤝\u200d👨🏽', name='woman and man holding hands: dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍🤝‍👨🏾': EmojiEntry(codepoint='1F469 1F3FF 200D 1F91D 200D 1F468 1F3FE', status='fully-qualified', emoji='👩🏿\u200d🤝\u200d👨🏾', name='woman and man holding hands: dark skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👫🏿': EmojiEntry(codepoint='1F46B 1F3FF', status='fully-qualified', emoji='👫🏿', name='woman and man holding hands: dark skin tone', group='People & Body', sub_group='family'),
    '👬': EmojiEntry(codepoint='1F46C', status='fully-qualified', emoji='👬', name='men holding hands', group='People & Body', sub_group='family'),
    '👬🏻': EmojiEntry(codepoint='1F46C 1F3FB', status='fully-qualified', emoji='👬🏻', name='men holding hands: light skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍🤝‍👨🏼': EmojiEntry(codepoint='1F468 1F3FB 200D 1F91D 200D 1F468 1F3FC', status='fully-qualified', emoji='👨🏻\u200d🤝\u200d👨🏼', name='men holding hands: light skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍🤝‍👨🏽': EmojiEntry(codepoint='1F468 1F3FB 200D 1F91D 200D 1F468 1F3FD', status='fully-qualified', emoji='👨🏻\u200d🤝\u200d👨🏽', name='men holding hands: light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍🤝‍👨🏾': EmojiEntry(codepoint='1F468 1F3FB 200D 1F91D 200D 1F468 1F3FE', status='fully-qualified', emoji='👨🏻\u200d🤝\u200d👨🏾', name='men holding hands: light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍🤝‍👨🏿': EmojiEntry(codepoint='1F468 1F3FB 200D 1F91D 200D 1F468 1F3FF', status='fully-qualified', emoji='👨🏻\u200d🤝\u200d👨🏿', name='men holding hands: light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍🤝‍👨🏻': EmojiEntry(codepoint='1F468 1F3FC 200D 1F91D 200D 1F468 1F3FB', status='fully-qualified', emoji='👨🏼\u200d🤝\u200d👨🏻', name='men holding hands: medium-light skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👬🏼': EmojiEntry(codepoint='1F46C 1F3FC', status='fully-qualified', emoji='👬🏼', name='men holding hands: medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍🤝‍👨🏽': EmojiEntry(codepoint='1F468 1F3FC 200D 1F91D 200D 1F468 1F3FD', status='fully-qualified', emoji='👨🏼\u200d🤝\u200d👨🏽', name='men holding hands: medium-light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍🤝‍👨🏾': EmojiEntry(codepoint='1F468 1F3FC 200D 1F91D 200D 1F468 1F3FE', status='fully-qualified', emoji='👨🏼\u200d🤝\u200d👨🏾', name='men holding hands: medium-light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍🤝‍👨🏿': EmojiEntry(codepoint='1F468 1F3FC 200D 1F91D 200D 1F468 1F3FF', status='fully-qualified', emoji='👨🏼\u200d🤝\u200d👨🏿', name='men holding hands: medium-light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍🤝‍👨🏻': EmojiEntry(codepoint='1F468 1F3FD 200D 1F91D 200D 1F468 1F3FB', status='fully-qualified', emoji='👨🏽\u200d🤝\u200d👨🏻', name='men holding hands: medium skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍🤝‍👨🏼': EmojiEntry(codepoint='1F468 1F3FD 200D 1F91D 200D 1F468 1F3FC', status='fully-qualified', emoji='👨🏽\u200d🤝\u200d👨🏼', name='men holding hands: medium skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👬🏽': EmojiEntry(codepoint='1F46C 1F3FD', status='fully-qualified', emoji='👬🏽', name='men holding hands: medium skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍🤝‍👨🏾': EmojiEntry(codepoint='1F468 1F3FD 200D 1F91D 200D 1F468 1F3FE', status='fully-qualified', emoji='👨🏽\u200d🤝\u200d👨🏾', name='men holding hands: medium skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍🤝‍👨🏿': EmojiEntry(codepoint='1F468 1F3FD 200D 1F91D 200D 1F468 1F3FF', status='fully-qualified', emoji='👨🏽\u200d🤝\u200d👨🏿', name='men holding hands: medium skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍🤝‍👨🏻': EmojiEntry(codepoint='1F468 1F3FE 200D 1F91D 200D 1F468 1F3FB', status='fully-qualified', emoji='👨🏾\u200d🤝\u200d👨🏻', name='men holding hands: medium-dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍🤝‍👨🏼': EmojiEntry(codepoint='1F468 1F3FE 200D 1F91D 200D 1F468 1F3FC', status='fully-qualified', emoji='👨🏾\u200d🤝\u200d👨🏼', name='men holding hands: medium-dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍🤝‍👨🏽': EmojiEntry(codepoint='1F468 1F3FE 200D 1F91D 200D 1F468 1F3FD', status='fully-qualified', emoji='👨🏾\u200d🤝\u200d👨🏽', name='men holding hands: medium-dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👬🏾': EmojiEntry(codepoint='1F46C 1F3FE', status='fully-qualified', emoji='👬🏾', name='men holding hands: medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍🤝‍👨🏿': EmojiEntry(codepoint='1F468 1F3FE 200D 1F91D 200D 1F468 1F3FF', status='fully-qualified', emoji='👨🏾\u200d🤝\u200d👨🏿', name='men holding hands: medium-dark skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍🤝‍👨🏻': EmojiEntry(codepoint='1F468 1F3FF 200D 1F91D 200D 1F468 1F3FB', status='fully-qualified', emoji='👨🏿\u200d🤝\u200d👨🏻', name='men holding hands: dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍🤝‍👨🏼': EmojiEntry(codepoint='1F468 1F3FF 200D 1F91D 200D 1F468 1F3FC', status='fully-qualified', emoji='👨🏿\u200d🤝\u200d👨🏼', name='men holding hands: dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍🤝‍👨🏽': EmojiEntry(codepoint='1F468 1F3FF 200D 1F91D 200D 1F468 1F3FD', status='fully-qualified', emoji='👨🏿\u200d🤝\u200d👨🏽', name='men holding hands: dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍🤝‍👨🏾': EmojiEntry(codepoint='1F468 1F3FF 200D 1F91D 200D 1F468 1F3FE', status='fully-qualified', emoji='👨🏿\u200d🤝\u200d👨🏾', name='men holding hands: dark skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👬🏿': EmojiEntry(codepoint='1F46C 1F3FF', status='fully-qualified', emoji='👬🏿', name='men holding hands: dark skin tone', group='People & Body', sub_group='family'),
    '💏': EmojiEntry(codepoint='1F48F', status='fully-qualified', emoji='💏', name='kiss', group='People & Body', sub_group='family'),
    '💏🏻': EmojiEntry(codepoint='1F48F 1F3FB', status='fully-qualified', emoji='💏🏻', name='kiss: light skin tone', group='People & Body', sub_group='family'),
    '💏🏼': EmojiEntry(codepoint='1F48F 1F3FC', status='fully-qualified', emoji='💏🏼', name='kiss: medium-light skin tone', group='People & Body', sub_group='family'),
    '💏🏽': EmojiEntry(codepoint='1F48F 1F3FD', status='fully-qualified', emoji='💏🏽', name='kiss: medium skin tone', group='People & Body', sub_group='family'),
    '💏🏾': EmojiEntry(codepoint='1F48F 1F3FE', status='fully-qualified', emoji='💏🏾', name='kiss: medium-dark skin tone', group='People & Body', sub_group='family'),
    '💏🏿': EmojiEntry(codepoint='1F48F 1F3FF', status='fully-qualified', emoji='💏🏿', name='kiss: dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏻‍❤️‍💋‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2764 FE0F 200D 1F48B 200D 1F9D1 1F3FC', status='fully-qualified', emoji='🧑🏻\u200d❤️\u200d💋\u200d🧑🏼', name='kiss: person, person, light skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏻‍❤‍💋‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2764 200D 1F48B 200D 1F9D1 1F3FC', status='minimally-qualified', emoji='🧑🏻\u200d❤\u200d💋\u200d🧑🏼', name='kiss: person, person, light skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏻‍❤️‍💋‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2764 FE0F 200D 1F48B 200D 1F9D1 1F3FD', status='fully-qualified', emoji='🧑🏻\u200d❤️\u200d💋\u200d🧑🏽', name='kiss: person, person, light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏻‍❤‍💋‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2764 200D 1F48B 200D 1F9D1 1F3FD', status='minimally-qualified', emoji='🧑🏻\u200d❤\u200d💋\u200d🧑🏽', name='kiss: person, person, light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏻‍❤️‍💋‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2764 FE0F 200D 1F48B 200D 1F9D1 1F3FE', status='fully-qualified', emoji='🧑🏻\u200d❤️\u200d💋\u200d🧑🏾', name='kiss: person, person, light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏻‍❤‍💋‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2764 200D 1F48B 200D 1F9D1 1F3FE', status='minimally-qualified', emoji='🧑🏻\u200d❤\u200d💋\u200d🧑🏾', name='kiss: person, person, light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏻‍❤️‍💋‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2764 FE0F 200D 1F48B 200D 1F9D1 1F3FF', status='fully-qualified', emoji='🧑🏻\u200d❤️\u200d💋\u200d🧑🏿', name='kiss: person, person, light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏻‍❤‍💋‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2764 200D 1F48B 200D 1F9D1 1F3FF', status='minimally-qualified', emoji='🧑🏻\u200d❤\u200d💋\u200d🧑🏿', name='kiss: person, person, light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍❤️‍💋‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2764 FE0F 200D 1F48B 200D 1F9D1 1F3FB', status='fully-qualified', emoji='🧑🏼\u200d❤️\u200d💋\u200d🧑🏻', name='kiss: person, person, medium-light skin tone, light skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍❤‍💋‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2764 200D 1F48B 200D 1F9D1 1F3FB', status='minimally-qualified', emoji='🧑🏼\u200d❤\u200d💋\u200d🧑🏻', name='kiss: person, person, medium-light skin tone, light skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍❤️‍💋‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2764 FE0F 200D 1F48B 200D 1F9D1 1F3FD', status='fully-qualified', emoji='🧑🏼\u200d❤️\u200d💋\u200d🧑🏽', name='kiss: person, person, medium-light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍❤‍💋‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2764 200D 1F48B 200D 1F9D1 1F3FD', status='minimally-qualified', emoji='🧑🏼\u200d❤\u200d💋\u200d🧑🏽', name='kiss: person, person, medium-light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍❤️‍💋‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2764 FE0F 200D 1F48B 200D 1F9D1 1F3FE', status='fully-qualified', emoji='🧑🏼\u200d❤️\u200d💋\u200d🧑🏾', name='kiss: person, person, medium-light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍❤‍💋‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2764 200D 1F48B 200D 1F9D1 1F3FE', status='minimally-qualified', emoji='🧑🏼\u200d❤\u200d💋\u200d🧑🏾', name='kiss: person, person, medium-light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍❤️‍💋‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2764 FE0F 200D 1F48B 200D 1F9D1 1F3FF', status='fully-qualified', emoji='🧑🏼\u200d❤️\u200d💋\u200d🧑🏿', name='kiss: person, person, medium-light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍❤‍💋‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2764 200D 1F48B 200D 1F9D1 1F3FF', status='minimally-qualified', emoji='🧑🏼\u200d❤\u200d💋\u200d🧑🏿', name='kiss: person, person, medium-light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍❤️‍💋‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2764 FE0F 200D 1F48B 200D 1F9D1 1F3FB', status='fully-qualified', emoji='🧑🏽\u200d❤️\u200d💋\u200d🧑🏻', name='kiss: person, person, medium skin tone, light skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍❤‍💋‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2764 200D 1F48B 200D 1F9D1 1F3FB', status='minimally-qualified', emoji='🧑🏽\u200d❤\u200d💋\u200d🧑🏻', name='kiss: person, person, medium skin tone, light skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍❤️‍💋‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2764 FE0F 200D 1F48B 200D 1F9D1 1F3FC', status='fully-qualified', emoji='🧑🏽\u200d❤️\u200d💋\u200d🧑🏼', name='kiss: person, person, medium skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍❤‍💋‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2764 200D 1F48B 200D 1F9D1 1F3FC', status='minimally-qualified', emoji='🧑🏽\u200d❤\u200d💋\u200d🧑🏼', name='kiss: person, person, medium skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍❤️‍💋‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2764 FE0F 200D 1F48B 200D 1F9D1 1F3FE', status='fully-qualified', emoji='🧑🏽\u200d❤️\u200d💋\u200d🧑🏾', name='kiss: person, person, medium skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍❤‍💋‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2764 200D 1F48B 200D 1F9D1 1F3FE', status='minimally-qualified', emoji='🧑🏽\u200d❤\u200d💋\u200d🧑🏾', name='kiss: person, person, medium skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍❤️‍💋‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2764 FE0F 200D 1F48B 200D 1F9D1 1F3FF', status='fully-qualified', emoji='🧑🏽\u200d❤️\u200d💋\u200d🧑🏿', name='kiss: person, person, medium skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍❤‍💋‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2764 200D 1F48B 200D 1F9D1 1F3FF', status='minimally-qualified', emoji='🧑🏽\u200d❤\u200d💋\u200d🧑🏿', name='kiss: person, person, medium skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍❤️‍💋‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2764 FE0F 200D 1F48B 200D 1F9D1 1F3FB', status='fully-qualified', emoji='🧑🏾\u200d❤️\u200d💋\u200d🧑🏻', name='kiss: person, person, medium-dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍❤‍💋‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2764 200D 1F48B 200D 1F9D1 1F3FB', status='minimally-qualified', emoji='🧑🏾\u200d❤\u200d💋\u200d🧑🏻', name='kiss: person, person, medium-dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍❤️‍💋‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2764 FE0F 200D 1F48B 200D 1F9D1 1F3FC', status='fully-qualified', emoji='🧑🏾\u200d❤️\u200d💋\u200d🧑🏼', name='kiss: person, person, medium-dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍❤‍💋‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2764 200D 1F48B 200D 1F9D1 1F3FC', status='minimally-qualified', emoji='🧑🏾\u200d❤\u200d💋\u200d🧑🏼', name='kiss: person, person, medium-dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍❤️‍💋‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2764 FE0F 200D 1F48B 200D 1F9D1 1F3FD', status='fully-qualified', emoji='🧑🏾\u200d❤️\u200d💋\u200d🧑🏽', name='kiss: person, person, medium-dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍❤‍💋‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2764 200D 1F48B 200D 1F9D1 1F3FD', status='minimally-qualified', emoji='🧑🏾\u200d❤\u200d💋\u200d🧑🏽', name='kiss: person, person, medium-dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍❤️‍💋‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2764 FE0F 200D 1F48B 200D 1F9D1 1F3FF', status='fully-qualified', emoji='🧑🏾\u200d❤️\u200d💋\u200d🧑🏿', name='kiss: person, person, medium-dark skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍❤‍💋‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2764 200D 1F48B 200D 1F9D1 1F3FF', status='minimally-qualified', emoji='🧑🏾\u200d❤\u200d💋\u200d🧑🏿', name='kiss: person, person, medium-dark skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍❤️‍💋‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2764 FE0F 200D 1F48B 200D 1F9D1 1F3FB', status='fully-qualified', emoji='🧑🏿\u200d❤️\u200d💋\u200d🧑🏻', name='kiss: person, person, dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍❤‍💋‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2764 200D 1F48B 200D 1F9D1 1F3FB', status='minimally-qualified', emoji='🧑🏿\u200d❤\u200d💋\u200d🧑🏻', name='kiss: person, person, dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍❤️‍💋‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2764 FE0F 200D 1F48B 200D 1F9D1 1F3FC', status='fully-qualified', emoji='🧑🏿\u200d❤️\u200d💋\u200d🧑🏼', name='kiss: person, person, dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍❤‍💋‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2764 200D 1F48B 200D 1F9D1 1F3FC', status='minimally-qualified', emoji='🧑🏿\u200d❤\u200d💋\u200d🧑🏼', name='kiss: person, person, dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍❤️‍💋‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2764 FE0F 200D 1F48B 200D 1F9D1 1F3FD', status='fully-qualified', emoji='🧑🏿\u200d❤️\u200d💋\u200d🧑🏽', name='kiss: person, person, dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍❤‍💋‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2764 200D 1F48B 200D 1F9D1 1F3FD', status='minimally-qualified', emoji='🧑🏿\u200d❤\u200d💋\u200d🧑🏽', name='kiss: person, person, dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍❤️‍💋‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2764 FE0F 200D 1F48B 200D 1F9D1 1F3FE', status='fully-qualified', emoji='🧑🏿\u200d❤️\u200d💋\u200d🧑🏾', name='kiss: person, person, dark skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍❤‍💋‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2764 200D 1F48B 200D 1F9D1 1F3FE', status='minimally-qualified', emoji='🧑🏿\u200d❤\u200d💋\u200d🧑🏾', name='kiss: person, person, dark skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩‍❤️‍💋‍👨': EmojiEntry(codepoint='1F469 200D 2764 FE0F 200D 1F48B 200D 1F468', status='fully-qualified', emoji='👩\u200d❤️\u200d💋\u200d👨', name='kiss: woman, man', group='People & Body', sub_group='family'),
    '👩‍❤‍💋‍👨': EmojiEntry(codepoint='1F469 200D 2764 200D 1F48B 200D 1F468', status='minimally-qualified', emoji='👩\u200d❤\u200d💋\u200d👨', name='kiss: woman, man', group='People & Body', sub_group='family'),
    '👩🏻‍❤️‍💋‍👨🏻': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FB', status='fully-qualified', emoji='👩🏻\u200d❤️\u200d💋\u200d👨🏻', name='kiss: woman, man, light skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤‍💋‍👨🏻': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 200D 1F48B 200D 1F468 1F3FB', status='minimally-qualified', emoji='👩🏻\u200d❤\u200d💋\u200d👨🏻', name='kiss: woman, man, light skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤️‍💋‍👨🏼': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FC', status='fully-qualified', emoji='👩🏻\u200d❤️\u200d💋\u200d👨🏼', name='kiss: woman, man, light skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤‍💋‍👨🏼': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 200D 1F48B 200D 1F468 1F3FC', status='minimally-qualified', emoji='👩🏻\u200d❤\u200d💋\u200d👨🏼', name='kiss: woman, man, light skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤️‍💋‍👨🏽': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FD', status='fully-qualified', emoji='👩🏻\u200d❤️\u200d💋\u200d👨🏽', name='kiss: woman, man, light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤‍💋‍👨🏽': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 200D 1F48B 200D 1F468 1F3FD', status='minimally-qualified', emoji='👩🏻\u200d❤\u200d💋\u200d👨🏽', name='kiss: woman, man, light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤️‍💋‍👨🏾': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FE', status='fully-qualified', emoji='👩🏻\u200d❤️\u200d💋\u200d👨🏾', name='kiss: woman, man, light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤‍💋‍👨🏾': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 200D 1F48B 200D 1F468 1F3FE', status='minimally-qualified', emoji='👩🏻\u200d❤\u200d💋\u200d👨🏾', name='kiss: woman, man, light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤️‍💋‍👨🏿': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FF', status='fully-qualified', emoji='👩🏻\u200d❤️\u200d💋\u200d👨🏿', name='kiss: woman, man, light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤‍💋‍👨🏿': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 200D 1F48B 200D 1F468 1F3FF', status='minimally-qualified', emoji='👩🏻\u200d❤\u200d💋\u200d👨🏿', name='kiss: woman, man, light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤️‍💋‍👨🏻': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FB', status='fully-qualified', emoji='👩🏼\u200d❤️\u200d💋\u200d👨🏻', name='kiss: woman, man, medium-light skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤‍💋‍👨🏻': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 200D 1F48B 200D 1F468 1F3FB', status='minimally-qualified', emoji='👩🏼\u200d❤\u200d💋\u200d👨🏻', name='kiss: woman, man, medium-light skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤️‍💋‍👨🏼': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FC', status='fully-qualified', emoji='👩🏼\u200d❤️\u200d💋\u200d👨🏼', name='kiss: woman, man, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤‍💋‍👨🏼': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 200D 1F48B 200D 1F468 1F3FC', status='minimally-qualified', emoji='👩🏼\u200d❤\u200d💋\u200d👨🏼', name='kiss: woman, man, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤️‍💋‍👨🏽': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FD', status='fully-qualified', emoji='👩🏼\u200d❤️\u200d💋\u200d👨🏽', name='kiss: woman, man, medium-light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤‍💋‍👨🏽': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 200D 1F48B 200D 1F468 1F3FD', status='minimally-qualified', emoji='👩🏼\u200d❤\u200d💋\u200d👨🏽', name='kiss: woman, man, medium-light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤️‍💋‍👨🏾': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FE', status='fully-qualified', emoji='👩🏼\u200d❤️\u200d💋\u200d👨🏾', name='kiss: woman, man, medium-light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤‍💋‍👨🏾': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 200D 1F48B 200D 1F468 1F3FE', status='minimally-qualified', emoji='👩🏼\u200d❤\u200d💋\u200d👨🏾', name='kiss: woman, man, medium-light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤️‍💋‍👨🏿': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FF', status='fully-qualified', emoji='👩🏼\u200d❤️\u200d💋\u200d👨🏿', name='kiss: woman, man, medium-light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤‍💋‍👨🏿': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 200D 1F48B 200D 1F468 1F3FF', status='minimally-qualified', emoji='👩🏼\u200d❤\u200d💋\u200d👨🏿', name='kiss: woman, man, medium-light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤️‍💋‍👨🏻': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FB', status='fully-qualified', emoji='👩🏽\u200d❤️\u200d💋\u200d👨🏻', name='kiss: woman, man, medium skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤‍💋‍👨🏻': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 200D 1F48B 200D 1F468 1F3FB', status='minimally-qualified', emoji='👩🏽\u200d❤\u200d💋\u200d👨🏻', name='kiss: woman, man, medium skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤️‍💋‍👨🏼': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FC', status='fully-qualified', emoji='👩🏽\u200d❤️\u200d💋\u200d👨🏼', name='kiss: woman, man, medium skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤‍💋‍👨🏼': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 200D 1F48B 200D 1F468 1F3FC', status='minimally-qualified', emoji='👩🏽\u200d❤\u200d💋\u200d👨🏼', name='kiss: woman, man, medium skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤️‍💋‍👨🏽': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FD', status='fully-qualified', emoji='👩🏽\u200d❤️\u200d💋\u200d👨🏽', name='kiss: woman, man, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤‍💋‍👨🏽': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 200D 1F48B 200D 1F468 1F3FD', status='minimally-qualified', emoji='👩🏽\u200d❤\u200d💋\u200d👨🏽', name='kiss: woman, man, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤️‍💋‍👨🏾': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FE', status='fully-qualified', emoji='👩🏽\u200d❤️\u200d💋\u200d👨🏾', name='kiss: woman, man, medium skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤‍💋‍👨🏾': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 200D 1F48B 200D 1F468 1F3FE', status='minimally-qualified', emoji='👩🏽\u200d❤\u200d💋\u200d👨🏾', name='kiss: woman, man, medium skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤️‍💋‍👨🏿': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FF', status='fully-qualified', emoji='👩🏽\u200d❤️\u200d💋\u200d👨🏿', name='kiss: woman, man, medium skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤‍💋‍👨🏿': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 200D 1F48B 200D 1F468 1F3FF', status='minimally-qualified', emoji='👩🏽\u200d❤\u200d💋\u200d👨🏿', name='kiss: woman, man, medium skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤️‍💋‍👨🏻': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FB', status='fully-qualified', emoji='👩🏾\u200d❤️\u200d💋\u200d👨🏻', name='kiss: woman, man, medium-dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤‍💋‍👨🏻': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 200D 1F48B 200D 1F468 1F3FB', status='minimally-qualified', emoji='👩🏾\u200d❤\u200d💋\u200d👨🏻', name='kiss: woman, man, medium-dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤️‍💋‍👨🏼': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FC', status='fully-qualified', emoji='👩🏾\u200d❤️\u200d💋\u200d👨🏼', name='kiss: woman, man, medium-dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤‍💋‍👨🏼': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 200D 1F48B 200D 1F468 1F3FC', status='minimally-qualified', emoji='👩🏾\u200d❤\u200d💋\u200d👨🏼', name='kiss: woman, man, medium-dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤️‍💋‍👨🏽': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FD', status='fully-qualified', emoji='👩🏾\u200d❤️\u200d💋\u200d👨🏽', name='kiss: woman, man, medium-dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤‍💋‍👨🏽': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 200D 1F48B 200D 1F468 1F3FD', status='minimally-qualified', emoji='👩🏾\u200d❤\u200d💋\u200d👨🏽', name='kiss: woman, man, medium-dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤️‍💋‍👨🏾': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FE', status='fully-qualified', emoji='👩🏾\u200d❤️\u200d💋\u200d👨🏾', name='kiss: woman, man, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤‍💋‍👨🏾': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 200D 1F48B 200D 1F468 1F3FE', status='minimally-qualified', emoji='👩🏾\u200d❤\u200d💋\u200d👨🏾', name='kiss: woman, man, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤️‍💋‍👨🏿': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FF', status='fully-qualified', emoji='👩🏾\u200d❤️\u200d💋\u200d👨🏿', name='kiss: woman, man, medium-dark skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤‍💋‍👨🏿': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 200D 1F48B 200D 1F468 1F3FF', status='minimally-qualified', emoji='👩🏾\u200d❤\u200d💋\u200d👨🏿', name='kiss: woman, man, medium-dark skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤️‍💋‍👨🏻': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FB', status='fully-qualified', emoji='👩🏿\u200d❤️\u200d💋\u200d👨🏻', name='kiss: woman, man, dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤‍💋‍👨🏻': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 200D 1F48B 200D 1F468 1F3FB', status='minimally-qualified', emoji='👩🏿\u200d❤\u200d💋\u200d👨🏻', name='kiss: woman, man, dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤️‍💋‍👨🏼': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FC', status='fully-qualified', emoji='👩🏿\u200d❤️\u200d💋\u200d👨🏼', name='kiss: woman, man, dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤‍💋‍👨🏼': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 200D 1F48B 200D 1F468 1F3FC', status='minimally-qualified', emoji='👩🏿\u200d❤\u200d💋\u200d👨🏼', name='kiss: woman, man, dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤️‍💋‍👨🏽': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FD', status='fully-qualified', emoji='👩🏿\u200d❤️\u200d💋\u200d👨🏽', name='kiss: woman, man, dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤‍💋‍👨🏽': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 200D 1F48B 200D 1F468 1F3FD', status='minimally-qualified', emoji='👩🏿\u200d❤\u200d💋\u200d👨🏽', name='kiss: woman, man, dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤️‍💋‍👨🏾': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FE', status='fully-qualified', emoji='👩🏿\u200d❤️\u200d💋\u200d👨🏾', name='kiss: woman, man, dark skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤‍💋‍👨🏾': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 200D 1F48B 200D 1F468 1F3FE', status='minimally-qualified', emoji='👩🏿\u200d❤\u200d💋\u200d👨🏾', name='kiss: woman, man, dark skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤️‍💋‍👨🏿': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FF', status='fully-qualified', emoji='👩🏿\u200d❤️\u200d💋\u200d👨🏿', name='kiss: woman, man, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤‍💋‍👨🏿': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 200D 1F48B 200D 1F468 1F3FF', status='minimally-qualified', emoji='👩🏿\u200d❤\u200d💋\u200d👨🏿', name='kiss: woman, man, dark skin tone', group='People & Body', sub_group='family'),
    '👨‍❤️‍💋‍👨': EmojiEntry(codepoint='1F468 200D 2764 FE0F 200D 1F48B 200D 1F468', status='fully-qualified', emoji='👨\u200d❤️\u200d💋\u200d👨', name='kiss: man, man', group='People & Body', sub_group='family'),
    '👨‍❤‍💋‍👨': EmojiEntry(codepoint='1F468 200D 2764 200D 1F48B 200D 1F468', status='minimally-qualified', emoji='👨\u200d❤\u200d💋\u200d👨', name='kiss: man, man', group='People & Body', sub_group='family'),
    '👨🏻‍❤️‍💋‍👨🏻': EmojiEntry(codepoint='1F468 1F3FB 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FB', status='fully-qualified', emoji='👨🏻\u200d❤️\u200d💋\u200d👨🏻', name='kiss: man, man, light skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍❤‍💋‍👨🏻': EmojiEntry(codepoint='1F468 1F3FB 200D 2764 200D 1F48B 200D 1F468 1F3FB', status='minimally-qualified', emoji='👨🏻\u200d❤\u200d💋\u200d👨🏻', name='kiss: man, man, light skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍❤️‍💋‍👨🏼': EmojiEntry(codepoint='1F468 1F3FB 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FC', status='fully-qualified', emoji='👨🏻\u200d❤️\u200d💋\u200d👨🏼', name='kiss: man, man, light skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍❤‍💋‍👨🏼': EmojiEntry(codepoint='1F468 1F3FB 200D 2764 200D 1F48B 200D 1F468 1F3FC', status='minimally-qualified', emoji='👨🏻\u200d❤\u200d💋\u200d👨🏼', name='kiss: man, man, light skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍❤️‍💋‍👨🏽': EmojiEntry(codepoint='1F468 1F3FB 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FD', status='fully-qualified', emoji='👨🏻\u200d❤️\u200d💋\u200d👨🏽', name='kiss: man, man, light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍❤‍💋‍👨🏽': EmojiEntry(codepoint='1F468 1F3FB 200D 2764 200D 1F48B 200D 1F468 1F3FD', status='minimally-qualified', emoji='👨🏻\u200d❤\u200d💋\u200d👨🏽', name='kiss: man, man, light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍❤️‍💋‍👨🏾': EmojiEntry(codepoint='1F468 1F3FB 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FE', status='fully-qualified', emoji='👨🏻\u200d❤️\u200d💋\u200d👨🏾', name='kiss: man, man, light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍❤‍💋‍👨🏾': EmojiEntry(codepoint='1F468 1F3FB 200D 2764 200D 1F48B 200D 1F468 1F3FE', status='minimally-qualified', emoji='👨🏻\u200d❤\u200d💋\u200d👨🏾', name='kiss: man, man, light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍❤️‍💋‍👨🏿': EmojiEntry(codepoint='1F468 1F3FB 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FF', status='fully-qualified', emoji='👨🏻\u200d❤️\u200d💋\u200d👨🏿', name='kiss: man, man, light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍❤‍💋‍👨🏿': EmojiEntry(codepoint='1F468 1F3FB 200D 2764 200D 1F48B 200D 1F468 1F3FF', status='minimally-qualified', emoji='👨🏻\u200d❤\u200d💋\u200d👨🏿', name='kiss: man, man, light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍❤️‍💋‍👨🏻': EmojiEntry(codepoint='1F468 1F3FC 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FB', status='fully-qualified', emoji='👨🏼\u200d❤️\u200d💋\u200d👨🏻', name='kiss: man, man, medium-light skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍❤‍💋‍👨🏻': EmojiEntry(codepoint='1F468 1F3FC 200D 2764 200D 1F48B 200D 1F468 1F3FB', status='minimally-qualified', emoji='👨🏼\u200d❤\u200d💋\u200d👨🏻', name='kiss: man, man, medium-light skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍❤️‍💋‍👨🏼': EmojiEntry(codepoint='1F468 1F3FC 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FC', status='fully-qualified', emoji='👨🏼\u200d❤️\u200d💋\u200d👨🏼', name='kiss: man, man, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍❤‍💋‍👨🏼': EmojiEntry(codepoint='1F468 1F3FC 200D 2764 200D 1F48B 200D 1F468 1F3FC', status='minimally-qualified', emoji='👨🏼\u200d❤\u200d💋\u200d👨🏼', name='kiss: man, man, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍❤️‍💋‍👨🏽': EmojiEntry(codepoint='1F468 1F3FC 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FD', status='fully-qualified', emoji='👨🏼\u200d❤️\u200d💋\u200d👨🏽', name='kiss: man, man, medium-light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍❤‍💋‍👨🏽': EmojiEntry(codepoint='1F468 1F3FC 200D 2764 200D 1F48B 200D 1F468 1F3FD', status='minimally-qualified', emoji='👨🏼\u200d❤\u200d💋\u200d👨🏽', name='kiss: man, man, medium-light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍❤️‍💋‍👨🏾': EmojiEntry(codepoint='1F468 1F3FC 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FE', status='fully-qualified', emoji='👨🏼\u200d❤️\u200d💋\u200d👨🏾', name='kiss: man, man, medium-light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍❤‍💋‍👨🏾': EmojiEntry(codepoint='1F468 1F3FC 200D 2764 200D 1F48B 200D 1F468 1F3FE', status='minimally-qualified', emoji='👨🏼\u200d❤\u200d💋\u200d👨🏾', name='kiss: man, man, medium-light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍❤️‍💋‍👨🏿': EmojiEntry(codepoint='1F468 1F3FC 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FF', status='fully-qualified', emoji='👨🏼\u200d❤️\u200d💋\u200d👨🏿', name='kiss: man, man, medium-light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍❤‍💋‍👨🏿': EmojiEntry(codepoint='1F468 1F3FC 200D 2764 200D 1F48B 200D 1F468 1F3FF', status='minimally-qualified', emoji='👨🏼\u200d❤\u200d💋\u200d👨🏿', name='kiss: man, man, medium-light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍❤️‍💋‍👨🏻': EmojiEntry(codepoint='1F468 1F3FD 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FB', status='fully-qualified', emoji='👨🏽\u200d❤️\u200d💋\u200d👨🏻', name='kiss: man, man, medium skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍❤‍💋‍👨🏻': EmojiEntry(codepoint='1F468 1F3FD 200D 2764 200D 1F48B 200D 1F468 1F3FB', status='minimally-qualified', emoji='👨🏽\u200d❤\u200d💋\u200d👨🏻', name='kiss: man, man, medium skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍❤️‍💋‍👨🏼': EmojiEntry(codepoint='1F468 1F3FD 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FC', status='fully-qualified', emoji='👨🏽\u200d❤️\u200d💋\u200d👨🏼', name='kiss: man, man, medium skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍❤‍💋‍👨🏼': EmojiEntry(codepoint='1F468 1F3FD 200D 2764 200D 1F48B 200D 1F468 1F3FC', status='minimally-qualified', emoji='👨🏽\u200d❤\u200d💋\u200d👨🏼', name='kiss: man, man, medium skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍❤️‍💋‍👨🏽': EmojiEntry(codepoint='1F468 1F3FD 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FD', status='fully-qualified', emoji='👨🏽\u200d❤️\u200d💋\u200d👨🏽', name='kiss: man, man, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍❤‍💋‍👨🏽': EmojiEntry(codepoint='1F468 1F3FD 200D 2764 200D 1F48B 200D 1F468 1F3FD', status='minimally-qualified', emoji='👨🏽\u200d❤\u200d💋\u200d👨🏽', name='kiss: man, man, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍❤️‍💋‍👨🏾': EmojiEntry(codepoint='1F468 1F3FD 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FE', status='fully-qualified', emoji='👨🏽\u200d❤️\u200d💋\u200d👨🏾', name='kiss: man, man, medium skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍❤‍💋‍👨🏾': EmojiEntry(codepoint='1F468 1F3FD 200D 2764 200D 1F48B 200D 1F468 1F3FE', status='minimally-qualified', emoji='👨🏽\u200d❤\u200d💋\u200d👨🏾', name='kiss: man, man, medium skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍❤️‍💋‍👨🏿': EmojiEntry(codepoint='1F468 1F3FD 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FF', status='fully-qualified', emoji='👨🏽\u200d❤️\u200d💋\u200d👨🏿', name='kiss: man, man, medium skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍❤‍💋‍👨🏿': EmojiEntry(codepoint='1F468 1F3FD 200D 2764 200D 1F48B 200D 1F468 1F3FF', status='minimally-qualified', emoji='👨🏽\u200d❤\u200d💋\u200d👨🏿', name='kiss: man, man, medium skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍❤️‍💋‍👨🏻': EmojiEntry(codepoint='1F468 1F3FE 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FB', status='fully-qualified', emoji='👨🏾\u200d❤️\u200d💋\u200d👨🏻', name='kiss: man, man, medium-dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍❤‍💋‍👨🏻': EmojiEntry(codepoint='1F468 1F3FE 200D 2764 200D 1F48B 200D 1F468 1F3FB', status='minimally-qualified', emoji='👨🏾\u200d❤\u200d💋\u200d👨🏻', name='kiss: man, man, medium-dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍❤️‍💋‍👨🏼': EmojiEntry(codepoint='1F468 1F3FE 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FC', status='fully-qualified', emoji='👨🏾\u200d❤️\u200d💋\u200d👨🏼', name='kiss: man, man, medium-dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍❤‍💋‍👨🏼': EmojiEntry(codepoint='1F468 1F3FE 200D 2764 200D 1F48B 200D 1F468 1F3FC', status='minimally-qualified', emoji='👨🏾\u200d❤\u200d💋\u200d👨🏼', name='kiss: man, man, medium-dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍❤️‍💋‍👨🏽': EmojiEntry(codepoint='1F468 1F3FE 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FD', status='fully-qualified', emoji='👨🏾\u200d❤️\u200d💋\u200d👨🏽', name='kiss: man, man, medium-dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍❤‍💋‍👨🏽': EmojiEntry(codepoint='1F468 1F3FE 200D 2764 200D 1F48B 200D 1F468 1F3FD', status='minimally-qualified', emoji='👨🏾\u200d❤\u200d💋\u200d👨🏽', name='kiss: man, man, medium-dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍❤️‍💋‍👨🏾': EmojiEntry(codepoint='1F468 1F3FE 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FE', status='fully-qualified', emoji='👨🏾\u200d❤️\u200d💋\u200d👨🏾', name='kiss: man, man, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍❤‍💋‍👨🏾': EmojiEntry(codepoint='1F468 1F3FE 200D 2764 200D 1F48B 200D 1F468 1F3FE', status='minimally-qualified', emoji='👨🏾\u200d❤\u200d💋\u200d👨🏾', name='kiss: man, man, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍❤️‍💋‍👨🏿': EmojiEntry(codepoint='1F468 1F3FE 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FF', status='fully-qualified', emoji='👨🏾\u200d❤️\u200d💋\u200d👨🏿', name='kiss: man, man, medium-dark skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍❤‍💋‍👨🏿': EmojiEntry(codepoint='1F468 1F3FE 200D 2764 200D 1F48B 200D 1F468 1F3FF', status='minimally-qualified', emoji='👨🏾\u200d❤\u200d💋\u200d👨🏿', name='kiss: man, man, medium-dark skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍❤️‍💋‍👨🏻': EmojiEntry(codepoint='1F468 1F3FF 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FB', status='fully-qualified', emoji='👨🏿\u200d❤️\u200d💋\u200d👨🏻', name='kiss: man, man, dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍❤‍💋‍👨🏻': EmojiEntry(codepoint='1F468 1F3FF 200D 2764 200D 1F48B 200D 1F468 1F3FB', status='minimally-qualified', emoji='👨🏿\u200d❤\u200d💋\u200d👨🏻', name='kiss: man, man, dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍❤️‍💋‍👨🏼': EmojiEntry(codepoint='1F468 1F3FF 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FC', status='fully-qualified', emoji='👨🏿\u200d❤️\u200d💋\u200d👨🏼', name='kiss: man, man, dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍❤‍💋‍👨🏼': EmojiEntry(codepoint='1F468 1F3FF 200D 2764 200D 1F48B 200D 1F468 1F3FC', status='minimally-qualified', emoji='👨🏿\u200d❤\u200d💋\u200d👨🏼', name='kiss: man, man, dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍❤️‍💋‍👨🏽': EmojiEntry(codepoint='1F468 1F3FF 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FD', status='fully-qualified', emoji='👨🏿\u200d❤️\u200d💋\u200d👨🏽', name='kiss: man, man, dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍❤‍💋‍👨🏽': EmojiEntry(codepoint='1F468 1F3FF 200D 2764 200D 1F48B 200D 1F468 1F3FD', status='minimally-qualified', emoji='👨🏿\u200d❤\u200d💋\u200d👨🏽', name='kiss: man, man, dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍❤️‍💋‍👨🏾': EmojiEntry(codepoint='1F468 1F3FF 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FE', status='fully-qualified', emoji='👨🏿\u200d❤️\u200d💋\u200d👨🏾', name='kiss: man, man, dark skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍❤‍💋‍👨🏾': EmojiEntry(codepoint='1F468 1F3FF 200D 2764 200D 1F48B 200D 1F468 1F3FE', status='minimally-qualified', emoji='👨🏿\u200d❤\u200d💋\u200d👨🏾', name='kiss: man, man, dark skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍❤️‍💋‍👨🏿': EmojiEntry(codepoint='1F468 1F3FF 200D 2764 FE0F 200D 1F48B 200D 1F468 1F3FF', status='fully-qualified', emoji='👨🏿\u200d❤️\u200d💋\u200d👨🏿', name='kiss: man, man, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍❤‍💋‍👨🏿': EmojiEntry(codepoint='1F468 1F3FF 200D 2764 200D 1F48B 200D 1F468 1F3FF', status='minimally-qualified', emoji='👨🏿\u200d❤\u200d💋\u200d👨🏿', name='kiss: man, man, dark skin tone', group='People & Body', sub_group='family'),
    '👩‍❤️‍💋‍👩': EmojiEntry(codepoint='1F469 200D 2764 FE0F 200D 1F48B 200D 1F469', status='fully-qualified', emoji='👩\u200d❤️\u200d💋\u200d👩', name='kiss: woman, woman', group='People & Body', sub_group='family'),
    '👩‍❤‍💋‍👩': EmojiEntry(codepoint='1F469 200D 2764 200D 1F48B 200D 1F469', status='minimally-qualified', emoji='👩\u200d❤\u200d💋\u200d👩', name='kiss: woman, woman', group='People & Body', sub_group='family'),
    '👩🏻‍❤️‍💋‍👩🏻': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FB', status='fully-qualified', emoji='👩🏻\u200d❤️\u200d💋\u200d👩🏻', name='kiss: woman, woman, light skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤‍💋‍👩🏻': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 200D 1F48B 200D 1F469 1F3FB', status='minimally-qualified', emoji='👩🏻\u200d❤\u200d💋\u200d👩🏻', name='kiss: woman, woman, light skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤️‍💋‍👩🏼': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FC', status='fully-qualified', emoji='👩🏻\u200d❤️\u200d💋\u200d👩🏼', name='kiss: woman, woman, light skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤‍💋‍👩🏼': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 200D 1F48B 200D 1F469 1F3FC', status='minimally-qualified', emoji='👩🏻\u200d❤\u200d💋\u200d👩🏼', name='kiss: woman, woman, light skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤️‍💋‍👩🏽': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FD', status='fully-qualified', emoji='👩🏻\u200d❤️\u200d💋\u200d👩🏽', name='kiss: woman, woman, light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤‍💋‍👩🏽': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 200D 1F48B 200D 1F469 1F3FD', status='minimally-qualified', emoji='👩🏻\u200d❤\u200d💋\u200d👩🏽', name='kiss: woman, woman, light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤️‍💋‍👩🏾': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FE', status='fully-qualified', emoji='👩🏻\u200d❤️\u200d💋\u200d👩🏾', name='kiss: woman, woman, light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤‍💋‍👩🏾': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 200D 1F48B 200D 1F469 1F3FE', status='minimally-qualified', emoji='👩🏻\u200d❤\u200d💋\u200d👩🏾', name='kiss: woman, woman, light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤️‍💋‍👩🏿': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FF', status='fully-qualified', emoji='👩🏻\u200d❤️\u200d💋\u200d👩🏿', name='kiss: woman, woman, light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤‍💋‍👩🏿': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 200D 1F48B 200D 1F469 1F3FF', status='minimally-qualified', emoji='👩🏻\u200d❤\u200d💋\u200d👩🏿', name='kiss: woman, woman, light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤️‍💋‍👩🏻': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FB', status='fully-qualified', emoji='👩🏼\u200d❤️\u200d💋\u200d👩🏻', name='kiss: woman, woman, medium-light skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤‍💋‍👩🏻': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 200D 1F48B 200D 1F469 1F3FB', status='minimally-qualified', emoji='👩🏼\u200d❤\u200d💋\u200d👩🏻', name='kiss: woman, woman, medium-light skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤️‍💋‍👩🏼': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FC', status='fully-qualified', emoji='👩🏼\u200d❤️\u200d💋\u200d👩🏼', name='kiss: woman, woman, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤‍💋‍👩🏼': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 200D 1F48B 200D 1F469 1F3FC', status='minimally-qualified', emoji='👩🏼\u200d❤\u200d💋\u200d👩🏼', name='kiss: woman, woman, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤️‍💋‍👩🏽': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FD', status='fully-qualified', emoji='👩🏼\u200d❤️\u200d💋\u200d👩🏽', name='kiss: woman, woman, medium-light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤‍💋‍👩🏽': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 200D 1F48B 200D 1F469 1F3FD', status='minimally-qualified', emoji='👩🏼\u200d❤\u200d💋\u200d👩🏽', name='kiss: woman, woman, medium-light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤️‍💋‍👩🏾': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FE', status='fully-qualified', emoji='👩🏼\u200d❤️\u200d💋\u200d👩🏾', name='kiss: woman, woman, medium-light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤‍💋‍👩🏾': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 200D 1F48B 200D 1F469 1F3FE', status='minimally-qualified', emoji='👩🏼\u200d❤\u200d💋\u200d👩🏾', name='kiss: woman, woman, medium-light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤️‍💋‍👩🏿': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FF', status='fully-qualified', emoji='👩🏼\u200d❤️\u200d💋\u200d👩🏿', name='kiss: woman, woman, medium-light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤‍💋‍👩🏿': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 200D 1F48B 200D 1F469 1F3FF', status='minimally-qualified', emoji='👩🏼\u200d❤\u200d💋\u200d👩🏿', name='kiss: woman, woman, medium-light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤️‍💋‍👩🏻': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FB', status='fully-qualified', emoji='👩🏽\u200d❤️\u200d💋\u200d👩🏻', name='kiss: woman, woman, medium skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤‍💋‍👩🏻': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 200D 1F48B 200D 1F469 1F3FB', status='minimally-qualified', emoji='👩🏽\u200d❤\u200d💋\u200d👩🏻', name='kiss: woman, woman, medium skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤️‍💋‍👩🏼': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FC', status='fully-qualified', emoji='👩🏽\u200d❤️\u200d💋\u200d👩🏼', name='kiss: woman, woman, medium skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤‍💋‍👩🏼': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 200D 1F48B 200D 1F469 1F3FC', status='minimally-qualified', emoji='👩🏽\u200d❤\u200d💋\u200d👩🏼', name='kiss: woman, woman, medium skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤️‍💋‍👩🏽': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FD', status='fully-qualified', emoji='👩🏽\u200d❤️\u200d💋\u200d👩🏽', name='kiss: woman, woman, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤‍💋‍👩🏽': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 200D 1F48B 200D 1F469 1F3FD', status='minimally-qualified', emoji='👩🏽\u200d❤\u200d💋\u200d👩🏽', name='kiss: woman, woman, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤️‍💋‍👩🏾': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FE', status='fully-qualified', emoji='👩🏽\u200d❤️\u200d💋\u200d👩🏾', name='kiss: woman, woman, medium skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤‍💋‍👩🏾': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 200D 1F48B 200D 1F469 1F3FE', status='minimally-qualified', emoji='👩🏽\u200d❤\u200d💋\u200d👩🏾', name='kiss: woman, woman, medium skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤️‍💋‍👩🏿': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FF', status='fully-qualified', emoji='👩🏽\u200d❤️\u200d💋\u200d👩🏿', name='kiss: woman, woman, medium skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤‍💋‍👩🏿': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 200D 1F48B 200D 1F469 1F3FF', status='minimally-qualified', emoji='👩🏽\u200d❤\u200d💋\u200d👩🏿', name='kiss: woman, woman, medium skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤️‍💋‍👩🏻': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FB', status='fully-qualified', emoji='👩🏾\u200d❤️\u200d💋\u200d👩🏻', name='kiss: woman, woman, medium-dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤‍💋‍👩🏻': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 200D 1F48B 200D 1F469 1F3FB', status='minimally-qualified', emoji='👩🏾\u200d❤\u200d💋\u200d👩🏻', name='kiss: woman, woman, medium-dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤️‍💋‍👩🏼': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FC', status='fully-qualified', emoji='👩🏾\u200d❤️\u200d💋\u200d👩🏼', name='kiss: woman, woman, medium-dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤‍💋‍👩🏼': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 200D 1F48B 200D 1F469 1F3FC', status='minimally-qualified', emoji='👩🏾\u200d❤\u200d💋\u200d👩🏼', name='kiss: woman, woman, medium-dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤️‍💋‍👩🏽': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FD', status='fully-qualified', emoji='👩🏾\u200d❤️\u200d💋\u200d👩🏽', name='kiss: woman, woman, medium-dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤‍💋‍👩🏽': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 200D 1F48B 200D 1F469 1F3FD', status='minimally-qualified', emoji='👩🏾\u200d❤\u200d💋\u200d👩🏽', name='kiss: woman, woman, medium-dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤️‍💋‍👩🏾': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FE', status='fully-qualified', emoji='👩🏾\u200d❤️\u200d💋\u200d👩🏾', name='kiss: woman, woman, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤‍💋‍👩🏾': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 200D 1F48B 200D 1F469 1F3FE', status='minimally-qualified', emoji='👩🏾\u200d❤\u200d💋\u200d👩🏾', name='kiss: woman, woman, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤️‍💋‍👩🏿': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FF', status='fully-qualified', emoji='👩🏾\u200d❤️\u200d💋\u200d👩🏿', name='kiss: woman, woman, medium-dark skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤‍💋‍👩🏿': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 200D 1F48B 200D 1F469 1F3FF', status='minimally-qualified', emoji='👩🏾\u200d❤\u200d💋\u200d👩🏿', name='kiss: woman, woman, medium-dark skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤️‍💋‍👩🏻': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FB', status='fully-qualified', emoji='👩🏿\u200d❤️\u200d💋\u200d👩🏻', name='kiss: woman, woman, dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤‍💋‍👩🏻': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 200D 1F48B 200D 1F469 1F3FB', status='minimally-qualified', emoji='👩🏿\u200d❤\u200d💋\u200d👩🏻', name='kiss: woman, woman, dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤️‍💋‍👩🏼': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FC', status='fully-qualified', emoji='👩🏿\u200d❤️\u200d💋\u200d👩🏼', name='kiss: woman, woman, dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤‍💋‍👩🏼': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 200D 1F48B 200D 1F469 1F3FC', status='minimally-qualified', emoji='👩🏿\u200d❤\u200d💋\u200d👩🏼', name='kiss: woman, woman, dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤️‍💋‍👩🏽': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FD', status='fully-qualified', emoji='👩🏿\u200d❤️\u200d💋\u200d👩🏽', name='kiss: woman, woman, dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤‍💋‍👩🏽': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 200D 1F48B 200D 1F469 1F3FD', status='minimally-qualified', emoji='👩🏿\u200d❤\u200d💋\u200d👩🏽', name='kiss: woman, woman, dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤️‍💋‍👩🏾': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FE', status='fully-qualified', emoji='👩🏿\u200d❤️\u200d💋\u200d👩🏾', name='kiss: woman, woman, dark skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤‍💋‍👩🏾': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 200D 1F48B 200D 1F469 1F3FE', status='minimally-qualified', emoji='👩🏿\u200d❤\u200d💋\u200d👩🏾', name='kiss: woman, woman, dark skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤️‍💋‍👩🏿': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 FE0F 200D 1F48B 200D 1F469 1F3FF', status='fully-qualified', emoji='👩🏿\u200d❤️\u200d💋\u200d👩🏿', name='kiss: woman, woman, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤‍💋‍👩🏿': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 200D 1F48B 200D 1F469 1F3FF', status='minimally-qualified', emoji='👩🏿\u200d❤\u200d💋\u200d👩🏿', name='kiss: woman, woman, dark skin tone', group='People & Body', sub_group='family'),
    '💑': EmojiEntry(codepoint='1F491', status='fully-qualified', emoji='💑', name='couple with heart', group='People & Body', sub_group='family'),
    '💑🏻': EmojiEntry(codepoint='1F491 1F3FB', status='fully-qualified', emoji='💑🏻', name='couple with heart: light skin tone', group='People & Body', sub_group='family'),
    '💑🏼': EmojiEntry(codepoint='1F491 1F3FC', status='fully-qualified', emoji='💑🏼', name='couple with heart: medium-light skin tone', group='People & Body', sub_group='family'),
    '💑🏽': EmojiEntry(codepoint='1F491 1F3FD', status='fully-qualified', emoji='💑🏽', name='couple with heart: medium skin tone', group='People & Body', sub_group='family'),
    '💑🏾': EmojiEntry(codepoint='1F491 1F3FE', status='fully-qualified', emoji='💑🏾', name='couple with heart: medium-dark skin tone', group='People & Body', sub_group='family'),
    '💑🏿': EmojiEntry(codepoint='1F491 1F3FF', status='fully-qualified', emoji='💑🏿', name='couple with heart: dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏻‍❤️‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2764 FE0F 200D 1F9D1 1F3FC', status='fully-qualified', emoji='🧑🏻\u200d❤️\u200d🧑🏼', name='couple with heart: person, person, light skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏻‍❤‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2764 200D 1F9D1 1F3FC', status='minimally-qualified', emoji='🧑🏻\u200d❤\u200d🧑🏼', name='couple with heart: person, person, light skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏻‍❤️‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2764 FE0F 200D 1F9D1 1F3FD', status='fully-qualified', emoji='🧑🏻\u200d❤️\u200d🧑🏽', name='couple with heart: person, person, light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏻‍❤‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2764 200D 1F9D1 1F3FD', status='minimally-qualified', emoji='🧑🏻\u200d❤\u200d🧑🏽', name='couple with heart: person, person, light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏻‍❤️‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2764 FE0F 200D 1F9D1 1F3FE', status='fully-qualified', emoji='🧑🏻\u200d❤️\u200d🧑🏾', name='couple with heart: person, person, light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏻‍❤‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2764 200D 1F9D1 1F3FE', status='minimally-qualified', emoji='🧑🏻\u200d❤\u200d🧑🏾', name='couple with heart: person, person, light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏻‍❤️‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2764 FE0F 200D 1F9D1 1F3FF', status='fully-qualified', emoji='🧑🏻\u200d❤️\u200d🧑🏿', name='couple with heart: person, person, light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏻‍❤‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FB 200D 2764 200D 1F9D1 1F3FF', status='minimally-qualified', emoji='🧑🏻\u200d❤\u200d🧑🏿', name='couple with heart: person, person, light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍❤️‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2764 FE0F 200D 1F9D1 1F3FB', status='fully-qualified', emoji='🧑🏼\u200d❤️\u200d🧑🏻', name='couple with heart: person, person, medium-light skin tone, light skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍❤‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2764 200D 1F9D1 1F3FB', status='minimally-qualified', emoji='🧑🏼\u200d❤\u200d🧑🏻', name='couple with heart: person, person, medium-light skin tone, light skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍❤️‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2764 FE0F 200D 1F9D1 1F3FD', status='fully-qualified', emoji='🧑🏼\u200d❤️\u200d🧑🏽', name='couple with heart: person, person, medium-light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍❤‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2764 200D 1F9D1 1F3FD', status='minimally-qualified', emoji='🧑🏼\u200d❤\u200d🧑🏽', name='couple with heart: person, person, medium-light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍❤️‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2764 FE0F 200D 1F9D1 1F3FE', status='fully-qualified', emoji='🧑🏼\u200d❤️\u200d🧑🏾', name='couple with heart: person, person, medium-light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍❤‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2764 200D 1F9D1 1F3FE', status='minimally-qualified', emoji='🧑🏼\u200d❤\u200d🧑🏾', name='couple with heart: person, person, medium-light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍❤️‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2764 FE0F 200D 1F9D1 1F3FF', status='fully-qualified', emoji='🧑🏼\u200d❤️\u200d🧑🏿', name='couple with heart: person, person, medium-light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏼‍❤‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FC 200D 2764 200D 1F9D1 1F3FF', status='minimally-qualified', emoji='🧑🏼\u200d❤\u200d🧑🏿', name='couple with heart: person, person, medium-light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍❤️‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2764 FE0F 200D 1F9D1 1F3FB', status='fully-qualified', emoji='🧑🏽\u200d❤️\u200d🧑🏻', name='couple with heart: person, person, medium skin tone, light skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍❤‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2764 200D 1F9D1 1F3FB', status='minimally-qualified', emoji='🧑🏽\u200d❤\u200d🧑🏻', name='couple with heart: person, person, medium skin tone, light skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍❤️‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2764 FE0F 200D 1F9D1 1F3FC', status='fully-qualified', emoji='🧑🏽\u200d❤️\u200d🧑🏼', name='couple with heart: person, person, medium skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍❤‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2764 200D 1F9D1 1F3FC', status='minimally-qualified', emoji='🧑🏽\u200d❤\u200d🧑🏼', name='couple with heart: person, person, medium skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍❤️‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2764 FE0F 200D 1F9D1 1F3FE', status='fully-qualified', emoji='🧑🏽\u200d❤️\u200d🧑🏾', name='couple with heart: person, person, medium skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍❤‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2764 200D 1F9D1 1F3FE', status='minimally-qualified', emoji='🧑🏽\u200d❤\u200d🧑🏾', name='couple with heart: person, person, medium skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍❤️‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2764 FE0F 200D 1F9D1 1F3FF', status='fully-qualified', emoji='🧑🏽\u200d❤️\u200d🧑🏿', name='couple with heart: person, person, medium skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏽‍❤‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FD 200D 2764 200D 1F9D1 1F3FF', status='minimally-qualified', emoji='🧑🏽\u200d❤\u200d🧑🏿', name='couple with heart: person, person, medium skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍❤️‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2764 FE0F 200D 1F9D1 1F3FB', status='fully-qualified', emoji='🧑🏾\u200d❤️\u200d🧑🏻', name='couple with heart: person, person, medium-dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍❤‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2764 200D 1F9D1 1F3FB', status='minimally-qualified', emoji='🧑🏾\u200d❤\u200d🧑🏻', name='couple with heart: person, person, medium-dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍❤️‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2764 FE0F 200D 1F9D1 1F3FC', status='fully-qualified', emoji='🧑🏾\u200d❤️\u200d🧑🏼', name='couple with heart: person, person, medium-dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍❤‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2764 200D 1F9D1 1F3FC', status='minimally-qualified', emoji='🧑🏾\u200d❤\u200d🧑🏼', name='couple with heart: person, person, medium-dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍❤️‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2764 FE0F 200D 1F9D1 1F3FD', status='fully-qualified', emoji='🧑🏾\u200d❤️\u200d🧑🏽', name='couple with heart: person, person, medium-dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍❤‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2764 200D 1F9D1 1F3FD', status='minimally-qualified', emoji='🧑🏾\u200d❤\u200d🧑🏽', name='couple with heart: person, person, medium-dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍❤️‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2764 FE0F 200D 1F9D1 1F3FF', status='fully-qualified', emoji='🧑🏾\u200d❤️\u200d🧑🏿', name='couple with heart: person, person, medium-dark skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏾‍❤‍🧑🏿': EmojiEntry(codepoint='1F9D1 1F3FE 200D 2764 200D 1F9D1 1F3FF', status='minimally-qualified', emoji='🧑🏾\u200d❤\u200d🧑🏿', name='couple with heart: person, person, medium-dark skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍❤️‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2764 FE0F 200D 1F9D1 1F3FB', status='fully-qualified', emoji='🧑🏿\u200d❤️\u200d🧑🏻', name='couple with heart: person, person, dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍❤‍🧑🏻': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2764 200D 1F9D1 1F3FB', status='minimally-qualified', emoji='🧑🏿\u200d❤\u200d🧑🏻', name='couple with heart: person, person, dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍❤️‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2764 FE0F 200D 1F9D1 1F3FC', status='fully-qualified', emoji='🧑🏿\u200d❤️\u200d🧑🏼', name='couple with heart: person, person, dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍❤‍🧑🏼': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2764 200D 1F9D1 1F3FC', status='minimally-qualified', emoji='🧑🏿\u200d❤\u200d🧑🏼', name='couple with heart: person, person, dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍❤️‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2764 FE0F 200D 1F9D1 1F3FD', status='fully-qualified', emoji='🧑🏿\u200d❤️\u200d🧑🏽', name='couple with heart: person, person, dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍❤‍🧑🏽': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2764 200D 1F9D1 1F3FD', status='minimally-qualified', emoji='🧑🏿\u200d❤\u200d🧑🏽', name='couple with heart: person, person, dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍❤️‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2764 FE0F 200D 1F9D1 1F3FE', status='fully-qualified', emoji='🧑🏿\u200d❤️\u200d🧑🏾', name='couple with heart: person, person, dark skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '🧑🏿‍❤‍🧑🏾': EmojiEntry(codepoint='1F9D1 1F3FF 200D 2764 200D 1F9D1 1F3FE', status='minimally-qualified', emoji='🧑🏿\u200d❤\u200d🧑🏾', name='couple with heart: person, person, dark skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩‍❤️‍👨': EmojiEntry(codepoint='1F469 200D 2764 FE0F 200D 1F468', status='fully-qualified', emoji='👩\u200d❤️\u200d👨', name='couple with heart: woman, man', group='People & Body', sub_group='family'),
    '👩‍❤‍👨': EmojiEntry(codepoint='1F469 200D 2764 200D 1F468', status='minimally-qualified', emoji='👩\u200d❤\u200d👨', name='couple with heart: woman, man', group='People & Body', sub_group='family'),
    '👩🏻‍❤️‍👨🏻': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 FE0F 200D 1F468 1F3FB', status='fully-qualified', emoji='👩🏻\u200d❤️\u200d👨🏻', name='couple with heart: woman, man, light skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤‍👨🏻': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 200D 1F468 1F3FB', status='minimally-qualified', emoji='👩🏻\u200d❤\u200d👨🏻', name='couple with heart: woman, man, light skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤️‍👨🏼': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 FE0F 200D 1F468 1F3FC', status='fully-qualified', emoji='👩🏻\u200d❤️\u200d👨🏼', name='couple with heart: woman, man, light skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤‍👨🏼': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 200D 1F468 1F3FC', status='minimally-qualified', emoji='👩🏻\u200d❤\u200d👨🏼', name='couple with heart: woman, man, light skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤️‍👨🏽': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 FE0F 200D 1F468 1F3FD', status='fully-qualified', emoji='👩🏻\u200d❤️\u200d👨🏽', name='couple with heart: woman, man, light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤‍👨🏽': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 200D 1F468 1F3FD', status='minimally-qualified', emoji='👩🏻\u200d❤\u200d👨🏽', name='couple with heart: woman, man, light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤️‍👨🏾': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 FE0F 200D 1F468 1F3FE', status='fully-qualified', emoji='👩🏻\u200d❤️\u200d👨🏾', name='couple with heart: woman, man, light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤‍👨🏾': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 200D 1F468 1F3FE', status='minimally-qualified', emoji='👩🏻\u200d❤\u200d👨🏾', name='couple with heart: woman, man, light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤️‍👨🏿': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 FE0F 200D 1F468 1F3FF', status='fully-qualified', emoji='👩🏻\u200d❤️\u200d👨🏿', name='couple with heart: woman, man, light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤‍👨🏿': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 200D 1F468 1F3FF', status='minimally-qualified', emoji='👩🏻\u200d❤\u200d👨🏿', name='couple with heart: woman, man, light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤️‍👨🏻': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 FE0F 200D 1F468 1F3FB', status='fully-qualified', emoji='👩🏼\u200d❤️\u200d👨🏻', name='couple with heart: woman, man, medium-light skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤‍👨🏻': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 200D 1F468 1F3FB', status='minimally-qualified', emoji='👩🏼\u200d❤\u200d👨🏻', name='couple with heart: woman, man, medium-light skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤️‍👨🏼': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 FE0F 200D 1F468 1F3FC', status='fully-qualified', emoji='👩🏼\u200d❤️\u200d👨🏼', name='couple with heart: woman, man, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤‍👨🏼': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 200D 1F468 1F3FC', status='minimally-qualified', emoji='👩🏼\u200d❤\u200d👨🏼', name='couple with heart: woman, man, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤️‍👨🏽': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 FE0F 200D 1F468 1F3FD', status='fully-qualified', emoji='👩🏼\u200d❤️\u200d👨🏽', name='couple with heart: woman, man, medium-light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤‍👨🏽': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 200D 1F468 1F3FD', status='minimally-qualified', emoji='👩🏼\u200d❤\u200d👨🏽', name='couple with heart: woman, man, medium-light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤️‍👨🏾': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 FE0F 200D 1F468 1F3FE', status='fully-qualified', emoji='👩🏼\u200d❤️\u200d👨🏾', name='couple with heart: woman, man, medium-light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤‍👨🏾': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 200D 1F468 1F3FE', status='minimally-qualified', emoji='👩🏼\u200d❤\u200d👨🏾', name='couple with heart: woman, man, medium-light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤️‍👨🏿': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 FE0F 200D 1F468 1F3FF', status='fully-qualified', emoji='👩🏼\u200d❤️\u200d👨🏿', name='couple with heart: woman, man, medium-light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤‍👨🏿': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 200D 1F468 1F3FF', status='minimally-qualified', emoji='👩🏼\u200d❤\u200d👨🏿', name='couple with heart: woman, man, medium-light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤️‍👨🏻': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 FE0F 200D 1F468 1F3FB', status='fully-qualified', emoji='👩🏽\u200d❤️\u200d👨🏻', name='couple with heart: woman, man, medium skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤‍👨🏻': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 200D 1F468 1F3FB', status='minimally-qualified', emoji='👩🏽\u200d❤\u200d👨🏻', name='couple with heart: woman, man, medium skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤️‍👨🏼': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 FE0F 200D 1F468 1F3FC', status='fully-qualified', emoji='👩🏽\u200d❤️\u200d👨🏼', name='couple with heart: woman, man, medium skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤‍👨🏼': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 200D 1F468 1F3FC', status='minimally-qualified', emoji='👩🏽\u200d❤\u200d👨🏼', name='couple with heart: woman, man, medium skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤️‍👨🏽': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 FE0F 200D 1F468 1F3FD', status='fully-qualified', emoji='👩🏽\u200d❤️\u200d👨🏽', name='couple with heart: woman, man, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤‍👨🏽': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 200D 1F468 1F3FD', status='minimally-qualified', emoji='👩🏽\u200d❤\u200d👨🏽', name='couple with heart: woman, man, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤️‍👨🏾': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 FE0F 200D 1F468 1F3FE', status='fully-qualified', emoji='👩🏽\u200d❤️\u200d👨🏾', name='couple with heart: woman, man, medium skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤‍👨🏾': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 200D 1F468 1F3FE', status='minimally-qualified', emoji='👩🏽\u200d❤\u200d👨🏾', name='couple with heart: woman, man, medium skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤️‍👨🏿': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 FE0F 200D 1F468 1F3FF', status='fully-qualified', emoji='👩🏽\u200d❤️\u200d👨🏿', name='couple with heart: woman, man, medium skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤‍👨🏿': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 200D 1F468 1F3FF', status='minimally-qualified', emoji='👩🏽\u200d❤\u200d👨🏿', name='couple with heart: woman, man, medium skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤️‍👨🏻': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 FE0F 200D 1F468 1F3FB', status='fully-qualified', emoji='👩🏾\u200d❤️\u200d👨🏻', name='couple with heart: woman, man, medium-dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤‍👨🏻': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 200D 1F468 1F3FB', status='minimally-qualified', emoji='👩🏾\u200d❤\u200d👨🏻', name='couple with heart: woman, man, medium-dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤️‍👨🏼': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 FE0F 200D 1F468 1F3FC', status='fully-qualified', emoji='👩🏾\u200d❤️\u200d👨🏼', name='couple with heart: woman, man, medium-dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤‍👨🏼': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 200D 1F468 1F3FC', status='minimally-qualified', emoji='👩🏾\u200d❤\u200d👨🏼', name='couple with heart: woman, man, medium-dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤️‍👨🏽': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 FE0F 200D 1F468 1F3FD', status='fully-qualified', emoji='👩🏾\u200d❤️\u200d👨🏽', name='couple with heart: woman, man, medium-dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤‍👨🏽': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 200D 1F468 1F3FD', status='minimally-qualified', emoji='👩🏾\u200d❤\u200d👨🏽', name='couple with heart: woman, man, medium-dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤️‍👨🏾': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 FE0F 200D 1F468 1F3FE', status='fully-qualified', emoji='👩🏾\u200d❤️\u200d👨🏾', name='couple with heart: woman, man, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤‍👨🏾': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 200D 1F468 1F3FE', status='minimally-qualified', emoji='👩🏾\u200d❤\u200d👨🏾', name='couple with heart: woman, man, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤️‍👨🏿': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 FE0F 200D 1F468 1F3FF', status='fully-qualified', emoji='👩🏾\u200d❤️\u200d👨🏿', name='couple with heart: woman, man, medium-dark skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤‍👨🏿': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 200D 1F468 1F3FF', status='minimally-qualified', emoji='👩🏾\u200d❤\u200d👨🏿', name='couple with heart: woman, man, medium-dark skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤️‍👨🏻': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 FE0F 200D 1F468 1F3FB', status='fully-qualified', emoji='👩🏿\u200d❤️\u200d👨🏻', name='couple with heart: woman, man, dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤‍👨🏻': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 200D 1F468 1F3FB', status='minimally-qualified', emoji='👩🏿\u200d❤\u200d👨🏻', name='couple with heart: woman, man, dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤️‍👨🏼': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 FE0F 200D 1F468 1F3FC', status='fully-qualified', emoji='👩🏿\u200d❤️\u200d👨🏼', name='couple with heart: woman, man, dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤‍👨🏼': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 200D 1F468 1F3FC', status='minimally-qualified', emoji='👩🏿\u200d❤\u200d👨🏼', name='couple with heart: woman, man, dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤️‍👨🏽': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 FE0F 200D 1F468 1F3FD', status='fully-qualified', emoji='👩🏿\u200d❤️\u200d👨🏽', name='couple with heart: woman, man, dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤‍👨🏽': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 200D 1F468 1F3FD', status='minimally-qualified', emoji='👩🏿\u200d❤\u200d👨🏽', name='couple with heart: woman, man, dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤️‍👨🏾': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 FE0F 200D 1F468 1F3FE', status='fully-qualified', emoji='👩🏿\u200d❤️\u200d👨🏾', name='couple with heart: woman, man, dark skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤‍👨🏾': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 200D 1F468 1F3FE', status='minimally-qualified', emoji='👩🏿\u200d❤\u200d👨🏾', name='couple with heart: woman, man, dark skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤️‍👨🏿': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 FE0F 200D 1F468 1F3FF', status='fully-qualified', emoji='👩🏿\u200d❤️\u200d👨🏿', name='couple with heart: woman, man, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤‍👨🏿': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 200D 1F468 1F3FF', status='minimally-qualified', emoji='👩🏿\u200d❤\u200d👨🏿', name='couple with heart: woman, man, dark skin tone', group='People & Body', sub_group='family'),
    '👨‍❤️‍👨': EmojiEntry(codepoint='1F468 200D 2764 FE0F 200D 1F468', status='fully-qualified', emoji='👨\u200d❤️\u200d👨', name='couple with heart: man, man', group='People & Body', sub_group='family'),
    '👨‍❤‍👨': EmojiEntry(codepoint='1F468 200D 2764 200D 1F468', status='minimally-qualified', emoji='👨\u200d❤\u200d👨', name='couple with heart: man, man', group='People & Body', sub_group='family'),
    '👨🏻‍❤️‍👨🏻': EmojiEntry(codepoint='1F468 1F3FB 200D 2764 FE0F 200D 1F468 1F3FB', status='fully-qualified', emoji='👨🏻\u200d❤️\u200d👨🏻', name='couple with heart: man, man, light skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍❤‍👨🏻': EmojiEntry(codepoint='1F468 1F3FB 200D 2764 200D 1F468 1F3FB', status='minimally-qualified', emoji='👨🏻\u200d❤\u200d👨🏻', name='couple with heart: man, man, light skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍❤️‍👨🏼': EmojiEntry(codepoint='1F468 1F3FB 200D 2764 FE0F 200D 1F468 1F3FC', status='fully-qualified', emoji='👨🏻\u200d❤️\u200d👨🏼', name='couple with heart: man, man, light skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍❤‍👨🏼': EmojiEntry(codepoint='1F468 1F3FB 200D 2764 200D 1F468 1F3FC', status='minimally-qualified', emoji='👨🏻\u200d❤\u200d👨🏼', name='couple with heart: man, man, light skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍❤️‍👨🏽': EmojiEntry(codepoint='1F468 1F3FB 200D 2764 FE0F 200D 1F468 1F3FD', status='fully-qualified', emoji='👨🏻\u200d❤️\u200d👨🏽', name='couple with heart: man, man, light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍❤‍👨🏽': EmojiEntry(codepoint='1F468 1F3FB 200D 2764 200D 1F468 1F3FD', status='minimally-qualified', emoji='👨🏻\u200d❤\u200d👨🏽', name='couple with heart: man, man, light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍❤️‍👨🏾': EmojiEntry(codepoint='1F468 1F3FB 200D 2764 FE0F 200D 1F468 1F3FE', status='fully-qualified', emoji='👨🏻\u200d❤️\u200d👨🏾', name='couple with heart: man, man, light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍❤‍👨🏾': EmojiEntry(codepoint='1F468 1F3FB 200D 2764 200D 1F468 1F3FE', status='minimally-qualified', emoji='👨🏻\u200d❤\u200d👨🏾', name='couple with heart: man, man, light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍❤️‍👨🏿': EmojiEntry(codepoint='1F468 1F3FB 200D 2764 FE0F 200D 1F468 1F3FF', status='fully-qualified', emoji='👨🏻\u200d❤️\u200d👨🏿', name='couple with heart: man, man, light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏻‍❤‍👨🏿': EmojiEntry(codepoint='1F468 1F3FB 200D 2764 200D 1F468 1F3FF', status='minimally-qualified', emoji='👨🏻\u200d❤\u200d👨🏿', name='couple with heart: man, man, light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍❤️‍👨🏻': EmojiEntry(codepoint='1F468 1F3FC 200D 2764 FE0F 200D 1F468 1F3FB', status='fully-qualified', emoji='👨🏼\u200d❤️\u200d👨🏻', name='couple with heart: man, man, medium-light skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍❤‍👨🏻': EmojiEntry(codepoint='1F468 1F3FC 200D 2764 200D 1F468 1F3FB', status='minimally-qualified', emoji='👨🏼\u200d❤\u200d👨🏻', name='couple with heart: man, man, medium-light skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍❤️‍👨🏼': EmojiEntry(codepoint='1F468 1F3FC 200D 2764 FE0F 200D 1F468 1F3FC', status='fully-qualified', emoji='👨🏼\u200d❤️\u200d👨🏼', name='couple with heart: man, man, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍❤‍👨🏼': EmojiEntry(codepoint='1F468 1F3FC 200D 2764 200D 1F468 1F3FC', status='minimally-qualified', emoji='👨🏼\u200d❤\u200d👨🏼', name='couple with heart: man, man, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍❤️‍👨🏽': EmojiEntry(codepoint='1F468 1F3FC 200D 2764 FE0F 200D 1F468 1F3FD', status='fully-qualified', emoji='👨🏼\u200d❤️\u200d👨🏽', name='couple with heart: man, man, medium-light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍❤‍👨🏽': EmojiEntry(codepoint='1F468 1F3FC 200D 2764 200D 1F468 1F3FD', status='minimally-qualified', emoji='👨🏼\u200d❤\u200d👨🏽', name='couple with heart: man, man, medium-light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍❤️‍👨🏾': EmojiEntry(codepoint='1F468 1F3FC 200D 2764 FE0F 200D 1F468 1F3FE', status='fully-qualified', emoji='👨🏼\u200d❤️\u200d👨🏾', name='couple with heart: man, man, medium-light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍❤‍👨🏾': EmojiEntry(codepoint='1F468 1F3FC 200D 2764 200D 1F468 1F3FE', status='minimally-qualified', emoji='👨🏼\u200d❤\u200d👨🏾', name='couple with heart: man, man, medium-light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍❤️‍👨🏿': EmojiEntry(codepoint='1F468 1F3FC 200D 2764 FE0F 200D 1F468 1F3FF', status='fully-qualified', emoji='👨🏼\u200d❤️\u200d👨🏿', name='couple with heart: man, man, medium-light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏼‍❤‍👨🏿': EmojiEntry(codepoint='1F468 1F3FC 200D 2764 200D 1F468 1F3FF', status='minimally-qualified', emoji='👨🏼\u200d❤\u200d👨🏿', name='couple with heart: man, man, medium-light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍❤️‍👨🏻': EmojiEntry(codepoint='1F468 1F3FD 200D 2764 FE0F 200D 1F468 1F3FB', status='fully-qualified', emoji='👨🏽\u200d❤️\u200d👨🏻', name='couple with heart: man, man, medium skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍❤‍👨🏻': EmojiEntry(codepoint='1F468 1F3FD 200D 2764 200D 1F468 1F3FB', status='minimally-qualified', emoji='👨🏽\u200d❤\u200d👨🏻', name='couple with heart: man, man, medium skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍❤️‍👨🏼': EmojiEntry(codepoint='1F468 1F3FD 200D 2764 FE0F 200D 1F468 1F3FC', status='fully-qualified', emoji='👨🏽\u200d❤️\u200d👨🏼', name='couple with heart: man, man, medium skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍❤‍👨🏼': EmojiEntry(codepoint='1F468 1F3FD 200D 2764 200D 1F468 1F3FC', status='minimally-qualified', emoji='👨🏽\u200d❤\u200d👨🏼', name='couple with heart: man, man, medium skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍❤️‍👨🏽': EmojiEntry(codepoint='1F468 1F3FD 200D 2764 FE0F 200D 1F468 1F3FD', status='fully-qualified', emoji='👨🏽\u200d❤️\u200d👨🏽', name='couple with heart: man, man, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍❤‍👨🏽': EmojiEntry(codepoint='1F468 1F3FD 200D 2764 200D 1F468 1F3FD', status='minimally-qualified', emoji='👨🏽\u200d❤\u200d👨🏽', name='couple with heart: man, man, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍❤️‍👨🏾': EmojiEntry(codepoint='1F468 1F3FD 200D 2764 FE0F 200D 1F468 1F3FE', status='fully-qualified', emoji='👨🏽\u200d❤️\u200d👨🏾', name='couple with heart: man, man, medium skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍❤‍👨🏾': EmojiEntry(codepoint='1F468 1F3FD 200D 2764 200D 1F468 1F3FE', status='minimally-qualified', emoji='👨🏽\u200d❤\u200d👨🏾', name='couple with heart: man, man, medium skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍❤️‍👨🏿': EmojiEntry(codepoint='1F468 1F3FD 200D 2764 FE0F 200D 1F468 1F3FF', status='fully-qualified', emoji='👨🏽\u200d❤️\u200d👨🏿', name='couple with heart: man, man, medium skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏽‍❤‍👨🏿': EmojiEntry(codepoint='1F468 1F3FD 200D 2764 200D 1F468 1F3FF', status='minimally-qualified', emoji='👨🏽\u200d❤\u200d👨🏿', name='couple with heart: man, man, medium skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍❤️‍👨🏻': EmojiEntry(codepoint='1F468 1F3FE 200D 2764 FE0F 200D 1F468 1F3FB', status='fully-qualified', emoji='👨🏾\u200d❤️\u200d👨🏻', name='couple with heart: man, man, medium-dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍❤‍👨🏻': EmojiEntry(codepoint='1F468 1F3FE 200D 2764 200D 1F468 1F3FB', status='minimally-qualified', emoji='👨🏾\u200d❤\u200d👨🏻', name='couple with heart: man, man, medium-dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍❤️‍👨🏼': EmojiEntry(codepoint='1F468 1F3FE 200D 2764 FE0F 200D 1F468 1F3FC', status='fully-qualified', emoji='👨🏾\u200d❤️\u200d👨🏼', name='couple with heart: man, man, medium-dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍❤‍👨🏼': EmojiEntry(codepoint='1F468 1F3FE 200D 2764 200D 1F468 1F3FC', status='minimally-qualified', emoji='👨🏾\u200d❤\u200d👨🏼', name='couple with heart: man, man, medium-dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍❤️‍👨🏽': EmojiEntry(codepoint='1F468 1F3FE 200D 2764 FE0F 200D 1F468 1F3FD', status='fully-qualified', emoji='👨🏾\u200d❤️\u200d👨🏽', name='couple with heart: man, man, medium-dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍❤‍👨🏽': EmojiEntry(codepoint='1F468 1F3FE 200D 2764 200D 1F468 1F3FD', status='minimally-qualified', emoji='👨🏾\u200d❤\u200d👨🏽', name='couple with heart: man, man, medium-dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍❤️‍👨🏾': EmojiEntry(codepoint='1F468 1F3FE 200D 2764 FE0F 200D 1F468 1F3FE', status='fully-qualified', emoji='👨🏾\u200d❤️\u200d👨🏾', name='couple with heart: man, man, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍❤‍👨🏾': EmojiEntry(codepoint='1F468 1F3FE 200D 2764 200D 1F468 1F3FE', status='minimally-qualified', emoji='👨🏾\u200d❤\u200d👨🏾', name='couple with heart: man, man, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍❤️‍👨🏿': EmojiEntry(codepoint='1F468 1F3FE 200D 2764 FE0F 200D 1F468 1F3FF', status='fully-qualified', emoji='👨🏾\u200d❤️\u200d👨🏿', name='couple with heart: man, man, medium-dark skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏾‍❤‍👨🏿': EmojiEntry(codepoint='1F468 1F3FE 200D 2764 200D 1F468 1F3FF', status='minimally-qualified', emoji='👨🏾\u200d❤\u200d👨🏿', name='couple with heart: man, man, medium-dark skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍❤️‍👨🏻': EmojiEntry(codepoint='1F468 1F3FF 200D 2764 FE0F 200D 1F468 1F3FB', status='fully-qualified', emoji='👨🏿\u200d❤️\u200d👨🏻', name='couple with heart: man, man, dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍❤‍👨🏻': EmojiEntry(codepoint='1F468 1F3FF 200D 2764 200D 1F468 1F3FB', status='minimally-qualified', emoji='👨🏿\u200d❤\u200d👨🏻', name='couple with heart: man, man, dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍❤️‍👨🏼': EmojiEntry(codepoint='1F468 1F3FF 200D 2764 FE0F 200D 1F468 1F3FC', status='fully-qualified', emoji='👨🏿\u200d❤️\u200d👨🏼', name='couple with heart: man, man, dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍❤‍👨🏼': EmojiEntry(codepoint='1F468 1F3FF 200D 2764 200D 1F468 1F3FC', status='minimally-qualified', emoji='👨🏿\u200d❤\u200d👨🏼', name='couple with heart: man, man, dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍❤️‍👨🏽': EmojiEntry(codepoint='1F468 1F3FF 200D 2764 FE0F 200D 1F468 1F3FD', status='fully-qualified', emoji='👨🏿\u200d❤️\u200d👨🏽', name='couple with heart: man, man, dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍❤‍👨🏽': EmojiEntry(codepoint='1F468 1F3FF 200D 2764 200D 1F468 1F3FD', status='minimally-qualified', emoji='👨🏿\u200d❤\u200d👨🏽', name='couple with heart: man, man, dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍❤️‍👨🏾': EmojiEntry(codepoint='1F468 1F3FF 200D 2764 FE0F 200D 1F468 1F3FE', status='fully-qualified', emoji='👨🏿\u200d❤️\u200d👨🏾', name='couple with heart: man, man, dark skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍❤‍👨🏾': EmojiEntry(codepoint='1F468 1F3FF 200D 2764 200D 1F468 1F3FE', status='minimally-qualified', emoji='👨🏿\u200d❤\u200d👨🏾', name='couple with heart: man, man, dark skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍❤️‍👨🏿': EmojiEntry(codepoint='1F468 1F3FF 200D 2764 FE0F 200D 1F468 1F3FF', status='fully-qualified', emoji='👨🏿\u200d❤️\u200d👨🏿', name='couple with heart: man, man, dark skin tone', group='People & Body', sub_group='family'),
    '👨🏿‍❤‍👨🏿': EmojiEntry(codepoint='1F468 1F3FF 200D 2764 200D 1F468 1F3FF', status='minimally-qualified', emoji='👨🏿\u200d❤\u200d👨🏿', name='couple with heart: man, man, dark skin tone', group='People & Body', sub_group='family'),
    '👩‍❤️‍👩': EmojiEntry(codepoint='1F469 200D 2764 FE0F 200D 1F469', status='fully-qualified', emoji='👩\u200d❤️\u200d👩', name='couple with heart: woman, woman', group='People & Body', sub_group='family'),
    '👩‍❤‍👩': EmojiEntry(codepoint='1F469 200D 2764 200D 1F469', status='minimally-qualified', emoji='👩\u200d❤\u200d👩', name='couple with heart: woman, woman', group='People & Body', sub_group='family'),
    '👩🏻‍❤️‍👩🏻': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 FE0F 200D 1F469 1F3FB', status='fully-qualified', emoji='👩🏻\u200d❤️\u200d👩🏻', name='couple with heart: woman, woman, light skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤‍👩🏻': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 200D 1F469 1F3FB', status='minimally-qualified', emoji='👩🏻\u200d❤\u200d👩🏻', name='couple with heart: woman, woman, light skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤️‍👩🏼': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 FE0F 200D 1F469 1F3FC', status='fully-qualified', emoji='👩🏻\u200d❤️\u200d👩🏼', name='couple with heart: woman, woman, light skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤‍👩🏼': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 200D 1F469 1F3FC', status='minimally-qualified', emoji='👩🏻\u200d❤\u200d👩🏼', name='couple with heart: woman, woman, light skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤️‍👩🏽': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 FE0F 200D 1F469 1F3FD', status='fully-qualified', emoji='👩🏻\u200d❤️\u200d👩🏽', name='couple with heart: woman, woman, light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤‍👩🏽': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 200D 1F469 1F3FD', status='minimally-qualified', emoji='👩🏻\u200d❤\u200d👩🏽', name='couple with heart: woman, woman, light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤️‍👩🏾': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 FE0F 200D 1F469 1F3FE', status='fully-qualified', emoji='👩🏻\u200d❤️\u200d👩🏾', name='couple with heart: woman, woman, light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤‍👩🏾': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 200D 1F469 1F3FE', status='minimally-qualified', emoji='👩🏻\u200d❤\u200d👩🏾', name='couple with heart: woman, woman, light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤️‍👩🏿': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 FE0F 200D 1F469 1F3FF', status='fully-qualified', emoji='👩🏻\u200d❤️\u200d👩🏿', name='couple with heart: woman, woman, light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏻‍❤‍👩🏿': EmojiEntry(codepoint='1F469 1F3FB 200D 2764 200D 1F469 1F3FF', status='minimally-qualified', emoji='👩🏻\u200d❤\u200d👩🏿', name='couple with heart: woman, woman, light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤️‍👩🏻': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 FE0F 200D 1F469 1F3FB', status='fully-qualified', emoji='👩🏼\u200d❤️\u200d👩🏻', name='couple with heart: woman, woman, medium-light skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤‍👩🏻': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 200D 1F469 1F3FB', status='minimally-qualified', emoji='👩🏼\u200d❤\u200d👩🏻', name='couple with heart: woman, woman, medium-light skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤️‍👩🏼': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 FE0F 200D 1F469 1F3FC', status='fully-qualified', emoji='👩🏼\u200d❤️\u200d👩🏼', name='couple with heart: woman, woman, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤‍👩🏼': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 200D 1F469 1F3FC', status='minimally-qualified', emoji='👩🏼\u200d❤\u200d👩🏼', name='couple with heart: woman, woman, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤️‍👩🏽': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 FE0F 200D 1F469 1F3FD', status='fully-qualified', emoji='👩🏼\u200d❤️\u200d👩🏽', name='couple with heart: woman, woman, medium-light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤‍👩🏽': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 200D 1F469 1F3FD', status='minimally-qualified', emoji='👩🏼\u200d❤\u200d👩🏽', name='couple with heart: woman, woman, medium-light skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤️‍👩🏾': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 FE0F 200D 1F469 1F3FE', status='fully-qualified', emoji='👩🏼\u200d❤️\u200d👩🏾', name='couple with heart: woman, woman, medium-light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤‍👩🏾': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 200D 1F469 1F3FE', status='minimally-qualified', emoji='👩🏼\u200d❤\u200d👩🏾', name='couple with heart: woman, woman, medium-light skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤️‍👩🏿': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 FE0F 200D 1F469 1F3FF', status='fully-qualified', emoji='👩🏼\u200d❤️\u200d👩🏿', name='couple with heart: woman, woman, medium-light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏼‍❤‍👩🏿': EmojiEntry(codepoint='1F469 1F3FC 200D 2764 200D 1F469 1F3FF', status='minimally-qualified', emoji='👩🏼\u200d❤\u200d👩🏿', name='couple with heart: woman, woman, medium-light skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤️‍👩🏻': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 FE0F 200D 1F469 1F3FB', status='fully-qualified', emoji='👩🏽\u200d❤️\u200d👩🏻', name='couple with heart: woman, woman, medium skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤‍👩🏻': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 200D 1F469 1F3FB', status='minimally-qualified', emoji='👩🏽\u200d❤\u200d👩🏻', name='couple with heart: woman, woman, medium skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤️‍👩🏼': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 FE0F 200D 1F469 1F3FC', status='fully-qualified', emoji='👩🏽\u200d❤️\u200d👩🏼', name='couple with heart: woman, woman, medium skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤‍👩🏼': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 200D 1F469 1F3FC', status='minimally-qualified', emoji='👩🏽\u200d❤\u200d👩🏼', name='couple with heart: woman, woman, medium skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤️‍👩🏽': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 FE0F 200D 1F469 1F3FD', status='fully-qualified', emoji='👩🏽\u200d❤️\u200d👩🏽', name='couple with heart: woman, woman, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤‍👩🏽': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 200D 1F469 1F3FD', status='minimally-qualified', emoji='👩🏽\u200d❤\u200d👩🏽', name='couple with heart: woman, woman, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤️‍👩🏾': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 FE0F 200D 1F469 1F3FE', status='fully-qualified', emoji='👩🏽\u200d❤️\u200d👩🏾', name='couple with heart: woman, woman, medium skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤‍👩🏾': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 200D 1F469 1F3FE', status='minimally-qualified', emoji='👩🏽\u200d❤\u200d👩🏾', name='couple with heart: woman, woman, medium skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤️‍👩🏿': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 FE0F 200D 1F469 1F3FF', status='fully-qualified', emoji='👩🏽\u200d❤️\u200d👩🏿', name='couple with heart: woman, woman, medium skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏽‍❤‍👩🏿': EmojiEntry(codepoint='1F469 1F3FD 200D 2764 200D 1F469 1F3FF', status='minimally-qualified', emoji='👩🏽\u200d❤\u200d👩🏿', name='couple with heart: woman, woman, medium skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤️‍👩🏻': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 FE0F 200D 1F469 1F3FB', status='fully-qualified', emoji='👩🏾\u200d❤️\u200d👩🏻', name='couple with heart: woman, woman, medium-dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤‍👩🏻': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 200D 1F469 1F3FB', status='minimally-qualified', emoji='👩🏾\u200d❤\u200d👩🏻', name='couple with heart: woman, woman, medium-dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤️‍👩🏼': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 FE0F 200D 1F469 1F3FC', status='fully-qualified', emoji='👩🏾\u200d❤️\u200d👩🏼', name='couple with heart: woman, woman, medium-dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤‍👩🏼': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 200D 1F469 1F3FC', status='minimally-qualified', emoji='👩🏾\u200d❤\u200d👩🏼', name='couple with heart: woman, woman, medium-dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤️‍👩🏽': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 FE0F 200D 1F469 1F3FD', status='fully-qualified', emoji='👩🏾\u200d❤️\u200d👩🏽', name='couple with heart: woman, woman, medium-dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤‍👩🏽': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 200D 1F469 1F3FD', status='minimally-qualified', emoji='👩🏾\u200d❤\u200d👩🏽', name='couple with heart: woman, woman, medium-dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤️‍👩🏾': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 FE0F 200D 1F469 1F3FE', status='fully-qualified', emoji='👩🏾\u200d❤️\u200d👩🏾', name='couple with heart: woman, woman, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤‍👩🏾': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 200D 1F469 1F3FE', status='minimally-qualified', emoji='👩🏾\u200d❤\u200d👩🏾', name='couple with heart: woman, woman, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤️‍👩🏿': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 FE0F 200D 1F469 1F3FF', status='fully-qualified', emoji='👩🏾\u200d❤️\u200d👩🏿', name='couple with heart: woman, woman, medium-dark skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏾‍❤‍👩🏿': EmojiEntry(codepoint='1F469 1F3FE 200D 2764 200D 1F469 1F3FF', status='minimally-qualified', emoji='👩🏾\u200d❤\u200d👩🏿', name='couple with heart: woman, woman, medium-dark skin tone, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤️‍👩🏻': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 FE0F 200D 1F469 1F3FB', status='fully-qualified', emoji='👩🏿\u200d❤️\u200d👩🏻', name='couple with heart: woman, woman, dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤‍👩🏻': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 200D 1F469 1F3FB', status='minimally-qualified', emoji='👩🏿\u200d❤\u200d👩🏻', name='couple with heart: woman, woman, dark skin tone, light skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤️‍👩🏼': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 FE0F 200D 1F469 1F3FC', status='fully-qualified', emoji='👩🏿\u200d❤️\u200d👩🏼', name='couple with heart: woman, woman, dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤‍👩🏼': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 200D 1F469 1F3FC', status='minimally-qualified', emoji='👩🏿\u200d❤\u200d👩🏼', name='couple with heart: woman, woman, dark skin tone, medium-light skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤️‍👩🏽': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 FE0F 200D 1F469 1F3FD', status='fully-qualified', emoji='👩🏿\u200d❤️\u200d👩🏽', name='couple with heart: woman, woman, dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤‍👩🏽': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 200D 1F469 1F3FD', status='minimally-qualified', emoji='👩🏿\u200d❤\u200d👩🏽', name='couple with heart: woman, woman, dark skin tone, medium skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤️‍👩🏾': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 FE0F 200D 1F469 1F3FE', status='fully-qualified', emoji='👩🏿\u200d❤️\u200d👩🏾', name='couple with heart: woman, woman, dark skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤‍👩🏾': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 200D 1F469 1F3FE', status='minimally-qualified', emoji='👩🏿\u200d❤\u200d👩🏾', name='couple with heart: woman, woman, dark skin tone, medium-dark skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤️‍👩🏿': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 FE0F 200D 1F469 1F3FF', status='fully-qualified', emoji='👩🏿\u200d❤️\u200d👩🏿', name='couple with heart: woman, woman, dark skin tone', group='People & Body', sub_group='family'),
    '👩🏿‍❤‍👩🏿': EmojiEntry(codepoint='1F469 1F3FF 200D 2764 200D 1F469 1F3FF', status='minimally-qualified', emoji='👩🏿\u200d❤\u200d👩🏿', name='couple with heart: woman, woman, dark skin tone', group='People & Body', sub_group='family'),
    '👪': EmojiEntry(codepoint='1F46A', status='fully-qualified', emoji='👪', name='family', group='People & Body', sub_group='family'),
    '👨‍👩‍👦': EmojiEntry(codepoint='1F468 200D 1F469 200D 1F466', status='fully-qualified', emoji='👨\u200d👩\u200d👦', name='family: man, woman, boy', group='People & Body', sub_group='family'),
    '👨‍👩‍👧': EmojiEntry(codepoint='1F468 200D 1F469 200D 1F467', status='fully-qualified', emoji='👨\u200d👩\u200d👧', name='family: man, woman, girl', group='People & Body', sub_group='family'),
    '👨‍👩‍👧‍👦': EmojiEntry(codepoint='1F468 200D 1F469 200D 1F467 200D 1F466', status='fully-qualified', emoji='👨\u200d👩\u200d👧\u200d👦', name='family: man, woman, girl, boy', group='People & Body', sub_group='family'),
    '👨‍👩‍👦‍👦': EmojiEntry(codepoint='1F468 200D 1F469 200D 1F466 200D 1F466', status='fully-qualified', emoji='👨\u200d👩\u200d👦\u200d👦', name='family: man, woman, boy, boy', group='People & Body', sub_group='family'),
    '👨‍👩‍👧‍👧': EmojiEntry(codepoint='1F468 200D 1F469 200D 1F467 200D 1F467', status='fully-qualified', emoji='👨\u200d👩\u200d👧\u200d👧', name='family: man, woman, girl, girl', group='People & Body', sub_group='family'),
    '👨‍👨‍👦': EmojiEntry(codepoint='1F468 200D 1F468 200D 1F466', status='fully-qualified', emoji='👨\u200d👨\u200d👦', name='family: man, man, boy', group='People & Body', sub_group='family'),
    '👨‍👨‍👧': EmojiEntry(codepoint='1F468 200D 1F468 200D 1F467', status='fully-qualified', emoji='👨\u200d👨\u200d👧', name='family: man, man, girl', group='People & Body', sub_group='family'),
    '👨‍👨‍👧‍👦': EmojiEntry(codepoint='1F468 200D 1F468 200D 1F467 200D 1F466', status='fully-qualified', emoji='👨\u200d👨\u200d👧\u200d👦', name='family: man, man, girl, boy', group='People & Body', sub_group='family'),
    '👨‍👨‍👦‍👦': EmojiEntry(codepoint='1F468 200D 1F468 200D 1F466 200D 1F466', status='fully-qualified', emoji='👨\u200d👨\u200d👦\u200d👦', name='family: man, man, boy, boy', group='People & Body', sub_group='family'),
    '👨‍👨‍👧‍👧': EmojiEntry(codepoint='1F468 200D 1F468 200D 1F467 200D 1F467', status='fully-qualified', emoji='👨\u200d👨\u200d👧\u200d👧', name='family: man, man, girl, girl', group='People & Body', sub_group='family'),
    '👩‍👩‍👦': EmojiEntry(codepoint='1F469 200D 1F469 200D 1F466', status='fully-qualified', emoji='👩\u200d👩\u200d👦', name='family: woman, woman, boy', group='People & Body', sub_group='family'),
    '👩‍👩‍👧': EmojiEntry(codepoint='1F469 200D 1F469 200D 1F467', status='fully-qualified', emoji='👩\u200d👩\u200d👧', name='family: woman, woman, girl', group='People & Body', sub_group='family'),
    '👩‍👩‍👧‍👦': EmojiEntry(codepoint='1F469 200D 1F469 200D 1F467 200D 1F466', status='fully-qualified', emoji='👩\u200d👩\u200d👧\u200d👦', name='family: woman, woman, girl, boy', group='People & Body', sub_group='family'),
    '👩‍👩‍👦‍👦': EmojiEntry(codepoint='1F469 200D 1F469 200D 1F466 200D 1F466', status='fully-qualified', emoji='👩\u200d👩\u200d👦\u200d👦', name='family: woman, woman, boy, boy', group='People & Body', sub_group='family'),
    '👩‍👩‍👧‍👧': EmojiEntry(codepoint='1F469 200D 1F469 200D 1F467 200D 1F467', status='fully-qualified', emoji='👩\u200d👩\u200d👧\u200d👧', name='family: woman, woman, girl, girl', group='People & Body', sub_group='family'),
    '👨‍👦': EmojiEntry(codepoint='1F468 200D 1F466', status='fully-qualified', emoji='👨\u200d👦', name='family: man, boy', group='People & Body', sub_group='family'),
    '👨‍👦‍👦': EmojiEntry(codepoint='1F468 200D 1F466 200D 1F466', status='fully-qualified', emoji='👨\u200d👦\u200d👦', name='family: man, boy, boy', group='People & Body', sub_group='family'),
    '👨‍👧': EmojiEntry(codepoint='1F468 200D 1F467', status='fully-qualified', emoji='👨\u200d👧', name='family: man, girl', group='People & Body', sub_group='family'),
    '👨‍👧‍👦': EmojiEntry(codepoint='1F468 200D 1F467 200D 1F466', status='fully-qualified', emoji='👨\u200d👧\u200d👦', name='family: man, girl, boy', group='People & Body', sub_group='family'),
    '👨‍👧‍👧': EmojiEntry(codepoint='1F468 200D 1F467 200D 1F467', status='fully-qualified', emoji='👨\u200d👧\u200d👧', name='family: man, girl, girl', group='People & Body', sub_group='family'),
    '👩‍👦': EmojiEntry(codepoint='1F469 200D 1F466', status='fully-qualified', emoji='👩\u200d👦', name='family: woman, boy', group='People & Body', sub_group='family'),
    '👩‍👦‍👦': EmojiEntry(codepoint='1F469 200D 1F466 200D 1F466', status='fully-qualified', emoji='👩\u200d👦\u200d👦', name='family: woman, boy, boy', group='People & Body', sub_group='family'),
    '👩‍👧': EmojiEntry(codepoint='1F469 200D 1F467', status='fully-qualified', emoji='👩\u200d👧', name='family: woman, girl', group='People & Body', sub_group='family'),
    '👩‍👧‍👦': EmojiEntry(codepoint='1F469 200D 1F467 200D 1F466', status='fully-qualified', emoji='👩\u200d👧\u200d👦', name='family: woman, girl, boy', group='People & Body', sub_group='family'),
    '👩‍👧‍👧': EmojiEntry(codepoint='1F469 200D 1F467 200D 1F467', status='fully-qualified', emoji='👩\u200d👧\u200d👧', name='family: woman, girl, girl', group='People & Body', sub_group='family'),
    '🗣️': EmojiEntry(codepoint='1F5E3 FE0F', status='fully-qualified', emoji='🗣️', name='speaking head', group='People & Body', sub_group='person-symbol'),
    '🗣': EmojiEntry(codepoint='1F5E3', status='unqualified', emoji='🗣', name='speaking head', group='People & Body', sub_group='person-symbol'),
    '👤': EmojiEntry(codepoint='1F464', status='fully-qualified', emoji='👤', name='bust in silhouette', group='People & Body', sub_group='person-symbol'),
    '👥': EmojiEntry(codepoint='1F465', status='fully-qualified', emoji='👥', name='busts in silhouette', group='People & Body', sub_group='person-symbol'),
    '🫂': EmojiEntry(codepoint='1FAC2', status='fully-qualified', emoji='🫂', name='people hugging', group='People & Body', sub_group='person-symbol'),
    '👣': EmojiEntry(codepoint='1F463', status='fully-qualified', emoji='👣', name='footprints', group='People & Body', sub_group='person-symbol'),
    '🐵': EmojiEntry(codepoint='1F435', status='fully-qualified', emoji='🐵', name='monkey face', group='Animals & Nature', sub_group='animal-mammal'),
    '🐒': EmojiEntry(codepoint='1F412', status='fully-qualified', emoji='🐒', name='monkey', group='Animals & Nature', sub_group='animal-mammal'),
    '🦍': EmojiEntry(codepoint='1F98D', status='fully-qualified', emoji='🦍', name='gorilla', group='Animals & Nature', sub_group='animal-mammal'),
    '🦧': EmojiEntry(codepoint='1F9A7', status='fully-qualified', emoji='🦧', name='orangutan', group='Animals & Nature', sub_group='animal-mammal'),
    '🐶': EmojiEntry(codepoint='1F436', status='fully-qualified', emoji='🐶', name='dog face', group='Animals & Nature', sub_group='animal-mammal'),
    '🐕': EmojiEntry(codepoint='1F415', status='fully-qualified', emoji='🐕', name='dog', group='Animals & Nature', sub_group='animal-mammal'),
    '🦮': EmojiEntry(codepoint='1F9AE', status='fully-qualified', emoji='🦮', name='guide dog', group='Animals & Nature', sub_group='animal-mammal'),
    '🐕‍🦺': EmojiEntry(codepoint='1F415 200D 1F9BA', status='fully-qualified', emoji='🐕\u200d🦺', name='service dog', group='Animals & Nature', sub_group='animal-mammal'),
    '🐩': EmojiEntry(codepoint='1F429', status='fully-qualified', emoji='🐩', name='poodle', group='Animals & Nature', sub_group='animal-mammal'),
    '🐺': EmojiEntry(codepoint='1F43A', status='fully-qualified', emoji='🐺', name='wolf', group='Animals & Nature', sub_group='animal-mammal'),
    '🦊': EmojiEntry(codepoint='1F98A', status='fully-qualified', emoji='🦊', name='fox', group='Animals & Nature', sub_group='animal-mammal'),
    '🦝': EmojiEntry(codepoint='1F99D', status='fully-qualified', emoji='🦝', name='raccoon', group='Animals & Nature', sub_group='animal-mammal'),
    '🐱': EmojiEntry(codepoint='1F431', status='fully-qualified', emoji='🐱', name='cat face', group='Animals & Nature', sub_group='animal-mammal'),
    '🐈': EmojiEntry(codepoint='1F408', status='fully-qualified', emoji='🐈', name='cat', group='Animals & Nature', sub_group='animal-mammal'),
    '🐈‍⬛': EmojiEntry(codepoint='1F408 200D 2B1B', status='fully-qualified', emoji='🐈\u200d⬛', name='black cat', group='Animals & Nature', sub_group='animal-mammal'),
    '🦁': EmojiEntry(codepoint='1F981', status='fully-qualified', emoji='🦁', name='lion', group='Animals & Nature', sub_group='animal-mammal'),
    '🐯': EmojiEntry(codepoint='1F42F', status='fully-qualified', emoji='🐯', name='tiger face', group='Animals & Nature', sub_group='animal-mammal'),
    '🐅': EmojiEntry(codepoint='1F405', status='fully-qualified', emoji='🐅', name='tiger', group='Animals & Nature', sub_group='animal-mammal'),
    '🐆': EmojiEntry(codepoint='1F406', status='fully-qualified', emoji='🐆', name='leopard', group='Animals & Nature', sub_group='animal-mammal'),
    '🐴': EmojiEntry(codepoint='1F434', status='fully-qualified', emoji='🐴', name='horse face', group='Animals & Nature', sub_group='animal-mammal'),
    '🐎': EmojiEntry(codepoint='1F40E', status='fully-qualified', emoji='🐎', name='horse', group='Animals & Nature', sub_group='animal-mammal'),
    '🦄': EmojiEntry(codepoint='1F984', status='fully-qualified', emoji='🦄', name='unicorn', group='Animals & Nature', sub_group='animal-mammal'),
    '🦓': EmojiEntry(codepoint='1F993', status='fully-qualified', emoji='🦓', name='zebra', group='Animals & Nature', sub_group='animal-mammal'),
    '🦌': EmojiEntry(codepoint='1F98C', status='fully-qualified', emoji='🦌', name='deer', group='Animals & Nature', sub_group='animal-mammal'),
    '🦬': EmojiEntry(codepoint='1F9AC', status='fully-qualified', emoji='🦬', name='bison', group='Animals & Nature', sub_group='animal-mammal'),
    '🐮': EmojiEntry(codepoint='1F42E', status='fully-qualified', emoji='🐮', name='cow face', group='Animals & Nature', sub_group='animal-mammal'),
    '🐂': EmojiEntry(codepoint='1F402', status='fully-qualified', emoji='🐂', name='ox', group='Animals & Nature', sub_group='animal-mammal'),
    '🐃': EmojiEntry(codepoint='1F403', status='fully-qualified', emoji='🐃', name='water buffalo', group='Animals & Nature', sub_group='animal-mammal'),
    '🐄': EmojiEntry(codepoint='1F404', status='fully-qualified', emoji='🐄', name='cow', group='Animals & Nature', sub_group='animal-mammal'),
    '🐷': EmojiEntry(codepoint='1F437', status='fully-qualified', emoji='🐷', name='pig face', group='Animals & Nature', sub_group='animal-mammal'),
    '🐖': EmojiEntry(codepoint='1F416', status='fully-qualified', emoji='🐖', name='pig', group='Animals & Nature', sub_group='animal-mammal'),
    '🐗': EmojiEntry(codepoint='1F417', status='fully-qualified', emoji='🐗', name='boar', group='Animals & Nature', sub_group='animal-mammal'),
    '🐽': EmojiEntry(codepoint='1F43D', status='fully-qualified', emoji='🐽', name='pig nose', group='Animals & Nature', sub_group='animal-mammal'),
    '🐏': EmojiEntry(codepoint='1F40F', status='fully-qualified', emoji='🐏', name='ram', group='Animals & Nature', sub_group='animal-mammal'),
    '🐑': EmojiEntry(codepoint='1F411', status='fully-qualified', emoji='🐑', name='ewe', group='Animals & Nature', sub_group='animal-mammal'),
    '🐐': EmojiEntry(codepoint='1F410', status='fully-qualified', emoji='🐐', name='goat', group='Animals & Nature', sub_group='animal-mammal'),
    '🐪': EmojiEntry(codepoint='1F42A', status='fully-qualified', emoji='🐪', name='camel', group='Animals & Nature', sub_group='animal-mammal'),
    '🐫': EmojiEntry(codepoint='1F42B', status='fully-qualified', emoji='🐫', name='two-hump camel', group='Animals & Nature', sub_group='animal-mammal'),
    '🦙': EmojiEntry(codepoint='1F999', status='fully-qualified', emoji='🦙', name='llama', group='Animals & Nature', sub_group='animal-mammal'),
    '🦒': EmojiEntry(codepoint='1F992', status='fully-qualified', emoji='🦒', name='giraffe', group='Animals & Nature', sub_group='animal-mammal'),
    '🐘': EmojiEntry(codepoint='1F418', status='fully-qualified', emoji='🐘', name='elephant', group='Animals & Nature', sub_group='animal-mammal'),
    '🦣': EmojiEntry(codepoint='1F9A3', status='fully-qualified', emoji='🦣', name='mammoth', group='Animals & Nature', sub_group='animal-mammal'),
    '🦏': EmojiEntry(codepoint='1F98F', status='fully-qualified', emoji='🦏', name='rhinoceros', group='Animals & Nature', sub_group='animal-mammal'),
    '🦛': EmojiEntry(codepoint='1F99B', status='fully-qualified', emoji='🦛', name='hippopotamus', group='Animals & Nature', sub_group='animal-mammal'),
    '🐭': EmojiEntry(codepoint='1F42D', status='fully-qualified', emoji='🐭', name='mouse face', group='Animals & Nature', sub_group='animal-mammal'),
    '🐁': EmojiEntry(codepoint='1F401', status='fully-qualified', emoji='🐁', name='mouse', group='Animals & Nature', sub_group='animal-mammal'),
    '🐀': EmojiEntry(codepoint='1F400', status='fully-qualified', emoji='🐀', name='rat', group='Animals & Nature', sub_group='animal-mammal'),
    '🐹': EmojiEntry(codepoint='1F439', status='fully-qualified', emoji='🐹', name='hamster', group='Animals & Nature', sub_group='animal-mammal'),
    '🐰': EmojiEntry(codepoint='1F430', status='fully-qualified', emoji='🐰', name='rabbit face', group='Animals & Nature', sub_group='animal-mammal'),
    '🐇': EmojiEntry(codepoint='1F407', status='fully-qualified', emoji='🐇', name='rabbit', group='Animals & Nature', sub_group='animal-mammal'),
    '🐿️': EmojiEntry(codepoint='1F43F FE0F', status='fully-qualified', emoji='🐿️', name='chipmunk', group='Animals & Nature', sub_group='animal-mammal'),
    '🐿': EmojiEntry(codepoint='1F43F', status='unqualified', emoji='🐿', name='chipmunk', group='Animals & Nature', sub_group='animal-mammal'),
    '🦫': EmojiEntry(codepoint='1F9AB', status='fully-qualified', emoji='🦫', name='beaver', group='Animals & Nature', sub_group='animal-mammal'),
    '🦔': EmojiEntry(codepoint='1F994', status='fully-qualified', emoji='🦔', name='hedgehog', group='Animals & Nature', sub_group='animal-mammal'),
    '🦇': EmojiEntry(codepoint='1F987', status='fully-qualified', emoji='🦇', name='bat', group='Animals & Nature', sub_group='animal-mammal'),
    '🐻': EmojiEntry(codepoint='1F43B', status='fully-qualified', emoji='🐻', name='bear', group='Animals & Nature', sub_group='animal-mammal'),
    '🐻‍❄️': EmojiEntry(codepoint='1F43B 200D 2744 FE0F', status='fully-qualified', emoji='🐻\u200d❄️', name='polar bear', group='Animals & Nature', sub_group='animal-mammal'),
    '🐻‍❄': EmojiEntry(codepoint='1F43B 200D 2744', status='minimally-qualified', emoji='🐻\u200d❄', name='polar bear', group='Animals & Nature', sub_group='animal-mammal'),
    '🐨': EmojiEntry(codepoint='1F428', status='fully-qualified', emoji='🐨', name='koala', group='Animals & Nature', sub_group='animal-mammal'),
    '🐼': EmojiEntry(codepoint='1F43C', status='fully-qualified', emoji='🐼', name='panda', group='Animals & Nature', sub_group='animal-mammal'),
    '🦥': EmojiEntry(codepoint='1F9A5', status='fully-qualified', emoji='🦥', name='sloth', group='Animals & Nature', sub_group='animal-mammal'),
    '🦦': EmojiEntry(codepoint='1F9A6', status='fully-qualified', emoji='🦦', name='otter', group='Animals & Nature', sub_group='animal-mammal'),
    '🦨': EmojiEntry(codepoint='1F9A8', status='fully-qualified', emoji='🦨', name='skunk', group='Animals & Nature', sub_group='animal-mammal'),
    '🦘': EmojiEntry(codepoint='1F998', status='fully-qualified', emoji='🦘', name='kangaroo', group='Animals & Nature', sub_group='animal-mammal'),
    '🦡': EmojiEntry(codepoint='1F9A1', status='fully-qualified', emoji='🦡', name='badger', group='Animals & Nature', sub_group='animal-mammal'),
    '🐾': EmojiEntry(codepoint='1F43E', status='fully-qualified', emoji='🐾', name='paw prints', group='Animals & Nature', sub_group='animal-mammal'),
    '🦃': EmojiEntry(codepoint='1F983', status='fully-qualified', emoji='🦃', name='turkey', group='Animals & Nature', sub_group='animal-bird'),
    '🐔': EmojiEntry(codepoint='1F414', status='fully-qualified', emoji='🐔', name='chicken', group='Animals & Nature', sub_group='animal-bird'),
    '🐓': EmojiEntry(codepoint='1F413', status='fully-qualified', emoji='🐓', name='rooster', group='Animals & Nature', sub_group='animal-bird'),
    '🐣': EmojiEntry(codepoint='1F423', status='fully-qualified', emoji='🐣', name='hatching chick', group='Animals & Nature', sub_group='animal-bird'),
    '🐤': EmojiEntry(codepoint='1F424', status='fully-qualified', emoji='🐤', name='baby chick', group='Animals & Nature', sub_group='animal-bird'),
    '🐥': EmojiEntry(codepoint='1F425', status='fully-qualified', emoji='🐥', name='front-facing baby chick', group='Animals & Nature', sub_group='animal-bird'),
    '🐦': EmojiEntry(codepoint='1F426', status='fully-qualified', emoji='🐦', name='bird', group='Animals & Nature', sub_group='animal-bird'),
    '🐧': EmojiEntry(codepoint='1F427', status='fully-qualified', emoji='🐧', name='penguin', group='Animals & Nature', sub_group='animal-bird'),
    '🕊️': EmojiEntry(codepoint='1F54A FE0F', status='fully-qualified', emoji='🕊️', name='dove', group='Animals & Nature', sub_group='animal-bird'),
    '🕊': EmojiEntry(codepoint='1F54A', status='unqualified', emoji='🕊', name='dove', group='Animals & Nature', sub_group='animal-bird'),
    '🦅': EmojiEntry(codepoint='1F985', status='fully-qualified', emoji='🦅', name='eagle', group='Animals & Nature', sub_group='animal-bird'),
    '🦆': EmojiEntry(codepoint='1F986', status='fully-qualified', emoji='🦆', name='duck', group='Animals & Nature', sub_group='animal-bird'),
    '🦢': EmojiEntry(codepoint='1F9A2', status='fully-qualified', emoji='🦢', name='swan', group='Animals & Nature', sub_group='animal-bird'),
    '🦉': EmojiEntry(codepoint='1F989', status='fully-qualified', emoji='🦉', name='owl', group='Animals & Nature', sub_group='animal-bird'),
    '🦤': EmojiEntry(codepoint='1F9A4', status='fully-qualified', emoji='🦤', name='dodo', group='Animals & Nature', sub_group='animal-bird'),
    '🪶': EmojiEntry(codepoint='1FAB6', status='fully-qualified', emoji='🪶', name='feather', group='Animals & Nature', sub_group='animal-bird'),
    '🦩': EmojiEntry(codepoint='1F9A9', status='fully-qualified', emoji='🦩', name='flamingo', group='Animals & Nature', sub_group='animal-bird'),
    '🦚': EmojiEntry(codepoint='1F99A', status='fully-qualified', emoji='🦚', name='peacock', group='Animals & Nature', sub_group='animal-bird'),
    '🦜': EmojiEntry(codepoint='1F99C', status='fully-qualified', emoji='🦜', name='parrot', group='Animals & Nature', sub_group='animal-bird'),
    '🐸': EmojiEntry(codepoint='1F438', status='fully-qualified', emoji='🐸', name='frog', group='Animals & Nature', sub_group='animal-amphibian'),
    '🐊': EmojiEntry(codepoint='1F40A', status='fully-qualified', emoji='🐊', name='crocodile', group='Animals & Nature', sub_group='animal-reptile'),
    '🐢': EmojiEntry(codepoint='1F422', status='fully-qualified', emoji='🐢', name='turtle', group='Animals & Nature', sub_group='animal-reptile'),
    '🦎': EmojiEntry(codepoint='1F98E', status='fully-qualified', emoji='🦎', name='lizard', group='Animals & Nature', sub_group='animal-reptile'),
    '🐍': EmojiEntry(codepoint='1F40D', status='fully-qualified', emoji='🐍', name='snake', group='Animals & Nature', sub_group='animal-reptile'),
    '🐲': EmojiEntry(codepoint='1F432', status='fully-qualified', emoji='🐲', name='dragon face', group='Animals & Nature', sub_group='animal-reptile'),
    '🐉': EmojiEntry(codepoint='1F409', status='fully-qualified', emoji='🐉', name='dragon', group='Animals & Nature', sub_group='animal-reptile'),
    '🦕': EmojiEntry(codepoint='1F995', status='fully-qualified', emoji='🦕', name='sauropod', group='Animals & Nature', sub_group='animal-reptile'),
    '🦖': EmojiEntry(codepoint='1F996', status='fully-qualified', emoji='🦖', name='T-Rex', group='Animals & Nature', sub_group='animal-reptile'),
    '🐳': EmojiEntry(codepoint='1F433', status='fully-qualified', emoji='🐳', name='spouting whale', group='Animals & Nature', sub_group='animal-marine'),
    '🐋': EmojiEntry(codepoint='1F40B', status='fully-qualified', emoji='🐋', name='whale', group='Animals & Nature', sub_group='animal-marine'),
    '🐬': EmojiEntry(codepoint='1F42C', status='fully-qualified', emoji='🐬', name='dolphin', group='Animals & Nature', sub_group='animal-marine'),
    '🦭': EmojiEntry(codepoint='1F9AD', status='fully-qualified', emoji='🦭', name='seal', group='Animals & Nature', sub_group='animal-marine'),
    '🐟': EmojiEntry(codepoint='1F41F', status='fully-qualified', emoji='🐟', name='fish', group='Animals & Nature', sub_group='animal-marine'),
    '🐠': EmojiEntry(codepoint='1F420', status='fully-qualified', emoji='🐠', name='tropical fish', group='Animals & Nature', sub_group='animal-marine'),
    '🐡': EmojiEntry(codepoint='1F421', status='fully-qualified', emoji='🐡', name='blowfish', group='Animals & Nature', sub_group='animal-marine'),
    '🦈': EmojiEntry(codepoint='1F988', status='fully-qualified', emoji='🦈', name='shark', group='Animals & Nature', sub_group='animal-marine'),
    '🐙': EmojiEntry(codepoint='1F419', status='fully-qualified', emoji='🐙', name='octopus', group='Animals & Nature', sub_group='animal-marine'),
    '🐚': EmojiEntry(codepoint='1F41A', status='fully-qualified', emoji='🐚', name='spiral shell', group='Animals & Nature', sub_group='animal-marine'),
    '🐌': EmojiEntry(codepoint='1F40C', status='fully-qualified', emoji='🐌', name='snail', group='Animals & Nature', sub_group='animal-bug'),
    '🦋': EmojiEntry(codepoint='1F98B', status='fully-qualified', emoji='🦋', name='butterfly', group='Animals & Nature', sub_group='animal-bug'),
    '🐛': EmojiEntry(codepoint='1F41B', status='fully-qualified', emoji='🐛', name='bug', group='Animals & Nature', sub_group='animal-bug'),
    '🐜': EmojiEntry(codepoint='1F41C', status='fully-qualified', emoji='🐜', name='ant', group='Animals & Nature', sub_group='animal-bug'),
    '🐝': EmojiEntry(codepoint='1F41D', status='fully-qualified', emoji='🐝', name='honeybee', group='Animals & Nature', sub_group='animal-bug'),
    '🪲': EmojiEntry(codepoint='1FAB2', status='fully-qualified', emoji='🪲', name='beetle', group='Animals & Nature', sub_group='animal-bug'),
    '🐞': EmojiEntry(codepoint='1F41E', status='fully-qualified', emoji='🐞', name='lady beetle', group='Animals & Nature', sub_group='animal-bug'),
    '🦗': EmojiEntry(codepoint='1F997', status='fully-qualified', emoji='🦗', name='cricket', group='Animals & Nature', sub_group='animal-bug'),
    '🪳': EmojiEntry(codepoint='1FAB3', status='fully-qualified', emoji='🪳', name='cockroach', group='Animals & Nature', sub_group='animal-bug'),
    '🕷️': EmojiEntry(codepoint='1F577 FE0F', status='fully-qualified', emoji='🕷️', name='spider', group='Animals & Nature', sub_group='animal-bug'),
    '🕷': EmojiEntry(codepoint='1F577', status='unqualified', emoji='🕷', name='spider', group='Animals & Nature', sub_group='animal-bug'),
    '🕸️': EmojiEntry(codepoint='1F578 FE0F', status='fully-qualified', emoji='🕸️', name='spider web', group='Animals & Nature', sub_group='animal-bug'),
    '🕸': EmojiEntry(codepoint='1F578', status='unqualified', emoji='🕸', name='spider web', group='Animals & Nature', sub_group='animal-bug'),
    '🦂': EmojiEntry(codepoint='1F982', status='fully-qualified', emoji='🦂', name='scorpion', group='Animals & Nature', sub_group='animal-bug'),
    '🦟': EmojiEntry(codepoint='1F99F', status='fully-qualified', emoji='🦟', name='mosquito', group='Animals & Nature', sub_group='animal-bug'),
    '🪰': EmojiEntry(codepoint='1FAB0', status='fully-qualified', emoji='🪰', name='fly', group='Animals & Nature', sub_group='animal-bug'),
    '🪱': EmojiEntry(codepoint='1FAB1', status='fully-qualified', emoji='🪱', name='worm', group='Animals & Nature', sub_group='animal-bug'),
    '🦠': EmojiEntry(codepoint='1F9A0', status='fully-qualified', emoji='🦠', name='microbe', group='Animals & Nature', sub_group='animal-bug'),
    '💐': EmojiEntry(codepoint='1F490', status='fully-qualified', emoji='💐', name='bouquet', group='Animals & Nature', sub_group='plant-flower'),
    '🌸': EmojiEntry(codepoint='1F338', status='fully-qualified', emoji='🌸', name='cherry blossom', group='Animals & Nature', sub_group='plant-flower'),
    '💮': EmojiEntry(codepoint='1F4AE', status='fully-qualified', emoji='💮', name='white flower', group='Animals & Nature', sub_group='plant-flower'),
    '🏵️': EmojiEntry(codepoint='1F3F5 FE0F', status='fully-qualified', emoji='🏵️', name='rosette', group='Animals & Nature', sub_group='plant-flower'),
    '🏵': EmojiEntry(codepoint='1F3F5', status='unqualified', emoji='🏵', name='rosette', group='Animals & Nature', sub_group='plant-flower'),
    '🌹': EmojiEntry(codepoint='1F339', status='fully-qualified', emoji='🌹', name='rose', group='Animals & Nature', sub_group='plant-flower'),
    '🥀': EmojiEntry(codepoint='1F940', status='fully-qualified', emoji='🥀', name='wilted flower', group='Animals & Nature', sub_group='plant-flower'),
    '🌺': EmojiEntry(codepoint='1F33A', status='fully-qualified', emoji='🌺', name='hibiscus', group='Animals & Nature', sub_group='plant-flower'),
    '🌻': EmojiEntry(codepoint='1F33B', status='fully-qualified', emoji='🌻', name='sunflower', group='Animals & Nature', sub_group='plant-flower'),
    '🌼': EmojiEntry(codepoint='1F33C', status='fully-qualified', emoji='🌼', name='blossom', group='Animals & Nature', sub_group='plant-flower'),
    '🌷': EmojiEntry(codepoint='1F337', status='fully-qualified', emoji='🌷', name='tulip', group='Animals & Nature', sub_group='plant-flower'),
    '🌱': EmojiEntry(codepoint='1F331', status='fully-qualified', emoji='🌱', name='seedling', group='Animals & Nature', sub_group='plant-other'),
    '🪴': EmojiEntry(codepoint='1FAB4', status='fully-qualified', emoji='🪴', name='potted plant', group='Animals & Nature', sub_group='plant-other'),
    '🌲': EmojiEntry(codepoint='1F332', status='fully-qualified', emoji='🌲', name='evergreen tree', group='Animals & Nature', sub_group='plant-other'),
    '🌳': EmojiEntry(codepoint='1F333', status='fully-qualified', emoji='🌳', name='deciduous tree', group='Animals & Nature', sub_group='plant-other'),
    '🌴': EmojiEntry(codepoint='1F334', status='fully-qualified', emoji='🌴', name='palm tree', group='Animals & Nature', sub_group='plant-other'),
    '🌵': EmojiEntry(codepoint='1F335', status='fully-qualified', emoji='🌵', name='cactus', group='Animals & Nature', sub_group='plant-other'),
    '🌾': EmojiEntry(codepoint='1F33E', status='fully-qualified', emoji='🌾', name='sheaf of rice', group='Animals & Nature', sub_group='plant-other'),
    '🌿': EmojiEntry(codepoint='1F33F', status='fully-qualified', emoji='🌿', name='herb', group='Animals & Nature', sub_group='plant-other'),
    '☘️': EmojiEntry(codepoint='2618 FE0F', status='fully-qualified', emoji='☘️', name='shamrock', group='Animals & Nature', sub_group='plant-other'),
    '☘': EmojiEntry(codepoint='2618', status='unqualified', emoji='☘', name='shamrock', group='Animals & Nature', sub_group='plant-other'),
    '🍀': EmojiEntry(codepoint='1F340', status='fully-qualified', emoji='🍀', name='four leaf clover', group='Animals & Nature', sub_group='plant-other'),
    '🍁': EmojiEntry(codepoint='1F341', status='fully-qualified', emoji='🍁', name='maple leaf', group='Animals & Nature', sub_group='plant-other'),
    '🍂': EmojiEntry(codepoint='1F342', status='fully-qualified', emoji='🍂', name='fallen leaf', group='Animals & Nature', sub_group='plant-other'),
    '🍃': EmojiEntry(codepoint='1F343', status='fully-qualified', emoji='🍃', name='leaf fluttering in wind', group='Animals & Nature', sub_group='plant-other'),
    '🍇': EmojiEntry(codepoint='1F347', status='fully-qualified', emoji='🍇', name='grapes', group='Food & Drink', sub_group='food-fruit'),
    '🍈': EmojiEntry(codepoint='1F348', status='fully-qualified', emoji='🍈', name='melon', group='Food & Drink', sub_group='food-fruit'),
    '🍉': EmojiEntry(codepoint='1F349', status='fully-qualified', emoji='🍉', name='watermelon', group='Food & Drink', sub_group='food-fruit'),
    '🍊': EmojiEntry(codepoint='1F34A', status='fully-qualified', emoji='🍊', name='tangerine', group='Food & Drink', sub_group='food-fruit'),
    '🍋': EmojiEntry(codepoint='1F34B', status='fully-qualified', emoji='🍋', name='lemon', group='Food & Drink', sub_group='food-fruit'),
    '🍌': EmojiEntry(codepoint='1F34C', status='fully-qualified', emoji='🍌', name='banana', group='Food & Drink', sub_group='food-fruit'),
    '🍍': EmojiEntry(codepoint='1F34D', status='fully-qualified', emoji='🍍', name='pineapple', group='Food & Drink', sub_group='food-fruit'),
    '🥭': EmojiEntry(codepoint='1F96D', status='fully-qualified', emoji='🥭', name='mango', group='Food & Drink', sub_group='food-fruit'),
    '🍎': EmojiEntry(codepoint='1F34E', status='fully-qualified', emoji='🍎', name='red apple', group='Food & Drink', sub_group='food-fruit'),
    '🍏': EmojiEntry(codepoint='1F34F', status='fully-qualified', emoji='🍏', name='green apple', group='Food & Drink', sub_group='food-fruit'),
    '🍐': EmojiEntry(codepoint='1F350', status='fully-qualified', emoji='🍐', name='pear', group='Food & Drink', sub_group='food-fruit'),
    '🍑': EmojiEntry(codepoint='1F351', status='fully-qualified', emoji='🍑', name='peach', group='Food & Drink', sub_group='food-fruit'),
    '🍒': EmojiEntry(codepoint='1F352', status='fully-qualified', emoji='🍒', name='cherries', group='Food & Drink', sub_group='food-fruit'),
    '🍓': EmojiEntry(codepoint='1F353', status='fully-qualified', emoji='🍓', name='strawberry', group='Food & Drink', sub_group='food-fruit'),
    '🫐': EmojiEntry(codepoint='1FAD0', status='fully-qualified', emoji='🫐', name='blueberries', group='Food & Drink', sub_group='food-fruit'),
    '🥝': EmojiEntry(codepoint='1F95D', status='fully-qualified', emoji='🥝', name='kiwi fruit', group='Food & Drink', sub_group='food-fruit'),
    '🍅': EmojiEntry(codepoint='1F345', status='fully-qualified', emoji='🍅', name='tomato', group='Food & Drink', sub_group='food-fruit'),
    '🫒': EmojiEntry(codepoint='1FAD2', status='fully-qualified', emoji='🫒', name='olive', group='Food & Drink', sub_group='food-fruit'),
    '🥥': EmojiEntry(codepoint='1F965', status='fully-qualified', emoji='🥥', name='coconut', group='Food & Drink', sub_group='food-fruit'),
    '🥑': EmojiEntry(codepoint='1F951', status='fully-qualified', emoji='🥑', name='avocado', group='Food & Drink', sub_group='food-vegetable'),
    '🍆': EmojiEntry(codepoint='1F346', status='fully-qualified', emoji='🍆', name='eggplant', group='Food & Drink', sub_group='food-vegetable'),
    '🥔': EmojiEntry(codepoint='1F954', status='fully-qualified', emoji='🥔', name='potato', group='Food & Drink', sub_group='food-vegetable'),
    '🥕': EmojiEntry(codepoint='1F955', status='fully-qualified', emoji='🥕', name='carrot', group='Food & Drink', sub_group='food-vegetable'),
    '🌽': EmojiEntry(codepoint='1F33D', status='fully-qualified', emoji='🌽', name='ear of corn', group='Food & Drink', sub_group='food-vegetable'),
    '🌶️': EmojiEntry(codepoint='1F336 FE0F', status='fully-qualified', emoji='🌶️', name='hot pepper', group='Food & Drink', sub_group='food-vegetable'),
    '🌶': EmojiEntry(codepoint='1F336', status='unqualified', emoji='🌶', name='hot pepper', group='Food & Drink', sub_group='food-vegetable'),
    '🫑': EmojiEntry(codepoint='1FAD1', status='fully-qualified', emoji='🫑', name='bell pepper', group='Food & Drink', sub_group='food-vegetable'),
    '🥒': EmojiEntry(codepoint='1F952', status='fully-qualified', emoji='🥒', name='cucumber', group='Food & Drink', sub_group='food-vegetable'),
    '🥬': EmojiEntry(codepoint='1F96C', status='fully-qualified', emoji='🥬', name='leafy green', group='Food & Drink', sub_group='food-vegetable'),
    '🥦': EmojiEntry(codepoint='1F966', status='fully-qualified', emoji='🥦', name='broccoli', group='Food & Drink', sub_group='food-vegetable'),
    '🧄': EmojiEntry(codepoint='1F9C4', status='fully-qualified', emoji='🧄', name='garlic', group='Food & Drink', sub_group='food-vegetable'),
    '🧅': EmojiEntry(codepoint='1F9C5', status='fully-qualified', emoji='🧅', name='onion', group='Food & Drink', sub_group='food-vegetable'),
    '🍄': EmojiEntry(codepoint='1F344', status='fully-qualified', emoji='🍄', name='mushroom', group='Food & Drink', sub_group='food-vegetable'),
    '🥜': EmojiEntry(codepoint='1F95C', status='fully-qualified', emoji='🥜', name='peanuts', group='Food & Drink', sub_group='food-vegetable'),
    '🌰': EmojiEntry(codepoint='1F330', status='fully-qualified', emoji='🌰', name='chestnut', group='Food & Drink', sub_group='food-vegetable'),
    '🍞': EmojiEntry(codepoint='1F35E', status='fully-qualified', emoji='🍞', name='bread', group='Food & Drink', sub_group='food-prepared'),
    '🥐': EmojiEntry(codepoint='1F950', status='fully-qualified', emoji='🥐', name='croissant', group='Food & Drink', sub_group='food-prepared'),
    '🥖': EmojiEntry(codepoint='1F956', status='fully-qualified', emoji='🥖', name='baguette bread', group='Food & Drink', sub_group='food-prepared'),
    '🫓': EmojiEntry(codepoint='1FAD3', status='fully-qualified', emoji='🫓', name='flatbread', group='Food & Drink', sub_group='food-prepared'),
    '🥨': EmojiEntry(codepoint='1F968', status='fully-qualified', emoji='🥨', name='pretzel', group='Food & Drink', sub_group='food-prepared'),
    '🥯': EmojiEntry(codepoint='1F96F', status='fully-qualified', emoji='🥯', name='bagel', group='Food & Drink', sub_group='food-prepared'),
    '🥞': EmojiEntry(codepoint='1F95E', status='fully-qualified', emoji='🥞', name='pancakes', group='Food & Drink', sub_group='food-prepared'),
    '🧇': EmojiEntry(codepoint='1F9C7', status='fully-qualified', emoji='🧇', name='waffle', group='Food & Drink', sub_group='food-prepared'),
    '🧀': EmojiEntry(codepoint='1F9C0', status='fully-qualified', emoji='🧀', name='cheese wedge', group='Food & Drink', sub_group='food-prepared'),
    '🍖': EmojiEntry(codepoint='1F356', status='fully-qualified', emoji='🍖', name='meat on bone', group='Food & Drink', sub_group='food-prepared'),
    '🍗': EmojiEntry(codepoint='1F357', status='fully-qualified', emoji='🍗', name='poultry leg', group='Food & Drink', sub_group='food-prepared'),
    '🥩': EmojiEntry(codepoint='1F969', status='fully-qualified', emoji='🥩', name='cut of meat', group='Food & Drink', sub_group='food-prepared'),
    '🥓': EmojiEntry(codepoint='1F953', status='fully-qualified', emoji='🥓', name='bacon', group='Food & Drink', sub_group='food-prepared'),
    '🍔': EmojiEntry(codepoint='1F354', status='fully-qualified', emoji='🍔', name='hamburger', group='Food & Drink', sub_group='food-prepared'),
    '🍟': EmojiEntry(codepoint='1F35F', status='fully-qualified', emoji='🍟', name='french fries', group='Food & Drink', sub_group='food-prepared'),
    '🍕': EmojiEntry(codepoint='1F355', status='fully-qualified', emoji='🍕', name='pizza', group='Food & Drink', sub_group='food-prepared'),
    '🌭': EmojiEntry(codepoint='1F32D', status='fully-qualified', emoji='🌭', name='hot dog', group='Food & Drink', sub_group='food-prepared'),
    '🥪': EmojiEntry(codepoint='1F96A', status='fully-qualified', emoji='🥪', name='sandwich', group='Food & Drink', sub_group='food-prepared'),
    '🌮': EmojiEntry(codepoint='1F32E', status='fully-qualified', emoji='🌮', name='taco', group='Food & Drink', sub_group='food-prepared'),
    '🌯': EmojiEntry(codepoint='1F32F', status='fully-qualified', emoji='🌯', name='burrito', group='Food & Drink', sub_group='food-prepared'),
    '🫔': EmojiEntry(codepoint='1FAD4', status='fully-qualified', emoji='🫔', name='tamale', group='Food & Drink', sub_group='food-prepared'),
    '🥙': EmojiEntry(codepoint='1F959', status='fully-qualified', emoji='🥙', name='stuffed flatbread', group='Food & Drink', sub_group='food-prepared'),
    '🧆': EmojiEntry(codepoint='1F9C6', status='fully-qualified', emoji='🧆', name='falafel', group='Food & Drink', sub_group='food-prepared'),
    '🥚': EmojiEntry(codepoint='1F95A', status='fully-qualified', emoji='🥚', name='egg', group='Food & Drink', sub_group='food-prepared'),
    '🍳': EmojiEntry(codepoint='1F373', status='fully-qualified', emoji='🍳', name='cooking', group='Food & Drink', sub_group='food-prepared'),
    '🥘': EmojiEntry(codepoint='1F958', status='fully-qualified', emoji='🥘', name='shallow pan of food', group='Food & Drink', sub_group='food-prepared'),
    '🍲': EmojiEntry(codepoint='1F372', status='fully-qualified', emoji='🍲', name='pot of food', group='Food & Drink', sub_group='food-prepared'),
    '🫕': EmojiEntry(codepoint='1FAD5', status='fully-qualified', emoji='🫕', name='fondue', group='Food & Drink', sub_group='food-prepared'),
    '🥣': EmojiEntry(codepoint='1F963', status='fully-qualified', emoji='🥣', name='bowl with spoon', group='Food & Drink', sub_group='food-prepared'),
    '🥗': EmojiEntry(codepoint='1F957', status='fully-qualified', emoji='🥗', name='green salad', group='Food & Drink', sub_group='food-prepared'),
    '🍿': EmojiEntry(codepoint='1F37F', status='fully-qualified', emoji='🍿', name='popcorn', group='Food & Drink', sub_group='food-prepared'),
    '🧈': EmojiEntry(codepoint='1F9C8', status='fully-qualified', emoji='🧈', name='butter', group='Food & Drink', sub_group='food-prepared'),
    '🧂': EmojiEntry(codepoint='1F9C2', status='fully-qualified', emoji='🧂', name='salt', group='Food & Drink', sub_group='food-prepared'),
    '🥫': EmojiEntry(codepoint='1F96B', status='fully-qualified', emoji='🥫', name='canned food', group='Food & Drink', sub_group='food-prepared'),
    '🍱': EmojiEntry(codepoint='1F371', status='fully-qualified', emoji='🍱', name='bento box', group='Food & Drink', sub_group='food-asian'),
    '🍘': EmojiEntry(codepoint='1F358', status='fully-qualified', emoji='🍘', name='rice cracker', group='Food & Drink', sub_group='food-asian'),
    '🍙': EmojiEntry(codepoint='1F359', status='fully-qualified', emoji='🍙', name='rice ball', group='Food & Drink', sub_group='food-asian'),
    '🍚': EmojiEntry(codepoint='1F35A', status='fully-qualified', emoji='🍚', name='cooked rice', group='Food & Drink', sub_group='food-asian'),
    '🍛': EmojiEntry(codepoint='1F35B', status='fully-qualified', emoji='🍛', name='curry rice', group='Food & Drink', sub_group='food-asian'),
    '🍜': EmojiEntry(codepoint='1F35C', status='fully-qualified', emoji='🍜', name='steaming bowl', group='Food & Drink', sub_group='food-asian'),
    '🍝': EmojiEntry(codepoint='1F35D', status='fully-qualified', emoji='🍝', name='spaghetti', group='Food & Drink', sub_group='food-asian'),
    '🍠': EmojiEntry(codepoint='1F360', status='fully-qualified', emoji='🍠', name='roasted sweet potato', group='Food & Drink', sub_group='food-asian'),
    '🍢': EmojiEntry(codepoint='1F362', status='fully-qualified', emoji='🍢', name='oden', group='Food & Drink', sub_group='food-asian'),
    '🍣': EmojiEntry(codepoint='1F363', status='fully-qualified', emoji='🍣', name='sushi', group='Food & Drink', sub_group='food-asian'),
    '🍤': EmojiEntry(codepoint='1F364', status='fully-qualified', emoji='🍤', name='fried shrimp', group='Food & Drink', sub_group='food-asian'),
    '🍥': EmojiEntry(codepoint='1F365', status='fully-qualified', emoji='🍥', name='fish cake with swirl', group='Food & Drink', sub_group='food-asian'),
    '🥮': EmojiEntry(codepoint='1F96E', status='fully-qualified', emoji='🥮', name='moon cake', group='Food & Drink', sub_group='food-asian'),
    '🍡': EmojiEntry(codepoint='1F361', status='fully-qualified', emoji='🍡', name='dango', group='Food & Drink', sub_group='food-asian'),
    '🥟': EmojiEntry(codepoint='1F95F', status='fully-qualified', emoji='🥟', name='dumpling', group='Food & Drink', sub_group='food-asian'),
    '🥠': EmojiEntry(codepoint='1F960', status='fully-qualified', emoji='🥠', name='fortune cookie', group='Food & Drink', sub_group='food-asian'),
    '🥡': EmojiEntry(codepoint='1F961', status='fully-qualified', emoji='🥡', name='takeout box', group='Food & Drink', sub_group='food-asian'),
    '🦀': EmojiEntry(codepoint='1F980', status='fully-qualified', emoji='🦀', name='crab', group='Food & Drink', sub_group='food-marine'),
    '🦞': EmojiEntry(codepoint='1F99E', status='fully-qualified', emoji='🦞', name='lobster', group='Food & Drink', sub_group='food-marine'),
    '🦐': EmojiEntry(codepoint='1F990', status='fully-qualified', emoji='🦐', name='shrimp', group='Food & Drink', sub_group='food-marine'),
    '🦑': EmojiEntry(codepoint='1F991', status='fully-qualified', emoji='🦑', name='squid', group='Food & Drink', sub_group='food-marine'),
    '🦪': EmojiEntry(codepoint='1F9AA', status='fully-qualified', emoji='🦪', name='oyster', group='Food & Drink', sub_group='food-marine'),
    '🍦': EmojiEntry(codepoint='1F366', status='fully-qualified', emoji='🍦', name='soft ice cream', group='Food & Drink', sub_group='food-sweet'),
    '🍧': EmojiEntry(codepoint='1F367', status='fully-qualified', emoji='🍧', name='shaved ice', group='Food & Drink', sub_group='food-sweet'),
    '🍨': EmojiEntry(codepoint='1F368', status='fully-qualified', emoji='🍨', name='ice cream', group='Food & Drink', sub_group='food-sweet'),
    '🍩': EmojiEntry(codepoint='1F369', status='fully-qualified', emoji='🍩', name='doughnut', group='Food & Drink', sub_group='food-sweet'),
    '🍪': EmojiEntry(codepoint='1F36A', status='fully-qualified', emoji='🍪', name='cookie', group='Food & Drink', sub_group='food-sweet'),
    '🎂': EmojiEntry(codepoint='1F382', status='fully-qualified', emoji='🎂', name='birthday cake', group='Food & Drink', sub_group='food-sweet'),
    '🍰': EmojiEntry(codepoint='1F370', status='fully-qualified', emoji='🍰', name='shortcake', group='Food & Drink', sub_group='food-sweet'),
    '🧁': EmojiEntry(codepoint='1F9C1', status='fully-qualified', emoji='🧁', name='cupcake', group='Food & Drink', sub_group='food-sweet'),
    '🥧': EmojiEntry(codepoint='1F967', status='fully-qualified', emoji='🥧', name='pie', group='Food & Drink', sub_group='food-sweet'),
    '🍫': EmojiEntry(codepoint='1F36B', status='fully-qualified', emoji='🍫', name='chocolate bar', group='Food & Drink', sub_group='food-sweet'),
    '🍬': EmojiEntry(codepoint='1F36C', status='fully-qualified', emoji='🍬', name='candy', group='Food & Drink', sub_group='food-sweet'),
    '🍭': EmojiEntry(codepoint='1F36D', status='fully-qualified', emoji='🍭', name='lollipop', group='Food & Drink', sub_group='food-sweet'),
    '🍮': EmojiEntry(codepoint='1F36E', status='fully-qualified', emoji='🍮', name='custard', group='Food & Drink', sub_group='food-sweet'),
    '🍯': EmojiEntry(codepoint='1F36F', status='fully-qualified', emoji='🍯', name='honey pot', group='Food & Drink', sub_group='food-sweet'),
    '🍼': EmojiEntry(codepoint='1F37C', status='fully-qualified', emoji='🍼', name='baby bottle', group='Food & Drink', sub_group='drink'),
    '🥛': EmojiEntry(codepoint='1F95B', status='fully-qualified', emoji='🥛', name='glass of milk', group='Food & Drink', sub_group='drink'),
    '☕': EmojiEntry(codepoint='2615', status='fully-qualified', emoji='☕', name='hot beverage', group='Food & Drink', sub_group='drink'),
    '🫖': EmojiEntry(codepoint='1FAD6', status='fully-qualified', emoji='🫖', name='teapot', group='Food & Drink', sub_group='drink'),
    '🍵': EmojiEntry(codepoint='1F375', status='fully-qualified', emoji='🍵', name='teacup without handle', group='Food & Drink', sub_group='drink'),
    '🍶': EmojiEntry(codepoint='1F376', status='fully-qualified', emoji='🍶', name='sake', group='Food & Drink', sub_group='drink'),
    '🍾': EmojiEntry(codepoint='1F37E', status='fully-qualified', emoji='🍾', name='bottle with popping cork', group='Food & Drink', sub_group='drink'),
    '🍷': EmojiEntry(codepoint='1F377', status='fully-qualified', emoji='🍷', name='wine glass', group='Food & Drink', sub_group='drink'),
    '🍸': EmojiEntry(codepoint='1F378', status='fully-qualified', emoji='🍸', name='cocktail glass', group='Food & Drink', sub_group='drink'),
    '🍹': EmojiEntry(codepoint='1F379', status='fully-qualified', emoji='🍹', name='tropical drink', group='Food & Drink', sub_group='drink'),
    '🍺': EmojiEntry(codepoint='1F37A', status='fully-qualified', emoji='🍺', name='beer mug', group='Food & Drink', sub_group='drink'),
    '🍻': EmojiEntry(codepoint='1F37B', status='fully-qualified', emoji='🍻', name='clinking beer mugs', group='Food & Drink', sub_group='drink'),
    '🥂': EmojiEntry(codepoint='1F942', status='fully-qualified', emoji='🥂', name='clinking glasses', group='Food & Drink', sub_group='drink'),
    '🥃': EmojiEntry(codepoint='1F943', status='fully-qualified', emoji='🥃', name='tumbler glass', group='Food & Drink', sub_group='drink'),
    '🥤': EmojiEntry(codepoint='1F964', status='fully-qualified', emoji='🥤', name='cup with straw', group='Food & Drink', sub_group='drink'),
    '🧋': EmojiEntry(codepoint='1F9CB', status='fully-qualified', emoji='🧋', name='bubble tea', group='Food & Drink', sub_group='drink'),
    '🧃': EmojiEntry(codepoint='1F9C3', status='fully-qualified', emoji='🧃', name='beverage box', group='Food & Drink', sub_group='drink'),
    '🧉': EmojiEntry(codepoint='1F9C9', status='fully-qualified', emoji='🧉', name='mate', group='Food & Drink', sub_group='drink'),
    '🧊': EmojiEntry(codepoint='1F9CA', status='fully-qualified', emoji='🧊', name='ice', group='Food & Drink', sub_group='drink'),
    '🥢': EmojiEntry(codepoint='1F962', status='fully-qualified', emoji='🥢', name='chopsticks', group='Food & Drink', sub_group='dishware'),
    '🍽️': EmojiEntry(codepoint='1F37D FE0F', status='fully-qualified', emoji='🍽️', name='fork and knife with plate', group='Food & Drink', sub_group='dishware'),
    '🍽': EmojiEntry(codepoint='1F37D', status='unqualified', emoji='🍽', name='fork and knife with plate', group='Food & Drink', sub_group='dishware'),
    '🍴': EmojiEntry(codepoint='1F374', status='fully-qualified', emoji='🍴', name='fork and knife', group='Food & Drink', sub_group='dishware'),
    '🥄': EmojiEntry(codepoint='1F944', status='fully-qualified', emoji='🥄', name='spoon', group='Food & Drink', sub_group='dishware'),
    '🔪': EmojiEntry(codepoint='1F52A', status='fully-qualified', emoji='🔪', name='kitchen knife', group='Food & Drink', sub_group='dishware'),
    '🏺': EmojiEntry(codepoint='1F3FA', status='fully-qualified', emoji='🏺', name='amphora', group='Food & Drink', sub_group='dishware'),
    '🌍': EmojiEntry(codepoint='1F30D', status='fully-qualified', emoji='🌍', name='globe showing Europe-Africa', group='Travel & Places', sub_group='place-map'),
    '🌎': EmojiEntry(codepoint='1F30E', status='fully-qualified', emoji='🌎', name='globe showing Americas', group='Travel & Places', sub_group='place-map'),
    '🌏': EmojiEntry(codepoint='1F30F', status='fully-qualified', emoji='🌏', name='globe showing Asia-Australia', group='Travel & Places', sub_group='place-map'),
    '🌐': EmojiEntry(codepoint='1F310', status='fully-qualified', emoji='🌐', name='globe with meridians', group='Travel & Places', sub_group='place-map'),
    '🗺️': EmojiEntry(codepoint='1F5FA FE0F', status='fully-qualified', emoji='🗺️', name='world map', group='Travel & Places', sub_group='place-map'),
    '🗺': EmojiEntry(codepoint='1F5FA', status='unqualified', emoji='🗺', name='world map', group='Travel & Places', sub_group='place-map'),
    '🗾': EmojiEntry(codepoint='1F5FE', status='fully-qualified', emoji='🗾', name='map of Japan', group='Travel & Places', sub_group='place-map'),
    '🧭': EmojiEntry(codepoint='1F9ED', status='fully-qualified', emoji='🧭', name='compass', group='Travel & Places', sub_group='place-map'),
    '🏔️': EmojiEntry(codepoint='1F3D4 FE0F', status='fully-qualified', emoji='🏔️', name='snow-capped mountain', group='Travel & Places', sub_group='place-geographic'),
    '🏔': EmojiEntry(codepoint='1F3D4', status='unqualified', emoji='🏔', name='snow-capped mountain', group='Travel & Places', sub_group='place-geographic'),
    '⛰️': EmojiEntry(codepoint='26F0 FE0F', status='fully-qualified', emoji='⛰️', name='mountain', group='Travel & Places', sub_group='place-geographic'),
    '⛰': EmojiEntry(codepoint='26F0', status='unqualified', emoji='⛰', name='mountain', group='Travel & Places', sub_group='place-geographic'),
    '🌋': EmojiEntry(codepoint='1F30B', status='fully-qualified', emoji='🌋', name='volcano', group='Travel & Places', sub_group='place-geographic'),
    '🗻': EmojiEntry(codepoint='1F5FB', status='fully-qualified', emoji='🗻', name='mount fuji', group='Travel & Places', sub_group='place-geographic'),
    '🏕️': EmojiEntry(codepoint='1F3D5 FE0F', status='fully-qualified', emoji='🏕️', name='camping', group='Travel & Places', sub_group='place-geographic'),
    '🏕': EmojiEntry(codepoint='1F3D5', status='unqualified', emoji='🏕', name='camping', group='Travel & Places', sub_group='place-geographic'),
    '🏖️': EmojiEntry(codepoint='1F3D6 FE0F', status='fully-qualified', emoji='🏖️', name='beach with umbrella', group='Travel & Places', sub_group='place-geographic'),
    '🏖': EmojiEntry(codepoint='1F3D6', status='unqualified', emoji='🏖', name='beach with umbrella', group='Travel & Places', sub_group='place-geographic'),
    '🏜️': EmojiEntry(codepoint='1F3DC FE0F', status='fully-qualified', emoji='🏜️', name='desert', group='Travel & Places', sub_group='place-geographic'),
    '🏜': EmojiEntry(codepoint='1F3DC', status='unqualified', emoji='🏜', name='desert', group='Travel & Places', sub_group='place-geographic'),
    '🏝️': EmojiEntry(codepoint='1F3DD FE0F', status='fully-qualified', emoji='🏝️', name='desert island', group='Travel & Places', sub_group='place-geographic'),
    '🏝': EmojiEntry(codepoint='1F3DD', status='unqualified', emoji='🏝', name='desert island', group='Travel & Places', sub_group='place-geographic'),
    '🏞️': EmojiEntry(codepoint='1F3DE FE0F', status='fully-qualified', emoji='🏞️', name='national park', group='Travel & Places', sub_group='place-geographic'),
    '🏞': EmojiEntry(codepoint='1F3DE', status='unqualified', emoji='🏞', name='national park', group='Travel & Places', sub_group='place-geographic'),
    '🏟️': EmojiEntry(codepoint='1F3DF FE0F', status='fully-qualified', emoji='🏟️', name='stadium', group='Travel & Places', sub_group='place-building'),
    '🏟': EmojiEntry(codepoint='1F3DF', status='unqualified', emoji='🏟', name='stadium', group='Travel & Places', sub_group='place-building'),
    '🏛️': EmojiEntry(codepoint='1F3DB FE0F', status='fully-qualified', emoji='🏛️', name='classical building', group='Travel & Places', sub_group='place-building'),
    '🏛': EmojiEntry(codepoint='1F3DB', status='unqualified', emoji='🏛', name='classical building', group='Travel & Places', sub_group='place-building'),
    '🏗️': EmojiEntry(codepoint='1F3D7 FE0F', status='fully-qualified', emoji='🏗️', name='building construction', group='Travel & Places', sub_group='place-building'),
    '🏗': EmojiEntry(codepoint='1F3D7', status='unqualified', emoji='🏗', name='building construction', group='Travel & Places', sub_group='place-building'),
    '🧱': EmojiEntry(codepoint='1F9F1', status='fully-qualified', emoji='🧱', name='brick', group='Travel & Places', sub_group='place-building'),
    '🪨': EmojiEntry(codepoint='1FAA8', status='fully-qualified', emoji='🪨', name='rock', group='Travel & Places', sub_group='place-building'),
    '🪵': EmojiEntry(codepoint='1FAB5', status='fully-qualified', emoji='🪵', name='wood', group='Travel & Places', sub_group='place-building'),
    '🛖': EmojiEntry(codepoint='1F6D6', status='fully-qualified', emoji='🛖', name='hut', group='Travel & Places', sub_group='place-building'),
    '🏘️': EmojiEntry(codepoint='1F3D8 FE0F', status='fully-qualified', emoji='🏘️', name='houses', group='Travel & Places', sub_group='place-building'),
    '🏘': EmojiEntry(codepoint='1F3D8', status='unqualified', emoji='🏘', name='houses', group='Travel & Places', sub_group='place-building'),
    '🏚️': EmojiEntry(codepoint='1F3DA FE0F', status='fully-qualified', emoji='🏚️', name='derelict house', group='Travel & Places', sub_group='place-building'),
    '🏚': EmojiEntry(codepoint='1F3DA', status='unqualified', emoji='🏚', name='derelict house', group='Travel & Places', sub_group='place-building'),
    '🏠': EmojiEntry(codepoint='1F3E0', status='fully-qualified', emoji='🏠', name='house', group='Travel & Places', sub_group='place-building'),
    '🏡': EmojiEntry(codepoint='1F3E1', status='fully-qualified', emoji='🏡', name='house with garden', group='Travel & Places', sub_group='place-building'),
    '🏢': EmojiEntry(codepoint='1F3E2', status='fully-qualified', emoji='🏢', name='office building', group='Travel & Places', sub_group='place-building'),
    '🏣': EmojiEntry(codepoint='1F3E3', status='fully-qualified', emoji='🏣', name='Japanese post office', group='Travel & Places', sub_group='place-building'),
    '🏤': EmojiEntry(codepoint='1F3E4', status='fully-qualified', emoji='🏤', name='post office', group='Travel & Places', sub_group='place-building'),
    '🏥': EmojiEntry(codepoint='1F3E5', status='fully-qualified', emoji='🏥', name='hospital', group='Travel & Places', sub_group='place-building'),
    '🏦': EmojiEntry(codepoint='1F3E6', status='fully-qualified', emoji='🏦', name='bank', group='Travel & Places', sub_group='place-building'),
    '🏨': EmojiEntry(codepoint='1F3E8', status='fully-qualified', emoji='🏨', name='hotel', group='Travel & Places', sub_group='place-building'),
    '🏩': EmojiEntry(codepoint='1F3E9', status='fully-qualified', emoji='🏩', name='love hotel', group='Travel & Places', sub_group='place-building'),
    '🏪': EmojiEntry(codepoint='1F3EA', status='fully-qualified', emoji='🏪', name='convenience store', group='Travel & Places', sub_group='place-building'),
    '🏫': EmojiEntry(codepoint='1F3EB', status='fully-qualified', emoji='🏫', name='school', group='Travel & Places', sub_group='place-building'),
    '🏬': EmojiEntry(codepoint='1F3EC', status='fully-qualified', emoji='🏬', name='department store', group='Travel & Places', sub_group='place-building'),
    '🏭': EmojiEntry(codepoint='1F3ED', status='fully-qualified', emoji='🏭', name='factory', group='Travel & Places', sub_group='place-building'),
    '🏯': EmojiEntry(codepoint='1F3EF', status='fully-qualified', emoji='🏯', name='Japanese castle', group='Travel & Places', sub_group='place-building'),
    '🏰': EmojiEntry(codepoint='1F3F0', status='fully-qualified', emoji='🏰', name='castle', group='Travel & Places', sub_group='place-building'),
    '💒': EmojiEntry(codepoint='1F492', status='fully-qualified', emoji='💒', name='wedding', group='Travel & Places', sub_group='place-building'),
    '🗼': EmojiEntry(codepoint='1F5FC', status='fully-qualified', emoji='🗼', name='Tokyo tower', group='Travel & Places', sub_group='place-building'),
    '🗽': EmojiEntry(codepoint='1F5FD', status='fully-qualified', emoji='🗽', name='Statue of Liberty', group='Travel & Places', sub_group='place-building'),
    '⛪': EmojiEntry(codepoint='26EA', status='fully-qualified', emoji='⛪', name='church', group='Travel & Places', sub_group='place-religious'),
    '🕌': EmojiEntry(codepoint='1F54C', status='fully-qualified', emoji='🕌', name='mosque', group='Travel & Places', sub_group='place-religious'),
    '🛕': EmojiEntry(codepoint='1F6D5', status='fully-qualified', emoji='🛕', name='hindu temple', group='Travel & Places', sub_group='place-religious'),
    '🕍': EmojiEntry(codepoint='1F54D', status='fully-qualified', emoji='🕍', name='synagogue', group='Travel & Places', sub_group='place-religious'),
    '⛩️': EmojiEntry(codepoint='26E9 FE0F', status='fully-qualified', emoji='⛩️', name='shinto shrine', group='Travel & Places', sub_group='place-religious'),
    '⛩': EmojiEntry(codepoint='26E9', status='unqualified', emoji='⛩', name='shinto shrine', group='Travel & Places', sub_group='place-religious'),
    '🕋': EmojiEntry(codepoint='1F54B', status='fully-qualified', emoji='🕋', name='kaaba', group='Travel & Places', sub_group='place-religious'),
    '⛲': EmojiEntry(codepoint='26F2', status='fully-qualified', emoji='⛲', name='fountain', group='Travel & Places', sub_group='place-other'),
    '⛺': EmojiEntry(codepoint='26FA', status='fully-qualified', emoji='⛺', name='tent', group='Travel & Places', sub_group='place-other'),
    '🌁': EmojiEntry(codepoint='1F301', status='fully-qualified', emoji='🌁', name='foggy', group='Travel & Places', sub_group='place-other'),
    '🌃': EmojiEntry(codepoint='1F303', status='fully-qualified', emoji='🌃', name='night with stars', group='Travel & Places', sub_group='place-other'),
    '🏙️': EmojiEntry(codepoint='1F3D9 FE0F', status='fully-qualified', emoji='🏙️', name='cityscape', group='Travel & Places', sub_group='place-other'),
    '🏙': EmojiEntry(codepoint='1F3D9', status='unqualified', emoji='🏙', name='cityscape', group='Travel & Places', sub_group='place-other'),
    '🌄': EmojiEntry(codepoint='1F304', status='fully-qualified', emoji='🌄', name='sunrise over mountains', group='Travel & Places', sub_group='place-other'),
    '🌅': EmojiEntry(codepoint='1F305', status='fully-qualified', emoji='🌅', name='sunrise', group='Travel & Places', sub_group='place-other'),
    '🌆': EmojiEntry(codepoint='1F306', status='fully-qualified', emoji='🌆', name='cityscape at dusk', group='Travel & Places', sub_group='place-other'),
    '🌇': EmojiEntry(codepoint='1F307', status='fully-qualified', emoji='🌇', name='sunset', group='Travel & Places', sub_group='place-other'),
    '🌉': EmojiEntry(codepoint='1F309', status='fully-qualified', emoji='🌉', name='bridge at night', group='Travel & Places', sub_group='place-other'),
    '♨️': EmojiEntry(codepoint='2668 FE0F', status='fully-qualified', emoji='♨️', name='hot springs', group='Travel & Places', sub_group='place-other'),
    '♨': EmojiEntry(codepoint='2668', status='unqualified', emoji='♨', name='hot springs', group='Travel & Places', sub_group='place-other'),
    '🎠': EmojiEntry(codepoint='1F3A0', status='fully-qualified', emoji='🎠', name='carousel horse', group='Travel & Places', sub_group='place-other'),
    '🎡': EmojiEntry(codepoint='1F3A1', status='fully-qualified', emoji='🎡', name='ferris wheel', group='Travel & Places', sub_group='place-other'),
    '🎢': EmojiEntry(codepoint='1F3A2', status='fully-qualified', emoji='🎢', name='roller coaster', group='Travel & Places', sub_group='place-other'),
    '💈': EmojiEntry(codepoint='1F488', status='fully-qualified', emoji='💈', name='barber pole', group='Travel & Places', sub_group='place-other'),
    '🎪': EmojiEntry(codepoint='1F3AA', status='fully-qualified', emoji='🎪', name='circus tent', group='Travel & Places', sub_group='place-other'),
    '🚂': EmojiEntry(codepoint='1F682', status='fully-qualified', emoji='🚂', name='locomotive', group='Travel & Places', sub_group='transport-ground'),
    '🚃': EmojiEntry(codepoint='1F683', status='fully-qualified', emoji='🚃', name='railway car', group='Travel & Places', sub_group='transport-ground'),
    '🚄': EmojiEntry(codepoint='1F684', status='fully-qualified', emoji='🚄', name='high-speed train', group='Travel & Places', sub_group='transport-ground'),
    '🚅': EmojiEntry(codepoint='1F685', status='fully-qualified', emoji='🚅', name='bullet train', group='Travel & Places', sub_group='transport-ground'),
    '🚆': EmojiEntry(codepoint='1F686', status='fully-qualified', emoji='🚆', name='train', group='Travel & Places', sub_group='transport-ground'),
    '🚇': EmojiEntry(codepoint='1F687', status='fully-qualified', emoji='🚇', name='metro', group='Travel & Places', sub_group='transport-ground'),
    '🚈': EmojiEntry(codepoint='1F688', status='fully-qualified', emoji='🚈', name='light rail', group='Travel & Places', sub_group='transport-ground'),
    '🚉': EmojiEntry(codepoint='1F689', status='fully-qualified', emoji='🚉', name='station', group='Travel & Places', sub_group='transport-ground'),
    '🚊': EmojiEntry(codepoint='1F68A', status='fully-qualified', emoji='🚊', name='tram', group='Travel & Places', sub_group='transport-ground'),
    '🚝': EmojiEntry(codepoint='1F69D', status='fully-qualified', emoji='🚝', name='monorail', group='Travel & Places', sub_group='transport-ground'),
    '🚞': EmojiEntry(codepoint='1F69E', status='fully-qualified', emoji='🚞', name='mountain railway', group='Travel & Places', sub_group='transport-ground'),
    '🚋': EmojiEntry(codepoint='1F68B', status='fully-qualified', emoji='🚋', name='tram car', group='Travel & Places', sub_group='transport-ground'),
    '🚌': EmojiEntry(codepoint='1F68C', status='fully-qualified', emoji='🚌', name='bus', group='Travel & Places', sub_group='transport-ground'),
    '🚍': EmojiEntry(codepoint='1F68D', status='fully-qualified', emoji='🚍', name='oncoming bus', group='Travel & Places', sub_group='transport-ground'),
    '🚎': EmojiEntry(codepoint='1F68E', status='fully-qualified', emoji='🚎', name='trolleybus', group='Travel & Places', sub_group='transport-ground'),
    '🚐': EmojiEntry(codepoint='1F690', status='fully-qualified', emoji='🚐', name='minibus', group='Travel & Places', sub_group='transport-ground'),
    '🚑': EmojiEntry(codepoint='1F691', status='fully-qualified', emoji='🚑', name='ambulance', group='Travel & Places', sub_group='transport-ground'),
    '🚒': EmojiEntry(codepoint='1F692', status='fully-qualified', emoji='🚒', name='fire engine', group='Travel & Places', sub_group='transport-ground'),
    '🚓': EmojiEntry(codepoint='1F693', status='fully-qualified', emoji='🚓', name='police car', group='Travel & Places', sub_group='transport-ground'),
    '🚔': EmojiEntry(codepoint='1F694', status='fully-qualified', emoji='🚔', name='oncoming police car', group='Travel & Places', sub_group='transport-ground'),
    '🚕': EmojiEntry(codepoint='1F695', status='fully-qualified', emoji='🚕', name='taxi', group='Travel & Places', sub_group='transport-ground'),
    '🚖': EmojiEntry(codepoint='1F696', status='fully-qualified', emoji='🚖', name='oncoming taxi', group='Travel & Places', sub_group='transport-ground'),
    '🚗': EmojiEntry(codepoint='1F697', status='fully-qualified', emoji='🚗', name='automobile', group='Travel & Places', sub_group='transport-ground'),
    '🚘': EmojiEntry(codepoint='1F698', status='fully-qualified', emoji='🚘', name='oncoming automobile', group='Travel & Places', sub_group='transport-ground'),
    '🚙': EmojiEntry(codepoint='1F699', status='fully-qualified', emoji='🚙', name='sport utility vehicle', group='Travel & Places', sub_group='transport-ground'),
    '🛻': EmojiEntry(codepoint='1F6FB', status='fully-qualified', emoji='🛻', name='pickup truck', group='Travel & Places', sub_group='transport-ground'),
    '🚚': EmojiEntry(codepoint='1F69A', status='fully-qualified', emoji='🚚', name='delivery truck', group='Travel & Places', sub_group='transport-ground'),
    '🚛': EmojiEntry(codepoint='1F69B', status='fully-qualified', emoji='🚛', name='articulated lorry', group='Travel & Places', sub_group='transport-ground'),
    '🚜': EmojiEntry(codepoint='1F69C', status='fully-qualified', emoji='🚜', name='tractor', group='Travel & Places', sub_group='transport-ground'),
    '🏎️': EmojiEntry(codepoint='1F3CE FE0F', status='fully-qualified', emoji='🏎️', name='racing car', group='Travel & Places', sub_group='transport-ground'),
    '🏎': EmojiEntry(codepoint='1F3CE', status='unqualified', emoji='🏎', name='racing car', group='Travel & Places', sub_group='transport-ground'),
    '🏍️': EmojiEntry(codepoint='1F3CD FE0F', status='fully-qualified', emoji='🏍️', name='motorcycle', group='Travel & Places', sub_group='transport-ground'),
    '🏍': EmojiEntry(codepoint='1F3CD', status='unqualified', emoji='🏍', name='motorcycle', group='Travel & Places', sub_group='transport-ground'),
    '🛵': EmojiEntry(codepoint='1F6F5', status='fully-qualified', emoji='🛵', name='motor scooter', group='Travel & Places', sub_group='transport-ground'),
    '🦽': EmojiEntry(codepoint='1F9BD', status='fully-qualified', emoji='🦽', name='manual wheelchair', group='Travel & Places', sub_group='transport-ground'),
    '🦼': EmojiEntry(codepoint='1F9BC', status='fully-qualified', emoji='🦼', name='motorized wheelchair', group='Travel & Places', sub_group='transport-ground'),
    '🛺': EmojiEntry(codepoint='1F6FA', status='fully-qualified', emoji='🛺', name='auto rickshaw', group='Travel & Places', sub_group='transport-ground'),
    '🚲': EmojiEntry(codepoint='1F6B2', status='fully-qualified', emoji='🚲', name='bicycle', group='Travel & Places', sub_group='transport-ground'),
    '🛴': EmojiEntry(codepoint='1F6F4', status='fully-qualified', emoji='🛴', name='kick scooter', group='Travel & Places', sub_group='transport-ground'),
    '🛹': EmojiEntry(codepoint='1F6F9', status='fully-qualified', emoji='🛹', name='skateboard', group='Travel & Places', sub_group='transport-ground'),
    '🛼': EmojiEntry(codepoint='1F6FC', status='fully-qualified', emoji='🛼', name='roller skate', group='Travel & Places', sub_group='transport-ground'),
    '🚏': EmojiEntry(codepoint='1F68F', status='fully-qualified', emoji='🚏', name='bus stop', group='Travel & Places', sub_group='transport-ground'),
    '🛣️': EmojiEntry(codepoint='1F6E3 FE0F', status='fully-qualified', emoji='🛣️', name='motorway', group='Travel & Places', sub_group='transport-ground'),
    '🛣': EmojiEntry(codepoint='1F6E3', status='unqualified', emoji='🛣', name='motorway', group='Travel & Places', sub_group='transport-ground'),
    '🛤️': EmojiEntry(codepoint='1F6E4 FE0F', status='fully-qualified', emoji='🛤️', name='railway track', group='Travel & Places', sub_group='transport-ground'),
    '🛤': EmojiEntry(codepoint='1F6E4', status='unqualified', emoji='🛤', name='railway track', group='Travel & Places', sub_group='transport-ground'),
    '🛢️': EmojiEntry(codepoint='1F6E2 FE0F', status='fully-qualified', emoji='🛢️', name='oil drum', group='Travel & Places', sub_group='transport-ground'),
    '🛢': EmojiEntry(codepoint='1F6E2', status='unqualified', emoji='🛢', name='oil drum', group='Travel & Places', sub_group='transport-ground'),
    '⛽': EmojiEntry(codepoint='26FD', status='fully-qualified', emoji='⛽', name='fuel pump', group='Travel & Places', sub_group='transport-ground'),
    '🚨': EmojiEntry(codepoint='1F6A8', status='fully-qualified', emoji='🚨', name='police car light', group='Travel & Places', sub_group='transport-ground'),
    '🚥': EmojiEntry(codepoint='1F6A5', status='fully-qualified', emoji='🚥', name='horizontal traffic light', group='Travel & Places', sub_group='transport-ground'),
    '🚦': EmojiEntry(codepoint='1F6A6', status='fully-qualified', emoji='🚦', name='vertical traffic light', group='Travel & Places', sub_group='transport-ground'),
    '🛑': EmojiEntry(codepoint='1F6D1', status='fully-qualified', emoji='🛑', name='stop sign', group='Travel & Places', sub_group='transport-ground'),
    '🚧': EmojiEntry(codepoint='1F6A7', status='fully-qualified', emoji='🚧', name='construction', group='Travel & Places', sub_group='transport-ground'),
    '⚓': EmojiEntry(codepoint='2693', status='fully-qualified', emoji='⚓', name='anchor', group='Travel & Places', sub_group='transport-water'),
    '⛵': EmojiEntry(codepoint='26F5', status='fully-qualified', emoji='⛵', name='sailboat', group='Travel & Places', sub_group='transport-water'),
    '🛶': EmojiEntry(codepoint='1F6F6', status='fully-qualified', emoji='🛶', name='canoe', group='Travel & Places', sub_group='transport-water'),
    '🚤': EmojiEntry(codepoint='1F6A4', status='fully-qualified', emoji='🚤', name='speedboat', group='Travel & Places', sub_group='transport-water'),
    '🛳️': EmojiEntry(codepoint='1F6F3 FE0F', status='fully-qualified', emoji='🛳️', name='passenger ship', group='Travel & Places', sub_group='transport-water'),
    '🛳': EmojiEntry(codepoint='1F6F3', status='unqualified', emoji='🛳', name='passenger ship', group='Travel & Places', sub_group='transport-water'),
    '⛴️': EmojiEntry(codepoint='26F4 FE0F', status='fully-qualified', emoji='⛴️', name='ferry', group='Travel & Places', sub_group='transport-water'),
    '⛴': EmojiEntry(codepoint='26F4', status='unqualified', emoji='⛴', name='ferry', group='Travel & Places', sub_group='transport-water'),
    '🛥️': EmojiEntry(codepoint='1F6E5 FE0F', status='fully-qualified', emoji='🛥️', name='motor boat', group='Travel & Places', sub_group='transport-water'),
    '🛥': EmojiEntry(codepoint='1F6E5', status='unqualified', emoji='🛥', name='motor boat', group='Travel & Places', sub_group='transport-water'),
    '🚢': EmojiEntry(codepoint='1F6A2', status='fully-qualified', emoji='🚢', name='ship', group='Travel & Places', sub_group='transport-water'),
    '✈️': EmojiEntry(codepoint='2708 FE0F', status='fully-qualified', emoji='✈️', name='airplane', group='Travel & Places', sub_group='transport-air'),
    '✈': EmojiEntry(codepoint='2708', status='unqualified', emoji='✈', name='airplane', group='Travel & Places', sub_group='transport-air'),
    '🛩️': EmojiEntry(codepoint='1F6E9 FE0F', status='fully-qualified', emoji='🛩️', name='small airplane', group='Travel & Places', sub_group='transport-air'),
    '🛩': EmojiEntry(codepoint='1F6E9', status='unqualified', emoji='🛩', name='small airplane', group='Travel & Places', sub_group='transport-air'),
    '🛫': EmojiEntry(codepoint='1F6EB', status='fully-qualified', emoji='🛫', name='airplane departure', group='Travel & Places', sub_group='transport-air'),
    '🛬': EmojiEntry(codepoint='1F6EC', status='fully-qualified', emoji='🛬', name='airplane arrival', group='Travel & Places', sub_group='transport-air'),
    '🪂': EmojiEntry(codepoint='1FA82', status='fully-qualified', emoji='🪂', name='parachute', group='Travel & Places', sub_group='transport-air'),
    '💺': EmojiEntry(codepoint='1F4BA', status='fully-qualified', emoji='💺', name='seat', group='Travel & Places', sub_group='transport-air'),
    '🚁': EmojiEntry(codepoint='1F681', status='fully-qualified', emoji='🚁', name='helicopter', group='Travel & Places', sub_group='transport-air'),
    '🚟': EmojiEntry(codepoint='1F69F', status='fully-qualified', emoji='🚟', name='suspension railway', group='Travel & Places', sub_group='transport-air'),
    '🚠': EmojiEntry(codepoint='1F6A0', status='fully-qualified', emoji='🚠', name='mountain cableway', group='Travel & Places', sub_group='transport-air'),
    '🚡': EmojiEntry(codepoint='1F6A1', status='fully-qualified', emoji='🚡', name='aerial tramway', group='Travel & Places', sub_group='transport-air'),
    '🛰️': EmojiEntry(codepoint='1F6F0 FE0F', status='fully-qualified', emoji='🛰️', name='satellite', group='Travel & Places', sub_group='transport-air'),
    '🛰': EmojiEntry(codepoint='1F6F0', status='unqualified', emoji='🛰', name='satellite', group='Travel & Places', sub_group='transport-air'),
    '🚀': EmojiEntry(codepoint='1F680', status='fully-qualified', emoji='🚀', name='rocket', group='Travel & Places', sub_group='transport-air'),
    '🛸': EmojiEntry(codepoint='1F6F8', status='fully-qualified', emoji='🛸', name='flying saucer', group='Travel & Places', sub_group='transport-air'),
    '🛎️': EmojiEntry(codepoint='1F6CE FE0F', status='fully-qualified', emoji='🛎️', name='bellhop bell', group='Travel & Places', sub_group='hotel'),
    '🛎': EmojiEntry(codepoint='1F6CE', status='unqualified', emoji='🛎', name='bellhop bell', group='Travel & Places', sub_group='hotel'),
    '🧳': EmojiEntry(codepoint='1F9F3', status='fully-qualified', emoji='🧳', name='luggage', group='Travel & Places', sub_group='hotel'),
    '⌛': EmojiEntry(codepoint='231B', status='fully-qualified', emoji='⌛', name='hourglass done', group='Travel & Places', sub_group='time'),
    '⏳': EmojiEntry(codepoint='23F3', status='fully-qualified', emoji='⏳', name='hourglass not done', group='Travel & Places', sub_group='time'),
    '⌚': EmojiEntry(codepoint='231A', status='fully-qualified', emoji='⌚', name='watch', group='Travel & Places', sub_group='time'),
    '⏰': EmojiEntry(codepoint='23F0', status='fully-qualified', emoji='⏰', name='alarm clock', group='Travel & Places', sub_group='time'),
    '⏱️': EmojiEntry(codepoint='23F1 FE0F', status='fully-qualified', emoji='⏱️', name='stopwatch', group='Travel & Places', sub_group='time'),
    '⏱': EmojiEntry(codepoint='23F1', status='unqualified', emoji='⏱', name='stopwatch', group='Travel & Places', sub_group='time'),
    '⏲️': EmojiEntry(codepoint='23F2 FE0F', status='fully-qualified', emoji='⏲️', name='timer clock', group='Travel & Places', sub_group='time'),
    '⏲': EmojiEntry(codepoint='23F2', status='unqualified', emoji='⏲', name='timer clock', group='Travel & Places', sub_group='time'),
    '🕰️': EmojiEntry(codepoint='1F570 FE0F', status='fully-qualified', emoji='🕰️', name='mantelpiece clock', group='Travel & Places', sub_group='time'),
    '🕰': EmojiEntry(codepoint='1F570', status='unqualified', emoji='🕰', name='mantelpiece clock', group='Travel & Places', sub_group='time'),
    '🕛': EmojiEntry(codepoint='1F55B', status='fully-qualified', emoji='🕛', name='twelve o’clock', group='Travel & Places', sub_group='time'),
    '🕧': EmojiEntry(codepoint='1F567', status='fully-qualified', emoji='🕧', name='twelve-thirty', group='Travel & Places', sub_group='time'),
    '🕐': EmojiEntry(codepoint='1F550', status='fully-qualified', emoji='🕐', name='one o’clock', group='Travel & Places', sub_group='time'),
    '🕜': EmojiEntry(codepoint='1F55C', status='fully-qualified', emoji='🕜', name='one-thirty', group='Travel & Places', sub_group='time'),
    '🕑': EmojiEntry(codepoint='1F551', status='fully-qualified', emoji='🕑', name='two o’clock', group='Travel & Places', sub_group='time'),
    '🕝': EmojiEntry(codepoint='1F55D', status='fully-qualified', emoji='🕝', name='two-thirty', group='Travel & Places', sub_group='time'),
    '🕒': EmojiEntry(codepoint='1F552', status='fully-qualified', emoji='🕒', name='three o’clock', group='Travel & Places', sub_group='time'),
    '🕞': EmojiEntry(codepoint='1F55E', status='fully-qualified', emoji='🕞', name='three-thirty', group='Travel & Places', sub_group='time'),
    '🕓': EmojiEntry(codepoint='1F553', status='fully-qualified', emoji='🕓', name='four o’clock', group='Travel & Places', sub_group='time'),
    '🕟': EmojiEntry(codepoint='1F55F', status='fully-qualified', emoji='🕟', name='four-thirty', group='Travel & Places', sub_group='time'),
    '🕔': EmojiEntry(codepoint='1F554', status='fully-qualified', emoji='🕔', name='five o’clock', group='Travel & Places', sub_group='time'),
    '🕠': EmojiEntry(codepoint='1F560', status='fully-qualified', emoji='🕠', name='five-thirty', group='Travel & Places', sub_group='time'),
    '🕕': EmojiEntry(codepoint='1F555', status='fully-qualified', emoji='🕕', name='six o’clock', group='Travel & Places', sub_group='time'),
    '🕡': EmojiEntry(codepoint='1F561', status='fully-qualified', emoji='🕡', name='six-thirty', group='Travel & Places', sub_group='time'),
    '🕖': EmojiEntry(codepoint='1F556', status='fully-qualified', emoji='🕖', name='seven o’clock', group='Travel & Places', sub_group='time'),
    '🕢': EmojiEntry(codepoint='1F562', status='fully-qualified', emoji='🕢', name='seven-thirty', group='Travel & Places', sub_group='time'),
    '🕗': EmojiEntry(codepoint='1F557', status='fully-qualified', emoji='🕗', name='eight o’clock', group='Travel & Places', sub_group='time'),
    '🕣': EmojiEntry(codepoint='1F563', status='fully-qualified', emoji='🕣', name='eight-thirty', group='Travel & Places', sub_group='time'),
    '🕘': EmojiEntry(codepoint='1F558', status='fully-qualified', emoji='🕘', name='nine o’clock', group='Travel & Places', sub_group='time'),
    '🕤': EmojiEntry(codepoint='1F564', status='fully-qualified', emoji='🕤', name='nine-thirty', group='Travel & Places', sub_group='time'),
    '🕙': EmojiEntry(codepoint='1F559', status='fully-qualified', emoji='🕙', name='ten o’clock', group='Travel & Places', sub_group='time'),
    '🕥': EmojiEntry(codepoint='1F565', status='fully-qualified', emoji='🕥', name='ten-thirty', group='Travel & Places', sub_group='time'),
    '🕚': EmojiEntry(codepoint='1F55A', status='fully-qualified', emoji='🕚', name='eleven o’clock', group='Travel & Places', sub_group='time'),
    '🕦': EmojiEntry(codepoint='1F566', status='fully-qualified', emoji='🕦', name='eleven-thirty', group='Travel & Places', sub_group='time'),
    '🌑': EmojiEntry(codepoint='1F311', status='fully-qualified', emoji='🌑', name='new moon', group='Travel & Places', sub_group='sky & weather'),
    '🌒': EmojiEntry(codepoint='1F312', status='fully-qualified', emoji='🌒', name='waxing crescent moon', group='Travel & Places', sub_group='sky & weather'),
    '🌓': EmojiEntry(codepoint='1F313', status='fully-qualified', emoji='🌓', name='first quarter moon', group='Travel & Places', sub_group='sky & weather'),
    '🌔': EmojiEntry(codepoint='1F314', status='fully-qualified', emoji='🌔', name='waxing gibbous moon', group='Travel & Places', sub_group='sky & weather'),
    '🌕': EmojiEntry(codepoint='1F315', status='fully-qualified', emoji='🌕', name='full moon', group='Travel & Places', sub_group='sky & weather'),
    '🌖': EmojiEntry(codepoint='1F316', status='fully-qualified', emoji='🌖', name='waning gibbous moon', group='Travel & Places', sub_group='sky & weather'),
    '🌗': EmojiEntry(codepoint='1F317', status='fully-qualified', emoji='🌗', name='last quarter moon', group='Travel & Places', sub_group='sky & weather'),
    '🌘': EmojiEntry(codepoint='1F318', status='fully-qualified', emoji='🌘', name='waning crescent moon', group='Travel & Places', sub_group='sky & weather'),
    '🌙': EmojiEntry(codepoint='1F319', status='fully-qualified', emoji='🌙', name='crescent moon', group='Travel & Places', sub_group='sky & weather'),
    '🌚': EmojiEntry(codepoint='1F31A', status='fully-qualified', emoji='🌚', name='new moon face', group='Travel & Places', sub_group='sky & weather'),
    '🌛': EmojiEntry(codepoint='1F31B', status='fully-qualified', emoji='🌛', name='first quarter moon face', group='Travel & Places', sub_group='sky & weather'),
    '🌜': EmojiEntry(codepoint='1F31C', status='fully-qualified', emoji='🌜', name='last quarter moon face', group='Travel & Places', sub_group='sky & weather'),
    '🌡️': EmojiEntry(codepoint='1F321 FE0F', status='fully-qualified', emoji='🌡️', name='thermometer', group='Travel & Places', sub_group='sky & weather'),
    '🌡': EmojiEntry(codepoint='1F321', status='unqualified', emoji='🌡', name='thermometer', group='Travel & Places', sub_group='sky & weather'),
    '☀️': EmojiEntry(codepoint='2600 FE0F', status='fully-qualified', emoji='☀️', name='sun', group='Travel & Places', sub_group='sky & weather'),
    '☀': EmojiEntry(codepoint='2600', status='unqualified', emoji='☀', name='sun', group='Travel & Places', sub_group='sky & weather'),
    '🌝': EmojiEntry(codepoint='1F31D', status='fully-qualified', emoji='🌝', name='full moon face', group='Travel & Places', sub_group='sky & weather'),
    '🌞': EmojiEntry(codepoint='1F31E', status='fully-qualified', emoji='🌞', name='sun with face', group='Travel & Places', sub_group='sky & weather'),
    '🪐': EmojiEntry(codepoint='1FA90', status='fully-qualified', emoji='🪐', name='ringed planet', group='Travel & Places', sub_group='sky & weather'),
    '⭐': EmojiEntry(codepoint='2B50', status='fully-qualified', emoji='⭐', name='star', group='Travel & Places', sub_group='sky & weather'),
    '🌟': EmojiEntry(codepoint='1F31F', status='fully-qualified', emoji='🌟', name='glowing star', group='Travel & Places', sub_group='sky & weather'),
    '🌠': EmojiEntry(codepoint='1F320', status='fully-qualified', emoji='🌠', name='shooting star', group='Travel & Places', sub_group='sky & weather'),
    '🌌': EmojiEntry(codepoint='1F30C', status='fully-qualified', emoji='🌌', name='milky way', group='Travel & Places', sub_group='sky & weather'),
    '☁️': EmojiEntry(codepoint='2601 FE0F', status='fully-qualified', emoji='☁️', name='cloud', group='Travel & Places', sub_group='sky & weather'),
    '☁': EmojiEntry(codepoint='2601', status='unqualified', emoji='☁', name='cloud', group='Travel & Places', sub_group='sky & weather'),
    '⛅': EmojiEntry(codepoint='26C5', status='fully-qualified', emoji='⛅', name='sun behind cloud', group='Travel & Places', sub_group='sky & weather'),
    '⛈️': EmojiEntry(codepoint='26C8 FE0F', status='fully-qualified', emoji='⛈️', name='cloud with lightning and rain', group='Travel & Places', sub_group='sky & weather'),
    '⛈': EmojiEntry(codepoint='26C8', status='unqualified', emoji='⛈', name='cloud with lightning and rain', group='Travel & Places', sub_group='sky & weather'),
    '🌤️': EmojiEntry(codepoint='1F324 FE0F', status='fully-qualified', emoji='🌤️', name='sun behind small cloud', group='Travel & Places', sub_group='sky & weather'),
    '🌤': EmojiEntry(codepoint='1F324', status='unqualified', emoji='🌤', name='sun behind small cloud', group='Travel & Places', sub_group='sky & weather'),
    '🌥️': EmojiEntry(codepoint='1F325 FE0F', status='fully-qualified', emoji='🌥️', name='sun behind large cloud', group='Travel & Places', sub_group='sky & weather'),
    '🌥': EmojiEntry(codepoint='1F325', status='unqualified', emoji='🌥', name='sun behind large cloud', group='Travel & Places', sub_group='sky & weather'),
    '🌦️': EmojiEntry(codepoint='1F326 FE0F', status='fully-qualified', emoji='🌦️', name='sun behind rain cloud', group='Travel & Places', sub_group='sky & weather'),
    '🌦': EmojiEntry(codepoint='1F326', status='unqualified', emoji='🌦', name='sun behind rain cloud', group='Travel & Places', sub_group='sky & weather'),
    '🌧️': EmojiEntry(codepoint='1F327 FE0F', status='fully-qualified', emoji='🌧️', name='cloud with rain', group='Travel & Places', sub_group='sky & weather'),
    '🌧': EmojiEntry(codepoint='1F327', status='unqualified', emoji='🌧', name='cloud with rain', group='Travel & Places', sub_group='sky & weather'),
    '🌨️': EmojiEntry(codepoint='1F328 FE0F', status='fully-qualified', emoji='🌨️', name='cloud with snow', group='Travel & Places', sub_group='sky & weather'),
    '🌨': EmojiEntry(codepoint='1F328', status='unqualified', emoji='🌨', name='cloud with snow', group='Travel & Places', sub_group='sky & weather'),
    '🌩️': EmojiEntry(codepoint='1F329 FE0F', status='fully-qualified', emoji='🌩️', name='cloud with lightning', group='Travel & Places', sub_group='sky & weather'),
    '🌩': EmojiEntry(codepoint='1F329', status='unqualified', emoji='🌩', name='cloud with lightning', group='Travel & Places', sub_group='sky & weather'),
    '🌪️': EmojiEntry(codepoint='1F32A FE0F', status='fully-qualified', emoji='🌪️', name='tornado', group='Travel & Places', sub_group='sky & weather'),
    '🌪': EmojiEntry(codepoint='1F32A', status='unqualified', emoji='🌪', name='tornado', group='Travel & Places', sub_group='sky & weather'),
    '🌫️': EmojiEntry(codepoint='1F32B FE0F', status='fully-qualified', emoji='🌫️', name='fog', group='Travel & Places', sub_group='sky & weather'),
    '🌫': EmojiEntry(codepoint='1F32B', status='unqualified', emoji='🌫', name='fog', group='Travel & Places', sub_group='sky & weather'),
    '🌬️': EmojiEntry(codepoint='1F32C FE0F', status='fully-qualified', emoji='🌬️', name='wind face', group='Travel & Places', sub_group='sky & weather'),
    '🌬': EmojiEntry(codepoint='1F32C', status='unqualified', emoji='🌬', name='wind face', group='Travel & Places', sub_group='sky & weather'),
    '🌀': EmojiEntry(codepoint='1F300', status='fully-qualified', emoji='🌀', name='cyclone', group='Travel & Places', sub_group='sky & weather'),
    '🌈': EmojiEntry(codepoint='1F308', status='fully-qualified', emoji='🌈', name='rainbow', group='Travel & Places', sub_group='sky & weather'),
    '🌂': EmojiEntry(codepoint='1F302', status='fully-qualified', emoji='🌂', name='closed umbrella', group='Travel & Places', sub_group='sky & weather'),
    '☂️': EmojiEntry(codepoint='2602 FE0F', status='fully-qualified', emoji='☂️', name='umbrella', group='Travel & Places', sub_group='sky & weather'),
    '☂': EmojiEntry(codepoint='2602', status='unqualified', emoji='☂', name='umbrella', group='Travel & Places', sub_group='sky & weather'),
    '☔': EmojiEntry(codepoint='2614', status='fully-qualified', emoji='☔', name='umbrella with rain drops', group='Travel & Places', sub_group='sky & weather'),
    '⛱️': EmojiEntry(codepoint='26F1 FE0F', status='fully-qualified', emoji='⛱️', name='umbrella on ground', group='Travel & Places', sub_group='sky & weather'),
    '⛱': EmojiEntry(codepoint='26F1', status='unqualified', emoji='⛱', name='umbrella on ground', group='Travel & Places', sub_group='sky & weather'),
    '⚡': EmojiEntry(codepoint='26A1', status='fully-qualified', emoji='⚡', name='high voltage', group='Travel & Places', sub_group='sky & weather'),
    '❄️': EmojiEntry(codepoint='2744 FE0F', status='fully-qualified', emoji='❄️', name='snowflake', group='Travel & Places', sub_group='sky & weather'),
    '❄': EmojiEntry(codepoint='2744', status='unqualified', emoji='❄', name='snowflake', group='Travel & Places', sub_group='sky & weather'),
    '☃️': EmojiEntry(codepoint='2603 FE0F', status='fully-qualified', emoji='☃️', name='snowman', group='Travel & Places', sub_group='sky & weather'),
    '☃': EmojiEntry(codepoint='2603', status='unqualified', emoji='☃', name='snowman', group='Travel & Places', sub_group='sky & weather'),
    '⛄': EmojiEntry(codepoint='26C4', status='fully-qualified', emoji='⛄', name='snowman without snow', group='Travel & Places', sub_group='sky & weather'),
    '☄️': EmojiEntry(codepoint='2604 FE0F', status='fully-qualified', emoji='☄️', name='comet', group='Travel & Places', sub_group='sky & weather'),
    '☄': EmojiEntry(codepoint='2604', status='unqualified', emoji='☄', name='comet', group='Travel & Places', sub_group='sky & weather'),
    '🔥': EmojiEntry(codepoint='1F525', status='fully-qualified', emoji='🔥', name='fire', group='Travel & Places', sub_group='sky & weather'),
    '💧': EmojiEntry(codepoint='1F4A7', status='fully-qualified', emoji='💧', name='droplet', group='Travel & Places', sub_group='sky & weather'),
    '🌊': EmojiEntry(codepoint='1F30A', status='fully-qualified', emoji='🌊', name='water wave', group='Travel & Places', sub_group='sky & weather'),
    '🎃': EmojiEntry(codepoint='1F383', status='fully-qualified', emoji='🎃', name='jack-o-lantern', group='Activities', sub_group='event'),
    '🎄': EmojiEntry(codepoint='1F384', status='fully-qualified', emoji='🎄', name='Christmas tree', group='Activities', sub_group='event'),
    '🎆': EmojiEntry(codepoint='1F386', status='fully-qualified', emoji='🎆', name='fireworks', group='Activities', sub_group='event'),
    '🎇': EmojiEntry(codepoint='1F387', status='fully-qualified', emoji='🎇', name='sparkler', group='Activities', sub_group='event'),
    '🧨': EmojiEntry(codepoint='1F9E8', status='fully-qualified', emoji='🧨', name='firecracker', group='Activities', sub_group='event'),
    '✨': EmojiEntry(codepoint='2728', status='fully-qualified', emoji='✨', name='sparkles', group='Activities', sub_group='event'),
    '🎈': EmojiEntry(codepoint='1F388', status='fully-qualified', emoji='🎈', name='balloon', group='Activities', sub_group='event'),
    '🎉': EmojiEntry(codepoint='1F389', status='fully-qualified', emoji='🎉', name='party popper', group='Activities', sub_group='event'),
    '🎊': EmojiEntry(codepoint='1F38A', status='fully-qualified', emoji='🎊', name='confetti ball', group='Activities', sub_group='event'),
    '🎋': EmojiEntry(codepoint='1F38B', status='fully-qualified', emoji='🎋', name='tanabata tree', group='Activities', sub_group='event'),
    '🎍': EmojiEntry(codepoint='1F38D', status='fully-qualified', emoji='🎍', name='pine decoration', group='Activities', sub_group='event'),
    '🎎': EmojiEntry(codepoint='1F38E', status='fully-qualified', emoji='🎎', name='Japanese dolls', group='Activities', sub_group='event'),
    '🎏': EmojiEntry(codepoint='1F38F', status='fully-qualified', emoji='🎏', name='carp streamer', group='Activities', sub_group='event'),
    '🎐': EmojiEntry(codepoint='1F390', status='fully-qualified', emoji='🎐', name='wind chime', group='Activities', sub_group='event'),
    '🎑': EmojiEntry(codepoint='1F391', status='fully-qualified', emoji='🎑', name='moon viewing ceremony', group='Activities', sub_group='event'),
    '🧧': EmojiEntry(codepoint='1F9E7', status='fully-qualified', emoji='🧧', name='red envelope', group='Activities', sub_group='event'),
    '🎀': EmojiEntry(codepoint='1F380', status='fully-qualified', emoji='🎀', name='ribbon', group='Activities', sub_group='event'),
    '🎁': EmojiEntry(codepoint='1F381', status='fully-qualified', emoji='🎁', name='wrapped gift', group='Activities', sub_group='event'),
    '🎗️': EmojiEntry(codepoint='1F397 FE0F', status='fully-qualified', emoji='🎗️', name='reminder ribbon', group='Activities', sub_group='event'),
    '🎗': EmojiEntry(codepoint='1F397', status='unqualified', emoji='🎗', name='reminder ribbon', group='Activities', sub_group='event'),
    '🎟️': EmojiEntry(codepoint='1F39F FE0F', status='fully-qualified', emoji='🎟️', name='admission tickets', group='Activities', sub_group='event'),
    '🎟': EmojiEntry(codepoint='1F39F', status='unqualified', emoji='🎟', name='admission tickets', group='Activities', sub_group='event'),
    '🎫': EmojiEntry(codepoint='1F3AB', status='fully-qualified', emoji='🎫', name='ticket', group='Activities', sub_group='event'),
    '🎖️': EmojiEntry(codepoint='1F396 FE0F', status='fully-qualified', emoji='🎖️', name='military medal', group='Activities', sub_group='award-medal'),
    '🎖': EmojiEntry(codepoint='1F396', status='unqualified', emoji='🎖', name='military medal', group='Activities', sub_group='award-medal'),
    '🏆': EmojiEntry(codepoint='1F3C6', status='fully-qualified', emoji='🏆', name='trophy', group='Activities', sub_group='award-medal'),
    '🏅': EmojiEntry(codepoint='1F3C5', status='fully-qualified', emoji='🏅', name='sports medal', group='Activities', sub_group='award-medal'),
    '🥇': EmojiEntry(codepoint='1F947', status='fully-qualified', emoji='🥇', name='1st place medal', group='Activities', sub_group='award-medal'),
    '🥈': EmojiEntry(codepoint='1F948', status='fully-qualified', emoji='🥈', name='2nd place medal', group='Activities', sub_group='award-medal'),
    '🥉': EmojiEntry(codepoint='1F949', status='fully-qualified', emoji='🥉', name='3rd place medal', group='Activities', sub_group='award-medal'),
    '⚽': EmojiEntry(codepoint='26BD', status='fully-qualified', emoji='⚽', name='soccer ball', group='Activities', sub_group='sport'),
    '⚾': EmojiEntry(codepoint='26BE', status='fully-qualified', emoji='⚾', name='baseball', group='Activities', sub_group='sport'),
    '🥎': EmojiEntry(codepoint='1F94E', status='fully-qualified', emoji='🥎', name='softball', group='Activities', sub_group='sport'),
    '🏀': EmojiEntry(codepoint='1F3C0', status='fully-qualified', emoji='🏀', name='basketball', group='Activities', sub_group='sport'),
    '🏐': EmojiEntry(codepoint='1F3D0', status='fully-qualified', emoji='🏐', name='volleyball', group='Activities', sub_group='sport'),
    '🏈': EmojiEntry(codepoint='1F3C8', status='fully-qualified', emoji='🏈', name='american football', group='Activities', sub_group='sport'),
    '🏉': EmojiEntry(codepoint='1F3C9', status='fully-qualified', emoji='🏉', name='rugby football', group='Activities', sub_group='sport'),
    '🎾': EmojiEntry(codepoint='1F3BE', status='fully-qualified', emoji='🎾', name='tennis', group='Activities', sub_group='sport'),
    '🥏': EmojiEntry(codepoint='1F94F', status='fully-qualified', emoji='🥏', name='flying disc', group='Activities', sub_group='sport'),
    '🎳': EmojiEntry(codepoint='1F3B3', status='fully-qualified', emoji='🎳', name='bowling', group='Activities', sub_group='sport'),
    '🏏': EmojiEntry(codepoint='1F3CF', status='fully-qualified', emoji='🏏', name='cricket game', group='Activities', sub_group='sport'),
    '🏑': EmojiEntry(codepoint='1F3D1', status='fully-qualified', emoji='🏑', name='field hockey', group='Activities', sub_group='sport'),
    '🏒': EmojiEntry(codepoint='1F3D2', status='fully-qualified', emoji='🏒', name='ice hockey', group='Activities', sub_group='sport'),
    '🥍': EmojiEntry(codepoint='1F94D', status='fully-qualified', emoji='🥍', name='lacrosse', group='Activities', sub_group='sport'),
    '🏓': EmojiEntry(codepoint='1F3D3', status='fully-qualified', emoji='🏓', name='ping pong', group='Activities', sub_group='sport'),
    '🏸': EmojiEntry(codepoint='1F3F8', status='fully-qualified', emoji='🏸', name='badminton', group='Activities', sub_group='sport'),
    '🥊': EmojiEntry(codepoint='1F94A', status='fully-qualified', emoji='🥊', name='boxing glove', group='Activities', sub_group='sport'),
    '🥋': EmojiEntry(codepoint='1F94B', status='fully-qualified', emoji='🥋', name='martial arts uniform', group='Activities', sub_group='sport'),
    '🥅': EmojiEntry(codepoint='1F945', status='fully-qualified', emoji='🥅', name='goal net', group='Activities', sub_group='sport'),
    '⛳': EmojiEntry(codepoint='26F3', status='fully-qualified', emoji='⛳', name='flag in hole', group='Activities', sub_group='sport'),
    '⛸️': EmojiEntry(codepoint='26F8 FE0F', status='fully-qualified', emoji='⛸️', name='ice skate', group='Activities', sub_group='sport'),
    '⛸': EmojiEntry(codepoint='26F8', status='unqualified', emoji='⛸', name='ice skate', group='Activities', sub_group='sport'),
    '🎣': EmojiEntry(codepoint='1F3A3', status='fully-qualified', emoji='🎣', name='fishing pole', group='Activities', sub_group='sport'),
    '🤿': EmojiEntry(codepoint='1F93F', status='fully-qualified', emoji='🤿', name='diving mask', group='Activities', sub_group='sport'),
    '🎽': EmojiEntry(codepoint='1F3BD', status='fully-qualified', emoji='🎽', name='running shirt', group='Activities', sub_group='sport'),
    '🎿': EmojiEntry(codepoint='1F3BF', status='fully-qualified', emoji='🎿', name='skis', group='Activities', sub_group='sport'),
    '🛷': EmojiEntry(codepoint='1F6F7', status='fully-qualified', emoji='🛷', name='sled', group='Activities', sub_group='sport'),
    '🥌': EmojiEntry(codepoint='1F94C', status='fully-qualified', emoji='🥌', name='curling stone', group='Activities', sub_group='sport'),
    '🎯': EmojiEntry(codepoint='1F3AF', status='fully-qualified', emoji='🎯', name='bullseye', group='Activities', sub_group='game'),
    '🪀': EmojiEntry(codepoint='1FA80', status='fully-qualified', emoji='🪀', name='yo-yo', group='Activities', sub_group='game'),
    '🪁': EmojiEntry(codepoint='1FA81', status='fully-qualified', emoji='🪁', name='kite', group='Activities', sub_group='game'),
    '🎱': EmojiEntry(codepoint='1F3B1', status='fully-qualified', emoji='🎱', name='pool 8 ball', group='Activities', sub_group='game'),
    '🔮': EmojiEntry(codepoint='1F52E', status='fully-qualified', emoji='🔮', name='crystal ball', group='Activities', sub_group='game'),
    '🪄': EmojiEntry(codepoint='1FA84', status='fully-qualified', emoji='🪄', name='magic wand', group='Activities', sub_group='game'),
    '🧿': EmojiEntry(codepoint='1F9FF', status='fully-qualified', emoji='🧿', name='nazar amulet', group='Activities', sub_group='game'),
    '🎮': EmojiEntry(codepoint='1F3AE', status='fully-qualified', emoji='🎮', name='video game', group='Activities', sub_group='game'),
    '🕹️': EmojiEntry(codepoint='1F579 FE0F', status='fully-qualified', emoji='🕹️', name='joystick', group='Activities', sub_group='game'),
    '🕹': EmojiEntry(codepoint='1F579', status='unqualified', emoji='🕹', name='joystick', group='Activities', sub_group='game'),
    '🎰': EmojiEntry(codepoint='1F3B0', status='fully-qualified', emoji='🎰', name='slot machine', group='Activities', sub_group='game'),
    '🎲': EmojiEntry(codepoint='1F3B2', status='fully-qualified', emoji='🎲', name='game die', group='Activities', sub_group='game'),
    '🧩': EmojiEntry(codepoint='1F9E9', status='fully-qualified', emoji='🧩', name='puzzle piece', group='Activities', sub_group='game'),
    '🧸': EmojiEntry(codepoint='1F9F8', status='fully-qualified', emoji='🧸', name='teddy bear', group='Activities', sub_group='game'),
    '🪅': EmojiEntry(codepoint='1FA85', status='fully-qualified', emoji='🪅', name='piñata', group='Activities', sub_group='game'),
    '🪆': EmojiEntry(codepoint='1FA86', status='fully-qualified', emoji='🪆', name='nesting dolls', group='Activities', sub_group='game'),
    '♠️': EmojiEntry(codepoint='2660 FE0F', status='fully-qualified', emoji='♠️', name='spade suit', group='Activities', sub_group='game'),
    '♠': EmojiEntry(codepoint='2660', status='unqualified', emoji='♠', name='spade suit', group='Activities', sub_group='game'),
    '♥️': EmojiEntry(codepoint='2665 FE0F', status='fully-qualified', emoji='♥️', name='heart suit', group='Activities', sub_group='game'),
    '♥': EmojiEntry(codepoint='2665', status='unqualified', emoji='♥', name='heart suit', group='Activities', sub_group='game'),
    '♦️': EmojiEntry(codepoint='2666 FE0F', status='fully-qualified', emoji='♦️', name='diamond suit', group='Activities', sub_group='game'),
    '♦': EmojiEntry(codepoint='2666', status='unqualified', emoji='♦', name='diamond suit', group='Activities', sub_group='game'),
    '♣️': EmojiEntry(codepoint='2663 FE0F', status='fully-qualified', emoji='♣️', name='club suit', group='Activities', sub_group='game'),
    '♣': EmojiEntry(codepoint='2663', status='unqualified', emoji='♣', name='club suit', group='Activities', sub_group='game'),
    '♟️': EmojiEntry(codepoint='265F FE0F', status='fully-qualified', emoji='♟️', name='chess pawn', group='Activities', sub_group='game'),
    '♟': EmojiEntry(codepoint='265F', status='unqualified', emoji='♟', name='chess pawn', group='Activities', sub_group='game'),
    '🃏': EmojiEntry(codepoint='1F0CF', status='fully-qualified', emoji='🃏', name='joker', group='Activities', sub_group='game'),
    '🀄': EmojiEntry(codepoint='1F004', status='fully-qualified', emoji='🀄', name='mahjong red dragon', group='Activities', sub_group='game'),
    '🎴': EmojiEntry(codepoint='1F3B4', status='fully-qualified', emoji='🎴', name='flower playing cards', group='Activities', sub_group='game'),
    '🎭': EmojiEntry(codepoint='1F3AD', status='fully-qualified', emoji='🎭', name='performing arts', group='Activities', sub_group='arts & crafts'),
    '🖼️': EmojiEntry(codepoint='1F5BC FE0F', status='fully-qualified', emoji='🖼️', name='framed picture', group='Activities', sub_group='arts & crafts'),
    '🖼': EmojiEntry(codepoint='1F5BC', status='unqualified', emoji='🖼', name='framed picture', group='Activities', sub_group='arts & crafts'),
    '🎨': EmojiEntry(codepoint='1F3A8', status='fully-qualified', emoji='🎨', name='artist palette', group='Activities', sub_group='arts & crafts'),
    '🧵': EmojiEntry(codepoint='1F9F5', status='fully-qualified', emoji='🧵', name='thread', group='Activities', sub_group='arts & crafts'),
    '🪡': EmojiEntry(codepoint='1FAA1', status='fully-qualified', emoji='🪡', name='sewing needle', group='Activities', sub_group='arts & crafts'),
    '🧶': EmojiEntry(codepoint='1F9F6', status='fully-qualified', emoji='🧶', name='yarn', group='Activities', sub_group='arts & crafts'),
    '🪢': EmojiEntry(codepoint='1FAA2', status='fully-qualified', emoji='🪢', name='knot', group='Activities', sub_group='arts & crafts'),
    '👓': EmojiEntry(codepoint='1F453', status='fully-qualified', emoji='👓', name='glasses', group='Objects', sub_group='clothing'),
    '🕶️': EmojiEntry(codepoint='1F576 FE0F', status='fully-qualified', emoji='🕶️', name='sunglasses', group='Objects', sub_group='clothing'),
    '🕶': EmojiEntry(codepoint='1F576', status='unqualified', emoji='🕶', name='sunglasses', group='Objects', sub_group='clothing'),
    '🥽': EmojiEntry(codepoint='1F97D', status='fully-qualified', emoji='🥽', name='goggles', group='Objects', sub_group='clothing'),
    '🥼': EmojiEntry(codepoint='1F97C', status='fully-qualified', emoji='🥼', name='lab coat', group='Objects', sub_group='clothing'),
    '🦺': EmojiEntry(codepoint='1F9BA', status='fully-qualified', emoji='🦺', name='safety vest', group='Objects', sub_group='clothing'),
    '👔': EmojiEntry(codepoint='1F454', status='fully-qualified', emoji='👔', name='necktie', group='Objects', sub_group='clothing'),
    '👕': EmojiEntry(codepoint='1F455', status='fully-qualified', emoji='👕', name='t-shirt', group='Objects', sub_group='clothing'),
    '👖': EmojiEntry(codepoint='1F456', status='fully-qualified', emoji='👖', name='jeans', group='Objects', sub_group='clothing'),
    '🧣': EmojiEntry(codepoint='1F9E3', status='fully-qualified', emoji='🧣', name='scarf', group='Objects', sub_group='clothing'),
    '🧤': EmojiEntry(codepoint='1F9E4', status='fully-qualified', emoji='🧤', name='gloves', group='Objects', sub_group='clothing'),
    '🧥': EmojiEntry(codepoint='1F9E5', status='fully-qualified', emoji='🧥', name='coat', group='Objects', sub_group='clothing'),
    '🧦': EmojiEntry(codepoint='1F9E6', status='fully-qualified', emoji='🧦', name='socks', group='Objects', sub_group='clothing'),
    '👗': EmojiEntry(codepoint='1F457', status='fully-qualified', emoji='👗', name='dress', group='Objects', sub_group='clothing'),
    '👘': EmojiEntry(codepoint='1F458', status='fully-qualified', emoji='👘', name='kimono', group='Objects', sub_group='clothing'),
    '🥻': EmojiEntry(codepoint='1F97B', status='fully-qualified', emoji='🥻', name='sari', group='Objects', sub_group='clothing'),
    '🩱': EmojiEntry(codepoint='1FA71', status='fully-qualified', emoji='🩱', name='one-piece swimsuit', group='Objects', sub_group='clothing'),
    '🩲': EmojiEntry(codepoint='1FA72', status='fully-qualified', emoji='🩲', name='briefs', group='Objects', sub_group='clothing'),
    '🩳': EmojiEntry(codepoint='1FA73', status='fully-qualified', emoji='🩳', name='shorts', group='Objects', sub_group='clothing'),
    '👙': EmojiEntry(codepoint='1F459', status='fully-qualified', emoji='👙', name='bikini', group='Objects', sub_group='clothing'),
    '👚': EmojiEntry(codepoint='1F45A', status='fully-qualified', emoji='👚', name='woman’s clothes', group='Objects', sub_group='clothing'),
    '👛': EmojiEntry(codepoint='1F45B', status='fully-qualified', emoji='👛', name='purse', group='Objects', sub_group='clothing'),
    '👜': EmojiEntry(codepoint='1F45C', status='fully-qualified', emoji='👜', name='handbag', group='Objects', sub_group='clothing'),
    '👝': EmojiEntry(codepoint='1F45D', status='fully-qualified', emoji='👝', name='clutch bag', group='Objects', sub_group='clothing'),
    '🛍️': EmojiEntry(codepoint='1F6CD FE0F', status='fully-qualified', emoji='🛍️', name='shopping bags', group='Objects', sub_group='clothing'),
    '🛍': EmojiEntry(codepoint='1F6CD', status='unqualified', emoji='🛍', name='shopping bags', group='Objects', sub_group='clothing'),
    '🎒': EmojiEntry(codepoint='1F392', status='fully-qualified', emoji='🎒', name='backpack', group='Objects', sub_group='clothing'),
    '🩴': EmojiEntry(codepoint='1FA74', status='fully-qualified', emoji='🩴', name='thong sandal', group='Objects', sub_group='clothing'),
    '👞': EmojiEntry(codepoint='1F45E', status='fully-qualified', emoji='👞', name='man’s shoe', group='Objects', sub_group='clothing'),
    '👟': EmojiEntry(codepoint='1F45F', status='fully-qualified', emoji='👟', name='running shoe', group='Objects', sub_group='clothing'),
    '🥾': EmojiEntry(codepoint='1F97E', status='fully-qualified', emoji='🥾', name='hiking boot', group='Objects', sub_group='clothing'),
    '🥿': EmojiEntry(codepoint='1F97F', status='fully-qualified', emoji='🥿', name='flat shoe', group='Objects', sub_group='clothing'),
    '👠': EmojiEntry(codepoint='1F460', status='fully-qualified', emoji='👠', name='high-heeled shoe', group='Objects', sub_group='clothing'),
    '👡': EmojiEntry(codepoint='1F461', status='fully-qualified', emoji='👡', name='woman’s sandal', group='Objects', sub_group='clothing'),
    '🩰': EmojiEntry(codepoint='1FA70', status='fully-qualified', emoji='🩰', name='ballet shoes', group='Objects', sub_group='clothing'),
    '👢': EmojiEntry(codepoint='1F462', status='fully-qualified', emoji='👢', name='woman’s boot', group='Objects', sub_group='clothing'),
    '👑': EmojiEntry(codepoint='1F451', status='fully-qualified', emoji='👑', name='crown', group='Objects', sub_group='clothing'),
    '👒': EmojiEntry(codepoint='1F452', status='fully-qualified', emoji='👒', name='woman’s hat', group='Objects', sub_group='clothing'),
    '🎩': EmojiEntry(codepoint='1F3A9', status='fully-qualified', emoji='🎩', name='top hat', group='Objects', sub_group='clothing'),
    '🎓': EmojiEntry(codepoint='1F393', status='fully-qualified', emoji='🎓', name='graduation cap', group='Objects', sub_group='clothing'),
    '🧢': EmojiEntry(codepoint='1F9E2', status='fully-qualified', emoji='🧢', name='billed cap', group='Objects', sub_group='clothing'),
    '🪖': EmojiEntry(codepoint='1FA96', status='fully-qualified', emoji='🪖', name='military helmet', group='Objects', sub_group='clothing'),
    '⛑️': EmojiEntry(codepoint='26D1 FE0F', status='fully-qualified', emoji='⛑️', name='rescue worker’s helmet', group='Objects', sub_group='clothing'),
    '⛑': EmojiEntry(codepoint='26D1', status='unqualified', emoji='⛑', name='rescue worker’s helmet', group='Objects', sub_group='clothing'),
    '📿': EmojiEntry(codepoint='1F4FF', status='fully-qualified', emoji='📿', name='prayer beads', group='Objects', sub_group='clothing'),
    '💄': EmojiEntry(codepoint='1F484', status='fully-qualified', emoji='💄', name='lipstick', group='Objects', sub_group='clothing'),
    '💍': EmojiEntry(codepoint='1F48D', status='fully-qualified', emoji='💍', name='ring', group='Objects', sub_group='clothing'),
    '💎': EmojiEntry(codepoint='1F48E', status='fully-qualified', emoji='💎', name='gem stone', group='Objects', sub_group='clothing'),
    '🔇': EmojiEntry(codepoint='1F507', status='fully-qualified', emoji='🔇', name='muted speaker', group='Objects', sub_group='sound'),
    '🔈': EmojiEntry(codepoint='1F508', status='fully-qualified', emoji='🔈', name='speaker low volume', group='Objects', sub_group='sound'),
    '🔉': EmojiEntry(codepoint='1F509', status='fully-qualified', emoji='🔉', name='speaker medium volume', group='Objects', sub_group='sound'),
    '🔊': EmojiEntry(codepoint='1F50A', status='fully-qualified', emoji='🔊', name='speaker high volume', group='Objects', sub_group='sound'),
    '📢': EmojiEntry(codepoint='1F4E2', status='fully-qualified', emoji='📢', name='loudspeaker', group='Objects', sub_group='sound'),
    '📣': EmojiEntry(codepoint='1F4E3', status='fully-qualified', emoji='📣', name='megaphone', group='Objects', sub_group='sound'),
    '📯': EmojiEntry(codepoint='1F4EF', status='fully-qualified', emoji='📯', name='postal horn', group='Objects', sub_group='sound'),
    '🔔': EmojiEntry(codepoint='1F514', status='fully-qualified', emoji='🔔', name='bell', group='Objects', sub_group='sound'),
    '🔕': EmojiEntry(codepoint='1F515', status='fully-qualified', emoji='🔕', name='bell with slash', group='Objects', sub_group='sound'),
    '🎼': EmojiEntry(codepoint='1F3BC', status='fully-qualified', emoji='🎼', name='musical score', group='Objects', sub_group='music'),
    '🎵': EmojiEntry(codepoint='1F3B5', status='fully-qualified', emoji='🎵', name='musical note', group='Objects', sub_group='music'),
    '🎶': EmojiEntry(codepoint='1F3B6', status='fully-qualified', emoji='🎶', name='musical notes', group='Objects', sub_group='music'),
    '🎙️': EmojiEntry(codepoint='1F399 FE0F', status='fully-qualified', emoji='🎙️', name='studio microphone', group='Objects', sub_group='music'),
    '🎙': EmojiEntry(codepoint='1F399', status='unqualified', emoji='🎙', name='studio microphone', group='Objects', sub_group='music'),
    '🎚️': EmojiEntry(codepoint='1F39A FE0F', status='fully-qualified', emoji='🎚️', name='level slider', group='Objects', sub_group='music'),
    '🎚': EmojiEntry(codepoint='1F39A', status='unqualified', emoji='🎚', name='level slider', group='Objects', sub_group='music'),
    '🎛️': EmojiEntry(codepoint='1F39B FE0F', status='fully-qualified', emoji='🎛️', name='control knobs', group='Objects', sub_group='music'),
    '🎛': EmojiEntry(codepoint='1F39B', status='unqualified', emoji='🎛', name='control knobs', group='Objects', sub_group='music'),
    '🎤': EmojiEntry(codepoint='1F3A4', status='fully-qualified', emoji='🎤', name='microphone', group='Objects', sub_group='music'),
    '🎧': EmojiEntry(codepoint='1F3A7', status='fully-qualified', emoji='🎧', name='headphone', group='Objects', sub_group='music'),
    '📻': EmojiEntry(codepoint='1F4FB', status='fully-qualified', emoji='📻', name='radio', group='Objects', sub_group='music'),
    '🎷': EmojiEntry(codepoint='1F3B7', status='fully-qualified', emoji='🎷', name='saxophone', group='Objects', sub_group='musical-instrument'),
    '🪗': EmojiEntry(codepoint='1FA97', status='fully-qualified', emoji='🪗', name='accordion', group='Objects', sub_group='musical-instrument'),
    '🎸': EmojiEntry(codepoint='1F3B8', status='fully-qualified', emoji='🎸', name='guitar', group='Objects', sub_group='musical-instrument'),
    '🎹': EmojiEntry(codepoint='1F3B9', status='fully-qualified', emoji='🎹', name='musical keyboard', group='Objects', sub_group='musical-instrument'),
    '🎺': EmojiEntry(codepoint='1F3BA', status='fully-qualified', emoji='🎺', name='trumpet', group='Objects', sub_group='musical-instrument'),
    '🎻': EmojiEntry(codepoint='1F3BB', status='fully-qualified', emoji='🎻', name='violin', group='Objects', sub_group='musical-instrument'),
    '🪕': EmojiEntry(codepoint='1FA95', status='fully-qualified', emoji='🪕', name='banjo', group='Objects', sub_group='musical-instrument'),
    '🥁': EmojiEntry(codepoint='1F941', status='fully-qualified', emoji='🥁', name='drum', group='Objects', sub_group='musical-instrument'),
    '🪘': EmojiEntry(codepoint='1FA98', status='fully-qualified', emoji='🪘', name='long drum', group='Objects', sub_group='musical-instrument'),
    '📱': EmojiEntry(codepoint='1F4F1', status='fully-qualified', emoji='📱', name='mobile phone', group='Objects', sub_group='phone'),
    '📲': EmojiEntry(codepoint='1F4F2', status='fully-qualified', emoji='📲', name='mobile phone with arrow', group='Objects', sub_group='phone'),
    '☎️': EmojiEntry(codepoint='260E FE0F', status='fully-qualified', emoji='☎️', name='telephone', group='Objects', sub_group='phone'),
    '☎': EmojiEntry(codepoint='260E', status='unqualified', emoji='☎', name='telephone', group='Objects', sub_group='phone'),
    '📞': EmojiEntry(codepoint='1F4DE', status='fully-qualified', emoji='📞', name='telephone receiver', group='Objects', sub_group='phone'),
    '📟': EmojiEntry(codepoint='1F4DF', status='fully-qualified', emoji='📟', name='pager', group='Objects', sub_group='phone'),
    '📠': EmojiEntry(codepoint='1F4E0', status='fully-qualified', emoji='📠', name='fax machine', group='Objects', sub_group='phone'),
    '🔋': EmojiEntry(codepoint='1F50B', status='fully-qualified', emoji='🔋', name='battery', group='Objects', sub_group='computer'),
    '🔌': EmojiEntry(codepoint='1F50C', status='fully-qualified', emoji='🔌', name='electric plug', group='Objects', sub_group='computer'),
    '💻': EmojiEntry(codepoint='1F4BB', status='fully-qualified', emoji='💻', name='laptop', group='Objects', sub_group='computer'),
    '🖥️': EmojiEntry(codepoint='1F5A5 FE0F', status='fully-qualified', emoji='🖥️', name='desktop computer', group='Objects', sub_group='computer'),
    '🖥': EmojiEntry(codepoint='1F5A5', status='unqualified', emoji='🖥', name='desktop computer', group='Objects', sub_group='computer'),
    '🖨️': EmojiEntry(codepoint='1F5A8 FE0F', status='fully-qualified', emoji='🖨️', name='printer', group='Objects', sub_group='computer'),
    '🖨': EmojiEntry(codepoint='1F5A8', status='unqualified', emoji='🖨', name='printer', group='Objects', sub_group='computer'),
    '⌨️': EmojiEntry(codepoint='2328 FE0F', status='fully-qualified', emoji='⌨️', name='keyboard', group='Objects', sub_group='computer'),
    '⌨': EmojiEntry(codepoint='2328', status='unqualified', emoji='⌨', name='keyboard', group='Objects', sub_group='computer'),
    '🖱️': EmojiEntry(codepoint='1F5B1 FE0F', status='fully-qualified', emoji='🖱️', name='computer mouse', group='Objects', sub_group='computer'),
    '🖱': EmojiEntry(codepoint='1F5B1', status='unqualified', emoji='🖱', name='computer mouse', group='Objects', sub_group='computer'),
    '🖲️': EmojiEntry(codepoint='1F5B2 FE0F', status='fully-qualified', emoji='🖲️', name='trackball', group='Objects', sub_group='computer'),
    '🖲': EmojiEntry(codepoint='1F5B2', status='unqualified', emoji='🖲', name='trackball', group='Objects', sub_group='computer'),
    '💽': EmojiEntry(codepoint='1F4BD', status='fully-qualified', emoji='💽', name='computer disk', group='Objects', sub_group='computer'),
    '💾': EmojiEntry(codepoint='1F4BE', status='fully-qualified', emoji='💾', name='floppy disk', group='Objects', sub_group='computer'),
    '💿': EmojiEntry(codepoint='1F4BF', status='fully-qualified', emoji='💿', name='optical disk', group='Objects', sub_group='computer'),
    '📀': EmojiEntry(codepoint='1F4C0', status='fully-qualified', emoji='📀', name='dvd', group='Objects', sub_group='computer'),
    '🧮': EmojiEntry(codepoint='1F9EE', status='fully-qualified', emoji='🧮', name='abacus', group='Objects', sub_group='computer'),
    '🎥': EmojiEntry(codepoint='1F3A5', status='fully-qualified', emoji='🎥', name='movie camera', group='Objects', sub_group='light & video'),
    '🎞️': EmojiEntry(codepoint='1F39E FE0F', status='fully-qualified', emoji='🎞️', name='film frames', group='Objects', sub_group='light & video'),
    '🎞': EmojiEntry(codepoint='1F39E', status='unqualified', emoji='🎞', name='film frames', group='Objects', sub_group='light & video'),
    '📽️': EmojiEntry(codepoint='1F4FD FE0F', status='fully-qualified', emoji='📽️', name='film projector', group='Objects', sub_group='light & video'),
    '📽': EmojiEntry(codepoint='1F4FD', status='unqualified', emoji='📽', name='film projector', group='Objects', sub_group='light & video'),
    '🎬': EmojiEntry(codepoint='1F3AC', status='fully-qualified', emoji='🎬', name='clapper board', group='Objects', sub_group='light & video'),
    '📺': EmojiEntry(codepoint='1F4FA', status='fully-qualified', emoji='📺', name='television', group='Objects', sub_group='light & video'),
    '📷': EmojiEntry(codepoint='1F4F7', status='fully-qualified', emoji='📷', name='camera', group='Objects', sub_group='light & video'),
    '📸': EmojiEntry(codepoint='1F4F8', status='fully-qualified', emoji='📸', name='camera with flash', group='Objects', sub_group='light & video'),
    '📹': EmojiEntry(codepoint='1F4F9', status='fully-qualified', emoji='📹', name='video camera', group='Objects', sub_group='light & video'),
    '📼': EmojiEntry(codepoint='1F4FC', status='fully-qualified', emoji='📼', name='videocassette', group='Objects', sub_group='light & video'),
    '🔍': EmojiEntry(codepoint='1F50D', status='fully-qualified', emoji='🔍', name='magnifying glass tilted left', group='Objects', sub_group='light & video'),
    '🔎': EmojiEntry(codepoint='1F50E', status='fully-qualified', emoji='🔎', name='magnifying glass tilted right', group='Objects', sub_group='light & video'),
    '🕯️': EmojiEntry(codepoint='1F56F FE0F', status='fully-qualified', emoji='🕯️', name='candle', group='Objects', sub_group='light & video'),
    '🕯': EmojiEntry(codepoint='1F56F', status='unqualified', emoji='🕯', name='candle', group='Objects', sub_group='light & video'),
    '💡': EmojiEntry(codepoint='1F4A1', status='fully-qualified', emoji='💡', name='light bulb', group='Objects', sub_group='light & video'),
    '🔦': EmojiEntry(codepoint='1F526', status='fully-qualified', emoji='🔦', name='flashlight', group='Objects', sub_group='light & video'),
    '🏮': EmojiEntry(codepoint='1F3EE', status='fully-qualified', emoji='🏮', name='red paper lantern', group='Objects', sub_group='light & video'),
    '🪔': EmojiEntry(codepoint='1FA94', status='fully-qualified', emoji='🪔', name='diya lamp', group='Objects', sub_group='light & video'),
    '📔': EmojiEntry(codepoint='1F4D4', status='fully-qualified', emoji='📔', name='notebook with decorative cover', group='Objects', sub_group='book-paper'),
    '📕': EmojiEntry(codepoint='1F4D5', status='fully-qualified', emoji='📕', name='closed book', group='Objects', sub_group='book-paper'),
    '📖': EmojiEntry(codepoint='1F4D6', status='fully-qualified', emoji='📖', name='open book', group='Objects', sub_group='book-paper'),
    '📗': EmojiEntry(codepoint='1F4D7', status='fully-qualified', emoji='📗', name='green book', group='Objects', sub_group='book-paper'),
    '📘': EmojiEntry(codepoint='1F4D8', status='fully-qualified', emoji='📘', name='blue book', group='Objects', sub_group='book-paper'),
    '📙': EmojiEntry(codepoint='1F4D9', status='fully-qualified', emoji='📙', name='orange book', group='Objects', sub_group='book-paper'),
    '📚': EmojiEntry(codepoint='1F4DA', status='fully-qualified', emoji='📚', name='books', group='Objects', sub_group='book-paper'),
    '📓': EmojiEntry(codepoint='1F4D3', status='fully-qualified', emoji='📓', name='notebook', group='Objects', sub_group='book-paper'),
    '📒': EmojiEntry(codepoint='1F4D2', status='fully-qualified', emoji='📒', name='ledger', group='Objects', sub_group='book-paper'),
    '📃': EmojiEntry(codepoint='1F4C3', status='fully-qualified', emoji='📃', name='page with curl', group='Objects', sub_group='book-paper'),
    '📜': EmojiEntry(codepoint='1F4DC', status='fully-qualified', emoji='📜', name='scroll', group='Objects', sub_group='book-paper'),
    '📄': EmojiEntry(codepoint='1F4C4', status='fully-qualified', emoji='📄', name='page facing up', group='Objects', sub_group='book-paper'),
    '📰': EmojiEntry(codepoint='1F4F0', status='fully-qualified', emoji='📰', name='newspaper', group='Objects', sub_group='book-paper'),
    '🗞️': EmojiEntry(codepoint='1F5DE FE0F', status='fully-qualified', emoji='🗞️', name='rolled-up newspaper', group='Objects', sub_group='book-paper'),
    '🗞': EmojiEntry(codepoint='1F5DE', status='unqualified', emoji='🗞', name='rolled-up newspaper', group='Objects', sub_group='book-paper'),
    '📑': EmojiEntry(codepoint='1F4D1', status='fully-qualified', emoji='📑', name='bookmark tabs', group='Objects', sub_group='book-paper'),
    '🔖': EmojiEntry(codepoint='1F516', status='fully-qualified', emoji='🔖', name='bookmark', group='Objects', sub_group='book-paper'),
    '🏷️': EmojiEntry(codepoint='1F3F7 FE0F', status='fully-qualified', emoji='🏷️', name='label', group='Objects', sub_group='book-paper'),
    '🏷': EmojiEntry(codepoint='1F3F7', status='unqualified', emoji='🏷', name='label', group='Objects', sub_group='book-paper'),
    '💰': EmojiEntry(codepoint='1F4B0', status='fully-qualified', emoji='💰', name='money bag', group='Objects', sub_group='money'),
    '🪙': EmojiEntry(codepoint='1FA99', status='fully-qualified', emoji='🪙', name='coin', group='Objects', sub_group='money'),
    '💴': EmojiEntry(codepoint='1F4B4', status='fully-qualified', emoji='💴', name='yen banknote', group='Objects', sub_group='money'),
    '💵': EmojiEntry(codepoint='1F4B5', status='fully-qualified', emoji='💵', name='dollar banknote', group='Objects', sub_group='money'),
    '💶': EmojiEntry(codepoint='1F4B6', status='fully-qualified', emoji='💶', name='euro banknote', group='Objects', sub_group='money'),
    '💷': EmojiEntry(codepoint='1F4B7', status='fully-qualified', emoji='💷', name='pound banknote', group='Objects', sub_group='money'),
    '💸': EmojiEntry(codepoint='1F4B8', status='fully-qualified', emoji='💸', name='money with wings', group='Objects', sub_group='money'),
    '💳': EmojiEntry(codepoint='1F4B3', status='fully-qualified', emoji='💳', name='credit card', group='Objects', sub_group='money'),
    '🧾': EmojiEntry(codepoint='1F9FE', status='fully-qualified', emoji='🧾', name='receipt', group='Objects', sub_group='money'),
    '💹': EmojiEntry(codepoint='1F4B9', status='fully-qualified', emoji='💹', name='chart increasing with yen', group='Objects', sub_group='money'),
    '✉️': EmojiEntry(codepoint='2709 FE0F', status='fully-qualified', emoji='✉️', name='envelope', group='Objects', sub_group='mail'),
    '✉': EmojiEntry(codepoint='2709', status='unqualified', emoji='✉', name='envelope', group='Objects', sub_group='mail'),
    '📧': EmojiEntry(codepoint='1F4E7', status='fully-qualified', emoji='📧', name='e-mail', group='Objects', sub_group='mail'),
    '📨': EmojiEntry(codepoint='1F4E8', status='fully-qualified', emoji='📨', name='incoming envelope', group='Objects', sub_group='mail'),
    '📩': EmojiEntry(codepoint='1F4E9', status='fully-qualified', emoji='📩', name='envelope with arrow', group='Objects', sub_group='mail'),
    '📤': EmojiEntry(codepoint='1F4E4', status='fully-qualified', emoji='📤', name='outbox tray', group='Objects', sub_group='mail'),
    '📥': EmojiEntry(codepoint='1F4E5', status='fully-qualified', emoji='📥', name='inbox tray', group='Objects', sub_group='mail'),
    '📦': EmojiEntry(codepoint='1F4E6', status='fully-qualified', emoji='📦', name='package', group='Objects', sub_group='mail'),
    '📫': EmojiEntry(codepoint='1F4EB', status='fully-qualified', emoji='📫', name='closed mailbox with raised flag', group='Objects', sub_group='mail'),
    '📪': EmojiEntry(codepoint='1F4EA', status='fully-qualified', emoji='📪', name='closed mailbox with lowered flag', group='Objects', sub_group='mail'),
    '📬': EmojiEntry(codepoint='1F4EC', status='fully-qualified', emoji='📬', name='open mailbox with raised flag', group='Objects', sub_group='mail'),
    '📭': EmojiEntry(codepoint='1F4ED', status='fully-qualified', emoji='📭', name='open mailbox with lowered flag', group='Objects', sub_group='mail'),
    '📮': EmojiEntry(codepoint='1F4EE', status='fully-qualified', emoji='📮', name='postbox', group='Objects', sub_group='mail'),
    '🗳️': EmojiEntry(codepoint='1F5F3 FE0F', status='fully-qualified', emoji='🗳️', name='ballot box with ballot', group='Objects', sub_group='mail'),
    '🗳': EmojiEntry(codepoint='1F5F3', status='unqualified', emoji='🗳', name='ballot box with ballot', group='Objects', sub_group='mail'),
    '✏️': EmojiEntry(codepoint='270F FE0F', status='fully-qualified', emoji='✏️', name='pencil', group='Objects', sub_group='writing'),
    '✏': EmojiEntry(codepoint='270F', status='unqualified', emoji='✏', name='pencil', group='Objects', sub_group='writing'),
    '✒️': EmojiEntry(codepoint='2712 FE0F', status='fully-qualified', emoji='✒️', name='black nib', group='Objects', sub_group='writing'),
    '✒': EmojiEntry(codepoint='2712', status='unqualified', emoji='✒', name='black nib', group='Objects', sub_group='writing'),
    '🖋️': EmojiEntry(codepoint='1F58B FE0F', status='fully-qualified', emoji='🖋️', name='fountain pen', group='Objects', sub_group='writing'),
    '🖋': EmojiEntry(codepoint='1F58B', status='unqualified', emoji='🖋', name='fountain pen', group='Objects', sub_group='writing'),
    '🖊️': EmojiEntry(codepoint='1F58A FE0F', status='fully-qualified', emoji='🖊️', name='pen', group='Objects', sub_group='writing'),
    '🖊': EmojiEntry(codepoint='1F58A', status='unqualified', emoji='🖊', name='pen', group='Objects', sub_group='writing'),
    '🖌️': EmojiEntry(codepoint='1F58C FE0F', status='fully-qualified', emoji='🖌️', name='paintbrush', group='Objects', sub_group='writing'),
    '🖌': EmojiEntry(codepoint='1F58C', status='unqualified', emoji='🖌', name='paintbrush', group='Objects', sub_group='writing'),
    '🖍️': EmojiEntry(codepoint='1F58D FE0F', status='fully-qualified', emoji='🖍️', name='crayon', group='Objects', sub_group='writing'),
    '🖍': EmojiEntry(codepoint='1F58D', status='unqualified', emoji='🖍', name='crayon', group='Objects', sub_group='writing'),
    '📝': EmojiEntry(codepoint='1F4DD', status='fully-qualified', emoji='📝', name='memo', group='Objects', sub_group='writing'),
    '💼': EmojiEntry(codepoint='1F4BC', status='fully-qualified', emoji='💼', name='briefcase', group='Objects', sub_group='office'),
    '📁': EmojiEntry(codepoint='1F4C1', status='fully-qualified', emoji='📁', name='file folder', group='Objects', sub_group='office'),
    '📂': EmojiEntry(codepoint='1F4C2', status='fully-qualified', emoji='📂', name='open file folder', group='Objects', sub_group='office'),
    '🗂️': EmojiEntry(codepoint='1F5C2 FE0F', status='fully-qualified', emoji='🗂️', name='card index dividers', group='Objects', sub_group='office'),
    '🗂': EmojiEntry(codepoint='1F5C2', status='unqualified', emoji='🗂', name='card index dividers', group='Objects', sub_group='office'),
    '📅': EmojiEntry(codepoint='1F4C5', status='fully-qualified', emoji='📅', name='calendar', group='Objects', sub_group='office'),
    '📆': EmojiEntry(codepoint='1F4C6', status='fully-qualified', emoji='📆', name='tear-off calendar', group='Objects', sub_group='office'),
    '🗒️': EmojiEntry(codepoint='1F5D2 FE0F', status='fully-qualified', emoji='🗒️', name='spiral notepad', group='Objects', sub_group='office'),
    '🗒': EmojiEntry(codepoint='1F5D2', status='unqualified', emoji='🗒', name='spiral notepad', group='Objects', sub_group='office'),
    '🗓️': EmojiEntry(codepoint='1F5D3 FE0F', status='fully-qualified', emoji='🗓️', name='spiral calendar', group='Objects', sub_group='office'),
    '🗓': EmojiEntry(codepoint='1F5D3', status='unqualified', emoji='🗓', name='spiral calendar', group='Objects', sub_group='office'),
    '📇': EmojiEntry(codepoint='1F4C7', status='fully-qualified', emoji='📇', name='card index', group='Objects', sub_group='office'),
    '📈': EmojiEntry(codepoint='1F4C8', status='fully-qualified', emoji='📈', name='chart increasing', group='Objects', sub_group='office'),
    '📉': EmojiEntry(codepoint='1F4C9', status='fully-qualified', emoji='📉', name='chart decreasing', group='Objects', sub_group='office'),
    '📊': EmojiEntry(codepoint='1F4CA', status='fully-qualified', emoji='📊', name='bar chart', group='Objects', sub_group='office'),
    '📋': EmojiEntry(codepoint='1F4CB', status='fully-qualified', emoji='📋', name='clipboard', group='Objects', sub_group='office'),
    '📌': EmojiEntry(codepoint='1F4CC', status='fully-qualified', emoji='📌', name='pushpin', group='Objects', sub_group='office'),
    '📍': EmojiEntry(codepoint='1F4CD', status='fully-qualified', emoji='📍', name='round pushpin', group='Objects', sub_group='office'),
    '📎': EmojiEntry(codepoint='1F4CE', status='fully-qualified', emoji='📎', name='paperclip', group='Objects', sub_group='office'),
    '🖇️': EmojiEntry(codepoint='1F587 FE0F', status='fully-qualified', emoji='🖇️', name='linked paperclips', group='Objects', sub_group='office'),
    '🖇': EmojiEntry(codepoint='1F587', status='unqualified', emoji='🖇', name='linked paperclips', group='Objects', sub_group='office'),
    '📏': EmojiEntry(codepoint='1F4CF', status='fully-qualified', emoji='📏', name='straight ruler', group='Objects', sub_group='office'),
    '📐': EmojiEntry(codepoint='1F4D0', status='fully-qualified', emoji='📐', name='triangular ruler', group='Objects', sub_group='office'),
    '✂️': EmojiEntry(codepoint='2702 FE0F', status='fully-qualified', emoji='✂️', name='scissors', group='Objects', sub_group='office'),
    '✂': EmojiEntry(codepoint='2702', status='unqualified', emoji='✂', name='scissors', group='Objects', sub_group='office'),
    '🗃️': EmojiEntry(codepoint='1F5C3 FE0F', status='fully-qualified', emoji='🗃️', name='card file box', group='Objects', sub_group='office'),
    '🗃': EmojiEntry(codepoint='1F5C3', status='unqualified', emoji='🗃', name='card file box', group='Objects', sub_group='office'),
    '🗄️': EmojiEntry(codepoint='1F5C4 FE0F', status='fully-qualified', emoji='🗄️', name='file cabinet', group='Objects', sub_group='office'),
    '🗄': EmojiEntry(codepoint='1F5C4', status='unqualified', emoji='🗄', name='file cabinet', group='Objects', sub_group='office'),
    '🗑️': EmojiEntry(codepoint='1F5D1 FE0F', status='fully-qualified', emoji='🗑️', name='wastebasket', group='Objects', sub_group='office'),
    '🗑': EmojiEntry(codepoint='1F5D1', status='unqualified', emoji='🗑', name='wastebasket', group='Objects', sub_group='office'),
    '🔒': EmojiEntry(codepoint='1F512', status='fully-qualified', emoji='🔒', name='locked', group='Objects', sub_group='lock'),
    '🔓': EmojiEntry(codepoint='1F513', status='fully-qualified', emoji='🔓', name='unlocked', group='Objects', sub_group='lock'),
    '🔏': EmojiEntry(codepoint='1F50F', status='fully-qualified', emoji='🔏', name='locked with pen', group='Objects', sub_group='lock'),
    '🔐': EmojiEntry(codepoint='1F510', status='fully-qualified', emoji='🔐', name='locked with key', group='Objects', sub_group='lock'),
    '🔑': EmojiEntry(codepoint='1F511', status='fully-qualified', emoji='🔑', name='key', group='Objects', sub_group='lock'),
    '🗝️': EmojiEntry(codepoint='1F5DD FE0F', status='fully-qualified', emoji='🗝️', name='old key', group='Objects', sub_group='lock'),
    '🗝': EmojiEntry(codepoint='1F5DD', status='unqualified', emoji='🗝', name='old key', group='Objects', sub_group='lock'),
    '🔨': EmojiEntry(codepoint='1F528', status='fully-qualified', emoji='🔨', name='hammer', group='Objects', sub_group='tool'),
    '🪓': EmojiEntry(codepoint='1FA93', status='fully-qualified', emoji='🪓', name='axe', group='Objects', sub_group='tool'),
    '⛏️': EmojiEntry(codepoint='26CF FE0F', status='fully-qualified', emoji='⛏️', name='pick', group='Objects', sub_group='tool'),
    '⛏': EmojiEntry(codepoint='26CF', status='unqualified', emoji='⛏', name='pick', group='Objects', sub_group='tool'),
    '⚒️': EmojiEntry(codepoint='2692 FE0F', status='fully-qualified', emoji='⚒️', name='hammer and pick', group='Objects', sub_group='tool'),
    '⚒': EmojiEntry(codepoint='2692', status='unqualified', emoji='⚒', name='hammer and pick', group='Objects', sub_group='tool'),
    '🛠️': EmojiEntry(codepoint='1F6E0 FE0F', status='fully-qualified', emoji='🛠️', name='hammer and wrench', group='Objects', sub_group='tool'),
    '🛠': EmojiEntry(codepoint='1F6E0', status='unqualified', emoji='🛠', name='hammer and wrench', group='Objects', sub_group='tool'),
    '🗡️': EmojiEntry(codepoint='1F5E1 FE0F', status='fully-qualified', emoji='🗡️', name='dagger', group='Objects', sub_group='tool'),
    '🗡': EmojiEntry(codepoint='1F5E1', status='unqualified', emoji='🗡', name='dagger', group='Objects', sub_group='tool'),
    '⚔️': EmojiEntry(codepoint='2694 FE0F', status='fully-qualified', emoji='⚔️', name='crossed swords', group='Objects', sub_group='tool'),
    '⚔': EmojiEntry(codepoint='2694', status='unqualified', emoji='⚔', name='crossed swords', group='Objects', sub_group='tool'),
    '🔫': EmojiEntry(codepoint='1F52B', status='fully-qualified', emoji='🔫', name='water pistol', group='Objects', sub_group='tool'),
    '🪃': EmojiEntry(codepoint='1FA83', status='fully-qualified', emoji='🪃', name='boomerang', group='Objects', sub_group='tool'),
    '🏹': EmojiEntry(codepoint='1F3F9', status='fully-qualified', emoji='🏹', name='bow and arrow', group='Objects', sub_group='tool'),
    '🛡️': EmojiEntry(codepoint='1F6E1 FE0F', status='fully-qualified', emoji='🛡️', name='shield', group='Objects', sub_group='tool'),
    '🛡': EmojiEntry(codepoint='1F6E1', status='unqualified', emoji='🛡', name='shield', group='Objects', sub_group='tool'),
    '🪚': EmojiEntry(codepoint='1FA9A', status='fully-qualified', emoji='🪚', name='carpentry saw', group='Objects', sub_group='tool'),
    '🔧': EmojiEntry(codepoint='1F527', status='fully-qualified', emoji='🔧', name='wrench', group='Objects', sub_group='tool'),
    '🪛': EmojiEntry(codepoint='1FA9B', status='fully-qualified', emoji='🪛', name='screwdriver', group='Objects', sub_group='tool'),
    '🔩': EmojiEntry(codepoint='1F529', status='fully-qualified', emoji='🔩', name='nut and bolt', group='Objects', sub_group='tool'),
    '⚙️': EmojiEntry(codepoint='2699 FE0F', status='fully-qualified', emoji='⚙️', name='gear', group='Objects', sub_group='tool'),
    '⚙': EmojiEntry(codepoint='2699', status='unqualified', emoji='⚙', name='gear', group='Objects', sub_group='tool'),
    '🗜️': EmojiEntry(codepoint='1F5DC FE0F', status='fully-qualified', emoji='🗜️', name='clamp', group='Objects', sub_group='tool'),
    '🗜': EmojiEntry(codepoint='1F5DC', status='unqualified', emoji='🗜', name='clamp', group='Objects', sub_group='tool'),
    '⚖️': EmojiEntry(codepoint='2696 FE0F', status='fully-qualified', emoji='⚖️', name='balance scale', group='Objects', sub_group='tool'),
    '⚖': EmojiEntry(codepoint='2696', status='unqualified', emoji='⚖', name='balance scale', group='Objects', sub_group='tool'),
    '🦯': EmojiEntry(codepoint='1F9AF', status='fully-qualified', emoji='🦯', name='white cane', group='Objects', sub_group='tool'),
    '🔗': EmojiEntry(codepoint='1F517', status='fully-qualified', emoji='🔗', name='link', group='Objects', sub_group='tool'),
    '⛓️': EmojiEntry(codepoint='26D3 FE0F', status='fully-qualified', emoji='⛓️', name='chains', group='Objects', sub_group='tool'),
    '⛓': EmojiEntry(codepoint='26D3', status='unqualified', emoji='⛓', name='chains', group='Objects', sub_group='tool'),
    '🪝': EmojiEntry(codepoint='1FA9D', status='fully-qualified', emoji='🪝', name='hook', group='Objects', sub_group='tool'),
    '🧰': EmojiEntry(codepoint='1F9F0', status='fully-qualified', emoji='🧰', name='toolbox', group='Objects', sub_group='tool'),
    '🧲': EmojiEntry(codepoint='1F9F2', status='fully-qualified', emoji='🧲', name='magnet', group='Objects', sub_group='tool'),
    '🪜': EmojiEntry(codepoint='1FA9C', status='fully-qualified', emoji='🪜', name='ladder', group='Objects', sub_group='tool'),
    '⚗️': EmojiEntry(codepoint='2697 FE0F', status='fully-qualified', emoji='⚗️', name='alembic', group='Objects', sub_group='science'),
    '⚗': EmojiEntry(codepoint='2697', status='unqualified', emoji='⚗', name='alembic', group='Objects', sub_group='science'),
    '🧪': EmojiEntry(codepoint='1F9EA', status='fully-qualified', emoji='🧪', name='test tube', group='Objects', sub_group='science'),
    '🧫': EmojiEntry(codepoint='1F9EB', status='fully-qualified', emoji='🧫', name='petri dish', group='Objects', sub_group='science'),
    '🧬': EmojiEntry(codepoint='1F9EC', status='fully-qualified', emoji='🧬', name='dna', group='Objects', sub_group='science'),
    '🔬': EmojiEntry(codepoint='1F52C', status='fully-qualified', emoji='🔬', name='microscope', group='Objects', sub_group='science'),
    '🔭': EmojiEntry(codepoint='1F52D', status='fully-qualified', emoji='🔭', name='telescope', group='Objects', sub_group='science'),
    '📡': EmojiEntry(codepoint='1F4E1', status='fully-qualified', emoji='📡', name='satellite antenna', group='Objects', sub_group='science'),
    '💉': EmojiEntry(codepoint='1F489', status='fully-qualified', emoji='💉', name='syringe', group='Objects', sub_group='medical'),
    '🩸': EmojiEntry(codepoint='1FA78', status='fully-qualified', emoji='🩸', name='drop of blood', group='Objects', sub_group='medical'),
    '💊': EmojiEntry(codepoint='1F48A', status='fully-qualified', emoji='💊', name='pill', group='Objects', sub_group='medical'),
    '🩹': EmojiEntry(codepoint='1FA79', status='fully-qualified', emoji='🩹', name='adhesive bandage', group='Objects', sub_group='medical'),
    '🩺': EmojiEntry(codepoint='1FA7A', status='fully-qualified', emoji='🩺', name='stethoscope', group='Objects', sub_group='medical'),
    '🚪': EmojiEntry(codepoint='1F6AA', status='fully-qualified', emoji='🚪', name='door', group='Objects', sub_group='household'),
    '🛗': EmojiEntry(codepoint='1F6D7', status='fully-qualified', emoji='🛗', name='elevator', group='Objects', sub_group='household'),
    '🪞': EmojiEntry(codepoint='1FA9E', status='fully-qualified', emoji='🪞', name='mirror', group='Objects', sub_group='household'),
    '🪟': EmojiEntry(codepoint='1FA9F', status='fully-qualified', emoji='🪟', name='window', group='Objects', sub_group='household'),
    '🛏️': EmojiEntry(codepoint='1F6CF FE0F', status='fully-qualified', emoji='🛏️', name='bed', group='Objects', sub_group='household'),
    '🛏': EmojiEntry(codepoint='1F6CF', status='unqualified', emoji='🛏', name='bed', group='Objects', sub_group='household'),
    '🛋️': EmojiEntry(codepoint='1F6CB FE0F', status='fully-qualified', emoji='🛋️', name='couch and lamp', group='Objects', sub_group='household'),
    '🛋': EmojiEntry(codepoint='1F6CB', status='unqualified', emoji='🛋', name='couch and lamp', group='Objects', sub_group='household'),
    '🪑': EmojiEntry(codepoint='1FA91', status='fully-qualified', emoji='🪑', name='chair', group='Objects', sub_group='household'),
    '🚽': EmojiEntry(codepoint='1F6BD', status='fully-qualified', emoji='🚽', name='toilet', group='Objects', sub_group='household'),
    '🪠': EmojiEntry(codepoint='1FAA0', status='fully-qualified', emoji='🪠', name='plunger', group='Objects', sub_group='household'),
    '🚿': EmojiEntry(codepoint='1F6BF', status='fully-qualified', emoji='🚿', name='shower', group='Objects', sub_group='household'),
    '🛁': EmojiEntry(codepoint='1F6C1', status='fully-qualified', emoji='🛁', name='bathtub', group='Objects', sub_group='household'),
    '🪤': EmojiEntry(codepoint='1FAA4', status='fully-qualified', emoji='🪤', name='mouse trap', group='Objects', sub_group='household'),
    '🪒': EmojiEntry(codepoint='1FA92', status='fully-qualified', emoji='🪒', name='razor', group='Objects', sub_group='household'),
    '🧴': EmojiEntry(codepoint='1F9F4', status='fully-qualified', emoji='🧴', name='lotion bottle', group='Objects', sub_group='household'),
    '🧷': EmojiEntry(codepoint='1F9F7', status='fully-qualified', emoji='🧷', name='safety pin', group='Objects', sub_group='household'),
    '🧹': EmojiEntry(codepoint='1F9F9', status='fully-qualified', emoji='🧹', name='broom', group='Objects', sub_group='household'),
    '🧺': EmojiEntry(codepoint='1F9FA', status='fully-qualified', emoji='🧺', name='basket', group='Objects', sub_group='household'),
    '🧻': EmojiEntry(codepoint='1F9FB', status='fully-qualified', emoji='🧻', name='roll of paper', group='Objects', sub_group='household'),
    '🪣': EmojiEntry(codepoint='1FAA3', status='fully-qualified', emoji='🪣', name='bucket', group='Objects', sub_group='household'),
    '🧼': EmojiEntry(codepoint='1F9FC', status='fully-qualified', emoji='🧼', name='soap', group='Objects', sub_group='household'),
    '🪥': EmojiEntry(codepoint='1FAA5', status='fully-qualified', emoji='🪥', name='toothbrush', group='Objects', sub_group='household'),
    '🧽': EmojiEntry(codepoint='1F9FD', status='fully-qualified', emoji='🧽', name='sponge', group='Objects', sub_group='household'),
    '🧯': EmojiEntry(codepoint='1F9EF', status='fully-qualified', emoji='🧯', name='fire extinguisher', group='Objects', sub_group='household'),
    '🛒': EmojiEntry(codepoint='1F6D2', status='fully-qualified', emoji='🛒', name='shopping cart', group='Objects', sub_group='household'),
    '🚬': EmojiEntry(codepoint='1F6AC', status='fully-qualified', emoji='🚬', name='cigarette', group='Objects', sub_group='other-object'),
    '⚰️': EmojiEntry(codepoint='26B0 FE0F', status='fully-qualified', emoji='⚰️', name='coffin', group='Objects', sub_group='other-object'),
    '⚰': EmojiEntry(codepoint='26B0', status='unqualified', emoji='⚰', name='coffin', group='Objects', sub_group='other-object'),
    '🪦': EmojiEntry(codepoint='1FAA6', status='fully-qualified', emoji='🪦', name='headstone', group='Objects', sub_group='other-object'),
    '⚱️': EmojiEntry(codepoint='26B1 FE0F', status='fully-qualified', emoji='⚱️', name='funeral urn', group='Objects', sub_group='other-object'),
    '⚱': EmojiEntry(codepoint='26B1', status='unqualified', emoji='⚱', name='funeral urn', group='Objects', sub_group='other-object'),
    '🗿': EmojiEntry(codepoint='1F5FF', status='fully-qualified', emoji='🗿', name='moai', group='Objects', sub_group='other-object'),
    '🪧': EmojiEntry(codepoint='1FAA7', status='fully-qualified', emoji='🪧', name='placard', group='Objects', sub_group='other-object'),
    '🏧': EmojiEntry(codepoint='1F3E7', status='fully-qualified', emoji='🏧', name='ATM sign', group='Symbols', sub_group='transport-sign'),
    '🚮': EmojiEntry(codepoint='1F6AE', status='fully-qualified', emoji='🚮', name='litter in bin sign', group='Symbols', sub_group='transport-sign'),
    '🚰': EmojiEntry(codepoint='1F6B0', status='fully-qualified', emoji='🚰', name='potable water', group='Symbols', sub_group='transport-sign'),
    '♿': EmojiEntry(codepoint='267F', status='fully-qualified', emoji='♿', name='wheelchair symbol', group='Symbols', sub_group='transport-sign'),
    '🚹': EmojiEntry(codepoint='1F6B9', status='fully-qualified', emoji='🚹', name='men’s room', group='Symbols', sub_group='transport-sign'),
    '🚺': EmojiEntry(codepoint='1F6BA', status='fully-qualified', emoji='🚺', name='women’s room', group='Symbols', sub_group='transport-sign'),
    '🚻': EmojiEntry(codepoint='1F6BB', status='fully-qualified', emoji='🚻', name='restroom', group='Symbols', sub_group='transport-sign'),
    '🚼': EmojiEntry(codepoint='1F6BC', status='fully-qualified', emoji='🚼', name='baby symbol', group='Symbols', sub_group='transport-sign'),
    '🚾': EmojiEntry(codepoint='1F6BE', status='fully-qualified', emoji='🚾', name='water closet', group='Symbols', sub_group='transport-sign'),
    '🛂': EmojiEntry(codepoint='1F6C2', status='fully-qualified', emoji='🛂', name='passport control', group='Symbols', sub_group='transport-sign'),
    '🛃': EmojiEntry(codepoint='1F6C3', status='fully-qualified', emoji='🛃', name='customs', group='Symbols', sub_group='transport-sign'),
    '🛄': EmojiEntry(codepoint='1F6C4', status='fully-qualified', emoji='🛄', name='baggage claim', group='Symbols', sub_group='transport-sign'),
    '🛅': EmojiEntry(codepoint='1F6C5', status='fully-qualified', emoji='🛅', name='left luggage', group='Symbols', sub_group='transport-sign'),
    '⚠️': EmojiEntry(codepoint='26A0 FE0F', status='fully-qualified', emoji='⚠️', name='warning', group='Symbols', sub_group='warning'),
    '⚠': EmojiEntry(codepoint='26A0', status='unqualified', emoji='⚠', name='warning', group='Symbols', sub_group='warning'),
    '🚸': EmojiEntry(codepoint='1F6B8', status='fully-qualified', emoji='🚸', name='children crossing', group='Symbols', sub_group='warning'),
    '⛔': EmojiEntry(codepoint='26D4', status='fully-qualified', emoji='⛔', name='no entry', group='Symbols', sub_group='warning'),
    '🚫': EmojiEntry(codepoint='1F6AB', status='fully-qualified', emoji='🚫', name='prohibited', group='Symbols', sub_group='warning'),
    '🚳': EmojiEntry(codepoint='1F6B3', status='fully-qualified', emoji='🚳', name='no bicycles', group='Symbols', sub_group='warning'),
    '🚭': EmojiEntry(codepoint='1F6AD', status='fully-qualified', emoji='🚭', name='no smoking', group='Symbols', sub_group='warning'),
    '🚯': EmojiEntry(codepoint='1F6AF', status='fully-qualified', emoji='🚯', name='no littering', group='Symbols', sub_group='warning'),
    '🚱': EmojiEntry(codepoint='1F6B1', status='fully-qualified', emoji='🚱', name='non-potable water', group='Symbols', sub_group='warning'),
    '🚷': EmojiEntry(codepoint='1F6B7', status='fully-qualified', emoji='🚷', name='no pedestrians', group='Symbols', sub_group='warning'),
    '📵': EmojiEntry(codepoint='1F4F5', status='fully-qualified', emoji='📵', name='no mobile phones', group='Symbols', sub_group='warning'),
    '🔞': EmojiEntry(codepoint='1F51E', status='fully-qualified', emoji='🔞', name='no one under eighteen', group='Symbols', sub_group='warning'),
    '☢️': EmojiEntry(codepoint='2622 FE0F', status='fully-qualified', emoji='☢️', name='radioactive', group='Symbols', sub_group='warning'),
    '☢': EmojiEntry(codepoint='2622', status='unqualified', emoji='☢', name='radioactive', group='Symbols', sub_group='warning'),
    '☣️': EmojiEntry(codepoint='2623 FE0F', status='fully-qualified', emoji='☣️', name='biohazard', group='Symbols', sub_group='warning'),
    '☣': EmojiEntry(codepoint='2623', status='unqualified', emoji='☣', name='biohazard', group='Symbols', sub_group='warning'),
    '⬆️': EmojiEntry(codepoint='2B06 FE0F', status='fully-qualified', emoji='⬆️', name='up arrow', group='Symbols', sub_group='arrow'),
    '⬆': EmojiEntry(codepoint='2B06', status='unqualified', emoji='⬆', name='up arrow', group='Symbols', sub_group='arrow'),
    '↗️': EmojiEntry(codepoint='2197 FE0F', status='fully-qualified', emoji='↗️', name='up-right arrow', group='Symbols', sub_group='arrow'),
    '↗': EmojiEntry(codepoint='2197', status='unqualified', emoji='↗', name='up-right arrow', group='Symbols', sub_group='arrow'),
    '➡️': EmojiEntry(codepoint='27A1 FE0F', status='fully-qualified', emoji='➡️', name='right arrow', group='Symbols', sub_group='arrow'),
    '➡': EmojiEntry(codepoint='27A1', status='unqualified', emoji='➡', name='right arrow', group='Symbols', sub_group='arrow'),
    '↘️': EmojiEntry(codepoint='2198 FE0F', status='fully-qualified', emoji='↘️', name='down-right arrow', group='Symbols', sub_group='arrow'),
    '↘': EmojiEntry(codepoint='2198', status='unqualified', emoji='↘', name='down-right arrow', group='Symbols', sub_group='arrow'),
    '⬇️': EmojiEntry(codepoint='2B07 FE0F', status='fully-qualified', emoji='⬇️', name='down arrow', group='Symbols', sub_group='arrow'),
    '⬇': EmojiEntry(codepoint='2B07', status='unqualified', emoji='⬇', name='down arrow', group='Symbols', sub_group='arrow'),
    '↙️': EmojiEntry(codepoint='2199 FE0F', status='fully-qualified', emoji='↙️', name='down-left arrow', group='Symbols', sub_group='arrow'),
    '↙': EmojiEntry(codepoint='2199', status='unqualified', emoji='↙', name='down-left arrow', group='Symbols', sub_group='arrow'),
    '⬅️': EmojiEntry(codepoint='2B05 FE0F', status='fully-qualified', emoji='⬅️', name='left arrow', group='Symbols', sub_group='arrow'),
    '⬅': EmojiEntry(codepoint='2B05', status='unqualified', emoji='⬅', name='left arrow', group='Symbols', sub_group='arrow'),
    '↖️': EmojiEntry(codepoint='2196 FE0F', status='fully-qualified', emoji='↖️', name='up-left arrow', group='Symbols', sub_group='arrow'),
    '↖': EmojiEntry(codepoint='2196', status='unqualified', emoji='↖', name='up-left arrow', group='Symbols', sub_group='arrow'),
    '↕️': EmojiEntry(codepoint='2195 FE0F', status='fully-qualified', emoji='↕️', name='up-down arrow', group='Symbols', sub_group='arrow'),
    '↕': EmojiEntry(codepoint='2195', status='unqualified', emoji='↕', name='up-down arrow', group='Symbols', sub_group='arrow'),
    '↔️': EmojiEntry(codepoint='2194 FE0F', status='fully-qualified', emoji='↔️', name='left-right arrow', group='Symbols', sub_group='arrow'),
    '↔': EmojiEntry(codepoint='2194', status='unqualified', emoji='↔', name='left-right arrow', group='Symbols', sub_group='arrow'),
    '↩️': EmojiEntry(codepoint='21A9 FE0F', status='fully-qualified', emoji='↩️', name='right arrow curving left', group='Symbols', sub_group='arrow'),
    '↩': EmojiEntry(codepoint='21A9', status='unqualified', emoji='↩', name='right arrow curving left', group='Symbols', sub_group='arrow'),
    '↪️': EmojiEntry(codepoint='21AA FE0F', status='fully-qualified', emoji='↪️', name='left arrow curving right', group='Symbols', sub_group='arrow'),
    '↪': EmojiEntry(codepoint='21AA', status='unqualified', emoji='↪', name='left arrow curving right', group='Symbols', sub_group='arrow'),
    '⤴️': EmojiEntry(codepoint='2934 FE0F', status='fully-qualified', emoji='⤴️', name='right arrow curving up', group='Symbols', sub_group='arrow'),
    '⤴': EmojiEntry(codepoint='2934', status='unqualified', emoji='⤴', name='right arrow curving up', group='Symbols', sub_group='arrow'),
    '⤵️': EmojiEntry(codepoint='2935 FE0F', status='fully-qualified', emoji='⤵️', name='right arrow curving down', group='Symbols', sub_group='arrow'),
    '⤵': EmojiEntry(codepoint='2935', status='unqualified', emoji='⤵', name='right arrow curving down', group='Symbols', sub_group='arrow'),
    '🔃': EmojiEntry(codepoint='1F503', status='fully-qualified', emoji='🔃', name='clockwise vertical arrows', group='Symbols', sub_group='arrow'),
    '🔄': EmojiEntry(codepoint='1F504', status='fully-qualified', emoji='🔄', name='counterclockwise arrows button', group='Symbols', sub_group='arrow'),
    '🔙': EmojiEntry(codepoint='1F519', status='fully-qualified', emoji='🔙', name='BACK arrow', group='Symbols', sub_group='arrow'),
    '🔚': EmojiEntry(codepoint='1F51A', status='fully-qualified', emoji='🔚', name='END arrow', group='Symbols', sub_group='arrow'),
    '🔛': EmojiEntry(codepoint='1F51B', status='fully-qualified', emoji='🔛', name='ON! arrow', group='Symbols', sub_group='arrow'),
    '🔜': EmojiEntry(codepoint='1F51C', status='fully-qualified', emoji='🔜', name='SOON arrow', group='Symbols', sub_group='arrow'),
    '🔝': EmojiEntry(codepoint='1F51D', status='fully-qualified', emoji='🔝', name='TOP arrow', group='Symbols', sub_group='arrow'),
    '🛐': EmojiEntry(codepoint='1F6D0', status='fully-qualified', emoji='🛐', name='place of worship', group='Symbols', sub_group='religion'),
    '⚛️': EmojiEntry(codepoint='269B FE0F', status='fully-qualified', emoji='⚛️', name='atom symbol', group='Symbols', sub_group='religion'),
    '⚛': EmojiEntry(codepoint='269B', status='unqualified', emoji='⚛', name='atom symbol', group='Symbols', sub_group='religion'),
    '🕉️': EmojiEntry(codepoint='1F549 FE0F', status='fully-qualified', emoji='🕉️', name='om', group='Symbols', sub_group='religion'),
    '🕉': EmojiEntry(codepoint='1F549', status='unqualified', emoji='🕉', name='om', group='Symbols', sub_group='religion'),
    '✡️': EmojiEntry(codepoint='2721 FE0F', status='fully-qualified', emoji='✡️', name='star of David', group='Symbols', sub_group='religion'),
    '✡': EmojiEntry(codepoint='2721', status='unqualified', emoji='✡', name='star of David', group='Symbols', sub_group='religion'),
    '☸️': EmojiEntry(codepoint='2638 FE0F', status='fully-qualified', emoji='☸️', name='wheel of dharma', group='Symbols', sub_group='religion'),
    '☸': EmojiEntry(codepoint='2638', status='unqualified', emoji='☸', name='wheel of dharma', group='Symbols', sub_group='religion'),
    '☯️': EmojiEntry(codepoint='262F FE0F', status='fully-qualified', emoji='☯️', name='yin yang', group='Symbols', sub_group='religion'),
    '☯': EmojiEntry(codepoint='262F', status='unqualified', emoji='☯', name='yin yang', group='Symbols', sub_group='religion'),
    '✝️': EmojiEntry(codepoint='271D FE0F', status='fully-qualified', emoji='✝️', name='latin cross', group='Symbols', sub_group='religion'),
    '✝': EmojiEntry(codepoint='271D', status='unqualified', emoji='✝', name='latin cross', group='Symbols', sub_group='religion'),
    '☦️': EmojiEntry(codepoint='2626 FE0F', status='fully-qualified', emoji='☦️', name='orthodox cross', group='Symbols', sub_group='religion'),
    '☦': EmojiEntry(codepoint='2626', status='unqualified', emoji='☦', name='orthodox cross', group='Symbols', sub_group='religion'),
    '☪️': EmojiEntry(codepoint='262A FE0F', status='fully-qualified', emoji='☪️', name='star and crescent', group='Symbols', sub_group='religion'),
    '☪': EmojiEntry(codepoint='262A', status='unqualified', emoji='☪', name='star and crescent', group='Symbols', sub_group='religion'),
    '☮️': EmojiEntry(codepoint='262E FE0F', status='fully-qualified', emoji='☮️', name='peace symbol', group='Symbols', sub_group='religion'),
    '☮': EmojiEntry(codepoint='262E', status='unqualified', emoji='☮', name='peace symbol', group='Symbols', sub_group='religion'),
    '🕎': EmojiEntry(codepoint='1F54E', status='fully-qualified', emoji='🕎', name='menorah', group='Symbols', sub_group='religion'),
    '🔯': EmojiEntry(codepoint='1F52F', status='fully-qualified', emoji='🔯', name='dotted six-pointed star', group='Symbols', sub_group='religion'),
    '♈': EmojiEntry(codepoint='2648', status='fully-qualified', emoji='♈', name='Aries', group='Symbols', sub_group='zodiac'),
    '♉': EmojiEntry(codepoint='2649', status='fully-qualified', emoji='♉', name='Taurus', group='Symbols', sub_group='zodiac'),
    '♊': EmojiEntry(codepoint='264A', status='fully-qualified', emoji='♊', name='Gemini', group='Symbols', sub_group='zodiac'),
    '♋': EmojiEntry(codepoint='264B', status='fully-qualified', emoji='♋', name='Cancer', group='Symbols', sub_group='zodiac'),
    '♌': EmojiEntry(codepoint='264C', status='fully-qualified', emoji='♌', name='Leo', group='Symbols', sub_group='zodiac'),
    '♍': EmojiEntry(codepoint='264D', status='fully-qualified', emoji='♍', name='Virgo', group='Symbols', sub_group='zodiac'),
    '♎': EmojiEntry(codepoint='264E', status='fully-qualified', emoji='♎', name='Libra', group='Symbols', sub_group='zodiac'),
    '♏': EmojiEntry(codepoint='264F', status='fully-qualified', emoji='♏', name='Scorpio', group='Symbols', sub_group='zodiac'),
    '♐': EmojiEntry(codepoint='2650', status='fully-qualified', emoji='♐', name='Sagittarius', group='Symbols', sub_group='zodiac'),
    '♑': EmojiEntry(codepoint='2651', status='fully-qualified', emoji='♑', name='Capricorn', group='Symbols', sub_group='zodiac'),
    '♒': EmojiEntry(codepoint='2652', status='fully-qualified', emoji='♒', name='Aquarius', group='Symbols', sub_group='zodiac'),
    '♓': EmojiEntry(codepoint='2653', status='fully-qualified', emoji='♓', name='Pisces', group='Symbols', sub_group='zodiac'),
    '⛎': EmojiEntry(codepoint='26CE', status='fully-qualified', emoji='⛎', name='Ophiuchus', group='Symbols', sub_group='zodiac'),
    '🔀': EmojiEntry(codepoint='1F500', status='fully-qualified', emoji='🔀', name='shuffle tracks button', group='Symbols', sub_group='av-symbol'),
    '🔁': EmojiEntry(codepoint='1F501', status='fully-qualified', emoji='🔁', name='repeat button', group='Symbols', sub_group='av-symbol'),
    '🔂': EmojiEntry(codepoint='1F502', status='fully-qualified', emoji='🔂', name='repeat single button', group='Symbols', sub_group='av-symbol'),
    '▶️': EmojiEntry(codepoint='25B6 FE0F', status='fully-qualified', emoji='▶️', name='play button', group='Symbols', sub_group='av-symbol'),
    '▶': EmojiEntry(codepoint='25B6', status='unqualified', emoji='▶', name='play button', group='Symbols', sub_group='av-symbol'),
    '⏩': EmojiEntry(codepoint='23E9', status='fully-qualified', emoji='⏩', name='fast-forward button', group='Symbols', sub_group='av-symbol'),
    '⏭️': EmojiEntry(codepoint='23ED FE0F', status='fully-qualified', emoji='⏭️', name='next track button', group='Symbols', sub_group='av-symbol'),
    '⏭': EmojiEntry(codepoint='23ED', status='unqualified', emoji='⏭', name='next track button', group='Symbols', sub_group='av-symbol'),
    '⏯️': EmojiEntry(codepoint='23EF FE0F', status='fully-qualified', emoji='⏯️', name='play or pause button', group='Symbols', sub_group='av-symbol'),
    '⏯': EmojiEntry(codepoint='23EF', status='unqualified', emoji='⏯', name='play or pause button', group='Symbols', sub_group='av-symbol'),
    '◀️': EmojiEntry(codepoint='25C0 FE0F', status='fully-qualified', emoji='◀️', name='reverse button', group='Symbols', sub_group='av-symbol'),
    '◀': EmojiEntry(codepoint='25C0', status='unqualified', emoji='◀', name='reverse button', group='Symbols', sub_group='av-symbol'),
    '⏪': EmojiEntry(codepoint='23EA', status='fully-qualified', emoji='⏪', name='fast reverse button', group='Symbols', sub_group='av-symbol'),
    '⏮️': EmojiEntry(codepoint='23EE FE0F', status='fully-qualified', emoji='⏮️', name='last track button', group='Symbols', sub_group='av-symbol'),
    '⏮': EmojiEntry(codepoint='23EE', status='unqualified', emoji='⏮', name='last track button', group='Symbols', sub_group='av-symbol'),
    '🔼': EmojiEntry(codepoint='1F53C', status='fully-qualified', emoji='🔼', name='upwards button', group='Symbols', sub_group='av-symbol'),
    '⏫': EmojiEntry(codepoint='23EB', status='fully-qualified', emoji='⏫', name='fast up button', group='Symbols', sub_group='av-symbol'),
    '🔽': EmojiEntry(codepoint='1F53D', status='fully-qualified', emoji='🔽', name='downwards button', group='Symbols', sub_group='av-symbol'),
    '⏬': EmojiEntry(codepoint='23EC', status='fully-qualified', emoji='⏬', name='fast down button', group='Symbols', sub_group='av-symbol'),
    '⏸️': EmojiEntry(codepoint='23F8 FE0F', status='fully-qualified', emoji='⏸️', name='pause button', group='Symbols', sub_group='av-symbol'),
    '⏸': EmojiEntry(codepoint='23F8', status='unqualified', emoji='⏸', name='pause button', group='Symbols', sub_group='av-symbol'),
    '⏹️': EmojiEntry(codepoint='23F9 FE0F', status='fully-qualified', emoji='⏹️', name='stop button', group='Symbols', sub_group='av-symbol'),
    '⏹': EmojiEntry(codepoint='23F9', status='unqualified', emoji='⏹', name='stop button', group='Symbols', sub_group='av-symbol'),
    '⏺️': EmojiEntry(codepoint='23FA FE0F', status='fully-qualified', emoji='⏺️', name='record button', group='Symbols', sub_group='av-symbol'),
    '⏺': EmojiEntry(codepoint='23FA', status='unqualified', emoji='⏺', name='record button', group='Symbols', sub_group='av-symbol'),
    '⏏️': EmojiEntry(codepoint='23CF FE0F', status='fully-qualified', emoji='⏏️', name='eject button', group='Symbols', sub_group='av-symbol'),
    '⏏': EmojiEntry(codepoint='23CF', status='unqualified', emoji='⏏', name='eject button', group='Symbols', sub_group='av-symbol'),
    '🎦': EmojiEntry(codepoint='1F3A6', status='fully-qualified', emoji='🎦', name='cinema', group='Symbols', sub_group='av-symbol'),
    '🔅': EmojiEntry(codepoint='1F505', status='fully-qualified', emoji='🔅', name='dim button', group='Symbols', sub_group='av-symbol'),
    '🔆': EmojiEntry(codepoint='1F506', status='fully-qualified', emoji='🔆', name='bright button', group='Symbols', sub_group='av-symbol'),
    '📶': EmojiEntry(codepoint='1F4F6', status='fully-qualified', emoji='📶', name='antenna bars', group='Symbols', sub_group='av-symbol'),
    '📳': EmojiEntry(codepoint='1F4F3', status='fully-qualified', emoji='📳', name='vibration mode', group='Symbols', sub_group='av-symbol'),
    '📴': EmojiEntry(codepoint='1F4F4', status='fully-qualified', emoji='📴', name='mobile phone off', group='Symbols', sub_group='av-symbol'),
    '♀️': EmojiEntry(codepoint='2640 FE0F', status='fully-qualified', emoji='♀️', name='female sign', group='Symbols', sub_group='gender'),
    '♀': EmojiEntry(codepoint='2640', status='unqualified', emoji='♀', name='female sign', group='Symbols', sub_group='gender'),
    '♂️': EmojiEntry(codepoint='2642 FE0F', status='fully-qualified', emoji='♂️', name='male sign', group='Symbols', sub_group='gender'),
    '♂': EmojiEntry(codepoint='2642', status='unqualified', emoji='♂', name='male sign', group='Symbols', sub_group='gender'),
    '⚧️': EmojiEntry(codepoint='26A7 FE0F', status='fully-qualified', emoji='⚧️', name='transgender symbol', group='Symbols', sub_group='gender'),
    '⚧': EmojiEntry(codepoint='26A7', status='unqualified', emoji='⚧', name='transgender symbol', group='Symbols', sub_group='gender'),
    '✖️': EmojiEntry(codepoint='2716 FE0F', status='fully-qualified', emoji='✖️', name='multiply', group='Symbols', sub_group='math'),
    '✖': EmojiEntry(codepoint='2716', status='unqualified', emoji='✖', name='multiply', group='Symbols', sub_group='math'),
    '➕': EmojiEntry(codepoint='2795', status='fully-qualified', emoji='➕', name='plus', group='Symbols', sub_group='math'),
    '➖': EmojiEntry(codepoint='2796', status='fully-qualified', emoji='➖', name='minus', group='Symbols', sub_group='math'),
    '➗': EmojiEntry(codepoint='2797', status='fully-qualified', emoji='➗', name='divide', group='Symbols', sub_group='math'),
    '♾️': EmojiEntry(codepoint='267E FE0F', status='fully-qualified', emoji='♾️', name='infinity', group='Symbols', sub_group='math'),
    '♾': EmojiEntry(codepoint='267E', status='unqualified', emoji='♾', name='infinity', group='Symbols', sub_group='math'),
    '‼️': EmojiEntry(codepoint='203C FE0F', status='fully-qualified', emoji='‼️', name='double exclamation mark', group='Symbols', sub_group='punctuation'),
    '‼': EmojiEntry(codepoint='203C', status='unqualified', emoji='‼', name='double exclamation mark', group='Symbols', sub_group='punctuation'),
    '⁉️': EmojiEntry(codepoint='2049 FE0F', status='fully-qualified', emoji='⁉️', name='exclamation question mark', group='Symbols', sub_group='punctuation'),
    '⁉': EmojiEntry(codepoint='2049', status='unqualified', emoji='⁉', name='exclamation question mark', group='Symbols', sub_group='punctuation'),
    '❓': EmojiEntry(codepoint='2753', status='fully-qualified', emoji='❓', name='red question mark', group='Symbols', sub_group='punctuation'),
    '❔': EmojiEntry(codepoint='2754', status='fully-qualified', emoji='❔', name='white question mark', group='Symbols', sub_group='punctuation'),
    '❕': EmojiEntry(codepoint='2755', status='fully-qualified', emoji='❕', name='white exclamation mark', group='Symbols', sub_group='punctuation'),
    '❗': EmojiEntry(codepoint='2757', status='fully-qualified', emoji='❗', name='red exclamation mark', group='Symbols', sub_group='punctuation'),
    '〰️': EmojiEntry(codepoint='3030 FE0F', status='fully-qualified', emoji='〰️', name='wavy dash', group='Symbols', sub_group='punctuation'),
    '〰': EmojiEntry(codepoint='3030', status='unqualified', emoji='〰', name='wavy dash', group='Symbols', sub_group='punctuation'),
    '💱': EmojiEntry(codepoint='1F4B1', status='fully-qualified', emoji='💱', name='currency exchange', group='Symbols', sub_group='currency'),
    '💲': EmojiEntry(codepoint='1F4B2', status='fully-qualified', emoji='💲', name='heavy dollar sign', group='Symbols', sub_group='currency'),
    '⚕️': EmojiEntry(codepoint='2695 FE0F', status='fully-qualified', emoji='⚕️', name='medical symbol', group='Symbols', sub_group='other-symbol'),
    '⚕': EmojiEntry(codepoint='2695', status='unqualified', emoji='⚕', name='medical symbol', group='Symbols', sub_group='other-symbol'),
    '♻️': EmojiEntry(codepoint='267B FE0F', status='fully-qualified', emoji='♻️', name='recycling symbol', group='Symbols', sub_group='other-symbol'),
    '♻': EmojiEntry(codepoint='267B', status='unqualified', emoji='♻', name='recycling symbol', group='Symbols', sub_group='other-symbol'),
    '⚜️': EmojiEntry(codepoint='269C FE0F', status='fully-qualified', emoji='⚜️', name='fleur-de-lis', group='Symbols', sub_group='other-symbol'),
    '⚜': EmojiEntry(codepoint='269C', status='unqualified', emoji='⚜', name='fleur-de-lis', group='Symbols', sub_group='other-symbol'),
    '🔱': EmojiEntry(codepoint='1F531', status='fully-qualified', emoji='🔱', name='trident emblem', group='Symbols', sub_group='other-symbol'),
    '📛': EmojiEntry(codepoint='1F4DB', status='fully-qualified', emoji='📛', name='name badge', group='Symbols', sub_group='other-symbol'),
    '🔰': EmojiEntry(codepoint='1F530', status='fully-qualified', emoji='🔰', name='Japanese symbol for beginner', group='Symbols', sub_group='other-symbol'),
    '⭕': EmojiEntry(codepoint='2B55', status='fully-qualified', emoji='⭕', name='hollow red circle', group='Symbols', sub_group='other-symbol'),
    '✅': EmojiEntry(codepoint='2705', status='fully-qualified', emoji='✅', name='check mark button', group='Symbols', sub_group='other-symbol'),
    '☑️': EmojiEntry(codepoint='2611 FE0F', status='fully-qualified', emoji='☑️', name='check box with check', group='Symbols', sub_group='other-symbol'),
    '☑': EmojiEntry(codepoint='2611', status='unqualified', emoji='☑', name='check box with check', group='Symbols', sub_group='other-symbol'),
    '✔️': EmojiEntry(codepoint='2714 FE0F', status='fully-qualified', emoji='✔️', name='check mark', group='Symbols', sub_group='other-symbol'),
    '✔': EmojiEntry(codepoint='2714', status='unqualified', emoji='✔', name='check mark', group='Symbols', sub_group='other-symbol'),
    '❌': EmojiEntry(codepoint='274C', status='fully-qualified', emoji='❌', name='cross mark', group='Symbols', sub_group='other-symbol'),
    '❎': EmojiEntry(codepoint='274E', status='fully-qualified', emoji='❎', name='cross mark button', group='Symbols', sub_group='other-symbol'),
    '➰': EmojiEntry(codepoint='27B0', status='fully-qualified', emoji='➰', name='curly loop', group='Symbols', sub_group='other-symbol'),
    '➿': EmojiEntry(codepoint='27BF', status='fully-qualified', emoji='➿', name='double curly loop', group='Symbols', sub_group='other-symbol'),
    '〽️': EmojiEntry(codepoint='303D FE0F', status='fully-qualified', emoji='〽️', name='part alternation mark', group='Symbols', sub_group='other-symbol'),
    '〽': EmojiEntry(codepoint='303D', status='unqualified', emoji='〽', name='part alternation mark', group='Symbols', sub_group='other-symbol'),
    '✳️': EmojiEntry(codepoint='2733 FE0F', status='fully-qualified', emoji='✳️', name='eight-spoked asterisk', group='Symbols', sub_group='other-symbol'),
    '✳': EmojiEntry(codepoint='2733', status='unqualified', emoji='✳', name='eight-spoked asterisk', group='Symbols', sub_group='other-symbol'),
    '✴️': EmojiEntry(codepoint='2734 FE0F', status='fully-qualified', emoji='✴️', name='eight-pointed star', group='Symbols', sub_group='other-symbol'),
    '✴': EmojiEntry(codepoint='2734', status='unqualified', emoji='✴', name='eight-pointed star', group='Symbols', sub_group='other-symbol'),
    '❇️': EmojiEntry(codepoint='2747 FE0F', status='fully-qualified', emoji='❇️', name='sparkle', group='Symbols', sub_group='other-symbol'),
    '❇': EmojiEntry(codepoint='2747', status='unqualified', emoji='❇', name='sparkle', group='Symbols', sub_group='other-symbol'),
    '©️': EmojiEntry(codepoint='00A9 FE0F', status='fully-qualified', emoji='©️', name='copyright', group='Symbols', sub_group='other-symbol'),
    '©': EmojiEntry(codepoint='00A9', status='unqualified', emoji='©', name='copyright', group='Symbols', sub_group='other-symbol'),
    '®️': EmojiEntry(codepoint='00AE FE0F', status='fully-qualified', emoji='®️', name='registered', group='Symbols', sub_group='other-symbol'),
    '®': EmojiEntry(codepoint='00AE', status='unqualified', emoji='®', name='registered', group='Symbols', sub_group='other-symbol'),
    '™️': EmojiEntry(codepoint='2122 FE0F', status='fully-qualified', emoji='™️', name='trade mark', group='Symbols', sub_group='other-symbol'),
    '™': EmojiEntry(codepoint='2122', status='unqualified', emoji='™', name='trade mark', group='Symbols', sub_group='other-symbol'),
    '#️⃣': EmojiEntry(codepoint='0023 FE0F 20E3', status='fully-qualified', emoji='#️⃣', name='#', group='Symbols', sub_group='keycap'),
    '#⃣': EmojiEntry(codepoint='0023 20E3', status='unqualified', emoji='#⃣', name='#', group='Symbols', sub_group='keycap'),
    '*️⃣': EmojiEntry(codepoint='002A FE0F 20E3', status='fully-qualified', emoji='*️⃣', name='keycap: *', group='Symbols', sub_group='keycap'),
    '*⃣': EmojiEntry(codepoint='002A 20E3', status='unqualified', emoji='*⃣', name='keycap: *', group='Symbols', sub_group='keycap'),
    '0️⃣': EmojiEntry(codepoint='0030 FE0F 20E3', status='fully-qualified', emoji='0️⃣', name='keycap: 0', group='Symbols', sub_group='keycap'),
    '0⃣': EmojiEntry(codepoint='0030 20E3', status='unqualified', emoji='0⃣', name='keycap: 0', group='Symbols', sub_group='keycap'),
    '1️⃣': EmojiEntry(codepoint='0031 FE0F 20E3', status='fully-qualified', emoji='1️⃣', name='keycap: 1', group='Symbols', sub_group='keycap'),
    '1⃣': EmojiEntry(codepoint='0031 20E3', status='unqualified', emoji='1⃣', name='keycap: 1', group='Symbols', sub_group='keycap'),
    '2️⃣': EmojiEntry(codepoint='0032 FE0F 20E3', status='fully-qualified', emoji='2️⃣', name='keycap: 2', group='Symbols', sub_group='keycap'),
    '2⃣': EmojiEntry(codepoint='0032 20E3', status='unqualified', emoji='2⃣', name='keycap: 2', group='Symbols', sub_group='keycap'),
    '3️⃣': EmojiEntry(codepoint='0033 FE0F 20E3', status='fully-qualified', emoji='3️⃣', name='keycap: 3', group='Symbols', sub_group='keycap'),
    '3⃣': EmojiEntry(codepoint='0033 20E3', status='unqualified', emoji='3⃣', name='keycap: 3', group='Symbols', sub_group='keycap'),
    '4️⃣': EmojiEntry(codepoint='0034 FE0F 20E3', status='fully-qualified', emoji='4️⃣', name='keycap: 4', group='Symbols', sub_group='keycap'),
    '4⃣': EmojiEntry(codepoint='0034 20E3', status='unqualified', emoji='4⃣', name='keycap: 4', group='Symbols', sub_group='keycap'),
    '5️⃣': EmojiEntry(codepoint='0035 FE0F 20E3', status='fully-qualified', emoji='5️⃣', name='keycap: 5', group='Symbols', sub_group='keycap'),
    '5⃣': EmojiEntry(codepoint='0035 20E3', status='unqualified', emoji='5⃣', name='keycap: 5', group='Symbols', sub_group='keycap'),
    '6️⃣': EmojiEntry(codepoint='0036 FE0F 20E3', status='fully-qualified', emoji='6️⃣', name='keycap: 6', group='Symbols', sub_group='keycap'),
    '6⃣': EmojiEntry(codepoint='0036 20E3', status='unqualified', emoji='6⃣', name='keycap: 6', group='Symbols', sub_group='keycap'),
    '7️⃣': EmojiEntry(codepoint='0037 FE0F 20E3', status='fully-qualified', emoji='7️⃣', name='keycap: 7', group='Symbols', sub_group='keycap'),
    '7⃣': EmojiEntry(codepoint='0037 20E3', status='unqualified', emoji='7⃣', name='keycap: 7', group='Symbols', sub_group='keycap'),
    '8️⃣': EmojiEntry(codepoint='0038 FE0F 20E3', status='fully-qualified', emoji='8️⃣', name='keycap: 8', group='Symbols', sub_group='keycap'),
    '8⃣': EmojiEntry(codepoint='0038 20E3', status='unqualified', emoji='8⃣', name='keycap: 8', group='Symbols', sub_group='keycap'),
    '9️⃣': EmojiEntry(codepoint='0039 FE0F 20E3', status='fully-qualified', emoji='9️⃣', name='keycap: 9', group='Symbols', sub_group='keycap'),
    '9⃣': EmojiEntry(codepoint='0039 20E3', status='unqualified', emoji='9⃣', name='keycap: 9', group='Symbols', sub_group='keycap'),
    '🔟': EmojiEntry(codepoint='1F51F', status='fully-qualified', emoji='🔟', name='keycap: 10', group='Symbols', sub_group='keycap'),
    '🔠': EmojiEntry(codepoint='1F520', status='fully-qualified', emoji='🔠', name='input latin uppercase', group='Symbols', sub_group='alphanum'),
    '🔡': EmojiEntry(codepoint='1F521', status='fully-qualified', emoji='🔡', name='input latin lowercase', group='Symbols', sub_group='alphanum'),
    '🔢': EmojiEntry(codepoint='1F522', status='fully-qualified', emoji='🔢', name='input numbers', group='Symbols', sub_group='alphanum'),
    '🔣': EmojiEntry(codepoint='1F523', status='fully-qualified', emoji='🔣', name='input symbols', group='Symbols', sub_group='alphanum'),
    '🔤': EmojiEntry(codepoint='1F524', status='fully-qualified', emoji='🔤', name='input latin letters', group='Symbols', sub_group='alphanum'),
    '🅰️': EmojiEntry(codepoint='1F170 FE0F', status='fully-qualified', emoji='🅰️', name='A button (blood type)', group='Symbols', sub_group='alphanum'),
    '🅰': EmojiEntry(codepoint='1F170', status='unqualified', emoji='🅰', name='A button (blood type)', group='Symbols', sub_group='alphanum'),
    '🆎': EmojiEntry(codepoint='1F18E', status='fully-qualified', emoji='🆎', name='AB button (blood type)', group='Symbols', sub_group='alphanum'),
    '🅱️': EmojiEntry(codepoint='1F171 FE0F', status='fully-qualified', emoji='🅱️', name='B button (blood type)', group='Symbols', sub_group='alphanum'),
    '🅱': EmojiEntry(codepoint='1F171', status='unqualified', emoji='🅱', name='B button (blood type)', group='Symbols', sub_group='alphanum'),
    '🆑': EmojiEntry(codepoint='1F191', status='fully-qualified', emoji='🆑', name='CL button', group='Symbols', sub_group='alphanum'),
    '🆒': EmojiEntry(codepoint='1F192', status='fully-qualified', emoji='🆒', name='COOL button', group='Symbols', sub_group='alphanum'),
    '🆓': EmojiEntry(codepoint='1F193', status='fully-qualified', emoji='🆓', name='FREE button', group='Symbols', sub_group='alphanum'),
    'ℹ️': EmojiEntry(codepoint='2139 FE0F', status='fully-qualified', emoji='ℹ️', name='information', group='Symbols', sub_group='alphanum'),
    'ℹ': EmojiEntry(codepoint='2139', status='unqualified', emoji='ℹ', name='information', group='Symbols', sub_group='alphanum'),
    '🆔': EmojiEntry(codepoint='1F194', status='fully-qualified', emoji='🆔', name='ID button', group='Symbols', sub_group='alphanum'),
    'Ⓜ️': EmojiEntry(codepoint='24C2 FE0F', status='fully-qualified', emoji='Ⓜ️', name='circled M', group='Symbols', sub_group='alphanum'),
    'Ⓜ': EmojiEntry(codepoint='24C2', status='unqualified', emoji='Ⓜ', name='circled M', group='Symbols', sub_group='alphanum'),
    '🆕': EmojiEntry(codepoint='1F195', status='fully-qualified', emoji='🆕', name='NEW button', group='Symbols', sub_group='alphanum'),
    '🆖': EmojiEntry(codepoint='1F196', status='fully-qualified', emoji='🆖', name='NG button', group='Symbols', sub_group='alphanum'),
    '🅾️': EmojiEntry(codepoint='1F17E FE0F', status='fully-qualified', emoji='🅾️', name='O button (blood type)', group='Symbols', sub_group='alphanum'),
    '🅾': EmojiEntry(codepoint='1F17E', status='unqualified', emoji='🅾', name='O button (blood type)', group='Symbols', sub_group='alphanum'),
    '🆗': EmojiEntry(codepoint='1F197', status='fully-qualified', emoji='🆗', name='OK button', group='Symbols', sub_group='alphanum'),
    '🅿️': EmojiEntry(codepoint='1F17F FE0F', status='fully-qualified', emoji='🅿️', name='P button', group='Symbols', sub_group='alphanum'),
    '🅿': EmojiEntry(codepoint='1F17F', status='unqualified', emoji='🅿', name='P button', group='Symbols', sub_group='alphanum'),
    '🆘': EmojiEntry(codepoint='1F198', status='fully-qualified', emoji='🆘', name='SOS button', group='Symbols', sub_group='alphanum'),
    '🆙': EmojiEntry(codepoint='1F199', status='fully-qualified', emoji='🆙', name='UP! button', group='Symbols', sub_group='alphanum'),
    '🆚': EmojiEntry(codepoint='1F19A', status='fully-qualified', emoji='🆚', name='VS button', group='Symbols', sub_group='alphanum'),
    '🈁': EmojiEntry(codepoint='1F201', status='fully-qualified', emoji='🈁', name='Japanese “here” button', group='Symbols', sub_group='alphanum'),
    '🈂️': EmojiEntry(codepoint='1F202 FE0F', status='fully-qualified', emoji='🈂️', name='Japanese “service charge” button', group='Symbols', sub_group='alphanum'),
    '🈂': EmojiEntry(codepoint='1F202', status='unqualified', emoji='🈂', name='Japanese “service charge” button', group='Symbols', sub_group='alphanum'),
    '🈷️': EmojiEntry(codepoint='1F237 FE0F', status='fully-qualified', emoji='🈷️', name='Japanese “monthly amount” button', group='Symbols', sub_group='alphanum'),
    '🈷': EmojiEntry(codepoint='1F237', status='unqualified', emoji='🈷', name='Japanese “monthly amount” button', group='Symbols', sub_group='alphanum'),
    '🈶': EmojiEntry(codepoint='1F236', status='fully-qualified', emoji='🈶', name='Japanese “not free of charge” button', group='Symbols', sub_group='alphanum'),
    '🈯': EmojiEntry(codepoint='1F22F', status='fully-qualified', emoji='🈯', name='Japanese “reserved” button', group='Symbols', sub_group='alphanum'),
    '🉐': EmojiEntry(codepoint='1F250', status='fully-qualified', emoji='🉐', name='Japanese “bargain” button', group='Symbols', sub_group='alphanum'),
    '🈹': EmojiEntry(codepoint='1F239', status='fully-qualified', emoji='🈹', name='Japanese “discount” button', group='Symbols', sub_group='alphanum'),
    '🈚': EmojiEntry(codepoint='1F21A', status='fully-qualified', emoji='🈚', name='Japanese “free of charge” button', group='Symbols', sub_group='alphanum'),
    '🈲': EmojiEntry(codepoint='1F232', status='fully-qualified', emoji='🈲', name='Japanese “prohibited” button', group='Symbols', sub_group='alphanum'),
    '🉑': EmojiEntry(codepoint='1F251', status='fully-qualified', emoji='🉑', name='Japanese “acceptable” button', group='Symbols', sub_group='alphanum'),
    '🈸': EmojiEntry(codepoint='1F238', status='fully-qualified', emoji='🈸', name='Japanese “application” button', group='Symbols', sub_group='alphanum'),
    '🈴': EmojiEntry(codepoint='1F234', status='fully-qualified', emoji='🈴', name='Japanese “passing grade” button', group='Symbols', sub_group='alphanum'),
    '🈳': EmojiEntry(codepoint='1F233', status='fully-qualified', emoji='🈳', name='Japanese “vacancy” button', group='Symbols', sub_group='alphanum'),
    '㊗️': EmojiEntry(codepoint='3297 FE0F', status='fully-qualified', emoji='㊗️', name='Japanese “congratulations” button', group='Symbols', sub_group='alphanum'),
    '㊗': EmojiEntry(codepoint='3297', status='unqualified', emoji='㊗', name='Japanese “congratulations” button', group='Symbols', sub_group='alphanum'),
    '㊙️': EmojiEntry(codepoint='3299 FE0F', status='fully-qualified', emoji='㊙️', name='Japanese “secret” button', group='Symbols', sub_group='alphanum'),
    '㊙': EmojiEntry(codepoint='3299', status='unqualified', emoji='㊙', name='Japanese “secret” button', group='Symbols', sub_group='alphanum'),
    '🈺': EmojiEntry(codepoint='1F23A', status='fully-qualified', emoji='🈺', name='Japanese “open for business” button', group='Symbols', sub_group='alphanum'),
    '🈵': EmojiEntry(codepoint='1F235', status='fully-qualified', emoji='🈵', name='Japanese “no vacancy” button', group='Symbols', sub_group='alphanum'),
    '🔴': EmojiEntry(codepoint='1F534', status='fully-qualified', emoji='🔴', name='red circle', group='Symbols', sub_group='geometric'),
    '🟠': EmojiEntry(codepoint='1F7E0', status='fully-qualified', emoji='🟠', name='orange circle', group='Symbols', sub_group='geometric'),
    '🟡': EmojiEntry(codepoint='1F7E1', status='fully-qualified', emoji='🟡', name='yellow circle', group='Symbols', sub_group='geometric'),
    '🟢': EmojiEntry(codepoint='1F7E2', status='fully-qualified', emoji='🟢', name='green circle', group='Symbols', sub_group='geometric'),
    '🔵': EmojiEntry(codepoint='1F535', status='fully-qualified', emoji='🔵', name='blue circle', group='Symbols', sub_group='geometric'),
    '🟣': EmojiEntry(codepoint='1F7E3', status='fully-qualified', emoji='🟣', name='purple circle', group='Symbols', sub_group='geometric'),
    '🟤': EmojiEntry(codepoint='1F7E4', status='fully-qualified', emoji='🟤', name='brown circle', group='Symbols', sub_group='geometric'),
    '⚫': EmojiEntry(codepoint='26AB', status='fully-qualified', emoji='⚫', name='black circle', group='Symbols', sub_group='geometric'),
    '⚪': EmojiEntry(codepoint='26AA', status='fully-qualified', emoji='⚪', name='white circle', group='Symbols', sub_group='geometric'),
    '🟥': EmojiEntry(codepoint='1F7E5', status='fully-qualified', emoji='🟥', name='red square', group='Symbols', sub_group='geometric'),
    '🟧': EmojiEntry(codepoint='1F7E7', status='fully-qualified', emoji='🟧', name='orange square', group='Symbols', sub_group='geometric'),
    '🟨': EmojiEntry(codepoint='1F7E8', status='fully-qualified', emoji='🟨', name='yellow square', group='Symbols', sub_group='geometric'),
    '🟩': EmojiEntry(codepoint='1F7E9', status='fully-qualified', emoji='🟩', name='green square', group='Symbols', sub_group='geometric'),
    '🟦': EmojiEntry(codepoint='1F7E6', status='fully-qualified', emoji='🟦', name='blue square', group='Symbols', sub_group='geometric'),
    '🟪': EmojiEntry(codepoint='1F7EA', status='fully-qualified', emoji='🟪', name='purple square', group='Symbols', sub_group='geometric'),
    '🟫': EmojiEntry(codepoint='1F7EB', status='fully-qualified', emoji='🟫', name='brown square', group='Symbols', sub_group='geometric'),
    '⬛': EmojiEntry(codepoint='2B1B', status='fully-qualified', emoji='⬛', name='black large square', group='Symbols', sub_group='geometric'),
    '⬜': EmojiEntry(codepoint='2B1C', status='fully-qualified', emoji='⬜', name='white large square', group='Symbols', sub_group='geometric'),
    '◼️': EmojiEntry(codepoint='25FC FE0F', status='fully-qualified', emoji='◼️', name='black medium square', group='Symbols', sub_group='geometric'),
    '◼': EmojiEntry(codepoint='25FC', status='unqualified', emoji='◼', name='black medium square', group='Symbols', sub_group='geometric'),
    '◻️': EmojiEntry(codepoint='25FB FE0F', status='fully-qualified', emoji='◻️', name='white medium square', group='Symbols', sub_group='geometric'),
    '◻': EmojiEntry(codepoint='25FB', status='unqualified', emoji='◻', name='white medium square', group='Symbols', sub_group='geometric'),
    '◾': EmojiEntry(codepoint='25FE', status='fully-qualified', emoji='◾', name='black medium-small square', group='Symbols', sub_group='geometric'),
    '◽': EmojiEntry(codepoint='25FD', status='fully-qualified', emoji='◽', name='white medium-small square', group='Symbols', sub_group='geometric'),
    '▪️': EmojiEntry(codepoint='25AA FE0F', status='fully-qualified', emoji='▪️', name='black small square', group='Symbols', sub_group='geometric'),
    '▪': EmojiEntry(codepoint='25AA', status='unqualified', emoji='▪', name='black small square', group='Symbols', sub_group='geometric'),
    '▫️': EmojiEntry(codepoint='25AB FE0F', status='fully-qualified', emoji='▫️', name='white small square', group='Symbols', sub_group='geometric'),
    '▫': EmojiEntry(codepoint='25AB', status='unqualified', emoji='▫', name='white small square', group='Symbols', sub_group='geometric'),
    '🔶': EmojiEntry(codepoint='1F536', status='fully-qualified', emoji='🔶', name='large orange diamond', group='Symbols', sub_group='geometric'),
    '🔷': EmojiEntry(codepoint='1F537', status='fully-qualified', emoji='🔷', name='large blue diamond', group='Symbols', sub_group='geometric'),
    '🔸': EmojiEntry(codepoint='1F538', status='fully-qualified', emoji='🔸', name='small orange diamond', group='Symbols', sub_group='geometric'),
    '🔹': EmojiEntry(codepoint='1F539', status='fully-qualified', emoji='🔹', name='small blue diamond', group='Symbols', sub_group='geometric'),
    '🔺': EmojiEntry(codepoint='1F53A', status='fully-qualified', emoji='🔺', name='red triangle pointed up', group='Symbols', sub_group='geometric'),
    '🔻': EmojiEntry(codepoint='1F53B', status='fully-qualified', emoji='🔻', name='red triangle pointed down', group='Symbols', sub_group='geometric'),
    '💠': EmojiEntry(codepoint='1F4A0', status='fully-qualified', emoji='💠', name='diamond with a dot', group='Symbols', sub_group='geometric'),
    '🔘': EmojiEntry(codepoint='1F518', status='fully-qualified', emoji='🔘', name='radio button', group='Symbols', sub_group='geometric'),
    '🔳': EmojiEntry(codepoint='1F533', status='fully-qualified', emoji='🔳', name='white square button', group='Symbols', sub_group='geometric'),
    '🔲': EmojiEntry(codepoint='1F532', status='fully-qualified', emoji='🔲', name='black square button', group='Symbols', sub_group='geometric'),
    '🏁': EmojiEntry(codepoint='1F3C1', status='fully-qualified', emoji='🏁', name='chequered flag', group='Flags', sub_group='flag'),
    '🚩': EmojiEntry(codepoint='1F6A9', status='fully-qualified', emoji='🚩', name='triangular flag', group='Flags', sub_group='flag'),
    '🎌': EmojiEntry(codepoint='1F38C', status='fully-qualified', emoji='🎌', name='crossed flags', group='Flags', sub_group='flag'),
    '🏴': EmojiEntry(codepoint='1F3F4', status='fully-qualified', emoji='🏴', name='black flag', group='Flags', sub_group='flag'),
    '🏳️': EmojiEntry(codepoint='1F3F3 FE0F', status='fully-qualified', emoji='🏳️', name='white flag', group='Flags', sub_group='flag'),
    '🏳': EmojiEntry(codepoint='1F3F3', status='unqualified', emoji='🏳', name='white flag', group='Flags', sub_group='flag'),
    '🏳️‍🌈': EmojiEntry(codepoint='1F3F3 FE0F 200D 1F308', status='fully-qualified', emoji='🏳️\u200d🌈', name='rainbow flag', group='Flags', sub_group='flag'),
    '🏳‍🌈': EmojiEntry(codepoint='1F3F3 200D 1F308', status='unqualified', emoji='🏳\u200d🌈', name='rainbow flag', group='Flags', sub_group='flag'),
    '🏳️‍⚧️': EmojiEntry(codepoint='1F3F3 FE0F 200D 26A7 FE0F', status='fully-qualified', emoji='🏳️\u200d⚧️', name='transgender flag', group='Flags', sub_group='flag'),
    '🏳‍⚧️': EmojiEntry(codepoint='1F3F3 200D 26A7 FE0F', status='unqualified', emoji='🏳\u200d⚧️', name='transgender flag', group='Flags', sub_group='flag'),
    '🏳️‍⚧': EmojiEntry(codepoint='1F3F3 FE0F 200D 26A7', status='unqualified', emoji='🏳️\u200d⚧', name='transgender flag', group='Flags', sub_group='flag'),
    '🏳‍⚧': EmojiEntry(codepoint='1F3F3 200D 26A7', status='unqualified', emoji='🏳\u200d⚧', name='transgender flag', group='Flags', sub_group='flag'),
    '🏴‍☠️': EmojiEntry(codepoint='1F3F4 200D 2620 FE0F', status='fully-qualified', emoji='🏴\u200d☠️', name='pirate flag', group='Flags', sub_group='flag'),
    '🏴‍☠': EmojiEntry(codepoint='1F3F4 200D 2620', status='minimally-qualified', emoji='🏴\u200d☠', name='pirate flag', group='Flags', sub_group='flag'),
    '🇦🇨': EmojiEntry(codepoint='1F1E6 1F1E8', status='fully-qualified', emoji='🇦🇨', name='flag: Ascension Island', group='Flags', sub_group='country-flag'),
    '🇦🇩': EmojiEntry(codepoint='1F1E6 1F1E9', status='fully-qualified', emoji='🇦🇩', name='flag: Andorra', group='Flags', sub_group='country-flag'),
    '🇦🇪': EmojiEntry(codepoint='1F1E6 1F1EA', status='fully-qualified', emoji='🇦🇪', name='flag: United Arab Emirates', group='Flags', sub_group='country-flag'),
    '🇦🇫': EmojiEntry(codepoint='1F1E6 1F1EB', status='fully-qualified', emoji='🇦🇫', name='flag: Afghanistan', group='Flags', sub_group='country-flag'),
    '🇦🇬': EmojiEntry(codepoint='1F1E6 1F1EC', status='fully-qualified', emoji='🇦🇬', name='flag: Antigua & Barbuda', group='Flags', sub_group='country-flag'),
    '🇦🇮': EmojiEntry(codepoint='1F1E6 1F1EE', status='fully-qualified', emoji='🇦🇮', name='flag: Anguilla', group='Flags', sub_group='country-flag'),
    '🇦🇱': EmojiEntry(codepoint='1F1E6 1F1F1', status='fully-qualified', emoji='🇦🇱', name='flag: Albania', group='Flags', sub_group='country-flag'),
    '🇦🇲': EmojiEntry(codepoint='1F1E6 1F1F2', status='fully-qualified', emoji='🇦🇲', name='flag: Armenia', group='Flags', sub_group='country-flag'),
    '🇦🇴': EmojiEntry(codepoint='1F1E6 1F1F4', status='fully-qualified', emoji='🇦🇴', name='flag: Angola', group='Flags', sub_group='country-flag'),
    '🇦🇶': EmojiEntry(codepoint='1F1E6 1F1F6', status='fully-qualified', emoji='🇦🇶', name='flag: Antarctica', group='Flags', sub_group='country-flag'),
    '🇦🇷': EmojiEntry(codepoint='1F1E6 1F1F7', status='fully-qualified', emoji='🇦🇷', name='flag: Argentina', group='Flags', sub_group='country-flag'),
    '🇦🇸': EmojiEntry(codepoint='1F1E6 1F1F8', status='fully-qualified', emoji='🇦🇸', name='flag: American Samoa', group='Flags', sub_group='country-flag'),
    '🇦🇹': EmojiEntry(codepoint='1F1E6 1F1F9', status='fully-qualified', emoji='🇦🇹', name='flag: Austria', group='Flags', sub_group='country-flag'),
    '🇦🇺': EmojiEntry(codepoint='1F1E6 1F1FA', status='fully-qualified', emoji='🇦🇺', name='flag: Australia', group='Flags', sub_group='country-flag'),
    '🇦🇼': EmojiEntry(codepoint='1F1E6 1F1FC', status='fully-qualified', emoji='🇦🇼', name='flag: Aruba', group='Flags', sub_group='country-flag'),
    '🇦🇽': EmojiEntry(codepoint='1F1E6 1F1FD', status='fully-qualified', emoji='🇦🇽', name='flag: Åland Islands', group='Flags', sub_group='country-flag'),
    '🇦🇿': EmojiEntry(codepoint='1F1E6 1F1FF', status='fully-qualified', emoji='🇦🇿', name='flag: Azerbaijan', group='Flags', sub_group='country-flag'),
    '🇧🇦': EmojiEntry(codepoint='1F1E7 1F1E6', status='fully-qualified', emoji='🇧🇦', name='flag: Bosnia & Herzegovina', group='Flags', sub_group='country-flag'),
    '🇧🇧': EmojiEntry(codepoint='1F1E7 1F1E7', status='fully-qualified', emoji='🇧🇧', name='flag: Barbados', group='Flags', sub_group='country-flag'),
    '🇧🇩': EmojiEntry(codepoint='1F1E7 1F1E9', status='fully-qualified', emoji='🇧🇩', name='flag: Bangladesh', group='Flags', sub_group='country-flag'),
    '🇧🇪': EmojiEntry(codepoint='1F1E7 1F1EA', status='fully-qualified', emoji='🇧🇪', name='flag: Belgium', group='Flags', sub_group='country-flag'),
    '🇧🇫': EmojiEntry(codepoint='1F1E7 1F1EB', status='fully-qualified', emoji='🇧🇫', name='flag: Burkina Faso', group='Flags', sub_group='country-flag'),
    '🇧🇬': EmojiEntry(codepoint='1F1E7 1F1EC', status='fully-qualified', emoji='🇧🇬', name='flag: Bulgaria', group='Flags', sub_group='country-flag'),
    '🇧🇭': EmojiEntry(codepoint='1F1E7 1F1ED', status='fully-qualified', emoji='🇧🇭', name='flag: Bahrain', group='Flags', sub_group='country-flag'),
    '🇧🇮': EmojiEntry(codepoint='1F1E7 1F1EE', status='fully-qualified', emoji='🇧🇮', name='flag: Burundi', group='Flags', sub_group='country-flag'),
    '🇧🇯': EmojiEntry(codepoint='1F1E7 1F1EF', status='fully-qualified', emoji='🇧🇯', name='flag: Benin', group='Flags', sub_group='country-flag'),
    '🇧🇱': EmojiEntry(codepoint='1F1E7 1F1F1', status='fully-qualified', emoji='🇧🇱', name='flag: St. Barthélemy', group='Flags', sub_group='country-flag'),
    '🇧🇲': EmojiEntry(codepoint='1F1E7 1F1F2', status='fully-qualified', emoji='🇧🇲', name='flag: Bermuda', group='Flags', sub_group='country-flag'),
    '🇧🇳': EmojiEntry(codepoint='1F1E7 1F1F3', status='fully-qualified', emoji='🇧🇳', name='flag: Brunei', group='Flags', sub_group='country-flag'),
    '🇧🇴': EmojiEntry(codepoint='1F1E7 1F1F4', status='fully-qualified', emoji='🇧🇴', name='flag: Bolivia', group='Flags', sub_group='country-flag'),
    '🇧🇶': EmojiEntry(codepoint='1F1E7 1F1F6', status='fully-qualified', emoji='🇧🇶', name='flag: Caribbean Netherlands', group='Flags', sub_group='country-flag'),
    '🇧🇷': EmojiEntry(codepoint='1F1E7 1F1F7', status='fully-qualified', emoji='🇧🇷', name='flag: Brazil', group='Flags', sub_group='country-flag'),
    '🇧🇸': EmojiEntry(codepoint='1F1E7 1F1F8', status='fully-qualified', emoji='🇧🇸', name='flag: Bahamas', group='Flags', sub_group='country-flag'),
    '🇧🇹': EmojiEntry(codepoint='1F1E7 1F1F9', status='fully-qualified', emoji='🇧🇹', name='flag: Bhutan', group='Flags', sub_group='country-flag'),
    '🇧🇻': EmojiEntry(codepoint='1F1E7 1F1FB', status='fully-qualified', emoji='🇧🇻', name='flag: Bouvet Island', group='Flags', sub_group='country-flag'),
    '🇧🇼': EmojiEntry(codepoint='1F1E7 1F1FC', status='fully-qualified', emoji='🇧🇼', name='flag: Botswana', group='Flags', sub_group='country-flag'),
    '🇧🇾': EmojiEntry(codepoint='1F1E7 1F1FE', status='fully-qualified', emoji='🇧🇾', name='flag: Belarus', group='Flags', sub_group='country-flag'),
    '🇧🇿': EmojiEntry(codepoint='1F1E7 1F1FF', status='fully-qualified', emoji='🇧🇿', name='flag: Belize', group='Flags', sub_group='country-flag'),
    '🇨🇦': EmojiEntry(codepoint='1F1E8 1F1E6', status='fully-qualified', emoji='🇨🇦', name='flag: Canada', group='Flags', sub_group='country-flag'),
    '🇨🇨': EmojiEntry(codepoint='1F1E8 1F1E8', status='fully-qualified', emoji='🇨🇨', name='flag: Cocos (Keeling) Islands', group='Flags', sub_group='country-flag'),
    '🇨🇩': EmojiEntry(codepoint='1F1E8 1F1E9', status='fully-qualified', emoji='🇨🇩', name='flag: Congo - Kinshasa', group='Flags', sub_group='country-flag'),
    '🇨🇫': EmojiEntry(codepoint='1F1E8 1F1EB', status='fully-qualified', emoji='🇨🇫', name='flag: Central African Republic', group='Flags', sub_group='country-flag'),
    '🇨🇬': EmojiEntry(codepoint='1F1E8 1F1EC', status='fully-qualified', emoji='🇨🇬', name='flag: Congo - Brazzaville', group='Flags', sub_group='country-flag'),
    '🇨🇭': EmojiEntry(codepoint='1F1E8 1F1ED', status='fully-qualified', emoji='🇨🇭', name='flag: Switzerland', group='Flags', sub_group='country-flag'),
    '🇨🇮': EmojiEntry(codepoint='1F1E8 1F1EE', status='fully-qualified', emoji='🇨🇮', name='flag: Côte d’Ivoire', group='Flags', sub_group='country-flag'),
    '🇨🇰': EmojiEntry(codepoint='1F1E8 1F1F0', status='fully-qualified', emoji='🇨🇰', name='flag: Cook Islands', group='Flags', sub_group='country-flag'),
    '🇨🇱': EmojiEntry(codepoint='1F1E8 1F1F1', status='fully-qualified', emoji='🇨🇱', name='flag: Chile', group='Flags', sub_group='country-flag'),
    '🇨🇲': EmojiEntry(codepoint='1F1E8 1F1F2', status='fully-qualified', emoji='🇨🇲', name='flag: Cameroon', group='Flags', sub_group='country-flag'),
    '🇨🇳': EmojiEntry(codepoint='1F1E8 1F1F3', status='fully-qualified', emoji='🇨🇳', name='flag: China', group='Flags', sub_group='country-flag'),
    '🇨🇴': EmojiEntry(codepoint='1F1E8 1F1F4', status='fully-qualified', emoji='🇨🇴', name='flag: Colombia', group='Flags', sub_group='country-flag'),
    '🇨🇵': EmojiEntry(codepoint='1F1E8 1F1F5', status='fully-qualified', emoji='🇨🇵', name='flag: Clipperton Island', group='Flags', sub_group='country-flag'),
    '🇨🇷': EmojiEntry(codepoint='1F1E8 1F1F7', status='fully-qualified', emoji='🇨🇷', name='flag: Costa Rica', group='Flags', sub_group='country-flag'),
    '🇨🇺': EmojiEntry(codepoint='1F1E8 1F1FA', status='fully-qualified', emoji='🇨🇺', name='flag: Cuba', group='Flags', sub_group='country-flag'),
    '🇨🇻': EmojiEntry(codepoint='1F1E8 1F1FB', status='fully-qualified', emoji='🇨🇻', name='flag: Cape Verde', group='Flags', sub_group='country-flag'),
    '🇨🇼': EmojiEntry(codepoint='1F1E8 1F1FC', status='fully-qualified', emoji='🇨🇼', name='flag: Curaçao', group='Flags', sub_group='country-flag'),
    '🇨🇽': EmojiEntry(codepoint='1F1E8 1F1FD', status='fully-qualified', emoji='🇨🇽', name='flag: Christmas Island', group='Flags', sub_group='country-flag'),
    '🇨🇾': EmojiEntry(codepoint='1F1E8 1F1FE', status='fully-qualified', emoji='🇨🇾', name='flag: Cyprus', group='Flags', sub_group='country-flag'),
    '🇨🇿': EmojiEntry(codepoint='1F1E8 1F1FF', status='fully-qualified', emoji='🇨🇿', name='flag: Czechia', group='Flags', sub_group='country-flag'),
    '🇩🇪': EmojiEntry(codepoint='1F1E9 1F1EA', status='fully-qualified', emoji='🇩🇪', name='flag: Germany', group='Flags', sub_group='country-flag'),
    '🇩🇬': EmojiEntry(codepoint='1F1E9 1F1EC', status='fully-qualified', emoji='🇩🇬', name='flag: Diego Garcia', group='Flags', sub_group='country-flag'),
    '🇩🇯': EmojiEntry(codepoint='1F1E9 1F1EF', status='fully-qualified', emoji='🇩🇯', name='flag: Djibouti', group='Flags', sub_group='country-flag'),
    '🇩🇰': EmojiEntry(codepoint='1F1E9 1F1F0', status='fully-qualified', emoji='🇩🇰', name='flag: Denmark', group='Flags', sub_group='country-flag'),
    '🇩🇲': EmojiEntry(codepoint='1F1E9 1F1F2', status='fully-qualified', emoji='🇩🇲', name='flag: Dominica', group='Flags', sub_group='country-flag'),
    '🇩🇴': EmojiEntry(codepoint='1F1E9 1F1F4', status='fully-qualified', emoji='🇩🇴', name='flag: Dominican Republic', group='Flags', sub_group='country-flag'),
    '🇩🇿': EmojiEntry(codepoint='1F1E9 1F1FF', status='fully-qualified', emoji='🇩🇿', name='flag: Algeria', group='Flags', sub_group='country-flag'),
    '🇪🇦': EmojiEntry(codepoint='1F1EA 1F1E6', status='fully-qualified', emoji='🇪🇦', name='flag: Ceuta & Melilla', group='Flags', sub_group='country-flag'),
    '🇪🇨': EmojiEntry(codepoint='1F1EA 1F1E8', status='fully-qualified', emoji='🇪🇨', name='flag: Ecuador', group='Flags', sub_group='country-flag'),
    '🇪🇪': EmojiEntry(codepoint='1F1EA 1F1EA', status='fully-qualified', emoji='🇪🇪', name='flag: Estonia', group='Flags', sub_group='country-flag'),
    '🇪🇬': EmojiEntry(codepoint='1F1EA 1F1EC', status='fully-qualified', emoji='🇪🇬', name='flag: Egypt', group='Flags', sub_group='country-flag'),
    '🇪🇭': EmojiEntry(codepoint='1F1EA 1F1ED', status='fully-qualified', emoji='🇪🇭', name='flag: Western Sahara', group='Flags', sub_group='country-flag'),
    '🇪🇷': EmojiEntry(codepoint='1F1EA 1F1F7', status='fully-qualified', emoji='🇪🇷', name='flag: Eritrea', group='Flags', sub_group='country-flag'),
    '🇪🇸': EmojiEntry(codepoint='1F1EA 1F1F8', status='fully-qualified', emoji='🇪🇸', name='flag: Spain', group='Flags', sub_group='country-flag'),
    '🇪🇹': EmojiEntry(codepoint='1F1EA 1F1F9', status='fully-qualified', emoji='🇪🇹', name='flag: Ethiopia', group='Flags', sub_group='country-flag'),
    '🇪🇺': EmojiEntry(codepoint='1F1EA 1F1FA', status='fully-qualified', emoji='🇪🇺', name='flag: European Union', group='Flags', sub_group='country-flag'),
    '🇫🇮': EmojiEntry(codepoint='1F1EB 1F1EE', status='fully-qualified', emoji='🇫🇮', name='flag: Finland', group='Flags', sub_group='country-flag'),
    '🇫🇯': EmojiEntry(codepoint='1F1EB 1F1EF', status='fully-qualified', emoji='🇫🇯', name='flag: Fiji', group='Flags', sub_group='country-flag'),
    '🇫🇰': EmojiEntry(codepoint='1F1EB 1F1F0', status='fully-qualified', emoji='🇫🇰', name='flag: Falkland Islands', group='Flags', sub_group='country-flag'),
    '🇫🇲': EmojiEntry(codepoint='1F1EB 1F1F2', status='fully-qualified', emoji='🇫🇲', name='flag: Micronesia', group='Flags', sub_group='country-flag'),
    '🇫🇴': EmojiEntry(codepoint='1F1EB 1F1F4', status='fully-qualified', emoji='🇫🇴', name='flag: Faroe Islands', group='Flags', sub_group='country-flag'),
    '🇫🇷': EmojiEntry(codepoint='1F1EB 1F1F7', status='fully-qualified', emoji='🇫🇷', name='flag: France', group='Flags', sub_group='country-flag'),
    '🇬🇦': EmojiEntry(codepoint='1F1EC 1F1E6', status='fully-qualified', emoji='🇬🇦', name='flag: Gabon', group='Flags', sub_group='country-flag'),
    '🇬🇧': EmojiEntry(codepoint='1F1EC 1F1E7', status='fully-qualified', emoji='🇬🇧', name='flag: United Kingdom', group='Flags', sub_group='country-flag'),
    '🇬🇩': EmojiEntry(codepoint='1F1EC 1F1E9', status='fully-qualified', emoji='🇬🇩', name='flag: Grenada', group='Flags', sub_group='country-flag'),
    '🇬🇪': EmojiEntry(codepoint='1F1EC 1F1EA', status='fully-qualified', emoji='🇬🇪', name='flag: Georgia', group='Flags', sub_group='country-flag'),
    '🇬🇫': EmojiEntry(codepoint='1F1EC 1F1EB', status='fully-qualified', emoji='🇬🇫', name='flag: French Guiana', group='Flags', sub_group='country-flag'),
    '🇬🇬': EmojiEntry(codepoint='1F1EC 1F1EC', status='fully-qualified', emoji='🇬🇬', name='flag: Guernsey', group='Flags', sub_group='country-flag'),
    '🇬🇭': EmojiEntry(codepoint='1F1EC 1F1ED', status='fully-qualified', emoji='🇬🇭', name='flag: Ghana', group='Flags', sub_group='country-flag'),
    '🇬🇮': EmojiEntry(codepoint='1F1EC 1F1EE', status='fully-qualified', emoji='🇬🇮', name='flag: Gibraltar', group='Flags', sub_group='country-flag'),
    '🇬🇱': EmojiEntry(codepoint='1F1EC 1F1F1', status='fully-qualified', emoji='🇬🇱', name='flag: Greenland', group='Flags', sub_group='country-flag'),
    '🇬🇲': EmojiEntry(codepoint='1F1EC 1F1F2', status='fully-qualified', emoji='🇬🇲', name='flag: Gambia', group='Flags', sub_group='country-flag'),
    '🇬🇳': EmojiEntry(codepoint='1F1EC 1F1F3', status='fully-qualified', emoji='🇬🇳', name='flag: Guinea', group='Flags', sub_group='country-flag'),
    '🇬🇵': EmojiEntry(codepoint='1F1EC 1F1F5', status='fully-qualified', emoji='🇬🇵', name='flag: Guadeloupe', group='Flags', sub_group='country-flag'),
    '🇬🇶': EmojiEntry(codepoint='1F1EC 1F1F6', status='fully-qualified', emoji='🇬🇶', name='flag: Equatorial Guinea', group='Flags', sub_group='country-flag'),
    '🇬🇷': EmojiEntry(codepoint='1F1EC 1F1F7', status='fully-qualified', emoji='🇬🇷', name='flag: Greece', group='Flags', sub_group='country-flag'),
    '🇬🇸': EmojiEntry(codepoint='1F1EC 1F1F8', status='fully-qualified', emoji='🇬🇸', name='flag: South Georgia & South Sandwich Islands', group='Flags', sub_group='country-flag'),
    '🇬🇹': EmojiEntry(codepoint='1F1EC 1F1F9', status='fully-qualified', emoji='🇬🇹', name='flag: Guatemala', group='Flags', sub_group='country-flag'),
    '🇬🇺': EmojiEntry(codepoint='1F1EC 1F1FA', status='fully-qualified', emoji='🇬🇺', name='flag: Guam', group='Flags', sub_group='country-flag'),
    '🇬🇼': EmojiEntry(codepoint='1F1EC 1F1FC', status='fully-qualified', emoji='🇬🇼', name='flag: Guinea-Bissau', group='Flags', sub_group='country-flag'),
    '🇬🇾': EmojiEntry(codepoint='1F1EC 1F1FE', status='fully-qualified', emoji='🇬🇾', name='flag: Guyana', group='Flags', sub_group='country-flag'),
    '🇭🇰': EmojiEntry(codepoint='1F1ED 1F1F0', status='fully-qualified', emoji='🇭🇰', name='flag: Hong Kong SAR China', group='Flags', sub_group='country-flag'),
    '🇭🇲': EmojiEntry(codepoint='1F1ED 1F1F2', status='fully-qualified', emoji='🇭🇲', name='flag: Heard & McDonald Islands', group='Flags', sub_group='country-flag'),
    '🇭🇳': EmojiEntry(codepoint='1F1ED 1F1F3', status='fully-qualified', emoji='🇭🇳', name='flag: Honduras', group='Flags', sub_group='country-flag'),
    '🇭🇷': EmojiEntry(codepoint='1F1ED 1F1F7', status='fully-qualified', emoji='🇭🇷', name='flag: Croatia', group='Flags', sub_group='country-flag'),
    '🇭🇹': EmojiEntry(codepoint='1F1ED 1F1F9', status='fully-qualified', emoji='🇭🇹', name='flag: Haiti', group='Flags', sub_group='country-flag'),
    '🇭🇺': EmojiEntry(codepoint='1F1ED 1F1FA', status='fully-qualified', emoji='🇭🇺', name='flag: Hungary', group='Flags', sub_group='country-flag'),
    '🇮🇨': EmojiEntry(codepoint='1F1EE 1F1E8', status='fully-qualified', emoji='🇮🇨', name='flag: Canary Islands', group='Flags', sub_group='country-flag'),
    '🇮🇩': EmojiEntry(codepoint='1F1EE 1F1E9', status='fully-qualified', emoji='🇮🇩', name='flag: Indonesia', group='Flags', sub_group='country-flag'),
    '🇮🇪': EmojiEntry(codepoint='1F1EE 1F1EA', status='fully-qualified', emoji='🇮🇪', name='flag: Ireland', group='Flags', sub_group='country-flag'),
    '🇮🇱': EmojiEntry(codepoint='1F1EE 1F1F1', status='fully-qualified', emoji='🇮🇱', name='flag: Israel', group='Flags', sub_group='country-flag'),
    '🇮🇲': EmojiEntry(codepoint='1F1EE 1F1F2', status='fully-qualified', emoji='🇮🇲', name='flag: Isle of Man', group='Flags', sub_group='country-flag'),
    '🇮🇳': EmojiEntry(codepoint='1F1EE 1F1F3', status='fully-qualified', emoji='🇮🇳', name='flag: India', group='Flags', sub_group='country-flag'),
    '🇮🇴': EmojiEntry(codepoint='1F1EE 1F1F4', status='fully-qualified', emoji='🇮🇴', name='flag: British Indian Ocean Territory', group='Flags', sub_group='country-flag'),
    '🇮🇶': EmojiEntry(codepoint='1F1EE 1F1F6', status='fully-qualified', emoji='🇮🇶', name='flag: Iraq', group='Flags', sub_group='country-flag'),
    '🇮🇷': EmojiEntry(codepoint='1F1EE 1F1F7', status='fully-qualified', emoji='🇮🇷', name='flag: Iran', group='Flags', sub_group='country-flag'),
    '🇮🇸': EmojiEntry(codepoint='1F1EE 1F1F8', status='fully-qualified', emoji='🇮🇸', name='flag: Iceland', group='Flags', sub_group='country-flag'),
    '🇮🇹': EmojiEntry(codepoint='1F1EE 1F1F9', status='fully-qualified', emoji='🇮🇹', name='flag: Italy', group='Flags', sub_group='country-flag'),
    '🇯🇪': EmojiEntry(codepoint='1F1EF 1F1EA', status='fully-qualified', emoji='🇯🇪', name='flag: Jersey', group='Flags', sub_group='country-flag'),
    '🇯🇲': EmojiEntry(codepoint='1F1EF 1F1F2', status='fully-qualified', emoji='🇯🇲', name='flag: Jamaica', group='Flags', sub_group='country-flag'),
    '🇯🇴': EmojiEntry(codepoint='1F1EF 1F1F4', status='fully-qualified', emoji='🇯🇴', name='flag: Jordan', group='Flags', sub_group='country-flag'),
    '🇯🇵': EmojiEntry(codepoint='1F1EF 1F1F5', status='fully-qualified', emoji='🇯🇵', name='flag: Japan', group='Flags', sub_group='country-flag'),
    '🇰🇪': EmojiEntry(codepoint='1F1F0 1F1EA', status='fully-qualified', emoji='🇰🇪', name='flag: Kenya', group='Flags', sub_group='country-flag'),
    '🇰🇬': EmojiEntry(codepoint='1F1F0 1F1EC', status='fully-qualified', emoji='🇰🇬', name='flag: Kyrgyzstan', group='Flags', sub_group='country-flag'),
    '🇰🇭': EmojiEntry(codepoint='1F1F0 1F1ED', status='fully-qualified', emoji='🇰🇭', name='flag: Cambodia', group='Flags', sub_group='country-flag'),
    '🇰🇮': EmojiEntry(codepoint='1F1F0 1F1EE', status='fully-qualified', emoji='🇰🇮', name='flag: Kiribati', group='Flags', sub_group='country-flag'),
    '🇰🇲': EmojiEntry(codepoint='1F1F0 1F1F2', status='fully-qualified', emoji='🇰🇲', name='flag: Comoros', group='Flags', sub_group='country-flag'),
    '🇰🇳': EmojiEntry(codepoint='1F1F0 1F1F3', status='fully-qualified', emoji='🇰🇳', name='flag: St. Kitts & Nevis', group='Flags', sub_group='country-flag'),
    '🇰🇵': EmojiEntry(codepoint='1F1F0 1F1F5', status='fully-qualified', emoji='🇰🇵', name='flag: North Korea', group='Flags', sub_group='country-flag'),
    '🇰🇷': EmojiEntry(codepoint='1F1F0 1F1F7', status='fully-qualified', emoji='🇰🇷', name='flag: South Korea', group='Flags', sub_group='country-flag'),
    '🇰🇼': EmojiEntry(codepoint='1F1F0 1F1FC', status='fully-qualified', emoji='🇰🇼', name='flag: Kuwait', group='Flags', sub_group='country-flag'),
    '🇰🇾': EmojiEntry(codepoint='1F1F0 1F1FE', status='fully-qualified', emoji='🇰🇾', name='flag: Cayman Islands', group='Flags', sub_group='country-flag'),
    '🇰🇿': EmojiEntry(codepoint='1F1F0 1F1FF', status='fully-qualified', emoji='🇰🇿', name='flag: Kazakhstan', group='Flags', sub_group='country-flag'),
    '🇱🇦': EmojiEntry(codepoint='1F1F1 1F1E6', status='fully-qualified', emoji='🇱🇦', name='flag: Laos', group='Flags', sub_group='country-flag'),
    '🇱🇧': EmojiEntry(codepoint='1F1F1 1F1E7', status='fully-qualified', emoji='🇱🇧', name='flag: Lebanon', group='Flags', sub_group='country-flag'),
    '🇱🇨': EmojiEntry(codepoint='1F1F1 1F1E8', status='fully-qualified', emoji='🇱🇨', name='flag: St. Lucia', group='Flags', sub_group='country-flag'),
    '🇱🇮': EmojiEntry(codepoint='1F1F1 1F1EE', status='fully-qualified', emoji='🇱🇮', name='flag: Liechtenstein', group='Flags', sub_group='country-flag'),
    '🇱🇰': EmojiEntry(codepoint='1F1F1 1F1F0', status='fully-qualified', emoji='🇱🇰', name='flag: Sri Lanka', group='Flags', sub_group='country-flag'),
    '🇱🇷': EmojiEntry(codepoint='1F1F1 1F1F7', status='fully-qualified', emoji='🇱🇷', name='flag: Liberia', group='Flags', sub_group='country-flag'),
    '🇱🇸': EmojiEntry(codepoint='1F1F1 1F1F8', status='fully-qualified', emoji='🇱🇸', name='flag: Lesotho', group='Flags', sub_group='country-flag'),
    '🇱🇹': EmojiEntry(codepoint='1F1F1 1F1F9', status='fully-qualified', emoji='🇱🇹', name='flag: Lithuania', group='Flags', sub_group='country-flag'),
    '🇱🇺': EmojiEntry(codepoint='1F1F1 1F1FA', status='fully-qualified', emoji='🇱🇺', name='flag: Luxembourg', group='Flags', sub_group='country-flag'),
    '🇱🇻': EmojiEntry(codepoint='1F1F1 1F1FB', status='fully-qualified', emoji='🇱🇻', name='flag: Latvia', group='Flags', sub_group='country-flag'),
    '🇱🇾': EmojiEntry(codepoint='1F1F1 1F1FE', status='fully-qualified', emoji='🇱🇾', name='flag: Libya', group='Flags', sub_group='country-flag'),
    '🇲🇦': EmojiEntry(codepoint='1F1F2 1F1E6', status='fully-qualified', emoji='🇲🇦', name='flag: Morocco', group='Flags', sub_group='country-flag'),
    '🇲🇨': EmojiEntry(codepoint='1F1F2 1F1E8', status='fully-qualified', emoji='🇲🇨', name='flag: Monaco', group='Flags', sub_group='country-flag'),
    '🇲🇩': EmojiEntry(codepoint='1F1F2 1F1E9', status='fully-qualified', emoji='🇲🇩', name='flag: Moldova', group='Flags', sub_group='country-flag'),
    '🇲🇪': EmojiEntry(codepoint='1F1F2 1F1EA', status='fully-qualified', emoji='🇲🇪', name='flag: Montenegro', group='Flags', sub_group='country-flag'),
    '🇲🇫': EmojiEntry(codepoint='1F1F2 1F1EB', status='fully-qualified', emoji='🇲🇫', name='flag: St. Martin', group='Flags', sub_group='country-flag'),
    '🇲🇬': EmojiEntry(codepoint='1F1F2 1F1EC', status='fully-qualified', emoji='🇲🇬', name='flag: Madagascar', group='Flags', sub_group='country-flag'),
    '🇲🇭': EmojiEntry(codepoint='1F1F2 1F1ED', status='fully-qualified', emoji='🇲🇭', name='flag: Marshall Islands', group='Flags', sub_group='country-flag'),
    '🇲🇰': EmojiEntry(codepoint='1F1F2 1F1F0', status='fully-qualified', emoji='🇲🇰', name='flag: North Macedonia', group='Flags', sub_group='country-flag'),
    '🇲🇱': EmojiEntry(codepoint='1F1F2 1F1F1', status='fully-qualified', emoji='🇲🇱', name='flag: Mali', group='Flags', sub_group='country-flag'),
    '🇲🇲': EmojiEntry(codepoint='1F1F2 1F1F2', status='fully-qualified', emoji='🇲🇲', name='flag: Myanmar (Burma)', group='Flags', sub_group='country-flag'),
    '🇲🇳': EmojiEntry(codepoint='1F1F2 1F1F3', status='fully-qualified', emoji='🇲🇳', name='flag: Mongolia', group='Flags', sub_group='country-flag'),
    '🇲🇴': EmojiEntry(codepoint='1F1F2 1F1F4', status='fully-qualified', emoji='🇲🇴', name='flag: Macao SAR China', group='Flags', sub_group='country-flag'),
    '🇲🇵': EmojiEntry(codepoint='1F1F2 1F1F5', status='fully-qualified', emoji='🇲🇵', name='flag: Northern Mariana Islands', group='Flags', sub_group='country-flag'),
    '🇲🇶': EmojiEntry(codepoint='1F1F2 1F1F6', status='fully-qualified', emoji='🇲🇶', name='flag: Martinique', group='Flags', sub_group='country-flag'),
    '🇲🇷': EmojiEntry(codepoint='1F1F2 1F1F7', status='fully-qualified', emoji='🇲🇷', name='flag: Mauritania', group='Flags', sub_group='country-flag'),
    '🇲🇸': EmojiEntry(codepoint='1F1F2 1F1F8', status='fully-qualified', emoji='🇲🇸', name='flag: Montserrat', group='Flags', sub_group='country-flag'),
    '🇲🇹': EmojiEntry(codepoint='1F1F2 1F1F9', status='fully-qualified', emoji='🇲🇹', name='flag: Malta', group='Flags', sub_group='country-flag'),
    '🇲🇺': EmojiEntry(codepoint='1F1F2 1F1FA', status='fully-qualified', emoji='🇲🇺', name='flag: Mauritius', group='Flags', sub_group='country-flag'),
    '🇲🇻': EmojiEntry(codepoint='1F1F2 1F1FB', status='fully-qualified', emoji='🇲🇻', name='flag: Maldives', group='Flags', sub_group='country-flag'),
    '🇲🇼': EmojiEntry(codepoint='1F1F2 1F1FC', status='fully-qualified', emoji='🇲🇼', name='flag: Malawi', group='Flags', sub_group='country-flag'),
    '🇲🇽': EmojiEntry(codepoint='1F1F2 1F1FD', status='fully-qualified', emoji='🇲🇽', name='flag: Mexico', group='Flags', sub_group='country-flag'),
    '🇲🇾': EmojiEntry(codepoint='1F1F2 1F1FE', status='fully-qualified', emoji='🇲🇾', name='flag: Malaysia', group='Flags', sub_group='country-flag'),
    '🇲🇿': EmojiEntry(codepoint='1F1F2 1F1FF', status='fully-qualified', emoji='🇲🇿', name='flag: Mozambique', group='Flags', sub_group='country-flag'),
    '🇳🇦': EmojiEntry(codepoint='1F1F3 1F1E6', status='fully-qualified', emoji='🇳🇦', name='flag: Namibia', group='Flags', sub_group='country-flag'),
    '🇳🇨': EmojiEntry(codepoint='1F1F3 1F1E8', status='fully-qualified', emoji='🇳🇨', name='flag: New Caledonia', group='Flags', sub_group='country-flag'),
    '🇳🇪': EmojiEntry(codepoint='1F1F3 1F1EA', status='fully-qualified', emoji='🇳🇪', name='flag: Niger', group='Flags', sub_group='country-flag'),
    '🇳🇫': EmojiEntry(codepoint='1F1F3 1F1EB', status='fully-qualified', emoji='🇳🇫', name='flag: Norfolk Island', group='Flags', sub_group='country-flag'),
    '🇳🇬': EmojiEntry(codepoint='1F1F3 1F1EC', status='fully-qualified', emoji='🇳🇬', name='flag: Nigeria', group='Flags', sub_group='country-flag'),
    '🇳🇮': EmojiEntry(codepoint='1F1F3 1F1EE', status='fully-qualified', emoji='🇳🇮', name='flag: Nicaragua', group='Flags', sub_group='country-flag'),
    '🇳🇱': EmojiEntry(codepoint='1F1F3 1F1F1', status='fully-qualified', emoji='🇳🇱', name='flag: Netherlands', group='Flags', sub_group='country-flag'),
    '🇳🇴': EmojiEntry(codepoint='1F1F3 1F1F4', status='fully-qualified', emoji='🇳🇴', name='flag: Norway', group='Flags', sub_group='country-flag'),
    '🇳🇵': EmojiEntry(codepoint='1F1F3 1F1F5', status='fully-qualified', emoji='🇳🇵', name='flag: Nepal', group='Flags', sub_group='country-flag'),
    '🇳🇷': EmojiEntry(codepoint='1F1F3 1F1F7', status='fully-qualified', emoji='🇳🇷', name='flag: Nauru', group='Flags', sub_group='country-flag'),
    '🇳🇺': EmojiEntry(codepoint='1F1F3 1F1FA', status='fully-qualified', emoji='🇳🇺', name='flag: Niue', group='Flags', sub_group='country-flag'),
    '🇳🇿': EmojiEntry(codepoint='1F1F3 1F1FF', status='fully-qualified', emoji='🇳🇿', name='flag: New Zealand', group='Flags', sub_group='country-flag'),
    '🇴🇲': EmojiEntry(codepoint='1F1F4 1F1F2', status='fully-qualified', emoji='🇴🇲', name='flag: Oman', group='Flags', sub_group='country-flag'),
    '🇵🇦': EmojiEntry(codepoint='1F1F5 1F1E6', status='fully-qualified', emoji='🇵🇦', name='flag: Panama', group='Flags', sub_group='country-flag'),
    '🇵🇪': EmojiEntry(codepoint='1F1F5 1F1EA', status='fully-qualified', emoji='🇵🇪', name='flag: Peru', group='Flags', sub_group='country-flag'),
    '🇵🇫': EmojiEntry(codepoint='1F1F5 1F1EB', status='fully-qualified', emoji='🇵🇫', name='flag: French Polynesia', group='Flags', sub_group='country-flag'),
    '🇵🇬': EmojiEntry(codepoint='1F1F5 1F1EC', status='fully-qualified', emoji='🇵🇬', name='flag: Papua New Guinea', group='Flags', sub_group='country-flag'),
    '🇵🇭': EmojiEntry(codepoint='1F1F5 1F1ED', status='fully-qualified', emoji='🇵🇭', name='flag: Philippines', group='Flags', sub_group='country-flag'),
    '🇵🇰': EmojiEntry(codepoint='1F1F5 1F1F0', status='fully-qualified', emoji='🇵🇰', name='flag: Pakistan', group='Flags', sub_group='country-flag'),
    '🇵🇱': EmojiEntry(codepoint='1F1F5 1F1F1', status='fully-qualified', emoji='🇵🇱', name='flag: Poland', group='Flags', sub_group='country-flag'),
    '🇵🇲': EmojiEntry(codepoint='1F1F5 1F1F2', status='fully-qualified', emoji='🇵🇲', name='flag: St. Pierre & Miquelon', group='Flags', sub_group='country-flag'),
    '🇵🇳': EmojiEntry(codepoint='1F1F5 1F1F3', status='fully-qualified', emoji='🇵🇳', name='flag: Pitcairn Islands', group='Flags', sub_group='country-flag'),
    '🇵🇷': EmojiEntry(codepoint='1F1F5 1F1F7', status='fully-qualified', emoji='🇵🇷', name='flag: Puerto Rico', group='Flags', sub_group='country-flag'),
    '🇵🇸': EmojiEntry(codepoint='1F1F5 1F1F8', status='fully-qualified', emoji='🇵🇸', name='flag: Palestinian Territories', group='Flags', sub_group='country-flag'),
    '🇵🇹': EmojiEntry(codepoint='1F1F5 1F1F9', status='fully-qualified', emoji='🇵🇹', name='flag: Portugal', group='Flags', sub_group='country-flag'),
    '🇵🇼': EmojiEntry(codepoint='1F1F5 1F1FC', status='fully-qualified', emoji='🇵🇼', name='flag: Palau', group='Flags', sub_group='country-flag'),
    '🇵🇾': EmojiEntry(codepoint='1F1F5 1F1FE', status='fully-qualified', emoji='🇵🇾', name='flag: Paraguay', group='Flags', sub_group='country-flag'),
    '🇶🇦': EmojiEntry(codepoint='1F1F6 1F1E6', status='fully-qualified', emoji='🇶🇦', name='flag: Qatar', group='Flags', sub_group='country-flag'),
    '🇷🇪': EmojiEntry(codepoint='1F1F7 1F1EA', status='fully-qualified', emoji='🇷🇪', name='flag: Réunion', group='Flags', sub_group='country-flag'),
    '🇷🇴': EmojiEntry(codepoint='1F1F7 1F1F4', status='fully-qualified', emoji='🇷🇴', name='flag: Romania', group='Flags', sub_group='country-flag'),
    '🇷🇸': EmojiEntry(codepoint='1F1F7 1F1F8', status='fully-qualified', emoji='🇷🇸', name='flag: Serbia', group='Flags', sub_group='country-flag'),
    '🇷🇺': EmojiEntry(codepoint='1F1F7 1F1FA', status='fully-qualified', emoji='🇷🇺', name='flag: Russia', group='Flags', sub_group='country-flag'),
    '🇷🇼': EmojiEntry(codepoint='1F1F7 1F1FC', status='fully-qualified', emoji='🇷🇼', name='flag: Rwanda', group='Flags', sub_group='country-flag'),
    '🇸🇦': EmojiEntry(codepoint='1F1F8 1F1E6', status='fully-qualified', emoji='🇸🇦', name='flag: Saudi Arabia', group='Flags', sub_group='country-flag'),
    '🇸🇧': EmojiEntry(codepoint='1F1F8 1F1E7', status='fully-qualified', emoji='🇸🇧', name='flag: Solomon Islands', group='Flags', sub_group='country-flag'),
    '🇸🇨': EmojiEntry(codepoint='1F1F8 1F1E8', status='fully-qualified', emoji='🇸🇨', name='flag: Seychelles', group='Flags', sub_group='country-flag'),
    '🇸🇩': EmojiEntry(codepoint='1F1F8 1F1E9', status='fully-qualified', emoji='🇸🇩', name='flag: Sudan', group='Flags', sub_group='country-flag'),
    '🇸🇪': EmojiEntry(codepoint='1F1F8 1F1EA', status='fully-qualified', emoji='🇸🇪', name='flag: Sweden', group='Flags', sub_group='country-flag'),
    '🇸🇬': EmojiEntry(codepoint='1F1F8 1F1EC', status='fully-qualified', emoji='🇸🇬', name='flag: Singapore', group='Flags', sub_group='country-flag'),
    '🇸🇭': EmojiEntry(codepoint='1F1F8 1F1ED', status='fully-qualified', emoji='🇸🇭', name='flag: St. Helena', group='Flags', sub_group='country-flag'),
    '🇸🇮': EmojiEntry(codepoint='1F1F8 1F1EE', status='fully-qualified', emoji='🇸🇮', name='flag: Slovenia', group='Flags', sub_group='country-flag'),
    '🇸🇯': EmojiEntry(codepoint='1F1F8 1F1EF', status='fully-qualified', emoji='🇸🇯', name='flag: Svalbard & Jan Mayen', group='Flags', sub_group='country-flag'),
    '🇸🇰': EmojiEntry(codepoint='1F1F8 1F1F0', status='fully-qualified', emoji='🇸🇰', name='flag: Slovakia', group='Flags', sub_group='country-flag'),
    '🇸🇱': EmojiEntry(codepoint='1F1F8 1F1F1', status='fully-qualified', emoji='🇸🇱', name='flag: Sierra Leone', group='Flags', sub_group='country-flag'),
    '🇸🇲': EmojiEntry(codepoint='1F1F8 1F1F2', status='fully-qualified', emoji='🇸🇲', name='flag: San Marino', group='Flags', sub_group='country-flag'),
    '🇸🇳': EmojiEntry(codepoint='1F1F8 1F1F3', status='fully-qualified', emoji='🇸🇳', name='flag: Senegal', group='Flags', sub_group='country-flag'),
    '🇸🇴': EmojiEntry(codepoint='1F1F8 1F1F4', status='fully-qualified', emoji='🇸🇴', name='flag: Somalia', group='Flags', sub_group='country-flag'),
    '🇸🇷': EmojiEntry(codepoint='1F1F8 1F1F7', status='fully-qualified', emoji='🇸🇷', name='flag: Suriname', group='Flags', sub_group='country-flag'),
    '🇸🇸': EmojiEntry(codepoint='1F1F8 1F1F8', status='fully-qualified', emoji='🇸🇸', name='flag: South Sudan', group='Flags', sub_group='country-flag'),
    '🇸🇹': EmojiEntry(codepoint='1F1F8 1F1F9', status='fully-qualified', emoji='🇸🇹', name='flag: São Tomé & Príncipe', group='Flags', sub_group='country-flag'),
    '🇸🇻': EmojiEntry(codepoint='1F1F8 1F1FB', status='fully-qualified', emoji='🇸🇻', name='flag: El Salvador', group='Flags', sub_group='country-flag'),
    '🇸🇽': EmojiEntry(codepoint='1F1F8 1F1FD', status='fully-qualified', emoji='🇸🇽', name='flag: Sint Maarten', group='Flags', sub_group='country-flag'),
    '🇸🇾': EmojiEntry(codepoint='1F1F8 1F1FE', status='fully-qualified', emoji='🇸🇾', name='flag: Syria', group='Flags', sub_group='country-flag'),
    '🇸🇿': EmojiEntry(codepoint='1F1F8 1F1FF', status='fully-qualified', emoji='🇸🇿', name='flag: Eswatini', group='Flags', sub_group='country-flag'),
    '🇹🇦': EmojiEntry(codepoint='1F1F9 1F1E6', status='fully-qualified', emoji='🇹🇦', name='flag: Tristan da Cunha', group='Flags', sub_group='country-flag'),
    '🇹🇨': EmojiEntry(codepoint='1F1F9 1F1E8', status='fully-qualified', emoji='🇹🇨', name='flag: Turks & Caicos Islands', group='Flags', sub_group='country-flag'),
    '🇹🇩': EmojiEntry(codepoint='1F1F9 1F1E9', status='fully-qualified', emoji='🇹🇩', name='flag: Chad', group='Flags', sub_group='country-flag'),
    '🇹🇫': EmojiEntry(codepoint='1F1F9 1F1EB', status='fully-qualified', emoji='🇹🇫', name='flag: French Southern Territories', group='Flags', sub_group='country-flag'),
    '🇹🇬': EmojiEntry(codepoint='1F1F9 1F1EC', status='fully-qualified', emoji='🇹🇬', name='flag: Togo', group='Flags', sub_group='country-flag'),
    '🇹🇭': EmojiEntry(codepoint='1F1F9 1F1ED', status='fully-qualified', emoji='🇹🇭', name='flag: Thailand', group='Flags', sub_group='country-flag'),
    '🇹🇯': EmojiEntry(codepoint='1F1F9 1F1EF', status='fully-qualified', emoji='🇹🇯', name='flag: Tajikistan', group='Flags', sub_group='country-flag'),
    '🇹🇰': EmojiEntry(codepoint='1F1F9 1F1F0', status='fully-qualified', emoji='🇹🇰', name='flag: Tokelau', group='Flags', sub_group='country-flag'),
    '🇹🇱': EmojiEntry(codepoint='1F1F9 1F1F1', status='fully-qualified', emoji='🇹🇱', name='flag: Timor-Leste', group='Flags', sub_group='country-flag'),
    '🇹🇲': EmojiEntry(codepoint='1F1F9 1F1F2', status='fully-qualified', emoji='🇹🇲', name='flag: Turkmenistan', group='Flags', sub_group='country-flag'),
    '🇹🇳': EmojiEntry(codepoint='1F1F9 1F1F3', status='fully-qualified', emoji='🇹🇳', name='flag: Tunisia', group='Flags', sub_group='country-flag'),
    '🇹🇴': EmojiEntry(codepoint='1F1F9 1F1F4', status='fully-qualified', emoji='🇹🇴', name='flag: Tonga', group='Flags', sub_group='country-flag'),
    '🇹🇷': EmojiEntry(codepoint='1F1F9 1F1F7', status='fully-qualified', emoji='🇹🇷', name='flag: Turkey', group='Flags', sub_group='country-flag'),
    '🇹🇹': EmojiEntry(codepoint='1F1F9 1F1F9', status='fully-qualified', emoji='🇹🇹', name='flag: Trinidad & Tobago', group='Flags', sub_group='country-flag'),
    '🇹🇻': EmojiEntry(codepoint='1F1F9 1F1FB', status='fully-qualified', emoji='🇹🇻', name='flag: Tuvalu', group='Flags', sub_group='country-flag'),
    '🇹🇼': EmojiEntry(codepoint='1F1F9 1F1FC', status='fully-qualified', emoji='🇹🇼', name='flag: Taiwan', group='Flags', sub_group='country-flag'),
    '🇹🇿': EmojiEntry(codepoint='1F1F9 1F1FF', status='fully-qualified', emoji='🇹🇿', name='flag: Tanzania', group='Flags', sub_group='country-flag'),
    '🇺🇦': EmojiEntry(codepoint='1F1FA 1F1E6', status='fully-qualified', emoji='🇺🇦', name='flag: Ukraine', group='Flags', sub_group='country-flag'),
    '🇺🇬': EmojiEntry(codepoint='1F1FA 1F1EC', status='fully-qualified', emoji='🇺🇬', name='flag: Uganda', group='Flags', sub_group='country-flag'),
    '🇺🇲': EmojiEntry(codepoint='1F1FA 1F1F2', status='fully-qualified', emoji='🇺🇲', name='flag: U.S. Outlying Islands', group='Flags', sub_group='country-flag'),
    '🇺🇳': EmojiEntry(codepoint='1F1FA 1F1F3', status='fully-qualified', emoji='🇺🇳', name='flag: United Nations', group='Flags', sub_group='country-flag'),
    '🇺🇸': EmojiEntry(codepoint='1F1FA 1F1F8', status='fully-qualified', emoji='🇺🇸', name='flag: United States', group='Flags', sub_group='country-flag'),
    '🇺🇾': EmojiEntry(codepoint='1F1FA 1F1FE', status='fully-qualified', emoji='🇺🇾', name='flag: Uruguay', group='Flags', sub_group='country-flag'),
    '🇺🇿': EmojiEntry(codepoint='1F1FA 1F1FF', status='fully-qualified', emoji='🇺🇿', name='flag: Uzbekistan', group='Flags', sub_group='country-flag'),
    '🇻🇦': EmojiEntry(codepoint='1F1FB 1F1E6', status='fully-qualified', emoji='🇻🇦', name='flag: Vatican City', group='Flags', sub_group='country-flag'),
    '🇻🇨': EmojiEntry(codepoint='1F1FB 1F1E8', status='fully-qualified', emoji='🇻🇨', name='flag: St. Vincent & Grenadines', group='Flags', sub_group='country-flag'),
    '🇻🇪': EmojiEntry(codepoint='1F1FB 1F1EA', status='fully-qualified', emoji='🇻🇪', name='flag: Venezuela', group='Flags', sub_group='country-flag'),
    '🇻🇬': EmojiEntry(codepoint='1F1FB 1F1EC', status='fully-qualified', emoji='🇻🇬', name='flag: British Virgin Islands', group='Flags', sub_group='country-flag'),
    '🇻🇮': EmojiEntry(codepoint='1F1FB 1F1EE', status='fully-qualified', emoji='🇻🇮', name='flag: U.S. Virgin Islands', group='Flags', sub_group='country-flag'),
    '🇻🇳': EmojiEntry(codepoint='1F1FB 1F1F3', status='fully-qualified', emoji='🇻🇳', name='flag: Vietnam', group='Flags', sub_group='country-flag'),
    '🇻🇺': EmojiEntry(codepoint='1F1FB 1F1FA', status='fully-qualified', emoji='🇻🇺', name='flag: Vanuatu', group='Flags', sub_group='country-flag'),
    '🇼🇫': EmojiEntry(codepoint='1F1FC 1F1EB', status='fully-qualified', emoji='🇼🇫', name='flag: Wallis & Futuna', group='Flags', sub_group='country-flag'),
    '🇼🇸': EmojiEntry(codepoint='1F1FC 1F1F8', status='fully-qualified', emoji='🇼🇸', name='flag: Samoa', group='Flags', sub_group='country-flag'),
    '🇽🇰': EmojiEntry(codepoint='1F1FD 1F1F0', status='fully-qualified', emoji='🇽🇰', name='flag: Kosovo', group='Flags', sub_group='country-flag'),
    '🇾🇪': EmojiEntry(codepoint='1F1FE 1F1EA', status='fully-qualified', emoji='🇾🇪', name='flag: Yemen', group='Flags', sub_group='country-flag'),
    '🇾🇹': EmojiEntry(codepoint='1F1FE 1F1F9', status='fully-qualified', emoji='🇾🇹', name='flag: Mayotte', group='Flags', sub_group='country-flag'),
    '🇿🇦': EmojiEntry(codepoint='1F1FF 1F1E6', status='fully-qualified', emoji='🇿🇦', name='flag: South Africa', group='Flags', sub_group='country-flag'),
    '🇿🇲': EmojiEntry(codepoint='1F1FF 1F1F2', status='fully-qualified', emoji='🇿🇲', name='flag: Zambia', group='Flags', sub_group='country-flag'),
    '🇿🇼': EmojiEntry(codepoint='1F1FF 1F1FC', status='fully-qualified', emoji='🇿🇼', name='flag: Zimbabwe', group='Flags', sub_group='country-flag'),
    '🏴󠁧󠁢󠁥󠁮󠁧󠁿': EmojiEntry(codepoint='1F3F4 E0067 E0062 E0065 E006E E0067 E007F', status='fully-qualified', emoji='🏴\U000e0067\U000e0062\U000e0065\U000e006e\U000e0067\U000e007f', name='flag: England', group='Flags', sub_group='subdivision-flag'),
    '🏴󠁧󠁢󠁳󠁣󠁴󠁿': EmojiEntry(codepoint='1F3F4 E0067 E0062 E0073 E0063 E0074 E007F', status='fully-qualified', emoji='🏴\U000e0067\U000e0062\U000e0073\U000e0063\U000e0074\U000e007f', name='flag: Scotland', group='Flags', sub_group='subdivision-flag'),
    '🏴󠁧󠁢󠁷󠁬󠁳󠁿': EmojiEntry(codepoint='1F3F4 E0067 E0062 E0077 E006C E0073 E007F', status='fully-qualified', emoji='🏴\U000e0067\U000e0062\U000e0077\U000e006c\U000e0073\U000e007f', name='flag: Wales', group='Flags', sub_group='subdivision-flag'),
}

# EMOJI_RAW = r'\U0001F469\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468|\U0001F468\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468|\U0001F469\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469|\U0001F9D1\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FF|\U0001F469\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468|\U0001F468\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468|\U0001F469\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469|\U0001F468\U0000200D\U0001F469\U0000200D\U0001F467\U0000200D\U0001F466|\U0001F468\U0000200D\U0001F469\U0000200D\U0001F466\U0000200D\U0001F466|\U0001F468\U0000200D\U0001F469\U0000200D\U0001F467\U0000200D\U0001F467|\U0001F468\U0000200D\U0001F468\U0000200D\U0001F467\U0000200D\U0001F466|\U0001F468\U0000200D\U0001F468\U0000200D\U0001F466\U0000200D\U0001F466|\U0001F468\U0000200D\U0001F468\U0000200D\U0001F467\U0000200D\U0001F467|\U0001F469\U0000200D\U0001F469\U0000200D\U0001F467\U0000200D\U0001F466|\U0001F469\U0000200D\U0001F469\U0000200D\U0001F466\U0000200D\U0001F466|\U0001F469\U0000200D\U0001F469\U0000200D\U0001F467\U0000200D\U0001F467|\U0001F3F4\U000E0067\U000E0062\U000E0065\U000E006E\U000E0067\U000E007F|\U0001F3F4\U000E0067\U000E0062\U000E0073\U000E0063\U000E0074\U000E007F|\U0001F3F4\U000E0067\U000E0062\U000E0077\U000E006C\U000E0073\U000E007F|\U0001F469\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468|\U0001F468\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468|\U0001F469\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469|\U0001F441\U0000FE0F\U0000200D\U0001F5E8\U0000FE0F|\U0001F471\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F471\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F471\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F471\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F471\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F471\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F471\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F471\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F471\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F471\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F64D\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F64D\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F64D\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F64D\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F64D\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F64D\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F64D\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F64D\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F64D\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F64D\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F64E\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F64E\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F64E\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F64E\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F64E\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F64E\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F64E\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F64E\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F64E\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F64E\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F645\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F645\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F645\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F645\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F645\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F645\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F645\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F645\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F645\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F645\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F646\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F646\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F646\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F646\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F646\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F646\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F646\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F646\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F646\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F646\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F481\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F481\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F481\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F481\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F481\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F481\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F481\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F481\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F481\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F481\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F64B\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F64B\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F64B\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F64B\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F64B\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F64B\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F64B\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F64B\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F64B\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F64B\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9CF\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9CF\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9CF\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9CF\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9CF\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9CF\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9CF\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9CF\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9CF\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9CF\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F647\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F647\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F647\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F647\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F647\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F647\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F647\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F647\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F647\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F647\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F926\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F926\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F926\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F926\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F926\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F926\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F926\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F926\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F926\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F926\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F937\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F937\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F937\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F937\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F937\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F937\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F937\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F937\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F937\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F937\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9D1\U0001F3FB\U0000200D\U00002695\U0000FE0F|\U0001F9D1\U0001F3FC\U0000200D\U00002695\U0000FE0F|\U0001F9D1\U0001F3FD\U0000200D\U00002695\U0000FE0F|\U0001F9D1\U0001F3FE\U0000200D\U00002695\U0000FE0F|\U0001F9D1\U0001F3FF\U0000200D\U00002695\U0000FE0F|\U0001F468\U0001F3FB\U0000200D\U00002695\U0000FE0F|\U0001F468\U0001F3FC\U0000200D\U00002695\U0000FE0F|\U0001F468\U0001F3FD\U0000200D\U00002695\U0000FE0F|\U0001F468\U0001F3FE\U0000200D\U00002695\U0000FE0F|\U0001F468\U0001F3FF\U0000200D\U00002695\U0000FE0F|\U0001F469\U0001F3FB\U0000200D\U00002695\U0000FE0F|\U0001F469\U0001F3FC\U0000200D\U00002695\U0000FE0F|\U0001F469\U0001F3FD\U0000200D\U00002695\U0000FE0F|\U0001F469\U0001F3FE\U0000200D\U00002695\U0000FE0F|\U0001F469\U0001F3FF\U0000200D\U00002695\U0000FE0F|\U0001F9D1\U0001F3FB\U0000200D\U00002696\U0000FE0F|\U0001F9D1\U0001F3FC\U0000200D\U00002696\U0000FE0F|\U0001F9D1\U0001F3FD\U0000200D\U00002696\U0000FE0F|\U0001F9D1\U0001F3FE\U0000200D\U00002696\U0000FE0F|\U0001F9D1\U0001F3FF\U0000200D\U00002696\U0000FE0F|\U0001F468\U0001F3FB\U0000200D\U00002696\U0000FE0F|\U0001F468\U0001F3FC\U0000200D\U00002696\U0000FE0F|\U0001F468\U0001F3FD\U0000200D\U00002696\U0000FE0F|\U0001F468\U0001F3FE\U0000200D\U00002696\U0000FE0F|\U0001F468\U0001F3FF\U0000200D\U00002696\U0000FE0F|\U0001F469\U0001F3FB\U0000200D\U00002696\U0000FE0F|\U0001F469\U0001F3FC\U0000200D\U00002696\U0000FE0F|\U0001F469\U0001F3FD\U0000200D\U00002696\U0000FE0F|\U0001F469\U0001F3FE\U0000200D\U00002696\U0000FE0F|\U0001F469\U0001F3FF\U0000200D\U00002696\U0000FE0F|\U0001F9D1\U0001F3FB\U0000200D\U00002708\U0000FE0F|\U0001F9D1\U0001F3FC\U0000200D\U00002708\U0000FE0F|\U0001F9D1\U0001F3FD\U0000200D\U00002708\U0000FE0F|\U0001F9D1\U0001F3FE\U0000200D\U00002708\U0000FE0F|\U0001F9D1\U0001F3FF\U0000200D\U00002708\U0000FE0F|\U0001F468\U0001F3FB\U0000200D\U00002708\U0000FE0F|\U0001F468\U0001F3FC\U0000200D\U00002708\U0000FE0F|\U0001F468\U0001F3FD\U0000200D\U00002708\U0000FE0F|\U0001F468\U0001F3FE\U0000200D\U00002708\U0000FE0F|\U0001F468\U0001F3FF\U0000200D\U00002708\U0000FE0F|\U0001F469\U0001F3FB\U0000200D\U00002708\U0000FE0F|\U0001F469\U0001F3FC\U0000200D\U00002708\U0000FE0F|\U0001F469\U0001F3FD\U0000200D\U00002708\U0000FE0F|\U0001F469\U0001F3FE\U0000200D\U00002708\U0000FE0F|\U0001F469\U0001F3FF\U0000200D\U00002708\U0000FE0F|\U0001F46E\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F46E\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F46E\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F46E\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F46E\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F46E\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F46E\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F46E\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F46E\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F46E\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F575\U0000FE0F\U0000200D\U00002642\U0000FE0F|\U0001F575\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F575\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F575\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F575\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F575\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F575\U0000FE0F\U0000200D\U00002640\U0000FE0F|\U0001F575\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F575\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F575\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F575\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F575\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F482\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F482\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F482\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F482\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F482\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F482\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F482\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F482\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F482\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F482\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F477\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F477\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F477\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F477\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F477\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F477\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F477\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F477\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F477\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F477\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F473\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F473\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F473\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F473\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F473\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F473\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F473\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F473\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F473\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F473\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F935\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F935\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F935\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F935\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F935\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F935\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F935\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F935\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F935\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F935\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F470\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F470\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F470\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F470\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F470\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F470\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F470\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F470\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F470\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F470\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9B8\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9B8\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9B8\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9B8\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9B8\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9B8\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9B8\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9B8\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9B8\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9B8\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9B9\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9B9\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9B9\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9B9\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9B9\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9B9\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9B9\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9B9\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9B9\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9B9\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9D9\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9D9\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9D9\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9D9\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9D9\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9D9\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9D9\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9D9\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9D9\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9D9\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9DA\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9DA\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9DA\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9DA\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9DA\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9DA\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9DA\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9DA\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9DA\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9DA\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9DB\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9DB\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9DB\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9DB\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9DB\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9DB\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9DB\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9DB\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9DB\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9DB\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9DC\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9DC\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9DC\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9DC\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9DC\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9DC\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9DC\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9DC\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9DC\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9DC\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9DD\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9DD\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9DD\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9DD\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9DD\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9DD\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9DD\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9DD\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9DD\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9DD\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F486\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F486\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F486\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F486\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F486\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F486\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F486\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F486\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F486\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F486\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F487\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F487\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F487\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F487\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F487\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F487\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F487\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F487\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F487\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F487\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F6B6\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F6B6\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F6B6\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F6B6\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F6B6\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F6B6\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F6B6\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F6B6\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F6B6\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F6B6\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9CD\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9CD\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9CD\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9CD\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9CD\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9CD\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9CD\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9CD\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9CD\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9CD\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9CE\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9CE\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9CE\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9CE\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9CE\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9CE\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9CE\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9CE\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9CE\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9CE\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F3C3\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F3C3\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F3C3\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F3C3\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F3C3\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F3C3\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F3C3\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F3C3\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F3C3\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F3C3\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9D6\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9D6\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9D6\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9D6\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9D6\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9D6\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9D6\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9D6\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9D6\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9D6\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9D7\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9D7\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9D7\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9D7\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9D7\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9D7\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9D7\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9D7\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9D7\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9D7\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F3CC\U0000FE0F\U0000200D\U00002642\U0000FE0F|\U0001F3CC\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F3CC\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F3CC\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F3CC\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F3CC\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F3CC\U0000FE0F\U0000200D\U00002640\U0000FE0F|\U0001F3CC\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F3CC\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F3CC\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F3CC\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F3CC\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F3C4\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F3C4\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F3C4\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F3C4\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F3C4\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F3C4\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F3C4\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F3C4\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F3C4\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F3C4\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F6A3\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F6A3\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F6A3\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F6A3\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F6A3\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F6A3\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F6A3\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F6A3\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F6A3\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F6A3\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F3CA\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F3CA\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F3CA\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F3CA\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F3CA\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F3CA\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F3CA\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F3CA\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F3CA\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F3CA\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U000026F9\U0000FE0F\U0000200D\U00002642\U0000FE0F|\U000026F9\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U000026F9\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U000026F9\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U000026F9\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U000026F9\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U000026F9\U0000FE0F\U0000200D\U00002640\U0000FE0F|\U000026F9\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U000026F9\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U000026F9\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U000026F9\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U000026F9\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F3CB\U0000FE0F\U0000200D\U00002642\U0000FE0F|\U0001F3CB\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F3CB\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F3CB\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F3CB\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F3CB\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F3CB\U0000FE0F\U0000200D\U00002640\U0000FE0F|\U0001F3CB\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F3CB\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F3CB\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F3CB\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F3CB\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F6B4\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F6B4\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F6B4\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F6B4\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F6B4\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F6B4\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F6B4\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F6B4\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F6B4\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F6B4\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F6B5\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F6B5\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F6B5\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F6B5\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F6B5\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F6B5\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F6B5\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F6B5\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F6B5\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F6B5\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F938\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F938\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F938\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F938\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F938\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F938\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F938\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F938\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F938\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F938\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F93D\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F93D\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F93D\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F93D\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F93D\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F93D\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F93D\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F93D\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F93D\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F93D\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F93E\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F93E\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F93E\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F93E\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F93E\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F93E\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F93E\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F93E\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F93E\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F93E\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F939\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F939\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F939\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F939\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F939\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F939\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F939\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F939\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F939\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F939\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9D8\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9D8\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9D8\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9D8\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9D8\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9D8\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9D8\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9D8\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9D8\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9D8\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9D1\U0000200D\U0001F91D\U0000200D\U0001F9D1|\U0001F469\U0000200D\U00002764\U0000200D\U0001F468|\U0001F468\U0000200D\U00002764\U0000200D\U0001F468|\U0001F469\U0000200D\U00002764\U0000200D\U0001F469|\U0001F468\U0000200D\U0001F469\U0000200D\U0001F466|\U0001F468\U0000200D\U0001F469\U0000200D\U0001F467|\U0001F468\U0000200D\U0001F468\U0000200D\U0001F466|\U0001F468\U0000200D\U0001F468\U0000200D\U0001F467|\U0001F469\U0000200D\U0001F469\U0000200D\U0001F466|\U0001F469\U0000200D\U0001F469\U0000200D\U0001F467|\U0001F468\U0000200D\U0001F466\U0000200D\U0001F466|\U0001F468\U0000200D\U0001F467\U0000200D\U0001F466|\U0001F468\U0000200D\U0001F467\U0000200D\U0001F467|\U0001F469\U0000200D\U0001F466\U0000200D\U0001F466|\U0001F469\U0000200D\U0001F467\U0000200D\U0001F466|\U0001F469\U0000200D\U0001F467\U0000200D\U0001F467|\U0001F3F3\U0000FE0F\U0000200D\U000026A7\U0000FE0F|\U0001F441\U0000200D\U0001F5E8\U0000FE0F|\U0001F441\U0000FE0F\U0000200D\U0001F5E8|\U0001F468\U0001F3FB\U0000200D\U0001F9B0|\U0001F468\U0001F3FC\U0000200D\U0001F9B0|\U0001F468\U0001F3FD\U0000200D\U0001F9B0|\U0001F468\U0001F3FE\U0000200D\U0001F9B0|\U0001F468\U0001F3FF\U0000200D\U0001F9B0|\U0001F468\U0001F3FB\U0000200D\U0001F9B1|\U0001F468\U0001F3FC\U0000200D\U0001F9B1|\U0001F468\U0001F3FD\U0000200D\U0001F9B1|\U0001F468\U0001F3FE\U0000200D\U0001F9B1|\U0001F468\U0001F3FF\U0000200D\U0001F9B1|\U0001F468\U0001F3FB\U0000200D\U0001F9B3|\U0001F468\U0001F3FC\U0000200D\U0001F9B3|\U0001F468\U0001F3FD\U0000200D\U0001F9B3|\U0001F468\U0001F3FE\U0000200D\U0001F9B3|\U0001F468\U0001F3FF\U0000200D\U0001F9B3|\U0001F468\U0001F3FB\U0000200D\U0001F9B2|\U0001F468\U0001F3FC\U0000200D\U0001F9B2|\U0001F468\U0001F3FD\U0000200D\U0001F9B2|\U0001F468\U0001F3FE\U0000200D\U0001F9B2|\U0001F468\U0001F3FF\U0000200D\U0001F9B2|\U0001F469\U0001F3FB\U0000200D\U0001F9B0|\U0001F469\U0001F3FC\U0000200D\U0001F9B0|\U0001F469\U0001F3FD\U0000200D\U0001F9B0|\U0001F469\U0001F3FE\U0000200D\U0001F9B0|\U0001F469\U0001F3FF\U0000200D\U0001F9B0|\U0001F9D1\U0001F3FB\U0000200D\U0001F9B0|\U0001F9D1\U0001F3FC\U0000200D\U0001F9B0|\U0001F9D1\U0001F3FD\U0000200D\U0001F9B0|\U0001F9D1\U0001F3FE\U0000200D\U0001F9B0|\U0001F9D1\U0001F3FF\U0000200D\U0001F9B0|\U0001F469\U0001F3FB\U0000200D\U0001F9B1|\U0001F469\U0001F3FC\U0000200D\U0001F9B1|\U0001F469\U0001F3FD\U0000200D\U0001F9B1|\U0001F469\U0001F3FE\U0000200D\U0001F9B1|\U0001F469\U0001F3FF\U0000200D\U0001F9B1|\U0001F9D1\U0001F3FB\U0000200D\U0001F9B1|\U0001F9D1\U0001F3FC\U0000200D\U0001F9B1|\U0001F9D1\U0001F3FD\U0000200D\U0001F9B1|\U0001F9D1\U0001F3FE\U0000200D\U0001F9B1|\U0001F9D1\U0001F3FF\U0000200D\U0001F9B1|\U0001F469\U0001F3FB\U0000200D\U0001F9B3|\U0001F469\U0001F3FC\U0000200D\U0001F9B3|\U0001F469\U0001F3FD\U0000200D\U0001F9B3|\U0001F469\U0001F3FE\U0000200D\U0001F9B3|\U0001F469\U0001F3FF\U0000200D\U0001F9B3|\U0001F9D1\U0001F3FB\U0000200D\U0001F9B3|\U0001F9D1\U0001F3FC\U0000200D\U0001F9B3|\U0001F9D1\U0001F3FD\U0000200D\U0001F9B3|\U0001F9D1\U0001F3FE\U0000200D\U0001F9B3|\U0001F9D1\U0001F3FF\U0000200D\U0001F9B3|\U0001F469\U0001F3FB\U0000200D\U0001F9B2|\U0001F469\U0001F3FC\U0000200D\U0001F9B2|\U0001F469\U0001F3FD\U0000200D\U0001F9B2|\U0001F469\U0001F3FE\U0000200D\U0001F9B2|\U0001F469\U0001F3FF\U0000200D\U0001F9B2|\U0001F9D1\U0001F3FB\U0000200D\U0001F9B2|\U0001F9D1\U0001F3FC\U0000200D\U0001F9B2|\U0001F9D1\U0001F3FD\U0000200D\U0001F9B2|\U0001F9D1\U0001F3FE\U0000200D\U0001F9B2|\U0001F9D1\U0001F3FF\U0000200D\U0001F9B2|\U0001F471\U0000200D\U00002640\U0000FE0F|\U0001F471\U0001F3FB\U0000200D\U00002640|\U0001F471\U0001F3FC\U0000200D\U00002640|\U0001F471\U0001F3FD\U0000200D\U00002640|\U0001F471\U0001F3FE\U0000200D\U00002640|\U0001F471\U0001F3FF\U0000200D\U00002640|\U0001F471\U0000200D\U00002642\U0000FE0F|\U0001F471\U0001F3FB\U0000200D\U00002642|\U0001F471\U0001F3FC\U0000200D\U00002642|\U0001F471\U0001F3FD\U0000200D\U00002642|\U0001F471\U0001F3FE\U0000200D\U00002642|\U0001F471\U0001F3FF\U0000200D\U00002642|\U0001F64D\U0000200D\U00002642\U0000FE0F|\U0001F64D\U0001F3FB\U0000200D\U00002642|\U0001F64D\U0001F3FC\U0000200D\U00002642|\U0001F64D\U0001F3FD\U0000200D\U00002642|\U0001F64D\U0001F3FE\U0000200D\U00002642|\U0001F64D\U0001F3FF\U0000200D\U00002642|\U0001F64D\U0000200D\U00002640\U0000FE0F|\U0001F64D\U0001F3FB\U0000200D\U00002640|\U0001F64D\U0001F3FC\U0000200D\U00002640|\U0001F64D\U0001F3FD\U0000200D\U00002640|\U0001F64D\U0001F3FE\U0000200D\U00002640|\U0001F64D\U0001F3FF\U0000200D\U00002640|\U0001F64E\U0000200D\U00002642\U0000FE0F|\U0001F64E\U0001F3FB\U0000200D\U00002642|\U0001F64E\U0001F3FC\U0000200D\U00002642|\U0001F64E\U0001F3FD\U0000200D\U00002642|\U0001F64E\U0001F3FE\U0000200D\U00002642|\U0001F64E\U0001F3FF\U0000200D\U00002642|\U0001F64E\U0000200D\U00002640\U0000FE0F|\U0001F64E\U0001F3FB\U0000200D\U00002640|\U0001F64E\U0001F3FC\U0000200D\U00002640|\U0001F64E\U0001F3FD\U0000200D\U00002640|\U0001F64E\U0001F3FE\U0000200D\U00002640|\U0001F64E\U0001F3FF\U0000200D\U00002640|\U0001F645\U0000200D\U00002642\U0000FE0F|\U0001F645\U0001F3FB\U0000200D\U00002642|\U0001F645\U0001F3FC\U0000200D\U00002642|\U0001F645\U0001F3FD\U0000200D\U00002642|\U0001F645\U0001F3FE\U0000200D\U00002642|\U0001F645\U0001F3FF\U0000200D\U00002642|\U0001F645\U0000200D\U00002640\U0000FE0F|\U0001F645\U0001F3FB\U0000200D\U00002640|\U0001F645\U0001F3FC\U0000200D\U00002640|\U0001F645\U0001F3FD\U0000200D\U00002640|\U0001F645\U0001F3FE\U0000200D\U00002640|\U0001F645\U0001F3FF\U0000200D\U00002640|\U0001F646\U0000200D\U00002642\U0000FE0F|\U0001F646\U0001F3FB\U0000200D\U00002642|\U0001F646\U0001F3FC\U0000200D\U00002642|\U0001F646\U0001F3FD\U0000200D\U00002642|\U0001F646\U0001F3FE\U0000200D\U00002642|\U0001F646\U0001F3FF\U0000200D\U00002642|\U0001F646\U0000200D\U00002640\U0000FE0F|\U0001F646\U0001F3FB\U0000200D\U00002640|\U0001F646\U0001F3FC\U0000200D\U00002640|\U0001F646\U0001F3FD\U0000200D\U00002640|\U0001F646\U0001F3FE\U0000200D\U00002640|\U0001F646\U0001F3FF\U0000200D\U00002640|\U0001F481\U0000200D\U00002642\U0000FE0F|\U0001F481\U0001F3FB\U0000200D\U00002642|\U0001F481\U0001F3FC\U0000200D\U00002642|\U0001F481\U0001F3FD\U0000200D\U00002642|\U0001F481\U0001F3FE\U0000200D\U00002642|\U0001F481\U0001F3FF\U0000200D\U00002642|\U0001F481\U0000200D\U00002640\U0000FE0F|\U0001F481\U0001F3FB\U0000200D\U00002640|\U0001F481\U0001F3FC\U0000200D\U00002640|\U0001F481\U0001F3FD\U0000200D\U00002640|\U0001F481\U0001F3FE\U0000200D\U00002640|\U0001F481\U0001F3FF\U0000200D\U00002640|\U0001F64B\U0000200D\U00002642\U0000FE0F|\U0001F64B\U0001F3FB\U0000200D\U00002642|\U0001F64B\U0001F3FC\U0000200D\U00002642|\U0001F64B\U0001F3FD\U0000200D\U00002642|\U0001F64B\U0001F3FE\U0000200D\U00002642|\U0001F64B\U0001F3FF\U0000200D\U00002642|\U0001F64B\U0000200D\U00002640\U0000FE0F|\U0001F64B\U0001F3FB\U0000200D\U00002640|\U0001F64B\U0001F3FC\U0000200D\U00002640|\U0001F64B\U0001F3FD\U0000200D\U00002640|\U0001F64B\U0001F3FE\U0000200D\U00002640|\U0001F64B\U0001F3FF\U0000200D\U00002640|\U0001F9CF\U0000200D\U00002642\U0000FE0F|\U0001F9CF\U0001F3FB\U0000200D\U00002642|\U0001F9CF\U0001F3FC\U0000200D\U00002642|\U0001F9CF\U0001F3FD\U0000200D\U00002642|\U0001F9CF\U0001F3FE\U0000200D\U00002642|\U0001F9CF\U0001F3FF\U0000200D\U00002642|\U0001F9CF\U0000200D\U00002640\U0000FE0F|\U0001F9CF\U0001F3FB\U0000200D\U00002640|\U0001F9CF\U0001F3FC\U0000200D\U00002640|\U0001F9CF\U0001F3FD\U0000200D\U00002640|\U0001F9CF\U0001F3FE\U0000200D\U00002640|\U0001F9CF\U0001F3FF\U0000200D\U00002640|\U0001F647\U0000200D\U00002642\U0000FE0F|\U0001F647\U0001F3FB\U0000200D\U00002642|\U0001F647\U0001F3FC\U0000200D\U00002642|\U0001F647\U0001F3FD\U0000200D\U00002642|\U0001F647\U0001F3FE\U0000200D\U00002642|\U0001F647\U0001F3FF\U0000200D\U00002642|\U0001F647\U0000200D\U00002640\U0000FE0F|\U0001F647\U0001F3FB\U0000200D\U00002640|\U0001F647\U0001F3FC\U0000200D\U00002640|\U0001F647\U0001F3FD\U0000200D\U00002640|\U0001F647\U0001F3FE\U0000200D\U00002640|\U0001F647\U0001F3FF\U0000200D\U00002640|\U0001F926\U0000200D\U00002642\U0000FE0F|\U0001F926\U0001F3FB\U0000200D\U00002642|\U0001F926\U0001F3FC\U0000200D\U00002642|\U0001F926\U0001F3FD\U0000200D\U00002642|\U0001F926\U0001F3FE\U0000200D\U00002642|\U0001F926\U0001F3FF\U0000200D\U00002642|\U0001F926\U0000200D\U00002640\U0000FE0F|\U0001F926\U0001F3FB\U0000200D\U00002640|\U0001F926\U0001F3FC\U0000200D\U00002640|\U0001F926\U0001F3FD\U0000200D\U00002640|\U0001F926\U0001F3FE\U0000200D\U00002640|\U0001F926\U0001F3FF\U0000200D\U00002640|\U0001F937\U0000200D\U00002642\U0000FE0F|\U0001F937\U0001F3FB\U0000200D\U00002642|\U0001F937\U0001F3FC\U0000200D\U00002642|\U0001F937\U0001F3FD\U0000200D\U00002642|\U0001F937\U0001F3FE\U0000200D\U00002642|\U0001F937\U0001F3FF\U0000200D\U00002642|\U0001F937\U0000200D\U00002640\U0000FE0F|\U0001F937\U0001F3FB\U0000200D\U00002640|\U0001F937\U0001F3FC\U0000200D\U00002640|\U0001F937\U0001F3FD\U0000200D\U00002640|\U0001F937\U0001F3FE\U0000200D\U00002640|\U0001F937\U0001F3FF\U0000200D\U00002640|\U0001F9D1\U0000200D\U00002695\U0000FE0F|\U0001F9D1\U0001F3FB\U0000200D\U00002695|\U0001F9D1\U0001F3FC\U0000200D\U00002695|\U0001F9D1\U0001F3FD\U0000200D\U00002695|\U0001F9D1\U0001F3FE\U0000200D\U00002695|\U0001F9D1\U0001F3FF\U0000200D\U00002695|\U0001F468\U0000200D\U00002695\U0000FE0F|\U0001F468\U0001F3FB\U0000200D\U00002695|\U0001F468\U0001F3FC\U0000200D\U00002695|\U0001F468\U0001F3FD\U0000200D\U00002695|\U0001F468\U0001F3FE\U0000200D\U00002695|\U0001F468\U0001F3FF\U0000200D\U00002695|\U0001F469\U0000200D\U00002695\U0000FE0F|\U0001F469\U0001F3FB\U0000200D\U00002695|\U0001F469\U0001F3FC\U0000200D\U00002695|\U0001F469\U0001F3FD\U0000200D\U00002695|\U0001F469\U0001F3FE\U0000200D\U00002695|\U0001F469\U0001F3FF\U0000200D\U00002695|\U0001F9D1\U0001F3FB\U0000200D\U0001F393|\U0001F9D1\U0001F3FC\U0000200D\U0001F393|\U0001F9D1\U0001F3FD\U0000200D\U0001F393|\U0001F9D1\U0001F3FE\U0000200D\U0001F393|\U0001F9D1\U0001F3FF\U0000200D\U0001F393|\U0001F468\U0001F3FB\U0000200D\U0001F393|\U0001F468\U0001F3FC\U0000200D\U0001F393|\U0001F468\U0001F3FD\U0000200D\U0001F393|\U0001F468\U0001F3FE\U0000200D\U0001F393|\U0001F468\U0001F3FF\U0000200D\U0001F393|\U0001F469\U0001F3FB\U0000200D\U0001F393|\U0001F469\U0001F3FC\U0000200D\U0001F393|\U0001F469\U0001F3FD\U0000200D\U0001F393|\U0001F469\U0001F3FE\U0000200D\U0001F393|\U0001F469\U0001F3FF\U0000200D\U0001F393|\U0001F9D1\U0001F3FB\U0000200D\U0001F3EB|\U0001F9D1\U0001F3FC\U0000200D\U0001F3EB|\U0001F9D1\U0001F3FD\U0000200D\U0001F3EB|\U0001F9D1\U0001F3FE\U0000200D\U0001F3EB|\U0001F9D1\U0001F3FF\U0000200D\U0001F3EB|\U0001F468\U0001F3FB\U0000200D\U0001F3EB|\U0001F468\U0001F3FC\U0000200D\U0001F3EB|\U0001F468\U0001F3FD\U0000200D\U0001F3EB|\U0001F468\U0001F3FE\U0000200D\U0001F3EB|\U0001F468\U0001F3FF\U0000200D\U0001F3EB|\U0001F469\U0001F3FB\U0000200D\U0001F3EB|\U0001F469\U0001F3FC\U0000200D\U0001F3EB|\U0001F469\U0001F3FD\U0000200D\U0001F3EB|\U0001F469\U0001F3FE\U0000200D\U0001F3EB|\U0001F469\U0001F3FF\U0000200D\U0001F3EB|\U0001F9D1\U0000200D\U00002696\U0000FE0F|\U0001F9D1\U0001F3FB\U0000200D\U00002696|\U0001F9D1\U0001F3FC\U0000200D\U00002696|\U0001F9D1\U0001F3FD\U0000200D\U00002696|\U0001F9D1\U0001F3FE\U0000200D\U00002696|\U0001F9D1\U0001F3FF\U0000200D\U00002696|\U0001F468\U0000200D\U00002696\U0000FE0F|\U0001F468\U0001F3FB\U0000200D\U00002696|\U0001F468\U0001F3FC\U0000200D\U00002696|\U0001F468\U0001F3FD\U0000200D\U00002696|\U0001F468\U0001F3FE\U0000200D\U00002696|\U0001F468\U0001F3FF\U0000200D\U00002696|\U0001F469\U0000200D\U00002696\U0000FE0F|\U0001F469\U0001F3FB\U0000200D\U00002696|\U0001F469\U0001F3FC\U0000200D\U00002696|\U0001F469\U0001F3FD\U0000200D\U00002696|\U0001F469\U0001F3FE\U0000200D\U00002696|\U0001F469\U0001F3FF\U0000200D\U00002696|\U0001F9D1\U0001F3FB\U0000200D\U0001F33E|\U0001F9D1\U0001F3FC\U0000200D\U0001F33E|\U0001F9D1\U0001F3FD\U0000200D\U0001F33E|\U0001F9D1\U0001F3FE\U0000200D\U0001F33E|\U0001F9D1\U0001F3FF\U0000200D\U0001F33E|\U0001F468\U0001F3FB\U0000200D\U0001F33E|\U0001F468\U0001F3FC\U0000200D\U0001F33E|\U0001F468\U0001F3FD\U0000200D\U0001F33E|\U0001F468\U0001F3FE\U0000200D\U0001F33E|\U0001F468\U0001F3FF\U0000200D\U0001F33E|\U0001F469\U0001F3FB\U0000200D\U0001F33E|\U0001F469\U0001F3FC\U0000200D\U0001F33E|\U0001F469\U0001F3FD\U0000200D\U0001F33E|\U0001F469\U0001F3FE\U0000200D\U0001F33E|\U0001F469\U0001F3FF\U0000200D\U0001F33E|\U0001F9D1\U0001F3FB\U0000200D\U0001F373|\U0001F9D1\U0001F3FC\U0000200D\U0001F373|\U0001F9D1\U0001F3FD\U0000200D\U0001F373|\U0001F9D1\U0001F3FE\U0000200D\U0001F373|\U0001F9D1\U0001F3FF\U0000200D\U0001F373|\U0001F468\U0001F3FB\U0000200D\U0001F373|\U0001F468\U0001F3FC\U0000200D\U0001F373|\U0001F468\U0001F3FD\U0000200D\U0001F373|\U0001F468\U0001F3FE\U0000200D\U0001F373|\U0001F468\U0001F3FF\U0000200D\U0001F373|\U0001F469\U0001F3FB\U0000200D\U0001F373|\U0001F469\U0001F3FC\U0000200D\U0001F373|\U0001F469\U0001F3FD\U0000200D\U0001F373|\U0001F469\U0001F3FE\U0000200D\U0001F373|\U0001F469\U0001F3FF\U0000200D\U0001F373|\U0001F9D1\U0001F3FB\U0000200D\U0001F527|\U0001F9D1\U0001F3FC\U0000200D\U0001F527|\U0001F9D1\U0001F3FD\U0000200D\U0001F527|\U0001F9D1\U0001F3FE\U0000200D\U0001F527|\U0001F9D1\U0001F3FF\U0000200D\U0001F527|\U0001F468\U0001F3FB\U0000200D\U0001F527|\U0001F468\U0001F3FC\U0000200D\U0001F527|\U0001F468\U0001F3FD\U0000200D\U0001F527|\U0001F468\U0001F3FE\U0000200D\U0001F527|\U0001F468\U0001F3FF\U0000200D\U0001F527|\U0001F469\U0001F3FB\U0000200D\U0001F527|\U0001F469\U0001F3FC\U0000200D\U0001F527|\U0001F469\U0001F3FD\U0000200D\U0001F527|\U0001F469\U0001F3FE\U0000200D\U0001F527|\U0001F469\U0001F3FF\U0000200D\U0001F527|\U0001F9D1\U0001F3FB\U0000200D\U0001F3ED|\U0001F9D1\U0001F3FC\U0000200D\U0001F3ED|\U0001F9D1\U0001F3FD\U0000200D\U0001F3ED|\U0001F9D1\U0001F3FE\U0000200D\U0001F3ED|\U0001F9D1\U0001F3FF\U0000200D\U0001F3ED|\U0001F468\U0001F3FB\U0000200D\U0001F3ED|\U0001F468\U0001F3FC\U0000200D\U0001F3ED|\U0001F468\U0001F3FD\U0000200D\U0001F3ED|\U0001F468\U0001F3FE\U0000200D\U0001F3ED|\U0001F468\U0001F3FF\U0000200D\U0001F3ED|\U0001F469\U0001F3FB\U0000200D\U0001F3ED|\U0001F469\U0001F3FC\U0000200D\U0001F3ED|\U0001F469\U0001F3FD\U0000200D\U0001F3ED|\U0001F469\U0001F3FE\U0000200D\U0001F3ED|\U0001F469\U0001F3FF\U0000200D\U0001F3ED|\U0001F9D1\U0001F3FB\U0000200D\U0001F4BC|\U0001F9D1\U0001F3FC\U0000200D\U0001F4BC|\U0001F9D1\U0001F3FD\U0000200D\U0001F4BC|\U0001F9D1\U0001F3FE\U0000200D\U0001F4BC|\U0001F9D1\U0001F3FF\U0000200D\U0001F4BC|\U0001F468\U0001F3FB\U0000200D\U0001F4BC|\U0001F468\U0001F3FC\U0000200D\U0001F4BC|\U0001F468\U0001F3FD\U0000200D\U0001F4BC|\U0001F468\U0001F3FE\U0000200D\U0001F4BC|\U0001F468\U0001F3FF\U0000200D\U0001F4BC|\U0001F469\U0001F3FB\U0000200D\U0001F4BC|\U0001F469\U0001F3FC\U0000200D\U0001F4BC|\U0001F469\U0001F3FD\U0000200D\U0001F4BC|\U0001F469\U0001F3FE\U0000200D\U0001F4BC|\U0001F469\U0001F3FF\U0000200D\U0001F4BC|\U0001F9D1\U0001F3FB\U0000200D\U0001F52C|\U0001F9D1\U0001F3FC\U0000200D\U0001F52C|\U0001F9D1\U0001F3FD\U0000200D\U0001F52C|\U0001F9D1\U0001F3FE\U0000200D\U0001F52C|\U0001F9D1\U0001F3FF\U0000200D\U0001F52C|\U0001F468\U0001F3FB\U0000200D\U0001F52C|\U0001F468\U0001F3FC\U0000200D\U0001F52C|\U0001F468\U0001F3FD\U0000200D\U0001F52C|\U0001F468\U0001F3FE\U0000200D\U0001F52C|\U0001F468\U0001F3FF\U0000200D\U0001F52C|\U0001F469\U0001F3FB\U0000200D\U0001F52C|\U0001F469\U0001F3FC\U0000200D\U0001F52C|\U0001F469\U0001F3FD\U0000200D\U0001F52C|\U0001F469\U0001F3FE\U0000200D\U0001F52C|\U0001F469\U0001F3FF\U0000200D\U0001F52C|\U0001F9D1\U0001F3FB\U0000200D\U0001F4BB|\U0001F9D1\U0001F3FC\U0000200D\U0001F4BB|\U0001F9D1\U0001F3FD\U0000200D\U0001F4BB|\U0001F9D1\U0001F3FE\U0000200D\U0001F4BB|\U0001F9D1\U0001F3FF\U0000200D\U0001F4BB|\U0001F468\U0001F3FB\U0000200D\U0001F4BB|\U0001F468\U0001F3FC\U0000200D\U0001F4BB|\U0001F468\U0001F3FD\U0000200D\U0001F4BB|\U0001F468\U0001F3FE\U0000200D\U0001F4BB|\U0001F468\U0001F3FF\U0000200D\U0001F4BB|\U0001F469\U0001F3FB\U0000200D\U0001F4BB|\U0001F469\U0001F3FC\U0000200D\U0001F4BB|\U0001F469\U0001F3FD\U0000200D\U0001F4BB|\U0001F469\U0001F3FE\U0000200D\U0001F4BB|\U0001F469\U0001F3FF\U0000200D\U0001F4BB|\U0001F9D1\U0001F3FB\U0000200D\U0001F3A4|\U0001F9D1\U0001F3FC\U0000200D\U0001F3A4|\U0001F9D1\U0001F3FD\U0000200D\U0001F3A4|\U0001F9D1\U0001F3FE\U0000200D\U0001F3A4|\U0001F9D1\U0001F3FF\U0000200D\U0001F3A4|\U0001F468\U0001F3FB\U0000200D\U0001F3A4|\U0001F468\U0001F3FC\U0000200D\U0001F3A4|\U0001F468\U0001F3FD\U0000200D\U0001F3A4|\U0001F468\U0001F3FE\U0000200D\U0001F3A4|\U0001F468\U0001F3FF\U0000200D\U0001F3A4|\U0001F469\U0001F3FB\U0000200D\U0001F3A4|\U0001F469\U0001F3FC\U0000200D\U0001F3A4|\U0001F469\U0001F3FD\U0000200D\U0001F3A4|\U0001F469\U0001F3FE\U0000200D\U0001F3A4|\U0001F469\U0001F3FF\U0000200D\U0001F3A4|\U0001F9D1\U0001F3FB\U0000200D\U0001F3A8|\U0001F9D1\U0001F3FC\U0000200D\U0001F3A8|\U0001F9D1\U0001F3FD\U0000200D\U0001F3A8|\U0001F9D1\U0001F3FE\U0000200D\U0001F3A8|\U0001F9D1\U0001F3FF\U0000200D\U0001F3A8|\U0001F468\U0001F3FB\U0000200D\U0001F3A8|\U0001F468\U0001F3FC\U0000200D\U0001F3A8|\U0001F468\U0001F3FD\U0000200D\U0001F3A8|\U0001F468\U0001F3FE\U0000200D\U0001F3A8|\U0001F468\U0001F3FF\U0000200D\U0001F3A8|\U0001F469\U0001F3FB\U0000200D\U0001F3A8|\U0001F469\U0001F3FC\U0000200D\U0001F3A8|\U0001F469\U0001F3FD\U0000200D\U0001F3A8|\U0001F469\U0001F3FE\U0000200D\U0001F3A8|\U0001F469\U0001F3FF\U0000200D\U0001F3A8|\U0001F9D1\U0000200D\U00002708\U0000FE0F|\U0001F9D1\U0001F3FB\U0000200D\U00002708|\U0001F9D1\U0001F3FC\U0000200D\U00002708|\U0001F9D1\U0001F3FD\U0000200D\U00002708|\U0001F9D1\U0001F3FE\U0000200D\U00002708|\U0001F9D1\U0001F3FF\U0000200D\U00002708|\U0001F468\U0000200D\U00002708\U0000FE0F|\U0001F468\U0001F3FB\U0000200D\U00002708|\U0001F468\U0001F3FC\U0000200D\U00002708|\U0001F468\U0001F3FD\U0000200D\U00002708|\U0001F468\U0001F3FE\U0000200D\U00002708|\U0001F468\U0001F3FF\U0000200D\U00002708|\U0001F469\U0000200D\U00002708\U0000FE0F|\U0001F469\U0001F3FB\U0000200D\U00002708|\U0001F469\U0001F3FC\U0000200D\U00002708|\U0001F469\U0001F3FD\U0000200D\U00002708|\U0001F469\U0001F3FE\U0000200D\U00002708|\U0001F469\U0001F3FF\U0000200D\U00002708|\U0001F9D1\U0001F3FB\U0000200D\U0001F680|\U0001F9D1\U0001F3FC\U0000200D\U0001F680|\U0001F9D1\U0001F3FD\U0000200D\U0001F680|\U0001F9D1\U0001F3FE\U0000200D\U0001F680|\U0001F9D1\U0001F3FF\U0000200D\U0001F680|\U0001F468\U0001F3FB\U0000200D\U0001F680|\U0001F468\U0001F3FC\U0000200D\U0001F680|\U0001F468\U0001F3FD\U0000200D\U0001F680|\U0001F468\U0001F3FE\U0000200D\U0001F680|\U0001F468\U0001F3FF\U0000200D\U0001F680|\U0001F469\U0001F3FB\U0000200D\U0001F680|\U0001F469\U0001F3FC\U0000200D\U0001F680|\U0001F469\U0001F3FD\U0000200D\U0001F680|\U0001F469\U0001F3FE\U0000200D\U0001F680|\U0001F469\U0001F3FF\U0000200D\U0001F680|\U0001F9D1\U0001F3FB\U0000200D\U0001F692|\U0001F9D1\U0001F3FC\U0000200D\U0001F692|\U0001F9D1\U0001F3FD\U0000200D\U0001F692|\U0001F9D1\U0001F3FE\U0000200D\U0001F692|\U0001F9D1\U0001F3FF\U0000200D\U0001F692|\U0001F468\U0001F3FB\U0000200D\U0001F692|\U0001F468\U0001F3FC\U0000200D\U0001F692|\U0001F468\U0001F3FD\U0000200D\U0001F692|\U0001F468\U0001F3FE\U0000200D\U0001F692|\U0001F468\U0001F3FF\U0000200D\U0001F692|\U0001F469\U0001F3FB\U0000200D\U0001F692|\U0001F469\U0001F3FC\U0000200D\U0001F692|\U0001F469\U0001F3FD\U0000200D\U0001F692|\U0001F469\U0001F3FE\U0000200D\U0001F692|\U0001F469\U0001F3FF\U0000200D\U0001F692|\U0001F46E\U0000200D\U00002642\U0000FE0F|\U0001F46E\U0001F3FB\U0000200D\U00002642|\U0001F46E\U0001F3FC\U0000200D\U00002642|\U0001F46E\U0001F3FD\U0000200D\U00002642|\U0001F46E\U0001F3FE\U0000200D\U00002642|\U0001F46E\U0001F3FF\U0000200D\U00002642|\U0001F46E\U0000200D\U00002640\U0000FE0F|\U0001F46E\U0001F3FB\U0000200D\U00002640|\U0001F46E\U0001F3FC\U0000200D\U00002640|\U0001F46E\U0001F3FD\U0000200D\U00002640|\U0001F46E\U0001F3FE\U0000200D\U00002640|\U0001F46E\U0001F3FF\U0000200D\U00002640|\U0001F575\U0000200D\U00002642\U0000FE0F|\U0001F575\U0000FE0F\U0000200D\U00002642|\U0001F575\U0001F3FB\U0000200D\U00002642|\U0001F575\U0001F3FC\U0000200D\U00002642|\U0001F575\U0001F3FD\U0000200D\U00002642|\U0001F575\U0001F3FE\U0000200D\U00002642|\U0001F575\U0001F3FF\U0000200D\U00002642|\U0001F575\U0000200D\U00002640\U0000FE0F|\U0001F575\U0000FE0F\U0000200D\U00002640|\U0001F575\U0001F3FB\U0000200D\U00002640|\U0001F575\U0001F3FC\U0000200D\U00002640|\U0001F575\U0001F3FD\U0000200D\U00002640|\U0001F575\U0001F3FE\U0000200D\U00002640|\U0001F575\U0001F3FF\U0000200D\U00002640|\U0001F482\U0000200D\U00002642\U0000FE0F|\U0001F482\U0001F3FB\U0000200D\U00002642|\U0001F482\U0001F3FC\U0000200D\U00002642|\U0001F482\U0001F3FD\U0000200D\U00002642|\U0001F482\U0001F3FE\U0000200D\U00002642|\U0001F482\U0001F3FF\U0000200D\U00002642|\U0001F482\U0000200D\U00002640\U0000FE0F|\U0001F482\U0001F3FB\U0000200D\U00002640|\U0001F482\U0001F3FC\U0000200D\U00002640|\U0001F482\U0001F3FD\U0000200D\U00002640|\U0001F482\U0001F3FE\U0000200D\U00002640|\U0001F482\U0001F3FF\U0000200D\U00002640|\U0001F477\U0000200D\U00002642\U0000FE0F|\U0001F477\U0001F3FB\U0000200D\U00002642|\U0001F477\U0001F3FC\U0000200D\U00002642|\U0001F477\U0001F3FD\U0000200D\U00002642|\U0001F477\U0001F3FE\U0000200D\U00002642|\U0001F477\U0001F3FF\U0000200D\U00002642|\U0001F477\U0000200D\U00002640\U0000FE0F|\U0001F477\U0001F3FB\U0000200D\U00002640|\U0001F477\U0001F3FC\U0000200D\U00002640|\U0001F477\U0001F3FD\U0000200D\U00002640|\U0001F477\U0001F3FE\U0000200D\U00002640|\U0001F477\U0001F3FF\U0000200D\U00002640|\U0001F473\U0000200D\U00002642\U0000FE0F|\U0001F473\U0001F3FB\U0000200D\U00002642|\U0001F473\U0001F3FC\U0000200D\U00002642|\U0001F473\U0001F3FD\U0000200D\U00002642|\U0001F473\U0001F3FE\U0000200D\U00002642|\U0001F473\U0001F3FF\U0000200D\U00002642|\U0001F473\U0000200D\U00002640\U0000FE0F|\U0001F473\U0001F3FB\U0000200D\U00002640|\U0001F473\U0001F3FC\U0000200D\U00002640|\U0001F473\U0001F3FD\U0000200D\U00002640|\U0001F473\U0001F3FE\U0000200D\U00002640|\U0001F473\U0001F3FF\U0000200D\U00002640|\U0001F935\U0000200D\U00002642\U0000FE0F|\U0001F935\U0001F3FB\U0000200D\U00002642|\U0001F935\U0001F3FC\U0000200D\U00002642|\U0001F935\U0001F3FD\U0000200D\U00002642|\U0001F935\U0001F3FE\U0000200D\U00002642|\U0001F935\U0001F3FF\U0000200D\U00002642|\U0001F935\U0000200D\U00002640\U0000FE0F|\U0001F935\U0001F3FB\U0000200D\U00002640|\U0001F935\U0001F3FC\U0000200D\U00002640|\U0001F935\U0001F3FD\U0000200D\U00002640|\U0001F935\U0001F3FE\U0000200D\U00002640|\U0001F935\U0001F3FF\U0000200D\U00002640|\U0001F470\U0000200D\U00002642\U0000FE0F|\U0001F470\U0001F3FB\U0000200D\U00002642|\U0001F470\U0001F3FC\U0000200D\U00002642|\U0001F470\U0001F3FD\U0000200D\U00002642|\U0001F470\U0001F3FE\U0000200D\U00002642|\U0001F470\U0001F3FF\U0000200D\U00002642|\U0001F470\U0000200D\U00002640\U0000FE0F|\U0001F470\U0001F3FB\U0000200D\U00002640|\U0001F470\U0001F3FC\U0000200D\U00002640|\U0001F470\U0001F3FD\U0000200D\U00002640|\U0001F470\U0001F3FE\U0000200D\U00002640|\U0001F470\U0001F3FF\U0000200D\U00002640|\U0001F469\U0001F3FB\U0000200D\U0001F37C|\U0001F469\U0001F3FC\U0000200D\U0001F37C|\U0001F469\U0001F3FD\U0000200D\U0001F37C|\U0001F469\U0001F3FE\U0000200D\U0001F37C|\U0001F469\U0001F3FF\U0000200D\U0001F37C|\U0001F468\U0001F3FB\U0000200D\U0001F37C|\U0001F468\U0001F3FC\U0000200D\U0001F37C|\U0001F468\U0001F3FD\U0000200D\U0001F37C|\U0001F468\U0001F3FE\U0000200D\U0001F37C|\U0001F468\U0001F3FF\U0000200D\U0001F37C|\U0001F9D1\U0001F3FB\U0000200D\U0001F37C|\U0001F9D1\U0001F3FC\U0000200D\U0001F37C|\U0001F9D1\U0001F3FD\U0000200D\U0001F37C|\U0001F9D1\U0001F3FE\U0000200D\U0001F37C|\U0001F9D1\U0001F3FF\U0000200D\U0001F37C|\U0001F9D1\U0001F3FB\U0000200D\U0001F384|\U0001F9D1\U0001F3FC\U0000200D\U0001F384|\U0001F9D1\U0001F3FD\U0000200D\U0001F384|\U0001F9D1\U0001F3FE\U0000200D\U0001F384|\U0001F9D1\U0001F3FF\U0000200D\U0001F384|\U0001F9B8\U0000200D\U00002642\U0000FE0F|\U0001F9B8\U0001F3FB\U0000200D\U00002642|\U0001F9B8\U0001F3FC\U0000200D\U00002642|\U0001F9B8\U0001F3FD\U0000200D\U00002642|\U0001F9B8\U0001F3FE\U0000200D\U00002642|\U0001F9B8\U0001F3FF\U0000200D\U00002642|\U0001F9B8\U0000200D\U00002640\U0000FE0F|\U0001F9B8\U0001F3FB\U0000200D\U00002640|\U0001F9B8\U0001F3FC\U0000200D\U00002640|\U0001F9B8\U0001F3FD\U0000200D\U00002640|\U0001F9B8\U0001F3FE\U0000200D\U00002640|\U0001F9B8\U0001F3FF\U0000200D\U00002640|\U0001F9B9\U0000200D\U00002642\U0000FE0F|\U0001F9B9\U0001F3FB\U0000200D\U00002642|\U0001F9B9\U0001F3FC\U0000200D\U00002642|\U0001F9B9\U0001F3FD\U0000200D\U00002642|\U0001F9B9\U0001F3FE\U0000200D\U00002642|\U0001F9B9\U0001F3FF\U0000200D\U00002642|\U0001F9B9\U0000200D\U00002640\U0000FE0F|\U0001F9B9\U0001F3FB\U0000200D\U00002640|\U0001F9B9\U0001F3FC\U0000200D\U00002640|\U0001F9B9\U0001F3FD\U0000200D\U00002640|\U0001F9B9\U0001F3FE\U0000200D\U00002640|\U0001F9B9\U0001F3FF\U0000200D\U00002640|\U0001F9D9\U0000200D\U00002642\U0000FE0F|\U0001F9D9\U0001F3FB\U0000200D\U00002642|\U0001F9D9\U0001F3FC\U0000200D\U00002642|\U0001F9D9\U0001F3FD\U0000200D\U00002642|\U0001F9D9\U0001F3FE\U0000200D\U00002642|\U0001F9D9\U0001F3FF\U0000200D\U00002642|\U0001F9D9\U0000200D\U00002640\U0000FE0F|\U0001F9D9\U0001F3FB\U0000200D\U00002640|\U0001F9D9\U0001F3FC\U0000200D\U00002640|\U0001F9D9\U0001F3FD\U0000200D\U00002640|\U0001F9D9\U0001F3FE\U0000200D\U00002640|\U0001F9D9\U0001F3FF\U0000200D\U00002640|\U0001F9DA\U0000200D\U00002642\U0000FE0F|\U0001F9DA\U0001F3FB\U0000200D\U00002642|\U0001F9DA\U0001F3FC\U0000200D\U00002642|\U0001F9DA\U0001F3FD\U0000200D\U00002642|\U0001F9DA\U0001F3FE\U0000200D\U00002642|\U0001F9DA\U0001F3FF\U0000200D\U00002642|\U0001F9DA\U0000200D\U00002640\U0000FE0F|\U0001F9DA\U0001F3FB\U0000200D\U00002640|\U0001F9DA\U0001F3FC\U0000200D\U00002640|\U0001F9DA\U0001F3FD\U0000200D\U00002640|\U0001F9DA\U0001F3FE\U0000200D\U00002640|\U0001F9DA\U0001F3FF\U0000200D\U00002640|\U0001F9DB\U0000200D\U00002642\U0000FE0F|\U0001F9DB\U0001F3FB\U0000200D\U00002642|\U0001F9DB\U0001F3FC\U0000200D\U00002642|\U0001F9DB\U0001F3FD\U0000200D\U00002642|\U0001F9DB\U0001F3FE\U0000200D\U00002642|\U0001F9DB\U0001F3FF\U0000200D\U00002642|\U0001F9DB\U0000200D\U00002640\U0000FE0F|\U0001F9DB\U0001F3FB\U0000200D\U00002640|\U0001F9DB\U0001F3FC\U0000200D\U00002640|\U0001F9DB\U0001F3FD\U0000200D\U00002640|\U0001F9DB\U0001F3FE\U0000200D\U00002640|\U0001F9DB\U0001F3FF\U0000200D\U00002640|\U0001F9DC\U0000200D\U00002642\U0000FE0F|\U0001F9DC\U0001F3FB\U0000200D\U00002642|\U0001F9DC\U0001F3FC\U0000200D\U00002642|\U0001F9DC\U0001F3FD\U0000200D\U00002642|\U0001F9DC\U0001F3FE\U0000200D\U00002642|\U0001F9DC\U0001F3FF\U0000200D\U00002642|\U0001F9DC\U0000200D\U00002640\U0000FE0F|\U0001F9DC\U0001F3FB\U0000200D\U00002640|\U0001F9DC\U0001F3FC\U0000200D\U00002640|\U0001F9DC\U0001F3FD\U0000200D\U00002640|\U0001F9DC\U0001F3FE\U0000200D\U00002640|\U0001F9DC\U0001F3FF\U0000200D\U00002640|\U0001F9DD\U0000200D\U00002642\U0000FE0F|\U0001F9DD\U0001F3FB\U0000200D\U00002642|\U0001F9DD\U0001F3FC\U0000200D\U00002642|\U0001F9DD\U0001F3FD\U0000200D\U00002642|\U0001F9DD\U0001F3FE\U0000200D\U00002642|\U0001F9DD\U0001F3FF\U0000200D\U00002642|\U0001F9DD\U0000200D\U00002640\U0000FE0F|\U0001F9DD\U0001F3FB\U0000200D\U00002640|\U0001F9DD\U0001F3FC\U0000200D\U00002640|\U0001F9DD\U0001F3FD\U0000200D\U00002640|\U0001F9DD\U0001F3FE\U0000200D\U00002640|\U0001F9DD\U0001F3FF\U0000200D\U00002640|\U0001F9DE\U0000200D\U00002642\U0000FE0F|\U0001F9DE\U0000200D\U00002640\U0000FE0F|\U0001F9DF\U0000200D\U00002642\U0000FE0F|\U0001F9DF\U0000200D\U00002640\U0000FE0F|\U0001F486\U0000200D\U00002642\U0000FE0F|\U0001F486\U0001F3FB\U0000200D\U00002642|\U0001F486\U0001F3FC\U0000200D\U00002642|\U0001F486\U0001F3FD\U0000200D\U00002642|\U0001F486\U0001F3FE\U0000200D\U00002642|\U0001F486\U0001F3FF\U0000200D\U00002642|\U0001F486\U0000200D\U00002640\U0000FE0F|\U0001F486\U0001F3FB\U0000200D\U00002640|\U0001F486\U0001F3FC\U0000200D\U00002640|\U0001F486\U0001F3FD\U0000200D\U00002640|\U0001F486\U0001F3FE\U0000200D\U00002640|\U0001F486\U0001F3FF\U0000200D\U00002640|\U0001F487\U0000200D\U00002642\U0000FE0F|\U0001F487\U0001F3FB\U0000200D\U00002642|\U0001F487\U0001F3FC\U0000200D\U00002642|\U0001F487\U0001F3FD\U0000200D\U00002642|\U0001F487\U0001F3FE\U0000200D\U00002642|\U0001F487\U0001F3FF\U0000200D\U00002642|\U0001F487\U0000200D\U00002640\U0000FE0F|\U0001F487\U0001F3FB\U0000200D\U00002640|\U0001F487\U0001F3FC\U0000200D\U00002640|\U0001F487\U0001F3FD\U0000200D\U00002640|\U0001F487\U0001F3FE\U0000200D\U00002640|\U0001F487\U0001F3FF\U0000200D\U00002640|\U0001F6B6\U0000200D\U00002642\U0000FE0F|\U0001F6B6\U0001F3FB\U0000200D\U00002642|\U0001F6B6\U0001F3FC\U0000200D\U00002642|\U0001F6B6\U0001F3FD\U0000200D\U00002642|\U0001F6B6\U0001F3FE\U0000200D\U00002642|\U0001F6B6\U0001F3FF\U0000200D\U00002642|\U0001F6B6\U0000200D\U00002640\U0000FE0F|\U0001F6B6\U0001F3FB\U0000200D\U00002640|\U0001F6B6\U0001F3FC\U0000200D\U00002640|\U0001F6B6\U0001F3FD\U0000200D\U00002640|\U0001F6B6\U0001F3FE\U0000200D\U00002640|\U0001F6B6\U0001F3FF\U0000200D\U00002640|\U0001F9CD\U0000200D\U00002642\U0000FE0F|\U0001F9CD\U0001F3FB\U0000200D\U00002642|\U0001F9CD\U0001F3FC\U0000200D\U00002642|\U0001F9CD\U0001F3FD\U0000200D\U00002642|\U0001F9CD\U0001F3FE\U0000200D\U00002642|\U0001F9CD\U0001F3FF\U0000200D\U00002642|\U0001F9CD\U0000200D\U00002640\U0000FE0F|\U0001F9CD\U0001F3FB\U0000200D\U00002640|\U0001F9CD\U0001F3FC\U0000200D\U00002640|\U0001F9CD\U0001F3FD\U0000200D\U00002640|\U0001F9CD\U0001F3FE\U0000200D\U00002640|\U0001F9CD\U0001F3FF\U0000200D\U00002640|\U0001F9CE\U0000200D\U00002642\U0000FE0F|\U0001F9CE\U0001F3FB\U0000200D\U00002642|\U0001F9CE\U0001F3FC\U0000200D\U00002642|\U0001F9CE\U0001F3FD\U0000200D\U00002642|\U0001F9CE\U0001F3FE\U0000200D\U00002642|\U0001F9CE\U0001F3FF\U0000200D\U00002642|\U0001F9CE\U0000200D\U00002640\U0000FE0F|\U0001F9CE\U0001F3FB\U0000200D\U00002640|\U0001F9CE\U0001F3FC\U0000200D\U00002640|\U0001F9CE\U0001F3FD\U0000200D\U00002640|\U0001F9CE\U0001F3FE\U0000200D\U00002640|\U0001F9CE\U0001F3FF\U0000200D\U00002640|\U0001F9D1\U0001F3FB\U0000200D\U0001F9AF|\U0001F9D1\U0001F3FC\U0000200D\U0001F9AF|\U0001F9D1\U0001F3FD\U0000200D\U0001F9AF|\U0001F9D1\U0001F3FE\U0000200D\U0001F9AF|\U0001F9D1\U0001F3FF\U0000200D\U0001F9AF|\U0001F468\U0001F3FB\U0000200D\U0001F9AF|\U0001F468\U0001F3FC\U0000200D\U0001F9AF|\U0001F468\U0001F3FD\U0000200D\U0001F9AF|\U0001F468\U0001F3FE\U0000200D\U0001F9AF|\U0001F468\U0001F3FF\U0000200D\U0001F9AF|\U0001F469\U0001F3FB\U0000200D\U0001F9AF|\U0001F469\U0001F3FC\U0000200D\U0001F9AF|\U0001F469\U0001F3FD\U0000200D\U0001F9AF|\U0001F469\U0001F3FE\U0000200D\U0001F9AF|\U0001F469\U0001F3FF\U0000200D\U0001F9AF|\U0001F9D1\U0001F3FB\U0000200D\U0001F9BC|\U0001F9D1\U0001F3FC\U0000200D\U0001F9BC|\U0001F9D1\U0001F3FD\U0000200D\U0001F9BC|\U0001F9D1\U0001F3FE\U0000200D\U0001F9BC|\U0001F9D1\U0001F3FF\U0000200D\U0001F9BC|\U0001F468\U0001F3FB\U0000200D\U0001F9BC|\U0001F468\U0001F3FC\U0000200D\U0001F9BC|\U0001F468\U0001F3FD\U0000200D\U0001F9BC|\U0001F468\U0001F3FE\U0000200D\U0001F9BC|\U0001F468\U0001F3FF\U0000200D\U0001F9BC|\U0001F469\U0001F3FB\U0000200D\U0001F9BC|\U0001F469\U0001F3FC\U0000200D\U0001F9BC|\U0001F469\U0001F3FD\U0000200D\U0001F9BC|\U0001F469\U0001F3FE\U0000200D\U0001F9BC|\U0001F469\U0001F3FF\U0000200D\U0001F9BC|\U0001F9D1\U0001F3FB\U0000200D\U0001F9BD|\U0001F9D1\U0001F3FC\U0000200D\U0001F9BD|\U0001F9D1\U0001F3FD\U0000200D\U0001F9BD|\U0001F9D1\U0001F3FE\U0000200D\U0001F9BD|\U0001F9D1\U0001F3FF\U0000200D\U0001F9BD|\U0001F468\U0001F3FB\U0000200D\U0001F9BD|\U0001F468\U0001F3FC\U0000200D\U0001F9BD|\U0001F468\U0001F3FD\U0000200D\U0001F9BD|\U0001F468\U0001F3FE\U0000200D\U0001F9BD|\U0001F468\U0001F3FF\U0000200D\U0001F9BD|\U0001F469\U0001F3FB\U0000200D\U0001F9BD|\U0001F469\U0001F3FC\U0000200D\U0001F9BD|\U0001F469\U0001F3FD\U0000200D\U0001F9BD|\U0001F469\U0001F3FE\U0000200D\U0001F9BD|\U0001F469\U0001F3FF\U0000200D\U0001F9BD|\U0001F3C3\U0000200D\U00002642\U0000FE0F|\U0001F3C3\U0001F3FB\U0000200D\U00002642|\U0001F3C3\U0001F3FC\U0000200D\U00002642|\U0001F3C3\U0001F3FD\U0000200D\U00002642|\U0001F3C3\U0001F3FE\U0000200D\U00002642|\U0001F3C3\U0001F3FF\U0000200D\U00002642|\U0001F3C3\U0000200D\U00002640\U0000FE0F|\U0001F3C3\U0001F3FB\U0000200D\U00002640|\U0001F3C3\U0001F3FC\U0000200D\U00002640|\U0001F3C3\U0001F3FD\U0000200D\U00002640|\U0001F3C3\U0001F3FE\U0000200D\U00002640|\U0001F3C3\U0001F3FF\U0000200D\U00002640|\U0001F46F\U0000200D\U00002642\U0000FE0F|\U0001F46F\U0000200D\U00002640\U0000FE0F|\U0001F9D6\U0000200D\U00002642\U0000FE0F|\U0001F9D6\U0001F3FB\U0000200D\U00002642|\U0001F9D6\U0001F3FC\U0000200D\U00002642|\U0001F9D6\U0001F3FD\U0000200D\U00002642|\U0001F9D6\U0001F3FE\U0000200D\U00002642|\U0001F9D6\U0001F3FF\U0000200D\U00002642|\U0001F9D6\U0000200D\U00002640\U0000FE0F|\U0001F9D6\U0001F3FB\U0000200D\U00002640|\U0001F9D6\U0001F3FC\U0000200D\U00002640|\U0001F9D6\U0001F3FD\U0000200D\U00002640|\U0001F9D6\U0001F3FE\U0000200D\U00002640|\U0001F9D6\U0001F3FF\U0000200D\U00002640|\U0001F9D7\U0000200D\U00002642\U0000FE0F|\U0001F9D7\U0001F3FB\U0000200D\U00002642|\U0001F9D7\U0001F3FC\U0000200D\U00002642|\U0001F9D7\U0001F3FD\U0000200D\U00002642|\U0001F9D7\U0001F3FE\U0000200D\U00002642|\U0001F9D7\U0001F3FF\U0000200D\U00002642|\U0001F9D7\U0000200D\U00002640\U0000FE0F|\U0001F9D7\U0001F3FB\U0000200D\U00002640|\U0001F9D7\U0001F3FC\U0000200D\U00002640|\U0001F9D7\U0001F3FD\U0000200D\U00002640|\U0001F9D7\U0001F3FE\U0000200D\U00002640|\U0001F9D7\U0001F3FF\U0000200D\U00002640|\U0001F3CC\U0000200D\U00002642\U0000FE0F|\U0001F3CC\U0000FE0F\U0000200D\U00002642|\U0001F3CC\U0001F3FB\U0000200D\U00002642|\U0001F3CC\U0001F3FC\U0000200D\U00002642|\U0001F3CC\U0001F3FD\U0000200D\U00002642|\U0001F3CC\U0001F3FE\U0000200D\U00002642|\U0001F3CC\U0001F3FF\U0000200D\U00002642|\U0001F3CC\U0000200D\U00002640\U0000FE0F|\U0001F3CC\U0000FE0F\U0000200D\U00002640|\U0001F3CC\U0001F3FB\U0000200D\U00002640|\U0001F3CC\U0001F3FC\U0000200D\U00002640|\U0001F3CC\U0001F3FD\U0000200D\U00002640|\U0001F3CC\U0001F3FE\U0000200D\U00002640|\U0001F3CC\U0001F3FF\U0000200D\U00002640|\U0001F3C4\U0000200D\U00002642\U0000FE0F|\U0001F3C4\U0001F3FB\U0000200D\U00002642|\U0001F3C4\U0001F3FC\U0000200D\U00002642|\U0001F3C4\U0001F3FD\U0000200D\U00002642|\U0001F3C4\U0001F3FE\U0000200D\U00002642|\U0001F3C4\U0001F3FF\U0000200D\U00002642|\U0001F3C4\U0000200D\U00002640\U0000FE0F|\U0001F3C4\U0001F3FB\U0000200D\U00002640|\U0001F3C4\U0001F3FC\U0000200D\U00002640|\U0001F3C4\U0001F3FD\U0000200D\U00002640|\U0001F3C4\U0001F3FE\U0000200D\U00002640|\U0001F3C4\U0001F3FF\U0000200D\U00002640|\U0001F6A3\U0000200D\U00002642\U0000FE0F|\U0001F6A3\U0001F3FB\U0000200D\U00002642|\U0001F6A3\U0001F3FC\U0000200D\U00002642|\U0001F6A3\U0001F3FD\U0000200D\U00002642|\U0001F6A3\U0001F3FE\U0000200D\U00002642|\U0001F6A3\U0001F3FF\U0000200D\U00002642|\U0001F6A3\U0000200D\U00002640\U0000FE0F|\U0001F6A3\U0001F3FB\U0000200D\U00002640|\U0001F6A3\U0001F3FC\U0000200D\U00002640|\U0001F6A3\U0001F3FD\U0000200D\U00002640|\U0001F6A3\U0001F3FE\U0000200D\U00002640|\U0001F6A3\U0001F3FF\U0000200D\U00002640|\U0001F3CA\U0000200D\U00002642\U0000FE0F|\U0001F3CA\U0001F3FB\U0000200D\U00002642|\U0001F3CA\U0001F3FC\U0000200D\U00002642|\U0001F3CA\U0001F3FD\U0000200D\U00002642|\U0001F3CA\U0001F3FE\U0000200D\U00002642|\U0001F3CA\U0001F3FF\U0000200D\U00002642|\U0001F3CA\U0000200D\U00002640\U0000FE0F|\U0001F3CA\U0001F3FB\U0000200D\U00002640|\U0001F3CA\U0001F3FC\U0000200D\U00002640|\U0001F3CA\U0001F3FD\U0000200D\U00002640|\U0001F3CA\U0001F3FE\U0000200D\U00002640|\U0001F3CA\U0001F3FF\U0000200D\U00002640|\U000026F9\U0000200D\U00002642\U0000FE0F|\U000026F9\U0000FE0F\U0000200D\U00002642|\U000026F9\U0001F3FB\U0000200D\U00002642|\U000026F9\U0001F3FC\U0000200D\U00002642|\U000026F9\U0001F3FD\U0000200D\U00002642|\U000026F9\U0001F3FE\U0000200D\U00002642|\U000026F9\U0001F3FF\U0000200D\U00002642|\U000026F9\U0000200D\U00002640\U0000FE0F|\U000026F9\U0000FE0F\U0000200D\U00002640|\U000026F9\U0001F3FB\U0000200D\U00002640|\U000026F9\U0001F3FC\U0000200D\U00002640|\U000026F9\U0001F3FD\U0000200D\U00002640|\U000026F9\U0001F3FE\U0000200D\U00002640|\U000026F9\U0001F3FF\U0000200D\U00002640|\U0001F3CB\U0000200D\U00002642\U0000FE0F|\U0001F3CB\U0000FE0F\U0000200D\U00002642|\U0001F3CB\U0001F3FB\U0000200D\U00002642|\U0001F3CB\U0001F3FC\U0000200D\U00002642|\U0001F3CB\U0001F3FD\U0000200D\U00002642|\U0001F3CB\U0001F3FE\U0000200D\U00002642|\U0001F3CB\U0001F3FF\U0000200D\U00002642|\U0001F3CB\U0000200D\U00002640\U0000FE0F|\U0001F3CB\U0000FE0F\U0000200D\U00002640|\U0001F3CB\U0001F3FB\U0000200D\U00002640|\U0001F3CB\U0001F3FC\U0000200D\U00002640|\U0001F3CB\U0001F3FD\U0000200D\U00002640|\U0001F3CB\U0001F3FE\U0000200D\U00002640|\U0001F3CB\U0001F3FF\U0000200D\U00002640|\U0001F6B4\U0000200D\U00002642\U0000FE0F|\U0001F6B4\U0001F3FB\U0000200D\U00002642|\U0001F6B4\U0001F3FC\U0000200D\U00002642|\U0001F6B4\U0001F3FD\U0000200D\U00002642|\U0001F6B4\U0001F3FE\U0000200D\U00002642|\U0001F6B4\U0001F3FF\U0000200D\U00002642|\U0001F6B4\U0000200D\U00002640\U0000FE0F|\U0001F6B4\U0001F3FB\U0000200D\U00002640|\U0001F6B4\U0001F3FC\U0000200D\U00002640|\U0001F6B4\U0001F3FD\U0000200D\U00002640|\U0001F6B4\U0001F3FE\U0000200D\U00002640|\U0001F6B4\U0001F3FF\U0000200D\U00002640|\U0001F6B5\U0000200D\U00002642\U0000FE0F|\U0001F6B5\U0001F3FB\U0000200D\U00002642|\U0001F6B5\U0001F3FC\U0000200D\U00002642|\U0001F6B5\U0001F3FD\U0000200D\U00002642|\U0001F6B5\U0001F3FE\U0000200D\U00002642|\U0001F6B5\U0001F3FF\U0000200D\U00002642|\U0001F6B5\U0000200D\U00002640\U0000FE0F|\U0001F6B5\U0001F3FB\U0000200D\U00002640|\U0001F6B5\U0001F3FC\U0000200D\U00002640|\U0001F6B5\U0001F3FD\U0000200D\U00002640|\U0001F6B5\U0001F3FE\U0000200D\U00002640|\U0001F6B5\U0001F3FF\U0000200D\U00002640|\U0001F938\U0000200D\U00002642\U0000FE0F|\U0001F938\U0001F3FB\U0000200D\U00002642|\U0001F938\U0001F3FC\U0000200D\U00002642|\U0001F938\U0001F3FD\U0000200D\U00002642|\U0001F938\U0001F3FE\U0000200D\U00002642|\U0001F938\U0001F3FF\U0000200D\U00002642|\U0001F938\U0000200D\U00002640\U0000FE0F|\U0001F938\U0001F3FB\U0000200D\U00002640|\U0001F938\U0001F3FC\U0000200D\U00002640|\U0001F938\U0001F3FD\U0000200D\U00002640|\U0001F938\U0001F3FE\U0000200D\U00002640|\U0001F938\U0001F3FF\U0000200D\U00002640|\U0001F93C\U0000200D\U00002642\U0000FE0F|\U0001F93C\U0000200D\U00002640\U0000FE0F|\U0001F93D\U0000200D\U00002642\U0000FE0F|\U0001F93D\U0001F3FB\U0000200D\U00002642|\U0001F93D\U0001F3FC\U0000200D\U00002642|\U0001F93D\U0001F3FD\U0000200D\U00002642|\U0001F93D\U0001F3FE\U0000200D\U00002642|\U0001F93D\U0001F3FF\U0000200D\U00002642|\U0001F93D\U0000200D\U00002640\U0000FE0F|\U0001F93D\U0001F3FB\U0000200D\U00002640|\U0001F93D\U0001F3FC\U0000200D\U00002640|\U0001F93D\U0001F3FD\U0000200D\U00002640|\U0001F93D\U0001F3FE\U0000200D\U00002640|\U0001F93D\U0001F3FF\U0000200D\U00002640|\U0001F93E\U0000200D\U00002642\U0000FE0F|\U0001F93E\U0001F3FB\U0000200D\U00002642|\U0001F93E\U0001F3FC\U0000200D\U00002642|\U0001F93E\U0001F3FD\U0000200D\U00002642|\U0001F93E\U0001F3FE\U0000200D\U00002642|\U0001F93E\U0001F3FF\U0000200D\U00002642|\U0001F93E\U0000200D\U00002640\U0000FE0F|\U0001F93E\U0001F3FB\U0000200D\U00002640|\U0001F93E\U0001F3FC\U0000200D\U00002640|\U0001F93E\U0001F3FD\U0000200D\U00002640|\U0001F93E\U0001F3FE\U0000200D\U00002640|\U0001F93E\U0001F3FF\U0000200D\U00002640|\U0001F939\U0000200D\U00002642\U0000FE0F|\U0001F939\U0001F3FB\U0000200D\U00002642|\U0001F939\U0001F3FC\U0000200D\U00002642|\U0001F939\U0001F3FD\U0000200D\U00002642|\U0001F939\U0001F3FE\U0000200D\U00002642|\U0001F939\U0001F3FF\U0000200D\U00002642|\U0001F939\U0000200D\U00002640\U0000FE0F|\U0001F939\U0001F3FB\U0000200D\U00002640|\U0001F939\U0001F3FC\U0000200D\U00002640|\U0001F939\U0001F3FD\U0000200D\U00002640|\U0001F939\U0001F3FE\U0000200D\U00002640|\U0001F939\U0001F3FF\U0000200D\U00002640|\U0001F9D8\U0000200D\U00002642\U0000FE0F|\U0001F9D8\U0001F3FB\U0000200D\U00002642|\U0001F9D8\U0001F3FC\U0000200D\U00002642|\U0001F9D8\U0001F3FD\U0000200D\U00002642|\U0001F9D8\U0001F3FE\U0000200D\U00002642|\U0001F9D8\U0001F3FF\U0000200D\U00002642|\U0001F9D8\U0000200D\U00002640\U0000FE0F|\U0001F9D8\U0001F3FB\U0000200D\U00002640|\U0001F9D8\U0001F3FC\U0000200D\U00002640|\U0001F9D8\U0001F3FD\U0000200D\U00002640|\U0001F9D8\U0001F3FE\U0000200D\U00002640|\U0001F9D8\U0001F3FF\U0000200D\U00002640|\U0001F43B\U0000200D\U00002744\U0000FE0F|\U0001F3F3\U0000FE0F\U0000200D\U0001F308|\U0001F3F3\U0000200D\U000026A7\U0000FE0F|\U0001F3F3\U0000FE0F\U0000200D\U000026A7|\U0001F3F4\U0000200D\U00002620\U0000FE0F|\U0001F441\U0000200D\U0001F5E8|\U0001F468\U0000200D\U0001F9B0|\U0001F468\U0000200D\U0001F9B1|\U0001F468\U0000200D\U0001F9B3|\U0001F468\U0000200D\U0001F9B2|\U0001F469\U0000200D\U0001F9B0|\U0001F9D1\U0000200D\U0001F9B0|\U0001F469\U0000200D\U0001F9B1|\U0001F9D1\U0000200D\U0001F9B1|\U0001F469\U0000200D\U0001F9B3|\U0001F9D1\U0000200D\U0001F9B3|\U0001F469\U0000200D\U0001F9B2|\U0001F9D1\U0000200D\U0001F9B2|\U0001F471\U0000200D\U00002640|\U0001F471\U0000200D\U00002642|\U0001F64D\U0000200D\U00002642|\U0001F64D\U0000200D\U00002640|\U0001F64E\U0000200D\U00002642|\U0001F64E\U0000200D\U00002640|\U0001F645\U0000200D\U00002642|\U0001F645\U0000200D\U00002640|\U0001F646\U0000200D\U00002642|\U0001F646\U0000200D\U00002640|\U0001F481\U0000200D\U00002642|\U0001F481\U0000200D\U00002640|\U0001F64B\U0000200D\U00002642|\U0001F64B\U0000200D\U00002640|\U0001F9CF\U0000200D\U00002642|\U0001F9CF\U0000200D\U00002640|\U0001F647\U0000200D\U00002642|\U0001F647\U0000200D\U00002640|\U0001F926\U0000200D\U00002642|\U0001F926\U0000200D\U00002640|\U0001F937\U0000200D\U00002642|\U0001F937\U0000200D\U00002640|\U0001F9D1\U0000200D\U00002695|\U0001F468\U0000200D\U00002695|\U0001F469\U0000200D\U00002695|\U0001F9D1\U0000200D\U0001F393|\U0001F468\U0000200D\U0001F393|\U0001F469\U0000200D\U0001F393|\U0001F9D1\U0000200D\U0001F3EB|\U0001F468\U0000200D\U0001F3EB|\U0001F469\U0000200D\U0001F3EB|\U0001F9D1\U0000200D\U00002696|\U0001F468\U0000200D\U00002696|\U0001F469\U0000200D\U00002696|\U0001F9D1\U0000200D\U0001F33E|\U0001F468\U0000200D\U0001F33E|\U0001F469\U0000200D\U0001F33E|\U0001F9D1\U0000200D\U0001F373|\U0001F468\U0000200D\U0001F373|\U0001F469\U0000200D\U0001F373|\U0001F9D1\U0000200D\U0001F527|\U0001F468\U0000200D\U0001F527|\U0001F469\U0000200D\U0001F527|\U0001F9D1\U0000200D\U0001F3ED|\U0001F468\U0000200D\U0001F3ED|\U0001F469\U0000200D\U0001F3ED|\U0001F9D1\U0000200D\U0001F4BC|\U0001F468\U0000200D\U0001F4BC|\U0001F469\U0000200D\U0001F4BC|\U0001F9D1\U0000200D\U0001F52C|\U0001F468\U0000200D\U0001F52C|\U0001F469\U0000200D\U0001F52C|\U0001F9D1\U0000200D\U0001F4BB|\U0001F468\U0000200D\U0001F4BB|\U0001F469\U0000200D\U0001F4BB|\U0001F9D1\U0000200D\U0001F3A4|\U0001F468\U0000200D\U0001F3A4|\U0001F469\U0000200D\U0001F3A4|\U0001F9D1\U0000200D\U0001F3A8|\U0001F468\U0000200D\U0001F3A8|\U0001F469\U0000200D\U0001F3A8|\U0001F9D1\U0000200D\U00002708|\U0001F468\U0000200D\U00002708|\U0001F469\U0000200D\U00002708|\U0001F9D1\U0000200D\U0001F680|\U0001F468\U0000200D\U0001F680|\U0001F469\U0000200D\U0001F680|\U0001F9D1\U0000200D\U0001F692|\U0001F468\U0000200D\U0001F692|\U0001F469\U0000200D\U0001F692|\U0001F46E\U0000200D\U00002642|\U0001F46E\U0000200D\U00002640|\U0001F575\U0000200D\U00002642|\U0001F575\U0000200D\U00002640|\U0001F482\U0000200D\U00002642|\U0001F482\U0000200D\U00002640|\U0001F477\U0000200D\U00002642|\U0001F477\U0000200D\U00002640|\U0001F473\U0000200D\U00002642|\U0001F473\U0000200D\U00002640|\U0001F935\U0000200D\U00002642|\U0001F935\U0000200D\U00002640|\U0001F470\U0000200D\U00002642|\U0001F470\U0000200D\U00002640|\U0001F469\U0000200D\U0001F37C|\U0001F468\U0000200D\U0001F37C|\U0001F9D1\U0000200D\U0001F37C|\U0001F9D1\U0000200D\U0001F384|\U0001F9B8\U0000200D\U00002642|\U0001F9B8\U0000200D\U00002640|\U0001F9B9\U0000200D\U00002642|\U0001F9B9\U0000200D\U00002640|\U0001F9D9\U0000200D\U00002642|\U0001F9D9\U0000200D\U00002640|\U0001F9DA\U0000200D\U00002642|\U0001F9DA\U0000200D\U00002640|\U0001F9DB\U0000200D\U00002642|\U0001F9DB\U0000200D\U00002640|\U0001F9DC\U0000200D\U00002642|\U0001F9DC\U0000200D\U00002640|\U0001F9DD\U0000200D\U00002642|\U0001F9DD\U0000200D\U00002640|\U0001F9DE\U0000200D\U00002642|\U0001F9DE\U0000200D\U00002640|\U0001F9DF\U0000200D\U00002642|\U0001F9DF\U0000200D\U00002640|\U0001F486\U0000200D\U00002642|\U0001F486\U0000200D\U00002640|\U0001F487\U0000200D\U00002642|\U0001F487\U0000200D\U00002640|\U0001F6B6\U0000200D\U00002642|\U0001F6B6\U0000200D\U00002640|\U0001F9CD\U0000200D\U00002642|\U0001F9CD\U0000200D\U00002640|\U0001F9CE\U0000200D\U00002642|\U0001F9CE\U0000200D\U00002640|\U0001F9D1\U0000200D\U0001F9AF|\U0001F468\U0000200D\U0001F9AF|\U0001F469\U0000200D\U0001F9AF|\U0001F9D1\U0000200D\U0001F9BC|\U0001F468\U0000200D\U0001F9BC|\U0001F469\U0000200D\U0001F9BC|\U0001F9D1\U0000200D\U0001F9BD|\U0001F468\U0000200D\U0001F9BD|\U0001F469\U0000200D\U0001F9BD|\U0001F3C3\U0000200D\U00002642|\U0001F3C3\U0000200D\U00002640|\U0001F46F\U0000200D\U00002642|\U0001F46F\U0000200D\U00002640|\U0001F9D6\U0000200D\U00002642|\U0001F9D6\U0000200D\U00002640|\U0001F9D7\U0000200D\U00002642|\U0001F9D7\U0000200D\U00002640|\U0001F3CC\U0000200D\U00002642|\U0001F3CC\U0000200D\U00002640|\U0001F3C4\U0000200D\U00002642|\U0001F3C4\U0000200D\U00002640|\U0001F6A3\U0000200D\U00002642|\U0001F6A3\U0000200D\U00002640|\U0001F3CA\U0000200D\U00002642|\U0001F3CA\U0000200D\U00002640|\U000026F9\U0000200D\U00002642|\U000026F9\U0000200D\U00002640|\U0001F3CB\U0000200D\U00002642|\U0001F3CB\U0000200D\U00002640|\U0001F6B4\U0000200D\U00002642|\U0001F6B4\U0000200D\U00002640|\U0001F6B5\U0000200D\U00002642|\U0001F6B5\U0000200D\U00002640|\U0001F938\U0000200D\U00002642|\U0001F938\U0000200D\U00002640|\U0001F93C\U0000200D\U00002642|\U0001F93C\U0000200D\U00002640|\U0001F93D\U0000200D\U00002642|\U0001F93D\U0000200D\U00002640|\U0001F93E\U0000200D\U00002642|\U0001F93E\U0000200D\U00002640|\U0001F939\U0000200D\U00002642|\U0001F939\U0000200D\U00002640|\U0001F9D8\U0000200D\U00002642|\U0001F9D8\U0000200D\U00002640|\U0001F468\U0000200D\U0001F466|\U0001F468\U0000200D\U0001F467|\U0001F469\U0000200D\U0001F466|\U0001F469\U0000200D\U0001F467|\U0001F415\U0000200D\U0001F9BA|\U0001F408\U0000200D\U00002B1B|\U0001F43B\U0000200D\U00002744|\U00000023\U0000FE0F\U000020E3|\U0000002A\U0000FE0F\U000020E3|\U00000030\U0000FE0F\U000020E3|\U00000031\U0000FE0F\U000020E3|\U00000032\U0000FE0F\U000020E3|\U00000033\U0000FE0F\U000020E3|\U00000034\U0000FE0F\U000020E3|\U00000035\U0000FE0F\U000020E3|\U00000036\U0000FE0F\U000020E3|\U00000037\U0000FE0F\U000020E3|\U00000038\U0000FE0F\U000020E3|\U00000039\U0000FE0F\U000020E3|\U0001F3F3\U0000200D\U0001F308|\U0001F3F3\U0000200D\U000026A7|\U0001F3F4\U0000200D\U00002620|\U0000263A\U0000FE0F|\U00002639\U0000FE0F|\U00002620\U0000FE0F|\U00002763\U0000FE0F|\U00002764\U0000FE0F|\U0001F573\U0000FE0F|\U0001F5E8\U0000FE0F|\U0001F5EF\U0000FE0F|\U0001F44B\U0001F3FB|\U0001F44B\U0001F3FC|\U0001F44B\U0001F3FD|\U0001F44B\U0001F3FE|\U0001F44B\U0001F3FF|\U0001F91A\U0001F3FB|\U0001F91A\U0001F3FC|\U0001F91A\U0001F3FD|\U0001F91A\U0001F3FE|\U0001F91A\U0001F3FF|\U0001F590\U0000FE0F|\U0001F590\U0001F3FB|\U0001F590\U0001F3FC|\U0001F590\U0001F3FD|\U0001F590\U0001F3FE|\U0001F590\U0001F3FF|\U0000270B\U0001F3FB|\U0000270B\U0001F3FC|\U0000270B\U0001F3FD|\U0000270B\U0001F3FE|\U0000270B\U0001F3FF|\U0001F596\U0001F3FB|\U0001F596\U0001F3FC|\U0001F596\U0001F3FD|\U0001F596\U0001F3FE|\U0001F596\U0001F3FF|\U0001F44C\U0001F3FB|\U0001F44C\U0001F3FC|\U0001F44C\U0001F3FD|\U0001F44C\U0001F3FE|\U0001F44C\U0001F3FF|\U0001F90C\U0001F3FB|\U0001F90C\U0001F3FC|\U0001F90C\U0001F3FD|\U0001F90C\U0001F3FE|\U0001F90C\U0001F3FF|\U0001F90F\U0001F3FB|\U0001F90F\U0001F3FC|\U0001F90F\U0001F3FD|\U0001F90F\U0001F3FE|\U0001F90F\U0001F3FF|\U0000270C\U0000FE0F|\U0000270C\U0001F3FB|\U0000270C\U0001F3FC|\U0000270C\U0001F3FD|\U0000270C\U0001F3FE|\U0000270C\U0001F3FF|\U0001F91E\U0001F3FB|\U0001F91E\U0001F3FC|\U0001F91E\U0001F3FD|\U0001F91E\U0001F3FE|\U0001F91E\U0001F3FF|\U0001F91F\U0001F3FB|\U0001F91F\U0001F3FC|\U0001F91F\U0001F3FD|\U0001F91F\U0001F3FE|\U0001F91F\U0001F3FF|\U0001F918\U0001F3FB|\U0001F918\U0001F3FC|\U0001F918\U0001F3FD|\U0001F918\U0001F3FE|\U0001F918\U0001F3FF|\U0001F919\U0001F3FB|\U0001F919\U0001F3FC|\U0001F919\U0001F3FD|\U0001F919\U0001F3FE|\U0001F919\U0001F3FF|\U0001F448\U0001F3FB|\U0001F448\U0001F3FC|\U0001F448\U0001F3FD|\U0001F448\U0001F3FE|\U0001F448\U0001F3FF|\U0001F449\U0001F3FB|\U0001F449\U0001F3FC|\U0001F449\U0001F3FD|\U0001F449\U0001F3FE|\U0001F449\U0001F3FF|\U0001F446\U0001F3FB|\U0001F446\U0001F3FC|\U0001F446\U0001F3FD|\U0001F446\U0001F3FE|\U0001F446\U0001F3FF|\U0001F595\U0001F3FB|\U0001F595\U0001F3FC|\U0001F595\U0001F3FD|\U0001F595\U0001F3FE|\U0001F595\U0001F3FF|\U0001F447\U0001F3FB|\U0001F447\U0001F3FC|\U0001F447\U0001F3FD|\U0001F447\U0001F3FE|\U0001F447\U0001F3FF|\U0000261D\U0000FE0F|\U0000261D\U0001F3FB|\U0000261D\U0001F3FC|\U0000261D\U0001F3FD|\U0000261D\U0001F3FE|\U0000261D\U0001F3FF|\U0001F44D\U0001F3FB|\U0001F44D\U0001F3FC|\U0001F44D\U0001F3FD|\U0001F44D\U0001F3FE|\U0001F44D\U0001F3FF|\U0001F44E\U0001F3FB|\U0001F44E\U0001F3FC|\U0001F44E\U0001F3FD|\U0001F44E\U0001F3FE|\U0001F44E\U0001F3FF|\U0000270A\U0001F3FB|\U0000270A\U0001F3FC|\U0000270A\U0001F3FD|\U0000270A\U0001F3FE|\U0000270A\U0001F3FF|\U0001F44A\U0001F3FB|\U0001F44A\U0001F3FC|\U0001F44A\U0001F3FD|\U0001F44A\U0001F3FE|\U0001F44A\U0001F3FF|\U0001F91B\U0001F3FB|\U0001F91B\U0001F3FC|\U0001F91B\U0001F3FD|\U0001F91B\U0001F3FE|\U0001F91B\U0001F3FF|\U0001F91C\U0001F3FB|\U0001F91C\U0001F3FC|\U0001F91C\U0001F3FD|\U0001F91C\U0001F3FE|\U0001F91C\U0001F3FF|\U0001F44F\U0001F3FB|\U0001F44F\U0001F3FC|\U0001F44F\U0001F3FD|\U0001F44F\U0001F3FE|\U0001F44F\U0001F3FF|\U0001F64C\U0001F3FB|\U0001F64C\U0001F3FC|\U0001F64C\U0001F3FD|\U0001F64C\U0001F3FE|\U0001F64C\U0001F3FF|\U0001F450\U0001F3FB|\U0001F450\U0001F3FC|\U0001F450\U0001F3FD|\U0001F450\U0001F3FE|\U0001F450\U0001F3FF|\U0001F932\U0001F3FB|\U0001F932\U0001F3FC|\U0001F932\U0001F3FD|\U0001F932\U0001F3FE|\U0001F932\U0001F3FF|\U0001F64F\U0001F3FB|\U0001F64F\U0001F3FC|\U0001F64F\U0001F3FD|\U0001F64F\U0001F3FE|\U0001F64F\U0001F3FF|\U0000270D\U0000FE0F|\U0000270D\U0001F3FB|\U0000270D\U0001F3FC|\U0000270D\U0001F3FD|\U0000270D\U0001F3FE|\U0000270D\U0001F3FF|\U0001F485\U0001F3FB|\U0001F485\U0001F3FC|\U0001F485\U0001F3FD|\U0001F485\U0001F3FE|\U0001F485\U0001F3FF|\U0001F933\U0001F3FB|\U0001F933\U0001F3FC|\U0001F933\U0001F3FD|\U0001F933\U0001F3FE|\U0001F933\U0001F3FF|\U0001F4AA\U0001F3FB|\U0001F4AA\U0001F3FC|\U0001F4AA\U0001F3FD|\U0001F4AA\U0001F3FE|\U0001F4AA\U0001F3FF|\U0001F9B5\U0001F3FB|\U0001F9B5\U0001F3FC|\U0001F9B5\U0001F3FD|\U0001F9B5\U0001F3FE|\U0001F9B5\U0001F3FF|\U0001F9B6\U0001F3FB|\U0001F9B6\U0001F3FC|\U0001F9B6\U0001F3FD|\U0001F9B6\U0001F3FE|\U0001F9B6\U0001F3FF|\U0001F442\U0001F3FB|\U0001F442\U0001F3FC|\U0001F442\U0001F3FD|\U0001F442\U0001F3FE|\U0001F442\U0001F3FF|\U0001F9BB\U0001F3FB|\U0001F9BB\U0001F3FC|\U0001F9BB\U0001F3FD|\U0001F9BB\U0001F3FE|\U0001F9BB\U0001F3FF|\U0001F443\U0001F3FB|\U0001F443\U0001F3FC|\U0001F443\U0001F3FD|\U0001F443\U0001F3FE|\U0001F443\U0001F3FF|\U0001F441\U0000FE0F|\U0001F476\U0001F3FB|\U0001F476\U0001F3FC|\U0001F476\U0001F3FD|\U0001F476\U0001F3FE|\U0001F476\U0001F3FF|\U0001F9D2\U0001F3FB|\U0001F9D2\U0001F3FC|\U0001F9D2\U0001F3FD|\U0001F9D2\U0001F3FE|\U0001F9D2\U0001F3FF|\U0001F466\U0001F3FB|\U0001F466\U0001F3FC|\U0001F466\U0001F3FD|\U0001F466\U0001F3FE|\U0001F466\U0001F3FF|\U0001F467\U0001F3FB|\U0001F467\U0001F3FC|\U0001F467\U0001F3FD|\U0001F467\U0001F3FE|\U0001F467\U0001F3FF|\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FF|\U0001F471\U0001F3FB|\U0001F471\U0001F3FC|\U0001F471\U0001F3FD|\U0001F471\U0001F3FE|\U0001F471\U0001F3FF|\U0001F468\U0001F3FB|\U0001F468\U0001F3FC|\U0001F468\U0001F3FD|\U0001F468\U0001F3FE|\U0001F468\U0001F3FF|\U0001F9D4\U0001F3FB|\U0001F9D4\U0001F3FC|\U0001F9D4\U0001F3FD|\U0001F9D4\U0001F3FE|\U0001F9D4\U0001F3FF|\U0001F469\U0001F3FB|\U0001F469\U0001F3FC|\U0001F469\U0001F3FD|\U0001F469\U0001F3FE|\U0001F469\U0001F3FF|\U0001F9D3\U0001F3FB|\U0001F9D3\U0001F3FC|\U0001F9D3\U0001F3FD|\U0001F9D3\U0001F3FE|\U0001F9D3\U0001F3FF|\U0001F474\U0001F3FB|\U0001F474\U0001F3FC|\U0001F474\U0001F3FD|\U0001F474\U0001F3FE|\U0001F474\U0001F3FF|\U0001F475\U0001F3FB|\U0001F475\U0001F3FC|\U0001F475\U0001F3FD|\U0001F475\U0001F3FE|\U0001F475\U0001F3FF|\U0001F64D\U0001F3FB|\U0001F64D\U0001F3FC|\U0001F64D\U0001F3FD|\U0001F64D\U0001F3FE|\U0001F64D\U0001F3FF|\U0001F64E\U0001F3FB|\U0001F64E\U0001F3FC|\U0001F64E\U0001F3FD|\U0001F64E\U0001F3FE|\U0001F64E\U0001F3FF|\U0001F645\U0001F3FB|\U0001F645\U0001F3FC|\U0001F645\U0001F3FD|\U0001F645\U0001F3FE|\U0001F645\U0001F3FF|\U0001F646\U0001F3FB|\U0001F646\U0001F3FC|\U0001F646\U0001F3FD|\U0001F646\U0001F3FE|\U0001F646\U0001F3FF|\U0001F481\U0001F3FB|\U0001F481\U0001F3FC|\U0001F481\U0001F3FD|\U0001F481\U0001F3FE|\U0001F481\U0001F3FF|\U0001F64B\U0001F3FB|\U0001F64B\U0001F3FC|\U0001F64B\U0001F3FD|\U0001F64B\U0001F3FE|\U0001F64B\U0001F3FF|\U0001F9CF\U0001F3FB|\U0001F9CF\U0001F3FC|\U0001F9CF\U0001F3FD|\U0001F9CF\U0001F3FE|\U0001F9CF\U0001F3FF|\U0001F647\U0001F3FB|\U0001F647\U0001F3FC|\U0001F647\U0001F3FD|\U0001F647\U0001F3FE|\U0001F647\U0001F3FF|\U0001F926\U0001F3FB|\U0001F926\U0001F3FC|\U0001F926\U0001F3FD|\U0001F926\U0001F3FE|\U0001F926\U0001F3FF|\U0001F937\U0001F3FB|\U0001F937\U0001F3FC|\U0001F937\U0001F3FD|\U0001F937\U0001F3FE|\U0001F937\U0001F3FF|\U0001F46E\U0001F3FB|\U0001F46E\U0001F3FC|\U0001F46E\U0001F3FD|\U0001F46E\U0001F3FE|\U0001F46E\U0001F3FF|\U0001F575\U0000FE0F|\U0001F575\U0001F3FB|\U0001F575\U0001F3FC|\U0001F575\U0001F3FD|\U0001F575\U0001F3FE|\U0001F575\U0001F3FF|\U0001F482\U0001F3FB|\U0001F482\U0001F3FC|\U0001F482\U0001F3FD|\U0001F482\U0001F3FE|\U0001F482\U0001F3FF|\U0001F977\U0001F3FB|\U0001F977\U0001F3FC|\U0001F977\U0001F3FD|\U0001F977\U0001F3FE|\U0001F977\U0001F3FF|\U0001F477\U0001F3FB|\U0001F477\U0001F3FC|\U0001F477\U0001F3FD|\U0001F477\U0001F3FE|\U0001F477\U0001F3FF|\U0001F934\U0001F3FB|\U0001F934\U0001F3FC|\U0001F934\U0001F3FD|\U0001F934\U0001F3FE|\U0001F934\U0001F3FF|\U0001F478\U0001F3FB|\U0001F478\U0001F3FC|\U0001F478\U0001F3FD|\U0001F478\U0001F3FE|\U0001F478\U0001F3FF|\U0001F473\U0001F3FB|\U0001F473\U0001F3FC|\U0001F473\U0001F3FD|\U0001F473\U0001F3FE|\U0001F473\U0001F3FF|\U0001F472\U0001F3FB|\U0001F472\U0001F3FC|\U0001F472\U0001F3FD|\U0001F472\U0001F3FE|\U0001F472\U0001F3FF|\U0001F9D5\U0001F3FB|\U0001F9D5\U0001F3FC|\U0001F9D5\U0001F3FD|\U0001F9D5\U0001F3FE|\U0001F9D5\U0001F3FF|\U0001F935\U0001F3FB|\U0001F935\U0001F3FC|\U0001F935\U0001F3FD|\U0001F935\U0001F3FE|\U0001F935\U0001F3FF|\U0001F470\U0001F3FB|\U0001F470\U0001F3FC|\U0001F470\U0001F3FD|\U0001F470\U0001F3FE|\U0001F470\U0001F3FF|\U0001F930\U0001F3FB|\U0001F930\U0001F3FC|\U0001F930\U0001F3FD|\U0001F930\U0001F3FE|\U0001F930\U0001F3FF|\U0001F931\U0001F3FB|\U0001F931\U0001F3FC|\U0001F931\U0001F3FD|\U0001F931\U0001F3FE|\U0001F931\U0001F3FF|\U0001F47C\U0001F3FB|\U0001F47C\U0001F3FC|\U0001F47C\U0001F3FD|\U0001F47C\U0001F3FE|\U0001F47C\U0001F3FF|\U0001F385\U0001F3FB|\U0001F385\U0001F3FC|\U0001F385\U0001F3FD|\U0001F385\U0001F3FE|\U0001F385\U0001F3FF|\U0001F936\U0001F3FB|\U0001F936\U0001F3FC|\U0001F936\U0001F3FD|\U0001F936\U0001F3FE|\U0001F936\U0001F3FF|\U0001F9B8\U0001F3FB|\U0001F9B8\U0001F3FC|\U0001F9B8\U0001F3FD|\U0001F9B8\U0001F3FE|\U0001F9B8\U0001F3FF|\U0001F9B9\U0001F3FB|\U0001F9B9\U0001F3FC|\U0001F9B9\U0001F3FD|\U0001F9B9\U0001F3FE|\U0001F9B9\U0001F3FF|\U0001F9D9\U0001F3FB|\U0001F9D9\U0001F3FC|\U0001F9D9\U0001F3FD|\U0001F9D9\U0001F3FE|\U0001F9D9\U0001F3FF|\U0001F9DA\U0001F3FB|\U0001F9DA\U0001F3FC|\U0001F9DA\U0001F3FD|\U0001F9DA\U0001F3FE|\U0001F9DA\U0001F3FF|\U0001F9DB\U0001F3FB|\U0001F9DB\U0001F3FC|\U0001F9DB\U0001F3FD|\U0001F9DB\U0001F3FE|\U0001F9DB\U0001F3FF|\U0001F9DC\U0001F3FB|\U0001F9DC\U0001F3FC|\U0001F9DC\U0001F3FD|\U0001F9DC\U0001F3FE|\U0001F9DC\U0001F3FF|\U0001F9DD\U0001F3FB|\U0001F9DD\U0001F3FC|\U0001F9DD\U0001F3FD|\U0001F9DD\U0001F3FE|\U0001F9DD\U0001F3FF|\U0001F486\U0001F3FB|\U0001F486\U0001F3FC|\U0001F486\U0001F3FD|\U0001F486\U0001F3FE|\U0001F486\U0001F3FF|\U0001F487\U0001F3FB|\U0001F487\U0001F3FC|\U0001F487\U0001F3FD|\U0001F487\U0001F3FE|\U0001F487\U0001F3FF|\U0001F6B6\U0001F3FB|\U0001F6B6\U0001F3FC|\U0001F6B6\U0001F3FD|\U0001F6B6\U0001F3FE|\U0001F6B6\U0001F3FF|\U0001F9CD\U0001F3FB|\U0001F9CD\U0001F3FC|\U0001F9CD\U0001F3FD|\U0001F9CD\U0001F3FE|\U0001F9CD\U0001F3FF|\U0001F9CE\U0001F3FB|\U0001F9CE\U0001F3FC|\U0001F9CE\U0001F3FD|\U0001F9CE\U0001F3FE|\U0001F9CE\U0001F3FF|\U0001F3C3\U0001F3FB|\U0001F3C3\U0001F3FC|\U0001F3C3\U0001F3FD|\U0001F3C3\U0001F3FE|\U0001F3C3\U0001F3FF|\U0001F483\U0001F3FB|\U0001F483\U0001F3FC|\U0001F483\U0001F3FD|\U0001F483\U0001F3FE|\U0001F483\U0001F3FF|\U0001F57A\U0001F3FB|\U0001F57A\U0001F3FC|\U0001F57A\U0001F3FD|\U0001F57A\U0001F3FE|\U0001F57A\U0001F3FF|\U0001F574\U0000FE0F|\U0001F574\U0001F3FB|\U0001F574\U0001F3FC|\U0001F574\U0001F3FD|\U0001F574\U0001F3FE|\U0001F574\U0001F3FF|\U0001F9D6\U0001F3FB|\U0001F9D6\U0001F3FC|\U0001F9D6\U0001F3FD|\U0001F9D6\U0001F3FE|\U0001F9D6\U0001F3FF|\U0001F9D7\U0001F3FB|\U0001F9D7\U0001F3FC|\U0001F9D7\U0001F3FD|\U0001F9D7\U0001F3FE|\U0001F9D7\U0001F3FF|\U0001F3C7\U0001F3FB|\U0001F3C7\U0001F3FC|\U0001F3C7\U0001F3FD|\U0001F3C7\U0001F3FE|\U0001F3C7\U0001F3FF|\U000026F7\U0000FE0F|\U0001F3C2\U0001F3FB|\U0001F3C2\U0001F3FC|\U0001F3C2\U0001F3FD|\U0001F3C2\U0001F3FE|\U0001F3C2\U0001F3FF|\U0001F3CC\U0000FE0F|\U0001F3CC\U0001F3FB|\U0001F3CC\U0001F3FC|\U0001F3CC\U0001F3FD|\U0001F3CC\U0001F3FE|\U0001F3CC\U0001F3FF|\U0001F3C4\U0001F3FB|\U0001F3C4\U0001F3FC|\U0001F3C4\U0001F3FD|\U0001F3C4\U0001F3FE|\U0001F3C4\U0001F3FF|\U0001F6A3\U0001F3FB|\U0001F6A3\U0001F3FC|\U0001F6A3\U0001F3FD|\U0001F6A3\U0001F3FE|\U0001F6A3\U0001F3FF|\U0001F3CA\U0001F3FB|\U0001F3CA\U0001F3FC|\U0001F3CA\U0001F3FD|\U0001F3CA\U0001F3FE|\U0001F3CA\U0001F3FF|\U000026F9\U0000FE0F|\U000026F9\U0001F3FB|\U000026F9\U0001F3FC|\U000026F9\U0001F3FD|\U000026F9\U0001F3FE|\U000026F9\U0001F3FF|\U0001F3CB\U0000FE0F|\U0001F3CB\U0001F3FB|\U0001F3CB\U0001F3FC|\U0001F3CB\U0001F3FD|\U0001F3CB\U0001F3FE|\U0001F3CB\U0001F3FF|\U0001F6B4\U0001F3FB|\U0001F6B4\U0001F3FC|\U0001F6B4\U0001F3FD|\U0001F6B4\U0001F3FE|\U0001F6B4\U0001F3FF|\U0001F6B5\U0001F3FB|\U0001F6B5\U0001F3FC|\U0001F6B5\U0001F3FD|\U0001F6B5\U0001F3FE|\U0001F6B5\U0001F3FF|\U0001F938\U0001F3FB|\U0001F938\U0001F3FC|\U0001F938\U0001F3FD|\U0001F938\U0001F3FE|\U0001F938\U0001F3FF|\U0001F93D\U0001F3FB|\U0001F93D\U0001F3FC|\U0001F93D\U0001F3FD|\U0001F93D\U0001F3FE|\U0001F93D\U0001F3FF|\U0001F93E\U0001F3FB|\U0001F93E\U0001F3FC|\U0001F93E\U0001F3FD|\U0001F93E\U0001F3FE|\U0001F93E\U0001F3FF|\U0001F939\U0001F3FB|\U0001F939\U0001F3FC|\U0001F939\U0001F3FD|\U0001F939\U0001F3FE|\U0001F939\U0001F3FF|\U0001F9D8\U0001F3FB|\U0001F9D8\U0001F3FC|\U0001F9D8\U0001F3FD|\U0001F9D8\U0001F3FE|\U0001F9D8\U0001F3FF|\U0001F6C0\U0001F3FB|\U0001F6C0\U0001F3FC|\U0001F6C0\U0001F3FD|\U0001F6C0\U0001F3FE|\U0001F6C0\U0001F3FF|\U0001F6CC\U0001F3FB|\U0001F6CC\U0001F3FC|\U0001F6CC\U0001F3FD|\U0001F6CC\U0001F3FE|\U0001F6CC\U0001F3FF|\U0001F46D\U0001F3FB|\U0001F46D\U0001F3FC|\U0001F46D\U0001F3FD|\U0001F46D\U0001F3FE|\U0001F46D\U0001F3FF|\U0001F46B\U0001F3FB|\U0001F46B\U0001F3FC|\U0001F46B\U0001F3FD|\U0001F46B\U0001F3FE|\U0001F46B\U0001F3FF|\U0001F46C\U0001F3FB|\U0001F46C\U0001F3FC|\U0001F46C\U0001F3FD|\U0001F46C\U0001F3FE|\U0001F46C\U0001F3FF|\U0001F5E3\U0000FE0F|\U0001F43F\U0000FE0F|\U0001F54A\U0000FE0F|\U0001F577\U0000FE0F|\U0001F578\U0000FE0F|\U0001F3F5\U0000FE0F|\U00002618\U0000FE0F|\U0001F336\U0000FE0F|\U0001F37D\U0000FE0F|\U0001F5FA\U0000FE0F|\U0001F3D4\U0000FE0F|\U000026F0\U0000FE0F|\U0001F3D5\U0000FE0F|\U0001F3D6\U0000FE0F|\U0001F3DC\U0000FE0F|\U0001F3DD\U0000FE0F|\U0001F3DE\U0000FE0F|\U0001F3DF\U0000FE0F|\U0001F3DB\U0000FE0F|\U0001F3D7\U0000FE0F|\U0001F3D8\U0000FE0F|\U0001F3DA\U0000FE0F|\U000026E9\U0000FE0F|\U0001F3D9\U0000FE0F|\U00002668\U0000FE0F|\U0001F3CE\U0000FE0F|\U0001F3CD\U0000FE0F|\U0001F6E3\U0000FE0F|\U0001F6E4\U0000FE0F|\U0001F6E2\U0000FE0F|\U0001F6F3\U0000FE0F|\U000026F4\U0000FE0F|\U0001F6E5\U0000FE0F|\U00002708\U0000FE0F|\U0001F6E9\U0000FE0F|\U0001F6F0\U0000FE0F|\U0001F6CE\U0000FE0F|\U000023F1\U0000FE0F|\U000023F2\U0000FE0F|\U0001F570\U0000FE0F|\U0001F321\U0000FE0F|\U00002600\U0000FE0F|\U00002601\U0000FE0F|\U000026C8\U0000FE0F|\U0001F324\U0000FE0F|\U0001F325\U0000FE0F|\U0001F326\U0000FE0F|\U0001F327\U0000FE0F|\U0001F328\U0000FE0F|\U0001F329\U0000FE0F|\U0001F32A\U0000FE0F|\U0001F32B\U0000FE0F|\U0001F32C\U0000FE0F|\U00002602\U0000FE0F|\U000026F1\U0000FE0F|\U00002744\U0000FE0F|\U00002603\U0000FE0F|\U00002604\U0000FE0F|\U0001F397\U0000FE0F|\U0001F39F\U0000FE0F|\U0001F396\U0000FE0F|\U000026F8\U0000FE0F|\U0001F579\U0000FE0F|\U00002660\U0000FE0F|\U00002665\U0000FE0F|\U00002666\U0000FE0F|\U00002663\U0000FE0F|\U0000265F\U0000FE0F|\U0001F5BC\U0000FE0F|\U0001F576\U0000FE0F|\U0001F6CD\U0000FE0F|\U000026D1\U0000FE0F|\U0001F399\U0000FE0F|\U0001F39A\U0000FE0F|\U0001F39B\U0000FE0F|\U0000260E\U0000FE0F|\U0001F5A5\U0000FE0F|\U0001F5A8\U0000FE0F|\U00002328\U0000FE0F|\U0001F5B1\U0000FE0F|\U0001F5B2\U0000FE0F|\U0001F39E\U0000FE0F|\U0001F4FD\U0000FE0F|\U0001F56F\U0000FE0F|\U0001F5DE\U0000FE0F|\U0001F3F7\U0000FE0F|\U00002709\U0000FE0F|\U0001F5F3\U0000FE0F|\U0000270F\U0000FE0F|\U00002712\U0000FE0F|\U0001F58B\U0000FE0F|\U0001F58A\U0000FE0F|\U0001F58C\U0000FE0F|\U0001F58D\U0000FE0F|\U0001F5C2\U0000FE0F|\U0001F5D2\U0000FE0F|\U0001F5D3\U0000FE0F|\U0001F587\U0000FE0F|\U00002702\U0000FE0F|\U0001F5C3\U0000FE0F|\U0001F5C4\U0000FE0F|\U0001F5D1\U0000FE0F|\U0001F5DD\U0000FE0F|\U000026CF\U0000FE0F|\U00002692\U0000FE0F|\U0001F6E0\U0000FE0F|\U0001F5E1\U0000FE0F|\U00002694\U0000FE0F|\U0001F6E1\U0000FE0F|\U00002699\U0000FE0F|\U0001F5DC\U0000FE0F|\U00002696\U0000FE0F|\U000026D3\U0000FE0F|\U00002697\U0000FE0F|\U0001F6CF\U0000FE0F|\U0001F6CB\U0000FE0F|\U000026B0\U0000FE0F|\U000026B1\U0000FE0F|\U000026A0\U0000FE0F|\U00002622\U0000FE0F|\U00002623\U0000FE0F|\U00002B06\U0000FE0F|\U00002197\U0000FE0F|\U000027A1\U0000FE0F|\U00002198\U0000FE0F|\U00002B07\U0000FE0F|\U00002199\U0000FE0F|\U00002B05\U0000FE0F|\U00002196\U0000FE0F|\U00002195\U0000FE0F|\U00002194\U0000FE0F|\U000021A9\U0000FE0F|\U000021AA\U0000FE0F|\U00002934\U0000FE0F|\U00002935\U0000FE0F|\U0000269B\U0000FE0F|\U0001F549\U0000FE0F|\U00002721\U0000FE0F|\U00002638\U0000FE0F|\U0000262F\U0000FE0F|\U0000271D\U0000FE0F|\U00002626\U0000FE0F|\U0000262A\U0000FE0F|\U0000262E\U0000FE0F|\U000025B6\U0000FE0F|\U000023ED\U0000FE0F|\U000023EF\U0000FE0F|\U000025C0\U0000FE0F|\U000023EE\U0000FE0F|\U000023F8\U0000FE0F|\U000023F9\U0000FE0F|\U000023FA\U0000FE0F|\U000023CF\U0000FE0F|\U00002640\U0000FE0F|\U00002642\U0000FE0F|\U000026A7\U0000FE0F|\U00002716\U0000FE0F|\U0000267E\U0000FE0F|\U0000203C\U0000FE0F|\U00002049\U0000FE0F|\U00003030\U0000FE0F|\U00002695\U0000FE0F|\U0000267B\U0000FE0F|\U0000269C\U0000FE0F|\U00002611\U0000FE0F|\U00002714\U0000FE0F|\U0000303D\U0000FE0F|\U00002733\U0000FE0F|\U00002734\U0000FE0F|\U00002747\U0000FE0F|\U000000A9\U0000FE0F|\U000000AE\U0000FE0F|\U00002122\U0000FE0F|\U00000023\U000020E3|\U0000002A\U000020E3|\U00000030\U000020E3|\U00000031\U000020E3|\U00000032\U000020E3|\U00000033\U000020E3|\U00000034\U000020E3|\U00000035\U000020E3|\U00000036\U000020E3|\U00000037\U000020E3|\U00000038\U000020E3|\U00000039\U000020E3|\U0001F170\U0000FE0F|\U0001F171\U0000FE0F|\U00002139\U0000FE0F|\U000024C2\U0000FE0F|\U0001F17E\U0000FE0F|\U0001F17F\U0000FE0F|\U0001F202\U0000FE0F|\U0001F237\U0000FE0F|\U00003297\U0000FE0F|\U00003299\U0000FE0F|\U000025FC\U0000FE0F|\U000025FB\U0000FE0F|\U000025AA\U0000FE0F|\U000025AB\U0000FE0F|\U0001F3F3\U0000FE0F|\U0001F1E6\U0001F1E8|\U0001F1E6\U0001F1E9|\U0001F1E6\U0001F1EA|\U0001F1E6\U0001F1EB|\U0001F1E6\U0001F1EC|\U0001F1E6\U0001F1EE|\U0001F1E6\U0001F1F1|\U0001F1E6\U0001F1F2|\U0001F1E6\U0001F1F4|\U0001F1E6\U0001F1F6|\U0001F1E6\U0001F1F7|\U0001F1E6\U0001F1F8|\U0001F1E6\U0001F1F9|\U0001F1E6\U0001F1FA|\U0001F1E6\U0001F1FC|\U0001F1E6\U0001F1FD|\U0001F1E6\U0001F1FF|\U0001F1E7\U0001F1E6|\U0001F1E7\U0001F1E7|\U0001F1E7\U0001F1E9|\U0001F1E7\U0001F1EA|\U0001F1E7\U0001F1EB|\U0001F1E7\U0001F1EC|\U0001F1E7\U0001F1ED|\U0001F1E7\U0001F1EE|\U0001F1E7\U0001F1EF|\U0001F1E7\U0001F1F1|\U0001F1E7\U0001F1F2|\U0001F1E7\U0001F1F3|\U0001F1E7\U0001F1F4|\U0001F1E7\U0001F1F6|\U0001F1E7\U0001F1F7|\U0001F1E7\U0001F1F8|\U0001F1E7\U0001F1F9|\U0001F1E7\U0001F1FB|\U0001F1E7\U0001F1FC|\U0001F1E7\U0001F1FE|\U0001F1E7\U0001F1FF|\U0001F1E8\U0001F1E6|\U0001F1E8\U0001F1E8|\U0001F1E8\U0001F1E9|\U0001F1E8\U0001F1EB|\U0001F1E8\U0001F1EC|\U0001F1E8\U0001F1ED|\U0001F1E8\U0001F1EE|\U0001F1E8\U0001F1F0|\U0001F1E8\U0001F1F1|\U0001F1E8\U0001F1F2|\U0001F1E8\U0001F1F3|\U0001F1E8\U0001F1F4|\U0001F1E8\U0001F1F5|\U0001F1E8\U0001F1F7|\U0001F1E8\U0001F1FA|\U0001F1E8\U0001F1FB|\U0001F1E8\U0001F1FC|\U0001F1E8\U0001F1FD|\U0001F1E8\U0001F1FE|\U0001F1E8\U0001F1FF|\U0001F1E9\U0001F1EA|\U0001F1E9\U0001F1EC|\U0001F1E9\U0001F1EF|\U0001F1E9\U0001F1F0|\U0001F1E9\U0001F1F2|\U0001F1E9\U0001F1F4|\U0001F1E9\U0001F1FF|\U0001F1EA\U0001F1E6|\U0001F1EA\U0001F1E8|\U0001F1EA\U0001F1EA|\U0001F1EA\U0001F1EC|\U0001F1EA\U0001F1ED|\U0001F1EA\U0001F1F7|\U0001F1EA\U0001F1F8|\U0001F1EA\U0001F1F9|\U0001F1EA\U0001F1FA|\U0001F1EB\U0001F1EE|\U0001F1EB\U0001F1EF|\U0001F1EB\U0001F1F0|\U0001F1EB\U0001F1F2|\U0001F1EB\U0001F1F4|\U0001F1EB\U0001F1F7|\U0001F1EC\U0001F1E6|\U0001F1EC\U0001F1E7|\U0001F1EC\U0001F1E9|\U0001F1EC\U0001F1EA|\U0001F1EC\U0001F1EB|\U0001F1EC\U0001F1EC|\U0001F1EC\U0001F1ED|\U0001F1EC\U0001F1EE|\U0001F1EC\U0001F1F1|\U0001F1EC\U0001F1F2|\U0001F1EC\U0001F1F3|\U0001F1EC\U0001F1F5|\U0001F1EC\U0001F1F6|\U0001F1EC\U0001F1F7|\U0001F1EC\U0001F1F8|\U0001F1EC\U0001F1F9|\U0001F1EC\U0001F1FA|\U0001F1EC\U0001F1FC|\U0001F1EC\U0001F1FE|\U0001F1ED\U0001F1F0|\U0001F1ED\U0001F1F2|\U0001F1ED\U0001F1F3|\U0001F1ED\U0001F1F7|\U0001F1ED\U0001F1F9|\U0001F1ED\U0001F1FA|\U0001F1EE\U0001F1E8|\U0001F1EE\U0001F1E9|\U0001F1EE\U0001F1EA|\U0001F1EE\U0001F1F1|\U0001F1EE\U0001F1F2|\U0001F1EE\U0001F1F3|\U0001F1EE\U0001F1F4|\U0001F1EE\U0001F1F6|\U0001F1EE\U0001F1F7|\U0001F1EE\U0001F1F8|\U0001F1EE\U0001F1F9|\U0001F1EF\U0001F1EA|\U0001F1EF\U0001F1F2|\U0001F1EF\U0001F1F4|\U0001F1EF\U0001F1F5|\U0001F1F0\U0001F1EA|\U0001F1F0\U0001F1EC|\U0001F1F0\U0001F1ED|\U0001F1F0\U0001F1EE|\U0001F1F0\U0001F1F2|\U0001F1F0\U0001F1F3|\U0001F1F0\U0001F1F5|\U0001F1F0\U0001F1F7|\U0001F1F0\U0001F1FC|\U0001F1F0\U0001F1FE|\U0001F1F0\U0001F1FF|\U0001F1F1\U0001F1E6|\U0001F1F1\U0001F1E7|\U0001F1F1\U0001F1E8|\U0001F1F1\U0001F1EE|\U0001F1F1\U0001F1F0|\U0001F1F1\U0001F1F7|\U0001F1F1\U0001F1F8|\U0001F1F1\U0001F1F9|\U0001F1F1\U0001F1FA|\U0001F1F1\U0001F1FB|\U0001F1F1\U0001F1FE|\U0001F1F2\U0001F1E6|\U0001F1F2\U0001F1E8|\U0001F1F2\U0001F1E9|\U0001F1F2\U0001F1EA|\U0001F1F2\U0001F1EB|\U0001F1F2\U0001F1EC|\U0001F1F2\U0001F1ED|\U0001F1F2\U0001F1F0|\U0001F1F2\U0001F1F1|\U0001F1F2\U0001F1F2|\U0001F1F2\U0001F1F3|\U0001F1F2\U0001F1F4|\U0001F1F2\U0001F1F5|\U0001F1F2\U0001F1F6|\U0001F1F2\U0001F1F7|\U0001F1F2\U0001F1F8|\U0001F1F2\U0001F1F9|\U0001F1F2\U0001F1FA|\U0001F1F2\U0001F1FB|\U0001F1F2\U0001F1FC|\U0001F1F2\U0001F1FD|\U0001F1F2\U0001F1FE|\U0001F1F2\U0001F1FF|\U0001F1F3\U0001F1E6|\U0001F1F3\U0001F1E8|\U0001F1F3\U0001F1EA|\U0001F1F3\U0001F1EB|\U0001F1F3\U0001F1EC|\U0001F1F3\U0001F1EE|\U0001F1F3\U0001F1F1|\U0001F1F3\U0001F1F4|\U0001F1F3\U0001F1F5|\U0001F1F3\U0001F1F7|\U0001F1F3\U0001F1FA|\U0001F1F3\U0001F1FF|\U0001F1F4\U0001F1F2|\U0001F1F5\U0001F1E6|\U0001F1F5\U0001F1EA|\U0001F1F5\U0001F1EB|\U0001F1F5\U0001F1EC|\U0001F1F5\U0001F1ED|\U0001F1F5\U0001F1F0|\U0001F1F5\U0001F1F1|\U0001F1F5\U0001F1F2|\U0001F1F5\U0001F1F3|\U0001F1F5\U0001F1F7|\U0001F1F5\U0001F1F8|\U0001F1F5\U0001F1F9|\U0001F1F5\U0001F1FC|\U0001F1F5\U0001F1FE|\U0001F1F6\U0001F1E6|\U0001F1F7\U0001F1EA|\U0001F1F7\U0001F1F4|\U0001F1F7\U0001F1F8|\U0001F1F7\U0001F1FA|\U0001F1F7\U0001F1FC|\U0001F1F8\U0001F1E6|\U0001F1F8\U0001F1E7|\U0001F1F8\U0001F1E8|\U0001F1F8\U0001F1E9|\U0001F1F8\U0001F1EA|\U0001F1F8\U0001F1EC|\U0001F1F8\U0001F1ED|\U0001F1F8\U0001F1EE|\U0001F1F8\U0001F1EF|\U0001F1F8\U0001F1F0|\U0001F1F8\U0001F1F1|\U0001F1F8\U0001F1F2|\U0001F1F8\U0001F1F3|\U0001F1F8\U0001F1F4|\U0001F1F8\U0001F1F7|\U0001F1F8\U0001F1F8|\U0001F1F8\U0001F1F9|\U0001F1F8\U0001F1FB|\U0001F1F8\U0001F1FD|\U0001F1F8\U0001F1FE|\U0001F1F8\U0001F1FF|\U0001F1F9\U0001F1E6|\U0001F1F9\U0001F1E8|\U0001F1F9\U0001F1E9|\U0001F1F9\U0001F1EB|\U0001F1F9\U0001F1EC|\U0001F1F9\U0001F1ED|\U0001F1F9\U0001F1EF|\U0001F1F9\U0001F1F0|\U0001F1F9\U0001F1F1|\U0001F1F9\U0001F1F2|\U0001F1F9\U0001F1F3|\U0001F1F9\U0001F1F4|\U0001F1F9\U0001F1F7|\U0001F1F9\U0001F1F9|\U0001F1F9\U0001F1FB|\U0001F1F9\U0001F1FC|\U0001F1F9\U0001F1FF|\U0001F1FA\U0001F1E6|\U0001F1FA\U0001F1EC|\U0001F1FA\U0001F1F2|\U0001F1FA\U0001F1F3|\U0001F1FA\U0001F1F8|\U0001F1FA\U0001F1FE|\U0001F1FA\U0001F1FF|\U0001F1FB\U0001F1E6|\U0001F1FB\U0001F1E8|\U0001F1FB\U0001F1EA|\U0001F1FB\U0001F1EC|\U0001F1FB\U0001F1EE|\U0001F1FB\U0001F1F3|\U0001F1FB\U0001F1FA|\U0001F1FC\U0001F1EB|\U0001F1FC\U0001F1F8|\U0001F1FD\U0001F1F0|\U0001F1FE\U0001F1EA|\U0001F1FE\U0001F1F9|\U0001F1FF\U0001F1E6|\U0001F1FF\U0001F1F2|\U0001F1FF\U0001F1FC|[\U000000a9\U000000ae\U0000203c\U00002049\U00002122\U00002139\U00002194-\U00002199\U000021a9-\U000021aa\U0000231a-\U0000231b\U00002328\U000023cf\U000023e9-\U000023f3\U000023f8-\U000023fa\U000024c2\U000025aa-\U000025ab\U000025b6\U000025c0\U000025fb-\U000025fe\U00002600-\U00002604\U0000260e\U00002611\U00002614-\U00002615\U00002618\U0000261d\U00002620\U00002622-\U00002623\U00002626\U0000262a\U0000262e-\U0000262f\U00002638-\U0000263a\U00002640\U00002642\U00002648-\U00002653\U0000265f-\U00002660\U00002663\U00002665-\U00002666\U00002668\U0000267b\U0000267e-\U0000267f\U00002692-\U00002697\U00002699\U0000269b-\U0000269c\U000026a0-\U000026a1\U000026a7\U000026aa-\U000026ab\U000026b0-\U000026b1\U000026bd-\U000026be\U000026c4-\U000026c5\U000026c8\U000026ce-\U000026cf\U000026d1\U000026d3-\U000026d4\U000026e9-\U000026ea\U000026f0-\U000026f5\U000026f7-\U000026fa\U000026fd\U00002702\U00002705\U00002708-\U0000270d\U0000270f\U00002712\U00002714\U00002716\U0000271d\U00002721\U00002728\U00002733-\U00002734\U00002744\U00002747\U0000274c\U0000274e\U00002753-\U00002755\U00002757\U00002763-\U00002764\U00002795-\U00002797\U000027a1\U000027b0\U000027bf\U00002934-\U00002935\U00002b05-\U00002b07\U00002b1b-\U00002b1c\U00002b50\U00002b55\U00003030\U0000303d\U00003297\U00003299\U0001f004\U0001f0cf\U0001f170-\U0001f171\U0001f17e-\U0001f17f\U0001f18e\U0001f191-\U0001f19a\U0001f201-\U0001f202\U0001f21a\U0001f22f\U0001f232-\U0001f23a\U0001f250-\U0001f251\U0001f300-\U0001f321\U0001f324-\U0001f393\U0001f396-\U0001f397\U0001f399-\U0001f39b\U0001f39e-\U0001f3f0\U0001f3f3-\U0001f3f5\U0001f3f7-\U0001f3fa\U0001f400-\U0001f4fd\U0001f4ff-\U0001f53d\U0001f549-\U0001f54e\U0001f550-\U0001f567\U0001f56f-\U0001f570\U0001f573-\U0001f57a\U0001f587\U0001f58a-\U0001f58d\U0001f590\U0001f595-\U0001f596\U0001f5a4-\U0001f5a5\U0001f5a8\U0001f5b1-\U0001f5b2\U0001f5bc\U0001f5c2-\U0001f5c4\U0001f5d1-\U0001f5d3\U0001f5dc-\U0001f5de\U0001f5e1\U0001f5e3\U0001f5e8\U0001f5ef\U0001f5f3\U0001f5fa-\U0001f64f\U0001f680-\U0001f6c5\U0001f6cb-\U0001f6d2\U0001f6d5-\U0001f6d7\U0001f6e0-\U0001f6e5\U0001f6e9\U0001f6eb-\U0001f6ec\U0001f6f0\U0001f6f3-\U0001f6fc\U0001f7e0-\U0001f7eb\U0001f90c-\U0001f93a\U0001f93c-\U0001f945\U0001f947-\U0001f978\U0001f97a-\U0001f9af\U0001f9b4-\U0001f9cb\U0001f9cd-\U0001f9ff\U0001fa70-\U0001fa74\U0001fa78-\U0001fa7a\U0001fa80-\U0001fa86\U0001fa90-\U0001faa8\U0001fab0-\U0001fab6\U0001fac0-\U0001fac2\U0001fad0-\U0001fad6]'
EMOJI_RAW =   r'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF|\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468|\U0001F468\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468|\U0001F469\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469|\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FE|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FF|\U0001F9D1\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FF|\U0001F469\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468|\U0001F468\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468|\U0001F469\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469|\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FE|\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FF|\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FB|\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FC|\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FD|\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FE|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF|\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB|\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC|\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD|\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE|\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FF|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FB|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FC|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FD|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FE|\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FF|\U0001F468\U0000200D\U0001F469\U0000200D\U0001F467\U0000200D\U0001F466|\U0001F468\U0000200D\U0001F469\U0000200D\U0001F466\U0000200D\U0001F466|\U0001F468\U0000200D\U0001F469\U0000200D\U0001F467\U0000200D\U0001F467|\U0001F468\U0000200D\U0001F468\U0000200D\U0001F467\U0000200D\U0001F466|\U0001F468\U0000200D\U0001F468\U0000200D\U0001F466\U0000200D\U0001F466|\U0001F468\U0000200D\U0001F468\U0000200D\U0001F467\U0000200D\U0001F467|\U0001F469\U0000200D\U0001F469\U0000200D\U0001F467\U0000200D\U0001F466|\U0001F469\U0000200D\U0001F469\U0000200D\U0001F466\U0000200D\U0001F466|\U0001F469\U0000200D\U0001F469\U0000200D\U0001F467\U0000200D\U0001F467|\U0001F3F4\U000E0067\U000E0062\U000E0065\U000E006E\U000E0067\U000E007F|\U0001F3F4\U000E0067\U000E0062\U000E0073\U000E0063\U000E0074\U000E007F|\U0001F3F4\U000E0067\U000E0062\U000E0077\U000E006C\U000E0073\U000E007F|\U0001F469\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468|\U0001F468\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468|\U0001F469\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469|\U0001F441\U0000FE0F\U0000200D\U0001F5E8\U0000FE0F|\U0001F9D4\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9D4\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9D4\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9D4\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9D4\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9D4\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9D4\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9D4\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9D4\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9D4\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F471\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F471\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F471\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F471\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F471\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F471\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F471\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F471\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F471\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F471\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F64D\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F64D\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F64D\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F64D\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F64D\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F64D\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F64D\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F64D\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F64D\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F64D\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F64E\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F64E\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F64E\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F64E\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F64E\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F64E\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F64E\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F64E\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F64E\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F64E\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F645\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F645\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F645\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F645\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F645\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F645\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F645\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F645\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F645\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F645\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F646\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F646\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F646\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F646\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F646\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F646\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F646\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F646\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F646\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F646\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F481\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F481\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F481\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F481\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F481\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F481\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F481\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F481\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F481\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F481\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F64B\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F64B\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F64B\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F64B\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F64B\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F64B\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F64B\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F64B\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F64B\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F64B\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9CF\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9CF\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9CF\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9CF\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9CF\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9CF\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9CF\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9CF\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9CF\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9CF\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F647\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F647\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F647\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F647\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F647\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F647\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F647\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F647\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F647\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F647\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F926\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F926\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F926\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F926\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F926\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F926\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F926\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F926\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F926\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F926\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F937\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F937\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F937\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F937\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F937\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F937\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F937\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F937\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F937\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F937\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9D1\U0001F3FB\U0000200D\U00002695\U0000FE0F|\U0001F9D1\U0001F3FC\U0000200D\U00002695\U0000FE0F|\U0001F9D1\U0001F3FD\U0000200D\U00002695\U0000FE0F|\U0001F9D1\U0001F3FE\U0000200D\U00002695\U0000FE0F|\U0001F9D1\U0001F3FF\U0000200D\U00002695\U0000FE0F|\U0001F468\U0001F3FB\U0000200D\U00002695\U0000FE0F|\U0001F468\U0001F3FC\U0000200D\U00002695\U0000FE0F|\U0001F468\U0001F3FD\U0000200D\U00002695\U0000FE0F|\U0001F468\U0001F3FE\U0000200D\U00002695\U0000FE0F|\U0001F468\U0001F3FF\U0000200D\U00002695\U0000FE0F|\U0001F469\U0001F3FB\U0000200D\U00002695\U0000FE0F|\U0001F469\U0001F3FC\U0000200D\U00002695\U0000FE0F|\U0001F469\U0001F3FD\U0000200D\U00002695\U0000FE0F|\U0001F469\U0001F3FE\U0000200D\U00002695\U0000FE0F|\U0001F469\U0001F3FF\U0000200D\U00002695\U0000FE0F|\U0001F9D1\U0001F3FB\U0000200D\U00002696\U0000FE0F|\U0001F9D1\U0001F3FC\U0000200D\U00002696\U0000FE0F|\U0001F9D1\U0001F3FD\U0000200D\U00002696\U0000FE0F|\U0001F9D1\U0001F3FE\U0000200D\U00002696\U0000FE0F|\U0001F9D1\U0001F3FF\U0000200D\U00002696\U0000FE0F|\U0001F468\U0001F3FB\U0000200D\U00002696\U0000FE0F|\U0001F468\U0001F3FC\U0000200D\U00002696\U0000FE0F|\U0001F468\U0001F3FD\U0000200D\U00002696\U0000FE0F|\U0001F468\U0001F3FE\U0000200D\U00002696\U0000FE0F|\U0001F468\U0001F3FF\U0000200D\U00002696\U0000FE0F|\U0001F469\U0001F3FB\U0000200D\U00002696\U0000FE0F|\U0001F469\U0001F3FC\U0000200D\U00002696\U0000FE0F|\U0001F469\U0001F3FD\U0000200D\U00002696\U0000FE0F|\U0001F469\U0001F3FE\U0000200D\U00002696\U0000FE0F|\U0001F469\U0001F3FF\U0000200D\U00002696\U0000FE0F|\U0001F9D1\U0001F3FB\U0000200D\U00002708\U0000FE0F|\U0001F9D1\U0001F3FC\U0000200D\U00002708\U0000FE0F|\U0001F9D1\U0001F3FD\U0000200D\U00002708\U0000FE0F|\U0001F9D1\U0001F3FE\U0000200D\U00002708\U0000FE0F|\U0001F9D1\U0001F3FF\U0000200D\U00002708\U0000FE0F|\U0001F468\U0001F3FB\U0000200D\U00002708\U0000FE0F|\U0001F468\U0001F3FC\U0000200D\U00002708\U0000FE0F|\U0001F468\U0001F3FD\U0000200D\U00002708\U0000FE0F|\U0001F468\U0001F3FE\U0000200D\U00002708\U0000FE0F|\U0001F468\U0001F3FF\U0000200D\U00002708\U0000FE0F|\U0001F469\U0001F3FB\U0000200D\U00002708\U0000FE0F|\U0001F469\U0001F3FC\U0000200D\U00002708\U0000FE0F|\U0001F469\U0001F3FD\U0000200D\U00002708\U0000FE0F|\U0001F469\U0001F3FE\U0000200D\U00002708\U0000FE0F|\U0001F469\U0001F3FF\U0000200D\U00002708\U0000FE0F|\U0001F46E\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F46E\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F46E\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F46E\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F46E\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F46E\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F46E\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F46E\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F46E\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F46E\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F575\U0000FE0F\U0000200D\U00002642\U0000FE0F|\U0001F575\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F575\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F575\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F575\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F575\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F575\U0000FE0F\U0000200D\U00002640\U0000FE0F|\U0001F575\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F575\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F575\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F575\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F575\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F482\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F482\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F482\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F482\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F482\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F482\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F482\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F482\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F482\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F482\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F477\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F477\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F477\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F477\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F477\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F477\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F477\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F477\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F477\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F477\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F473\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F473\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F473\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F473\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F473\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F473\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F473\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F473\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F473\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F473\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F935\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F935\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F935\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F935\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F935\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F935\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F935\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F935\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F935\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F935\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F470\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F470\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F470\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F470\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F470\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F470\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F470\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F470\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F470\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F470\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9B8\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9B8\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9B8\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9B8\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9B8\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9B8\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9B8\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9B8\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9B8\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9B8\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9B9\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9B9\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9B9\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9B9\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9B9\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9B9\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9B9\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9B9\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9B9\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9B9\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9D9\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9D9\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9D9\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9D9\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9D9\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9D9\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9D9\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9D9\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9D9\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9D9\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9DA\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9DA\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9DA\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9DA\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9DA\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9DA\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9DA\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9DA\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9DA\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9DA\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9DB\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9DB\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9DB\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9DB\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9DB\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9DB\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9DB\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9DB\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9DB\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9DB\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9DC\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9DC\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9DC\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9DC\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9DC\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9DC\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9DC\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9DC\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9DC\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9DC\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9DD\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9DD\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9DD\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9DD\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9DD\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9DD\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9DD\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9DD\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9DD\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9DD\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F486\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F486\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F486\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F486\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F486\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F486\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F486\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F486\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F486\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F486\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F487\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F487\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F487\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F487\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F487\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F487\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F487\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F487\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F487\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F487\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F6B6\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F6B6\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F6B6\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F6B6\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F6B6\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F6B6\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F6B6\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F6B6\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F6B6\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F6B6\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9CD\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9CD\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9CD\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9CD\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9CD\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9CD\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9CD\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9CD\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9CD\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9CD\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9CE\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9CE\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9CE\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9CE\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9CE\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9CE\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9CE\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9CE\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9CE\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9CE\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F3C3\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F3C3\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F3C3\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F3C3\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F3C3\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F3C3\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F3C3\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F3C3\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F3C3\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F3C3\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9D6\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9D6\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9D6\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9D6\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9D6\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9D6\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9D6\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9D6\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9D6\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9D6\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9D7\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9D7\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9D7\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9D7\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9D7\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9D7\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9D7\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9D7\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9D7\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9D7\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F3CC\U0000FE0F\U0000200D\U00002642\U0000FE0F|\U0001F3CC\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F3CC\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F3CC\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F3CC\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F3CC\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F3CC\U0000FE0F\U0000200D\U00002640\U0000FE0F|\U0001F3CC\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F3CC\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F3CC\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F3CC\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F3CC\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F3C4\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F3C4\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F3C4\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F3C4\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F3C4\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F3C4\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F3C4\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F3C4\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F3C4\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F3C4\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F6A3\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F6A3\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F6A3\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F6A3\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F6A3\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F6A3\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F6A3\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F6A3\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F6A3\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F6A3\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F3CA\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F3CA\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F3CA\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F3CA\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F3CA\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F3CA\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F3CA\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F3CA\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F3CA\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F3CA\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U000026F9\U0000FE0F\U0000200D\U00002642\U0000FE0F|\U000026F9\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U000026F9\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U000026F9\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U000026F9\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U000026F9\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U000026F9\U0000FE0F\U0000200D\U00002640\U0000FE0F|\U000026F9\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U000026F9\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U000026F9\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U000026F9\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U000026F9\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F3CB\U0000FE0F\U0000200D\U00002642\U0000FE0F|\U0001F3CB\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F3CB\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F3CB\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F3CB\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F3CB\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F3CB\U0000FE0F\U0000200D\U00002640\U0000FE0F|\U0001F3CB\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F3CB\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F3CB\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F3CB\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F3CB\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F6B4\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F6B4\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F6B4\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F6B4\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F6B4\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F6B4\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F6B4\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F6B4\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F6B4\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F6B4\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F6B5\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F6B5\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F6B5\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F6B5\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F6B5\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F6B5\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F6B5\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F6B5\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F6B5\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F6B5\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F938\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F938\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F938\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F938\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F938\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F938\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F938\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F938\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F938\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F938\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F93D\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F93D\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F93D\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F93D\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F93D\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F93D\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F93D\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F93D\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F93D\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F93D\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F93E\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F93E\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F93E\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F93E\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F93E\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F93E\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F93E\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F93E\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F93E\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F93E\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F939\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F939\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F939\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F939\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F939\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F939\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F939\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F939\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F939\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F939\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9D8\U0001F3FB\U0000200D\U00002642\U0000FE0F|\U0001F9D8\U0001F3FC\U0000200D\U00002642\U0000FE0F|\U0001F9D8\U0001F3FD\U0000200D\U00002642\U0000FE0F|\U0001F9D8\U0001F3FE\U0000200D\U00002642\U0000FE0F|\U0001F9D8\U0001F3FF\U0000200D\U00002642\U0000FE0F|\U0001F9D8\U0001F3FB\U0000200D\U00002640\U0000FE0F|\U0001F9D8\U0001F3FC\U0000200D\U00002640\U0000FE0F|\U0001F9D8\U0001F3FD\U0000200D\U00002640\U0000FE0F|\U0001F9D8\U0001F3FE\U0000200D\U00002640\U0000FE0F|\U0001F9D8\U0001F3FF\U0000200D\U00002640\U0000FE0F|\U0001F9D1\U0000200D\U0001F91D\U0000200D\U0001F9D1|\U0001F469\U0000200D\U00002764\U0000200D\U0001F468|\U0001F468\U0000200D\U00002764\U0000200D\U0001F468|\U0001F469\U0000200D\U00002764\U0000200D\U0001F469|\U0001F468\U0000200D\U0001F469\U0000200D\U0001F466|\U0001F468\U0000200D\U0001F469\U0000200D\U0001F467|\U0001F468\U0000200D\U0001F468\U0000200D\U0001F466|\U0001F468\U0000200D\U0001F468\U0000200D\U0001F467|\U0001F469\U0000200D\U0001F469\U0000200D\U0001F466|\U0001F469\U0000200D\U0001F469\U0000200D\U0001F467|\U0001F468\U0000200D\U0001F466\U0000200D\U0001F466|\U0001F468\U0000200D\U0001F467\U0000200D\U0001F466|\U0001F468\U0000200D\U0001F467\U0000200D\U0001F467|\U0001F469\U0000200D\U0001F466\U0000200D\U0001F466|\U0001F469\U0000200D\U0001F467\U0000200D\U0001F466|\U0001F469\U0000200D\U0001F467\U0000200D\U0001F467|\U0001F3F3\U0000FE0F\U0000200D\U000026A7\U0000FE0F|\U0001F636\U0000200D\U0001F32B\U0000FE0F|\U00002764\U0000FE0F\U0000200D\U0001F525|\U00002764\U0000FE0F\U0000200D\U0001FA79|\U0001F441\U0000200D\U0001F5E8\U0000FE0F|\U0001F441\U0000FE0F\U0000200D\U0001F5E8|\U0001F9D4\U0000200D\U00002642\U0000FE0F|\U0001F9D4\U0001F3FB\U0000200D\U00002642|\U0001F9D4\U0001F3FC\U0000200D\U00002642|\U0001F9D4\U0001F3FD\U0000200D\U00002642|\U0001F9D4\U0001F3FE\U0000200D\U00002642|\U0001F9D4\U0001F3FF\U0000200D\U00002642|\U0001F9D4\U0000200D\U00002640\U0000FE0F|\U0001F9D4\U0001F3FB\U0000200D\U00002640|\U0001F9D4\U0001F3FC\U0000200D\U00002640|\U0001F9D4\U0001F3FD\U0000200D\U00002640|\U0001F9D4\U0001F3FE\U0000200D\U00002640|\U0001F9D4\U0001F3FF\U0000200D\U00002640|\U0001F468\U0001F3FB\U0000200D\U0001F9B0|\U0001F468\U0001F3FC\U0000200D\U0001F9B0|\U0001F468\U0001F3FD\U0000200D\U0001F9B0|\U0001F468\U0001F3FE\U0000200D\U0001F9B0|\U0001F468\U0001F3FF\U0000200D\U0001F9B0|\U0001F468\U0001F3FB\U0000200D\U0001F9B1|\U0001F468\U0001F3FC\U0000200D\U0001F9B1|\U0001F468\U0001F3FD\U0000200D\U0001F9B1|\U0001F468\U0001F3FE\U0000200D\U0001F9B1|\U0001F468\U0001F3FF\U0000200D\U0001F9B1|\U0001F468\U0001F3FB\U0000200D\U0001F9B3|\U0001F468\U0001F3FC\U0000200D\U0001F9B3|\U0001F468\U0001F3FD\U0000200D\U0001F9B3|\U0001F468\U0001F3FE\U0000200D\U0001F9B3|\U0001F468\U0001F3FF\U0000200D\U0001F9B3|\U0001F468\U0001F3FB\U0000200D\U0001F9B2|\U0001F468\U0001F3FC\U0000200D\U0001F9B2|\U0001F468\U0001F3FD\U0000200D\U0001F9B2|\U0001F468\U0001F3FE\U0000200D\U0001F9B2|\U0001F468\U0001F3FF\U0000200D\U0001F9B2|\U0001F469\U0001F3FB\U0000200D\U0001F9B0|\U0001F469\U0001F3FC\U0000200D\U0001F9B0|\U0001F469\U0001F3FD\U0000200D\U0001F9B0|\U0001F469\U0001F3FE\U0000200D\U0001F9B0|\U0001F469\U0001F3FF\U0000200D\U0001F9B0|\U0001F9D1\U0001F3FB\U0000200D\U0001F9B0|\U0001F9D1\U0001F3FC\U0000200D\U0001F9B0|\U0001F9D1\U0001F3FD\U0000200D\U0001F9B0|\U0001F9D1\U0001F3FE\U0000200D\U0001F9B0|\U0001F9D1\U0001F3FF\U0000200D\U0001F9B0|\U0001F469\U0001F3FB\U0000200D\U0001F9B1|\U0001F469\U0001F3FC\U0000200D\U0001F9B1|\U0001F469\U0001F3FD\U0000200D\U0001F9B1|\U0001F469\U0001F3FE\U0000200D\U0001F9B1|\U0001F469\U0001F3FF\U0000200D\U0001F9B1|\U0001F9D1\U0001F3FB\U0000200D\U0001F9B1|\U0001F9D1\U0001F3FC\U0000200D\U0001F9B1|\U0001F9D1\U0001F3FD\U0000200D\U0001F9B1|\U0001F9D1\U0001F3FE\U0000200D\U0001F9B1|\U0001F9D1\U0001F3FF\U0000200D\U0001F9B1|\U0001F469\U0001F3FB\U0000200D\U0001F9B3|\U0001F469\U0001F3FC\U0000200D\U0001F9B3|\U0001F469\U0001F3FD\U0000200D\U0001F9B3|\U0001F469\U0001F3FE\U0000200D\U0001F9B3|\U0001F469\U0001F3FF\U0000200D\U0001F9B3|\U0001F9D1\U0001F3FB\U0000200D\U0001F9B3|\U0001F9D1\U0001F3FC\U0000200D\U0001F9B3|\U0001F9D1\U0001F3FD\U0000200D\U0001F9B3|\U0001F9D1\U0001F3FE\U0000200D\U0001F9B3|\U0001F9D1\U0001F3FF\U0000200D\U0001F9B3|\U0001F469\U0001F3FB\U0000200D\U0001F9B2|\U0001F469\U0001F3FC\U0000200D\U0001F9B2|\U0001F469\U0001F3FD\U0000200D\U0001F9B2|\U0001F469\U0001F3FE\U0000200D\U0001F9B2|\U0001F469\U0001F3FF\U0000200D\U0001F9B2|\U0001F9D1\U0001F3FB\U0000200D\U0001F9B2|\U0001F9D1\U0001F3FC\U0000200D\U0001F9B2|\U0001F9D1\U0001F3FD\U0000200D\U0001F9B2|\U0001F9D1\U0001F3FE\U0000200D\U0001F9B2|\U0001F9D1\U0001F3FF\U0000200D\U0001F9B2|\U0001F471\U0000200D\U00002640\U0000FE0F|\U0001F471\U0001F3FB\U0000200D\U00002640|\U0001F471\U0001F3FC\U0000200D\U00002640|\U0001F471\U0001F3FD\U0000200D\U00002640|\U0001F471\U0001F3FE\U0000200D\U00002640|\U0001F471\U0001F3FF\U0000200D\U00002640|\U0001F471\U0000200D\U00002642\U0000FE0F|\U0001F471\U0001F3FB\U0000200D\U00002642|\U0001F471\U0001F3FC\U0000200D\U00002642|\U0001F471\U0001F3FD\U0000200D\U00002642|\U0001F471\U0001F3FE\U0000200D\U00002642|\U0001F471\U0001F3FF\U0000200D\U00002642|\U0001F64D\U0000200D\U00002642\U0000FE0F|\U0001F64D\U0001F3FB\U0000200D\U00002642|\U0001F64D\U0001F3FC\U0000200D\U00002642|\U0001F64D\U0001F3FD\U0000200D\U00002642|\U0001F64D\U0001F3FE\U0000200D\U00002642|\U0001F64D\U0001F3FF\U0000200D\U00002642|\U0001F64D\U0000200D\U00002640\U0000FE0F|\U0001F64D\U0001F3FB\U0000200D\U00002640|\U0001F64D\U0001F3FC\U0000200D\U00002640|\U0001F64D\U0001F3FD\U0000200D\U00002640|\U0001F64D\U0001F3FE\U0000200D\U00002640|\U0001F64D\U0001F3FF\U0000200D\U00002640|\U0001F64E\U0000200D\U00002642\U0000FE0F|\U0001F64E\U0001F3FB\U0000200D\U00002642|\U0001F64E\U0001F3FC\U0000200D\U00002642|\U0001F64E\U0001F3FD\U0000200D\U00002642|\U0001F64E\U0001F3FE\U0000200D\U00002642|\U0001F64E\U0001F3FF\U0000200D\U00002642|\U0001F64E\U0000200D\U00002640\U0000FE0F|\U0001F64E\U0001F3FB\U0000200D\U00002640|\U0001F64E\U0001F3FC\U0000200D\U00002640|\U0001F64E\U0001F3FD\U0000200D\U00002640|\U0001F64E\U0001F3FE\U0000200D\U00002640|\U0001F64E\U0001F3FF\U0000200D\U00002640|\U0001F645\U0000200D\U00002642\U0000FE0F|\U0001F645\U0001F3FB\U0000200D\U00002642|\U0001F645\U0001F3FC\U0000200D\U00002642|\U0001F645\U0001F3FD\U0000200D\U00002642|\U0001F645\U0001F3FE\U0000200D\U00002642|\U0001F645\U0001F3FF\U0000200D\U00002642|\U0001F645\U0000200D\U00002640\U0000FE0F|\U0001F645\U0001F3FB\U0000200D\U00002640|\U0001F645\U0001F3FC\U0000200D\U00002640|\U0001F645\U0001F3FD\U0000200D\U00002640|\U0001F645\U0001F3FE\U0000200D\U00002640|\U0001F645\U0001F3FF\U0000200D\U00002640|\U0001F646\U0000200D\U00002642\U0000FE0F|\U0001F646\U0001F3FB\U0000200D\U00002642|\U0001F646\U0001F3FC\U0000200D\U00002642|\U0001F646\U0001F3FD\U0000200D\U00002642|\U0001F646\U0001F3FE\U0000200D\U00002642|\U0001F646\U0001F3FF\U0000200D\U00002642|\U0001F646\U0000200D\U00002640\U0000FE0F|\U0001F646\U0001F3FB\U0000200D\U00002640|\U0001F646\U0001F3FC\U0000200D\U00002640|\U0001F646\U0001F3FD\U0000200D\U00002640|\U0