"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaToS3 = void 0;
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
/**
 * @summary The LambdaToS3 class.
 */
class LambdaToS3 extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the LambdaToS3 class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {LambdaToS3Props} props - user provided props for the construct.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the Lambda function
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        // Setup the S3 bucket
        this.s3Bucket = defaults.buildS3Bucket(this, {
            existingBucketObj: props.existingBucketObj,
            bucketProps: props.bucketProps
        });
        // Configure environment variables
        this.lambdaFunction.addEnvironment('S3_BUCKET_NAME', this.s3Bucket.bucketName);
        // Add the requested or default bucket permissions
        if (props.bucketPermissions) {
            if (props.bucketPermissions.includes('Delete')) {
                this.s3Bucket.grantDelete(this.lambdaFunction.grantPrincipal);
            }
            if (props.bucketPermissions.includes('Put')) {
                this.s3Bucket.grantPut(this.lambdaFunction.grantPrincipal);
            }
            if (props.bucketPermissions.includes('Read')) {
                this.s3Bucket.grantRead(this.lambdaFunction.grantPrincipal);
            }
            if (props.bucketPermissions.includes('ReadWrite')) {
                this.s3Bucket.grantReadWrite(this.lambdaFunction.grantPrincipal);
            }
            if (props.bucketPermissions.includes('Write')) {
                this.s3Bucket.grantWrite(this.lambdaFunction.grantPrincipal);
            }
        }
        else {
            this.s3Bucket.grantReadWrite(this.lambdaFunction.grantPrincipal);
        }
        // Add appropriate metadata
        const s3BucketResource = this.s3Bucket.node.findChild('Resource');
        s3BucketResource.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W51',
                        reason: `This S3 bucket Bucket does not need a bucket policy`
                    }]
            }
        };
    }
}
exports.LambdaToS3 = LambdaToS3;
//# sourceMappingURL=data:application/json;base64,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