"use strict";
const assert_1 = require("@aws-cdk/assert");
const dynamodb = require("@aws-cdk/aws-dynamodb");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const sources = require("../lib");
const test_function_1 = require("./test-function");
module.exports = {
    'sufficiently complex example'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // WHEN
        fn.addEventSource(new sources.DynamoEventSource(table, {
            startingPosition: lambda.StartingPosition.TRIM_HORIZON
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": "dynamodb:ListStreams",
                        "Effect": "Allow",
                        "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["TD925BC7E", "Arn"] }, "/stream/*"]] }
                    },
                    {
                        "Action": [
                            "dynamodb:DescribeStream",
                            "dynamodb:GetRecords",
                            "dynamodb:GetShardIterator",
                        ],
                        "Effect": "Allow",
                        "Resource": {
                            "Fn::GetAtt": [
                                "TD925BC7E",
                                "StreamArn"
                            ]
                        }
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "FnServiceRoleDefaultPolicyC6A839BF",
            "Roles": [{
                    "Ref": "FnServiceRoleB9001A96"
                }]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            "EventSourceArn": {
                "Fn::GetAtt": [
                    "TD925BC7E",
                    "StreamArn"
                ]
            },
            "FunctionName": {
                "Ref": "Fn9270CBC0"
            },
            "BatchSize": 100,
            "StartingPosition": "TRIM_HORIZON"
        }));
        test.done();
    },
    'specific batch size'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // WHEN
        fn.addEventSource(new sources.DynamoEventSource(table, {
            batchSize: 50,
            startingPosition: lambda.StartingPosition.LATEST
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            "EventSourceArn": {
                "Fn::GetAtt": [
                    "TD925BC7E",
                    "StreamArn"
                ]
            },
            "FunctionName": {
                "Ref": "Fn9270CBC0"
            },
            "BatchSize": 50,
            "StartingPosition": "LATEST"
        }));
        test.done();
    },
    'fails if streaming not enabled on table'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            }
        });
        // WHEN
        test.throws(() => fn.addEventSource(new sources.DynamoEventSource(table, {
            batchSize: 50,
            startingPosition: lambda.StartingPosition.LATEST
        })), /DynamoDB Streams must be enabled on the table T/);
        test.done();
    },
    'fails if batch size < 1'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // WHEN
        test.throws(() => fn.addEventSource(new sources.DynamoEventSource(table, {
            batchSize: 0,
            startingPosition: lambda.StartingPosition.LATEST
        })), /Maximum batch size must be between 1 and 1000 inclusive \(given 0\)/);
        test.done();
    },
    'fails if batch size > 1000'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // WHEN
        test.throws(() => fn.addEventSource(new sources.DynamoEventSource(table, {
            batchSize: 1001,
            startingPosition: lambda.StartingPosition.LATEST
        })), /Maximum batch size must be between 1 and 1000 inclusive \(given 1001\)/);
        test.done();
    },
    'specific maxBatchingWindow'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // WHEN
        fn.addEventSource(new sources.DynamoEventSource(table, {
            maxBatchingWindow: cdk.Duration.minutes(2),
            startingPosition: lambda.StartingPosition.LATEST
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            "EventSourceArn": {
                "Fn::GetAtt": [
                    "TD925BC7E",
                    "StreamArn"
                ]
            },
            "FunctionName": {
                "Ref": "Fn9270CBC0"
            },
            "MaximumBatchingWindowInSeconds": 120,
            "StartingPosition": "LATEST"
        }));
        test.done();
    },
    'throws if maxBatchingWindow > 300 seconds'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // THEN
        test.throws(() => fn.addEventSource(new sources.DynamoEventSource(table, {
            maxBatchingWindow: cdk.Duration.seconds(301),
            startingPosition: lambda.StartingPosition.LATEST
        })), /maxBatchingWindow cannot be over 300 seconds/);
        test.done();
    },
    'contains eventSourceMappingId after lambda binding'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        const eventSource = new sources.DynamoEventSource(table, {
            startingPosition: lambda.StartingPosition.TRIM_HORIZON
        });
        // WHEN
        fn.addEventSource(eventSource);
        // THEN
        test.ok(eventSource.eventSourceMappingId);
        test.done();
    },
    'eventSourceMappingId throws error before binding to lambda'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        const eventSource = new sources.DynamoEventSource(table, {
            startingPosition: lambda.StartingPosition.TRIM_HORIZON
        });
        // WHEN/THEN
        test.throws(() => eventSource.eventSourceMappingId, /DynamoEventSource is not yet bound to an event source mapping/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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