"""
    Plotting and saving Alpha Beta results
"""

import pickle
import pathlib
import logging
import copy
import datetime
import shapefile
import numpy as np
import matplotlib.pyplot as plt
import matplotlib as mpl
from mpl_toolkits.axes_grid1 import make_axes_locatable

# Local imports
import avaframe.out3Plot.plotUtils as pU

# create local logger
log = logging.getLogger(__name__)

colors = ["#393955", "#8A8A9B", "#E9E940"]
mpl.rcParams['axes.prop_cycle'] = mpl.cycler(color=colors)


def writeABtoSHP(resAB):
    """ Write com2AB results to shapefile

    Parameters
    ----------
    resAB : dict
        dict with com2AB results

    Returns
    -------
    saveOutFile: str
        path to shapefile
    """

    saveOutFile = pathlib.Path(resAB['saveOutPath']) / 'com2AB_Results'
    avaPath = resAB['AvaPath']

    # write projection information file
    with open(str(saveOutFile)+'.prj', 'w') as prjf:
        prjf.write(avaPath['sks'])

    # open shapefile writer with point shapetype
    w = shapefile.Writer(str(saveOutFile), shapeType=1)

    # set fields
    w.field('fid', 'N')
    w.field('Name', 'C', '60')
    w.field('Angle', 'F', 5, 1)
    # w.field('Distance', 'F', '40')

    # loop through the profiles
    for i, profile in enumerate(avaPath['Name']):

        resAB = processABresults(resAB, profile)
        cuProf = resAB[profile]

        pointName = profile + '_Beta'
        w.point(cuProf['x'][cuProf['ids10Point']], cuProf['y'][cuProf['ids10Point']])
        w.record(i, pointName, cuProf['beta'])

        pointName = profile + '_Alpha'
        w.point(cuProf['x'][cuProf['ids_alpha']], cuProf['y'][cuProf['ids_alpha']])
        w.record(i, pointName, cuProf['alpha'])

        pointName = profile + '_AlphaPlus1SD'
        w.point(cuProf['x'][cuProf['ids_alphaP1SD']], cuProf['y'][cuProf['ids_alphaP1SD']])
        w.record(i, pointName, cuProf['alphaSD'][0])

        pointName = profile + '_AlphaMinus1SD'
        w.point(cuProf['x'][cuProf['ids_alphaM1SD']], cuProf['y'][cuProf['ids_alphaM1SD']])
        w.record(i, pointName, cuProf['alphaSD'][1])

        pointName = profile + '_AlphaMinus2SD'
        w.point(cuProf['x'][cuProf['ids_alphaM2SD']], cuProf['y'][cuProf['ids_alphaM2SD']])
        w.record(i, pointName, cuProf['alphaSD'][2])

    w.close()

    log.info('Writing com2AB to shapefile: %s.shp', saveOutFile)

    return saveOutFile


def readABresults(saveOutPath, name, flags):
    """ read the results generated by com2AB """
    savename = name + '_com2AB_eqparam.pickle'
    save_file = pathlib.Path(saveOutPath, savename)
    with open(save_file, 'rb') as handle:
        eqParams = pickle.load(handle)
    if not flags.getboolean('fullOut'):
        save_file.unlink()
    savename = name + '_com2AB_eqout.pickle'
    save_file = pathlib.Path(saveOutPath, savename)
    with open(save_file, 'rb') as handle:
        eqOut = pickle.load(handle)
    if not flags.getboolean('fullOut'):
        save_file.unlink()

    return eqParams, eqOut


def processABresults(resAB, name):
    """ prepare AlphaBeta results for plotting and writing results """
    s = resAB[name]['s']
    z = resAB[name]['z']
    CuSplit = resAB[name]['CuSplit']
    alpha = resAB[name]['alpha']
    alphaSD = resAB[name]['alphaSD']

    # Line down to alpha
    f = z[0] + np.tan(np.deg2rad(-alpha)) * s
    fplus1SD = z[0] + np.tan(np.deg2rad(-alphaSD[0])) * s
    fminus1SD = z[0] + np.tan(np.deg2rad(-alphaSD[1])) * s
    fminus2SD = z[0] + np.tan(np.deg2rad(-alphaSD[2])) * s

    # First it calculates f - g and the corresponding signs
    # using np.sign. Applying np.diff reveals all
    # the positions, where the sign changes (e.g. the lines cross).
    ids_alpha = np.argwhere(np.diff(np.sign(f - z))).flatten()
    ids_alphaP1SD = np.argwhere(np.diff(np.sign(fplus1SD - z))).flatten()
    ids_alphaM1SD = np.argwhere(np.diff(np.sign(fminus1SD - z))).flatten()
    ids_alphaM2SD = np.argwhere(np.diff(np.sign(fminus2SD - z))).flatten()

    # Only get the first index past the splitpoint
    try:
        ids_alpha = ids_alpha[s[ids_alpha] > CuSplit][0]
    except IndexError:
        log.warning('Alpha out of profile')
        ids_alpha = None

    try:
        ids_alphaP1SD = ids_alphaP1SD[s[ids_alphaP1SD] > CuSplit][0]
    except IndexError:
        log.warning('+1 SD above beta point')
        ids_alphaP1SD = None

    try:
        ids_alphaM1SD = ids_alphaM1SD[s[ids_alphaM1SD] > CuSplit][0]
    except IndexError:
        log.warning('-1 SD out of profile')
        ids_alphaM1SD = None

    try:
        ids_alphaM2SD = ids_alphaM2SD[s[ids_alphaM2SD] > CuSplit][0]
    except IndexError:
        log.warning('-2 SD out of profile')
        ids_alphaM2SD = None

    resAB[name]['f'] = f
    resAB[name]['ids_alpha'] = ids_alpha
    resAB[name]['ids_alphaP1SD'] = ids_alphaP1SD
    resAB[name]['ids_alphaM1SD'] = ids_alphaM1SD
    resAB[name]['ids_alphaM2SD'] = ids_alphaM2SD

    return resAB


def writeABpostOut(resAB, cfg, reportDictList):
    """ Loops on the given Avapath, runs AlpahBeta Postprocessing
    plots Results and Write Results
    """
    saveOutPath = resAB['saveOutPath']
    flags = cfg['FLAGS']
    AvaPath = resAB['AvaPath']
    NameAva = AvaPath['Name']
    FileNamePlot_ext = [None] * len(NameAva)
    FileNameWrite_ext = [None] * len(NameAva)
    for i in range(len(NameAva)):
        name = NameAva[i]
        resAB = processABresults(resAB, name)
        # Plot the whole profile with beta, alpha ... points and lines
        savename = name + '_AlphaBeta'
        savePath = pathlib.Path(saveOutPath, savename)
        plotPath(resAB, name, flags)
        FileNamePlot_ext[i] = plotProfile(resAB, name, savePath, flags)
        reportAB, FileNameWrite_ext[i] = WriteResults(resAB, name, flags)
        reportAB['AlphaBeta plots'][name] = savePath
        # Add to report dictionary list
        reportDictList.append(reportAB)
    if flags.getboolean('plotPath') or flags.getboolean('plotProfile'):
        plt.pause(0.001)
        input("Press [enter] to continue.")
    return reportDictList, FileNamePlot_ext, FileNameWrite_ext


def plotPath(resAB, name, flags):
    """ Plot and save results depending on flags options"""
    splitPoint = resAB['splitPoint']
    header = resAB['dem']['header']
    rasterdata = resAB['dem']['rasterData']
    x = resAB[name]['x']
    y = resAB[name]['y']
    indSplit = resAB[name]['indSplit']

    if flags.getboolean('plotPath'):
        # Plot raster and path
        fig, ax = plt.subplots(figsize=(pU.figW, pU.figH))
        titleText = name
        plt.title(titleText)
        cmap = copy.copy(mpl.cm.get_cmap("Greys"))
        cmap.set_bad(color='white')
        im = plt.imshow(rasterdata, cmap, origin='lower')
        divider = make_axes_locatable(ax)
        cax = divider.append_axes("right", size="5%", pad=0.1)
        fig.colorbar(im, cax=cax)
        # path1 = ax1.plot((x-header['xllcorner'])/header['cellsize'],
        #                  (y-header['yllcorner'])/header['cellsize'])
        ax.plot((x-header['xllcorner'])/header['cellsize'],
                (y-header['yllcorner'])/header['cellsize'], 'k',
                label='avapath')
        ax.plot((splitPoint['x']-header['xllcorner'])/header['cellsize'],
                (splitPoint['y']-header['yllcorner'])/header['cellsize'], '.',
                color='0.3', label='Split points')
        ax.plot((x[indSplit]-header['xllcorner'])/header['cellsize'],
                (y[indSplit]-header['yllcorner'])/header['cellsize'], '.',
                color='0.6', label='Projection of Split Point on ava path')
        fig.legend(frameon=False, loc='lower center')
        pU.putAvaNameOnPlot(ax, name)
        plt.show(block=False)


def plotProfile(resAB, name, save_file, flags):
    """ Plot and save results depending on flags options"""
    s = resAB[name]['s']
    z = resAB[name]['z']
    ids10Point = resAB[name]['ids10Point']
    poly = resAB[name]['poly']
    beta = resAB[name]['beta']
    alpha = resAB[name]['alpha']
    f = resAB[name]['f']
    ids_alphaM1SD = resAB[name]['ids_alphaM1SD']
    ids_alphaM2SD = resAB[name]['ids_alphaM2SD']
    indSplit = resAB[name]['indSplit']
    ParameterSet = resAB['eqParams']['ParameterSet']
    # Plot the whole profile with beta, alpha ... points and lines
    fig_prof = plt.figure(figsize=(1.5*pU.figW, 1*pU.figH))
    titleText = name
    plt.title(titleText)
    xlabelText = 'Distance [m]\nBeta: ' + str(round(beta, 1)) + \
        '°' + '  Alpha: ' + str(round(alpha, 1)) + '°'
    plt.xlabel(xlabelText, multialignment='center')
    plt.ylabel('Height [m]')
    plt.plot(s, z, '-', label='DEM')
    plt.plot(s, poly(s), ':', label='QuadFit')
    plt.axvline(x=s[indSplit], color='0.7',
                linewidth=1, linestyle='--', label='Split point')
    plt.axvline(x=s[ids10Point], color='0.8',
                linewidth=1, linestyle='-.', label='Beta')
    plt.plot(s, f, '-', label='AlphaLine')
    if ids_alphaM1SD:
        plt.plot(s[ids_alphaM1SD], z[ids_alphaM1SD], 'x', markersize=8,
                 label='Alpha - 1SD')
    if ids_alphaM2SD:
        plt.plot(s[ids_alphaM2SD], z[ids_alphaM2SD], 'x', markersize=8,
                 label='Alpha - 2SD')

    ax = plt.gca()
    fig_prof.tight_layout()
    versionText = datetime.datetime.now().strftime("%d.%m.%y") + \
        '; ' + 'AlphaBeta ' + ParameterSet
    plt.text(00, 0, versionText, fontsize=8, verticalalignment='bottom',
             horizontalalignment='left', transform=ax.transAxes,
             color='0.5')
    # plt.text(-0.2, 0, 'matplotlib -2', \
    #          verticalalignment='center', transform=ax.transAxes)

    plt.gca().set_aspect('equal', adjustable='box')
    plt.grid(linestyle=':', color='0.9')
    plt.legend(frameon=False)
    plt.draw()
    if flags.getboolean('plotProfile'):
        plt.show(block=False)
    if flags.getboolean('saveProfile'):
        log.debug('Saving profile figure to: %s', save_file)
        fig_prof.savefig(save_file)

    plt.close("all")

    return save_file


def WriteResults(resAB, name, flags):
    """ Write AB results to file """
    saveOutPath = resAB['saveOutPath']
    s = resAB[name]['s']
    x = resAB[name]['x']
    y = resAB[name]['y']
    z = resAB[name]['z']
    ids10Point = resAB[name]['ids10Point']
    beta = resAB[name]['beta']
    alpha = resAB[name]['alpha']
    alphaSD = resAB[name]['alphaSD']
    ids_alpha = resAB[name]['ids_alpha']
    ids_alphaP1SD = resAB[name]['ids_alphaP1SD']
    ids_alphaM1SD = resAB[name]['ids_alphaM1SD']
    ids_alphaM2SD = resAB[name]['ids_alphaM2SD']
    eqParameters = resAB['eqParams']
    ParameterSet = eqParameters['ParameterSet']
    # prepare report dictionary
    # Create dictionary
    reportAB = {}
    reportAB = {'headerLine': {'type': 'title', 'title': 'com1AB Simulation'},
                'avaPath': {'type': 'simName', 'name': name},
                ParameterSet + 'setup': {'type': 'list',
                                         'k1': eqParameters['k1'],
                                         'k2': eqParameters['k2'],
                                         'k3': eqParameters['k3'],
                                         'k4': eqParameters['k4'],
                                         'SD': eqParameters['SD']},
                'AlphaBeta results': {'type': 'list',
                                      'beta': '',
                                      'alpha': '',
                                      'alphaM1SD': '',
                                      'alphaM2SD': '',
                                      'alphaP1SD': ''},
                'AlphaBeta plots': {'type': 'image'}}

    IND = [ids_alpha, ids10Point, ids_alphaM1SD, ids_alphaM2SD,
           ids_alphaP1SD]
    ANGLE = [alpha, beta, alphaSD[1], alphaSD[2], alphaSD[0]]
    LABEL = ['alpha', 'beta', 'alphaM1SD', 'alphaM2SD', 'alphaP1SD']

    # write report and log
    log.info('Profile: %s with %s parameter set',  name, ParameterSet)
    parameterName = ['k1', 'k2', 'k3', 'k4', 'SD']
    for paramName in parameterName:
        log.info('%s = %g' % (paramName, eqParameters[paramName]))
    log.info(('{:<13s}'*6).format(
        ' ', 'x [m]', 'y [m]', 'z [m]', 's [m]', 'angle [°]'))
    for ind, label, angle in zip(IND, LABEL, ANGLE):
        reportAB = addLine2Report(ind, reportAB, x, y, z, s, label, angle)

    # write results to txt file
    FileName_ext = ''
    if flags.getboolean('writeRes'):
        FileName_ext = pathlib.Path(saveOutPath, name + '_AB_results.txt')
        with open(FileName_ext, 'w') as outfile:
            outfile.write('Profile name %s\n' % name)
            outfile.write('Parameter Set %s\n' % ParameterSet)
            for paramName in parameterName:
                outfile.write('%s = %g\n' % (paramName, eqParameters[paramName]))
            outfile.write('Alpha Beta AlMinus1SD AlMinus2SD AlPlus1SD\n')
            outfile.write(('{:<13s}'*5 + '\n').format(
                'x', 'y', 'z', 's', 'angle'))
            for ind, angle in zip(IND, ANGLE):
                writeLine(ind, outfile, x, y, z, s, angle)

    return reportAB, FileName_ext


def writeLine(ind, outfile, x, y, z, s, angle):
    if ind:
        outfile.write(('{:<13.2f}'*5 + '\n').format(
            x[ind], y[ind], z[ind], s[ind], angle))
    else:
        outfile.write(('{:<13.2f}'*5 + '\n').format(0, 0, 0, 0, 0))


def addLine2Report(ind, reportAB, x, y, z, s, label, angle):
    if ind:
        log.info(('{:<13s}'+'{:<13.2f}'*5).format(label, x[ind], y[ind],
                                                  z[ind], s[ind], angle))
        strAlpha = ('{:.2f}' + '{:s}').format(angle, '°')
    else:
        strAlpha = label + 'point out of profile'
        log.warning(strAlpha)
    reportAB['AlphaBeta results'].update({label: strAlpha})

    return reportAB
