"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
/**
 * A Lambda-based custom resource handler that provisions S3 bucket
 * notifications for a bucket.
 *
 * The resource property schema is:
 *
 * {
 *   BucketName: string, NotificationConfiguration: { see
 *   PutBucketNotificationConfiguration }
 * }
 *
 * For 'Delete' operations, we send an empty NotificationConfiguration as
 * required. We propagate errors and results as-is.
 *
 * Sadly, we can't use @aws-cdk/aws-lambda as it will introduce a dependency
 * cycle, so this uses raw `cdk.Resource`s.
 */
class NotificationsResourceHandler extends cdk.Construct {
    constructor(scope, id) {
        super(scope, id);
        const role = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole')
            ]
        });
        // handler allows to put bucket notification on s3 buckets.
        role.addToPolicy(new iam.PolicyStatement({
            actions: ['s3:PutBucketNotification'],
            resources: ['*']
        }));
        const resourceType = 'AWS::Lambda::Function';
        class InLineLambda extends cdk.CfnResource {
            constructor() {
                super(...arguments);
                this.tags = new cdk.TagManager(cdk.TagType.STANDARD, resourceType);
            }
            renderProperties(properties) {
                properties.Tags = cdk.listMapper(cdk.cfnTagToCloudFormation)(this.tags.renderTags());
                delete properties.tags;
                return properties;
            }
        }
        const resource = new InLineLambda(this, 'Resource', {
            type: resourceType,
            properties: {
                Description: 'AWS CloudFormation handler for "Custom::S3BucketNotifications" resources (@aws-cdk/aws-s3)',
                Code: { ZipFile: `exports.handler = ${handler.toString()};` },
                Handler: 'index.handler',
                Role: role.roleArn,
                Runtime: 'nodejs10.x',
                Timeout: 300,
            }
        });
        resource.node.addDependency(role);
        this.functionArn = resource.getAtt('Arn').toString();
    }
    /**
     * Defines a stack-singleton lambda function with the logic for a CloudFormation custom
     * resource that provisions bucket notification configuration for a bucket.
     *
     * @returns The ARN of the custom resource lambda function.
     */
    static singleton(context) {
        const root = cdk.Stack.of(context);
        // well-known logical id to ensure stack singletonity
        const logicalId = 'BucketNotificationsHandler050a0587b7544547bf325f094a3db834';
        let lambda = root.node.tryFindChild(logicalId);
        if (!lambda) {
            lambda = new NotificationsResourceHandler(root, logicalId);
        }
        return lambda.functionArn;
    }
}
exports.NotificationsResourceHandler = NotificationsResourceHandler;
// tslint:disable:no-console
/**
 * Lambda event handler for the custom resource. Bear in mind that we are going
 * to .toString() this function and inline it as Lambda code.
 *
 * The function will issue a putBucketNotificationConfiguration request for the
 * specified bucket.
 */
const handler = (event, context) => {
    // eslint-disable-next-line @typescript-eslint/no-require-imports, import/no-extraneous-dependencies
    const s3 = new (require('aws-sdk').S3)();
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const https = require('https');
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const url = require('url');
    log(JSON.stringify(event, undefined, 2));
    const props = event.ResourceProperties;
    if (event.RequestType === 'Delete') {
        props.NotificationConfiguration = {}; // this is how you clean out notifications
    }
    const req = {
        Bucket: props.BucketName,
        NotificationConfiguration: props.NotificationConfiguration
    };
    return s3.putBucketNotificationConfiguration(req, (err, data) => {
        log({ err, data });
        if (err) {
            return submitResponse('FAILED', err.message + `\nMore information in CloudWatch Log Stream: ${context.logStreamName}`);
        }
        else {
            return submitResponse('SUCCESS');
        }
    });
    function log(obj) {
        console.error(event.RequestId, event.StackId, event.LogicalResourceId, obj);
    }
    // tslint:disable-next-line:max-line-length
    // adapted from https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html#cfn-lambda-function-code-cfnresponsemodule
    // to allow sending an error messge as a reason.
    function submitResponse(responseStatus, reason) {
        const responseBody = JSON.stringify({
            Status: responseStatus,
            Reason: reason || 'See the details in CloudWatch Log Stream: ' + context.logStreamName,
            PhysicalResourceId: event.PhysicalResourceId || event.LogicalResourceId,
            StackId: event.StackId,
            RequestId: event.RequestId,
            LogicalResourceId: event.LogicalResourceId,
            NoEcho: false,
        });
        log({ responseBody });
        const parsedUrl = url.parse(event.ResponseURL);
        const options = {
            hostname: parsedUrl.hostname,
            port: 443,
            path: parsedUrl.path,
            method: 'PUT',
            headers: {
                'content-type': '',
                'content-length': responseBody.length
            }
        };
        const request = https.request(options, (r) => {
            log({ statusCode: r.statusCode, statusMessage: r.statusMessage });
            context.done();
        });
        request.on('error', (error) => {
            log({ sendError: error });
            context.done();
        });
        request.write(responseBody);
        request.end();
    }
};
//# sourceMappingURL=data:application/json;base64,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