"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToSqs = void 0;
// Imports
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
const cdk = require("@aws-cdk/core");
/**
 * @summary The ApiGatewayToSqs class.
 */
class ApiGatewayToSqs extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayToSqs class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {ApiGatewayToSqsProps} props - user provided props for the construct.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the dead letter queue, if applicable
        if (!props.deployDeadLetterQueue || props.deployDeadLetterQueue === true) {
            const dlq = defaults.buildQueue(this, 'deadLetterQueue', {
                queueProps: props.queueProps
            });
            this.deadLetterQueue = defaults.buildDeadLetterQueue({
                deadLetterQueue: dlq,
                maxReceiveCount: (props.maxReceiveCount) ? props.maxReceiveCount : 3
            });
        }
        // Setup the queue
        this.sqsQueue = defaults.buildQueue(this, 'queue', {
            queueProps: props.queueProps,
            deadLetterQueue: this.deadLetterQueue
        });
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] = defaults.GlobalRestApi(this);
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        // Setup the API Gateway resource
        const apiGatewayResource = this.apiGateway.root.addResource('message');
        // Setup API Gateway methods
        // Create
        if (props.allowCreateOperation && props.allowCreateOperation === true && props.createRequestTemplate) {
            const createRequestTemplate = "Action=SendMessage&MessageBody=$util.urlEncode(\"$input.body\")";
            this.addActionToPolicy("sqs:SendMessage");
            this.addMethod(this.apiGateway.root, createRequestTemplate, "POST");
        }
        // Read
        if (!props.allowReadOperation || props.allowReadOperation === true) {
            const getRequestTemplate = "Action=ReceiveMessage";
            this.addActionToPolicy("sqs:ReceiveMessage");
            this.addMethod(this.apiGateway.root, getRequestTemplate, "GET");
        }
        // Delete
        if (props.allowDeleteOperation && props.allowDeleteOperation === true) {
            const deleteRequestTemplate = "Action=DeleteMessage&ReceiptHandle=$util.urlEncode($input.params('receiptHandle'))";
            this.addActionToPolicy("sqs:DeleteMessage");
            this.addMethod(apiGatewayResource, deleteRequestTemplate, "DELETE");
        }
    }
    addActionToPolicy(action) {
        this.apiGatewayRole.addToPolicy(new iam.PolicyStatement({
            resources: [
                this.sqsQueue.queueArn
            ],
            actions: [`${action}`]
        }));
    }
    addMethod(apiResource, requestTemplate, apiMethod) {
        // Add the integration
        const apiGatewayIntegration = new api.AwsIntegration({
            service: "sqs",
            path: `${cdk.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
            integrationHttpMethod: "POST",
            options: {
                passthroughBehavior: api.PassthroughBehavior.NEVER,
                credentialsRole: this.apiGatewayRole,
                requestParameters: {
                    "integration.request.header.Content-Type": "'application/x-www-form-urlencoded'"
                },
                requestTemplates: {
                    "application/json": requestTemplate
                },
                integrationResponses: [
                    {
                        statusCode: "200"
                    },
                    {
                        statusCode: "500",
                        responseTemplates: {
                            "text/html": "Error"
                        },
                        selectionPattern: "500"
                    }
                ]
            }
        });
        // Add the method to the resource
        apiResource.addMethod(apiMethod, apiGatewayIntegration, {
            authorizationType: api.AuthorizationType.IAM,
            methodResponses: [
                {
                    statusCode: "200",
                    responseParameters: {
                        "method.response.header.Content-Type": true
                    }
                },
                {
                    statusCode: "500",
                    responseParameters: {
                        "method.response.header.Content-Type": true
                    },
                }
            ]
        });
    }
}
exports.ApiGatewayToSqs = ApiGatewayToSqs;
//# sourceMappingURL=data:application/json;base64,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