import * as cdk from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { IApplication } from './application';
/**
 * Properties for a Service Catalog AppRegistry AutoApplication
 */
export interface ApplicationAssociatorProps {
    /**
     * Enforces a particular physical application name.
     *
     * @default - No name.
     */
    readonly applicationName?: string;
    /**
     * Enforces a particular application arn.
     *
     * @default - No application arn.
     */
    readonly applicationArnValue?: string;
    /**
     * Application description.
     *
     * @default - No description.
     */
    readonly description?: string;
    /**
     * Stack properties.
     *
     */
    readonly stackProps: cdk.StackProps;
}
/**
 * An AppRegistry construct to automatically create an application with the given name and description.
 *
 * The application name must be unique at the account level and it's immutable.
 * This construct will automatically associate all stacks in the given scope, however
 * in case of a `Pipeline` stack, stage underneath the pipeline will not automatically be associated and
 * needs to be associated separately.
 *
 * If cross account stack is detected, then this construct will automatically share the application to consumer accounts.
 * Cross account feature will only work for non environment agnostic stacks.
 */
export declare class ApplicationAssociator extends Construct {
    /**
     * Created or imported application.
     */
    private readonly application;
    private readonly associatedStages;
    constructor(scope: cdk.App, id: string, props: ApplicationAssociatorProps);
    /**
     * Associate this application with the given stage.
     *
     */
    associateStage(stage: cdk.Stage): cdk.Stage;
    /**
     * Validates if a stage is already associated to the application.
     *
     */
    isStageAssociated(stage: cdk.Stage): boolean;
    /**
     * Get the AppRegistry application.
     *
     */
    get appRegistryApplication(): IApplication;
}
