"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3ToLambda = void 0;
const core_1 = require("@aws-cdk/core");
const defaults = require("@aws-solutions-constructs/core");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
class S3ToLambda extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the S3ToLambda class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {S3ToLambdaProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            deployLambda: props.deployLambda,
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        this.s3Bucket = defaults.buildS3Bucket(this, {
            deployBucket: props.deployBucket,
            existingBucketObj: props.existingBucketObj,
            bucketProps: props.bucketProps
        });
        // Create S3 trigger to invoke lambda function
        this.lambdaFunction.addEventSource(new aws_lambda_event_sources_1.S3EventSource(this.s3Bucket, defaults.S3EventSourceProps(props.s3EventSourceProps)));
        this.addCfnNagSuppress();
    }
    addCfnNagSuppress() {
        // Extract the CfnBucket from the s3Bucket
        const s3BucketResource = this.s3Bucket.node.findChild('Resource');
        s3BucketResource.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W51',
                        reason: `This S3 bucket Bucket does not need a bucket policy`
                    }]
            }
        };
        const root = core_1.Stack.of(this);
        const logicalId = 'BucketNotificationsHandler050a0587b7544547bf325f094a3db834';
        const notificationsResourceHandler = root.node.tryFindChild(logicalId);
        const notificationsResourceHandlerRoleRole = notificationsResourceHandler.node.findChild('Role');
        const notificationsResourceHandlerRolePolicy = notificationsResourceHandlerRoleRole.node.findChild('DefaultPolicy');
        // Extract the CfnFunction from the Function
        const fnResource = notificationsResourceHandler.node.findChild('Resource');
        fnResource.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W58',
                        reason: `Lambda function has permission to write CloudWatch Logs via AWSLambdaBasicExecutionRole policy attached to the lambda role`
                    }]
            }
        };
        // Extract the CfnPolicy from the iam.Policy
        const policyResource = notificationsResourceHandlerRolePolicy.node.findChild('Resource');
        policyResource.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W12',
                        reason: `Bucket resource is '*' due to circular dependency with bucket and role creation at the same time`
                    }]
            }
        };
    }
}
exports.S3ToLambda = S3ToLambda;
//# sourceMappingURL=data:application/json;base64,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