"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const nodeunit_shim_1 = require("nodeunit-shim");
const codepipeline = require("../lib");
const validations = require("../lib/validation");
const fake_build_action_1 = require("./fake-build-action");
const fake_source_action_1 = require("./fake-source-action");
/* eslint-disable quote-props */
nodeunit_shim_1.nodeunitShim({
    'artifact bounds validation': {
        'artifacts count exceed maximum'(test) {
            const result = boundsValidationResult(1, 0, 0);
            test.deepEqual(result.length, 1);
            test.ok(result[0].match(/cannot have more than 0/), 'the validation should have failed');
            test.done();
        },
        'artifacts count below minimum'(test) {
            const result = boundsValidationResult(1, 2, 2);
            test.deepEqual(result.length, 1);
            test.ok(result[0].match(/must have at least 2/), 'the validation should have failed');
            test.done();
        },
        'artifacts count within bounds'(test) {
            const result = boundsValidationResult(1, 0, 2);
            test.deepEqual(result.length, 0);
            test.done();
        },
    },
    'action type validation': {
        'must be source and is source'(test) {
            const result = validations.validateSourceAction(true, codepipeline.ActionCategory.SOURCE, 'test action', 'test stage');
            test.deepEqual(result.length, 0);
            test.done();
        },
        'must be source and is not source'(test) {
            const result = validations.validateSourceAction(true, codepipeline.ActionCategory.DEPLOY, 'test action', 'test stage');
            test.deepEqual(result.length, 1);
            test.ok(result[0].match(/may only contain Source actions/), 'the validation should have failed');
            test.done();
        },
        'cannot be source and is source'(test) {
            const result = validations.validateSourceAction(false, codepipeline.ActionCategory.SOURCE, 'test action', 'test stage');
            test.deepEqual(result.length, 1);
            test.ok(result[0].match(/may only occur in first stage/), 'the validation should have failed');
            test.done();
        },
        'cannot be source and is not source'(test) {
            const result = validations.validateSourceAction(false, codepipeline.ActionCategory.DEPLOY, 'test action', 'test stage');
            test.deepEqual(result.length, 0);
            test.done();
        },
    },
    'action name validation': {
        'throws an exception when adding an Action with an empty name to the Pipeline'(test) {
            const stack = new cdk.Stack();
            const action = new fake_source_action_1.FakeSourceAction({
                actionName: '',
                output: new codepipeline.Artifact(),
            });
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
            const stage = pipeline.addStage({ stageName: 'Source' });
            test.throws(() => {
                stage.addAction(action);
            }, /Action name must match regular expression:/);
            test.done();
        },
    },
    'action Artifacts validation': {
        'validates that input Artifacts are within bounds'(test) {
            const stack = new cdk.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const extraOutput1 = new codepipeline.Artifact('A1');
            const extraOutput2 = new codepipeline.Artifact('A2');
            const extraOutput3 = new codepipeline.Artifact('A3');
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'Source',
                                output: sourceOutput,
                                extraOutputs: [
                                    extraOutput1,
                                    extraOutput2,
                                    extraOutput3,
                                ],
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'Build',
                                input: sourceOutput,
                                extraInputs: [
                                    extraOutput1,
                                    extraOutput2,
                                    extraOutput3,
                                ],
                            }),
                        ],
                    },
                ],
            });
            test.throws(() => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {}));
            }, /Build\/Fake cannot have more than 3 input artifacts/);
            test.done();
        },
        'validates that output Artifacts are within bounds'(test) {
            const stack = new cdk.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const extraOutput1 = new codepipeline.Artifact('A1');
            const extraOutput2 = new codepipeline.Artifact('A2');
            const extraOutput3 = new codepipeline.Artifact('A3');
            const extraOutput4 = new codepipeline.Artifact('A4');
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'Source',
                                output: sourceOutput,
                                extraOutputs: [
                                    extraOutput1,
                                    extraOutput2,
                                    extraOutput3,
                                    extraOutput4,
                                ],
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'Build',
                                input: sourceOutput,
                            }),
                        ],
                    },
                ],
            });
            test.throws(() => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {}));
            }, /Source\/Fake cannot have more than 4 output artifacts/);
            test.done();
        },
    },
    'automatically assigns artifact names to the Actions'(test) {
        const stack = new cdk.Stack();
        const pipeline = new codepipeline.Pipeline(stack, 'pipeline');
        const sourceOutput = new codepipeline.Artifact();
        const sourceAction = new fake_source_action_1.FakeSourceAction({
            actionName: 'CodeCommit',
            output: sourceOutput,
        });
        pipeline.addStage({
            stageName: 'Source',
            actions: [sourceAction],
        });
        pipeline.addStage({
            stageName: 'Build',
            actions: [
                new fake_build_action_1.FakeBuildAction({
                    actionName: 'CodeBuild',
                    input: sourceOutput,
                    output: new codepipeline.Artifact(),
                }),
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            'Stages': [
                {
                    'Name': 'Source',
                    'Actions': [
                        {
                            'Name': 'CodeCommit',
                            'OutputArtifacts': [
                                {
                                    'Name': 'Artifact_Source_CodeCommit',
                                },
                            ],
                        },
                    ],
                },
                {
                    'Name': 'Build',
                    'Actions': [
                        {
                            'Name': 'CodeBuild',
                            'InputArtifacts': [
                                {
                                    'Name': 'Artifact_Source_CodeCommit',
                                },
                            ],
                            'OutputArtifacts': [
                                {
                                    'Name': 'Artifact_Build_CodeBuild',
                                },
                            ],
                        },
                    ],
                },
            ],
        }));
        test.done();
    },
    'the same Action can be safely added to 2 different Stages'(test) {
        const stack = new cdk.Stack();
        const sourceOutput = new codepipeline.Artifact();
        const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
            stages: [
                {
                    stageName: 'Source',
                    actions: [
                        new fake_source_action_1.FakeSourceAction({
                            actionName: 'Source',
                            output: sourceOutput,
                        }),
                    ],
                },
            ],
        });
        const action = new fake_build_action_1.FakeBuildAction({ actionName: 'FakeAction', input: sourceOutput });
        const stage2 = {
            stageName: 'Stage2',
            actions: [action],
        };
        const stage3 = {
            stageName: 'Stage3',
            actions: [action],
        };
        pipeline.addStage(stage2);
        test.doesNotThrow(() => {
            pipeline.addStage(stage3);
        }, /FakeAction/);
        test.done();
    },
    'input Artifacts': {
        'can be added multiple times to an Action safely'(test) {
            const artifact = new codepipeline.Artifact('SomeArtifact');
            test.doesNotThrow(() => {
                new fake_build_action_1.FakeBuildAction({
                    actionName: 'CodeBuild',
                    input: artifact,
                    extraInputs: [artifact],
                });
            });
            test.done();
        },
        'can have duplicate names'(test) {
            const artifact1 = new codepipeline.Artifact('SomeArtifact');
            const artifact2 = new codepipeline.Artifact('SomeArtifact');
            test.doesNotThrow(() => {
                new fake_build_action_1.FakeBuildAction({
                    actionName: 'CodeBuild',
                    input: artifact1,
                    extraInputs: [artifact2],
                });
            });
            test.done();
        },
    },
    'output Artifacts': {
        'accept multiple Artifacts with the same name safely'(test) {
            test.doesNotThrow(() => {
                new fake_source_action_1.FakeSourceAction({
                    actionName: 'CodeBuild',
                    output: new codepipeline.Artifact('Artifact1'),
                    extraOutputs: [
                        new codepipeline.Artifact('Artifact1'),
                        new codepipeline.Artifact('Artifact1'),
                    ],
                });
            });
            test.done();
        },
    },
    'an Action with a non-AWS owner cannot have a Role passed for it'(test) {
        const stack = new cdk.Stack();
        const sourceOutput = new codepipeline.Artifact();
        const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
            stages: [
                {
                    stageName: 'Source',
                    actions: [
                        new fake_source_action_1.FakeSourceAction({
                            actionName: 'source',
                            output: sourceOutput,
                        }),
                    ],
                },
            ],
        });
        const buildStage = pipeline.addStage({ stageName: 'Build' });
        // constructing it is fine
        const buildAction = new fake_build_action_1.FakeBuildAction({
            actionName: 'build',
            input: sourceOutput,
            owner: 'ThirdParty',
            role: new iam.Role(stack, 'Role', {
                assumedBy: new iam.AnyPrincipal(),
            }),
        });
        // an attempt to add it to the Pipeline is where things blow up
        test.throws(() => {
            buildStage.addAction(buildAction);
        }, /Role is not supported for actions with an owner different than 'AWS' - got 'ThirdParty' \(Action: 'build' in Stage: 'Build'\)/);
        test.done();
    },
    'actions can be retrieved from stages they have been added to'(test) {
        const stack = new cdk.Stack();
        const sourceOutput = new codepipeline.Artifact();
        const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
            stages: [
                {
                    stageName: 'Source',
                    actions: [
                        new fake_source_action_1.FakeSourceAction({
                            actionName: 'source',
                            output: sourceOutput,
                        }),
                    ],
                },
            ],
        });
        const sourceStage = pipeline.stages[0];
        const buildStage = pipeline.addStage({
            stageName: 'Build',
            actions: [
                new fake_build_action_1.FakeBuildAction({
                    actionName: 'build1',
                    input: sourceOutput,
                    runOrder: 11,
                }),
                new fake_build_action_1.FakeBuildAction({
                    actionName: 'build2',
                    input: sourceOutput,
                    runOrder: 2,
                }),
            ],
        });
        test.equal(sourceStage.actions.length, 1);
        test.equal(sourceStage.actions[0].actionProperties.actionName, 'source');
        test.equal(buildStage.actions.length, 2);
        test.equal(buildStage.actions[0].actionProperties.actionName, 'build1');
        test.equal(buildStage.actions[1].actionProperties.actionName, 'build2');
        test.done();
    },
});
function boundsValidationResult(numberOfArtifacts, min, max) {
    const artifacts = [];
    for (let i = 0; i < numberOfArtifacts; i++) {
        artifacts.push(new codepipeline.Artifact(`TestArtifact${i}`));
    }
    return validations.validateArtifactBounds('output', artifacts, min, max, 'testCategory', 'testProvider');
}
//# sourceMappingURL=data:application/json;base64,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