# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['babble', 'babble.scripts', 'babble.templates.aws']

package_data = \
{'': ['*'], 'babble': ['providers/*']}

entry_points = \
{'console_scripts': ['babble = babble.pip_entry:main']}

setup_kwargs = {
    'name': 'babble-cloud',
    'version': '0.1.4',
    'description': 'Rapidly deploy serverless HTTP APIs to AWS through a low-code Pythonic YAML configuration file.',
    'long_description': '\n# Babble Cloud\n\nBabble is a tool that enables beginner programmers to rapidly deploy serverless HTTP APIs to AWS through a low-code Pythonic YAML configuration file.\n\n\n## Table of Contents\n\n[Requirements](#requirements)  \n[Installation](#installation)  \n[Usage](#usage)  \n[Configuration](#configuration)  \n[Demonstration](#demonstration)\n## Requirements\n\n[Python](https://www.python.org/downloads/)  \n[Docker](https://docs.docker.com/engine/install/)  \n[AWS](https://aws.amazon.com/free/)\n\n## Installation\n\nTo install Babble, run the following command:\n```bash\n  pip install babble-cloud\n```\n    \n## Usage\n\nTo start a new project, run the following command:\n```bash\n  babble create\n```\nTo set the configuration file for a project, run the following command:\n```bash\n  babble push <source file>\n```\nTo retrieve the configuration file from a project, run the following command:\n```bash\n  babble pull <destination file>\n```\nTo view the current configuration of a project, run the following command:\n```bash\n  babble view\n```\nTo activate a project, run the following command:\n```bash\n  babble activate\n```\nTo deactivate your project, run the following command:\n```bash\n  babble deactivate\n```\nTo delete your project, run the following command:\n```bash\n  babble delete\n```\n**Note:**&nbsp; Upon running the commands `babble activate`, `babble deactivate`, and in some cases `babble delete`, you will be prompted for your AWS account credentials in order to add or remove the necessary resources from your AWS account.  For information about obtaining these credentials, please refer to the [AWS documentation](https://docs.aws.amazon.com/singlesignon/latest/userguide/howtogetcredentials.html) on getting IAM role credentials.\n## Configuration\n\nTo configure your application infrastructure, you will need to create a YAML configuration file containing your resources.  Currently, resources fall under five categories: `tables`, `folders`, `packages`, `modules`, and `endpoints`.\n\n### Tables\nTable definitions are used to deploy DynamoDB tables to the user\'s AWS account.  The following methods can be accessed via dot notation from module and endpoint source code:\n\n**`put_item(item: dict)`** - Creates a new item in the table, or replaces an old item with a new item.  It is required that the item contains the "key" attribute specified in the table\'s configuration. If an item that has the same "key" attribute as the new item already exists in the table, the new item completely replaces the existing item. Otherwise, an entirely new item will be created.\n\n**`get_item(item: dict)`** - Retrieves an item from the table.  It is required that the item contains the "key" attribute specified in the table\'s configuration.  If there is no matching item, get_item will return an empty dict.\n\n**`delete_item(item: dict)`** - Retrieves an item from the table.  It is required that the item contains the "key" attribute specified in the table\'s configuration.\n\n**`scan()`** - Performs a full table scan, returning a list of all items in the table.\n\n```yaml\ntables:\n    <table name>:\n        key: <key name>\n    <table name>:\n        key: <key name>\n    ...\n```\n### Folders\nFolder definitions are used to deploy S3 buckets to the user\'s AWS account.  The following methods can be accessed via dot notation from module and endpoint source code:\n\n**`open(path: str, mode: str)`** - Opens or creates an S3 file based on the `path` specification, and returns it as a file object. This method emulates the behavior of the standard Python `open` function.\n\n**`url(path: str, expiration: int = 3600)`** - Returns a temporary download URL for an S3 file. This URL will be valid for the number of seconds specified by `expiration`, with a default value of 3600 seconds, or one hour.\n\n\n```yaml\nfolders:\n    <folder name>:\n    <folder name>:\n    ...\n```\n### Packages\nPackage definitions are used to include external packages in both module and endpoint source code.  These packages can be sourced from the Python Standard Library (lib), or from the Python Package Index (pip).\n```yaml\npackages:\n    <package name>:\n        source: <lib or pip>\n        version: <version number or "~">\n    <package name>:\n        source: <lib or pip>\n        version: <version number or "~">\n    ...\n```\n### Modules\nModule definitions are used to write source code for the application.  All functions, classes, and variables defined within these modules can be accessed via dot notation from module and endpoint source code. Please note that all imports are handled implicitly and there is no need for any import statements.  All tables, folders, packages, and other modules can be accessed via dot notation.\n```yaml\nmodules:\n    <module name>:\n        content: |\n            # source code goes here\n    <module name>:\n        content: |\n            # source code goes here\n    ...\n```\n### Endpoints\nEndpoint definitions are used to write source code that processes the request data and determines the response data. Please note that all imports are handled implicitly and there is no need for any import statements.  All tables, folders, packages, and modules can be accessed via dot notation. It is imperative that all endpoints contain a function named `request` with the parameters `headers`, `path`, `query`, and `body` as shown below.\n```yaml\nendpoints:\n    <endpoint method> <endpoint path>:\n        content: |\n            def request(headers, path, query, body):\n                # source code goes here\n                return response\n    <endpoint method> <endpoint path>:\n        content: |\n            def request(headers, path, query, body):\n                # source code goes here\n                return response\n```\n## Demonstration\n\n### Creation and Activation\n\n**Step 1:**  Create a new project. Upon running this command, you will be prompted for a name for your new project.\n```bash\nbabble create\n```\n**Step 2:**  Push your YAML configuration file to the project.  Upon running this command, you will be prompted to select a project you previously created.\n```bash\nbabble push <filename>\n```\n**Step 3:**  Activate your project.  Upon running this command, you will be prompted to select a project you previously created.\n```bash\nbabble activate\n```\n**Step 4:**  Obtain the URL for your newly created HTTP API.  Upon running this command, you will be prompted to select a project you previously created.\n```bash\nbabble view\n```\n\n### Deactivation and Deletion\n**Step 1:**  Deactivate your project.   Upon running this command, you will be prompted to select a project you previously created.\n```bash\nbabble deactivate\n```\n**Step 2:**  Delete your project.   Upon running this command, you will be prompted to select a project you previously created.  If the selected project is not deactivated, it will be deactivated automatically.\n```bash\nbabble delete\n```\n\n### Configuration File\nBelow is an example of a configuration file that can be used with Babble to deploy a serverless HTTP API with the following endpoints:\n\n**`PUT /items/{item_id}`** - Creates a new entry in `my_table`, using the path parameter `item_id` as the key, and the JSON-formatted body for all other attributes.\n\n**`GET /items/{item_id}`** - Returns an entry from `my_table`, using the path parameter `item_id` to find the entry.\n\n**`DELETE /items/{item_id}`** - Deletes an entry from `my_table`, using the path parameter `item_id` to find the entry.\n\n**`GET /items`** - Reads all entries in `my_table`, creates a Pandas DataFrame containing these entries, writes the DataFrame to the file `output.txt` in `my_folder`, and returns a temporary download URL for the newly created file.\n\n\n```yaml\ntables:\n  my_table:\n    key: my_key\nfolders:\n  my_folder:\npackages:\n  pandas:\n    source: pip\n    version: 1.5.3\n  json:\n    source: lib\n    version: ~\nmodules:\n  my_module:\n    content: |\n      def get_item(item_key):\n        return my_table.get_item({"my_key": item_key})\n      def put_item(item_key, body):\n        item = json.loads(body)\n        item["my_key"] = item_key\n        my_table.put_item(item)\n      def delete_item(item_key):\n        my_table.delete_item({"my_key": item_key})\n      def get_items():\n        items = my_table.scan()\n        df = pandas.DataFrame.from_dict(items)\n        with my_folder.open("output.txt", "w") as fp:\n          fp.write(df.to_string())\n        return my_folder.url("output.txt")\nendpoints:\n  GET /items/{item_key}:\n    content: |\n      def request(headers, path, query, body):\n        return my_module.get_item(path["item_key"])\n  PUT /items/{item_key}:\n    content: |\n      def request(headers, path, query, body):\n        return my_module.put_item(path["item_key"], body)\n  DELETE /items/{item_key}:\n    content: |\n      def request(headers, path, query, body):\n        return my_module.delete_item(path["item_key"])\n  GET /items:\n    content: |\n      def request(headers, path, query, body):\n        return my_module.get_items()\n```\n\n',
    'author': 'mkearney2023',
    'author_email': 'mkearney2023@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'entry_points': entry_points,
    'python_requires': '>=3.0,<4.0',
}


setup(**setup_kwargs)
