"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderJsonPath = exports.renderList = exports.State = void 0;
const cdk = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const fields_1 = require("../fields");
const types_1 = require("../types");
/**
 * Base class for all other state classes.
 *
 * @stability stable
 */
class State extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * @stability stable
         */
        this.branches = [];
        this.retries = [];
        this.catches = [];
        this.choices = [];
        this.prefixes = [];
        /**
         * States with references to this state.
         *
         * Used for finding complete connected graph that a state is part of.
         */
        this.incomingStates = [];
        this.startState = this;
        this.comment = props.comment;
        this.inputPath = props.inputPath;
        this.parameters = props.parameters;
        this.outputPath = props.outputPath;
        this.resultPath = props.resultPath;
    }
    /**
     * Add a prefix to the stateId of all States found in a construct tree.
     *
     * @stability stable
     */
    static prefixStates(root, prefix) {
        const queue = [root];
        while (queue.length > 0) {
            const el = queue.splice(0, 1)[0];
            if (isPrefixable(el)) {
                el.addPrefix(prefix);
            }
            queue.push(...constructs_1.Node.of(el).children);
        }
    }
    /**
     * Find the set of states reachable through transitions from the given start state.
     *
     * This does not retrieve states from within sub-graphs, such as states within a Parallel state's branch.
     *
     * @stability stable
     */
    static findReachableStates(start, options = {}) {
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            queue.push(...outgoing);
            ret.add(state);
        }
        return Array.from(ret);
    }
    /**
     * Find the set of end states states reachable through transitions from the given start state.
     *
     * @stability stable
     */
    static findReachableEndStates(start, options = {}) {
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            if (outgoing.length > 0) {
                // We can continue
                queue.push(...outgoing);
            }
            else {
                // Terminal state
                ret.add(state);
            }
        }
        return Array.from(ret);
    }
    /**
     * Return only the states that allow chaining from an array of states.
     *
     * @stability stable
     */
    static filterNextables(states) {
        return states.filter(isNextable);
    }
    /**
     * Descriptive identifier for this chainable.
     *
     * @stability stable
     */
    get id() {
        return this.node.id;
    }
    /**
     * Tokenized string that evaluates to the state's ID.
     *
     * @stability stable
     */
    get stateId() {
        return this.prefixes.concat(this.id).join('');
    }
    /**
     * Add a prefix to the stateId of this state.
     *
     * @stability stable
     */
    addPrefix(x) {
        if (x !== '') {
            this.prefixes.splice(0, 0, x);
        }
    }
    /**
     * Register this state as part of the given graph.
     *
     * Don't call this. It will be called automatically when you work
     * with states normally.
     *
     * @stability stable
     */
    bindToGraph(graph) {
        if (this.containingGraph === graph) {
            return;
        }
        if (this.containingGraph) {
            // eslint-disable-next-line max-len
            throw new Error(`Trying to use state '${this.stateId}' in ${graph}, but is already in ${this.containingGraph}. Every state can only be used in one graph.`);
        }
        this.containingGraph = graph;
        this.whenBoundToGraph(graph);
        for (const incoming of this.incomingStates) {
            incoming.bindToGraph(graph);
        }
        for (const outgoing of this.outgoingTransitions({ includeErrorHandlers: true })) {
            outgoing.bindToGraph(graph);
        }
        for (const branch of this.branches) {
            branch.registerSuperGraph(this.containingGraph);
        }
        if (!!this.iteration) {
            this.iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a retrier to the retry list of this state
     * @internal
     */
    _addRetry(props = {}) {
        validateErrors(props.errors);
        this.retries.push({
            ...props,
            errors: props.errors ? props.errors : [types_1.Errors.ALL],
        });
    }
    /**
     * Add an error handler to the catch list of this state
     * @internal
     */
    _addCatch(handler, props = {}) {
        validateErrors(props.errors);
        this.catches.push({
            next: handler,
            props: {
                errors: props.errors ? props.errors : [types_1.Errors.ALL],
                resultPath: props.resultPath,
            },
        });
        handler.addIncoming(this);
        if (this.containingGraph) {
            handler.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default transition of this state.
     *
     * @stability stable
     */
    makeNext(next) {
        // Can't be called 'setNext' because of JSII
        if (this._next) {
            throw new Error(`State '${this.id}' already has a next state`);
        }
        this._next = next;
        next.addIncoming(this);
        if (this.containingGraph) {
            next.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a choice branch to this state.
     *
     * @stability stable
     */
    addChoice(condition, next) {
        this.choices.push({ condition, next });
        next.startState.addIncoming(this);
        if (this.containingGraph) {
            next.startState.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a paralle branch to this state.
     *
     * @stability stable
     */
    addBranch(branch) {
        this.branches.push(branch);
        if (this.containingGraph) {
            branch.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a map iterator to this state.
     *
     * @stability stable
     */
    addIterator(iteration) {
        this.iteration = iteration;
        if (this.containingGraph) {
            iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default choice transition of this state.
     *
     * @stability stable
     */
    makeDefault(def) {
        // Can't be called 'setDefault' because of JSII
        if (this.defaultChoice) {
            throw new Error(`Choice '${this.id}' already has a default next state`);
        }
        this.defaultChoice = def;
    }
    /**
     * Render the default next state in ASL JSON format.
     *
     * @stability stable
     */
    renderNextEnd() {
        if (this._next) {
            return { Next: this._next.stateId };
        }
        else {
            return { End: true };
        }
    }
    /**
     * Render the choices in ASL JSON format.
     *
     * @stability stable
     */
    renderChoices() {
        return {
            Choices: renderList(this.choices, renderChoice),
            Default: this.defaultChoice ? this.defaultChoice.stateId : undefined,
        };
    }
    /**
     * Render InputPath/Parameters/OutputPath in ASL JSON format.
     *
     * @stability stable
     */
    renderInputOutput() {
        return {
            InputPath: renderJsonPath(this.inputPath),
            Parameters: this.parameters,
            OutputPath: renderJsonPath(this.outputPath),
        };
    }
    /**
     * Render parallel branches in ASL JSON format.
     *
     * @stability stable
     */
    renderBranches() {
        return {
            Branches: this.branches.map(b => b.toGraphJson()),
        };
    }
    /**
     * Render map iterator in ASL JSON format.
     *
     * @stability stable
     */
    renderIterator() {
        if (!this.iteration) {
            throw new Error('Iterator must not be undefined !');
        }
        return {
            Iterator: this.iteration.toGraphJson(),
        };
    }
    /**
     * Render error recovery options in ASL JSON format.
     *
     * @stability stable
     */
    renderRetryCatch() {
        return {
            Retry: renderList(this.retries, renderRetry, (a, b) => compareErrors(a.errors, b.errors)),
            Catch: renderList(this.catches, renderCatch, (a, b) => compareErrors(a.props.errors, b.props.errors)),
        };
    }
    /**
     * Called whenever this state is bound to a graph.
     *
     * Can be overridden by subclasses.
     *
     * @stability stable
     */
    whenBoundToGraph(graph) {
        graph.registerState(this);
    }
    /**
     * Add a state to the incoming list
     */
    addIncoming(source) {
        this.incomingStates.push(source);
    }
    /**
     * Return all states this state can transition to
     */
    outgoingTransitions(options) {
        const ret = new Array();
        if (this._next) {
            ret.push(this._next);
        }
        if (this.defaultChoice) {
            ret.push(this.defaultChoice);
        }
        for (const c of this.choices) {
            ret.push(c.next);
        }
        if (options.includeErrorHandlers) {
            for (const c of this.catches) {
                ret.push(c.next);
            }
        }
        return ret;
    }
}
exports.State = State;
/**
 * Render a choice transition
 */
function renderChoice(c) {
    return {
        ...c.condition.renderCondition(),
        Next: c.next.stateId,
    };
}
/**
 * Render a Retry object to ASL
 */
function renderRetry(retry) {
    return {
        ErrorEquals: retry.errors,
        IntervalSeconds: retry.interval && retry.interval.toSeconds(),
        MaxAttempts: retry.maxAttempts,
        BackoffRate: retry.backoffRate,
    };
}
/**
 * Render a Catch object to ASL
 */
function renderCatch(c) {
    return {
        ErrorEquals: c.props.errors,
        ResultPath: renderJsonPath(c.props.resultPath),
        Next: c.next.stateId,
    };
}
/**
 * Compares a list of Errors to move Errors.ALL last in a sort function
 */
function compareErrors(a, b) {
    if (a === null || a === void 0 ? void 0 : a.includes(types_1.Errors.ALL)) {
        return 1;
    }
    if (b === null || b === void 0 ? void 0 : b.includes(types_1.Errors.ALL)) {
        return -1;
    }
    return 0;
}
/**
 * Validates an errors list
 */
function validateErrors(errors) {
    if ((errors === null || errors === void 0 ? void 0 : errors.includes(types_1.Errors.ALL)) && errors.length > 1) {
        throw new Error(`${types_1.Errors.ALL} must appear alone in an error list`);
    }
}
/**
 * Render a list or return undefined for an empty list
 */
function renderList(xs, mapFn, sortFn) {
    if (xs.length === 0) {
        return undefined;
    }
    let list = xs;
    if (sortFn) {
        list = xs.sort(sortFn);
    }
    return list.map(mapFn);
}
exports.renderList = renderList;
/**
 * Render JSON path, respecting the special value DISCARD
 */
function renderJsonPath(jsonPath) {
    if (jsonPath === undefined) {
        return undefined;
    }
    if (jsonPath === fields_1.JsonPath.DISCARD) {
        return null;
    }
    if (!jsonPath.startsWith('$')) {
        throw new Error(`Expected JSON path to start with '$', got: ${jsonPath}`);
    }
    return jsonPath;
}
exports.renderJsonPath = renderJsonPath;
/**
 * Whether an object is a Prefixable
 */
function isPrefixable(x) {
    return typeof (x) === 'object' && x.addPrefix;
}
/**
 * Whether an object is INextable
 */
function isNextable(x) {
    return typeof (x) === 'object' && x.next;
}
//# sourceMappingURL=data:application/json;base64,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