"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FieldUtils = exports.Context = exports.Data = exports.JsonPath = void 0;
const core_1 = require("@aws-cdk/core");
const json_path_1 = require("./json-path");
/**
 * Extract a field from the State Machine data or context that gets passed around between states.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-paths.html
 * @stability stable
 */
class JsonPath {
    constructor() { }
    /**
     * Instead of using a literal string, get the value from a JSON path.
     *
     * @stability stable
     */
    static stringAt(path) {
        validateJsonPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * Instead of using a literal string list, get the value from a JSON path.
     *
     * @stability stable
     */
    static listAt(path) {
        // does not apply to task context
        validateDataPath(path);
        return core_1.Token.asList(new json_path_1.JsonPathToken(path));
    }
    /**
     * Instead of using a literal number, get the value from a JSON path.
     *
     * @stability stable
     */
    static numberAt(path) {
        validateJsonPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * Use the entire data structure.
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     *
     * @stability stable
     */
    static get entirePayload() {
        return new json_path_1.JsonPathToken('$').toString();
    }
    /**
     * Determines if the indicated string is an encoded JSON path.
     *
     * @param value string to be evaluated.
     * @stability stable
     */
    static isEncodedJsonPath(value) {
        return !!json_path_1.jsonPathString(value);
    }
    /**
     * Return the Task Token field.
     *
     * External actions will need this token to report step completion
     * back to StepFunctions using the `SendTaskSuccess` or `SendTaskFailure`
     * calls.
     *
     * @stability stable
     */
    static get taskToken() {
        return new json_path_1.JsonPathToken('$$.Task.Token').toString();
    }
    /**
     * Use the entire context data structure.
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     *
     * @stability stable
     */
    static get entireContext() {
        return new json_path_1.JsonPathToken('$$').toString();
    }
}
exports.JsonPath = JsonPath;
/**
 * Special string value to discard state input, output or result.
 *
 * @stability stable
 */
JsonPath.DISCARD = 'DISCARD';
/**
 * (deprecated) Extract a field from the State Machine data that gets passed around between states.
 *
 * @deprecated replaced by `JsonPath`
 */
class Data {
    /**
     * (deprecated) Instead of using a literal string, get the value from a JSON path.
     *
     * @deprecated
     */
    static stringAt(path) {
        validateDataPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * (deprecated) Instead of using a literal string list, get the value from a JSON path.
     *
     * @deprecated
     */
    static listAt(path) {
        validateDataPath(path);
        return core_1.Token.asList(new json_path_1.JsonPathToken(path));
    }
    /**
     * (deprecated) Instead of using a literal number, get the value from a JSON path.
     *
     * @deprecated
     */
    static numberAt(path) {
        validateDataPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * (deprecated) Use the entire data structure.
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     *
     * @deprecated
     */
    static get entirePayload() {
        return new json_path_1.JsonPathToken('$').toString();
    }
    /**
     * (deprecated) Determines if the indicated string is an encoded JSON path.
     *
     * @param value string to be evaluated.
     * @deprecated
     */
    static isJsonPathString(value) {
        return !!json_path_1.jsonPathString(value);
    }
    constructor() { }
}
exports.Data = Data;
/**
 * (deprecated) Extract a field from the State Machine Context data.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#wait-token-contextobject
 * @deprecated replaced by `JsonPath`
 */
class Context {
    /**
     * (deprecated) Instead of using a literal string, get the value from a JSON path.
     *
     * @deprecated
     */
    static stringAt(path) {
        validateContextPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * (deprecated) Instead of using a literal number, get the value from a JSON path.
     *
     * @deprecated
     */
    static numberAt(path) {
        validateContextPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * (deprecated) Return the Task Token field.
     *
     * External actions will need this token to report step completion
     * back to StepFunctions using the `SendTaskSuccess` or `SendTaskFailure`
     * calls.
     *
     * @deprecated
     */
    static get taskToken() {
        return new json_path_1.JsonPathToken('$$.Task.Token').toString();
    }
    /**
     * (deprecated) Use the entire context data structure.
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     *
     * @deprecated
     */
    static get entireContext() {
        return new json_path_1.JsonPathToken('$$').toString();
    }
    constructor() { }
}
exports.Context = Context;
/**
 * Helper functions to work with structures containing fields.
 *
 * @stability stable
 */
class FieldUtils {
    /**
     * Render a JSON structure containing fields to the right StepFunctions structure.
     *
     * @stability stable
     */
    static renderObject(obj) {
        return json_path_1.renderObject(obj);
    }
    /**
     * Return all JSON paths used in the given structure.
     *
     * @stability stable
     */
    static findReferencedPaths(obj) {
        return Array.from(json_path_1.findReferencedPaths(obj)).sort();
    }
    /**
     * Returns whether the given task structure contains the TaskToken field anywhere.
     *
     * The field is considered included if the field itself or one of its containing
     * fields occurs anywhere in the payload.
     *
     * @stability stable
     */
    static containsTaskToken(obj) {
        const paths = json_path_1.findReferencedPaths(obj);
        return paths.has('$$.Task.Token') || paths.has('$$.Task') || paths.has('$$');
    }
    constructor() { }
}
exports.FieldUtils = FieldUtils;
function validateJsonPath(path) {
    if (path !== '$'
        && !path.startsWith('$.')
        && path !== '$$'
        && !path.startsWith('$$.')
        && !path.startsWith('$[')) {
        throw new Error(`JSON path values must be exactly '$', '$$', start with '$.', start with '$$.' or start with '$[' Received: ${path}`);
    }
}
function validateDataPath(path) {
    if (path !== '$' && !path.startsWith('$.')) {
        throw new Error("Data JSON path values must either be exactly equal to '$' or start with '$.'");
    }
}
function validateContextPath(path) {
    if (path !== '$$' && !path.startsWith('$$.')) {
        throw new Error("Context JSON path values must either be exactly equal to '$$' or start with '$$.'");
    }
}
//# sourceMappingURL=data:application/json;base64,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