# Original source: /labspace/models/aimnet/batch_opt_script/

import torch
import warnings
import os
try:
    import torchani
except:
    pass
from collections import defaultdict
from openbabel import pybel
try:
    from .ANI2xt import ANI2xt
except:
    pass
from tqdm import tqdm


hartree2eV = 27.211385
@torch.jit.script
class FIRE():
    """a general optimization program """
    # For a list of documentation for different optimization programs: https://wiki.fysik.dtu.dk/ase/ase/optimize.html
    def __init__(self, coord):
        ## default parameters
        self.dt_max = 1.0
        self.Nmin = 5
        self.maxstep = 0.2
        self.finc = 1.1
        self.fdec = 0.5
        self.astart = 0.1
        self.fa = 0.99
        self.v = torch.zeros_like(coord)
        self.Nsteps = torch.zeros(coord.shape[0], dtype=torch.long, device=coord.device)
        self.dt = torch.full(coord.shape[:1], 0.1, device=coord.device)
        self.a = torch.full(coord.shape[:1], 0.1, device=coord.device)

    def __call__(self, coord, forces):
        """Moving atoms based on forces
        
        Arguments:
            coord: coordinates of atoms. Size (Batch, N, 3), where Batch is
                   the number of structures, N is the number of atom in each structure.
            forces: forces on each atom. Size (Batch, N, 3).
            
        Return:
            new coordinates that are moved based on input forces. Size (Batch, N, 3)"""
        vf = (forces * self.v).flatten(-2, -1).sum(-1)
        w_vf = vf > 0.0
        if w_vf.all():
            a = self.a.unsqueeze(-1).unsqueeze(-1)
            v = self.v
            f = forces
            self.v = (1.0 - a) * v + a * v.flatten(-2, -1).norm(p=2, dim=-1).unsqueeze(-1).unsqueeze(-1) * f / f.flatten(-2, -1).norm(p=2, dim=-1).unsqueeze(-1).unsqueeze(-1)
            self.Nsteps += 1
        elif w_vf.any():
            a = self.a[w_vf].unsqueeze(-1).unsqueeze(-1)
            v = self.v[w_vf]
            f = forces[w_vf]
            self.v[w_vf] = (1.0 - a) * v + a * v.flatten(-2, -1).norm(p=2, dim=-1).unsqueeze(-1).unsqueeze(-1) * f / f.flatten(-2, -1).norm(p=2, dim=-1).unsqueeze(-1).unsqueeze(-1)

            w_N = self.Nsteps > self.Nmin
            w_vfN = w_vf & w_N
            self.dt[w_vfN] = (self.dt[w_vfN] * self.finc).clamp(max=self.dt_max)
            self.a[w_vfN] *= self.fa
            self.Nsteps[w_vfN] += 1

        w_vf = ~w_vf
        if w_vf.all():
            self.v[:] = 0.0
            self.a[:] = torch.tensor(self.astart, device=self.a.device)
            self.dt[:] *= self.fdec
            self.Nsteps[:] = 0
        elif w_vf.any():
            self.v[w_vf] = torch.tensor(0.0, device=self.v.device)
            self.a[w_vf] = torch.tensor(self.astart, device=self.a.device)
            self.dt[w_vf] *= self.fdec
            self.Nsteps[w_vf] = torch.tensor(0, device=self.v.device)

        dt = self.dt.unsqueeze(-1).unsqueeze(-1)
        self.v += dt * forces
        dr = dt * self.v
        normdr = dr.flatten(-2, -1).norm(p=2, dim=-1).unsqueeze(-1).unsqueeze(-1)
        dr *= (self.maxstep / normdr).clamp(max=1.0)
        return coord + dr

    def clean(self, mask):
        # types: (Tensor) -> bool
        self.v = self.v[mask]
        self.Nsteps = self.Nsteps[mask]
        self.dt = self.dt[mask]
        self.a = self.a[mask]
        return True


class EnForce_ANI(torch.nn.Module):
    """Takes in an torch model, then defines two forward functions for it.
    The input model should be able to calculate energy and disp_energy given
    coordiantes, species and charges of a molecule. 

    Arguments:
        ani: model
        batchsize_atoms: the maximum nmber atoms that can be handled in one batch.

    Returns:
        the energies and forces for the input molecules. One time calculation.
    """
    def __init__(self, ani, name, batchsize_atoms=1024*16):
        super().__init__()
        self.add_module('ani', ani)
        self.name = name
        self.batchsize_atoms = batchsize_atoms

    def forward(self, coord, numbers, charges):
        """Calculate the energies and forces for input molecules. Called by self.forward_batched
        
        Arguments:
            coord: coordinates for all input structures. size (B, N, 3), where
                  B is the number of structures in coord, N is the number of
                  atoms in each structure, 3 represents xyz dimensions.
            numbers: the periodic numbers for all atoms.
            
        Returns:
            energies
            forces
        """
        charge = torch.zeros_like(numbers[:, 0])

        if self.name == "AIMNET":
            d = self.ani(dict(coord=coord, numbers=numbers, charge=charges))  #Output from the model
            e = (d['energy'] + d['disp_energy']).to(torch.double)
            g = torch.autograd.grad([e.sum()], [coord])[0]  # size(100, 23, 3)
            assert g is not None
            f = -g
        elif self.name == "ANI2xt":
            e, f = self.ani(numbers, coord)
        elif self.name == "ANI2x":
            e = self.ani((numbers, coord)).energies
            e = e * hartree2eV  #ANI2x (torch.models.ANI2x()) output energy unit is Hatree;
                                     #ANI ASE interface unit is eV
            g = torch.autograd.grad([e.sum()], [coord])[0]
            f = -g
        return e, f

#    @torch.jit.script_method
    def forward_batched(self, coord, numbers, charges):
        """Calculate the energies and forces for input molecules.
        
        Arguments:
            coord: coordinates for all input structures. size (B, N, 3), where
                  B is the number of structures in coord, N is the number of
                  atoms in each structure, 3 represents xyz dimensions.
            numbers: the periodic numbers for all atoms. size (N, )
            
        Returns:
            energies
            forces
        """
        B, N = coord.shape[:2]
        e = []
        f = []
        idx = torch.arange(B, device=coord.device)
        # for batch in idx.split(self.batchsize_atoms // N):  #How was the batchsize_atoms decided?
        for batch in idx.split(self.batchsize_atoms // N):  #How was the batchsize_atoms decided?
            _e, _f = self(coord[batch], numbers[batch], charges[batch])
            e.append(_e)
            f.append(_f)
        return torch.cat(e, dim=0), torch.cat(f, dim=0)

def print_stats(state):
    """Print the optimization status"""
    numbers = state['numbers']
    num_total = numbers.size()[0]
    num_converged = torch.sum(state['converged_mask']).to('cpu')
    num_active = num_total - num_converged
    print("Total 3D structures: %i  Converged: %i   Active: %i" % (num_total, num_converged, num_active))

def n_steps(state, n, opttol):
    """Doing n steps optimization for each input. Only converged structures are 
    modified at each step.
    n_steps does not change input conformer order
    
    Argument:
        state: an dictionary containing all information about this optimization step
        n: optimization step"""
    # t0 = perf_counter()
    numbers = state['numbers']
    charges = state['charges']
    num_total = numbers.size()[0]
    coord = state['coord']
    optimizer = FIRE(coord)
    for istep in tqdm(range(1, (n+1), 1)):
        not_converged = ~ state['converged_mask']  #Size fixed
        
        # stop optimization if all structures converged.
        if not not_converged.any():
            break

        coord = state['coord'][not_converged]  #Subset coordinates, size=not_converged.
        numbers = state['numbers'][not_converged]
        charges = state['charges'][not_converged]

        coord.requires_grad_(True)
        e, f = state['nn'].forward_batched(coord, numbers, charges)  #Key step to calculate all coordinates and forces.
        coord.requires_grad_(False)

        coord = optimizer(coord, f)
        fmax = f.norm(dim=-1).max(dim=-1)[0]  #Size(100), Norm is the length of each vector. Here it returns the maximum force length for ecah conformer. Size (100)
        not_converged_post = fmax > opttol

        optimizer.clean(not_converged_post)  #Subset v, a in FIRE for next optimization
        
        state['converged_mask'][not_converged] = ~ not_converged_post  #Update converged_mask, so that converged structures will not be updated in future steps.
        state['fmax'][not_converged] = fmax  #Update fmax for conformers that are optimized in this iteration
        state['energy'][not_converged] = e.detach()  #Update energy for conformers that are optimized in this iteration
        state['coord'][not_converged] = coord  #Update coordinates for conformers that are optimized in this iteration
    
        if (istep % (n//10)) == 0:
            print_stats(state)
    if istep == (n):
        print("Reaching maximum optimization step:   ", end="")
    else:
        print(f"All structures converged at step {istep}:   ", end="")
    print_stats(state)


def ensemble_opt(net, coord, numbers, charges, param, model, device):
    """Optimizing a group of molecules
    
    Arguments:
    net: a EnForce_ANI object
    coord: coordinates of input molecules (N, m, 3). N is the number of structures
           m is the number of atoms in each structure.
    numbers: periodict number of atoms in the molecule (include H). size is m.
    param: a dictionary containing parameters
    
    """
    coord = torch.tensor(coord, dtype=torch.float, device=device)
    numbers = torch.tensor(numbers, dtype=torch.long, device=device)
    charges = torch.tensor(charges, dtype=torch.long, device=device)
    converged_mask = torch.zeros(coord.shape[0], dtype=torch.bool, device=device)
    fmax = torch.full(coord.shape[:1], 999.0, device=coord.device)  # size=N, a tensored filled with 999.0, representing the current maximum forces at each conformer.
    energy = torch.full(coord.shape[:1], 999.0, dtype=torch.double, device=coord.device)
    ids = torch.arange(coord.shape[0], device=coord.device)  #Returns a 1D tensor
    optimizer = FIRE(coord)

    state = dict(
        ids=ids,
        coord=coord, numbers=numbers, converged_mask=converged_mask,
        optimizer=optimizer, nn=net, fmax=fmax, energy=energy,
        timing=defaultdict(float), charges=charges,
        he=list(), close=list()  #!!! he and close?
    )

    n_steps(state, param['opt_steps'], param['opttol'])

    return dict(
        coord=state['coord'].tolist(),
        ids=state['ids'].tolist(),
        energy=state['energy'].tolist(),
        fmax=state['fmax'].tolist(),
        he=state['he'],
        close=state['close'],
        timing=dict(state['timing']),
        numbers=state['numbers'].tolist()
    )


def padding_coords(lists, pad_value=0.0):
    lengths = [len(lst) for lst in lists]
    pad_length = [max(lengths) - len(lst) for lst in lists]
    assert (len(pad_length) == len(lists))
    
    lists_padded = []
    for i in range(len(pad_length)):
        lst_i = lists[i]
        pad_i = [(pad_value, pad_value, pad_value) for _ in range(pad_length[i])]
        lst_i_padded = lst_i + pad_i
        lists_padded.append(lst_i_padded)
    return lists_padded

def padding_species(lists, pad_value=-1):
    lengths = [len(lst) for lst in lists]
    pad_length = [max(lengths) - len(lst) for lst in lists]
    assert (len(pad_length) == len(lists))
    
    lists_padded = []
    for i in range(len(pad_length)):
        lst_i = lists[i]
        pad_i = [pad_value for _ in range(pad_length[i])]
        lst_i_padded = lst_i + pad_i
        lists_padded.append(lst_i_padded)
    return lists_padded

def mols2lists(mols, model):
    species_order = ("H", 'C', 'N', 'O', 'S', 'F', 'Cl')
    # ani2xt_index = {1:0, 6:1, 7:2, 8:3, 16:4, 9:5, 17:6}
    ani2xt_index = {1:0, 6:1, 7:2, 8:3, 9:4, 16:5, 17:6}
    coord = [[a.coords for a in mol.atoms] for mol in mols]
    charges = [mol.charge for mol in mols]
    if model == "ANI2xt":
        numbers = [[ani2xt_index[a.atomicnum] for a in mol.atoms] for mol in mols]
    else:
        numbers = [[a.atomicnum for a in mol.atoms] for mol in mols]
    return coord, numbers, charges


class optimizing(object):
    def __init__(self, in_f, out_f, model, device, config):
        self.in_f = in_f
        self.out_f = out_f
        # self.energies = energies
        self.model = model
        self.device = device
        self.config = config
        root = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
        if model == "AIMNET":
            self.ani = torch.jit.load(os.path.join(root, "models/aimnet2nqed_pc14iall_b97m_sae.jpt"), map_location=device) # Return a ScriptModule object
        elif model == "ANI2xt":
            self.ani = ANI2xt(device, state_dict=os.path.join(root, "models/ani2xt_seed0.pt"))
        elif model == "ANI2x":
            self.ani = torchani.models.ANI2x(periodic_table_index=True).to(device)
        else:
            raise ValueError("Model has to be ANI2x, ANI2xt or AIMNET.")
        
    def run(self):
        print("Preparing for parallel optimizing... (Max optimization steps: %i)" % self.config["opt_steps"])
        mols = list(pybel.readfile('sdf', self.in_f))
        print(f"Total 3D conformers: {len(mols)}")
        coord, numbers, charges = mols2lists(mols, self.model)
        if self.model == "AIMNET":
            coord_padded = padding_coords(coord, 0)
            numbers_padded = padding_species(numbers, 0)
        else:
            coord_padded = padding_coords(coord, 0)
            numbers_padded = padding_species(numbers, -1)

        for p in self.ani.parameters():  
            p.requires_grad_(False)

        ani = EnForce_ANI(self.ani, self.model)  # Interesting, EnForce_ANI inherites nn.module, bu can still accept a ScriptModule object as the input

        with torch.jit.optimized_execution(False):
            optdict = ensemble_opt(ani, coord_padded, numbers_padded, charges,
                                   self.config, self.model, self.device)  #Magic step

        energies = optdict['energy']
        fmax = optdict['fmax']
        convergence_mask = list(map(lambda x: (x <= self.config['opttol']), fmax))
        with open(self.out_f, 'w') as f:
            for i in range(len(mols)):
                mol = mols[i]
                idx = mol.title
                fmax_i = fmax[i]
                mol.data['E_tot'] = energies[i]
                mol.data['fmax'] = fmax_i
                mol.data['Converged'] = str(convergence_mask[i])
                mol.data['ID'] = idx
                coord = optdict['coord'][i]
                for atom, c in zip(mol.atoms, coord):
                    atom.OBAtom.SetVector(*c)
                f.write(mol.write('sdf'))
